/*
 * Copyright (C) 2013-current Broadcom Corporation
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation version 2.
 *
 * This program is distributed "as is" WITHOUT ANY WARRANTY of any
 * kind, whether express or implied; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/bitops.h>
#include <linux/buffer_head.h>
#include <linux/device.h>
#include <linux/brcmstb/eio_boot.h>
#include <linux/brcmstb/bootloader_message.h>
#include <linux/errno.h>
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/io.h>
#include <linux/jiffies.h>
#include <linux/notifier.h>
#include <linux/of_address.h>
#include <linux/of_irq.h>
#include <linux/of_platform.h>
#include <linux/platform_device.h>
#include <linux/printk.h>
#include <linux/reboot.h>
#include <linux/smp.h>
#include <linux/uaccess.h>

/* Other AON Register Definitions can be found in 'mach-bcm/aon_defs.h'*/
#define AON_REG_ANDROID_RESTART_CAUSE	0x24
#define AON_REG_ANDROID_RESTART_TIME	0x10
#define AON_REG_ANDROID_RESTART_TIME_N	0x14

#define RESTART_CAUSE_QUIESCENT_MARKER 0x100
#define RESTART_CAUSE_DMVERITY_EIO_MARKER 0x200
#define RESTART_CAUSE_FASTBOOT_MARKER 0x400

static void __iomem *aon_sram_base;
static struct eio_boot eiob;

static void reboot_data_save_time_misc(u32 sec)
{
	struct file *filp = NULL;
	int ret = 0;
	loff_t offset;

	/* Open MISC partition */
	filp = filp_open("/dev/block/by-name/misc", O_RDWR, 0);
	if (IS_ERR(filp)) {
		ret = PTR_ERR(filp);
		pr_err("Could not open MISC device (err=%d).\n", ret);
		return;
	}

	/* Write sec to reserved[4-7] to partition */
	offset = offsetof(struct bootloader_message_ab, reserved)
		+ sizeof(u32);
	ret = kernel_write(filp, (char *)&sec, sizeof(u32), &offset);
	if (ret < 0) {
		pr_err("Could not write restart sec to MISC (err=%d).\n", ret);
		goto out;
	} else if (ret != sizeof(u32)) {
		pr_err("MISC sec write bytes mismatch (actual:%d != expected:%u).\n",
			ret, (unsigned int)sizeof(u32));
		goto out;
	}

	/* Write ~sec to reserved[8-11] to partition */
	offset = offsetof(struct bootloader_message_ab, reserved)
		+ (2 * sizeof(u32));
	sec = ~sec;
	ret = kernel_write(filp, (char *)&sec, sizeof(u32), &offset);
	if (ret < 0) {
		pr_err("Could not write restart ~sec to MISC (err=%d).\n", ret);
		goto out;
	} else if (ret != sizeof(u32)) {
		pr_err("MISC ~sec write bytes mismatch (actual:%d != expected:%u).\n",
			ret, (unsigned int)sizeof(u32));
		goto out;
	}

	/* Ensure updated data is properly synced */
	vfs_fsync(filp, 0);
out:
	filp_close(filp, NULL);
}

/* Reboot time is saved in either the AON SRAM or the MISC partition. */
static inline void reboot_data_save_time(bool aon_sram_present)
{
	/* Sync the wall clock into AON SRAM since wktmr gets reset */
	struct timespec64 now;
	u32 sec;

	ktime_get_real_ts64(&now);
	sec = (u32)now.tv_sec + (now.tv_nsec + 500000000) / 1000000000;

	if (aon_sram_present) {
		writel(sec, aon_sram_base + AON_REG_ANDROID_RESTART_TIME);
		writel(~sec, aon_sram_base + AON_REG_ANDROID_RESTART_TIME_N);
	} else
		reboot_data_save_time_misc(sec);
}

/*
 * Process reboot cmd string. Accordingly construct and return reboot
 * data value.
 */
static u32 get_reboot_data(void *cmd)
{
	u32 val;
	char *command = cmd;
	char *quiescent = NULL;
	char *dmverity_corrupt = NULL;
	char *fastboot = NULL;

	/* Save the reboot reason to AON SRAM for bootloader to use.
	 * Do this before writing to the sw master reset reg.
	 * If there is an error in writing to sw master reset reg,
	 * then we will likely need power-on-reset and the AON SRAM
	 * will be cleared as part of the power-on-reset.
	 */
	if (command) {
		quiescent = strstr(command, "quiescent");
		dmverity_corrupt = strstr(command, "dm-verity device corrupted");
		fastboot = strstr(command, " fastboot");
	}

	if (command && command[0] != 0) {
		val = 0;
		/* Save first letter of the reboot command argument to
		 * distinguish different reboot reason. Expected 'cmd':
		 *   - 'bootloader': Boot into bootloader
		 *   - 'recovery': Boot into recovery mode
		 *   - otherwise: Boot into normal Android mode
		 */
		if (strstr(command, "recovery") ||
		    strstr(command, "bootloader"))
			val = command[0];

		pr_info("reboot_data_save: cmd='%s', val=%u\n", command, val);
	} else {
		val = 0;
		pr_info("reboot_data_save: empty cmd string\n");
	}
	if (quiescent) {
		pr_info("reboot_data_save: quiescent mode requested\n");
		val |= RESTART_CAUSE_QUIESCENT_MARKER;
	}
	if (dmverity_corrupt) {
		pr_info("reboot_data_save: dm-verity device corrupted reported\n");
		val |= RESTART_CAUSE_DMVERITY_EIO_MARKER;
	}
	if (fastboot) {
		pr_info("reboot_data_save: fastboot mode requested\n");
		val |= RESTART_CAUSE_FASTBOOT_MARKER;
	}

	return val;
}

static void reboot_data_save_aon(void *cmd)
{
	u32 val;

	val = get_reboot_data(cmd);
	writel(val, aon_sram_base + AON_REG_ANDROID_RESTART_CAUSE);
}

static int reboot_data_save_eio(void *cmd)
{
	struct file *filp = NULL;
	int ret = 0;
	loff_t offset = 0;

	/* Open EIO partition */
	filp = filp_open("/dev/block/by-name/eio", O_RDWR, 0);
	if (IS_ERR(filp)) {
		ret = PTR_ERR(filp);
		pr_err("Could not open EIO device (err=%d).\n", ret);
		return ret;
	}

	/* Read EIO boot header from parition */
	ret = kernel_read(filp, (char *)&eiob, sizeof(struct eio_boot), &offset);
	if (ret < 0) {
		pr_err("Could not read EIO header (err=%d).\n", ret);
		goto out;
	} else if (ret != sizeof(struct eio_boot)) {
		pr_err("EIO header read bytes mismatch (actual:%d != expected:%u).\n",
			ret, (unsigned int)sizeof(struct eio_boot));
		ret = -EIO;
		goto out;
	}

	if (eiob.magic != EIO_BOOT_MAGIC) {
		pr_err("Invalid eio header magic value (actual:%x != expected:%x).\n",
			eiob.magic, EIO_BOOT_MAGIC);
		ret = -EINVAL;
		goto out;
	}

	eiob.spare[EIO_BOOT_REBOOT_DATA_SPARE_SLOT] =
		get_reboot_data(cmd);

	/* Write updated EIO boot header to partition */
	offset = 0;
	ret = kernel_write(filp, (char *)&eiob, sizeof(struct eio_boot), &offset);
	if (ret < 0) {
		pr_err("Could not write EIO header (err=%d).\n", ret);
		goto out;
	} else if (ret != sizeof(struct eio_boot)) {
		pr_err("EIO header write bytes mismatch (actual:%d != expected:%u).\n",
			ret, (unsigned int)sizeof(struct eio_boot));
		ret = -EIO;
		goto out;
	}

	/* Ensure updated data is properly synced */
	vfs_fsync(filp, 0);
out:
	filp_close(filp, NULL);
	return ret;
}

static int reboot_data_save_misc(void *cmd)
{
	struct file *filp = NULL;
	int ret = 0;
	u32 reboot_data;
	/* Use the first reserved of the bootloader_message_ab, this is fine
	 * because all devices using this method are A|B and will be using
	 * this structure.  The reserved data is unused at the moment.
	 */
	loff_t offset = offsetof(struct bootloader_message_ab, reserved);

	/* Open MISC partition */
	filp = filp_open("/dev/block/by-name/misc", O_RDWR, 0);
	if (IS_ERR(filp)) {
		ret = PTR_ERR(filp);
		pr_err("Could not open MISC device (err=%d).\n", ret);
		return ret;
	}

	reboot_data = get_reboot_data(cmd);

	/* Write updated MISC data to partition */
	ret = kernel_write(filp, (char *)&reboot_data, sizeof(u32), &offset);
	if (ret < 0) {
		pr_err("Could not write reboot data to MISC (err=%d).\n", ret);
		goto out;
	} else if (ret != sizeof(u32)) {
		pr_err("MISC data write bytes mismatch (actual:%d != expected:%u).\n",
			ret, (unsigned int)sizeof(u32));
		ret = -EIO;
		goto out;
	}

	/* Ensure updated data is properly synced */
	vfs_fsync(filp, 0);
out:
	filp_close(filp, NULL);
	return ret;
}

static int reboot_data_save_handler(struct notifier_block *this,
				  unsigned long mode, void *cmd)
{
	int rc;

	if (mode != SYS_RESTART)
		return NOTIFY_DONE;

	/*
	 * Save Android reboot data in either AON SRAM (if available)
	 * or either of EIO partition (if available) and MISC partition
	 * (bootloader message). Devices with EIO partition also have a
	 * MISC partition, but not the other way around. If both approaches
	 * fail, complain and move on.
	 */
	if (aon_sram_base) {
		reboot_data_save_aon(cmd);
		reboot_data_save_time(true);
	} else {
		rc = reboot_data_save_eio(cmd);
		if (rc < 0 && rc != -ENOENT)
			pr_err("reboot_data_save: reboot data cannot be saved (eio).\n");
		else {
			rc = reboot_data_save_misc(cmd);
			if (rc < 0)
				pr_err("reboot_data_save: reboot data cannot be saved (misc).\n");

			reboot_data_save_time(false);
		}
	}

	return NOTIFY_DONE;
}

static struct notifier_block reboot_data_save_nb = {
	.notifier_call = reboot_data_save_handler,
	.priority = 128,
};

static int __init reboot_data_save_init(void)
{
	int rc;
	struct device_node *np_aon_ctrl;

	/* The System Data RAM in the AON_CTRL block is used by different
	 * drivers for preserving state across software reset. For backward
	 * compatiblity, we only issue a warning if brcm,brcmstb-aon-ctrl
	 * device tree node cannot be found.*/
	np_aon_ctrl =
		of_find_compatible_node(NULL, NULL, "brcm,brcmstb-aon-ctrl");
	if (!np_aon_ctrl)
		aon_sram_base = NULL;
	else {
		aon_sram_base = of_iomap(np_aon_ctrl, 1);
		WARN(!aon_sram_base, "failed to map aon-sram base");
		of_node_put(np_aon_ctrl);
	}

	rc = register_reboot_notifier(&reboot_data_save_nb);
	if (rc)
		pr_err("cannot register reboot notifier (err=%d)\n", rc);

	return rc;
}

static void __exit reboot_data_save_exit(void)
{
	if (aon_sram_base)
		iounmap(aon_sram_base);

	unregister_reboot_notifier(&reboot_data_save_nb);
}

subsys_initcall(reboot_data_save_init);
module_exit(reboot_data_save_exit);

MODULE_AUTHOR("Broadcom");
MODULE_DESCRIPTION("Broadcom STB reboot data save driver");
MODULE_IMPORT_NS(VFS_internal_I_am_really_a_filesystem_and_am_NOT_a_driver);
MODULE_LICENSE("GPL v2");
