////////////////////////////////////////////////////////////////////////////////
// base.js
////////////////////////////////////////////////////////////////////////////////

// Copyright 2013 the V8 project authors. All rights reserved.
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
//     * Redistributions of source code must retain the above copyright
//       notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above
//       copyright notice, this list of conditions and the following
//       disclaimer in the documentation and/or other materials provided
//       with the distribution.
//     * Neither the name of Google Inc. nor the names of its
//       contributors may be used to endorse or promote products derived
//       from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

if(typeof(WScript) === "undefined")
{
    var WScript = {
        Echo: print
    }
}

// Performance.now is used in latency benchmarks, the fallback is Date.now.
var performance = performance || {};
performance.now = (function() {
  return performance.now       ||
         performance.mozNow    ||
         performance.msNow     ||
         performance.oNow      ||
         performance.webkitNow ||
         Date.now;
})();

// Simple framework for running the benchmark suites and
// computing a score based on the timing measurements.


// A benchmark has a name (string) and a function that will be run to
// do the performance measurement. The optional setup and tearDown
// arguments are functions that will be invoked before and after
// running the benchmark, but the running time of these functions will
// not be accounted for in the benchmark score.
function Benchmark(name, doWarmup, doDeterministic, deterministicIterations, 
                   run, setup, tearDown, rmsResult, minIterations) {
  this.name = name;
  this.doWarmup = doWarmup;
  this.doDeterministic = doDeterministic;
  this.deterministicIterations = deterministicIterations;
  this.run = run;
  this.Setup = setup ? setup : function() { };
  this.TearDown = tearDown ? tearDown : function() { };
  this.rmsResult = rmsResult ? rmsResult : null; 
  this.minIterations = minIterations ? minIterations : 32;
}


// Benchmark results hold the benchmark and the measured time used to
// run the benchmark. The benchmark score is computed later once a
// full benchmark suite has run to completion. If latency is set to 0
// then there is no latency score for this benchmark.
function BenchmarkResult(benchmark, time, latency) {
  this.benchmark = benchmark;
  this.time = time;
  this.latency = latency;
}


// Automatically convert results to numbers. Used by the geometric
// mean computation.
BenchmarkResult.prototype.valueOf = function() {
  return this.time;
}


// Suites of benchmarks consist of a name and the set of benchmarks in
// addition to the reference timing that the final score will be based
// on. This way, all scores are relative to a reference run and higher
// scores implies better performance.
function BenchmarkSuite(name, reference, benchmarks) {
  this.name = name;
  this.reference = reference;
  this.benchmarks = benchmarks;
  BenchmarkSuite.suites.push(this);
}


// Keep track of all declared benchmark suites.
BenchmarkSuite.suites = [];

// Scores are not comparable across versions. Bump the version if
// you're making changes that will affect that scores, e.g. if you add
// a new benchmark or change an existing one.
BenchmarkSuite.version = '9';


// Defines global benchsuite running mode that overrides benchmark suite 
// behavior. Intended to be set by the benchmark driver. Undefined 
// values here allow a benchmark to define behaviour itself.
BenchmarkSuite.config = {
  doWarmup: undefined,
  doDeterministic: undefined
};


// Override the alert function to throw an exception instead.
alert = function(s) {
  throw "Alert called with argument: " + s;
};


// To make the benchmark results predictable, we replace Math.random
// with a 100% deterministic alternative.
BenchmarkSuite.ResetRNG = function() {
  Math.random = (function() {
    var seed = 49734321;
    return function() {
      // Robert Jenkins' 32 bit integer hash function.
      seed = ((seed + 0x7ed55d16) + (seed << 12))  & 0xffffffff;
      seed = ((seed ^ 0xc761c23c) ^ (seed >>> 19)) & 0xffffffff;
      seed = ((seed + 0x165667b1) + (seed << 5))   & 0xffffffff;
      seed = ((seed + 0xd3a2646c) ^ (seed << 9))   & 0xffffffff;
      seed = ((seed + 0xfd7046c5) + (seed << 3))   & 0xffffffff;
      seed = ((seed ^ 0xb55a4f09) ^ (seed >>> 16)) & 0xffffffff;
      return (seed & 0xfffffff) / 0x10000000;
    };
  })();
}


// Runs all registered benchmark suites and optionally yields between
// each individual benchmark to avoid running for too long in the
// context of browsers. Once done, the final score is reported to the
// runner.
BenchmarkSuite.RunSuites = function(runner, skipBenchmarks) {
  skipBenchmarks = typeof skipBenchmarks === 'undefined' ? [] : skipBenchmarks;
  var continuation = null;
  var suites = BenchmarkSuite.suites;
  var length = suites.length;
  BenchmarkSuite.scores = [];
  var index = 0;
  function RunStep() {
    while (continuation || index < length) {
      if (continuation) {
        continuation = continuation();
      } else {
        var suite = suites[index++];
        if (runner.NotifyStart) runner.NotifyStart(suite.name);
        if (skipBenchmarks.indexOf(suite.name) > -1) {
          suite.NotifySkipped(runner);
        } else {
          continuation = suite.RunStep(runner);
        }
      }
      if (continuation && typeof window != 'undefined' && window.setTimeout) {
        window.setTimeout(RunStep, 25);
        return;
      }
    }

    // show final result
    if (runner.NotifyScore) {
      var score = BenchmarkSuite.GeometricMean(BenchmarkSuite.scores);
      var formatted = BenchmarkSuite.FormatScore(100 * score);
      runner.NotifyScore(formatted);
    }
  }
  RunStep();
}


// Counts the total number of registered benchmarks. Useful for
// showing progress as a percentage.
BenchmarkSuite.CountBenchmarks = function() {
  var result = 0;
  var suites = BenchmarkSuite.suites;
  for (var i = 0; i < suites.length; i++) {
    result += suites[i].benchmarks.length;
  }
  return result;
}


// Computes the geometric mean of a set of numbers.
BenchmarkSuite.GeometricMean = function(numbers) {
  var log = 0;
  for (var i = 0; i < numbers.length; i++) {
    log += Math.log(numbers[i]);
  }
  return Math.pow(Math.E, log / numbers.length);
}


// Computes the geometric mean of a set of throughput time measurements.
BenchmarkSuite.GeometricMeanTime = function(measurements) {
  var log = 0;
  for (var i = 0; i < measurements.length; i++) {
    log += Math.log(measurements[i].time);
  }
  return Math.pow(Math.E, log / measurements.length);
}


// Computes the geometric mean of a set of rms measurements.
BenchmarkSuite.GeometricMeanLatency = function(measurements) {
  var log = 0;
  var hasLatencyResult = false;
  for (var i = 0; i < measurements.length; i++) {
    if (measurements[i].latency != 0) {
      log += Math.log(measurements[i].latency);
      hasLatencyResult = true;
    }
  }
  if (hasLatencyResult) {
    return Math.pow(Math.E, log / measurements.length);
  } else {
    return 0;
  }
}


// Converts a score value to a string with at least three significant
// digits.
BenchmarkSuite.FormatScore = function(value) {
  if (value > 100) {
    return value.toFixed(0);
  } else {
    return value.toPrecision(3);
  }
}

// Notifies the runner that we're done running a single benchmark in
// the benchmark suite. This can be useful to report progress.
BenchmarkSuite.prototype.NotifyStep = function(result) {
  this.results.push(result);
  if (this.runner.NotifyStep) this.runner.NotifyStep(result.benchmark.name);
}


// Notifies the runner that we're done with running a suite and that
// we have a result which can be reported to the user if needed.
BenchmarkSuite.prototype.NotifyResult = function() {
  var mean = BenchmarkSuite.GeometricMeanTime(this.results);
  var score = this.reference[0] / mean;
  BenchmarkSuite.scores.push(score);
  if (this.runner.NotifyResult) {
    var formatted = BenchmarkSuite.FormatScore(100 * score);
    this.runner.NotifyResult("### SCORE:", formatted);
  }
  if (this.reference.length == 2) {
    var meanLatency = BenchmarkSuite.GeometricMeanLatency(this.results);
    if (meanLatency != 0) {
      var scoreLatency = this.reference[1] / meanLatency;
      BenchmarkSuite.scores.push(scoreLatency);
      if (this.runner.NotifyResult) {
        var formattedLatency = BenchmarkSuite.FormatScore(100 * scoreLatency)
        this.runner.NotifyResult("### LATENCY:", formattedLatency);
      }
    }
  }
}


BenchmarkSuite.prototype.NotifySkipped = function(runner) {
  BenchmarkSuite.scores.push(1);  // push default reference score.
  if (runner.NotifyResult) {
    runner.NotifyResult(this.name, "Skipped");
  }
}


// Notifies the runner that running a benchmark resulted in an error.
BenchmarkSuite.prototype.NotifyError = function(error) {
  if (this.runner.NotifyError) {
    this.runner.NotifyError(this.name, error);
  }
  if (this.runner.NotifyStep) {
    this.runner.NotifyStep(this.name);
  }
}


// Runs a single benchmark for at least a second and computes the
// average time it takes to run a single iteration.
BenchmarkSuite.prototype.RunSingleBenchmark = function(benchmark, data) {
  var config = BenchmarkSuite.config;
  var doWarmup = config.doWarmup !== undefined 
                 ? config.doWarmup 
                 : benchmark.doWarmup;
  var doDeterministic = config.doDeterministic !== undefined 
                        ? config.doDeterministic 
                        : benchmark.doDeterministic;

  function Measure(data) {
    var elapsed = 0;
    var start = new Date();
  
  // Run either for 1 second or for the number of iterations specified
  // by minIterations, depending on the config flag doDeterministic.
    for (var i = 0; (doDeterministic ? 
      i<benchmark.deterministicIterations : elapsed < 1000); i++) {
      benchmark.run();
      elapsed = new Date() - start;
    }
    if (data != null) {
      data.runs += i;
      data.elapsed += elapsed;
    }
  }

  // Sets up data in order to skip or not the warmup phase.
  if (!doWarmup && data == null) {
    data = { runs: 0, elapsed: 0 };
  }

  if (data == null) {
    Measure(null);
    return { runs: 0, elapsed: 0 };
  } else {
    Measure(data);
    // If we've run too few iterations, we continue for another second.
    if (data.runs < benchmark.minIterations) return data;
    var usec = (data.elapsed * 1000) / data.runs;
    var rms = (benchmark.rmsResult != null) ? benchmark.rmsResult() : 0;
    this.NotifyStep(new BenchmarkResult(benchmark, usec, rms));
    return null;
  }
}


// This function starts running a suite, but stops between each
// individual benchmark in the suite and returns a continuation
// function which can be invoked to run the next benchmark. Once the
// last benchmark has been executed, null is returned.
BenchmarkSuite.prototype.RunStep = function(runner) {
  BenchmarkSuite.ResetRNG();
  this.results = [];
  this.runner = runner;
  var length = this.benchmarks.length;
  var index = 0;
  var suite = this;
  var data;

  // Run the setup, the actual benchmark, and the tear down in three
  // separate steps to allow the framework to yield between any of the
  // steps.

  function RunNextSetup() {
    if (index < length) {
      try {
        suite.benchmarks[index].Setup();
      } catch (e) {
        suite.NotifyError(e);
        return null;
      }
      return RunNextBenchmark;
    }
    suite.NotifyResult();
    return null;
  }

  function RunNextBenchmark() {
    try {
      data = suite.RunSingleBenchmark(suite.benchmarks[index], data);
    } catch (e) {
      suite.NotifyError(e);
      return null;
    }
    // If data is null, we're done with this benchmark.
    return (data == null) ? RunNextTearDown : RunNextBenchmark();
  }

  function RunNextTearDown() {
    try {
      suite.benchmarks[index++].TearDown();
    } catch (e) {
      suite.NotifyError(e);
      return null;
    }
    return RunNextSetup;
  }

  // Start out running the setup.
  return RunNextSetup();
}

//////////////////////////////////////////////////////////////////////
//      mandreel.js
//////////////////////////////////////////////////////////////////////
// Portions copyright 2012 Google, Inc.
// Copyright 2012 Onan Games. All rights reserved.
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
//     * Redistributions of source code must retain the above copyright
//       notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above
//       copyright notice, this list of conditions and the following
//       disclaimer in the documentation and/or other materials provided
//       with the distribution.
//     * Neither the name of Google Inc. nor the names of its
//       contributors may be used to endorse or promote products derived
//       from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

var MandreelBenchmark = new BenchmarkSuite('Mandreel', [16831872, 49852],
                                            [new Benchmark('Mandreel',
                                                           false,
                                                           false,
                                                           15,
                                                           runMandreel,
                                                           setupMandreel,
                                                           tearDownMandreel,
                                                           RMSMandreel,
                                                           4)]);

var mandreelSumSquaredPauses = 0;
var mandreelSamples = 0;
var mandreelSampleTimeStart = 0.0;

function setupMandreel() {
  // Check for Typed Arrays support, throw error if not.
  if (!(typeof Uint8Array != "undefined" &&
    typeof Float64Array != "undefined" &&
    typeof (new Uint8Array(0)).subarray != "undefined")) {
      throw "TypedArrayUnsupported";
  }

  my_old_constructors = mandreel_call_constructors;
  mandreel_call_constructors = my_mandreel_call_constructors;
  startMandreelTimer();
  startApp();
  _mandreelAppAudioReady();
}

function runMandreel() {
  Mandreel_currentTime = 0;
  var sp = g_stack_pointer+800*1024;
  for (var i=0;i<mandreel_total_memory/4;i++) {
    heap32[i] = my_heap32[i];
  }
  tlsf_ptr = 0;
  heapNewPos = my_heapNewPos;
  my_old_constructors(llvm_2E_global_ctors,5,sp);
  heapU32[sp>>2] = 640;
  heapU32[(sp+4)>>2] = 480;
  __mandreel_internal_SetResolution(sp);
  __mandreel_internal_init(g_stack_pointer+800*1024);
  __init(g_stack_pointer+800*1024);
  for (var i = 0; i < 20; i++) {
    render();
    Mandreel_flushTimeouts();
    updateMandreelStats(performance.now());
  }
  Mandreel_checkState();
}

function updateMandreelStats(time) {
  var pause = time - mandreelSampleTimeStart;
  mandreelSampleTimeStart = time;
  mandreelSumSquaredPauses += (pause * pause);
  mandreelSamples++;
}

function startMandreelTimer() {
  mandreelSampleTimeStart = performance.now();
}

function RMSMandreel() {
  return Math.round(Math.sqrt(mandreelSumSquaredPauses / mandreelSamples) * 100);
}

function tearDownMandreel() {
  my_old_constructors = null;
  my_heap = null;
  my_heap8 = null;
  my_heap32 = null;

  heap = null;
  heap8 = null;
  heapU8 = null;
  heap16 = null;
  heapU16 = null;
  heap32 = null;
  heapU32 = null;
  heapFloat = null;
  heapDouble = null;
  mandreelAppUsePackAsyncTexture = null;
  mandreelSumSquaredPauses = 0;
  mandreelSamples = 0;
}

// Mocks for browser functions.

function Mandreel_setTimeout(cmd, delay) {
  timeouts.push(cmd);
}

function Mandreel_flushTimeouts() {
  while (Mandreel_timeouts.length != 0) {
    var next = Mandreel_timeouts.pop();
    eval(next);
  }
}

Mandreel_timeouts = new Array();

Mandreel_XMLHttpRequest = function() {
  this.open = function(type, url, some_bool) {
    this.url = url;
  }
  this.overrideMimeType = function() {
  }
  this.send = function() {
    this.response = null;
    this.readyState = 4;
    this.status = 0;
    this.onreadystatechange();
  }
};

function Mandreel_Element(type) {
  this.element_type = type;
  this.insertBefore = function() {
  }
}


function Mandreel_Context() {
}

function Mandreel_Canvas() {
}

function Mandreel_CanvasDiv() {
}

Mandreel_document = {
  createElement : function(element_type) {
    var element = new Mandreel_Element(element_type);
    element.parentNode = new Mandreel_Element("dummy_parent");
    return element;
  },
  getElementById : function(name) {
    if (name === "canvas") {
      return new Mandreel_Canvas();
    } else if (name === "canvasDiv") {
      return new Mandreel_CanvasDiv();
    } else {
      return undefined;
    }
  },
  getElementsByTagName : function(element) {
    if (element === "script") {
      return new Array(new this.createElement(element));
    }
  }

};

Mandreel_window = {
  WebGLRenderingContext: {},
  Float64Array: Float64Array,
  Float32Array: Float32Array,
  Int32Array: Int32Array,
  Uint32Array: Uint32Array,
  Int16Array: Int16Array,
  Uint16Array: Uint16Array,
  Int8Array: Int8Array,
  Uint8Array: Uint8Array,
  setTimeout: Mandreel_setTimeout,
  addEventListener : function () {},
  document: Mandreel_document
};

function dump(x) { }

alert = typeof alert != "undefined" ? alert : function(x) {
  print(x);
}

var my_old_constructors;
var my_heap;
var my_heap8;
var my_heap32;
var my_heapNewPos;

function my_mandreel_call_constructors(_ptr, size,stackPos) {
  my_heapNewPos = heapNewPos;
  my_heap = new ArrayBuffer(mandreel_total_memory);
  my_heap8 = new Int8Array(my_heap);
  my_heap32 = new Int32Array(my_heap);
  for (var i=0;i<mandreel_total_memory/4;i++) {
    my_heap32[i] = heap32[i];
  }
  my_old_constructors(_ptr,size,stackPos);
}


var Mandreel_currentTime = 0;

function Date_now() {
   Mandreel_currentTime += 16;
   return Mandreel_currentTime;
}

function Mandreel_checkState() {
  var sum = 0;
  for (var i = 0; i < heap32.length; i += 100) {
    sum = (sum * 3 + heap32[i]) & 0xFFFFFF;
  }
  if (sum != 8001026) {
    alert("Check sum mismatch: expected ???, actual " + sum);
  }
}

// Original Mandreel code follows.
// Modifications for benchmarking are marked in comments.

// Start of js/main.js file:

////////////////////////////////////////////
function startApp(_platform)
{
        // Start mandreel
        var params =
        {
                platform : _platform,
                width : 1024,
                height : 768,
                webglCanvas : "canvas",
                flashCanvas : "FlashDiv",
                workingFolderFlash : "data/as3/",
                workingFolderWebgl : "data/js/",
                swfFlash : "mandreel.swf",
                log : true
    };
        mandreelAppStart(appStartState,params);
}

////////////////////////////////////////////
function appStartState(state,param)
{
        // mandreel.js program is been loaded
        if ( state == "loadingScript" )
        {
    }

        // mandreel.js program has been loaded
        if ( state == "scriptLoaded" )
        {
    }

        // loading pack data file
        if (state == "loadingData")
        {
                // param bytes loaded
    }

        // Audio system is been started
        if ( state == "loadingAudio" )
        {
    }

        // Audio system is ready and the default audio preloading has been done
        if ( state == "audioLoaded" )
        {
    }



        // Mandreel has been started, render will start automatically
        if ( state == "ready" )
        {
                // Hide loading image
                var canvasElement = Mandreel_document.getElementById('loading');
                if ( canvasElement != null )
                        canvasElement.style.visibility = "hidden";
    }

        // An error has been produced during the start process and the app must quit
        if ( state == "error" )
        {
                if ( param == "webgl_not_found" )
                {
                        window.location = "http://get.webgl.org";
                        return;
            }
                alert(param);
    }
}

// End of js/main.js file.

// Start of js/mandreelapp.js file.

var mandreelAppMandreelJs = "mandreel.js";
var mandreelAppMandreelJsCompressed = false;
var mandreelAppWorkingFolder = "data/js/";
var mandreelAppLog = false;
var mandreelAppLocalHost = "http://localhost";
var mandreelAppReadDataFromLocalHost = false;
var mandreelAppReadMandreelJsFromLocalHost = false;
var mandreelAppHostedAudioServer = null;
var mandreelAppHostedAudioUrl = null;
var mandrelCurrentFatVersion = "1.4";
var mandreelAppPlatform = "webgl";
var mandreelAppCanvasWidth = 1024;
var mandreelAppCanvasHeight = 768;
var mandreelAppWidthSrc = 1024;
var mandreelAppHeightSrc = 768;
var mandreelAppCanvasName = "canvas";
var mandreelAppCanvasDiv = "canvasDiv";
var mandreelAppUseFlashSockets = false;
var mandreelAppUsePackAsyncTexture = new Array();
//var mandreelBufferPackAsyncTexture = null;
var mandreelAppForceFocus = true;
var _imandreel_pause_game = false;

/* The following code was removed for benchmarking:
navigator.pointer = navigator.pointer || navigator.webkitPointer || navigator.mozPointer || null;*/

/* The following code was removed for benchmarking:
 if (!Date_now) {
   Date_now = function() {
     return +new Date();
   };
 };*/

////////////////////////////////////////////

if (Mandreel_window["console"])
{
        if (!Mandreel_window["dump"]) Mandreel_window["dump"] = function dump(str){ if ( mandreelAppLog ) console.log(str) };
}
else
{
        if (!Mandreel_window["dump"]) Mandreel_window["dump"] = function dump(str){ };
}

/* The following code is removed for benchmarking:
var mandreel_BrowserDetect = {
        init: function () {
                this.browser = this.searchString(this.dataBrowser) || "An unknown browser";
                this.version = this.searchVersion(navigator.userAgent)
                        || this.searchVersion(navigator.appVersion)
                        || "an unknown version";
                this.OS = this.searchString(this.dataOS) || "an unknown OS";
    },
        searchString: function (data) {
                for (var i=0;i<data.length;i++) {
                        var dataString = data[i].string;
                        var dataProp = data[i].prop;
                        this.versionSearchString = data[i].versionSearch || data[i].identity;
                        if (dataString) {
                                if (dataString.indexOf(data[i].subString) != -1)
                                        return data[i].identity;
                    }
                        else if (dataProp)
                                return data[i].identity;
            }
    },
        searchVersion: function (dataString) {
                var index = dataString.indexOf(this.versionSearchString);
                if (index == -1) return;
                return parseFloat(dataString.substring(index+this.versionSearchString.length+1));
    },
        dataBrowser: [
                {
                        string: navigator.userAgent,
                        subString: "Chrome",
                        identity: "Chrome"
            },
                {       string: navigator.userAgent,
                        subString: "OmniWeb",
                        versionSearch: "OmniWeb/",
                        identity: "OmniWeb"
            },
                {
                        string: navigator.vendor,
                        subString: "Apple",
                        identity: "Safari",
                        versionSearch: "Version"
            },
                {
                        prop: window.opera,
                        identity: "Opera",
                        versionSearch: "Version"
            },
                {
                        string: navigator.vendor,
                        subString: "iCab",
                        identity: "iCab"
            },
                {
                        string: navigator.vendor,
                        subString: "KDE",
                        identity: "Konqueror"
            },
                {
                        string: navigator.userAgent,
                        subString: "Firefox",
                        identity: "Firefox"
            },
                {
                        string: navigator.vendor,
                        subString: "Camino",
                        identity: "Camino"
            },
                {               // for newer Netscapes (6+)
                        string: navigator.userAgent,
                        subString: "Netscape",
                        identity: "Netscape"
            },
                {
                        string: navigator.userAgent,
                        subString: "MSIE",
                        identity: "Explorer",
                        versionSearch: "MSIE"
            },
                {
                        string: navigator.userAgent,
                        subString: "Gecko",
                        identity: "Mozilla",
                        versionSearch: "rv"
            },
                {               // for older Netscapes (4-)
                        string: navigator.userAgent,
                        subString: "Mozilla",
                        identity: "Netscape",
                        versionSearch: "Mozilla"
            }
        ],
        dataOS : [
                {
                        string: navigator.platform,
                        subString: "Win",
                        identity: "Windows"
            },
                {
                        string: navigator.platform,
                        subString: "Mac",
                        identity: "Mac"
            },
                {
                           string: navigator.userAgent,
                           subString: "iPhone",
                           identity: "iPhone/iPod"
            },
                {
                        string: navigator.platform,
                        subString: "Linux",
                        identity: "Linux"
            }
        ]

};
mandreel_BrowserDetect.init(); */

////////////////////////////////////////////
var mandreel_packfiledata_name = null;
var mandreel_packfiledata_compressed = false;
var mandreel_packfiledata_size = 0;
var mandreel_total_packtexture_size = 0;
var mandreel_total_pogfile_size = 0;
var mandreel_loaded_packtexture_size = 0;
var mandreel_jslzma_size = 0;
var mandreel_swf_size = 0;

var mandreelJsScriptLoaded_loaded = false;

var mandreel_swf_last_total_size = 0;
function mandreel_swf_size_updated(str)
{
        var params = str.split(',');
        if ( mandreel_swf_size == 0 )
        {
                mandreel_swf_last_loaded_size = 0;
    }
        mandreel_swf_size = parseInt(params[0]);
        var loaded = parseInt(params[1]);
        var delta_size = loaded - mandreel_swf_last_loaded_size;
        mandreel_swf_last_loaded_size = loaded;

        var percentage = ((100*loaded)/mandreel_swf_size)|0;
        if (percentage>100)
                percentage = 100;

        if ( mandreelAppStartStateFunc )
                mandreelAppStartStateFunc("loadingScriptUpdate",percentage);

        imandreel_update_load(delta_size,0);
}

function mandreel_swf_size_loaded(str)
{
        if ( mandreelAppStartStateFunc )
                mandreelAppStartStateFunc("scriptLoaded","");
}

function mandreelNextDecompress(mandreel_result_lzma)
{

        if ( mandreelAppStartStateFunc )
        {
                var totalBytesLeft = mandreel_result_lzma.totalSize - mandreel_result_lzma.remainingBytes;

                var percentage;

                if (totalBytesLeft == 0)
                        percentage = 0;
                else
                        percentage = ((100*totalBytesLeft)/mandreel_result_lzma.totalSize)|0;

                mandreelAppStartStateFunc("uncompressingDataUpdate",percentage);
    }

        var old_result = mandreel_result_lzma;
        mandreel_result_lzma = LZMA.decompress2(mandreel_result_lzma.inStream,mandreel_result_lzma.inStream,mandreel_result_lzma.outStream,mandreel_result_lzma);

        if (mandreel_result_lzma == null)
        {
                //setTimeout(mandreelLoadScript,10,old_result.my_outStream.data);

                //mandreel_fs_saveFile('data.bin', old_result.my_outStream.arrayBuffer);
                //callback(old_result.my_outStream.arrayBuffer);
                //alert('done');
                mandreelLoadPackData(old_result.my_outStream.arrayBuffer,true);
    }
        else
        {
                Mandreel_setTimeout(mandreelNextDecompress,10,mandreel_result_lzma);
    }


}

function mandreel_load_packfile(array_buffer)
{
        if (array_buffer)
        {
                mandreelLoadPackData(array_buffer,false);
                return;
    }

        var working_folder = mandreelAppWorkingFolder;
        if ( mandreelAppReadDataFromLocalHost )
                working_folder = mandreelAppLocalHost+"/"+mandreelAppWorkingFolder;
        var packdata_request = new XMLHttpRequest();
        var url = working_folder+mandreel_packfiledata_name;

        packdata_request.open("GET", url, true);

        if("responseType" in packdata_request)
                packdata_request.responseType="arraybuffer";
        else
                packdata_request.overrideMimeType('text/plain; charset=x-user-defined');

        var last_loaded_size = 0;

        packdata_request.onreadystatechange = function()
        {
                if (packdata_request.readyState != 4) return;

                if (packdata_request.status == 200)
                {
                        var buffer;
                        if (packdata_request.responseType=="arraybuffer")
                                buffer=packdata_request.response;
                        else if (packdata_request.mozResponseArrayBuffer != null)
                                buffer = packdata_request.mozResponseArrayBuffer;
                        else
                                buffer=packdata_request.response;

                        if (mandreel_packfiledata_compressed)
                        {
                                var inStream = {
                                  data: new Uint8Array(buffer),
                                  offset: 0,
                                  readByte: function(){
                                        return this.data[this.offset ++];
                                  }
                            };

                                var outStream = {
                                  data: null,
                                  offset: 0,
                                  binary_mode : true,
                                  writeByte: function(value){
                                        this.data[this.offset ++] = value;
                                  }
                            };


                                var result = LZMA.decompress2(inStream,inStream,outStream,null);

                                if (result == null)
                                        mandreelLoadPackData(outStream.arrayBuffer,true);
                                else
                                        Mandreel_setTimeout(mandreelNextDecompress,10,result);
                    }
                        else
                                mandreelLoadPackData(buffer,true);
            }
                else
                {
                        if ( mandreelAppStartStateFunc )
                                mandreelAppStartStateFunc("error","can't load packfile data");
            }
    }
        packdata_request.onprogress = function(e)
        {
                var delta_size = e.loaded - last_loaded_size;
                last_loaded_size = e.loaded;
                var percentage = ((100*e.loaded)/mandreel_packfiledata_size)|0;


                imandreel_update_load(delta_size,0);

                if (percentage>100)
                        percentage = 100;

                if ( mandreelAppStartStateFunc )
                        mandreelAppStartStateFunc("loadingData",percentage);
    }

        packdata_request.send();
}

function mandreelJsScriptLoaded()
{
        if (mandreelJsScriptLoaded_loaded)
                return;

        if (typeof(mandreel_cache_files)=="undefined")
        {
                Mandreel_setTimeout(mandreelJsScriptLoaded,10);
                return;
    }

        mandreelJsScriptLoaded_loaded = true;

        if ( mandreelAppStartStateFunc )
                mandreelAppStartStateFunc("scriptLoaded","");

        if (mandreel_packfiledata_name)
        {

                mandreel_fs_load_binary(mandreel_packfiledata_name, mandreel_load_packfile);


    }
        else
        {
                mandreelCacheMandreelFat();
                imandreelJsScriptLoaded();
    }
}

function imandreelLoadAudio()
{
        g_mandreel_working_folder = mandreelAppWorkingFolder;
        if ( mandreelAppReadDataFromLocalHost )
                g_mandreel_working_folder = mandreelAppLocalHost+"/"+mandreelAppWorkingFolder;

        // load audio
        if ( mandreelAppStartStateFunc )
                mandreelAppStartStateFunc("loadingAudio","");
        mandreel_start_audio(mandreelAppHostedAudioServer,mandreelAppHostedAudioUrl,_mandreelAppAudioReady);
}

var _mandreel_currentPackTexture = '';

function imandreel_packasynctexture_loaded(buffer,save_file)
{
        if (save_file)
                mandreel_fs_saveFile(_mandreel_currentPackTexture,buffer);

        //mandreelBufferPackAsyncTexture = buffer;
        Mandreel_TextureAsync_PackBufferData[_mandreel_currentPackTexture] = buffer;

        imandreelLoadNextTexturePack();
}

var mandreel_update_total_size = 0;

function imandreel_update_load(size, total)
{
        if (total == 0)
                total = mandreel_packfiledata_size + mandreel_total_packtexture_size + mandreel_total_pogfile_size + mandreel_jslzma_size + mandreel_swf_size;


        mandreel_update_total_size+=size;

        var percentage = ((100*mandreel_update_total_size)/total)|0;

        if (percentage>100)
                percentage = 100;
        if (mandreelAppStartStateFunc)
                mandreelAppStartStateFunc("loadingProgress",percentage);

}


function imandreel_packasynctexture_load(array_buffer)
{
        if (array_buffer)
        {
                imandreel_packasynctexture_loaded(array_buffer, false);
                return;
    }


        var working_folder = mandreelAppWorkingFolder;
        if ( mandreelAppReadDataFromLocalHost )
                working_folder = mandreelAppLocalHost+"/"+mandreelAppWorkingFolder;
        var packdata_request = new XMLHttpRequest();
        var url = working_folder+_mandreel_currentPackTexture;

        packdata_request.open("GET", url, true);

        if("responseType" in packdata_request)
                packdata_request.responseType="arraybuffer";
        else
                packdata_request.overrideMimeType('text/plain; charset=x-user-defined');

        var last_loaded_size = 0;

        packdata_request.onreadystatechange = function()
        {
                if (packdata_request.readyState != 4) return;

                if (packdata_request.status == 200)
                {
                        var buffer;
                        if (packdata_request.responseType=="arraybuffer")
                                buffer=packdata_request.response;
                        else if (packdata_request.mozResponseArrayBuffer != null)
                                buffer = packdata_request.mozResponseArrayBuffer;
                        else
                                buffer=packdata_request.response;

                        imandreel_packasynctexture_loaded(buffer, true);
            }
                else
                {
                        if ( mandreelAppStartStateFunc )
                                mandreelAppStartStateFunc("error","can't load textureasync pack data");
            }
    }
        packdata_request.onprogress = function(e)
        {
                var delta_size = e.loaded - last_loaded_size;
                last_loaded_size = e.loaded;
                mandreel_loaded_packtexture_size+=delta_size;

                imandreel_update_load(delta_size,0);

                var percentage = ((100*mandreel_loaded_packtexture_size)/mandreel_total_packtexture_size)|0;

                if (percentage>100)
                        percentage = 100;

                if ( mandreelAppStartStateFunc )
                        mandreelAppStartStateFunc("loadingTextureAsyncPack",percentage);
    }

        packdata_request.send();

}

function imandreelLoadNextTexturePack()
{
        if (mandreelAppUsePackAsyncTexture.length)
        {
                _mandreel_currentPackTexture = mandreelAppUsePackAsyncTexture.pop();
                mandreel_fs_load_binary(_mandreel_currentPackTexture, imandreel_packasynctexture_load);
    }
        else
                imandreelLoadAudio();
}

function imandreelJsScriptLoaded()
{
        imandreelLoadNextTexturePack();
}

////////////////////////////////////////////

function mandreelDecompressJSReady(code, save_file)
{
        if (save_file)
                mandreel_fs_saveFile(mandreelAppMandreelJs + ".lzma", code);

        var ga = Mandreel_document.createElement('script');
        ga.type = "text/javascript";
        ga.text = code;
        var s = Mandreel_document.getElementsByTagName('script')[0]; s.parentNode.insertBefore(ga, s);
        s.parentNode.removeChild(ga);
        mandreelJsScriptLoaded();
}

function mandreelNextDecompressJS(mandreel_result_lzma)
{

        if ( mandreelAppStartStateFunc )
        {
                var totalBytesLeft = mandreel_result_lzma.totalSize - mandreel_result_lzma.remainingBytes;

                var percentage;

                if (totalBytesLeft == 0)
                        percentage = 0;
                else
                        percentage = ((100*totalBytesLeft)/mandreel_result_lzma.totalSize)|0;

                mandreelAppStartStateFunc("uncompressingScriptUpdate",percentage);
    }

        var old_result = mandreel_result_lzma;
        mandreel_result_lzma = LZMA.decompress2(mandreel_result_lzma.inStream,mandreel_result_lzma.inStream,mandreel_result_lzma.outStream,mandreel_result_lzma);

        if (mandreel_result_lzma == null)
        {
                mandreelDecompressJSReady(old_result.my_outStream.data,true);
    }
        else
        {
                Mandreel_setTimeout(mandreelNextDecompressJS,10,mandreel_result_lzma);
    }


}

function mandreel_load_compressed_js(code_js)
{
        if (code_js)
        {
                mandreelDecompressJSReady(code_js,false);
                return;
    }
         var xmlhttp_get = new XMLHttpRequest();

         var url = mandreelAppMandreelJs + ".lzma";
         if ( mandreelAppReadMandreelJsFromLocalHost )
                url = mandreelAppLocalHost+"/"+url;

         xmlhttp_get.open('GET',url);


        if("responseType" in xmlhttp_get)
                xmlhttp_get.responseType="arraybuffer";
        else
                xmlhttp_get.overrideMimeType('text/plain; charset=x-user-defined');

        var last_loaded_size = 0;

        xmlhttp_get.onreadystatechange = function()
        {
                if (xmlhttp_get.readyState==4)
                {
                        if (xmlhttp_get.status==200 || xmlhttp_get.status==0)
                        {

                                var inStream = {
                                  data: new Uint8Array(xmlhttp_get.response),
                                  offset: 0,
                                  readByte: function(){
                                        return this.data[this.offset ++];
                                  }
                            };

                                var outStream = {
                                  data: new String(""),
                                  offset: 0,
                                  binary_mode : false,
                                  writeByte: function(value){
                                        this.data+=String.fromCharCode(value);
                                  }
                            };


                                var result = LZMA.decompress2(inStream,inStream,outStream,null);
                                if (result == null)
                                        mandreelDecompressJSReady(outStream.data,true);
                                else
                                        Mandreel_setTimeout(mandreelNextDecompressJS,10,result);
                    }
                        else
                        {
                                alert('error ' + xmlhttp_get.status);
                    }
            }

    }

        xmlhttp_get.onprogress = function(e)
        {
                var delta_size = e.loaded - last_loaded_size;
                last_loaded_size = e.loaded;

                var percentage = ((100*e.loaded)/e.total)|0;

                mandreel_jslzma_size = e.total;

                imandreel_update_load(delta_size,0);

                if (percentage>100)
                        percentage = 100;

                if ( mandreelAppStartStateFunc )
                        mandreelAppStartStateFunc("loadingScriptUpdate",percentage);
    }

        xmlhttp_get.send();

}

function mandreelLoadMandreelJsScript()
{
        if (mandreelAppMandreelJsCompressed)
        {
                mandreel_fs_load_text(mandreelAppMandreelJs + ".lzma", mandreel_load_compressed_js);

    }
        else
        {
                var ga = Mandreel_document.createElement('script');
                ga.type = 'text/javascript';
                ga.async = true;
                ga.onload = ga.onreadystatechange = mandreelJsScriptLoaded;
                var url = mandreelAppMandreelJs;
                if ( mandreelAppReadMandreelJsFromLocalHost )
                        ga.src = mandreelAppLocalHost+"/"+url;
                else
                        ga.src = url;
                var s = Mandreel_document.getElementsByTagName('script')[0];
                if ( mandreelAppStartStateFunc )
                        mandreelAppStartStateFunc("loadingScript","");
                s.parentNode.insertBefore(ga, s);
    }
}

////////////////////////////////////////////
function mandreelFatLoaded()
{
        if ( mandreelAppPlatform == "nacl" )
        {
                g_mandreel_working_folder = mandreelAppWorkingFolder;
                if ( mandreelAppReadDataFromLocalHost )
                        g_mandreel_working_folder = mandreelAppLocalHost+"/"+mandreelAppWorkingFolder;

                // load audio
                if ( mandreelAppStartStateFunc )
                        mandreelAppStartStateFunc("loadingAudio","");
                _mandreelAppAudioReady();
    }
        else
                mandreelLoadMandreelJsScript();
}

var mandreelFatData = "";
var mandreelFatPreloadRequest = 0;
////////////////////////////////////////////

function mandreel_chanka_fixfile(fileName)
{
        var fileNameNoExt = fileName.toLowerCase();
        fileNameNoExt = fileNameNoExt.replace(/\\/g,"/");

        fileNameNoExt =  fileNameNoExt.substr(1);

        var new_fileName = '';

        for(var j = 0; j < fileNameNoExt.length; j++)
        {
                var data = fileNameNoExt.charCodeAt(j);

                if (data != 13)
                {
                         var t = String.fromCharCode(data);
                         new_fileName+=t;
            }
    }

        return  new_fileName;
}

function mandreel_removecr(my_string)
{
        var new_string = '';
        for(var j = 0; j < my_string.length; j++)
        {
                var data = my_string.charCodeAt(j);

                if (data != 13)
                {
                         var t = String.fromCharCode(data);
                         new_string+=t;
            }
    }

        return  new_string;
}

function mandreelCacheMandreelFat()
{
        var array_mandreel_fat = new ArrayBuffer(mandreelFatData.length+1);


        {
                var j;
                var len = mandreelFatData.length;
                var data_char;

                var my_bytes = new Uint8Array(array_mandreel_fat);

                for(j = 0; j < len; j++)
                {
                        data_char = mandreelFatData.charCodeAt(j);

                        my_bytes[j] = data_char;
            }
                my_bytes[j] = 0;
    }


        mandreel_cache_files['mandreel.fat'] = array_mandreel_fat;
}

function mandreelLoadPackData(data, save_pack)
{
        var files = mandreelFatData.split('\n');

        var current_dir = '';
        var current_offset = 0;

        if (save_pack)
                mandreel_fs_saveFile(mandreel_packfiledata_name,data);


        for (var i=0;i<files.length;++i)
        {
                var file_data = files[i].split(',');

                if (file_data[0] == 'dir')
                {
                        current_dir = file_data[1];
                        //current_dir = current_dir.substr(0,current_dir.lengh-1);
            }
                else if (file_data[0] == 'file')
                {
                        var file_name = current_dir + file_data[1];
                        file_name = mandreel_chanka_fixfile(file_name) ;
                        //dump('new file ' + file_name + ' ' + current_offset);
                        //dump(file_name);
                        var file_size = parseInt(file_data[2]);

                        var my_array = new ArrayBuffer(file_size);

                        var my_bytes = new Uint8Array(my_array);

                        var data_bytes = new Uint8Array(data,current_offset,file_size);

                        my_bytes.set(data_bytes);

                        mandreel_cache_files[file_name] = my_array;


                        current_offset+=file_size;


            }
    }



        g_mandreel_datafiles_sufix = '';
        mandreelCacheMandreelFat();

        imandreelJsScriptLoaded();
}

var preCreatedWebAudioContext = null;

function mandreelLoadFat()
{
        mandreelFatPreloadRequest = new Mandreel_XMLHttpRequest();
        var working_folder = mandreelAppWorkingFolder;
        if ( mandreelAppReadDataFromLocalHost )
                working_folder = mandreelAppLocalHost+"/"+mandreelAppWorkingFolder;
        var url = working_folder+"mandreel.fat.dat";
        if ( mandreelAppPlatform == "nacl" )
                url = working_folder+"mandreel.fat";
        mandreelFatPreloadRequest.open("GET", url, true);
        mandreelFatPreloadRequest.onreadystatechange = function()
        {
                if (mandreelFatPreloadRequest.readyState != 4) return;
                if ( mandreelFatPreloadRequest.status != 404 && mandreelFatPreloadRequest.response != null )
                {
                        mandreelFatData = mandreelFatPreloadRequest.response;
            }
                if ( mandreelFatData == "" )
                {
                        dump("error loading mandreel.fat file, Maybe the working folder is not correctly set???");
            }

                var packfiledata = null;
                var compressed = false;
                var packfiledata_size = 0;
                var total_packtexture_size = 0;
                var pogsize = 0;

                // Check version
                var FatLines = mandreelFatData.split('\n');
                for ( var i=0;i<FatLines.length;++i )
                {
                        var line = mandreel_removecr(FatLines[i]);
                        var params = line.split(',');
                        if ( params[0] == "version" )
                        {
                                if ( params[1] != mandrelCurrentFatVersion )
                                        dump("warning: mandreel.fat version number is ("+params[1]+") and it should be ("+mandrelCurrentFatVersion+")");

                    }
                        else if ( params[0] == "platform" && params[1] != "js" && (mandreelAppPlatform == "webgl" || mandreelAppPlatform == "canvas")  )
                                dump("warning: mandreel.fat platform is ("+params[1]+") and it should be (js)");
                        else if ( params[0] == "platform" && params[1] != "nacl" && mandreelAppPlatform == "nacl"  )
                                dump("warning: mandreel.fat platform is ("+params[1]+") and it should be (nacl)");
                        else if (params[0] == 'packdata')
                        {
                                packfiledata = params[1];
                                compressed = params[2].charAt(0) == '1';
                                packfiledata_size =  parseInt(params[3], 10);
                    }
                        else if (params[0] == 'flashsockets')
                        {
                                mandreelAppUseFlashSockets = true;
                    }
                        else if (params[0] == 'packtexture')
                        {
                                var filename = params[1];
                                var size_packtexture = parseInt(params[3], 10);
                                total_packtexture_size+=size_packtexture;
                                mandreelAppUsePackAsyncTexture.push(filename);
                    }
                        else if (params[0] == 'audiopreloadfile')
                        {
                                pogsize = parseInt(params[2],10);
                    }
                        else if (params[0] == 'audiodriver' && params[1] == 'webaudio')
                        {
                                try     { preCreatedWebAudioContext = new webkitAudioContext(); } catch(err) { preCreatedWebAudioContext = null; }
                    }
            }

                if ( preCreatedWebAudioContext != null )
                        mandreel_total_pogfile_size = pogsize;
                mandreel_packfiledata_name = packfiledata;
                mandreel_packfiledata_compressed = compressed;
                mandreel_packfiledata_size = packfiledata_size;
                mandreel_total_packtexture_size = total_packtexture_size;
                mandreelFatLoaded();
    }
        mandreelFatPreloadRequest.send();
}

var mandreelAppStartStateFunc = 0;
var mandreelDisableSpaceKey = true;
////////////////////////////////////////////
function mandreelAppStart(startStateFunc,params)
{
        mandreelAppStartStateFunc = startStateFunc;




        if ( typeof(params.log) != 'undefined' )
                mandreelAppLog = params.log;

        if ( typeof(params.platform) != 'undefined' )
                mandreelAppPlatform = params.platform;

        if (typeof(params.mandreelJSCompressed) != 'undefined' )
                mandreelAppMandreelJsCompressed = params.mandreelJSCompressed;


/*       The following code is removed for benchmarking:
        if ((mandreel_BrowserDetect.browser == 'Chrome' || mandreel_BrowserDetect.browser == 'Safari') && mandreel_BrowserDetect.OS == 'Mac' && mandreelAppPlatform == "flash")
                mandreelDisableSpaceKey = false; */

        if ( mandreelAppPlatform != "webgl" && mandreelAppPlatform != "flash" && mandreelAppPlatform != "nacl" && mandreelAppPlatform != "canvas" && mandreelAppPlatform != "plugin")
        {
                mandreelAppStartStateFunc("error",'platform ('+mandreelAppPlatform+') not supported');
                return;
    }

        if ( mandreelAppPlatform == "webgl" || mandreelAppPlatform == "canvas" )
        {
                mandreelAppWorkingFolder = "data/js/";
                if ( typeof(params.workingFolderWebgl) != 'undefined' )
                        mandreelAppWorkingFolder = params.workingFolderWebgl;
                // Check Float64Array availability
                if ( !Mandreel_window["Float64Array"] )
                {
                        mandreelAppStartStateFunc("error",'Browser unsupported: Float64Array not available');
                        return;
            }

                var flashElement = Mandreel_document.getElementById('FlashWrapper');
                if ( flashElement != null )
                {
                        flashElement.style.visibility = "hidden";
                        flashElement.style.width = "0px";
                        flashElement.style.height = "0px";
            }
                var flashElement = Mandreel_document.getElementById('FlashDiv');
                if ( flashElement != null )
                {
                        flashElement.style.visibility = "hidden";
                        flashElement.style.width = "0px";
                        flashElement.style.height = "0px";
            }

                // Setup WebGL
                if ( typeof(params.webglCanvas) == 'undefined' )
                {
                        mandreelAppStartStateFunc("error",'canvas parameter not found');
                        return;
            }
                var canvas = Mandreel_document.getElementById(params.webglCanvas);
                if ( canvas == null )
                {
                        mandreelAppStartStateFunc("error",'canvas object ('+params.webglCanvas+') not found');
                        return;
            }
                if ( params.width != null )
                {
                        canvas.width = params.width;
                        mandreelAppWidthSrc = params.width;
            }
                if ( params.height != null )
                {
                        canvas.height = params.height;
                        mandreelAppHeightSrc = params.height;
            }
                if ( mandreelAppPlatform == "webgl" )
                {
                        // The following code is removed for benchmarking:
                        // imandreel_gl = WebGLUtils.setupWebGL(canvas,{premultipliedAlpha:false,alpha:false});
                        // if (imandreel_gl == null)
                        // {
                        //      mandreelAppStartStateFunc("error","webgl_not_found");
                        //      return;
                        // }
            }

                if ( mandreelAppPlatform == "canvas" )
                {
                        imandreel_ctx_canvas = canvas.getContext('2d');
                        if ( imandreel_ctx_canvas == null )
                        {
                                mandreelAppStartStateFunc("error","canvas context 2d not found");
                                return;
                    }
            }

                if (params.cache != null)
                {
                        //alert( params.cache.size + params.cache.url);
                        mandreel_fs_init(function() { if ( mandreelAppStartStateFunc )
                                mandreelAppStartStateFunc("loadingFat","");
                        mandreelLoadFat();}, params.cache.size,params.cache.url);
            }
                else
                {
                        // load Fat
                        if ( mandreelAppStartStateFunc )
                                mandreelAppStartStateFunc("loadingFat","");
                        mandreelLoadFat();
            }
    }

        if ( mandreelAppPlatform == "flash" )
        {
                mandreelAppWorkingFolder = "data/as3/";
                if ( typeof(params.workingFolderFlash) != 'undefined' )
                        mandreelAppWorkingFolder = params.workingFolderFlash;
                if (!swfobject.hasFlashPlayerVersion('11.2.0'))
                {
                        mandreelAppStartStateFunc("error","flash 11 not found");
                        return;
            }

                if ( typeof(params.flashCanvas) == 'undefined' )
                {
                        mandreelAppStartStateFunc("error",'canvas parameter not found');
                        return;
            }

                if ( true ) // hide webgl canvas
                {
                        var canvas = Mandreel_document.getElementById(mandreelAppCanvasDiv);
                        if ( canvas != null )
                        {
                                canvas.style.visibility = "hidden";
                                canvas.style.width = "0px";
                                canvas.style.height = "0px";
                    }
            }

                if ( params.width != null )
                {
                        mandreelAppCanvasWidth = params.width;
            }
                if ( params.height != null )
                {
                        mandreelAppCanvasHeight = params.height;
            }

                mandreelAppCanvasDiv = params.flashCanvas;

                try
                {
                        var mandreelSocketsSwf = "mandreel.swf";
                        if ( typeof(params.swfFlash) != 'undefined' )
                                mandreelSocketsSwf = params.swfFlash;

                        var my_flashvars = "workingFolder=" + encodeURIComponent(mandreelAppWorkingFolder);
                        if ( typeof(params.log) != 'undefined' && params.log == true)
                          my_flashvars += "&log=true"
                        my_flashvars += "&width=" + params.width;
                        my_flashvars += "&height=" + params.height;
                        my_flashvars += "&swfloader=" + mandreelSocketsSwf;

                        if (typeof(params.restore_context) != 'undefined' && params.restore_context == true)
                                my_flashvars += "&restore_context=1";

                        if (typeof(params.antialiasing) != 'undefined')
                                my_flashvars += "&antialiasing=" + params.antialiasing;

                        if (typeof(params.right_click_enable) != 'undefined' && params.right_click_enable == true)
                                my_flashvars += "&right_click=1";

                        if (typeof(params.disable_depth) != 'undefined' && params.disable_depth == true)
                                my_flashvars += "&disable_depth=1";

                        var swfname = "mandreelloader.swf";
                        if ( typeof(params.swfPreloader) != 'undefined' && params.swfPreloader == false)
                                swfname = mandreelSocketsSwf;

                        var swf = swfobject.createSWF({ data:swfname, width:"100%", height:"100%" }, { menu:"false",allowScriptAccess:"always",allowFullScreen:"true",wmode:"direct",scale:"noscale",salign :"tl",bgcolor:"#000000",flashvars:my_flashvars}, params.flashCanvas);
                        if ( !swf )
                        {
                                mandreelAppStartStateFunc("error","error loading " + swfname);
                                return;
                    }
                        else
                        {
                                if ( mandreelAppStartStateFunc )
                                        mandreelAppStartStateFunc("loadingScript","");
                    }
            }
                catch(err)
                {
                        mandreelAppStartStateFunc("error","exception " + err + " while loading " + mandreelSocketsSwf);
                        return;
            }

                appStartState('loadingScript');
    }

        if ( mandreelAppPlatform == "nacl" )
        {
                mandreelAppWorkingFolder = "data/nacl/";

                // load Fat
                if ( mandreelAppStartStateFunc )
                        mandreelAppStartStateFunc("loadingFat","");
                mandreelLoadFat();

    }

        if ( mandreelAppPlatform == "plugin" )
        {
                mandreelInitPluginPlatform(params);
    }
}


////////////////////////////////////////////
function MandreelAudioStartedNacl()
{
        var helloWorldModule = null;
        helloWorldModule = Mandreel_document.getElementById('hello_world');
        helloWorldModule.postMessage('init');
        var flashElement = Mandreel_document.getElementById('FlashWrapper');
        if ( flashElement != null )
                flashElement.style.visibility = "hidden";
        mandreelAppStartStateFunc("ready",mandreelAppCanvasWidth,mandreelAppCanvasHeight);
}

////////////////////////////////////////////
function _mandreelAppStartReady()
{
        if ( mandreelAppPlatform == "nacl" )
        {
                wa_initWebAudio();
                mandreel_webAudio_PreloadAssets();
    }
        else
        {
                if ( mandreelAppStartStateFunc )
                {
                        mandreelAppStartRenderWebGL();
                        mandreelAppStartStateFunc("ready",mandreelAppCanvasWidth,mandreelAppCanvasHeight);
            }
    }
}
////////////////////////////////////////////
function _mandreelAppAudioReady()
{
        if ( mandreelAppStartStateFunc )
                mandreelAppStartStateFunc("audioLoaded","");

        if ( mandreelAppUseFlashSockets )
                mandreel_flash_sockets_load_flash(_mandreelAppStartReady);
        else
                _mandreelAppStartReady();
}

////////////////////////////////////////////
function mandreelAppInit()
{
        if ( mandreelAppPlatform == "webgl" || mandreelAppPlatform == "canvas" )
        {
                global_init(g_stack_pointer+800*1024);

                //Mandreel_TextureAsync_PackBufferData[_mandreel_currentPackTexture] = mandreelBufferPackAsyncTexture;
                //mandreelBufferPackAsyncTexture = null;
                var sp = g_stack_pointer+800*1024;
                heapU32[sp>>2] = mandreelAppCanvasWidth;
                heapU32[(sp+4)>>2] = mandreelAppCanvasHeight;
                __mandreel_internal_SetResolution(sp);
                __mandreel_internal_init(g_stack_pointer+800*1024);
                __init(g_stack_pointer+800*1024);
    }
}

////////////////////////////////////////////
function mandreelAppDraw(elapsed)
{
        if ( mandreelAppPlatform == "webgl" || mandreelAppPlatform == "canvas" )
        {
                if ( mandreelAppPlatform == "canvas" && imandreel_ctx_canvas != null )
                {
                        var canvas = Mandreel_document.getElementById(mandreelAppCanvasName);
                        imandreel_ctx_canvas.clearRect(0,0,canvas.width,canvas.height);
            }
                var sp = g_stack_pointer+800*1024;
                __mandreel_internal_preupdate(sp);
                heapU32[sp>>2] = elapsed;
                __draw(sp);
                __mandreel_internal_update(sp);
                __mandreel_process_async_calls();
    }
}

////////////////////////////////////////////
function mandreelAppMouseWheel(delta)
{
        if ( mandreelAppPlatform == "webgl" || mandreelAppPlatform == "canvas" )
        {
                var i7 = MandreelLockFrame();
                heap32[(i7+0)>>2] = delta;
                __mouseWheelDelta(i7);
                MandreelUnlockFrame();
    }
}

////////////////////////////////////////////
function mandreelAppMouseMove(x,y)
{
        if ( mandreelAppPlatform == "webgl" || mandreelAppPlatform == "canvas" )
        {
                var i7 = MandreelLockFrame();
                heap32[(i7+0)>>2] = x;
                heap32[(i7+4)>>2] = y;
                __mouseMove(i7);
                MandreelUnlockFrame();
    }
}

////////////////////////////////////////////
function mandreelAppResize(x,y)
{
        if ( mandreelAppPlatform == "webgl" || mandreelAppPlatform == "canvas" )
        {
                var i7 = MandreelLockFrame();
                heap32[(i7+0)>>2] = x;
                heap32[(i7+4)>>2] = y;
                __resize(i7);
                heap32[(i7+0)>>2] = x;
                heap32[(i7+4)>>2] = y;
                __mandreel_internal_SetResolution(i7);
                MandreelUnlockFrame();
    }
}

////////////////////////////////////////////
function mandreelAppMouseButton(down,x,y)
{
        if ( mandreelAppPlatform == "webgl" || mandreelAppPlatform == "canvas" )
        {
                var i7 = MandreelLockFrame();
                heap32[(i7+0)>>2] = down;
                heap32[(i7+4)>>2] = x;
                heap32[(i7+8)>>2] = y;
                __mouseButton(i7);
                MandreelUnlockFrame();
    }
}

////////////////////////////////////////////
function mandreelAppMouseDblClick(x,y)
{
        if ( mandreelAppPlatform == "webgl" || mandreelAppPlatform == "canvas" )
        {
                var i7 = MandreelLockFrame();
                heap32[(i7+0)>>2] = x;
                heap32[(i7+4)>>2] = y;
                __mouseDoubleClick(i7);
                MandreelUnlockFrame();
    }
}

////////////////////////////////////////////
function mandreelAppKeyEvent(down,keyId)
{
        if ( mandreelAppPlatform == "webgl" || mandreelAppPlatform == "canvas" )
        {
                var i7 = MandreelLockFrame();
                heap32[(i7+0)>>2] = down;
                heap32[(i7+4)>>2] = keyId;
                __keyEvent(i7);
                MandreelUnlockFrame();
    }
}

////////////////////////////////////////////
function mandreelAppGetPlatform()
{
        return mandreelAppPlatform;
}

////////////////////////////////////////////
function mandreelAppGetElementAbsolutePos(elementName)
{
        var element = Mandreel_document.getElementById(elementName);
        var res = new Object();
        res.x = 0; res.y = 0;
        if (element !== null)
        {
                if (element.getBoundingClientRect)
                {
                        var viewportElement = Mandreel_document.documentElement;
                        var box = element.getBoundingClientRect();
                        var scrollLeft = viewportElement.scrollLeft;
                        var scrollTop = viewportElement.scrollTop;
                        res.x = box.left + scrollLeft;
                        res.y = box.top + scrollTop;
            }
                else
                { //for old browsers
                        res.x = element.offsetLeft;
                        res.y = element.offsetTop;
                        var parentNode = element.parentNode;
                        var borderWidth = null;
                        while (offsetParent != null)
                        {
                                res.x += offsetParent.offsetLeft;
                                res.y += offsetParent.offsetTop;
                                var parentTagName = offsetParent.tagName.toLowerCase();
                                if ((__isIEOld && parentTagName != "table") ||
                                        ((__isFireFoxNew || __isChrome) &&
                                                parentTagName == "td"))
                                {
                                        borderWidth = kGetBorderWidth(offsetParent);
                                        res.x += borderWidth.left;
                                        res.y += borderWidth.top;
                            }

                                if (offsetParent != Mandreel_document.body &&
                                offsetParent != Mandreel_document.documentElement)
                                {
                                        res.x -= offsetParent.scrollLeft;
                                        res.y -= offsetParent.scrollTop;
                            }

                                //next lines are necessary to fix the problem
                                //with offsetParent
                                if (!__isIE && !__isOperaOld || __isIENew)
                                {
                                        while (offsetParent != parentNode &&
                                                parentNode !== null) {
                                                res.x -= parentNode.scrollLeft;
                                                res.y -= parentNode.scrollTop;
                                                if (__isFireFoxOld || __isWebKit)
                                                {
                                                    borderWidth =
                                                     kGetBorderWidth(parentNode);
                                                    res.x += borderWidth.left;
                                                    res.y += borderWidth.top;
                                            }
                                                parentNode = parentNode.parentNode;
                                    }
                            }

                                parentNode = offsetParent.parentNode;
                                offsetParent = offsetParent.offsetParent;
                    }
            }
    }
        return res;
}
function __getIEVersion()
{
        var rv = -1; // Return value assumes failure.
        if (navigator.appName == 'Microsoft Internet Explorer')
        {
                var ua = navigator.userAgent;
                var re = new RegExp("MSIE ([0-9]{1,}[\.0-9]{0,})");
                if (re.exec(ua) != null)
                        rv = parseFloat(RegExp.$1);
    }
        return rv;
}
function __getOperaVersion()
{
        var rv = 0; // Default value
        if (Mandreel_window.opera)
        {
                var sver = Mandreel_window.opera.version();
                rv = parseFloat(sver);
    }
        return rv;
}
/* The following code was removed for benchmarking:
var __userAgent = navigator.userAgent;
var __isIE =  navigator.appVersion.match(/MSIE/) != null;
var __IEVersion = __getIEVersion();
var __isIENew = __isIE && __IEVersion >= 8;
var __isIEOld = __isIE && !__isIENew;
var __isFireFox = __userAgent.match(/firefox/i) != null;
var __isFireFoxOld = __isFireFox && ((__userAgent.match(/firefox\/2./i) != null) || (__userAgent.match(/firefox\/1./i) != null));
var __isFireFoxNew = __isFireFox && !__isFireFoxOld;
var __isWebKit =  navigator.appVersion.match(/WebKit/) != null;
var __isChrome =  navigator.appVersion.match(/Chrome/) != null;
var __isOpera =  Mandreel_window.opera != null;
var __operaVersion = __getOperaVersion();
var __isOperaOld = __isOpera && (__operaVersion < 10); */
function __parseBorderWidth(width)
{
        var res = 0;
        if (typeof(width) == "string" && width != null && width != "" )
        {
                var p = width.indexOf("px");
                if (p >= 0)
                {
                        res = parseInt(width.substring(0, p));
            }
                else
                {
                        //do not know how to calculate other values (such as 0.5em or 0.1cm) correctly now so just set the width to 1 pixel
                        res = 1;
            }
    }
        return res;
}
function __getBorderWidth(element)
{
        var res = new Object();
        res.left = 0; res.top = 0; res.right = 0; res.bottom = 0;
        if (Mandreel_window.getComputedStyle)
        {
                //for Firefox
                var elStyle = Mandreel_window.getComputedStyle(element, null);
                res.left = parseInt(elStyle.borderLeftWidth.slice(0, -2));
                res.top = parseInt(elStyle.borderTopWidth.slice(0, -2));
                res.right = parseInt(elStyle.borderRightWidth.slice(0, -2));
                res.bottom = parseInt(elStyle.borderBottomWidth.slice(0, -2));
    }
        else
        {
                //for other browsers
                res.left = __parseBorderWidth(element.style.borderLeftWidth);
                res.top = __parseBorderWidth(element.style.borderTopWidth);
                res.right = __parseBorderWidth(element.style.borderRightWidth);
                res.bottom = __parseBorderWidth(element.style.borderBottomWidth);
    }
        return res;
}


////////////////////////////////////////////
// WebGL
////////////////////////////////////////////
        var imandreel_gl = null;
        var imandreel_ctx_canvas = null;
        var imandreel_is_ready = 0;
        var imandreel_oldTime = Date_now();

        function mandreel_wheel(event) { imandreel_onMouseWheel(event);event.preventDefault(); event.returnValue=false; }

        ////////////////////////////////////////////
        function mandreelAppStartRenderWebGL()
        {
/* The following code was removed for benchmarking:
                var canvas = document.getElementById(mandreelAppCanvasName);

                mandreelAppCanvasWidth = canvas.width;
                mandreelAppCanvasHeight = canvas.height;

                // Register event handlers
                if(window.addEventListener){ window.addEventListener('DOMMouseScroll',mandreel_wheel,false); }  window.onmousewheel=document.onmousewheel=mandreel_wheel;

                window.addEventListener('mousedown',imandreel_onMouseDown,false);
                window.addEventListener('mouseup',imandreel_onMouseUp,false);
                window.addEventListener('mousemove',imandreel_onMouseMove,false);
                window.addEventListener('dblclick',imandreel_onMouseDblClick,false);

                document.body.oncontextmenu = function() { return false;};

                var canvasDiv = document.getElementById(mandreelAppCanvasDiv);
                canvasDiv.addEventListener('keydown',imandreel_onKeyDown,false);
                canvasDiv.addEventListener('keyup',imandreel_onKeyUp,false);
                canvasDiv.focus();*/

                // Call mandreel app init funtion (__init())
                mandreelAppInit();

                // Start rendering
                imandreel_is_ready = 1;
                //imandreel_render();
    }

        ////////////////////////////////////////////
        function imandreel_render()
        {
                if( ABORT )
                        return;

                var canvas = Mandreel_document.getElementById(mandreelAppCanvasName);
                WebGLUtils.requestAnimationFrame(canvas, imandreel_render);


                // Reshape
                if (canvas.clientWidth != mandreelAppCanvasWidth || canvas.clientHeight != mandreelAppCanvasHeight)
                {
                        mandreelAppCanvasWidth = canvas.clientWidth;
                        mandreelAppCanvasHeight = canvas.clientHeight;
                        imandreel_gl.viewport(0, 0, mandreelAppCanvasWidth, mandreelAppCanvasHeight);
            }

                // Set the focus to the canvas div
                if (mandreelAppForceFocus)
                {
                        var canvasDiv = Mandreel_document.getElementById(mandreelAppCanvasDiv);
                        canvasDiv.focus();
            }

                // Call mandreel app draw funtion (__draw())
                var nowTime = Date_now();
                if (!g_mandreel_frame_locked)
                {
                        g_mandreel_frame_inframe = true;
                        if (!_imandreel_pause_game)
                                mandreelAppDraw(nowTime-imandreel_oldTime);
                        g_mandreel_frame_inframe = false;
            }
                imandreel_oldTime = nowTime;
    }

        function render()
        {
                // Call mandreel app draw funtion (__draw())
                var nowTime = Date_now();
                if (!g_mandreel_frame_locked)
                {
                        g_mandreel_frame_inframe = true;
                        if (!_imandreel_pause_game)
                                mandreelAppDraw(nowTime-imandreel_oldTime);
                        g_mandreel_frame_inframe = false;
            }
                imandreel_oldTime = nowTime;
    }

        var imandreel_last_movex = 0;
        var imandreel_last_movey = 0;

        ////////////////////////////////////////////
        function imandreel_onMouseMove(e)
        {
                if (!imandreel_is_ready)
                        return;

                if (_imandreel_pause_game)
                        return;

                var pos = mandreelAppGetElementAbsolutePos(mandreelAppCanvasName);
                var setX = e.clientX - pos.x;
                var setY = e.clientY - pos.y;

                  if(navigator.pointer && navigator.pointer.isLocked) {
            var deltaX = event.movementX || event.webkitMovementX || event.mozMovementX || 0;
            var deltaY = event.movementY || event.webkitMovementY || event.mozMovementX || 0;

                        setX = imandreel_last_movex+deltaX;
                        setY = imandreel_last_movey+deltaY;
                    }


                imandreel_last_movex = setX;
                imandreel_last_movey = setY;
                mandreelAppMouseMove(setX,setY);
    }

        ////////////////////////////////////////////
        function imandreel_onMouseDblClick(e)
        {
                if (!imandreel_is_ready)
                        return;

                if (_imandreel_pause_game)
                        return;

                var pos = mandreelAppGetElementAbsolutePos(mandreelAppCanvasName);
                var setX = e.clientX - pos.x;
                var setY = e.clientY - pos.y;
                mandreelAppMouseDblClick(setX,setY);
    }

        ////////////////////////////////////////////
        var mandreel_mouse_down = false;
        var mandreel_rmouse_down = false;
        function imandreel_onMouseDown(e)
        {
                if (!imandreel_is_ready)
                        return;

                if (_imandreel_pause_game)
                        return;

                var rightclick;
                if (!e) var e = Mandreel_window.event;
                if (e.which) rightclick = (e.which == 3);
                else if (e.button) rightclick = (e.button == 2);

                var pos = mandreelAppGetElementAbsolutePos(mandreelAppCanvasName);
                var setX = e.clientX - pos.x;
                var setY = e.clientY - pos.y;

                if (!rightclick)
                {
                        if (mandreel_mouse_down)
                                return;

                        mandreel_mouse_down = true;
                        mandreelAppMouseButton(1,setX,setY);
            }
                else
                {
                        if (mandreel_rmouse_down)
                                return;

                        mandreel_rmouse_down = true;
                        mandreelAppMouseButton(3,setX,setY);
            }
    }

        ////////////////////////////////////////////
        function imandreel_onMouseUp(e)
        {
                if (!imandreel_is_ready)
                        return;

                if (_imandreel_pause_game)
                        return;

                var rightclick;
                if (!e) var e = Mandreel_window.event;
                if (e.which) rightclick = (e.which == 3);
                else if (e.button) rightclick = (e.button == 2);

                var pos = mandreelAppGetElementAbsolutePos(mandreelAppCanvasName);
                var setX = e.clientX - pos.x;
                var setY = e.clientY - pos.y;


                if (!rightclick)
                {
                        if (mandreel_mouse_down == false)
                                return;

                        mandreel_mouse_down = false;
                        mandreelAppMouseButton(0,setX,setY);
            }
                else
                {
                        if (mandreel_rmouse_down == false)
                                return;

                        mandreel_rmouse_down = false;
                        mandreelAppMouseButton(2,setX,setY);
            }
    }

        ////////////////////////////////////////////
        function imandreel_onMouseWheel(e)
        {
                if (!imandreel_is_ready)
                        return;

                if (_imandreel_pause_game)
                        return;

                mandreelAppMouseWheel(e.wheelDelta);
    }

        ////////////////////////////////////////////
        function imandreel_onKeyUp(e)
        {
                if (!imandreel_is_ready)
                        return;

                if (_imandreel_pause_game)
                        return;

                var KeyID = e.keyCode;
                mandreelAppKeyEvent(0,KeyID);
    }

        ////////////////////////////////////////////
        Mandreel_window.onkeydown = function(e)
        {
                if (mandreelDisableSpaceKey == false && e.keyCode == 32)
                        return;
                return !(e.keyCode == 32 || e.keyCode == 9);
    };

        ////////////////////////////////////////////
        function imandreel_onKeyDown(e)
        {
                if (e.which === 8) // disable back button on browser
        {
                e.preventDefault();
            }
                if (!imandreel_is_ready)
                        return;

                if (_imandreel_pause_game)
                        return;

                var KeyID = e.keyCode;
                mandreelAppKeyEvent(1,KeyID);
    }


////////////////////////////////////////////
// Flash
////////////////////////////////////////////

(function(){
                try {
                        // Disabling SWFObject's Autohide feature
                        if (typeof swfobject.switchOffAutoHideShow === "function") {
                                swfobject.switchOffAutoHideShow();
                    }
            } catch(e) {}
    })();

var imandreel_flash_global_sp = 0;
var imandreel_flash_global_sp2 = 0;
var imandreel_flash_current_sp = 0;

function MandreelInterSwfLoaded2()
{
}
function MandreelInterSwfLoaded()
{
        appStartState('scriptLoaded');
        var flashMovie = swfobject.getObjectById(mandreelAppCanvasDiv)
        dump(flashMovie.width);
        flashMovie.MandreelInit();

        imandreel_flash_global_sp = MandreelInterGetSWF().MandreelInterGetGlobalStack();
        imandreel_flash_global_sp2 = MandreelInterGetSWF().MandreelInterGetGlobalStack2();
        if ( mandreelAppStartStateFunc )
                mandreelAppStartStateFunc("ready",mandreelAppCanvasWidth,mandreelAppCanvasHeight);
}



var g_mandreel_swf = null;

function MandreelInterGetSWF()
{
        if (g_mandreel_swf)
                return g_mandreel_swf;

        g_mandreel_swf = swfobject.getObjectById(mandreelAppCanvasDiv)
        return g_mandreel_swf;
}

function MandreelInterWriteInt(ptr, value)
{
        MandreelInterGetSWF().MandreelInterWriteInt(ptr,value);
}

function MandreelInterWriteFloat(ptr, value)
{
        MandreelInterGetSWF().MandreelInterWriteFloat(ptr,value);
}

function MandreelInterWriteString(ptr, value)
{
        MandreelInterGetSWF().MandreelInterWriteString(ptr,value);
}

function MandreelInterWriteWString(ptr, value)
{
        MandreelInterGetSWF().MandreelInterWriteWString(ptr,value);
}

function MandreelInterCallFunctionRaw(sp, func_name, returnType)
{
        return MandreelInterGetSWF().MandreelInterCallFunctionRaw(sp,func_name,returnType);
}


function assert(condition, _text) {
//console.assert(condition, _text);
    if (!condition) {
      var text = "Assertion failed: " + _text;
      alert(text + ':\n' + (new Error).stack);
      ABORT = true;
      throw "Assertion: " + text;
    }
  }



function MandreelLockFrame()
{
        MandreelInterGetSWF().MandreelLockFrame();

        imandreel_flash_current_sp += 300*1024;

        assert(imandreel_flash_current_sp<1024*1024);

        return imandreel_flash_current_sp;
}

function MandreelPause()
{
        MandreelInterGetSWF().MandreelPause();
}

function MandreelResume()
{
        MandreelInterGetSWF().MandreelResume();
}


function MandreelUnlockFrame()
{
        imandreel_flash_current_sp -= 300*1024;
        MandreelInterGetSWF().MandreelUnlockFrame();
}



function MandreelInterCallFunctionAsync(func_name,param)
{
        return MandreelInterGetSWF().MandreelInterCallFunctionAsync(func_name, param);
}


function MandreelInterCallFunction(returnType,func_name)
{
        var size_params = 0;

        var i;
        var num_params = (arguments.length-2)/2;
        num_params|=0;
        for (i=2;i<num_params*2+2;i+=2)
        {
                var type = arguments[i];

                var size_arg = 0;
                switch(type)
                {
                        case 'int':
                                size_arg = 4;
                                break;
                        case 'float':
                                size_arg = 4;
                                break;
                        case 'string':
                                size_arg = 4;
                                size_arg += ((arguments[i+1].length + 4) & 0xfffffffc);
                                break;
                        case 'wstring':
                                size_arg = 4;
                                size_arg += ((arguments[i+1].length*2 + 4) & 0xfffffffc);
                                break;
                        default:
                                assert(false);
            }

                size_params += size_arg;
    }

        // stack always 8 byte aligned
        size_params=((size_params+7)& 0xfffffff8);

        var sp = 0;

        if (i<(arguments.length))
                sp = arguments[i];
        else
        {
                assert(false);
    }


        sp-=size_params;

        MandreelLockFrame();

        var offset = 0;
        var ptr_data = num_params*4+sp;
    for (i=2;i<num_params*2+2;i+=2)
        {
                var type = arguments[i];


                var size_arg = 0;
                switch(type)
                {
                        case 'int':
                                MandreelInterWriteInt((sp+offset),arguments[i+1]);
                                break;
                        case 'float':
                                MandreelInterWriteFloat((sp+offset),arguments[i+1]);
                                break;
                        case 'string':
                                {
                                        MandreelInterWriteInt((sp+offset),ptr_data);
                                        var string = arguments[i+1];
                                        MandreelInterWriteString(ptr_data,string);

                                        ptr_data += ((string.length + 4) & 0xfffffffc);
                            }
                                break;
                        case 'wstring':
                                {
                                        MandreelInterWriteInt((sp+offset),ptr_data);
                                        var string = arguments[i+1];
                                        MandreelInterWriteWString(ptr_data,string);

                                        ptr_data += ((string.length*2 + 4) & 0xfffffffc);
                            }
                                break;
                        default:
                                assert(false);
            }
                offset+=4;
    }

        return_value = MandreelInterCallFunctionRaw(sp, func_name, returnType);

        MandreelUnlockFrame();

        if (returnType == 'int')
                return parseInt(return_value);
        else if (returnType == 'float')
                return parseFloat(return_value);

        return;
}



// file system

var g_mandreel_timestamp_fs = 0;
var g_mandreel_fs = null;

function onMandreelQuotaOk(fs)
{
        g_mandreel_fs = fs;
        dump('onMandreelQuotaOk');
}


function MandreelFsErrorHandler(e) {
  var msg = '';

  switch (e.code) {
    case FileError.QUOTA_EXCEEDED_ERR:
      msg = 'QUOTA_EXCEEDED_ERR';
      break;
    case FileError.NOT_FOUND_ERR:
      msg = 'NOT_FOUND_ERR';
      break;
    case FileError.SECURITY_ERR:
      msg = 'SECURITY_ERR';
      break;
    case FileError.INVALID_MODIFICATION_ERR:
      msg = 'INVALID_MODIFICATION_ERR';
      break;
    case FileError.INVALID_STATE_ERR:
      msg = 'INVALID_STATE_ERR';
      break;
    default:
      msg = 'Unknown Error';
      break;
  };

  dump('Error: ' + msg);
  dump(e);
}

var indexedDB = Mandreel_window.indexedDB || Mandreel_window.webkitIndexedDB ||
                Mandreel_window.mozIndexedDB;

if ('webkitIndexedDB' in Mandreel_window) {
  Mandreel_window.IDBTransaction = Mandreel_window.webkitIDBTransaction;
  Mandreel_window.IDBKeyRange = Mandreel_window.webkitIDBKeyRange;
}


var mandreel_indexedDB = {};
mandreel_indexedDB.db = null;
mandreel_indexedDB.callback_init = false;

mandreel_indexedDB.onerror = function(e) {
  dump(e);
};

mandreel_indexedDB.init = function(callback)
{
 try {
        var request = indexedDB.open('my_cache', 2);

        request.onerror = function(event) {
                callback(false);
            };
        request.onupgradeneeded = function(event) {
        dump('onupgradeneeded\n');
        mandreel_indexedDB.db = event.target.result;
    var db = mandreel_indexedDB.db;
        var objectStore = db.createObjectStore("cache",{keyPath: "fileName"});
    }

  request.onsuccess = function(event) {
  dump('onsuccess\n');
    mandreel_indexedDB.db = event.target.result;
        callback(true);

  };
  }
  catch(e)
  {
        callback(false);
  }

}

function mandreel_fs_init(callback, size_bytes, url_time)
{

        g_mandreel_timestamp_fs = 99999999999999;




        var aux = null;
        try     { aux = webkitStorageInfo; } catch(err) { aux = null; }
        if (aux == null)
        {
                //callback(false);

                var time_request = new XMLHttpRequest();
        time_request.open('GET',url_time);
        time_request.onreadystatechange = function()
        {
                if (time_request.readyState==4)
                {
                        if (time_request.status==200 || time_request.status==0)
                        {

                                if (time_request.responseText)
                                {
                                        g_mandreel_timestamp_fs = parseFloat(time_request.responseText);
                                        dump('mandreel_init_fs time ' + g_mandreel_timestamp_fs);
                            }
                    }

                        mandreel_indexedDB.init(callback);
            }
    }
        time_request.send();


                return;
    }

        webkitStorageInfo.requestQuota(
                webkitStorageInfo.PERSISTENT ,   // or PERSISTENT
                size_bytes,
                function(bytes) {  dump('request quota succeed');},
                function () { dump('request quota failed');} );


//      g_mandreel_timestamp_fs =0;

        var time_request = new XMLHttpRequest();
        time_request.open('GET',url_time);
        time_request.onreadystatechange = function()
        {
                if (time_request.readyState==4)
                {
                        if (time_request.status==200 || time_request.status==0)
                        {

                                if (time_request.responseText)
                                {
                                        g_mandreel_timestamp_fs = parseFloat(time_request.responseText);
                                        dump('mandreel_init_fs time ' + g_mandreel_timestamp_fs);
                            }
                    }

                        var my_requestFileSystem  = Mandreel_window.requestFileSystem || Mandreel_window.webkitRequestFileSystem;

                        my_requestFileSystem(Mandreel_window.PERSISTENT, size_bytes,
                                function(result)
                                {
                                        onMandreelQuotaOk(result);


                                        if (callback)
                                                Mandreel_setTimeout(callback, 100, true);
                            } , function(e) { MandreelFsErrorHandler(e); if (callback) callback(false); } );
            }
    }
        time_request.send();
}


mandreel_indexedDB.load = function(file_name, callback)
{
        var db = mandreel_indexedDB.db;
        var trans = db.transaction(["cache"]);
        var store = trans.objectStore("cache");

        var key = file_name;

         var getReq = store.get(key);
           getReq.onsuccess = function (ev) {
                        if (getReq.result)
        {
        dump('chanka ' + g_mandreel_timestamp_fs + ' ' + getReq.result.timeStamp + '\n');
                                if (getReq.result.timeStamp>=g_mandreel_timestamp_fs)
                                        callback(getReq.result.data);
                                else
                                        callback(null);
                    }
                        else
                                callback(null);
           }
           getReq.onerror = function (ev) {
                   console.log("index.get failed. Error: " + ev.message);
           }
}

function mandreel_fs_get_url(file_name, callback)
{
        if (mandreel_indexedDB.db)
        {
                callback(null);
                return;
    }


        if (g_mandreel_fs == null)
        {
                callback(null);
                return;
    }

        g_mandreel_fs.root.getFile(file_name, {}, function(fileEntry) {

        fileEntry.getMetadata(function(metaData){
        var my_seconds = metaData.modificationTime.getTime()/1000;

        if (g_mandreel_timestamp_fs>my_seconds)
        {
                callback(null);
    }
        else
        {
                //alert('mandreel_fscachefile2');


                if (Mandreel_window.localStorage.getItem(mandreel_fs_get_key(file_name)) != null)
                        callback(fileEntry.toURL());
                else
                        callback(null);

    }


            }, MandreelFsErrorHandler);


  }, function(error) {callback(null);});

         return;

}

function mandreel_fs_load_binary(file_name, callback)
 {
        if (mandreel_indexedDB.db)
        {
                mandreel_indexedDB.load(file_name,callback);
                return;
    }


        if (g_mandreel_fs == null)
        {
                callback(null);
                return;
    }


        g_mandreel_fs.root.getFile(file_name, {}, function(fileEntry) {

        fileEntry.getMetadata(function(metaData){
        var my_seconds = metaData.modificationTime.getTime()/1000;

        if (g_mandreel_timestamp_fs>my_seconds)
        {
                dump('mandreel_fscachefile1');
                Mandreel_window.localStorage.removeItem(mandreel_fs_get_key(file_name));
                fileEntry.remove(function() {
                                callback(null);
                    }, MandreelFsErrorHandler);

    }
        else
        {
                //alert('mandreel_fscachefile2');
                dump('mandreel_fscachefile2 ' + my_seconds);

                fileEntry.file(function(file) {

                                var reader = new FileReader();


                           reader.onloadend = function(e) {

                                   if (this.result.byteLength && Mandreel_window.localStorage.getItem(mandreel_fs_get_key(file_name)) != null)
                                   {
                                                dump('mandreel_fs_loadFile ' + file_name);
                                                callback(this.result);
                                    }
                                        else
                                                callback(null);

                           };


                                 reader.readAsArrayBuffer(file);


                    }, MandreelFsErrorHandler);

    }


            }, MandreelFsErrorHandler);


  }, function(error) {dump('mandreel_fscachefile3');    callback(null);});

         return;
 }


 mandreel_indexedDB.save = function(file_name, data)
{
        var db = mandreel_indexedDB.db;


  var trans = db.transaction(["cache"], IDBTransaction.READ_WRITE);
  var store = trans.objectStore("cache");
  var request = store.put({
    "data": data,
        "timeStamp" : Date_now()/1000,
    "fileName" : file_name
  });

  request.onsuccess = function(e) {
    // Re-render all the todo's
   // html5rocks.indexedDB.getAllTodoItems();
   dump('mandreel_indexedDB.save ok ');
  };

  request.onerror = function(e) {
    dump('mandreel_indexedDB.save onerror ' + file_name);
        dump(e);
  };
}

function mandreel_fs_get_key(name)
{
        return 'mandreel_fs_' + name;
}

function mandreel_is_filesystem()
{
        if (mandreel_indexedDB.db)
                return false;

        if (!g_mandreel_fs)
                return false;

        return true;
}

function mandreel_is_indexeddb()
{
        if (mandreel_indexedDB.db)
                return true;

        return false;
}

 function mandreel_fs_saveFile(name, my_arrayBuffer)
{
        if (mandreel_indexedDB.db)
        {
                mandreel_indexedDB.save(name,my_arrayBuffer);
                return;
    }

        if (!g_mandreel_fs)
                return;

        Mandreel_window.localStorage.removeItem(mandreel_fs_get_key(name));
        g_mandreel_fs.root.getFile(name, {create: true}, function(fileEntry) {

    // Create a FileWriter object for our FileEntry (log.txt).
    fileEntry.createWriter(function(fileWriter) {

      fileWriter.onwriteend = function(e) {
                Mandreel_window.localStorage.setItem(mandreel_fs_get_key(name),'valid');
        dump('Write completed.');
      };

      fileWriter.onerror = function(e) {
        dump('Write failed: ' + e.toString());
      };

          var my_BlobBuilder = Mandreel_window.MozBlobBuilder || Mandreel_window.WebKitBlobBuilder || Mandreel_window.BlobBuilder;

          var bb = new my_BlobBuilder(); // Note: window.WebKitBlobBuilder in Chrome 12.
      bb.append(my_arrayBuffer);
      fileWriter.write(bb.getBlob('text/plain'));



    }, MandreelFsErrorHandler);

  }, MandreelFsErrorHandler);
}


function mandreel_fs_load_text(file_name, callback)
 {

        if (mandreel_indexedDB.db)
        {
                mandreel_indexedDB.load(file_name,callback);
                return;
    }

        if (g_mandreel_fs == null)
        {
                callback(null);
                return;
    }


        g_mandreel_fs.root.getFile(file_name, {}, function(fileEntry) {

        fileEntry.getMetadata(function(metaData){
        var my_seconds = metaData.modificationTime.getTime()/1000;

        if (g_mandreel_timestamp_fs>my_seconds)
        {
                dump('mandreel_fscachefile1');
                Mandreel_window.localStorage.removeItem(mandreel_fs_get_key(file_name));
                fileEntry.remove(function() {
                                callback(null);
                    }, MandreelFsErrorHandler);

    }
        else
        {
                //alert('mandreel_fscachefile2');
                dump('mandreel_fscachefile2 ' + my_seconds);

                fileEntry.file(function(file) {

                                var reader = new FileReader();


                           reader.onloadend = function(e) {


                                   if (this.result.length && Mandreel_window.localStorage.getItem(mandreel_fs_get_key(file_name)) != null)
                                   {
                                                dump('mandreel_fs_loadFile ' + file_name);
                                                callback(this.result);
                                    }
                                        else
                                                callback(null);

                           };


                                 reader.readAsText(file);


                    }, MandreelFsErrorHandler);

    }


            }, MandreelFsErrorHandler);


  }, function(error) {dump('mandreel_fscachefile3');    callback(null);});

         return;
 }


 // full screen


/* The following code was removed for benchmarking:
 var __screen = screen; */

        try
        {
        (function(global) {
    "use strict";
        var elementPrototype = (global.HTMLElement || global.Element)["prototype"];

         // document.isFullScreen
    if(!Mandreel_document.hasOwnProperty("fullscreenEnabled")) {
        var getter = (function() {
            // These are the functions that match the spec, and should be preferred
            if("webkitIsFullScreen" in Mandreel_document) {
                return function() { return Mandreel_document.webkitIsFullScreen; }
            }
            if("mozFullScreen" in Mandreel_document) {
                return function() { return Mandreel_document.mozFullScreen; }
            }
            return function() { return false }; // not supported, never fullscreen
        })();

        Object.defineProperty(Mandreel_document, "fullscreenEnabled", {
            enumerable: true, configurable: false, writeable: false,
            get: getter
        });
    }

    if(!Mandreel_document.hasOwnProperty("fullscreenElement")) {
        var getter = (function() {
            // These are the functions that match the spec, and should be preferred
            if("webkitFullscreenElement" in Mandreel_document) {
                return function() { return Mandreel_document.webkitFullscreenElement; }
            }
            if("mozFullscreenElemen" in Mandreel_document) {
                return function() { return Mandreel_document.mozFullscreenElemen; }
            }
            return function() { return null }; // not supported
        })();

        Object.defineProperty(Mandreel_document, "fullscreenElement", {
            enumerable: true, configurable: false, writeable: false,
            get: getter
        });
    }

    // Document event: fullscreenchange
    function fullscreenchange(oldEvent) {
        var newEvent = Mandreel_document.createEvent("CustomEvent");
        newEvent.initCustomEvent("fullscreenchange", true, false, null);
        // TODO: Any need for variable copy?
        Mandreel_document.dispatchEvent(newEvent);
    }
    Mandreel_document.addEventListener("webkitfullscreenchange", fullscreenchange, false);
    Mandreel_document.addEventListener("mozfullscreenchange", fullscreenchange, false);

    // Document event: fullscreenerror
    function fullscreenerror(oldEvent) {
        var newEvent = Mandreel_document.createEvent("CustomEvent");
        newEvent.initCustomEvent("fullscreenerror", true, false, null);
        // TODO: Any need for variable copy?
        Mandreel_document.dispatchEvent(newEvent);
    }
    Mandreel_document.addEventListener("webkitfullscreenerror", fullscreenerror, false);
    Mandreel_document.addEventListener("mozfullscreenerror", fullscreenerror, false);

    // element.requestFullScreen
    if(!elementPrototype.requestFullScreen) {
        elementPrototype.requestFullScreen = (function() {
            if(elementPrototype.webkitRequestFullScreen) {
                return function() {
                    this.webkitRequestFullScreen(Element.ALLOW_KEYBOARD_INPUT);
                }
            }

            return  elementPrototype.mozRequestFullScreen    ||
                    function(){ /* unsupported, fail silently */ };
        })();
    }

    // document.cancelFullscreen
    if(!Mandreel_document.cancelFullScreen) {
        Mandreel_document.cancelFullScreen = (function() {
            return  Mandreel_document.webkitCancelFullScreen ||
                    Mandreel_document.mozCancelFullScreen ||
                    function(){ /* unsupported, fail silently */ };
        })();
    }
    })((typeof(exports) != 'undefined') ? global : Mandreel_window);
    }
        catch(e)
        {
    }


try
        {
var __onErrorFullscreen = function() {
  dump("Mouse lock was not successful.");
};

Mandreel_document.addEventListener('fullscreenchange', function(e) {

var width;
var height;
 var canvas = Mandreel_document.getElementById(mandreelAppCanvasName);
 if(Mandreel_document.fullscreenEnabled) {


            width = __screen.width;
            height = __screen.height;

        } else {
            width = mandreelAppWidthSrc;
            height = mandreelAppHeightSrc;
        }

                canvas.width = width;
                canvas.height = height;
                mandreelAppResize(width, height);

  if (Mandreel_document.fullscreenEnabled && navigator.pointer) {
    navigator.pointer.lock(Mandreel_document.body, function() {
      // Locked and ready to play.
    }, __onErrorFullscreen);
  }
}, false);

}
catch(e)
{
}



function mandreelAppFullscreen(enable)
{
        var canvas = Mandreel_document.getElementById(mandreelAppCanvasName);
        if ( mandreelAppPlatform == "webgl" || mandreelAppPlatform == "canvas" )
        {
                if (canvas)
                {
                        if (enable)
                                canvas.requestFullScreen();
                        else
                                Mandreel_document.cancelFullScreen();
            }
    }
}


function mandreelAppDisableForceFocus()
{
        mandreelAppForceFocus = false;
}

function mandreelAppEnableForceFocus()
{
        mandreelAppForceFocus = true;
}

var imandreel_base64 = {};
imandreel_base64.PADCHAR = '=';
imandreel_base64.ALPHA = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/';

imandreel_base64.makeDOMException = function() {
    // sadly in FF,Safari,Chrome you can't make a DOMException
    var e, tmp;

    try {
        return new DOMException(DOMException.INVALID_CHARACTER_ERR);
    } catch (tmp) {
        // not available, just passback a duck-typed equiv
        // https://developer.mozilla.org/en/Core_JavaScript_1.5_Reference/Global_Objects/Error
        // https://developer.mozilla.org/en/Core_JavaScript_1.5_Reference/Global_Objects/Error/prototype
        var ex = new Error("DOM Exception 5");

        // ex.number and ex.description is IE-specific.
        ex.code = ex.number = 5;
        ex.name = ex.description = "INVALID_CHARACTER_ERR";

        // Safari/Chrome output format
        ex.toString = function() { return 'Error: ' + ex.name + ': ' + ex.message; };
        return ex;
    }
}

imandreel_base64.getbyte64 = function(s,i) {
    // This is oddly fast, except on Chrome/V8.
    //  Minimal or no improvement in performance by using a
    //   object with properties mapping chars to value (eg. 'A': 0)
    var idx = imandreel_base64.ALPHA.indexOf(s.charAt(i));
    if (idx === -1) {
        throw imandreel_base64.makeDOMException();
    }
    return idx;
}

imandreel_base64.decode = function(s) {
    // convert to string
    s = '' + s;
    var getbyte64 = imandreel_base64.getbyte64;
    var pads, i, b10;
    var imax = s.length
    if (imax === 0) {
        return s;
    }

    if (imax % 4 !== 0) {
        throw imandreel_base64.makeDOMException();
    }

    pads = 0
    if (s.charAt(imax - 1) === imandreel_base64.PADCHAR) {
        pads = 1;
        if (s.charAt(imax - 2) === imandreel_base64.PADCHAR) {
            pads = 2;
        }
        // either way, we want to ignore this last block
        imax -= 4;
    }

    var x = [];
    for (i = 0; i < imax; i += 4) {
        b10 = (getbyte64(s,i) << 18) | (getbyte64(s,i+1) << 12) |
            (getbyte64(s,i+2) << 6) | getbyte64(s,i+3);
        x.push(String.fromCharCode(b10 >> 16, (b10 >> 8) & 0xff, b10 & 0xff));
    }

    switch (pads) {
    case 1:
        b10 = (getbyte64(s,i) << 18) | (getbyte64(s,i+1) << 12) | (getbyte64(s,i+2) << 6);
        x.push(String.fromCharCode(b10 >> 16, (b10 >> 8) & 0xff));
        break;
    case 2:
        b10 = (getbyte64(s,i) << 18) | (getbyte64(s,i+1) << 12);
        x.push(String.fromCharCode(b10 >> 16));
        break;
    }
    return x.join('');
}

if (!Mandreel_window.atob) {
Mandreel_window.atob = function(a) { return  imandreel_base64.decode(a); }
}


function imandreel_interop_callbridge(new_method, sp)
{

        var n = new Array();

        for( var i = 2; i < arguments.length; i++ )
        {

                if (typeof arguments[i] == 'string')
                        n.push(atob(arguments[i]));
                else
                        n.push(arguments[i]);
    }

        var total_args = arguments.length-2;

        switch(total_args)
        {
                case 0:
                        return Mandreel_window[new_method](sp);
                case 1:
                        return Mandreel_window[new_method](sp, n[0]);
                case 2:
                        return Mandreel_window[new_method](sp, n[0], n[1]);
                case 3:
                        return Mandreel_window[new_method](sp, n[0], n[1], n[2]);
                case 4:
                        return Mandreel_window[new_method](sp, n[0], n[1], n[2], n[3]);
                case 5:
                        return Mandreel_window[new_method](sp, n[0], n[1], n[2], n[3], n[4]);
                case 6:
                        return Mandreel_window[new_method](sp, n[0], n[1], n[2], n[3], n[4], n[5]);
                case 7:
                        return Mandreel_window[new_method](sp, n[0], n[1], n[2], n[3], n[4], n[5], n[6]);
                case 8:
                        return Mandreel_window[new_method](sp, n[0], n[1], n[2], n[3], n[4], n[5], n[6], n[7]);
                case 9:
                        return Mandreel_window[new_method](sp, n[0], n[1], n[2], n[3], n[4], n[5], n[6], n[7], n[8]);
                case 10:
                        return Mandreel_window[new_method](sp, n[0], n[1], n[2], n[3], n[4], n[5], n[6], n[7], n[8], n[9]);
                case 11:
                        return Mandreel_window[new_method](sp, n[0], n[1], n[2], n[3], n[4], n[5], n[6], n[7], n[8], n[9], n[10]);
                case 12:
                        return Mandreel_window[new_method](sp, n[0], n[1], n[2], n[3], n[4], n[5], n[6], n[7], n[8], n[9], n[10], n[11]);
                case 13:
                        return Mandreel_window[new_method](sp, n[0], n[1], n[2], n[3], n[4], n[5], n[6], n[7], n[8], n[9], n[10], n[11], n[12]);
                case 14:
                        return Mandreel_window[new_method](sp, n[0], n[1], n[2], n[3], n[4], n[5], n[6], n[7], n[8], n[9], n[10], n[11], n[12], n[13]);
                case 15:
                        return Mandreel_window[new_method](sp, n[0], n[1], n[2], n[3], n[4], n[5], n[6], n[7], n[8], n[9], n[10], n[11], n[12], n[13], n[14]);
                case 16:
                        return Mandreel_window[new_method](sp, n[0], n[1], n[2], n[3], n[4], n[5], n[6], n[7], n[8], n[9], n[10], n[11], n[12], n[13], n[14], n[15]);
                break;
    }

}
function imandreel_as3_dump(param)
{
        dump(atob(param));
}


var mandreel_embed_plugin;

function PluginMandreelInterWriteInt(ptr, value)
{
        mandreel_embed_plugin.MandreelInterCalls("WriteInt", ptr,value);
}

function PluginMandreelInterWriteFloat(ptr, value)
{
        mandreel_embed_plugin.MandreelInterCalls("WriteFloat", ptr,value);
}

function PluginMandreelInterWriteString(ptr, value)
{
        mandreel_embed_plugin.MandreelInterCalls("WriteString", ptr,value);
}

function PluginMandreelInterWriteWString(ptr, value)
{
        mandreel_embed_plugin.MandreelInterCalls("WriteWString", ptr,value);
}

var mandreel_plugin_current_sp = 0;

function PluginMandreelLockFrame()
{
        var result = mandreel_embed_plugin.MandreelInterCalls("LockFrame");

        mandreel_plugin_current_sp+=300*1024;

        assert(mandreel_plugin_current_sp<1024*1024);

        return result+mandreel_plugin_current_sp;
}

function PluginMandreelPause()
{
        mandreel_embed_plugin.MandreelInterCalls("Pause");
}

function PluginMandreelResume()
{
        mandreel_embed_plugin.MandreelInterCalls("Resume");
}


function PluginMandreelUnlockFrame()
{
        mandreel_embed_plugin.MandreelInterCalls("UnlockFrame");

        mandreel_plugin_current_sp-=300*1024;
}

function PluginMandreelInterCallFunction()
{
        var total_args = arguments.length;

        switch(total_args)
        {
                case 0:
                        return mandreel_embed_plugin.MandreelInterCallFunction();
                case 1:
                        return mandreel_embed_plugin.MandreelInterCallFunction(arguments[0]);
                case 2:
                        return mandreel_embed_plugin.MandreelInterCallFunction(arguments[0], arguments[1]);
                case 3:
                        return mandreel_embed_plugin.MandreelInterCallFunction(arguments[0], arguments[1], arguments[2]);
                case 4:
                        return mandreel_embed_plugin.MandreelInterCallFunction(arguments[0], arguments[1], arguments[2], arguments[3]);
                case 5:
                        return mandreel_embed_plugin.MandreelInterCallFunction(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4]);
                case 6:
                        return mandreel_embed_plugin.MandreelInterCallFunction(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4], arguments[5]);
                case 7:
                        return mandreel_embed_plugin.MandreelInterCallFunction(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4], arguments[5], arguments[6]);
                case 8:
                        return mandreel_embed_plugin.MandreelInterCallFunction(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4], arguments[5], arguments[6], arguments[7]);
                case 9:
                        return mandreel_embed_plugin.MandreelInterCallFunction(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4], arguments[5], arguments[6], arguments[7], arguments[8]);
                case 10:
                        return mandreel_embed_plugin.MandreelInterCallFunction(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4], arguments[5], arguments[6], arguments[7], arguments[8], arguments[9]);
                case 11:
                        return mandreel_embed_plugin.MandreelInterCallFunction(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4], arguments[5], arguments[6], arguments[7], arguments[8], arguments[9], arguments[10]);
                case 12:
                        return mandreel_embed_plugin.MandreelInterCallFunction(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4], arguments[5], arguments[6], arguments[7], arguments[8], arguments[9], arguments[10], arguments[11]);
                case 13:
                        return mandreel_embed_plugin.MandreelInterCallFunction(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4], arguments[5], arguments[6], arguments[7], arguments[8], arguments[9], arguments[10], arguments[11], arguments[12]);
                case 14:
                        return mandreel_embed_plugin.MandreelInterCallFunction(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4], arguments[5], arguments[6], arguments[7], arguments[8], arguments[9], arguments[10], arguments[11], arguments[12], arguments[13]);
                case 15:
                        return mandreel_embed_plugin.MandreelInterCallFunction(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4], arguments[5], arguments[6], arguments[7], arguments[8], arguments[9], arguments[10], arguments[11], arguments[12], arguments[13], arguments[14]);
                case 16:
                        return mandreel_embed_plugin.MandreelInterCallFunction(arguments[0], arguments[1], arguments[2], arguments[3], arguments[4], arguments[5], arguments[6], arguments[7], arguments[8], arguments[9], arguments[10], arguments[11], arguments[12], arguments[13], arguments[14], arguments[15]);
                break;
    }
}

function mandreel_plugin_draw()
{
        var canvas = Mandreel_document.getElementById('canvasDiv');
        WebGLUtils.requestAnimationFrame(canvas, mandreel_plugin_draw);
        mandreel_embed_plugin.MandreelInterCalls("AppDraw");

}

function mandreelInitPluginPlatform(params)
{

        if ( params.width != null )
                mandreelAppWidthSrc = params.width;

        if ( params.height != null )
                mandreelAppHeightSrc = params.height;


        mandreel_embed_plugin = Mandreel_document.createElement('embed');
        mandreel_embed_plugin.setAttribute('width',mandreelAppWidthSrc);
        mandreel_embed_plugin.setAttribute('height',mandreelAppHeightSrc);
        mandreel_embed_plugin.setAttribute('type',"application/halfbrick-npruntime-scriptable-plugin");

        var div = Mandreel_document.getElementById('canvasDiv');


        var oChild=div.firstChild;

        div.replaceChild(mandreel_embed_plugin, oChild);

        var flashElement = Mandreel_document.getElementById('FlashWrapper');
        if ( flashElement != null )
        {
                flashElement.style.visibility = "hidden";
                flashElement.style.width = "0px";
                flashElement.style.height = "0px";
    }





        Mandreel_window.MandreelInterWriteInt     = PluginMandreelInterWriteInt;
        Mandreel_window.MandreelInterWriteFloat   = PluginMandreelInterWriteInt;
        Mandreel_window.MandreelInterWriteString  = PluginMandreelInterWriteString;
        Mandreel_window.MandreelInterWriteWString = PluginMandreelInterWriteWString;
        Mandreel_window.MandreelLockFrame         = PluginMandreelLockFrame;
        Mandreel_window.MandreelUnlockFrame       = PluginMandreelUnlockFrame;
        Mandreel_window.MandreelInterCallFunction = PluginMandreelInterCallFunction;
        Mandreel_window.MandreelPause                    = PluginMandreelPause;
        Mandreel_window.MandreelResume                   = PluginMandreelResume;

        Mandreel_setTimeout(function () {

        if ( typeof(params.pluginSolutionName) != 'undefined' )
                mandreel_embed_plugin.init(params.pluginDLL, params.pluginWorkingFolder,params.pluginSolutionName);
        else
                mandreel_embed_plugin.init(params.pluginDLL, params.pluginWorkingFolder);


        mandreelAppStartStateFunc("ready",mandreelAppWidthSrc,mandreelAppHeightSrc);

        Mandreel_setTimeout("mandreel_plugin_draw()", 16);
    }, 100);

}


function MandreelInterSwfProgress(mode, percentage, bytes, totalbytes)
{
        imandreel_update_load(bytes, totalbytes);
        if (mode == 'files')
        {
                if ( mandreelAppStartStateFunc )
                        mandreelAppStartStateFunc("loadingData",percentage);
    }
        else if (mode == 'audio')
        {
                if ( mandreelAppStartStateFunc )
                        mandreelAppStartStateFunc("loadingAudioUpdate",percentage);
    }
        else if (mode == 'textureasync')
        {
                if ( mandreelAppStartStateFunc )
                        mandreelAppStartStateFunc("loadingTextureAsyncPack",percentage);
    }

}

function MandreelInterSwfCheckMethod(method)
{
        if (typeof(Mandreel_window[method])=="undefined")
                return 0;
        else
                return 1;
}

// End of js/mandreelapp.js file.

// Start of mandreel.js file.

var mandreel_total_memory = 15908864;
var mandreel_stack_memory = 1048576;
var mandreel_heap_memory = 13591752; //init_memory = 744248
/////////////////////////////////////////////
// Heap
/////////////////////////////////////////////
var heap;
var heap8;
var heapU8;
var heap16;
var heapU16;
var heap32;
var heapU32;
var heapFloat;
var heapDouble;
var heapNewPos = 512;

var ABORT = false;

var g_mandreel_cache_files = false;


var g_mandreel_frame_locked = false;
var g_mandreel_frame_inframe = false;

var mandreel_cache_files = [];
var g_mandreel_working_folder = 'DataPC/';
var g_mandreel_datafiles_sufix = '.dat';
var __FUNCTION_TABLE__ = [];
var mandreel_pos_function_table = 1;
function register_delegate(ptr_func)
{
        var functionId = mandreel_pos_function_table;
        __FUNCTION_TABLE__[functionId] = ptr_func;

        mandreel_pos_function_table++;
        return functionId*4;
}


var g_addr_emit = 0;
function emit_start(addr)
{
        g_addr_emit = addr;
}

function emit_8(data)
{
        heapU8[g_addr_emit] = data;
        g_addr_emit++;
}

function emit_16(data)
{
        heapU16[g_addr_emit>>1] = data;
        g_addr_emit+=2;
}

function emit_32(data)
{
        heapU32[g_addr_emit>>2] = data;
        g_addr_emit+=4;
}

function emit_fill(data, size)
{
        var j;
        for (j=0;j<size;j++)
        {
                heapU8[g_addr_emit] = data;
                g_addr_emit++;
    }
}

function emit_string(v)
{
        var j;
        var len = v.length;
        var data;

        for(j = 0; j < len; j++)
        {
                data = v.charCodeAt(j);

                heapU8[g_addr_emit] = data;
                g_addr_emit++;
    }
}





var g_stack_pointer = Malloc(mandreel_stack_memory);

function assert_unalign()
{
        dump("fatal error: unaligned memory access detected!!!!");
        assert(false);
}

function _assert(sp)
{
        var p0 = heap32[sp>>2];sp+=4;
  var p1 = heap32[sp>>2];sp+=4;
  var line = heap32[sp>>2];sp+=4;
        var name = get_string_from_ptr(p0);
        var file = get_string_from_ptr(p1);
        assert(false, name + file + ' ' + line);
}
__cxa_pure_virtual.__index__ = 0;
function __cxa_pure_virtual()
{
        assert(0);
}

// operator delete[]
function _ZdaPv(sp)
{
        free(sp);
}

// operator delete
function _ZdlPv(sp)
{
        free(sp);
}

// operator new[](unsigned int)
function _Znaj(sp)
{
        malloc(sp);
}
// operator new[](unsigned int)
function _Znwj(sp)
{
        malloc(sp);
}

function abort(sp)
{
        assert(0);
}

var r_g0 = 0;
var r_g1 = 0;
var f_g0 = 0;

//isFinite(aux)
//isNaN(aux)

var tlsf_ptr = 0;


function initHeap()
{
        heap = new ArrayBuffer(mandreel_total_memory);
        heap8 = new Int8Array(heap);
        heapU8 = new Uint8Array(heap);
        heap16 = new Int16Array(heap);
        heapU16 = new Uint16Array(heap);
        heap32 = new Int32Array(heap);
        heapU32 = new Uint32Array(heap);
        heapFloat = new Float32Array(heap);
        heapDouble = new Float64Array(heap);


        for (var i=0;i<mandreel_total_memory/4;i++)
        {
                heapU32[i] = 0;
    }
}

function Malloc(bytes)
{
        if ( heap == undefined )
        {
                //initHeap();
    }
        var newOffset = heapNewPos;
        // Always 32 bit aligned
        heapNewPos += ((bytes + 3) & 0xfffffffc);

        if (heapNewPos>mandreel_total_memory)
        {
                assert(false);
    }

        return newOffset;
}

function assert(condition, _text) {
//console.assert(condition, _text);
    if (!condition) {
      var text = "Assertion failed: " + _text;
      alert(text + ':\n' + (new Error).stack);
      ABORT = true;
      throw "Assertion: " + text;
    }
  }

  function my_assert(sp)
  {
        var p0 = heap32[sp>>2];sp+=4;
  var p1 = heap32[sp>>2];sp+=4;
        //var name = get_string_from_ptr(p1);

        assert(false, 'hola');
  }

  function WriteHeapDouble(addr, value)
  {
  //assert((addr&7)==0);
        heapDouble[addr>>3] = value;
  }

    function WriteHeapU64(addr, value)
  {
        heap32[addr>>2] = value.l;
        heap32[(addr>>2)+1] = value.h;
  }


var arg_test_local = Malloc(8);
function my_arg_test(sp)
{
        var ptr = heapU32[sp>>2];
        var size = heapU32[(sp+4)>>2];

        var arg = heapU32[ptr>>2];


        if (size == 4)
        {
        heap32[ptr>>2] = arg+4;

        arg = heap32[arg>>2];

        heap32[arg_test_local>>2] = arg;

        //dump('my_arg_test ' + arg + ' ' + ptr + '\n');

    }
        else
        {
                arg = (arg+7) & ~7;

                heap32[ptr>>2] = arg+8;

        //assert((arg&7)==0);
        var value0 = heap32[arg>>2];
        var value1 = heap32[(arg+4)>>2];
        //arg = llvm_readDouble(arg);

        //assert((arg_test_local&7)==0);

        heap32[arg_test_local>>2] = value0;
        heap32[(arg_test_local+4)>>2] = value1;

        //llvm_writeDouble(arg_test_local,arg);

        //dump('my_arg_test ' + arg + ' ' + ptr + '\n');


    }




        r_g0 = arg_test_local;
}






function uint(value) {
    if (value >= 0) return value;
    return 4294967296 + value;
  }



function puts(sp)
{
        var addr = heapU32[sp>>2];

        var name = get_string_from_ptr(addr);

        name+='\n';

        dump(name);

}

function _Z11print_valued(_stack_pos, value)
{
        dump(value);
        dump('\n');
}

function _Z11print_labelPKc(_stack_pos, addr)
{
        puts(_stack_pos,addr);
        dump('\n');
}




function gettimeofday(sp)
  {
  var ptr = heap32[sp>>2];
  var time_ms = Date_now();
        heap32[ptr>>2] = time_ms/1000;
        heap32[(ptr>>2)+1] = (time_ms%1000)*1000;
        r_g0 = 0;
  }


  function free(sp)
  {
        var ptr = heapU32[sp>>2];
        sp-=8;

        heap32[(sp)>>2] = tlsf_ptr;
        heap32[(sp+4)>>2] = ptr;
        tlsf_free(sp);
  }

  function malloc_size(sp)
  {
  var ptr = heapU32[sp>>2];

        sp-=4;

        heap32[(sp)>>2] = ptr;
        tlsf_block_size(sp);
  }


  function realloc(sp)
  {
        var ptr = heapU32[sp>>2];
        var size = heapU32[(sp+4)>>2];

        //assert(ptr == 0);

        sp-=12;

        //dump('realloc ' + sp + ' ' + ptr + ' ' + size + '\n');

        heap32[(sp)>>2] = tlsf_ptr;
        heap32[(sp+4)>>2] = ptr;
        heap32[(sp+8)>>2] = size;
        tlsf_realloc(sp);

        //dump('return ' + r_g0 + '\n');
  }

  var llvm_double_addr = Malloc(8);

  function llvm_writeDouble(addr,src)
  {
  //assert((llvm_double_addr&7)==0);
        heapDouble[llvm_double_addr>>3] = src;

        //assert((addr&7)==0);

        var val0 = heap32[(llvm_double_addr)>>2];
        var val1 = heap32[(llvm_double_addr+4)>>2];

        heap32[(addr)>>2] = val0;
        heap32[(addr+4)>>2] = val1;
  }

  function llvm_readDouble(addr)
  {
        //assert((addr&7)==0);

        var val0 = heap32[(addr)>>2];
        var val1 = heap32[(addr+4)>>2];

        heap32[(llvm_double_addr)>>2] = val0;
        heap32[(llvm_double_addr+4)>>2] = val1;


//      assert((llvm_double_addr&7)==0);
        var result = heapDouble[llvm_double_addr>>3];


        return result;

  }

  function llvm_move_double(addr_dst, addr_src)
  {

        var val0 = heapU32[(addr_src)>>2];
        var val1 = heapU32[(addr_src+4)>>2];

        heapU32[(addr_dst)>>2] = val0;
        heapU32[(addr_dst+4)>>2] = val1;

  }

  function llvm_move_float(addr_dst, addr_src)
  {
        heapU32[(addr_dst)] = heapU32[(addr_src)];
  }

  function malloc(sp)
  {
        var size = heapU32[sp>>2];

        if (size == 0)
        {
                size = 4;
    }


        if (tlsf_ptr == 0)
        {
                var addr = Malloc(mandreel_heap_memory);

                sp-=8;
                heap32[(sp)>>2] = addr;
                heap32[(sp+4)>>2] = mandreel_heap_memory;
                tlsf_create(sp);
                tlsf_ptr = r_g0;
    }

        sp-=8;

        heap32[(sp)>>2] = tlsf_ptr;
        heap32[(sp+4)>>2] = size;
        tlsf_malloc(sp);

        if (r_g0 == 0)
        {
                dump('malloc failed ' + size + '\n');
                assert(false);
    }
  }


   function log10f(sp)
  {
  var value = heapFloat[sp>>2];
        f_g0 = Math.log(value)/Math.LN10;
  }

   function log10(sp)
  {
  var value = heapDouble[sp>>3];
        f_g0 = Math.log(value)/Math.LN10;
  }

function logf(sp)
  {
  var value = heapFloat[sp>>2];
        f_g0 = Math.log(value);
  }

  function log(sp)
  {
  var value = heapDouble[sp>>3];
        f_g0 = Math.log(value);
  }



  function cosf(sp)
  {
  var value = heapFloat[sp>>2];
        f_g0 = Math.cos(value);
        //assert (isNaN(f_g0) == false);
  }

  function acosf(sp)
  {
  var value = heapFloat[sp>>2];
        f_g0 = Math.acos(value);
  }

  function asinf(sp)
  {
  var value = heapFloat[sp>>2];
        f_g0 = Math.asin(value);
  }

  function asin(sp)
  {
  var value = heapDouble[sp>>3];
        f_g0 = Math.asin(value);
  }

  function acos(sp)
  {
  var value = heapDouble[sp>>3];
        f_g0 = Math.acos(value);
  }

  function floor(sp)
  {
  var value = heapDouble[sp>>3];
        f_g0 = Math.floor(value);
  }

  function floorf(sp)
  {
  var value = heapFloat[sp>>2];
        f_g0 = Math.floor(value);
  }

  function round(sp)
  {
  var value = heapDouble[sp>>3];
        f_g0 = Math.round(value);
  }

  function roundf(sp)
  {
  var value = heapFloat[sp>>2];
        f_g0 = Math.round(value);
  }

  function ceilf(sp)
  {
  var value = heapFloat[sp>>2];
        f_g0 = Math.ceil(value);
  }

  function ceil(sp)
  {
  var value = heapDouble[sp>>3];
        f_g0 = Math.ceil(value);
  }


  function exp2(sp)
  {
  var value = heapDouble[sp>>3];

        f_g0 = Math.pow(2,value);
  }

  function exp2f(sp)
  {
  var value = heapFloat[sp>>2];

        f_g0 = Math.pow(2,value);
  }



  function pow(sp)
  {
  var value = heapDouble[sp>>3];
  var value2 = heapDouble[(sp+8)>>3];
        f_g0 = Math.pow(value,value2);
  }

  function powf(sp)
  {
  var value = heapFloat[sp>>2];
  var value2 = heapFloat[(sp+4)>>2];
        f_g0 = Math.pow(value,value2);
  }

  function cos(sp)
  {
  var value = heapDouble[sp>>3];
        f_g0 = Math.cos(value);
        //assert (isNaN(f_g0) == false);
  }

  function tan(sp)
  {
  var value = heapDouble[sp>>3];
        f_g0 = Math.tan(value);
        //assert (isNaN(f_g0) == false);
  }

   function sinf(sp)
  {
  var value = heapFloat[sp>>2];
        f_g0 = Math.sin(value);

        //assert (isNaN(f_g0) == false);
  }

  function expf(sp)
  {
  var value = heapFloat[sp>>2];
        f_g0 = Math.exp(value);
  }

  function exp(sp)
  {
  var value = heapDouble[sp>>3];
        f_g0 = Math.exp(value);
  }

  function tanf(sp)
  {
  var value = heapFloat[sp>>2];
        f_g0 = Math.tan(value);
  }

 function atanf(sp)
  {
  var value = heapFloat[sp>>2];
        f_g0 = Math.atan(value);
  }

  function atan(sp)
  {
  var value = heapDouble[sp>>3];
        f_g0 = Math.atan(value);
  }

  function abs(sp)
  {
  var value = heap32[sp>>2];
  if (value<0)
  r_g0 = -value;
  else
  r_g0 = value;
  }

  function sin(sp)
  {
  var value = heapDouble[sp>>3];
        f_g0 = Math.sin(value);
  }

  function sqrtf(sp)
  {
  var value = heapFloat[sp>>2];
        f_g0 = Math.sqrt(value);
  }

  function sqrt(sp)
  {
  var value = heapDouble[sp>>3];
        f_g0 = Math.sqrt(value);
  }

  function fmod(sp)
  {
  var value = heapDouble[sp>>3];sp+=8;
  var value2 = heapDouble[sp>>3];
        f_g0 = value % value2;
  }

   function fmodf(sp)
  {
  var value = heapFloat[sp>>2];sp+=4;
  var value2 = heapFloat[sp>>2];
        f_g0 = value % value2;
  }


  function atan2f(sp)
  {
  var x = heapFloat[sp>>2];sp+=4;
  var y = heapFloat[sp>>2];
        f_g0 = Math.atan2(x,y);
  }

  function atan2(sp)
  {
  var x = heapDouble[sp>>3];
  var y = heapDouble[(sp+8)>>3];
        f_g0 = Math.atan2(x,y);
  }

  function fabs(sp)
  {
  var value = heapDouble[sp>>3];
        f_g0 = Math.abs(value);
  }


  function _Z18OutputDebugStringAPKc(sp)
  {
  puts(sp);

  }


  function getenv(sp)
  {
  r_g0 = 0;
  }


  function mandreel_fcmp_ord(X, Y)
  {
        return (X == X && Y == Y);
  }

  function mandreel_fcmp_uno(X, Y)
{

        return (X != X || Y != Y);
}

var llvm_errno = Malloc(4);
function _errno(sp)
{
        r_g0 = llvm_errno;
}




if (!Mandreel_window["dump"])
        Mandreel_window["dump"] = function dump(str){console.log(str)} ;




  function get_string_from_ptr(ptr)
  {
        var ret = "";

        if (ptr == 0)
                return ret;

        var i = 0;
        while (1) {
  //    if ((ptr.pos + i) >= ptr.slab.length) { return "<< Invalid read: " + (ptr.pos+i) + " : " + ptr.slab.length + " >>"; } else {}
        if (heapU8[ptr + i] == 0)
                break;

      var t = String.fromCharCode(heapU8[ptr + i]);
      ret += t;
      i += 1;
    }

        return ret;
  }

  function fill_ptr_from_string(ptr, v)
  {
        var j;
        var len = v.length;
        var data;

        for(j = 0; j < len; j++)
        {
                data = v.charCodeAt(j);

                heapU8[ptr] = data;
                ptr++;
    }
                heapU8[ptr] = 0;
    }

  var file_ids = [];
  var current_file_id = 20;

  function create_file_id(buffer)
  {
        this.buffer = buffer;
        this.offset = 0;
        this.byteArray = new Uint8Array(buffer);
  }

  function mandreel_rewind(sp)
  {
  var file_id = heap32[sp>>2];sp+=4;

  file_ids[file_id].offset = 0;

        r_g0 = 0;

        //return 0;
  }


  function mandreel_fseek(sp)
  {
  var file_id = heap32[sp>>2];sp+=4;
  var pos = heap32[sp>>2];sp+=4;
  var type = heap32[sp>>2];sp+=4;

        if (type == 2)
        {
                file_ids[file_id].offset = file_ids[file_id].buffer.byteLength + pos;
    }
        else if (type == 1)
        {
                file_ids[file_id].offset = file_ids[file_id].offset + pos;

    }
        else if (type == 0)
        {
                file_ids[file_id].offset = pos;

    }

        r_g0 = 0;

        //return 0;
  }

  function mandreel_fclose(sp)
  {
  var file_id = heap32[sp>>2];sp+=4;

        file_ids[file_id] = null;
        r_g0 = 0;
        //return 0;
  }



  function mandreel_feof(sp)
  {
  var file_id = heap32[sp>>2];sp+=4;

  var offset = file_ids[file_id].offset;
  var total = file_ids[file_id].buffer.byteLength;

  if (offset>=total)
  r_g0 = 1;
  else
  r_g0 = 0;

  }

  function mandreel_getc(sp)
  {
  var file_id = heap32[sp>>2];sp+=4;


  var offset = file_ids[file_id].offset;


        var buffer = file_ids[file_id].buffer;

        var byteArray = file_ids[file_id].byteArray;

        var total = 1;

        var result;

        if ((offset+total)>buffer.byteLength)
        {
                result = -1;
    }
        else
        {
                result = byteArray[offset];
                file_ids[file_id].offset+=total;
    }

        r_g0 = result;
  }



  function mandreel_fread(sp)
  {
  var ptr = heap32[sp>>2];sp+=4;
  var size = heap32[sp>>2];sp+=4;
  var count = heap32[sp>>2];sp+=4;
  var file_id = heap32[sp>>2];sp+=4;

  var offset = file_ids[file_id].offset;

  //dump('fread ' + ptr + ' ' + size + ' ' + count + ' ' + file_id + ' ' + offset + '\n');

        var buffer = file_ids[file_id].buffer;

        var total = size*count;

        if ((offset+total)>buffer.byteLength)
                total = buffer.byteLength-offset;

        var byteArray = file_ids[file_id].byteArray;


        var sub_array = byteArray.subarray(offset, offset+total);

        heapU8.set(sub_array,ptr);


        //heapU8.set(byteArray, ptr);
        //for (var i=0;i<total;++i)
        //{
//              heapU8[ptr+i] = byteArray[i+offset];
//    }


        file_ids[file_id].offset+=total;

        r_g0 = total/size;
        //return total;
  }

  function mandreel_ftell(sp)
  {
  var file_id = heap32[sp>>2];sp+=4;

        var value = file_ids[file_id].offset;
        //dump('offset ftell ' + value + '\n');
        r_g0 = value;
        //return value;
  }

  function _Z30mandreel_fopen_enable_checkfatb(sp)
  {
  }

  function mandreel_ungetc(sp)
  {
        var my_char = heap32[sp>>2];sp+=4;
        var file_id = heap32[sp>>2];sp+=4;

        var offset = file_ids[file_id].offset-1;

        var byteArray = file_ids[file_id].byteArray;

        assert(byteArray[offset] == my_char);

        file_ids[file_id].offset = offset;

        return my_char;
  }
  function mandreel_fopen(sp)
  {
  var ptr_name = heap32[sp>>2];sp+=4;
  var ptr_flags = heap32[sp>>2];sp+=4;


        var name = get_string_from_ptr(ptr_name);
        var flags = get_string_from_ptr(ptr_flags);
        //dump('fopen\n');
        //dump(name);
        //dump('\n');
        //dump(flags);
        //dump('\n');

        var buffer;

        var full_name;

        name = name.toLowerCase();

        name = name.replace(/\\/g,"/");

        full_name       = g_mandreel_working_folder + name + g_mandreel_datafiles_sufix;


        buffer =mandreel_cache_files[name];

        if (buffer == null)
        {
        r_g0 = 0;
        return;
    }



        //dump('\nopening file ' + full_name + ' ' + buffer.byteLength + '\n');


        file_ids[current_file_id] = new create_file_id(buffer);

        var old_id = current_file_id;
        current_file_id++;

        r_g0 = old_id;
        //return old_id;
  }

  function llvm_store_unalign32_float(addr, value)
  {
        heapFloat[0] = value;
        var data = heap32[0];
        heap8[addr] = data&0xff;
        heap8[addr+1] = (data>>>8)&0xff;
        heap8[addr+2] = (data>>>16)&0xff;
        heap8[addr+3] = (data>>>24)&0xff;
  }
  function llvm_store_unalign32(addr, value)
  {
        heap8[addr] = value&0xff;
        heap8[addr+1] = (value>>>8)&0xff;
        heap8[addr+2] = (value>>>16)&0xff;
        heap8[addr+3] = (value>>>24)&0xff;
  }

  function llvm_read_unalign32(addr)
  {
        var value;
        value = heapU8[addr];
        value |= heapU8[addr+1]<<8;
        value |= heapU8[addr+2]<<16;
        value |= heapU8[addr+3]<<24;
        return value;
  }

  function llvm_read_unalign32_float(addr)
  {
        var value;
        value = heapU8[addr];
        value |= heapU8[addr+1]<<8;
        value |= heapU8[addr+2]<<16;
        value |= heapU8[addr+3]<<24;

        heap32[0] = value;
        return  heapFloat[0];
  }

  function mandreel_getlocalstorage()
  {
        return Mandreel_window.localStorage;
        //return Mandreel_window.sessionStorage;
  }

  function mandreel_openls(sp)
  {
        var ptr_name = heap32[sp>>2];sp+=4;

        var key = get_string_from_ptr(ptr_name);

        var my_localStorage = mandreel_getlocalstorage();

        var value = my_localStorage.getItem(key);

        if (value == null)
        {
                r_g0 = -1;
                return;
    }


        var length = my_localStorage.getItem(key + '_size');

        if (length == null)
        {
                r_g0 = -1;
                return;
    }




        dump('mandreel_openls ' + key + ' return ' + length);


        r_g0 = parseInt(length);



        return;

  }

  function mandreel_readls(sp)
  {
        var ptr_name = heap32[sp>>2];sp+=4;
        var data_dst = heap32[sp>>2];sp+=4;
        var data_len = heap32[sp>>2];sp+=4;

        var key = get_string_from_ptr(ptr_name);

        var my_localStorage = mandreel_getlocalstorage();

        var value = my_localStorage.getItem(key);

        var data = JSON.parse(value);


        for (var i=0;i<data_len;++i)
        {
                heapU8[data_dst+i] = data[i];
    }

        r_g0 =  data_len;
        return;

}

function mandreel_removels(sp)
 {
 var ptr_name_a = heap32[sp>>2];sp+=4;
 var key_a = get_string_from_ptr(ptr_name_a);

 var my_localStorage = mandreel_getlocalstorage();

        my_localStorage.removeItem(key_a);
        my_localStorage.removeItem(key_a + '_size');
        r_g0 = 0;

 }


function mandreel_renamels(sp)
 {
 var ptr_name_a = heap32[sp>>2];sp+=4;
  var ptr_name_b = heap32[sp>>2];sp+=4;

  var key_a = get_string_from_ptr(ptr_name_a);
  var key_b = get_string_from_ptr(ptr_name_b);

  var my_localStorage = mandreel_getlocalstorage();


  var value = my_localStorage.getItem(key_a);
  var value2 = my_localStorage.getItem(key_a + '_size');

  if (value != null && value2 != null)
  {
        my_localStorage.setItem(key_b, value);
        my_localStorage.setItem(key_b + '_size', value2);

        my_localStorage.removeItem(key_a);
        my_localStorage.removeItem(key_a + '_size');


        r_g0 = 0;
}
else
 r_g0 = -1;
 }

function mandreel_writels(sp)
  {
        var ptr_name = heap32[sp>>2];sp+=4;
        var data_src = heap32[sp>>2];sp+=4;
        var data_len = heap32[sp>>2];sp+=4;

        var key = get_string_from_ptr(ptr_name);



        var data = new Uint8Array(heap,data_src,data_len);

        var value = JSON.stringify(data);

        var my_localStorage = mandreel_getlocalstorage();

        try
        {
                my_localStorage.setItem(key, value);
    } catch(e)
        {
                dump('error saving ' + key);
                dump(e.message);
                r_g0 =  0;
                return;
    }

        try
        {
                my_localStorage.setItem(key + '_size', data_len);
    } catch(e)
        {
                dump('error saving ' + key);
                dump(e.message);
                r_g0 =  0;
                return;
    }


        r_g0 =  data_len;
        return;

}

function mandreel_call_constructors(_ptr, size,stackPos)
{
var ptr = _ptr;

ptr = ptr >> 2;

for (var i=0;i<size;++i)
{


var tag = heap32[ptr];
var ptr_id = heap32[ptr+1];

__FUNCTION_TABLE__[(ptr_id)>>2](stackPos);

ptr+=2;

}
}

function get_string_from_wptr(ptr)
  {
        var ret = "";

        if (ptr == 0)
                return ret;

        assert((ptr&1)==0);
        ptr>>=1;
        var i = 0;
        while (1) {
  //    if ((ptr.pos + i) >= ptr.slab.length) { return "<< Invalid read: " + (ptr.pos+i) + " : " + ptr.slab.length + " >>"; } else {}
        if (heapU16[ptr + i] == 0)
                break;

      var t = String.fromCharCode(heapU16[ptr + i]);
     // if (t == "\0") { break; } else {}
      ret += t;
      i += 1;
    }

        return ret;
  }

  function fill_wptr_from_string(ptr, v)
  {
        var j;
        var len = v.length;
        var data;

        assert((ptr&1)==0);
        ptr>>=1;

        for(j = 0; j < len; j++)
        {
                data = v.charCodeAt(j);

                heapU16[ptr] = data;
                ptr++;
    }
                heapU16[ptr] = 0;
    }

function mandreelInterGetParams(sp)
{
        var params = [];

        var offset = 0;
        for (i=1;i<arguments.length;++i)
        {
                var type = arguments[i];

                switch(type)
                {
                        case 'int':
                                params[i-1] = heap32[(sp+offset)>>2];
                                break;
                        case 'float':
                                params[i-1] = heapFloat[(sp+offset)>>2];
                                break;
                        case 'string':
                                params[i-1] = get_string_from_ptr(heap32[(sp+offset)>>2]);
                                break;
                        default:
                                assert(false);
            }
                offset+=4;
    }

        return params;
}

function mandreelInterRetParam(type, value)
{
        switch(type)
        {
                case 'int':
                        r_g0 = value;
                        break;
                case 'float':
                        f_g0 = value;
                        break;
                default:
                        assert(false);
    }

        return 0;
}

function MandreelInterGetFunctionPtr(value)
{
        return __FUNCTION_TABLE__[value >> 2];
}


function MandreelInterCallFunction(returnType,func_name)
{
        var size_params = 0;

        var i;
        var num_params = (arguments.length-2)/2;
        num_params|=0;
        for (i=2;i<num_params*2+2;i+=2)
        {
                var type = arguments[i];

                var size_arg = 0;
                switch(type)
                {
                        case 'int':
                                size_arg = 4;
                                break;
                        case 'float':
                                size_arg = 4;
                                break;
                        case 'string':
                                size_arg = 4;
                                size_arg += ((arguments[i+1].length + 4) & 0xfffffffc);
                                break;
                        case 'wstring':
                                size_arg = 4;
                                size_arg += ((arguments[i+1].length*2 + 4) & 0xfffffffc);
                                break;
                        default:
                                assert(false);
            }

                size_params += size_arg;
    }

        // stack always 8 byte aligned
        size_params=((size_params+7)& 0xfffffff8);

        var sp = 0;

        if (i<(arguments.length))
                sp = arguments[i];
        else
        {
                assert(false,"MandreelInterCallFunction missing stack pointer paramenter");
                //assert(g_mandreel_frame_locked == true);
                //sp = g_stack_pointer+800*1024;
    }

        sp-=size_params;

        var offset = 0;
        var ptr_data = num_params*4+sp;
    for (i=2;i<num_params*2+2;i+=2)
        {
                var type = arguments[i];

                var size_arg = 0;
                switch(type)
                {
                        case 'int':
                                heap32[(sp+offset)>>2] = arguments[i+1];
                                break;
                        case 'float':
                                heapFloat[(sp+offset)>>2] = arguments[i+1];
                                break;
                        case 'string':
                                {
                                        heap32[(sp+offset)>>2] = ptr_data;
                                        var string = arguments[i+1];
                                        fill_ptr_from_string(ptr_data,string);

                                        ptr_data += ((string.length + 4) & 0xfffffffc);
                            }
                                break;
                        case 'wstring':
                                {
                                        MandreelInterWriteInt((sp+offset),ptr_data);
                                        var string = arguments[i+1];
                                        MandreelInterWriteWString(ptr_data,string);

                                        ptr_data += ((string.length*2 + 4) & 0xfffffffc);
                            }
                                break;
                        default:
                                assert(false);
            }
                offset+=4;
    }

        Mandreel_window[func_name](sp);

        if (returnType == 'int')
                return r_g0;
        else if (returnType == 'float')
                return f_g0;
        else
        {
                assert(returnType == 'void');
                return;
    }
}


function MandreelInterCallFunctionPtr(returnType,func_ptr)
{
        var size_params = 0;

        var i;
        var num_params = (arguments.length-2)/2;
        num_params|=0;
        for (i=2;i<num_params*2+2;i+=2)
        {
                var type = arguments[i];

                var size_arg = 0;
                switch(type)
                {
                        case 'int':
                                size_arg = 4;
                                break;
                        case 'float':
                                size_arg = 4;
                                break;
                        case 'string':
                                size_arg = 4;
                                size_arg += ((arguments[i+1].length + 4) & 0xfffffffc);
                                break;
                        case 'wstring':
                                size_arg = 4;
                                size_arg += ((arguments[i+1].length*2 + 4) & 0xfffffffc);
                                break;
                        default:
                                assert(false);
            }

                size_params += size_arg;
    }

        // stack always 8 byte aligned
        size_params=((size_params+7)& 0xfffffff8);

        var sp = 0;

        if (i<(arguments.length))
                sp = arguments[i];
        else
        {
                assert(false);
                //assert(g_mandreel_frame_locked == true);
                //sp = g_stack_pointer+800*1024;
    }

        sp-=size_params;

        var offset = 0;
        var ptr_data = num_params*4+sp;
    for (i=2;i<num_params*2+2;i+=2)
        {
                var type = arguments[i];

                var size_arg = 0;
                switch(type)
                {
                        case 'int':
                                heap32[(sp+offset)>>2] = arguments[i+1];
                                break;
                        case 'float':
                                heapFloat[(sp+offset)>>2] = arguments[i+1];
                                break;
                        case 'string':
                                {
                                        heap32[(sp+offset)>>2] = ptr_data;
                                        var string = arguments[i+1];
                                        fill_ptr_from_string(ptr_data,string);

                                        ptr_data += ((string.length + 4) & 0xfffffffc);
                            }
                                break;
                        case 'wstring':
                                {
                                        MandreelInterWriteInt((sp+offset),ptr_data);
                                        var string = arguments[i+1];
                                        MandreelInterWriteWString(ptr_data,string);

                                        ptr_data += ((string.length*2 + 4) & 0xfffffffc);
                            }
                                break;
                        default:
                                assert(false);
            }
                offset+=4;
    }

        __FUNCTION_TABLE__[(func_ptr)>>2](sp);

        if (returnType == 'int')
                return r_g0;
        else if (returnType == 'float')
                return f_g0;
        else
        {
                assert(returnType == 'void');
                return;
    }
}


var MANDREEL_HTTP_REQUEST_MODE_GET = 0;
var MANDREEL_HTTP_REQUEST_MODE_POST = 1;
var MANDREEL_HTTP_REQUEST_MODE_PUT = 2;

var MANDREEL_HTTP_REQUEST_STATUS_ERROR = 0;
var MANDREEL_HTTP_REQUEST_STATUS_BUSY = 1;
var MANDREEL_HTTP_REQUEST_STATUS_FINISHED = 2;
var MANDREEL_HTTP_REQUEST_STATUS_INIT = 3;


var mandreel_js_mapping_ids = [];
var mandreel_js_mapping_ids_free = [];


function Mandreel_HttpRequest_Create(sp)
{
        var ptr_url = heap32[sp>>2];sp+=4;
        var type = heap32[sp>>2];sp+=4;


        var url = get_string_from_ptr(ptr_url);


        var str_type = 'GET';
        if (type == MANDREEL_HTTP_REQUEST_MODE_GET)
                str_type = 'GET';
        else if (type == MANDREEL_HTTP_REQUEST_MODE_PUT)
                str_type = 'PUT';
        else if (type == MANDREEL_HTTP_REQUEST_MODE_POST)
                str_type = 'POST';

        var xmlhttp_get = new XMLHttpRequest();
        xmlhttp_get.open(str_type,url);

        if("responseType" in xmlhttp_get)
                xmlhttp_get.responseType="arraybuffer";
    else
        {
                xmlhttp_get.overrideMimeType('text/plain; charset=x-user-defined');
    }

        if (mandreel_js_mapping_ids_free.length == 0)
                mandreel_js_mapping_ids_free.push(mandreel_js_mapping_ids.length);

        var new_id = mandreel_js_mapping_ids_free.pop();

        var my_state = {
        buffer : null,
        httpRequest : xmlhttp_get,
        status : MANDREEL_HTTP_REQUEST_STATUS_INIT,
        offset_read : 0
  };



        mandreel_js_mapping_ids[new_id] = my_state;

        r_g0 = new_id+1;
}

function Mandreel_HttpRequest_Send(sp)
{
        var _id = heap32[sp>>2];sp+=4;
        var ptr_data = heap32[sp>>2];sp+=4;
        var id = _id-1;

        var data;

        if (ptr_data)
                data = get_string_from_ptr(ptr_data);
        else
                data = null;

        var my_state = mandreel_js_mapping_ids[id];


        my_state.status = MANDREEL_HTTP_REQUEST_STATUS_BUSY;

        my_state.httpRequest.onreadystatechange = function()
        {
                if (my_state.httpRequest.readyState==4)
                {
                        if (my_state.httpRequest.status==200)
                        {
                                var buffer;

                                if (my_state.httpRequest.responseType=="arraybuffer")
                                        buffer=my_state.httpRequest.response;
                                else if (my_state.httpRequest.mozResponseArrayBuffer != null)
                                        buffer = my_state.httpRequest.mozResponseArrayBuffer;
                                else
                                        buffer=my_state.httpRequest.response;

                                my_state.status = MANDREEL_HTTP_REQUEST_STATUS_FINISHED;
                                my_state.buffer =  new Uint8Array(buffer);
                                //alert(my_state.buffer.length);

                                //alert(mandreel_js_mapping_ids[id].buffer);

                    }
                        else
                                my_state.status = MANDREEL_HTTP_REQUEST_STATUS_ERROR;
            }
    }

        my_state.httpRequest.send(data);
}


function Mandreel_HttpRequest_Status(sp)
{
        var _id = heap32[sp>>2];sp+=4;
        var id = _id-1;


        r_g0 = mandreel_js_mapping_ids[id].status;
}

function Mandreel_HttpRequest_Read(sp)
{
        var _id = heap32[sp>>2];sp+=4;
        var ptr = heap32[sp>>2];sp+=4;
        var size = heap32[sp>>2];sp+=4;
        var id = _id-1;

        var remaining_bytes =  mandreel_js_mapping_ids[id].buffer.length - mandreel_js_mapping_ids[id].offset_read;

        if (size>remaining_bytes)
                size = remaining_bytes;

        var sub_array = mandreel_js_mapping_ids[id].buffer.subarray(mandreel_js_mapping_ids[id].offset_read, mandreel_js_mapping_ids[id].offset_read+size);
        heapU8.set(sub_array,ptr);

        mandreel_js_mapping_ids[id].offset_read+=size;

        r_g0 = size;
}

function Mandreel_HttpRequest_BytesAvalable(sp)
{
        var _id = heap32[sp>>2];sp+=4;
        var id = _id-1;


        if (mandreel_js_mapping_ids[id].buffer)
                r_g0 = mandreel_js_mapping_ids[id].buffer.length - mandreel_js_mapping_ids[id].offset_read;
        else
                r_g0 = 0;
}

function Mandreel_HttpRequest_Close(sp)
{
        var _id = heap32[sp>>2];sp+=4;
        var id = _id-1;

        mandreel_js_mapping_ids[id] = null;
        mandreel_js_mapping_ids_free.push(id);
}

function Mandreel_HttpRequest_SetRequestHeader(sp)
{
        var _id = heap32[sp>>2];sp+=4;
        var ptr_a = heap32[sp>>2];sp+=4;
        var ptr_b = heap32[sp>>2];sp+=4;
        var id = _id-1;

        var str_a = get_string_from_ptr(ptr_a);
        var str_b = get_string_from_ptr(ptr_b);

        var my_state = mandreel_js_mapping_ids[id];

        my_state.httpRequest.setRequestHeader(str_a, str_b);
}


var Mandreel_TextureAsync_textures = 0;
var Mandreel_TextureAsync_textures_loaded = 0;

var Mandreel_TextureAsync_PackBufferData = new Array();

function Mandreel_TextureAsync_SetData(sp)
{
        var texture_id = heap32[sp>>2];sp+=4;

        var tex = array_ids_ogl[texture_id];

        if ( mandreelAppPlatform != "canvas" )
        {
                imandreel_gl.texImage2D(imandreel_gl.TEXTURE_2D, 0, imandreel_gl.RGBA, imandreel_gl.RGBA, imandreel_gl.UNSIGNED_BYTE, tex.image);
                tex.image = null;
    }
}

function Mandreel_TextureAsync_CheckPending(sp)
{
        r_g0 = Mandreel_TextureAsync_textures - Mandreel_TextureAsync_textures_loaded;
}

function Mandreel_TextureAsync_GetProperties(sp)
{
        var texture_id = heap32[sp>>2];sp+=4;
        var ptr_width = heap32[sp>>2];sp+=4;
        var ptr_height = heap32[sp>>2];sp+=4;

        var tex = array_ids_ogl[texture_id];

        if (tex == null || tex.mandreel_width == undefined)
                r_g0 = 0;
        else
        {
                heap32[ptr_width>>2] = tex.mandreel_width;
                heap32[ptr_height>>2] = tex.mandreel_height;
                r_g0 = 1;
    }
}

function mandreel_arrayBufferDataUri(offset, size, buffer) {
var bytes = new Uint8Array(buffer,offset,size)
   var ascii = '';
   for (var i=0; i<bytes.length; i++)
     ascii += String.fromCharCode(bytes[i]);
   var base64 = btoa(ascii);

  if (/^\x89PNG/.test(ascii))
    return 'data:image/png;base64,'+base64;
  else
    return 'data:image/jpeg;base64,'+base64;
  }

 function mandreel_texture_async_fix_name(name)
{
        var ascii = '';

        var j;
        var len = name.length;


        for(j = 0; j < len; j++)
        {
                var my_char = name[j];

                if (my_char == '/')
                        my_char = '_';

                ascii+=my_char;

    }

   return ascii;
}



function Mandreel_TextureAsync_Load(sp)
{
        var ptr_name = heap32[sp>>2];sp+=4;
        var texture_id = heap32[sp>>2];sp+=4;

        var name = get_string_from_ptr(ptr_name);

        var nameSrc = name;

        name = name.toLowerCase();


        var full_name   = g_mandreel_working_folder + name;

        var image  = new Image();


        Mandreel_TextureAsync_textures++;

        var imgURL = null;




        image.onerror = function() {
          dump('error loading texture ' + image.src + '\n');
                  Mandreel_TextureAsync_textures_loaded++;
      }
        image.onload = function()
        {
                if (imgURL)
                {
                        var URL = Mandreel_window.URL || Mandreel_window.webkitURL;
                        URL.revokeObjectURL(imgURL);
            }
                if ( mandreelAppPlatform == "canvas" )
                {
                        array_ids_ogl[texture_id] = image;
                        Mandreel_TextureAsync_textures_loaded++;
            }
                else
                {
                        var tex = array_ids_ogl[texture_id];
                        if (tex)
                        {
                                tex.image = image;
                                tex.mandreel_width = image.width;
                                tex.mandreel_height = image.height;
                                Mandreel_TextureAsync_textures_loaded++;

                                var sp = MandreelLockFrame();
                                MandreelInterCallFunction('void',"Mandreel_TextureAsync_Loaded",'int',texture_id,'int',image.width,'int',image.height, sp);
                                MandreelUnlockFrame();
            }
                        else
                        {
                                dump('texture not valid ' + texture_id + ' ' + name + '\n');
                                Mandreel_TextureAsync_textures_loaded++;
                    }
            }


    }

        var new_sp = sp-4096;

        var packfile = new_sp + 2048;
        var offset_ptr = new_sp + 2048+1024;
        var size_ptr = new_sp + 2048+1024+4;

        fill_ptr_from_string(new_sp + 1024,name);

        heap32[(new_sp)>>2] = new_sp + 1024;
        heap32[(new_sp+4)>>2] = offset_ptr;
        heap32[(new_sp+8)>>2] = size_ptr;
        heap32[(new_sp+12)>>2] = packfile;
        iMandreel_TextureAsync_GetPackOffset(new_sp);


        var image_src;

        var image_src_valid = true;

        if (r_g0)
        {
                var packfilename = get_string_from_ptr(packfile);
                image_src = mandreel_arrayBufferDataUri(heap32[offset_ptr>>2],heap32[size_ptr>>2],Mandreel_TextureAsync_PackBufferData[packfilename]);
    }
        else
        {

                if (nameSrc.search('http:') != -1 || nameSrc.search('https:') != -1)
                {
                        image.crossOrigin = 'anonymous'; // no credentials flag. Same as
                        image_src = nameSrc;
            }
                else
                {
                        if (mandreel_is_filesystem())
                        {
                                image_src_valid = false;

                                var new_name = mandreel_texture_async_fix_name(full_name);
                                mandreel_fs_get_url(new_name, function Mandreel_TextureAsync_Load_FS(data) {
                                        if (data)
                                        {
                                                image.src = data;
                                    }
                                        else
                                        {
                                                var packdata_request = new XMLHttpRequest();

                                                packdata_request.open("GET", full_name, true);

                                                if("responseType" in packdata_request)
                                                        packdata_request.responseType="arraybuffer";
                                                else
                                                        packdata_request.overrideMimeType('text/plain; charset=x-user-defined');

                                                packdata_request.onreadystatechange = function()
                                                {
                                                        if (packdata_request.readyState != 4) return;

                                                        if (packdata_request.status == 200)
                                                        {
                                                                var buffer;
                                                                if (packdata_request.responseType=="arraybuffer")
                                                                        buffer=packdata_request.response;
                                                                else if (packdata_request.mozResponseArrayBuffer != null)
                                                                        buffer = packdata_request.mozResponseArrayBuffer;
                                                                else
                                                                        buffer=packdata_request.response;

                                                                mandreel_fs_saveFile(new_name, buffer);

                                                                var uri = mandreel_arrayBufferDataUri(0,buffer.byteLength,buffer);

                                                                image.src = uri;

                                                    }
                                                        else
                                                        {

                                                                Mandreel_TextureAsync_textures_loaded++;

                                                    }
                                            }

                                                packdata_request.send();

                                    }
                            }
                                );
                    }
                        else if (mandreel_is_indexeddb())
                        {
                                image_src_valid = false;

                                var new_name = mandreel_texture_async_fix_name(full_name);
                                mandreel_indexedDB.load(new_name,function Mandreel_TextureAsync_Load_IDB(data) {
                                        if (data)
                                        {
                                                 var URL = Mandreel_window.URL || Mandreel_window.webkitURL;


                                                // Create and revoke ObjectURL
                                                imgURL = URL.createObjectURL(data);

                                                image.src = imgURL;
                                    }
                                        else
                                        {
                                                var packdata_request = new XMLHttpRequest();

                                                packdata_request.open("GET", full_name, true);

                                                 packdata_request.responseType = "blob";



                                                packdata_request.onreadystatechange = function()
                                                {
                                                        if (packdata_request.readyState != 4) return;

                                                        if (packdata_request.status == 200)
                                                        {
                                                                var buffer=packdata_request.response;

                                                                 var URL = Mandreel_window.URL || Mandreel_window.webkitURL;

                                                                // Create and revoke ObjectURL
                                                                imgURL = URL.createObjectURL(buffer);

                                                                image.src = imgURL;

                                                                mandreel_fs_saveFile(new_name, buffer);

                                                    }
                                                        else
                                                        {

                                                                Mandreel_TextureAsync_textures_loaded++;

                                                    }
                                            }

                                                packdata_request.send();
                                    }

                            }
                                );
                    }
                        else
                                image_src = full_name;
            }
    }


        if (image_src_valid)
        {
                setTimeout( function Mandreel_TextureAsync_Load_callback() {
                                                image.src = image_src;
                                    }, 1);
    }
}



function __sandbox_OutputDebugString(sp)
{
        puts(sp);
}




var MANDREELCALLJS_TYPE_RETURN_VOID = 0;
var MANDREELCALLJS_TYPE_INT = 1;
var MANDREELCALLJS_TYPE_FLOAT =  2;
var MANDREELCALLJS_TYPE_STRING =  3;
var MANDREELCALLJS_TYPE_RETURN_INT =  4;
var MANDREELCALLJS_TYPE_RETURN_FLOAT =  5;

function MandreelInterWriteString(ptr, value)
{
        fill_ptr_from_string(ptr,value);
}

function MandreelInterWriteWString(ptr, value)
{
        fill_wptr_from_string(ptr, value);
}

function MandreelInterWriteFloat(ptr, value)
{
        heapFloat[ptr>>2] = value;
}

function MandreelPause()
{
        _imandreel_pause_game = true;
}

function MandreelResume()
{
        _imandreel_pause_game = false;
}


function MandreelLockFrame()
{
        assert(g_mandreel_frame_inframe == false, "calling lockframe during render frame");
        assert(g_mandreel_frame_locked == false, "calling lockframe twice");
        g_mandreel_frame_locked = true;

        return g_stack_pointer+800*1024;
}

function MandreelUnlockFrame()
{
        assert(g_mandreel_frame_inframe == false);
        g_mandreel_frame_locked = false;
}


function MandreelInterWriteInt(ptr, value)
{
        heap32[ptr>>2] = value;
}

function MandreelInterStringFromWPtr(ptr)
{
        return get_string_from_wptr(ptr);
}

function MandreelInterStringFromPtr(ptr)
{
        return get_string_from_ptr(ptr);
}

function mandreel_my_call_external_array(method, params)
{
        var result
        var resultString;
        try
        {
                switch(params.length)
                {
                        case 1:
                                resultString = Mandreel_window[method](params[0]);
                                break;
                        case 2:
                                resultString = Mandreel_window[method](params[0],params[1]);
                                break;
                        case 3:
                                resultString = Mandreel_window[method](params[0],params[1],params[2]);
                                break;
                        case 4:
                                resultString = Mandreel_window[method](params[0],params[1],params[2],params[3]);
                                break;
                        case 5:
                                resultString = Mandreel_window[method](params[0],params[1],params[2],params[3],params[4]);
                                break;
                        case 6:
                                resultString = Mandreel_window[method](params[0],params[1],params[2],params[3],params[4],params[5]);
                                break;
                        case 7:
                                resultString = Mandreel_window[method](params[0],params[1],params[2],params[3],params[4],params[5],params[6]);
                                break;
                        case 8:
                                resultString = Mandreel_window[method](params[0],params[1],params[2],params[3],params[4],params[5],params[6],params[7]);
                                break;
                        case 9:
                                resultString = Mandreel_window[method](params[0],params[1],params[2],params[3],params[4],params[5],params[6],params[7],params[8]);
                                break;
                        case 10:
                                resultString = Mandreel_window[method](params[0],params[1],params[2],params[3],params[4],params[5],params[6],params[7],params[8],params[9]);
                                break;
                        case 11:
                                resultString = Mandreel_window[method](params[0],params[1],params[2],params[3],params[4],params[5],params[6],params[7],params[8],params[9],params[10]);
                                break;
                        case 12:
                                resultString = Mandreel_window[method](params[0],params[1],params[2],params[3],params[4],params[5],params[6],params[7],params[8],params[9],params[10],params[11]);
                                break;
                        case 13:
                                resultString = Mandreel_window[method](params[0],params[1],params[2],params[3],params[4],params[5],params[6],params[7],params[8],params[9],params[10],params[11],params[12]);
                                break;
                        case 14:
                                resultString = Mandreel_window[method](params[0],params[1],params[2],params[3],params[4],params[5],params[6],params[7],params[8],params[9],params[10],params[11],params[12],params[13]);
                                break;
                        case 15:
                                resultString = Mandreel_window[method](params[0],params[1],params[2],params[3],params[4],params[5],params[6],params[7],params[8],params[9],params[10],params[11],params[12],params[13],params[14]);
                                break;
                        default:
                                assert(false);
            }
                result = 0;
    } catch(e) { dump('error calling ' + method + '\n'); dump(e); result = 1;}

        return [result,resultString];
}


function Mandreel_InterJS_Call(sp)
{
        var new_sp = sp;
        var method_ptr = heap32[sp>>2];sp+=4;
        var method = get_string_from_ptr(method_ptr);

        var params = new Array();


        params.push(new_sp);

        var var_int;
        var var_string;
        var var_double;

        var return_type;
        var return_ptr;
        while (true)
        {
                var my_type = heap32[sp>>2];sp+=4;


                if (my_type == MANDREELCALLJS_TYPE_RETURN_VOID)
                {
                        return_type = my_type;
                        break;
            }
                else if (my_type == MANDREELCALLJS_TYPE_INT)
                {
                        var_int = heap32[sp>>2];

                        params.push(var_int);
                        sp+=4;
            }
                else if (my_type == MANDREELCALLJS_TYPE_FLOAT)
                {
                        sp = (sp+7) & ~7;

                        var_double = llvm_readDouble(sp);

                        params.push(var_double);
                        sp+=8;
            }
                else if (my_type == MANDREELCALLJS_TYPE_STRING)
                {
                        var_int = heap32[sp>>2];
                        var_string = get_string_from_ptr(var_int);

                        params.push(var_string);
                        sp+=4;
            }
                else if (my_type == MANDREELCALLJS_TYPE_RETURN_INT)
                {
                        return_type = my_type;
                        return_ptr = heap32[sp>>2];
                        break;
            }
                else if (my_type == MANDREELCALLJS_TYPE_RETURN_FLOAT)
                {
                        return_type = my_type;
                        return_ptr = heap32[sp>>2];
                        break;
            }
                else
                {
                        assert(false, "invalid arguments calling Mandreel_InterJS_Call");
            }
    }

        var result = mandreel_my_call_external_array(method,params);

        r_g0 = result[0];


        if (r_g0 == 0)
        {
                if (return_type == MANDREELCALLJS_TYPE_RETURN_INT)
                {
                        heap32[return_ptr>>2] = result[1];
            }
                else if (return_type == MANDREELCALLJS_TYPE_RETURN_FLOAT)
                {
                        heapFloat[return_ptr>>2] = result[1];
            }

    }
}

function iMandreelRegisterExternalCallback()
{
}

function __mandreel_internal_CreateWindow()
{
}

var __mandreel_async_calls_mandreel = [];
var __mandreel_async_calls_js = [];


function Mandreel_InterJS_AsyncCall(sp)
{
        var method_ptr = heap32[sp>>2];sp+=4;
        var _func_name = get_string_from_ptr(method_ptr);
        var param_ptr = heap32[sp>>2];sp+=4;
        var _param = get_string_from_ptr(param_ptr);

        __mandreel_async_calls_js.push({func_name:_func_name,param:_param});
}




function MandreelInterCallFunctionAsync(_func_name, _param)
{
        __mandreel_async_calls_mandreel.push({func_name:_func_name,param:_param});


}

function __mandreel_process_async_calls()
{
        if (__mandreel_async_calls_mandreel.length)
        {
                var temp_list = __mandreel_async_calls_mandreel.slice(0);

                __mandreel_async_calls_mandreel = [];

                for (var i=0;i<temp_list.length;++i)
                {
                        var param = temp_list[i].param;
                        var func_name = temp_list[i].func_name;

                        var size = ((param.length + 1)+7)&0xFFFFFFF8;

                        var sp = g_stack_pointer+800*1024;

                        var str_ptr = sp - size;
                        fill_ptr_from_string(str_ptr,param);

                        sp = str_ptr - 4;
                        heap32[sp>>2] = str_ptr;

                        Mandreel_window[func_name](sp);
            }
    }

        if (__mandreel_async_calls_js.length)
        {
                var temp_list = __mandreel_async_calls_js.slice(0);

                __mandreel_async_calls_js = [];

                for (var i=0;i<temp_list.length;++i)
                {
                        var param = temp_list[i].param;
                        var func_name = temp_list[i].func_name;

                        Mandreel_window[func_name](param);

            }
    }
}

function mandreel_internal_isCanvas(sp)
{
        if ( mandreelAppPlatform == "canvas" )
                r_g0 = 1;
        else
                r_g0 = 0;
}

function Mandreel_Device_SetFullScreen(sp)
{
        var enable = heap32[sp>>2];sp+=4;
        mandreelAppFullscreen(enable);
}

var array_ids_ogl = [];

var max_ogl_id = 8192;

var array_ids_ogl_enable = [];
var g_current_program_id = 0;


var uniformArrays2 = [];
var uniformArrays3 = [];
var uniformArrays4 = [];
var uniformArraysCreated = 0;
var mandreel_draw_enable = true;


if (typeof imandreel_gl=="undefined")
{
        alert('using old template, update code');
}

function myglCreateUniformArrays()
{
        if ( uniformArraysCreated == 0 )
        {
                for(var i=0; i<256;i++ )
                {
                        uniformArrays2[i] = new Float32Array(i*2);
                        uniformArrays3[i] = new Float32Array(i*3);
                        uniformArrays4[i] = new Float32Array(i*4);
            }
                uniformArraysCreated = 1;
    }
}

var my_super_id = 1;
function myglNewSlot()
{
        //var id = array_ids_ogl_enable.pop();
        var id = my_super_id;
        my_super_id++;
        return id;
}

function myglFreeSlot(id)
{
        //array_ids_ogl_enable.push(id);
}


function myglCreateProgram(sp)
{
        var id = myglNewSlot();
        var program = imandreel_gl.createProgram();

        program.uniform_locations_current_id = 0;
        program.array_uniform_locations = [];

        array_ids_ogl[id] = program;


        r_g0 = id;
}

function myglCreateShader(sp)
{
        var type = heap32[sp>>2];sp+=4;
        var id = myglNewSlot();

        array_ids_ogl[id] = imandreel_gl.createShader(type);

        r_g0 = id;
}

function myglAttachShader(sp)
{
        var program_id = heap32[sp>>2];sp+=4;
        var shader_id = heap32[sp>>2];sp+=4;

        imandreel_gl.attachShader(array_ids_ogl[program_id], array_ids_ogl[shader_id]);
}

function myglBindAttribLocation(sp)
{
        var program_id = heap32[sp>>2];sp+=4;
        var index = heap32[sp>>2];sp+=4;
        var ptr_string = heap32[sp>>2];sp+=4;

        var string = get_string_from_ptr(ptr_string);

        imandreel_gl.bindAttribLocation(array_ids_ogl[program_id], index, string);
}

function myglLinkProgram(sp)
{
        var program_id = heap32[sp>>2];sp+=4;

        imandreel_gl.linkProgram(array_ids_ogl[program_id]);
}

function myglShaderSource(sp)
{
        var id = heap32[sp>>2];sp+=4;
        var ptr_string = heap32[sp>>2];sp+=4;

        var shader = array_ids_ogl[id];

        var shader_code = get_string_from_ptr(ptr_string);

        //dump(shader_code);


        imandreel_gl.shaderSource(shader, shader_code);
}


function myglDrawArrays(sp)
{
        var mode = heap32[sp>>2];sp+=4;
        var first = heap32[sp>>2];sp+=4;
        var count = heap32[sp>>2];sp+=4;

        if (mandreel_draw_enable)
                imandreel_gl.drawArrays(mode, first, count);


        //dump('draw arrays ' + mode + ' ' + first + ' ' + count + '\n');
 }

function myglDrawElements(sp)
{
        var mode = heapU32[sp>>2]; sp+=4;
        var count = heapU32[sp>>2]; sp+=4;
        var type = heapU32[sp>>2]; sp+=4;
        var offset = heapU32[sp>>2]; sp+=4;


        if (mandreel_draw_enable)
                imandreel_gl.drawElements(mode, count, type, offset);



}

function myglCreateTexture(sp)
{
        var id = myglNewSlot();
        array_ids_ogl[id] = imandreel_gl.createTexture();

        r_g0 = id;
}

function myglCreateRenderBuffer(sp) {
        var id = myglNewSlot();
    array_ids_ogl[id] = imandreel_gl.createRenderbuffer();

    r_g0 = id;
}

function myglCreateFrameBuffer(sp) {
        var id = myglNewSlot();
    array_ids_ogl[id] = imandreel_gl.createFramebuffer();

    r_g0 = id;
}

function myglBindFramebuffer(sp)
{
    var target = heap32[sp >> 2]; sp += 4;
    var framebuffer_id = heap32[sp >> 2]; sp += 4;

        if (framebuffer_id != 0)
        {
                var framebuffer = array_ids_ogl[framebuffer_id];
                imandreel_gl.bindFramebuffer(target,framebuffer);
    }
        else
                imandreel_gl.bindFramebuffer(target,null);

}

function myglBindRenderbuffer(sp)
{
    var target = heap32[sp >> 2]; sp += 4;
    var renderbuffer_id = heap32[sp >> 2]; sp += 4;

    var renderbuffer = array_ids_ogl[renderbuffer_id];

    imandreel_gl.bindRenderbuffer(target,renderbuffer);

}


function myglRenderbufferStorage(sp) {
    var target = heap32[sp >> 2]; sp += 4;
    var internalformat = heap32[sp >> 2]; sp += 4;
    var witdth = heap32[sp >> 2]; sp += 4;
    var height = heap32[sp >> 2]; sp += 4;

    imandreel_gl.renderbufferStorage(target, internalformat, witdth, height);

}

function myglFramebufferRenderbuffer (sp)
{
  var target = heap32[sp>>2];sp+=4;
  var attachment = heap32[sp>>2];sp+=4;
  var renderbuffertarget = heap32[sp>>2];sp+=4;
  var renderbuffer_id = heap32[sp>>2];sp+=4;

    var renderbuffer = array_ids_ogl[renderbuffer_id];

    imandreel_gl.framebufferRenderbuffer(target, attachment, renderbuffertarget, renderbuffer);

}

function myglFramebufferTexture2D (sp)
{
  var target = heap32[sp>>2];sp+=4;
  var attachment = heap32[sp>>2];sp+=4;
  var textarget = heap32[sp>>2];sp+=4;
  var texture_id = heap32[sp>>2];sp+=4;
  var level = heap32[sp>>2];sp+=4;

  var texture = array_ids_ogl[texture_id];

  imandreel_gl.framebufferTexture2D(target, attachment, textarget, texture, level);


}

function myglTexImage2D(sp)
 {
  var target = heap32[sp>>2];sp+=4;
  var level = heap32[sp>>2];sp+=4;
  var internalFormat = heap32[sp>>2];sp+=4;
  var width = heap32[sp>>2];sp+=4;
  var height = heap32[sp>>2];sp+=4;
  var border = heap32[sp>>2];sp+=4;
  var format = heap32[sp>>2];sp+=4;
  var type = heap32[sp>>2];sp+=4;
  var data = heap32[sp>>2];sp+=4;

  if (level>0 && target==imandreel_gl.TEXTURE_2D)
        return;

   if (data == 0)
  {
        //imandreel_gl.texImage2D(target, level, internalFormat, width, height, border, format, type, null);
        var buffer;
        var bufferView;

        if (type == imandreel_gl.UNSIGNED_SHORT_5_6_5 || type == imandreel_gl.UNSIGNED_SHORT_4_4_4_4 || type == imandreel_gl.UNSIGNED_SHORT_5_5_5_1)
    {
                buffer = new ArrayBuffer(width*height*2);
                bufferView = new Uint16Array(buffer);
    }
        else
        {
                var size;
                if (format == imandreel_gl.LUMINANCE)
                        size = width*height;
                else if (format == imandreel_gl.RGB)
                        size = width*height*3;
                else if (format == imandreel_gl.RGBA)
                        size = width*height*4;
                else if (format == imandreel_gl.ALPHA)
                        size = width*height;
                else if (format == imandreel_gl.LUMINANCE_ALPHA)
                        size = width*height*2;

                buffer = new ArrayBuffer(size);
                bufferView = new Uint8Array(buffer);
    }

        imandreel_gl.texImage2D(target, level, internalFormat, width, height, border, format, type, bufferView);
        return;
  }


        var bufferView;
   if (type == imandreel_gl.UNSIGNED_SHORT_5_6_5 || type == imandreel_gl.UNSIGNED_SHORT_4_4_4_4 || type == imandreel_gl.UNSIGNED_SHORT_5_5_5_1)
   {
                bufferView = new Uint16Array(heap,data,width*height);
    }
        else if (type == imandreel_gl.UNSIGNED_BYTE)
        {
                if (format == imandreel_gl.LUMINANCE)
                        bufferView = new Uint8Array(heap,data,width*height);
                else if (format == imandreel_gl.RGB)
                        bufferView = new Uint8Array(heap,data,width*height*3);
                else if (format == imandreel_gl.RGBA)
                        bufferView = new Uint8Array(heap,data,width*height*4);
                else if (format == imandreel_gl.ALPHA)
                        bufferView = new Uint8Array(heap,data,width*height);
                else if (format == imandreel_gl.LUMINANCE_ALPHA)
                        bufferView = new Uint8Array(heap,data,width*height*2);
                else
                {
                        dump('format unknown ' + format + '\n');
                        assert(false);
            }
    }
        else
        {
        dump('type unknown ' + type + '\n');
                assert(false);
    }

  imandreel_gl.texImage2D(target, level, internalFormat, width, height, border, format, type, bufferView);
  if ((width&(width-1))==0 && (height&(height-1))==0)
  {
    if (target==imandreel_gl.TEXTURE_2D)
        imandreel_gl.generateMipmap(target);
}
 }
  function myglStencilFunc(sp)
  {
  var func = heap32[sp>>2];sp+=4;
  var ref = heap32[sp>>2];sp+=4;
  var mask = heap32[sp>>2];sp+=4;

  imandreel_gl.stencilFunc(func, ref, mask);
  }

  function myglStencilFuncSeparate(sp)
  {
  var face = heap32[sp>>2];sp+=4;
  var func = heap32[sp>>2];sp+=4;
  var ref = heap32[sp>>2];sp+=4;
  var mask = heap32[sp>>2];sp+=4;

  imandreel_gl.stencilFuncSeparate(face,func,ref,mask);
  }

  function myglStencilMaskSeparate(sp)
  {
  var face = heap32[sp>>2];sp+=4;
   var mask = heap32[sp>>2];sp+=4;

   imandreel_gl.stencilMaskSeparate(face,mask);
  }

  function myglStencilMask(sp)
  {
   var mask = heap32[sp>>2];sp+=4;

   imandreel_gl.stencilMask(mask);
  }
  function myglStencilOp (sp)
  {
   var fail = heap32[sp>>2];sp+=4;
  var zfail = heap32[sp>>2];sp+=4;
   var zpass = heap32[sp>>2];sp+=4;

  imandreel_gl.stencilOp(fail, zfail, zpass);
  }

  function myglStencilOpSeparate (sp)
  {
  var face = heap32[sp>>2];sp+=4;
   var fail = heap32[sp>>2];sp+=4;
  var zfail = heap32[sp>>2];sp+=4;
   var zpass = heap32[sp>>2];sp+=4;

  imandreel_gl.stencilOpSeparate(face, fail, zfail, zpass);
  }

 function myglTexSubImage2D(sp)
 {
  var target = heap32[sp>>2];sp+=4;
  var level = heap32[sp>>2];sp+=4;
  var xoffset = heap32[sp>>2];sp+=4;
  var yoffset = heap32[sp>>2];sp+=4;
  var width = heap32[sp>>2];sp+=4;
  var height = heap32[sp>>2];sp+=4;
  var format = heap32[sp>>2];sp+=4;
  var type = heap32[sp>>2];sp+=4;
  var data = heap32[sp>>2];sp+=4;



        var bufferView;
   if (type == imandreel_gl.UNSIGNED_SHORT_5_6_5 || type == imandreel_gl.UNSIGNED_SHORT_4_4_4_4 || type == imandreel_gl.UNSIGNED_SHORT_5_5_5_1)
   {
                bufferView = new Uint16Array(heap,data,width*height);
    }
        else if (type == imandreel_gl.UNSIGNED_BYTE)
        {
                if (format == imandreel_gl.LUMINANCE)
                        bufferView = new Uint8Array(heap,data,width*height);
                else if (format == imandreel_gl.RGB)
                        bufferView = new Uint8Array(heap,data,width*height*3);
                else if (format == imandreel_gl.RGBA)
                        bufferView = new Uint8Array(heap,data,width*height*4);
                else if (format == imandreel_gl.ALPHA)
                        bufferView = new Uint8Array(heap,data,width*height);
    }

  imandreel_gl.texSubImage2D(target, level, xoffset, yoffset, width, height, format, type, bufferView);
 }


  function myglCreateBuffer(sp)
{
        var id = myglNewSlot();
        array_ids_ogl[id] = imandreel_gl.createBuffer();

        r_g0 = id;
}

var glBufferDataArray = [];

function myglBufferData(sp)
{
        var target = heapU32[sp>>2]; sp+=4;
        var size = heapU32[sp>>2]; sp+=4;
        var data = heapU32[sp>>2]; sp+=4;
        var usage = heapU32[sp>>2]; sp+=4;

        if (data == 0)
                imandreel_gl.bufferData(target, size, usage);
        else
        {
                if (usage == imandreel_gl.STATIC_DRAW || true)
                {
                        var buffer_data = new Int8Array(heap, data, size);
                        imandreel_gl.bufferData(target, buffer_data, usage);
            }
                else
                {
                        var new_size = size/4;
                        var buffer_data = glBufferDataArray[new_size];

                        if (buffer_data == null)
                        {
                                buffer_data =  new Int32Array(new_size);
                                glBufferDataArray[new_size] = buffer_data;
                    }

                        var new_data = data>>2;

                        for ( var i = 0 ; i < new_size ; ++i )
                        {
                                buffer_data[i] = heap32[new_data+i];
                    }

                        imandreel_gl.bufferData(target, buffer_data, usage);
            }
    }
}

function myglBufferSubData(sp)
{
        var target = heapU32[sp>>2]; sp+=4;
        var offset = heapU32[sp>>2]; sp+=4;
        var size = heapU32[sp>>2]; sp+=4;
        var data = heapU32[sp>>2]; sp+=4;


        var buffer_data = new Int8Array(heap, data, size);
        imandreel_gl.bufferSubData(target, offset, buffer_data);

//      dump('buffer sub data ' + offset + ' ' + size + ' ' + data + '\n')

}


function myglBindBuffer(sp)
{
        var target = heapU32[sp>>2]; sp+=4;
        var id = heapU32[sp>>2]; sp+=4;

        imandreel_gl.bindBuffer(target, array_ids_ogl[id]);
}


function myglUseProgram(sp)
{
        var program_id = heap32[sp>>2];sp+=4;

        g_current_program_id = program_id;

        imandreel_gl.useProgram(array_ids_ogl[program_id]);

}

function myglDisableVertexAttribArray(sp)
{
        var idx = heapU32[sp>>2];sp+=4;
        imandreel_gl.disableVertexAttribArray(idx);
}
function myglEnableVertexAttribArray(sp)
{
        var idx = heapU32[sp>>2];sp+=4;
        imandreel_gl.enableVertexAttribArray(idx);
}

function myglVertexAttribPointer(sp)
{
        var idx = heapU32[sp>>2];sp+=4;
        var size = heapU32[sp>>2];sp+=4;
        var type = heapU32[sp>>2];sp+=4;
        var normalized = heapU32[sp>>2];sp+=4;
        var stride = heapU32[sp>>2];sp+=4;
        var ptr = heapU32[sp>>2];sp+=4;

        //dump(idx + ' ' + size + ' ' + type + ' ' + normalized + ' ' + stride + ' ' + ptr + '\n');


        imandreel_gl.vertexAttribPointer(idx, size, type, normalized, stride, ptr);
}

function myglPolygonOffset(sp)
{
        var factor = heapFloat[sp>>2]; sp+=4;
        var units = heapFloat[sp>>2]; sp+=4;
        imandreel_gl.polygonOffset(factor, units);
}

function myglEnable(sp)
 {
        var value = heap32[sp>>2];sp+=4;

        imandreel_gl.enable(value);
  }

function myglDisable(sp)
 {
        var value = heap32[sp>>2];sp+=4;

        imandreel_gl.disable(value);
  }

  function myglDepthFunc(sp)
  {
        var func = heapU32[sp>>2];sp+=4;

        imandreel_gl.depthFunc(func);

  }

  function myglGenerateMipmap(sp)
  {
        var texture_type = heap32[sp>>2];sp+=4;
        imandreel_gl.generateMipmap(texture_type);
  }

  function myglPixelStorei (sp)
  {
        var pname = heap32[sp>>2];sp+=4;
        var param = heap32[sp>>2];sp+=4;
        imandreel_gl.pixelStorei(pname,param);
  }


  function myglBindTexture(sp)
  {
  var texture_type = heap32[sp>>2];sp+=4;
  var texture_id = heap32[sp>>2];sp+=4;

if (texture_id == 0)
{
        imandreel_gl.bindTexture(texture_type, null);
}
else
{
        var tex = array_ids_ogl[texture_id];
        imandreel_gl.bindTexture(texture_type, tex);
    }

  }

  function myglActiveTexture(sp)
{
        var param = heapU32[sp>>2];sp+=4;
        imandreel_gl.activeTexture(param);
}

function myglCompileShader(sp)
{
        var id = heap32[sp>>2];sp+=4;

        var shader = array_ids_ogl[id];

        imandreel_gl.compileShader(shader);
}

function myglGetUniformLocation(sp)
{
        var program_id = heap32[sp>>2];sp+=4;
        var ptr_string = heap32[sp>>2];sp+=4;

        var string = get_string_from_ptr(ptr_string);
        var program = array_ids_ogl[program_id];
        var result = imandreel_gl.getUniformLocation(program, string);

        if (result != null)
        {
                program.array_uniform_locations[program.uniform_locations_current_id] = result;
                r_g0 = program.uniform_locations_current_id;
                program.uniform_locations_current_id++;
    }
        else
                r_g0 = -1;
}

function myglIsEnabled(sp)
{
        var cap = heap32[sp>>2];sp+=4;

        r_g0 = imandreel_gl.isEnabled(cap);
}


function myglUniform1i(sp)
{
        var index = heap32[sp>>2];sp+=4;
        var value = heap32[sp>>2];sp+=4;

        var program = array_ids_ogl[g_current_program_id];

        var uniform_value = program.array_uniform_locations[index];

        imandreel_gl.uniform1i(uniform_value, value);
}

function myglUniform2i(sp)
{
        var index = heap32[sp>>2];sp+=4;
        var x = heap32[sp>>2];sp+=4;
        var y = heap32[sp>>2];sp+=4;

        var program = array_ids_ogl[g_current_program_id];

        var uniform_value = program.array_uniform_locations[index];

        imandreel_gl.uniform2i(uniform_value, x,y);
}

function myglUniform3i(sp)
{
        var index = heap32[sp>>2];sp+=4;
        var x = heap32[sp>>2];sp+=4;
        var y = heap32[sp>>2];sp+=4;
        var z = heap32[sp>>2];sp+=4;

        var program = array_ids_ogl[g_current_program_id];

        var uniform_value = program.array_uniform_locations[index];

        imandreel_gl.uniform3i(uniform_value, x,y,z);
}

function myglUniform4i(sp)
{
        var index = heap32[sp>>2];sp+=4;
        var x = heap32[sp>>2];sp+=4;
        var y = heap32[sp>>2];sp+=4;
        var z = heap32[sp>>2];sp+=4;
        var w = heap32[sp>>2];sp+=4;

        var program = array_ids_ogl[g_current_program_id];

        var uniform_value = program.array_uniform_locations[index];

        imandreel_gl.uniform4i(uniform_value, x,y,z,w);
}

function myglUniform1f(sp)
{
        var index = heap32[sp>>2];sp+=4;
        var value = heapFloat[sp>>2];sp+=4;

        var program = array_ids_ogl[g_current_program_id];

        var uniform_value = program.array_uniform_locations[index];

        imandreel_gl.uniform1f(uniform_value, value);
}

function myglUniform3f(sp)
{
        var index = heap32[sp>>2];sp+=4;
        var x = heapFloat[sp>>2];sp+=4;
        var y = heapFloat[sp>>2];sp+=4;
        var z = heapFloat[sp>>2];sp+=4;

        var program = array_ids_ogl[g_current_program_id];

        var uniform_value = program.array_uniform_locations[index];

        imandreel_gl.uniform3f(uniform_value, x,y,z);
}

function myglUniform2f(sp)
{
        var index = heap32[sp>>2];sp+=4;
        var x = heapFloat[sp>>2];sp+=4;
        var y = heapFloat[sp>>2];sp+=4;

        var program = array_ids_ogl[g_current_program_id];

        var uniform_value = program.array_uniform_locations[index];

        imandreel_gl.uniform2f(uniform_value, x,y);
}


function myglUniform4f(sp)
{
        var index = heap32[sp>>2];sp+=4;
        var x = heapFloat[sp>>2];sp+=4;
        var y = heapFloat[sp>>2];sp+=4;
        var z = heapFloat[sp>>2];sp+=4;
        var w = heapFloat[sp>>2];sp+=4;

        var program = array_ids_ogl[g_current_program_id];
        var uniform_value = program.array_uniform_locations[index];

        imandreel_gl.uniform4f(uniform_value, x,y,z,w);
}

function myglUniform1fv(sp)
{
        var index = heap32[sp>>2];sp+=4;
        var count = heap32[sp>>2];sp+=4;
        var data = heap32[sp>>2];sp+=4;

        var new_data = data>>2;
        var new_count = count;
        var bufferView = new Float32Array(new_count);

        for ( var i = 0 ; i < new_count ; ++i )
        {
                bufferView[i] = heapFloat[new_data+i];
    }

        var program = array_ids_ogl[g_current_program_id];
        var uniform_value = program.array_uniform_locations[index];
        imandreel_gl.uniform1fv(uniform_value, bufferView);


}

function myglUniform1iv(sp)
{
        var index = heap32[sp>>2];sp+=4;
        var count = heap32[sp>>2];sp+=4;
        var data = heap32[sp>>2];sp+=4;

        var new_data = data>>2;
        var new_count = count;
        var bufferView = new Int32Array(new_count);

        for ( var i = 0 ; i < new_count ; ++i )
        {
                bufferView[i] = heap32[new_data+i];
    }

        var program = array_ids_ogl[g_current_program_id];
        var uniform_value = program.array_uniform_locations[index];
        imandreel_gl.uniform1iv(uniform_value, bufferView);
}

function myglUniform2iv(sp)
{
        var index = heap32[sp>>2];sp+=4;
        var count = heap32[sp>>2];sp+=4;
        var data = heap32[sp>>2];sp+=4;

        var new_data = data>>2;
        var new_count = count*2;
        var bufferView = new Int32Array(new_count);

        for ( var i = 0 ; i < new_count ; ++i )
        {
                bufferView[i] = heap32[new_data+i];
    }

        var program = array_ids_ogl[g_current_program_id];
        var uniform_value = program.array_uniform_locations[index];
        imandreel_gl.uniform2iv(uniform_value, bufferView);
}

function myglUniform3iv(sp)
{
        var index = heap32[sp>>2];sp+=4;
        var count = heap32[sp>>2];sp+=4;
        var data = heap32[sp>>2];sp+=4;

        var new_data = data>>2;
        var new_count = count*3;
        var bufferView = new Int32Array(new_count);

        for ( var i = 0 ; i < new_count ; ++i )
        {
                bufferView[i] = heap32[new_data+i];
    }

        var program = array_ids_ogl[g_current_program_id];
        var uniform_value = program.array_uniform_locations[index];
        imandreel_gl.uniform3iv(uniform_value, bufferView);
}

function myglUniform4iv(sp)
{
        var index = heap32[sp>>2];sp+=4;
        var count = heap32[sp>>2];sp+=4;
        var data = heap32[sp>>2];sp+=4;

        var new_data = data>>2;
        var new_count = count*4;
        var bufferView = new Int32Array(new_count);

        for ( var i = 0 ; i < new_count ; ++i )
        {
                bufferView[i] = heap32[new_data+i];
    }

        var program = array_ids_ogl[g_current_program_id];
        var uniform_value = program.array_uniform_locations[index];
        imandreel_gl.uniform4iv(uniform_value, bufferView);
}



function myglUniform3fv(sp)
{
        myglCreateUniformArrays();

        var index = heap32[sp>>2];sp+=4;
        var count = heap32[sp>>2];sp+=4;
        var data = heap32[sp>>2];sp+=4;

        var new_data = data>>2;
        var new_count = count*3;
        var bufferView = uniformArrays3[count];

        for ( var i = 0 ; i < new_count ; ++i )
        {
                bufferView[i] = heapFloat[new_data+i];
    }

        var program = array_ids_ogl[g_current_program_id];
        var uniform_value = program.array_uniform_locations[index];
        imandreel_gl.uniform3fv(uniform_value, bufferView);
}

function myglUniform2fv(sp)
{
        myglCreateUniformArrays();

    var index = heap32[sp >> 2]; sp += 4;
    var count = heap32[sp >> 2]; sp += 4;
    var data = heap32[sp >> 2]; sp += 4;

        var new_data = data>>2;
        var new_count = count*2;
        var bufferView = uniformArrays2[count];

        for ( var i = 0 ; i < new_count ; ++i )
        {
                bufferView[i] = heapFloat[new_data+i];
    }


        var program = array_ids_ogl[g_current_program_id];
    var uniform_value = program.array_uniform_locations[index];
    imandreel_gl.uniform2fv(uniform_value, bufferView);
}


function myglUniform4fv(sp)
{
        myglCreateUniformArrays();

        var index = heap32[sp>>2];sp+=4;
        var count = heap32[sp>>2];sp+=4;
        var data = heap32[sp>>2];sp+=4;


        var new_data = data>>2;
        var new_count = count*4;
        var bufferView = uniformArrays4[count];

        for ( var i = 0 ; i < new_count ; ++i )
        {
                bufferView[i] = heapFloat[new_data+i];
    }


        var program = array_ids_ogl[g_current_program_id];
        var uniform_value = program.array_uniform_locations[index];
        imandreel_gl.uniform4fv(uniform_value, bufferView);
}


function myglUniformMatrix4fv(sp)
{
        myglCreateUniformArrays();

        var index = heap32[sp>>2];sp+=4;
        var count = heap32[sp>>2];sp+=4;
        var transpose = heap32[sp>>2];sp+=4;
        var ptr = heap32[sp>>2];sp+=4;

        var program = array_ids_ogl[g_current_program_id];
        var uniform_value = program.array_uniform_locations[index];

        //var buffer_data = new Float32Array(heap, ptr, count*16);
        for ( var i = 0 ; i < count*16 ; ++i )
        {
                uniformArrays4[count*4][i] = heapFloat[(ptr>>2)+i];
    }

        //imandreel_gl.uniformMatrix4fv(uniform_value, transpose, buffer_data);
        //imandreel_gl.uniformMatrix4fv(uniform_value, transpose, heapFloat.subarray(ptr/4,(ptr/4)+(count*16)));
        imandreel_gl.uniformMatrix4fv(uniform_value, transpose, uniformArrays4[count*4]);
}

function myglUniformMatrix3fv(sp)
{
        myglCreateUniformArrays();

        var index = heap32[sp>>2];sp+=4;
        var count = heap32[sp>>2];sp+=4;
        var transpose = heap32[sp>>2];sp+=4;
        var ptr = heap32[sp>>2];sp+=4;

        var program = array_ids_ogl[g_current_program_id];
        var uniform_value = program.array_uniform_locations[index];

        //var buffer_data = new Float32Array(heap, ptr, count*9);
        for ( var i = 0 ; i < count*9 ; ++i )
        {
                uniformArrays3[count*3][i] = heapFloat[(ptr>>2)+i];
    }

        //imandreel_gl.uniformMatrix3fv(uniform_value, transpose, buffer_data);
        //imandreel_gl.uniformMatrix3fv(uniform_value, transpose, heapFloat.subarray(ptr/4,(ptr/4)+(count*9)));
        imandreel_gl.uniformMatrix3fv(uniform_value, transpose, uniformArrays3[count*3]);
}



function myglValidateProgram(sp)
{
        var program_id = heap32[sp>>2];sp+=4;

        imandreel_gl.validateProgram(array_ids_ogl[program_id]);
}

function myglGetAttribLocation(sp)
{
        var program_id = heap32[sp>>2];sp+=4;
        var ptr_string = heap32[sp>>2];sp+=4;

        var string = get_string_from_ptr(ptr_string);
        var result = imandreel_gl.getAttribLocation(array_ids_ogl[program_id], string);

        r_g0 = result;
}

function myglGetProgramInfoLogLength(sp)
{
        var program_id = heap32[sp>>2];sp+=4;

        var info_log = imandreel_gl.getProgramInfoLog(array_ids_ogl[program_id]);

        if (info_log)
                r_g0 = info_log.length+1;
        else
                r_g0 = 0;
}


function myglGetProgramInfoLog(sp)
{
        var program_id = heap32[sp>>2];sp+=4;
        var ptr_string = heap32[sp>>2];sp+=4;

        var info_log = imandreel_gl.getProgramInfoLog(array_ids_ogl[program_id]);

        fill_ptr_from_string(ptr_string, info_log);
}

function myglGetShaderInfoLogLength(sp)
{
        var program_id = heap32[sp>>2];sp+=4;

        var info_log = imandreel_gl.getShaderInfoLog(array_ids_ogl[program_id]);

        if (info_log)
                r_g0 = info_log.length+1;
        else
                r_g0 = 0;
}

function myglGetShaderInfoLog(sp)
{
        var program_id = heap32[sp>>2];sp+=4;
        var ptr_string = heap32[sp>>2];sp+=4;

        var info_log = imandreel_gl.getShaderInfoLog(array_ids_ogl[program_id]);

        fill_ptr_from_string(ptr_string, info_log);
}

function myglViewport(sp) {
    var x = heap32[sp >> 2]; sp += 4;
    var y = heap32[sp >> 2]; sp += 4;
    var width = heap32[sp >> 2]; sp += 4;
    var height = heap32[sp >> 2]; sp += 4;

    imandreel_gl.viewport(x,y,width,height);

}

function myglScissor(sp)
{
    var x = heap32[sp >> 2]; sp += 4;
    var y = heap32[sp >> 2]; sp += 4;
    var width = heap32[sp >> 2]; sp += 4;
    var height = heap32[sp >> 2]; sp += 4;

    imandreel_gl.scissor(x,y,width,height);
}



function myglClearColor(sp)
{
  var r = heapFloat[sp>>2];sp+=4;
  var g = heapFloat[sp>>2];sp+=4;
  var b = heapFloat[sp>>2];sp+=4;
  var a = heapFloat[sp>>2];sp+=4;

  imandreel_gl.clearColor(r,g,b,a);


}

function myglClearStencil(sp)
{
        var stencil = heap32[sp>>2];sp+=4;
        imandreel_gl.clearStencil(stencil);
}


function myglClearDepthf(sp)
{
        var depth = heapFloat[sp>>2];sp+=4;
        imandreel_gl.clearDepth(depth);
}

function myglClear(sp)
  {
  var mask = heap32[sp>>2];sp+=4;


  //dump('clear ' + mask + '\n');
        if (mandreel_draw_enable)
                imandreel_gl.clear(mask);
  }

  function myglGetError(sp)
  {
//      r_g0 = imandreel_gl.getError();
        r_g0 = 0;
  }

  function myglGetProgramParameter(sp)
  {
        var program_id = heap32[sp>>2];sp+=4;
        var pname = heap32[sp>>2];sp+=4;

        r_g0 = imandreel_gl.getProgramParameter(array_ids_ogl[program_id], pname);
  }

  function myglGetActiveAttrib (sp)
  {
        var program_id = heap32[sp>>2];sp+=4;
        var index = heap32[sp>>2];sp+=4;
        var ptr = heap32[sp>>2];sp+=4;

         var result = imandreel_gl.getActiveAttrib(array_ids_ogl[program_id], index);

         if (result != null)
         {
                heap32[(ptr)>>2] = result.size;
                heap32[(ptr+4)>>2] = result.type;
                fill_ptr_from_string(ptr+8, result.name);
                r_g0 = 0;
    }
        else
           r_g0 = 1;
  }

    function myglGetActiveUniform (sp)
  {
        var program_id = heap32[sp>>2];sp+=4;
        var index = heap32[sp>>2];sp+=4;
        var ptr = heap32[sp>>2];sp+=4;

         var result = imandreel_gl.getActiveUniform(array_ids_ogl[program_id], index);

         if (result != null)
         {
                heap32[(ptr)>>2] = result.size;
                heap32[(ptr+4)>>2] = result.type;
                fill_ptr_from_string(ptr+8, result.name);
                r_g0 = 0;
    }
        else
           r_g0 = 1;
  }

  function myglTexParameterf (sp)
  {
        var target = heap32[sp>>2];sp+=4;
        var pname = heap32[sp>>2];sp+=4;
        var value = heapFloat[sp>>2];sp+=4;

        imandreel_gl.texParameterf(target,pname,value);
}

function myglTexParameteri (sp)
  {
        var target = heap32[sp>>2];sp+=4;
        var pname = heap32[sp>>2];sp+=4;
        var value = heap32[sp>>2];sp+=4;

        imandreel_gl.texParameteri(target,pname,value);
}

function myglCullFace (sp)
 {
        var mode = heap32[sp>>2];sp+=4;
        imandreel_gl.cullFace(mode);
 }

 function myglDepthMask (sp)
 {
        var flag = heap32[sp>>2];sp+=4;
        imandreel_gl.depthMask(flag);
 }

 function myglDepthRangef (sp)
 {
        var zNear = heapFloat[sp>>2];sp+=4;
        var zFar = heapFloat[sp>>2];sp+=4;
        imandreel_gl.depthRange(zNear, zFar);
 }

function myglFrontFace (sp)
 {
        var mode = heap32[sp>>2];sp+=4;
        imandreel_gl.frontFace(mode);
 }

 function myglBlendFunc (sp)
 {
        var sfactor = heap32[sp>>2];sp+=4;
        var dfactor = heap32[sp>>2];sp+=4;
        imandreel_gl.blendFunc(sfactor,dfactor);
 }

 function myglBlendColor(sp)
 {
        var red = heapFloat[sp>>2];sp+=4;
        var green = heapFloat[sp>>2];sp+=4;
        var blue = heapFloat[sp>>2];sp+=4;
        var alpha = heapFloat[sp>>2];sp+=4;
        imandreel_gl.blendColor(red,green,blue,alpha);
 }

 function myglBlendEquation(sp)
 {
        var mode = heap32[sp>>2];sp+=4;
        imandreel_gl.blendEquation(mode);
 }

 function myglBlendEquationSeparate(sp)
 {
        var modeRGB = heap32[sp>>2];sp+=4;
        var modeAlpha = heap32[sp>>2];sp+=4;
        imandreel_gl.blendEquationSeparate(modeRGB,modeAlpha);
 }

 function myglBlendFuncSeparate(sp)
 {
        var srcRGB = heap32[sp>>2];sp+=4;
        var dstRGB = heap32[sp>>2];sp+=4;
        var srcAlpha = heap32[sp>>2];sp+=4;
        var dstAlpha = heap32[sp>>2];sp+=4;

        imandreel_gl.blendFuncSeparate(srcRGB,dstRGB,srcAlpha,dstAlpha);
 }


 function myglColorMask (sp)
 {
        var red = heap32[sp>>2];sp+=4;
        var green = heap32[sp>>2];sp+=4;
        var blue = heap32[sp>>2];sp+=4;
        var alpha = heap32[sp>>2];sp+=4;
        imandreel_gl.colorMask(red,green,blue,alpha);
 }

 function removeByElement(arrayName,arrayElement)
 {
    for(var i=0; i<arrayName.length;i++ )
     {
        if(arrayName[i]==arrayElement)
                {
            arrayName.splice(i,1);
                        return;
            }
      }
  }


 function mygetParameter(sp)
 {
        var pname = heap32[sp>>2];sp+=4;
        r_g0 = imandreel_gl.getParameter(pname);
 }


 function mygetProgramParameter(sp)
 {
        var program_id = heap32[sp>>2];sp+=4;
        var pname = heap32[sp>>2];sp+=4;
        r_g0 = imandreel_gl.getProgramParameter(array_ids_ogl[program_id], pname);
 }

 function mygetShaderParameter(sp)
 {
        var shader_id = heap32[sp>>2];sp+=4;
        var pname = heap32[sp>>2];sp+=4;
        r_g0 = imandreel_gl.getShaderParameter(array_ids_ogl[shader_id], pname);
 }

 function myglVertexAttrib1f(sp)
 {
        var index = heap32[sp>>2];sp+=4;
        var x = heapFloat[sp>>2];sp+=4;
        imandreel_gl.vertexAttrib1f(index,x);
 }

  function myglVertexAttrib2f(sp)
 {
        var index = heap32[sp>>2];sp+=4;
        var x = heapFloat[sp>>2];sp+=4;
        var y = heapFloat[sp>>2];sp+=4;
        imandreel_gl.vertexAttrib2f(index,x,y);
 }

  function myglVertexAttrib3f(sp)
 {
        var index = heap32[sp>>2];sp+=4;
        var x = heapFloat[sp>>2];sp+=4;
        var y = heapFloat[sp>>2];sp+=4;
        var z = heapFloat[sp>>2];sp+=4;
        imandreel_gl.vertexAttrib3f(index,x,y,z);
 }

  function myglVertexAttrib4f(sp)
 {
        var index = heap32[sp>>2];sp+=4;
        var x = heapFloat[sp>>2];sp+=4;
        var y = heapFloat[sp>>2];sp+=4;
        var z = heapFloat[sp>>2];sp+=4;
        var w = heapFloat[sp>>2];sp+=4;
        imandreel_gl.vertexAttrib4f(index,x,y,z,w);
 }

 function myglVertexAttrib1fv(sp)
 {
        var index = heap32[sp>>2];sp+=4;
        var ptr = heap32[sp>>2];sp+=4;
        var x = heap32[ptr>>2];ptr+=4;
        imandreel_gl.vertexAttrib1f(index,x);
 }

 function myglVertexAttrib2fv(sp)
 {
        var index = heap32[sp>>2];sp+=4;
        var ptr = heap32[sp>>2];sp+=4;
        var x = heap32[ptr>>2];ptr+=4;
        var y = heap32[ptr>>2];ptr+=4;

        imandreel_gl.vertexAttrib2f(index,x,y);
 }

 function myglVertexAttrib3fv(sp)
 {
        var index = heap32[sp>>2];sp+=4;
        var ptr = heap32[sp>>2];sp+=4;
        var x = heap32[ptr>>2];ptr+=4;
        var y = heap32[ptr>>2];ptr+=4;
        var z = heap32[ptr>>2];ptr+=4;

        imandreel_gl.vertexAttrib3f(index,x,y,z);
 }

 function myglVertexAttrib4fv(sp)
 {
        var index = heap32[sp>>2];sp+=4;
        var ptr = heap32[sp>>2];sp+=4;
        var x = heap32[ptr>>2];ptr+=4;
        var y = heap32[ptr>>2];ptr+=4;
        var z = heap32[ptr>>2];ptr+=4;
        var w = heap32[ptr>>2];ptr+=4;

        imandreel_gl.vertexAttrib4f(index,x,y,z,w);
 }


 function myglDeleteTexture (sp)
 {

        var texture_id = heap32[sp>>2];sp+=4;

        var texture = array_ids_ogl[texture_id];

        imandreel_gl.deleteTexture(texture);

        array_ids_ogl[texture_id] = null;

        myglFreeSlot(texture_id);
 }

 function myglDeleteBuffer (sp)
 {

        var buffer_id = heap32[sp>>2];sp+=4;

        var buffer = array_ids_ogl[buffer_id];

        imandreel_gl.deleteBuffer(buffer);

        array_ids_ogl[buffer_id] = null;

        myglFreeSlot(buffer_id);
 }

 function myglDeleteFrameBuffer (sp)
 {
        var framebuffer_id = heap32[sp>>2];sp+=4;

        var framebuffer = array_ids_ogl[framebuffer_id];

        imandreel_gl.deleteFramebuffer(framebuffer);

        array_ids_ogl[framebuffer_id] = null;

        myglFreeSlot(framebuffer_id);
 }


 function myglDeleteProgram (sp)
 {
        var program_id = heap32[sp>>2];sp+=4;

        var program = array_ids_ogl[program_id];

        imandreel_gl.deleteProgram(program);

        array_ids_ogl[program_id] = null;

        myglFreeSlot(program_id);
 }

 function myglDeleteRenderBuffer (sp)
 {
        var renderbuffer_id = heap32[sp>>2];sp+=4;

        var renderbuffer = array_ids_ogl[renderbuffer_id];

        imandreel_gl.deleteRenderbuffer(renderbuffer);

        array_ids_ogl[renderbuffer_id] = null;

        myglFreeSlot(renderbuffer_id);
 }

 function myglDeleteShader (sp)
 {
        var shader_id = heap32[sp>>2];sp+=4;

        var shader = array_ids_ogl[shader_id];

        imandreel_gl.deleteShader(shader);

        array_ids_ogl[shader_id] = null;

        myglFreeSlot(shader_id);
 }

 function myglInit(sp)
 {

        for (var i=0;i<max_ogl_id;++i)
        {
                array_ids_ogl_enable.push(i+1);
    }
 }

 function myglReadPixels(sp)
 {
  var x = heap32[sp>>2];sp+=4;
  var y = heap32[sp>>2];sp+=4;
  var width = heap32[sp>>2];sp+=4;
  var height = heap32[sp>>2];sp+=4;
  var format = heap32[sp>>2];sp+=4;
  var type = heap32[sp>>2];sp+=4;
  var pixels = heap32[sp>>2];sp+=4;

        var bufferView = new Uint8Array(heap,pixels,width*height*4);
  imandreel_gl.readPixels(x,y,width,height,format,type,bufferView);
}

function mandreel_internal_DrawSprite(sp)
{
        if ( imandreel_ctx_canvas == null )
        {
                console.log("Mandreel_2D_DrawSprite error: canvas context is null");
                return;
    }

        var hw_id = heap32[sp>>2]; sp+=4;
        var u0 = heapFloat[sp>>2]; sp+=4;
        var u1 = heapFloat[sp>>2]; sp+=4;
        var v0 = heapFloat[sp>>2]; sp+=4;
        var v1 = heapFloat[sp>>2]; sp+=4;
        //var x = heapFloat[sp>>2]; sp+=4;
        //var y = heapFloat[sp>>2]; sp+=4;
        var sx = heapFloat[sp>>2]; sp+=4;
        var sy = heapFloat[sp>>2]; sp+=4;
        //var rot = heapFloat[sp>>2]; sp+=4;
        var m11 = heapFloat[sp>>2]; sp+=4;
        var m12 = heapFloat[sp>>2]; sp+=4;
        var m21 = heapFloat[sp>>2]; sp+=4;
        var m22 = heapFloat[sp>>2]; sp+=4;
        var tx = heapFloat[sp>>2]; sp+=4;
        var ty = heapFloat[sp>>2]; sp+=4;
        var color = heap32[sp>>2]; sp+=4;

        //ctx_canvas.fillStyle="#FF0000";
        //ctx_canvas.fillRect(tx,ty,sx,sy);
        var texture = array_ids_ogl[hw_id];

        if ( texture == null )
        {
                console.log("Mandreel_2D_DrawSprite error: texture invalid ("+hw_id+") or loading yet...");
                return;
    }

        var width = (u1-u0)*texture.width;
        var height = (v1-v0)*texture.height;

        if (width == 0)
                return;
        if (height == 0)
                return;

        var x_offset = u0*texture.width;
        var y_offset = v0*texture.height;

        //dump(x_offset + ' ' + y_offset + ' ' + texture.width + ' ' + texture.height + ' ' + x + ' ' +y + ' ' + width + ' ' + height + '\n');

        x_offset = x_offset % texture.width;
        y_offset = y_offset % texture.height;

        var scale_x, scale_y;
        if (sx<0)
                scale_x = -1;
        else
                scale_x = 1;
        if (sy<0)
                scale_y = -1;
        else
                scale_y = 1;

        var simple_draw = false;//scale_x == 1 && scale_y == 1 && rot == 0;
        var x_pos = (0.5*sx + tx/* + 240*/);
        var y_pos = (/*320*/ + ((0.5*sy + ty) /*+ 160*/));
        var new_sx = sx;
        var new_sy = sy;

        imandreel_ctx_canvas.globalAlpha = (color>>>24)/255;

        if (!simple_draw)
        {
                imandreel_ctx_canvas.save()
                /*ctx_canvas.translate(x_pos + new_sx/2, y_pos + new_sy/2)
                ctx_canvas.rotate(-rot*Math.PI/180);
                ctx_canvas.scale(scale_x, scale_y);
                ctx_canvas.translate(-(x_pos + new_sx/2), -(y_pos + new_sy/2))*/
                imandreel_ctx_canvas.setTransform(m11,m12,m21,m22,tx,ty);
    }

        /*if (x_offset<0 || y_offset<0 || (x_offset+width)>texture.width || (y_offset+height)>texture.height)
        {
                dump(x_offset + ' ' + y_offset + ' ' + texture.width + ' ' + texture.height + ' ' + x + ' ' +y + ' ' + width + ' ' + height + '\n');
    }*/

        if (new_sx<0)
        {
                x_pos += new_sx;
                new_sx = -new_sx;
    }
        if (new_sy<0)
        {
                y_pos += new_sy;
                new_sy = -new_sy;
    }

        //ctx_canvas.drawImage(texture,x_offset,y_offset, width, height, x_pos, y_pos , new_sx, new_sy);
        imandreel_ctx_canvas.drawImage(texture,x_offset,y_offset, width, height, 0,0, 1,1);

        if (!simple_draw)
                imandreel_ctx_canvas.restore()
}

function mandreel_internal_UpdateTexture(sp)
{
        if ( imandreel_ctx_canvas == null )
        {
                console.log("Mandreel_2D_UpdateTexture error: canvas context is null");
                return;
    }

        var hw_id = heap32[sp>>2]; sp+=4;
        var dataptr = heap32[sp>>2]; sp+=4;
        var width = heap32[sp>>2]; sp+=4;
        var height = heap32[sp>>2]; sp+=4;

        var imageData = imandreel_ctx_canvas.getImageData(0,0,width,height);
        var data = imageData.data;
        for (var y = 0; y < (height*width*4); ++y)
        {
                data[y] = heapU8[dataptr + y];
    }
        imandreel_ctx_canvas.putImageData(imageData,0,0);

        var dataurl = imandreel_ctx_canvas.canvas.toDataURL();

        var image = new Image();
        image.onerror = function()
        {
                dump('error updating texture '+hw_id+'\n');
    }
        image.onload = function()
        {
                //dump('texture '+hw_id+' updated\n'+'width '+image.width+' height '+image.height);
                array_ids_ogl[hw_id] = image;
    }
        image.src = dataurl;
}

function mandreel_internal_WriteFramebuffer(sp)
{
        if ( imandreel_ctx_canvas == null )
        {
                console.log("Mandreel_2D_UpdateTexture error: canvas context is null");
                return;
    }

        var dataptr = heap32[sp>>2]; sp+=4;
        var width = heap32[sp>>2]; sp+=4;
        var height = heap32[sp>>2]; sp+=4;

        var imageData = imandreel_ctx_canvas.getImageData(0,0,width,height);
        if ( imageData != null )
        {
                var data = imageData.data;
                if ( data != null )
                {
                        var size = (height*width*4);
                        if ( typeof imageData.data.set != "undefined" )
                        {
                                var sub = heapU8.subarray(dataptr,dataptr+size);
                                imageData.data.set(sub);
                    }
                        else
                        {
                                for (var y = 0; y < size; ++y)
                                {
                                        data[y] = heapU8[dataptr + y];
                            }
                    }

                        imandreel_ctx_canvas.putImageData(imageData,0,0);
            }
                else
                {
                        dump("WriteFramebuffer canvas data null");
            }
    }
        else
        {
                dump("WriteFramebuffer canvas imageData null");
    }
}
var mandreel_audio_init = null_mandreel_audio;
var mandreel_audio_end = null_mandreel_audio;
var mandreel_audio_update = null_mandreel_audio;
var mandreel_audio_createBuffer = null_mandreel_audio;
var mandreel_audio_playChannel = null_mandreel_audio;
var mandreel_audio_stopChannel = null_mandreel_audio;
var mandreel_audio_setChannelVolume = null_mandreel_audio;
var mandreel_audio_setChannelPan = null_mandreel_audio;
var mandreel_audio_setChannelPitch = null_mandreel_audio;
var mandreel_audio_playMusic = null_mandreel_audio;
var mandreel_audio_stopMusic = null_mandreel_audio;
var mandreel_audio_useMusicFunctions = _mandreel_audio_useMusicFunctions;
var mandreel_audio_checkBuffersPending = null_mandreel_audio;
var mandreel_audio_setMusicVol = null_mandreel_audio;

var mandreel_audio_startedFunction = 0;

var MandreelAudioDriver = "null";

function mandreel_start_audio(audioServer, audioUrl,startedFunction)
{
        mandreel_audio_startedFunction = startedFunction;

        // Check audio driver data availability
        var webAudioDataAvailable = false;
        var flashAudioDataAvailable = false;
        var flashLiteAudioDataAvailable = false;
        var audiotagsDataAvailable = false;
        var FatLines = mandreelFatData.split('\n');
        for ( var i=0;i<FatLines.length;++i )
        {
                var params = FatLines[i].split(',');
                if ( params[0] == "audiodriver" )
                {
                        var data = params[1];
                        data = data.replace('\r','');
                        if ( data == "webaudio" )
                                webAudioDataAvailable = true;
                        else if ( data == "flash" )
                                flashAudioDataAvailable = true;
                        else if ( data == "flashlite" )
                                flashLiteAudioDataAvailable = true;
                        else if ( data == "audiotag" )
                                audiotagsDataAvailable = true;
            }
    }


        // Init audio driver
        {
                // webaudio
                if ( webAudioDataAvailable && MandreelAudioDriver == "null" )
                {
                        try     { webAudioContext = new webkitAudioContext(); } catch(err) { webAudioContext = 0; }
                        if ( webAudioContext )
                        {
                                wa_MainAudioDriver();
                                MandreelAudioDriver = "webAudio";
                    }
            }
                // flash
                if ( flashAudioDataAvailable && MandreelAudioDriver == "null" )
                {
                        MandreelAudioDriver = "flash";
                        if ( audioServer == null )
                        {
                                audioServer = "";
                                audioUrl = "";
                    }
                        fl_MainAudioDriver(audioServer,audioUrl);
            }
                // flashlite
                if ( flashLiteAudioDataAvailable && MandreelAudioDriver == "null" )
                {
                        MandreelAudioDriver = "flashlite";
                        mandreel_flashaudio_lite = true;
                        fl_MainAudioDriver("","");
            }
                // audiotags
                if ( audiotagsDataAvailable && MandreelAudioDriver == "null" )
                {
                        MandreelAudioDriver = "audiotag";
                        at_MainAudioDriver();
            }
                // null
                if ( MandreelAudioDriver == "null" )
                {
                        null_MainAudioDriver();
            }
    }
        dump("AudioDriver ("+MandreelAudioDriver+")");
}

function mandreel_audio_isLogEnabled(sp)
{
        r_g0 = 0;
}

function _mandreel_audio_useMusicFunctions(sp)
{
        r_g0 = 0;
}

function MandreelAudioStarted()
{
        setTimeout(mandreel_audio_startedFunction, 10);
}


function mandreel_audio_getAudioDriverName(sp)
{
        var name_ptr = heap32[sp>>2];sp+=4;
        fill_ptr_from_string(name_ptr, MandreelAudioDriver);
}
var webAudioUseMusicFunctions = true;

function wa_mandreel_audio_init(sp)
{
}

function wa_mandreel_audio_end(sp)
{
}

function wa_mandreel_audio_update(sp)
{
}

function wa_mandreel_audio_checkBuffersPending(sp)
{
        r_g0 = wa_imp_mandreel_audio_checkBuffersPending();
}

function wa_mandreel_audio_createBuffer(sp)
{
        var ptr_fileName = heap32[sp>>2];sp+=4;
        var maxChannels = heap32[sp>>2];sp+=4;
        var fileName = get_string_from_ptr(ptr_fileName).toLowerCase();
        wa_imp_mandreel_audio_createBuffer(fileName);

        r_g0 = 0;
}

function wa_mandreel_audio_playChannel(sp)
{
        var ptr_fileName = heap32[sp>>2];sp+=4;
        var channel = heap32[sp>>2];sp+=4;
        var vol = heapFloat[sp>>2];sp+=4;
        var loop = heap32[sp>>2];sp+=4;
        var pitch = heapFloat[sp>>2];sp+=4;
        var fileName = get_string_from_ptr(ptr_fileName).toLowerCase();
        wa_imp_mandreel_audio_playChannel(fileName,channel,vol,loop,pitch);
        r_g0 = 0;
}

function wa_mandreel_audio_stopChannel(sp)
{
        var ptr_fileName = heap32[sp>>2];sp+=4;
        var channel = heap32[sp>>2];sp+=4;
        var index = heapFloat[sp>>2];sp+=4;
        wa_imp_mandreel_audio_stopChannel(channel);
}

function wa_mandreel_audio_setChannelVolume(sp)
{
        var channel = heap32[sp>>2];sp+=4;
        var vol = heapFloat[sp>>2];sp+=4;
        wa_imp_mandreel_audio_setChannelVolume(channel,vol);
}

function wa_mandreel_audio_setChannelPan(sp)
{
        var channel = heap32[sp>>2];sp+=4;
        var pan = heapFloat[sp>>2];sp+=4;
        wa_imp_mandreel_audio_setChannelPan(channel,pan);
}

function wa_mandreel_audio_setChannelPitch(sp)
{
        var channel = heap32[sp>>2];sp+=4;
        var pitch = heapFloat[sp>>2];sp+=4;
        wa_imp_mandreel_audio_setChannelPitch(channel,pitch);
}

function wa_mandreel_audio_useMusicFunctions(sp)
{
        r_g0 = webAudioUseMusicFunctions ? 1 : 0;
}

function wa_mandreel_audio_playMusic(sp)
{
        var ptr_fileName = heap32[sp>>2];sp+=4;
        var fileName = get_string_from_ptr(ptr_fileName).toLowerCase();
        wa_imp_mandreel_audio_playMusic(fileName);
}

function wa_mandreel_audio_stopMusic(sp)
{
        wa_imp_mandreel_audio_stopMusic();
}

function wa_mandreel_audio_setMusicVol(sp)
{
        var vol = heapFloat[sp>>2];sp+=4;
        wa_imp_mandreel_audio_setMusicVol(vol);
}

function wa_MainAudioDriver()
{
        mandreel_audio_init = wa_mandreel_audio_init;
        mandreel_audio_end = wa_mandreel_audio_end;
        mandreel_audio_update = wa_mandreel_audio_update;
        mandreel_audio_createBuffer = wa_mandreel_audio_createBuffer;
        mandreel_audio_playChannel = wa_mandreel_audio_playChannel;
        mandreel_audio_stopChannel = wa_mandreel_audio_stopChannel;
        mandreel_audio_setChannelVolume = wa_mandreel_audio_setChannelVolume;
        mandreel_audio_setChannelPan = wa_mandreel_audio_setChannelPan;
        mandreel_audio_setChannelPitch = wa_mandreel_audio_setChannelPitch;
        mandreel_audio_useMusicFunctions = wa_mandreel_audio_useMusicFunctions;
        mandreel_audio_playMusic = wa_mandreel_audio_playMusic;
        mandreel_audio_stopMusic = wa_mandreel_audio_stopMusic;
        mandreel_audio_checkBuffersPending = wa_mandreel_audio_checkBuffersPending;
        mandreel_audio_setMusicVol = wa_mandreel_audio_setMusicVol;

        setTimeout("mandreel_webAudio_PreloadAssets()", 10);
}


var webAudioBuffers = new Array();
var webAudioChannels = new Array(32);
var webAudioGain = new Array(32);
var webAudioContext = 0;


function wa_initWebAudio()
{
        if ( preCreatedWebAudioContext != null )
        {
                webAudioContext = preCreatedWebAudioContext;
    }
        else
        {
                try     { webAudioContext = new webkitAudioContext(); } catch(err) { webAudioContext = 0; }
    }
}

function wa_imp_callfunction(params)
{
        if ( params[0] == "#MandreelAudio" )
        {
                if ( params[1] == "playChannel" )
                        wa_imp_mandreel_audio_playChannel(params[2],params[3],params[4],params[5],params[6],params[7]);
                else if ( params[1] == "stopChannel" )
                        wa_imp_mandreel_audio_stopChannel(params[2]);
                else if ( params[1] == "setChannelVol" )
                        wa_imp_mandreel_audio_setChannelVol(params[2],params[3]);
                else if ( params[1] == "setChannelPan" )
                        wa_imp_mandreel_audio_setChannelPan(params[2],params[3]);
                else if ( params[1] == "playMusic" )
                        wa_imp_mandreel_audio_playMusic(params[2]);
                else if ( params[1] == "stopMusic" )
                        wa_imp_mandreel_audio_stopMusic(params[2]);
                else if ( params[1] == "setMusicVol" )
                        wa_imp_mandreel_audio_setMusicVol(params[2]);

                return true;
    }
        return false;
}

function wa_getFileNameNoExt(fileName)
{
        var fileNameNoExt = fileName.toLowerCase();
        var indexDot = fileNameNoExt.lastIndexOf('.');
        if ( indexDot != -1 )
                fileNameNoExt = fileNameNoExt.substr(0,indexDot);
        fileNameNoExt = fileNameNoExt.replace(/\\/g,"/");
        if ( fileNameNoExt.length > 0 )
        {
                if ( fileNameNoExt.charAt(0) == "/" )
                {
                        if (fileNameNoExt.length > 1 )
                                fileNameNoExt = fileNameNoExt.substr(1,fileNameNoExt.length-1);
                        else
                                fileNameNoExt = "";
            }
    }
        return fileNameNoExt;
}

var wa_mandreel_cache_audio_files = false;
var wa_mandreel_cache_audio_files_path = '';

function mandreel_webAudio_reloadfile(fileName,callback)
{
        var fileNameNoExt = wa_getFileNameNoExt(fileName);
        var url = wa_mandreel_cache_audio_files_path+fileNameNoExt+".ogg";
        dump("webAudio: loading buffer ("+fileName+") url("+url+")");
        var request = new XMLHttpRequest();
        request.open("GET", url, true);
        request.responseType = "arraybuffer";
        request.onreadystatechange = function()
        {
                if (request.readyState == 4)
                {

                        if (request.status == 404) callback(null);

                        callback(request.response);

            }
    }
        request.send();
}

Mandreel_window.BlobBuilder = Mandreel_window.MozBlobBuilder || Mandreel_window.WebKitBlobBuilder || Mandreel_window.BlobBuilder;

function mandreel_webaudio_saveFile(name, my_arrayBuffer)
{
        dump('mandreel_webaudio_saveFile ' + name);
        g_mandreel_fs.root.getFile(name, {create: true}, function(fileEntry) {

    // Create a FileWriter object for our FileEntry (log.txt).
    fileEntry.createWriter(function(fileWriter) {

      fileWriter.onwriteend = function(e) {
        dump('Write completed.');
      };

      fileWriter.onerror = function(e) {
        dump('Write failed: ' + e.toString());
      };

          var bb = new Mandreel_window.BlobBuilder(); // Note: window.WebKitBlobBuilder in Chrome 12.
      bb.append(my_arrayBuffer);
      fileWriter.write(bb.getBlob('text/plain'));



    }, function(e) { dump('error 1 mandreel_webaudio_saveFile ' + name);MandreelFsErrorHandler(e) });

  }, function(e) { dump('error 2 mandreel_webaudio_saveFile ' + name);MandreelFsErrorHandler(e) });
}

function mandreel_webaudio_loadFile(name, callback, callback2)
{
        g_mandreel_fs.root.getFile(name, {}, function(fileEntry) {

        fileEntry.file(function(file) {

                var reader = new FileReader();


       reader.onloadend = function(e) {

           dump('mandreel_fs_loadFile ' + name);
           if (this.result.byteLength)
                callback(this.result);
           else
           callback2(this.result);

       };


                reader.readAsArrayBuffer(file);



    }, function(e) { dump('error 1 webaudio_loadFile ' + name);MandreelFsErrorHandler(e) } );

  }, function(e) { dump('error 2 webaudio_loadFile ' + name);MandreelFsErrorHandler(e) } );
}


function mandreel_webAudio_cacheFile(fileName, callback)
{
        if (!g_mandreel_fs)
        {
                mandreel_webAudio_reloadfile(fileName, callback);

                return;
    }
        fileName = fileName.toLowerCase();

        fileName = fileName.replace(/\\/g,"/");

        if (fileName[0] == '/')
                fileName = fileName.substring(1);

        dump('chanka name ' + fileName);

        g_mandreel_fs.root.getFile(fileName, {}, function(fileEntry) {

        fileEntry.getMetadata(function(metaData){
        var my_seconds = metaData.modificationTime.getTime()/1000;

        if (g_mandreel_timestamp_fs>my_seconds)
        {
                dump('mandreel_webAudio_cacheFile1 ');
                fileEntry.remove(function() {
                                mandreel_webAudio_reloadfile(fileName, function(response) { callback(response); if (response) mandreel_webaudio_saveFile(fileName, response); } );
                    }, function(e) { dump('error 1 mandreel_webAudio_cacheFile ' + fileName);MandreelFsErrorHandler(e); mandreel_webAudio_reloadfile(fileName, function(response) { callback(response); if (response) mandreel_webaudio_saveFile(fileName, response); } );});

    }
        else
        {
                //alert('mandreel_fscachefile2');
                dump('mandreel_webAudio_cacheFile2');
                mandreel_webaudio_loadFile(fileName, function(response) { callback(response);  } ,
                function() {
                                mandreel_webAudio_reloadfile(fileName, function(response) { callback(response); if (response) mandreel_webaudio_saveFile(fileName, response); } );
                    }
                );



    }


            }, function(e) { dump('error 2 mandreel_webAudio_cacheFile ' + fileName);MandreelFsErrorHandler(e) });


  }, function(error) {dump('mandreel_webAudio_cacheFile3');     mandreel_webAudio_reloadfile(fileName, function(response) { callback(response); if (response) mandreel_webaudio_saveFile(fileName, response); });});
}

function mandreel_webAudio_queueLoadBuffer(fileName, callback)
{
        //dump("mandreel_webAudio_queueLoadBuffer "+fileName);
        if ( webAudioContext )
        {
                var fileNameNoExt = wa_getFileNameNoExt(fileName);
                var buffer = webAudioBuffers[fileNameNoExt];
                if ( buffer == null && buffer != "invalid" )
                {
                        if ( fileNameNoExt != "" )
                        {
                                if (wa_mandreel_cache_audio_files == true)
                                {
                                        webAudioBuffers[fileNameNoExt] = "invalid";

                                        mandreel_webAudio_cacheFile(fileName, function(response) {

                                                if (callback != null)
                                                        callback();

                                                if ( response != null )
                                                        webAudioBuffers[fileNameNoExt] = webAudioContext.createBuffer(response, false);
                                                else
                                                        webAudioBuffers[fileNameNoExt] = "invalid";
                                    } );
                                        return;
                            }

                                var url = g_mandreel_working_folder+fileNameNoExt+".ogg";
                                dump("webAudio: loading buffer ("+fileName+") url("+url+")");
                                webAudioBuffers[fileNameNoExt] = "invalid";
                                var request = new XMLHttpRequest();
                                request.open("GET", url, true);
                                request.responseType = "arraybuffer";
                                request.onreadystatechange = function()
                                {
                                        if (request.readyState == 4)
                                        {
                                                if (callback != null)
                                                        callback();

                                                if (request.status == 404) return;
                                                //dump("webAudio: loaded buffer "+request.status);
                                                if ( request.response != null )
                                                        webAudioBuffers[fileNameNoExt] = webAudioContext.createBuffer(request.response, false);
                                                else
                                                        webAudioBuffers[fileNameNoExt] = "invalid";
                                    }
                            }
                                request.send();
                    }
                        else
                                webAudioBuffers[fileNameNoExt] = "invalid";
            }
    }
}

var webAudioPreloadAudioFiles = new Array();
var webAudioPreloadBytes = null;
var webAudioPreloadCurrentFile = 0;
var webAudioPreloadAsync = false;

function webAudioParsePreloadFile(response)
{
        var pos = 4;
        var bytes = new Uint8Array(response);
        var i = 0;
        if ( webAudioPreloadAsync )
                webAudioPreloadBytes = bytes;
        while ( pos < bytes.byteLength )
        {
                // filename
                var fileName = "";
                while ( bytes[pos] != 0 )
                {
                        fileName += String.fromCharCode(bytes[pos]);
                        pos++;
            }
                pos++;
                // filesize
                var filesize = bytes[pos] | (bytes[pos+1]<<8) | (bytes[pos+2]<<16) | (bytes[pos+3]<<24);
                pos += 4;

                // contents
                var fileNameNoExt = wa_getFileNameNoExt(fileName);
                if ( webAudioPreloadAsync )
                {
                        var audioFile =
                        {
                                fileName : fileNameNoExt,
                                fileSize : filesize,
                                position : pos
                    };
                        webAudioPreloadAudioFiles[i] = audioFile;
            }
                else
                {
                        var bufferdata = new Uint8Array(filesize);
                        bufferdata.set(bytes.subarray(pos,pos+filesize));
                        webAudioBuffers[fileNameNoExt] = webAudioContext.createBuffer(bufferdata.buffer, false);
                        dump("preload audio file ("+fileName+")");
            }

                pos += filesize;
                i++;
    }
        if ( webAudioPreloadAsync )
        {
                dump("started preloading audio files async");
                setTimeout("mandreel_webAudio_preloadNextAudioFile()",10);
    }
}

function mandreel_webAudio_preloadNextAudioFile()
{
        if ( webAudioPreloadAudioFiles.length > webAudioPreloadCurrentFile )
        {
                var audioFile = webAudioPreloadAudioFiles[webAudioPreloadCurrentFile];
                if ( audioFile.fileName == null )
                {
                        webAudioPreloadCurrentFile++;
                        setTimeout("mandreel_webAudio_preloadNextAudioFile()",10);
            }
                else
                {
                        var bufferdata = new Uint8Array(audioFile.fileSize);
                        bufferdata.set(webAudioPreloadBytes.subarray(audioFile.position,audioFile.position+audioFile.fileSize));
                        dump("async preload audio file ("+audioFile.fileName+")");
                        webAudioContext.decodeAudioData(bufferdata.buffer,mandreel_webAudio_preloadNextAudioFileOK,mandreel_webAudio_preloadNextAudioFileError);
            }
    }
        else
        {
                webAudioPreloadAudioFiles = null;
                webAudioPreloadBytes = null;
                dump("finished preloading audio files async");
    }
}

function mandreel_webAudio_preloadNextAudioFileOK(audioBuffer)
{
        var audioFile = webAudioPreloadAudioFiles[webAudioPreloadCurrentFile];
        webAudioPreloadCurrentFile++;
        if ( audioFile.fileName != null )
        {
                webAudioBuffers[audioFile.fileName] = audioBuffer;
                setTimeout("mandreel_webAudio_preloadNextAudioFile()",10);
    }
}

function mandreel_webAudio_preloadNextAudioFileError()
{
        var audioFile = webAudioPreloadAudioFiles[webAudioPreloadCurrentFile];
        webAudioPreloadCurrentFile++;
        if ( audioFile.fileName != null )
                dump("FAILED async preload audio file ("+audioFile.fileName+")");
        setTimeout("mandreel_webAudio_preloadNextAudioFile()",10);
}

function mandreel_webAudio_queueLoadPackedBuffers(fileName, callback)
{
        if ( webAudioContext )
        {
                // TODO
                /*if (wa_mandreel_cache_audio_files == true)
                {
                        webAudioBuffers[fileNameNoExt] = "invalid";

                        mandreel_webAudio_cacheFile(fileName, function(response) {

                                if (callback != null)
                                        callback();

                                if ( response != null )
                                        webAudioBuffers[fileNameNoExt] = webAudioContext.createBuffer(response, true);
                                else
                                        webAudioBuffers[fileNameNoExt] = "invalid";
                    } );
                        return;
            }*/

                mandreel_fs_load_binary(fileName, function mandreel_webAudio_queueLoadPackedBuffers_cb(data){

                if (data)
                {
                        if (callback != null)
                                callback();
                        webAudioParsePreloadFile(data);
            }
                else
                {
                        var url = g_mandreel_working_folder+fileName;
                        dump("webAudio: loading preload buffers ("+fileName+") url("+url+")");
                        var request = new XMLHttpRequest();
                        request.open("GET", url, true);
                        request.responseType = "arraybuffer";
                        var last_loaded_size = 0;
                        request.onreadystatechange = function()
                        {
                                if (request.readyState == 4)
                                {
                                        if (callback != null)
                                                callback();

                                        if (request.status == 404)
                                                return;
                                        if ( request.response != null )
                                        {
                                                mandreel_fs_saveFile(fileName, request.response);
                                                webAudioParsePreloadFile(request.response);
                                    }
                            }
                    }
                        request.onprogress = function(e)
                        {
                                var delta_size = e.loaded - last_loaded_size;
                                last_loaded_size = e.loaded;

                                imandreel_update_load(delta_size,0);

                                var percentage = ((100*e.loaded)/mandreel_total_pogfile_size)|0;

                                if (percentage>100)
                                        percentage = 100;

                                if ( mandreelAppStartStateFunc )
                                        mandreelAppStartStateFunc("loadingAudioUpdate",percentage);
                    }
                        request.send();
            }
            });
    }
}



var wa_mandreel_audio_buffers_num = 0;
var wa_mandreel_audio_buffers_loaded = 0;

function wa_imp_mandreel_audio_checkBuffersPending()
{
        return wa_mandreel_audio_buffers_num - wa_mandreel_audio_buffers_loaded;
}

function wa_imp_mandreel_audio_createBuffer(fileName)
{
        if ( webAudioContext )
        {
                var fileNameNoExt = wa_getFileNameNoExt(fileName);
                var buffer = webAudioBuffers[fileNameNoExt];
                if ( buffer == null && buffer != "invalid" )
                {
                        wa_mandreel_audio_buffers_num++;
                        mandreel_webAudio_queueLoadBuffer(fileName, function () {wa_mandreel_audio_buffers_loaded++;});
            }
    }
}

function wa_imp_mandreel_audio_playChannel(fileName,channel,vol,loop,pitch)
{
        if ( webAudioContext )
        {
                var fileNameNoExt = wa_getFileNameNoExt(fileName);
                var buffer = webAudioBuffers[fileNameNoExt];
                if ( buffer == null || buffer == "invalid" )
                {
                        if ( webAudioPreloadAudioFiles != null )
                        {
                                var i = 0;
                                while ( i < webAudioPreloadAudioFiles.length )
                                {
                                        var audioFile = webAudioPreloadAudioFiles[i];
                                        if ( audioFile.fileName == fileNameNoExt )
                                        {
                                                var bufferdata = new Uint8Array(audioFile.fileSize);
                                                bufferdata.set(webAudioPreloadBytes.subarray(audioFile.position,audioFile.position+audioFile.fileSize));
                                                webAudioBuffers[audioFile.fileName] = webAudioContext.createBuffer(bufferdata.buffer, false);
                                                dump("**** preload audio file ("+audioFile.fileName+"), forced by playChannel");
                                                audioFile.fileName = null;
                                                i = webAudioPreloadAudioFiles.length;
                                    }
                                        else
                                        {
                                                ++i;
                                    }
                            }
                    }
            }
                if ( buffer == null || buffer == "invalid" )
                {
                        mandreel_webAudio_queueLoadBuffer(fileName);
                        buffer = webAudioBuffers[fileNameNoExt];
            }
                if ( buffer != null && buffer != "invalid" )
                {
                        var chn = webAudioContext.createBufferSource();
                        var gain = webAudioContext.createGainNode();
                        if ( chn && gain )
                        {
                                webAudioChannels[channel] = chn;
                                webAudioGain[channel] = gain;
                                chn.buffer = buffer;
                                chn.connect(gain);
                                gain.connect(webAudioContext.destination);
                                var bLoop = loop != 0;
                                chn.loop = bLoop;
                                gain.gain.value = vol;
                                chn.playbackRate.value = pitch;
                                chn.noteOn(0);
                                //dump("webAudio: PlayChannel "+channel+" "+fileName+" "+vol+" "+bLoop+" "+pitch);
                    }
            }
    }
}

function wa_imp_mandreel_audio_stopChannel(channel)
{
        if ( webAudioContext )
        {
                var chn = webAudioChannels[channel];
                if ( chn != null )
                {
                        //dump("webAudio: StopChannel "+channel);
                        chn.noteOff(0);
                        webAudioChannels[channel] = null;
                        webAudioGain[channel] = null;
            }
    }
}

function wa_imp_mandreel_audio_setChannelVolume(channel,vol)
{
        if ( webAudioContext )
        {
                var gain = webAudioGain[channel];
                if ( gain != null )
                        gain.gain.value = vol;
    }
}

function wa_imp_mandreel_audio_setChannelPan(channel,pan)
{
        if ( webAudioContext )
        {
    }
}

function wa_imp_mandreel_audio_setChannelPitch(channel,pitch)
{
        if ( webAudioContext )
        {
                var chn = webAudioChannels[channel];
                if ( chn != null )
                {
                        chn.playbackRate.value = pitch;
            }
    }
}

var mwebAudioPreloadState = "start";
//var mwebAudioPreloadRequest = 0;
var mwebAudioPreloadAssets = 0;
var mwebAudioCurrentPreloadAsset = 0;
var mwebAudioAsyncFiles = '';
var mwebListAudioAsyncFiles ='';
var mwebListAudioAsyncFilesPos = 0;
var mwebTotalPreloadingFiles = 0;
var mwebCurrentPreloadingFiles = 0;
function mwebCallbackAsync()
{
        mwebListAudioAsyncFilesPos++;
        if ( mwebListAudioAsyncFilesPos >= mwebListAudioAsyncFiles.length )
        {
                mwebListAudioAsyncFiles = null;
                mwebAudioAsyncFiles = null;
                return;
    }

        setTimeout("mandreel_webAudio_queueLoadBuffer(mwebListAudioAsyncFiles[mwebListAudioAsyncFilesPos], mwebCallbackAsync);",10);
}

function mwebCallbackAsyncPreload()
{
        mwebCurrentPreloadingFiles++;
}

function mandreel_webAudio_PreloadAssets()
{
        /*if ( mwebAudioPreloadState == "start" )
        {
                //dump("webaudio start");
                        mwebAudioPreloadRequest = new XMLHttpRequest();
                        var url = g_mandreel_working_folder+"mandreel.fat.dat";
                        mwebAudioPreloadRequest.open("GET", url, true);
                        mwebAudioPreloadRequest.onreadystatechange = function()
                        {
                                if (mwebAudioPreloadRequest.readyState != 4) return;
                                if ( mwebAudioPreloadRequest.status != 404 && mwebAudioPreloadRequest.response != null )
                                        mwebAudioPreloadState = "parseFat";
                                else
                                {
                                        mwebAudioPreloadState = "done";
                                        dump("error pre-loading audio assets, status("+mwebAudioPreloadRequest.status+")");
                            }
                    }
                        mwebAudioPreloadState = "loadingFat";
                        mwebAudioPreloadRequest.send();
    }
        else if ( mwebAudioPreloadState == "parseFat" )*/
        if ( mwebAudioPreloadState == "start" )
        {
                //mwebAudioPreloadAssets = mwebAudioPreloadRequest.response.split('\n');
                //mwebAudioPreloadRequest = 0;
                mwebAudioPreloadAssets = mandreelFatData.split('\n');
                mwebAudioCurrentPreloadAsset = 0;
                mwebAudioPreloadState = "preloading";
    }
        else if ( mwebAudioPreloadState == "preloading" )
        {
                //dump("webaudio preloading");
                var queued = false;
                while ( !queued && mwebAudioPreloadState != "done" )
                {
                        if ( mwebAudioCurrentPreloadAsset < mwebAudioPreloadAssets.length )
                        {
                                var params = mwebAudioPreloadAssets[mwebAudioCurrentPreloadAsset].split(',');
                                if ( params[0] == "audiofile" && params[1])
                                {
                                        var sync_load = true;
                                        if (params[2] && (params[2]&1) == 0)
                                                sync_load = false;

                                        if (sync_load)
                                        {
                                                mandreel_webAudio_queueLoadBuffer(params[1],mwebCallbackAsyncPreload);
                                                mwebTotalPreloadingFiles++;
                                                queued = true;
                                    }

                            }
                                else if ( params[0] == "audiopreloadfile" )
                                {
                                        mandreel_webAudio_queueLoadPackedBuffers(params[1],mwebCallbackAsyncPreload);
                                        var size_pog_file = parseInt(params[2], 10);
                                        mandreel_total_pogfile_size = size_pog_file;
                                        mwebTotalPreloadingFiles++;
                                        queued = true;
                            }
                                else if ( params[0] == "audiopreloadasync" )
                                {
                                        webAudioPreloadAsync = true;
                            }
                                mwebAudioCurrentPreloadAsset++;
                    }
                        else
                                queued = true;

                        if ( mwebAudioCurrentPreloadAsset >= mwebAudioPreloadAssets.length )
                        {
                                if (mwebCurrentPreloadingFiles == mwebTotalPreloadingFiles)
                                {
                                        mwebAudioPreloadState = "done";
                                        mwebAudioPreloadAssets = 0;
                            }
                    }
            }
    }

        if ( mwebAudioPreloadState == "done" )
        {
                if ( mandreelAppPlatform == "nacl" )
                        setTimeout("MandreelAudioStartedNacl()", 10);
                else
                        setTimeout("MandreelAudioStarted()", 10);
    }
        else
                setTimeout("mandreel_webAudio_PreloadAssets()", 10);
}

var wa_mandreelMusicElement = null;
var wa_mandreelMusicElementFilename = "";
var wa_mandreelMusicElementVolume = 1.0;
function wa_imp_mandreel_audio_playMusic(fileName)
{
        var fileNameNoExt = wa_getFileNameNoExt(fileName);
        var fileNameFull = g_mandreel_working_folder + fileNameNoExt + ".ogg";

        if ( wa_mandreelMusicElementFilename != fileNameFull )
        {
                wa_imp_mandreel_audio_stopMusic(0);
                var audio = document.createElement("audio");
                var type = fileNameFull.slice(fileNameFull.lastIndexOf(".")+1);
                switch(type){
                        case "mp3" : type = "mpeg"; break;
                        case "ogg" : type = "ogg"; break;
                        case "wav" : type = "wav"; break;
                        default : throw("'" + fileNameFull + "' is not a recognized audio file");
            }

                // set correct id for lookup, loading method and data types
                audio.setAttribute("type", "audio/" + type);
                var is_chrome = navigator.userAgent.toLowerCase().indexOf('chrome') > -1;
                if ( is_chrome )
                        audio.setAttribute("loop", "loop");
                else
                        audio.addEventListener('ended', function(){this.currentTime = 0;}, false);
                audio.volume = wa_mandreelMusicElementVolume;
                audio.setAttribute("autoplay", "true");
                audio.setAttribute("src", fileNameFull);


                // include into list and document
                document.body.appendChild(audio);
                wa_mandreelMusicElement = audio;
                wa_mandreelMusicElementFilename = fileNameFull;
    }
}

function wa_imp_mandreel_audio_stopMusic()
{
        if ( wa_mandreelMusicElement != null )
        {
                document.body.removeChild(wa_mandreelMusicElement);
                wa_mandreelMusicElement = null;
                wa_mandreelMusicElementFilename = "";
    }
}

function wa_imp_mandreel_audio_setMusicVol(vol)
{
        wa_mandreelMusicElementVolume = vol;
        if ( wa_mandreelMusicElement != null )
        {
                wa_mandreelMusicElement.volume = wa_mandreelMusicElementVolume;
    }
}

var mandreel_audio_stream_func_ptr = 0;
function mandreel_audio_stream_process(e)
{
        var l = e.outputBuffer.getChannelData(0);
        var l2 = e.outputBuffer.getChannelData(1);
        var n = e.outputBuffer.length;

        var offset2 = 0;
        var inc = 44.1 / 48.0;
        while ( n > 0 )
        {
                var n2 = ((n*inc)|0)/4;
                if ( n2 > 1024 )
                        n2 = 1024;

                var sp = g_stack_pointer+100*1024;
                var ptr = g_stack_pointer+200*1024;
                var _sp = sp>>2;
                heap32[_sp]=ptr;
                heap32[_sp+1]=n2*4;
                __FUNCTION_TABLE__[(mandreel_audio_stream_func_ptr)>>2](sp);

                var offset = ptr>>2;
                var size = n2*4;
                /*for (var i=0;i<size;++i)
                {
                        l[i+offset2] = heapFloat[offset+(i*2)];
                        l2[i+offset2] = heapFloat[offset+(i*2)+1];
            }*/
                var i = 0;
                var j = 0;
                while ( i < size )
                {
                        l[j+offset2] = heapFloat[offset+((i|0)*2)];
                        l2[j+offset2] = heapFloat[offset+((i|0)*2)+1];
                        i += inc;
                        j++;
            }

                //offset2 += n2*4;
                //n -= n2*4;
                offset2 += j;
                n -= j;
    }
}

function mandreel_Audio_requestSoundData(soundData)
{
        var n = soundData.length/2;
        var offset2 = 0;
        while ( n > 0 )
        {
                var n2 = n;
                if ( n2 > 1024 )
                        n2 = 1024;
                var sp = g_stack_pointer+100*1024;
                var ptr = g_stack_pointer+200*1024;
                var _sp = sp>>2;
                heap32[_sp]=ptr;
                heap32[_sp+1]=n2;
                __FUNCTION_TABLE__[(mandreel_audio_stream_func_ptr)>>2](sp);

                var offset = ptr>>2;
                var size = n2*2;
                var buf = heapFloat.subarray(offset,offset+size);
                soundData.set(buf,offset2);
                /*for (var i=0; i<size; i++)
                  soundData[i+offset2] = heapFloat[offset+i];*/
                offset2 += n2*2;
                n -= n2;
    }
 }

var __Mandreel_Audio_CreateStream_created = false;
function Mandreel_Audio_CreateStream(sp)
{
        if ( !__Mandreel_Audio_CreateStream_created )
        {
                if (webAudioContext)
                {
                        mandreel_audio_stream_func_ptr = heap32[sp>>2];
                        var source =  webAudioContext.createJavaScriptNode(1024*4, 0, 2);
                        source.connect(webAudioContext.destination);
                        source.onaudioprocess = mandreel_audio_stream_process;
            }
                else
                {
                        mandreel_audio_stream_func_ptr = heap32[sp>>2];
                        AudioDataDestination(44100,mandreel_Audio_requestSoundData);
            }
                __Mandreel_Audio_CreateStream_created = true;
    }
}
function mandreel_webAudio_dummyStream()
{
        var sp = g_stack_pointer+100*1024;
        var ptr = g_stack_pointer+200*1024;
        var _sp = sp>>2;
        heap32[_sp]=ptr;
        heap32[_sp+1]=4096;
        __FUNCTION_TABLE__[(mandreel_audio_stream_func_ptr)>>2](sp);
        setTimeout("mandreel_webAudio_dummyStream()",10);
}

function AudioDataDestination(sampleRate, readFn)
{
        // Initialize the audio output.
        var audio = new Audio();
        if ( audio.mozSetup == null )
        {
                setTimeout("mandreel_webAudio_dummyStream()",10);
                return;
    }
        audio.mozSetup(2, sampleRate);

        var currentWritePosition = 0;
        var prebufferSize = sampleRate / 2; // buffer 500ms
        var tail = null, tailPosition;

        // The function called with regular interval to populate
        // the audio output buffer.
        setInterval(function() {
          var written;
          // Check if some data was not written in previous attempts.
          if(tail) {
                written = audio.mozWriteAudio(tail.subarray(tailPosition));
                currentWritePosition += written;
                tailPosition += written;
                if(tailPosition < tail.length) {
                  // Not all the data was written, saving the tail...
                  return; // ... and exit the function.
            }
                tail = null;
          }

          // Check if we need add some data to the audio output.
          var currentPosition = audio.mozCurrentSampleOffset();
          var available = currentPosition + prebufferSize - currentWritePosition;
          if(available > 0) {
                // Request some sound data from the callback function.
                var soundData = new Float32Array(available);
                readFn(soundData);

                // Writting the data.
                written = audio.mozWriteAudio(soundData);
                if(written < soundData.length) {
                  // Not all the data was written, saving the tail.
                  tail = soundData;
                  tailPosition = written;
            }
                currentWritePosition += written;
          }
    }, 100);
}
var mandreel_flashaudio_server = "";
var mandreel_flashaudio_lite = false;
var mandreel_flashaudio_musicaudiotag = true;

function as3Error(str)
{
        var params = str.split(' ');
        if ( params[0] == "createdBuffer" )
                mandreel_audio_flash_lastBufferCreatedSwf = parseInt(params[1]);
        dump("as3Log: "+str);
}

function mandreel_audio_getFlashMovieObject(movieName)
{
        if (Mandreel_window.document[movieName])
        {
                return Mandreel_window.document[movieName];
    }
        if (navigator.appName.indexOf("Microsoft Internet")==-1)
        {
                if (document.embeds && document.embeds[movieName])
                        return document.embeds[movieName];
    }
        else
        {
                return document.getElementById(movieName);
    }
}

function mandreel_sendmsg_flash(msg)
{
        if ( mandreel_flashaudio_server != "" )
        {
                var iframeWin = document.getElementById("ninja-iframe").contentWindow;
                iframeWin.postMessage(msg,mandreel_flashaudio_server);
    }
        else
        {
                var flashMovie=mandreel_audio_getFlashMovieObject("FlashDivAudio");
                if ( flashMovie != null )
                        flashMovie.receiveMessage(msg);
                else
                        dump("error: flashMovie not found");
    }
}

function fl_mandreel_audio_init(sp)
{
        mandreel_sendmsg_flash("init "+g_mandreel_working_folder);
}

function fl_mandreel_audio_end(sp)
{
}

function fl_mandreel_audio_update(sp)
{
}

function mandreel_flashaudiolite_createBuffer(fileName)
{
        mandreel_audio_flash_lastBufferCreated++;
        mandreel_sendmsg_flash("createBuffer "+wa_getFileNameNoExt(fileName)+" "+mandreel_audio_flash_lastBufferCreated);
}

var mandreel_audio_flash_lastBufferCreated = 0;
function fl_mandreel_audio_createBuffer(sp)
{
        var ptr_fileName = heap32[sp>>2];sp+=4;
        var fileName = get_string_from_ptr(ptr_fileName).toLowerCase();
        mandreel_flashaudiolite_createBuffer(fileName);
}

function fl_internal_mandreel_audio_checkBuffersPending()
{
        return mandreel_flashaudio_lite && (mandreel_audio_flash_lastBufferCreatedSwf != mandreel_audio_flash_lastBufferCreated);
}

var mandreel_audio_flash_lastBufferCreatedSwf = 0;
function fl_mandreel_audio_checkBuffersPending(sp)
{
        r_g0 = 0;
        if ( fl_internal_mandreel_audio_checkBuffersPending() )
                r_g0 = 1;
        dump("fl_mandreel_audio_checkBuffersPending ("+r_g0+") ("+mandreel_audio_flash_lastBufferCreatedSwf+") ("+mandreel_audio_flash_lastBufferCreated+")");
}


function fl_mandreel_audio_playChannel(sp)
{
        var ptr_fileName = heap32[sp>>2];sp+=4;
        var channel = heap32[sp>>2];sp+=4;
        var vol = heapFloat[sp>>2];sp+=4;
        var loop = heap32[sp>>2];sp+=4;
        var pitch = heapFloat[sp>>2];sp+=4;
        var fileName = get_string_from_ptr(ptr_fileName).toLowerCase();
        mandreel_sendmsg_flash("playChannel "+fileName+" "+channel+" "+loop+" "+vol+" "+pitch);
        r_g0 = 0;
}

function fl_mandreel_audio_stopChannel(sp)
{
        var ptr_fileName = heap32[sp>>2];sp+=4;
        var channel = heap32[sp>>2];sp+=4;
        var index = heapFloat[sp>>2];sp+=4;

        mandreel_sendmsg_flash("stopChannel "+channel);
}

function fl_mandreel_audio_setChannelVolume(sp)
{
        var channel = heap32[sp>>2];sp+=4;
        var vol = heapFloat[sp>>2];sp+=4;

        mandreel_sendmsg_flash("setChannelVolume "+channel+" "+vol);
}

function fl_mandreel_audio_setChannelPan(sp)
{
        var channel = heap32[sp>>2];sp+=4;
        var pan = heapFloat[sp>>2];sp+=4;
        mandreel_sendmsg_flash("setChannelPan "+channel+" "+pan);
}

function fl_mandreel_audio_setChannelPitch(sp)
{
        var channel = heap32[sp>>2];sp+=4;
        var pitch = heapFloat[sp>>2];sp+=4;
        mandreel_sendmsg_flash("setChannelPitch "+channel+" "+pitch);
}


function mandreel_audio_load_flash()
{
        var failed = "";


        try
        {
                var mandreelAudioSwf = g_mandreel_working_folder+"mandreelaudio.swf";
                if ( mandreel_flashaudio_lite )
                        mandreelAudioSwf = g_mandreel_working_folder+"mandreelaudiolite.swf";
                var swf = swfobject.createSWF({ data:mandreelAudioSwf, width:"0", height:"0", allowScriptAccess:"always" }, { menu:"false" }, "FlashDivAudio");
                if ( !swf )
                        failed  = "swfobject.js not avaiable or Unable to open "+mandreelAudioSwf;
    }
        catch(err)
        {
                failed  = err;
    }

        if (failed == "" && !swfobject.hasFlashPlayerVersion("9.0.0"))
                failed = "flash player not found";

        if ( failed != "" )
        {
                dump("Failed to create flash audio driver ("+failed+"). Null driver will be used.");
                MandreelAudioDriver = "null";
                null_MainAudioDriver();
    }
}

function fl_MainAudioDriver(audioServer, audioUrl)
{
        mandreel_flashaudio_server = audioServer;
        if ( mandreel_flashaudio_lite )
                mandreel_flashaudio_server = "";
        if ( mandreel_flashaudio_server != "" )
        {
                Mandreel_window.addEventListener("message", receiveMessage2, false);
                var el = document.createElement("iframe");
                el.setAttribute('id', 'ninja-iframe');
                el.setAttribute('width', '0');
                el.setAttribute('height', '0');
                el.setAttribute('frameborder', '0');
                document.body.appendChild(el);
                el.onerror = function(message) { alert(message); };
                el.setAttribute('src', audioServer+audioUrl+"/MandreelAudio.html");
                setTimeout("CheckNinjaFrameReady()", 1000);
    }
        else
        {
                setTimeout("mandreel_audio_load_flash()", 10);
    }
}
var ninjaLoaded = false;
function CheckNinjaFrameReady()
{
        try
        {
                mandreel_sendmsg_flash("loadFlash");
    }
        catch(err)
        {
    }
        if ( !ninjaLoaded )
                setTimeout("CheckNinjaFrameReady()", 1000);
}

function fl_map_mandreel_audio_functions()
{
        mandreel_audio_init = fl_mandreel_audio_init;
        mandreel_audio_end = fl_mandreel_audio_end;
        mandreel_audio_update = fl_mandreel_audio_update;
        mandreel_audio_createBuffer = fl_mandreel_audio_createBuffer;
        mandreel_audio_playChannel = fl_mandreel_audio_playChannel;
        mandreel_audio_stopChannel = fl_mandreel_audio_stopChannel;
        mandreel_audio_setChannelVolume = fl_mandreel_audio_setChannelVolume;
        mandreel_audio_setChannelPan = fl_mandreel_audio_setChannelPan;
        mandreel_audio_setChannelPitch = fl_mandreel_audio_setChannelPitch;
        if ( mandreel_flashaudio_musicaudiotag )
        {
                mandreel_audio_useMusicFunctions = wa_mandreel_audio_useMusicFunctions;
                mandreel_audio_playMusic = wa_mandreel_audio_playMusic;
                mandreel_audio_stopMusic = wa_mandreel_audio_stopMusic;
                mandreel_audio_setMusicVol = wa_mandreel_audio_setMusicVol;
    }
        else
                dump("WARNING: flash music functions not supported");
        mandreel_audio_checkBuffersPending = fl_mandreel_audio_checkBuffersPending;
}

function receiveMessage2(event)
{
        ninjaLoaded = true;
        fl_map_mandreel_audio_functions();
        setTimeout("MandreelAudioStarted()", 10);
}

function mandreel_flashlite_checkPreloadFinished()
{
        if ( !fl_internal_mandreel_audio_checkBuffersPending() )
                MandreelAudioStarted();
        else
                setTimeout("mandreel_flashlite_checkPreloadFinished()", 10);
}

function mandreel_flashlite_startPreload()
{
        mandreel_sendmsg_flash("init "+g_mandreel_working_folder);
        // preload sounds
        var FatLines = mandreelFatData.split('\n');
        for ( var i=0;i<FatLines.length;++i )
        {
                var params = FatLines[i].replace('\r','').split(',');
                if ( params[0] == "audiofile" && params[1] )
                {
                        var sync_load = true;
                        if (params[2] && (params[2]&1) == 0)
                                sync_load = false;

                        if (sync_load)
                        {

                                mandreel_flashaudiolite_createBuffer(params[1]);
                    }
            }
    }
        setTimeout("mandreel_flashlite_checkPreloadFinished()", 10);
}

function flashReady()
{
        fl_map_mandreel_audio_functions();
        setTimeout("mandreel_flashlite_startPreload()",10);
}

function null_mandreel_audio(sp)
{
        r_g0 = 0;
}

function null_MainAudioDriver()
{
        mandreel_audio_init = null_mandreel_audio;
        mandreel_audio_end = null_mandreel_audio;
        mandreel_audio_update = null_mandreel_audio;
        mandreel_audio_createBuffer = null_mandreel_audio;
        mandreel_audio_playChannel = null_mandreel_audio;
        mandreel_audio_stopChannel = null_mandreel_audio;
        mandreel_audio_setChannelVolume = null_mandreel_audio;
        mandreel_audio_setChannelPan = null_mandreel_audio;
        mandreel_audio_setChannelPitch = null_mandreel_audio;
        mandreel_audio_useMusicFunctions = wa_mandreel_audio_useMusicFunctions;
        mandreel_audio_playMusic = wa_mandreel_audio_playMusic;
        mandreel_audio_stopMusic = wa_mandreel_audio_stopMusic;
        mandreel_audio_setMusicVol = wa_mandreel_audio_setMusicVol;
        setTimeout("MandreelAudioStarted()", 10);
}
function at_mandreel_audio_init(sp)
{
}

function at_mandreel_audio_end(sp)
{
}

function at_mandreel_audio_update(sp)
{
        for ( i = 0 ; i < 32 ; ++i )
        {
                var end = maudioChannelEnd[i];
                if ( end != null )
                {
                        var sound = maudiotagChannels[i];
                        if ( sound != null )
                        {
                                if ( sound.currentTime > end )
                                {
                                        sound.pause();
                                        maudioChannelEnd[i] = null;
                            }
                    }
            }
    }
}

function at_mandreel_audio_checkBuffersPending(sp)
{
        r_g0 = 0;//wa_imp_mandreel_audio_checkBuffersPending();
}

function at_mandreel_audio_createBuffer(sp)
{
        var ptr_fileName = heap32[sp>>2];sp+=4;
        var maxChannels = heap32[sp>>2];sp+=4;
        var fileName = get_string_from_ptr(ptr_fileName).toLowerCase();
        //wa_imp_mandreel_audio_createBuffer(fileName);

        r_g0 = 0;
}

function at_mandreel_audio_playChannel(sp)
{
        var ptr_fileName = heap32[sp>>2];sp+=4;
        var channel = heap32[sp>>2];sp+=4;
        var vol = heapFloat[sp>>2];sp+=4;
        var loop = heap32[sp>>2];sp+=4;
        var pitch = heapFloat[sp>>2];sp+=4;
        var fileName = get_string_from_ptr(ptr_fileName).toLowerCase();
        at_imp_mandreel_audio_playChannel(fileName,channel,vol,loop,pitch);
        r_g0 = 0;
}

function at_mandreel_audio_stopChannel(sp)
{
        var ptr_fileName = heap32[sp>>2];sp+=4;
        var channel = heap32[sp>>2];sp+=4;
        var index = heapFloat[sp>>2];sp+=4;
        at_imp_mandreel_audio_stopChannel(channel);
}

function at_mandreel_audio_setChannelVolume(sp)
{
        var channel = heap32[sp>>2];sp+=4;
        var vol = heapFloat[sp>>2];sp+=4;
        //wa_imp_mandreel_audio_setChannelVolume(channel,vol);
}

function at_mandreel_audio_setChannelPan(sp)
{
        var channel = heap32[sp>>2];sp+=4;
        var pan = heapFloat[sp>>2];sp+=4;
        //wa_imp_mandreel_audio_setChannelPan(channel,pan);
}

function at_mandreel_audio_setChannelPitch(sp)
{
        var channel = heap32[sp>>2];sp+=4;
        var pitch = heapFloat[sp>>2];sp+=4;
        //wa_imp_mandreel_audio_setChannelPitch(channel,pitch);
}

function at_mandreel_audio_useMusicFunctions(sp)
{
        r_g0 = 1;
}

function at_MainAudioDriver()
{
        mandreel_audio_init = at_mandreel_audio_init;
        mandreel_audio_end = at_mandreel_audio_end;
        mandreel_audio_update = at_mandreel_audio_update;
        mandreel_audio_createBuffer = at_mandreel_audio_createBuffer;
        mandreel_audio_playChannel = at_mandreel_audio_playChannel;
        mandreel_audio_stopChannel = at_mandreel_audio_stopChannel;
        mandreel_audio_setChannelVolume = at_mandreel_audio_setChannelVolume;
        mandreel_audio_setChannelPan = at_mandreel_audio_setChannelPan;
        mandreel_audio_setChannelPitch = at_mandreel_audio_setChannelPitch;
        mandreel_audio_useMusicFunctions = at_mandreel_audio_useMusicFunctions;
        mandreel_audio_playMusic = wa_mandreel_audio_playMusic;
        mandreel_audio_stopMusic = wa_mandreel_audio_stopMusic;
        mandreel_audio_checkBuffersPending = at_mandreel_audio_checkBuffersPending;
        mandreel_audio_setMusicVol = wa_mandreel_audio_setMusicVol;

        setTimeout("mandreel_audiotag_PreloadAssets()", 10);
}

var maudiotagPreloadState = "start";
var maudiotagPreloadAssets = 0;
var maudiotagCurrentPreloadAsset = 0;
var maudiotagPreloadAsync = false;
var maudiotagDurations = Array();
var maudiotagChannelsCreated = 0;
var maudiotagChannels = Array();
var maudiotagChannelsLoaded = Array();
var maudiotagAudioSpriteFound = false;
var maudiotagAvailableChannels = new Array();
var maudioSecondsPos = new Array();
var maudioChannelEnd = new Array();

function mandreel_audiotag_PreloadAssets()
{
        if ( maudiotagPreloadState == "start" )
        {
                maudiotagPreloadAssets = mandreelFatData.split('\n');
                maudiotagCurrentPreloadAsset = 0;
                maudiotagPreloadState = "preloading";
    }
        else if ( maudiotagPreloadState == "preloading" )
        {
                var queued = false;
                while ( !queued && maudiotagPreloadState != "done" )
                {
                        if ( maudiotagCurrentPreloadAsset < maudiotagPreloadAssets.length )
                        {
                                var params = maudiotagPreloadAssets[maudiotagCurrentPreloadAsset].split(',');
                                if ( params[0] == "audiofile" && params[1])
                                {
                                        var duration = params[3];
                                        var fileNameNoExt = wa_getFileNameNoExt(params[1]);
                                        maudiotagDurations[fileNameNoExt] = duration|0;
                                        dump("audiotag duration ("+params[1]+") "+duration);
                            }
                                else if ( params[0] == "audiotagfile" )
                                {
                                        var type = params[1];
                                        var filesize = params[3];
                                        var numsounds = params[4];
                                        var at = new Audio();
                                        if ( !maudiotagAudioSpriteFound && at.canPlayType && at.canPlayType("audio/"+type) != "" )
                                        {
                                                maudiotagLoadPackFile(params[2],filesize,numsounds,type);
                                                maudiotagAudioSpriteFound = true;
                                                queued = true;
                                    }
                            }
                                else if ( params[0] == "audiopreloadasync" )
                                {
                                        maudiotagPreloadAsync = true;
                            }
                                maudiotagCurrentPreloadAsset++;
                    }
                        else
                                queued = true;

                        if ( maudiotagCurrentPreloadAsset >= maudiotagPreloadAssets.length )
                        {
                                maudiotagPreloadState = "done";
                                maudiotagPreloadAssets = 0;
                    }
            }
    }

        if ( maudiotagPreloadState == "done" )
        {
                setTimeout("MandreelAudioStarted()", 10);
    }
        else
                setTimeout("mandreel_audiotag_PreloadAssets()", 10);
}

function maudiotagLoadPackFile(filename,filesize,numsounds,atype)
{
        dump("audiotag packfile ("+filename+") ("+filesize+") ("+numsounds+")");
        var url = g_mandreel_working_folder+filename;
        dump("audiotag: loading packed data ("+filename+") url("+url+")");
        var request = new XMLHttpRequest();
        request.open("GET", url, true);
        request.responseType = "arraybuffer";
        request.onreadystatechange = function()
        {
                if (request.readyState == 4)
                {
                        if (request.status == 404)
                                return;
                        if ( request.response != null )
                                audiotagParsePreloadFile(request.response,numsounds,atype);
            }
    }
        request.send();
}

function audiotagParsePreloadFile(response,numsounds,atype)
{
        var pos = 0;
        var bytes = new Uint8Array(response);
        var i = 0;
        if ( webAudioPreloadAsync )
                webAudioPreloadBytes = bytes;
        var secondsPos = 0.0;
        while ( pos < bytes.byteLength && i < numsounds)
        {
                // filename
                var fileName = "";
                while ( bytes[pos] != 0 )
                {
                        fileName += String.fromCharCode(bytes[pos]);
                        pos++;
            }
                pos++;
                // filesize
                var filesize = bytes[pos] | (bytes[pos+1]<<8) | (bytes[pos+2]<<16) | (bytes[pos+3]<<24);
                pos += 4;
                var fileNameNoExt = wa_getFileNameNoExt(fileName);
                var duration = maudiotagDurations[fileNameNoExt];
                dump("afile ("+fileName+") duration("+duration+") posseconds("+secondsPos+")");
                maudioSecondsPos[fileNameNoExt] = secondsPos;
                secondsPos += duration / 1000.0;
                secondsPos += 0.25;
                dump("second "+i+" "+secondsPos);

                i++;
    }

        // contents
        var contentSize = bytes.byteLength - pos;
        var bufferdata = new Uint8Array(contentSize);
        bufferdata.set(bytes.subarray(pos,pos+contentSize));
        var ascii = '';
        for (var i=0; i<bufferdata.length; i++)
                ascii += String.fromCharCode(bufferdata[i]);
        var base64 = btoa(ascii);
        audiotagAudioSprite = "data:audio/"+atype+";base64," + base64;

        if ( webAudioPreloadAsync )
        {
                dump("started preloading audio files async");
                setTimeout("mandreel_webAudio_preloadNextAudioFile()",10);
    }

        audiotagCreateChannel(0);
}

function audiotagCreateChannel(index)
{
        console.log("audiotagCreateChannel "+index);
        var sound = new Audio();
        sound.addEventListener("canplaythrough",function()
        {
                if ( maudiotagChannelsLoaded[index] == null )
                {
                        maudiotagChannelsLoaded[index] = sound;
                        maudiotagAvailableChannels.push(sound);
                        console.log("************** loaded channel "+index);
                        if ( index < 8 )
                                setTimeout("audiotagCreateChannel("+index+"+1)", 10);
            }
    }, false);
        sound.addEventListener("error",function()
        {
                console.log("************** error loading channel "+index);
    }, false);
        //sound.src = "data:audio/mp3;base64,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";
        sound.src = audiotagAudioSprite;
        sound.load();
        console.log("--audiotagCreateChannel "+index);
}

function at_imp_mandreel_audio_playChannel(filename,channel,vol,loop,pitch)
{
        at_imp_mandreel_audio_stopChannel(channel);
        var fileNameNoExt = wa_getFileNameNoExt(filename);
        var time = maudioSecondsPos[fileNameNoExt];
        if ( time != null )
        {
                console.log("play "+fileNameNoExt+" "+time);
                if ( maudiotagAvailableChannels.length > 0 )
                {
                        var sound = maudiotagAvailableChannels.pop();
                        var duration = maudiotagDurations[fileNameNoExt];
                        sound.currentTime = time;
                        console.log("OK "+sound.currentTime);
                        sound.play();
                        maudiotagChannels[channel] = sound;
                        maudioChannelEnd[channel] = time+(duration/1000.0);
            }
    }
        else
        {
                console.log("not found play "+fileNameNoExt);
    }
}

function at_imp_mandreel_audio_stopChannel(channel)
{
        var sound = maudiotagChannels[channel];
        if ( sound != null )
        {
                sound.pause();
                maudiotagAvailableChannels.push(sound);
                maudiotagChannels[channel] = null;
    }
}

var mandreel_flash_socket_callback = null

function mandreel_flash_sockets_load_flash(callback)
{
        mandreel_flash_socket_callback = callback;
        var failed = "";
        try
        {
                var mandreelSocketsSwf = g_mandreel_working_folder+"mandreelflashsockets.swf";
                var swf = swfobject.createSWF({ data:mandreelSocketsSwf, width:"0", height:"0", allowScriptAccess:"always" }, { menu:"false" }, "FlashDivSockets");
                if ( !swf )
                        failed  = "Unable to open MandreelFlashSockets.swf";
    }
        catch(err)
        {
                failed  = err;
    }
}

var js_mandreel_flash_socket_swf_loaded = false;

function js_mandreel_flash_socket_ready()
{
        js_mandreel_flash_socket_swf_loaded = true;
        if (mandreel_flash_socket_callback)
                mandreel_flash_socket_callback();
}


function Mandreel_Socket_InitLibrary(sp)
{
        //mandreel_flash_sockets_load_flash();
}


function mandreel_flash_sockets_getFlashMovieObject(movieName)
{
        if (Mandreel_window.document[movieName])
        {
                return Mandreel_window.document[movieName];
    }
        if (navigator.appName.indexOf("Microsoft Internet")==-1)
        {
                if (document.embeds && document.embeds[movieName])
                        return document.embeds[movieName];
    }
        else
        {
                return document.getElementById(movieName);
    }
}

function js_mandreel_flash_socket_onError(message)
{
        var id = parseInt(message);

        var sp = g_stack_pointer+512*1024;

        dump('on error ' + id);

        heap32[sp>>2] = id;
        mandreel_flash_tcp_onError(sp);

}
function js_mandreel_flash_socket_onConnect(message)
{
        var id = parseInt(message);

        var sp = g_stack_pointer+512*1024;

        dump('connected ' + id);

        heap32[sp>>2] = id;
        mandreel_flash_tcp_onConnect(sp);
}

function js_mandreel_flash_tcp_receive_callbak(message)
{
        var strings = message.split(' ');

        var id = parseInt(strings[0]);
        var data = strings[1];

        var sp = g_stack_pointer+512*1024;

        var data_ptr = sp + 1024;
        fill_ptr_from_string(data_ptr,data);

        heap32[sp>>2] = id;
        heap32[(sp+4)>>2] = data_ptr;
        mandreel_flash_tcp_receive_callbak(sp);
}

function js_mandreel_flash_tcp_update(sp)
{
        var id = heap32[sp>>2];sp+=4;
        var size = heap32[sp>>2];sp+=4;

        var flashMovie=mandreel_flash_sockets_getFlashMovieObject("FlashDivSockets");
        flashMovie.receiveMessage("receive " + id + " " + size);
}

function js_mandreel_flash_tcp_create(sp)
{
        var id = heap32[sp>>2];sp+=4;

        var flashMovie=mandreel_flash_sockets_getFlashMovieObject("FlashDivSockets");
        flashMovie.receiveMessage("create " + id);
}

function js_mandreel_flash_tcp_connectTo(sp)
{
        var id = heap32[sp>>2];sp+=4;
        var ptr_string = heap32[sp>>2];sp+=4;
        var port = heap32[sp>>2];sp+=4;

        var server_name = get_string_from_ptr(ptr_string);

        var flashMovie=mandreel_flash_sockets_getFlashMovieObject("FlashDivSockets");
        flashMovie.receiveMessage("connect " + id + " " + server_name + " " + port);
}

function js_mandreel_flash_tcp_close(sp)
{
        var id = heap32[sp>>2];sp+=4;


        var flashMovie=mandreel_flash_sockets_getFlashMovieObject("FlashDivSockets");
        flashMovie.receiveMessage("close " + id);
        dump("js_mandreel_flash_tcp_close " + id);
}
function js_mandreel_flash_tcp_write(sp)
{
        var id = heap32[sp>>2];sp+=4;
        var ptr = heap32[sp>>2];sp+=4;

        var message = get_string_from_ptr(ptr);

        dump('js_mandreel_flash_tcp_write ' + message);

        var flashMovie=mandreel_flash_sockets_getFlashMovieObject("FlashDivSockets");
        r_g0 = flashMovie.receiveMessage("send " + id + " " + message);
}



function js_mandreel_flash_tcp_dump(msg)
{
        dump(msg);
}

function _GLOBAL__I_Landscape02Vtx(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = _ZL10raycastBar;
        heap32[(g0)] = 8;
        r0 = r0 >> 2;
        _Znwj(i7);
        heap32[(r0+10006)] = r_g0;
        heap32[(g0)] = r_g0;
        heap32[(g0+1)] = 0;
        gettimeofday(i7);
        heap32[(r0+10001)] = 0;
        heap32[(r0+10005)] = 0;
        heap32[(r0+10004)] = 9999;
        heap32[(r0+10003)] = 0;
        heap32[(r0+10002)] = 0;
        return;
}

function _GLOBAL__D_Landscape02Vtx(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = _ZL10raycastBar;
        r0 = r0 >> 2;
        r0 = heap32[(r0+10006)];
        heap32[(g0)] = r0;
        _ZdlPv(i7);
        return;
}

function _ZN15DemoApplication6myinitEv(sp)
{
        var i7;
        var fp = sp>>2;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        return;
}

function _ZN15DemoApplication16getDynamicsWorldEv(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = r0 >> 2;
        r0 = heap32[(r0+1)];
        r_g0 = r0;
        return;
}

function _ZNK16btCollisionWorld17RayResultCallback14needsCollisionEP17btBroadphaseProxy(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = heap32[(fp+1)];
        r2 = heapU16[(r0+14)>>1];
        r3 = heapU16[(r1+4)>>1];
        r2 = r2 & r3;
        r2 = r2 & 65535;
        if(r2 ==0) //_LBB4_2
{
        r0 = 0;
        r_g0 = r0;
        return;
}
else{
        r1 = heapU16[(r1+6)>>1];
        r0 = heapU16[(r0+12)>>1];
        r0 = r1 & r0;
        r0 = r0 & 65535;
        r1 = 0;
        r0 = r0 != r1;
        r0 = r0 & 1;
        r_g0 = r0;
        return;
}
}

function _ZN16btCollisionWorld24ClosestRayResultCallbackD1Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTVN16btCollisionWorld24ClosestRayResultCallbackE;
        r0 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r0)] = r1;
        return;
}

function _ZN16btCollisionWorld24ClosestRayResultCallbackD0Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTVN16btCollisionWorld24ClosestRayResultCallbackE;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        heap32[(g0)] = r0;
        _ZdlPv(i7);
        return;
}

function _ZN16btCollisionWorld24ClosestRayResultCallback15addSingleResultERNS_14LocalRayResultEb(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var f0;
        var f1;
        var f2;
        var f3;
        var f4;
        var f5;
        var f6;
        var f7;
        var f8;
        var f9;
        var f10;
        var f11;
var __label__ = 0;
        i7 = sp + -16;var g0 = i7>>2; // save stack
        r0 = heap32[(fp+1)];
        r0 = r0 >> 2;
        r1 = heap32[(fp)];
        f0 = heapFloat[(r0+6)];
        r1 = r1 >> 2;
        f1 = heapFloat[(r1+1)];
        if(f0 <=f1) //_LBB7_2
{
        r2 = heap32[(fp+2)];
        heapFloat[(r1+1)] = f0;
        r3 = heap32[(r0)];
        heap32[(r1+2)] = r3;
        if(r2 ==0) //_LBB7_4
{
        r2 = r3 >> 2;
        f0 = heapFloat[(r0+2)];
        f1 = heapFloat[(r2+1)];
        f2 = heapFloat[(r0+3)];
        f3 = heapFloat[(r2+2)];
        f4 = heapFloat[(r2+5)];
        f5 = heapFloat[(r2+6)];
        f1 = f1*f0;
        f3 = f3*f2;
        f6 = heapFloat[(r0+4)];
        f7 = heapFloat[(r2+3)];
        f8 = heapFloat[(r2+9)];
        f9 = heapFloat[(r2+10)];
        f10 = heapFloat[(r2+11)];
        f11 = heapFloat[(r2+7)];
        f4 = f4*f0;
        f5 = f5*f2;
        f1 = f1+f3;
        f3 = f7*f6;
        f0 = f8*f0;
        f2 = f9*f2;
        f4 = f4+f5;
        f5 = f11*f6;
        f1 = f1+f3;
        f0 = f0+f2;
        f2 = f10*f6;
        f3 = f4+f5;
        heapFloat[(r1+13)] = f1;
        f0 = f0+f2;
        heapFloat[(r1+14)] = f3;
        heapFloat[(r1+15)] = f0;
        heap32[(r1+16)] = 0;
}
else{
        heap32[(r1+13)] = heap32[(r0+2)];
        heap32[(r1+14)] = heap32[(r0+3)];
        heap32[(r1+15)] = heap32[(r0+4)];
        heap32[(r1+16)] = heap32[(r0+5)];
}
        f0 =                         1;
        f1 = heapFloat[(r0+6)];
        f2 = heapFloat[(r1+9)];
        f3 = heapFloat[(r1+5)];
        f0 = f0-f1;
        f3 = f3*f0;
        f2 = f2*f1;
        f2 = f3+f2;
        heapFloat[(r1+17)] = f2;
        f2 = heapFloat[(r1+6)];
        f3 = heapFloat[(r1+10)];
        f2 = f2*f0;
        f3 = f3*f1;
        f2 = f2+f3;
        heapFloat[(r1+18)] = f2;
        f2 = heapFloat[(r1+7)];
        f3 = heapFloat[(r1+11)];
        f0 = f2*f0;
        f1 = f3*f1;
        f0 = f0+f1;
        heapFloat[(r1+19)] = f0;
        f0 = heapFloat[(r0+6)];
        f_g0 = f0;
        return;
}
else{
        r0 = _2E_str3;
        r1 = _2E_str4;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 250;
        _assert(i7);
}
}

function _ZN20btDefaultMotionStateD1Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV20btDefaultMotionState;
        r0 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r0)] = r1;
        return;
}

function _ZN20btDefaultMotionStateD0Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV20btDefaultMotionState;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        heap32[(g0)] = r0;
        _ZdlPv(i7);
        return;
}

function _ZNK20btDefaultMotionState17getWorldTransformER11btTransform(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var f0;
        var f1;
        var f2;
        var f3;
        var f4;
        var f5;
        var f6;
        var f7;
        var f8;
        var f9;
        var f10;
        var f11;
        var f12;
        var f13;
        var f14;
        var f15;
        var f16;
        var f17;
        var f18;
        var f19;
        var f20;
        var f21;
        var f22;
        var f23;
        var f24;
        var f25;
        var f26;
        var f27;
        var f28;
        var f29;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = r0 >> 2;
        f0 = heapFloat[(r0+17)];
        f1 = heapFloat[(r0+1)];
        f2 = heapFloat[(r0+21)];
        f3 = heapFloat[(r0+5)];
        f4 = heapFloat[(r0+2)];
        f5 = heapFloat[(r0+6)];
        f6 = f1*f0;
        f7 = f3*f2;
        f8 = heapFloat[(r0+25)];
        f9 = heapFloat[(r0+9)];
        f10 = heapFloat[(r0+18)];
        f11 = heapFloat[(r0+22)];
        f12 = heapFloat[(r0+19)];
        f13 = heapFloat[(r0+13)];
        f14 = heapFloat[(r0+3)];
        f15 = heapFloat[(r0+23)];
        f16 = heapFloat[(r0+30)];
        f17 = heapFloat[(r0+14)];
        f18 = heapFloat[(r0+7)];
        r1 = heap32[(fp+1)];
        f19 = heapFloat[(r0+26)];
        f20 = heapFloat[(r0+31)];
        f21 = heapFloat[(r0+27)];
        f22 = heapFloat[(r0+15)];
        f23 = heapFloat[(r0+11)];
        f24 = heapFloat[(r0+10)];
        f25 = heapFloat[(r0+29)];
        f26 = f4*f0;
        f27 = f5*f2;
        f6 = f6+f7;
        f7 = f9*f8;
        r0 = r1 >> 2;
        f28 = f14*f0;
        f29 = f18*f2;
        f26 = f26+f27;
        f27 = f24*f8;
        f6 = f6+f7;
        f7 = f28+f29;
        f28 = f23*f8;
        f26 = f26+f27;
        heapFloat[(r0)] = f6;
        f6 = f1*f10;
        f27 = f3*f11;
        f7 = f7+f28;
        heapFloat[(r0+1)] = f26;
        heapFloat[(r0+2)] = f7;
        f7 = f4*f10;
        f26 = f5*f11;
        f6 = f6+f27;
        f27 = f9*f19;
        f28 = f14*f10;
        f29 = f18*f11;
        f7 = f7+f26;
        f26 = f24*f19;
        f6 = f6+f27;
        heap32[(r0+3)] = 0;
        f27 = f28+f29;
        f28 = f23*f19;
        f7 = f7+f26;
        heapFloat[(r0+4)] = f6;
        f1 = f1*f12;
        f3 = f3*f15;
        f6 = f27+f28;
        heapFloat[(r0+5)] = f7;
        heapFloat[(r0+6)] = f6;
        f4 = f4*f12;
        f5 = f5*f15;
        f1 = f1+f3;
        f3 = f9*f21;
        f6 = -f25;
        f7 = f14*f12;
        f9 = f18*f15;
        f4 = f4+f5;
        f5 = f24*f21;
        f1 = f1+f3;
        heap32[(r0+7)] = 0;
        f3 = f0*f13;
        f14 = f2*f17;
        f0 = f0*f6;
        f2 = f2*f16;
        f7 = f7+f9;
        f9 = f23*f21;
        f4 = f4+f5;
        heapFloat[(r0+8)] = f1;
        f1 = f10*f13;
        f5 = f11*f17;
        f10 = f10*f6;
        f11 = f11*f16;
        f3 = f3+f14;
        f14 = f8*f22;
        f0 = f0-f2;
        f2 = f8*f20;
        f7 = f7+f9;
        heapFloat[(r0+9)] = f4;
        heapFloat[(r0+10)] = f7;
        f4 = f12*f13;
        f7 = f15*f17;
        f6 = f12*f6;
        f8 = f15*f16;
        f1 = f1+f5;
        f5 = f19*f22;
        f9 = f10-f11;
        f10 = f19*f20;
        f3 = f3+f14;
        f0 = f0-f2;
        f2 = f4+f7;
        f4 = f21*f22;
        f6 = f6-f8;
        f7 = f21*f20;
        f1 = f1+f5;
        f5 = f9-f10;
        f0 = f3+f0;
        heap32[(r0+11)] = 0;
        f2 = f2+f4;
        f3 = f6-f7;
        f1 = f1+f5;
        heapFloat[(r0+12)] = f0;
        f0 = f2+f3;
        heapFloat[(r0+13)] = f1;
        heapFloat[(r0+14)] = f0;
        heap32[(r0+15)] = 0;
        return;
}

function _ZN20btDefaultMotionState17setWorldTransformERK11btTransform(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var f0;
        var f1;
        var f2;
        var f3;
        var f4;
        var f5;
        var f6;
        var f7;
        var f8;
        var f9;
        var f10;
        var f11;
        var f12;
        var f13;
        var f14;
        var f15;
        var f16;
        var f17;
        var f18;
        var f19;
        var f20;
        var f21;
        var f22;
        var f23;
        var f24;
        var f25;
        var f26;
        var f27;
        var f28;
        var f29;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = heap32[(fp+1)];
        r0 = r0 >> 2;
        r1 = r1 >> 2;
        f0 = heapFloat[(r1)];
        f1 = heapFloat[(r0+17)];
        f2 = heapFloat[(r1+1)];
        f3 = heapFloat[(r0+21)];
        f4 = heapFloat[(r0+18)];
        f5 = heapFloat[(r0+22)];
        f6 = f1*f0;
        f7 = f3*f2;
        f8 = heapFloat[(r1+2)];
        f9 = heapFloat[(r0+25)];
        f10 = heapFloat[(r1+8)];
        f11 = heapFloat[(r0+29)];
        f12 = heapFloat[(r1+4)];
        f13 = heapFloat[(r0+19)];
        f14 = heapFloat[(r1+9)];
        f15 = heapFloat[(r0+30)];
        f16 = heapFloat[(r1+5)];
        f17 = heapFloat[(r0+23)];
        f18 = heapFloat[(r1+10)];
        f19 = heapFloat[(r0+31)];
        f20 = heapFloat[(r1+6)];
        f21 = heapFloat[(r0+27)];
        f22 = heapFloat[(r0+26)];
        f23 = f4*f0;
        f24 = f5*f2;
        f6 = f6+f7;
        f7 = f9*f8;
        f25 = heapFloat[(r1+14)];
        f26 = heapFloat[(r1+13)];
        f27 = heapFloat[(r1+12)];
        f28 = f13*f0;
        f29 = f17*f2;
        f23 = f23+f24;
        f24 = f22*f8;
        f6 = f6+f7;
        f7 = f28+f29;
        f28 = f21*f8;
        f23 = f23+f24;
        heapFloat[(r0+1)] = f6;
        f6 = f1*f12;
        f24 = f3*f16;
        f7 = f7+f28;
        heapFloat[(r0+2)] = f23;
        heapFloat[(r0+3)] = f7;
        f7 = f4*f12;
        f23 = f5*f16;
        f6 = f6+f24;
        f24 = f9*f20;
        f28 = f13*f12;
        f29 = f17*f16;
        f7 = f7+f23;
        f23 = f22*f20;
        f6 = f6+f24;
        heap32[(r0+4)] = 0;
        f24 = f28+f29;
        f28 = f21*f20;
        f7 = f7+f23;
        heapFloat[(r0+5)] = f6;
        f1 = f1*f10;
        f3 = f3*f14;
        f6 = f24+f28;
        heapFloat[(r0+6)] = f7;
        heapFloat[(r0+7)] = f6;
        f4 = f4*f10;
        f5 = f5*f14;
        f1 = f1+f3;
        f3 = f9*f18;
        f6 = f13*f10;
        f7 = f17*f14;
        f4 = f4+f5;
        f5 = f22*f18;
        f1 = f1+f3;
        heap32[(r0+8)] = 0;
        f0 = f0*f11;
        f2 = f2*f15;
        f3 = f6+f7;
        f6 = f21*f18;
        f4 = f4+f5;
        heapFloat[(r0+9)] = f1;
        f1 = f12*f11;
        f5 = f16*f15;
        f0 = f0+f2;
        f2 = f8*f19;
        f3 = f3+f6;
        heapFloat[(r0+10)] = f4;
        f0 = f0+f2;
        heapFloat[(r0+11)] = f3;
        f2 = f10*f11;
        f3 = f14*f15;
        f1 = f1+f5;
        f4 = f20*f19;
        f1 = f1+f4;
        f2 = f2+f3;
        f3 = f18*f19;
        f0 = f0+f27;
        heap32[(r0+12)] = 0;
        f2 = f2+f3;
        f1 = f1+f26;
        heapFloat[(r0+13)] = f0;
        f0 = f2+f25;
        heapFloat[(r0+14)] = f1;
        heapFloat[(r0+15)] = f0;
        heap32[(r0+16)] = 0;
        return;
}

function _ZN17btTypedConstraint21setupSolverConstraintER20btAlignedObjectArrayI18btSolverConstraintEiif(sp)
{
        var i7;
        var fp = sp>>2;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        return;
}

function _ZN17btTypedConstraint23solveConstraintObsoleteER11btRigidBodyS1_f(sp)
{
        var i7;
        var fp = sp>>2;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        return;
}

function _ZN13BenchmarkDemo15displayCallbackEv(sp)
{
        var i7;
        var fp = sp>>2;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        return;
}

function _ZN13BenchmarkDemo11exitPhysicsEv(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = r0 >> 2;
        r1 = heap32[(r0+9)];
if(!(r1 <1)) //_LBB15_5
{
        r1 = 0;
_3: while(true){
        r2 = heap32[(r0+11)];
        r3 = r1 << 2;
        r2 = (r2 + r3)|0;
        r2 = r2 >> 2;
        r2 = heap32[(r2)];
if(!(r2 ==0)) //_LBB15_4
{
        r3 = r2 >> 2;
        r3 = heap32[(r3)];
        r3 = r3 >> 2;
        r3 = heap32[(r3+1)];
        heap32[(g0)] = r2;
        __FUNCTION_TABLE__[(r3)>>2](i7);
}
        r1 = (r1 + 1)|0;
        r2 = heap32[(r0+9)];
        if(r2 >r1) //_LBB15_2
{
continue _3;
}
else{
break _3;
}
}
}
        r1 = heap32[(r0+1)];
        r2 = r1 >> 2;
        r2 = heap32[(r2+2)];
        r3 = (r2 + -1)|0;
_9: do {
if(!(r3 <0)) //_LBB15_16
{
        r3 = 1;
        r2 = (r3 - r2)|0;
_11: while(true){
        r3 = r1 >> 2;
        r3 = heap32[(r3+4)];
        r4 = r2 << 2;
        r3 = (r3 - r4)|0;
        r3 = r3 >> 2;
        r3 = heap32[(r3)];
        r4 = heapU8[r3+232];
        r4 = r4 & 2;
if(!(r4 ==0)) //_LBB15_9
{
        if(r3 !=0) //_LBB15_10
{
        r4 = r3 >> 2;
        r4 = heap32[(r4+118)];
if(!(r4 ==0)) //_LBB15_9
{
        r1 = r4 >> 2;
        r1 = heap32[(r1)];
        r1 = r1 >> 2;
        r1 = heap32[(r1+1)];
        heap32[(g0)] = r4;
        __FUNCTION_TABLE__[(r1)>>2](i7);
        r1 = heap32[(r0+1)];
}
}
}
        r4 = r1 >> 2;
        r4 = heap32[(r4)];
        r4 = r4 >> 2;
        r4 = heap32[(r4+9)];
        heap32[(g0)] = r1;
        heap32[(g0+1)] = r3;
        __FUNCTION_TABLE__[(r4)>>2](i7);
        if(r3 !=0) //_LBB15_15
{
        r1 = r3 >> 2;
        r1 = heap32[(r1)];
        r1 = r1 >> 2;
        r1 = heap32[(r1+2)];
        heap32[(g0)] = r3;
        __FUNCTION_TABLE__[(r1)>>2](i7);
}
        if(r2 ==0) //_LBB15_16
{
break _9;
}
else{
        r2 = (r2 + 1)|0;
        r1 = heap32[(r0+1)];
continue _11;
}
}
}
} while(0);
        r1 = heap32[(r0+4)];
_23: do {
if(!(r1 <1)) //_LBB15_21
{
        r1 = 0;
_25: while(true){
        r2 = heap32[(r0+6)];
        r3 = r1 << 2;
        r2 = (r2 + r3)|0;
        r2 = r2 >> 2;
        r2 = heap32[(r2)];
if(!(r2 ==0)) //_LBB15_20
{
        r3 = r2 >> 2;
        r3 = heap32[(r3)];
        r3 = r3 >> 2;
        r3 = heap32[(r3+1)];
        heap32[(g0)] = r2;
        __FUNCTION_TABLE__[(r3)>>2](i7);
}
        r1 = (r1 + 1)|0;
        r2 = heap32[(r0+4)];
        if(r2 >r1) //_LBB15_18
{
continue _25;
}
else{
break _23;
}
}
}
} while(0);
        r1 = heap32[(r0+1)];
if(!(r1 ==0)) //_LBB15_23
{
        r2 = r1 >> 2;
        r2 = heap32[(r2)];
        r2 = r2 >> 2;
        r2 = heap32[(r2+1)];
        heap32[(g0)] = r1;
        __FUNCTION_TABLE__[(r2)>>2](i7);
}
        r1 = heap32[(r0+15)];
if(!(r1 ==0)) //_LBB15_25
{
        r2 = r1 >> 2;
        r2 = heap32[(r2)];
        r2 = r2 >> 2;
        r2 = heap32[(r2+1)];
        heap32[(g0)] = r1;
        __FUNCTION_TABLE__[(r2)>>2](i7);
}
        r1 = heap32[(r0+13)];
if(!(r1 ==0)) //_LBB15_27
{
        r2 = r1 >> 2;
        r2 = heap32[(r2)];
        r2 = r2 >> 2;
        r2 = heap32[(r2+1)];
        heap32[(g0)] = r1;
        __FUNCTION_TABLE__[(r2)>>2](i7);
}
        r1 = heap32[(r0+14)];
if(!(r1 ==0)) //_LBB15_29
{
        r2 = r1 >> 2;
        r2 = heap32[(r2)];
        r2 = r2 >> 2;
        r2 = heap32[(r2+1)];
        heap32[(g0)] = r1;
        __FUNCTION_TABLE__[(r2)>>2](i7);
}
        r0 = heap32[(r0+16)];
if(!(r0 ==0)) //_LBB15_31
{
        r1 = r0 >> 2;
        r1 = heap32[(r1)];
        r1 = r1 >> 2;
        r1 = heap32[(r1+1)];
        heap32[(g0)] = r0;
        __FUNCTION_TABLE__[(r1)>>2](i7);
}
        return;
}

function _ZN7RagDollD0Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV7RagDoll;
        r2 = 0;
        r3 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r3)] = r1;
        r1 = r2;
_1: while(true){
        r4 = heap32[(r3+1)];
        r5 = r1 << 2;
        r6 = r4 >> 2;
        r5 = (r0 - r5)|0;
        r6 = heap32[(r6)];
        r6 = r6 >> 2;
        r5 = r5 >> 2;
        r6 = heap32[(r6+14)];
        r7 = heap32[(r5+24)];
        heap32[(g0)] = r4;
        heap32[(g0+1)] = r7;
        __FUNCTION_TABLE__[(r6)>>2](i7);
        r4 = heap32[(r5+24)];
if(!(r4 ==0)) //_LBB16_3
{
        r5 = r4 >> 2;
        r5 = heap32[(r5)];
        r5 = r5 >> 2;
        r5 = heap32[(r5+1)];
        heap32[(g0)] = r4;
        __FUNCTION_TABLE__[(r5)>>2](i7);
}
        r4 = (r2 - r1)|0;
        r4 = r4 << 2;
        r4 = (r0 + r4)|0;
        r1 = (r1 + -1)|0;
        r4 = r4 >> 2;
        heap32[(r4+24)] = 0;
        if(r1 !=-10) //_LBB16_1
{
continue _1;
}
else{
break _1;
}
}
        r1 = 2;
_7: while(true){
        r2 = heap32[(r3+1)];
        r4 = r2 >> 2;
        r5 = r1 << 2;
        r4 = heap32[(r4)];
        r5 = (r0 + r5)|0;
        r4 = r4 >> 2;
        r5 = r5 >> 2;
        r4 = heap32[(r4+21)];
        r6 = heap32[(r5+11)];
        heap32[(g0)] = r2;
        heap32[(g0+1)] = r6;
        __FUNCTION_TABLE__[(r4)>>2](i7);
        r2 = heap32[(r5+11)];
        r4 = r2 >> 2;
        r4 = heap32[(r4+118)];
        if(r4 !=0) //_LBB16_7
{
        r2 = r4 >> 2;
        r2 = heap32[(r2)];
        r2 = r2 >> 2;
        r2 = heap32[(r2+1)];
        heap32[(g0)] = r4;
        __FUNCTION_TABLE__[(r2)>>2](i7);
        r2 = heap32[(r5+11)];
}
if(!(r2 ==0)) //_LBB16_10
{
        r4 = r2 >> 2;
        r4 = heap32[(r4)];
        r4 = r4 >> 2;
        r4 = heap32[(r4+2)];
        heap32[(g0)] = r2;
        __FUNCTION_TABLE__[(r4)>>2](i7);
}
        heap32[(r5+11)] = 0;
        r2 = heap32[(r5)];
if(!(r2 ==0)) //_LBB16_12
{
        r4 = r2 >> 2;
        r4 = heap32[(r4)];
        r4 = r4 >> 2;
        r4 = heap32[(r4+1)];
        heap32[(g0)] = r2;
        __FUNCTION_TABLE__[(r4)>>2](i7);
}
        r1 = (r1 + 1)|0;
        heap32[(r5)] = 0;
        if(r1 !=13) //_LBB16_5
{
continue _7;
}
else{
break _7;
}
}
        heap32[(g0)] = r0;
        _ZdlPv(i7);
        return;
}

function _ZN7RagDollD1Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV7RagDoll;
        r2 = 0;
        r3 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r3)] = r1;
        r1 = r2;
_1: while(true){
        r4 = heap32[(r3+1)];
        r5 = r1 << 2;
        r6 = r4 >> 2;
        r5 = (r0 - r5)|0;
        r6 = heap32[(r6)];
        r6 = r6 >> 2;
        r5 = r5 >> 2;
        r6 = heap32[(r6+14)];
        r7 = heap32[(r5+24)];
        heap32[(g0)] = r4;
        heap32[(g0+1)] = r7;
        __FUNCTION_TABLE__[(r6)>>2](i7);
        r4 = heap32[(r5+24)];
if(!(r4 ==0)) //_LBB17_3
{
        r5 = r4 >> 2;
        r5 = heap32[(r5)];
        r5 = r5 >> 2;
        r5 = heap32[(r5+1)];
        heap32[(g0)] = r4;
        __FUNCTION_TABLE__[(r5)>>2](i7);
}
        r4 = (r2 - r1)|0;
        r4 = r4 << 2;
        r4 = (r0 + r4)|0;
        r1 = (r1 + -1)|0;
        r4 = r4 >> 2;
        heap32[(r4+24)] = 0;
        if(r1 !=-10) //_LBB17_1
{
continue _1;
}
else{
break _1;
}
}
        r1 = 2;
_7: while(true){
        r2 = heap32[(r3+1)];
        r4 = r2 >> 2;
        r5 = r1 << 2;
        r4 = heap32[(r4)];
        r5 = (r0 + r5)|0;
        r4 = r4 >> 2;
        r5 = r5 >> 2;
        r4 = heap32[(r4+21)];
        r6 = heap32[(r5+11)];
        heap32[(g0)] = r2;
        heap32[(g0+1)] = r6;
        __FUNCTION_TABLE__[(r4)>>2](i7);
        r2 = heap32[(r5+11)];
        r4 = r2 >> 2;
        r4 = heap32[(r4+118)];
        if(r4 !=0) //_LBB17_7
{
        r2 = r4 >> 2;
        r2 = heap32[(r2)];
        r2 = r2 >> 2;
        r2 = heap32[(r2+1)];
        heap32[(g0)] = r4;
        __FUNCTION_TABLE__[(r2)>>2](i7);
        r2 = heap32[(r5+11)];
}
if(!(r2 ==0)) //_LBB17_10
{
        r4 = r2 >> 2;
        r4 = heap32[(r4)];
        r4 = r4 >> 2;
        r4 = heap32[(r4+2)];
        heap32[(g0)] = r2;
        __FUNCTION_TABLE__[(r4)>>2](i7);
}
        heap32[(r5+11)] = 0;
        r2 = heap32[(r5)];
if(!(r2 ==0)) //_LBB17_12
{
        r4 = r2 >> 2;
        r4 = heap32[(r4)];
        r4 = r4 >> 2;
        r4 = heap32[(r4+1)];
        heap32[(g0)] = r2;
        __FUNCTION_TABLE__[(r4)>>2](i7);
}
        r1 = (r1 + 1)|0;
        heap32[(r5)] = 0;
        if(r1 !=13) //_LBB17_5
{
continue _7;
}
else{
break _7;
}
}
        return;
}

function _ZN13BenchmarkDemoD0Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV13BenchmarkDemo;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        heap32[(g0)] = r0;
        _ZN13BenchmarkDemo11exitPhysicsEv(i7);
        r1 = heap32[(r2+11)];
if(!(r1 ==0)) //_LBB18_4
{
        r3 = heapU8[r0+48];
if(!(r3 ==0)) //_LBB18_3
{
        r3 = gNumAlignedFree;
        r3 = r3 >> 2;
        r4 = heap32[(r3)];
        r4 = (r4 + 1)|0;
        r1 = r1 >> 2;
        heap32[(r3)] = r4;
        r1 = heap32[(r1+-1)];
        heap32[(g0)] = r1;
        free(i7);
}
        heap32[(r2+11)] = 0;
}
        r1 = 1;
        heap8[r0+48] = r1;
        heap32[(r2+11)] = 0;
        heap32[(r2+9)] = 0;
        heap32[(r2+10)] = 0;
        r3 = heap32[(r2+6)];
if(!(r3 ==0)) //_LBB18_8
{
        r4 = heapU8[r0+28];
if(!(r4 ==0)) //_LBB18_7
{
        r4 = gNumAlignedFree;
        r4 = r4 >> 2;
        r5 = heap32[(r4)];
        r5 = (r5 + 1)|0;
        r3 = r3 >> 2;
        heap32[(r4)] = r5;
        r3 = heap32[(r3+-1)];
        heap32[(g0)] = r3;
        free(i7);
}
        heap32[(r2+6)] = 0;
}
        heap8[r0+28] = r1;
        heap32[(r2+6)] = 0;
        heap32[(r2+4)] = 0;
        heap32[(r2+5)] = 0;
        heap32[(g0)] = r0;
        _ZdlPv(i7);
        return;
}

function _ZN15DemoApplication20localCreateRigidBodyEfRK11btTransformP16btCollisionShape(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var f0;
        var f1;
var __label__ = 0;
        i7 = sp + -168;var g0 = i7>>2; // save stack
        r0 = heap32[(fp+3)];
if(!(r0 ==0)) //_LBB19_3
{
        r1 = r0 >> 2;
        r1 = heap32[(r1+1)];
if(!(r1 !=35)) //_LBB19_3
{
        r0 = _2E_str5;
        r1 = _2E_str6;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 1297;
        _assert(i7);
}
}
        f0 = heapFloat[(fp+1)];
        r1 = sp + -152;
        r2 = r1 >> 2;
        heap32[(fp+-38)] = 0;
        heap32[(r2+1)] = 0;
        heap32[(r2+2)] = 0;
        f1 =                         0;
        heap32[(r2+3)] = 0;
if(!(f0 ==f1)) //_LBB19_5
{
        r3 = r0 >> 2;
        r3 = heap32[(r3)];
        r3 = r3 >> 2;
        r3 = heap32[(r3+8)];
        heap32[(g0)] = r0;
        heapFloat[(g0+1)] = f0;
        heap32[(g0+2)] = r1;
        __FUNCTION_TABLE__[(r3)>>2](i7);
}
        r1 = heap32[(fp)];
        r3 = heap32[(fp+2)];
        r4 = gNumAlignedAllocs;
        r4 = r4 >> 2;
        r5 = heap32[(r4)];
        r5 = (r5 + 1)|0;
        heap32[(r4)] = r5;
        heap32[(g0)] = 627;
        malloc(i7);
        r4 = r_g0;
        if(r4 !=0) //_LBB19_7
{
        r5 = 0;
        r6 = (r4 + 4)|0;
        r5 = (r5 - r6)|0;
        r5 = r5 & 15;
        r5 = (r4 + r5)|0;
        r6 = (r5 + 4)|0;
        r5 = r5 >> 2;
        heap32[(r5)] = r4;
        r4 = r6;
}
        r5 = r4 >> 2;
        heap32[(r5+41)] = 1065353216;
        heap32[(r5+42)] = 1065353216;
        heap32[(r5+43)] = 1065353216;
        heap32[(r5+44)] = 0;
        heap32[(r5+45)] = 0;
        heap32[(r5+46)] = 1566444395;
        heap32[(r5+47)] = 0;
        heap32[(r5+48)] = 0;
        heap32[(r5+49)] = 0;
        heap32[(r5+50)] = 0;
        heap32[(r5+51)] = 1;
        heap32[(r5+52)] = -1;
        heap32[(r5+53)] = -1;
        heap32[(r5+54)] = 1;
        heap32[(r5+55)] = 0;
        heap32[(r5+56)] = 1056964608;
        heap32[(r5+57)] = 0;
        heap32[(r5+58)] = 1;
        heap32[(r5+59)] = 0;
        heap32[(r5+60)] = 1065353216;
        heap32[(r5+61)] = 0;
        heap32[(r5+62)] = 0;
        heap32[(r5+63)] = 0;
        heap32[(r5+1)] = 1065353216;
        heap32[(r5+2)] = 0;
        heap32[(r5+3)] = 0;
        heap32[(r5+4)] = 0;
        heap32[(r5+5)] = 0;
        heap32[(r5+6)] = 1065353216;
        heap32[(r5+7)] = 0;
        heap32[(r5+8)] = 0;
        heap32[(r5+9)] = 0;
        heap32[(r5+10)] = 0;
        heap32[(r5+11)] = 1065353216;
        heap32[(r5+12)] = 0;
        heap32[(r5+13)] = 0;
        heap32[(r5+14)] = 0;
        r6 = _ZTV11btRigidBody;
        heap32[(r5+15)] = 0;
        r6 = (r6 + 8)|0;
        heap32[(r5+16)] = 0;
        r7 = 1;
        heap32[(r5)] = r6;
        heap8[r4+492] = r7;
        heap32[(r5+122)] = 0;
        heap32[(r5+120)] = 0;
        r6 = sp + -136;
        heap32[(r5+121)] = 0;
        r7 = r6 >> 2;
        heapFloat[(fp+-34)] = f0;
        heap32[(r7+1)] = 0;
        heap32[(r7+18)] = r0;
        heap32[(r7+19)] = heap32[(fp+-38)];
        heap32[(r7+20)] = heap32[(r2+1)];
        heap32[(r7+21)] = heap32[(r2+2)];
        heap32[(r7+22)] = heap32[(r2+3)];
        heap32[(r7+23)] = 0;
        heap32[(r7+24)] = 0;
        heap32[(r7+25)] = 1056964608;
        heap32[(r7+26)] = 0;
        heap32[(r7+27)] = 1061997773;
        r0 = 0;
        heap32[(r7+28)] = 1065353216;
        heap8[sp+-20] = r0;
        heap32[(r7+30)] = 1000593162;
        heap32[(r7+31)] = 1008981770;
        heap32[(r7+32)] = 1008981770;
        heap32[(r7+33)] = 1008981770;
        heap32[(r7+2)] = 1065353216;
        heap32[(r7+3)] = 0;
        heap32[(r7+4)] = 0;
        heap32[(r7+5)] = 0;
        heap32[(r7+6)] = 0;
        heap32[(r7+7)] = 1065353216;
        heap32[(r7+8)] = 0;
        heap32[(r7+9)] = 0;
        heap32[(r7+10)] = 0;
        heap32[(r7+11)] = 0;
        heap32[(r7+12)] = 1065353216;
        heap32[(r7+13)] = 0;
        heap32[(r7+14)] = 0;
        heap32[(r7+15)] = 0;
        heap32[(r7+16)] = 0;
        heap32[(r7+17)] = 0;
        heap32[(g0)] = r4;
        heap32[(g0+1)] = r6;
        _ZN11btRigidBody14setupRigidBodyERKNS_27btRigidBodyConstructionInfoE(i7);
        r0 = r3 >> 2;
        heap32[(r5+1)] = heap32[(r0)];
        heap32[(r5+2)] = heap32[(r0+1)];
        heap32[(r5+3)] = heap32[(r0+2)];
        heap32[(r5+4)] = heap32[(r0+3)];
        heap32[(r5+5)] = heap32[(r0+4)];
        heap32[(r5+6)] = heap32[(r0+5)];
        heap32[(r5+7)] = heap32[(r0+6)];
        heap32[(r5+8)] = heap32[(r0+7)];
        heap32[(r5+9)] = heap32[(r0+8)];
        heap32[(r5+10)] = heap32[(r0+9)];
        heap32[(r5+11)] = heap32[(r0+10)];
        heap32[(r5+12)] = heap32[(r0+11)];
        heap32[(r5+13)] = heap32[(r0+12)];
        heap32[(r5+14)] = heap32[(r0+13)];
        heap32[(r5+15)] = heap32[(r0+14)];
        r1 = r1 >> 2;
        heap32[(r5+16)] = heap32[(r0+15)];
        heap32[(r5+46)] = heap32[(r1+2)];
        r0 = heap32[(r1+1)];
        r1 = r0 >> 2;
        r1 = heap32[(r1)];
        r1 = r1 >> 2;
        r1 = heap32[(r1+20)];
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r4;
        __FUNCTION_TABLE__[(r1)>>2](i7);
        r_g0 = r4;
        return;
}

function _ZN13BenchmarkDemoD1Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV13BenchmarkDemo;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        heap32[(g0)] = r0;
        _ZN13BenchmarkDemo11exitPhysicsEv(i7);
        r1 = heap32[(r2+11)];
if(!(r1 ==0)) //_LBB20_4
{
        r3 = heapU8[r0+48];
if(!(r3 ==0)) //_LBB20_3
{
        r3 = gNumAlignedFree;
        r3 = r3 >> 2;
        r4 = heap32[(r3)];
        r4 = (r4 + 1)|0;
        r1 = r1 >> 2;
        heap32[(r3)] = r4;
        r1 = heap32[(r1+-1)];
        heap32[(g0)] = r1;
        free(i7);
}
        heap32[(r2+11)] = 0;
}
        r1 = 1;
        heap8[r0+48] = r1;
        heap32[(r2+11)] = 0;
        heap32[(r2+9)] = 0;
        heap32[(r2+10)] = 0;
        r3 = heap32[(r2+6)];
if(!(r3 ==0)) //_LBB20_8
{
        r4 = heapU8[r0+28];
if(!(r4 ==0)) //_LBB20_7
{
        r4 = gNumAlignedFree;
        r4 = r4 >> 2;
        r5 = heap32[(r4)];
        r5 = (r5 + 1)|0;
        r3 = r3 >> 2;
        heap32[(r4)] = r5;
        r3 = heap32[(r3+-1)];
        heap32[(g0)] = r3;
        free(i7);
}
        heap32[(r2+6)] = 0;
}
        heap8[r0+28] = r1;
        heap32[(r2+6)] = 0;
        heap32[(r2+4)] = 0;
        heap32[(r2+5)] = 0;
        return;
}

function _ZN13BenchmarkDemo20clientMoveAndDisplayEv(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var f0;
        var f1;
        var f2;
var __label__ = 0;
        i7 = sp + -128;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = r0 >> 2;
        r1 = heap32[(r0+1)];
if(!(r1 ==0)) //_LBB21_2
{
        r2 = r1 >> 2;
        r2 = heap32[(r2)];
        r2 = r2 >> 2;
        r2 = heap32[(r2+12)];
        heap32[(g0)] = r1;
        heap32[(g0+1)] = 1015580809;
        heap32[(g0+2)] = 1;
        heap32[(g0+3)] = 1015580809;
        __FUNCTION_TABLE__[(r2)>>2](i7);
        r1 = heap32[(r0+1)];
        r2 = r1 >> 2;
        r2 = heap32[(r2)];
        r2 = r2 >> 2;
        r2 = heap32[(r2+5)];
        heap32[(g0)] = r1;
        __FUNCTION_TABLE__[(r2)>>2](i7);
}
        r1 = heap32[(r0+17)];
if(!(r1 !=7)) //_LBB21_10
{
        r1 = _ZL10raycastBar;
        r2 = r1 >> 2;
        r0 = heap32[(r0+1)];
        r3 = heap32[(r2+10006)];
        heap32[(g0)] = r3;
        heap32[(g0+1)] = 0;
        r3 = 0;
        gettimeofday(i7);
_6: while(true){
        r4 = sp + -96;
        r5 = r4 >> 2;
        heap32[(r5+1)] = 1065353216;
        r6 = 1;
        heap32[(r5+2)] = 0;
        r7 = -1;
        heap16[(sp+-84)>>1] = r6;
        r6 = _ZTVN16btCollisionWorld24ClosestRayResultCallbackE;
        heap16[(sp+-82)>>1] = r7;
        r7 = (r1 + r3)|0;
        r6 = (r6 + 8)|0;
        heap32[(r5+4)] = 0;
        r8 = r7 >> 2;
        heap32[(fp+-24)] = r6;
        heap32[(r5+5)] = heap32[(r8)];
        heap32[(r5+6)] = heap32[(r8+1)];
        heap32[(r5+7)] = heap32[(r8+2)];
        heap32[(r5+8)] = heap32[(r8+3)];
        heap32[(r5+9)] = heap32[(r8+2000)];
        heap32[(r5+10)] = heap32[(r8+2001)];
        heap32[(r5+11)] = heap32[(r8+2002)];
        r9 = r0 >> 2;
        heap32[(r5+12)] = heap32[(r8+2003)];
        r9 = heap32[(r9)];
        r9 = r9 >> 2;
        r9 = heap32[(r9+7)];
        r10 = (r7 + 8000)|0;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r7;
        heap32[(g0+2)] = r10;
        heap32[(g0+3)] = r4;
        __FUNCTION_TABLE__[(r9)>>2](i7);
        r4 = heap32[(r5+2)];
        if(r4 ==0) //_LBB21_6
{
        r4 = r7 >> 2;
        r5 = r7 >> 2;
        r9 = r7 >> 2;
        r10 = r7 >> 2;
        heap32[(r4+6000)] = heap32[(r5+2000)];
        r4 = r7 >> 2;
        r5 = r7 >> 2;
        heap32[(r9+6001)] = heap32[(r10+2001)];
        r9 = r7 >> 2;
        r10 = r7 >> 2;
        heap32[(r4+6002)] = heap32[(r5+2002)];
        r4 = r7 >> 2;
        heap32[(r9+6003)] = heap32[(r10+2003)];
        r5 = r7 >> 2;
        heap32[(r4+8000)] = 1065353216;
        r4 = r7 >> 2;
        heap32[(r5+8001)] = 0;
        heap32[(r4+8002)] = 0;
        heap32[(r8+8003)] = 0;
}
else{
        r4 = r7 >> 2;
        r9 = r7 >> 2;
        heap32[(r4+6000)] = heap32[(r5+17)];
        r4 = r7 >> 2;
        heap32[(r9+6001)] = heap32[(r5+18)];
        r9 = r7 >> 2;
        heap32[(r4+6002)] = heap32[(r5+19)];
        heap32[(r9+6003)] = heap32[(r5+20)];
        f0 = heapFloat[(r5+13)];
        r4 = r7 >> 2;
        heapFloat[(r4+8000)] = f0;
        f1 = heapFloat[(r5+14)];
        r9 = r7 >> 2;
        heapFloat[(r9+8001)] = f1;
        f2 = heapFloat[(r5+15)];
        r7 = r7 >> 2;
        f0 = f0*f0;
        f1 = f1*f1;
        heapFloat[(r7+8002)] = f2;
        heap32[(r8+8003)] = heap32[(r5+16)];
        f0 = f0+f1;
        f1 = f2*f2;
        f0 = f0+f1;
        heapFloat[(g0)] = f0;
        sqrtf(i7);
        f1 =                         1;
        f0 = f1/f_g0;
        f1 = heapFloat[(r4+8000)];
        f1 = f1*f0;
        heapFloat[(r4+8000)] = f1;
        f1 = heapFloat[(r9+8001)];
        f1 = f1*f0;
        heapFloat[(r9+8001)] = f1;
        f1 = heapFloat[(r7+8002)];
        f0 = f1*f0;
        heapFloat[(r7+8002)] = f0;
}
        r3 = (r3 + 16)|0;
        heap32[(fp+-24)] = r6;
if(!(r3 !=8000)) //_LBB21_4
{
break _6;
}
}
        r0 = heap32[(r2+10001)];
        r1 = sp + -8;
        heap32[(g0)] = r1;
        heap32[(g0+1)] = 0;
        gettimeofday(i7);
        r3 = heap32[(r2+10006)];
        r3 = r3 >> 2;
        r1 = r1 >> 2;
        r4 = heap32[(fp+-2)];
        r5 = heap32[(r3)];
        r1 = heap32[(r1+1)];
        r3 = heap32[(r3+1)];
        r1 = (r1 - r3)|0;
        r3 = (r4 - r5)|0;
        r1 = (r1 / 1000)|0;
        r3 = (r3 * 1000)|0;
        r1 = (r1 + r3)|0;
        r0 = (r1 + r0)|0;
        heap32[(r2+10001)] = r0;
        r1 = heap32[(r2+10000)];
        r1 = (r1 + 1)|0;
        heap32[(r2+10000)] = r1;
if(!(r1 <51)) //_LBB21_10
{
        r3 = heap32[(r2+10004)];
        r3 = r0 < r3 ? r0 : r3;
        heap32[(r2+10004)] = r3;
        r4 = heap32[(r2+10005)];
        r4 = r0 > r4 ? r0 : r4;
        heap32[(r2+10005)] = r4;
        r5 = heap32[(r2+10002)];
        r5 = (r0 + r5)|0;
        heap32[(r2+10002)] = r5;
        r6 = heap32[(r2+10003)];
        r6 = (r6 + 1)|0;
        f0 = r5; //fitos r5, f0
        f1 = r6; //fitos r6, f1
        f0 = f0/f1;
        heap32[(r2+10003)] = r6;
        r5 = _2E_str7;
        r1 = (r1 * 500)|0;
        f0 = f0; //fstod f0, f0
        heap32[(g0)] = r5;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = r0;
        heap32[(g0+3)] = r3;
        heap32[(g0+4)] = r4;
        llvm_writeDouble((i7+24),f0);
        printf(i7);
        heap32[(r2+10001)] = 0;
        heap32[(r2+10000)] = 0;
}
}
        return;
}

function _ZN13BenchmarkDemo10createWallERK9btVector3iS2_(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var f0;
        var f1;
        var f2;
        var f3;
        var f4;
        var f5;
        var f6;
        var f7;
        var f8;
        var f9;
var __label__ = 0;
        i7 = sp + -96;var g0 = i7>>2; // save stack
        r0 = heap32[(fp+1)];
        r1 = gNumAlignedAllocs;
        r0 = r0 >> 2;
        r1 = r1 >> 2;
        r2 = heap32[(r1)];
        f0 = heapFloat[(r0+2)];
        f1 = heapFloat[(r0+1)];
        f2 = heapFloat[(r0)];
        r2 = (r2 + 1)|0;
        heap32[(r1)] = r2;
        heap32[(g0)] = 71;
        malloc(i7);
        r1 = r_g0;
        r2 = heap32[(fp)];
        if(r1 !=0) //_LBB22_2
{
        r3 = 0;
        r4 = (r1 + 4)|0;
        r3 = (r3 - r4)|0;
        r3 = r3 & 15;
        r3 = (r1 + r3)|0;
        r4 = (r3 + 4)|0;
        r3 = r3 >> 2;
        heap32[(r3)] = r1;
        r1 = r4;
}
        r3 = r1 >> 2;
        heap32[(r3+2)] = 0;
        heap32[(r3+3)] = 1065353216;
        heap32[(r3+4)] = 1065353216;
        heap32[(r3+5)] = 1065353216;
        r4 = _ZTV10btBoxShape;
        heap32[(r3+6)] = 0;
        r4 = (r4 + 8)|0;
        heap32[(r3+11)] = 1025758986;
        heap32[(r3)] = r4;
        f3 =     -0.039999999105930328;
        f2 = f2+f3;
        heap32[(r3+1)] = 0;
        f1 = f1+f3;
        heapFloat[(r3+7)] = f2;
        f0 = f0+f3;
        heapFloat[(r3+8)] = f1;
        heapFloat[(r3+9)] = f0;
        r4 = sp + -16;
        heap32[(r3+10)] = 0;
        r3 = r4 >> 2;
        heap32[(fp+-4)] = 0;
        heap32[(r3+1)] = 0;
        heap32[(r3+2)] = 0;
        heap32[(r3+3)] = 0;
        heap32[(g0)] = r1;
        heap32[(g0+1)] = 1065353216;
        heap32[(g0+2)] = r4;
        _ZNK10btBoxShape21calculateLocalInertiaEfR9btVector3(i7);
        f4 = heapFloat[(r0+1)];
        f0 = heapFloat[(r0+2)];
        r0 = sp + -80;
        r3 = r0 >> 2;
        heap32[(fp+-20)] = 1065353216;
        heap32[(r3+1)] = 0;
        heap32[(r3+2)] = 0;
        heap32[(r3+3)] = 0;
        heap32[(r3+4)] = 0;
        heap32[(r3+5)] = 1065353216;
        heap32[(r3+6)] = 0;
        heap32[(r3+7)] = 0;
        heap32[(r3+8)] = 0;
        heap32[(r3+9)] = 0;
        heap32[(r3+10)] = 1065353216;
        heap32[(r3+11)] = 0;
        f1 = f0+f0;
        f2 =                       -12;
        heap32[(r3+12)] = 0;
        heap32[(r3+13)] = 0;
        f2 = f1*f2;
        f3 =                       0.5;
        f3 = f2*f3;
        f2 = f4+f4;
        r5 = 12;
        heap32[(r3+14)] = 0;
        heap32[(r3+15)] = 0;
_4: while(true){
if(!(r5 <1)) //_LBB22_4
{
        r4 = 0;
_8: while(true){
        r6 = r2 >> 2;
        f5 = r4; //fitos r4, f5
        f6 =                         0;
        f7 = heapFloat[(r6)];
        f8 = heapFloat[(r6+2)];
        f9 = heapFloat[(r6+1)];
        f5 = f5*f1;
        f6 = f7+f6;
        f5 = f5+f3;
        f7 = f9+f4;
        heapFloat[(r3+12)] = f6;
        f5 = f8+f5;
        heapFloat[(r3+13)] = f7;
        r6 = _ZL14benchmarkDemo4;
        heapFloat[(r3+14)] = f5;
        r7 = r6 >> 2;
        heap32[(r3+15)] = 0;
        r7 = heap32[(r7)];
        r7 = r7 >> 2;
        r7 = heap32[(r7+2)];
        heap32[(g0)] = r6;
        heap32[(g0+1)] = 1065353216;
        heap32[(g0+2)] = r0;
        heap32[(g0+3)] = r1;
        r4 = (r4 + 1)|0;
        __FUNCTION_TABLE__[(r7)>>2](i7);
        if(r5 !=r4) //_LBB22_3
{
continue _8;
}
else{
break _8;
}
}
}
        r5 = (r5 + -1)|0;
        f4 = f4+f2;
        f3 = f3+f0;
        if(r5 ==0) //_LBB22_8
{
break _4;
}
else{
continue _4;
}
}
        return;
}

function _ZN13BenchmarkDemo19createLargeMeshBodyEv(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var r12;
        var r13;
        var r14;
var __label__ = 0;
        i7 = sp + -80;var g0 = i7>>2; // save stack
        r0 = sp + -64;
        r1 = r0 >> 2;
        heap32[(fp+-16)] = 1065353216;
        heap32[(r1+1)] = 0;
        heap32[(r1+2)] = 0;
        heap32[(r1+3)] = 0;
        heap32[(r1+4)] = 0;
        heap32[(r1+5)] = 1065353216;
        heap32[(r1+6)] = 0;
        heap32[(r1+7)] = 0;
        heap32[(r1+8)] = 0;
        heap32[(r1+9)] = 0;
        heap32[(r1+10)] = 1065353216;
        heap32[(r1+11)] = 0;
        heap32[(r1+12)] = 0;
        heap32[(r1+13)] = 0;
        r2 = 0;
        heap32[(r1+14)] = 0;
        heap32[(r1+15)] = 0;
_1: while(true){
        r3 = gNumAlignedAllocs;
        r3 = r3 >> 2;
        r4 = heap32[(r3)];
        r5 = (r4 + 1)|0;
        heap32[(r3)] = r5;
        heap32[(g0)] = 103;
        malloc(i7);
        r5 = r_g0;
        if(r5 !=0) //_LBB23_3
{
        r6 = 0;
        r7 = (r5 + 4)|0;
        r6 = (r6 - r7)|0;
        r6 = r6 & 15;
        r6 = (r5 + r6)|0;
        r7 = (r6 + 4)|0;
        r6 = r6 >> 2;
        heap32[(r6)] = r5;
        r5 = r7;
}
        r6 = r5 >> 2;
        heap32[(r6+1)] = 1065353216;
        heap32[(r6+2)] = 1065353216;
        r7 = _ZTV26btTriangleIndexVertexArray;
        heap32[(r6+3)] = 1065353216;
        r7 = (r7 + 8)|0;
        heap32[(r6+4)] = 0;
        r8 = 1;
        heap32[(r6)] = r7;
        heap8[r5+36] = r8;
        heap32[(r6+8)] = 0;
        heap32[(r6+6)] = 0;
        r7 = LandscapeVtx;
        r9 = r2 << 2;
        r10 = LandscapeVtxCount;
        r11 = LandscapeIdx;
        r12 = LandscapeIdxCount;
        r7 = (r7 + r9)|0;
        r10 = (r10 + r9)|0;
        r11 = (r11 + r9)|0;
        r9 = (r12 + r9)|0;
        heap32[(r6+7)] = 0;
        r7 = r7 >> 2;
        heap32[(r6+12)] = 0;
        r10 = r10 >> 2;
        r11 = r11 >> 2;
        r9 = r9 >> 2;
        r7 = heap32[(r7)];
        r10 = heap32[(r10)];
        r11 = heap32[(r11)];
        r9 = heap32[(r9)];
        r12 = (r4 + 2)|0;
        heap32[(r3)] = r12;
        heap32[(g0)] = 51;
        malloc(i7);
        r12 = r_g0;
        r9 = (r9 / 3)|0;
        if(r12 !=0) //_LBB23_6
{
        r13 = 0;
        r14 = (r12 + 4)|0;
        r13 = (r13 - r14)|0;
        r13 = r13 & 15;
        r13 = (r12 + r13)|0;
        r14 = (r13 + 4)|0;
        r13 = r13 >> 2;
        heap32[(r13)] = r12;
        r12 = r14;
}
        heap8[r5+36] = r8;
        heap32[(r6+8)] = r12;
        heap32[(r6+7)] = 1;
        r13 = heap32[(r6+6)];
        r13 = r13 << 5;
        r12 = (r12 + r13)|0;
        r12 = r12 >> 2;
        heap32[(r12)] = r9;
        heap32[(r12+1)] = r11;
        heap32[(r12+2)] = 6;
        heap32[(r12+3)] = r10;
        heap32[(r12+4)] = r7;
        heap32[(r12+5)] = 12;
        heap32[(r12+6)] = 3;
        heap32[(r12+7)] = 0;
        r7 = heap32[(r6+6)];
        r9 = (r7 + 1)|0;
        heap32[(r6+6)] = r9;
        r7 = r7 << 5;
        r9 = heap32[(r6+8)];
        r7 = (r9 + r7)|0;
        r7 = r7 >> 2;
        r4 = (r4 + 3)|0;
        heap32[(r7+6)] = 3;
        heap32[(r3)] = r4;
        heap32[(g0)] = 95;
        malloc(i7);
        r3 = r_g0;
        if(r3 !=0) //_LBB23_9
{
        r4 = 0;
        r7 = (r3 + 4)|0;
        r4 = (r4 - r7)|0;
        r4 = r4 & 15;
        r4 = (r3 + r4)|0;
        r7 = (r4 + 4)|0;
        r4 = r4 >> 2;
        heap32[(r4)] = r3;
        r3 = r7;
}
        r4 = r3 >> 2;
        r7 = _ZTV19btTriangleMeshShape;
        heap32[(r4+2)] = 0;
        r7 = (r7 + 8)|0;
        heap32[(r4+3)] = 0;
        heap32[(r4)] = r7;
        heap32[(r4+12)] = r5;
        heap32[(r4+1)] = 21;
        r7 = heap32[(r6)];
        r7 = r7 >> 2;
        r7 = heap32[(r7+10)];
        heap32[(g0)] = r5;
        __FUNCTION_TABLE__[(r7)>>2](i7);
        r7 = r_g0;
        if(r7 ==0) //_LBB23_12
{
        heap32[(g0)] = r3;
        _ZN19btTriangleMeshShape15recalcLocalAabbEv(i7);
}
else{
        r6 = heap32[(r6)];
        r6 = r6 >> 2;
        r6 = heap32[(r6+12)];
        r7 = (r3 + 16)|0;
        r9 = (r3 + 32)|0;
        heap32[(g0)] = r5;
        heap32[(g0+1)] = r7;
        heap32[(g0+2)] = r9;
        __FUNCTION_TABLE__[(r6)>>2](i7);
}
        r5 = _ZTV22btBvhTriangleMeshShape;
        r5 = (r5 + 8)|0;
        heap32[(r4)] = r5;
        heap32[(r4+13)] = 0;
        heap32[(r4+14)] = 0;
        r5 = 0;
        heap8[r3+60] = r8;
        heap8[r3+61] = r5;
        heap32[(r4+1)] = 21;
        heap32[(g0)] = r3;
        _ZN22btBvhTriangleMeshShape17buildOptimizedBvhEv(i7);
        heap32[(r1+12)] = 0;
        heap32[(r1+13)] = -1043857408;
        r4 = _ZL14benchmarkDemo4;
        heap32[(r1+14)] = 0;
        r5 = r4 >> 2;
        heap32[(r1+15)] = 0;
        r5 = heap32[(r5)];
        r5 = r5 >> 2;
        r5 = heap32[(r5+2)];
        heap32[(g0)] = r4;
        heap32[(g0+1)] = 0;
        heap32[(g0+2)] = r0;
        heap32[(g0+3)] = r3;
        __FUNCTION_TABLE__[(r5)>>2](i7);
        r2 = (r2 + 1)|0;
        r3 = r_g0 >> 2;
        heap32[(r3+56)] = 1063675494;
        if(r2 !=8) //_LBB23_1
{
continue _1;
}
else{
break _1;
}
}
        return;
}

function _ZN13BenchmarkDemo11createTest6Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var f0;
        var f1;
        var f2;
        var f3;
        var f4;
        var f5;
        var f6;
        var f7;
        var f8;
        var f9;
var __label__ = 0;
        i7 = sp + -112;var g0 = i7>>2; // save stack
        r0 = gNumAlignedAllocs;
        r0 = r0 >> 2;
        r1 = heap32[(r0)];
        r1 = (r1 + 1)|0;
        heap32[(r0)] = r1;
        heap32[(g0)] = 127;
        malloc(i7);
        r0 = r_g0;
        if(r0 !=0) //_LBB24_2
{
        r1 = 0;
        r2 = (r0 + 4)|0;
        r1 = (r1 - r2)|0;
        r1 = r1 & 15;
        r1 = (r0 + r1)|0;
        r2 = (r1 + 4)|0;
        r1 = r1 >> 2;
        heap32[(r1)] = r0;
        r0 = r2;
}
        heap32[(g0)] = r0;
        heap32[(g0+1)] = 0;
        heap32[(g0+2)] = 0;
        r1 = 0;
        _ZN17btConvexHullShapeC1EPKfii(i7);
_4: while(true){
        r2 = (r1 * -3)|0;
        r3 = _ZL7TaruVtx;
        r2 = r2 << 2;
        r2 = (r3 + r2)|0;
        r3 = sp + -96;
        r2 = r2 >> 2;
        r4 = r3 >> 2;
        heap32[(fp+-24)] = heap32[(r2)];
        heap32[(r4+1)] = heap32[(r2+1)];
        heap32[(r4+2)] = heap32[(r2+2)];
        heap32[(r4+3)] = 0;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r3;
        r1 = (r1 + -1)|0;
        _ZN17btConvexHullShape8addPointERK9btVector3(i7);
        if(r1 ==-43) //_LBB24_10
{
break _4;
}
else{
continue _4;
}
}
        r1 = sp + -64;
        r2 = r1 >> 2;
        heap32[(fp+-16)] = 1065353216;
        heap32[(r2+1)] = 0;
        heap32[(r2+2)] = 0;
        heap32[(r2+3)] = 0;
        heap32[(r2+4)] = 0;
        heap32[(r2+5)] = 1065353216;
        heap32[(r2+6)] = 0;
        heap32[(r2+7)] = 0;
        heap32[(r2+8)] = 0;
        heap32[(r2+9)] = 0;
        heap32[(r2+10)] = 1065353216;
        heap32[(r2+11)] = 0;
        heap32[(r2+12)] = 0;
        heap32[(r2+13)] = 0;
        heap32[(r2+14)] = 0;
        r3 = sp + -80;
        heap32[(r2+15)] = 0;
        r4 = r3 >> 2;
        heap32[(fp+-20)] = 0;
        heap32[(r4+1)] = 0;
        heap32[(r4+2)] = 0;
        r5 = r0 >> 2;
        heap32[(r4+3)] = 0;
        r4 = heap32[(r5)];
        r4 = r4 >> 2;
        r4 = heap32[(r4+8)];
        heap32[(g0)] = r0;
        heap32[(g0+1)] = 1065353216;
        heap32[(g0+2)] = r3;
        r3 = 10;
        f0 =                        20;
        f1 =                         2;
        f2 =                       -25;
        __FUNCTION_TABLE__[(r4)>>2](i7);
        r7 = _ZL14benchmarkDemo4;
_7: while(true){
        f3 =                         3;
        f4 =                        25;
        f5 = f1+f3;
        f4 = f0+f4;
        r4 = 0;
        r5 = r4;
_9: while(true){
        f6 = r5; //fitos r5, f6
        f6 = f6*f5;
        f6 = f6+f2;
        f7 =                         5;
        f6 = f6*f7;
        f8 =                         0;
        f6 = f6+f8;
        r6 = r4;
_11: while(true){
        f9 = r6; //fitos r6, f9
        f9 = f9*f5;
        f9 = f9+f2;
        f9 = f9*f7;
        f9 = f9+f8;
        heapFloat[(r2+12)] = f9;
        heapFloat[(r2+13)] = f4;
        heapFloat[(r2+14)] = f6;
        r8 = r7 >> 2;
        heap32[(r2+15)] = 0;
        r8 = heap32[(r8)];
        r8 = r8 >> 2;
        r8 = heap32[(r8+2)];
        heap32[(g0)] = r7;
        heap32[(g0+1)] = 1065353216;
        heap32[(g0+2)] = r1;
        heap32[(g0+3)] = r0;
        r6 = (r6 + 1)|0;
        __FUNCTION_TABLE__[(r8)>>2](i7);
        if(r6 !=10) //_LBB24_7
{
continue _11;
}
else{
break _11;
}
}
        r5 = (r5 + 1)|0;
        if(r5 !=10) //_LBB24_6
{
continue _9;
}
else{
break _9;
}
}
        f4 =        1.1000000238418579;
        f5 =      -0.05000000074505806;
        f4 = f1*f4;
        f1 = f1*f5;
        f5 =                         9;
        f3 = f4+f3;
        f1 = f1*f5;
        r3 = (r3 + -1)|0;
        f0 = f0+f3;
        f2 = f1+f2;
        f1 = f4;
        if(r3 ==0) //_LBB24_11
{
break _7;
}
else{
continue _7;
}
}
        _ZN13BenchmarkDemo19createLargeMeshBodyEv(i7);
        return;
}

function _ZN7RagDoll20localCreateRigidBodyEfRK11btTransformP16btCollisionShape(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
var __label__ = 0;
        i7 = sp + -168;var g0 = i7>>2; // save stack
        r0 = sp + -16;
        r1 = r0 >> 2;
        heap32[(fp+-4)] = 0;
        heap32[(r1+1)] = 0;
        r2 = heap32[(fp+2)];
        heap32[(r1+2)] = 0;
        r3 = r2 >> 2;
        heap32[(r1+3)] = 0;
        r3 = heap32[(r3)];
        r3 = r3 >> 2;
        r3 = heap32[(r3+8)];
        heap32[(g0)] = r2;
        heap32[(g0+1)] = 1065353216;
        heap32[(g0+2)] = r0;
        r0 = _ZGVZN11btTransform11getIdentityEvE17identityTransform;
        __FUNCTION_TABLE__[(r3)>>2](i7);
        r3 = heapU8[r0];
if(!(r3 !=0)) //_LBB25_4
{
        r3 = _ZGVZN11btMatrix3x311getIdentityEvE14identityMatrix;
        r4 = heapU8[r3];
if(!(r4 !=0)) //_LBB25_3
{
        r4 = _ZZN11btMatrix3x311getIdentityEvE14identityMatrix_2E_0_2E_0_2E_0_2E_0;
        r5 = _ZZN11btMatrix3x311getIdentityEvE14identityMatrix_2E_0_2E_1_2E_0_2E_1;
        r4 = r4 >> 2;
        r6 = _ZZN11btMatrix3x311getIdentityEvE14identityMatrix_2E_0_2E_2_2E_0_2E_2;
        r5 = r5 >> 2;
        heap32[(r4)] = 1065353216;
        r4 = r6 >> 2;
        heap32[(r5)] = 1065353216;
        r5 = 1;
        heap32[(r4)] = 1065353216;
        heap8[r3] = r5;
}
        r3 = _ZZN11btTransform11getIdentityEvE17identityTransform;
        r4 = _ZZN11btMatrix3x311getIdentityEvE14identityMatrix_2E_0_2E_0_2E_0_2E_0;
        r3 = r3 >> 2;
        r4 = r4 >> 2;
        heap32[(r3)] = heap32[(r4)];
        heap32[(r3+1)] = 0;
        heap32[(r3+2)] = 0;
        r4 = _ZZN11btMatrix3x311getIdentityEvE14identityMatrix_2E_0_2E_1_2E_0_2E_1;
        heap32[(r3+3)] = 0;
        r4 = r4 >> 2;
        heap32[(r3+4)] = 0;
        heap32[(r3+5)] = heap32[(r4)];
        heap32[(r3+6)] = 0;
        heap32[(r3+7)] = 0;
        r4 = _ZZN11btMatrix3x311getIdentityEvE14identityMatrix_2E_0_2E_2_2E_0_2E_2;
        heap32[(r3+8)] = 0;
        r4 = r4 >> 2;
        heap32[(r3+9)] = 0;
        heap32[(r3+10)] = heap32[(r4)];
        heap32[(r3+11)] = 0;
        heap32[(r3+12)] = 0;
        heap32[(r3+13)] = 0;
        heap32[(r3+14)] = 0;
        r4 = 1;
        heap32[(r3+15)] = 0;
        heap8[r0] = r4;
}
        r0 = heap32[(fp)];
        r3 = heap32[(fp+1)];
        heap32[(g0)] = 200;
        r4 = _ZTV20btDefaultMotionState;
        _Znwj(i7);
        r6 = r_g0 >> 2;
        r4 = (r4 + 8)|0;
        r3 = r3 >> 2;
        heap32[(r6)] = r4;
        heap32[(r6+1)] = heap32[(r3)];
        heap32[(r6+2)] = heap32[(r3+1)];
        heap32[(r6+3)] = heap32[(r3+2)];
        heap32[(r6+4)] = heap32[(r3+3)];
        heap32[(r6+5)] = heap32[(r3+4)];
        heap32[(r6+6)] = heap32[(r3+5)];
        heap32[(r6+7)] = heap32[(r3+6)];
        heap32[(r6+8)] = heap32[(r3+7)];
        heap32[(r6+9)] = heap32[(r3+8)];
        heap32[(r6+10)] = heap32[(r3+9)];
        heap32[(r6+11)] = heap32[(r3+10)];
        heap32[(r6+12)] = heap32[(r3+11)];
        heap32[(r6+13)] = heap32[(r3+12)];
        heap32[(r6+14)] = heap32[(r3+13)];
        r4 = _ZZN11btTransform11getIdentityEvE17identityTransform;
        heap32[(r6+15)] = heap32[(r3+14)];
        r4 = r4 >> 2;
        heap32[(r6+16)] = heap32[(r3+15)];
        heap32[(r6+17)] = heap32[(r4)];
        heap32[(r6+18)] = heap32[(r4+1)];
        heap32[(r6+19)] = heap32[(r4+2)];
        heap32[(r6+20)] = heap32[(r4+3)];
        heap32[(r6+21)] = heap32[(r4+4)];
        heap32[(r6+22)] = heap32[(r4+5)];
        heap32[(r6+23)] = heap32[(r4+6)];
        heap32[(r6+24)] = heap32[(r4+7)];
        heap32[(r6+25)] = heap32[(r4+8)];
        heap32[(r6+26)] = heap32[(r4+9)];
        heap32[(r6+27)] = heap32[(r4+10)];
        heap32[(r6+28)] = heap32[(r4+11)];
        heap32[(r6+29)] = heap32[(r4+12)];
        heap32[(r6+30)] = heap32[(r4+13)];
        heap32[(r6+31)] = heap32[(r4+14)];
        heap32[(r6+32)] = heap32[(r4+15)];
        heap32[(r6+33)] = heap32[(r3)];
        heap32[(r6+34)] = heap32[(r3+1)];
        heap32[(r6+35)] = heap32[(r3+2)];
        heap32[(r6+36)] = heap32[(r3+3)];
        heap32[(r6+37)] = heap32[(r3+4)];
        heap32[(r6+38)] = heap32[(r3+5)];
        heap32[(r6+39)] = heap32[(r3+6)];
        heap32[(r6+40)] = heap32[(r3+7)];
        heap32[(r6+41)] = heap32[(r3+8)];
        heap32[(r6+42)] = heap32[(r3+9)];
        heap32[(r6+43)] = heap32[(r3+10)];
        heap32[(r6+44)] = heap32[(r3+11)];
        heap32[(r6+45)] = heap32[(r3+12)];
        heap32[(r6+46)] = heap32[(r3+13)];
        heap32[(r6+47)] = heap32[(r3+14)];
        heap32[(r6+48)] = heap32[(r3+15)];
        r3 = sp + -152;
        heap32[(r6+49)] = 0;
        r4 = r3 >> 2;
        heap32[(fp+-38)] = 1065353216;
        heap32[(r4+1)] = r_g0;
        heap32[(r4+18)] = r2;
        heap32[(r4+19)] = heap32[(fp+-4)];
        heap32[(r4+20)] = heap32[(r1+1)];
        heap32[(r4+21)] = heap32[(r1+2)];
        heap32[(r4+22)] = heap32[(r1+3)];
        heap32[(r4+23)] = 0;
        heap32[(r4+24)] = 0;
        heap32[(r4+25)] = 1056964608;
        heap32[(r4+26)] = 0;
        heap32[(r4+27)] = 1061997773;
        r1 = 0;
        heap32[(r4+28)] = 1065353216;
        heap8[sp+-36] = r1;
        heap32[(r4+30)] = 1000593162;
        heap32[(r4+31)] = 1008981770;
        heap32[(r4+32)] = 1008981770;
        heap32[(r4+33)] = 1008981770;
        heap32[(r4+2)] = 1065353216;
        heap32[(r4+3)] = 0;
        heap32[(r4+4)] = 0;
        heap32[(r4+5)] = 0;
        heap32[(r4+6)] = 0;
        heap32[(r4+7)] = 1065353216;
        heap32[(r4+8)] = 0;
        heap32[(r4+9)] = 0;
        heap32[(r4+10)] = 0;
        heap32[(r4+11)] = 0;
        heap32[(r4+12)] = 1065353216;
        heap32[(r4+13)] = 0;
        heap32[(r4+14)] = 0;
        heap32[(r4+15)] = 0;
        r2 = gNumAlignedAllocs;
        heap32[(r4+16)] = 0;
        r2 = r2 >> 2;
        heap32[(r4+17)] = 0;
        r4 = heap32[(r2)];
        r4 = (r4 + 1)|0;
        heap32[(r2)] = r4;
        heap32[(g0)] = 627;
        malloc(i7);
        r2 = r_g0;
        if(r2 !=0) //_LBB25_6
{
        r4 = (r2 + 4)|0;
        r1 = (r1 - r4)|0;
        r1 = r1 & 15;
        r1 = (r2 + r1)|0;
        r4 = (r1 + 4)|0;
        r1 = r1 >> 2;
        heap32[(r1)] = r2;
        r2 = r4;
}
        r1 = r2 >> 2;
        heap32[(r1+41)] = 1065353216;
        heap32[(r1+42)] = 1065353216;
        heap32[(r1+43)] = 1065353216;
        heap32[(r1+44)] = 0;
        heap32[(r1+45)] = 0;
        heap32[(r1+46)] = 1566444395;
        heap32[(r1+47)] = 0;
        heap32[(r1+48)] = 0;
        heap32[(r1+49)] = 0;
        heap32[(r1+50)] = 0;
        heap32[(r1+51)] = 1;
        heap32[(r1+52)] = -1;
        heap32[(r1+53)] = -1;
        heap32[(r1+54)] = 1;
        heap32[(r1+55)] = 0;
        heap32[(r1+56)] = 1056964608;
        heap32[(r1+57)] = 0;
        heap32[(r1+58)] = 1;
        heap32[(r1+59)] = 0;
        heap32[(r1+60)] = 1065353216;
        heap32[(r1+61)] = 0;
        heap32[(r1+62)] = 0;
        heap32[(r1+63)] = 0;
        heap32[(r1+1)] = 1065353216;
        heap32[(r1+2)] = 0;
        heap32[(r1+3)] = 0;
        heap32[(r1+4)] = 0;
        heap32[(r1+5)] = 0;
        heap32[(r1+6)] = 1065353216;
        heap32[(r1+7)] = 0;
        heap32[(r1+8)] = 0;
        heap32[(r1+9)] = 0;
        heap32[(r1+10)] = 0;
        heap32[(r1+11)] = 1065353216;
        heap32[(r1+12)] = 0;
        heap32[(r1+13)] = 0;
        heap32[(r1+14)] = 0;
        r4 = _ZTV11btRigidBody;
        heap32[(r1+15)] = 0;
        r4 = (r4 + 8)|0;
        heap32[(r1+16)] = 0;
        r5 = 1;
        heap32[(r1)] = r4;
        heap8[r2+492] = r5;
        heap32[(r1+122)] = 0;
        heap32[(r1+120)] = 0;
        heap32[(r1+121)] = 0;
        heap32[(g0)] = r2;
        heap32[(g0+1)] = r3;
        _ZN11btRigidBody14setupRigidBodyERKNS_27btRigidBodyConstructionInfoE(i7);
        r0 = r0 >> 2;
        r0 = heap32[(r0+1)];
        r1 = r0 >> 2;
        r1 = heap32[(r1)];
        r1 = r1 >> 2;
        r1 = heap32[(r1+20)];
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r2;
        __FUNCTION_TABLE__[(r1)>>2](i7);
        r_g0 = r2;
        return;
}

function _ZN13BenchmarkDemo11initPhysicsEv(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var r12;
        var r13;
        var r14;
        var r15;
        var f0;
        var f1;
        var f2;
        var f3;
        var f4;
        var f5;
        var f6;
        var f7;
        var f8;
        var f9;
        var f10;
        var f11;
        var f12;
        var f13;
        var f14;
        var f15;
        var f16;
        var f17;
        var f18;
        var f19;
        var f20;
        var f21;
var __label__ = 0;
        i7 = sp + -41432;var g0 = i7>>2; // save stack
        heap32[(g0)] = 88;
        _Znwj(i7);
        r0 = r_g0;
        r1 = _ZTV31btDefaultCollisionConfiguration;
        r2 = gNumAlignedAllocs;
        r3 = r0 >> 2;
        r1 = (r1 + 8)|0;
        r2 = r2 >> 2;
        heap32[(r3)] = r1;
        r1 = heap32[(r2)];
        r3 = (r1 + 1)|0;
        heap32[(r2)] = r3;
        heap32[(g0)] = 379;
        malloc(i7);
        r3 = r_g0;
        if(r3 !=0) //_LBB26_2
{
        r4 = 0;
        r5 = (r3 + 4)|0;
        r4 = (r4 - r5)|0;
        r4 = r4 & 15;
        r4 = (r3 + r4)|0;
        r5 = (r4 + 4)|0;
        r4 = r4 >> 2;
        heap32[(r4)] = r3;
        r3 = r5;
}
        r4 = r3 >> 2;
        heap32[(r4+77)] = 953267991;
        r4 = heapU8[r3+332];
        r4 = r4 & 240;
        r5 = r0 >> 2;
        heap8[r3+332] = r4;
        r4 = (r1 + 2)|0;
        heap32[(r5+8)] = r3;
        heap32[(r2)] = r4;
        heap32[(g0)] = 23;
        malloc(i7);
        r3 = r_g0;
        if(r3 !=0) //_LBB26_5
{
        r4 = 0;
        r6 = (r3 + 4)|0;
        r4 = (r4 - r6)|0;
        r4 = r4 & 15;
        r4 = (r3 + r4)|0;
        r6 = (r4 + 4)|0;
        r4 = r4 >> 2;
        heap32[(r4)] = r3;
        r3 = r6;
}
        r4 = _ZTV30btGjkEpaPenetrationDepthSolver;
        r6 = r3 >> 2;
        r4 = (r4 + 8)|0;
        heap32[(r6)] = r4;
        r4 = (r1 + 3)|0;
        heap32[(r5+9)] = r3;
        heap32[(r2)] = r4;
        heap32[(g0)] = 43;
        malloc(i7);
        r4 = r_g0;
        if(r4 !=0) //_LBB26_8
{
        r3 = 0;
        r6 = (r4 + 4)|0;
        r3 = (r3 - r6)|0;
        r3 = r3 & 15;
        r3 = (r4 + r3)|0;
        r6 = r3 >> 2;
        heap32[(r6)] = r4;
        r4 = (r3 + 4)|0;
        r3 = heap32[(r5+9)];
}
        r6 = _ZTVN23btConvexConvexAlgorithm10CreateFuncE;
        r7 = heap32[(r5+8)];
        r8 = 0;
        r9 = r4 >> 2;
        r6 = (r6 + 8)|0;
        heap8[r4+4] = r8;
        heap32[(r9)] = r6;
        heap32[(r9+4)] = 0;
        heap32[(r9+5)] = 3;
        heap32[(r9+3)] = r7;
        heap32[(r9+2)] = r3;
        r3 = (r1 + 4)|0;
        heap32[(r5+10)] = r4;
        heap32[(r2)] = r3;
        heap32[(g0)] = 27;
        malloc(i7);
        r3 = r_g0;
        if(r3 !=0) //_LBB26_11
{
        r4 = (r3 + 4)|0;
        r4 = (r8 - r4)|0;
        r4 = r4 & 15;
        r4 = (r3 + r4)|0;
        r6 = (r4 + 4)|0;
        r4 = r4 >> 2;
        heap32[(r4)] = r3;
        r3 = r6;
}
        r4 = _ZTVN33btConvexConcaveCollisionAlgorithm10CreateFuncE;
        r6 = r3 >> 2;
        r4 = (r4 + 8)|0;
        heap8[r3+4] = r8;
        heap32[(r6)] = r4;
        r4 = (r1 + 5)|0;
        heap32[(r5+11)] = r3;
        heap32[(r2)] = r4;
        heap32[(g0)] = 27;
        malloc(i7);
        r3 = r_g0;
        if(r3 !=0) //_LBB26_14
{
        r4 = (r3 + 4)|0;
        r4 = (r8 - r4)|0;
        r4 = r4 & 15;
        r4 = (r3 + r4)|0;
        r6 = (r4 + 4)|0;
        r4 = r4 >> 2;
        heap32[(r4)] = r3;
        r3 = r6;
}
        r4 = _ZTVN33btConvexConcaveCollisionAlgorithm17SwappedCreateFuncE;
        r6 = r3 >> 2;
        r4 = (r4 + 8)|0;
        heap8[r3+4] = r8;
        heap32[(r6)] = r4;
        r4 = (r1 + 6)|0;
        heap32[(r5+12)] = r3;
        heap32[(r2)] = r4;
        heap32[(g0)] = 27;
        malloc(i7);
        r3 = r_g0;
        if(r3 !=0) //_LBB26_17
{
        r4 = (r3 + 4)|0;
        r4 = (r8 - r4)|0;
        r4 = r4 & 15;
        r4 = (r3 + r4)|0;
        r6 = (r4 + 4)|0;
        r4 = r4 >> 2;
        heap32[(r4)] = r3;
        r3 = r6;
}
        r4 = _ZTVN28btCompoundCollisionAlgorithm10CreateFuncE;
        r6 = r3 >> 2;
        r4 = (r4 + 8)|0;
        heap8[r3+4] = r8;
        heap32[(r6)] = r4;
        r4 = (r1 + 7)|0;
        heap32[(r5+13)] = r3;
        heap32[(r2)] = r4;
        heap32[(g0)] = 27;
        malloc(i7);
        r3 = r_g0;
        if(r3 !=0) //_LBB26_20
{
        r4 = (r3 + 4)|0;
        r4 = (r8 - r4)|0;
        r4 = r4 & 15;
        r4 = (r3 + r4)|0;
        r6 = (r4 + 4)|0;
        r4 = r4 >> 2;
        heap32[(r4)] = r3;
        r3 = r6;
}
        r4 = _ZTVN28btCompoundCollisionAlgorithm17SwappedCreateFuncE;
        r6 = r3 >> 2;
        r4 = (r4 + 8)|0;
        heap8[r3+4] = r8;
        heap32[(r6)] = r4;
        r4 = (r1 + 8)|0;
        heap32[(r5+14)] = r3;
        heap32[(r2)] = r4;
        heap32[(g0)] = 27;
        malloc(i7);
        r3 = r_g0;
        if(r3 !=0) //_LBB26_23
{
        r4 = (r3 + 4)|0;
        r4 = (r8 - r4)|0;
        r4 = r4 & 15;
        r4 = (r3 + r4)|0;
        r6 = (r4 + 4)|0;
        r4 = r4 >> 2;
        heap32[(r4)] = r3;
        r3 = r6;
}
        r4 = _ZTVN16btEmptyAlgorithm10CreateFuncE;
        r6 = r3 >> 2;
        r4 = (r4 + 8)|0;
        heap8[r3+4] = r8;
        heap32[(r6)] = r4;
        r4 = (r1 + 9)|0;
        heap32[(r5+15)] = r3;
        heap32[(r2)] = r4;
        heap32[(g0)] = 27;
        malloc(i7);
        r3 = r_g0;
        if(r3 !=0) //_LBB26_26
{
        r4 = (r3 + 4)|0;
        r4 = (r8 - r4)|0;
        r4 = r4 & 15;
        r4 = (r3 + r4)|0;
        r6 = (r4 + 4)|0;
        r4 = r4 >> 2;
        heap32[(r4)] = r3;
        r3 = r6;
}
        r4 = _ZTVN32btSphereSphereCollisionAlgorithm10CreateFuncE;
        r6 = r3 >> 2;
        r4 = (r4 + 8)|0;
        heap8[r3+4] = r8;
        heap32[(r6)] = r4;
        r4 = (r1 + 10)|0;
        heap32[(r5+16)] = r3;
        heap32[(r2)] = r4;
        heap32[(g0)] = 27;
        malloc(i7);
        r3 = r_g0;
        if(r3 !=0) //_LBB26_29
{
        r4 = (r3 + 4)|0;
        r4 = (r8 - r4)|0;
        r4 = r4 & 15;
        r4 = (r3 + r4)|0;
        r6 = (r4 + 4)|0;
        r4 = r4 >> 2;
        heap32[(r4)] = r3;
        r3 = r6;
}
        r4 = _ZTVN34btSphereTriangleCollisionAlgorithm10CreateFuncE;
        r6 = r3 >> 2;
        r4 = (r4 + 8)|0;
        heap8[r3+4] = r8;
        heap32[(r6)] = r4;
        r6 = (r1 + 11)|0;
        heap32[(r5+18)] = r3;
        heap32[(r2)] = r6;
        heap32[(g0)] = 27;
        malloc(i7);
        r3 = r_g0;
        if(r3 !=0) //_LBB26_32
{
        r6 = (r3 + 4)|0;
        r6 = (r8 - r6)|0;
        r6 = r6 & 15;
        r6 = (r3 + r6)|0;
        r7 = (r6 + 4)|0;
        r6 = r6 >> 2;
        heap32[(r6)] = r3;
        r3 = r7;
}
        r6 = r3 >> 2;
        heap8[r3+4] = r8;
        heap32[(r6)] = r4;
        r4 = 1;
        heap32[(r5+19)] = r3;
        r6 = (r1 + 12)|0;
        heap8[r3+4] = r4;
        heap32[(r2)] = r6;
        heap32[(g0)] = 27;
        malloc(i7);
        r3 = r_g0;
        if(r3 !=0) //_LBB26_35
{
        r6 = (r3 + 4)|0;
        r6 = (r8 - r6)|0;
        r6 = r6 & 15;
        r6 = (r3 + r6)|0;
        r7 = (r6 + 4)|0;
        r6 = r6 >> 2;
        heap32[(r6)] = r3;
        r3 = r7;
}
        r6 = _ZTVN26btBoxBoxCollisionAlgorithm10CreateFuncE;
        r7 = r3 >> 2;
        r6 = (r6 + 8)|0;
        heap8[r3+4] = r8;
        heap32[(r7)] = r6;
        r6 = (r1 + 13)|0;
        heap32[(r5+17)] = r3;
        heap32[(r2)] = r6;
        heap32[(g0)] = 35;
        malloc(i7);
        r3 = r_g0;
        if(r3 !=0) //_LBB26_38
{
        r6 = (r3 + 4)|0;
        r6 = (r8 - r6)|0;
        r6 = r6 & 15;
        r6 = (r3 + r6)|0;
        r7 = (r6 + 4)|0;
        r6 = r6 >> 2;
        heap32[(r6)] = r3;
        r3 = r7;
}
        r6 = _ZTVN31btConvexPlaneCollisionAlgorithm10CreateFuncE;
        r7 = r3 >> 2;
        r6 = (r6 + 8)|0;
        heap8[r3+4] = r8;
        heap32[(r7)] = r6;
        heap32[(r7+2)] = 1;
        heap32[(r7+3)] = 1;
        r7 = (r1 + 14)|0;
        heap32[(r5+21)] = r3;
        heap32[(r2)] = r7;
        heap32[(g0)] = 35;
        malloc(i7);
        r3 = r_g0;
        if(r3 !=0) //_LBB26_41
{
        r7 = (r3 + 4)|0;
        r7 = (r8 - r7)|0;
        r7 = r7 & 15;
        r7 = (r3 + r7)|0;
        r9 = (r7 + 4)|0;
        r7 = r7 >> 2;
        heap32[(r7)] = r3;
        r3 = r9;
}
        r7 = r3 >> 2;
        heap8[r3+4] = r8;
        heap32[(r7)] = r6;
        heap32[(r7+2)] = 1;
        heap32[(r7+3)] = 1;
        heap32[(r5+20)] = r3;
        heap8[r3+4] = r4;
        r3 = (r1 + 15)|0;
        heap8[r0+12] = r4;
        heap32[(r2)] = r3;
        heap32[(g0)] = 39;
        malloc(i7);
        r3 = r_g0;
        if(r3 !=0) //_LBB26_44
{
        r6 = (r3 + 4)|0;
        r6 = (r8 - r6)|0;
        r6 = r6 & 15;
        r6 = (r3 + r6)|0;
        r7 = (r6 + 4)|0;
        r6 = r6 >> 2;
        heap32[(r6)] = r3;
        r3 = r7;
}
        r6 = r3 >> 2;
        heap32[(r6+1)] = 0;
        heap32[(r6+3)] = 0;
        heap8[r3+16] = r8;
        heap32[(r6)] = 0;
        r7 = (r1 + 16)|0;
        heap32[(r6+2)] = 0;
        heap32[(r2)] = r7;
        heap32[(g0)] = 19;
        malloc(i7);
        r7 = r_g0;
        if(r7 !=0) //_LBB26_47
{
        r9 = (r7 + 4)|0;
        r9 = (r8 - r9)|0;
        r9 = r9 & 15;
        r9 = (r7 + r9)|0;
        r10 = (r9 + 4)|0;
        r9 = r9 >> 2;
        heap32[(r9)] = r7;
        r7 = r10;
}
        heap32[(r6)] = r7;
        heap32[(r6+1)] = 0;
        heap32[(r5+2)] = r3;
        r3 = (r1 + 17)|0;
        heap8[r0+20] = r4;
        heap32[(r2)] = r3;
        heap32[(g0)] = 39;
        malloc(i7);
        r3 = r_g0;
        if(r3 !=0) //_LBB26_50
{
        r6 = (r3 + 4)|0;
        r6 = (r8 - r6)|0;
        r6 = r6 & 15;
        r6 = (r3 + r6)|0;
        r7 = (r6 + 4)|0;
        r6 = r6 >> 2;
        heap32[(r6)] = r3;
        r3 = r7;
}
        r6 = r3 >> 2;
        heap32[(r6)] = 1140;
        r7 = (r1 + 18)|0;
        heap32[(r6+1)] = 4096;
        heap32[(r2)] = r7;
        heap32[(g0)] = 4669459;
        malloc(i7);
        r7 = r_g0;
        if(r7 !=0) //_LBB26_53
{
        r9 = (r7 + 4)|0;
        r9 = (r8 - r9)|0;
        r9 = r9 & 15;
        r9 = (r7 + r9)|0;
        r10 = (r9 + 4)|0;
        r9 = r9 >> 2;
        heap32[(r9)] = r7;
        r7 = r10;
}
        heap32[(r6+4)] = r7;
        r9 = 4095;
        r10 = 1140;
        heap32[(r6+3)] = r7;
        heap32[(r6+2)] = 4096;
        r11 = r7;
_55: while(true){
        r7 = r7 >> 2;
        r10 = (r11 + r10)|0;
        heap32[(r7)] = r10;
        r10 = heap32[(r6)];
        r7 = (r11 + r10)|0;
        r9 = (r9 + -1)|0;
        r11 = r7;
        if(r9 !=0) //_LBB26_55
{
continue _55;
}
else{
break _55;
}
}
        r6 = r7 >> 2;
        heap32[(r6)] = 0;
        heap32[(r5+4)] = r3;
        r3 = (r1 + 19)|0;
        heap8[r0+28] = r4;
        heap32[(r2)] = r3;
        heap32[(g0)] = 39;
        malloc(i7);
        r3 = r_g0;
        if(r3 !=0) //_LBB26_58
{
        r6 = (r3 + 4)|0;
        r6 = (r8 - r6)|0;
        r6 = r6 & 15;
        r6 = (r3 + r6)|0;
        r7 = (r6 + 4)|0;
        r6 = r6 >> 2;
        heap32[(r6)] = r3;
        r3 = r7;
}
        r6 = r3 >> 2;
        heap32[(r6)] = 80;
        r1 = (r1 + 20)|0;
        heap32[(r6+1)] = 4096;
        heap32[(r2)] = r1;
        heap32[(g0)] = 327699;
        malloc(i7);
        r1 = r_g0;
        if(r1 !=0) //_LBB26_61
{
        r7 = (r1 + 4)|0;
        r7 = (r8 - r7)|0;
        r7 = r7 & 15;
        r7 = (r1 + r7)|0;
        r9 = (r7 + 4)|0;
        r7 = r7 >> 2;
        heap32[(r7)] = r1;
        r1 = r9;
}
        heap32[(r6+4)] = r1;
        r7 = 4095;
        r9 = 80;
        heap32[(r6+3)] = r1;
        heap32[(r6+2)] = 4096;
        r10 = r1;
_64: while(true){
        r1 = r1 >> 2;
        r9 = (r10 + r9)|0;
        heap32[(r1)] = r9;
        r9 = heap32[(r6)];
        r1 = (r10 + r9)|0;
        r7 = (r7 + -1)|0;
        r10 = r1;
        if(r7 !=0) //_LBB26_63
{
continue _64;
}
else{
break _64;
}
}
        r1 = r1 >> 2;
        r6 = _ZL14benchmarkDemo4;
        heap32[(r1)] = 0;
        r1 = r6 >> 2;
        heap32[(r5+6)] = r3;
        heap32[(r1+16)] = r0;
        heap32[(g0)] = 5388;
        _Znwj(i7);
        r0 = r_g0;
        r3 = _ZTV21btCollisionDispatcher;
        r5 = heap32[(r1+16)];
        r3 = (r3 + 8)|0;
        r7 = r0 >> 2;
        heap32[(r7)] = r3;
        heap32[(r7+1)] = 2;
        heap8[r0+24] = r4;
        heap32[(r7+5)] = 0;
        r3 = _ZTV16btManifoldResult;
        heap32[(r7+3)] = 0;
        r3 = (r3 + 8)|0;
        heap32[(r7+4)] = 0;
        heap32[(r7+7)] = r3;
        r3 = _ZN21btCollisionDispatcher19defaultNearCallbackER16btBroadphasePairRS_RK16btDispatcherInfo__index__;
        heap32[(r7+1346)] = r5;
        heap32[(r7+47)] = r3;
        r3 = r5 >> 2;
        r9 = heap32[(r3)];
        r9 = r9 >> 2;
        r9 = heap32[(r9+3)];
        heap32[(g0)] = r5;
        __FUNCTION_TABLE__[(r9)>>2](i7);
        heap32[(r7+48)] = r_g0;
        r3 = heap32[(r3)];
        r3 = r3 >> 2;
        r3 = heap32[(r3+2)];
        heap32[(g0)] = r5;
        __FUNCTION_TABLE__[(r3)>>2](i7);
        r5 = (r0 + 200)|0;
        heap32[(r7+49)] = r_g0;
_67: while(true){
        if(r8 >35) //_LBB26_72
{
__label__ = 52;
break _67;
}
else{
        r3 = 0;
_70: while(true){
        if(r3 <36) //_LBB26_65
{
        r7 = r0 >> 2;
        r7 = heap32[(r7+1346)];
        r9 = r7 >> 2;
        r9 = heap32[(r9)];
        r9 = r9 >> 2;
        r9 = heap32[(r9+5)];
        r10 = r3 << 2;
        heap32[(g0)] = r7;
        heap32[(g0+1)] = r8;
        heap32[(g0+2)] = r3;
        r7 = (r5 + r10)|0;
        r7 = r7 >> 2;
        __FUNCTION_TABLE__[(r9)>>2](i7);
        r9 = r_g0;
        heap32[(r7)] = r9;
        if(r9 !=0) //_LBB26_67
{
        r3 = (r3 + 1)|0;
}
else{
__label__ = 46;
break _67;
}
}
else{
break _70;
}
}
        r8 = (r8 + 1)|0;
        r5 = (r5 + 144)|0;
continue _67;
}
}
switch(__label__ ){//multiple entries
case 52:
        heap32[(r1+14)] = r0;
        heap32[(g0)] = 76;
        _Znwj(i7);
        r0 = r_g0;
        heap32[(g0)] = r0;
        _ZN28btHashedOverlappingPairCacheC1Ev(i7);
        r3 = heap32[(r2)];
        r5 = (r3 + 1)|0;
        heap32[(r2)] = r5;
        heap32[(g0)] = 135;
        malloc(i7);
        r7 = r_g0;
        if(r7 !=0) //_LBB26_74
{
        r8 = 0;
        r9 = (r7 + 4)|0;
        r8 = (r8 - r9)|0;
        r8 = r8 & 15;
        r8 = (r7 + r8)|0;
        r9 = (r8 + 4)|0;
        r8 = r8 >> 2;
        heap32[(r8)] = r7;
        r7 = r9;
}
        r8 = _ZTV20btAxisSweep3InternalItE;
        r9 = r7 >> 2;
        r8 = (r8 + 8)|0;
        r10 = -2;
        heap32[(r9)] = r8;
        r8 = -1;
        heap16[(r7+4)>>1] = r10;
        heap16[(r7+6)>>1] = r8;
        heap32[(r9+23)] = r0;
        r8 = 0;
        heap32[(r9+24)] = 0;
        heap8[r7+100] = r8;
        heap32[(r9+26)] = 0;
        heap32[(r9+27)] = 0;
        if(r0 ==0) //_LBB26_77
{
        r5 = (r3 + 2)|0;
        heap32[(r2)] = r5;
        heap32[(g0)] = 95;
        malloc(i7);
        r5 = r_g0;
        if(r5 !=0) //_LBB26_79
{
        r0 = 0;
        r3 = (r5 + 4)|0;
        r0 = (r0 - r3)|0;
        r0 = r0 & 15;
        r0 = (r5 + r0)|0;
        r3 = (r0 + 4)|0;
        r0 = r0 >> 2;
        heap32[(r0)] = r5;
        r5 = r3;
}
        heap32[(g0)] = r5;
        _ZN28btHashedOverlappingPairCacheC1Ev(i7);
        heap32[(r9+23)] = r5;
        heap8[r7+100] = r4;
        r5 = heap32[(r2)];
}
        r0 = (r5 + 1)|0;
        heap32[(r2)] = r0;
        heap32[(g0)] = 43;
        malloc(i7);
        r0 = r_g0;
        if(r0 !=0) //_LBB26_83
{
        r3 = (r0 + 4)|0;
        r3 = (r8 - r3)|0;
        r3 = r3 & 15;
        r3 = (r0 + r3)|0;
        r10 = (r3 + 4)|0;
        r3 = r3 >> 2;
        heap32[(r3)] = r0;
        r0 = r10;
}
        r3 = _ZTV15btNullPairCache;
        r10 = r0 >> 2;
        r3 = (r3 + 8)|0;
        heap32[(r10)] = r3;
        heap8[r0+20] = r4;
        heap32[(r10+4)] = 0;
        heap32[(r10+2)] = 0;
        heap32[(r10+3)] = 0;
        r3 = (r5 + 2)|0;
        heap32[(r9+28)] = r0;
        heap32[(r2)] = r3;
        heap32[(g0)] = 175;
        malloc(i7);
        r10 = r_g0;
        if(r10 !=0) //_LBB26_86
{
        r11 = (r10 + 4)|0;
        r11 = (r8 - r11)|0;
        r11 = r11 & 15;
        r11 = (r10 + r11)|0;
        r12 = (r11 + 4)|0;
        r11 = r11 >> 2;
        heap32[(r11)] = r10;
        r10 = r12;
}
        r11 = _ZTV16btDbvtBroadphase;
        r12 = r10 >> 2;
        r11 = (r11 + 8)|0;
        heap32[(r12)] = r11;
_92: while(true){
        r11 = (r10 + r8)|0;
        r13 = r11 >> 2;
        heap8[r11+40] = r4;
        heap32[(r13+9)] = 0;
        heap32[(r13+7)] = 0;
        heap32[(r13+8)] = 0;
        heap32[(r13+1)] = 0;
        heap32[(r13+2)] = 0;
        heap32[(r13+3)] = -1;
        r8 = (r8 + 40)|0;
        heap32[(r13+4)] = 0;
        heap32[(r13+5)] = 0;
        if(r8 !=80) //_LBB26_88
{
continue _92;
}
else{
break _92;
}
}
        r8 = 0;
        r11 = r0 == r8;
        heap8[r10+153] = r8;
        r11 = r11 & 1;
        heap8[r10+154] = r4;
        heap8[r10+152] = r11;
        heap32[(r12+25)] = 0;
        heap32[(r12+26)] = 0;
        heap32[(r12+31)] = 0;
        heap32[(r12+27)] = 1;
        heap32[(r12+28)] = 0;
        heap32[(r12+29)] = 10;
        heap32[(r12+30)] = 1;
        heap32[(r12+32)] = 0;
        heap32[(r12+33)] = 0;
        heap32[(r12+34)] = 0;
        if(r0 ==0) //_LBB26_91
{
        r0 = (r5 + 3)|0;
        heap32[(r2)] = r0;
        heap32[(g0)] = 95;
        malloc(i7);
        r0 = r_g0;
        if(r0 !=0) //_LBB26_93
{
        r3 = 0;
        r5 = (r0 + 4)|0;
        r3 = (r3 - r5)|0;
        r3 = r3 & 15;
        r3 = (r0 + r3)|0;
        r5 = (r3 + 4)|0;
        r3 = r3 >> 2;
        heap32[(r3)] = r0;
        r0 = r5;
}
        heap32[(g0)] = r0;
        _ZN28btHashedOverlappingPairCacheC1Ev(i7);
        r3 = heap32[(r2)];
}
        heap32[(r12+24)] = r0;
        heap32[(r12+37)] = 0;
        heap32[(r12+35)] = 0;
        heap32[(r12+36)] = 0;
        heap32[(r12+21)] = 0;
        heap32[(r12+22)] = 0;
        heap32[(r12+23)] = 0;
        heap32[(r9+27)] = r10;
        heap8[r10+153] = r4;
        heap32[(r9+2)] = -998637568;
        heap32[(r9+3)] = -998637568;
        heap32[(r9+4)] = -998637568;
        heap32[(r9+5)] = 0;
        heap32[(r9+6)] = 1148846080;
        heap32[(r9+7)] = 1148846080;
        heap32[(r9+8)] = 1148846080;
        heap32[(r9+9)] = 0;
        r0 = heapU16[(r7+6)>>1];
        f0 = uint(r0); //fuitos r0, f0
        f1 =                      2000;
        f0 = f0/f1;
        heapFloat[(r9+10)] = f0;
        heapFloat[(r9+11)] = f0;
        heapFloat[(r9+12)] = f0;
        r0 = (r3 + 1)|0;
        heap32[(r9+13)] = 0;
        heap32[(r2)] = r0;
        heap32[(g0)] = 224083;
        malloc(i7);
        r0 = r_g0;
        if(r0 !=0) //_LBB26_97
{
        r5 = (r0 + 4)|0;
        r5 = (r8 - r5)|0;
        r5 = r5 & 15;
        r5 = (r0 + r5)|0;
        r10 = (r5 + 4)|0;
        r5 = r5 >> 2;
        heap32[(r5)] = r0;
        r0 = r10;
}
_103: while(true){
        r5 = r8 << 6;
        r5 = (r0 - r5)|0;
        r5 = r5 >> 2;
        r8 = (r8 + -1)|0;
        heap32[(r5)] = 0;
        heap32[(r5+2)] = 0;
        if(r8 !=-3501) //_LBB26_98
{
continue _103;
}
else{
break _103;
}
}
        r5 = 3501;
        heap32[(r9+15)] = r0;
        r8 = 0;
        heap16[(r7+58)>>1] = r5;
        r5 = -112;
        r10 = 2;
        heap16[(r7+56)>>1] = r8;
        heap16[(r7+64)>>1] = r4;
_106: while(true){
        r0 = (r0 - r5)|0;
        heap16[(r0)>>1] = r10;
        r5 = (r5 + -64)|0;
        r0 = heap32[(r9+15)];
        r10 = (r10 + 1)|0;
        if(r5 !=-224112) //_LBB26_100
{
continue _106;
}
else{
break _106;
}
}
        r3 = (r3 + 2)|0;
        heap16[(r0+224048)>>1] = r8;
_109: while(true){
        r0 = (r3 + r8)|0;
        heap32[(r2)] = r0;
        heap32[(g0)] = 28027;
        malloc(i7);
        r0 = r_g0;
        if(r0 !=0) //_LBB26_104
{
        r5 = 0;
        r10 = (r0 + 4)|0;
        r5 = (r5 - r10)|0;
        r5 = r5 & 15;
        r5 = (r0 + r5)|0;
        r10 = (r5 + 4)|0;
        r5 = r5 >> 2;
        heap32[(r5)] = r0;
        r0 = r10;
}
        r5 = r8 << 2;
        r5 = (r7 + r5)|0;
        r5 = r5 >> 2;
        r8 = (r8 + 1)|0;
        heap32[(r5+20)] = r0;
        heap32[(r5+17)] = r0;
        if(r8 !=3) //_LBB26_102
{
continue _109;
}
else{
break _109;
}
}
        r0 = heap32[(r9+15)];
        r0 = r0 >> 2;
        heap32[(r0)] = 0;
        r0 = heap32[(r9+15)];
        r3 = 0;
        heap16[(r0+48)>>1] = r3;
        r0 = heap32[(r9+15)];
        heap16[(r0+54)>>1] = r4;
        r0 = heap32[(r9+17)];
        heap16[(r0)>>1] = r3;
        r0 = heap32[(r9+17)];
        heap16[(r0+2)>>1] = r3;
        r0 = heap32[(r9+17)];
        r5 = heapU16[(r7+6)>>1];
        heap16[(r0+4)>>1] = r5;
        r0 = heap32[(r9+17)];
        heap16[(r0+6)>>1] = r3;
        r0 = heap32[(r9+15)];
        heap16[(r0+50)>>1] = r3;
        r0 = heap32[(r9+15)];
        heap16[(r0+56)>>1] = r4;
        r0 = heap32[(r9+18)];
        heap16[(r0)>>1] = r3;
        r0 = heap32[(r9+18)];
        heap16[(r0+2)>>1] = r3;
        r0 = heap32[(r9+18)];
        r5 = heapU16[(r7+6)>>1];
        heap16[(r0+4)>>1] = r5;
        r0 = heap32[(r9+18)];
        heap16[(r0+6)>>1] = r3;
        r0 = heap32[(r9+15)];
        heap16[(r0+52)>>1] = r3;
        r0 = heap32[(r9+15)];
        heap16[(r0+58)>>1] = r4;
        r0 = heap32[(r9+19)];
        heap16[(r0)>>1] = r3;
        r0 = heap32[(r9+19)];
        heap16[(r0+2)>>1] = r3;
        r0 = heap32[(r9+19)];
        r5 = heapU16[(r7+6)>>1];
        heap16[(r0+4)>>1] = r5;
        r0 = heap32[(r9+19)];
        r5 = _ZTV12btAxisSweep3;
        r5 = (r5 + 8)|0;
        heap16[(r0+6)>>1] = r3;
        heap32[(r9)] = r5;
        heap32[(r1+13)] = r7;
        heap32[(g0)] = 128;
        r0 = _ZTV35btSequentialImpulseConstraintSolver;
        _Znwj(i7);
        r7 = r_g0 >> 2;
        r0 = (r0 + 8)|0;
        heap32[(r7)] = r0;
        heap8[r_g0+20] = r4;
        heap32[(r7+4)] = 0;
        heap32[(r7+2)] = 0;
        heap32[(r7+3)] = 0;
        heap8[r_g0+40] = r4;
        heap32[(r7+9)] = 0;
        heap32[(r7+7)] = 0;
        heap32[(r7+8)] = 0;
        heap8[r_g0+60] = r4;
        heap32[(r7+14)] = 0;
        heap32[(r7+12)] = 0;
        heap32[(r7+13)] = 0;
        heap8[r_g0+80] = r4;
        heap32[(r7+19)] = 0;
        heap32[(r7+17)] = 0;
        heap32[(r7+18)] = 0;
        heap8[r_g0+100] = r4;
        heap32[(r7+24)] = 0;
        heap32[(r7+22)] = 0;
        heap32[(r7+23)] = 0;
        heap8[r_g0+120] = r4;
        heap32[(r7+29)] = 0;
        heap32[(r7+27)] = 0;
        heap32[(r7+28)] = 0;
        heap32[(r7+31)] = 0;
        heap32[(r1+15)] = r_g0;
        heap32[(g0)] = 272;
        _Znwj(i7);
        r5 = r_g0;
        r7 = heap32[(r1+16)];
        r8 = _ZTV16btCollisionWorld;
        r9 = heap32[(r1+15)];
        r10 = heap32[(r1+13)];
        r11 = heap32[(r1+14)];
        r8 = (r8 + 8)|0;
        r12 = r5 >> 2;
        heap32[(r12)] = r8;
        heap8[r5+20] = r4;
        heap32[(r12+4)] = 0;
        heap32[(r12+2)] = 0;
        heap32[(r12+3)] = 0;
        heap32[(r12+6)] = r11;
        heap32[(r12+7)] = 0;
        heap32[(r12+8)] = 0;
        heap32[(r12+9)] = 1;
        heap32[(r12+10)] = 1065353216;
        heap8[r5+44] = r3;
        heap32[(r12+12)] = 0;
        heap8[r5+52] = r3;
        heap8[r5+53] = r4;
        heap8[r5+54] = r4;
        heap32[(r12+14)] = 1025758986;
        heap8[r5+60] = r3;
        heap32[(r12+16)] = 0;
        heap8[r5+68] = r3;
        heap32[(r12+18)] = 0;
        heap32[(r12+20)] = r10;
        heap32[(r12+21)] = 0;
        r8 = r7 >> 2;
        heap8[r5+88] = r4;
        r8 = heap32[(r8)];
        r8 = r8 >> 2;
        r8 = heap32[(r8+4)];
        heap32[(g0)] = r7;
        __FUNCTION_TABLE__[(r8)>>2](i7);
        heap32[(r12+19)] = r_g0;
        heap32[(r12+18)] = r_g0;
        heap32[(r12+23)] = 0;
        heap32[(r12+24)] = 0;
        heap32[(r12+25)] = 0;
        heap32[(r12+26)] = 1058642330;
        heap32[(r12+27)] = 1065353216;
        heap32[(r12+28)] = 1050253722;
        heap32[(r12+30)] = 0;
        heap32[(r12+32)] = 1101004800;
        heap32[(r12+31)] = 10;
        heap32[(r12+34)] = 1045220557;
        heap32[(r12+35)] = 1036831949;
        heap32[(r12+36)] = 0;
        heap32[(r12+33)] = 1065353216;
        heap32[(r12+37)] = 0;
        heap32[(r12+38)] = -1130113270;
        heap32[(r12+39)] = 0;
        heap32[(r12+40)] = 1062836634;
        heap32[(r12+41)] = 260;
        r7 = _ZTV23btDiscreteDynamicsWorld;
        heap32[(r12+42)] = 2;
        r7 = (r7 + 8)|0;
        heap32[(r12+43)] = 128;
        heap32[(r12)] = r7;
        heap32[(r12+44)] = r9;
        heap8[r5+200] = r4;
        heap32[(r12+49)] = 0;
        heap32[(r12+47)] = 0;
        heap32[(r12+48)] = 0;
        heap8[r5+220] = r4;
        heap32[(r12+54)] = 0;
        heap32[(r12+52)] = 0;
        heap32[(r12+53)] = 0;
        heap32[(r12+56)] = 0;
        heap32[(r12+57)] = -1054867456;
        heap32[(r12+58)] = 0;
        heap32[(r12+59)] = 0;
        heap32[(r12+60)] = 0;
        heap8[r5+246] = r3;
        heap8[r5+264] = r4;
        heap32[(r12+65)] = 0;
        heap32[(r12+63)] = 0;
        heap32[(r12+64)] = 0;
        heap32[(r12+67)] = 0;
        if(r9 !=0) //_LBB26_111
{
        heap8[r5+245] = r3;
        r7 = heap32[(r2)];
}
else{
        r7 = heap32[(r2)];
        r7 = (r7 + 1)|0;
        heap32[(r2)] = r7;
        heap32[(g0)] = 147;
        malloc(i7);
        r8 = r_g0;
        if(r8 !=0) //_LBB26_109
{
        r9 = 0;
        r10 = (r8 + 4)|0;
        r9 = (r9 - r10)|0;
        r9 = r9 & 15;
        r9 = (r8 + r9)|0;
        r10 = (r9 + 4)|0;
        r9 = r9 >> 2;
        heap32[(r9)] = r8;
        r8 = r10;
}
        r9 = r8 >> 2;
        heap32[(r9)] = r0;
        heap8[r8+20] = r4;
        heap32[(r9+4)] = 0;
        heap32[(r9+2)] = 0;
        heap32[(r9+3)] = 0;
        heap8[r8+40] = r4;
        heap32[(r9+9)] = 0;
        heap32[(r9+7)] = 0;
        heap32[(r9+8)] = 0;
        heap8[r8+60] = r4;
        heap32[(r9+14)] = 0;
        heap32[(r9+12)] = 0;
        heap32[(r9+13)] = 0;
        heap8[r8+80] = r4;
        heap32[(r9+19)] = 0;
        heap32[(r9+17)] = 0;
        heap32[(r9+18)] = 0;
        heap8[r8+100] = r4;
        heap32[(r9+24)] = 0;
        heap32[(r9+22)] = 0;
        heap32[(r9+23)] = 0;
        heap8[r8+120] = r4;
        heap32[(r9+29)] = 0;
        heap32[(r9+27)] = 0;
        heap32[(r9+28)] = 0;
        r0 = r5 >> 2;
        heap32[(r9+31)] = 0;
        heap32[(r0+44)] = r8;
        heap8[r5+245] = r4;
}
        r0 = (r7 + 1)|0;
        heap32[(r2)] = r0;
        heap32[(g0)] = 87;
        malloc(i7);
        r0 = r_g0;
        if(r0 !=0) //_LBB26_114
{
        r7 = (r0 + 4)|0;
        r7 = (r3 - r7)|0;
        r7 = r7 & 15;
        r7 = (r0 + r7)|0;
        r8 = (r7 + 4)|0;
        r7 = r7 >> 2;
        heap32[(r7)] = r0;
        r0 = r8;
}
        r7 = _ZTV25btSimulationIslandManager;
        r8 = r0 >> 2;
        r7 = (r7 + 8)|0;
        heap32[(r8)] = r7;
        heap8[r0+20] = r4;
        heap32[(r8+4)] = 0;
        heap32[(r8+2)] = 0;
        heap32[(r8+3)] = 0;
        heap8[r0+40] = r4;
        heap32[(r8+9)] = 0;
        heap32[(r8+7)] = 0;
        heap32[(r8+8)] = 0;
        heap8[r0+60] = r4;
        heap32[(r8+14)] = 0;
        heap32[(r8+12)] = 0;
        heap32[(r8+13)] = 0;
        r7 = r5 >> 2;
        heap8[r0+64] = r4;
        heap32[(r7+45)] = r0;
        heap8[r5+244] = r4;
        heap32[(r1+1)] = r5;
        r0 = heap32[(r7+41)];
        r0 = r0 | 32;
        heap32[(r7+41)] = r0;
        heap32[(r7+31)] = 5;
        heap32[(r1+2)] = 0;
        r0 = heap32[(r1+1)];
        r5 = r0 >> 2;
        r5 = heap32[(r5)];
        r5 = r5 >> 2;
        r5 = heap32[(r5+17)];
        r7 = sp + -41280;
        r8 = r7 >> 2;
        heap32[(fp+-10320)] = 0;
        heap32[(r8+1)] = -1054867456;
        heap32[(r8+2)] = 0;
        heap32[(r8+3)] = 0;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r7;
        __FUNCTION_TABLE__[(r5)>>2](i7);
        r0 = heap32[(r1+17)];
        if(r0 <5) //_LBB26_117
{
        r0 = heap32[(r2)];
        r5 = (r0 + 1)|0;
        heap32[(r2)] = r5;
        heap32[(g0)] = 71;
        malloc(i7);
        r5 = r_g0;
        if(r5 !=0) //_LBB26_119
{
        r7 = 0;
        r8 = (r5 + 4)|0;
        r7 = (r7 - r8)|0;
        r7 = r7 & 15;
        r7 = (r5 + r7)|0;
        r8 = (r7 + 4)|0;
        r7 = r7 >> 2;
        heap32[(r7)] = r5;
        r5 = r8;
}
        r7 = r5 >> 2;
        heap32[(r7+2)] = 0;
        heap32[(r7+3)] = 1065353216;
        heap32[(r7+4)] = 1065353216;
        heap32[(r7+5)] = 1065353216;
        r8 = _ZTV10btBoxShape;
        heap32[(r7+6)] = 0;
        r8 = (r8 + 8)|0;
        heap32[(r7+11)] = 1025758986;
        heap32[(r7)] = r8;
        heap32[(r7+1)] = 0;
        heap32[(r7+7)] = 1132066243;
        heap32[(r7+8)] = 1112004362;
        heap32[(r7+9)] = 1132066243;
        heap32[(r7+10)] = 0;
        r7 = heap32[(r1+5)];
        r8 = heap32[(r1+4)];
        if(r7 ==r8) //_LBB26_122
{
        r9 = r8 << 1;
        r9 = r8 == 0 ? r4 : r9;
if(!(r7 >=r9)) //_LBB26_121
{
        if(r9 !=0) //_LBB26_125
{
        r7 = r9 << 2;
        r0 = (r0 + 2)|0;
        r7 = r7 | 3;
        heap32[(r2)] = r0;
        r0 = (r7 + 16)|0;
        heap32[(g0)] = r0;
        malloc(i7);
        r0 = r_g0;
        if(r0 !=0) //_LBB26_127
{
        r7 = 0;
        r10 = (r0 + 4)|0;
        r7 = (r7 - r10)|0;
        r7 = r7 & 15;
        r7 = (r0 + r7)|0;
        r10 = (r7 + 4)|0;
        r7 = r7 >> 2;
        heap32[(r7)] = r0;
        r0 = r10;
}
}
else{
        r0 = 0;
}
_138: do {
        if(r8 <1) //_LBB26_130
{
        r10 = heap32[(r1+6)];
}
else{
        r7 = 0;
_141: while(true){
        r10 = heap32[(r1+6)];
        r11 = r7 << 2;
        r12 = (r10 + r11)|0;
        r12 = r12 >> 2;
        r11 = (r0 + r11)|0;
        r12 = heap32[(r12)];
        r7 = (r7 + 1)|0;
        r11 = r11 >> 2;
        heap32[(r11)] = r12;
if(!(r8 !=r7)) //_LBB26_131
{
break _138;
}
}
}
} while(0);
        if(r10 !=0) //_LBB26_134
{
        r7 = heapU8[r6+28];
        if(r7 !=0) //_LBB26_136
{
        r8 = gNumAlignedFree;
        r8 = r8 >> 2;
        r7 = heap32[(r8)];
        r7 = (r7 + 1)|0;
        r10 = r10 >> 2;
        heap32[(r8)] = r7;
        r8 = heap32[(r10+-1)];
        heap32[(g0)] = r8;
        free(i7);
        r8 = heap32[(r1+4)];
}
        heap32[(r1+6)] = 0;
}
        heap8[r6+28] = r4;
        heap32[(r1+6)] = r0;
        heap32[(r1+5)] = r9;
}
}
        r0 = r8 << 2;
        r7 = heap32[(r1+6)];
        r0 = (r7 + r0)|0;
        r0 = r0 >> 2;
        heap32[(r0)] = r5;
        r0 = heap32[(r1+4)];
        r0 = (r0 + 1)|0;
        r7 = _ZGVZN11btTransform11getIdentityEvE17identityTransform;
        heap32[(r1+4)] = r0;
        r0 = heapU8[r7];
if(!(r0 !=0)) //_LBB26_143
{
        r0 = _ZGVZN11btMatrix3x311getIdentityEvE14identityMatrix;
        r8 = heapU8[r0];
if(!(r8 !=0)) //_LBB26_142
{
        r8 = _ZZN11btMatrix3x311getIdentityEvE14identityMatrix_2E_0_2E_0_2E_0_2E_0;
        r9 = _ZZN11btMatrix3x311getIdentityEvE14identityMatrix_2E_0_2E_1_2E_0_2E_1;
        r8 = r8 >> 2;
        r10 = _ZZN11btMatrix3x311getIdentityEvE14identityMatrix_2E_0_2E_2_2E_0_2E_2;
        r9 = r9 >> 2;
        heap32[(r8)] = 1065353216;
        r8 = r10 >> 2;
        heap32[(r9)] = 1065353216;
        heap32[(r8)] = 1065353216;
        heap8[r0] = r4;
}
        r0 = _ZZN11btTransform11getIdentityEvE17identityTransform;
        r8 = _ZZN11btMatrix3x311getIdentityEvE14identityMatrix_2E_0_2E_0_2E_0_2E_0;
        r0 = r0 >> 2;
        r8 = r8 >> 2;
        heap32[(r0)] = heap32[(r8)];
        heap32[(r0+1)] = 0;
        heap32[(r0+2)] = 0;
        r8 = _ZZN11btMatrix3x311getIdentityEvE14identityMatrix_2E_0_2E_1_2E_0_2E_1;
        heap32[(r0+3)] = 0;
        r8 = r8 >> 2;
        heap32[(r0+4)] = 0;
        heap32[(r0+5)] = heap32[(r8)];
        heap32[(r0+6)] = 0;
        heap32[(r0+7)] = 0;
        r8 = _ZZN11btMatrix3x311getIdentityEvE14identityMatrix_2E_0_2E_2_2E_0_2E_2;
        heap32[(r0+8)] = 0;
        r8 = r8 >> 2;
        heap32[(r0+9)] = 0;
        heap32[(r0+10)] = heap32[(r8)];
        heap32[(r0+11)] = 0;
        heap32[(r0+12)] = 0;
        heap32[(r0+13)] = 0;
        heap32[(r0+14)] = 0;
        heap32[(r0+15)] = 0;
        heap8[r7] = r4;
}
        heap32[(g0)] = 200;
        r0 = _ZTV20btDefaultMotionState;
        _Znwj(i7);
        r8 = r_g0 >> 2;
        r0 = (r0 + 8)|0;
        heap32[(r8)] = r0;
        heap32[(r8+1)] = 1065353216;
        heap32[(r8+2)] = 0;
        heap32[(r8+3)] = 0;
        heap32[(r8+4)] = 0;
        heap32[(r8+5)] = 0;
        heap32[(r8+6)] = 1065353216;
        heap32[(r8+7)] = 0;
        heap32[(r8+8)] = 0;
        heap32[(r8+9)] = 0;
        heap32[(r8+10)] = 0;
        heap32[(r8+11)] = 1065353216;
        heap32[(r8+12)] = 0;
        heap32[(r8+13)] = 0;
        heap32[(r8+14)] = -1035468800;
        r0 = _ZZN11btTransform11getIdentityEvE17identityTransform;
        heap32[(r8+15)] = 0;
        r0 = r0 >> 2;
        heap32[(r8+16)] = 0;
        heap32[(r8+17)] = heap32[(r0)];
        heap32[(r8+18)] = heap32[(r0+1)];
        heap32[(r8+19)] = heap32[(r0+2)];
        heap32[(r8+20)] = heap32[(r0+3)];
        heap32[(r8+21)] = heap32[(r0+4)];
        heap32[(r8+22)] = heap32[(r0+5)];
        heap32[(r8+23)] = heap32[(r0+6)];
        heap32[(r8+24)] = heap32[(r0+7)];
        heap32[(r8+25)] = heap32[(r0+8)];
        heap32[(r8+26)] = heap32[(r0+9)];
        heap32[(r8+27)] = heap32[(r0+10)];
        heap32[(r8+28)] = heap32[(r0+11)];
        heap32[(r8+29)] = heap32[(r0+12)];
        heap32[(r8+30)] = heap32[(r0+13)];
        heap32[(r8+31)] = heap32[(r0+14)];
        heap32[(r8+32)] = heap32[(r0+15)];
        heap32[(r8+33)] = 1065353216;
        heap32[(r8+34)] = 0;
        heap32[(r8+35)] = 0;
        heap32[(r8+36)] = 0;
        heap32[(r8+37)] = 0;
        heap32[(r8+38)] = 1065353216;
        heap32[(r8+39)] = 0;
        heap32[(r8+40)] = 0;
        heap32[(r8+41)] = 0;
        heap32[(r8+42)] = 0;
        heap32[(r8+43)] = 1065353216;
        heap32[(r8+44)] = 0;
        heap32[(r8+45)] = 0;
        heap32[(r8+46)] = -1035468800;
        heap32[(r8+47)] = 0;
        heap32[(r8+48)] = 0;
        r0 = sp + -41416;
        heap32[(r8+49)] = 0;
        r8 = r0 >> 2;
        heap32[(fp+-10354)] = 0;
        heap32[(r8+1)] = r_g0;
        heap32[(r8+18)] = r5;
        heap32[(r8+19)] = 0;
        heap32[(r8+20)] = 0;
        heap32[(r8+21)] = 0;
        heap32[(r8+22)] = 0;
        heap32[(r8+23)] = 0;
        heap32[(r8+24)] = 0;
        heap32[(r8+25)] = 1056964608;
        heap32[(r8+26)] = 0;
        heap32[(r8+27)] = 1061997773;
        r5 = 0;
        heap32[(r8+28)] = 1065353216;
        heap8[sp+-41300] = r5;
        heap32[(r8+30)] = 1000593162;
        heap32[(r8+31)] = 1008981770;
        heap32[(r8+32)] = 1008981770;
        heap32[(r8+33)] = 1008981770;
        heap32[(r8+2)] = 1065353216;
        heap32[(r8+3)] = 0;
        heap32[(r8+4)] = 0;
        heap32[(r8+5)] = 0;
        heap32[(r8+6)] = 0;
        heap32[(r8+7)] = 1065353216;
        heap32[(r8+8)] = 0;
        heap32[(r8+9)] = 0;
        heap32[(r8+10)] = 0;
        heap32[(r8+11)] = 0;
        heap32[(r8+12)] = 1065353216;
        heap32[(r8+13)] = 0;
        heap32[(r8+14)] = 0;
        heap32[(r8+15)] = 0;
        heap32[(r8+16)] = 0;
        heap32[(r8+17)] = 0;
        r7 = heap32[(r2)];
        r7 = (r7 + 1)|0;
        heap32[(r2)] = r7;
        heap32[(g0)] = 627;
        malloc(i7);
        r7 = r_g0;
        if(r7 !=0) //_LBB26_145
{
        r8 = (r7 + 4)|0;
        r5 = (r5 - r8)|0;
        r5 = r5 & 15;
        r5 = (r7 + r5)|0;
        r8 = (r5 + 4)|0;
        r5 = r5 >> 2;
        heap32[(r5)] = r7;
        r7 = r8;
}
        r5 = r7 >> 2;
        heap32[(r5+41)] = 1065353216;
        heap32[(r5+42)] = 1065353216;
        heap32[(r5+43)] = 1065353216;
        heap32[(r5+44)] = 0;
        heap32[(r5+45)] = 0;
        heap32[(r5+46)] = 1566444395;
        heap32[(r5+47)] = 0;
        heap32[(r5+48)] = 0;
        heap32[(r5+49)] = 0;
        heap32[(r5+50)] = 0;
        heap32[(r5+51)] = 1;
        heap32[(r5+52)] = -1;
        heap32[(r5+53)] = -1;
        heap32[(r5+54)] = 1;
        heap32[(r5+55)] = 0;
        heap32[(r5+56)] = 1056964608;
        heap32[(r5+57)] = 0;
        heap32[(r5+58)] = 1;
        heap32[(r5+59)] = 0;
        heap32[(r5+60)] = 1065353216;
        heap32[(r5+61)] = 0;
        heap32[(r5+62)] = 0;
        heap32[(r5+63)] = 0;
        heap32[(r5+1)] = 1065353216;
        heap32[(r5+2)] = 0;
        heap32[(r5+3)] = 0;
        heap32[(r5+4)] = 0;
        heap32[(r5+5)] = 0;
        heap32[(r5+6)] = 1065353216;
        heap32[(r5+7)] = 0;
        heap32[(r5+8)] = 0;
        heap32[(r5+9)] = 0;
        heap32[(r5+10)] = 0;
        heap32[(r5+11)] = 1065353216;
        heap32[(r5+12)] = 0;
        heap32[(r5+13)] = 0;
        heap32[(r5+14)] = 0;
        r8 = _ZTV11btRigidBody;
        heap32[(r5+15)] = 0;
        r8 = (r8 + 8)|0;
        heap32[(r5+16)] = 0;
        heap32[(r5)] = r8;
        heap8[r7+492] = r4;
        heap32[(r5+122)] = 0;
        heap32[(r5+120)] = 0;
        heap32[(r5+121)] = 0;
        heap32[(g0)] = r7;
        heap32[(g0+1)] = r0;
        _ZN11btRigidBody14setupRigidBodyERKNS_27btRigidBodyConstructionInfoE(i7);
        r0 = heap32[(r1+1)];
        r5 = r0 >> 2;
        r5 = heap32[(r5)];
        r5 = r5 >> 2;
        r5 = heap32[(r5+20)];
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r7;
        __FUNCTION_TABLE__[(r5)>>2](i7);
        r0 = heap32[(r1+17)];
}
_161: do {
        if(r0 >3) //_LBB26_151
{
        if(r0 >5) //_LBB26_154
{
        if(r0 ==6) //_LBB26_325
{
        _ZN13BenchmarkDemo11createTest6Ev(i7);
        return;
}
else{
        if(r0 ==7) //_LBB26_326
{
        _ZN13BenchmarkDemo11createTest6Ev(i7);
        r0 = sp + -41264;
        heap32[(g0)] = 8;
        r1 = r0 >> 2;
        _Znwj(i7);
        heap32[(r1+10006)] = r_g0;
        heap32[(g0)] = r_g0;
        heap32[(g0+1)] = 0;
        gettimeofday(i7);
        heap32[(r1+10000)] = 0;
        heap32[(r1+10001)] = 0;
        heap32[(r1+10005)] = 0;
        heap32[(r1+10004)] = 9999;
        heap32[(r1+10003)] = 0;
        heap32[(r1+10002)] = 0;
        heap32[(r1+10007)] = 1092616192;
        heap32[(r1+10008)] = 0;
        heap32[(r1+10009)] = 0;
        heap32[(r1+10010)] = 1112014848;
        heap32[(r1+10011)] = 1065353216;
        f3 =                         0;
_169: while(true){
        f0 = r3; //fitos r3, f0
        f1 =      0.025132741779088974;
        f0 = f0*f1;
        f1 =                       0.5;
        f0 = f0*f1;
        heapFloat[(g0)] = f0;
        sinf(i7);
        f2 =                         1;
        f1 = f_g0/f2;
        f2 = f1*f3;
        heapFloat[(g0)] = f0;
        cosf(i7);
        f4 = -f2;
        f5 = f_g0*f3;
        f6 = f2*f3;
        f7 = f4-f2;
        f8 = f_g0+f2;
        f8 = f8-f6;
        f7 = f7-f6;
        f9 = f5+f2;
        f9 = f9-f6;
        f5 = f5+f6;
        f6 = f7*f4;
        f10 = f8*f_g0;
        f5 = f5-f1;
        f11 = f9*f_g0;
        f7 = f7*f1;
        f10 = f6+f10;
        f4 = f9*f4;
        r2 = r3 << 4;
        f0 = f5*f_g0;
        f7 = f11-f7;
        f9 = f5*f2;
        f10 = f10+f4;
        f5 = f5*f1;
        r2 = (r0 + r2)|0;
        f0 = f6+f0;
        f1 = f8*f1;
        f6 = f7-f9;
        f2 = f8*f2;
        f5 = f10+f5;
        f7 =                      2500;
        f0 = f0-f1;
        f1 = f6+f2;
        r2 = r2 >> 2;
        f2 = f5*f7;
        f0 = f0-f4;
        f1 = f1*f7;
        heapFloat[(r2+4000)] = f2;
        f0 = f0*f7;
        heapFloat[(r2+4001)] = f1;
        heapFloat[(r2+4002)] = f0;
        heap32[(r2+4003)] = 0;
        f1 = heapFloat[(r1+10008)];
        heapFloat[(r2)] = f1;
        heap32[(r2+1)] = 1112014848;
        heap32[(r2+2)] = 0;
        f1 = f1+f2;
        heap32[(r2+3)] = 0;
        f0 = f0+f3;
        heapFloat[(r2+2000)] = f1;
        heapFloat[(r2+2002)] = f0;
        heap32[(r2+2003)] = 0;
        heap32[(r2+2001)] = -998637568;
        heap32[(r2+8000)] = 1065353216;
        heap32[(r2+8001)] = 0;
        r3 = (r3 + 1)|0;
        heap32[(r2+8002)] = 0;
        heap32[(r2+8003)] = 0;
if(!(r3 !=500)) //_LBB26_327
{
break _169;
}
}
        r2 = _ZL10raycastBar;
        heap32[(g0)] = r2;
        heap32[(g0+1)] = r0;
        heap32[(g0+2)] = 8000;
        memcpy(i7);
        r3 = (r2 + 8000)|0;
        r4 = (r0 + 8000)|0;
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r4;
        heap32[(g0+2)] = 8000;
        memcpy(i7);
        r3 = (r2 + 16000)|0;
        r4 = (r0 + 16000)|0;
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r4;
        heap32[(g0+2)] = 8000;
        memcpy(i7);
        r3 = (r2 + 24000)|0;
        r4 = (r0 + 24000)|0;
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r4;
        heap32[(g0+2)] = 8000;
        memcpy(i7);
        r3 = (r2 + 32000)|0;
        r0 = (r0 + 32000)|0;
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r0;
        heap32[(g0+2)] = 8000;
        memcpy(i7);
        r0 = r2 >> 2;
        r2 = heap32[(r1+10000)];
        heap32[(r0+10000)] = r2;
        r2 = heap32[(r1+10001)];
        heap32[(r0+10001)] = r2;
        r2 = heap32[(r1+10002)];
        heap32[(r0+10002)] = r2;
        r2 = heap32[(r1+10003)];
        heap32[(r0+10003)] = r2;
        r2 = heap32[(r1+10004)];
        heap32[(r0+10004)] = r2;
        r2 = heap32[(r1+10005)];
        heap32[(r0+10005)] = r2;
        r2 = heap32[(r1+10006)];
        r2 = r2 >> 2;
        r3 = heap32[(r0+10006)];
        r4 = heap32[(r2+1)];
        r2 = heap32[(r2)];
        r3 = r3 >> 2;
        heap32[(r3)] = r2;
        heap32[(r3+1)] = r4;
        heap32[(r0+10007)] = heap32[(r1+10007)];
        heap32[(r0+10008)] = heap32[(r1+10008)];
        heap32[(r0+10009)] = heap32[(r1+10009)];
        heap32[(r0+10010)] = heap32[(r1+10010)];
        heap32[(r0+10011)] = heap32[(r1+10011)];
        r0 = heap32[(r1+10006)];
        heap32[(g0)] = r0;
        _ZdlPv(i7);
        return;
}
else{
break _161;
}
}
}
else{
        if(r0 ==4) //_LBB26_291
{
        r3 = heap32[(r2)];
        r3 = (r3 + 1)|0;
        heap32[(r2)] = r3;
        heap32[(g0)] = 127;
        malloc(i7);
        r2 = r_g0;
        if(r2 !=0) //_LBB26_293
{
        r3 = 0;
        r4 = (r2 + 4)|0;
        r3 = (r3 - r4)|0;
        r3 = r3 & 15;
        r3 = (r2 + r3)|0;
        r4 = (r3 + 4)|0;
        r3 = r3 >> 2;
        heap32[(r3)] = r2;
        r2 = r4;
}
        heap32[(g0)] = r2;
        heap32[(g0+1)] = 0;
        heap32[(g0+2)] = 0;
        r3 = r2 >> 2;
        _ZN17btConvexHullShapeC1EPKfii(i7);
        r4 = heap32[(r3)];
        r4 = r4 >> 2;
        r4 = heap32[(r4+6)];
        r0 = sp + -96;
        r5 = r0 >> 2;
        heap32[(fp+-24)] = 1065353216;
        heap32[(r5+1)] = 1065353216;
        heap32[(r5+2)] = 1065353216;
        heap32[(r5+3)] = 0;
        heap32[(g0)] = r2;
        heap32[(g0+1)] = r0;
        r0 = 0;
        __FUNCTION_TABLE__[(r4)>>2](i7);
_178: while(true){
        r4 = (r0 * -3)|0;
        r5 = _ZL7TaruVtx;
        r4 = r4 << 2;
        r4 = (r5 + r4)|0;
        r5 = sp + -80;
        r4 = r4 >> 2;
        r7 = r5 >> 2;
        heap32[(fp+-20)] = heap32[(r4)];
        heap32[(r7+1)] = heap32[(r4+1)];
        heap32[(r7+2)] = heap32[(r4+2)];
        heap32[(r7+3)] = 0;
        heap32[(g0)] = r2;
        heap32[(g0+1)] = r5;
        r0 = (r0 + -1)|0;
        _ZN17btConvexHullShape8addPointERK9btVector3(i7);
        if(r0 ==-43) //_LBB26_301
{
break _178;
}
}
        r4 = sp + -160;
        r0 = r4 >> 2;
        heap32[(fp+-40)] = 1065353216;
        heap32[(r0+1)] = 0;
        heap32[(r0+2)] = 0;
        heap32[(r0+3)] = 0;
        heap32[(r0+4)] = 0;
        heap32[(r0+5)] = 1065353216;
        heap32[(r0+6)] = 0;
        heap32[(r0+7)] = 0;
        heap32[(r0+8)] = 0;
        heap32[(r0+9)] = 0;
        heap32[(r0+10)] = 1065353216;
        heap32[(r0+11)] = 0;
        heap32[(r0+12)] = 0;
        heap32[(r0+13)] = 0;
        heap32[(r0+14)] = 0;
        r5 = sp + -176;
        heap32[(r0+15)] = 0;
        r7 = r5 >> 2;
        heap32[(fp+-44)] = 0;
        heap32[(r7+1)] = 0;
        heap32[(r7+2)] = 0;
        heap32[(r7+3)] = 0;
        r3 = heap32[(r3)];
        r3 = r3 >> 2;
        r7 = heap32[(r3+8)];
        heap32[(g0)] = r2;
        heap32[(g0+1)] = 1065353216;
        heap32[(g0+2)] = r5;
        r3 = 15;
        f0 =                         3;
        f2 =                       -18;
        f3 =                       1.5;
        __FUNCTION_TABLE__[(r7)>>2](i7);
        f1 = f0;
_181: while(true){
        f4 = f3+f0;
        r5 = 0;
        r7 = r5;
_183: while(true){
        f5 = r7; //fitos r7, f5
        f5 = f5*f4;
        f5 = f5+f2;
        r8 = r5;
_185: while(true){
        f6 = r8; //fitos r8, f6
        f6 = f6*f4;
        f6 = f6+f2;
        heapFloat[(r0+12)] = f6;
        heapFloat[(r0+13)] = f1;
        heapFloat[(r0+14)] = f5;
        heap32[(r0+15)] = 0;
        r9 = heap32[(r1)];
        r9 = r9 >> 2;
        r9 = heap32[(r9+2)];
        heap32[(g0)] = r6;
        heap32[(g0+1)] = 1065353216;
        heap32[(g0+2)] = r4;
        heap32[(g0+3)] = r2;
        r8 = (r8 + 1)|0;
        __FUNCTION_TABLE__[(r9)>>2](i7);
if(!(r8 !=8)) //_LBB26_298
{
break _185;
}
}
        r7 = (r7 + 1)|0;
if(!(r7 !=8)) //_LBB26_297
{
break _183;
}
}
        f4 =        1.0099999904632568;
        f5 =      -0.05000000074505806;
        f4 = f3*f4;
        f6 =                         3;
        f3 = f3*f5;
        f5 =                         7;
        f6 = f4+f6;
        f3 = f3*f5;
        r3 = (r3 + -1)|0;
        f1 = f1+f6;
        f2 = f3+f2;
        f3 = f4;
        if(r3 ==0) //_LBB26_165
{
break _161;
}
else{
continue _181;
}
}
}
else{
        if(r0 ==5) //_LBB26_302
{
        r3 = 10;
        f0 =                        20;
        f1 =                         2;
        f2 =                       -25;
_191: while(true){
        f3 =                         3;
        f4 =                        25;
        f5 = f1+f3;
        f4 = f0+f4;
        r0 = 0;
        r5 = r0;
_193: while(true){
        f6 = r5; //fitos r5, f6
        f6 = f6*f5;
        f6 = f6+f2;
        f7 =                         5;
        f6 = f6*f7;
        f8 =                         0;
        f6 = f6+f8;
        r7 = r0;
_195: while(true){
        r8 = _ZL8nextRand;
        r8 = r8 >> 2;
        r9 = heap32[(r8)];
        r9 = (r9 * 214013)|0;
        r9 = (r9 + 2531011)|0;
        r10 = sp + -64;
        heap32[(r8)] = r9;
        r8 = r10 >> 2;
        heap32[(fp+-16)] = 1065353216;
        heap32[(r8+1)] = 0;
        heap32[(r8+2)] = 0;
        heap32[(r8+3)] = 0;
        heap32[(r8+4)] = 0;
        heap32[(r8+5)] = 1065353216;
        heap32[(r8+6)] = 0;
        heap32[(r8+7)] = 0;
        f9 = r7; //fitos r7, f9
        heap32[(r8+8)] = 0;
        f9 = f9*f5;
        heap32[(r8+9)] = 0;
        f9 = f9+f2;
        heap32[(r8+10)] = 1065353216;
        f9 = f9*f7;
        f9 = f9+f8;
        heap32[(r8+11)] = 0;
        r9 = r9 >>> 16;
        heapFloat[(r8+12)] = f9;
        r9 = r9 & 32767;
        heapFloat[(r8+13)] = f4;
        r9 = Math.floor(uint(r9) % uint(9));
        heapFloat[(r8+14)] = f6;
        heap32[(r8+15)] = 0;
if(!(uint(r9) >uint(8))) //_LBB26_321
{
        r8 = r4 << r9;
        r11 = r8 & 7;
        if(r11 !=0) //_LBB26_309
{
        r8 = heap32[(r2)];
        r8 = (r8 + 1)|0;
        heap32[(r2)] = r8;
        r9 = (r9 + 1)|0;
        f9 = r9; //fitos r9, f9
        f10 =                       0.5;
        heap32[(g0)] = 71;
        f9 = f9*f10;
        f10 =                       1.5;
        malloc(i7);
        r9 = r_g0;
        f10 = f9*f10;
        if(r9 !=0) //_LBB26_311
{
        r8 = 0;
        r11 = (r9 + 4)|0;
        r8 = (r8 - r11)|0;
        r8 = r8 & 15;
        r8 = (r9 + r8)|0;
        r11 = (r8 + 4)|0;
        r8 = r8 >> 2;
        heap32[(r8)] = r9;
        r9 = r11;
}
        r8 = r9 >> 2;
        heap32[(r8+2)] = 0;
        heap32[(r8+3)] = 1065353216;
        heap32[(r8+4)] = 1065353216;
        heap32[(r8+5)] = 1065353216;
        r11 = _ZTV10btBoxShape;
        heap32[(r8+6)] = 0;
        r11 = (r11 + 8)|0;
        heap32[(r8+11)] = 1025758986;
        heap32[(r8)] = r11;
        f11 =     -0.039999999105930328;
        f10 = f10+f11;
        heap32[(r8+1)] = 0;
        heapFloat[(r8+7)] = f10;
        heapFloat[(r8+8)] = f10;
        heapFloat[(r8+9)] = f10;
        heap32[(r8+10)] = 0;
        r8 = heap32[(r1)];
        r8 = r8 >> 2;
        r8 = heap32[(r8+2)];
        heap32[(g0)] = r6;
        heapFloat[(g0+1)] = f9;
        heap32[(g0+2)] = r10;
        heap32[(g0+3)] = r9;
        __FUNCTION_TABLE__[(r8)>>2](i7);
}
else{
        r11 = r8 & 56;
        if(r11 !=0) //_LBB26_313
{
        r8 = heap32[(r2)];
        r8 = (r8 + 1)|0;
        heap32[(r2)] = r8;
        r9 = (r9 + -2)|0;
        heap32[(g0)] = 71;
        f9 = r9; //fitos r9, f9
        f10 =                       0.5;
        malloc(i7);
        r9 = r_g0;
        f9 = f9*f10;
        if(r9 !=0) //_LBB26_315
{
        r8 = 0;
        r11 = (r9 + 4)|0;
        r8 = (r8 - r11)|0;
        r8 = r8 & 15;
        r8 = (r9 + r8)|0;
        r11 = (r8 + 4)|0;
        r8 = r8 >> 2;
        heap32[(r8)] = r9;
        r9 = r11;
}
        r8 = r9 >> 2;
        heap32[(r8+2)] = 0;
        heap32[(r8+3)] = 1065353216;
        heap32[(r8+4)] = 1065353216;
        r11 = _ZTV13btSphereShape;
        heap32[(r8+5)] = 1065353216;
        r11 = (r11 + 8)|0;
        heap32[(r8+6)] = 0;
        heap32[(r8)] = r11;
        f10 =                       1.5;
        f10 = f9*f10;
        heap32[(r8+1)] = 8;
        heapFloat[(r8+7)] = f10;
        heapFloat[(r8+11)] = f10;
        r8 = heap32[(r1)];
        r8 = r8 >> 2;
        r8 = heap32[(r8+2)];
        heap32[(g0)] = r6;
        heapFloat[(g0+1)] = f9;
        heap32[(g0+2)] = r10;
        heap32[(g0+3)] = r9;
        __FUNCTION_TABLE__[(r8)>>2](i7);
}
else{
        r8 = r8 & 448;
        if(r8 !=0) //_LBB26_317
{
        r8 = heap32[(r2)];
        r8 = (r8 + 1)|0;
        heap32[(r2)] = r8;
        r8 = (r9 + -5)|0;
        heap32[(g0)] = 75;
        f9 = r8; //fitos r8, f9
        f10 =                       0.5;
        malloc(i7);
        r8 = r_g0;
        f9 = f9*f10;
        if(r8 !=0) //_LBB26_319
{
        r9 = 0;
        r11 = (r8 + 4)|0;
        r9 = (r9 - r11)|0;
        r9 = r9 & 15;
        r9 = (r8 + r9)|0;
        r11 = (r9 + 4)|0;
        r9 = r9 >> 2;
        heap32[(r9)] = r8;
        r8 = r11;
}
        r9 = r8 >> 2;
        heap32[(r9+2)] = 0;
        heap32[(r9+3)] = 1065353216;
        heap32[(r9+4)] = 1065353216;
        heap32[(r9+5)] = 1065353216;
        r11 = _ZTV14btCapsuleShape;
        heap32[(r9+6)] = 0;
        r11 = (r11 + 8)|0;
        heap32[(r9+11)] = 1025758986;
        heap32[(r9)] = r11;
        heap32[(r9+1)] = 10;
        heap32[(r9+13)] = 1;
        f11 = f9+f9;
        f10 = f11*f10;
        heapFloat[(r9+7)] = f9;
        heapFloat[(r9+8)] = f10;
        heapFloat[(r9+9)] = f9;
        heap32[(r9+10)] = 0;
        r9 = heap32[(r1)];
        r9 = r9 >> 2;
        r9 = heap32[(r9+2)];
        heap32[(g0)] = r6;
        heapFloat[(g0+1)] = f9;
        heap32[(g0+2)] = r10;
        heap32[(g0+3)] = r8;
        __FUNCTION_TABLE__[(r9)>>2](i7);
}
}
}
}
        r7 = (r7 + 1)|0;
if(!(r7 !=10)) //_LBB26_305
{
break _195;
}
}
        r5 = (r5 + 1)|0;
if(!(r5 !=10)) //_LBB26_304
{
break _193;
}
}
        f4 =        1.1000000238418579;
        f5 =      -0.05000000074505806;
        f4 = f1*f4;
        f1 = f1*f5;
        f5 =                         9;
        f3 = f4+f3;
        f1 = f1*f5;
        r3 = (r3 + -1)|0;
        f0 = f0+f3;
        f2 = f1+f2;
        f1 = f4;
if(!(r3 !=0)) //_LBB26_303
{
break _191;
}
}
        _ZN13BenchmarkDemo19createLargeMeshBodyEv(i7);
        return;
}
else{
break _161;
}
}
}
}
else{
        if(r0 ==1) //_LBB26_156
{
        r3 = heap32[(r2)];
        r3 = (r3 + 1)|0;
        heap32[(r2)] = r3;
        heap32[(g0)] = 71;
        malloc(i7);
        r2 = r_g0;
        if(r2 !=0) //_LBB26_158
{
        r3 = 0;
        r4 = (r2 + 4)|0;
        r3 = (r3 - r4)|0;
        r3 = r3 & 15;
        r3 = (r2 + r3)|0;
        r4 = (r3 + 4)|0;
        r3 = r3 >> 2;
        heap32[(r3)] = r2;
        r2 = r4;
}
        r3 = r2 >> 2;
        heap32[(r3+2)] = 0;
        heap32[(r3+3)] = 1065353216;
        heap32[(r3+4)] = 1065353216;
        heap32[(r3+5)] = 1065353216;
        r4 = _ZTV10btBoxShape;
        heap32[(r3+6)] = 0;
        r4 = (r4 + 8)|0;
        heap32[(r3+11)] = 1025758986;
        heap32[(r3)] = r4;
        heap32[(r3+1)] = 0;
        heap32[(r3+7)] = 1064682127;
        heap32[(r3+8)] = 1064682127;
        heap32[(r3+9)] = 1064682127;
        r4 = sp + -1152;
        heap32[(r3+10)] = 0;
        r3 = r4 >> 2;
        heap32[(fp+-288)] = 0;
        heap32[(r3+1)] = 0;
        heap32[(r3+2)] = 0;
        heap32[(r3+3)] = 0;
        heap32[(g0)] = r2;
        heap32[(g0+1)] = 1073741824;
        heap32[(g0+2)] = r4;
        _ZNK10btBoxShape21calculateLocalInertiaEfR9btVector3(i7);
        r3 = sp + -1216;
        r4 = r3 >> 2;
        heap32[(fp+-304)] = 1065353216;
        heap32[(r4+1)] = 0;
        heap32[(r4+2)] = 0;
        heap32[(r4+3)] = 0;
        heap32[(r4+4)] = 0;
        heap32[(r4+5)] = 1065353216;
        heap32[(r4+6)] = 0;
        heap32[(r4+7)] = 0;
        heap32[(r4+8)] = 0;
        heap32[(r4+9)] = 0;
        heap32[(r4+10)] = 1065353216;
        heap32[(r4+11)] = 0;
        heap32[(r4+12)] = 0;
        heap32[(r4+13)] = 0;
        r0 = 47;
        f0 =                       -12;
        f1 =                         2;
        heap32[(r4+14)] = 0;
        heap32[(r4+15)] = 0;
_225: while(true){
        r5 = 0;
        r7 = r5;
_227: while(true){
        f2 = r7; //fitos r7, f2
        f3 =                         3;
        f2 = f2*f3;
        f2 = f2+f0;
        r8 = r5;
_229: while(true){
        f4 = r8; //fitos r8, f4
        f4 = f4*f3;
        f4 = f4+f0;
        heapFloat[(r4+12)] = f4;
        heapFloat[(r4+13)] = f1;
        heapFloat[(r4+14)] = f2;
        heap32[(r4+15)] = 0;
        r9 = heap32[(r1)];
        r9 = r9 >> 2;
        r9 = heap32[(r9+2)];
        heap32[(g0)] = r6;
        heap32[(g0+1)] = 1073741824;
        heap32[(g0+2)] = r3;
        heap32[(g0+3)] = r2;
        r8 = (r8 + 1)|0;
        __FUNCTION_TABLE__[(r9)>>2](i7);
if(!(r8 !=8)) //_LBB26_161
{
break _229;
}
}
        r7 = (r7 + 1)|0;
if(!(r7 !=8)) //_LBB26_160
{
break _227;
}
}
        f2 =      -0.34999999403953552;
        r0 = (r0 + -1)|0;
        f1 = f1+f3;
        f0 = f0+f2;
        if(r0 ==0) //_LBB26_165
{
break _161;
}
else{
continue _225;
}
}
}
else{
        if(r0 ==2) //_LBB26_166
{
        r3 = heap32[(r2)];
        r3 = (r3 + 1)|0;
        heap32[(r2)] = r3;
        heap32[(g0)] = 71;
        malloc(i7);
        r3 = r_g0;
        if(r3 !=0) //_LBB26_168
{
        r4 = 0;
        r0 = (r3 + 4)|0;
        r4 = (r4 - r0)|0;
        r4 = r4 & 15;
        r4 = (r3 + r4)|0;
        r0 = (r4 + 4)|0;
        r4 = r4 >> 2;
        heap32[(r4)] = r3;
        r3 = r0;
}
        r4 = r3 >> 2;
        heap32[(r4+2)] = 0;
        heap32[(r4+3)] = 1065353216;
        heap32[(r4+4)] = 1065353216;
        heap32[(r4+5)] = 1065353216;
        r0 = _ZTV10btBoxShape;
        heap32[(r4+6)] = 0;
        r0 = (r0 + 8)|0;
        heap32[(r4+11)] = 1025758986;
        heap32[(r4)] = r0;
        heap32[(r4+1)] = 0;
        heap32[(r4+7)] = 1064682127;
        heap32[(r4+8)] = 1064682127;
        heap32[(r4+9)] = 1064682127;
        r5 = sp + -1024;
        heap32[(r4+10)] = 0;
        r4 = r5 >> 2;
        heap32[(fp+-256)] = 1065353216;
        heap32[(r4+1)] = 0;
        heap32[(r4+2)] = 0;
        heap32[(r4+3)] = 0;
        heap32[(r4+4)] = 0;
        heap32[(r4+5)] = 1065353216;
        heap32[(r4+6)] = 0;
        heap32[(r4+7)] = 0;
        heap32[(r4+8)] = 0;
        heap32[(r4+9)] = 0;
        heap32[(r4+10)] = 1065353216;
        heap32[(r4+11)] = 0;
        heap32[(r4+12)] = 0;
        heap32[(r4+13)] = 0;
        heap32[(r4+14)] = 0;
        r7 = sp + -1040;
        heap32[(r4+15)] = 0;
        r8 = r7 >> 2;
        heap32[(fp+-260)] = 0;
        heap32[(r8+1)] = 0;
        heap32[(r8+2)] = 0;
        heap32[(r8+3)] = 0;
        heap32[(g0)] = r3;
        heap32[(g0+1)] = 1065353216;
        heap32[(g0+2)] = r7;
        r7 = 12;
        f0 =       -12.240598678588867;
        f1 =                         1;
        _ZNK10btBoxShape21calculateLocalInertiaEfR9btVector3(i7);
        f2 = f0;
        f8 =                       -20;
_239: while(true){
if(!(r7 <1)) //_LBB26_173
{
        f4 =                         0;
        f3 = f1+f4;
        r8 = 0;
        r9 = r8;
_243: while(true){
        f5 = r9; //fitos r9, f5
        f6 =        2.0400998592376709;
        f5 = f5*f6;
        f5 = f5+f0;
        f5 = f5+f4;
        r10 = r8;
_245: while(true){
        f7 = r10; //fitos r10, f7
        f7 = f7*f6;
        f7 = f7+f2;
        f7 = f7+f8;
        heapFloat[(r4+12)] = f7;
        heapFloat[(r4+13)] = f3;
        heapFloat[(r4+14)] = f5;
        heap32[(r4+15)] = 0;
        r11 = heap32[(r1)];
        r11 = r11 >> 2;
        r11 = heap32[(r11+2)];
        heap32[(g0)] = r6;
        heap32[(g0+1)] = 1065353216;
        heap32[(g0+2)] = r5;
        heap32[(g0+3)] = r3;
        r10 = (r10 + 1)|0;
        __FUNCTION_TABLE__[(r11)>>2](i7);
if(!(r7 !=r10)) //_LBB26_171
{
break _245;
}
}
        r9 = (r9 + 1)|0;
if(!(r7 !=r9)) //_LBB26_170
{
break _243;
}
}
}
        f3 =        2.0400998592376709;
        f4 =        1.0199999809265137;
        r7 = (r7 + -1)|0;
        f1 = f1+f3;
        f0 = f0+f4;
        f2 = f2+f4;
        if(r7 ==0) //_LBB26_176
{
break _239;
}
}
        r3 = sp + -1136;
        r4 = r3 >> 2;
        heap32[(fp+-284)] = 1065353216;
        heap32[(r4+1)] = 1065353216;
        heap32[(r4+2)] = 1065353216;
        r5 = sp + -1120;
        heap32[(r4+3)] = 0;
        r4 = r5 >> 2;
        heap32[(fp+-280)] = -1073741824;
        heap32[(r4+1)] = 0;
        heap32[(r4+2)] = 0;
        heap32[(r4+3)] = 0;
        heap32[(g0)] = r5;
        heap32[(g0+1)] = r3;
        r3 = sp + -1104;
        _ZN13BenchmarkDemo10createWallERK9btVector3iS2_(i7);
        r4 = r3 >> 2;
        heap32[(fp+-276)] = 1065353216;
        heap32[(r4+1)] = 1065353216;
        heap32[(r4+2)] = 1065353216;
        r5 = sp + -1088;
        heap32[(r4+3)] = 0;
        r4 = r5 >> 2;
        heap32[(fp+-272)] = 1082130432;
        heap32[(r4+1)] = 0;
        heap32[(r4+2)] = 0;
        heap32[(r4+3)] = 0;
        heap32[(g0)] = r5;
        heap32[(g0+1)] = r3;
        r3 = sp + -1072;
        _ZN13BenchmarkDemo10createWallERK9btVector3iS2_(i7);
        r4 = r3 >> 2;
        heap32[(fp+-268)] = 1065353216;
        heap32[(r4+1)] = 1065353216;
        heap32[(r4+2)] = 1065353216;
        r5 = sp + -1056;
        heap32[(r4+3)] = 0;
        r4 = r5 >> 2;
        heap32[(fp+-264)] = 1092616192;
        heap32[(r4+1)] = 0;
        heap32[(r4+2)] = 0;
        heap32[(r4+3)] = 0;
        heap32[(g0)] = r5;
        heap32[(g0+1)] = r3;
        _ZN13BenchmarkDemo10createWallERK9btVector3iS2_(i7);
        r3 = heap32[(r2)];
        r3 = (r3 + 1)|0;
        heap32[(r2)] = r3;
        heap32[(g0)] = 71;
        malloc(i7);
        r2 = r_g0;
        if(r2 !=0) //_LBB26_178
{
        r3 = 0;
        r4 = (r2 + 4)|0;
        r3 = (r3 - r4)|0;
        r3 = r3 & 15;
        r3 = (r2 + r3)|0;
        r4 = (r3 + 4)|0;
        r3 = r3 >> 2;
        heap32[(r3)] = r2;
        r2 = r4;
}
        r3 = r2 >> 2;
        heap32[(r3+2)] = 0;
        heap32[(r3+3)] = 1065353216;
        heap32[(r3+4)] = 1065353216;
        heap32[(r3+5)] = 1065353216;
        heap32[(r3+6)] = 0;
        heap32[(r3+11)] = 1025758986;
        heap32[(r3)] = r0;
        heap32[(r3+1)] = 0;
        heap32[(r3+7)] = 1064682127;
        heap32[(r3+8)] = 1064682127;
        heap32[(r3+9)] = 1064682127;
        r4 = sp + -944;
        heap32[(r3+10)] = 0;
        r3 = r4 >> 2;
        heap32[(fp+-236)] = 1065353216;
        heap32[(r3+1)] = 0;
        heap32[(r3+2)] = 0;
        heap32[(r3+3)] = 0;
        heap32[(r3+4)] = 0;
        heap32[(r3+5)] = 1065353216;
        heap32[(r3+6)] = 0;
        heap32[(r3+7)] = 0;
        heap32[(r3+8)] = 0;
        heap32[(r3+9)] = 0;
        heap32[(r3+10)] = 1065353216;
        heap32[(r3+11)] = 0;
        heap32[(r3+12)] = 0;
        heap32[(r3+13)] = 0;
        heap32[(r3+14)] = 0;
        r0 = sp + -960;
        heap32[(r3+15)] = 0;
        r5 = r0 >> 2;
        heap32[(fp+-240)] = 0;
        heap32[(r5+1)] = 0;
        heap32[(r5+2)] = 0;
        heap32[(r5+3)] = 0;
        heap32[(g0)] = r2;
        heap32[(g0+1)] = 1065353216;
        heap32[(g0+2)] = r0;
        r0 = 8;
        f0 =                         1;
        f1 =                         0;
        _ZNK10btBoxShape21calculateLocalInertiaEfR9btVector3(i7);
        f2 = f1;
        f3 = f0;
        f4 = f1;
_253: while(true){
        r5 = 24;
_255: while(true){
        f5 =        9.9312677383422852;
        f6 =                         0;
        f7 = f3*f0;
        f8 = f4*f6;
        f9 = f3*f5;
        f10 = f1*f6;
        f7 = f8+f7;
        f11 = f2*f5;
        f9 = f10+f9;
        f12 = f2*f0;
        f13 = f1*f0;
        f14 = f2*f6;
        f7 = f7+f11;
        f9 = f9-f12;
        f11 = f13+f14;
        f12 = f4*f5;
        f5 = f1*f5;
        f13 = f4*f0;
        f11 = f11-f12;
        f12 = f4*f4;
        f15 = f3*f3;
        f16 = f4*f7;
        f17 = f1*f9;
        f5 = f5+f13;
        f13 = f3*f6;
        f5 = f5-f13;
        f12 = f12+f15;
        f15 = f2*f2;
        f18 = f3*f7;
        f19 = f1*f11;
        f16 = f16+f17;
        f17 = f2*f11;
        f12 = f12+f15;
        f15 = f1*f1;
        f7 = f2*f7;
        f20 = f1*f5;
        f18 = f18+f19;
        f19 = f4*f5;
        f16 = f16-f17;
        f5 = f3*f5;
        f17 =                         2;
        f12 = f12+f15;
        f7 = f7+f20;
        f15 = f3*f9;
        f18 = f18-f19;
        f9 = f2*f9;
        f5 = f16+f5;
        f16 =                        25;
        f12 = f17/f12;
        f9 = f18+f9;
        f5 = f5+f16;
        f7 = f7-f15;
        f11 = f4*f11;
        f15 = f2*f12;
        f16 = f3*f12;
        f7 = f7+f11;
        f9 = f9+f6;
        heapFloat[(r3+12)] = f5;
        f5 = f3*f16;
        f11 = f2*f15;
        f7 = f7+f6;
        heapFloat[(r3+13)] = f9;
        heapFloat[(r3+14)] = f7;
        f7 =                         1;
        f9 = f5+f11;
        f18 = f4*f16;
        f19 = f1*f15;
        f9 = f7-f9;
        heap32[(r3+15)] = 0;
        f20 = f4*f15;
        f16 = f1*f16;
        f21 = f18-f19;
        heapFloat[(fp+-236)] = f9;
        f9 = f4*f12;
        f12 = f20+f16;
        heapFloat[(r3+1)] = f21;
        f21 = f4*f9;
        heapFloat[(r3+2)] = f12;
        f11 = f21+f11;
        f12 = f18+f19;
        heap32[(r3+3)] = 0;
        f15 = f3*f15;
        f9 = f1*f9;
        f11 = f7-f11;
        heapFloat[(r3+4)] = f12;
        f12 = f15-f9;
        heapFloat[(r3+5)] = f11;
        heapFloat[(r3+6)] = f12;
        f11 = f20-f16;
        heap32[(r3+7)] = 0;
        f5 = f21+f5;
        f9 = f15+f9;
        heapFloat[(r3+8)] = f11;
        f5 = f7-f5;
        heapFloat[(r3+9)] = f9;
        heapFloat[(r3+10)] = f5;
        heap32[(r3+11)] = 0;
        r7 = heap32[(r1)];
        r7 = r7 >> 2;
        f5 =       0.99144488573074341;
        f7 =       0.13052619993686676;
        r7 = heap32[(r7+2)];
        f9 = f1*f5;
        f11 = f2*f5;
        f12 = f4*f5;
        f1 = f1*f7;
        f5 = f3*f5;
        f1 = f1+f5;
        f5 = f10+f12;
        f9 = f9-f8;
        f3 = f3*f7;
        f10 = f10+f11;
        f4 = f4*f7;
        f3 = f9-f3;
        f4 = f10+f4;
        f9 = f1+f14;
        heap32[(g0)] = r6;
        heap32[(g0+1)] = 1065353216;
        heap32[(g0+2)] = r4;
        heap32[(g0+3)] = r2;
        f5 = f5+f13;
        f7 = f2*f7;
        r5 = (r5 + -1)|0;
        f1 = f3-f14;
        f2 = f4-f13;
        f3 = f9-f8;
        f4 = f5-f7;
        __FUNCTION_TABLE__[(r7)>>2](i7);
if(!(r5 !=0)) //_LBB26_180
{
break _255;
}
}
        f5 =       0.99785894155502319;
        f7 =      0.065403133630752563;
        f8 = f4*f5;
        f9 = f1*f6;
        f10 = f2*f5;
        f11 = f1*f7;
        f12 = f3*f5;
        f1 = f1*f5;
        f5 = f4*f6;
        f8 = f9+f8;
        f13 = f3*f6;
        f11 = f11+f12;
        f6 = f2*f6;
        f9 = f9+f10;
        f4 = f4*f7;
        f1 = f1-f5;
        f3 = f3*f7;
        f10 = f11+f6;
        f9 = f9+f4;
        f1 = f1-f3;
        f3 = f8+f13;
        f2 = f2*f7;
        r0 = (r0 + -1)|0;
        f4 = f3-f2;
        f3 = f10-f5;
        f2 = f9-f13;
        f1 = f1-f6;
        f0 = f0+f17;
        if(r0 ==0) //_LBB26_165
{
break _161;
}
else{
continue _253;
}
}
}
else{
        if(r0 ==3) //_LBB26_183
{
        r3 = -16;
        r0 = 16;
        f0 =                         0;
        f1 =                         1;
        f2 = f0;
_260: while(true){
        f3 = r3; //fitos r3, f3
        f4 =                         6;
        f3 = f3*f4;
        f5 =                       0.5;
        f6 =                       3.5;
        f7 =        4.2000002861022949;
        f8 =        5.5999999046325684;
        f9 =        2.2749998569488525;
        f10 =       0.69999998807907104;
        f11 =        5.0750002861022949;
        f3 = f3*f5;
        f5 = f2+f0;
        f6 = f1+f6;
        f7 = f1+f7;
        f8 = f1+f8;
        f9 = f1+f9;
        f10 = f1+f10;
        f11 = f1+f11;
        r5 = 0;
_262: while(true){
        heap32[(g0)] = 136;
        _Znwj(i7);
        r7 = r_g0;
        r8 = _ZTV7RagDoll;
        r9 = r7 >> 2;
        r10 = heap32[(r1+1)];
        r8 = (r8 + 8)|0;
        heap32[(r9)] = r8;
        heap32[(r9+1)] = r10;
        r8 = heap32[(r2)];
        r9 = (r8 + 1)|0;
        heap32[(r2)] = r9;
        f12 = r5; //fitos r5, f12
        heap32[(g0)] = 75;
        f12 = f12*f4;
        malloc(i7);
        r9 = r_g0;
        f12 = f12+f3;
        if(r9 !=0) //_LBB26_187
{
        r10 = 0;
        r11 = (r9 + 4)|0;
        r10 = (r10 - r11)|0;
        r10 = r10 & 15;
        r10 = (r9 + r10)|0;
        r11 = (r10 + 4)|0;
        r10 = r10 >> 2;
        heap32[(r10)] = r9;
        r9 = r11;
}
        r10 = r9 >> 2;
        heap32[(r10+2)] = 0;
        heap32[(r10+3)] = 1065353216;
        heap32[(r10+4)] = 1065353216;
        heap32[(r10+5)] = 1065353216;
        r11 = _ZTV14btCapsuleShape;
        heap32[(r10+6)] = 0;
        r11 = (r11 + 8)|0;
        heap32[(r10+11)] = 1025758986;
        heap32[(r10)] = r11;
        heap32[(r10+1)] = 10;
        heap32[(r10+13)] = 1;
        heap32[(r10+7)] = 1057384039;
        heap32[(r10+8)] = 1051931443;
        heap32[(r10+9)] = 1057384039;
        r12 = r7 >> 2;
        heap32[(r10+10)] = 0;
        r10 = (r8 + 2)|0;
        heap32[(r12+2)] = r9;
        heap32[(r2)] = r10;
        heap32[(g0)] = 75;
        malloc(i7);
        r9 = r_g0;
        if(r9 !=0) //_LBB26_190
{
        r10 = 0;
        r13 = (r9 + 4)|0;
        r10 = (r10 - r13)|0;
        r10 = r10 & 15;
        r10 = (r9 + r10)|0;
        r13 = (r10 + 4)|0;
        r10 = r10 >> 2;
        heap32[(r10)] = r9;
        r9 = r13;
}
        r10 = r9 >> 2;
        heap32[(r10+2)] = 0;
        heap32[(r10+3)] = 1065353216;
        heap32[(r10+4)] = 1065353216;
        heap32[(r10+5)] = 1065353216;
        heap32[(r10+6)] = 0;
        heap32[(r10+11)] = 1025758986;
        heap32[(r10)] = r11;
        heap32[(r10+1)] = 10;
        heap32[(r10+13)] = 1;
        heap32[(r10+7)] = 1057384039;
        heap32[(r10+8)] = 1056629064;
        heap32[(r10+9)] = 1057384039;
        heap32[(r10+10)] = 0;
        r10 = (r8 + 3)|0;
        heap32[(r12+3)] = r9;
        heap32[(r2)] = r10;
        heap32[(g0)] = 75;
        malloc(i7);
        r9 = r_g0;
        if(r9 !=0) //_LBB26_193
{
        r10 = 0;
        r13 = (r9 + 4)|0;
        r10 = (r10 - r13)|0;
        r10 = r10 & 15;
        r10 = (r9 + r10)|0;
        r13 = (r10 + 4)|0;
        r10 = r10 >> 2;
        heap32[(r10)] = r9;
        r9 = r13;
}
        r10 = r9 >> 2;
        heap32[(r10+2)] = 0;
        heap32[(r10+3)] = 1065353216;
        heap32[(r10+4)] = 1065353216;
        heap32[(r10+5)] = 1065353216;
        heap32[(r10+6)] = 0;
        heap32[(r10+11)] = 1025758986;
        heap32[(r10)] = r11;
        heap32[(r10+1)] = 10;
        heap32[(r10+13)] = 1;
        heap32[(r10+7)] = 1051931443;
        heap32[(r10+8)] = 1035154227;
        heap32[(r10+9)] = 1051931443;
        heap32[(r10+10)] = 0;
        r10 = (r8 + 4)|0;
        heap32[(r12+4)] = r9;
        heap32[(r2)] = r10;
        heap32[(g0)] = 75;
        malloc(i7);
        r9 = r_g0;
        if(r9 !=0) //_LBB26_196
{
        r10 = 0;
        r13 = (r9 + 4)|0;
        r10 = (r10 - r13)|0;
        r10 = r10 & 15;
        r10 = (r9 + r10)|0;
        r13 = (r10 + 4)|0;
        r10 = r10 >> 2;
        heap32[(r10)] = r9;
        r9 = r13;
}
        r10 = r9 >> 2;
        heap32[(r10+2)] = 0;
        heap32[(r10+3)] = 1065353216;
        heap32[(r10+4)] = 1065353216;
        heap32[(r10+5)] = 1065353216;
        heap32[(r10+6)] = 0;
        heap32[(r10+11)] = 1025758986;
        heap32[(r10)] = r11;
        heap32[(r10+1)] = 10;
        heap32[(r10+13)] = 1;
        heap32[(r10+7)] = 1048240456;
        heap32[(r10+8)] = 1061788057;
        heap32[(r10+9)] = 1048240456;
        heap32[(r10+10)] = 0;
        r10 = (r8 + 5)|0;
        heap32[(r12+5)] = r9;
        heap32[(r2)] = r10;
        heap32[(g0)] = 75;
        malloc(i7);
        r9 = r_g0;
        if(r9 !=0) //_LBB26_199
{
        r10 = 0;
        r13 = (r9 + 4)|0;
        r10 = (r10 - r13)|0;
        r10 = r10 & 15;
        r10 = (r9 + r10)|0;
        r13 = (r10 + 4)|0;
        r10 = r10 >> 2;
        heap32[(r10)] = r9;
        r9 = r13;
}
        r10 = r9 >> 2;
        heap32[(r10+2)] = 0;
        heap32[(r10+3)] = 1065353216;
        heap32[(r10+4)] = 1065353216;
        heap32[(r10+5)] = 1065353216;
        heap32[(r10+6)] = 0;
        heap32[(r10+11)] = 1025758986;
        heap32[(r10)] = r11;
        heap32[(r10+1)] = 10;
        heap32[(r10+13)] = 1;
        heap32[(r10+7)] = 1043542835;
        heap32[(r10+8)] = 1059439248;
        heap32[(r10+9)] = 1043542835;
        heap32[(r10+10)] = 0;
        r10 = (r8 + 6)|0;
        heap32[(r12+6)] = r9;
        heap32[(r2)] = r10;
        heap32[(g0)] = 75;
        malloc(i7);
        r9 = r_g0;
        if(r9 !=0) //_LBB26_202
{
        r10 = 0;
        r13 = (r9 + 4)|0;
        r10 = (r10 - r13)|0;
        r10 = r10 & 15;
        r10 = (r9 + r10)|0;
        r13 = (r10 + 4)|0;
        r10 = r10 >> 2;
        heap32[(r10)] = r9;
        r9 = r13;
}
        r10 = r9 >> 2;
        heap32[(r10+2)] = 0;
        heap32[(r10+3)] = 1065353216;
        heap32[(r10+4)] = 1065353216;
        heap32[(r10+5)] = 1065353216;
        heap32[(r10+6)] = 0;
        heap32[(r10+11)] = 1025758986;
        heap32[(r10)] = r11;
        heap32[(r10+1)] = 10;
        heap32[(r10+13)] = 1;
        heap32[(r10+7)] = 1048240456;
        heap32[(r10+8)] = 1061788057;
        heap32[(r10+9)] = 1048240456;
        heap32[(r10+10)] = 0;
        r10 = (r8 + 7)|0;
        heap32[(r12+7)] = r9;
        heap32[(r2)] = r10;
        heap32[(g0)] = 75;
        malloc(i7);
        r9 = r_g0;
        if(r9 !=0) //_LBB26_205
{
        r10 = 0;
        r13 = (r9 + 4)|0;
        r10 = (r10 - r13)|0;
        r10 = r10 & 15;
        r10 = (r9 + r10)|0;
        r13 = (r10 + 4)|0;
        r10 = r10 >> 2;
        heap32[(r10)] = r9;
        r9 = r13;
}
        r10 = r9 >> 2;
        heap32[(r10+2)] = 0;
        heap32[(r10+3)] = 1065353216;
        heap32[(r10+4)] = 1065353216;
        heap32[(r10+5)] = 1065353216;
        heap32[(r10+6)] = 0;
        heap32[(r10+11)] = 1025758986;
        heap32[(r10)] = r11;
        heap32[(r10+1)] = 10;
        heap32[(r10+13)] = 1;
        heap32[(r10+7)] = 1043542835;
        heap32[(r10+8)] = 1059439248;
        heap32[(r10+9)] = 1043542835;
        heap32[(r10+10)] = 0;
        r10 = (r8 + 8)|0;
        heap32[(r12+8)] = r9;
        heap32[(r2)] = r10;
        heap32[(g0)] = 75;
        malloc(i7);
        r9 = r_g0;
        if(r9 !=0) //_LBB26_208
{
        r10 = 0;
        r13 = (r9 + 4)|0;
        r10 = (r10 - r13)|0;
        r10 = r10 & 15;
        r10 = (r9 + r10)|0;
        r13 = (r10 + 4)|0;
        r10 = r10 >> 2;
        heap32[(r10)] = r9;
        r9 = r13;
}
        r10 = r9 >> 2;
        heap32[(r10+2)] = 0;
        heap32[(r10+3)] = 1065353216;
        heap32[(r10+4)] = 1065353216;
        heap32[(r10+5)] = 1065353216;
        heap32[(r10+6)] = 0;
        heap32[(r10+11)] = 1025758986;
        heap32[(r10)] = r11;
        heap32[(r10+1)] = 10;
        heap32[(r10+13)] = 1;
        heap32[(r10+7)] = 1043542835;
        heap32[(r10+8)] = 1058264843;
        heap32[(r10+9)] = 1043542835;
        heap32[(r10+10)] = 0;
        r10 = (r8 + 9)|0;
        heap32[(r12+9)] = r9;
        heap32[(r2)] = r10;
        heap32[(g0)] = 75;
        malloc(i7);
        r9 = r_g0;
        if(r9 !=0) //_LBB26_211
{
        r10 = 0;
        r13 = (r9 + 4)|0;
        r10 = (r10 - r13)|0;
        r10 = r10 & 15;
        r10 = (r9 + r10)|0;
        r13 = (r10 + 4)|0;
        r10 = r10 >> 2;
        heap32[(r10)] = r9;
        r9 = r13;
}
        r10 = r9 >> 2;
        heap32[(r10+2)] = 0;
        heap32[(r10+3)] = 1065353216;
        heap32[(r10+4)] = 1065353216;
        heap32[(r10+5)] = 1065353216;
        heap32[(r10+6)] = 0;
        heap32[(r10+11)] = 1025758986;
        heap32[(r10)] = r11;
        heap32[(r10+1)] = 10;
        heap32[(r10+13)] = 1;
        heap32[(r10+7)] = 1041194025;
        heap32[(r10+8)] = 1054867456;
        heap32[(r10+9)] = 1041194025;
        heap32[(r10+10)] = 0;
        r10 = (r8 + 10)|0;
        heap32[(r12+10)] = r9;
        heap32[(r2)] = r10;
        heap32[(g0)] = 75;
        malloc(i7);
        r9 = r_g0;
        if(r9 !=0) //_LBB26_214
{
        r10 = 0;
        r13 = (r9 + 4)|0;
        r10 = (r10 - r13)|0;
        r10 = r10 & 15;
        r10 = (r9 + r10)|0;
        r13 = (r10 + 4)|0;
        r10 = r10 >> 2;
        heap32[(r10)] = r9;
        r9 = r13;
}
        r10 = r9 >> 2;
        heap32[(r10+2)] = 0;
        heap32[(r10+3)] = 1065353216;
        heap32[(r10+4)] = 1065353216;
        heap32[(r10+5)] = 1065353216;
        heap32[(r10+6)] = 0;
        heap32[(r10+11)] = 1025758986;
        heap32[(r10)] = r11;
        heap32[(r10+1)] = 10;
        heap32[(r10+13)] = 1;
        heap32[(r10+7)] = 1043542835;
        heap32[(r10+8)] = 1058264843;
        heap32[(r10+9)] = 1043542835;
        heap32[(r10+10)] = 0;
        r8 = (r8 + 11)|0;
        heap32[(r12+11)] = r9;
        heap32[(r2)] = r8;
        heap32[(g0)] = 75;
        malloc(i7);
        r8 = r_g0;
        if(r8 !=0) //_LBB26_217
{
        r9 = 0;
        r10 = (r8 + 4)|0;
        r9 = (r9 - r10)|0;
        r9 = r9 & 15;
        r9 = (r8 + r9)|0;
        r10 = (r9 + 4)|0;
        r9 = r9 >> 2;
        heap32[(r9)] = r8;
        r8 = r10;
}
        r9 = r8 >> 2;
        heap32[(r9+2)] = 0;
        heap32[(r9+3)] = 1065353216;
        heap32[(r9+4)] = 1065353216;
        heap32[(r9+5)] = 1065353216;
        heap32[(r9+6)] = 0;
        heap32[(r9+11)] = 1025758986;
        heap32[(r9)] = r11;
        heap32[(r9+1)] = 10;
        heap32[(r9+13)] = 1;
        heap32[(r9+7)] = 1041194025;
        heap32[(r9+8)] = 1054867456;
        heap32[(r9+9)] = 1041194025;
        heap32[(r9+10)] = 0;
        heap32[(r12+12)] = r8;
        r8 = heap32[(r12+2)];
        r9 = sp + -880;
        r10 = r9 >> 2;
        heap32[(fp+-220)] = 1065353216;
        heap32[(r10+1)] = 0;
        heap32[(r10+2)] = 0;
        heap32[(r10+3)] = 0;
        heap32[(r10+4)] = 0;
        heap32[(r10+5)] = 1065353216;
        heap32[(r10+6)] = 0;
        heap32[(r10+7)] = 0;
        heap32[(r10+8)] = 0;
        heap32[(r10+9)] = 0;
        heap32[(r10+10)] = 1065353216;
        f13 =                         0;
        f13 = f12+f13;
        heap32[(r10+11)] = 0;
        heapFloat[(r10+12)] = f13;
        heapFloat[(r10+13)] = f6;
        heapFloat[(r10+14)] = f5;
        heap32[(r10+15)] = 0;
        heap32[(g0)] = r7;
        heap32[(g0+1)] = r9;
        heap32[(g0+2)] = r8;
        _ZN7RagDoll20localCreateRigidBodyEfRK11btTransformP16btCollisionShape(i7);
        heap32[(r12+13)] = r_g0;
        r8 = heap32[(r12+3)];
        r9 = sp + -816;
        r10 = r9 >> 2;
        heap32[(fp+-204)] = 1065353216;
        heap32[(r10+1)] = 0;
        heap32[(r10+2)] = 0;
        heap32[(r10+3)] = 0;
        heap32[(r10+4)] = 0;
        heap32[(r10+5)] = 1065353216;
        heap32[(r10+6)] = 0;
        heap32[(r10+7)] = 0;
        heap32[(r10+8)] = 0;
        heap32[(r10+9)] = 0;
        heap32[(r10+10)] = 1065353216;
        heap32[(r10+11)] = 0;
        heapFloat[(r10+12)] = f13;
        heapFloat[(r10+13)] = f7;
        heapFloat[(r10+14)] = f5;
        heap32[(r10+15)] = 0;
        heap32[(g0)] = r7;
        heap32[(g0+1)] = r9;
        heap32[(g0+2)] = r8;
        _ZN7RagDoll20localCreateRigidBodyEfRK11btTransformP16btCollisionShape(i7);
        heap32[(r12+14)] = r_g0;
        r8 = heap32[(r12+4)];
        r9 = sp + -752;
        r10 = r9 >> 2;
        heap32[(fp+-188)] = 1065353216;
        heap32[(r10+1)] = 0;
        heap32[(r10+2)] = 0;
        heap32[(r10+3)] = 0;
        heap32[(r10+4)] = 0;
        heap32[(r10+5)] = 1065353216;
        heap32[(r10+6)] = 0;
        heap32[(r10+7)] = 0;
        heap32[(r10+8)] = 0;
        heap32[(r10+9)] = 0;
        heap32[(r10+10)] = 1065353216;
        heap32[(r10+11)] = 0;
        heapFloat[(r10+12)] = f13;
        heapFloat[(r10+13)] = f8;
        heapFloat[(r10+14)] = f5;
        heap32[(r10+15)] = 0;
        heap32[(g0)] = r7;
        heap32[(g0+1)] = r9;
        heap32[(g0+2)] = r8;
        _ZN7RagDoll20localCreateRigidBodyEfRK11btTransformP16btCollisionShape(i7);
        heap32[(r12+15)] = r_g0;
        r8 = heap32[(r12+5)];
        r9 = sp + -688;
        r10 = r9 >> 2;
        heap32[(fp+-172)] = 1065353216;
        heap32[(r10+1)] = 0;
        heap32[(r10+2)] = 0;
        heap32[(r10+3)] = 0;
        heap32[(r10+4)] = 0;
        heap32[(r10+5)] = 1065353216;
        heap32[(r10+6)] = 0;
        heap32[(r10+7)] = 0;
        heap32[(r10+8)] = 0;
        heap32[(r10+9)] = 0;
        f13 =      -0.62999999523162842;
        heap32[(r10+10)] = 1065353216;
        f13 = f12+f13;
        heap32[(r10+11)] = 0;
        heapFloat[(r10+12)] = f13;
        heapFloat[(r10+13)] = f9;
        heapFloat[(r10+14)] = f5;
        heap32[(r10+15)] = 0;
        heap32[(g0)] = r7;
        heap32[(g0+1)] = r9;
        heap32[(g0+2)] = r8;
        _ZN7RagDoll20localCreateRigidBodyEfRK11btTransformP16btCollisionShape(i7);
        heap32[(r12+16)] = r_g0;
        r8 = heap32[(r12+6)];
        r9 = sp + -624;
        r10 = r9 >> 2;
        heap32[(fp+-156)] = 1065353216;
        heap32[(r10+1)] = 0;
        heap32[(r10+2)] = 0;
        heap32[(r10+3)] = 0;
        heap32[(r10+4)] = 0;
        heap32[(r10+5)] = 1065353216;
        heap32[(r10+6)] = 0;
        heap32[(r10+7)] = 0;
        heap32[(r10+8)] = 0;
        heap32[(r10+9)] = 0;
        heap32[(r10+10)] = 1065353216;
        heap32[(r10+11)] = 0;
        heapFloat[(r10+12)] = f13;
        heapFloat[(r10+13)] = f10;
        heapFloat[(r10+14)] = f5;
        heap32[(r10+15)] = 0;
        heap32[(g0)] = r7;
        heap32[(g0+1)] = r9;
        heap32[(g0+2)] = r8;
        _ZN7RagDoll20localCreateRigidBodyEfRK11btTransformP16btCollisionShape(i7);
        heap32[(r12+17)] = r_g0;
        r8 = heap32[(r12+7)];
        r9 = sp + -560;
        r10 = r9 >> 2;
        heap32[(fp+-140)] = 1065353216;
        heap32[(r10+1)] = 0;
        heap32[(r10+2)] = 0;
        heap32[(r10+3)] = 0;
        heap32[(r10+4)] = 0;
        heap32[(r10+5)] = 1065353216;
        heap32[(r10+6)] = 0;
        heap32[(r10+7)] = 0;
        heap32[(r10+8)] = 0;
        heap32[(r10+9)] = 0;
        f13 =       0.62999999523162842;
        heap32[(r10+10)] = 1065353216;
        f13 = f12+f13;
        heap32[(r10+11)] = 0;
        heapFloat[(r10+12)] = f13;
        heapFloat[(r10+13)] = f9;
        heapFloat[(r10+14)] = f5;
        heap32[(r10+15)] = 0;
        heap32[(g0)] = r7;
        heap32[(g0+1)] = r9;
        heap32[(g0+2)] = r8;
        _ZN7RagDoll20localCreateRigidBodyEfRK11btTransformP16btCollisionShape(i7);
        heap32[(r12+18)] = r_g0;
        r8 = heap32[(r12+8)];
        r9 = sp + -496;
        r10 = r9 >> 2;
        heap32[(fp+-124)] = 1065353216;
        heap32[(r10+1)] = 0;
        heap32[(r10+2)] = 0;
        heap32[(r10+3)] = 0;
        heap32[(r10+4)] = 0;
        heap32[(r10+5)] = 1065353216;
        heap32[(r10+6)] = 0;
        heap32[(r10+7)] = 0;
        heap32[(r10+8)] = 0;
        heap32[(r10+9)] = 0;
        heap32[(r10+10)] = 1065353216;
        heap32[(r10+11)] = 0;
        heapFloat[(r10+12)] = f13;
        heapFloat[(r10+13)] = f10;
        heapFloat[(r10+14)] = f5;
        heap32[(r10+15)] = 0;
        heap32[(g0)] = r7;
        heap32[(g0+1)] = r9;
        heap32[(g0+2)] = r8;
        _ZN7RagDoll20localCreateRigidBodyEfRK11btTransformP16btCollisionShape(i7);
        heap32[(r12+19)] = r_g0;
        r8 = heap32[(r12+9)];
        r9 = sp + -432;
        r10 = r9 >> 2;
        heap32[(fp+-108)] = -1287930578;
        heap32[(r10+1)] = -1082130432;
        heap32[(r10+2)] = 0;
        heap32[(r10+3)] = 0;
        heap32[(r10+4)] = 1065353216;
        heap32[(r10+5)] = -1287930578;
        heap32[(r10+6)] = 0;
        heap32[(r10+7)] = 0;
        heap32[(r10+8)] = 0;
        heap32[(r10+9)] = 0;
        f13 =       -1.2250000238418579;
        heap32[(r10+10)] = 1065353216;
        f13 = f12+f13;
        heap32[(r10+11)] = 0;
        heapFloat[(r10+12)] = f13;
        heapFloat[(r10+13)] = f11;
        heapFloat[(r10+14)] = f5;
        heap32[(r10+15)] = 0;
        heap32[(g0)] = r7;
        heap32[(g0+1)] = r9;
        heap32[(g0+2)] = r8;
        _ZN7RagDoll20localCreateRigidBodyEfRK11btTransformP16btCollisionShape(i7);
        heap32[(r12+20)] = r_g0;
        r8 = heap32[(r12+10)];
        r9 = sp + -368;
        r10 = r9 >> 2;
        heap32[(fp+-92)] = -1287930578;
        heap32[(r10+1)] = -1082130432;
        heap32[(r10+2)] = 0;
        heap32[(r10+3)] = 0;
        heap32[(r10+4)] = 1065353216;
        heap32[(r10+5)] = -1287930578;
        heap32[(r10+6)] = 0;
        heap32[(r10+7)] = 0;
        heap32[(r10+8)] = 0;
        heap32[(r10+9)] = 0;
        f13 =       -2.4500000476837158;
        heap32[(r10+10)] = 1065353216;
        f13 = f12+f13;
        heap32[(r10+11)] = 0;
        heapFloat[(r10+12)] = f13;
        heapFloat[(r10+13)] = f11;
        heapFloat[(r10+14)] = f5;
        heap32[(r10+15)] = 0;
        heap32[(g0)] = r7;
        heap32[(g0+1)] = r9;
        heap32[(g0+2)] = r8;
        _ZN7RagDoll20localCreateRigidBodyEfRK11btTransformP16btCollisionShape(i7);
        heap32[(r12+21)] = r_g0;
        r8 = heap32[(r12+11)];
        r9 = sp + -304;
        r10 = r9 >> 2;
        heap32[(fp+-76)] = -1287930578;
        heap32[(r10+1)] = 1065353216;
        heap32[(r10+2)] = 0;
        heap32[(r10+3)] = 0;
        heap32[(r10+4)] = -1082130432;
        heap32[(r10+5)] = -1287930578;
        heap32[(r10+6)] = 0;
        heap32[(r10+7)] = 0;
        heap32[(r10+8)] = -2147483648;
        heap32[(r10+9)] = 0;
        f13 =        1.2250000238418579;
        heap32[(r10+10)] = 1065353216;
        f13 = f12+f13;
        heap32[(r10+11)] = 0;
        heapFloat[(r10+12)] = f13;
        heapFloat[(r10+13)] = f11;
        heapFloat[(r10+14)] = f5;
        heap32[(r10+15)] = 0;
        heap32[(g0)] = r7;
        heap32[(g0+1)] = r9;
        heap32[(g0+2)] = r8;
        _ZN7RagDoll20localCreateRigidBodyEfRK11btTransformP16btCollisionShape(i7);
        heap32[(r12+22)] = r_g0;
        r8 = heap32[(r12+12)];
        r9 = sp + -240;
        r10 = r9 >> 2;
        heap32[(fp+-60)] = -1287930578;
        heap32[(r10+1)] = 1065353216;
        heap32[(r10+2)] = 0;
        heap32[(r10+3)] = 0;
        heap32[(r10+4)] = -1082130432;
        heap32[(r10+5)] = -1287930578;
        heap32[(r10+6)] = 0;
        heap32[(r10+7)] = 0;
        heap32[(r10+8)] = -2147483648;
        heap32[(r10+9)] = 0;
        f13 =        2.4500000476837158;
        heap32[(r10+10)] = 1065353216;
        f12 = f12+f13;
        heap32[(r10+11)] = 0;
        heapFloat[(r10+12)] = f12;
        heapFloat[(r10+13)] = f11;
        heapFloat[(r10+14)] = f5;
        heap32[(r10+15)] = 0;
        heap32[(g0)] = r7;
        heap32[(g0+1)] = r9;
        heap32[(g0+2)] = r8;
        _ZN7RagDoll20localCreateRigidBodyEfRK11btTransformP16btCollisionShape(i7);
        r9 = 0;
        heap32[(r12+23)] = r_g0;
_297: while(true){
        r8 = r9 << 2;
        r8 = (r7 - r8)|0;
        r8 = r8 >> 2;
        r10 = heap32[(r8+13)];
        r10 = r10 >> 2;
        heap32[(r10+109)] = 1028443341;
        heap32[(r10+110)] = 1062836634;
        r10 = heap32[(r8+13)];
        r10 = r10 >> 2;
        heap32[(r10+55)] = 1061997773;
        r8 = heap32[(r8+13)];
        r8 = r8 >> 2;
        r9 = (r9 + -1)|0;
        heap32[(r8+116)] = 1070386381;
        heap32[(r8+117)] = 1075838976;
if(!(r9 !=-11)) //_LBB26_219
{
break _297;
}
}
        heap32[(g0)] = 748;
        _Znwj(i7);
        r8 = r_g0;
        r9 = heap32[(r12+14)];
        r10 = heap32[(r12+13)];
        r11 = r8 >> 2;
        heap32[(r11+1)] = 4;
        heap32[(r11+2)] = -1;
        r13 = 0;
        heap32[(r11+3)] = -1;
        heap8[r8+16] = r13;
        heap32[(r11+5)] = r10;
        heap32[(r11+6)] = r9;
        r9 = _ZTV17btHingeConstraint;
        heap32[(r11+7)] = 0;
        r9 = (r9 + 8)|0;
        heap32[(r11+8)] = 1050253722;
        heap32[(r11)] = r9;
        heap32[(r11+135)] = -1287930578;
        heap32[(r11+136)] = 0;
        heap32[(r11+137)] = 1065353216;
        heap32[(r11+138)] = 0;
        heap32[(r11+139)] = -2147483648;
        heap32[(r11+140)] = 1065353216;
        heap32[(r11+141)] = 0;
        heap32[(r11+142)] = 0;
        heap32[(r11+143)] = -1082130432;
        heap32[(r11+144)] = -2147483648;
        heap32[(r11+145)] = -1287930578;
        heap32[(r11+146)] = 0;
        heap32[(r11+147)] = 0;
        heap32[(r11+148)] = 1057384039;
        heap32[(r11+149)] = 0;
        heap32[(r11+150)] = 0;
        heap32[(r11+151)] = -1287930578;
        heap32[(r11+152)] = 0;
        heap32[(r11+153)] = 1065353216;
        heap32[(r11+154)] = 0;
        heap32[(r11+155)] = -2147483648;
        heap32[(r11+156)] = 1065353216;
        heap32[(r11+157)] = 0;
        heap32[(r11+158)] = 0;
        heap32[(r11+159)] = -1082130432;
        heap32[(r11+160)] = -2147483648;
        heap32[(r11+161)] = -1287930578;
        heap32[(r11+162)] = 0;
        heap32[(r11+163)] = 0;
        heap32[(r11+164)] = -1090099609;
        heap32[(r11+165)] = 0;
        heap32[(r11+166)] = 0;
        heap8[r8+720] = r13;
        heap8[r8+721] = r13;
        heap8[r8+723] = r13;
        heap8[r8+724] = r4;
        heap8[r8+725] = r13;
        heap32[(r11+183)] = 0;
        heap32[(r11+172)] = 1065353216;
        heap32[(r11+173)] = -1082130432;
        heap32[(r11+170)] = 1050253722;
        heap32[(r11+171)] = 1065353216;
        heap32[(r11+169)] = 1063675494;
        heap8[r8+722] = r13;
        heap32[(r11+179)] = 1065353216;
        heap32[(g0)] = -1085730853;
        heap32[(g0+1)] = 1086918619;
        fmodf(i7);
        f12 = f_g0;
        f13 =       -3.1415927410125732;
        if(f12 >=f13) //_LBB26_222
{
        f14 =        3.1415927410125732;
        if(f12 >f14) //_LBB26_224
{
        f14 =       -6.2831854820251465;
        f12 = f12+f14;
}
}
else{
        f14 =        6.2831854820251465;
        f12 = f12+f14;
}
        r10 = r8 >> 2;
        heapFloat[(r10+172)] = f12;
        heap32[(g0)] = 1070141403;
        heap32[(g0+1)] = 1086918619;
        fmodf(i7);
        f12 = f_g0;
        if(f12 >=f13) //_LBB26_227
{
        f14 =        3.1415927410125732;
        if(f12 >f14) //_LBB26_229
{
        f14 =       -6.2831854820251465;
        f12 = f12+f14;
}
}
else{
        f14 =        6.2831854820251465;
        f12 = f12+f14;
}
        heapFloat[(r10+173)] = f12;
        heap32[(r10+169)] = 1063675494;
        heap32[(r10+170)] = 1050253722;
        heap32[(r10+171)] = 1065353216;
        heap32[(r12+24)] = r8;
        r10 = heap32[(r12+1)];
        r11 = r10 >> 2;
        r11 = heap32[(r11)];
        r11 = r11 >> 2;
        r11 = heap32[(r11+13)];
        heap32[(g0)] = r10;
        heap32[(g0+1)] = r8;
        heap32[(g0+2)] = 1;
        __FUNCTION_TABLE__[(r11)>>2](i7);
        heap32[(g0)] = 596;
        _Znwj(i7);
        r10 = heap32[(r12+15)];
        r11 = heap32[(r12+14)];
        r14 = r_g0 >> 2;
        heap32[(r14+1)] = 5;
        heap32[(r14+2)] = -1;
        heap32[(r14+3)] = -1;
        heap8[r_g0+16] = r13;
        heap32[(r14+5)] = r11;
        heap32[(r14+6)] = r10;
        heap32[(r14+7)] = 0;
        r10 = _ZTV21btConeTwistConstraint;
        heap32[(r14+8)] = 1050253722;
        r10 = (r10 + 8)|0;
        heap32[(r14)] = r10;
        heap32[(r14+72)] = -1287930578;
        heap32[(r14+73)] = -1082130432;
        heap32[(r14+74)] = 0;
        heap32[(r14+75)] = 0;
        heap32[(r14+76)] = 1065353216;
        heap32[(r14+77)] = -1287930578;
        heap32[(r14+78)] = 0;
        heap32[(r14+79)] = 0;
        heap32[(r14+80)] = -2147483648;
        heap32[(r14+81)] = 0;
        heap32[(r14+82)] = 1065353216;
        heap32[(r14+83)] = 0;
        heap32[(r14+84)] = 0;
        heap32[(r14+85)] = 1065772647;
        heap32[(r14+86)] = 0;
        heap32[(r14+87)] = 0;
        heap32[(r14+88)] = -1287930578;
        heap32[(r14+89)] = -1082130432;
        heap32[(r14+90)] = 0;
        heap32[(r14+91)] = 0;
        heap32[(r14+92)] = 1065353216;
        heap32[(r14+93)] = -1287930578;
        heap32[(r14+94)] = 0;
        heap32[(r14+95)] = 0;
        heap32[(r14+96)] = -2147483648;
        heap32[(r14+97)] = 0;
        heap32[(r14+98)] = 1065353216;
        heap32[(r14+99)] = 0;
        heap32[(r14+100)] = 0;
        heap32[(r14+101)] = -1090854584;
        heap32[(r14+102)] = 0;
        heap32[(r14+103)] = 0;
        heap8[r_g0+540] = r13;
        heap32[(r14+128)] = 0;
        heap32[(r14+140)] = -1082130432;
        heap32[(r14+107)] = 1008981770;
        heap32[(r14+111)] = 1028443341;
        heap32[(r14+145)] = 0;
        heap32[(r14+146)] = 0;
        heap32[(r14+147)] = 1060320051;
        heap32[(r14+148)] = 0;
        heap32[(r14+108)] = 1061752795;
        heap32[(r14+109)] = 1061752795;
        heap32[(r14+110)] = 1070141403;
        heap32[(r14+104)] = 1065353216;
        heap32[(r14+105)] = 1050253722;
        heap32[(r14+106)] = 1065353216;
        heap32[(r12+25)] = r_g0;
        r11 = heap32[(r12+1)];
        r14 = r11 >> 2;
        r14 = heap32[(r14)];
        r14 = r14 >> 2;
        r14 = heap32[(r14+13)];
        heap32[(g0)] = r11;
        heap32[(g0+1)] = r_g0;
        heap32[(g0+2)] = 1;
        __FUNCTION_TABLE__[(r14)>>2](i7);
        heap32[(g0)] = 596;
        _Znwj(i7);
        r11 = heap32[(r12+16)];
        r14 = heap32[(r12+13)];
        r15 = r_g0 >> 2;
        heap32[(r15+1)] = 5;
        heap32[(r15+2)] = -1;
        heap32[(r15+3)] = -1;
        heap8[r_g0+16] = r13;
        heap32[(r15+5)] = r14;
        heap32[(r15+6)] = r11;
        heap32[(r15+7)] = 0;
        heap32[(r15+8)] = 1050253722;
        heap32[(r15)] = r10;
        heap32[(r15+72)] = -1087044364;
        heap32[(r15+73)] = -1087044366;
        heap32[(r15+74)] = 0;
        heap32[(r15+75)] = 0;
        heap32[(r15+76)] = 1060439282;
        heap32[(r15+77)] = -1087044364;
        heap32[(r15+78)] = 0;
        heap32[(r15+79)] = 0;
        heap32[(r15+80)] = -2147483648;
        heap32[(r15+81)] = 0;
        heap32[(r15+82)] = 1065353216;
        heap32[(r15+83)] = 0;
        heap32[(r15+84)] = -1088338002;
        heap32[(r15+85)] = -1095552205;
        heap32[(r15+86)] = 0;
        heap32[(r15+87)] = 0;
        heap32[(r15+88)] = -1087044364;
        heap32[(r15+89)] = -1087044366;
        heap32[(r15+90)] = 0;
        heap32[(r15+91)] = 0;
        heap32[(r15+92)] = 1060439282;
        heap32[(r15+93)] = -1087044364;
        heap32[(r15+94)] = 0;
        heap32[(r15+95)] = 0;
        heap32[(r15+96)] = -2147483648;
        heap32[(r15+97)] = 0;
        heap32[(r15+98)] = 1065353216;
        heap32[(r15+99)] = 0;
        heap32[(r15+100)] = 0;
        heap32[(r15+101)] = 1061788057;
        heap32[(r15+102)] = 0;
        heap32[(r15+103)] = 0;
        heap8[r_g0+540] = r13;
        heap32[(r15+128)] = 0;
        heap32[(r15+140)] = -1082130432;
        heap32[(r15+107)] = 1008981770;
        heap32[(r15+111)] = 1028443341;
        heap32[(r15+145)] = 0;
        heap32[(r15+146)] = 0;
        heap32[(r15+147)] = 1060320051;
        heap32[(r15+148)] = 0;
        heap32[(r15+108)] = 1061752795;
        heap32[(r15+109)] = 1061752795;
        heap32[(r15+110)] = 0;
        heap32[(r15+104)] = 1065353216;
        heap32[(r15+105)] = 1050253722;
        heap32[(r15+106)] = 1065353216;
        heap32[(r12+26)] = r_g0;
        r11 = heap32[(r12+1)];
        r14 = r11 >> 2;
        r14 = heap32[(r14)];
        r14 = r14 >> 2;
        r14 = heap32[(r14+13)];
        heap32[(g0)] = r11;
        heap32[(g0+1)] = r_g0;
        heap32[(g0+2)] = 1;
        __FUNCTION_TABLE__[(r14)>>2](i7);
        heap32[(g0)] = 748;
        _Znwj(i7);
        r8 = r_g0;
        r11 = heap32[(r12+17)];
        r14 = heap32[(r12+16)];
        r15 = r8 >> 2;
        heap32[(r15+1)] = 4;
        heap32[(r15+2)] = -1;
        heap32[(r15+3)] = -1;
        heap8[r8+16] = r13;
        heap32[(r15+5)] = r14;
        heap32[(r15+6)] = r11;
        heap32[(r15+7)] = 0;
        heap32[(r15+8)] = 1050253722;
        heap32[(r15)] = r9;
        heap32[(r15+135)] = -1287930578;
        heap32[(r15+136)] = 0;
        heap32[(r15+137)] = 1065353216;
        heap32[(r15+138)] = 0;
        heap32[(r15+139)] = -2147483648;
        heap32[(r15+140)] = 1065353216;
        heap32[(r15+141)] = 0;
        heap32[(r15+142)] = 0;
        heap32[(r15+143)] = -1082130432;
        heap32[(r15+144)] = -2147483648;
        heap32[(r15+145)] = -1287930578;
        heap32[(r15+146)] = 0;
        heap32[(r15+147)] = 0;
        heap32[(r15+148)] = -1085695591;
        heap32[(r15+149)] = 0;
        heap32[(r15+150)] = 0;
        heap32[(r15+151)] = -1287930578;
        heap32[(r15+152)] = 0;
        heap32[(r15+153)] = 1065353216;
        heap32[(r15+154)] = 0;
        heap32[(r15+155)] = -2147483648;
        heap32[(r15+156)] = 1065353216;
        heap32[(r15+157)] = 0;
        heap32[(r15+158)] = 0;
        heap32[(r15+159)] = -1082130432;
        heap32[(r15+160)] = -2147483648;
        heap32[(r15+161)] = -1287930578;
        heap32[(r15+162)] = 0;
        heap32[(r15+163)] = 0;
        heap32[(r15+164)] = 1059439248;
        heap32[(r15+165)] = 0;
        heap32[(r15+166)] = 0;
        heap8[r8+720] = r13;
        heap8[r8+721] = r13;
        heap8[r8+723] = r13;
        heap8[r8+724] = r4;
        heap8[r8+725] = r13;
        heap32[(r15+183)] = 0;
        heap32[(r15+172)] = 1065353216;
        heap32[(r15+173)] = -1082130432;
        heap32[(r15+170)] = 1050253722;
        heap32[(r15+171)] = 1065353216;
        heap32[(r15+169)] = 1063675494;
        heap8[r8+722] = r13;
        heap32[(r15+179)] = 1065353216;
        heap32[(g0)] = 0;
        heap32[(g0+1)] = 1086918619;
        fmodf(i7);
        f12 = f_g0;
        if(f12 >=f13) //_LBB26_232
{
        f14 =        3.1415927410125732;
        if(f12 >f14) //_LBB26_234
{
        f14 =       -6.2831854820251465;
        f12 = f12+f14;
}
}
else{
        f14 =        6.2831854820251465;
        f12 = f12+f14;
}
        r11 = r8 >> 2;
        heapFloat[(r11+172)] = f12;
        heap32[(g0)] = 1070141403;
        heap32[(g0+1)] = 1086918619;
        fmodf(i7);
        f12 = f_g0;
        if(f12 >=f13) //_LBB26_237
{
        f14 =        3.1415927410125732;
        if(f12 >f14) //_LBB26_239
{
        f14 =       -6.2831854820251465;
        f12 = f12+f14;
}
}
else{
        f14 =        6.2831854820251465;
        f12 = f12+f14;
}
        heapFloat[(r11+173)] = f12;
        heap32[(r11+169)] = 1063675494;
        heap32[(r11+170)] = 1050253722;
        heap32[(r11+171)] = 1065353216;
        heap32[(r12+27)] = r8;
        r11 = heap32[(r12+1)];
        r14 = r11 >> 2;
        r14 = heap32[(r14)];
        r14 = r14 >> 2;
        r14 = heap32[(r14+13)];
        heap32[(g0)] = r11;
        heap32[(g0+1)] = r8;
        heap32[(g0+2)] = 1;
        __FUNCTION_TABLE__[(r14)>>2](i7);
        heap32[(g0)] = 596;
        _Znwj(i7);
        r11 = heap32[(r12+18)];
        r14 = heap32[(r12+13)];
        r15 = r_g0 >> 2;
        heap32[(r15+1)] = 5;
        heap32[(r15+2)] = -1;
        heap32[(r15+3)] = -1;
        heap8[r_g0+16] = r13;
        heap32[(r15+5)] = r14;
        heap32[(r15+6)] = r11;
        heap32[(r15+7)] = 0;
        heap32[(r15+8)] = 1050253722;
        heap32[(r15)] = r10;
        heap32[(r15+72)] = 1060439283;
        heap32[(r15+73)] = -1087044365;
        heap32[(r15+74)] = 0;
        heap32[(r15+75)] = 0;
        heap32[(r15+76)] = 1060439283;
        heap32[(r15+77)] = 1060439283;
        heap32[(r15+78)] = 0;
        heap32[(r15+79)] = 0;
        heap32[(r15+80)] = -2147483648;
        heap32[(r15+81)] = 0;
        heap32[(r15+82)] = 1065353216;
        heap32[(r15+83)] = 0;
        heap32[(r15+84)] = 1059145646;
        heap32[(r15+85)] = -1095552205;
        heap32[(r15+86)] = 0;
        heap32[(r15+87)] = 0;
        heap32[(r15+88)] = 1060439283;
        heap32[(r15+89)] = -1087044365;
        heap32[(r15+90)] = 0;
        heap32[(r15+91)] = 0;
        heap32[(r15+92)] = 1060439283;
        heap32[(r15+93)] = 1060439283;
        heap32[(r15+94)] = 0;
        heap32[(r15+95)] = 0;
        heap32[(r15+96)] = -2147483648;
        heap32[(r15+97)] = 0;
        heap32[(r15+98)] = 1065353216;
        heap32[(r15+99)] = 0;
        heap32[(r15+100)] = 0;
        heap32[(r15+101)] = 1061788057;
        heap32[(r15+102)] = 0;
        heap32[(r15+103)] = 0;
        heap8[r_g0+540] = r13;
        heap32[(r15+128)] = 0;
        heap32[(r15+140)] = -1082130432;
        heap32[(r15+107)] = 1008981770;
        heap32[(r15+111)] = 1028443341;
        heap32[(r15+145)] = 0;
        heap32[(r15+146)] = 0;
        heap32[(r15+147)] = 1060320051;
        heap32[(r15+148)] = 0;
        heap32[(r15+108)] = 1061752795;
        heap32[(r15+109)] = 1061752795;
        heap32[(r15+110)] = 0;
        heap32[(r15+104)] = 1065353216;
        heap32[(r15+105)] = 1050253722;
        heap32[(r15+106)] = 1065353216;
        heap32[(r12+28)] = r_g0;
        r11 = heap32[(r12+1)];
        r14 = r11 >> 2;
        r14 = heap32[(r14)];
        r14 = r14 >> 2;
        r14 = heap32[(r14+13)];
        heap32[(g0)] = r11;
        heap32[(g0+1)] = r_g0;
        heap32[(g0+2)] = 1;
        __FUNCTION_TABLE__[(r14)>>2](i7);
        heap32[(g0)] = 748;
        _Znwj(i7);
        r8 = r_g0;
        r11 = heap32[(r12+19)];
        r14 = heap32[(r12+18)];
        r15 = r8 >> 2;
        heap32[(r15+1)] = 4;
        heap32[(r15+2)] = -1;
        heap32[(r15+3)] = -1;
        heap8[r8+16] = r13;
        heap32[(r15+5)] = r14;
        heap32[(r15+6)] = r11;
        heap32[(r15+7)] = 0;
        heap32[(r15+8)] = 1050253722;
        heap32[(r15)] = r9;
        heap32[(r15+135)] = -1287930578;
        heap32[(r15+136)] = 0;
        heap32[(r15+137)] = 1065353216;
        heap32[(r15+138)] = 0;
        heap32[(r15+139)] = -2147483648;
        heap32[(r15+140)] = 1065353216;
        heap32[(r15+141)] = 0;
        heap32[(r15+142)] = 0;
        heap32[(r15+143)] = -1082130432;
        heap32[(r15+144)] = -2147483648;
        heap32[(r15+145)] = -1287930578;
        heap32[(r15+146)] = 0;
        heap32[(r15+147)] = 0;
        heap32[(r15+148)] = -1085695591;
        heap32[(r15+149)] = 0;
        heap32[(r15+150)] = 0;
        heap32[(r15+151)] = -1287930578;
        heap32[(r15+152)] = 0;
        heap32[(r15+153)] = 1065353216;
        heap32[(r15+154)] = 0;
        heap32[(r15+155)] = -2147483648;
        heap32[(r15+156)] = 1065353216;
        heap32[(r15+157)] = 0;
        heap32[(r15+158)] = 0;
        heap32[(r15+159)] = -1082130432;
        heap32[(r15+160)] = -2147483648;
        heap32[(r15+161)] = -1287930578;
        heap32[(r15+162)] = 0;
        heap32[(r15+163)] = 0;
        heap32[(r15+164)] = 1059439248;
        heap32[(r15+165)] = 0;
        heap32[(r15+166)] = 0;
        heap8[r8+720] = r13;
        heap8[r8+721] = r13;
        heap8[r8+723] = r13;
        heap8[r8+724] = r4;
        heap8[r8+725] = r13;
        heap32[(r15+183)] = 0;
        heap32[(r15+172)] = 1065353216;
        heap32[(r15+173)] = -1082130432;
        heap32[(r15+170)] = 1050253722;
        heap32[(r15+171)] = 1065353216;
        heap32[(r15+169)] = 1063675494;
        heap8[r8+722] = r13;
        heap32[(r15+179)] = 1065353216;
        heap32[(g0)] = 0;
        heap32[(g0+1)] = 1086918619;
        fmodf(i7);
        f12 = f_g0;
        if(f12 >=f13) //_LBB26_242
{
        f14 =        3.1415927410125732;
        if(f12 >f14) //_LBB26_244
{
        f14 =       -6.2831854820251465;
        f12 = f12+f14;
}
}
else{
        f14 =        6.2831854820251465;
        f12 = f12+f14;
}
        r11 = r8 >> 2;
        heapFloat[(r11+172)] = f12;
        heap32[(g0)] = 1070141403;
        heap32[(g0+1)] = 1086918619;
        fmodf(i7);
        f12 = f_g0;
        if(f12 >=f13) //_LBB26_247
{
        f14 =        3.1415927410125732;
        if(f12 >f14) //_LBB26_249
{
        f14 =       -6.2831854820251465;
        f12 = f12+f14;
}
}
else{
        f14 =        6.2831854820251465;
        f12 = f12+f14;
}
        heapFloat[(r11+173)] = f12;
        heap32[(r11+169)] = 1063675494;
        heap32[(r11+170)] = 1050253722;
        heap32[(r11+171)] = 1065353216;
        heap32[(r12+29)] = r8;
        r11 = heap32[(r12+1)];
        r14 = r11 >> 2;
        r14 = heap32[(r14)];
        r14 = r14 >> 2;
        r14 = heap32[(r14+13)];
        heap32[(g0)] = r11;
        heap32[(g0+1)] = r8;
        heap32[(g0+2)] = 1;
        __FUNCTION_TABLE__[(r14)>>2](i7);
        heap32[(g0)] = 596;
        _Znwj(i7);
        r11 = heap32[(r12+20)];
        r14 = heap32[(r12+14)];
        r15 = r_g0 >> 2;
        heap32[(r15+1)] = 5;
        heap32[(r15+2)] = -1;
        heap32[(r15+3)] = -1;
        heap8[r_g0+16] = r13;
        heap32[(r15+5)] = r14;
        heap32[(r15+6)] = r11;
        heap32[(r15+7)] = 0;
        heap32[(r15+8)] = 1050253722;
        heap32[(r15)] = r10;
        heap32[(r15+72)] = -1082130432;
        heap32[(r15+73)] = 867941678;
        heap32[(r15+74)] = -2147483648;
        heap32[(r15+75)] = 0;
        heap32[(r15+76)] = -1279541970;
        heap32[(r15+77)] = -1082130432;
        heap32[(r15+78)] = 0;
        heap32[(r15+79)] = 0;
        heap32[(r15+80)] = -2147483648;
        heap32[(r15+81)] = 0;
        heap32[(r15+82)] = 1065353216;
        heap32[(r15+83)] = 0;
        heap32[(r15+84)] = -1087163597;
        heap32[(r15+85)] = 1057384039;
        heap32[(r15+86)] = 0;
        heap32[(r15+87)] = 0;
        heap32[(r15+88)] = -1287930578;
        heap32[(r15+89)] = -1082130432;
        heap32[(r15+90)] = 0;
        heap32[(r15+91)] = 0;
        heap32[(r15+92)] = 1065353216;
        heap32[(r15+93)] = -1287930578;
        heap32[(r15+94)] = 0;
        heap32[(r15+95)] = 0;
        heap32[(r15+96)] = -2147483648;
        heap32[(r15+97)] = 0;
        heap32[(r15+98)] = 1065353216;
        heap32[(r15+99)] = 0;
        heap32[(r15+100)] = 0;
        heap32[(r15+101)] = -1088338002;
        heap32[(r15+102)] = 0;
        heap32[(r15+103)] = 0;
        heap8[r_g0+540] = r13;
        heap32[(r15+128)] = 0;
        heap32[(r15+140)] = -1082130432;
        heap32[(r15+107)] = 1008981770;
        heap32[(r15+111)] = 1028443341;
        heap32[(r15+145)] = 0;
        heap32[(r15+146)] = 0;
        heap32[(r15+147)] = 1060320051;
        heap32[(r15+148)] = 0;
        heap32[(r15+108)] = 1070141403;
        heap32[(r15+109)] = 1070141403;
        heap32[(r15+110)] = 0;
        heap32[(r15+104)] = 1065353216;
        heap32[(r15+105)] = 1050253722;
        heap32[(r15+106)] = 1065353216;
        heap32[(r12+30)] = r_g0;
        r11 = heap32[(r12+1)];
        r14 = r11 >> 2;
        r14 = heap32[(r14)];
        r14 = r14 >> 2;
        r14 = heap32[(r14+13)];
        heap32[(g0)] = r11;
        heap32[(g0+1)] = r_g0;
        heap32[(g0+2)] = 1;
        __FUNCTION_TABLE__[(r14)>>2](i7);
        heap32[(g0)] = 748;
        _Znwj(i7);
        r8 = r_g0;
        r11 = heap32[(r12+21)];
        r14 = heap32[(r12+20)];
        r15 = r8 >> 2;
        heap32[(r15+1)] = 4;
        heap32[(r15+2)] = -1;
        heap32[(r15+3)] = -1;
        heap8[r8+16] = r13;
        heap32[(r15+5)] = r14;
        heap32[(r15+6)] = r11;
        heap32[(r15+7)] = 0;
        heap32[(r15+8)] = 1050253722;
        heap32[(r15)] = r9;
        heap32[(r15+135)] = -1287930578;
        heap32[(r15+136)] = 0;
        heap32[(r15+137)] = 1065353216;
        heap32[(r15+138)] = 0;
        heap32[(r15+139)] = -2147483648;
        heap32[(r15+140)] = 1065353216;
        heap32[(r15+141)] = 0;
        heap32[(r15+142)] = 0;
        heap32[(r15+143)] = -1082130432;
        heap32[(r15+144)] = -2147483648;
        heap32[(r15+145)] = -1287930578;
        heap32[(r15+146)] = 0;
        heap32[(r15+147)] = 0;
        heap32[(r15+148)] = 1059145646;
        heap32[(r15+149)] = 0;
        heap32[(r15+150)] = 0;
        heap32[(r15+151)] = -1287930578;
        heap32[(r15+152)] = 0;
        heap32[(r15+153)] = 1065353216;
        heap32[(r15+154)] = 0;
        heap32[(r15+155)] = -2147483648;
        heap32[(r15+156)] = 1065353216;
        heap32[(r15+157)] = 0;
        heap32[(r15+158)] = 0;
        heap32[(r15+159)] = -1082130432;
        heap32[(r15+160)] = -2147483648;
        heap32[(r15+161)] = -1287930578;
        heap32[(r15+162)] = 0;
        heap32[(r15+163)] = 0;
        heap32[(r15+164)] = -1090854584;
        heap32[(r15+165)] = 0;
        heap32[(r15+166)] = 0;
        heap8[r8+720] = r13;
        heap8[r8+721] = r13;
        heap8[r8+723] = r13;
        heap8[r8+724] = r4;
        heap8[r8+725] = r13;
        heap32[(r15+183)] = 0;
        heap32[(r15+172)] = 1065353216;
        heap32[(r15+173)] = -1082130432;
        heap32[(r15+170)] = 1050253722;
        heap32[(r15+171)] = 1065353216;
        heap32[(r15+169)] = 1063675494;
        heap8[r8+722] = r13;
        heap32[(r15+179)] = 1065353216;
        heap32[(g0)] = -1077342245;
        heap32[(g0+1)] = 1086918619;
        fmodf(i7);
        f12 = f_g0;
        if(f12 >=f13) //_LBB26_252
{
        f14 =        3.1415927410125732;
        if(f12 >f14) //_LBB26_254
{
        f14 =       -6.2831854820251465;
        f12 = f12+f14;
}
}
else{
        f14 =        6.2831854820251465;
        f12 = f12+f14;
}
        r11 = r8 >> 2;
        heapFloat[(r11+172)] = f12;
        heap32[(g0)] = 0;
        heap32[(g0+1)] = 1086918619;
        fmodf(i7);
        f12 = f_g0;
        if(f12 >=f13) //_LBB26_257
{
        f14 =        3.1415927410125732;
        if(f12 >f14) //_LBB26_259
{
        f14 =       -6.2831854820251465;
        f12 = f12+f14;
}
}
else{
        f14 =        6.2831854820251465;
        f12 = f12+f14;
}
        heapFloat[(r11+173)] = f12;
        heap32[(r11+169)] = 1063675494;
        heap32[(r11+170)] = 1050253722;
        heap32[(r11+171)] = 1065353216;
        heap32[(r12+31)] = r8;
        r11 = heap32[(r12+1)];
        r14 = r11 >> 2;
        r14 = heap32[(r14)];
        r14 = r14 >> 2;
        r14 = heap32[(r14+13)];
        heap32[(g0)] = r11;
        heap32[(g0+1)] = r8;
        heap32[(g0+2)] = 1;
        __FUNCTION_TABLE__[(r14)>>2](i7);
        heap32[(g0)] = 596;
        _Znwj(i7);
        r11 = heap32[(r12+22)];
        r14 = heap32[(r12+14)];
        r15 = r_g0 >> 2;
        heap32[(r15+1)] = 5;
        heap32[(r15+2)] = -1;
        heap32[(r15+3)] = -1;
        heap8[r_g0+16] = r13;
        heap32[(r15+5)] = r14;
        heap32[(r15+6)] = r11;
        heap32[(r15+7)] = 0;
        heap32[(r15+8)] = 1050253722;
        heap32[(r15)] = r10;
        heap32[(r15+72)] = 1065353216;
        heap32[(r15+73)] = 0;
        heap32[(r15+74)] = 0;
        heap32[(r15+75)] = 0;
        heap32[(r15+76)] = 0;
        heap32[(r15+77)] = 1065353216;
        heap32[(r15+78)] = 0;
        heap32[(r15+79)] = 0;
        heap32[(r15+80)] = -2147483648;
        heap32[(r15+81)] = 0;
        heap32[(r15+82)] = 1065353216;
        heap32[(r15+83)] = 0;
        heap32[(r15+84)] = 1060320051;
        heap32[(r15+85)] = 1057384039;
        heap32[(r15+86)] = 0;
        heap32[(r15+87)] = 0;
        heap32[(r15+88)] = -1287930578;
        heap32[(r15+89)] = -1082130432;
        heap32[(r15+90)] = 0;
        heap32[(r15+91)] = 0;
        heap32[(r15+92)] = 1065353216;
        heap32[(r15+93)] = -1287930578;
        heap32[(r15+94)] = 0;
        heap32[(r15+95)] = 0;
        heap32[(r15+96)] = -2147483648;
        heap32[(r15+97)] = 0;
        heap32[(r15+98)] = 1065353216;
        heap32[(r15+99)] = 0;
        heap32[(r15+100)] = 0;
        heap32[(r15+101)] = -1088338002;
        heap32[(r15+102)] = 0;
        heap32[(r15+103)] = 0;
        heap8[r_g0+540] = r13;
        heap32[(r15+128)] = 0;
        heap32[(r15+140)] = -1082130432;
        heap32[(r15+107)] = 1008981770;
        heap32[(r15+111)] = 1028443341;
        heap32[(r15+145)] = 0;
        heap32[(r15+146)] = 0;
        heap32[(r15+147)] = 1060320051;
        heap32[(r15+148)] = 0;
        heap32[(r15+108)] = 1070141403;
        heap32[(r15+109)] = 1070141403;
        heap32[(r15+110)] = 0;
        heap32[(r15+104)] = 1065353216;
        heap32[(r15+105)] = 1050253722;
        heap32[(r15+106)] = 1065353216;
        heap32[(r12+32)] = r_g0;
        r10 = heap32[(r12+1)];
        r11 = r10 >> 2;
        r11 = heap32[(r11)];
        r11 = r11 >> 2;
        r11 = heap32[(r11+13)];
        heap32[(g0)] = r10;
        heap32[(g0+1)] = r_g0;
        heap32[(g0+2)] = 1;
        __FUNCTION_TABLE__[(r11)>>2](i7);
        heap32[(g0)] = 748;
        _Znwj(i7);
        r8 = r_g0;
        r10 = heap32[(r12+23)];
        r11 = heap32[(r12+22)];
        r14 = r8 >> 2;
        heap32[(r14+1)] = 4;
        heap32[(r14+2)] = -1;
        heap32[(r14+3)] = -1;
        heap8[r8+16] = r13;
        heap32[(r14+5)] = r11;
        heap32[(r14+6)] = r10;
        heap32[(r14+7)] = 0;
        heap32[(r14+8)] = 1050253722;
        heap32[(r14)] = r9;
        heap32[(r14+135)] = -1287930578;
        heap32[(r14+136)] = 0;
        heap32[(r14+137)] = 1065353216;
        heap32[(r14+138)] = 0;
        heap32[(r14+139)] = -2147483648;
        heap32[(r14+140)] = 1065353216;
        heap32[(r14+141)] = 0;
        heap32[(r14+142)] = 0;
        heap32[(r14+143)] = -1082130432;
        heap32[(r14+144)] = -2147483648;
        heap32[(r14+145)] = -1287930578;
        heap32[(r14+146)] = 0;
        heap32[(r14+147)] = 0;
        heap32[(r14+148)] = 1059145646;
        heap32[(r14+149)] = 0;
        heap32[(r14+150)] = 0;
        heap32[(r14+151)] = -1287930578;
        heap32[(r14+152)] = 0;
        heap32[(r14+153)] = 1065353216;
        heap32[(r14+154)] = 0;
        heap32[(r14+155)] = -2147483648;
        heap32[(r14+156)] = 1065353216;
        heap32[(r14+157)] = 0;
        heap32[(r14+158)] = 0;
        heap32[(r14+159)] = -1082130432;
        heap32[(r14+160)] = -2147483648;
        heap32[(r14+161)] = -1287930578;
        heap32[(r14+162)] = 0;
        heap32[(r14+163)] = 0;
        heap32[(r14+164)] = -1090854584;
        heap32[(r14+165)] = 0;
        heap32[(r14+166)] = 0;
        heap8[r8+720] = r13;
        heap8[r8+721] = r13;
        heap8[r8+723] = r13;
        heap8[r8+724] = r4;
        heap8[r8+725] = r13;
        heap32[(r14+183)] = 0;
        heap32[(r14+172)] = 1065353216;
        heap32[(r14+173)] = -1082130432;
        heap32[(r14+170)] = 1050253722;
        heap32[(r14+171)] = 1065353216;
        heap32[(r14+169)] = 1063675494;
        heap8[r8+722] = r13;
        heap32[(r14+179)] = 1065353216;
        heap32[(g0)] = -1077342245;
        heap32[(g0+1)] = 1086918619;
        fmodf(i7);
        f12 = f_g0;
        if(f12 >=f13) //_LBB26_262
{
        f14 =        3.1415927410125732;
        if(f12 >f14) //_LBB26_264
{
        f14 =       -6.2831854820251465;
        f12 = f12+f14;
}
}
else{
        f14 =        6.2831854820251465;
        f12 = f12+f14;
}
        r9 = r8 >> 2;
        heapFloat[(r9+172)] = f12;
        heap32[(g0)] = 0;
        heap32[(g0+1)] = 1086918619;
        fmodf(i7);
        f12 = f_g0;
        if(f12 >=f13) //_LBB26_267
{
        f13 =        3.1415927410125732;
        if(f12 >f13) //_LBB26_269
{
        f13 =       -6.2831854820251465;
        f12 = f12+f13;
}
}
else{
        f13 =        6.2831854820251465;
        f12 = f12+f13;
}
        heapFloat[(r9+173)] = f12;
        heap32[(r9+169)] = 1063675494;
        heap32[(r9+170)] = 1050253722;
        heap32[(r9+171)] = 1065353216;
        heap32[(r12+33)] = r8;
        r9 = heap32[(r12+1)];
        r10 = r9 >> 2;
        r10 = heap32[(r10)];
        r10 = r10 >> 2;
        r10 = heap32[(r10+13)];
        heap32[(g0)] = r9;
        heap32[(g0+1)] = r8;
        heap32[(g0+2)] = 1;
        __FUNCTION_TABLE__[(r10)>>2](i7);
        r8 = heap32[(r1+10)];
        r9 = heap32[(r1+9)];
        if(r8 ==r9) //_LBB26_272
{
        r10 = r9 << 1;
        r10 = r9 == 0 ? r4 : r10;
if(!(r8 >=r10)) //_LBB26_271
{
        if(r10 !=0) //_LBB26_275
{
        r8 = heap32[(r2)];
        r11 = r10 << 2;
        r8 = (r8 + 1)|0;
        r11 = r11 | 3;
        heap32[(r2)] = r8;
        r8 = (r11 + 16)|0;
        heap32[(g0)] = r8;
        malloc(i7);
        r8 = r_g0;
        if(r8 !=0) //_LBB26_277
{
        r11 = (r8 + 4)|0;
        r11 = (r13 - r11)|0;
        r11 = r11 & 15;
        r11 = (r8 + r11)|0;
        r12 = (r11 + 4)|0;
        r11 = r11 >> 2;
        heap32[(r11)] = r8;
        r8 = r12;
}
}
else{
        r8 = 0;
}
_358: do {
        if(r9 <1) //_LBB26_280
{
        r11 = heap32[(r1+11)];
}
else{
_360: while(true){
        r11 = heap32[(r1+11)];
        r12 = r13 << 2;
        r14 = (r11 + r12)|0;
        r14 = r14 >> 2;
        r12 = (r8 + r12)|0;
        r14 = heap32[(r14)];
        r13 = (r13 + 1)|0;
        r12 = r12 >> 2;
        heap32[(r12)] = r14;
if(!(r9 !=r13)) //_LBB26_281
{
break _358;
}
}
}
} while(0);
        if(r11 !=0) //_LBB26_284
{
        r12 = heapU8[r6+48];
        if(r12 !=0) //_LBB26_286
{
        r9 = gNumAlignedFree;
        r9 = r9 >> 2;
        r12 = heap32[(r9)];
        r12 = (r12 + 1)|0;
        r11 = r11 >> 2;
        heap32[(r9)] = r12;
        r9 = heap32[(r11+-1)];
        heap32[(g0)] = r9;
        free(i7);
        r9 = heap32[(r1+9)];
}
        heap32[(r1+11)] = 0;
}
        heap8[r6+48] = r4;
        heap32[(r1+11)] = r8;
        heap32[(r1+10)] = r10;
}
}
        r8 = r9 << 2;
        r9 = heap32[(r1+11)];
        r8 = (r9 + r8)|0;
        r8 = r8 >> 2;
        heap32[(r8)] = r7;
        r7 = heap32[(r1+9)];
        r5 = (r5 + 1)|0;
        r7 = (r7 + 1)|0;
        heap32[(r1+9)] = r7;
if(!(r0 !=r5)) //_LBB26_185
{
break _262;
}
}
        f3 =                         7;
        f4 =                        -2;
        r0 = (r0 + -1)|0;
        f1 = f1+f3;
        f2 = f2+f4;
        r3 = (r3 + 1)|0;
        if(r0 ==0) //_LBB26_165
{
break _161;
}
else{
continue _260;
}
}
}
}
}
}
} while(0);
        return;
break;
case 46:
        r0 = _2E_str674;
        r1 = _2E_str573;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 53;
        _assert(i7);
break;
}
}

function _GLOBAL__D__Z6mymainiPPc(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = _ZL14benchmarkDemo4;
        r1 = _ZTV13BenchmarkDemo;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        heap32[(g0)] = r0;
        _ZN13BenchmarkDemo11exitPhysicsEv(i7);
        r1 = heap32[(r2+11)];
if(!(r1 ==0)) //_LBB27_4
{
        r3 = heapU8[r0+48];
if(!(r3 ==0)) //_LBB27_3
{
        r3 = gNumAlignedFree;
        r3 = r3 >> 2;
        r4 = heap32[(r3)];
        r4 = (r4 + 1)|0;
        r1 = r1 >> 2;
        heap32[(r3)] = r4;
        r1 = heap32[(r1+-1)];
        heap32[(g0)] = r1;
        free(i7);
}
        heap32[(r2+11)] = 0;
}
        r1 = 1;
        heap8[r0+48] = r1;
        heap32[(r2+11)] = 0;
        heap32[(r2+9)] = 0;
        heap32[(r2+10)] = 0;
        r3 = heap32[(r2+6)];
if(!(r3 ==0)) //_LBB27_8
{
        r4 = heapU8[r0+28];
if(!(r4 ==0)) //_LBB27_7
{
        r4 = gNumAlignedFree;
        r4 = r4 >> 2;
        r5 = heap32[(r4)];
        r5 = (r5 + 1)|0;
        r3 = r3 >> 2;
        heap32[(r4)] = r5;
        r3 = heap32[(r3+-1)];
        heap32[(g0)] = r3;
        free(i7);
}
        heap32[(r2+6)] = 0;
}
        heap8[r0+28] = r1;
        heap32[(r2+6)] = 0;
        heap32[(r2+4)] = 0;
        heap32[(r2+5)] = 0;
        return;
}

function _ZN14BenchmarkDemo4D1Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV14BenchmarkDemo4;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        heap32[(g0)] = r0;
        _ZN13BenchmarkDemoD2Ev(i7);
        return;
}

function _ZN14BenchmarkDemo4D0Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV14BenchmarkDemo4;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        heap32[(g0)] = r0;
        _ZN13BenchmarkDemoD2Ev(i7);
        heap32[(g0)] = r0;
        _ZdlPv(i7);
        return;
}

function __draw(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = _ZL14benchmarkDemo4;
        heap32[(g0)] = r0;
        _ZN13BenchmarkDemo20clientMoveAndDisplayEv(i7);
        return;
}

function _ZN13BenchmarkDemoD2Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV13BenchmarkDemo;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        heap32[(g0)] = r0;
        _ZN13BenchmarkDemo11exitPhysicsEv(i7);
        r1 = heap32[(r2+11)];
if(!(r1 ==0)) //_LBB31_4
{
        r3 = heapU8[r0+48];
if(!(r3 ==0)) //_LBB31_3
{
        r3 = gNumAlignedFree;
        r3 = r3 >> 2;
        r4 = heap32[(r3)];
        r4 = (r4 + 1)|0;
        r1 = r1 >> 2;
        heap32[(r3)] = r4;
        r1 = heap32[(r1+-1)];
        heap32[(g0)] = r1;
        free(i7);
}
        heap32[(r2+11)] = 0;
}
        r1 = 1;
        heap8[r0+48] = r1;
        heap32[(r2+11)] = 0;
        heap32[(r2+9)] = 0;
        heap32[(r2+10)] = 0;
        r3 = heap32[(r2+6)];
if(!(r3 ==0)) //_LBB31_8
{
        r4 = heapU8[r0+28];
if(!(r4 ==0)) //_LBB31_7
{
        r4 = gNumAlignedFree;
        r4 = r4 >> 2;
        r5 = heap32[(r4)];
        r5 = (r5 + 1)|0;
        r3 = r3 >> 2;
        heap32[(r4)] = r5;
        r3 = heap32[(r3+-1)];
        heap32[(g0)] = r3;
        free(i7);
}
        heap32[(r2+6)] = 0;
}
        heap8[r0+28] = r1;
        heap32[(r2+6)] = 0;
        heap32[(r2+4)] = 0;
        heap32[(r2+5)] = 0;
        return;
}

function __init(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = gDisableDeactivation;
        r1 = 1;
        heap8[r0] = r1;
        _ZN13BenchmarkDemo11initPhysicsEv(i7);
        return;
}

function _ZN15btNullPairCache23getOverlappingPairArrayEv(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = (r0 + 4)|0;
        r_g0 = r0;
        return;
}

function _ZN15btNullPairCache20cleanOverlappingPairER16btBroadphasePairP12btDispatcher(sp)
{
        var i7;
        var fp = sp>>2;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        return;
}

function _ZNK15btNullPairCache22getNumOverlappingPairsEv(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = 0;
        r_g0 = r0;
        return;
}

function _ZN15btNullPairCache19cleanProxyFromPairsEP17btBroadphaseProxyP12btDispatcher(sp)
{
        var i7;
        var fp = sp>>2;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        return;
}

function _ZN15btNullPairCache24setOverlapFilterCallbackEP23btOverlapFilterCallback(sp)
{
        var i7;
        var fp = sp>>2;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        return;
}

function _ZN15btNullPairCache26processAllOverlappingPairsEP17btOverlapCallbackP12btDispatcher(sp)
{
        var i7;
        var fp = sp>>2;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        return;
}

function _ZN15btNullPairCache8findPairEP17btBroadphaseProxyS1_(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = 0;
        r_g0 = r0;
        return;
}

function _ZN15btNullPairCache18hasDeferredRemovalEv(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = 1;
        r_g0 = r0;
        return;
}

function _ZN15btNullPairCache28setInternalGhostPairCallbackEP25btOverlappingPairCallback(sp)
{
        var i7;
        var fp = sp>>2;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        return;
}

function _ZN15btNullPairCache18addOverlappingPairEP17btBroadphaseProxyS1_(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = 0;
        r_g0 = r0;
        return;
}

function _ZN15btNullPairCache21removeOverlappingPairEP17btBroadphaseProxyS1_P12btDispatcher(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = 0;
        r_g0 = r0;
        return;
}

function _ZN15btNullPairCache37removeOverlappingPairsContainingProxyEP17btBroadphaseProxyP12btDispatcher(sp)
{
        var i7;
        var fp = sp>>2;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        return;
}

function _ZN15btNullPairCache20sortOverlappingPairsEP12btDispatcher(sp)
{
        var i7;
        var fp = sp>>2;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        return;
}

function _ZN15btNullPairCache26getOverlappingPairArrayPtrEv(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = r0 >> 2;
        r0 = heap32[(r0+4)];
        r_g0 = r0;
        return;
}

function _ZNK15btNullPairCache26getOverlappingPairArrayPtrEv(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = r0 >> 2;
        r0 = heap32[(r0+4)];
        r_g0 = r0;
        return;
}

function _ZN15btNullPairCacheD1Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV15btNullPairCache;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        r1 = heap32[(r2+4)];
if(!(r1 ==0)) //_LBB48_4
{
        r3 = heapU8[r0+20];
if(!(r3 ==0)) //_LBB48_3
{
        r3 = gNumAlignedFree;
        r3 = r3 >> 2;
        r4 = heap32[(r3)];
        r4 = (r4 + 1)|0;
        r1 = r1 >> 2;
        heap32[(r3)] = r4;
        r1 = heap32[(r1+-1)];
        heap32[(g0)] = r1;
        free(i7);
}
        heap32[(r2+4)] = 0;
}
        r1 = 1;
        heap8[r0+20] = r1;
        heap32[(r2+4)] = 0;
        heap32[(r2+2)] = 0;
        heap32[(r2+3)] = 0;
        return;
}

function _ZN15btNullPairCacheD0Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV15btNullPairCache;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        r1 = heap32[(r2+4)];
if(!(r1 ==0)) //_LBB49_4
{
        r3 = heapU8[r0+20];
if(!(r3 ==0)) //_LBB49_3
{
        r3 = gNumAlignedFree;
        r3 = r3 >> 2;
        r4 = heap32[(r3)];
        r4 = (r4 + 1)|0;
        r1 = r1 >> 2;
        heap32[(r3)] = r4;
        r1 = heap32[(r1+-1)];
        heap32[(g0)] = r1;
        free(i7);
}
        heap32[(r2+4)] = 0;
}
        r1 = 1;
        heap8[r0+20] = r1;
        heap32[(r2+4)] = 0;
        heap32[(r2+2)] = 0;
        heap32[(r2+3)] = 0;
        heap32[(g0)] = r0;
        _ZdlPv(i7);
        return;
}

function _ZNK20btAxisSweep3InternalItE7getAabbEP17btBroadphaseProxyR9btVector3S4_(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp+1)];
        r1 = heap32[(fp+2)];
        r0 = r0 >> 2;
        r1 = r1 >> 2;
        heap32[(r1)] = heap32[(r0+4)];
        heap32[(r1+1)] = heap32[(r0+5)];
        r2 = heap32[(fp+3)];
        heap32[(r1+2)] = heap32[(r0+6)];
        r2 = r2 >> 2;
        heap32[(r1+3)] = heap32[(r0+7)];
        heap32[(r2)] = heap32[(r0+8)];
        heap32[(r2+1)] = heap32[(r0+9)];
        heap32[(r2+2)] = heap32[(r0+10)];
        heap32[(r2+3)] = heap32[(r0+11)];
        return;
}

function _ZN20btAxisSweep3InternalItE23getOverlappingPairCacheEv(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = r0 >> 2;
        r0 = heap32[(r0+23)];
        r_g0 = r0;
        return;
}

function _ZNK20btAxisSweep3InternalItE23getOverlappingPairCacheEv(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = r0 >> 2;
        r0 = heap32[(r0+23)];
        r_g0 = r0;
        return;
}

function _ZNK20btAxisSweep3InternalItE17getBroadphaseAabbER9btVector3S2_(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = heap32[(fp+1)];
        r0 = r0 >> 2;
        r1 = r1 >> 2;
        heap32[(r1)] = heap32[(r0+2)];
        heap32[(r1+1)] = heap32[(r0+3)];
        r2 = heap32[(fp+2)];
        heap32[(r1+2)] = heap32[(r0+4)];
        r2 = r2 >> 2;
        heap32[(r1+3)] = heap32[(r0+5)];
        heap32[(r2)] = heap32[(r0+6)];
        heap32[(r2+1)] = heap32[(r0+7)];
        heap32[(r2+2)] = heap32[(r0+8)];
        heap32[(r2+3)] = heap32[(r0+9)];
        return;
}

function _ZN20btAxisSweep3InternalItE9resetPoolEP12btDispatcher(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = heapU16[(r0+56)>>1];
if(!(r1 !=0)) //_LBB54_6
{
        r1 = 1;
        heap16[(r0+64)>>1] = r1;
        r2 = heapU16[(r0+58)>>1];
        if(uint(r2) >uint(1)) //_LBB54_3
{
        r3 = 2;
_5: while(true){
        r2 = r1 & 65535;
        r4 = r0 >> 2;
        r2 = r2 << 6;
        r4 = heap32[(r4+15)];
        r2 = (r4 + r2)|0;
        heap16[(r2+48)>>1] = r3;
        r1 = (r1 + 1)|0;
        r2 = heapU16[(r0+58)>>1];
        r3 = (r3 + 1)|0;
        r4 = r1 & 65535;
if(!(uint(r2) >uint(r4))) //_LBB54_4
{
break _5;
}
}
}
        r1 = r2 & 65535;
        r0 = r0 >> 2;
        r1 = r1 << 6;
        r0 = heap32[(r0+15)];
        r0 = (r1 + r0)|0;
        r1 = 0;
        heap16[(r0+-16)>>1] = r1;
}
        return;
}

function _ZN20btAxisSweep3InternalItE10printStatsEv(sp)
{
        var i7;
        var fp = sp>>2;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        return;
}

function _ZN20btAxisSweep3InternalItE8aabbTestERK9btVector3S3_R24btBroadphaseAabbCallback(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var f0;
        var f1;
var __label__ = 0;
        i7 = sp + -16;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = r0 >> 2;
        r2 = heap32[(r1+27)];
        r3 = heap32[(fp+1)];
        r4 = heap32[(fp+2)];
        r5 = heap32[(fp+3)];
        if(r2 !=0) //_LBB56_2
{
        r0 = r2 >> 2;
        r0 = heap32[(r0)];
        r0 = r0 >> 2;
        r0 = heap32[(r0+7)];
        heap32[(g0)] = r2;
        heap32[(g0+1)] = r3;
        heap32[(g0+2)] = r4;
        heap32[(g0+3)] = r5;
        __FUNCTION_TABLE__[(r0)>>2](i7);
        return;
}
else{
        r2 = heapU16[(r0+56)>>1];
        r2 = r2 << 1;
        r2 = r2 | 1;
if(!(r2 ==1)) //_LBB56_18
{
        r2 = 1;
        r6 = 2;
_6: while(true){
        r7 = r6;
        r6 = r2 & 65535;
        r8 = heap32[(r1+17)];
        r6 = r6 << 2;
        r9 = heapU8[r8+r6];
        r9 = r9 & 1;
if(!(r9 ==0)) //_LBB56_17
{
        r6 = (r8 + r6)|0;
        r6 = heapU16[(r6+2)>>1];
        r8 = heap32[(r1+15)];
        r6 = r6 << 6;
        r6 = (r8 + r6)|0;
        r8 = r3 >> 2;
        r9 = r6 >> 2;
        f0 = heapFloat[(r8)];
        f1 = heapFloat[(r9+8)];
        if(f0 >f1) //_LBB56_8
{
__label__ = 8;
}
else{
        r10 = r4 >> 2;
        f0 = heapFloat[(r10)];
        f1 = heapFloat[(r9+4)];
        if(f0 <f1) //_LBB56_8
{
__label__ = 8;
}
else{
        r10 = 1;
__label__ = 9;
}
}
if (__label__ == 8){
        r10 = 0;
}
        f0 = heapFloat[(r8+2)];
        f1 = heapFloat[(r9+10)];
        if(f0 >f1) //_LBB56_12
{
__label__ = 11;
}
else{
        r11 = r4 >> 2;
        f0 = heapFloat[(r11+2)];
        f1 = heapFloat[(r9+6)];
        if(f0 <f1) //_LBB56_12
{
__label__ = 11;
}
else{
__label__ = 12;
}
}
if (__label__ == 11){
        r10 = 0;
}
        f0 = heapFloat[(r8+1)];
        f1 = heapFloat[(r9+9)];
if(!(f0 >f1)) //_LBB56_17
{
        r8 = r4 >> 2;
        f0 = heapFloat[(r8+1)];
        f1 = heapFloat[(r9+5)];
if(!(f0 <f1)) //_LBB56_17
{
        r8 = r10 & 255;
if(!(r8 ==0)) //_LBB56_17
{
        r8 = r5 >> 2;
        r8 = heap32[(r8)];
        r8 = r8 >> 2;
        r8 = heap32[(r8+2)];
        heap32[(g0)] = r5;
        heap32[(g0+1)] = r6;
        __FUNCTION_TABLE__[(r8)>>2](i7);
}
}
}
}
        r8 = heapU16[(r0+56)>>1];
        r6 = (r7 + 1)|0;
        r2 = (r2 + 1)|0;
        r8 = r8 << 1;
        r7 = r7 & 65535;
        r8 = r8 | 1;
if(!(uint(r7) <uint(r8))) //_LBB56_4
{
break _6;
}
}
}
        return;
}
}

function _ZN20btAxisSweep3InternalItE7rayTestERK9btVector3S3_R23btBroadphaseRayCallbackS3_S3_(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
var __label__ = 0;
        i7 = sp + -24;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = r0 >> 2;
        r2 = heap32[(r1+27)];
        r3 = heap32[(fp+3)];
        if(r2 !=0) //_LBB57_2
{
        r0 = heap32[(fp+1)];
        r1 = heap32[(fp+2)];
        r4 = heap32[(fp+4)];
        r5 = heap32[(fp+5)];
        r6 = r2 >> 2;
        r6 = heap32[(r6)];
        r6 = r6 >> 2;
        r6 = heap32[(r6+6)];
        heap32[(g0)] = r2;
        heap32[(g0+1)] = r0;
        heap32[(g0+2)] = r1;
        heap32[(g0+3)] = r3;
        heap32[(g0+4)] = r4;
        heap32[(g0+5)] = r5;
        __FUNCTION_TABLE__[(r6)>>2](i7);
        return;
}
else{
        r2 = heapU16[(r0+56)>>1];
        r2 = r2 << 1;
        r2 = r2 | 1;
if(!(r2 ==1)) //_LBB57_7
{
        r2 = 1;
        r4 = 2;
_6: while(true){
        r5 = r2 & 65535;
        r6 = heap32[(r1+17)];
        r5 = r5 << 2;
        r7 = heapU8[r6+r5];
        r7 = r7 & 1;
if(!(r7 ==0)) //_LBB57_6
{
        r7 = r3 >> 2;
        r7 = heap32[(r7)];
        r7 = r7 >> 2;
        r5 = (r6 + r5)|0;
        r5 = heapU16[(r5+2)>>1];
        r6 = heap32[(r7+2)];
        r7 = heap32[(r1+15)];
        r5 = r5 << 6;
        r5 = (r7 + r5)|0;
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r5;
        __FUNCTION_TABLE__[(r6)>>2](i7);
}
        r5 = heapU16[(r0+56)>>1];
        r6 = (r4 + 1)|0;
        r2 = (r2 + 1)|0;
        r5 = r5 << 1;
        r7 = r4 & 65535;
        r5 = r5 | 1;
        r4 = r6;
if(!(uint(r7) <uint(r5))) //_LBB57_4
{
break _6;
}
}
}
        return;
}
}

function _ZNK20btAxisSweep3InternalItE8quantizeEPtRK9btVector3i(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var f0;
        var f1;
        var f2;
        var f3;
        var f4;
        var f5;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = r0 >> 2;
        f0 = heapFloat[(fp+2)];
        f1 = heapFloat[(r1+2)];
        f2 = heapFloat[(fp+4)];
        f3 = heapFloat[(r1+4)];
        f4 = heapFloat[(fp+3)];
        f5 = heapFloat[(r1+3)];
        f0 = f0-f1;
        f1 = heapFloat[(r1+10)];
        f2 = f2-f3;
        f3 = heapFloat[(r1+12)];
        f4 = f4-f5;
        f5 = heapFloat[(r1+11)];
        f0 = f0*f1;
        r1 = heap32[(fp+1)];
        r2 = heap32[(fp+5)];
        f1 = f2*f3;
        f2 = f4*f5;
        f3 =                         0;
        if(f0 >f3) //_LBB58_2
{
        r3 = heapU16[(r0+6)>>1];
        f4 = r3; //fitos r3, f4
        if(f0 <f4) //_LBB58_4
{
        r3 = heapU16[(r0+4)>>1];
        r4 = Math.floor(f0);
        r3 = r3 & r4;
        r3 = r3 | r2;
}
else{
        r4 = heapU16[(r0+4)>>1];
        r3 = r4 & r3;
        r3 = r3 | r2;
}
}
else{
        r3 = r2;
}
        heap16[(r1)>>1] = r3;
        if(f2 >f3) //_LBB58_7
{
        r3 = heapU16[(r0+6)>>1];
        f0 = r3; //fitos r3, f0
        if(f2 <f0) //_LBB58_9
{
        r3 = heapU16[(r0+4)>>1];
        r4 = Math.floor(f2);
        r3 = r3 & r4;
        r3 = r3 | r2;
}
else{
        r4 = heapU16[(r0+4)>>1];
        r3 = r4 & r3;
        r3 = r3 | r2;
}
}
else{
        r3 = r2;
}
        heap16[(r1+2)>>1] = r3;
        if(f1 >f3) //_LBB58_12
{
        r3 = heapU16[(r0+6)>>1];
        f0 = r3; //fitos r3, f0
        if(f1 <f0) //_LBB58_14
{
        r0 = heapU16[(r0+4)>>1];
        r3 = Math.floor(f1);
        r0 = r0 & r3;
        r2 = r0 | r2;
}
else{
        r0 = heapU16[(r0+4)>>1];
        r0 = r0 & r3;
        r2 = r0 | r2;
}
}
        heap16[(r1+4)>>1] = r2;
        return;
}

function _ZN20btAlignedObjectArrayI16btBroadphasePairE17quickSortInternalI29btBroadphasePairSortPredicateEEvT_ii(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var r12;
        var r13;
        var r14;
        var r15;
        var r16;
        var r17;
        var r18;
        var r19;
        var r20;
        var r21;
        var r22;
        var r23;
        var r24;
        var r25;
var __label__ = 0;
        i7 = sp + -16;var g0 = i7>>2; // save stack
        r0 = heap32[(fp+2)];
        r1 = heap32[(fp+1)];
        r2 = (r0 + r1)|0;
        r3 = r2 >>> 31;
        r4 = heap32[(fp)];
        r2 = (r2 + r3)|0;
        r3 = r4 >> 2;
        r2 = r2 & 536870910;
        r5 = heap32[(r3+3)];
        r2 = r2 << 3;
        r2 = (r5 + r2)|0;
        r2 = r2 >> 2;
        r5 = heap32[(r2)];
        r6 = heap32[(r2+1)];
        r2 = heap32[(r2+2)];
        r7 = r1;
        r8 = r0;
_1: while(true){
        r9 = heap32[(r3+3)];
        r10 = r7 << 4;
        r10 = (r9 + r10)|0;
        r11 = 0;
_3: while(true){
        r12 = r11 << 4;
        r12 = (r10 + r12)|0;
        r12 = r12 >> 2;
        r13 = r11 << 2;
        r14 = heap32[(r12)];
        if(r14 !=0) //_LBB59_5
{
        r15 = r14 >> 2;
        r15 = heap32[(r15+3)];
}
else{
        r15 = -1;
}
        if(r5 !=0) //_LBB59_8
{
        r16 = r5 >> 2;
        r16 = heap32[(r16+3)];
}
else{
        r16 = -1;
}
        r17 = r13 << 2;
        r17 = (r10 + r17)|0;
        r17 = r17 >> 2;
        r17 = heap32[(r17+1)];
        if(r17 !=0) //_LBB59_11
{
        r18 = r17 >> 2;
        r18 = heap32[(r18+3)];
}
else{
        r18 = -1;
}
        if(r6 !=0) //_LBB59_14
{
        r19 = r6 >> 2;
        r19 = heap32[(r19+3)];
}
else{
        r19 = -1;
}
_21: do {
if(!(r15 >r16)) //_LBB59_2
{
if(!(r14 !=r5)) //_LBB59_18
{
        if(r18 >r19) //_LBB59_2
{
break _21;
}
}
        if(r14 !=r5) //_LBB59_22
{
break _3;
}
else{
        if(r17 !=r6) //_LBB59_22
{
break _3;
}
else{
        r15 = r13 << 2;
        r15 = (r10 + r15)|0;
        r15 = r15 >> 2;
        r15 = heap32[(r15+2)];
if(!(uint(r15) >uint(r2))) //_LBB59_2
{
break _3;
}
}
}
}
} while(0);
        r11 = (r11 + 1)|0;
continue _3;
}
        r18 = r8 << 4;
        r15 = (r7 + r11)|0;
        r9 = (r9 + r18)|0;
        r16 = 0;
_30: while(true){
        r19 = r16 << 2;
        if(r5 !=0) //_LBB59_25
{
        r20 = r5 >> 2;
        r20 = heap32[(r20+3)];
}
else{
        r20 = -1;
}
        r21 = r19 << 2;
        r21 = (r9 + r21)|0;
        r21 = r21 >> 2;
        r21 = heap32[(r21)];
        if(r21 !=0) //_LBB59_28
{
        r22 = r21 >> 2;
        r22 = heap32[(r22+3)];
}
else{
        r22 = -1;
}
        if(r6 !=0) //_LBB59_31
{
        r23 = r6 >> 2;
        r23 = heap32[(r23+3)];
}
else{
        r23 = -1;
}
        r24 = r19 << 2;
        r24 = (r9 + r24)|0;
        r24 = r24 >> 2;
        r24 = heap32[(r24+1)];
        if(r24 !=0) //_LBB59_34
{
        r25 = r24 >> 2;
        r25 = heap32[(r25+3)];
}
else{
        r25 = -1;
}
_48: do {
if(!(r20 >r22)) //_LBB59_21
{
if(!(r5 !=r21)) //_LBB59_38
{
        if(r23 >r25) //_LBB59_21
{
break _48;
}
}
        if(r5 !=r21) //_LBB59_41
{
break _30;
}
else{
        if(r6 !=r24) //_LBB59_41
{
break _30;
}
else{
        r20 = r19 << 2;
        r20 = (r9 + r20)|0;
        r20 = r20 >> 2;
        r20 = heap32[(r20+2)];
if(!(uint(r2) >uint(r20))) //_LBB59_21
{
break _30;
}
}
}
}
} while(0);
        r16 = (r16 + -1)|0;
continue _30;
}
        r20 = (r8 + r16)|0;
        if(r15 <=r20) //_LBB59_43
{
        r13 = r13 << 2;
        r10 = (r10 + r13)|0;
        r10 = r10 >> 2;
        r13 = heap32[(r12+3)];
        r15 = heap32[(r10+2)];
        r19 = r19 << 2;
        r19 = (r9 + r19)|0;
        heap32[(r10)] = r21;
        r20 = r16 << 4;
        r19 = r19 >> 2;
        heap32[(r10+1)] = r24;
        r19 = heap32[(r19+2)];
        r9 = (r9 + r20)|0;
        r9 = r9 >> 2;
        heap32[(r10+2)] = r19;
        r9 = heap32[(r9+3)];
        heap32[(r12+3)] = r9;
        r9 = heap32[(r3+3)];
        r9 = (r9 + r18)|0;
        r9 = (r9 + r20)|0;
        r9 = r9 >> 2;
        heap32[(r9)] = r14;
        r7 = (r7 + r11)|0;
        r8 = (r8 + r16)|0;
        heap32[(r9+1)] = r17;
        r7 = (r7 + 1)|0;
        r8 = (r8 + -1)|0;
        heap32[(r9+2)] = r15;
        heap32[(r9+3)] = r13;
}
else{
        r7 = r15;
        r8 = r20;
}
        if(r7 <=r8) //_LBB59_1
{
continue _1;
}
else{
break _1;
}
}
if(!(r8 <=r1)) //_LBB59_47
{
        heap32[(g0)] = r4;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = r8;
        _ZN20btAlignedObjectArrayI16btBroadphasePairE17quickSortInternalI29btBroadphasePairSortPredicateEEvT_ii(i7);
}
if(!(r7 >=r0)) //_LBB59_49
{
        heap32[(g0)] = r4;
        heap32[(g0+1)] = r7;
        heap32[(g0+2)] = r0;
        _ZN20btAlignedObjectArrayI16btBroadphasePairE17quickSortInternalI29btBroadphasePairSortPredicateEEvT_ii(i7);
}
        return;
}

function _ZN20btAxisSweep3InternalItE11sortMinDownEitP12btDispatcherb(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var r12;
        var r13;
        var r14;
        var r15;
        var r16;
        var r17;
var __label__ = 0;
        i7 = sp + -16;var g0 = i7>>2; // save stack
        r0 = heap32[(fp+1)];
        r1 = heap32[(fp)];
        r2 = r0 << 2;
        r2 = (r1 + r2)|0;
        r3 = heap32[(fp+2)];
        r2 = r2 >> 2;
        r2 = heap32[(r2+17)];
        r3 = r3 << 2;
        r4 = (r2 + r3)|0;
        r2 = heapU16[(r2+r3)>>1];
        r3 = heapU16[(r4+-4)>>1];
if(!(uint(r2) >=uint(r3))) //_LBB60_14
{
        r1 = r1 >> 2;
        r2 = heap32[(fp+3)];
        r3 = heap32[(r1+15)];
        r5 = heapU16[(r4+2)>>1];
        r6 = 1;
        r7 = r6 << r0;
        r7 = r7 & 3;
        r6 = r6 << r7;
        r5 = r5 << 6;
        r5 = (r3 + r5)|0;
        r6 = r6 & 3;
        r0 = r0 << 1;
        r8 = (r5 + 48)|0;
        r7 = r7 << 1;
        r9 = (r5 + 54)|0;
        r6 = r6 << 1;
        r10 = (r8 + r0)|0;
        r11 = (r9 + r7)|0;
        r12 = (r8 + r7)|0;
        r9 = (r9 + r6)|0;
        r8 = (r8 + r6)|0;
        r4 = (r4 + 2)|0;
_3: while(true){
        r13 = heapU8[r4+-6];
        r14 = heapU16[(r4+-4)>>1];
        r13 = r13 & 1;
        if(r13 ==0) //_LBB60_11
{
        r13 = r14 << 6;
        r3 = (r3 + r13)|0;
        r3 = (r3 + r0)|0;
        r13 = heapU16[(r3+48)>>1];
        r13 = (r13 + 1)|0;
        heap16[(r3+48)>>1] = r13;
}
else{
if(!(r2 ==0)) //_LBB60_10
{
        r13 = r14 << 6;
        r13 = (r3 + r13)|0;
        r15 = (r13 + r7)|0;
        r16 = heapU16[(r11)>>1];
        r17 = heapU16[(r15+48)>>1];
if(!(uint(r16) <uint(r17))) //_LBB60_10
{
        r15 = heapU16[(r15+54)>>1];
        r16 = heapU16[(r12)>>1];
if(!(uint(r15) <uint(r16))) //_LBB60_10
{
        r15 = (r13 + r6)|0;
        r16 = heapU16[(r9)>>1];
        r17 = heapU16[(r15+48)>>1];
if(!(uint(r16) <uint(r17))) //_LBB60_10
{
        r15 = heapU16[(r15+54)>>1];
        r16 = heapU16[(r8)>>1];
if(!(uint(r15) <uint(r16))) //_LBB60_10
{
        r15 = heap32[(r1+23)];
        r16 = r15 >> 2;
        r16 = heap32[(r16)];
        r16 = r16 >> 2;
        r16 = heap32[(r16+2)];
        heap32[(g0)] = r15;
        heap32[(g0+1)] = r5;
        heap32[(g0+2)] = r13;
        __FUNCTION_TABLE__[(r16)>>2](i7);
        r15 = heap32[(r1+24)];
if(!(r15 ==0)) //_LBB60_10
{
        r16 = r15 >> 2;
        r16 = heap32[(r16)];
        r16 = r16 >> 2;
        r16 = heap32[(r16+2)];
        heap32[(g0)] = r15;
        heap32[(g0+1)] = r5;
        heap32[(g0+2)] = r13;
        __FUNCTION_TABLE__[(r16)>>2](i7);
}
}
}
}
}
}
        r14 = r14 << 6;
        r3 = (r3 + r14)|0;
        r3 = (r3 + r0)|0;
        r14 = heapU16[(r3+54)>>1];
        r14 = (r14 + 1)|0;
        heap16[(r3+54)>>1] = r14;
}
        r3 = heapU16[(r10)>>1];
        r3 = (r3 + -1)|0;
        heap16[(r10)>>1] = r3;
        r3 = heapU16[(r4+-2)>>1];
        r13 = heapU16[(r4)>>1];
        r14 = heapU16[(r4+-4)>>1];
        r15 = heapU16[(r4+-6)>>1];
        heap16[(r4+-2)>>1] = r15;
        heap16[(r4)>>1] = r14;
        heap16[(r4+-6)>>1] = r3;
        heap16[(r4+-4)>>1] = r13;
        r13 = heapU16[(r4+-10)>>1];
        if(uint(r3) >=uint(r13)) //_LBB60_14
{
break _3;
}
else{
        r3 = heap32[(r1+15)];
        r4 = (r4 + -4)|0;
continue _3;
}
}
}
        return;
}

function _ZN20btAxisSweep3InternalItE9sortMaxUpEitP12btDispatcherb(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var r12;
        var r13;
        var r14;
        var r15;
        var r16;
var __label__ = 0;
        i7 = sp + -16;var g0 = i7>>2; // save stack
        r0 = heap32[(fp+1)];
        r1 = heap32[(fp)];
        r2 = r0 << 2;
        r2 = (r1 + r2)|0;
        r2 = r2 >> 2;
        r3 = heap32[(fp+2)];
        r4 = 1;
        r2 = heap32[(r2+17)];
        r3 = r3 << 2;
        r2 = (r2 + r3)|0;
        r3 = r4 << r0;
        r3 = r3 & 3;
        r1 = r1 >> 2;
        r5 = heapU16[(r2+2)>>1];
        r4 = r4 << r3;
        r6 = heap32[(r1+15)];
        r5 = r5 << 6;
        r4 = r4 & 3;
        r5 = (r6 + r5)|0;
        r3 = r3 << 1;
        r6 = (r5 + 48)|0;
        r4 = r4 << 1;
        r5 = (r5 + 54)|0;
        r0 = r0 << 1;
        r7 = heap32[(fp+3)];
        r8 = (r5 + r3)|0;
        r9 = (r6 + r3)|0;
        r10 = (r5 + r4)|0;
        r6 = (r6 + r4)|0;
        r5 = (r5 + r0)|0;
        r2 = (r2 + 4)|0;
_1: while(true){
        r11 = heapU16[(r2+2)>>1];
        if(r11 ==0) //_LBB61_14
{
break _1;
}
else{
        r12 = heapU16[(r2)>>1];
        r13 = heapU16[(r2+-4)>>1];
        if(uint(r13) >=uint(r12)) //_LBB61_1
{
        r13 = heap32[(r1+15)];
        r11 = r11 & 65535;
        r12 = r12 & 1;
        if(r12 != 0) //_LBB61_10
{
        r11 = r11 << 6;
        r11 = (r13 + r11)|0;
        r11 = (r11 + r0)|0;
        r12 = heapU16[(r11+54)>>1];
        r12 = (r12 + -1)|0;
        heap16[(r11+54)>>1] = r12;
}
else{
if(!(r7 ==0)) //_LBB61_9
{
        r12 = r11 << 6;
        r12 = (r13 + r12)|0;
        r14 = (r12 + r3)|0;
        r15 = heapU16[(r8)>>1];
        r16 = heapU16[(r14+48)>>1];
if(!(uint(r15) <uint(r16))) //_LBB61_9
{
        r14 = heapU16[(r14+54)>>1];
        r15 = heapU16[(r9)>>1];
if(!(uint(r14) <uint(r15))) //_LBB61_9
{
        r14 = (r12 + r4)|0;
        r15 = heapU16[(r10)>>1];
        r16 = heapU16[(r14+48)>>1];
if(!(uint(r15) <uint(r16))) //_LBB61_9
{
        r14 = heapU16[(r14+54)>>1];
        r15 = heapU16[(r6)>>1];
if(!(uint(r14) <uint(r15))) //_LBB61_9
{
        r14 = heap32[(r1+23)];
        r15 = r14 >> 2;
        r15 = heap32[(r15)];
        r15 = r15 >> 2;
        r16 = heapU16[(r2+-2)>>1];
        r15 = heap32[(r15+2)];
        r16 = r16 << 6;
        r16 = (r13 + r16)|0;
        heap32[(g0)] = r14;
        heap32[(g0+1)] = r16;
        heap32[(g0+2)] = r12;
        __FUNCTION_TABLE__[(r15)>>2](i7);
        r14 = heap32[(r1+24)];
if(!(r14 ==0)) //_LBB61_9
{
        r15 = r14 >> 2;
        r15 = heap32[(r15)];
        r15 = r15 >> 2;
        r15 = heap32[(r15+2)];
        heap32[(g0)] = r14;
        heap32[(g0+1)] = r16;
        heap32[(g0+2)] = r12;
        __FUNCTION_TABLE__[(r15)>>2](i7);
}
}
}
}
}
}
        r11 = r11 << 6;
        r11 = (r13 + r11)|0;
        r11 = (r11 + r0)|0;
        r12 = heapU16[(r11+48)>>1];
        r12 = (r12 + -1)|0;
        heap16[(r11+48)>>1] = r12;
}
        r11 = heapU16[(r5)>>1];
        r11 = (r11 + 1)|0;
        heap16[(r5)>>1] = r11;
        r11 = heapU16[(r2+-2)>>1];
        r12 = heapU16[(r2+-4)>>1];
        r13 = heapU16[(r2+2)>>1];
        r14 = heapU16[(r2)>>1];
        heap16[(r2+-4)>>1] = r14;
        heap16[(r2+-2)>>1] = r13;
        r13 = (r2 + 4)|0;
        heap16[(r2)>>1] = r12;
        heap16[(r2+2)>>1] = r11;
        r2 = r13;
continue _1;
}
else{
break _1;
}
}
}
        return;
}

function _ZN20btAxisSweep3InternalItE9sortMinUpEitP12btDispatcherb(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var r12;
        var r13;
        var r14;
        var r15;
var __label__ = 0;
        i7 = sp + -16;var g0 = i7>>2; // save stack
        r0 = heap32[(fp+1)];
        r1 = heap32[(fp)];
        r2 = r0 << 2;
        r2 = (r1 + r2)|0;
        r2 = r2 >> 2;
        r3 = heap32[(fp+2)];
        r2 = heap32[(r2+17)];
        r3 = r3 << 2;
        r2 = (r2 + r3)|0;
        r1 = r1 >> 2;
        r3 = heapU16[(r2+2)>>1];
        r4 = 1;
        r5 = r4 << r0;
        r6 = heap32[(r1+15)];
        r3 = r3 << 6;
        r5 = r5 & 3;
        r3 = (r6 + r3)|0;
        r0 = r0 << 1;
        r3 = (r3 + r0)|0;
        r4 = r4 << r5;
        r6 = heap32[(fp+3)];
        r7 = heap32[(fp+4)];
        r3 = (r3 + 48)|0;
        r4 = r4 & 3;
        r2 = (r2 + 4)|0;
_1: while(true){
        r8 = heapU16[(r2+2)>>1];
        if(r8 ==0) //_LBB62_14
{
break _1;
}
else{
        r9 = heapU16[(r2)>>1];
        r10 = heapU16[(r2+-4)>>1];
        if(uint(r10) >=uint(r9)) //_LBB62_1
{
        r10 = heap32[(r1+15)];
        r8 = r8 & 65535;
        r9 = r9 & 1;
        if(r9 ==0) //_LBB62_10
{
        r8 = r8 << 6;
        r8 = (r10 + r8)|0;
        r8 = (r8 + r0)|0;
        r9 = heapU16[(r8+48)>>1];
        r9 = (r9 + -1)|0;
        heap16[(r8+48)>>1] = r9;
}
else{
if(!(r7 ==0)) //_LBB62_9
{
        r9 = heapU16[(r2+-2)>>1];
        r9 = r9 << 6;
        r11 = r8 << 6;
        r9 = (r10 + r9)|0;
        r12 = r5 << 1;
        r11 = (r10 + r11)|0;
        r13 = (r9 + r12)|0;
        r12 = (r11 + r12)|0;
        r14 = heapU16[(r13+54)>>1];
        r15 = heapU16[(r12+48)>>1];
if(!(uint(r14) <uint(r15))) //_LBB62_9
{
        r12 = heapU16[(r12+54)>>1];
        r13 = heapU16[(r13+48)>>1];
if(!(uint(r12) <uint(r13))) //_LBB62_9
{
        r12 = r4 << 1;
        r13 = (r9 + r12)|0;
        r12 = (r11 + r12)|0;
        r14 = heapU16[(r13+54)>>1];
        r15 = heapU16[(r12+48)>>1];
if(!(uint(r14) <uint(r15))) //_LBB62_9
{
        r12 = heapU16[(r12+54)>>1];
        r13 = heapU16[(r13+48)>>1];
if(!(uint(r12) <uint(r13))) //_LBB62_9
{
        r12 = heap32[(r1+23)];
        r13 = r12 >> 2;
        r13 = heap32[(r13)];
        r13 = r13 >> 2;
        r13 = heap32[(r13+3)];
        heap32[(g0)] = r12;
        heap32[(g0+1)] = r9;
        heap32[(g0+2)] = r11;
        heap32[(g0+3)] = r6;
        __FUNCTION_TABLE__[(r13)>>2](i7);
        r12 = heap32[(r1+24)];
if(!(r12 ==0)) //_LBB62_9
{
        r13 = r12 >> 2;
        r13 = heap32[(r13)];
        r13 = r13 >> 2;
        r13 = heap32[(r13+3)];
        heap32[(g0)] = r12;
        heap32[(g0+1)] = r9;
        heap32[(g0+2)] = r11;
        heap32[(g0+3)] = r6;
        __FUNCTION_TABLE__[(r13)>>2](i7);
}
}
}
}
}
}
        r8 = r8 << 6;
        r8 = (r10 + r8)|0;
        r8 = (r8 + r0)|0;
        r9 = heapU16[(r8+54)>>1];
        r9 = (r9 + -1)|0;
        heap16[(r8+54)>>1] = r9;
}
        r8 = heapU16[(r3)>>1];
        r8 = (r8 + 1)|0;
        heap16[(r3)>>1] = r8;
        r8 = heapU16[(r2+-2)>>1];
        r9 = heapU16[(r2+-4)>>1];
        r10 = heapU16[(r2+2)>>1];
        r11 = heapU16[(r2)>>1];
        heap16[(r2+-4)>>1] = r11;
        heap16[(r2+-2)>>1] = r10;
        r10 = (r2 + 4)|0;
        heap16[(r2)>>1] = r9;
        heap16[(r2+2)>>1] = r8;
        r2 = r10;
continue _1;
}
else{
break _1;
}
}
}
        return;
}

function _ZN20btAxisSweep3InternalItE11sortMaxDownEitP12btDispatcherb(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var r12;
        var r13;
        var r14;
        var r15;
var __label__ = 0;
        i7 = sp + -16;var g0 = i7>>2; // save stack
        r0 = heap32[(fp+1)];
        r1 = heap32[(fp)];
        r2 = r0 << 2;
        r2 = (r1 + r2)|0;
        r3 = heap32[(fp+2)];
        r2 = r2 >> 2;
        r2 = heap32[(r2+17)];
        r4 = r3 << 2;
        r5 = (r2 + r4)|0;
        r4 = heapU16[(r2+r4)>>1];
        r6 = heapU16[(r5+-4)>>1];
if(!(uint(r4) >=uint(r6))) //_LBB63_14
{
        r1 = r1 >> 2;
        r4 = heap32[(fp+3)];
        r6 = heap32[(fp+4)];
        r7 = heap32[(r1+15)];
        r3 = r3 << 2;
        r2 = (r2 + r3)|0;
        r2 = heapU16[(r2+2)>>1];
        r3 = 1;
        r8 = r3 << r0;
        r2 = r2 << 6;
        r8 = r8 & 3;
        r2 = (r7 + r2)|0;
        r0 = r0 << 1;
        r3 = r3 << r8;
        r2 = (r2 + r0)|0;
        r3 = r3 & 3;
        r2 = (r2 + 54)|0;
        r5 = (r5 + 2)|0;
_3: while(true){
        r9 = heapU8[r5+-6];
        r10 = heapU16[(r5+-4)>>1];
        r9 = r9 & 1;
        if(r9 != 0) //_LBB63_11
{
        r9 = r10 << 6;
        r7 = (r7 + r9)|0;
        r7 = (r7 + r0)|0;
        r9 = heapU16[(r7+54)>>1];
        r9 = (r9 + 1)|0;
        heap16[(r7+54)>>1] = r9;
}
else{
if(!(r6 ==0)) //_LBB63_10
{
        r9 = heapU16[(r5)>>1];
        r9 = r9 << 6;
        r11 = r10 << 6;
        r9 = (r7 + r9)|0;
        r12 = r8 << 1;
        r11 = (r7 + r11)|0;
        r13 = (r9 + r12)|0;
        r12 = (r11 + r12)|0;
        r14 = heapU16[(r13+54)>>1];
        r15 = heapU16[(r12+48)>>1];
if(!(uint(r14) <uint(r15))) //_LBB63_10
{
        r12 = heapU16[(r12+54)>>1];
        r13 = heapU16[(r13+48)>>1];
if(!(uint(r12) <uint(r13))) //_LBB63_10
{
        r12 = r3 << 1;
        r13 = (r9 + r12)|0;
        r12 = (r11 + r12)|0;
        r14 = heapU16[(r13+54)>>1];
        r15 = heapU16[(r12+48)>>1];
if(!(uint(r14) <uint(r15))) //_LBB63_10
{
        r12 = heapU16[(r12+54)>>1];
        r13 = heapU16[(r13+48)>>1];
if(!(uint(r12) <uint(r13))) //_LBB63_10
{
        r12 = heap32[(r1+23)];
        r13 = r12 >> 2;
        r13 = heap32[(r13)];
        r13 = r13 >> 2;
        r13 = heap32[(r13+3)];
        heap32[(g0)] = r12;
        heap32[(g0+1)] = r9;
        heap32[(g0+2)] = r11;
        heap32[(g0+3)] = r4;
        __FUNCTION_TABLE__[(r13)>>2](i7);
        r12 = heap32[(r1+24)];
if(!(r12 ==0)) //_LBB63_10
{
        r13 = r12 >> 2;
        r13 = heap32[(r13)];
        r13 = r13 >> 2;
        r13 = heap32[(r13+3)];
        heap32[(g0)] = r12;
        heap32[(g0+1)] = r9;
        heap32[(g0+2)] = r11;
        heap32[(g0+3)] = r4;
        __FUNCTION_TABLE__[(r13)>>2](i7);
}
}
}
}
}
}
        r10 = r10 << 6;
        r7 = (r7 + r10)|0;
        r7 = (r7 + r0)|0;
        r10 = heapU16[(r7+48)>>1];
        r10 = (r10 + 1)|0;
        heap16[(r7+48)>>1] = r10;
}
        r7 = heapU16[(r2)>>1];
        r7 = (r7 + -1)|0;
        heap16[(r2)>>1] = r7;
        r7 = heapU16[(r5+-2)>>1];
        r9 = heapU16[(r5)>>1];
        r10 = heapU16[(r5+-4)>>1];
        r11 = heapU16[(r5+-6)>>1];
        heap16[(r5+-2)>>1] = r11;
        heap16[(r5)>>1] = r10;
        heap16[(r5+-6)>>1] = r7;
        heap16[(r5+-4)>>1] = r9;
        r9 = heapU16[(r5+-10)>>1];
        if(uint(r7) >=uint(r9)) //_LBB63_14
{
break _3;
}
else{
        r7 = heap32[(r1+15)];
        r5 = (r5 + -4)|0;
continue _3;
}
}
}
        return;
}

function _ZN20btAxisSweep3InternalItE7setAabbEP17btBroadphaseProxyRK9btVector3S5_P12btDispatcher(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var r12;
        var r13;
        var r14;
        var r15;
        var r16;
        var r17;
        var r18;
        var r19;
        var r20;
        var f0;
        var f1;
        var f2;
var __label__ = 0;
        i7 = sp + -40;var g0 = i7>>2; // save stack
        r0 = heap32[(fp+2)];
        r1 = heap32[(fp+1)];
        r2 = r0 >> 2;
        r3 = r1 >> 2;
        heap32[(r3+4)] = heap32[(r2)];
        heap32[(r3+5)] = heap32[(r2+1)];
        r4 = heap32[(fp+3)];
        heap32[(r3+6)] = heap32[(r2+2)];
        heap32[(r3+7)] = heap32[(r2+3)];
        r5 = r4 >> 2;
        heap32[(r3+8)] = heap32[(r5)];
        heap32[(r3+9)] = heap32[(r5+1)];
        r6 = heap32[(fp)];
        heap32[(r3+10)] = heap32[(r5+2)];
        heap32[(r3+11)] = heap32[(r5+3)];
        r7 = r6 >> 2;
        r1 = heapU16[(r1+12)>>1];
        r8 = heap32[(r7+15)];
        f0 = heapFloat[(r2)];
        f1 = heapFloat[(r2+1)];
        f2 = heapFloat[(r2+2)];
        r2 = sp + -6;
        heap32[(g0)] = r6;
        heap32[(g0+1)] = r2;
        heapFloat[(g0+2)] = f0;
        heapFloat[(g0+3)] = f1;
        heapFloat[(g0+4)] = f2;
        heap32[(g0+5)] = 0;
        _ZNK20btAxisSweep3InternalItE8quantizeEPtRK9btVector3i(i7);
        f0 = heapFloat[(r5)];
        f1 = heapFloat[(r5+1)];
        f2 = heapFloat[(r5+2)];
        r1 = r1 << 6;
        r5 = sp + -12;
        r1 = (r8 + r1)|0;
        heap32[(g0)] = r6;
        heap32[(g0+1)] = r5;
        heapFloat[(g0+2)] = f0;
        heapFloat[(g0+3)] = f1;
        heapFloat[(g0+4)] = f2;
        heap32[(g0+5)] = 1;
        r8 = heap32[(fp+4)];
        r1 = (r1 + 48)|0;
        r9 = 0;
        _ZNK20btAxisSweep3InternalItE8quantizeEPtRK9btVector3i(i7);
        r10 = r9;
_1: while(true){
        r11 = r9 << 2;
        r12 = r9 << 1;
        r13 = (r1 - r12)|0;
        r11 = (r6 - r11)|0;
        r14 = heapU16[(r13)>>1];
        r13 = heapU16[(r13+6)>>1];
        r11 = r11 >> 2;
        r15 = (r2 - r12)|0;
        r12 = (r5 - r12)|0;
        r16 = heap32[(r11+17)];
        r17 = r14 << 2;
        r18 = r13 << 2;
        r15 = heapU16[(r15)>>1];
        r19 = heapU16[(r16+r17)>>1];
        r20 = heapU16[(r16+r18)>>1];
        r12 = heapU16[(r12)>>1];
        heap16[(r16+r17)>>1] = r15;
        r11 = heap32[(r11+17)];
        r15 = (r15 - r19)|0;
        heap16[(r11+r18)>>1] = r12;
if(!(r15 >-1)) //_LBB64_3
{
        heap32[(g0)] = r6;
        heap32[(g0+1)] = r10;
        heap32[(g0+2)] = r14;
        heap32[(g0+3)] = 1;
        _ZN20btAxisSweep3InternalItE11sortMinDownEitP12btDispatcherb(i7);
}
        r11 = (r12 - r20)|0;
if(!(r11 <1)) //_LBB64_5
{
        heap32[(g0)] = r6;
        heap32[(g0+1)] = r10;
        heap32[(g0+2)] = r13;
        heap32[(g0+3)] = 1;
        _ZN20btAxisSweep3InternalItE9sortMaxUpEitP12btDispatcherb(i7);
}
if(!(r15 <1)) //_LBB64_7
{
        heap32[(g0)] = r6;
        heap32[(g0+1)] = r10;
        heap32[(g0+2)] = r14;
        heap32[(g0+3)] = r8;
        heap32[(g0+4)] = 1;
        _ZN20btAxisSweep3InternalItE9sortMinUpEitP12btDispatcherb(i7);
}
if(!(r11 >-1)) //_LBB64_9
{
        heap32[(g0)] = r6;
        heap32[(g0+1)] = r10;
        heap32[(g0+2)] = r13;
        heap32[(g0+3)] = r8;
        heap32[(g0+4)] = 1;
        _ZN20btAxisSweep3InternalItE11sortMaxDownEitP12btDispatcherb(i7);
}
        r9 = (r9 + -1)|0;
        r10 = (r10 + 1)|0;
        if(r9 !=-3) //_LBB64_1
{
continue _1;
}
else{
break _1;
}
}
        r1 = heap32[(r7+27)];
if(!(r1 ==0)) //_LBB64_12
{
        r2 = r1 >> 2;
        r2 = heap32[(r2)];
        r2 = r2 >> 2;
        r2 = heap32[(r2+4)];
        r3 = heap32[(r3+15)];
        heap32[(g0)] = r1;
        heap32[(g0+1)] = r3;
        heap32[(g0+2)] = r0;
        heap32[(g0+3)] = r4;
        heap32[(g0+4)] = r8;
        __FUNCTION_TABLE__[(r2)>>2](i7);
}
        return;
}

function _ZN20btAxisSweep3InternalItE12destroyProxyEP17btBroadphaseProxyP12btDispatcher(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var r12;
        var r13;
var __label__ = 0;
        i7 = sp + -24;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = r0 >> 2;
        r2 = heap32[(r1+27)];
        r3 = heap32[(fp+1)];
        r4 = heap32[(fp+2)];
if(!(r2 ==0)) //_LBB65_2
{
        r5 = r2 >> 2;
        r5 = heap32[(r5)];
        r5 = r5 >> 2;
        r6 = r3 >> 2;
        r5 = heap32[(r5+3)];
        r6 = heap32[(r6+15)];
        heap32[(g0)] = r2;
        heap32[(g0+1)] = r6;
        heap32[(g0+2)] = r4;
        __FUNCTION_TABLE__[(r5)>>2](i7);
}
        r2 = heap32[(r1+23)];
        r5 = r2 >> 2;
        r5 = heap32[(r5)];
        r5 = r5 >> 2;
        r3 = r3 >> 2;
        r3 = heap32[(r3+3)];
        r6 = heap32[(r1+15)];
        r5 = heap32[(r5+14)];
        heap32[(g0)] = r2;
        __FUNCTION_TABLE__[(r5)>>2](i7);
        r2 = r_g0;
if(!(r2 !=0)) //_LBB65_4
{
        r2 = heap32[(r1+23)];
        r5 = r2 >> 2;
        r5 = heap32[(r5)];
        r5 = r5 >> 2;
        r5 = heap32[(r5+4)];
        r7 = r3 & 65535;
        r7 = r7 << 6;
        r7 = (r6 + r7)|0;
        heap32[(g0)] = r2;
        heap32[(g0+1)] = r7;
        heap32[(g0+2)] = r4;
        __FUNCTION_TABLE__[(r5)>>2](i7);
}
        r2 = heap32[(r1+15)];
        r5 = heapU16[(r0+56)>>1];
        r7 = heapU16[(r2+54)>>1];
        r7 = (r7 + -2)|0;
        heap16[(r2+54)>>1] = r7;
        r2 = heap32[(r1+15)];
        r7 = heapU16[(r2+56)>>1];
        r7 = (r7 + -2)|0;
        r8 = r3 & 65535;
        heap16[(r2+56)>>1] = r7;
        r2 = heap32[(r1+15)];
        r7 = r8 << 6;
        r9 = heapU16[(r2+58)>>1];
        r6 = (r6 + r7)|0;
        r5 = r5 << 1;
        r6 = (r6 + 54)|0;
        r10 = 0;
        r9 = (r9 + -2)|0;
        heap16[(r2+58)>>1] = r9;
        r2 = r10;
_7: while(true){
        r9 = r2 << 2;
        r9 = (r0 + r9)|0;
        r9 = r9 >> 2;
        r11 = heapU16[(r6)>>1];
        r12 = heapU16[(r0+6)>>1];
        r9 = heap32[(r9+17)];
        r13 = r11 << 2;
        heap16[(r9+r13)>>1] = r12;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r2;
        heap32[(g0+2)] = r11;
        heap32[(g0+3)] = 0;
        _ZN20btAxisSweep3InternalItE9sortMaxUpEitP12btDispatcherb(i7);
        r11 = heapU16[(r6+-6)>>1];
        r12 = heapU16[(r0+6)>>1];
        r13 = r11 << 2;
        heap16[(r9+r13)>>1] = r12;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r2;
        heap32[(g0+2)] = r11;
        heap32[(g0+3)] = r4;
        heap32[(g0+4)] = 0;
        r11 = r5 << 2;
        _ZN20btAxisSweep3InternalItE9sortMinUpEitP12btDispatcherb(i7);
        r9 = (r9 + r11)|0;
        heap16[(r9+-2)>>1] = r10;
        r11 = heapU16[(r0+6)>>1];
        r2 = (r2 + 1)|0;
        r6 = (r6 + 2)|0;
        heap16[(r9+-4)>>1] = r11;
        if(r2 !=3) //_LBB65_5
{
continue _7;
}
else{
break _7;
}
}
if(!(r8 ==0)) //_LBB65_8
{
        r2 = heapU16[(r0+58)>>1];
        if(uint(r2) >uint(r8)) //_LBB65_9
{
        r1 = heap32[(r1+15)];
        r2 = heapU16[(r0+64)>>1];
        r1 = (r1 + r7)|0;
        heap16[(r1+48)>>1] = r2;
        heap16[(r0+64)>>1] = r3;
        r1 = heapU16[(r0+56)>>1];
        r1 = (r1 + -1)|0;
        heap16[(r0+56)>>1] = r1;
        return;
}
}
        r0 = _2E_str11;
        r1 = _2E_str112;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 495;
        _assert(i7);
}

function _ZN20btAxisSweep3InternalItE11createProxyERK9btVector3S3_iPvssP12btDispatcherS4_(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var r12;
        var r13;
        var r14;
        var r15;
        var r16;
        var r17;
        var r18;
        var f0;
        var f1;
        var f2;
var __label__ = 0;
        i7 = sp + -56;var g0 = i7>>2; // save stack
        r0 = heap32[(fp+1)];
        r1 = r0 >> 2;
        f0 = heapFloat[(r1)];
        f1 = heapFloat[(r1+1)];
        f2 = heapFloat[(r1+2)];
        r1 = heap32[(fp)];
        r2 = sp + -6;
        heap32[(g0)] = r1;
        heap32[(g0+1)] = r2;
        heapFloat[(g0+2)] = f0;
        heapFloat[(g0+3)] = f1;
        heapFloat[(g0+4)] = f2;
        heap32[(g0+5)] = 0;
        r2 = heap32[(fp+2)];
        _ZNK20btAxisSweep3InternalItE8quantizeEPtRK9btVector3i(i7);
        r3 = r2 >> 2;
        f0 = heapFloat[(r3)];
        f1 = heapFloat[(r3+1)];
        f2 = heapFloat[(r3+2)];
        r3 = sp + -12;
        heap32[(g0)] = r1;
        heap32[(g0+1)] = r3;
        heapFloat[(g0+2)] = f0;
        heapFloat[(g0+3)] = f1;
        heapFloat[(g0+4)] = f2;
        heap32[(g0+5)] = 1;
        _ZNK20btAxisSweep3InternalItE8quantizeEPtRK9btVector3i(i7);
        r3 = heapU16[(r1+64)>>1];
        if(r3 !=0) //_LBB66_2
{
        r4 = heap32[(fp+4)];
        r5 = heap32[(fp+5)];
        r6 = heap32[(fp+6)];
        r7 = heap32[(fp+7)];
        r8 = heap32[(fp+8)];
        r9 = r1 >> 2;
        r10 = r3 << 6;
        r11 = heap32[(r9+15)];
        r10 = (r11 + r10)|0;
        r11 = heapU16[(r10+48)>>1];
        heap16[(r1+64)>>1] = r11;
        r11 = heapU16[(r1+56)>>1];
        r11 = (r11 + 1)|0;
        r12 = r10 >> 2;
        heap16[(r1+56)>>1] = r11;
        heap32[(r12+3)] = r3;
        heap32[(r12)] = r4;
        heap16[(r10+4)>>1] = r5;
        heap16[(r10+6)>>1] = r6;
        heap32[(r12+2)] = r8;
        r8 = heap32[(r9+15)];
        r11 = heapU16[(r1+56)>>1];
        r11 = r11 << 1;
        r12 = heapU16[(r8+54)>>1];
        r13 = r11 & 65534;
        r12 = (r12 + 2)|0;
        r14 = (r13 + -1)|0;
        heap16[(r8+54)>>1] = r12;
        r8 = r14 << 2;
        r12 = heap32[(r9+17)];
        r14 = (r12 + r8)|0;
        r15 = r13 | 1;
        r14 = heapU16[(r14+2)>>1];
        r16 = heapU16[(r12+r8)>>1];
        r15 = r15 << 2;
        r17 = (r12 + r15)|0;
        heap16[(r12+r15)>>1] = r16;
        heap16[(r17+2)>>1] = r14;
        r12 = heap32[(r9+17)];
        r14 = heapU16[(sp+-6)>>1];
        heap16[(r12+r8)>>1] = r14;
        r12 = heap32[(r9+17)];
        r12 = (r12 + r8)|0;
        heap16[(r12+2)>>1] = r3;
        r12 = heap32[(r9+17)];
        r14 = heapU16[(sp+-12)>>1];
        r13 = r13 << 2;
        heap16[(r12+r13)>>1] = r14;
        r12 = heap32[(r9+17)];
        r12 = (r12 + r13)|0;
        r14 = (r11 + -1)|0;
        heap16[(r12+2)>>1] = r3;
        heap16[(r10+48)>>1] = r14;
        heap16[(r10+54)>>1] = r11;
        r12 = heap32[(r9+15)];
        r16 = heapU16[(r12+56)>>1];
        r16 = (r16 + 2)|0;
        heap16[(r12+56)>>1] = r16;
        r12 = heap32[(r9+18)];
        r16 = (r12 + r8)|0;
        r16 = heapU16[(r16+2)>>1];
        r17 = heapU16[(r12+r8)>>1];
        r18 = (r12 + r15)|0;
        heap16[(r12+r15)>>1] = r17;
        heap16[(r18+2)>>1] = r16;
        r12 = heap32[(r9+18)];
        r16 = heapU16[(sp+-4)>>1];
        heap16[(r12+r8)>>1] = r16;
        r12 = heap32[(r9+18)];
        r12 = (r12 + r8)|0;
        heap16[(r12+2)>>1] = r3;
        r12 = heap32[(r9+18)];
        r16 = heapU16[(sp+-10)>>1];
        heap16[(r12+r13)>>1] = r16;
        r12 = heap32[(r9+18)];
        r12 = (r12 + r13)|0;
        heap16[(r12+2)>>1] = r3;
        heap16[(r10+50)>>1] = r14;
        heap16[(r10+56)>>1] = r11;
        r12 = heap32[(r9+15)];
        r16 = heapU16[(r12+58)>>1];
        r16 = (r16 + 2)|0;
        heap16[(r12+58)>>1] = r16;
        r12 = heap32[(r9+19)];
        r16 = (r12 + r8)|0;
        r16 = heapU16[(r16+2)>>1];
        r17 = heapU16[(r12+r8)>>1];
        r18 = (r12 + r15)|0;
        heap16[(r12+r15)>>1] = r17;
        heap16[(r18+2)>>1] = r16;
        r12 = heap32[(r9+19)];
        r15 = heapU16[(sp+-2)>>1];
        heap16[(r12+r8)>>1] = r15;
        r12 = heap32[(r9+19)];
        r8 = (r12 + r8)|0;
        heap16[(r8+2)>>1] = r3;
        r8 = heap32[(r9+19)];
        r12 = heapU16[(sp+-8)>>1];
        heap16[(r8+r13)>>1] = r12;
        r8 = heap32[(r9+19)];
        r8 = (r8 + r13)|0;
        heap16[(r8+2)>>1] = r3;
        heap16[(r10+52)>>1] = r14;
        heap16[(r10+58)>>1] = r11;
        r8 = heapU16[(r10+48)>>1];
        heap32[(g0)] = r1;
        heap32[(g0+1)] = 0;
        heap32[(g0+2)] = r8;
        heap32[(g0+3)] = 0;
        _ZN20btAxisSweep3InternalItE11sortMinDownEitP12btDispatcherb(i7);
        r8 = heapU16[(r10+54)>>1];
        heap32[(g0)] = r1;
        heap32[(g0+1)] = 0;
        heap32[(g0+2)] = r8;
        heap32[(g0+3)] = r7;
        heap32[(g0+4)] = 0;
        _ZN20btAxisSweep3InternalItE11sortMaxDownEitP12btDispatcherb(i7);
        r8 = heapU16[(r10+50)>>1];
        heap32[(g0)] = r1;
        heap32[(g0+1)] = 1;
        heap32[(g0+2)] = r8;
        heap32[(g0+3)] = 0;
        _ZN20btAxisSweep3InternalItE11sortMinDownEitP12btDispatcherb(i7);
        r8 = heapU16[(r10+56)>>1];
        heap32[(g0)] = r1;
        heap32[(g0+1)] = 1;
        heap32[(g0+2)] = r8;
        heap32[(g0+3)] = r7;
        heap32[(g0+4)] = 0;
        _ZN20btAxisSweep3InternalItE11sortMaxDownEitP12btDispatcherb(i7);
        r8 = heapU16[(r10+52)>>1];
        heap32[(g0)] = r1;
        heap32[(g0+1)] = 2;
        heap32[(g0+2)] = r8;
        heap32[(g0+3)] = 1;
        _ZN20btAxisSweep3InternalItE11sortMinDownEitP12btDispatcherb(i7);
        r8 = heapU16[(r10+58)>>1];
        heap32[(g0)] = r1;
        heap32[(g0+1)] = 2;
        heap32[(g0+2)] = r8;
        heap32[(g0+3)] = r7;
        heap32[(g0+4)] = 1;
        _ZN20btAxisSweep3InternalItE11sortMaxDownEitP12btDispatcherb(i7);
        updateMandreelStats(performance.now());
        r1 = heap32[(r9+27)];
        r8 = heap32[(r9+15)];
if(!(r1 ==0)) //_LBB66_4
{
        r9 = heap32[(fp+3)];
        r10 = r1 >> 2;
        r10 = heap32[(r10)];
        r10 = r10 >> 2;
        r10 = heap32[(r10+2)];
        r11 = r3 << 6;
        heap32[(g0)] = r1;
        heap32[(g0+1)] = r0;
        heap32[(g0+2)] = r2;
        heap32[(g0+3)] = r9;
        heap32[(g0+4)] = r4;
        heap32[(g0+5)] = r5;
        heap32[(g0+6)] = r6;
        heap32[(g0+7)] = r7;
        heap32[(g0+8)] = 0;
        r0 = (r8 + r11)|0;
        r0 = r0 >> 2;
        __FUNCTION_TABLE__[(r10)>>2](i7);
        heap32[(r0+15)] = r_g0;
}
        r0 = r3 << 6;
        r0 = (r8 + r0)|0;
        r_g0 = r0;
        return;
}
else{
        r0 = _2E_str213;
        r1 = _2E_str112;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 483;
        _assert(i7);
}
}

function _ZN20btAlignedObjectArrayI16btBroadphasePairE6resizeEiRKS0_(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = r0 >> 2;
        r2 = heap32[(r1+1)];
        r3 = heap32[(fp+1)];
_1: do {
if(!(r2 >r3)) //_LBB67_20
{
if(!(r2 >=r3)) //_LBB67_20
{
        r4 = heap32[(r1+2)];
if(!(r4 >=r3)) //_LBB67_18
{
        if(r3 !=0) //_LBB67_5
{
        r4 = gNumAlignedAllocs;
        r4 = r4 >> 2;
        r5 = heap32[(r4)];
        r6 = r3 << 4;
        r5 = (r5 + 1)|0;
        r6 = r6 | 3;
        heap32[(r4)] = r5;
        r4 = (r6 + 16)|0;
        heap32[(g0)] = r4;
        malloc(i7);
        r4 = r_g0;
        if(r4 !=0) //_LBB67_7
{
        r5 = 0;
        r6 = (r4 + 4)|0;
        r5 = (r5 - r6)|0;
        r5 = r5 & 15;
        r5 = (r4 + r5)|0;
        r6 = (r5 + 4)|0;
        r5 = r5 >> 2;
        heap32[(r5)] = r4;
        r4 = r6;
}
}
else{
        r4 = 0;
}
        r5 = (r0 + 12)|0;
        if(r2 <1) //_LBB67_10
{
        r6 = r5 >> 2;
        r8 = heap32[(r6)];
}
else{
        r6 = 0;
        r7 = (r6 - r2)|0;
_14: while(true){
        r8 = r5 >> 2;
        r8 = heap32[(r8)];
        r9 = r6 << 4;
        r10 = (r8 - r9)|0;
        r10 = r10 >> 2;
        r9 = (r4 - r9)|0;
        r11 = heap32[(r10)];
        r9 = r9 >> 2;
        heap32[(r9)] = r11;
        r11 = heap32[(r10+1)];
        heap32[(r9+1)] = r11;
        r11 = heap32[(r10+2)];
        heap32[(r9+2)] = r11;
        r10 = heap32[(r10+3)];
        r6 = (r6 + -1)|0;
        heap32[(r9+3)] = r10;
if(!(r7 !=r6)) //_LBB67_11
{
break _14;
}
}
        r5 = (r0 + 12)|0;
}
if(!(r8 ==0)) //_LBB67_17
{
        r6 = heapU8[r0+16];
if(!(r6 ==0)) //_LBB67_16
{
        r6 = gNumAlignedFree;
        r6 = r6 >> 2;
        r7 = heap32[(r6)];
        r7 = (r7 + 1)|0;
        r8 = r8 >> 2;
        heap32[(r6)] = r7;
        r6 = heap32[(r8+-1)];
        heap32[(g0)] = r6;
        free(i7);
}
        r6 = r5 >> 2;
        heap32[(r6)] = 0;
}
        r6 = 1;
        r5 = r5 >> 2;
        heap8[r0+16] = r6;
        heap32[(r5)] = r4;
        heap32[(r1+2)] = r3;
        if(r2 >=r3) //_LBB67_20
{
break _1;
}
}
        r0 = heap32[(fp+2)];
_25: while(true){
        r4 = r0 >> 2;
        r5 = r2 << 4;
        r6 = heap32[(r1+3)];
        r5 = (r6 + r5)|0;
        r6 = heap32[(r4)];
        r5 = r5 >> 2;
        heap32[(r5)] = r6;
        r6 = heap32[(r4+1)];
        heap32[(r5+1)] = r6;
        r6 = heap32[(r4+2)];
        heap32[(r5+2)] = r6;
        r4 = heap32[(r4+3)];
        r2 = (r2 + 1)|0;
        heap32[(r5+3)] = r4;
        if(r3 !=r2) //_LBB67_19
{
continue _25;
}
else{
break _1;
}
}
}
}
} while(0);
        heap32[(r1+1)] = r3;
        return;
}

function _ZN20btAxisSweep3InternalItE25calculateOverlappingPairsEP12btDispatcher(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var r12;
        var r13;
        var r14;
var __label__ = 0;
        i7 = sp + -48;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = r0 >> 2;
        r1 = heap32[(r0+23)];
        r2 = r1 >> 2;
        r2 = heap32[(r2)];
        r2 = r2 >> 2;
        r2 = heap32[(r2+14)];
        heap32[(g0)] = r1;
        __FUNCTION_TABLE__[(r2)>>2](i7);
        r1 = r_g0;
if(!(r1 ==0)) //_LBB68_22
{
        r1 = heap32[(fp+1)];
        r2 = heap32[(r0+23)];
        r3 = r2 >> 2;
        r3 = heap32[(r3)];
        r3 = r3 >> 2;
        r3 = heap32[(r3+7)];
        heap32[(g0)] = r2;
        __FUNCTION_TABLE__[(r3)>>2](i7);
        r2 = r_g0;
        r3 = r2 >> 2;
        r3 = heap32[(r3+1)];
        if(r3 >1) //_LBB68_3
{
        r3 = (r3 + -1)|0;
        heap32[(g0)] = r2;
        heap32[(g0+1)] = 0;
        heap32[(g0+2)] = r3;
        r3 = r2 >> 2;
        _ZN20btAlignedObjectArrayI16btBroadphasePairE17quickSortInternalI29btBroadphasePairSortPredicateEEvT_ii(i7);
        r3 = heap32[(r3+1)];
}
        r4 = sp + -32;
        r5 = r4 >> 2;
        heap32[(fp+-8)] = 0;
        heap32[(r5+1)] = 0;
        heap32[(r5+2)] = 0;
        heap32[(r5+3)] = 0;
        r5 = heap32[(r0+26)];
        r3 = (r3 - r5)|0;
        heap32[(g0)] = r2;
        heap32[(g0+1)] = r3;
        heap32[(g0+2)] = r4;
        r3 = 0;
        _ZN20btAlignedObjectArrayI16btBroadphasePairE6resizeEiRKS0_(i7);
        heap32[(r0+26)] = 0;
        r4 = r3;
        r5 = r3;
        r6 = r3;
        r7 = r3;
_6: while(true){
        r8 = r2 >> 2;
        r9 = heap32[(r8+1)];
        if(r9 >r7) //_LBB68_5
{
        r8 = heap32[(r8+3)];
        r9 = r7 << 4;
        r9 = (r8 + r9)|0;
        r9 = r9 >> 2;
        r10 = heap32[(r9)];
        r11 = r7 << 2;
        r12 = heap32[(r9+1)];
        if(r10 !=r5) //_LBB68_7
{
__label__ = 6;
}
else{
        if(r12 ==r6) //_LBB68_13
{
        r4 = heap32[(r9+2)];
        if(r4 ==0) //_LBB68_15
{
__label__ = 13;
}
else{
__label__ = 12;
break _6;
}
}
else{
__label__ = 6;
}
}
if (__label__ == 6){
        r5 = (r12 + 54)|0;
        r6 = (r10 + 48)|0;
        r9 = 0;
_14: while(true){
        if(r9 <3) //_LBB68_8
{
        r13 = heapU16[(r6+6)>>1];
        r14 = heapU16[(r5+-6)>>1];
        if(uint(r13) <uint(r14)) //_LBB68_15
{
__label__ = 13;
break _14;
}
else{
        r13 = heapU16[(r5)>>1];
        r14 = heapU16[(r6)>>1];
        if(uint(r13) <uint(r14)) //_LBB68_15
{
__label__ = 13;
break _14;
}
else{
        r9 = (r9 + 1)|0;
        r5 = (r5 + 2)|0;
        r6 = (r6 + 2)|0;
}
}
}
else{
__label__ = 14;
break _14;
}
}
}
if (__label__ == 13){
        r4 = heap32[(r0+23)];
        r5 = r4 >> 2;
        r5 = heap32[(r5)];
        r5 = r5 >> 2;
        r5 = heap32[(r5+8)];
        r6 = (r8 + r3)|0;
        r9 = r11 << 2;
        heap32[(g0)] = r4;
        heap32[(g0+1)] = r6;
        heap32[(g0+2)] = r1;
        r4 = (r8 + r9)|0;
        r4 = r4 >> 2;
        __FUNCTION_TABLE__[(r5)>>2](i7);
        heap32[(r4)] = 0;
        heap32[(r4+1)] = 0;
        r4 = heap32[(r0+26)];
        r4 = (r4 + 1)|0;
        r5 = gOverlappingPairs;
        r5 = r5 >> 2;
        heap32[(r0+26)] = r4;
        r6 = heap32[(r5)];
        r6 = (r6 + -1)|0;
        heap32[(r5)] = r6;
}
        r7 = (r7 + 1)|0;
        r3 = (r3 + 16)|0;
        r5 = r10;
        r6 = r12;
}
else{
__label__ = 16;
break _6;
}
}
switch(__label__ ){//multiple entries
case 16:
        if(r9 >1) //_LBB68_20
{
        r4 = (r9 + -1)|0;
        heap32[(g0)] = r2;
        heap32[(g0+1)] = 0;
        heap32[(g0+2)] = r4;
        _ZN20btAlignedObjectArrayI16btBroadphasePairE17quickSortInternalI29btBroadphasePairSortPredicateEEvT_ii(i7);
        r9 = heap32[(r8+1)];
        r4 = heap32[(r0+26)];
}
        r1 = sp + -16;
        r3 = r1 >> 2;
        heap32[(fp+-4)] = 0;
        heap32[(r3+1)] = 0;
        heap32[(r3+2)] = 0;
        heap32[(r3+3)] = 0;
        r3 = (r9 - r4)|0;
        heap32[(g0)] = r2;
        heap32[(g0+1)] = r3;
        heap32[(g0+2)] = r1;
        _ZN20btAlignedObjectArrayI16btBroadphasePairE6resizeEiRKS0_(i7);
        heap32[(r0+26)] = 0;
break;
case 12:
        r8 = _2E_str314;
        r0 = _2E_str112;
        heap32[(g0)] = r8;
        heap32[(g0+1)] = r0;
        heap32[(g0+2)] = 687;
        _assert(i7);
break;
}
}
        return;
}

function _ZN20btAxisSweep3InternalItED2Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV20btAxisSweep3InternalItE;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        r1 = heap32[(r2+27)];
if(!(r1 ==0)) //_LBB69_5
{
        r1 = heap32[(r2+28)];
        r3 = r1 >> 2;
        r3 = heap32[(r3)];
        r3 = r3 >> 2;
        r3 = heap32[(r3)];
        heap32[(g0)] = r1;
        __FUNCTION_TABLE__[(r3)>>2](i7);
        r1 = heap32[(r2+28)];
if(!(r1 ==0)) //_LBB69_3
{
        r3 = gNumAlignedFree;
        r3 = r3 >> 2;
        r4 = heap32[(r3)];
        r4 = (r4 + 1)|0;
        r1 = r1 >> 2;
        heap32[(r3)] = r4;
        r1 = heap32[(r1+-1)];
        heap32[(g0)] = r1;
        free(i7);
}
        r1 = heap32[(r2+27)];
        r3 = r1 >> 2;
        r3 = heap32[(r3)];
        r3 = r3 >> 2;
        r3 = heap32[(r3)];
        heap32[(g0)] = r1;
        __FUNCTION_TABLE__[(r3)>>2](i7);
        r1 = heap32[(r2+27)];
if(!(r1 ==0)) //_LBB69_5
{
        r3 = gNumAlignedFree;
        r3 = r3 >> 2;
        r4 = heap32[(r3)];
        r4 = (r4 + 1)|0;
        r1 = r1 >> 2;
        heap32[(r3)] = r4;
        r1 = heap32[(r1+-1)];
        heap32[(g0)] = r1;
        free(i7);
}
}
        r1 = 0;
_8: while(true){
        r3 = r1 << 2;
        r3 = (r0 + r3)|0;
        r3 = r3 >> 2;
        r3 = heap32[(r3+22)];
if(!(r3 ==0)) //_LBB69_8
{
        r4 = gNumAlignedFree;
        r4 = r4 >> 2;
        r5 = heap32[(r4)];
        r5 = (r5 + 1)|0;
        r3 = r3 >> 2;
        heap32[(r4)] = r5;
        r3 = heap32[(r3+-1)];
        heap32[(g0)] = r3;
        free(i7);
}
        r1 = (r1 + -1)|0;
        if(r1 !=-3) //_LBB69_6
{
continue _8;
}
else{
break _8;
}
}
        r1 = heap32[(r2+15)];
if(!(r1 ==0)) //_LBB69_11
{
        r3 = gNumAlignedFree;
        r3 = r3 >> 2;
        r4 = heap32[(r3)];
        r4 = (r4 + 1)|0;
        r1 = r1 >> 2;
        heap32[(r3)] = r4;
        r1 = heap32[(r1+-1)];
        heap32[(g0)] = r1;
        free(i7);
}
        r0 = heapU8[r0+100];
if(!(r0 ==0)) //_LBB69_14
{
        r0 = heap32[(r2+23)];
        r1 = r0 >> 2;
        r1 = heap32[(r1)];
        r1 = r1 >> 2;
        r1 = heap32[(r1)];
        heap32[(g0)] = r0;
        __FUNCTION_TABLE__[(r1)>>2](i7);
        r0 = heap32[(r2+23)];
if(!(r0 ==0)) //_LBB69_14
{
        r1 = gNumAlignedFree;
        r1 = r1 >> 2;
        r2 = heap32[(r1)];
        r2 = (r2 + 1)|0;
        r0 = r0 >> 2;
        heap32[(r1)] = r2;
        r0 = heap32[(r0+-1)];
        heap32[(g0)] = r0;
        free(i7);
}
}
        return;
}

function _ZN20btAxisSweep3InternalItED1Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        heap32[(g0)] = r0;
        _ZN20btAxisSweep3InternalItED2Ev(i7);
        return;
}

function _ZN20btAxisSweep3InternalItED0Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV20btAxisSweep3InternalItE;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        r1 = heap32[(r2+27)];
if(!(r1 ==0)) //_LBB71_5
{
        r1 = heap32[(r2+28)];
        r3 = r1 >> 2;
        r3 = heap32[(r3)];
        r3 = r3 >> 2;
        r3 = heap32[(r3)];
        heap32[(g0)] = r1;
        __FUNCTION_TABLE__[(r3)>>2](i7);
        r1 = heap32[(r2+28)];
if(!(r1 ==0)) //_LBB71_3
{
        r3 = gNumAlignedFree;
        r3 = r3 >> 2;
        r4 = heap32[(r3)];
        r4 = (r4 + 1)|0;
        r1 = r1 >> 2;
        heap32[(r3)] = r4;
        r1 = heap32[(r1+-1)];
        heap32[(g0)] = r1;
        free(i7);
}
        r1 = heap32[(r2+27)];
        r3 = r1 >> 2;
        r3 = heap32[(r3)];
        r3 = r3 >> 2;
        r3 = heap32[(r3)];
        heap32[(g0)] = r1;
        __FUNCTION_TABLE__[(r3)>>2](i7);
        r1 = heap32[(r2+27)];
if(!(r1 ==0)) //_LBB71_5
{
        r3 = gNumAlignedFree;
        r3 = r3 >> 2;
        r4 = heap32[(r3)];
        r4 = (r4 + 1)|0;
        r1 = r1 >> 2;
        heap32[(r3)] = r4;
        r1 = heap32[(r1+-1)];
        heap32[(g0)] = r1;
        free(i7);
}
}
        r1 = 0;
_8: while(true){
        r3 = r1 << 2;
        r3 = (r0 + r3)|0;
        r3 = r3 >> 2;
        r3 = heap32[(r3+22)];
if(!(r3 ==0)) //_LBB71_8
{
        r4 = gNumAlignedFree;
        r4 = r4 >> 2;
        r5 = heap32[(r4)];
        r5 = (r5 + 1)|0;
        r3 = r3 >> 2;
        heap32[(r4)] = r5;
        r3 = heap32[(r3+-1)];
        heap32[(g0)] = r3;
        free(i7);
}
        r1 = (r1 + -1)|0;
        if(r1 !=-3) //_LBB71_6
{
continue _8;
}
else{
break _8;
}
}
        r1 = heap32[(r2+15)];
if(!(r1 ==0)) //_LBB71_11
{
        r3 = gNumAlignedFree;
        r3 = r3 >> 2;
        r4 = heap32[(r3)];
        r4 = (r4 + 1)|0;
        r1 = r1 >> 2;
        heap32[(r3)] = r4;
        r1 = heap32[(r1+-1)];
        heap32[(g0)] = r1;
        free(i7);
}
        r0 = heapU8[r0+100];
if(!(r0 ==0)) //_LBB71_14
{
        r0 = heap32[(r2+23)];
        r1 = r0 >> 2;
        r1 = heap32[(r1)];
        r1 = r1 >> 2;
        r1 = heap32[(r1)];
        heap32[(g0)] = r0;
        __FUNCTION_TABLE__[(r1)>>2](i7);
        r0 = heap32[(r2+23)];
if(!(r0 ==0)) //_LBB71_14
{
        r1 = gNumAlignedFree;
        r1 = r1 >> 2;
        r3 = heap32[(r1)];
        r3 = (r3 + 1)|0;
        r0 = r0 >> 2;
        heap32[(r1)] = r3;
        r0 = heap32[(r0+-1)];
        heap32[(g0)] = r0;
        free(i7);
}
}
        r0 = gNumAlignedFree;
        r0 = r0 >> 2;
        r1 = heap32[(r0)];
        r1 = (r1 + 1)|0;
        heap32[(r0)] = r1;
        r0 = heap32[(r2+-1)];
        heap32[(g0)] = r0;
        free(i7);
        return;
}

function _ZN12btAxisSweep3D0Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV12btAxisSweep3;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        heap32[(g0)] = r0;
        _ZN20btAxisSweep3InternalItED2Ev(i7);
if(!(r0 ==0)) //_LBB72_2
{
        r0 = gNumAlignedFree;
        r0 = r0 >> 2;
        r1 = heap32[(r0)];
        r1 = (r1 + 1)|0;
        heap32[(r0)] = r1;
        r0 = heap32[(r2+-1)];
        heap32[(g0)] = r0;
        free(i7);
}
        return;
}

function _ZN12btAxisSweep3D1Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV12btAxisSweep3;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        heap32[(g0)] = r0;
        _ZN20btAxisSweep3InternalItED2Ev(i7);
        return;
}

function _ZN20btCollisionAlgorithmD1Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV20btCollisionAlgorithm;
        r0 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r0)] = r1;
        return;
}

function _ZN20btCollisionAlgorithmD0Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV20btCollisionAlgorithm;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        heap32[(g0)] = r0;
        _ZdlPv(i7);
        return;
}

function _ZN6btDbvt8ICollide7ProcessEPK10btDbvtNodeS3_(sp)
{
        var i7;
        var fp = sp>>2;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        return;
}

function _ZN6btDbvt8ICollide7ProcessEPK10btDbvtNodef(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = r0 >> 2;
        r1 = heap32[(r1)];
        r1 = r1 >> 2;
        r1 = heap32[(r1+3)];
        r2 = heap32[(fp+1)];
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r2;
        __FUNCTION_TABLE__[(r1)>>2](i7);
        return;
}

function _ZN6btDbvt8ICollide7DescentEPK10btDbvtNode(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = 1;
        r_g0 = r0;
        return;
}

function _ZN6btDbvt8ICollide9AllLeavesEPK10btDbvtNode(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = 1;
        r_g0 = r0;
        return;
}

function _ZL10insertleafP6btDbvtP10btDbvtNodeS2_(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var f0;
        var f1;
        var f2;
        var f3;
        var f4;
        var f5;
        var f6;
        var f7;
        var f8;
        var f9;
        var f10;
        var f11;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = heap32[(fp+2)];
        r0 = r0 >> 2;
        r2 = heap32[(r0)];
        if(r2 !=0) //_LBB80_2
{
        r2 = heap32[(fp+1)];
        r3 = r2 >> 2;
        r3 = heap32[(r3+10)];
        if(r3 !=0) //_LBB80_4
{
        r4 = r1 >> 2;
        f0 = heapFloat[(r4+2)];
        f1 = heapFloat[(r4+6)];
        f2 = heapFloat[(r4+1)];
        f3 = heapFloat[(r4+5)];
        f4 = heapFloat[(r4)];
        f5 = heapFloat[(r4+4)];
        f0 = f0+f1;
        f1 = f2+f3;
        f2 = f4+f5;
_5: while(true){
        r4 = r2 >> 2;
        r4 = heap32[(r4+9)];
        r4 = r4 >> 2;
        f3 = heapFloat[(r4)];
        f4 = heapFloat[(r4+4)];
        f5 = heapFloat[(r4+2)];
        f6 = heapFloat[(r4+6)];
        f7 = heapFloat[(r4+1)];
        f8 = heapFloat[(r4+5)];
        f3 = f3+f4;
        f4 = f5+f6;
        f5 = f7+f8;
        f3 = f2-f3;
        f4 = f0-f4;
        f5 = f1-f5;
        f6 =                         0;
        if(f3 <f6) //_LBB80_7
{
        f3 = -f3;
}
        if(f5 <f6) //_LBB80_10
{
        f5 = -f5;
}
        f3 = f3+f5;
        if(f4 <f6) //_LBB80_13
{
        f4 = -f4;
}
        r3 = r3 >> 2;
        f5 = heapFloat[(r3)];
        f7 = heapFloat[(r3+4)];
        f8 = heapFloat[(r3+2)];
        f9 = heapFloat[(r3+6)];
        f10 = heapFloat[(r3+1)];
        f11 = heapFloat[(r3+5)];
        f5 = f5+f7;
        f7 = f8+f9;
        f8 = f10+f11;
        f5 = f2-f5;
        f3 = f3+f4;
        f4 = f0-f7;
        f7 = f1-f8;
        if(f5 <f6) //_LBB80_16
{
        f5 = -f5;
}
        if(f7 <f6) //_LBB80_19
{
        f7 = -f7;
}
        f5 = f5+f7;
        if(f4 <f6) //_LBB80_22
{
        f4 = -f4;
}
        f4 = f5+f4;
        r3 = f3 >= f4;
        r3 = r3 & 1;
        r3 = r3 << 2;
        r2 = (r2 + r3)|0;
        r2 = r2 >> 2;
        r2 = heap32[(r2+9)];
        r3 = r2 >> 2;
        r3 = heap32[(r3+10)];
if(!(r3 !=0)) //_LBB80_5
{
break _5;
}
}
}
        r3 = r2 >> 2;
        r4 = heap32[(r0+1)];
        r5 = heap32[(r3+8)];
        if(r4 ==0) //_LBB80_26
{
        r4 = gNumAlignedAllocs;
        r4 = r4 >> 2;
        r6 = heap32[(r4)];
        r6 = (r6 + 1)|0;
        heap32[(r4)] = r6;
        heap32[(g0)] = 63;
        malloc(i7);
        r4 = r_g0;
        if(r4 !=0) //_LBB80_28
{
        r6 = 0;
        r7 = (r4 + 4)|0;
        r6 = (r6 - r7)|0;
        r6 = r6 & 15;
        r6 = (r4 + r6)|0;
        r7 = (r6 + 4)|0;
        r6 = r6 >> 2;
        heap32[(r6)] = r4;
        r4 = r7;
}
}
else{
        heap32[(r0+1)] = 0;
}
        r6 = r4 >> 2;
        heap32[(r6+8)] = r5;
        heap32[(r6+9)] = 0;
        heap32[(r6+10)] = 0;
        r7 = r1 >> 2;
        f0 = heapFloat[(r7)];
        f1 = heapFloat[(r3)];
        f0 = f0 < f1 ? f0 : f1;
        heapFloat[(r6)] = f0;
        f0 = heapFloat[(r7+4)];
        f1 = heapFloat[(r3+4)];
        f0 = f0 > f1 ? f0 : f1;
        heapFloat[(r6+4)] = f0;
        f0 = heapFloat[(r7+1)];
        f1 = heapFloat[(r3+1)];
        f0 = f0 < f1 ? f0 : f1;
        heapFloat[(r6+1)] = f0;
        f0 = heapFloat[(r7+5)];
        f1 = heapFloat[(r3+5)];
        f0 = f0 > f1 ? f0 : f1;
        heapFloat[(r6+5)] = f0;
        f0 = heapFloat[(r7+2)];
        f1 = heapFloat[(r3+2)];
        f0 = f0 < f1 ? f0 : f1;
        heapFloat[(r6+2)] = f0;
        f0 = heapFloat[(r7+6)];
        f1 = heapFloat[(r3+6)];
        f0 = f0 > f1 ? f0 : f1;
        heapFloat[(r6+6)] = f0;
_31: do {
        if(r5 ==0) //_LBB80_38
{
        heap32[(r6+9)] = r2;
        heap32[(r3+8)] = r4;
        heap32[(r6+10)] = r1;
        heap32[(r7+8)] = r4;
        heap32[(r0)] = r4;
}
else{
        r0 = heap32[(r3+8)];
        r0 = r0 >> 2;
        r0 = heap32[(r0+10)];
        r0 = r0 == r2;
        r0 = r0 & 1;
        r0 = r0 << 2;
        r0 = (r5 + r0)|0;
        r0 = r0 >> 2;
        heap32[(r0+9)] = r4;
        heap32[(r6+9)] = r2;
        heap32[(r3+8)] = r4;
        heap32[(r6+10)] = r1;
        heap32[(r7+8)] = r4;
        f0 = heapFloat[(r6)];
_34: while(true){
        r0 = r5;
        r1 = r0 >> 2;
        f1 = heapFloat[(r1)];
if(!(f1 >f0)) //_LBB80_37
{
        r2 = r4 >> 2;
        f0 = heapFloat[(r1+1)];
        f1 = heapFloat[(r2+1)];
if(!(f0 >f1)) //_LBB80_37
{
        f0 = heapFloat[(r1+2)];
        f1 = heapFloat[(r2+2)];
if(!(f0 >f1)) //_LBB80_37
{
        f0 = heapFloat[(r1+4)];
        f1 = heapFloat[(r2+4)];
if(!(f0 <f1)) //_LBB80_37
{
        f0 = heapFloat[(r1+5)];
        f1 = heapFloat[(r2+5)];
if(!(f0 <f1)) //_LBB80_37
{
        f0 = heapFloat[(r1+6)];
        f1 = heapFloat[(r2+6)];
        if(f0 >=f1) //_LBB80_39
{
break _31;
}
}
}
}
}
}
        r2 = heap32[(r1+10)];
        r3 = heap32[(r1+9)];
        r2 = r2 >> 2;
        r3 = r3 >> 2;
        f0 = heapFloat[(r3)];
        f1 = heapFloat[(r2)];
        f0 = f0 < f1 ? f0 : f1;
        heapFloat[(r1)] = f0;
        f1 = heapFloat[(r3+4)];
        f2 = heapFloat[(r2+4)];
        f1 = f1 > f2 ? f1 : f2;
        heapFloat[(r1+4)] = f1;
        f1 = heapFloat[(r3+1)];
        f2 = heapFloat[(r2+1)];
        f1 = f1 < f2 ? f1 : f2;
        heapFloat[(r1+1)] = f1;
        f1 = heapFloat[(r3+5)];
        f2 = heapFloat[(r2+5)];
        f1 = f1 > f2 ? f1 : f2;
        heapFloat[(r1+5)] = f1;
        f1 = heapFloat[(r3+2)];
        f2 = heapFloat[(r2+2)];
        f1 = f1 < f2 ? f1 : f2;
        heapFloat[(r1+2)] = f1;
        f1 = heapFloat[(r3+6)];
        f2 = heapFloat[(r2+6)];
        f1 = f1 > f2 ? f1 : f2;
        heapFloat[(r1+6)] = f1;
        r5 = heap32[(r1+8)];
        r4 = r0;
        if(r5 ==0) //_LBB80_39
{
break _31;
}
}
}
} while(0);
        return;
}
else{
        r2 = r1 >> 2;
        heap32[(r0)] = r1;
        heap32[(r2+8)] = 0;
        return;
}
}

function _ZL10removeleafP6btDbvtP10btDbvtNode(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var f0;
        var f1;
        var f2;
        var f3;
        var f4;
        var f5;
        var f6;
        var f7;
        var f8;
        var f9;
        var f10;
        var f11;
        var f12;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = heap32[(fp+1)];
        r0 = r0 >> 2;
        r2 = heap32[(r0)];
        if(r2 !=r1) //_LBB81_2
{
        r2 = r1 >> 2;
        r2 = heap32[(r2+8)];
        r3 = r2 >> 2;
        r4 = heap32[(r3+10)];
        r1 = r4 != r1;
        r1 = r1 & 1;
        r1 = r1 << 2;
        r1 = (r2 + r1)|0;
        r1 = r1 >> 2;
        r3 = heap32[(r3+8)];
        r1 = heap32[(r1+9)];
_3: do {
        if(r3 ==0) //_LBB81_11
{
        r3 = r1 >> 2;
        heap32[(r0)] = r1;
        heap32[(r3+8)] = 0;
        r3 = heap32[(r0+1)];
if(!(r3 ==0)) //_LBB81_13
{
        r1 = gNumAlignedFree;
        r1 = r1 >> 2;
        r4 = heap32[(r1)];
        r4 = (r4 + 1)|0;
        r3 = r3 >> 2;
        heap32[(r1)] = r4;
        r3 = heap32[(r3+-1)];
        heap32[(g0)] = r3;
        free(i7);
}
        heap32[(r0+1)] = r2;
        r3 = heap32[(r0)];
}
else{
        r4 = r3 >> 2;
        r4 = heap32[(r4+10)];
        r4 = r4 == r2;
        r4 = r4 & 1;
        r4 = r4 << 2;
        r4 = (r3 + r4)|0;
        r4 = r4 >> 2;
        r5 = r1 >> 2;
        heap32[(r4+9)] = r1;
        heap32[(r5+8)] = r3;
        r1 = heap32[(r0+1)];
if(!(r1 ==0)) //_LBB81_5
{
        r4 = gNumAlignedFree;
        r4 = r4 >> 2;
        r5 = heap32[(r4)];
        r5 = (r5 + 1)|0;
        r1 = r1 >> 2;
        heap32[(r4)] = r5;
        r1 = heap32[(r1+-1)];
        heap32[(g0)] = r1;
        free(i7);
}
        heap32[(r0+1)] = r2;
_12: while(true){
        if(r3 !=0) //_LBB81_6
{
        r1 = r3 >> 2;
        r2 = heap32[(r1+10)];
        r4 = heap32[(r1+9)];
        r2 = r2 >> 2;
        r4 = r4 >> 2;
        f0 = heapFloat[(r1+6)];
        f1 = heapFloat[(r1)];
        f2 = heapFloat[(r1+1)];
        f3 = heapFloat[(r1+2)];
        f4 = heapFloat[(r1+4)];
        f5 = heapFloat[(r1+5)];
        f6 = heapFloat[(r4)];
        f7 = heapFloat[(r2)];
        f6 = f6 < f7 ? f6 : f7;
        heapFloat[(r1)] = f6;
        f7 = heapFloat[(r4+4)];
        f8 = heapFloat[(r2+4)];
        f7 = f7 > f8 ? f7 : f8;
        heapFloat[(r1+4)] = f7;
        f8 = heapFloat[(r4+1)];
        f9 = heapFloat[(r2+1)];
        f8 = f8 < f9 ? f8 : f9;
        heapFloat[(r1+1)] = f8;
        f9 = heapFloat[(r4+5)];
        f10 = heapFloat[(r2+5)];
        f9 = f9 > f10 ? f9 : f10;
        heapFloat[(r1+5)] = f9;
        f10 = heapFloat[(r4+2)];
        f11 = heapFloat[(r2+2)];
        f10 = f10 < f11 ? f10 : f11;
        heapFloat[(r1+2)] = f10;
        f11 = heapFloat[(r4+6)];
        f12 = heapFloat[(r2+6)];
        f11 = f11 > f12 ? f11 : f12;
        heapFloat[(r1+6)] = f11;
if(!(f0 !=f11)) //_LBB81_8
{
        r2 = f1 == f6;
        r4 = f2 == f8;
        r2 = r2 & r4;
        r4 = f3 == f10;
        r2 = r2 & r4;
        r4 = f4 == f7;
        r2 = r2 & r4;
        r4 = f5 == f9;
        r2 = r2 & r4;
        if(r2 != 0) //_LBB81_14
{
break _3;
}
}
        r3 = heap32[(r1+8)];
}
else{
break _12;
}
}
        r0 = heap32[(r0)];
        r_g0 = r0;
        return;
}
} while(0);
        r_g0 = r3;
        return;
}
else{
        heap32[(r0)] = 0;
        r0 = 0;
        r_g0 = r0;
        return;
}
}

function _ZN6btDbvt6updateEP10btDbvtNodeR12btDbvtAabbMmRK9btVector3f(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var f0;
        var f1;
        var f2;
        var f3;
        var f4;
        var f5;
        var f6;
        var f7;
var __label__ = 0;
        i7 = sp + -16;var g0 = i7>>2; // save stack
        r0 = heap32[(fp+2)];
        r0 = r0 >> 2;
        r1 = heap32[(fp+1)];
        f0 = heapFloat[(r0)];
        r2 = heap32[(fp)];
        r3 = heap32[(fp+3)];
        f1 = heapFloat[(fp+4)];
        r4 = r1 >> 2;
        f2 = heapFloat[(r4)];
        if(f2 <=f0) //_LBB82_2
{
        f2 = heapFloat[(r0+1)];
        f3 = heapFloat[(r4+1)];
        if(f3 <=f2) //_LBB82_4
{
        f3 = heapFloat[(r4+2)];
        f4 = heapFloat[(r0+2)];
if(!(f3 >f4)) //_LBB82_3
{
        f3 = heapFloat[(r4+4)];
        f4 = heapFloat[(r0+4)];
if(!(f3 <f4)) //_LBB82_3
{
        f3 = heapFloat[(r4+5)];
        f4 = heapFloat[(r0+5)];
if(!(f3 <f4)) //_LBB82_3
{
        f3 = heapFloat[(r4+6)];
        f4 = heapFloat[(r0+6)];
if(!(f3 <f4)) //_LBB82_3
{
        r0 = 0;
        r_g0 = r0;
        return;
}
}
}
}
}
}
else{
        f2 = heapFloat[(r0+1)];
}
        f0 = f0-f1;
        f2 = f2-f1;
        heapFloat[(r0)] = f0;
        heapFloat[(r0+1)] = f2;
        f3 = heapFloat[(r0+2)];
        f3 = f3-f1;
        heapFloat[(r0+2)] = f3;
        f4 = heapFloat[(r0+4)];
        f4 = f4+f1;
        heapFloat[(r0+4)] = f4;
        f5 = heapFloat[(r0+5)];
        f5 = f5+f1;
        heapFloat[(r0+5)] = f5;
        f6 = heapFloat[(r0+6)];
        f1 = f6+f1;
        r3 = r3 >> 2;
        heapFloat[(r0+6)] = f1;
        f6 = heapFloat[(r3)];
        f7 =                         0;
        if(f6 <=f7) //_LBB82_10
{
        f0 = f0+f6;
        heapFloat[(r0)] = f0;
}
else{
        f0 = f4+f6;
        heapFloat[(r0+4)] = f0;
}
        f0 = heapFloat[(r3+1)];
        if(f0 <=f7) //_LBB82_13
{
        f0 = f2+f0;
        heapFloat[(r0+1)] = f0;
}
else{
        f0 = f5+f0;
        heapFloat[(r0+5)] = f0;
}
        f0 = heapFloat[(r3+2)];
        if(f0 <=f7) //_LBB82_16
{
        f0 = f3+f0;
        heapFloat[(r0+2)] = f0;
}
else{
        f0 = f1+f0;
        heapFloat[(r0+6)] = f0;
}
        heap32[(g0)] = r2;
        heap32[(g0+1)] = r1;
        _ZL10removeleafP6btDbvtP10btDbvtNode(i7);
        r3 = r_g0;
_22: do {
        if(r3 !=0) //_LBB82_19
{
        r5 = r2 >> 2;
        r6 = heap32[(r5+2)];
        if(r6 <0) //_LBB82_24
{
        r5 = heap32[(r5)];
}
else{
        r7 = -1;
_27: while(true){
        r5 = r3;
        r7 = (r7 + 1)|0;
        if(r6 >r7) //_LBB82_23
{
        r3 = r5 >> 2;
        r3 = heap32[(r3+8)];
        if(r3 ==0) //_LBB82_22
{
break _22;
}
else{
continue _27;
}
}
else{
break _22;
}
}
}
}
else{
        r5 = 0;
}
} while(0);
        heap32[(r4)] = heap32[(r0)];
        heap32[(r4+1)] = heap32[(r0+1)];
        heap32[(r4+2)] = heap32[(r0+2)];
        heap32[(r4+3)] = heap32[(r0+3)];
        heap32[(r4+4)] = heap32[(r0+4)];
        heap32[(r4+5)] = heap32[(r0+5)];
        heap32[(r4+6)] = heap32[(r0+6)];
        heap32[(r4+7)] = heap32[(r0+7)];
        heap32[(g0)] = r2;
        heap32[(g0+1)] = r5;
        heap32[(g0+2)] = r1;
        _ZL10insertleafP6btDbvtP10btDbvtNodeS2_(i7);
        r0 = 1;
        r_g0 = r0;
        return;
}

function _ZL17recursedeletenodeP6btDbvtP10btDbvtNode(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp+1)];
        r1 = heap32[(fp)];
        r2 = r0 >> 2;
        r3 = heap32[(r2+10)];
if(!(r3 ==0)) //_LBB83_2
{
        r3 = heap32[(r2+9)];
        heap32[(g0)] = r1;
        heap32[(g0+1)] = r3;
        _ZL17recursedeletenodeP6btDbvtP10btDbvtNode(i7);
        r2 = heap32[(r2+10)];
        heap32[(g0)] = r1;
        heap32[(g0+1)] = r2;
        _ZL17recursedeletenodeP6btDbvtP10btDbvtNode(i7);
}
        r1 = r1 >> 2;
        r2 = heap32[(r1)];
if(!(r2 !=r0)) //_LBB83_4
{
        heap32[(r1)] = 0;
}
        r2 = heap32[(r1+1)];
if(!(r2 ==0)) //_LBB83_6
{
        r3 = gNumAlignedFree;
        r3 = r3 >> 2;
        r4 = heap32[(r3)];
        r4 = (r4 + 1)|0;
        r2 = r2 >> 2;
        heap32[(r3)] = r4;
        r2 = heap32[(r2+-1)];
        heap32[(g0)] = r2;
        free(i7);
}
        heap32[(r1+1)] = r0;
        return;
}

function _ZN6btDbvt19optimizeIncrementalEi(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var r12;
        var f0;
        var f1;
        var f2;
        var f3;
        var f4;
        var f5;
        var f6;
        var f7;
var __label__ = 0;
        i7 = sp + -16;var g0 = i7>>2; // save stack
        r0 = heap32[(fp+1)];
        r1 = heap32[(fp)];
        if(r0 <0) //_LBB84_2
{
        r0 = r1 >> 2;
        r0 = heap32[(r0+3)];
}
        r2 = r1 >> 2;
        r3 = heap32[(r2)];
_4: do {
if(!(r3 ==0)) //_LBB84_20
{
if(!(r0 <1)) //_LBB84_20
{
_6: while(true){
        r3 = 0;
        r4 = r1;
_8: while(true){
        r4 = r4 >> 2;
        r4 = heap32[(r4)];
        r5 = r4 >> 2;
        r6 = heap32[(r5+10)];
        if(r6 !=0) //_LBB84_6
{
        r6 = heap32[(r5+8)];
        if(uint(r6) >uint(r4)) //_LBB84_8
{
        r7 = r6 >> 2;
        r8 = heap32[(r7+10)];
        r8 = r8 == r4;
        r8 = r8 & 1;
        r9 = (r6 + 36)|0;
        r10 = r8 << 2;
        r11 = (r9 + r10)|0;
        r11 = r11 >> 2;
        r11 = heap32[(r11)];
        if(r11 ==r4) //_LBB84_10
{
        r8 = r8 ^ 1;
        r8 = r8 << 2;
        r9 = (r9 + r8)|0;
        r9 = r9 >> 2;
        r9 = heap32[(r9)];
        r11 = heap32[(r7+8)];
        if(r11 ==0) //_LBB84_12
{
        heap32[(r2)] = r4;
}
else{
        r12 = r11 >> 2;
        r12 = heap32[(r12+10)];
        r12 = r12 == r6;
        r12 = r12 & 1;
        r12 = r12 << 2;
        r12 = (r11 + r12)|0;
        r12 = r12 >> 2;
        heap32[(r12+9)] = r4;
}
        r12 = r9 >> 2;
        heap32[(r12+8)] = r4;
        heap32[(r7+8)] = r4;
        heap32[(r5+8)] = r11;
        r11 = heap32[(r5+9)];
        heap32[(r7+9)] = r11;
        r11 = heap32[(r5+10)];
        heap32[(r7+10)] = r11;
        r11 = heap32[(r5+9)];
        r11 = r11 >> 2;
        heap32[(r11+8)] = r6;
        r11 = heap32[(r5+10)];
        r4 = (r4 + 36)|0;
        r10 = (r4 + r10)|0;
        r11 = r11 >> 2;
        r4 = (r4 + r8)|0;
        r8 = r10 >> 2;
        heap32[(r11+8)] = r6;
        r4 = r4 >> 2;
        heap32[(r8)] = r6;
        heap32[(r4)] = r9;
        f0 = heapFloat[(r7+7)];
        f1 = heapFloat[(r7+6)];
        f2 = heapFloat[(r7+5)];
        f3 = heapFloat[(r7+4)];
        f4 = heapFloat[(r7+3)];
        f5 = heapFloat[(r7+2)];
        f6 = heapFloat[(r7+1)];
        f7 = heapFloat[(r7)];
        heap32[(r7)] = heap32[(r5)];
        heap32[(r7+1)] = heap32[(r5+1)];
        heap32[(r7+2)] = heap32[(r5+2)];
        heap32[(r7+3)] = heap32[(r5+3)];
        heap32[(r7+4)] = heap32[(r5+4)];
        heap32[(r7+5)] = heap32[(r5+5)];
        heap32[(r7+6)] = heap32[(r5+6)];
        heap32[(r7+7)] = heap32[(r5+7)];
        heapFloat[(r5)] = f7;
        heapFloat[(r5+1)] = f6;
        heapFloat[(r5+2)] = f5;
        heapFloat[(r5+3)] = f4;
        heapFloat[(r5+4)] = f3;
        heapFloat[(r5+5)] = f2;
        heapFloat[(r5+6)] = f1;
        heapFloat[(r5+7)] = f0;
}
else{
break _6;
}
}
else{
        r6 = r4;
}
        r4 = heap32[(r2+4)];
        r4 = r4 >>> r3;
        r4 = r4 & 1;
        r4 = r4 << 2;
        r4 = (r6 + r4)|0;
        r3 = (r3 + 1)|0;
        r4 = (r4 + 36)|0;
        r3 = r3 & 31;
}
else{
break _8;
}
}
        heap32[(g0)] = r1;
        heap32[(g0+1)] = r4;
        _ZL10removeleafP6btDbvtP10btDbvtNode(i7);
        r3 = r_g0;
        if(r3 !=0) //_LBB84_18
{
        r3 = heap32[(r2)];
}
else{
        r3 = 0;
}
        heap32[(g0)] = r1;
        heap32[(g0+1)] = r3;
        heap32[(g0+2)] = r4;
        _ZL10insertleafP6btDbvtP10btDbvtNodeS2_(i7);
        r3 = heap32[(r2+4)];
        r0 = (r0 + -1)|0;
        r3 = (r3 + 1)|0;
        heap32[(r2+4)] = r3;
if(!(r0 !=0)) //_LBB84_5
{
break _4;
}
}
        r4 = _2E_str22;
        r0 = _2E_str1118;
        heap32[(g0)] = r4;
        heap32[(g0+1)] = r0;
        heap32[(g0+2)] = 379;
        _assert(i7);
}
}
} while(0);
        return;
}

function _ZN18btDbvtTreeColliderD1Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV18btDbvtTreeCollider;
        r0 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r0)] = r1;
        return;
}

function _ZN18btDbvtTreeColliderD0Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV18btDbvtTreeCollider;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        heap32[(g0)] = r0;
        _ZdlPv(i7);
        return;
}

function _ZN18btDbvtTreeCollider7ProcessEPK10btDbvtNodeS2_(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
var __label__ = 0;
        i7 = sp + -16;var g0 = i7>>2; // save stack
        r0 = heap32[(fp+1)];
        r1 = heap32[(fp+2)];
if(!(r0 ==r1)) //_LBB87_2
{
        r2 = heap32[(fp)];
        r2 = r2 >> 2;
        r3 = heap32[(r2+1)];
        r3 = r3 >> 2;
        r3 = heap32[(r3+24)];
        r4 = r3 >> 2;
        r4 = heap32[(r4)];
        r4 = r4 >> 2;
        r1 = r1 >> 2;
        r0 = r0 >> 2;
        r4 = heap32[(r4+2)];
        r1 = heap32[(r1+9)];
        r0 = heap32[(r0+9)];
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r0;
        heap32[(g0+2)] = r1;
        __FUNCTION_TABLE__[(r4)>>2](i7);
        r0 = heap32[(r2+1)];
        r0 = r0 >> 2;
        r1 = heap32[(r0+30)];
        r1 = (r1 + 1)|0;
        heap32[(r0+30)] = r1;
}
        return;
}

function _ZN18btDbvtTreeCollider7ProcessEPK10btDbvtNode(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
var __label__ = 0;
        i7 = sp + -16;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = r0 >> 2;
        r2 = heap32[(r1)];
        r1 = heap32[(r1+2)];
        r2 = r2 >> 2;
        r1 = r1 >> 2;
        r2 = heap32[(r2+2)];
        r1 = heap32[(r1+12)];
        r3 = heap32[(fp+1)];
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r3;
        heap32[(g0+2)] = r1;
        __FUNCTION_TABLE__[(r2)>>2](i7);
        return;
}

function _ZNK16btDbvtBroadphase7getAabbEP17btBroadphaseProxyR9btVector3S3_(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp+1)];
        r1 = heap32[(fp+2)];
        r0 = r0 >> 2;
        r1 = r1 >> 2;
        heap32[(r1)] = heap32[(r0+4)];
        heap32[(r1+1)] = heap32[(r0+5)];
        r2 = heap32[(fp+3)];
        heap32[(r1+2)] = heap32[(r0+6)];
        r2 = r2 >> 2;
        heap32[(r1+3)] = heap32[(r0+7)];
        heap32[(r2)] = heap32[(r0+8)];
        heap32[(r2+1)] = heap32[(r0+9)];
        heap32[(r2+2)] = heap32[(r0+10)];
        heap32[(r2+3)] = heap32[(r0+11)];
        return;
}

function _ZN19BroadphaseRayTesterD1Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV19BroadphaseRayTester;
        r0 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r0)] = r1;
        return;
}

function _ZN19BroadphaseRayTesterD0Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV19BroadphaseRayTester;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        heap32[(g0)] = r0;
        _ZdlPv(i7);
        return;
}

function _ZN19BroadphaseRayTester7ProcessEPK10btDbvtNode(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = r0 >> 2;
        r0 = heap32[(r0+1)];
        r1 = r0 >> 2;
        r2 = heap32[(fp+1)];
        r1 = heap32[(r1)];
        r1 = r1 >> 2;
        r2 = r2 >> 2;
        r1 = heap32[(r1+2)];
        r2 = heap32[(r2+9)];
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r2;
        __FUNCTION_TABLE__[(r1)>>2](i7);
        return;
}

function _ZN20BroadphaseAabbTesterD1Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV20BroadphaseAabbTester;
        r0 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r0)] = r1;
        return;
}

function _ZN20BroadphaseAabbTesterD0Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV20BroadphaseAabbTester;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        heap32[(g0)] = r0;
        _ZdlPv(i7);
        return;
}

function _ZN20BroadphaseAabbTester7ProcessEPK10btDbvtNode(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = r0 >> 2;
        r0 = heap32[(r0+1)];
        r1 = r0 >> 2;
        r2 = heap32[(fp+1)];
        r1 = heap32[(r1)];
        r1 = r1 >> 2;
        r2 = r2 >> 2;
        r1 = heap32[(r1+2)];
        r2 = heap32[(r2+9)];
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r2;
        __FUNCTION_TABLE__[(r1)>>2](i7);
        return;
}

function _ZN16btDbvtBroadphase23getOverlappingPairCacheEv(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = r0 >> 2;
        r0 = heap32[(r0+24)];
        r_g0 = r0;
        return;
}

function _ZNK16btDbvtBroadphase23getOverlappingPairCacheEv(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = r0 >> 2;
        r0 = heap32[(r0+24)];
        r_g0 = r0;
        return;
}

function _ZNK16btDbvtBroadphase17getBroadphaseAabbER9btVector3S1_(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var f0;
        var f1;
        var f2;
        var f3;
        var f4;
        var f5;
        var f6;
        var f7;
        var f8;
        var f9;
        var f10;
        var f11;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = r0 >> 2;
        r1 = heap32[(r0+1)];
        r2 = heap32[(fp+1)];
        r3 = heap32[(fp+2)];
        r0 = heap32[(r0+11)];
        if(r1 ==0) //_LBB98_4
{
        if(r0 !=0) //_LBB98_6
{
        r0 = r0 >> 2;
        f0 = heapFloat[(r0)];
        f2 = heapFloat[(r0+1)];
        f4 = heapFloat[(r0+2)];
        f6 = heapFloat[(r0+3)];
        f1 = heapFloat[(r0+4)];
        f3 = heapFloat[(r0+5)];
        f5 = heapFloat[(r0+6)];
        f7 = heapFloat[(r0+7)];
}
else{
        f6 =                         0;
        f4 = f6;
        f2 = f6;
        f0 = f6;
        f1 = f6;
        f3 = f6;
        f5 = f6;
        f7 = f6;
}
}
else{
        if(r0 ==0) //_LBB98_3
{
        r0 = r1 >> 2;
        f0 = heapFloat[(r0)];
        f2 = heapFloat[(r0+1)];
        f4 = heapFloat[(r0+2)];
        f6 = heapFloat[(r0+3)];
        f1 = heapFloat[(r0+4)];
        f3 = heapFloat[(r0+5)];
        f5 = heapFloat[(r0+6)];
        f7 = heapFloat[(r0+7)];
}
else{
        r1 = r1 >> 2;
        r0 = r0 >> 2;
        f0 = heapFloat[(r1)];
        f1 = heapFloat[(r0)];
        f2 = heapFloat[(r1+4)];
        f3 = heapFloat[(r0+4)];
        f4 = heapFloat[(r1+1)];
        f5 = heapFloat[(r0+1)];
        f6 = heapFloat[(r1+5)];
        f7 = heapFloat[(r0+5)];
        f8 = heapFloat[(r1+2)];
        f9 = heapFloat[(r0+2)];
        f10 = heapFloat[(r1+6)];
        f11 = heapFloat[(r0+6)];
        f0 = f0 < f1 ? f0 : f1;
        f1 = f2 > f3 ? f2 : f3;
        f2 = f4 < f5 ? f4 : f5;
        f3 = f6 > f7 ? f6 : f7;
        f4 = f8 < f9 ? f8 : f9;
        f5 = f10 > f11 ? f10 : f11;
}
}
        r0 = r2 >> 2;
        heapFloat[(r0)] = f0;
        heapFloat[(r0+1)] = f2;
        heapFloat[(r0+2)] = f4;
        r1 = r3 >> 2;
        heapFloat[(r0+3)] = f6;
        heapFloat[(r1)] = f1;
        heapFloat[(r1+1)] = f3;
        heapFloat[(r1+2)] = f5;
        heapFloat[(r1+3)] = f7;
        return;
}

function _ZN16btDbvtBroadphase10printStatsEv(sp)
{
        var i7;
        var fp = sp>>2;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        return;
}

function _ZN16btDbvtBroadphase9resetPoolEP12btDispatcher(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = r0 >> 2;
        r2 = 0;
        r3 = heap32[(r1+14)];
        r4 = heap32[(r1+4)];
        r3 = (r2 - r3)|0;
if(!(r4 !=r3)) //_LBB100_18
{
        r3 = heap32[(r1+1)];
if(!(r3 ==0)) //_LBB100_3
{
        r4 = (r0 + 4)|0;
        heap32[(g0)] = r4;
        heap32[(g0+1)] = r3;
        _ZL17recursedeletenodeP6btDbvtP10btDbvtNode(i7);
}
        r3 = heap32[(r1+2)];
if(!(r3 ==0)) //_LBB100_5
{
        r4 = gNumAlignedFree;
        r4 = r4 >> 2;
        r5 = heap32[(r4)];
        r5 = (r5 + 1)|0;
        r3 = r3 >> 2;
        heap32[(r4)] = r5;
        r3 = heap32[(r3+-1)];
        heap32[(g0)] = r3;
        free(i7);
}
        heap32[(r1+2)] = 0;
        heap32[(r1+3)] = -1;
        r3 = heap32[(r1+9)];
if(!(r3 ==0)) //_LBB100_9
{
        r4 = heapU8[r0+40];
if(!(r4 ==0)) //_LBB100_8
{
        r4 = gNumAlignedFree;
        r4 = r4 >> 2;
        r5 = heap32[(r4)];
        r5 = (r5 + 1)|0;
        r3 = r3 >> 2;
        heap32[(r4)] = r5;
        r3 = heap32[(r3+-1)];
        heap32[(g0)] = r3;
        free(i7);
}
        heap32[(r1+9)] = 0;
}
        r3 = 1;
        heap8[r0+40] = r3;
        heap32[(r1+9)] = 0;
        heap32[(r1+7)] = 0;
        heap32[(r1+8)] = 0;
        heap32[(r1+5)] = 0;
        r4 = heap32[(r1+11)];
if(!(r4 ==0)) //_LBB100_11
{
        r5 = (r0 + 44)|0;
        heap32[(g0)] = r5;
        heap32[(g0+1)] = r4;
        _ZL17recursedeletenodeP6btDbvtP10btDbvtNode(i7);
}
        r4 = heap32[(r1+12)];
if(!(r4 ==0)) //_LBB100_13
{
        r5 = gNumAlignedFree;
        r5 = r5 >> 2;
        r6 = heap32[(r5)];
        r6 = (r6 + 1)|0;
        r4 = r4 >> 2;
        heap32[(r5)] = r6;
        r4 = heap32[(r4+-1)];
        heap32[(g0)] = r4;
        free(i7);
}
        heap32[(r1+12)] = 0;
        heap32[(r1+13)] = -1;
        r4 = heap32[(r1+19)];
if(!(r4 ==0)) //_LBB100_17
{
        r5 = heapU8[r0+80];
if(!(r5 ==0)) //_LBB100_16
{
        r5 = gNumAlignedFree;
        r5 = r5 >> 2;
        r6 = heap32[(r5)];
        r6 = (r6 + 1)|0;
        r4 = r4 >> 2;
        heap32[(r5)] = r6;
        r4 = heap32[(r4+-1)];
        heap32[(g0)] = r4;
        free(i7);
}
        heap32[(r1+19)] = 0;
}
        heap8[r0+80] = r3;
        heap32[(r1+19)] = 0;
        heap32[(r1+17)] = 0;
        heap32[(r1+18)] = 0;
        heap32[(r1+15)] = 0;
        heap8[r0+153] = r2;
        heap8[r0+154] = r3;
        heap32[(r1+26)] = 0;
        heap32[(r1+31)] = 0;
        heap32[(r1+27)] = 1;
        heap32[(r1+28)] = 0;
        heap32[(r1+29)] = 10;
        heap32[(r1+30)] = 1;
        heap32[(r1+32)] = 0;
        heap32[(r1+33)] = 0;
        heap32[(r1+34)] = 0;
        heap32[(r1+37)] = 0;
        heap32[(r1+35)] = 0;
        heap32[(r1+36)] = 0;
        heap32[(r1+21)] = 0;
        heap32[(r1+22)] = 0;
        heap32[(r1+23)] = 0;
}
        return;
}

function _ZN6btDbvt24collideTTpersistentStackEPK10btDbvtNodeS2_RNS_8ICollideE(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var r12;
        var r13;
        var r14;
        var r15;
        var r16;
        var f0;
        var f1;
var __label__ = 0;
        i7 = sp + -16;var g0 = i7>>2; // save stack
        r0 = heap32[(fp+1)];
_1: do {
if(!(r0 ==0)) //_LBB101_57
{
        r1 = heap32[(fp+2)];
if(!(r1 ==0)) //_LBB101_57
{
        r2 = heap32[(fp)];
        r3 = r2 >> 2;
        r4 = heap32[(r3+6)];
if(!(r4 >127)) //_LBB101_17
{
        r5 = heap32[(r3+7)];
if(!(r5 >127)) //_LBB101_17
{
        r5 = gNumAlignedAllocs;
        r5 = r5 >> 2;
        r6 = heap32[(r5)];
        r6 = (r6 + 1)|0;
        heap32[(r5)] = r6;
        heap32[(g0)] = 1043;
        malloc(i7);
        r5 = r_g0;
        if(r5 !=0) //_LBB101_6
{
        r6 = 0;
        r7 = (r5 + 4)|0;
        r6 = (r6 - r7)|0;
        r6 = r6 & 15;
        r6 = (r5 + r6)|0;
        r7 = (r6 + 4)|0;
        r6 = r6 >> 2;
        heap32[(r6)] = r5;
        r5 = r7;
}
        r6 = (r2 + 32)|0;
        if(r4 <1) //_LBB101_9
{
        r4 = r6 >> 2;
        r8 = heap32[(r4)];
}
else{
        r7 = 0;
_13: while(true){
        r8 = r6 >> 2;
        r8 = heap32[(r8)];
        r9 = r7 << 3;
        r10 = (r8 + r9)|0;
        r10 = r10 >> 2;
        r9 = (r5 + r9)|0;
        r11 = heap32[(r10+1)];
        r10 = heap32[(r10)];
        r9 = r9 >> 2;
        r7 = (r7 + 1)|0;
        heap32[(r9)] = r10;
        heap32[(r9+1)] = r11;
if(!(r4 !=r7)) //_LBB101_10
{
break _13;
}
}
        r6 = (r2 + 32)|0;
}
if(!(r8 ==0)) //_LBB101_16
{
        r4 = heapU8[r2+36];
if(!(r4 ==0)) //_LBB101_15
{
        r4 = gNumAlignedFree;
        r4 = r4 >> 2;
        r7 = heap32[(r4)];
        r7 = (r7 + 1)|0;
        r8 = r8 >> 2;
        heap32[(r4)] = r7;
        r4 = heap32[(r8+-1)];
        heap32[(g0)] = r4;
        free(i7);
}
        r4 = r6 >> 2;
        heap32[(r4)] = 0;
}
        r4 = 1;
        r6 = r6 >> 2;
        heap8[r2+36] = r4;
        heap32[(r6)] = r5;
        heap32[(r3+7)] = 128;
}
}
        r4 = heap32[(fp+3)];
        heap32[(r3+6)] = 128;
        r5 = heap32[(r3+8)];
        r5 = r5 >> 2;
        r6 = 1;
        r7 = 124;
        heap32[(r5)] = r0;
        heap32[(r5+1)] = r1;
_24: while(true){
        r0 = r6;
        r6 = (r0 + -1)|0;
        r1 = heap32[(r3+8)];
        r5 = r6 << 3;
        r5 = (r1 + r5)|0;
        r5 = r5 >> 2;
        r8 = heap32[(r5)];
        r5 = heap32[(r5+1)];
        if(r6 >r7) //_LBB101_20
{
        r7 = heap32[(r3+6)];
        r9 = r7 << 1;
_28: do {
if(!(r7 >r9)) //_LBB101_38
{
if(!(r7 >=r9)) //_LBB101_38
{
        r10 = heap32[(r3+7)];
        if(r10 <r9) //_LBB101_24
{
        if(r9 !=0) //_LBB101_26
{
        r10 = gNumAlignedAllocs;
        r10 = r10 >> 2;
        r11 = heap32[(r10)];
        r12 = r7 << 4;
        r11 = (r11 + 1)|0;
        r12 = r12 | 3;
        heap32[(r10)] = r11;
        r10 = (r12 + 16)|0;
        heap32[(g0)] = r10;
        malloc(i7);
        r10 = r_g0;
        if(r10 !=0) //_LBB101_28
{
        r11 = 0;
        r12 = (r10 + 4)|0;
        r11 = (r11 - r12)|0;
        r11 = r11 & 15;
        r11 = (r10 + r11)|0;
        r12 = (r11 + 4)|0;
        r11 = r11 >> 2;
        heap32[(r11)] = r10;
        r10 = r12;
}
}
else{
        r10 = 0;
}
if(!(r7 <1)) //_LBB101_32
{
        r11 = (r1 + 4)|0;
        r12 = (r10 + 4)|0;
        r13 = r7;
_40: while(true){
        r14 = r11 >> 2;
        r15 = heap32[(r14)];
        r14 = heap32[(r14+-1)];
        r16 = r12 >> 2;
        r13 = (r13 + -1)|0;
        r11 = (r11 + 8)|0;
        r12 = (r12 + 8)|0;
        heap32[(r16+-1)] = r14;
        heap32[(r16)] = r15;
if(!(r13 !=0)) //_LBB101_31
{
break _40;
}
}
}
if(!(r1 ==0)) //_LBB101_36
{
        r11 = heapU8[r2+36];
if(!(r11 ==0)) //_LBB101_35
{
        r11 = gNumAlignedFree;
        r11 = r11 >> 2;
        r12 = heap32[(r11)];
        r12 = (r12 + 1)|0;
        r1 = r1 >> 2;
        heap32[(r11)] = r12;
        r1 = heap32[(r1+-1)];
        heap32[(g0)] = r1;
        free(i7);
}
        heap32[(r3+8)] = 0;
}
        r1 = 1;
        heap8[r2+36] = r1;
        heap32[(r3+8)] = r10;
        heap32[(r3+7)] = r9;
if(!(r7 <r9)) //_LBB101_23
{
break _28;
}
}
_49: while(true){
        r7 = (r7 + -1)|0;
if(!(r7 !=0)) //_LBB101_37
{
break _28;
}
}
}
}
} while(0);
        r7 = (r9 + -4)|0;
        heap32[(r3+6)] = r9;
}
        if(r8 !=r5) //_LBB101_43
{
        r1 = r8 >> 2;
        r9 = r5 >> 2;
        f0 = heapFloat[(r1)];
        f1 = heapFloat[(r9+4)];
if(!(f0 >f1)) //_LBB101_41
{
        f0 = heapFloat[(r1+4)];
        f1 = heapFloat[(r9)];
if(!(f0 <f1)) //_LBB101_41
{
        f0 = heapFloat[(r1+1)];
        f1 = heapFloat[(r9+5)];
if(!(f0 >f1)) //_LBB101_41
{
        f0 = heapFloat[(r1+5)];
        f1 = heapFloat[(r9+1)];
if(!(f0 <f1)) //_LBB101_41
{
        f0 = heapFloat[(r1+2)];
        f1 = heapFloat[(r9+6)];
if(!(f0 >f1)) //_LBB101_41
{
        f0 = heapFloat[(r1+6)];
        f1 = heapFloat[(r9+2)];
if(!(f0 <f1)) //_LBB101_41
{
        r10 = heap32[(r9+10)];
        r11 = heap32[(r1+10)];
        if(r11 ==0) //_LBB101_53
{
        if(r10 ==0) //_LBB101_55
{
        r0 = r4 >> 2;
        r0 = heap32[(r0)];
        r0 = r0 >> 2;
        r0 = heap32[(r0+2)];
        heap32[(g0)] = r4;
        heap32[(g0+1)] = r8;
        heap32[(g0+2)] = r5;
        __FUNCTION_TABLE__[(r0)>>2](i7);
}
else{
        r5 = r0 << 3;
        r6 = heap32[(r3+8)];
        r6 = (r6 + r5)|0;
        r1 = heap32[(r9+9)];
        r6 = r6 >> 2;
        heap32[(r6+-2)] = r8;
        heap32[(r6+-1)] = r1;
        r6 = heap32[(r3+8)];
        r5 = (r6 + r5)|0;
        r1 = heap32[(r9+10)];
        r5 = r5 >> 2;
        r6 = (r0 + 1)|0;
        heap32[(r5)] = r8;
        heap32[(r5+1)] = r1;
}
}
else{
        r6 = heap32[(r3+8)];
        if(r10 ==0) //_LBB101_52
{
        r8 = r0 << 3;
        r6 = (r6 + r8)|0;
        r6 = r6 >> 2;
        r9 = heap32[(r1+9)];
        heap32[(r6+-2)] = r9;
        heap32[(r6+-1)] = r5;
        r6 = heap32[(r3+8)];
        r6 = (r6 + r8)|0;
        r8 = heap32[(r1+10)];
        r9 = r6 >> 2;
        r6 = (r0 + 1)|0;
        heap32[(r9)] = r8;
        heap32[(r9+1)] = r5;
}
else{
        r5 = r0 << 3;
        r6 = (r6 + r5)|0;
        r8 = heap32[(r9+9)];
        r10 = heap32[(r1+9)];
        r6 = r6 >> 2;
        heap32[(r6+-2)] = r10;
        heap32[(r6+-1)] = r8;
        r6 = heap32[(r3+8)];
        r6 = (r6 + r5)|0;
        r8 = heap32[(r9+9)];
        r10 = heap32[(r1+10)];
        r6 = r6 >> 2;
        heap32[(r6)] = r10;
        heap32[(r6+1)] = r8;
        r6 = heap32[(r3+8)];
        r6 = (r5 + r6)|0;
        r8 = heap32[(r9+10)];
        r10 = heap32[(r1+9)];
        r6 = r6 >> 2;
        heap32[(r6+2)] = r10;
        heap32[(r6+3)] = r8;
        r6 = heap32[(r3+8)];
        r6 = (r5 + r6)|0;
        r5 = heap32[(r9+10)];
        r1 = heap32[(r1+10)];
        r8 = r6 >> 2;
        r6 = (r0 + 3)|0;
        heap32[(r8+4)] = r1;
        heap32[(r8+5)] = r5;
}
}
}
}
}
}
}
}
}
else{
        r5 = r8 >> 2;
        r8 = heap32[(r5+10)];
        if(r8 !=0) //_LBB101_42
{
        r6 = r0 << 3;
        r8 = heap32[(r3+8)];
        r8 = (r8 + r6)|0;
        r1 = heap32[(r5+9)];
        r8 = r8 >> 2;
        heap32[(r8+-2)] = r1;
        heap32[(r8+-1)] = r1;
        r8 = heap32[(r3+8)];
        r8 = (r8 + r6)|0;
        r1 = heap32[(r5+10)];
        r8 = r8 >> 2;
        heap32[(r8)] = r1;
        heap32[(r8+1)] = r1;
        r8 = heap32[(r3+8)];
        r6 = (r6 + r8)|0;
        r8 = heap32[(r5+10)];
        r5 = heap32[(r5+9)];
        r1 = r6 >> 2;
        r6 = (r0 + 2)|0;
        heap32[(r1+2)] = r5;
        heap32[(r1+3)] = r8;
}
}
        if(r6 !=0) //_LBB101_18
{
continue _24;
}
else{
break _1;
}
}
}
}
} while(0);
        return;
}

function _ZN6btDbvt9collideTVEPK10btDbvtNodeRK12btDbvtAabbMmRNS_8ICollideE(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var r12;
        var f0;
        var f1;
        var f2;
        var f3;
        var f4;
        var f5;
        var f6;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
if(!(r0 ==0)) //_LBB102_46
{
        r1 = heap32[(fp+1)];
        r2 = heap32[(fp+2)];
        r3 = gNumAlignedAllocs;
        r1 = r1 >> 2;
        r3 = r3 >> 2;
        r4 = heap32[(r3)];
        f0 = heapFloat[(r1)];
        f1 = heapFloat[(r1+1)];
        f2 = heapFloat[(r1+2)];
        f3 = heapFloat[(r1+4)];
        f4 = heapFloat[(r1+5)];
        f5 = heapFloat[(r1+6)];
        r1 = (r4 + 1)|0;
        heap32[(r3)] = r1;
        heap32[(g0)] = 275;
        malloc(i7);
        r1 = r_g0;
        if(r1 !=0) //_LBB102_3
{
        r4 = 0;
        r5 = (r1 + 4)|0;
        r4 = (r4 - r5)|0;
        r4 = r4 & 15;
        r4 = (r1 + r4)|0;
        r5 = (r4 + 4)|0;
        r4 = r4 >> 2;
        heap32[(r4)] = r1;
        r1 = r5;
}
        r4 = 1;
        r5 = 64;
        r6 = r1 >> 2;
        heap32[(r6)] = r0;
_6: while(true){
        r0 = r4;
        r4 = (r0 + -1)|0;
        r6 = r4 << 2;
        r6 = (r1 + r6)|0;
        r6 = r6 >> 2;
        r6 = heap32[(r6)];
        r7 = r6 >> 2;
        f6 = heapFloat[(r7)];
        if(f6 <=f3) //_LBB102_7
{
        f6 = heapFloat[(r7+4)];
if(!(f6 <f0)) //_LBB102_6
{
        f6 = heapFloat[(r7+1)];
if(!(f6 >f4)) //_LBB102_6
{
        f6 = heapFloat[(r7+5)];
if(!(f6 <f1)) //_LBB102_6
{
        f6 = heapFloat[(r7+2)];
if(!(f6 >f5)) //_LBB102_6
{
        f6 = heapFloat[(r7+6)];
if(!(f6 <f2)) //_LBB102_6
{
        r8 = heap32[(r7+10)];
        if(r8 ==0) //_LBB102_42
{
        r0 = r2 >> 2;
        r0 = heap32[(r0)];
        r0 = r0 >> 2;
        r0 = heap32[(r0+3)];
        heap32[(g0)] = r2;
        heap32[(g0+1)] = r6;
        __FUNCTION_TABLE__[(r0)>>2](i7);
}
else{
        r6 = heap32[(r7+9)];
        if(r5 ==r4) //_LBB102_15
{
        r8 = 1;
        r9 = r4 << 1;
        r8 = r4 == 0 ? r8 : r9;
if(!(r5 >=r8)) //_LBB102_14
{
        if(r8 !=0) //_LBB102_18
{
        r5 = heap32[(r3)];
        r9 = r8 << 2;
        r5 = (r5 + 1)|0;
        r9 = r9 | 3;
        heap32[(r3)] = r5;
        r5 = (r9 + 16)|0;
        heap32[(g0)] = r5;
        malloc(i7);
        r9 = r_g0;
        if(r9 !=0) //_LBB102_20
{
        r5 = 0;
        r10 = (r9 + 4)|0;
        r5 = (r5 - r10)|0;
        r5 = r5 & 15;
        r5 = (r9 + r5)|0;
        r10 = (r5 + 4)|0;
        r5 = r5 >> 2;
        heap32[(r5)] = r9;
        r9 = r10;
}
}
else{
        r9 = 0;
}
if(!(r4 <1)) //_LBB102_24
{
        r4 = (r0 + -1)|0;
        r5 = r1;
        r10 = r9;
_28: while(true){
        r11 = r5 >> 2;
        r4 = (r4 + -1)|0;
        r12 = (r10 + 4)|0;
        r5 = (r5 + 4)|0;
        r10 = r10 >> 2;
        r11 = heap32[(r11)];
        heap32[(r10)] = r11;
        r10 = r12;
if(!(r4 !=0)) //_LBB102_23
{
break _28;
}
}
}
        if(r1 !=0) //_LBB102_26
{
        r4 = gNumAlignedFree;
        r4 = r4 >> 2;
        r5 = heap32[(r4)];
        r5 = (r5 + 1)|0;
        r1 = r1 >> 2;
        heap32[(r4)] = r5;
        r1 = heap32[(r1+-1)];
        heap32[(g0)] = r1;
        free(i7);
        r5 = r8;
        r1 = r9;
}
else{
        r5 = r8;
        r1 = r9;
}
}
}
        r4 = r0 << 2;
        r8 = (r1 + r4)|0;
        r8 = r8 >> 2;
        heap32[(r8+-1)] = r6;
        r6 = heap32[(r7+10)];
        if(r5 ==r0) //_LBB102_29
{
        r7 = 1;
        r8 = r0 << 1;
        r7 = r0 == 0 ? r7 : r8;
if(!(r5 >=r7)) //_LBB102_28
{
        if(r7 !=0) //_LBB102_32
{
        r5 = heap32[(r3)];
        r8 = r7 << 2;
        r5 = (r5 + 1)|0;
        r8 = r8 | 3;
        heap32[(r3)] = r5;
        r5 = (r8 + 16)|0;
        heap32[(g0)] = r5;
        malloc(i7);
        r8 = r_g0;
        if(r8 !=0) //_LBB102_34
{
        r5 = 0;
        r9 = (r8 + 4)|0;
        r5 = (r5 - r9)|0;
        r5 = r5 & 15;
        r5 = (r8 + r5)|0;
        r9 = (r5 + 4)|0;
        r5 = r5 >> 2;
        heap32[(r5)] = r8;
        r8 = r9;
}
}
else{
        r8 = 0;
}
_43: do {
if(!(r0 <1)) //_LBB102_38
{
        r5 = r1;
        r9 = r8;
        r10 = r0;
_45: while(true){
        r11 = r5 >> 2;
        r10 = (r10 + -1)|0;
        r12 = (r9 + 4)|0;
        r5 = (r5 + 4)|0;
        r9 = r9 >> 2;
        r11 = heap32[(r11)];
        heap32[(r9)] = r11;
        r9 = r12;
if(!(r10 !=0)) //_LBB102_37
{
break _43;
}
}
}
} while(0);
        if(r1 !=0) //_LBB102_40
{
        r5 = gNumAlignedFree;
        r5 = r5 >> 2;
        r9 = heap32[(r5)];
        r9 = (r9 + 1)|0;
        r1 = r1 >> 2;
        heap32[(r5)] = r9;
        r1 = heap32[(r1+-1)];
        heap32[(g0)] = r1;
        free(i7);
        r5 = r7;
        r1 = r8;
}
else{
        r5 = r7;
        r1 = r8;
}
}
}
        r7 = (r1 + r4)|0;
        r4 = (r0 + 1)|0;
        r0 = r7 >> 2;
        heap32[(r0)] = r6;
}
}
}
}
}
}
}
if(!(r4 >0)) //_LBB102_5
{
break _6;
}
}
if(!(r1 ==0)) //_LBB102_46
{
        r0 = gNumAlignedFree;
        r0 = r0 >> 2;
        r2 = heap32[(r0)];
        r2 = (r2 + 1)|0;
        r1 = r1 >> 2;
        heap32[(r0)] = r2;
        r0 = heap32[(r1+-1)];
        heap32[(g0)] = r0;
        free(i7);
}
}
        return;
}

function _ZN16btDbvtBroadphase8aabbTestERK9btVector3S2_R24btBroadphaseAabbCallback(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
var __label__ = 0;
        i7 = sp + -56;var g0 = i7>>2; // save stack
        r0 = _ZTV20BroadphaseAabbTester;
        r1 = sp + -8;
        r0 = (r0 + 8)|0;
        r2 = heap32[(fp+1)];
        r3 = r1 >> 2;
        r4 = heap32[(fp+3)];
        heap32[(fp+-2)] = r0;
        r0 = sp + -40;
        r2 = r2 >> 2;
        heap32[(r3+1)] = r4;
        r3 = r0 >> 2;
        heap32[(fp+-10)] = heap32[(r2)];
        heap32[(r3+1)] = heap32[(r2+1)];
        r4 = heap32[(fp+2)];
        heap32[(r3+2)] = heap32[(r2+2)];
        r4 = r4 >> 2;
        heap32[(r3+3)] = heap32[(r2+3)];
        heap32[(r3+4)] = heap32[(r4)];
        heap32[(r3+5)] = heap32[(r4+1)];
        r2 = heap32[(fp)];
        heap32[(r3+6)] = heap32[(r4+2)];
        r2 = r2 >> 2;
        heap32[(r3+7)] = heap32[(r4+3)];
        r3 = heap32[(r2+1)];
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r0;
        heap32[(g0+2)] = r1;
        _ZN6btDbvt9collideTVEPK10btDbvtNodeRK12btDbvtAabbMmRNS_8ICollideE(i7);
        r2 = heap32[(r2+11)];
        heap32[(g0)] = r2;
        heap32[(g0+1)] = r0;
        heap32[(g0+2)] = r1;
        _ZN6btDbvt9collideTVEPK10btDbvtNodeRK12btDbvtAabbMmRNS_8ICollideE(i7);
        return;
}

function _ZN16btDbvtBroadphase11createProxyERK9btVector3S2_iPvssP12btDispatcherS3_(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var f0;
        var f1;
        var f2;
        var f3;
        var f4;
        var f5;
        var f6;
        var f7;
var __label__ = 0;
        i7 = sp + -64;var g0 = i7>>2; // save stack
        r0 = gNumAlignedAllocs;
        r0 = r0 >> 2;
        r1 = heap32[(r0)];
        r2 = (r1 + 1)|0;
        heap32[(r0)] = r2;
        heap32[(g0)] = 83;
        malloc(i7);
        r2 = r_g0;
        r3 = heap32[(fp)];
        r4 = heap32[(fp+1)];
        r5 = heap32[(fp+2)];
        r6 = heap32[(fp+4)];
        r7 = heap32[(fp+5)];
        r8 = heap32[(fp+6)];
        if(r2 !=0) //_LBB104_2
{
        r9 = 0;
        r10 = (r2 + 4)|0;
        r9 = (r9 - r10)|0;
        r9 = r9 & 15;
        r9 = (r2 + r9)|0;
        r10 = (r9 + 4)|0;
        r9 = r9 >> 2;
        heap32[(r9)] = r2;
        r2 = r10;
}
        r9 = r2 >> 2;
        heap32[(r9)] = r6;
        heap16[(r2+4)>>1] = r7;
        r4 = r4 >> 2;
        heap16[(r2+6)>>1] = r8;
        f0 = heapFloat[(r4)];
        heapFloat[(r9+4)] = f0;
        f1 = heapFloat[(r4+1)];
        heapFloat[(r9+5)] = f1;
        f2 = heapFloat[(r4+2)];
        heapFloat[(r9+6)] = f2;
        f3 = heapFloat[(r4+3)];
        r4 = r5 >> 2;
        heapFloat[(r9+7)] = f3;
        f4 = heapFloat[(r4)];
        heapFloat[(r9+8)] = f4;
        f5 = heapFloat[(r4+1)];
        heapFloat[(r9+9)] = f5;
        f6 = heapFloat[(r4+2)];
        heapFloat[(r9+10)] = f6;
        f7 = heapFloat[(r4+3)];
        heapFloat[(r9+11)] = f7;
        heap32[(r9+2)] = 0;
        heap32[(r9+14)] = 0;
        r4 = sp + -32;
        heap32[(r9+13)] = 0;
        r5 = r4 >> 2;
        heapFloat[(fp+-8)] = f0;
        heapFloat[(r5+1)] = f1;
        heapFloat[(r5+2)] = f2;
        heapFloat[(r5+3)] = f3;
        heapFloat[(r5+4)] = f4;
        heapFloat[(r5+5)] = f5;
        heapFloat[(r5+6)] = f6;
        r6 = r3 >> 2;
        heapFloat[(r5+7)] = f7;
        r5 = heap32[(r6+26)];
        heap32[(r9+15)] = r5;
        r5 = heap32[(r6+37)];
        r5 = (r5 + 1)|0;
        heap32[(r6+37)] = r5;
        heap32[(r9+3)] = r5;
        r5 = heap32[(r6+2)];
        r7 = (r3 + 4)|0;
        if(r5 ==0) //_LBB104_5
{
        r5 = (r1 + 2)|0;
        heap32[(r0)] = r5;
        heap32[(g0)] = 63;
        malloc(i7);
        r5 = r_g0;
        if(r5 !=0) //_LBB104_7
{
        r0 = 0;
        r1 = (r5 + 4)|0;
        r0 = (r0 - r1)|0;
        r0 = r0 & 15;
        r0 = (r5 + r0)|0;
        r1 = (r0 + 4)|0;
        r0 = r0 >> 2;
        heap32[(r0)] = r5;
        r5 = r1;
}
}
else{
        heap32[(r6+2)] = 0;
}
        r0 = r5 >> 2;
        heap32[(r0+8)] = 0;
        heap32[(r0+9)] = r2;
        heap32[(r0+10)] = 0;
        heapFloat[(r0)] = f0;
        heapFloat[(r0+1)] = f1;
        heapFloat[(r0+2)] = f2;
        heapFloat[(r0+3)] = f3;
        heapFloat[(r0+4)] = f4;
        heapFloat[(r0+5)] = f5;
        heapFloat[(r0+6)] = f6;
        heapFloat[(r0+7)] = f7;
        r0 = heap32[(r6+1)];
        heap32[(g0)] = r7;
        heap32[(g0+1)] = r0;
        heap32[(g0+2)] = r5;
        _ZL10insertleafP6btDbvtP10btDbvtNodeS2_(i7);
        r0 = heap32[(r6+4)];
        r0 = (r0 + 1)|0;
        heap32[(r6+4)] = r0;
        heap32[(r9+12)] = r5;
        r0 = heap32[(r6+26)];
        r0 = r0 << 2;
        r0 = (r3 + r0)|0;
        r0 = r0 >> 2;
        heap32[(r9+13)] = 0;
        r1 = heap32[(r0+21)];
        heap32[(r9+14)] = r1;
if(!(r1 ==0)) //_LBB104_10
{
        r1 = r1 >> 2;
        heap32[(r1+13)] = r2;
}
        heap32[(r0+21)] = r2;
        r0 = heapU8[r3+153];
if(!(r0 !=0)) //_LBB104_12
{
        r0 = _ZTV18btDbvtTreeCollider;
        r0 = (r0 + 8)|0;
        r1 = sp + -48;
        r5 = r1 >> 2;
        heap32[(fp+-12)] = r0;
        heap32[(r5+1)] = r3;
        heap32[(r5+2)] = r2;
        r3 = heap32[(r6+1)];
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r4;
        heap32[(g0+2)] = r1;
        _ZN6btDbvt9collideTVEPK10btDbvtNodeRK12btDbvtAabbMmRNS_8ICollideE(i7);
        r3 = heap32[(r6+11)];
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r4;
        heap32[(g0+2)] = r1;
        _ZN6btDbvt9collideTVEPK10btDbvtNodeRK12btDbvtAabbMmRNS_8ICollideE(i7);
        heap32[(fp+-12)] = r0;
}
        r_g0 = r2;
        return;
}

function _ZNK6btDbvt15rayTestInternalEPK10btDbvtNodeRK9btVector3S5_S5_PjfS5_S5_RNS_8ICollideE(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var r12;
        var r13;
        var r14;
        var r15;
        var r16;
        var r17;
        var r18;
        var r19;
        var r20;
        var r21;
        var r22;
        var r23;
        var f0;
        var f1;
        var f2;
        var f3;
        var f4;
        var f5;
        var f6;
        var f7;
        var f8;
var __label__ = 0;
        i7 = sp + -40;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
if(!(r0 ==0)) //_LBB105_43
{
        r1 = heap32[(fp+1)];
        r2 = heap32[(fp+2)];
        r3 = heap32[(fp+3)];
        f0 = heapFloat[(fp+4)];
        r4 = heap32[(fp+5)];
        r5 = heap32[(fp+6)];
        r6 = heap32[(fp+7)];
        r7 = gNumAlignedAllocs;
        r7 = r7 >> 2;
        r8 = heap32[(r7)];
        r8 = (r8 + 1)|0;
        heap32[(r7)] = r8;
        heap32[(g0)] = 531;
        malloc(i7);
        r8 = r_g0;
        if(r8 !=0) //_LBB105_3
{
        r9 = 0;
        r10 = (r8 + 4)|0;
        r9 = (r9 - r10)|0;
        r9 = r9 & 15;
        r9 = (r8 + r9)|0;
        r10 = (r9 + 4)|0;
        r9 = r9 >> 2;
        heap32[(r9)] = r8;
        r8 = r10;
}
        r9 = 0;
_6: while(true){
        r10 = r9 << 2;
        r10 = (r8 + r10)|0;
        r9 = (r9 + 1)|0;
        r10 = r10 >> 2;
        heap32[(r10)] = 0;
if(!(r9 !=128)) //_LBB105_5
{
break _6;
}
}
        r9 = 128;
        r10 = 1;
        r11 = 126;
        r12 = r8 >> 2;
        heap32[(r12)] = r0;
        r0 = r9;
        r12 = r10;
_9: while(true){
        r13 = r12;
        r12 = (r13 + -1)|0;
        r14 = r12 << 2;
        r14 = (r8 + r14)|0;
        r14 = r14 >> 2;
        r14 = heap32[(r14)];
        r15 = r14 >> 2;
        r16 = r5 >> 2;
        f1 = heapFloat[(r15+2)];
        f2 = heapFloat[(r16+2)];
        f3 = heapFloat[(r15+1)];
        f4 = heapFloat[(r16+1)];
        f5 = heapFloat[(r15)];
        f6 = heapFloat[(r16)];
        r16 = sp + -32;
        f5 = f5-f6;
        r17 = r16 >> 2;
        f3 = f3-f4;
        heapFloat[(fp+-8)] = f5;
        f1 = f1-f2;
        heapFloat[(r17+1)] = f3;
        heapFloat[(r17+2)] = f1;
        heap32[(r17+3)] = 0;
        r18 = r4 >> 2;
        f1 = heapFloat[(r15+6)];
        f2 = heapFloat[(r18+2)];
        f3 = heapFloat[(r15+5)];
        f4 = heapFloat[(r18+1)];
        f5 = heapFloat[(r15+4)];
        f6 = heapFloat[(r18)];
        f5 = f5-f6;
        f3 = f3-f4;
        heapFloat[(r17+4)] = f5;
        f1 = f1-f2;
        heapFloat[(r17+5)] = f3;
        heapFloat[(r17+6)] = f1;
        r18 = r3 >> 2;
        heap32[(r17+7)] = 0;
        r17 = heap32[(r18)];
        r19 = heap32[(r18+1)];
        r20 = (r10 - r19)|0;
        r21 = r17 << 4;
        r20 = r20 << 4;
        r21 = (r16 + r21)|0;
        r20 = (r16 + r20)|0;
        r22 = r1 >> 2;
        r21 = r21 >> 2;
        r20 = r20 >> 2;
        r23 = r2 >> 2;
        f1 = heapFloat[(r21)];
        f2 = heapFloat[(r22)];
        f3 = heapFloat[(r20+1)];
        f4 = heapFloat[(r22+1)];
        f1 = f1-f2;
        f5 = heapFloat[(r23)];
        f3 = f3-f4;
        f6 = heapFloat[(r23+1)];
        f1 = f1*f5;
        f3 = f3*f6;
if(!(f1 >f3)) //_LBB105_9
{
        r17 = (r10 - r17)|0;
        r17 = r17 << 4;
        r19 = r19 << 4;
        r17 = (r16 + r17)|0;
        r19 = (r16 + r19)|0;
        r17 = r17 >> 2;
        r19 = r19 >> 2;
        f7 = heapFloat[(r17)];
        f8 = heapFloat[(r19+1)];
        f2 = f7-f2;
        f4 = f8-f4;
        f2 = f2*f5;
        f4 = f4*f6;
        if(f4 <=f2) //_LBB105_10
{
        r17 = heap32[(r18+2)];
        r18 = 1;
        r18 = (r18 - r17)|0;
        r18 = r18 << 4;
        r18 = (r16 + r18)|0;
        r18 = r18 >> 2;
        f5 = heapFloat[(r18+2)];
        f6 = heapFloat[(r22+2)];
        f5 = f5-f6;
        f7 = heapFloat[(r23+2)];
        f1 = f1 < f4 ? f4 : f1;
        f4 = f5*f7;
if(!(f1 >f4)) //_LBB105_9
{
        r17 = r17 << 4;
        r16 = (r16 + r17)|0;
        r16 = r16 >> 2;
        f5 = heapFloat[(r16+2)];
        f5 = f5-f6;
        f2 = f3 < f2 ? f3 : f2;
        f3 = f5*f7;
if(!(f3 >f2)) //_LBB105_9
{
        f1 = f1 < f3 ? f3 : f1;
if(!(f1 >=f0)) //_LBB105_9
{
        f1 = f4 < f2 ? f4 : f2;
        f2 =                         0;
if(!(f1 <=f2)) //_LBB105_9
{
        r16 = heap32[(r15+10)];
        if(r16 ==0) //_LBB105_39
{
        r13 = r6 >> 2;
        r13 = heap32[(r13)];
        r13 = r13 >> 2;
        r13 = heap32[(r13+3)];
        heap32[(g0)] = r6;
        heap32[(g0+1)] = r14;
        __FUNCTION_TABLE__[(r13)>>2](i7);
}
else{
        if(r12 >r11) //_LBB105_17
{
        r14 = r0 << 1;
_23: do {
        if(r0 <=r14) //_LBB105_19
{
if(!(r0 >=r14)) //_LBB105_18
{
        if(r9 <r14) //_LBB105_22
{
        if(r14 !=0) //_LBB105_24
{
        r9 = heap32[(r7)];
        r11 = r0 << 3;
        r9 = (r9 + 1)|0;
        r11 = r11 | 3;
        heap32[(r7)] = r9;
        r9 = (r11 + 16)|0;
        heap32[(g0)] = r9;
        malloc(i7);
        r9 = r_g0;
        if(r9 !=0) //_LBB105_26
{
        r11 = 0;
        r12 = (r9 + 4)|0;
        r11 = (r11 - r12)|0;
        r11 = r11 & 15;
        r11 = (r9 + r11)|0;
        r12 = (r11 + 4)|0;
        r11 = r11 >> 2;
        heap32[(r11)] = r9;
        r9 = r12;
}
}
else{
        r9 = 0;
}
if(!(r0 <1)) //_LBB105_30
{
        r11 = r8;
        r12 = r9;
        r16 = r0;
_35: while(true){
        r17 = r11 >> 2;
        r16 = (r16 + -1)|0;
        r18 = (r12 + 4)|0;
        r11 = (r11 + 4)|0;
        r12 = r12 >> 2;
        r17 = heap32[(r17)];
        heap32[(r12)] = r17;
        r12 = r18;
if(!(r16 !=0)) //_LBB105_29
{
break _35;
}
}
}
if(!(r8 ==0)) //_LBB105_32
{
        r11 = gNumAlignedFree;
        r11 = r11 >> 2;
        r12 = heap32[(r11)];
        r12 = (r12 + 1)|0;
        r8 = r8 >> 2;
        heap32[(r11)] = r12;
        r8 = heap32[(r8+-1)];
        heap32[(g0)] = r8;
        free(i7);
}
        if(r0 <r14) //_LBB105_34
{
        r8 = r9;
        r9 = r14;
}
else{
        r8 = r9;
        r9 = r14;
break _23;
}
}
        r11 = r0;
_45: while(true){
        r12 = r0 << 2;
        r12 = (r8 + r12)|0;
        r11 = (r11 + -1)|0;
        r0 = (r0 + 1)|0;
        r12 = r12 >> 2;
        heap32[(r12)] = 0;
if(!(r11 !=0)) //_LBB105_36
{
break _23;
}
}
}
}
} while(0);
        r11 = (r14 + -2)|0;
}
else{
        r14 = r0;
}
        r0 = r13 << 2;
        r0 = (r8 + r0)|0;
        r0 = r0 >> 2;
        r12 = heap32[(r15+9)];
        heap32[(r0+-1)] = r12;
        r15 = heap32[(r15+10)];
        r12 = (r13 + 1)|0;
        heap32[(r0)] = r15;
        r0 = r14;
}
}
}
}
}
}
}
if(!(r12 !=0)) //_LBB105_7
{
break _9;
}
}
if(!(r8 ==0)) //_LBB105_43
{
        r0 = gNumAlignedFree;
        r0 = r0 >> 2;
        r1 = heap32[(r0)];
        r1 = (r1 + 1)|0;
        r2 = r8 >> 2;
        heap32[(r0)] = r1;
        r0 = heap32[(r2+-1)];
        heap32[(g0)] = r0;
        free(i7);
}
}
        return;
}

function _ZN16btDbvtBroadphase7rayTestERK9btVector3S2_R23btBroadphaseRayCallbackS2_S2_(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var f0;
var __label__ = 0;
        i7 = sp + -40;var g0 = i7>>2; // save stack
        r0 = _ZTV19BroadphaseRayTester;
        r1 = sp + -8;
        r0 = (r0 + 8)|0;
        r2 = heap32[(fp+3)];
        r3 = heap32[(fp)];
        r4 = r1 >> 2;
        heap32[(fp+-2)] = r0;
        heap32[(r4+1)] = r2;
        r0 = r2 >> 2;
        r3 = r3 >> 2;
        f0 = heapFloat[(r0+8)];
        r4 = heap32[(r3+1)];
        r5 = heap32[(fp+1)];
        r6 = (r2 + 4)|0;
        r2 = (r2 + 20)|0;
        r7 = heap32[(fp+4)];
        r8 = heap32[(fp+5)];
        heap32[(g0)] = r4;
        heap32[(g0+1)] = r5;
        heap32[(g0+2)] = r6;
        heap32[(g0+3)] = r2;
        heapFloat[(g0+4)] = f0;
        heap32[(g0+5)] = r7;
        heap32[(g0+6)] = r8;
        heap32[(g0+7)] = r1;
        _ZNK6btDbvt15rayTestInternalEPK10btDbvtNodeRK9btVector3S5_S5_PjfS5_S5_RNS_8ICollideE(i7);
        f0 = heapFloat[(r0+8)];
        r0 = heap32[(r3+11)];
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r5;
        heap32[(g0+2)] = r6;
        heap32[(g0+3)] = r2;
        heapFloat[(g0+4)] = f0;
        heap32[(g0+5)] = r7;
        heap32[(g0+6)] = r8;
        heap32[(g0+7)] = r1;
        _ZNK6btDbvt15rayTestInternalEPK10btDbvtNodeRK9btVector3S5_S5_PjfS5_S5_RNS_8ICollideE(i7);
        return;
}

function _ZN16btDbvtBroadphase12destroyProxyEP17btBroadphaseProxyP12btDispatcher(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
var __label__ = 0;
        i7 = sp + -16;var g0 = i7>>2; // save stack
        r0 = heap32[(fp+1)];
        r1 = r0 >> 2;
        r2 = heap32[(fp)];
        r3 = heap32[(fp+2)];
        r4 = heap32[(r1+12)];
        r5 = heap32[(r1+15)];
        if(r5 !=2) //_LBB107_4
{
        r5 = (r2 + 4)|0;
        heap32[(g0)] = r5;
        heap32[(g0+1)] = r4;
        r5 = r2 >> 2;
        _ZL10removeleafP6btDbvtP10btDbvtNode(i7);
        r6 = heap32[(r5+2)];
if(!(r6 ==0)) //_LBB107_6
{
        r7 = gNumAlignedFree;
        r7 = r7 >> 2;
        r8 = heap32[(r7)];
        r8 = (r8 + 1)|0;
        r6 = r6 >> 2;
        heap32[(r7)] = r8;
        r6 = heap32[(r6+-1)];
        heap32[(g0)] = r6;
        free(i7);
}
        heap32[(r5+2)] = r4;
        r4 = heap32[(r5+4)];
        r4 = (r4 + -1)|0;
        heap32[(r5+4)] = r4;
}
else{
        r5 = (r2 + 44)|0;
        heap32[(g0)] = r5;
        heap32[(g0+1)] = r4;
        r5 = r2 >> 2;
        _ZL10removeleafP6btDbvtP10btDbvtNode(i7);
        r6 = heap32[(r5+12)];
if(!(r6 ==0)) //_LBB107_3
{
        r7 = gNumAlignedFree;
        r7 = r7 >> 2;
        r8 = heap32[(r7)];
        r8 = (r8 + 1)|0;
        r6 = r6 >> 2;
        heap32[(r7)] = r8;
        r6 = heap32[(r6+-1)];
        heap32[(g0)] = r6;
        free(i7);
}
        heap32[(r5+12)] = r4;
        r4 = heap32[(r5+14)];
        r4 = (r4 + -1)|0;
        heap32[(r5+14)] = r4;
}
        r4 = heap32[(r1+13)];
        r5 = heap32[(r1+14)];
        if(r4 ==0) //_LBB107_9
{
        r4 = heap32[(r1+15)];
        r4 = r4 << 2;
        r4 = (r2 + r4)|0;
        r4 = r4 >> 2;
        heap32[(r4+21)] = r5;
}
else{
        r4 = r4 >> 2;
        heap32[(r4+14)] = r5;
}
        r4 = heap32[(r1+14)];
if(!(r4 ==0)) //_LBB107_12
{
        r4 = r4 >> 2;
        r5 = heap32[(r1+13)];
        heap32[(r4+13)] = r5;
}
        r4 = r2 >> 2;
        r4 = heap32[(r4+24)];
        r5 = r4 >> 2;
        r5 = heap32[(r5)];
        r5 = r5 >> 2;
        r5 = heap32[(r5+4)];
        heap32[(g0)] = r4;
        heap32[(g0+1)] = r0;
        heap32[(g0+2)] = r3;
        __FUNCTION_TABLE__[(r5)>>2](i7);
if(!(r0 ==0)) //_LBB107_14
{
        r0 = gNumAlignedFree;
        r0 = r0 >> 2;
        r3 = heap32[(r0)];
        r3 = (r3 + 1)|0;
        heap32[(r0)] = r3;
        r0 = heap32[(r1+-1)];
        heap32[(g0)] = r0;
        free(i7);
}
        r0 = 1;
        heap8[r2+154] = r0;
        return;
}

function _ZN16btDbvtBroadphase25calculateOverlappingPairsEP12btDispatcher(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var r12;
        var f0;
        var f1;
        var f2;
        var f3;
        var f4;
        var f5;
        var f6;
        var f7;
var __label__ = 0;
        i7 = sp + -48;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = r0 >> 2;
        r2 = heap32[(r1+4)];
        r3 = heap32[(r1+28)];
        r2 = (r3 * r2)|0;
        r2 = (r2 / 100)|0;
        r3 = (r0 + 4)|0;
        r2 = (r2 + 1)|0;
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r2;
        _ZN6btDbvt19optimizeIncrementalEi(i7);
        r2 = heap32[(r1+31)];
if(!(r2 ==0)) //_LBB108_2
{
        r2 = heap32[(r1+14)];
        r4 = heap32[(r1+27)];
        r2 = (r4 * r2)|0;
        r4 = (r2 / 100)|0;
        r5 = (r0 + 44)|0;
        r4 = (r4 + 1)|0;
        heap32[(g0)] = r5;
        heap32[(g0+1)] = r4;
        _ZN6btDbvt19optimizeIncrementalEi(i7);
        r2 = (r2 / -100)|0;
        r4 = heap32[(r1+31)];
        r2 = (r2 + r4)|0;
        r2 = (r2 + -1)|0;
        r4 = 0;
        r2 = r2 < 0 ? r4 : r2;
        heap32[(r1+31)] = r2;
}
        r2 = heap32[(r1+26)];
        r2 = (r2 + 1)|0;
        r4 = r2 >>> 31;
        r4 = (r2 + r4)|0;
        r4 = r4 & -2;
        r2 = (r2 - r4)|0;
        r4 = r2 << 2;
        r4 = (r0 + r4)|0;
        r4 = r4 >> 2;
        heap32[(r1+26)] = r2;
        r2 = heap32[(r4+21)];
if(!(r2 ==0)) //_LBB108_20
{
        r4 = (r0 + 44)|0;
_6: while(true){
        r5 = r2 >> 2;
        r6 = heap32[(r5+13)];
        r7 = heap32[(r5+14)];
        if(r6 ==0) //_LBB108_6
{
        r6 = heap32[(r5+15)];
        r6 = r6 << 2;
        r6 = (r0 + r6)|0;
        r6 = r6 >> 2;
        heap32[(r6+21)] = r7;
}
else{
        r6 = r6 >> 2;
        heap32[(r6+14)] = r7;
}
        r6 = heap32[(r5+14)];
if(!(r6 ==0)) //_LBB108_9
{
        r6 = r6 >> 2;
        r8 = heap32[(r5+13)];
        heap32[(r6+13)] = r8;
}
        heap32[(r5+13)] = 0;
        r6 = heap32[(r1+23)];
        heap32[(r5+14)] = r6;
        r6 = heap32[(r1+23)];
if(!(r6 ==0)) //_LBB108_11
{
        r6 = r6 >> 2;
        heap32[(r6+13)] = r2;
}
        heap32[(r1+23)] = r2;
        r6 = heap32[(r5+12)];
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r6;
        _ZL10removeleafP6btDbvtP10btDbvtNode(i7);
        r8 = heap32[(r1+2)];
if(!(r8 ==0)) //_LBB108_13
{
        r9 = gNumAlignedFree;
        r9 = r9 >> 2;
        r10 = heap32[(r9)];
        r10 = (r10 + 1)|0;
        r8 = r8 >> 2;
        heap32[(r9)] = r10;
        r8 = heap32[(r8+-1)];
        heap32[(g0)] = r8;
        free(i7);
}
        heap32[(r1+2)] = r6;
        r6 = heap32[(r1+4)];
        r6 = (r6 + -1)|0;
        heap32[(r1+4)] = r6;
        r6 = heap32[(r1+12)];
        f0 = heapFloat[(r5+4)];
        f1 = heapFloat[(r5+5)];
        f2 = heapFloat[(r5+6)];
        f3 = heapFloat[(r5+7)];
        f4 = heapFloat[(r5+8)];
        f5 = heapFloat[(r5+9)];
        f6 = heapFloat[(r5+10)];
        f7 = heapFloat[(r5+11)];
        if(r6 ==0) //_LBB108_15
{
        r6 = gNumAlignedAllocs;
        r6 = r6 >> 2;
        r8 = heap32[(r6)];
        r8 = (r8 + 1)|0;
        heap32[(r6)] = r8;
        heap32[(g0)] = 63;
        malloc(i7);
        r6 = r_g0;
        if(r6 !=0) //_LBB108_17
{
        r8 = 0;
        r9 = (r6 + 4)|0;
        r8 = (r8 - r9)|0;
        r8 = r8 & 15;
        r8 = (r6 + r8)|0;
        r9 = (r8 + 4)|0;
        r8 = r8 >> 2;
        heap32[(r8)] = r6;
        r6 = r9;
}
}
else{
        heap32[(r1+12)] = 0;
}
        r8 = r6 >> 2;
        heap32[(r8+8)] = 0;
        heap32[(r8+9)] = r2;
        heap32[(r8+10)] = 0;
        heapFloat[(r8)] = f0;
        heapFloat[(r8+1)] = f1;
        heapFloat[(r8+2)] = f2;
        heapFloat[(r8+3)] = f3;
        heapFloat[(r8+4)] = f4;
        heapFloat[(r8+5)] = f5;
        heapFloat[(r8+6)] = f6;
        heapFloat[(r8+7)] = f7;
        r2 = heap32[(r1+11)];
        heap32[(g0)] = r4;
        heap32[(g0+1)] = r2;
        heap32[(g0+2)] = r6;
        _ZL10insertleafP6btDbvtP10btDbvtNodeS2_(i7);
        r2 = heap32[(r1+14)];
        r2 = (r2 + 1)|0;
        heap32[(r1+14)] = r2;
        heap32[(r5+12)] = r6;
        heap32[(r5+15)] = 2;
        r2 = r7;
if(!(r7 !=0)) //_LBB108_4
{
break _6;
}
}
        r2 = heap32[(r1+14)];
        r4 = 1;
        heap32[(r1+31)] = r2;
        heap8[r0+154] = r4;
}
        r2 = _ZTV18btDbvtTreeCollider;
        r4 = sp + -32;
        r2 = (r2 + 8)|0;
        r5 = r4 >> 2;
        heap32[(fp+-8)] = r2;
        heap32[(r5+1)] = r0;
        r5 = heapU8[r0+153];
if(!(r5 ==0)) //_LBB108_23
{
        r5 = heap32[(r1+11)];
        r6 = heap32[(r1+1)];
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r6;
        heap32[(g0+2)] = r5;
        heap32[(g0+3)] = r4;
        _ZN6btDbvt24collideTTpersistentStackEPK10btDbvtNodeS2_RNS_8ICollideE(i7);
        r5 = heapU8[r0+153];
if(!(r5 ==0)) //_LBB108_23
{
        r5 = heap32[(r1+1)];
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r5;
        heap32[(g0+2)] = r5;
        heap32[(g0+3)] = r4;
        _ZN6btDbvt24collideTTpersistentStackEPK10btDbvtNodeS2_RNS_8ICollideE(i7);
}
}
        r3 = heap32[(fp+1)];
        heap32[(fp+-8)] = r2;
        r2 = heapU8[r0+154];
_32: do {
if(!(r2 ==0)) //_LBB108_40
{
        r2 = heap32[(r1+24)];
        r4 = r2 >> 2;
        r4 = heap32[(r4)];
        r4 = r4 >> 2;
        r4 = heap32[(r4+7)];
        heap32[(g0)] = r2;
        __FUNCTION_TABLE__[(r4)>>2](i7);
        r2 = r_g0;
        r4 = r2 >> 2;
        r4 = heap32[(r4+1)];
if(!(r4 <1)) //_LBB108_40
{
        r5 = heap32[(r1+29)];
        r5 = (r5 * r4)|0;
        r6 = heap32[(r1+30)];
        r5 = (r5 / 100)|0;
        r5 = r6 > r5 ? r6 : r5;
        r5 = r4 < r5 ? r4 : r5;
        if(r5 >0) //_LBB108_27
{
        r4 = 0;
_37: while(true){
        r6 = r2 >> 2;
        r7 = heap32[(r1+36)];
        r7 = (r7 + r4)|0;
        r8 = heap32[(r6+1)];
        r7 = (r7 % r8)|0;
        r8 = heap32[(r6+3)];
        r7 = r7 << 4;
        r7 = (r8 + r7)|0;
        r7 = r7 >> 2;
        r8 = heap32[(r7+1)];
        r7 = heap32[(r7)];
        r9 = r8 >> 2;
        r10 = r7 >> 2;
        r9 = heap32[(r9+12)];
        r10 = heap32[(r10+12)];
        r10 = r10 >> 2;
        r9 = r9 >> 2;
        f0 = heapFloat[(r10)];
        f1 = heapFloat[(r9+4)];
        if(f0 >f1) //_LBB108_35
{
__label__ = 32;
}
else{
        f0 = heapFloat[(r10+4)];
        f1 = heapFloat[(r9)];
        if(f0 <f1) //_LBB108_35
{
__label__ = 32;
}
else{
        f0 = heapFloat[(r10+1)];
        f1 = heapFloat[(r9+5)];
        if(f0 >f1) //_LBB108_35
{
__label__ = 32;
}
else{
        f0 = heapFloat[(r10+5)];
        f1 = heapFloat[(r9+1)];
        if(f0 <f1) //_LBB108_35
{
__label__ = 32;
}
else{
        f0 = heapFloat[(r10+2)];
        f1 = heapFloat[(r9+6)];
        if(f0 >f1) //_LBB108_35
{
__label__ = 32;
}
else{
        f0 = heapFloat[(r10+6)];
        f1 = heapFloat[(r9+2)];
        if(f0 <f1) //_LBB108_35
{
__label__ = 32;
}
else{
__label__ = 33;
}
}
}
}
}
}
if (__label__ == 32){
        r9 = heap32[(r1+24)];
        r10 = r9 >> 2;
        r10 = heap32[(r10)];
        r10 = r10 >> 2;
        r10 = heap32[(r10+3)];
        heap32[(g0)] = r9;
        heap32[(g0+1)] = r7;
        heap32[(g0+2)] = r8;
        heap32[(g0+3)] = r3;
        r5 = (r5 + -1)|0;
        r4 = (r4 + -1)|0;
        __FUNCTION_TABLE__[(r10)>>2](i7);
}
        r4 = (r4 + 1)|0;
if(!(r4 <r5)) //_LBB108_28
{
break _37;
}
}
        r4 = heap32[(r6+1)];
        if(r4 <1) //_LBB108_39
{
        heap32[(r1+36)] = 0;
break _32;
}
}
        r2 = heap32[(r1+36)];
        r2 = (r2 + r5)|0;
        r2 = (r2 % r4)|0;
        heap32[(r1+36)] = r2;
}
}
} while(0);
        r2 = heap32[(r1+35)];
        r2 = (r2 + 1)|0;
        heap32[(r1+35)] = r2;
        r2 = 0;
        heap32[(r1+30)] = 1;
        heap8[r0+154] = r2;
        r0 = heap32[(r1+32)];
        if(r0 ==0) //_LBB108_42
{
        heap32[(r1+34)] = 0;
        r4 = heap32[(r1+33)];
}
else{
        r4 = heap32[(r1+33)];
        f0 = uint(r4); //fuitos r4, f0
        f1 = uint(r0); //fuitos r0, f1
        f0 = f0/f1;
        heapFloat[(r1+34)] = f0;
}
        r4 = r4 >>> 1;
        r0 = r0 >>> 1;
        heap32[(r1+33)] = r4;
        heap32[(r1+32)] = r0;
        r0 = heap32[(r1+24)];
        r4 = r0 >> 2;
        r4 = heap32[(r4)];
        r4 = r4 >> 2;
        r4 = heap32[(r4+14)];
        heap32[(g0)] = r0;
        __FUNCTION_TABLE__[(r4)>>2](i7);
        r0 = r_g0;
if(!(r0 ==0)) //_LBB108_65
{
        r0 = heap32[(r1+24)];
        r4 = r0 >> 2;
        r4 = heap32[(r4)];
        r4 = r4 >> 2;
        r4 = heap32[(r4+7)];
        heap32[(g0)] = r0;
        __FUNCTION_TABLE__[(r4)>>2](i7);
        r0 = r_g0;
        r4 = r0 >> 2;
        r4 = heap32[(r4+1)];
if(!(r4 <2)) //_LBB108_46
{
        r4 = (r4 + -1)|0;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = 0;
        heap32[(g0+2)] = r4;
        _ZN20btAlignedObjectArrayI16btBroadphasePairE17quickSortInternalI29btBroadphasePairSortPredicateEEvT_ii(i7);
}
        r4 = r2;
        r5 = r2;
        r6 = r2;
        r7 = r2;
_61: while(true){
        r8 = r0 >> 2;
        r9 = heap32[(r8+1)];
        if(r9 >r7) //_LBB108_47
{
        r8 = heap32[(r8+3)];
        r9 = r7 << 4;
        r9 = (r8 + r9)|0;
        r9 = r9 >> 2;
        r10 = heap32[(r9)];
        r11 = r7 << 2;
        r12 = heap32[(r9+1)];
        if(r10 !=r4) //_LBB108_49
{
__label__ = 46;
}
else{
        if(r12 ==r5) //_LBB108_56
{
        r4 = heap32[(r9+2)];
        if(r4 ==0) //_LBB108_58
{
__label__ = 54;
}
else{
__label__ = 53;
break _61;
}
}
else{
__label__ = 46;
}
}
if (__label__ == 46){
        r4 = r12 >> 2;
        r5 = r10 >> 2;
        r4 = heap32[(r4+12)];
        r5 = heap32[(r5+12)];
        r5 = r5 >> 2;
        r4 = r4 >> 2;
        f0 = heapFloat[(r5)];
        f1 = heapFloat[(r4+4)];
        if(f0 >f1) //_LBB108_58
{
__label__ = 54;
}
else{
        f0 = heapFloat[(r5+4)];
        f1 = heapFloat[(r4)];
        if(f0 <f1) //_LBB108_58
{
__label__ = 54;
}
else{
        f0 = heapFloat[(r5+1)];
        f1 = heapFloat[(r4+5)];
        if(f0 >f1) //_LBB108_58
{
__label__ = 54;
}
else{
        f0 = heapFloat[(r5+5)];
        f1 = heapFloat[(r4+1)];
        if(f0 <f1) //_LBB108_58
{
__label__ = 54;
}
else{
        f0 = heapFloat[(r5+2)];
        f1 = heapFloat[(r4+6)];
        if(f0 >f1) //_LBB108_58
{
__label__ = 54;
}
else{
        f0 = heapFloat[(r5+6)];
        f1 = heapFloat[(r4+2)];
        if(f0 <f1) //_LBB108_58
{
__label__ = 54;
}
else{
__label__ = 55;
}
}
}
}
}
}
}
if (__label__ == 54){
        r4 = heap32[(r1+24)];
        r5 = r4 >> 2;
        r5 = heap32[(r5)];
        r5 = r5 >> 2;
        r5 = heap32[(r5+8)];
        r9 = (r8 + r2)|0;
        r11 = r11 << 2;
        heap32[(g0)] = r4;
        heap32[(g0+1)] = r9;
        heap32[(g0+2)] = r3;
        r4 = (r8 + r11)|0;
        r4 = r4 >> 2;
        __FUNCTION_TABLE__[(r5)>>2](i7);
        r6 = (r6 + 1)|0;
        heap32[(r4)] = 0;
        heap32[(r4+1)] = 0;
}
        r7 = (r7 + 1)|0;
        r2 = (r2 + 16)|0;
        r4 = r10;
        r5 = r12;
}
else{
__label__ = 57;
break _61;
}
}
switch(__label__ ){//multiple entries
case 57:
        if(r9 >1) //_LBB108_63
{
        r9 = (r9 + -1)|0;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = 0;
        heap32[(g0+2)] = r9;
        _ZN20btAlignedObjectArrayI16btBroadphasePairE17quickSortInternalI29btBroadphasePairSortPredicateEEvT_ii(i7);
        r9 = heap32[(r8+1)];
}
        r1 = sp + -16;
        r2 = r1 >> 2;
        heap32[(fp+-4)] = 0;
        heap32[(r2+1)] = 0;
        heap32[(r2+2)] = 0;
        heap32[(r2+3)] = 0;
        r2 = (r9 - r6)|0;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r2;
        heap32[(g0+2)] = r1;
        _ZN20btAlignedObjectArrayI16btBroadphasePairE6resizeEiRKS0_(i7);
break;
case 53:
        r8 = _2E_str314;
        r0 = _2E_str18;
        heap32[(g0)] = r8;
        heap32[(g0+1)] = r0;
        heap32[(g0+2)] = 469;
        _assert(i7);
break;
}
}
        return;
}

function _ZN16btDbvtBroadphase7setAabbEP17btBroadphaseProxyRK9btVector3S4_P12btDispatcher(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var r12;
        var f0;
        var f1;
        var f2;
        var f3;
        var f4;
        var f5;
        var f6;
        var f7;
        var f8;
        var f9;
        var f10;
var __label__ = 0;
        i7 = sp + -88;var g0 = i7>>2; // save stack
        r0 = heap32[(fp+2)];
        r0 = r0 >> 2;
        f0 = heapFloat[(r0)];
        r1 = sp + -32;
        heapFloat[(fp+-8)] = f0;
        f1 = heapFloat[(r0+1)];
        r2 = r1 >> 2;
        heapFloat[(r2+1)] = f1;
        f2 = heapFloat[(r0+2)];
        heapFloat[(r2+2)] = f2;
        f3 = heapFloat[(r0+3)];
        r3 = heap32[(fp+3)];
        r3 = r3 >> 2;
        heapFloat[(r2+3)] = f3;
        f4 = heapFloat[(r3)];
        heapFloat[(r2+4)] = f4;
        f5 = heapFloat[(r3+1)];
        heapFloat[(r2+5)] = f5;
        f6 = heapFloat[(r3+2)];
        heapFloat[(r2+6)] = f6;
        f7 = heapFloat[(r3+3)];
        r4 = heap32[(fp+1)];
        r5 = heap32[(fp)];
        r6 = r4 >> 2;
        heapFloat[(r2+7)] = f7;
        r7 = heap32[(r6+15)];
_1: do {
        if(r7 !=2) //_LBB109_9
{
        r2 = r5 >> 2;
        r8 = heap32[(r2+32)];
        r8 = (r8 + 1)|0;
        heap32[(r2+32)] = r8;
        r8 = heap32[(r6+12)];
        r7 = r8 >> 2;
        f8 = heapFloat[(r7)];
if(!(f8 >f4)) //_LBB109_24
{
        f8 = heapFloat[(r7+4)];
if(!(f8 <f0)) //_LBB109_24
{
        f8 = heapFloat[(r7+1)];
if(!(f8 >f5)) //_LBB109_24
{
        f8 = heapFloat[(r7+5)];
if(!(f8 <f1)) //_LBB109_24
{
        f8 = heapFloat[(r7+2)];
if(!(f8 >f6)) //_LBB109_24
{
        f8 = heapFloat[(r7+6)];
if(!(f8 <f2)) //_LBB109_24
{
        f0 = heapFloat[(r6+4)];
        f1 = heapFloat[(r6+8)];
        f1 = f1-f0;
        f2 =                       0.5;
        f3 = heapFloat[(r6+5)];
        f4 = heapFloat[(r6+9)];
        f4 = f4-f3;
        f5 = heapFloat[(r0+1)];
        f6 = heapFloat[(r0)];
        f1 = f1*f2;
        f7 = heapFloat[(r2+25)];
        f8 = heapFloat[(r0+2)];
        f9 = heapFloat[(r6+6)];
        f10 = heapFloat[(r6+10)];
        r7 = sp + -48;
        f10 = f10-f9;
        f4 = f4*f2;
        f1 = f1*f7;
        f2 = f10*f2;
        f4 = f4*f7;
        r9 = r7 >> 2;
        heapFloat[(fp+-12)] = f1;
        f2 = f2*f7;
        heapFloat[(r9+1)] = f4;
        heapFloat[(r9+2)] = f2;
        heap32[(r9+3)] = 0;
        f0 = f6-f0;
        f6 =                         0;
if(!(f0 >=f6)) //_LBB109_17
{
        f0 = -f1;
        heapFloat[(fp+-12)] = f0;
}
        f0 = f5-f3;
if(!(f0 >=f6)) //_LBB109_19
{
        f0 = -f4;
        heapFloat[(r9+1)] = f0;
}
        f0 = f8-f9;
if(!(f0 >=f6)) //_LBB109_21
{
        f0 = -f2;
        heapFloat[(r9+2)] = f0;
}
        r9 = (r5 + 4)|0;
        heap32[(g0)] = r9;
        heap32[(g0+1)] = r8;
        heap32[(g0+2)] = r1;
        heap32[(g0+3)] = r7;
        heap32[(g0+4)] = 1028443341;
        _ZN6btDbvt6updateEP10btDbvtNodeR12btDbvtAabbMmRK9btVector3f(i7);
        r8 = r_g0;
        if(r8 !=0) //_LBB109_23
{
        r7 = heap32[(r2+33)];
        r8 = 1;
        r7 = (r7 + 1)|0;
        heap32[(r2+33)] = r7;
break _1;
}
else{
        r8 = 0;
break _1;
}
}
}
}
}
}
}
        r1 = (r5 + 4)|0;
        heap32[(g0)] = r1;
        heap32[(g0+1)] = r8;
        _ZL10removeleafP6btDbvtP10btDbvtNode(i7);
        r9 = r_g0;
_23: do {
        if(r9 !=0) //_LBB109_26
{
        r11 = heap32[(r2+3)];
        if(r11 <0) //_LBB109_31
{
        r10 = heap32[(r2+1)];
}
else{
        r12 = -1;
_28: while(true){
        r10 = r9;
        r12 = (r12 + 1)|0;
        if(r11 >r12) //_LBB109_30
{
        r9 = r10 >> 2;
        r9 = heap32[(r9+8)];
        if(r9 ==0) //_LBB109_29
{
break _23;
}
}
else{
break _23;
}
}
}
}
else{
        r10 = 0;
}
} while(0);
        heapFloat[(r7)] = f0;
        heapFloat[(r7+1)] = f1;
        heapFloat[(r7+2)] = f2;
        heapFloat[(r7+3)] = f3;
        heapFloat[(r7+4)] = f4;
        heapFloat[(r7+5)] = f5;
        heapFloat[(r7+6)] = f6;
        heapFloat[(r7+7)] = f7;
        heap32[(g0)] = r1;
        heap32[(g0+1)] = r10;
        heap32[(g0+2)] = r8;
        _ZL10insertleafP6btDbvtP10btDbvtNodeS2_(i7);
        r1 = heap32[(r2+33)];
        r8 = 1;
        r1 = (r1 + 1)|0;
        heap32[(r2+33)] = r1;
}
else{
        r1 = heap32[(r6+12)];
        r7 = (r5 + 44)|0;
        heap32[(g0)] = r7;
        heap32[(g0+1)] = r1;
        r7 = r5 >> 2;
        _ZL10removeleafP6btDbvtP10btDbvtNode(i7);
        r8 = heap32[(r7+12)];
if(!(r8 ==0)) //_LBB109_3
{
        r9 = gNumAlignedFree;
        r9 = r9 >> 2;
        r10 = heap32[(r9)];
        r10 = (r10 + 1)|0;
        r8 = r8 >> 2;
        heap32[(r9)] = r10;
        r8 = heap32[(r8+-1)];
        heap32[(g0)] = r8;
        free(i7);
}
        heap32[(r7+12)] = r1;
        r1 = heap32[(r7+14)];
        r1 = (r1 + -1)|0;
        heap32[(r7+14)] = r1;
        r1 = heap32[(r7+2)];
        r8 = (r5 + 4)|0;
        if(r1 ==0) //_LBB109_5
{
        r1 = gNumAlignedAllocs;
        r1 = r1 >> 2;
        r9 = heap32[(r1)];
        r9 = (r9 + 1)|0;
        heap32[(r1)] = r9;
        heap32[(g0)] = 63;
        malloc(i7);
        r1 = r_g0;
        if(r1 !=0) //_LBB109_7
{
        r9 = 0;
        r10 = (r1 + 4)|0;
        r9 = (r9 - r10)|0;
        r9 = r9 & 15;
        r9 = (r1 + r9)|0;
        r10 = (r9 + 4)|0;
        r9 = r9 >> 2;
        heap32[(r9)] = r1;
        r1 = r10;
}
}
else{
        heap32[(r7+2)] = 0;
}
        r9 = r1 >> 2;
        heap32[(r9+8)] = 0;
        heap32[(r9+9)] = r4;
        heap32[(r9+10)] = 0;
        heap32[(r9)] = heap32[(fp+-8)];
        heap32[(r9+1)] = heap32[(r2+1)];
        heap32[(r9+2)] = heap32[(r2+2)];
        heap32[(r9+3)] = heap32[(r2+3)];
        heapFloat[(r9+4)] = f4;
        heapFloat[(r9+5)] = f5;
        heapFloat[(r9+6)] = f6;
        heapFloat[(r9+7)] = f7;
        r2 = heap32[(r7+1)];
        heap32[(g0)] = r8;
        heap32[(g0+1)] = r2;
        heap32[(g0+2)] = r1;
        _ZL10insertleafP6btDbvtP10btDbvtNodeS2_(i7);
        r2 = heap32[(r7+4)];
        r2 = (r2 + 1)|0;
        r8 = 1;
        heap32[(r7+4)] = r2;
        heap32[(r6+12)] = r1;
}
} while(0);
        r1 = heap32[(r6+13)];
        r2 = heap32[(r6+14)];
        if(r1 ==0) //_LBB109_35
{
        r1 = heap32[(r6+15)];
        r1 = r1 << 2;
        r1 = (r5 + r1)|0;
        r1 = r1 >> 2;
        heap32[(r1+21)] = r2;
}
else{
        r1 = r1 >> 2;
        heap32[(r1+14)] = r2;
}
        r1 = heap32[(r6+14)];
if(!(r1 ==0)) //_LBB109_38
{
        r1 = r1 >> 2;
        r2 = heap32[(r6+13)];
        heap32[(r1+13)] = r2;
}
        heap32[(r6+4)] = heap32[(r0)];
        heap32[(r6+5)] = heap32[(r0+1)];
        heap32[(r6+6)] = heap32[(r0+2)];
        heap32[(r6+7)] = heap32[(r0+3)];
        heap32[(r6+8)] = heap32[(r3)];
        heap32[(r6+9)] = heap32[(r3+1)];
        heap32[(r6+10)] = heap32[(r3+2)];
        r0 = r5 >> 2;
        heap32[(r6+11)] = heap32[(r3+3)];
        r1 = heap32[(r0+26)];
        heap32[(r6+15)] = r1;
        r1 = heap32[(r0+26)];
        r1 = r1 << 2;
        r1 = (r5 + r1)|0;
        r1 = r1 >> 2;
        heap32[(r6+13)] = 0;
        r2 = heap32[(r1+21)];
        heap32[(r6+14)] = r2;
        r2 = heap32[(r1+21)];
if(!(r2 ==0)) //_LBB109_40
{
        r2 = r2 >> 2;
        heap32[(r2+13)] = r4;
}
        r2 = r8 & 1;
        heap32[(r1+21)] = r4;
if(!(r2 ==0)) //_LBB109_43
{
        r1 = 1;
        heap8[r5+154] = r1;
        r1 = heapU8[r5+153];
if(!(r1 !=0)) //_LBB109_43
{
        r1 = _ZTV18btDbvtTreeCollider;
        r2 = sp + -64;
        r1 = (r1 + 8)|0;
        r3 = r2 >> 2;
        heap32[(fp+-16)] = r1;
        heap32[(r3+1)] = r5;
        r1 = heap32[(r6+12)];
        r3 = heap32[(r0+11)];
        r4 = (r5 + 44)|0;
        heap32[(g0)] = r4;
        heap32[(g0+1)] = r3;
        heap32[(g0+2)] = r1;
        heap32[(g0+3)] = r2;
        _ZN6btDbvt24collideTTpersistentStackEPK10btDbvtNodeS2_RNS_8ICollideE(i7);
        r1 = heap32[(r6+12)];
        r0 = heap32[(r0+1)];
        r3 = (r5 + 4)|0;
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r0;
        heap32[(g0+2)] = r1;
        heap32[(g0+3)] = r2;
        _ZN6btDbvt24collideTTpersistentStackEPK10btDbvtNodeS2_RNS_8ICollideE(i7);
}
}
        return;
}

function _ZN16btDbvtBroadphaseD0Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV16btDbvtBroadphase;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        r1 = heapU8[r0+152];
if(!(r1 ==0)) //_LBB110_3
{
        r1 = heap32[(r2+24)];
        r3 = r1 >> 2;
        r3 = heap32[(r3)];
        r3 = r3 >> 2;
        r3 = heap32[(r3)];
        heap32[(g0)] = r1;
        __FUNCTION_TABLE__[(r3)>>2](i7);
        r1 = heap32[(r2+24)];
if(!(r1 ==0)) //_LBB110_3
{
        r2 = gNumAlignedFree;
        r2 = r2 >> 2;
        r3 = heap32[(r2)];
        r3 = (r3 + 1)|0;
        r1 = r1 >> 2;
        heap32[(r2)] = r3;
        r1 = heap32[(r1+-1)];
        heap32[(g0)] = r1;
        free(i7);
}
}
        r1 = (r0 + 4)|0;
if(!(r1 ==0)) //_LBB110_15
{
        r2 = (r1 + 80)|0;
if(!(r2 ==r1)) //_LBB110_15
{
        r1 = (r0 + 44)|0;
        r2 = 0;
_8: while(true){
        r3 = (r0 + r2)|0;
        r4 = r3 >> 2;
        r5 = heap32[(r4+11)];
if(!(r5 ==0)) //_LBB110_8
{
        r6 = (r1 + r2)|0;
        heap32[(g0)] = r6;
        heap32[(g0+1)] = r5;
        _ZL17recursedeletenodeP6btDbvtP10btDbvtNode(i7);
}
        r5 = heap32[(r4+12)];
if(!(r5 ==0)) //_LBB110_10
{
        r6 = gNumAlignedFree;
        r6 = r6 >> 2;
        r7 = heap32[(r6)];
        r7 = (r7 + 1)|0;
        r5 = r5 >> 2;
        heap32[(r6)] = r7;
        r5 = heap32[(r5+-1)];
        heap32[(g0)] = r5;
        free(i7);
}
        heap32[(r4+12)] = 0;
        heap32[(r4+13)] = -1;
        r5 = heap32[(r4+19)];
if(!(r5 ==0)) //_LBB110_14
{
        r6 = heapU8[r3+80];
if(!(r6 ==0)) //_LBB110_13
{
        r6 = gNumAlignedFree;
        r6 = r6 >> 2;
        r7 = heap32[(r6)];
        r7 = (r7 + 1)|0;
        r5 = r5 >> 2;
        heap32[(r6)] = r7;
        r5 = heap32[(r5+-1)];
        heap32[(g0)] = r5;
        free(i7);
}
        heap32[(r4+19)] = 0;
}
        r5 = 1;
        heap32[(r4+15)] = 0;
        heap8[r3+80] = r5;
        heap32[(r4+19)] = 0;
        r2 = (r2 + -40)|0;
        heap32[(r4+17)] = 0;
        heap32[(r4+18)] = 0;
        if(r2 !=-80) //_LBB110_6
{
continue _8;
}
else{
break _8;
}
}
}
}
        heap32[(g0)] = r0;
        _ZdlPv(i7);
        return;
}

function _ZN16btDbvtBroadphaseD1Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV16btDbvtBroadphase;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        r1 = heapU8[r0+152];
if(!(r1 ==0)) //_LBB111_3
{
        r1 = heap32[(r2+24)];
        r3 = r1 >> 2;
        r3 = heap32[(r3)];
        r3 = r3 >> 2;
        r3 = heap32[(r3)];
        heap32[(g0)] = r1;
        __FUNCTION_TABLE__[(r3)>>2](i7);
        r1 = heap32[(r2+24)];
if(!(r1 ==0)) //_LBB111_3
{
        r2 = gNumAlignedFree;
        r2 = r2 >> 2;
        r3 = heap32[(r2)];
        r3 = (r3 + 1)|0;
        r1 = r1 >> 2;
        heap32[(r2)] = r3;
        r1 = heap32[(r1+-1)];
        heap32[(g0)] = r1;
        free(i7);
}
}
        r1 = (r0 + 4)|0;
if(!(r1 ==0)) //_LBB111_15
{
        r2 = (r1 + 80)|0;
if(!(r2 ==r1)) //_LBB111_15
{
        r1 = (r0 + 44)|0;
        r2 = 0;
_8: while(true){
        r3 = (r0 + r2)|0;
        r4 = r3 >> 2;
        r5 = heap32[(r4+11)];
if(!(r5 ==0)) //_LBB111_8
{
        r6 = (r1 + r2)|0;
        heap32[(g0)] = r6;
        heap32[(g0+1)] = r5;
        _ZL17recursedeletenodeP6btDbvtP10btDbvtNode(i7);
}
        r5 = heap32[(r4+12)];
if(!(r5 ==0)) //_LBB111_10
{
        r6 = gNumAlignedFree;
        r6 = r6 >> 2;
        r7 = heap32[(r6)];
        r7 = (r7 + 1)|0;
        r5 = r5 >> 2;
        heap32[(r6)] = r7;
        r5 = heap32[(r5+-1)];
        heap32[(g0)] = r5;
        free(i7);
}
        heap32[(r4+12)] = 0;
        heap32[(r4+13)] = -1;
        r5 = heap32[(r4+19)];
if(!(r5 ==0)) //_LBB111_14
{
        r6 = heapU8[r3+80];
if(!(r6 ==0)) //_LBB111_13
{
        r6 = gNumAlignedFree;
        r6 = r6 >> 2;
        r7 = heap32[(r6)];
        r7 = (r7 + 1)|0;
        r5 = r5 >> 2;
        heap32[(r6)] = r7;
        r5 = heap32[(r5+-1)];
        heap32[(g0)] = r5;
        free(i7);
}
        heap32[(r4+19)] = 0;
}
        r5 = 1;
        heap32[(r4+15)] = 0;
        heap8[r3+80] = r5;
        heap32[(r4+19)] = 0;
        r2 = (r2 + -40)|0;
        heap32[(r4+17)] = 0;
        heap32[(r4+18)] = 0;
        if(r2 !=-80) //_LBB111_6
{
continue _8;
}
else{
break _8;
}
}
}
}
        return;
}

function _ZN12btDispatcherD0Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV12btDispatcher;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        heap32[(g0)] = r0;
        _ZdlPv(i7);
        return;
}

function _ZN12btDispatcherD1Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV12btDispatcher;
        r0 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r0)] = r1;
        return;
}

function _ZNK14btQuantizedBvh8quantizeEPtRK9btVector3i(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var f0;
        var f1;
        var f2;
        var f3;
        var f4;
        var f5;
var __label__ = 0;
        i7 = sp + -16;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = heapU8[r0+60];
        if(r1 !=0) //_LBB114_2
{
        r1 = heap32[(fp+2)];
        r1 = r1 >> 2;
        f0 = heapFloat[(r1)];
        r0 = r0 >> 2;
        f1 = heapFloat[(r0+5)];
        if(f0 <=f1) //_LBB114_4
{
        f1 = heapFloat[(r1+1)];
        f2 = heapFloat[(r0+6)];
        if(f1 <=f2) //_LBB114_6
{
        f2 = heapFloat[(r1+2)];
        f3 = heapFloat[(r0+7)];
        if(f2 <=f3) //_LBB114_8
{
        f3 = heapFloat[(r0+1)];
        if(f0 >=f3) //_LBB114_10
{
        f4 = heapFloat[(r0+2)];
        if(f1 >=f4) //_LBB114_12
{
        f5 = heapFloat[(r0+3)];
        if(f2 >=f5) //_LBB114_14
{
        r1 = heap32[(fp+1)];
        r2 = heap32[(fp+3)];
        f2 = f2-f5;
        f5 = heapFloat[(r0+11)];
        f1 = f1-f4;
        f4 = heapFloat[(r0+10)];
        f0 = f0-f3;
        f3 = heapFloat[(r0+9)];
        f2 = f2*f5;
        f1 = f1*f4;
        f0 = f0*f3;
        if(r2 ==0) //_LBB114_16
{
        r0 = Math.floor(f0);
        r2 = Math.floor(f1);
        r0 = r0 & 65534;
        r3 = Math.floor(f2);
        r2 = r2 & 65534;
        heap16[(r1)>>1] = r0;
        r0 = r3 & 65534;
}
else{
        f3 =                         1;
        f0 = f0+f3;
        f1 = f1+f3;
        r0 = Math.floor(f0);
        f0 = f2+f3;
        r2 = Math.floor(f1);
        r0 = r0 | 1;
        r3 = Math.floor(f0);
        r2 = r2 | 1;
        heap16[(r1)>>1] = r0;
        r0 = r3 | 1;
}
        heap16[(r1+2)>>1] = r2;
        heap16[(r1+4)>>1] = r0;
        return;
}
else{
        r0 = _2E_str9;
        r1 = _2E_str313;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 361;
        _assert(i7);
}
}
else{
        r0 = _2E_str820;
        r1 = _2E_str313;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 360;
        _assert(i7);
}
}
else{
        r0 = _2E_str717;
        r1 = _2E_str313;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 359;
        _assert(i7);
}
}
else{
        r0 = _2E_str616;
        r1 = _2E_str313;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 357;
        _assert(i7);
}
}
else{
        r0 = _2E_str515;
        r1 = _2E_str313;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 356;
        _assert(i7);
}
}
else{
        r0 = _2E_str414;
        r1 = _2E_str313;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 355;
        _assert(i7);
}
}
else{
        r0 = _2E_str212;
        r1 = _2E_str313;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 353;
        _assert(i7);
}
}

function _ZN20btAlignedObjectArrayI18btQuantizedBvhNodeE7reserveEi(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = heap32[(fp+1)];
        r2 = r0 >> 2;
        r3 = heap32[(r2+2)];
if(!(r3 >=r1)) //_LBB115_16
{
        if(r1 !=0) //_LBB115_3
{
        r3 = gNumAlignedAllocs;
        r3 = r3 >> 2;
        r4 = heap32[(r3)];
        r5 = r1 << 4;
        r4 = (r4 + 1)|0;
        r5 = r5 | 3;
        heap32[(r3)] = r4;
        r3 = (r5 + 16)|0;
        heap32[(g0)] = r3;
        malloc(i7);
        r3 = r_g0;
        if(r3 !=0) //_LBB115_5
{
        r4 = 0;
        r5 = (r3 + 4)|0;
        r4 = (r4 - r5)|0;
        r4 = r4 & 15;
        r4 = (r3 + r4)|0;
        r5 = (r4 + 4)|0;
        r4 = r4 >> 2;
        heap32[(r4)] = r3;
        r3 = r5;
}
}
else{
        r3 = 0;
}
        r4 = heap32[(r2+1)];
        r5 = (r0 + 12)|0;
        if(r4 <1) //_LBB115_8
{
        r4 = r5 >> 2;
        r7 = heap32[(r4)];
}
else{
        r6 = 0;
_11: while(true){
        r7 = r5 >> 2;
        r7 = heap32[(r7)];
        r8 = r6 << 4;
        r9 = heapU16[(r7+r8)>>1];
        r10 = (r7 + r8)|0;
        heap16[(r3+r8)>>1] = r9;
        r8 = (r3 + r8)|0;
        r9 = heapU16[(r10+2)>>1];
        heap16[(r8+2)>>1] = r9;
        r9 = heapU16[(r10+4)>>1];
        heap16[(r8+4)>>1] = r9;
        r9 = heapU16[(r10+6)>>1];
        heap16[(r8+6)>>1] = r9;
        r9 = heapU16[(r10+8)>>1];
        heap16[(r8+8)>>1] = r9;
        r9 = heapU16[(r10+10)>>1];
        r10 = r10 >> 2;
        heap16[(r8+10)>>1] = r9;
        r6 = (r6 + 1)|0;
        r8 = r8 >> 2;
        r9 = heap32[(r10+3)];
        heap32[(r8+3)] = r9;
if(!(r4 !=r6)) //_LBB115_9
{
break _11;
}
}
        r5 = (r0 + 12)|0;
}
if(!(r7 ==0)) //_LBB115_15
{
        r4 = heapU8[r0+16];
if(!(r4 ==0)) //_LBB115_14
{
        r4 = gNumAlignedFree;
        r4 = r4 >> 2;
        r6 = heap32[(r4)];
        r6 = (r6 + 1)|0;
        r7 = r7 >> 2;
        heap32[(r4)] = r6;
        r4 = heap32[(r7+-1)];
        heap32[(g0)] = r4;
        free(i7);
}
        r4 = r5 >> 2;
        heap32[(r4)] = 0;
}
        r4 = 1;
        r5 = r5 >> 2;
        heap8[r0+16] = r4;
        heap32[(r5)] = r3;
        heap32[(r2+2)] = r1;
}
        return;
}

function _ZN28btHashedOverlappingPairCache23getOverlappingPairArrayEv(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = (r0 + 4)|0;
        r_g0 = r0;
        return;
}

function _ZN28btHashedOverlappingPairCache24setOverlapFilterCallbackEP23btOverlapFilterCallback(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = r0 >> 2;
        r1 = heap32[(fp+1)];
        heap32[(r0+6)] = r1;
        return;
}

function _ZN28btHashedOverlappingPairCache18hasDeferredRemovalEv(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = 0;
        r_g0 = r0;
        return;
}

function _ZN28btHashedOverlappingPairCache28setInternalGhostPairCallbackEP25btOverlappingPairCallback(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = r0 >> 2;
        r1 = heap32[(fp+1)];
        heap32[(r0+18)] = r1;
        return;
}

function _ZN28btHashedOverlappingPairCache20cleanOverlappingPairER16btBroadphasePairP12btDispatcher(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp+1)];
        r0 = r0 >> 2;
        r1 = heap32[(r0+2)];
if(!(r1 ==0)) //_LBB120_2
{
        r2 = heap32[(fp+2)];
        r3 = r1 >> 2;
        r3 = heap32[(r3)];
        r3 = r3 >> 2;
        r3 = heap32[(r3)];
        heap32[(g0)] = r1;
        __FUNCTION_TABLE__[(r3)>>2](i7);
        r1 = r2 >> 2;
        r1 = heap32[(r1)];
        r1 = r1 >> 2;
        r1 = heap32[(r1+13)];
        r3 = heap32[(r0+2)];
        heap32[(g0)] = r2;
        heap32[(g0+1)] = r3;
        __FUNCTION_TABLE__[(r1)>>2](i7);
        heap32[(r0+2)] = 0;
}
        return;
}

function _ZZN28btHashedOverlappingPairCache19cleanProxyFromPairsEP17btBroadphaseProxyP12btDispatcherEN17CleanPairCallbackD1Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTVZN28btHashedOverlappingPairCache19cleanProxyFromPairsEP17btBroadphaseProxyP12btDispatcherE17CleanPairCallback;
        r0 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r0)] = r1;
        return;
}

function _ZZN28btHashedOverlappingPairCache19cleanProxyFromPairsEP17btBroadphaseProxyP12btDispatcherEN17CleanPairCallbackD0Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTVZN28btHashedOverlappingPairCache19cleanProxyFromPairsEP17btBroadphaseProxyP12btDispatcherE17CleanPairCallback;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        heap32[(g0)] = r0;
        _ZdlPv(i7);
        return;
}

function _ZZN28btHashedOverlappingPairCache19cleanProxyFromPairsEP17btBroadphaseProxyP12btDispatcherEN17CleanPairCallback14processOverlapER16btBroadphasePair(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
var __label__ = 0;
        i7 = sp + -16;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = r0 >> 2;
        r1 = heap32[(fp+1)];
        r2 = heap32[(r0+1)];
        r3 = r1 >> 2;
        r4 = heap32[(r3)];
        if(r4 ==r2) //_LBB123_2
{
__label__ = 2;
}
else{
        r3 = heap32[(r3+1)];
        if(r3 !=r2) //_LBB123_3
{
__label__ = 3;
}
else{
__label__ = 2;
}
}
if (__label__ == 2){
        r2 = heap32[(r0+2)];
        r3 = r2 >> 2;
        r3 = heap32[(r3)];
        r3 = r3 >> 2;
        r3 = heap32[(r3+8)];
        r0 = heap32[(r0+3)];
        heap32[(g0)] = r2;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = r0;
        __FUNCTION_TABLE__[(r3)>>2](i7);
}
        r0 = 0;
        r_g0 = r0;
        return;
}

function _ZZN28btHashedOverlappingPairCache37removeOverlappingPairsContainingProxyEP17btBroadphaseProxyP12btDispatcherEN18RemovePairCallbackD1Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTVZN28btHashedOverlappingPairCache37removeOverlappingPairsContainingProxyEP17btBroadphaseProxyP12btDispatcherE18RemovePairCallback;
        r0 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r0)] = r1;
        return;
}

function _ZZN28btHashedOverlappingPairCache37removeOverlappingPairsContainingProxyEP17btBroadphaseProxyP12btDispatcherEN18RemovePairCallbackD0Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTVZN28btHashedOverlappingPairCache37removeOverlappingPairsContainingProxyEP17btBroadphaseProxyP12btDispatcherE18RemovePairCallback;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        heap32[(g0)] = r0;
        _ZdlPv(i7);
        return;
}

function _ZZN28btHashedOverlappingPairCache37removeOverlappingPairsContainingProxyEP17btBroadphaseProxyP12btDispatcherEN18RemovePairCallback14processOverlapER16btBroadphasePair(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = r0 >> 2;
        r1 = heap32[(fp+1)];
        r0 = heap32[(r0+1)];
        r1 = r1 >> 2;
        r2 = heap32[(r1)];
        if(r2 ==r0) //_LBB126_2
{
        r0 = 1;
        r_g0 = r0;
        return;
}
else{
        r1 = heap32[(r1+1)];
        r0 = r1 == r0;
        r0 = r0 & 1;
        r_g0 = r0;
        return;
}
}

function _ZN28btHashedOverlappingPairCache26getOverlappingPairArrayPtrEv(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = r0 >> 2;
        r0 = heap32[(r0+4)];
        r_g0 = r0;
        return;
}

function _ZNK28btHashedOverlappingPairCache26getOverlappingPairArrayPtrEv(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = r0 >> 2;
        r0 = heap32[(r0+4)];
        r_g0 = r0;
        return;
}

function _ZNK28btHashedOverlappingPairCache22getNumOverlappingPairsEv(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = r0 >> 2;
        r0 = heap32[(r0+2)];
        r_g0 = r0;
        return;
}

function _ZN28btHashedOverlappingPairCache26processAllOverlappingPairsEP17btOverlapCallbackP12btDispatcher(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
var __label__ = 0;
        i7 = sp + -16;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = heap32[(fp+1)];
        r2 = heap32[(fp+2)];
        r3 = 0;
        r4 = r3;
_1: while(true){
        r5 = r4 << 2;
_3: while(true){
        r6 = r0 >> 2;
        r7 = heap32[(r6+2)];
        if(r7 >r4) //_LBB130_1
{
        r7 = r1 >> 2;
        r7 = heap32[(r7)];
        r7 = r7 >> 2;
        r8 = heap32[(r6+4)];
        r7 = heap32[(r7+2)];
        r9 = (r8 + r3)|0;
        heap32[(g0)] = r1;
        heap32[(g0+1)] = r9;
        __FUNCTION_TABLE__[(r7)>>2](i7);
        r7 = r_g0;
        if(r7 ==0) //_LBB130_3
{
break _3;
}
else{
        r7 = r5 << 2;
        r6 = heap32[(r6)];
        r7 = (r8 + r7)|0;
        r7 = r7 >> 2;
        r6 = r6 >> 2;
        r6 = heap32[(r6+3)];
        r8 = heap32[(r7+1)];
        r7 = heap32[(r7)];
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r7;
        heap32[(g0+2)] = r8;
        heap32[(g0+3)] = r2;
        r7 = gOverlappingPairs;
        r7 = r7 >> 2;
        __FUNCTION_TABLE__[(r6)>>2](i7);
        r6 = heap32[(r7)];
        r6 = (r6 + -1)|0;
        heap32[(r7)] = r6;
continue _3;
}
}
else{
break _1;
}
}
        r4 = (r4 + 1)|0;
        r3 = (r3 + 16)|0;
continue _1;
}
        return;
}

function _ZN28btHashedOverlappingPairCache37removeOverlappingPairsContainingProxyEP17btBroadphaseProxyP12btDispatcher(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
var __label__ = 0;
        i7 = sp + -24;var g0 = i7>>2; // save stack
        r0 = _ZTVZN28btHashedOverlappingPairCache37removeOverlappingPairsContainingProxyEP17btBroadphaseProxyP12btDispatcherE18RemovePairCallback;
        r1 = sp + -8;
        r0 = (r0 + 8)|0;
        r2 = heap32[(fp)];
        r3 = r1 >> 2;
        r4 = heap32[(fp+1)];
        heap32[(fp+-2)] = r0;
        r0 = r2 >> 2;
        heap32[(r3+1)] = r4;
        r0 = heap32[(r0)];
        r0 = r0 >> 2;
        r0 = heap32[(r0+12)];
        r3 = heap32[(fp+2)];
        heap32[(g0)] = r2;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = r3;
        __FUNCTION_TABLE__[(r0)>>2](i7);
        return;
}

function _ZN28btHashedOverlappingPairCache19cleanProxyFromPairsEP17btBroadphaseProxyP12btDispatcher(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
var __label__ = 0;
        i7 = sp + -32;var g0 = i7>>2; // save stack
        r0 = _ZTVZN28btHashedOverlappingPairCache19cleanProxyFromPairsEP17btBroadphaseProxyP12btDispatcherE17CleanPairCallback;
        r1 = sp + -16;
        r0 = (r0 + 8)|0;
        r2 = r1 >> 2;
        r3 = heap32[(fp+1)];
        heap32[(fp+-4)] = r0;
        r0 = heap32[(fp)];
        heap32[(r2+1)] = r3;
        r3 = heap32[(fp+2)];
        heap32[(r2+2)] = r0;
        r4 = r0 >> 2;
        heap32[(r2+3)] = r3;
        r2 = heap32[(r4)];
        r2 = r2 >> 2;
        r2 = heap32[(r2+12)];
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = r3;
        __FUNCTION_TABLE__[(r2)>>2](i7);
        return;
}

function _ZN28btHashedOverlappingPairCacheD0Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV28btHashedOverlappingPairCache;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        r1 = heap32[(r2+16)];
if(!(r1 ==0)) //_LBB133_4
{
        r3 = heapU8[r0+68];
if(!(r3 ==0)) //_LBB133_3
{
        r3 = gNumAlignedFree;
        r3 = r3 >> 2;
        r4 = heap32[(r3)];
        r4 = (r4 + 1)|0;
        r1 = r1 >> 2;
        heap32[(r3)] = r4;
        r1 = heap32[(r1+-1)];
        heap32[(g0)] = r1;
        free(i7);
}
        heap32[(r2+16)] = 0;
}
        r1 = 1;
        heap8[r0+68] = r1;
        heap32[(r2+16)] = 0;
        heap32[(r2+14)] = 0;
        heap32[(r2+15)] = 0;
        r3 = heap32[(r2+11)];
if(!(r3 ==0)) //_LBB133_8
{
        r4 = heapU8[r0+48];
if(!(r4 ==0)) //_LBB133_7
{
        r4 = gNumAlignedFree;
        r4 = r4 >> 2;
        r5 = heap32[(r4)];
        r5 = (r5 + 1)|0;
        r3 = r3 >> 2;
        heap32[(r4)] = r5;
        r3 = heap32[(r3+-1)];
        heap32[(g0)] = r3;
        free(i7);
}
        heap32[(r2+11)] = 0;
}
        heap8[r0+48] = r1;
        heap32[(r2+11)] = 0;
        heap32[(r2+9)] = 0;
        heap32[(r2+10)] = 0;
        r3 = heap32[(r2+4)];
if(!(r3 ==0)) //_LBB133_12
{
        r4 = heapU8[r0+20];
if(!(r4 ==0)) //_LBB133_11
{
        r4 = gNumAlignedFree;
        r4 = r4 >> 2;
        r5 = heap32[(r4)];
        r5 = (r5 + 1)|0;
        r3 = r3 >> 2;
        heap32[(r4)] = r5;
        r3 = heap32[(r3+-1)];
        heap32[(g0)] = r3;
        free(i7);
}
        heap32[(r2+4)] = 0;
}
        heap8[r0+20] = r1;
        heap32[(r2+4)] = 0;
        heap32[(r2+2)] = 0;
        heap32[(r2+3)] = 0;
        heap32[(g0)] = r0;
        _ZdlPv(i7);
        return;
}

function _ZN28btHashedOverlappingPairCacheD1Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV28btHashedOverlappingPairCache;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        r1 = heap32[(r2+16)];
if(!(r1 ==0)) //_LBB134_4
{
        r3 = heapU8[r0+68];
if(!(r3 ==0)) //_LBB134_3
{
        r3 = gNumAlignedFree;
        r3 = r3 >> 2;
        r4 = heap32[(r3)];
        r4 = (r4 + 1)|0;
        r1 = r1 >> 2;
        heap32[(r3)] = r4;
        r1 = heap32[(r1+-1)];
        heap32[(g0)] = r1;
        free(i7);
}
        heap32[(r2+16)] = 0;
}
        r1 = 1;
        heap8[r0+68] = r1;
        heap32[(r2+16)] = 0;
        heap32[(r2+14)] = 0;
        heap32[(r2+15)] = 0;
        r3 = heap32[(r2+11)];
if(!(r3 ==0)) //_LBB134_8
{
        r4 = heapU8[r0+48];
if(!(r4 ==0)) //_LBB134_7
{
        r4 = gNumAlignedFree;
        r4 = r4 >> 2;
        r5 = heap32[(r4)];
        r5 = (r5 + 1)|0;
        r3 = r3 >> 2;
        heap32[(r4)] = r5;
        r3 = heap32[(r3+-1)];
        heap32[(g0)] = r3;
        free(i7);
}
        heap32[(r2+11)] = 0;
}
        heap8[r0+48] = r1;
        heap32[(r2+11)] = 0;
        heap32[(r2+9)] = 0;
        heap32[(r2+10)] = 0;
        r3 = heap32[(r2+4)];
if(!(r3 ==0)) //_LBB134_12
{
        r4 = heapU8[r0+20];
if(!(r4 ==0)) //_LBB134_11
{
        r4 = gNumAlignedFree;
        r4 = r4 >> 2;
        r5 = heap32[(r4)];
        r5 = (r5 + 1)|0;
        r3 = r3 >> 2;
        heap32[(r4)] = r5;
        r3 = heap32[(r3+-1)];
        heap32[(g0)] = r3;
        free(i7);
}
        heap32[(r2+4)] = 0;
}
        heap8[r0+20] = r1;
        heap32[(r2+4)] = 0;
        heap32[(r2+2)] = 0;
        heap32[(r2+3)] = 0;
        return;
}

function _ZN28btHashedOverlappingPairCache18addOverlappingPairEP17btBroadphaseProxyS1_(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var r12;
        var r13;
        var r14;
var __label__ = 0;
        i7 = sp + -16;var g0 = i7>>2; // save stack
        r0 = gAddedPairs;
        r0 = r0 >> 2;
        r1 = heap32[(r0)];
        r2 = heap32[(fp)];
        r1 = (r1 + 1)|0;
        r3 = r2 >> 2;
        heap32[(r0)] = r1;
        r0 = heap32[(r3+6)];
        r1 = heap32[(fp+1)];
        r4 = heap32[(fp+2)];
        if(r0 ==0) //_LBB135_2
{
        r0 = heapU16[(r4+6)>>1];
        r5 = heapU16[(r1+4)>>1];
        r0 = r0 & r5;
        r0 = r0 & 65535;
        if(r0 ==0) //_LBB135_25
{
__label__ = 23;
}
else{
        r0 = heapU16[(r1+6)>>1];
        r5 = heapU16[(r4+4)>>1];
        r0 = r0 & r5;
        r0 = r0 & 65535;
        r5 = 0;
        r0 = r0 != r5;
        r0 = r0 & 1;
__label__ = 4;
}
}
else{
        r5 = r0 >> 2;
        r5 = heap32[(r5)];
        r5 = r5 >> 2;
        r5 = heap32[(r5+2)];
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = r4;
        __FUNCTION_TABLE__[(r5)>>2](i7);
        r0 = r_g0;
__label__ = 4;
}
if (__label__ == 4){
        r0 = r0 & 255;
if(!(r0 ==0)) //_LBB135_25
{
        r0 = r1 >> 2;
        r5 = r4 >> 2;
        r0 = heap32[(r0+3)];
        r5 = heap32[(r5+3)];
        r6 = r0 > r5 ? r1 : r4;
        r0 = r0 > r5 ? r4 : r1;
        r1 = r6 >> 2;
        r4 = heap32[(r1+3)];
        r5 = r0 >> 2;
        r7 = heap32[(r5+3)];
        r8 = r4 << 16;
        r8 = r8 | r7;
        r9 = r8 << 15;
        r9 = r9 ^ -1;
        r8 = (r8 + r9)|0;
        r9 = r8 >> 10;
        r8 = r9 ^ r8;
        r8 = (r8 * 9)|0;
        r9 = r8 >> 6;
        r8 = r9 ^ r8;
        r9 = r8 << 11;
        r9 = r9 ^ -1;
        r8 = (r8 + r9)|0;
        r9 = heap32[(r3+3)];
        r10 = r8 >> 16;
        r8 = r10 ^ r8;
        r10 = (r9 + -1)|0;
        r10 = r8 & r10;
        r11 = heap32[(r3+11)];
        r12 = r10 << 2;
        r11 = (r11 + r12)|0;
_8: while(true){
        r11 = r11 >> 2;
        r11 = heap32[(r11)];
        if(r11 ==-1) //_LBB135_14
{
__label__ = 13;
break _8;
}
else{
        r12 = heap32[(r3+4)];
        r13 = r11 << 4;
        r12 = (r12 + r13)|0;
        r13 = r12 >> 2;
        r14 = heap32[(r13)];
        r14 = r14 >> 2;
        r14 = heap32[(r14+3)];
if(!(r14 !=r7)) //_LBB135_6
{
        r13 = heap32[(r13+1)];
        r13 = r13 >> 2;
        r13 = heap32[(r13+3)];
if(!(r13 !=r4)) //_LBB135_6
{
__label__ = 10;
break _8;
}
}
        r12 = heap32[(r3+16)];
        r11 = r11 << 2;
        r11 = (r12 + r11)|0;
}
}
_14: do {
switch(__label__ ){//multiple entries
case 13:
        r4 = heap32[(r3+2)];
__label__ = 14;
break _14;
break;
case 10:
        r4 = heap32[(r3+2)];
        if(r4 >r11) //_LBB135_12
{
        if(r12 !=0) //_LBB135_24
{
__label__ = 22;
}
else{
__label__ = 14;
}
}
else{
        r0 = _2E_str222;
        r1 = _2E_str323;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 250;
        _assert(i7);
}
break;
}
} while(0);
if (__label__ == 14){
        r12 = (r2 + 4)|0;
        heap32[(g0)] = r12;
        _ZN20btAlignedObjectArrayI16btBroadphasePairE21expandNonInitializingEv(i7);
        r12 = r_g0;
        r7 = heap32[(r3+18)];
if(!(r7 ==0)) //_LBB135_17
{
        r11 = r7 >> 2;
        r11 = heap32[(r11)];
        r11 = r11 >> 2;
        r11 = heap32[(r11+2)];
        heap32[(g0)] = r7;
        heap32[(g0+1)] = r0;
        heap32[(g0+2)] = r6;
        __FUNCTION_TABLE__[(r11)>>2](i7);
}
        r7 = heap32[(r3+3)];
        if(r9 <r7) //_LBB135_19
{
        heap32[(g0)] = r2;
        _ZN28btHashedOverlappingPairCache10growTablesEv(i7);
        r10 = heap32[(r3+3)];
        r10 = (r10 + -1)|0;
        r10 = r10 & r8;
}
        r2 = heap32[(r5+3)];
        r1 = heap32[(r1+3)];
        if(r2 >=r1) //_LBB135_22
{
        r1 = r12 >> 2;
        heap32[(r1)] = r6;
        heap32[(r1+1)] = r0;
}
else{
        r1 = r12 >> 2;
        heap32[(r1)] = r0;
        heap32[(r1+1)] = r6;
}
        r0 = r12 >> 2;
        heap32[(r0+2)] = 0;
        heap32[(r0+3)] = 0;
        r0 = r10 << 2;
        r1 = heap32[(r3+11)];
        r1 = (r1 + r0)|0;
        r1 = r1 >> 2;
        r2 = r4 << 2;
        r5 = heap32[(r3+16)];
        r2 = (r5 + r2)|0;
        r1 = heap32[(r1)];
        r2 = r2 >> 2;
        heap32[(r2)] = r1;
        r1 = heap32[(r3+11)];
        r0 = (r1 + r0)|0;
        r0 = r0 >> 2;
        heap32[(r0)] = r4;
}
        r_g0 = r12;
        return;
}
}
        r0 = 0;
        r_g0 = r0;
        return;
}

function _ZN28btHashedOverlappingPairCache21removeOverlappingPairEP17btBroadphaseProxyS1_P12btDispatcher(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
var __label__ = 0;
        i7 = sp + -16;var g0 = i7>>2; // save stack
        r0 = gRemovePairs;
        r0 = r0 >> 2;
        r1 = heap32[(r0)];
        r2 = heap32[(fp+2)];
        r3 = heap32[(fp+1)];
        r1 = (r1 + 1)|0;
        r4 = r3 >> 2;
        heap32[(r0)] = r1;
        r0 = r2 >> 2;
        r1 = heap32[(r4+3)];
        r0 = heap32[(r0+3)];
        r4 = r1 > r0 ? r3 : r2;
        r0 = r1 > r0 ? r2 : r3;
        r1 = r4 >> 2;
        r1 = heap32[(r1+3)];
        r2 = r0 >> 2;
        r2 = heap32[(r2+3)];
        r3 = r1 << 16;
        r3 = r3 | r2;
        r5 = r3 << 15;
        r5 = r5 ^ -1;
        r3 = (r3 + r5)|0;
        r5 = r3 >> 10;
        r3 = r5 ^ r3;
        r3 = (r3 * 9)|0;
        r5 = r3 >> 6;
        r3 = r5 ^ r3;
        r5 = r3 << 11;
        r6 = heap32[(fp)];
        r5 = r5 ^ -1;
        r7 = r6 >> 2;
        r3 = (r3 + r5)|0;
        r5 = r3 >> 16;
        r8 = heap32[(r7+3)];
        r3 = r5 ^ r3;
        r5 = (r8 + -1)|0;
        r3 = r3 & r5;
        r5 = heap32[(r7+11)];
        r3 = r3 << 2;
        r8 = heap32[(fp+3)];
        r5 = (r5 + r3)|0;
_1: while(true){
        r5 = r5 >> 2;
        r5 = heap32[(r5)];
        if(r5 ==-1) //_LBB136_35
{
__label__ = 33;
break _1;
}
else{
        r9 = heap32[(r7+4)];
        r10 = r5 << 4;
        r9 = (r9 + r10)|0;
        r10 = r9 >> 2;
        r11 = heap32[(r10)];
        r11 = r11 >> 2;
        r11 = heap32[(r11+3)];
if(!(r11 !=r2)) //_LBB136_1
{
        r11 = heap32[(r10+1)];
        r11 = r11 >> 2;
        r11 = heap32[(r11+3)];
if(!(r11 !=r1)) //_LBB136_1
{
__label__ = 5;
break _1;
}
}
        r9 = heap32[(r7+16)];
        r5 = r5 << 2;
        r5 = (r9 + r5)|0;
continue _1;
}
}
if (__label__ == 5){
        r11 = heap32[(r7+2)];
        if(r11 >r5) //_LBB136_7
{
if(!(r9 ==0)) //_LBB136_35
{
        r5 = heap32[(r7)];
        r5 = r5 >> 2;
        r5 = heap32[(r5+8)];
        heap32[(g0)] = r6;
        heap32[(g0+1)] = r9;
        heap32[(g0+2)] = r8;
        __FUNCTION_TABLE__[(r5)>>2](i7);
        r5 = heap32[(r10)];
        r5 = r5 >> 2;
        r5 = heap32[(r5+3)];
        if(r5 ==r2) //_LBB136_10
{
        r2 = heap32[(r10+1)];
        r2 = r2 >> 2;
        r2 = heap32[(r2+3)];
        if(r2 ==r1) //_LBB136_12
{
        r1 = heap32[(r7+4)];
        r1 = (r9 - r1)|0;
        r1 = r1 >> 4;
        r2 = heap32[(r7+2)];
        if(r2 >r1) //_LBB136_14
{
        r2 = heap32[(r7+11)];
        r2 = (r2 + r3)|0;
        r2 = r2 >> 2;
        r3 = heap32[(r2)];
        if(r3 ==-1) //_LBB136_17
{
        r0 = _2E_str727;
        r1 = _2E_str121;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 304;
        _assert(i7);
}
else{
        r5 = heap32[(r10+3)];
        r6 = heap32[(r7+16)];
        if(r3 ==r1) //_LBB136_21
{
__label__ = 20;
}
else{
_22: while(true){
        r9 = r3;
        r3 = r9 << 2;
        r3 = (r6 + r3)|0;
        r10 = r3 >> 2;
        r3 = heap32[(r10)];
if(!(r3 !=r1)) //_LBB136_18
{
break _22;
}
}
        if(r9 ==-1) //_LBB136_21
{
__label__ = 20;
}
else{
        r2 = r1 << 2;
        r2 = (r6 + r2)|0;
        r2 = r2 >> 2;
        r2 = heap32[(r2)];
        heap32[(r10)] = r2;
__label__ = 21;
}
}
if (__label__ == 20){
        r3 = r1 << 2;
        r3 = (r6 + r3)|0;
        r3 = r3 >> 2;
        r3 = heap32[(r3)];
        heap32[(r2)] = r3;
}
        r2 = heap32[(r7+2)];
        r3 = heap32[(r7+18)];
if(!(r3 ==0)) //_LBB136_24
{
        r6 = r3 >> 2;
        r6 = heap32[(r6)];
        r6 = r6 >> 2;
        r6 = heap32[(r6+3)];
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r0;
        heap32[(g0+2)] = r4;
        heap32[(g0+3)] = r8;
        __FUNCTION_TABLE__[(r6)>>2](i7);
}
        r0 = (r2 + -1)|0;
        if(r0 !=r1) //_LBB136_26
{
        r3 = heap32[(r7+4)];
        r4 = r2 << 4;
        r3 = (r3 + r4)|0;
        r3 = r3 >> 2;
        r6 = heap32[(r3+-3)];
        r3 = heap32[(r3+-4)];
        r6 = r6 >> 2;
        r3 = r3 >> 2;
        r6 = heap32[(r6+3)];
        r3 = heap32[(r3+3)];
        r6 = r6 << 16;
        r3 = r3 | r6;
        r6 = r3 << 15;
        r6 = r6 ^ -1;
        r3 = (r3 + r6)|0;
        r6 = r3 >> 10;
        r3 = r6 ^ r3;
        r3 = (r3 * 9)|0;
        r6 = r3 >> 6;
        r3 = r6 ^ r3;
        r6 = r3 << 11;
        r6 = r6 ^ -1;
        r3 = (r3 + r6)|0;
        r6 = r3 >> 16;
        r8 = heap32[(r7+3)];
        r3 = r6 ^ r3;
        r6 = (r8 + -1)|0;
        r3 = r3 & r6;
        r6 = heap32[(r7+11)];
        r3 = r3 << 2;
        r6 = (r6 + r3)|0;
        r6 = r6 >> 2;
        r8 = heap32[(r6)];
        if(r8 ==-1) //_LBB136_29
{
        r0 = _2E_str727;
        r1 = _2E_str121;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 345;
        _assert(i7);
}
else{
        r9 = heap32[(r7+16)];
        if(r8 ==r0) //_LBB136_33
{
__label__ = 31;
}
else{
_38: while(true){
        r10 = r8;
        r8 = r10 << 2;
        r8 = (r9 + r8)|0;
        r11 = r8 >> 2;
        r8 = heap32[(r11)];
if(!(r8 !=r0)) //_LBB136_30
{
break _38;
}
}
        if(r10 ==-1) //_LBB136_33
{
__label__ = 31;
}
else{
        r2 = r2 << 2;
        r2 = (r9 + r2)|0;
        r2 = r2 >> 2;
        r2 = heap32[(r2+-1)];
        heap32[(r11)] = r2;
__label__ = 32;
}
}
if (__label__ == 31){
        r0 = r2 << 2;
        r0 = (r9 + r0)|0;
        r0 = r0 >> 2;
        r0 = heap32[(r0+-1)];
        heap32[(r6)] = r0;
}
        r0 = heap32[(r7+4)];
        r2 = (r0 + r4)|0;
        r4 = r1 << 4;
        r2 = r2 >> 2;
        r0 = (r0 + r4)|0;
        r4 = heap32[(r2+-4)];
        r0 = r0 >> 2;
        heap32[(r0)] = r4;
        r4 = heap32[(r2+-3)];
        heap32[(r0+1)] = r4;
        r4 = heap32[(r2+-2)];
        heap32[(r0+2)] = r4;
        r2 = heap32[(r2+-1)];
        heap32[(r0+3)] = r2;
        r0 = heap32[(r7+11)];
        r0 = (r0 + r3)|0;
        r0 = r0 >> 2;
        r2 = r1 << 2;
        r4 = heap32[(r7+16)];
        r2 = (r4 + r2)|0;
        r0 = heap32[(r0)];
        r2 = r2 >> 2;
        heap32[(r2)] = r0;
        r0 = heap32[(r7+11)];
        r0 = (r0 + r3)|0;
        r0 = r0 >> 2;
        heap32[(r0)] = r1;
}
}
        r0 = heap32[(r7+2)];
        r0 = (r0 + -1)|0;
        heap32[(r7+2)] = r0;
        r_g0 = r5;
        return;
}
}
else{
        r0 = _2E_str626;
        r1 = _2E_str121;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 300;
        _assert(i7);
}
}
else{
        r0 = _2E_str525;
        r3 = _2E_str121;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r3;
        heap32[(g0+2)] = 297;
        _assert(i7);
}
}
else{
        r0 = _2E_str424;
        r1 = _2E_str121;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 296;
        _assert(i7);
}
}
}
else{
        r0 = _2E_str222;
        r1 = _2E_str323;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 250;
        _assert(i7);
}
}
        r0 = 0;
        r_g0 = r0;
        return;
}

function _ZN28btHashedOverlappingPairCache8findPairEP17btBroadphaseProxyS1_(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
var __label__ = 0;
        i7 = sp + -16;var g0 = i7>>2; // save stack
        r0 = gFindPairs;
        r0 = r0 >> 2;
        r1 = heap32[(r0)];
        r2 = heap32[(fp+1)];
        r3 = heap32[(fp+2)];
        r1 = (r1 + 1)|0;
        r4 = r2 >> 2;
        heap32[(r0)] = r1;
        r0 = r3 >> 2;
        r1 = heap32[(r4+3)];
        r0 = heap32[(r0+3)];
        r4 = r1 > r0 ? r2 : r3;
        r0 = r1 > r0 ? r3 : r2;
        r1 = r4 >> 2;
        r1 = heap32[(r1+3)];
        r0 = r0 >> 2;
        r0 = heap32[(r0+3)];
        r2 = r1 << 16;
        r2 = r2 | r0;
        r3 = r2 << 15;
        r3 = r3 ^ -1;
        r2 = (r2 + r3)|0;
        r3 = r2 >> 10;
        r2 = r3 ^ r2;
        r2 = (r2 * 9)|0;
        r3 = r2 >> 6;
        r2 = r3 ^ r2;
        r3 = r2 << 11;
        r4 = heap32[(fp)];
        r3 = r3 ^ -1;
        r4 = r4 >> 2;
        r2 = (r2 + r3)|0;
        r3 = r2 >> 16;
        r5 = heap32[(r4+3)];
        r2 = r3 ^ r2;
        r3 = (r5 + -1)|0;
        r2 = r2 & r3;
        r3 = heap32[(r4+9)];
_1: do {
if(!(r3 <=r2)) //_LBB137_9
{
        r3 = heap32[(r4+11)];
        r2 = r2 << 2;
        r2 = (r3 + r2)|0;
_3: while(true){
        r2 = r2 >> 2;
        r2 = heap32[(r2)];
        if(r2 ==-1) //_LBB137_9
{
break _1;
}
else{
        r3 = heap32[(r4+4)];
        r5 = r2 << 4;
        r3 = (r3 + r5)|0;
        r5 = r3 >> 2;
        r6 = heap32[(r5)];
        r6 = r6 >> 2;
        r6 = heap32[(r6+3)];
if(!(r6 !=r0)) //_LBB137_2
{
        r5 = heap32[(r5+1)];
        r5 = r5 >> 2;
        r5 = heap32[(r5+3)];
if(!(r5 !=r1)) //_LBB137_2
{
break _3;
}
}
        r3 = heap32[(r4+16)];
        r2 = r2 << 2;
        r2 = (r3 + r2)|0;
}
}
        r0 = heap32[(r4+2)];
        if(r0 >r2) //_LBB137_8
{
        r_g0 = r3;
        return;
}
else{
        r3 = _2E_str222;
        r0 = _2E_str121;
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r0;
        heap32[(g0+2)] = 165;
        _assert(i7);
}
}
} while(0);
        r0 = 0;
        r_g0 = r0;
        return;
}

function _ZN28btHashedOverlappingPairCache20sortOverlappingPairsEP12btDispatcher(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var r12;
        var r13;
        var r14;
        var r15;
        var r16;
var __label__ = 0;
        i7 = sp + -40;var g0 = i7>>2; // save stack
        r0 = sp + -24;
        r1 = 1;
        r2 = r0 >> 2;
        heap8[sp+-8] = r1;
        heap32[(r2+3)] = 0;
        r3 = heap32[(fp)];
        heap32[(r2+1)] = 0;
        r4 = r3 >> 2;
        heap32[(r2+2)] = 0;
        r5 = heap32[(r4+2)];
_1: do {
        if(r5 >0) //_LBB138_2
{
        r6 = heap32[(fp+1)];
        r7 = 0;
        r5 = r7;
_3: while(true){
        r8 = heap32[(r4+4)];
        if(r7 ==r5) //_LBB138_5
{
        r9 = 1;
        r10 = r5 << 1;
        r10 = r5 == 0 ? r9 : r10;
if(!(r7 >=r10)) //_LBB138_4
{
        if(r10 !=0) //_LBB138_8
{
        r7 = gNumAlignedAllocs;
        r7 = r7 >> 2;
        r11 = heap32[(r7)];
        r12 = r10 << 4;
        r11 = (r11 + 1)|0;
        r12 = r12 | 3;
        heap32[(r7)] = r11;
        r7 = (r12 + 16)|0;
        heap32[(g0)] = r7;
        malloc(i7);
        r7 = r_g0;
        if(r7 !=0) //_LBB138_10
{
        r11 = 0;
        r12 = (r7 + 4)|0;
        r11 = (r11 - r12)|0;
        r11 = r11 & 15;
        r11 = (r7 + r11)|0;
        r12 = (r11 + 4)|0;
        r11 = r11 >> 2;
        heap32[(r11)] = r7;
        r7 = r12;
}
}
else{
        r7 = 0;
}
        if(r5 <1) //_LBB138_13
{
        r13 = heap32[(r2+3)];
}
else{
        r11 = 0;
        r12 = (r11 - r5)|0;
_16: while(true){
        r13 = heap32[(r2+3)];
        r14 = r11 << 4;
        r15 = (r13 - r14)|0;
        r15 = r15 >> 2;
        r14 = (r7 - r14)|0;
        r16 = heap32[(r15)];
        r14 = r14 >> 2;
        heap32[(r14)] = r16;
        r16 = heap32[(r15+1)];
        heap32[(r14+1)] = r16;
        r16 = heap32[(r15+2)];
        heap32[(r14+2)] = r16;
        r15 = heap32[(r15+3)];
        r11 = (r11 + -1)|0;
        heap32[(r14+3)] = r15;
if(!(r12 !=r11)) //_LBB138_14
{
break _16;
}
}
}
        if(r13 !=0) //_LBB138_17
{
        r11 = heapU8[sp+-8];
        if(r11 !=0) //_LBB138_19
{
        r5 = gNumAlignedFree;
        r5 = r5 >> 2;
        r11 = heap32[(r5)];
        r11 = (r11 + 1)|0;
        r12 = r13 >> 2;
        heap32[(r5)] = r11;
        r5 = heap32[(r12+-1)];
        heap32[(g0)] = r5;
        free(i7);
        r5 = heap32[(r2+1)];
}
        heap32[(r2+3)] = 0;
}
        heap8[sp+-8] = r9;
        heap32[(r2+3)] = r7;
        heap32[(r2+2)] = r10;
}
}
        r7 = r1 << 4;
        r7 = (r8 + r7)|0;
        r7 = r7 >> 2;
        r5 = r5 << 4;
        r8 = heap32[(r2+3)];
        r5 = (r8 + r5)|0;
        r8 = heap32[(r7+-4)];
        r5 = r5 >> 2;
        heap32[(r5)] = r8;
        r8 = heap32[(r7+-3)];
        heap32[(r5+1)] = r8;
        r8 = heap32[(r7+-2)];
        heap32[(r5+2)] = r8;
        r7 = heap32[(r7+-1)];
        heap32[(r5+3)] = r7;
        r5 = heap32[(r2+1)];
        r5 = (r5 + 1)|0;
        heap32[(r2+1)] = r5;
        r7 = heap32[(r4+2)];
        if(r7 <=r1) //_LBB138_24
{
break _3;
}
else{
        r7 = heap32[(r2+2)];
        r1 = (r1 + 1)|0;
}
}
        if(r5 >0) //_LBB138_26
{
        r1 = 0;
_29: while(true){
        r5 = heap32[(r2+3)];
        r7 = r1 << 4;
        r8 = heap32[(r4)];
        r5 = (r5 + r7)|0;
        r5 = r5 >> 2;
        r7 = r8 >> 2;
        r7 = heap32[(r7+3)];
        r8 = heap32[(r5+1)];
        r5 = heap32[(r5)];
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r5;
        heap32[(g0+2)] = r8;
        heap32[(g0+3)] = r6;
        __FUNCTION_TABLE__[(r7)>>2](i7);
        r5 = heap32[(r2+1)];
        r1 = (r1 + 1)|0;
        if(r5 >r1) //_LBB138_27
{
continue _29;
}
else{
break _1;
}
}
}
}
else{
        r5 = 0;
}
} while(0);
        r1 = heap32[(r4+14)];
        if(r1 >0) //_LBB138_30
{
        r5 = 0;
_35: while(true){
        r1 = r5 << 2;
        r6 = heap32[(r4+16)];
        r1 = (r6 + r1)|0;
        r5 = (r5 + 1)|0;
        r1 = r1 >> 2;
        heap32[(r1)] = -1;
        r1 = heap32[(r4+14)];
if(!(r1 >r5)) //_LBB138_31
{
break _35;
}
}
        r5 = heap32[(r2+1)];
}
        if(r5 >1) //_LBB138_35
{
        r5 = (r5 + -1)|0;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = 0;
        heap32[(g0+2)] = r5;
        _ZN20btAlignedObjectArrayI16btBroadphasePairE17quickSortInternalI29btBroadphasePairSortPredicateEEvT_ii(i7);
        r5 = heap32[(r2+1)];
}
_42: do {
if(!(r5 <1)) //_LBB138_39
{
        r0 = 0;
_44: while(true){
        r1 = heap32[(r2+3)];
        r5 = r0 << 4;
        r6 = heap32[(r4)];
        r1 = (r1 + r5)|0;
        r1 = r1 >> 2;
        r5 = r6 >> 2;
        r5 = heap32[(r5+2)];
        r6 = heap32[(r1+1)];
        r1 = heap32[(r1)];
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = r6;
        r0 = (r0 + 1)|0;
        __FUNCTION_TABLE__[(r5)>>2](i7);
        r1 = heap32[(r2+1)];
        if(r1 >r0) //_LBB138_38
{
continue _44;
}
else{
break _42;
}
}
}
} while(0);
        r0 = heap32[(r2+3)];
if(!(r0 ==0)) //_LBB138_42
{
        r1 = heapU8[sp+-8];
if(!(r1 ==0)) //_LBB138_42
{
        r1 = gNumAlignedFree;
        r1 = r1 >> 2;
        r2 = heap32[(r1)];
        r2 = (r2 + 1)|0;
        r0 = r0 >> 2;
        heap32[(r1)] = r2;
        r0 = heap32[(r0+-1)];
        heap32[(g0)] = r0;
        free(i7);
}
}
        return;
}

function _ZN20btAlignedObjectArrayI16btBroadphasePairE21expandNonInitializingEv(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var r12;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = r0 >> 2;
        r2 = heap32[(r1+2)];
        r3 = heap32[(r1+1)];
        if(r2 ==r3) //_LBB139_2
{
        r4 = 1;
        r5 = r3 << 1;
        r5 = r3 == 0 ? r4 : r5;
        if(r2 >=r5) //_LBB139_1
{
__label__ = 1;
}
else{
        if(r5 !=0) //_LBB139_5
{
        r2 = gNumAlignedAllocs;
        r2 = r2 >> 2;
        r6 = heap32[(r2)];
        r7 = r5 << 4;
        r6 = (r6 + 1)|0;
        r7 = r7 | 3;
        heap32[(r2)] = r6;
        r2 = (r7 + 16)|0;
        heap32[(g0)] = r2;
        malloc(i7);
        r6 = r_g0;
        if(r6 !=0) //_LBB139_7
{
        r2 = 0;
        r7 = (r6 + 4)|0;
        r2 = (r2 - r7)|0;
        r2 = r2 & 15;
        r2 = (r6 + r2)|0;
        r7 = (r2 + 4)|0;
        r2 = r2 >> 2;
        heap32[(r2)] = r6;
        r6 = r7;
}
}
else{
        r6 = 0;
}
        r7 = (r0 + 12)|0;
        if(r3 <1) //_LBB139_10
{
        r2 = r7 >> 2;
        r9 = heap32[(r2)];
}
else{
        r2 = 0;
        r8 = (r2 - r3)|0;
_12: while(true){
        r9 = r7 >> 2;
        r9 = heap32[(r9)];
        r10 = r2 << 4;
        r11 = (r9 - r10)|0;
        r11 = r11 >> 2;
        r10 = (r6 - r10)|0;
        r12 = heap32[(r11)];
        r10 = r10 >> 2;
        heap32[(r10)] = r12;
        r12 = heap32[(r11+1)];
        heap32[(r10+1)] = r12;
        r12 = heap32[(r11+2)];
        heap32[(r10+2)] = r12;
        r11 = heap32[(r11+3)];
        r2 = (r2 + -1)|0;
        heap32[(r10+3)] = r11;
if(!(r8 !=r2)) //_LBB139_11
{
break _12;
}
}
        r7 = (r0 + 12)|0;
}
        if(r9 !=0) //_LBB139_15
{
        r2 = heapU8[r0+16];
        if(r2 !=0) //_LBB139_17
{
        r2 = gNumAlignedFree;
        r2 = r2 >> 2;
        r8 = heap32[(r2)];
        r8 = (r8 + 1)|0;
        r9 = r9 >> 2;
        heap32[(r2)] = r8;
        r2 = heap32[(r9+-1)];
        heap32[(g0)] = r2;
        free(i7);
        r2 = heap32[(r1+1)];
}
else{
        r2 = r3;
}
        r8 = r7 >> 2;
        heap32[(r8)] = 0;
}
else{
        r2 = r3;
}
        r7 = r7 >> 2;
        heap8[r0+16] = r4;
        heap32[(r7)] = r6;
        heap32[(r1+2)] = r5;
__label__ = 19;
}
}
else{
__label__ = 1;
}
if (__label__ == 1){
        r2 = r3;
}
        r0 = (r2 + 1)|0;
        heap32[(r1+1)] = r0;
        r0 = heap32[(r1+3)];
        r1 = r3 << 4;
        r0 = (r0 + r1)|0;
        r_g0 = r0;
        return;
}

function _ZN28btHashedOverlappingPairCache10growTablesEv(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = r0 >> 2;
        r2 = heap32[(r1+9)];
        r3 = heap32[(r1+3)];
_1: do {
if(!(r2 >=r3)) //_LBB140_48
{
_3: do {
if(!(r2 >r3)) //_LBB140_20
{
        r4 = heap32[(r1+10)];
if(!(r4 >=r3)) //_LBB140_18
{
        if(r3 !=0) //_LBB140_5
{
        r4 = gNumAlignedAllocs;
        r4 = r4 >> 2;
        r5 = heap32[(r4)];
        r6 = r3 << 2;
        r5 = (r5 + 1)|0;
        r6 = r6 | 3;
        heap32[(r4)] = r5;
        r4 = (r6 + 16)|0;
        heap32[(g0)] = r4;
        malloc(i7);
        r4 = r_g0;
        if(r4 !=0) //_LBB140_7
{
        r5 = 0;
        r6 = (r4 + 4)|0;
        r5 = (r5 - r6)|0;
        r5 = r5 & 15;
        r5 = (r4 + r5)|0;
        r6 = (r5 + 4)|0;
        r5 = r5 >> 2;
        heap32[(r5)] = r4;
        r4 = r6;
}
}
else{
        r4 = 0;
}
        r5 = (r0 + 44)|0;
        if(r2 <1) //_LBB140_10
{
        r6 = r5 >> 2;
        r7 = heap32[(r6)];
}
else{
        r6 = 0;
_15: while(true){
        r7 = r5 >> 2;
        r7 = heap32[(r7)];
        r8 = r6 << 2;
        r9 = (r7 + r8)|0;
        r9 = r9 >> 2;
        r8 = (r4 + r8)|0;
        r9 = heap32[(r9)];
        r6 = (r6 + 1)|0;
        r8 = r8 >> 2;
        heap32[(r8)] = r9;
if(!(r2 !=r6)) //_LBB140_11
{
break _15;
}
}
        r5 = (r0 + 44)|0;
}
if(!(r7 ==0)) //_LBB140_17
{
        r6 = heapU8[r0+48];
if(!(r6 ==0)) //_LBB140_16
{
        r6 = gNumAlignedFree;
        r6 = r6 >> 2;
        r8 = heap32[(r6)];
        r8 = (r8 + 1)|0;
        r7 = r7 >> 2;
        heap32[(r6)] = r8;
        r6 = heap32[(r7+-1)];
        heap32[(g0)] = r6;
        free(i7);
}
        r6 = r5 >> 2;
        heap32[(r6)] = 0;
}
        r6 = 1;
        r5 = r5 >> 2;
        heap8[r0+48] = r6;
        heap32[(r5)] = r4;
        heap32[(r1+10)] = r3;
}
        r4 = r2;
_26: while(true){
        r5 = r4 << 2;
        r6 = heap32[(r1+11)];
        r5 = (r6 + r5)|0;
        r4 = (r4 + 1)|0;
        r5 = r5 >> 2;
        heap32[(r5)] = 0;
if(!(r3 !=r4)) //_LBB140_19
{
break _3;
}
}
}
} while(0);
        heap32[(r1+9)] = r3;
        r4 = heap32[(r1+14)];
_29: do {
if(!(r4 >r3)) //_LBB140_39
{
if(!(r4 >=r3)) //_LBB140_39
{
        r5 = heap32[(r1+15)];
if(!(r5 >=r3)) //_LBB140_38
{
        if(r3 !=0) //_LBB140_25
{
        r5 = gNumAlignedAllocs;
        r5 = r5 >> 2;
        r6 = heap32[(r5)];
        r7 = r3 << 2;
        r6 = (r6 + 1)|0;
        r7 = r7 | 3;
        heap32[(r5)] = r6;
        r5 = (r7 + 16)|0;
        heap32[(g0)] = r5;
        malloc(i7);
        r5 = r_g0;
        if(r5 !=0) //_LBB140_27
{
        r6 = 0;
        r7 = (r5 + 4)|0;
        r6 = (r6 - r7)|0;
        r6 = r6 & 15;
        r6 = (r5 + r6)|0;
        r7 = (r6 + 4)|0;
        r6 = r6 >> 2;
        heap32[(r6)] = r5;
        r5 = r7;
}
}
else{
        r5 = 0;
}
        r6 = (r0 + 64)|0;
        if(r4 <1) //_LBB140_30
{
        r7 = r6 >> 2;
        r8 = heap32[(r7)];
}
else{
        r7 = 0;
_42: while(true){
        r8 = r6 >> 2;
        r8 = heap32[(r8)];
        r9 = r7 << 2;
        r10 = (r8 + r9)|0;
        r10 = r10 >> 2;
        r9 = (r5 + r9)|0;
        r10 = heap32[(r10)];
        r7 = (r7 + 1)|0;
        r9 = r9 >> 2;
        heap32[(r9)] = r10;
if(!(r4 !=r7)) //_LBB140_31
{
break _42;
}
}
        r6 = (r0 + 64)|0;
}
if(!(r8 ==0)) //_LBB140_37
{
        r7 = heapU8[r0+68];
if(!(r7 ==0)) //_LBB140_36
{
        r7 = gNumAlignedFree;
        r7 = r7 >> 2;
        r9 = heap32[(r7)];
        r9 = (r9 + 1)|0;
        r8 = r8 >> 2;
        heap32[(r7)] = r9;
        r7 = heap32[(r8+-1)];
        heap32[(g0)] = r7;
        free(i7);
}
        r7 = r6 >> 2;
        heap32[(r7)] = 0;
}
        r7 = 1;
        r6 = r6 >> 2;
        heap8[r0+68] = r7;
        heap32[(r6)] = r5;
        heap32[(r1+15)] = r3;
        if(r4 >=r3) //_LBB140_39
{
break _29;
}
}
_52: while(true){
        r0 = r4 << 2;
        r5 = heap32[(r1+16)];
        r0 = (r5 + r0)|0;
        r4 = (r4 + 1)|0;
        r0 = r0 >> 2;
        heap32[(r0)] = 0;
if(!(r3 !=r4)) //_LBB140_38
{
break _29;
}
}
}
}
} while(0);
        heap32[(r1+14)] = r3;
_55: do {
if(!(r3 <1)) //_LBB140_45
{
        r0 = 0;
_57: while(true){
        r4 = r0 << 2;
        r5 = heap32[(r1+11)];
        r4 = (r5 + r4)|0;
        r0 = (r0 + 1)|0;
        r4 = r4 >> 2;
        heap32[(r4)] = -1;
if(!(r3 !=r0)) //_LBB140_41
{
break _57;
}
}
if(!(r3 <1)) //_LBB140_45
{
        r0 = 0;
_61: while(true){
        r4 = r0 << 2;
        r5 = heap32[(r1+16)];
        r4 = (r5 + r4)|0;
        r0 = (r0 + 1)|0;
        r4 = r4 >> 2;
        heap32[(r4)] = -1;
if(!(r3 !=r0)) //_LBB140_44
{
break _55;
}
}
}
}
} while(0);
if(!(r2 <1)) //_LBB140_48
{
        r0 = 0;
_65: while(true){
        r3 = heap32[(r1+4)];
        r4 = r0 << 4;
        r3 = (r3 + r4)|0;
        r3 = r3 >> 2;
        r4 = heap32[(r3+1)];
        r3 = heap32[(r3)];
        r4 = r4 >> 2;
        r3 = r3 >> 2;
        r4 = heap32[(r4+3)];
        r4 = r4 << 16;
        r3 = heap32[(r3+3)];
        r3 = r4 | r3;
        r4 = r3 << 15;
        r4 = r4 ^ -1;
        r3 = (r3 + r4)|0;
        r4 = r3 >> 10;
        r3 = r4 ^ r3;
        r3 = (r3 * 9)|0;
        r4 = r3 >> 6;
        r3 = r4 ^ r3;
        r4 = r3 << 11;
        r4 = r4 ^ -1;
        r3 = (r3 + r4)|0;
        r4 = r3 >> 16;
        r5 = heap32[(r1+3)];
        r3 = r4 ^ r3;
        r4 = (r5 + -1)|0;
        r3 = r3 & r4;
        r3 = r3 << 2;
        r4 = heap32[(r1+11)];
        r4 = (r4 + r3)|0;
        r4 = r4 >> 2;
        r5 = r0 << 2;
        r6 = heap32[(r1+16)];
        r5 = (r6 + r5)|0;
        r4 = heap32[(r4)];
        r5 = r5 >> 2;
        heap32[(r5)] = r4;
        r4 = heap32[(r1+11)];
        r3 = (r4 + r3)|0;
        r4 = (r0 + 1)|0;
        r3 = r3 >> 2;
        heap32[(r3)] = r0;
        r0 = r4;
        if(r2 !=r4) //_LBB140_47
{
continue _65;
}
else{
break _1;
}
}
}
}
} while(0);
        return;
}

function _ZN28btHashedOverlappingPairCacheC1Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV28btHashedOverlappingPairCache;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        r3 = 1;
        heap32[(r2)] = r1;
        heap8[r0+20] = r3;
        heap32[(r2+4)] = 0;
        heap32[(r2+2)] = 0;
        heap32[(r2+3)] = 0;
        r1 = 0;
        heap32[(r2+6)] = 0;
        heap8[r0+28] = r1;
        heap8[r0+48] = r3;
        heap32[(r2+11)] = 0;
        heap32[(r2+9)] = 0;
        heap32[(r2+10)] = 0;
        heap8[r0+68] = r3;
        heap32[(r2+16)] = 0;
        heap32[(r2+14)] = 0;
        r4 = gNumAlignedAllocs;
        heap32[(r2+15)] = 0;
        r4 = r4 >> 2;
        heap32[(r2+18)] = 0;
        r5 = heap32[(r4)];
        r5 = (r5 + 1)|0;
        heap32[(r4)] = r5;
        heap32[(g0)] = 51;
        malloc(i7);
        r4 = r_g0;
        if(r4 !=0) //_LBB141_2
{
        r5 = (r4 + 4)|0;
        r1 = (r1 - r5)|0;
        r1 = r1 & 15;
        r1 = (r4 + r1)|0;
        r5 = (r1 + 4)|0;
        r1 = r1 >> 2;
        heap32[(r1)] = r4;
        r4 = r5;
}
        heap8[r0+20] = r3;
        heap32[(r2+4)] = r4;
        heap32[(r2+3)] = 2;
        heap32[(g0)] = r0;
        _ZN28btHashedOverlappingPairCache10growTablesEv(i7);
        return;
}

function _ZNK14btQuantizedBvh31calculateSerializeBufferSizeNewEv(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = 84;
        r_g0 = r0;
        return;
}

function _ZNK14btQuantizedBvh9serializeEPvP12btSerializer(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
var __label__ = 0;
        i7 = sp + -24;var g0 = i7>>2; // save stack
        r0 = heap32[(fp+1)];
        r1 = heap32[(fp)];
        r0 = r0 >> 2;
        r2 = r1 >> 2;
        heap32[(r0+4)] = heap32[(r2+5)];
        heap32[(r0+5)] = heap32[(r2+6)];
        heap32[(r0+6)] = heap32[(r2+7)];
        heap32[(r0+7)] = heap32[(r2+8)];
        heap32[(r0)] = heap32[(r2+1)];
        heap32[(r0+1)] = heap32[(r2+2)];
        heap32[(r0+2)] = heap32[(r2+3)];
        heap32[(r0+3)] = heap32[(r2+4)];
        heap32[(r0+8)] = heap32[(r2+9)];
        heap32[(r0+9)] = heap32[(r2+10)];
        heap32[(r0+10)] = heap32[(r2+11)];
        heap32[(r0+11)] = heap32[(r2+12)];
        r3 = heap32[(r2+14)];
        heap32[(r0+12)] = r3;
        r1 = heapU8[r1+60];
        heap32[(r0+13)] = r1;
        r1 = heap32[(r2+22)];
        r3 = heap32[(fp+2)];
        heap32[(r0+14)] = r1;
        r1 = heap32[(r2+22)];
        if(r1 !=0) //_LBB143_2
{
        r1 = r3 >> 2;
        r4 = heap32[(r1)];
        r4 = r4 >> 2;
        r4 = heap32[(r4+7)];
        r5 = heap32[(r2+24)];
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r5;
        __FUNCTION_TABLE__[(r4)>>2](i7);
        r4 = r_g0;
        heap32[(r0+16)] = r4;
if(!(r4 ==0)) //_LBB143_7
{
        r4 = heap32[(r1)];
        r4 = r4 >> 2;
        r5 = heap32[(r2+22)];
        r4 = heap32[(r4+4)];
        heap32[(g0)] = r3;
        heap32[(g0+1)] = 48;
        heap32[(g0+2)] = r5;
        __FUNCTION_TABLE__[(r4)>>2](i7);
        r4 = r_g0;
if(!(r5 <1)) //_LBB143_6
{
        r6 = r4 >> 2;
        r7 = 0;
        r6 = heap32[(r6+2)];
        r5 = (r7 - r5)|0;
_6: while(true){
        r8 = (r7 * -12)|0;
        r8 = r8 << 2;
        r9 = r7 << 6;
        r10 = heap32[(r2+24)];
        r8 = (r6 + r8)|0;
        r10 = (r10 - r9)|0;
        r8 = r8 >> 2;
        r10 = r10 >> 2;
        heap32[(r8+4)] = heap32[(r10+4)];
        heap32[(r8+5)] = heap32[(r10+5)];
        heap32[(r8+6)] = heap32[(r10+6)];
        heap32[(r8+7)] = heap32[(r10+7)];
        r10 = heap32[(r2+24)];
        r10 = (r10 - r9)|0;
        r10 = r10 >> 2;
        heap32[(r8)] = heap32[(r10)];
        heap32[(r8+1)] = heap32[(r10+1)];
        heap32[(r8+2)] = heap32[(r10+2)];
        heap32[(r8+3)] = heap32[(r10+3)];
        r10 = heap32[(r2+24)];
        r10 = (r10 - r9)|0;
        r10 = r10 >> 2;
        r10 = heap32[(r10+8)];
        heap32[(r8+8)] = r10;
        r10 = heap32[(r2+24)];
        r10 = (r10 - r9)|0;
        r10 = r10 >> 2;
        r10 = heap32[(r10+9)];
        heap32[(r8+9)] = r10;
        r10 = heap32[(r2+24)];
        r9 = (r10 - r9)|0;
        r9 = r9 >> 2;
        r9 = heap32[(r9+10)];
        r7 = (r7 + -1)|0;
        heap32[(r8+10)] = r9;
if(!(r5 !=r7)) //_LBB143_5
{
break _6;
}
}
}
        r1 = heap32[(r1)];
        r1 = r1 >> 2;
        r1 = heap32[(r1+5)];
        r5 = heap32[(r2+24)];
        r6 = _2E_str32;
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r4;
        heap32[(g0+2)] = r6;
        heap32[(g0+3)] = 1497453121;
        heap32[(g0+4)] = r5;
        __FUNCTION_TABLE__[(r1)>>2](i7);
}
}
else{
        heap32[(r0+16)] = 0;
}
        r1 = heap32[(r2+32)];
        heap32[(r0+15)] = r1;
        r1 = heap32[(r2+32)];
        if(r1 !=0) //_LBB143_9
{
        r1 = r3 >> 2;
        r4 = heap32[(r1)];
        r4 = r4 >> 2;
        r4 = heap32[(r4+7)];
        r5 = heap32[(r2+34)];
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r5;
        __FUNCTION_TABLE__[(r4)>>2](i7);
        r4 = r_g0;
        heap32[(r0+17)] = r4;
if(!(r4 ==0)) //_LBB143_14
{
        r4 = heap32[(r1)];
        r4 = r4 >> 2;
        r5 = heap32[(r2+32)];
        r4 = heap32[(r4+4)];
        heap32[(g0)] = r3;
        heap32[(g0+1)] = 16;
        heap32[(g0+2)] = r5;
        __FUNCTION_TABLE__[(r4)>>2](i7);
        r4 = r_g0;
_14: do {
if(!(r5 <1)) //_LBB143_13
{
        r6 = r4 >> 2;
        r7 = 0;
        r6 = heap32[(r6+2)];
        r5 = (r7 - r5)|0;
_16: while(true){
        r8 = r7 << 4;
        r9 = heap32[(r2+34)];
        r9 = (r9 - r8)|0;
        r9 = r9 >> 2;
        r10 = (r6 - r8)|0;
        r9 = heap32[(r9+3)];
        r11 = r10 >> 2;
        heap32[(r11+3)] = r9;
        r9 = heap32[(r2+34)];
        r9 = (r9 - r8)|0;
        r9 = heapU16[(r9+6)>>1];
        heap16[(r10+6)>>1] = r9;
        r9 = heap32[(r2+34)];
        r9 = (r9 - r8)|0;
        r9 = heapU16[(r9+8)>>1];
        heap16[(r10+8)>>1] = r9;
        r9 = heap32[(r2+34)];
        r9 = (r9 - r8)|0;
        r9 = heapU16[(r9+10)>>1];
        heap16[(r10+10)>>1] = r9;
        r9 = heap32[(r2+34)];
        r9 = (r9 - r8)|0;
        r9 = heapU16[(r9)>>1];
        heap16[(r10)>>1] = r9;
        r9 = heap32[(r2+34)];
        r9 = (r9 - r8)|0;
        r9 = heapU16[(r9+2)>>1];
        heap16[(r10+2)>>1] = r9;
        r9 = heap32[(r2+34)];
        r8 = (r9 - r8)|0;
        r8 = heapU16[(r8+4)>>1];
        r7 = (r7 + -1)|0;
        heap16[(r10+4)>>1] = r8;
if(!(r5 !=r7)) //_LBB143_12
{
break _14;
}
}
}
} while(0);
        r1 = heap32[(r1)];
        r1 = r1 >> 2;
        r1 = heap32[(r1+5)];
        r5 = heap32[(r2+34)];
        r6 = _2E_str133;
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r4;
        heap32[(g0+2)] = r6;
        heap32[(g0+3)] = 1497453121;
        heap32[(g0+4)] = r5;
        __FUNCTION_TABLE__[(r1)>>2](i7);
}
}
else{
        heap32[(r0+17)] = 0;
}
        r1 = heap32[(r2+36)];
        heap32[(r0+19)] = r1;
        r1 = heap32[(r2+38)];
        heap32[(r0+20)] = r1;
        r1 = heap32[(r2+38)];
        if(r1 !=0) //_LBB143_16
{
        r1 = r3 >> 2;
        r4 = heap32[(r1)];
        r4 = r4 >> 2;
        r4 = heap32[(r4+7)];
        r5 = heap32[(r2+40)];
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r5;
        __FUNCTION_TABLE__[(r4)>>2](i7);
        r4 = r_g0;
        heap32[(r0+18)] = r4;
if(!(r4 ==0)) //_LBB143_21
{
        r0 = heap32[(r1)];
        r0 = r0 >> 2;
        r4 = heap32[(r2+38)];
        r0 = heap32[(r0+4)];
        heap32[(g0)] = r3;
        heap32[(g0+1)] = 20;
        heap32[(g0+2)] = r4;
        __FUNCTION_TABLE__[(r0)>>2](i7);
        r0 = r_g0;
_24: do {
if(!(r4 <1)) //_LBB143_20
{
        r5 = r0 >> 2;
        r6 = 0;
        r5 = heap32[(r5+2)];
        r4 = (r6 - r4)|0;
_26: while(true){
        r7 = r6 << 5;
        r8 = heap32[(r2+40)];
        r9 = (r6 * -10)|0;
        r8 = (r8 - r7)|0;
        r8 = heapU16[(r8+6)>>1];
        r9 = r9 << 1;
        r9 = (r5 + r9)|0;
        heap16[(r9+14)>>1] = r8;
        r8 = heap32[(r2+40)];
        r8 = (r8 - r7)|0;
        r8 = heapU16[(r8+8)>>1];
        heap16[(r9+16)>>1] = r8;
        r8 = heap32[(r2+40)];
        r8 = (r8 - r7)|0;
        r8 = heapU16[(r8+10)>>1];
        heap16[(r9+18)>>1] = r8;
        r8 = heap32[(r2+40)];
        r8 = (r8 - r7)|0;
        r8 = heapU16[(r8)>>1];
        heap16[(r9+8)>>1] = r8;
        r8 = heap32[(r2+40)];
        r8 = (r8 - r7)|0;
        r8 = heapU16[(r8+2)>>1];
        heap16[(r9+10)>>1] = r8;
        r8 = heap32[(r2+40)];
        r8 = (r8 - r7)|0;
        r8 = heapU16[(r8+4)>>1];
        heap16[(r9+12)>>1] = r8;
        r8 = heap32[(r2+40)];
        r9 = (r6 * -5)|0;
        r8 = (r8 - r7)|0;
        r9 = r9 << 2;
        r8 = r8 >> 2;
        r9 = (r5 + r9)|0;
        r8 = heap32[(r8+3)];
        r9 = r9 >> 2;
        heap32[(r9)] = r8;
        r8 = heap32[(r2+40)];
        r7 = (r8 - r7)|0;
        r7 = r7 >> 2;
        r7 = heap32[(r7+4)];
        r6 = (r6 + -1)|0;
        heap32[(r9+1)] = r7;
if(!(r4 !=r6)) //_LBB143_19
{
break _24;
}
}
}
} while(0);
        r1 = heap32[(r1)];
        r1 = r1 >> 2;
        r1 = heap32[(r1+5)];
        r2 = heap32[(r2+40)];
        r4 = _2E_str234;
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r0;
        heap32[(g0+2)] = r4;
        heap32[(g0+3)] = 1497453121;
        heap32[(g0+4)] = r2;
        __FUNCTION_TABLE__[(r1)>>2](i7);
}
}
else{
        heap32[(r0+18)] = 0;
}
        r0 = _2E_str335;
        r_g0 = r0;
        return;
}

function _ZN14btQuantizedBvhD1Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        heap32[(g0)] = r0;
        _ZN14btQuantizedBvhD2Ev(i7);
        return;
}

function _ZN14btQuantizedBvhD0Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV14btQuantizedBvh;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        r1 = heap32[(r2+40)];
if(!(r1 ==0)) //_LBB145_4
{
        r3 = heapU8[r0+164];
if(!(r3 ==0)) //_LBB145_3
{
        r3 = gNumAlignedFree;
        r3 = r3 >> 2;
        r4 = heap32[(r3)];
        r4 = (r4 + 1)|0;
        r1 = r1 >> 2;
        heap32[(r3)] = r4;
        r1 = heap32[(r1+-1)];
        heap32[(g0)] = r1;
        free(i7);
}
        heap32[(r2+40)] = 0;
}
        r1 = 1;
        heap8[r0+164] = r1;
        heap32[(r2+40)] = 0;
        heap32[(r2+38)] = 0;
        heap32[(r2+39)] = 0;
        r3 = heap32[(r2+34)];
if(!(r3 ==0)) //_LBB145_8
{
        r4 = heapU8[r0+140];
if(!(r4 ==0)) //_LBB145_7
{
        r4 = gNumAlignedFree;
        r4 = r4 >> 2;
        r5 = heap32[(r4)];
        r5 = (r5 + 1)|0;
        r3 = r3 >> 2;
        heap32[(r4)] = r5;
        r3 = heap32[(r3+-1)];
        heap32[(g0)] = r3;
        free(i7);
}
        heap32[(r2+34)] = 0;
}
        heap8[r0+140] = r1;
        heap32[(r2+34)] = 0;
        heap32[(r2+32)] = 0;
        heap32[(r2+33)] = 0;
        r3 = heap32[(r2+29)];
if(!(r3 ==0)) //_LBB145_12
{
        r4 = heapU8[r0+120];
if(!(r4 ==0)) //_LBB145_11
{
        r4 = gNumAlignedFree;
        r4 = r4 >> 2;
        r5 = heap32[(r4)];
        r5 = (r5 + 1)|0;
        r3 = r3 >> 2;
        heap32[(r4)] = r5;
        r3 = heap32[(r3+-1)];
        heap32[(g0)] = r3;
        free(i7);
}
        heap32[(r2+29)] = 0;
}
        heap8[r0+120] = r1;
        heap32[(r2+29)] = 0;
        heap32[(r2+27)] = 0;
        heap32[(r2+28)] = 0;
        r3 = heap32[(r2+24)];
if(!(r3 ==0)) //_LBB145_16
{
        r4 = heapU8[r0+100];
if(!(r4 ==0)) //_LBB145_15
{
        r4 = gNumAlignedFree;
        r4 = r4 >> 2;
        r5 = heap32[(r4)];
        r5 = (r5 + 1)|0;
        r3 = r3 >> 2;
        heap32[(r4)] = r5;
        r3 = heap32[(r3+-1)];
        heap32[(g0)] = r3;
        free(i7);
}
        heap32[(r2+24)] = 0;
}
        heap8[r0+100] = r1;
        heap32[(r2+24)] = 0;
        heap32[(r2+22)] = 0;
        heap32[(r2+23)] = 0;
        r3 = heap32[(r2+19)];
if(!(r3 ==0)) //_LBB145_20
{
        r4 = heapU8[r0+80];
if(!(r4 ==0)) //_LBB145_19
{
        r4 = gNumAlignedFree;
        r4 = r4 >> 2;
        r5 = heap32[(r4)];
        r5 = (r5 + 1)|0;
        r3 = r3 >> 2;
        heap32[(r4)] = r5;
        r3 = heap32[(r3+-1)];
        heap32[(g0)] = r3;
        free(i7);
}
        heap32[(r2+19)] = 0;
}
        heap8[r0+80] = r1;
        heap32[(r2+19)] = 0;
        heap32[(r2+17)] = 0;
        heap32[(r2+18)] = 0;
if(!(r0 ==0)) //_LBB145_22
{
        r0 = gNumAlignedFree;
        r0 = r0 >> 2;
        r1 = heap32[(r0)];
        r1 = (r1 + 1)|0;
        heap32[(r0)] = r1;
        r0 = heap32[(r2+-1)];
        heap32[(g0)] = r0;
        free(i7);
}
        return;
}

function _ZNK14btQuantizedBvh9serializeEPvjb(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var r12;
var __label__ = 0;
        i7 = sp + -16;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = r0 >> 2;
        r2 = heap32[(r1+38)];
        r3 = heap32[(r1+42)];
        if(r3 ==r2) //_LBB146_2
{
        r3 = heap32[(fp+1)];
        r4 = heap32[(fp+3)];
        r5 = _ZTV14btQuantizedBvh;
        r6 = r3 >> 2;
        r5 = (r5 + 8)|0;
        heap32[(r1+42)] = r2;
        heap32[(r6)] = r5;
        r2 = 0;
        heap32[(r6+13)] = 277;
        r5 = 1;
        heap8[r3+60] = r2;
        heap8[r3+80] = r5;
        heap32[(r6+19)] = 0;
        heap32[(r6+17)] = 0;
        heap32[(r6+18)] = 0;
        heap8[r3+100] = r5;
        heap32[(r6+24)] = 0;
        heap32[(r6+22)] = 0;
        heap32[(r6+23)] = 0;
        heap8[r3+120] = r5;
        heap32[(r6+29)] = 0;
        heap32[(r6+27)] = 0;
        heap32[(r6+28)] = 0;
        heap8[r3+140] = r5;
        heap32[(r6+34)] = 0;
        heap32[(r6+32)] = 0;
        heap32[(r6+33)] = 0;
        heap32[(r6+36)] = 0;
        heap8[r3+164] = r5;
        heap32[(r6+40)] = 0;
        heap32[(r6+38)] = 0;
        heap32[(r6+39)] = 0;
        heap32[(r6+42)] = 0;
        heap32[(r6+1)] = -8388609;
        heap32[(r6+2)] = -8388609;
        heap32[(r6+3)] = -8388609;
        heap32[(r6+4)] = 0;
        heap32[(r6+5)] = 2139095039;
        heap32[(r6+6)] = 2139095039;
        heap32[(r6+7)] = 2139095039;
        heap32[(r6+8)] = 0;
        r7 = heap32[(r1+14)];
        if(r4 ==0) //_LBB146_4
{
        heap32[(r6+14)] = r7;
        heap32[(r6+1)] = heap32[(r1+1)];
        heap32[(r6+2)] = heap32[(r1+2)];
        heap32[(r6+3)] = heap32[(r1+3)];
        heap32[(r6+4)] = heap32[(r1+4)];
        heap32[(r6+5)] = heap32[(r1+5)];
        heap32[(r6+6)] = heap32[(r1+6)];
        heap32[(r6+7)] = heap32[(r1+7)];
        heap32[(r6+8)] = heap32[(r1+8)];
        heap32[(r6+9)] = heap32[(r1+9)];
        heap32[(r6+10)] = heap32[(r1+10)];
        heap32[(r6+11)] = heap32[(r1+11)];
        heap32[(r6+12)] = heap32[(r1+12)];
        r7 = heap32[(r1+36)];
        heap32[(r6+36)] = r7;
        r7 = heap32[(r1+42)];
}
else{
        r8 = r7 << 8;
        r9 = r7 >>> 8;
        r10 = r7 << 24;
        r8 = r8 & 16711680;
        r9 = r9 & 65280;
        r7 = r7 >>> 24;
        r8 = r10 | r8;
        r7 = r9 | r7;
        r7 = r8 | r7;
        heap32[(r6+14)] = r7;
        r7 = heapU8[r0+7];
        heap8[r3+4] = r7;
        r7 = heapU8[r0+6];
        heap8[r3+5] = r7;
        r7 = heapU8[r0+5];
        heap8[r3+6] = r7;
        r7 = heapU8[r0+4];
        heap8[r3+7] = r7;
        r7 = heapU8[r0+11];
        heap8[r3+8] = r7;
        r7 = heapU8[r0+10];
        heap8[r3+9] = r7;
        r7 = heapU8[r0+9];
        heap8[r3+10] = r7;
        r7 = heapU8[r0+8];
        heap8[r3+11] = r7;
        r7 = heapU8[r0+15];
        heap8[r3+12] = r7;
        r7 = heapU8[r0+14];
        heap8[r3+13] = r7;
        r7 = heapU8[r0+13];
        heap8[r3+14] = r7;
        r7 = heapU8[r0+12];
        heap8[r3+15] = r7;
        r7 = heapU8[r0+19];
        heap8[r3+16] = r7;
        r7 = heapU8[r0+18];
        heap8[r3+17] = r7;
        r7 = heapU8[r0+17];
        heap8[r3+18] = r7;
        r7 = heapU8[r0+16];
        heap8[r3+19] = r7;
        r7 = heapU8[r0+23];
        heap8[r3+20] = r7;
        r7 = heapU8[r0+22];
        heap8[r3+21] = r7;
        r7 = heapU8[r0+21];
        heap8[r3+22] = r7;
        r7 = heapU8[r0+20];
        heap8[r3+23] = r7;
        r7 = heapU8[r0+27];
        heap8[r3+24] = r7;
        r7 = heapU8[r0+26];
        heap8[r3+25] = r7;
        r7 = heapU8[r0+25];
        heap8[r3+26] = r7;
        r7 = heapU8[r0+24];
        heap8[r3+27] = r7;
        r7 = heapU8[r0+31];
        heap8[r3+28] = r7;
        r7 = heapU8[r0+30];
        heap8[r3+29] = r7;
        r7 = heapU8[r0+29];
        heap8[r3+30] = r7;
        r7 = heapU8[r0+28];
        heap8[r3+31] = r7;
        r7 = heapU8[r0+35];
        heap8[r3+32] = r7;
        r7 = heapU8[r0+34];
        heap8[r3+33] = r7;
        r7 = heapU8[r0+33];
        heap8[r3+34] = r7;
        r7 = heapU8[r0+32];
        heap8[r3+35] = r7;
        r7 = heapU8[r0+39];
        heap8[r3+36] = r7;
        r7 = heapU8[r0+38];
        heap8[r3+37] = r7;
        r7 = heapU8[r0+37];
        heap8[r3+38] = r7;
        r7 = heapU8[r0+36];
        heap8[r3+39] = r7;
        r7 = heapU8[r0+43];
        heap8[r3+40] = r7;
        r7 = heapU8[r0+42];
        heap8[r3+41] = r7;
        r7 = heapU8[r0+41];
        heap8[r3+42] = r7;
        r7 = heapU8[r0+40];
        heap8[r3+43] = r7;
        r7 = heapU8[r0+47];
        heap8[r3+44] = r7;
        r7 = heapU8[r0+46];
        heap8[r3+45] = r7;
        r7 = heapU8[r0+45];
        heap8[r3+46] = r7;
        r7 = heapU8[r0+44];
        heap8[r3+47] = r7;
        r7 = heapU8[r0+51];
        heap8[r3+48] = r7;
        r7 = heapU8[r0+50];
        heap8[r3+49] = r7;
        r7 = heapU8[r0+49];
        heap8[r3+50] = r7;
        r7 = heapU8[r0+48];
        heap8[r3+51] = r7;
        r7 = heap32[(r1+36)];
        r8 = r7 << 8;
        r9 = r7 >>> 8;
        r10 = r7 << 24;
        r8 = r8 & 16711680;
        r9 = r9 & 65280;
        r7 = r7 >>> 24;
        r8 = r10 | r8;
        r7 = r9 | r7;
        r7 = r8 | r7;
        heap32[(r6+36)] = r7;
        r7 = heap32[(r1+42)];
        r8 = r7 << 8;
        r9 = r7 >>> 8;
        r10 = r7 << 24;
        r8 = r8 & 16711680;
        r9 = r9 & 65280;
        r7 = r7 >>> 24;
        r8 = r10 | r8;
        r7 = r9 | r7;
        r7 = r8 | r7;
}
        heap32[(r6+42)] = r7;
        r7 = heapU8[r0+60];
        heap8[r3+60] = r7;
        r7 = (r3 + 172)|0;
        r8 = heap32[(r1+14)];
        r0 = heapU8[r0+60];
        if(r0 ==0) //_LBB146_20
{
        r0 = 0;
        heap8[r3+100] = r0;
        heap32[(r6+24)] = r7;
        heap32[(r6+22)] = r8;
        heap32[(r6+23)] = r8;
_9: do {
        if(r4 !=0) //_LBB146_22
{
if(!(r8 <1)) //_LBB146_29
{
        r5 = (r5 - r8)|0;
_12: while(true){
        r2 = r0 << 6;
        r9 = heap32[(r1+24)];
        r9 = (r9 - r2)|0;
        r10 = heapU8[r9+3];
        r7 = (r7 - r2)|0;
        heap8[r7] = r10;
        r10 = heapU8[r9+2];
        heap8[r7+1] = r10;
        r10 = heapU8[r9+1];
        heap8[r7+2] = r10;
        r10 = heapU8[r9];
        heap8[r7+3] = r10;
        r10 = heapU8[r9+7];
        heap8[r7+4] = r10;
        r10 = heapU8[r9+6];
        heap8[r7+5] = r10;
        r10 = heapU8[r9+5];
        heap8[r7+6] = r10;
        r10 = heapU8[r9+4];
        heap8[r7+7] = r10;
        r10 = heapU8[r9+11];
        heap8[r7+8] = r10;
        r10 = heapU8[r9+10];
        heap8[r7+9] = r10;
        r10 = heapU8[r9+9];
        heap8[r7+10] = r10;
        r10 = heapU8[r9+8];
        heap8[r7+11] = r10;
        r10 = heapU8[r9+15];
        heap8[r7+12] = r10;
        r10 = heapU8[r9+14];
        heap8[r7+13] = r10;
        r10 = heapU8[r9+13];
        heap8[r7+14] = r10;
        r9 = heapU8[r9+12];
        heap8[r7+15] = r9;
        r7 = heap32[(r1+24)];
        r7 = (r7 - r2)|0;
        r9 = heap32[(r6+24)];
        r10 = heapU8[r7+19];
        r9 = (r9 - r2)|0;
        heap8[r9+16] = r10;
        r10 = heapU8[r7+18];
        heap8[r9+17] = r10;
        r10 = heapU8[r7+17];
        heap8[r9+18] = r10;
        r10 = heapU8[r7+16];
        heap8[r9+19] = r10;
        r10 = heapU8[r7+23];
        heap8[r9+20] = r10;
        r10 = heapU8[r7+22];
        heap8[r9+21] = r10;
        r10 = heapU8[r7+21];
        heap8[r9+22] = r10;
        r10 = heapU8[r7+20];
        heap8[r9+23] = r10;
        r10 = heapU8[r7+27];
        heap8[r9+24] = r10;
        r10 = heapU8[r7+26];
        heap8[r9+25] = r10;
        r10 = heapU8[r7+25];
        heap8[r9+26] = r10;
        r10 = heapU8[r7+24];
        heap8[r9+27] = r10;
        r10 = heapU8[r7+31];
        heap8[r9+28] = r10;
        r10 = heapU8[r7+30];
        heap8[r9+29] = r10;
        r10 = heapU8[r7+29];
        heap8[r9+30] = r10;
        r7 = heapU8[r7+28];
        heap8[r9+31] = r7;
        r7 = heap32[(r1+24)];
        r7 = (r7 - r2)|0;
        r7 = r7 >> 2;
        r7 = heap32[(r7+8)];
        r9 = r7 << 8;
        r10 = r7 >>> 8;
        r11 = heap32[(r6+24)];
        r12 = r7 << 24;
        r9 = r9 & 16711680;
        r10 = r10 & 65280;
        r7 = r7 >>> 24;
        r11 = (r11 - r2)|0;
        r9 = r12 | r9;
        r7 = r10 | r7;
        r10 = r11 >> 2;
        r7 = r9 | r7;
        heap32[(r10+8)] = r7;
        r7 = heap32[(r1+24)];
        r7 = (r7 - r2)|0;
        r7 = r7 >> 2;
        r7 = heap32[(r7+9)];
        r9 = r7 << 8;
        r10 = r7 >>> 8;
        r11 = heap32[(r6+24)];
        r12 = r7 << 24;
        r9 = r9 & 16711680;
        r10 = r10 & 65280;
        r7 = r7 >>> 24;
        r11 = (r11 - r2)|0;
        r9 = r12 | r9;
        r7 = r10 | r7;
        r10 = r11 >> 2;
        r7 = r9 | r7;
        heap32[(r10+9)] = r7;
        r7 = heap32[(r1+24)];
        r7 = (r7 - r2)|0;
        r7 = r7 >> 2;
        r7 = heap32[(r7+10)];
        r9 = r7 << 8;
        r10 = r7 >>> 8;
        r11 = heap32[(r6+24)];
        r12 = r7 << 24;
        r9 = r9 & 16711680;
        r10 = r10 & 65280;
        r7 = r7 >>> 24;
        r2 = (r11 - r2)|0;
        r9 = r12 | r9;
        r7 = r10 | r7;
        r2 = r2 >> 2;
        r7 = r9 | r7;
        heap32[(r2+10)] = r7;
        if(r5 ==r0) //_LBB146_29
{
break _9;
}
else{
        r7 = heap32[(r6+24)];
        r0 = (r0 + -1)|0;
}
}
}
}
else{
        if(r8 >0) //_LBB146_26
{
        r5 = (r5 - r8)|0;
        r0 = 0;
_17: while(true){
        r2 = r0 << 6;
        r9 = heap32[(r1+24)];
        r7 = (r7 - r2)|0;
        r9 = (r9 - r2)|0;
        r7 = r7 >> 2;
        r9 = r9 >> 2;
        heap32[(r7)] = heap32[(r9)];
        heap32[(r7+1)] = heap32[(r9+1)];
        heap32[(r7+2)] = heap32[(r9+2)];
        heap32[(r7+3)] = heap32[(r9+3)];
        r7 = heap32[(r6+24)];
        r9 = heap32[(r1+24)];
        r7 = (r7 - r2)|0;
        r9 = (r9 - r2)|0;
        r7 = r7 >> 2;
        r9 = r9 >> 2;
        heap32[(r7+4)] = heap32[(r9+4)];
        heap32[(r7+5)] = heap32[(r9+5)];
        heap32[(r7+6)] = heap32[(r9+6)];
        heap32[(r7+7)] = heap32[(r9+7)];
        r7 = heap32[(r1+24)];
        r7 = (r7 - r2)|0;
        r7 = r7 >> 2;
        r9 = heap32[(r6+24)];
        r9 = (r9 - r2)|0;
        r7 = heap32[(r7+8)];
        r9 = r9 >> 2;
        heap32[(r9+8)] = r7;
        r7 = heap32[(r1+24)];
        r7 = (r7 - r2)|0;
        r7 = r7 >> 2;
        r9 = heap32[(r6+24)];
        r9 = (r9 - r2)|0;
        r7 = heap32[(r7+9)];
        r9 = r9 >> 2;
        heap32[(r9+9)] = r7;
        r7 = heap32[(r1+24)];
        r7 = (r7 - r2)|0;
        r7 = r7 >> 2;
        r9 = heap32[(r6+24)];
        r2 = (r9 - r2)|0;
        r7 = heap32[(r7+10)];
        r2 = r2 >> 2;
        heap32[(r2+10)] = r7;
        if(r5 ==r0) //_LBB146_29
{
break _9;
}
else{
        r7 = heap32[(r6+24)];
        r0 = (r0 + -1)|0;
}
}
}
}
} while(0);
        r5 = heap32[(r6+24)];
if(!(r5 ==0)) //_LBB146_33
{
        r0 = heapU8[r3+100];
if(!(r0 ==0)) //_LBB146_32
{
        r0 = gNumAlignedFree;
        r0 = r0 >> 2;
        r2 = heap32[(r0)];
        r2 = (r2 + 1)|0;
        r5 = r5 >> 2;
        heap32[(r0)] = r2;
        r5 = heap32[(r5+-1)];
        heap32[(g0)] = r5;
        free(i7);
}
        heap32[(r6+24)] = 0;
}
        r5 = r8 << 6;
        r0 = 0;
        heap8[r3+100] = r0;
        heap32[(r6+24)] = 0;
        heap32[(r6+22)] = 0;
        heap32[(r6+23)] = 0;
}
else{
        heap8[r3+140] = r2;
        heap32[(r6+34)] = r7;
        heap32[(r6+32)] = r8;
        heap32[(r6+33)] = r8;
_28: do {
        if(r4 !=0) //_LBB146_8
{
if(!(r8 <1)) //_LBB146_15
{
        r5 = (r5 - r8)|0;
_31: while(true){
        r0 = heap32[(r1+34)];
        r9 = r2 << 4;
        r0 = (r0 - r9)|0;
        r0 = heapU16[(r0)>>1];
        r10 = r0 << 8;
        r0 = r0 << 24;
        r10 = r10 & 16711680;
        r0 = r0 | r10;
        r7 = (r7 - r9)|0;
        r0 = r0 >>> 16;
        heap16[(r7)>>1] = r0;
        r7 = heap32[(r1+34)];
        r7 = (r7 - r9)|0;
        r7 = heapU16[(r7+2)>>1];
        r0 = r7 << 8;
        r7 = r7 << 24;
        r0 = r0 & 16711680;
        r10 = heap32[(r6+34)];
        r7 = r7 | r0;
        r0 = (r10 - r9)|0;
        r7 = r7 >>> 16;
        heap16[(r0+2)>>1] = r7;
        r7 = heap32[(r1+34)];
        r7 = (r7 - r9)|0;
        r7 = heapU16[(r7+4)>>1];
        r0 = r7 << 8;
        r7 = r7 << 24;
        r0 = r0 & 16711680;
        r10 = heap32[(r6+34)];
        r7 = r7 | r0;
        r0 = (r10 - r9)|0;
        r7 = r7 >>> 16;
        heap16[(r0+4)>>1] = r7;
        r7 = heap32[(r1+34)];
        r7 = (r7 - r9)|0;
        r7 = heapU16[(r7+6)>>1];
        r0 = r7 << 8;
        r7 = r7 << 24;
        r0 = r0 & 16711680;
        r10 = heap32[(r6+34)];
        r7 = r7 | r0;
        r0 = (r10 - r9)|0;
        r7 = r7 >>> 16;
        heap16[(r0+6)>>1] = r7;
        r7 = heap32[(r1+34)];
        r7 = (r7 - r9)|0;
        r7 = heapU16[(r7+8)>>1];
        r0 = r7 << 8;
        r7 = r7 << 24;
        r0 = r0 & 16711680;
        r10 = heap32[(r6+34)];
        r7 = r7 | r0;
        r0 = (r10 - r9)|0;
        r7 = r7 >>> 16;
        heap16[(r0+8)>>1] = r7;
        r7 = heap32[(r1+34)];
        r7 = (r7 - r9)|0;
        r7 = heapU16[(r7+10)>>1];
        r0 = r7 << 8;
        r7 = r7 << 24;
        r0 = r0 & 16711680;
        r10 = heap32[(r6+34)];
        r7 = r7 | r0;
        r0 = (r10 - r9)|0;
        r7 = r7 >>> 16;
        heap16[(r0+10)>>1] = r7;
        r7 = heap32[(r1+34)];
        r7 = (r7 - r9)|0;
        r7 = r7 >> 2;
        r7 = heap32[(r7+3)];
        r0 = r7 << 8;
        r10 = r7 >>> 8;
        r11 = heap32[(r6+34)];
        r12 = r7 << 24;
        r0 = r0 & 16711680;
        r10 = r10 & 65280;
        r7 = r7 >>> 24;
        r9 = (r11 - r9)|0;
        r0 = r12 | r0;
        r7 = r10 | r7;
        r9 = r9 >> 2;
        r7 = r0 | r7;
        heap32[(r9+3)] = r7;
        if(r5 ==r2) //_LBB146_15
{
break _28;
}
else{
        r7 = heap32[(r6+34)];
        r2 = (r2 + -1)|0;
}
}
}
}
else{
        if(r8 >0) //_LBB146_12
{
        r5 = (r5 - r8)|0;
        r0 = 0;
_36: while(true){
        r2 = r0 << 4;
        r9 = heap32[(r1+34)];
        r9 = (r9 - r2)|0;
        r9 = heapU16[(r9)>>1];
        r7 = (r7 - r2)|0;
        heap16[(r7)>>1] = r9;
        r7 = heap32[(r1+34)];
        r7 = (r7 - r2)|0;
        r9 = heap32[(r6+34)];
        r7 = heapU16[(r7+2)>>1];
        r9 = (r9 - r2)|0;
        heap16[(r9+2)>>1] = r7;
        r7 = heap32[(r1+34)];
        r7 = (r7 - r2)|0;
        r9 = heap32[(r6+34)];
        r7 = heapU16[(r7+4)>>1];
        r9 = (r9 - r2)|0;
        heap16[(r9+4)>>1] = r7;
        r7 = heap32[(r1+34)];
        r7 = (r7 - r2)|0;
        r9 = heap32[(r6+34)];
        r7 = heapU16[(r7+6)>>1];
        r9 = (r9 - r2)|0;
        heap16[(r9+6)>>1] = r7;
        r7 = heap32[(r1+34)];
        r7 = (r7 - r2)|0;
        r9 = heap32[(r6+34)];
        r7 = heapU16[(r7+8)>>1];
        r9 = (r9 - r2)|0;
        heap16[(r9+8)>>1] = r7;
        r7 = heap32[(r1+34)];
        r7 = (r7 - r2)|0;
        r9 = heap32[(r6+34)];
        r7 = heapU16[(r7+10)>>1];
        r9 = (r9 - r2)|0;
        heap16[(r9+10)>>1] = r7;
        r7 = heap32[(r1+34)];
        r7 = (r7 - r2)|0;
        r7 = r7 >> 2;
        r9 = heap32[(r6+34)];
        r2 = (r9 - r2)|0;
        r7 = heap32[(r7+3)];
        r2 = r2 >> 2;
        heap32[(r2+3)] = r7;
        if(r5 ==r0) //_LBB146_15
{
break _28;
}
else{
        r7 = heap32[(r6+34)];
        r0 = (r0 + -1)|0;
}
}
}
}
} while(0);
        r5 = heap32[(r6+34)];
if(!(r5 ==0)) //_LBB146_19
{
        r7 = heapU8[r3+140];
if(!(r7 ==0)) //_LBB146_18
{
        r7 = gNumAlignedFree;
        r7 = r7 >> 2;
        r0 = heap32[(r7)];
        r0 = (r0 + 1)|0;
        r5 = r5 >> 2;
        heap32[(r7)] = r0;
        r5 = heap32[(r5+-1)];
        heap32[(g0)] = r5;
        free(i7);
}
        heap32[(r6+34)] = 0;
}
        r5 = r8 << 4;
        r7 = 0;
        heap8[r3+140] = r7;
        heap32[(r6+34)] = 0;
        heap32[(r6+32)] = 0;
        heap32[(r6+33)] = 0;
}
        r0 = heap32[(r6+40)];
        r2 = heap32[(r1+42)];
if(!(r0 ==0)) //_LBB146_38
{
        r7 = heapU8[r3+164];
if(!(r7 ==0)) //_LBB146_37
{
        r7 = gNumAlignedFree;
        r7 = r7 >> 2;
        r8 = heap32[(r7)];
        r8 = (r8 + 1)|0;
        r0 = r0 >> 2;
        heap32[(r7)] = r8;
        r0 = heap32[(r0+-1)];
        heap32[(g0)] = r0;
        free(i7);
}
        heap32[(r6+40)] = 0;
}
        r0 = (r5 + r3)|0;
        r0 = (r0 + 172)|0;
        r5 = 0;
        heap8[r3+164] = r5;
        heap32[(r6+40)] = r0;
        heap32[(r6+38)] = r2;
        heap32[(r6+39)] = r2;
        r2 = heap32[(r1+42)];
_53: do {
        if(r4 !=0) //_LBB146_40
{
if(!(r2 <1)) //_LBB146_46
{
__label__ = 41; //SET chanka
_55: while(true){
        r2 = r5 << 5;
        r4 = heap32[(r1+40)];
        r4 = heapU16[(r4+r2)>>1];
        r7 = r4 << 8;
        r4 = r4 << 24;
        r7 = r7 & 16711680;
        r4 = r4 | r7;
        r4 = r4 >>> 16;
        heap16[(r0+r2)>>1] = r4;
        r0 = heap32[(r1+40)];
        r0 = (r0 + r2)|0;
        r0 = heapU16[(r0+2)>>1];
        r4 = r0 << 8;
        r0 = r0 << 24;
        r4 = r4 & 16711680;
        r7 = heap32[(r6+40)];
        r0 = r0 | r4;
        r4 = (r7 + r2)|0;
        r0 = r0 >>> 16;
        heap16[(r4+2)>>1] = r0;
        r0 = heap32[(r1+40)];
        r0 = (r0 + r2)|0;
        r0 = heapU16[(r0+4)>>1];
        r4 = r0 << 8;
        r0 = r0 << 24;
        r4 = r4 & 16711680;
        r7 = heap32[(r6+40)];
        r0 = r0 | r4;
        r4 = (r7 + r2)|0;
        r0 = r0 >>> 16;
        heap16[(r4+4)>>1] = r0;
        r0 = heap32[(r1+40)];
        r0 = (r0 + r2)|0;
        r0 = heapU16[(r0+6)>>1];
        r4 = r0 << 8;
        r0 = r0 << 24;
        r4 = r4 & 16711680;
        r7 = heap32[(r6+40)];
        r0 = r0 | r4;
        r4 = (r7 + r2)|0;
        r0 = r0 >>> 16;
        heap16[(r4+6)>>1] = r0;
        r0 = heap32[(r1+40)];
        r0 = (r0 + r2)|0;
        r0 = heapU16[(r0+8)>>1];
        r4 = r0 << 8;
        r0 = r0 << 24;
        r4 = r4 & 16711680;
        r7 = heap32[(r6+40)];
        r0 = r0 | r4;
        r4 = (r7 + r2)|0;
        r0 = r0 >>> 16;
        heap16[(r4+8)>>1] = r0;
        r0 = heap32[(r1+40)];
        r0 = (r0 + r2)|0;
        r0 = heapU16[(r0+10)>>1];
        r4 = r0 << 8;
        r0 = r0 << 24;
        r4 = r4 & 16711680;
        r7 = heap32[(r6+40)];
        r0 = r0 | r4;
        r4 = (r7 + r2)|0;
        r0 = r0 >>> 16;
        heap16[(r4+10)>>1] = r0;
        r0 = heap32[(r1+40)];
        r0 = (r0 + r2)|0;
        r0 = r0 >> 2;
        r0 = heap32[(r0+3)];
        r4 = r0 << 8;
        r7 = r0 >>> 8;
        r8 = heap32[(r6+40)];
        r9 = r0 << 24;
        r4 = r4 & 16711680;
        r7 = r7 & 65280;
        r0 = r0 >>> 24;
        r8 = (r8 + r2)|0;
        r4 = r9 | r4;
        r0 = r7 | r0;
        r7 = r8 >> 2;
        r0 = r4 | r0;
        heap32[(r7+3)] = r0;
        r0 = heap32[(r1+40)];
        r0 = (r0 + r2)|0;
        r0 = r0 >> 2;
        r0 = heap32[(r0+4)];
        r4 = r0 << 8;
        r7 = r0 >>> 8;
        r8 = heap32[(r6+40)];
        r9 = r0 << 24;
        r4 = r4 & 16711680;
        r7 = r7 & 65280;
        r0 = r0 >>> 24;
        r2 = (r8 + r2)|0;
        r4 = r9 | r4;
        r0 = r7 | r0;
        r5 = (r5 + 1)|0;
        r2 = r2 >> 2;
        r0 = r4 | r0;
        heap32[(r2+4)] = r0;
        r0 = heap32[(r1+42)];
        if(r0 <=r5) //_LBB146_46
{
break _53;
}
else{
        r0 = heap32[(r6+40)];
}
}
}
}
else{
        if(r2 >0) //_LBB146_43
{
        r2 = 0;
_60: while(true){
        r4 = r2 << 5;
        r5 = heap32[(r1+40)];
        r5 = heapU16[(r5+r4)>>1];
        heap16[(r0+r4)>>1] = r5;
        r0 = heap32[(r1+40)];
        r0 = (r0 + r4)|0;
        r5 = heap32[(r6+40)];
        r0 = heapU16[(r0+2)>>1];
        r5 = (r5 + r4)|0;
        heap16[(r5+2)>>1] = r0;
        r0 = heap32[(r1+40)];
        r0 = (r0 + r4)|0;
        r5 = heap32[(r6+40)];
        r0 = heapU16[(r0+4)>>1];
        r5 = (r5 + r4)|0;
        heap16[(r5+4)>>1] = r0;
        r0 = heap32[(r1+40)];
        r0 = (r0 + r4)|0;
        r5 = heap32[(r6+40)];
        r0 = heapU16[(r0+6)>>1];
        r5 = (r5 + r4)|0;
        heap16[(r5+6)>>1] = r0;
        r0 = heap32[(r1+40)];
        r0 = (r0 + r4)|0;
        r5 = heap32[(r6+40)];
        r0 = heapU16[(r0+8)>>1];
        r5 = (r5 + r4)|0;
        heap16[(r5+8)>>1] = r0;
        r0 = heap32[(r1+40)];
        r0 = (r0 + r4)|0;
        r5 = heap32[(r6+40)];
        r0 = heapU16[(r0+10)>>1];
        r5 = (r5 + r4)|0;
        heap16[(r5+10)>>1] = r0;
        r0 = heap32[(r1+40)];
        r0 = (r0 + r4)|0;
        r0 = r0 >> 2;
        r5 = heap32[(r6+40)];
        r5 = (r5 + r4)|0;
        r0 = heap32[(r0+3)];
        r5 = r5 >> 2;
        heap32[(r5+3)] = r0;
        r0 = heap32[(r1+40)];
        r0 = (r0 + r4)|0;
        r0 = r0 >> 2;
        r5 = heap32[(r6+40)];
        r5 = (r5 + r4)|0;
        r0 = heap32[(r0+4)];
        r5 = r5 >> 2;
        heap32[(r5+4)] = r0;
        r0 = heap32[(r6+40)];
        r0 = (r0 + r4)|0;
        r0 = r0 >> 2;
        heap32[(r0+5)] = 0;
        r0 = heap32[(r6+40)];
        r0 = (r0 + r4)|0;
        r0 = r0 >> 2;
        heap32[(r0+6)] = 0;
        r0 = heap32[(r6+40)];
        r0 = (r0 + r4)|0;
        r2 = (r2 + 1)|0;
        r0 = r0 >> 2;
        heap32[(r0+7)] = 0;
        r0 = heap32[(r1+42)];
        if(r0 <=r2) //_LBB146_46
{
break _53;
}
else{
        r0 = heap32[(r6+40)];
}
}
}
}
} while(0);
        r0 = heap32[(r6+40)];
if(!(r0 ==0)) //_LBB146_50
{
        r1 = heapU8[r3+164];
if(!(r1 ==0)) //_LBB146_49
{
        r1 = gNumAlignedFree;
        r1 = r1 >> 2;
        r2 = heap32[(r1)];
        r2 = (r2 + 1)|0;
        r0 = r0 >> 2;
        heap32[(r1)] = r2;
        r0 = heap32[(r0+-1)];
        heap32[(g0)] = r0;
        free(i7);
}
        heap32[(r6+40)] = 0;
}
        r0 = 0;
        heap8[r3+164] = r0;
        heap32[(r6+40)] = 0;
        heap32[(r6+38)] = 0;
        heap32[(r6+39)] = 0;
        heap32[(r6)] = 0;
        r0 = 1;
        r_g0 = r0;
        return;
}
else{
        r0 = _2E_str638;
        r1 = _2E_str537;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 847;
        _assert(i7);
}
}

function _ZN14btQuantizedBvh16deSerializeFloatER23btQuantizedBvhFloatData(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var r12;
        var r13;
        var r14;
        var r15;
        var r16;
        var r17;
        var r18;
var __label__ = 0;
        i7 = sp + -16;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = heap32[(fp+1)];
        r1 = r1 >> 2;
        r2 = r0 >> 2;
        heap32[(r2+5)] = heap32[(r1+4)];
        heap32[(r2+6)] = heap32[(r1+5)];
        heap32[(r2+7)] = heap32[(r1+6)];
        heap32[(r2+8)] = heap32[(r1+7)];
        heap32[(r2+1)] = heap32[(r1)];
        heap32[(r2+2)] = heap32[(r1+1)];
        heap32[(r2+3)] = heap32[(r1+2)];
        heap32[(r2+4)] = heap32[(r1+3)];
        heap32[(r2+9)] = heap32[(r1+8)];
        heap32[(r2+10)] = heap32[(r1+9)];
        heap32[(r2+11)] = heap32[(r1+10)];
        heap32[(r2+12)] = heap32[(r1+11)];
        r3 = heap32[(r1+12)];
        heap32[(r2+14)] = r3;
        r3 = 0;
        r4 = heap32[(r1+13)];
        r4 = r4 != r3;
        r4 = r4 & 1;
        heap8[r0+60] = r4;
        r4 = heap32[(r2+22)];
        r5 = heap32[(r1+14)];
if(!(r4 >=r5)) //_LBB147_17
{
        r6 = heap32[(r2+23)];
if(!(r6 >=r5)) //_LBB147_17
{
        if(r5 !=0) //_LBB147_4
{
        r6 = gNumAlignedAllocs;
        r6 = r6 >> 2;
        r7 = heap32[(r6)];
        r7 = (r7 + 1)|0;
        r8 = r5 << 6;
        heap32[(r6)] = r7;
        r6 = r8 | 19;
        heap32[(g0)] = r6;
        malloc(i7);
        r6 = r_g0;
        if(r6 !=0) //_LBB147_6
{
        r7 = 0;
        r8 = (r6 + 4)|0;
        r7 = (r7 - r8)|0;
        r7 = r7 & 15;
        r7 = (r6 + r7)|0;
        r8 = (r7 + 4)|0;
        r7 = r7 >> 2;
        heap32[(r7)] = r6;
        r6 = r8;
}
}
else{
        r6 = 0;
}
        r7 = (r0 + 96)|0;
        if(r4 <1) //_LBB147_9
{
        r4 = r7 >> 2;
        r9 = heap32[(r4)];
}
else{
        r8 = 0;
_12: while(true){
        r9 = r7 >> 2;
        r9 = heap32[(r9)];
        r10 = (r6 + r8)|0;
        r11 = (r9 + r8)|0;
        heap32[(g0)] = r10;
        heap32[(g0+1)] = r11;
        heap32[(g0+2)] = 64;
        r4 = (r4 + -1)|0;
        r8 = (r8 + 64)|0;
        memcpy(i7);
if(!(r4 !=0)) //_LBB147_10
{
break _12;
}
}
        r7 = (r0 + 96)|0;
}
if(!(r9 ==0)) //_LBB147_16
{
        r4 = heapU8[r0+100];
if(!(r4 ==0)) //_LBB147_15
{
        r4 = gNumAlignedFree;
        r4 = r4 >> 2;
        r8 = heap32[(r4)];
        r8 = (r8 + 1)|0;
        r9 = r9 >> 2;
        heap32[(r4)] = r8;
        r4 = heap32[(r9+-1)];
        heap32[(g0)] = r4;
        free(i7);
}
        r4 = r7 >> 2;
        heap32[(r4)] = 0;
}
        r4 = 1;
        r7 = r7 >> 2;
        heap8[r0+100] = r4;
        heap32[(r7)] = r6;
        heap32[(r2+23)] = r5;
}
}
        heap32[(r2+22)] = r5;
_23: do {
if(!(r5 <1)) //_LBB147_20
{
        r4 = 0;
        r6 = heap32[(r1+16)];
        r5 = (r4 - r5)|0;
_25: while(true){
        r7 = (r4 * -12)|0;
        r7 = r7 << 2;
        r8 = r4 << 6;
        r9 = heap32[(r2+24)];
        r7 = (r6 + r7)|0;
        r9 = (r9 - r8)|0;
        r7 = r7 >> 2;
        r9 = r9 >> 2;
        heap32[(r9+4)] = heap32[(r7+4)];
        heap32[(r9+5)] = heap32[(r7+5)];
        heap32[(r9+6)] = heap32[(r7+6)];
        heap32[(r9+7)] = heap32[(r7+7)];
        r9 = heap32[(r2+24)];
        r9 = (r9 - r8)|0;
        r9 = r9 >> 2;
        heap32[(r9)] = heap32[(r7)];
        heap32[(r9+1)] = heap32[(r7+1)];
        heap32[(r9+2)] = heap32[(r7+2)];
        heap32[(r9+3)] = heap32[(r7+3)];
        r9 = heap32[(r2+24)];
        r9 = (r9 - r8)|0;
        r10 = heap32[(r7+8)];
        r9 = r9 >> 2;
        heap32[(r9+8)] = r10;
        r9 = heap32[(r2+24)];
        r9 = (r9 - r8)|0;
        r10 = heap32[(r7+9)];
        r9 = r9 >> 2;
        heap32[(r9+9)] = r10;
        r9 = heap32[(r2+24)];
        r8 = (r9 - r8)|0;
        r7 = heap32[(r7+10)];
        r4 = (r4 + -1)|0;
        r8 = r8 >> 2;
        heap32[(r8+10)] = r7;
        if(r5 !=r4) //_LBB147_19
{
continue _25;
}
else{
break _23;
}
}
}
} while(0);
        r4 = heap32[(r2+32)];
        r5 = heap32[(r1+15)];
_28: do {
if(!(r4 >=r5)) //_LBB147_23
{
        r6 = (r0 + 124)|0;
        heap32[(g0)] = r6;
        heap32[(g0+1)] = r5;
        r6 = (r5 - r4)|0;
        r4 = r4 << 4;
        _ZN20btAlignedObjectArrayI18btQuantizedBvhNodeE7reserveEi(i7);
_30: while(true){
        r7 = heap32[(r2+34)];
        r7 = (r7 + r4)|0;
        r7 = r7 >> 2;
        r6 = (r6 + -1)|0;
        r4 = (r4 + 16)|0;
        heap32[(r7)] = 0;
        heap32[(r7+1)] = 0;
        heap32[(r7+2)] = 0;
        heap32[(r7+3)] = 0;
        if(r6 !=0) //_LBB147_22
{
continue _30;
}
else{
break _28;
}
}
}
} while(0);
        heap32[(r2+32)] = r5;
_33: do {
if(!(r5 <1)) //_LBB147_26
{
        r4 = 0;
        r6 = heap32[(r1+17)];
        r5 = (r4 - r5)|0;
_35: while(true){
        r7 = r4 << 4;
        r8 = (r6 - r7)|0;
        r9 = r8 >> 2;
        r10 = heap32[(r2+34)];
        r10 = (r10 - r7)|0;
        r9 = heap32[(r9+3)];
        r10 = r10 >> 2;
        heap32[(r10+3)] = r9;
        r9 = heap32[(r2+34)];
        r10 = heapU16[(r8+6)>>1];
        r9 = (r9 - r7)|0;
        heap16[(r9+6)>>1] = r10;
        r9 = heap32[(r2+34)];
        r10 = heapU16[(r8+8)>>1];
        r9 = (r9 - r7)|0;
        heap16[(r9+8)>>1] = r10;
        r9 = heap32[(r2+34)];
        r10 = heapU16[(r8+10)>>1];
        r9 = (r9 - r7)|0;
        heap16[(r9+10)>>1] = r10;
        r9 = heap32[(r2+34)];
        r10 = heapU16[(r8)>>1];
        r9 = (r9 - r7)|0;
        heap16[(r9)>>1] = r10;
        r9 = heap32[(r2+34)];
        r10 = heapU16[(r8+2)>>1];
        r9 = (r9 - r7)|0;
        heap16[(r9+2)>>1] = r10;
        r9 = heap32[(r2+34)];
        r8 = heapU16[(r8+4)>>1];
        r4 = (r4 + -1)|0;
        r7 = (r9 - r7)|0;
        heap16[(r7+4)>>1] = r8;
        if(r5 !=r4) //_LBB147_25
{
continue _35;
}
else{
break _33;
}
}
}
} while(0);
        r4 = heap32[(r1+19)];
        heap32[(r2+36)] = r4;
        r4 = heap32[(r2+38)];
        r5 = heap32[(r1+20)];
if(!(r4 >=r5)) //_LBB147_43
{
        r6 = heap32[(r2+39)];
if(!(r6 >=r5)) //_LBB147_43
{
        if(r5 !=0) //_LBB147_30
{
        r6 = gNumAlignedAllocs;
        r6 = r6 >> 2;
        r7 = heap32[(r6)];
        r7 = (r7 + 1)|0;
        r8 = r5 << 5;
        heap32[(r6)] = r7;
        r6 = r8 | 19;
        heap32[(g0)] = r6;
        malloc(i7);
        r6 = r_g0;
        if(r6 !=0) //_LBB147_32
{
        r7 = 0;
        r8 = (r6 + 4)|0;
        r7 = (r7 - r8)|0;
        r7 = r7 & 15;
        r7 = (r6 + r7)|0;
        r8 = (r7 + 4)|0;
        r7 = r7 >> 2;
        heap32[(r7)] = r6;
        r6 = r8;
}
}
else{
        r6 = 0;
}
        r7 = (r0 + 160)|0;
        if(r4 <1) //_LBB147_35
{
        r4 = r7 >> 2;
        r9 = heap32[(r4)];
}
else{
        r8 = 0;
_49: while(true){
        r9 = r7 >> 2;
        r9 = heap32[(r9)];
        r10 = (r9 + r8)|0;
        r10 = r10 >> 2;
        r11 = (r6 + r8)|0;
        r4 = (r4 + -1)|0;
        r8 = (r8 + 32)|0;
        r12 = heap32[(r10)];
        r11 = r11 >> 2;
        r13 = heap32[(r10+1)];
        r14 = heap32[(r10+2)];
        r15 = heap32[(r10+3)];
        r16 = heap32[(r10+4)];
        r17 = heap32[(r10+5)];
        r18 = heap32[(r10+6)];
        r10 = heap32[(r10+7)];
        heap32[(r11)] = r12;
        heap32[(r11+1)] = r13;
        heap32[(r11+2)] = r14;
        heap32[(r11+3)] = r15;
        heap32[(r11+4)] = r16;
        heap32[(r11+5)] = r17;
        heap32[(r11+6)] = r18;
        heap32[(r11+7)] = r10;
if(!(r4 !=0)) //_LBB147_36
{
break _49;
}
}
        r7 = (r0 + 160)|0;
}
if(!(r9 ==0)) //_LBB147_42
{
        r4 = heapU8[r0+164];
if(!(r4 ==0)) //_LBB147_41
{
        r4 = gNumAlignedFree;
        r4 = r4 >> 2;
        r8 = heap32[(r4)];
        r8 = (r8 + 1)|0;
        r9 = r9 >> 2;
        heap32[(r4)] = r8;
        r4 = heap32[(r9+-1)];
        heap32[(g0)] = r4;
        free(i7);
}
        r4 = r7 >> 2;
        heap32[(r4)] = 0;
}
        r4 = 1;
        r7 = r7 >> 2;
        heap8[r0+164] = r4;
        heap32[(r7)] = r6;
        heap32[(r2+39)] = r5;
}
}
        heap32[(r2+38)] = r5;
_60: do {
if(!(r5 <1)) //_LBB147_46
{
        r0 = heap32[(r1+18)];
        r1 = (r3 - r5)|0;
_62: while(true){
        r4 = (r3 * -10)|0;
        r4 = r4 << 1;
        r4 = (r0 + r4)|0;
        r5 = r3 << 5;
        r6 = heap32[(r2+40)];
        r7 = heapU16[(r4+14)>>1];
        r6 = (r6 - r5)|0;
        heap16[(r6+6)>>1] = r7;
        r6 = heap32[(r2+40)];
        r7 = heapU16[(r4+16)>>1];
        r6 = (r6 - r5)|0;
        heap16[(r6+8)>>1] = r7;
        r6 = heap32[(r2+40)];
        r7 = heapU16[(r4+18)>>1];
        r6 = (r6 - r5)|0;
        heap16[(r6+10)>>1] = r7;
        r6 = heap32[(r2+40)];
        r7 = heapU16[(r4+8)>>1];
        r6 = (r6 - r5)|0;
        heap16[(r6)>>1] = r7;
        r6 = heap32[(r2+40)];
        r7 = heapU16[(r4+10)>>1];
        r6 = (r6 - r5)|0;
        heap16[(r6+2)>>1] = r7;
        r6 = (r3 * -5)|0;
        r7 = heap32[(r2+40)];
        r4 = heapU16[(r4+12)>>1];
        r6 = r6 << 2;
        r7 = (r7 - r5)|0;
        r6 = (r0 + r6)|0;
        heap16[(r7+4)>>1] = r4;
        r4 = r6 >> 2;
        r6 = heap32[(r2+40)];
        r6 = (r6 - r5)|0;
        r7 = heap32[(r4)];
        r6 = r6 >> 2;
        heap32[(r6+3)] = r7;
        r6 = heap32[(r2+40)];
        r5 = (r6 - r5)|0;
        r4 = heap32[(r4+1)];
        r3 = (r3 + -1)|0;
        r5 = r5 >> 2;
        heap32[(r5+4)] = r4;
        if(r1 !=r3) //_LBB147_45
{
continue _62;
}
else{
break _60;
}
}
}
} while(0);
        return;
}

function _ZN14btQuantizedBvh17deSerializeDoubleER24btQuantizedBvhDoubleData(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var r12;
        var r13;
        var r14;
        var r15;
        var r16;
        var r17;
        var r18;
        var f0;
var __label__ = 0;
        i7 = sp + -16;var g0 = i7>>2; // save stack
        r0 = heap32[(fp+1)];
        r1 = heap32[(fp)];
        f0 = llvm_readDouble((r0+32));
        r2 = r1 >> 2;
        f0 = f0; //fdtos f0, f0
        heapFloat[(r2+5)] = f0;
        f0 = llvm_readDouble((r0+40));
        f0 = f0; //fdtos f0, f0
        heapFloat[(r2+6)] = f0;
        f0 = llvm_readDouble((r0+48));
        f0 = f0; //fdtos f0, f0
        heapFloat[(r2+7)] = f0;
        f0 = llvm_readDouble((r0+56));
        f0 = f0; //fdtos f0, f0
        heapFloat[(r2+8)] = f0;
        f0 = llvm_readDouble((r0));
        f0 = f0; //fdtos f0, f0
        heapFloat[(r2+1)] = f0;
        f0 = llvm_readDouble((r0+8));
        f0 = f0; //fdtos f0, f0
        heapFloat[(r2+2)] = f0;
        f0 = llvm_readDouble((r0+16));
        f0 = f0; //fdtos f0, f0
        heapFloat[(r2+3)] = f0;
        f0 = llvm_readDouble((r0+24));
        f0 = f0; //fdtos f0, f0
        heapFloat[(r2+4)] = f0;
        f0 = llvm_readDouble((r0+64));
        f0 = f0; //fdtos f0, f0
        heapFloat[(r2+9)] = f0;
        f0 = llvm_readDouble((r0+72));
        f0 = f0; //fdtos f0, f0
        heapFloat[(r2+10)] = f0;
        f0 = llvm_readDouble((r0+80));
        f0 = f0; //fdtos f0, f0
        heapFloat[(r2+11)] = f0;
        f0 = llvm_readDouble((r0+88));
        f0 = f0; //fdtos f0, f0
        r0 = r0 >> 2;
        heapFloat[(r2+12)] = f0;
        r3 = heap32[(r0+24)];
        heap32[(r2+14)] = r3;
        r3 = 0;
        r4 = heap32[(r0+25)];
        r4 = r4 != r3;
        r4 = r4 & 1;
        heap8[r1+60] = r4;
        r4 = heap32[(r2+22)];
        r5 = heap32[(r0+26)];
if(!(r4 >=r5)) //_LBB148_17
{
        r6 = heap32[(r2+23)];
if(!(r6 >=r5)) //_LBB148_17
{
        if(r5 !=0) //_LBB148_4
{
        r6 = gNumAlignedAllocs;
        r6 = r6 >> 2;
        r7 = heap32[(r6)];
        r7 = (r7 + 1)|0;
        r8 = r5 << 6;
        heap32[(r6)] = r7;
        r6 = r8 | 19;
        heap32[(g0)] = r6;
        malloc(i7);
        r6 = r_g0;
        if(r6 !=0) //_LBB148_6
{
        r7 = 0;
        r8 = (r6 + 4)|0;
        r7 = (r7 - r8)|0;
        r7 = r7 & 15;
        r7 = (r6 + r7)|0;
        r8 = (r7 + 4)|0;
        r7 = r7 >> 2;
        heap32[(r7)] = r6;
        r6 = r8;
}
}
else{
        r6 = 0;
}
        r7 = (r1 + 96)|0;
        if(r4 <1) //_LBB148_9
{
        r4 = r7 >> 2;
        r9 = heap32[(r4)];
}
else{
        r8 = 0;
_12: while(true){
        r9 = r7 >> 2;
        r9 = heap32[(r9)];
        r10 = (r6 + r8)|0;
        r11 = (r9 + r8)|0;
        heap32[(g0)] = r10;
        heap32[(g0+1)] = r11;
        heap32[(g0+2)] = 64;
        r4 = (r4 + -1)|0;
        r8 = (r8 + 64)|0;
        memcpy(i7);
if(!(r4 !=0)) //_LBB148_10
{
break _12;
}
}
        r7 = (r1 + 96)|0;
}
if(!(r9 ==0)) //_LBB148_16
{
        r4 = heapU8[r1+100];
if(!(r4 ==0)) //_LBB148_15
{
        r4 = gNumAlignedFree;
        r4 = r4 >> 2;
        r8 = heap32[(r4)];
        r8 = (r8 + 1)|0;
        r9 = r9 >> 2;
        heap32[(r4)] = r8;
        r4 = heap32[(r9+-1)];
        heap32[(g0)] = r4;
        free(i7);
}
        r4 = r7 >> 2;
        heap32[(r4)] = 0;
}
        r4 = 1;
        r7 = r7 >> 2;
        heap8[r1+100] = r4;
        heap32[(r7)] = r6;
        heap32[(r2+23)] = r5;
}
}
        heap32[(r2+22)] = r5;
_23: do {
if(!(r5 <1)) //_LBB148_20
{
        r4 = 0;
        r6 = heap32[(r0+28)];
        r5 = (r4 - r5)|0;
_25: while(true){
        r7 = (r4 * -10)|0;
        r7 = r7 << 3;
        r8 = (r6 + r7)|0;
        r9 = r4 << 6;
        r10 = heap32[(r2+24)];
        f0 = llvm_readDouble((r8+32));
        r10 = (r10 - r9)|0;
        r10 = r10 >> 2;
        f0 = f0; //fdtos f0, f0
        heapFloat[(r10+4)] = f0;
        f0 = llvm_readDouble((r8+40));
        f0 = f0; //fdtos f0, f0
        heapFloat[(r10+5)] = f0;
        f0 = llvm_readDouble((r8+48));
        f0 = f0; //fdtos f0, f0
        heapFloat[(r10+6)] = f0;
        f0 = llvm_readDouble((r8+56));
        f0 = f0; //fdtos f0, f0
        heapFloat[(r10+7)] = f0;
        r10 = heap32[(r2+24)];
        f0 = llvm_readDouble((r6+r7));
        r7 = (r10 - r9)|0;
        r7 = r7 >> 2;
        f0 = f0; //fdtos f0, f0
        heapFloat[(r7)] = f0;
        f0 = llvm_readDouble((r8+8));
        f0 = f0; //fdtos f0, f0
        heapFloat[(r7+1)] = f0;
        f0 = llvm_readDouble((r8+16));
        f0 = f0; //fdtos f0, f0
        heapFloat[(r7+2)] = f0;
        r10 = (r4 * -20)|0;
        f0 = llvm_readDouble((r8+24));
        r8 = r10 << 2;
        f0 = f0; //fdtos f0, f0
        r8 = (r6 + r8)|0;
        heapFloat[(r7+3)] = f0;
        r7 = r8 >> 2;
        r8 = heap32[(r2+24)];
        r8 = (r8 - r9)|0;
        r10 = heap32[(r7+16)];
        r8 = r8 >> 2;
        heap32[(r8+8)] = r10;
        r8 = heap32[(r2+24)];
        r8 = (r8 - r9)|0;
        r10 = heap32[(r7+17)];
        r8 = r8 >> 2;
        heap32[(r8+9)] = r10;
        r8 = heap32[(r2+24)];
        r8 = (r8 - r9)|0;
        r7 = heap32[(r7+18)];
        r4 = (r4 + -1)|0;
        r8 = r8 >> 2;
        heap32[(r8+10)] = r7;
        if(r5 !=r4) //_LBB148_19
{
continue _25;
}
else{
break _23;
}
}
}
} while(0);
        r4 = heap32[(r2+32)];
        r5 = heap32[(r0+27)];
_28: do {
if(!(r4 >=r5)) //_LBB148_23
{
        r6 = (r1 + 124)|0;
        heap32[(g0)] = r6;
        heap32[(g0+1)] = r5;
        r6 = (r5 - r4)|0;
        r4 = r4 << 4;
        _ZN20btAlignedObjectArrayI18btQuantizedBvhNodeE7reserveEi(i7);
_30: while(true){
        r7 = heap32[(r2+34)];
        r7 = (r7 + r4)|0;
        r7 = r7 >> 2;
        r6 = (r6 + -1)|0;
        r4 = (r4 + 16)|0;
        heap32[(r7)] = 0;
        heap32[(r7+1)] = 0;
        heap32[(r7+2)] = 0;
        heap32[(r7+3)] = 0;
        if(r6 !=0) //_LBB148_22
{
continue _30;
}
else{
break _28;
}
}
}
} while(0);
        heap32[(r2+32)] = r5;
_33: do {
if(!(r5 <1)) //_LBB148_26
{
        r4 = 0;
        r6 = heap32[(r0+29)];
        r5 = (r4 - r5)|0;
_35: while(true){
        r7 = r4 << 4;
        r8 = (r6 - r7)|0;
        r9 = r8 >> 2;
        r10 = heap32[(r2+34)];
        r10 = (r10 - r7)|0;
        r9 = heap32[(r9+3)];
        r10 = r10 >> 2;
        heap32[(r10+3)] = r9;
        r9 = heap32[(r2+34)];
        r10 = heapU16[(r8+6)>>1];
        r9 = (r9 - r7)|0;
        heap16[(r9+6)>>1] = r10;
        r9 = heap32[(r2+34)];
        r10 = heapU16[(r8+8)>>1];
        r9 = (r9 - r7)|0;
        heap16[(r9+8)>>1] = r10;
        r9 = heap32[(r2+34)];
        r10 = heapU16[(r8+10)>>1];
        r9 = (r9 - r7)|0;
        heap16[(r9+10)>>1] = r10;
        r9 = heap32[(r2+34)];
        r10 = heapU16[(r8)>>1];
        r9 = (r9 - r7)|0;
        heap16[(r9)>>1] = r10;
        r9 = heap32[(r2+34)];
        r10 = heapU16[(r8+2)>>1];
        r9 = (r9 - r7)|0;
        heap16[(r9+2)>>1] = r10;
        r9 = heap32[(r2+34)];
        r8 = heapU16[(r8+4)>>1];
        r4 = (r4 + -1)|0;
        r7 = (r9 - r7)|0;
        heap16[(r7+4)>>1] = r8;
        if(r5 !=r4) //_LBB148_25
{
continue _35;
}
else{
break _33;
}
}
}
} while(0);
        r4 = heap32[(r0+30)];
        heap32[(r2+36)] = r4;
        r4 = heap32[(r2+38)];
        r5 = heap32[(r0+31)];
if(!(r4 >=r5)) //_LBB148_43
{
        r6 = heap32[(r2+39)];
if(!(r6 >=r5)) //_LBB148_43
{
        if(r5 !=0) //_LBB148_30
{
        r6 = gNumAlignedAllocs;
        r6 = r6 >> 2;
        r7 = heap32[(r6)];
        r7 = (r7 + 1)|0;
        r8 = r5 << 5;
        heap32[(r6)] = r7;
        r6 = r8 | 19;
        heap32[(g0)] = r6;
        malloc(i7);
        r6 = r_g0;
        if(r6 !=0) //_LBB148_32
{
        r7 = 0;
        r8 = (r6 + 4)|0;
        r7 = (r7 - r8)|0;
        r7 = r7 & 15;
        r7 = (r6 + r7)|0;
        r8 = (r7 + 4)|0;
        r7 = r7 >> 2;
        heap32[(r7)] = r6;
        r6 = r8;
}
}
else{
        r6 = 0;
}
        r7 = (r1 + 160)|0;
        if(r4 <1) //_LBB148_35
{
        r4 = r7 >> 2;
        r9 = heap32[(r4)];
}
else{
        r8 = 0;
_49: while(true){
        r9 = r7 >> 2;
        r9 = heap32[(r9)];
        r10 = (r9 + r8)|0;
        r10 = r10 >> 2;
        r11 = (r6 + r8)|0;
        r4 = (r4 + -1)|0;
        r8 = (r8 + 32)|0;
        r12 = heap32[(r10)];
        r11 = r11 >> 2;
        r13 = heap32[(r10+1)];
        r14 = heap32[(r10+2)];
        r15 = heap32[(r10+3)];
        r16 = heap32[(r10+4)];
        r17 = heap32[(r10+5)];
        r18 = heap32[(r10+6)];
        r10 = heap32[(r10+7)];
        heap32[(r11)] = r12;
        heap32[(r11+1)] = r13;
        heap32[(r11+2)] = r14;
        heap32[(r11+3)] = r15;
        heap32[(r11+4)] = r16;
        heap32[(r11+5)] = r17;
        heap32[(r11+6)] = r18;
        heap32[(r11+7)] = r10;
if(!(r4 !=0)) //_LBB148_36
{
break _49;
}
}
        r7 = (r1 + 160)|0;
}
if(!(r9 ==0)) //_LBB148_42
{
        r4 = heapU8[r1+164];
if(!(r4 ==0)) //_LBB148_41
{
        r4 = gNumAlignedFree;
        r4 = r4 >> 2;
        r8 = heap32[(r4)];
        r8 = (r8 + 1)|0;
        r9 = r9 >> 2;
        heap32[(r4)] = r8;
        r4 = heap32[(r9+-1)];
        heap32[(g0)] = r4;
        free(i7);
}
        r4 = r7 >> 2;
        heap32[(r4)] = 0;
}
        r4 = 1;
        r7 = r7 >> 2;
        heap8[r1+164] = r4;
        heap32[(r7)] = r6;
        heap32[(r2+39)] = r5;
}
}
        heap32[(r2+38)] = r5;
_60: do {
if(!(r5 <1)) //_LBB148_46
{
        r0 = heap32[(r0+32)];
        r1 = (r3 - r5)|0;
_62: while(true){
        r4 = (r3 * -10)|0;
        r4 = r4 << 1;
        r4 = (r0 + r4)|0;
        r5 = r3 << 5;
        r6 = heap32[(r2+40)];
        r7 = heapU16[(r4+14)>>1];
        r6 = (r6 - r5)|0;
        heap16[(r6+6)>>1] = r7;
        r6 = heap32[(r2+40)];
        r7 = heapU16[(r4+16)>>1];
        r6 = (r6 - r5)|0;
        heap16[(r6+8)>>1] = r7;
        r6 = heap32[(r2+40)];
        r7 = heapU16[(r4+18)>>1];
        r6 = (r6 - r5)|0;
        heap16[(r6+10)>>1] = r7;
        r6 = heap32[(r2+40)];
        r7 = heapU16[(r4+8)>>1];
        r6 = (r6 - r5)|0;
        heap16[(r6)>>1] = r7;
        r6 = heap32[(r2+40)];
        r7 = heapU16[(r4+10)>>1];
        r6 = (r6 - r5)|0;
        heap16[(r6+2)>>1] = r7;
        r6 = (r3 * -5)|0;
        r7 = heap32[(r2+40)];
        r4 = heapU16[(r4+12)>>1];
        r6 = r6 << 2;
        r7 = (r7 - r5)|0;
        r6 = (r0 + r6)|0;
        heap16[(r7+4)>>1] = r4;
        r4 = r6 >> 2;
        r6 = heap32[(r2+40)];
        r6 = (r6 - r5)|0;
        r7 = heap32[(r4)];
        r6 = r6 >> 2;
        heap32[(r6+3)] = r7;
        r6 = heap32[(r2+40)];
        r5 = (r6 - r5)|0;
        r4 = heap32[(r4+1)];
        r3 = (r3 + -1)|0;
        r5 = r5 >> 2;
        heap32[(r5+4)] = r4;
        if(r1 !=r3) //_LBB148_45
{
continue _62;
}
else{
break _60;
}
}
}
} while(0);
        return;
}

function _ZN14btQuantizedBvhD2Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV14btQuantizedBvh;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        r1 = heap32[(r2+40)];
if(!(r1 ==0)) //_LBB149_4
{
        r3 = heapU8[r0+164];
if(!(r3 ==0)) //_LBB149_3
{
        r3 = gNumAlignedFree;
        r3 = r3 >> 2;
        r4 = heap32[(r3)];
        r4 = (r4 + 1)|0;
        r1 = r1 >> 2;
        heap32[(r3)] = r4;
        r1 = heap32[(r1+-1)];
        heap32[(g0)] = r1;
        free(i7);
}
        heap32[(r2+40)] = 0;
}
        r1 = 1;
        heap8[r0+164] = r1;
        heap32[(r2+40)] = 0;
        heap32[(r2+38)] = 0;
        heap32[(r2+39)] = 0;
        r3 = heap32[(r2+34)];
if(!(r3 ==0)) //_LBB149_8
{
        r4 = heapU8[r0+140];
if(!(r4 ==0)) //_LBB149_7
{
        r4 = gNumAlignedFree;
        r4 = r4 >> 2;
        r5 = heap32[(r4)];
        r5 = (r5 + 1)|0;
        r3 = r3 >> 2;
        heap32[(r4)] = r5;
        r3 = heap32[(r3+-1)];
        heap32[(g0)] = r3;
        free(i7);
}
        heap32[(r2+34)] = 0;
}
        heap8[r0+140] = r1;
        heap32[(r2+34)] = 0;
        heap32[(r2+32)] = 0;
        heap32[(r2+33)] = 0;
        r3 = heap32[(r2+29)];
if(!(r3 ==0)) //_LBB149_12
{
        r4 = heapU8[r0+120];
if(!(r4 ==0)) //_LBB149_11
{
        r4 = gNumAlignedFree;
        r4 = r4 >> 2;
        r5 = heap32[(r4)];
        r5 = (r5 + 1)|0;
        r3 = r3 >> 2;
        heap32[(r4)] = r5;
        r3 = heap32[(r3+-1)];
        heap32[(g0)] = r3;
        free(i7);
}
        heap32[(r2+29)] = 0;
}
        heap8[r0+120] = r1;
        heap32[(r2+29)] = 0;
        heap32[(r2+27)] = 0;
        heap32[(r2+28)] = 0;
        r3 = heap32[(r2+24)];
if(!(r3 ==0)) //_LBB149_16
{
        r4 = heapU8[r0+100];
if(!(r4 ==0)) //_LBB149_15
{
        r4 = gNumAlignedFree;
        r4 = r4 >> 2;
        r5 = heap32[(r4)];
        r5 = (r5 + 1)|0;
        r3 = r3 >> 2;
        heap32[(r4)] = r5;
        r3 = heap32[(r3+-1)];
        heap32[(g0)] = r3;
        free(i7);
}
        heap32[(r2+24)] = 0;
}
        heap8[r0+100] = r1;
        heap32[(r2+24)] = 0;
        heap32[(r2+22)] = 0;
        heap32[(r2+23)] = 0;
        r3 = heap32[(r2+19)];
if(!(r3 ==0)) //_LBB149_20
{
        r4 = heapU8[r0+80];
if(!(r4 ==0)) //_LBB149_19
{
        r4 = gNumAlignedFree;
        r4 = r4 >> 2;
        r5 = heap32[(r4)];
        r5 = (r5 + 1)|0;
        r3 = r3 >> 2;
        heap32[(r4)] = r5;
        r3 = heap32[(r3+-1)];
        heap32[(g0)] = r3;
        free(i7);
}
        heap32[(r2+19)] = 0;
}
        heap8[r0+80] = r1;
        heap32[(r2+19)] = 0;
        heap32[(r2+17)] = 0;
        heap32[(r2+18)] = 0;
        return;
}

function _ZNK14btQuantizedBvh26walkStacklessQuantizedTreeEP21btNodeOverlapCallbackPtS2_ii(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var r12;
var __label__ = 0;
        i7 = sp + -16;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = heapU8[r0+60];
        if(r1 !=0) //_LBB150_2
{
        r1 = heap32[(fp+1)];
        r2 = heap32[(fp+2)];
        r3 = heap32[(fp+3)];
        r4 = heap32[(fp+4)];
        r5 = heap32[(fp+5)];
        r0 = r0 >> 2;
        r0 = heap32[(r0+34)];
        r6 = r4 << 4;
        r7 = (r5 - r4)|0;
        r0 = (r0 + r6)|0;
        r6 = 0;
_3: while(true){
        if(r4 <r5) //_LBB150_3
{
        if(r6 <r7) //_LBB150_5
{
        r8 = heapU16[(r2)>>1];
        r9 = heapU16[(r0+6)>>1];
        r10 = heapU16[(r3)>>1];
        r11 = heapU16[(r0)>>1];
        r8 = uint(r8) > uint(r9);
        r9 = uint(r10) < uint(r11);
        r10 = heapU16[(r2+4)>>1];
        r11 = heapU16[(r0+10)>>1];
        r8 = r8 | r9;
        r9 = uint(r10) > uint(r11);
        r10 = heapU16[(r3+4)>>1];
        r11 = heapU16[(r0+4)>>1];
        r8 = r8 | r9;
        r9 = uint(r10) < uint(r11);
        r10 = heapU16[(r2+2)>>1];
        r11 = heapU16[(r0+8)>>1];
        r8 = r8 | r9;
        r9 = uint(r10) > uint(r11);
        r10 = heapU16[(r3+2)>>1];
        r11 = heapU16[(r0+2)>>1];
        r12 = r0 >> 2;
        r8 = r8 | r9;
        r9 = uint(r10) < uint(r11);
        r10 = heap32[(r12+3)];
        r6 = (r6 + 1)|0;
        r8 = r8 | r9;
        if(r10 <0) //_LBB150_10
{
__label__ = 10;
}
else{
        if(r8 != 0) //_LBB150_10
{
__label__ = 10;
}
else{
        if(r10 >-1) //_LBB150_9
{
        r8 = r1 >> 2;
        r8 = heap32[(r8)];
        r8 = r8 >> 2;
        r8 = heap32[(r8+2)];
        r12 = r10 >> 21;
        r10 = r10 & 2097151;
        heap32[(g0)] = r1;
        heap32[(g0+1)] = r12;
        heap32[(g0+2)] = r10;
        __FUNCTION_TABLE__[(r8)>>2](i7);
__label__ = 11;
}
else{
__label__ = 8;
break _3;
}
}
}
if (__label__ == 10){
        r9 = 0;
        r9 = r10 < r9;
        r8 = r9 & r8;
        if(r8 != 0) //_LBB150_12
{
        r8 = heap32[(r12+3)];
        if(r8 <0) //_LBB150_14
{
        r9 = r8 << 4;
        r0 = (r0 - r9)|0;
        r4 = (r4 - r8)|0;
continue _3;
}
else{
__label__ = 13;
break _3;
}
}
}
        r0 = (r0 + 16)|0;
        r4 = (r4 + 1)|0;
}
else{
__label__ = 4;
break _3;
}
}
else{
__label__ = 16;
break _3;
}
}
switch(__label__ ){//multiple entries
case 16:
        r0 = maxIterations;
        r0 = r0 >> 2;
        r1 = heap32[(r0)];
if(!(r1 >=r6)) //_LBB150_18
{
        heap32[(r0)] = r6;
}
        return;
break;
case 13:
        r0 = _2E_str941;
        r1 = _2E_str313;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 75;
        _assert(i7);
break;
case 8:
        r0 = _2E_str739;
        r1 = _2E_str313;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 80;
        _assert(i7);
break;
case 4:
        r0 = _2E_str1143;
        r1 = _2E_str537;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 703;
        _assert(i7);
break;
}
}
else{
        r0 = _2E_str212;
        r1 = _2E_str537;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 669;
        _assert(i7);
}
}

function _ZNK14btQuantizedBvh17quantizeWithClampEPtRK9btVector3i(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var f0;
        var f1;
        var f2;
        var f3;
        var f4;
var __label__ = 0;
        i7 = sp + -32;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = heapU8[r0+60];
        if(r1 !=0) //_LBB151_2
{
        r1 = heap32[(fp+1)];
        r2 = heap32[(fp+2)];
        r3 = heap32[(fp+3)];
        r2 = r2 >> 2;
        f0 = heapFloat[(r2)];
        r4 = sp + -16;
        heapFloat[(fp+-4)] = f0;
        f1 = heapFloat[(r2+1)];
        r5 = r4 >> 2;
        heapFloat[(r5+1)] = f1;
        f2 = heapFloat[(r2+2)];
        heapFloat[(r5+2)] = f2;
        f3 = heapFloat[(r2+3)];
        r2 = r0 >> 2;
        heapFloat[(r5+3)] = f3;
        f4 = heapFloat[(r2+1)];
        if(f0 <f4) //_LBB151_4
{
        heapFloat[(fp+-4)] = f4;
        f0 = f4;
}
        f4 = heapFloat[(r2+2)];
        if(f1 <f4) //_LBB151_7
{
        heapFloat[(r5+1)] = f4;
        f1 = f4;
}
        f4 = heapFloat[(r2+3)];
        if(f2 <f4) //_LBB151_10
{
        heapFloat[(r5+2)] = f4;
        f2 = f4;
}
        f4 = heapFloat[(r2+4)];
        if(f3 <f4) //_LBB151_13
{
        heapFloat[(r5+3)] = f4;
        f3 = f4;
}
        f4 = heapFloat[(r2+5)];
if(!(f4 >=f0)) //_LBB151_16
{
        heapFloat[(fp+-4)] = f4;
}
        f0 = heapFloat[(r2+6)];
if(!(f0 >=f1)) //_LBB151_18
{
        heapFloat[(r5+1)] = f0;
}
        f0 = heapFloat[(r2+7)];
if(!(f0 >=f2)) //_LBB151_20
{
        heapFloat[(r5+2)] = f0;
}
        f0 = heapFloat[(r2+8)];
if(!(f0 >=f3)) //_LBB151_22
{
        heapFloat[(r5+3)] = f0;
}
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = r4;
        heap32[(g0+3)] = r3;
        _ZNK14btQuantizedBvh8quantizeEPtRK9btVector3i(i7);
        return;
}
else{
        r0 = _2E_str212;
        r1 = _2E_str313;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 420;
        _assert(i7);
}
}

function _ZNK14btQuantizedBvh42walkRecursiveQuantizedTreeAgainstQueryAabbEPK18btQuantizedBvhNodeP21btNodeOverlapCallbackPtS5_(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
var __label__ = 0;
        i7 = sp + -24;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = heap32[(fp+1)];
        r2 = heap32[(fp+2)];
        r3 = heap32[(fp+3)];
        r4 = heap32[(fp+4)];
_1: while(true){
        r5 = heapU8[r0+60];
        if(r5 !=0) //_LBB152_3
{
        r5 = heapU16[(r3)>>1];
        r6 = heapU16[(r1+6)>>1];
        if(uint(r5) >uint(r6)) //_LBB152_14
{
__label__ = 14;
break _1;
}
else{
        r5 = heapU16[(r4)>>1];
        r6 = heapU16[(r1)>>1];
        r5 = r5 & 65535;
        r6 = r6 & 65535;
        if(uint(r5) <uint(r6)) //_LBB152_14
{
__label__ = 14;
break _1;
}
else{
        r5 = heapU16[(r3+4)>>1];
        r6 = heapU16[(r1+10)>>1];
        r5 = r5 & 65535;
        r6 = r6 & 65535;
        if(uint(r5) >uint(r6)) //_LBB152_14
{
__label__ = 14;
break _1;
}
else{
        r5 = heapU16[(r4+4)>>1];
        r6 = heapU16[(r1+4)>>1];
        r5 = r5 & 65535;
        r6 = r6 & 65535;
        if(uint(r5) <uint(r6)) //_LBB152_14
{
__label__ = 14;
break _1;
}
else{
        r5 = heapU16[(r3+2)>>1];
        r6 = heapU16[(r1+8)>>1];
        r5 = r5 & 65535;
        r6 = r6 & 65535;
        if(uint(r5) >uint(r6)) //_LBB152_14
{
__label__ = 14;
break _1;
}
else{
        r5 = heapU16[(r4+2)>>1];
        r6 = heapU16[(r1+2)>>1];
        r5 = r5 & 65535;
        r6 = r6 & 65535;
        if(uint(r5) <uint(r6)) //_LBB152_14
{
__label__ = 14;
break _1;
}
else{
        r5 = r1 >> 2;
        r6 = heap32[(r5+3)];
        if(r6 <0) //_LBB152_11
{
        r6 = (r1 + 16)|0;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r6;
        heap32[(g0+2)] = r2;
        heap32[(g0+3)] = r3;
        heap32[(g0+4)] = r4;
        _ZNK14btQuantizedBvh42walkRecursiveQuantizedTreeAgainstQueryAabbEPK18btQuantizedBvhNodeP21btNodeOverlapCallbackPtS5_(i7);
        r5 = heap32[(r5+7)];
        if(r5 <0) //_LBB152_13
{
        r6 = 1;
        r5 = (r6 - r5)|0;
        r5 = r5 << 4;
        r1 = (r1 + r5)|0;
continue _1;
}
else{
        r1 = (r1 + 32)|0;
continue _1;
}
}
else{
__label__ = 10;
break _1;
}
}
}
}
}
}
}
}
else{
__label__ = 2;
break _1;
}
}
switch(__label__ ){//multiple entries
case 14:
        return;
break;
case 2:
        r0 = _2E_str212;
        r1 = _2E_str537;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 419;
        _assert(i7);
break;
case 10:
        r0 = r2 >> 2;
        r0 = heap32[(r0)];
        r0 = r0 >> 2;
        r0 = heap32[(r0+2)];
        r1 = r6 >> 21;
        r3 = r6 & 2097151;
        heap32[(g0)] = r2;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = r3;
        __FUNCTION_TABLE__[(r0)>>2](i7);
        return;
break;
}
}

function _ZNK14btQuantizedBvh26reportAabbOverlappingNodexEP21btNodeOverlapCallbackRK9btVector3S4_(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var r12;
        var f0;
        var f1;
var __label__ = 0;
        i7 = sp + -40;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = heap32[(fp+1)];
        r2 = heap32[(fp+2)];
        r3 = heap32[(fp+3)];
        r4 = heapU8[r0+60];
_1: do {
        if(r4 ==0) //_LBB153_20
{
        r0 = r0 >> 2;
        r4 = heap32[(r0+24)];
        r5 = 0;
        r6 = r5;
_3: while(true){
        r7 = heap32[(r0+14)];
        if(r7 >r5) //_LBB153_21
{
        if(r7 >r6) //_LBB153_23
{
        r7 = r2 >> 2;
        r8 = r4 >> 2;
        r6 = (r6 + 1)|0;
        f0 = heapFloat[(r7)];
        f1 = heapFloat[(r8+4)];
        if(f0 >f1) //_LBB153_26
{
__label__ = 26;
}
else{
        r9 = r3 >> 2;
        f0 = heapFloat[(r9)];
        f1 = heapFloat[(r8)];
        if(f0 <f1) //_LBB153_26
{
__label__ = 26;
}
else{
        r9 = 1;
__label__ = 27;
}
}
if (__label__ == 26){
        r9 = 0;
}
        f0 = heapFloat[(r7+2)];
        f1 = heapFloat[(r8+6)];
        if(f0 >f1) //_LBB153_30
{
__label__ = 29;
}
else{
        r10 = r3 >> 2;
        f0 = heapFloat[(r10+2)];
        f1 = heapFloat[(r8+2)];
        if(f0 <f1) //_LBB153_30
{
__label__ = 29;
}
else{
__label__ = 30;
}
}
if (__label__ == 29){
        r9 = 0;
}
        f0 = heapFloat[(r7+1)];
        f1 = heapFloat[(r8+5)];
        if(f0 <=f1) //_LBB153_33
{
        r7 = r3 >> 2;
        f0 = heapFloat[(r7+1)];
        f1 = heapFloat[(r8+1)];
        r7 = 0;
        r10 = heap32[(r8+8)];
        r11 = -1;
        r9 = f0 < f1 ? r7 : r9;
        r7 = r10 == r11;
        r12 = r9 & 255;
if(!(r12 ==0)) //_LBB153_35
{
        r10 = r10 != r11;
        r10 = r10 & 1;
        if(r10 ==0) //_LBB153_36
{
        r10 = r1 >> 2;
        r10 = heap32[(r10)];
        r10 = r10 >> 2;
        r10 = heap32[(r10+2)];
        r11 = heap32[(r8+10)];
        r12 = heap32[(r8+9)];
        heap32[(g0)] = r1;
        heap32[(g0+1)] = r12;
        heap32[(g0+2)] = r11;
        __FUNCTION_TABLE__[(r10)>>2](i7);
}
}
}
else{
        r7 = heap32[(r8+8)];
        r9 = -1;
        r7 = r7 == r9;
        r9 = 0;
}
        r9 = r9 & 255;
if(!(r9 !=0)) //_LBB153_39
{
        r7 = r7 & 1;
        if(r7 ==0) //_LBB153_40
{
        r7 = heap32[(r8+8)];
        r8 = r7 << 6;
        r4 = (r4 + r8)|0;
        r5 = (r7 + r5)|0;
continue _3;
}
}
        r4 = (r4 + 64)|0;
        r5 = (r5 + 1)|0;
}
else{
__label__ = 22;
break _3;
}
}
else{
__label__ = 40;
break _3;
}
}
switch(__label__ ){//multiple entries
case 40:
        r0 = maxIterations;
        r0 = r0 >> 2;
        r1 = heap32[(r0)];
        if(r1 >=r6) //_LBB153_17
{
break _1;
}
else{
        heap32[(r0)] = r6;
        return;
}
break;
case 22:
        r0 = _2E_str1921;
        r1 = _2E_str537;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 367;
        _assert(i7);
break;
}
}
else{
        r4 = sp + -6;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r4;
        heap32[(g0+2)] = r2;
        heap32[(g0+3)] = 0;
        _ZNK14btQuantizedBvh17quantizeWithClampEPtRK9btVector3i(i7);
        r2 = sp + -12;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r2;
        heap32[(g0+2)] = r3;
        heap32[(g0+3)] = 1;
        r3 = r0 >> 2;
        _ZNK14btQuantizedBvh17quantizeWithClampEPtRK9btVector3i(i7);
        r5 = heap32[(r3+36)];
        if(r5 ==2) //_LBB153_18
{
        r3 = heap32[(r3+34)];
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r3;
        heap32[(g0+2)] = r1;
        heap32[(g0+3)] = r4;
        heap32[(g0+4)] = r2;
        _ZNK14btQuantizedBvh42walkRecursiveQuantizedTreeAgainstQueryAabbEPK18btQuantizedBvhNodeP21btNodeOverlapCallbackPtS5_(i7);
        return;
}
else{
        if(r5 ==1) //_LBB153_5
{
        r5 = heapU8[r0+60];
        if(r5 ==0) //_LBB153_7
{
        r2 = _2E_str212;
        r3 = _2E_str537;
        heap32[(g0)] = r2;
        heap32[(g0+1)] = r3;
        heap32[(g0+2)] = 735;
        _assert(i7);
}
else{
        r5 = heap32[(r3+38)];
        if(r5 >0) //_LBB153_8
{
        r5 = 0;
_42: while(true){
        r6 = heap32[(r3+40)];
        r7 = r5 << 5;
        r8 = (r6 + r7)|0;
        r9 = heapU16[(sp+-6)>>1];
        r10 = heapU16[(r8+6)>>1];
if(!(uint(r9) >uint(r10))) //_LBB153_16
{
        r9 = heapU16[(sp+-12)>>1];
        r6 = heapU16[(r6+r7)>>1];
        r7 = r9 & 65535;
        r6 = r6 & 65535;
if(!(uint(r7) <uint(r6))) //_LBB153_16
{
        r6 = heapU16[(sp+-2)>>1];
        r7 = heapU16[(r8+10)>>1];
        r6 = r6 & 65535;
        r7 = r7 & 65535;
if(!(uint(r6) >uint(r7))) //_LBB153_16
{
        r6 = heapU16[(sp+-8)>>1];
        r7 = heapU16[(r8+4)>>1];
        r6 = r6 & 65535;
        r7 = r7 & 65535;
if(!(uint(r6) <uint(r7))) //_LBB153_16
{
        r6 = heapU16[(sp+-4)>>1];
        r7 = heapU16[(r8+8)>>1];
        r6 = r6 & 65535;
        r7 = r7 & 65535;
if(!(uint(r6) >uint(r7))) //_LBB153_16
{
        r6 = heapU16[(sp+-10)>>1];
        r7 = heapU16[(r8+2)>>1];
        r6 = r6 & 65535;
        r7 = r7 & 65535;
if(!(uint(r6) <uint(r7))) //_LBB153_16
{
        r6 = r8 >> 2;
        r7 = heap32[(r6+3)];
        r6 = heap32[(r6+4)];
        r6 = (r6 + r7)|0;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = r4;
        heap32[(g0+3)] = r2;
        heap32[(g0+4)] = r7;
        heap32[(g0+5)] = r6;
        _ZNK14btQuantizedBvh26walkStacklessQuantizedTreeEP21btNodeOverlapCallbackPtS2_ii(i7);
}
}
}
}
}
}
        r5 = (r5 + 1)|0;
        r6 = heap32[(r3+38)];
        if(r6 >r5) //_LBB153_9
{
continue _42;
}
else{
break _1;
}
}
}
else{
break _1;
}
}
}
else{
        if(r5 !=0) //_LBB153_19
{
        r0 = _2E_str10;
        r1 = _2E_str537;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 341;
        _assert(i7);
}
else{
        r3 = heap32[(r3+14)];
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = r4;
        heap32[(g0+3)] = r2;
        heap32[(g0+4)] = 0;
        heap32[(g0+5)] = r3;
        _ZNK14btQuantizedBvh26walkStacklessQuantizedTreeEP21btNodeOverlapCallbackPtS2_ii(i7);
        return;
}
}
}
}
} while(0);
        return;
}

function _ZN14btQuantizedBvh9buildTreeEii(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var r12;
        var r13;
        var r14;
        var r15;
        var r16;
        var r17;
        var r18;
        var r19;
        var r20;
        var r21;
        var r22;
        var r23;
        var r24;
        var r25;
        var f0;
        var f1;
        var f2;
        var f3;
        var f4;
        var f5;
        var f6;
        var f7;
        var f8;
        var f9;
        var f10;
        var f11;
        var f12;
        var f13;
        var f14;
        var f15;
        var f16;
        var f17;
var __label__ = 0;
        i7 = sp + -112;var g0 = i7>>2; // save stack
        r0 = heap32[(fp+1)];
        r1 = heap32[(fp+2)];
        r2 = (r1 - r0)|0;
        if(r2 >0) //_LBB154_2
{
        r3 = heap32[(fp)];
        r4 = r3 >> 2;
        r5 = heap32[(r4+14)];
        if(r2 !=1) //_LBB154_7
{
_5: do {
        if(r0 <r1) //_LBB154_9
{
        r6 = heapU8[r3+60];
        f1 =                         0;
        r7 = r0;
        f2 = f1;
        f3 = f1;
_7: while(true){
        r8 = r6 & 255;
        if(r8 ==0) //_LBB154_12
{
        r9 = heap32[(r4+19)];
        r10 = r7 << 6;
        r9 = (r9 + r10)|0;
        r9 = r9 >> 2;
        f0 = heapFloat[(r9)];
        f5 = heapFloat[(r9+1)];
        f8 = heapFloat[(r9+2)];
        f4 = heapFloat[(r9+4)];
        f6 = heapFloat[(r9+5)];
        f7 = heapFloat[(r9+6)];
}
else{
        r9 = heap32[(r4+29)];
        r10 = r7 << 4;
        r11 = (r9 + r10)|0;
        r9 = heapU16[(r9+r10)>>1];
        r10 = heapU16[(r11+2)>>1];
        r12 = heapU16[(r11+4)>>1];
        r13 = heapU16[(r11+6)>>1];
        r14 = heapU16[(r11+8)>>1];
        r11 = heapU16[(r11+10)>>1];
        f0 = uint(r9); //fuitos r9, f0
        f4 = heapFloat[(r4+9)];
        f5 = uint(r13); //fuitos r13, f5
        f6 = uint(r10); //fuitos r10, f6
        f7 = heapFloat[(r4+10)];
        f8 = uint(r14); //fuitos r14, f8
        f9 = uint(r12); //fuitos r12, f9
        f10 = heapFloat[(r4+11)];
        f11 = uint(r11); //fuitos r11, f11
        f0 = f0/f4;
        f12 = heapFloat[(r4+1)];
        f4 = f5/f4;
        f5 = f6/f7;
        f6 = heapFloat[(r4+2)];
        f7 = f8/f7;
        f8 = f9/f10;
        f9 = heapFloat[(r4+3)];
        f10 = f11/f10;
        f0 = f0+f12;
        f5 = f5+f6;
        f8 = f8+f9;
        f4 = f4+f12;
        f6 = f7+f6;
        f7 = f10+f9;
}
        f0 = f4+f0;
        f4 =                       0.5;
        f5 = f6+f5;
        f6 = f7+f8;
        f0 = f0*f4;
        f5 = f5*f4;
        f6 = f6*f4;
        r7 = (r7 + 1)|0;
        f3 = f3+f0;
        f2 = f2+f5;
        f1 = f1+f6;
if(!(r1 !=r7)) //_LBB154_10
{
break _7;
}
}
        f0 = r2; //fitos r2, f0
        if(r0 <r1) //_LBB154_16
{
        f5 =                         1;
        f5 = f5/f0;
        f6 = f3*f5;
        f7 = f2*f5;
        f5 = f1*f5;
        f1 =                         0;
        r6 = r0;
        f2 = f1;
        f3 = f1;
_16: while(true){
        if(r8 ==0) //_LBB154_19
{
        r7 = heap32[(r4+19)];
        r9 = r6 << 6;
        r7 = (r7 + r9)|0;
        r7 = r7 >> 2;
        f8 = heapFloat[(r7)];
        f10 = heapFloat[(r7+1)];
        f13 = heapFloat[(r7+2)];
        f9 = heapFloat[(r7+4)];
        f11 = heapFloat[(r7+5)];
        f12 = heapFloat[(r7+6)];
}
else{
        r7 = heap32[(r4+29)];
        r9 = r6 << 4;
        r10 = (r7 + r9)|0;
        r7 = heapU16[(r7+r9)>>1];
        r9 = heapU16[(r10+2)>>1];
        r11 = heapU16[(r10+4)>>1];
        r12 = heapU16[(r10+6)>>1];
        r13 = heapU16[(r10+8)>>1];
        r10 = heapU16[(r10+10)>>1];
        f8 = uint(r7); //fuitos r7, f8
        f9 = heapFloat[(r4+9)];
        f10 = uint(r12); //fuitos r12, f10
        f11 = uint(r9); //fuitos r9, f11
        f12 = heapFloat[(r4+10)];
        f13 = uint(r13); //fuitos r13, f13
        f14 = uint(r11); //fuitos r11, f14
        f15 = heapFloat[(r4+11)];
        f16 = uint(r10); //fuitos r10, f16
        f8 = f8/f9;
        f17 = heapFloat[(r4+1)];
        f9 = f10/f9;
        f10 = f11/f12;
        f11 = heapFloat[(r4+2)];
        f12 = f13/f12;
        f13 = f14/f15;
        f14 = heapFloat[(r4+3)];
        f15 = f16/f15;
        f8 = f8+f17;
        f10 = f10+f11;
        f13 = f13+f14;
        f9 = f9+f17;
        f11 = f12+f11;
        f12 = f15+f14;
}
        f8 = f9+f8;
        f9 = f11+f10;
        f10 = f12+f13;
        f8 = f8*f4;
        f9 = f9*f4;
        f10 = f10*f4;
        f8 = f8-f6;
        f9 = f9-f7;
        f10 = f10-f5;
        f8 = f8*f8;
        f9 = f9*f9;
        f10 = f10*f10;
        r6 = (r6 + 1)|0;
        f3 = f3+f8;
        f2 = f2+f9;
        f1 = f1+f10;
if(!(r1 !=r6)) //_LBB154_17
{
break _5;
}
}
}
else{
        f1 =                         0;
        f2 = f1;
        f3 = f1;
}
}
else{
        f0 = r2; //fitos r2, f0
        f1 =                         0;
        f2 = f1;
        f3 = f1;
}
} while(0);
        f4 =                        -1;
        f5 =                         1;
        f0 = f0+f4;
        f0 = f5/f0;
        f3 = f3*f0;
        f2 = f2*f0;
        f0 = f1*f0;
        if(f3 >=f2) //_LBB154_23
{
        r6 = 2;
        r7 = 0;
        r6 = f3 < f0 ? r6 : r7;
}
else{
        r6 = 2;
        r7 = 1;
        r6 = f2 < f0 ? r6 : r7;
}
        r7 = sp + -32;
        r8 = r7 >> 2;
        heap32[(fp+-8)] = 0;
        heap32[(r8+1)] = 0;
        heap32[(r8+2)] = 0;
        heap32[(r8+3)] = 0;
        if(r0 <r1) //_LBB154_26
{
        r9 = heapU8[r3+60];
        f0 =                         0;
        r10 = r0;
        f1 = f0;
        f2 = f0;
_31: while(true){
        r11 = r9 & 255;
        if(r11 ==0) //_LBB154_29
{
        r11 = heap32[(r4+19)];
        r12 = r10 << 6;
        r11 = (r11 + r12)|0;
        r11 = r11 >> 2;
        f3 = heapFloat[(r11)];
        f6 = heapFloat[(r11+1)];
        f9 = heapFloat[(r11+2)];
        f4 = heapFloat[(r11+4)];
        f7 = heapFloat[(r11+5)];
        f8 = heapFloat[(r11+6)];
}
else{
        r11 = heap32[(r4+29)];
        r12 = r10 << 4;
        r13 = (r11 + r12)|0;
        r11 = heapU16[(r11+r12)>>1];
        r12 = heapU16[(r13+2)>>1];
        r14 = heapU16[(r13+4)>>1];
        r15 = heapU16[(r13+6)>>1];
        r16 = heapU16[(r13+8)>>1];
        r13 = heapU16[(r13+10)>>1];
        f3 = uint(r11); //fuitos r11, f3
        f4 = heapFloat[(r4+9)];
        f6 = uint(r15); //fuitos r15, f6
        f7 = uint(r12); //fuitos r12, f7
        f8 = heapFloat[(r4+10)];
        f9 = uint(r16); //fuitos r16, f9
        f10 = uint(r14); //fuitos r14, f10
        f11 = heapFloat[(r4+11)];
        f12 = uint(r13); //fuitos r13, f12
        f3 = f3/f4;
        f13 = heapFloat[(r4+1)];
        f4 = f6/f4;
        f6 = f7/f8;
        f7 = heapFloat[(r4+2)];
        f8 = f9/f8;
        f9 = f10/f11;
        f10 = heapFloat[(r4+3)];
        f11 = f12/f11;
        f3 = f3+f13;
        f6 = f6+f7;
        f9 = f9+f10;
        f4 = f4+f13;
        f7 = f8+f7;
        f8 = f11+f10;
}
        f3 = f4+f3;
        f4 =                       0.5;
        f6 = f7+f6;
        f7 = f8+f9;
        f3 = f3*f4;
        f6 = f6*f4;
        f4 = f7*f4;
        r10 = (r10 + 1)|0;
        f2 = f2+f3;
        f1 = f1+f6;
        f0 = f0+f4;
if(!(r1 !=r10)) //_LBB154_27
{
break _31;
}
}
        heapFloat[(r8+2)] = f0;
        heapFloat[(r8+1)] = f1;
        heapFloat[(fp+-8)] = f2;
}
else{
        f0 =                         0;
        f1 = f0;
        f2 = f0;
}
        f3 = r2; //fitos r2, f3
        f3 = f5/f3;
        f2 = f2*f3;
        f1 = f1*f3;
        heapFloat[(fp+-8)] = f2;
        f0 = f0*f3;
        heapFloat[(r8+1)] = f1;
        heapFloat[(r8+2)] = f0;
_40: do {
        if(r0 <r1) //_LBB154_34
{
        r8 = r6 << 2;
        r6 = (r7 + r8)|0;
        r6 = r6 >> 2;
        f0 = heapFloat[(r6)];
        r6 = r0 << 4;
        r7 = (r0 - r1)|0;
        r9 = r0 << 6;
        r10 = r6 | 12;
        r11 = 0;
        r12 = r9;
        r6 = r0;
_42: while(true){
        r13 = heapU8[r3+60];
        if(r13 ==0) //_LBB154_37
{
        r14 = r11 << 4;
        r15 = heap32[(r4+19)];
        r15 = (r15 + r9)|0;
        r14 = r14 << 2;
        r14 = (r15 - r14)|0;
        r14 = r14 >> 2;
        f1 = heapFloat[(r14)];
        f3 = heapFloat[(r14+1)];
        f6 = heapFloat[(r14+2)];
        f2 = heapFloat[(r14+4)];
        f4 = heapFloat[(r14+5)];
        f5 = heapFloat[(r14+6)];
}
else{
        r14 = heap32[(r4+29)];
        r14 = (r14 + r10)|0;
        r15 = r11 << 4;
        r14 = (r14 - r15)|0;
        r15 = heapU16[(r14+-12)>>1];
        r16 = heapU16[(r14+-10)>>1];
        r17 = heapU16[(r14+-8)>>1];
        r18 = heapU16[(r14+-6)>>1];
        r19 = heapU16[(r14+-4)>>1];
        r14 = heapU16[(r14+-2)>>1];
        f1 = uint(r15); //fuitos r15, f1
        f2 = heapFloat[(r4+9)];
        f3 = uint(r18); //fuitos r18, f3
        f4 = uint(r16); //fuitos r16, f4
        f5 = heapFloat[(r4+10)];
        f6 = uint(r19); //fuitos r19, f6
        f7 = uint(r17); //fuitos r17, f7
        f8 = heapFloat[(r4+11)];
        f9 = uint(r14); //fuitos r14, f9
        f1 = f1/f2;
        f10 = heapFloat[(r4+1)];
        f2 = f3/f2;
        f3 = f4/f5;
        f4 = heapFloat[(r4+2)];
        f5 = f6/f5;
        f6 = f7/f8;
        f7 = heapFloat[(r4+3)];
        f8 = f9/f8;
        f1 = f1+f10;
        f3 = f3+f4;
        f6 = f6+f7;
        f2 = f2+f10;
        f4 = f5+f4;
        f5 = f8+f7;
}
        f1 = f2+f1;
        f2 =                       0.5;
        r14 = sp + -48;
        f3 = f4+f3;
        f1 = f1*f2;
        f4 = f5+f6;
        r15 = r14 >> 2;
        f3 = f3*f2;
        heapFloat[(fp+-12)] = f1;
        f1 = f4*f2;
        heapFloat[(r15+1)] = f3;
        r14 = (r14 + r8)|0;
        heapFloat[(r15+2)] = f1;
        r14 = r14 >> 2;
        heap32[(r15+3)] = 0;
        f1 = heapFloat[(r14)];
        if(f1 >f0) //_LBB154_40
{
        if(r13 ==0) //_LBB154_42
{
        r13 = r11 << 4;
        r14 = heap32[(r4+19)];
        r15 = (r14 + r9)|0;
        r13 = r13 << 2;
        r13 = (r15 - r13)|0;
        r13 = r13 >> 2;
        r15 = heap32[(r13+15)];
        r16 = heap32[(r13+14)];
        r17 = heap32[(r13+13)];
        r18 = heap32[(r13+12)];
        r19 = heap32[(r13+11)];
        r20 = heap32[(r13+10)];
        r21 = heap32[(r13+9)];
        r22 = heap32[(r13+8)];
        f1 = heapFloat[(r13+7)];
        f2 = heapFloat[(r13+6)];
        f3 = heapFloat[(r13+5)];
        f4 = heapFloat[(r13+4)];
        f5 = heapFloat[(r13+3)];
        f6 = heapFloat[(r13+2)];
        f7 = heapFloat[(r13+1)];
        f8 = heapFloat[(r13)];
        r13 = r6 << 6;
        r23 = (r14 + r12)|0;
        r14 = (r14 + r13)|0;
        heap32[(g0)] = r23;
        heap32[(g0+1)] = r14;
        heap32[(g0+2)] = 64;
        memcpy(i7);
        r14 = heap32[(r4+19)];
        r13 = (r14 + r13)|0;
        r13 = r13 >> 2;
        heapFloat[(r13)] = f8;
        heapFloat[(r13+1)] = f7;
        heapFloat[(r13+2)] = f6;
        heapFloat[(r13+3)] = f5;
        heapFloat[(r13+4)] = f4;
        heapFloat[(r13+5)] = f3;
        heapFloat[(r13+6)] = f2;
        heapFloat[(r13+7)] = f1;
        heap32[(r13+8)] = r22;
        heap32[(r13+9)] = r21;
        heap32[(r13+10)] = r20;
        heap32[(r13+11)] = r19;
        heap32[(r13+12)] = r18;
        heap32[(r13+13)] = r17;
        heap32[(r13+14)] = r16;
        heap32[(r13+15)] = r15;
}
else{
        r13 = heap32[(r4+29)];
        r14 = (r13 + r10)|0;
        r15 = r11 << 4;
        r14 = (r14 - r15)|0;
        r15 = r6 << 4;
        r16 = r14 >> 2;
        r17 = heap32[(r16)];
        r18 = heapU16[(r14+-2)>>1];
        r19 = heapU16[(r14+-4)>>1];
        r20 = heapU16[(r14+-6)>>1];
        r21 = heapU16[(r14+-8)>>1];
        r22 = heapU16[(r14+-10)>>1];
        r23 = heapU16[(r13+r15)>>1];
        r24 = heapU16[(r14+-12)>>1];
        r13 = (r13 + r15)|0;
        heap16[(r14+-12)>>1] = r23;
        r23 = heapU16[(r13+2)>>1];
        heap16[(r14+-10)>>1] = r23;
        r23 = heapU16[(r13+4)>>1];
        heap16[(r14+-8)>>1] = r23;
        r23 = heapU16[(r13+6)>>1];
        heap16[(r14+-6)>>1] = r23;
        r23 = heapU16[(r13+8)>>1];
        heap16[(r14+-4)>>1] = r23;
        r23 = heapU16[(r13+10)>>1];
        r13 = r13 >> 2;
        heap16[(r14+-2)>>1] = r23;
        r13 = heap32[(r13+3)];
        heap32[(r16)] = r13;
        r13 = heap32[(r4+29)];
        r14 = (r13 + r15)|0;
        heap16[(r13+r15)>>1] = r24;
        heap16[(r14+2)>>1] = r22;
        heap16[(r14+4)>>1] = r21;
        heap16[(r14+6)>>1] = r20;
        heap16[(r14+8)>>1] = r19;
        r13 = r14 >> 2;
        heap16[(r14+10)>>1] = r18;
        heap32[(r13+3)] = r17;
}
        r6 = (r6 + 1)|0;
}
        r11 = (r11 + -1)|0;
        r12 = (r12 + 64)|0;
if(!(r7 !=r11)) //_LBB154_35
{
break _40;
}
}
}
else{
        r6 = r0;
}
} while(0);
        r7 = (r2 / 3)|0;
        r8 = (r7 + r0)|0;
        if(r8 >=r6) //_LBB154_48
{
__label__ = 46;
}
else{
        r8 = (r1 + -1)|0;
        r7 = (r8 - r7)|0;
        if(r7 <=r6) //_LBB154_48
{
__label__ = 46;
}
else{
__label__ = 47;
}
}
if (__label__ == 46){
        r6 = r2 >> 1;
        r6 = (r6 + r0)|0;
}
if(!(r6 ==r0)) //_LBB154_51
{
        if(r6 !=r1) //_LBB154_52
{
        r2 = heap32[(r4+14)];
        heap32[(fp+-21)] = r2;
        r2 = heapU8[r3+60];
        if(r2 ==0) //_LBB154_54
{
        r2 = heap32[(fp+-21)];
        r2 = r2 << 6;
        r7 = heap32[(r4+24)];
        r2 = (r7 + r2)|0;
        r2 = r2 >> 2;
        heap32[(r2)] = heap32[(r4+5)];
        heap32[(r2+1)] = heap32[(r4+6)];
        heap32[(r2+2)] = heap32[(r4+7)];
        heap32[(r2+3)] = heap32[(r4+8)];
}
else{
        r2 = heap32[(r4+34)];
        r7 = heap32[(fp+-21)];
        r7 = r7 << 4;
        r2 = (r2 + r7)|0;
        r7 = (r3 + 20)|0;
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r2;
        heap32[(g0+2)] = r7;
        heap32[(g0+3)] = 0;
        _ZNK14btQuantizedBvh8quantizeEPtRK9btVector3i(i7);
}
        r2 = heap32[(r4+14)];
        r7 = heapU8[r3+60];
        if(r7 ==0) //_LBB154_57
{
        r2 = r2 << 6;
        r7 = heap32[(r4+24)];
        r2 = (r7 + r2)|0;
        r2 = r2 >> 2;
        heap32[(r2+4)] = heap32[(r4+1)];
        heap32[(r2+5)] = heap32[(r4+2)];
        heap32[(r2+6)] = heap32[(r4+3)];
        heap32[(r2+7)] = heap32[(r4+4)];
}
else{
        r7 = heap32[(r4+34)];
        r2 = r2 << 4;
        r2 = (r7 + r2)|0;
        r2 = (r2 + 6)|0;
        r7 = (r3 + 4)|0;
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r2;
        heap32[(g0+2)] = r7;
        heap32[(g0+3)] = 1;
        _ZNK14btQuantizedBvh8quantizeEPtRK9btVector3i(i7);
}
_73: do {
if(!(r0 >=r1)) //_LBB154_95
{
        r2 = r0;
_75: while(true){
        r7 = heapU8[r3+60];
_77: do {
        if(r7 ==0) //_LBB154_69
{
        r7 = heap32[(r4+19)];
        r8 = r2 << 6;
        r7 = (r7 + r8)|0;
        r7 = r7 >> 2;
        f0 = heapFloat[(r7+4)];
        r8 = sp + -80;
        heapFloat[(fp+-20)] = f0;
        f1 = heapFloat[(r7+5)];
        r8 = r8 >> 2;
        heapFloat[(r8+1)] = f1;
        f2 = heapFloat[(r7+6)];
        heapFloat[(r8+2)] = f2;
        f3 = heapFloat[(r7+7)];
        heapFloat[(r8+3)] = f3;
        f4 = heapFloat[(r7)];
        r8 = sp + -64;
        heapFloat[(fp+-16)] = f4;
        f5 = heapFloat[(r7+1)];
        r8 = r8 >> 2;
        heapFloat[(r8+1)] = f5;
        f6 = heapFloat[(r7+2)];
        heapFloat[(r8+2)] = f6;
        f7 = heapFloat[(r7+3)];
        heapFloat[(r8+3)] = f7;
        r7 = heap32[(r4+14)];
        r8 = heap32[(r4+24)];
        r7 = r7 << 6;
        r8 = (r8 + r7)|0;
        r8 = r8 >> 2;
        f8 = heapFloat[(r8)];
if(!(f4 >=f8)) //_LBB154_71
{
        heapFloat[(r8)] = f4;
}
        f4 = heapFloat[(r8+1)];
if(!(f5 >=f4)) //_LBB154_73
{
        heapFloat[(r8+1)] = f5;
}
        f4 = heapFloat[(r8+2)];
if(!(f6 >=f4)) //_LBB154_75
{
        heapFloat[(r8+2)] = f6;
}
        f4 = heapFloat[(r8+3)];
if(!(f7 >=f4)) //_LBB154_77
{
        heapFloat[(r8+3)] = f7;
}
        r8 = heap32[(r4+24)];
        r7 = (r8 + r7)|0;
        r7 = r7 >> 2;
        f4 = heapFloat[(r7+4)];
if(!(f4 >=f0)) //_LBB154_79
{
        heapFloat[(r7+4)] = f0;
}
        f0 = heapFloat[(r7+5)];
if(!(f0 >=f1)) //_LBB154_81
{
        heapFloat[(r7+5)] = f1;
}
        f0 = heapFloat[(r7+6)];
if(!(f0 >=f2)) //_LBB154_83
{
        heapFloat[(r7+6)] = f2;
}
        f0 = heapFloat[(r7+7)];
        if(f0 >=f3) //_LBB154_94
{
break _77;
}
else{
        heapFloat[(r7+7)] = f3;
}
}
else{
        r7 = heap32[(r4+29)];
        r8 = r2 << 4;
        r9 = (r7 + r8)|0;
        r10 = sp + -80;
        r11 = heapU16[(r9+10)>>1];
        r12 = heapU16[(r9+8)>>1];
        r13 = heapU16[(r9+6)>>1];
        f0 = heapFloat[(r4+11)];
        f1 = heapFloat[(r4+10)];
        f2 = heapFloat[(r4+9)];
        r14 = r10 >> 2;
        f3 = uint(r13); //fuitos r13, f3
        heap32[(r14+3)] = 0;
        f3 = f3/f2;
        f4 = heapFloat[(r4+1)];
        f3 = f3+f4;
        f5 = uint(r12); //fuitos r12, f5
        heapFloat[(fp+-20)] = f3;
        f3 = f5/f1;
        f5 = heapFloat[(r4+2)];
        f3 = f3+f5;
        f6 = uint(r11); //fuitos r11, f6
        heapFloat[(r14+1)] = f3;
        f3 = f6/f0;
        f6 = heapFloat[(r4+3)];
        f3 = f3+f6;
        heapFloat[(r14+2)] = f3;
        r7 = heapU16[(r7+r8)>>1];
        r8 = sp + -64;
        f3 = uint(r7); //fuitos r7, f3
        r7 = heapU16[(r9+4)>>1];
        r9 = heapU16[(r9+2)>>1];
        f7 = uint(r9); //fuitos r9, f7
        f2 = f3/f2;
        r9 = r8 >> 2;
        f3 = uint(r7); //fuitos r7, f3
        f1 = f7/f1;
        f2 = f2+f4;
        heap32[(r9+3)] = 0;
        f0 = f3/f0;
        f1 = f1+f5;
        heapFloat[(fp+-16)] = f2;
        f0 = f0+f6;
        heapFloat[(r9+1)] = f1;
        heapFloat[(r9+2)] = f0;
        r7 = heap32[(r4+14)];
        r9 = sp + -6;
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r9;
        heap32[(g0+2)] = r8;
        heap32[(g0+3)] = 0;
        _ZNK14btQuantizedBvh8quantizeEPtRK9btVector3i(i7);
        r8 = sp + -12;
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r8;
        heap32[(g0+2)] = r10;
        heap32[(g0+3)] = 1;
        _ZNK14btQuantizedBvh8quantizeEPtRK9btVector3i(i7);
        r8 = heapU16[(sp+-6)>>1];
        r9 = heap32[(r4+34)];
        r7 = r7 << 4;
        r10 = heapU16[(r9+r7)>>1];
        if(uint(r10) >uint(r8)) //_LBB154_63
{
        heap16[(r9+r7)>>1] = r8;
        r9 = heap32[(r4+34)];
}
        r8 = (r9 + r7)|0;
        r10 = heapU16[(sp+-12)>>1];
        r11 = heapU16[(r8+6)>>1];
        if(uint(r11) <uint(r10)) //_LBB154_66
{
        heap16[(r8+6)>>1] = r10;
        r9 = heap32[(r4+34)];
}
        r8 = (r9 + r7)|0;
        r10 = heapU16[(sp+-4)>>1];
        r11 = heapU16[(r8+2)>>1];
        if(uint(r11) >uint(r10)) //_LBB154_89
{
        heap16[(r8+2)>>1] = r10;
        r9 = heap32[(r4+34)];
}
        r8 = (r9 + r7)|0;
        r10 = heapU16[(sp+-10)>>1];
        r11 = heapU16[(r8+8)>>1];
        if(uint(r11) <uint(r10)) //_LBB154_90
{
        heap16[(r8+8)>>1] = r10;
        r9 = heap32[(r4+34)];
}
        r8 = (r9 + r7)|0;
        r10 = heapU16[(sp+-2)>>1];
        r11 = heapU16[(r8+4)>>1];
        if(uint(r11) >uint(r10)) //_LBB154_92
{
        heap16[(r8+4)>>1] = r10;
        r9 = heap32[(r4+34)];
}
        r7 = (r9 + r7)|0;
        r8 = heapU16[(sp+-8)>>1];
        r9 = heapU16[(r7+10)>>1];
        if(uint(r9) <uint(r8)) //_LBB154_93
{
        heap16[(r7+10)>>1] = r8;
}
}
} while(0);
        r2 = (r2 + 1)|0;
if(!(r1 !=r2)) //_LBB154_60
{
break _73;
}
}
}
} while(0);
        r2 = heap32[(r4+14)];
        r7 = (r2 + 1)|0;
        heap32[(r4+14)] = r7;
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r0;
        heap32[(g0+2)] = r6;
        _ZN14btQuantizedBvh9buildTreeEii(i7);
        r0 = heap32[(r4+14)];
        heap32[(fp+-23)] = r0;
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r6;
        heap32[(g0+2)] = r1;
        _ZN14btQuantizedBvh9buildTreeEii(i7);
        r0 = heap32[(r4+14)];
        r0 = (r0 - r5)|0;
        heap32[(fp+-22)] = r0;
        r0 = heapU8[r3+60];
_120: do {
if(!(r0 ==0)) //_LBB154_143
{
        r0 = heap32[(fp+-22)];
        r0 = r0 << 4;
if(!(r0 <2049)) //_LBB154_142
{
        r0 = heap32[(r4+34)];
        r1 = heap32[(fp+-23)];
        r1 = r1 << 4;
        heap32[(fp+-24)] = r1;
        r2 = r2 << 4;
        r1 = (r0 + r1)|0;
        r2 = (r0 + r2)|0;
        r5 = r1 >> 2;
        r6 = r2 >> 2;
        r5 = heap32[(r5+3)];
        r8 = 0;
        r6 = heap32[(r6+7)];
        r9 = 1;
        r10 = (r8 - r6)|0;
        r6 = r6 < 0 ? r10 : r9;
        r10 = r6 << 4;
if(!(r10 >2048)) //_LBB154_119
{
        r10 = heap32[(r4+39)];
        r11 = heap32[(r4+38)];
        if(r10 ==r11) //_LBB154_100
{
        r12 = r11 << 1;
        r12 = r11 == 0 ? r9 : r12;
        if(r10 >=r12) //_LBB154_99
{
__label__ = 92;
}
else{
        if(r12 !=0) //_LBB154_103
{
        r10 = gNumAlignedAllocs;
        r10 = r10 >> 2;
        r13 = heap32[(r10)];
        r13 = (r13 + 1)|0;
        r14 = r12 << 5;
        heap32[(r10)] = r13;
        r10 = r14 | 19;
        heap32[(g0)] = r10;
        malloc(i7);
        r13 = r_g0;
        if(r13 !=0) //_LBB154_105
{
        r10 = 0;
        r14 = (r13 + 4)|0;
        r10 = (r10 - r14)|0;
        r10 = r10 & 15;
        r10 = (r13 + r10)|0;
        r14 = (r10 + 4)|0;
        r10 = r10 >> 2;
        heap32[(r10)] = r13;
        r13 = r14;
}
}
else{
        r13 = 0;
}
        r14 = (r3 + 160)|0;
        if(r11 <1) //_LBB154_108
{
        r10 = r14 >> 2;
        r16 = heap32[(r10)];
}
else{
        r10 = 0;
        r15 = r11;
_137: while(true){
        r16 = r14 >> 2;
        r16 = heap32[(r16)];
        r17 = (r16 + r10)|0;
        r17 = r17 >> 2;
        r18 = (r13 + r10)|0;
        r15 = (r15 + -1)|0;
        r10 = (r10 + 32)|0;
        r19 = heap32[(r17)];
        r18 = r18 >> 2;
        r20 = heap32[(r17+1)];
        r21 = heap32[(r17+2)];
        r22 = heap32[(r17+3)];
        r23 = heap32[(r17+4)];
        r24 = heap32[(r17+5)];
        r25 = heap32[(r17+6)];
        r17 = heap32[(r17+7)];
        heap32[(r18)] = r19;
        heap32[(r18+1)] = r20;
        heap32[(r18+2)] = r21;
        heap32[(r18+3)] = r22;
        heap32[(r18+4)] = r23;
        heap32[(r18+5)] = r24;
        heap32[(r18+6)] = r25;
        heap32[(r18+7)] = r17;
if(!(r15 !=0)) //_LBB154_109
{
break _137;
}
}
        r14 = (r3 + 160)|0;
}
        if(r16 !=0) //_LBB154_113
{
        r10 = heapU8[r3+164];
        if(r10 !=0) //_LBB154_115
{
        r10 = gNumAlignedFree;
        r10 = r10 >> 2;
        r15 = heap32[(r10)];
        r15 = (r15 + 1)|0;
        r16 = r16 >> 2;
        heap32[(r10)] = r15;
        r10 = heap32[(r16+-1)];
        heap32[(g0)] = r10;
        free(i7);
        r10 = heap32[(r4+38)];
}
else{
        r10 = r11;
}
        r15 = r14 >> 2;
        heap32[(r15)] = 0;
}
else{
        r10 = r11;
}
        r14 = r14 >> 2;
        heap8[r3+164] = r9;
        heap32[(r14)] = r13;
        heap32[(r4+39)] = r12;
__label__ = 110;
}
}
else{
__label__ = 92;
}
if (__label__ == 92){
        r10 = r11;
}
        r10 = (r10 + 1)|0;
        heap32[(r4+38)] = r10;
        r10 = heap32[(r4+40)];
        r12 = heapU16[(r2+16)>>1];
        r11 = r11 << 5;
        heap16[(r10+r11)>>1] = r12;
        r10 = (r10 + r11)|0;
        r11 = heapU16[(r2+18)>>1];
        heap16[(r10+2)>>1] = r11;
        r11 = heapU16[(r2+20)>>1];
        heap16[(r10+4)>>1] = r11;
        r11 = heapU16[(r2+22)>>1];
        heap16[(r10+6)>>1] = r11;
        r11 = heapU16[(r2+24)>>1];
        heap16[(r10+8)>>1] = r11;
        r2 = heapU16[(r2+26)>>1];
        r11 = r10 >> 2;
        heap16[(r10+10)>>1] = r2;
        heap32[(r11+3)] = r7;
        heap32[(r11+4)] = r6;
}
        r2 = (r8 - r5)|0;
        r2 = r5 < 0 ? r2 : r9;
        r5 = r2 << 4;
if(!(r5 >2048)) //_LBB154_141
{
        r5 = heap32[(r4+39)];
        r6 = heap32[(r4+38)];
        if(r5 ==r6) //_LBB154_122
{
        r7 = r6 << 1;
        r7 = r6 == 0 ? r9 : r7;
        if(r5 >=r7) //_LBB154_121
{
__label__ = 113;
}
else{
        if(r7 !=0) //_LBB154_125
{
        r5 = gNumAlignedAllocs;
        r5 = r5 >> 2;
        r10 = heap32[(r5)];
        r10 = (r10 + 1)|0;
        r11 = r7 << 5;
        heap32[(r5)] = r10;
        r5 = r11 | 19;
        heap32[(g0)] = r5;
        malloc(i7);
        r10 = r_g0;
        if(r10 !=0) //_LBB154_127
{
        r5 = (r10 + 4)|0;
        r5 = (r8 - r5)|0;
        r5 = r5 & 15;
        r5 = (r10 + r5)|0;
        r11 = (r5 + 4)|0;
        r5 = r5 >> 2;
        heap32[(r5)] = r10;
        r10 = r11;
}
}
else{
        r10 = 0;
}
        r11 = (r3 + 160)|0;
        if(r6 <1) //_LBB154_130
{
        r5 = r11 >> 2;
        r12 = heap32[(r5)];
}
else{
        r5 = r6;
_166: while(true){
        r12 = r11 >> 2;
        r12 = heap32[(r12)];
        r13 = (r12 + r8)|0;
        r13 = r13 >> 2;
        r14 = (r10 + r8)|0;
        r5 = (r5 + -1)|0;
        r8 = (r8 + 32)|0;
        r15 = heap32[(r13)];
        r14 = r14 >> 2;
        r16 = heap32[(r13+1)];
        r17 = heap32[(r13+2)];
        r18 = heap32[(r13+3)];
        r19 = heap32[(r13+4)];
        r20 = heap32[(r13+5)];
        r21 = heap32[(r13+6)];
        r13 = heap32[(r13+7)];
        heap32[(r14)] = r15;
        heap32[(r14+1)] = r16;
        heap32[(r14+2)] = r17;
        heap32[(r14+3)] = r18;
        heap32[(r14+4)] = r19;
        heap32[(r14+5)] = r20;
        heap32[(r14+6)] = r21;
        heap32[(r14+7)] = r13;
if(!(r5 !=0)) //_LBB154_131
{
break _166;
}
}
        r11 = (r3 + 160)|0;
}
        if(r12 !=0) //_LBB154_135
{
        r5 = heapU8[r3+164];
        if(r5 !=0) //_LBB154_137
{
        r5 = gNumAlignedFree;
        r5 = r5 >> 2;
        r8 = heap32[(r5)];
        r8 = (r8 + 1)|0;
        r12 = r12 >> 2;
        heap32[(r5)] = r8;
        r5 = heap32[(r12+-1)];
        heap32[(g0)] = r5;
        free(i7);
        r5 = heap32[(r4+38)];
}
else{
        r5 = r6;
}
        r8 = r11 >> 2;
        heap32[(r8)] = 0;
}
else{
        r5 = r6;
}
        r8 = r11 >> 2;
        heap8[r3+164] = r9;
        heap32[(r8)] = r10;
        heap32[(r4+39)] = r7;
__label__ = 131;
}
}
else{
__label__ = 113;
}
if (__label__ == 113){
        r5 = r6;
}
        r5 = (r5 + 1)|0;
        heap32[(r4+38)] = r5;
        r5 = heap32[(r4+40)];
        r7 = heap32[(fp+-24)];
        r0 = heapU16[(r0+r7)>>1];
        r6 = r6 << 5;
        heap16[(r5+r6)>>1] = r0;
        r0 = (r5 + r6)|0;
        r5 = heapU16[(r1+2)>>1];
        heap16[(r0+2)>>1] = r5;
        r5 = heapU16[(r1+4)>>1];
        heap16[(r0+4)>>1] = r5;
        r5 = heapU16[(r1+6)>>1];
        heap16[(r0+6)>>1] = r5;
        r5 = heapU16[(r1+8)>>1];
        heap16[(r0+8)>>1] = r5;
        r1 = heapU16[(r1+10)>>1];
        r5 = r0 >> 2;
        heap16[(r0+10)>>1] = r1;
        r0 = heap32[(fp+-23)];
        heap32[(r5+3)] = r0;
        heap32[(r5+4)] = r2;
}
        r0 = heap32[(r4+38)];
        heap32[(r4+42)] = r0;
        r0 = heapU8[r3+60];
        if(r0 ==0) //_LBB154_143
{
break _120;
}
}
        r0 = heap32[(fp+-21)];
        r0 = r0 << 4;
        r4 = heap32[(r4+34)];
        r4 = (r4 + r0)|0;
        r0 = 0;
        r4 = r4 >> 2;
        r1 = heap32[(fp+-22)];
        r0 = (r0 - r1)|0;
        heap32[(r4+3)] = r0;
        return;
}
} while(0);
        r0 = heap32[(fp+-21)];
        r0 = r0 << 6;
        r1 = heap32[(r4+24)];
        r0 = (r1 + r0)|0;
        r0 = r0 >> 2;
        r1 = heap32[(fp+-22)];
        heap32[(r0+8)] = r1;
        return;
}
}
        r0 = _2E_str21;
        r1 = _2E_str537;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 278;
        _assert(i7);
}
else{
        r1 = heapU8[r3+60];
        if(r1 ==0) //_LBB154_5
{
        r1 = heap32[(r4+24)];
        r2 = heap32[(r4+19)];
        r3 = r5 << 6;
        r0 = r0 << 6;
        r1 = (r1 + r3)|0;
        r0 = (r2 + r0)|0;
        heap32[(g0)] = r1;
        heap32[(g0+1)] = r0;
        heap32[(g0+2)] = 64;
        memcpy(i7);
}
else{
        r0 = r0 << 4;
        r1 = heap32[(r4+29)];
        r2 = heap32[(r4+34)];
        r3 = heapU16[(r1+r0)>>1];
        r5 = r5 << 4;
        r0 = (r1 + r0)|0;
        heap16[(r2+r5)>>1] = r3;
        r1 = (r2 + r5)|0;
        r2 = heapU16[(r0+2)>>1];
        heap16[(r1+2)>>1] = r2;
        r2 = heapU16[(r0+4)>>1];
        heap16[(r1+4)>>1] = r2;
        r2 = heapU16[(r0+6)>>1];
        heap16[(r1+6)>>1] = r2;
        r2 = heapU16[(r0+8)>>1];
        heap16[(r1+8)>>1] = r2;
        r2 = heapU16[(r0+10)>>1];
        r0 = r0 >> 2;
        heap16[(r1+10)>>1] = r2;
        r1 = r1 >> 2;
        r0 = heap32[(r0+3)];
        heap32[(r1+3)] = r0;
}
        r0 = heap32[(r4+14)];
        r0 = (r0 + 1)|0;
        heap32[(r4+14)] = r0;
        return;
}
}
else{
        r0 = _2E_str2246;
        r1 = _2E_str537;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 127;
        _assert(i7);
}
}

function _ZNK14btQuantizedBvh36walkStacklessQuantizedTreeAgainstRayEP21btNodeOverlapCallbackRK9btVector3S4_S4_S4_ii(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var r12;
        var r13;
        var r14;
        var r15;
        var r16;
        var r17;
        var r18;
        var r19;
        var r20;
        var r21;
        var f0;
        var f1;
        var f2;
        var f3;
        var f4;
        var f5;
        var f6;
        var f7;
        var f8;
        var f9;
        var f10;
        var f11;
        var f12;
        var f13;
        var f14;
        var f15;
        var f16;
        var f17;
        var f18;
        var f19;
var __label__ = 0;
        i7 = sp + -96;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = heapU8[r0+60];
        if(r1 !=0) //_LBB155_2
{
        r1 = heap32[(fp+1)];
        r2 = heap32[(fp+2)];
        r3 = heap32[(fp+3)];
        r4 = heap32[(fp+4)];
        r5 = heap32[(fp+5)];
        r6 = heap32[(fp+6)];
        r3 = r3 >> 2;
        r2 = r2 >> 2;
        f0 = heapFloat[(r3)];
        f1 = heapFloat[(r2)];
        f2 = heapFloat[(r3+1)];
        f3 = heapFloat[(r2+1)];
        f0 = f0-f1;
        r7 = r0 >> 2;
        f1 = f2-f3;
        f2 = heapFloat[(r3+2)];
        f3 = heapFloat[(r2+2)];
        r8 = heap32[(r7+34)];
        f2 = f2-f3;
        f3 = f0*f0;
        f4 = f1*f1;
        f3 = f3+f4;
        f4 = f2*f2;
        f3 = f3+f4;
        heapFloat[(g0)] = f3;
        sqrtf(i7);
        f4 =                         1;
        f3 = f4/f_g0;
        f5 = heapFloat[(r3)];
        f6 = heapFloat[(r2)];
        f7 = heapFloat[(r3+1)];
        f8 = heapFloat[(r2+1)];
        f0 = f0*f3;
        f9 = f5-f6;
        f1 = f1*f3;
        f10 = f7-f8;
        f11 = heapFloat[(r3+2)];
        f12 = heapFloat[(r2+2)];
        f2 = f2*f3;
        f3 = f11-f12;
        f9 = f0*f9;
        f10 = f1*f10;
        f9 = f9+f10;
        f3 = f2*f3;
        f3 = f9+f3;
        f9 =                         0;
        if(f0 !=f9) //_LBB155_4
{
        f0 = f4/f0;
}
else{
        f0 =        999999984306749440;
}
        if(f1 !=f9) //_LBB155_7
{
        f1 = f4/f1;
}
else{
        f1 =        999999984306749440;
}
        if(f2 !=f9) //_LBB155_10
{
        f2 = f4/f2;
}
else{
        f2 =        999999984306749440;
}
        r9 = heapU8[r0+60];
        if(r9 !=0) //_LBB155_13
{
        r4 = r4 >> 2;
        r5 = r5 >> 2;
        r9 = f0 < f9;
        r10 = f1 < f9;
        r11 = f2 < f9;
        f4 = heapFloat[(r2+3)];
        f10 = heapFloat[(r3+3)];
        f13 = f5 < f6 ? f5 : f6;
        f14 = heapFloat[(r4)];
        f15 = f7 < f8 ? f7 : f8;
        f16 = heapFloat[(r4+1)];
        f17 = f11 < f12 ? f11 : f12;
        f18 = heapFloat[(r4+2)];
        f5 = f6 < f5 ? f5 : f6;
        f6 = heapFloat[(r5)];
        f7 = f8 < f7 ? f7 : f8;
        f8 = heapFloat[(r5+1)];
        f11 = f12 < f11 ? f11 : f12;
        f12 = heapFloat[(r5+2)];
        r3 = r9 & 1;
        r9 = r10 & 1;
        r10 = r11 & 1;
        f19 = f10 < f4 ? f10 : f4;
        f4 = f4 < f10 ? f10 : f4;
        f10 = f13+f14;
        f13 = f15+f16;
        f14 = f17+f18;
        f5 = f5+f6;
        f6 = f7+f8;
        f7 = f11+f12;
        r11 = sp + -16;
        r12 = r11 >> 2;
        heapFloat[(fp+-4)] = f10;
        heapFloat[(r12+1)] = f13;
        heapFloat[(r12+2)] = f14;
        heapFloat[(r12+3)] = f19;
        f8 = heapFloat[(r7+1)];
        if(f10 <f8) //_LBB155_15
{
        heapFloat[(fp+-4)] = f8;
        f10 = f8;
}
        f8 = heapFloat[(r7+2)];
        if(f13 <f8) //_LBB155_18
{
        heapFloat[(r12+1)] = f8;
        f13 = f8;
}
        f8 = heapFloat[(r7+3)];
        if(f14 <f8) //_LBB155_21
{
        heapFloat[(r12+2)] = f8;
        f14 = f8;
}
        f8 = heapFloat[(r7+4)];
        if(f19 <f8) //_LBB155_24
{
        heapFloat[(r12+3)] = f8;
        f19 = f8;
}
        f8 = heapFloat[(r7+5)];
if(!(f8 >=f10)) //_LBB155_27
{
        heapFloat[(fp+-4)] = f8;
}
        f8 = heapFloat[(r7+6)];
if(!(f8 >=f13)) //_LBB155_29
{
        heapFloat[(r12+1)] = f8;
}
        f8 = heapFloat[(r7+7)];
if(!(f8 >=f14)) //_LBB155_31
{
        heapFloat[(r12+2)] = f8;
}
        f8 = heapFloat[(r7+8)];
if(!(f8 >=f19)) //_LBB155_33
{
        heapFloat[(r12+3)] = f8;
}
        r12 = sp + -38;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r12;
        heap32[(g0+2)] = r11;
        heap32[(g0+3)] = 0;
        _ZNK14btQuantizedBvh8quantizeEPtRK9btVector3i(i7);
        r11 = heapU8[r0+60];
        if(r11 !=0) //_LBB155_35
{
        r11 = sp + -32;
        r12 = r11 >> 2;
        heapFloat[(fp+-8)] = f5;
        heapFloat[(r12+1)] = f6;
        heapFloat[(r12+2)] = f7;
        heapFloat[(r12+3)] = f4;
        f8 = heapFloat[(r7+1)];
        if(f5 <f8) //_LBB155_37
{
        heapFloat[(fp+-8)] = f8;
        f5 = f8;
}
        f8 = heapFloat[(r7+2)];
        if(f6 <f8) //_LBB155_40
{
        heapFloat[(r12+1)] = f8;
        f6 = f8;
}
        f8 = heapFloat[(r7+3)];
        if(f7 <f8) //_LBB155_43
{
        heapFloat[(r12+2)] = f8;
        f7 = f8;
}
        f8 = heapFloat[(r7+4)];
        if(f4 <f8) //_LBB155_46
{
        heapFloat[(r12+3)] = f8;
        f4 = f8;
}
        f8 = heapFloat[(r7+5)];
if(!(f8 >=f5)) //_LBB155_49
{
        heapFloat[(fp+-8)] = f8;
}
        f5 = heapFloat[(r7+6)];
if(!(f5 >=f6)) //_LBB155_51
{
        heapFloat[(r12+1)] = f5;
}
        f5 = heapFloat[(r7+7)];
if(!(f5 >=f7)) //_LBB155_53
{
        heapFloat[(r12+2)] = f5;
}
        f5 = heapFloat[(r7+8)];
if(!(f5 >=f4)) //_LBB155_55
{
        heapFloat[(r12+3)] = f5;
}
        r12 = sp + -44;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r12;
        heap32[(g0+2)] = r11;
        heap32[(g0+3)] = 1;
        r0 = r3 ^ 1;
        r11 = r9 ^ 1;
        r12 = r10 ^ 1;
        r13 = 0;
        _ZNK14btQuantizedBvh8quantizeEPtRK9btVector3i(i7);
        r14 = r13;
_66: while(true){
        if(r13 <r6) //_LBB155_56
{
        if(r14 <r6) //_LBB155_58
{
        r15 = r8 >> 2;
        r16 = heap32[(r15+3)];
        r14 = (r14 + 1)|0;
        r16 = r16 >>> 31;
        r17 = heapU16[(sp+-38)>>1];
        r18 = heapU16[(r8+6)>>1];
        if(uint(r17) >uint(r18)) //_LBB155_64
{
__label__ = 55;
}
else{
        r17 = heapU16[(r8)>>1];
        r18 = heapU16[(sp+-44)>>1];
        r18 = r18 & 65535;
        r17 = r17 & 65535;
        if(uint(r18) <uint(r17)) //_LBB155_64
{
__label__ = 55;
}
else{
        r18 = heapU16[(sp+-34)>>1];
        r19 = heapU16[(r8+10)>>1];
        r18 = r18 & 65535;
        r19 = r19 & 65535;
        if(uint(r18) >uint(r19)) //_LBB155_64
{
__label__ = 55;
}
else{
        r18 = heapU16[(r8+4)>>1];
        r19 = heapU16[(sp+-40)>>1];
        r19 = r19 & 65535;
        r18 = r18 & 65535;
        if(uint(r19) <uint(r18)) //_LBB155_64
{
__label__ = 55;
}
else{
        r19 = heapU16[(sp+-36)>>1];
        r20 = heapU16[(r8+8)>>1];
        r19 = r19 & 65535;
        r20 = r20 & 65535;
        if(uint(r19) >uint(r20)) //_LBB155_64
{
__label__ = 55;
}
else{
        r19 = heapU16[(r8+2)>>1];
        r20 = heapU16[(sp+-42)>>1];
        r20 = r20 & 65535;
        r19 = r19 & 65535;
        if(uint(r20) >=uint(r19)) //_LBB155_65
{
        f4 = uint(r17); //fuitos r17, f4
        f5 = heapFloat[(r7+9)];
        f6 = uint(r19); //fuitos r19, f6
        f7 = heapFloat[(r7+10)];
        f4 = f4/f5;
        f8 = heapFloat[(r7+1)];
        f10 = heapFloat[(r7+11)];
        f11 = heapFloat[(r7+2)];
        f12 = heapFloat[(r7+3)];
        r17 = sp + -80;
        f13 = uint(r18); //fuitos r18, f13
        f6 = f6/f7;
        f4 = f4+f8;
        f13 = f13/f10;
        r18 = r17 >> 2;
        f6 = f6+f11;
        heapFloat[(fp+-20)] = f4;
        f13 = f13+f12;
        heapFloat[(r18+1)] = f6;
        heapFloat[(r18+2)] = f13;
        heap32[(r18+3)] = 0;
        r19 = heapU16[(r8+10)>>1];
        r20 = heapU16[(r8+8)>>1];
        r21 = heapU16[(r8+6)>>1];
        heap32[(r18+7)] = 0;
        f14 = heapFloat[(r5)];
        f4 = f4-f14;
        heapFloat[(fp+-20)] = f4;
        f4 = heapFloat[(r5+1)];
        f4 = f6-f4;
        heapFloat[(r18+1)] = f4;
        f4 = heapFloat[(r5+2)];
        f6 = uint(r21); //fuitos r21, f6
        f4 = f13-f4;
        f5 = f6/f5;
        heapFloat[(r18+2)] = f4;
        f4 = f5+f8;
        f5 = heapFloat[(r4)];
        f6 = uint(r20); //fuitos r20, f6
        f4 = f4-f5;
        f5 = f6/f7;
        heapFloat[(r18+4)] = f4;
        f4 = f5+f11;
        f5 = heapFloat[(r4+1)];
        f6 = uint(r19); //fuitos r19, f6
        f4 = f4-f5;
        f5 = f6/f10;
        heapFloat[(r18+5)] = f4;
        r19 = r3 << 4;
        r20 = r11 << 4;
        f4 = f5+f12;
        f5 = heapFloat[(r4+2)];
        r19 = (r17 + r19)|0;
        r20 = (r17 + r20)|0;
        f4 = f4-f5;
        r19 = r19 >> 2;
        heapFloat[(r18+6)] = f4;
        r18 = r20 >> 2;
        f4 = heapFloat[(r19)];
        f5 = heapFloat[(r2)];
        f6 = heapFloat[(r18+1)];
        f7 = heapFloat[(r2+1)];
        f4 = f4-f5;
        f6 = f6-f7;
        f4 = f4*f0;
        f6 = f6*f1;
        if(f4 >f6) //_LBB155_64
{
__label__ = 55;
}
else{
        r18 = r0 << 4;
        r19 = r9 << 4;
        r18 = (r17 + r18)|0;
        r19 = (r17 + r19)|0;
        r18 = r18 >> 2;
        r19 = r19 >> 2;
        f8 = heapFloat[(r18)];
        f10 = heapFloat[(r19+1)];
        f5 = f8-f5;
        f7 = f10-f7;
        f5 = f5*f0;
        f7 = f7*f1;
        if(f7 >f5) //_LBB155_64
{
__label__ = 55;
}
else{
        r18 = r12 << 4;
        r18 = (r17 + r18)|0;
        r18 = r18 >> 2;
        f8 = heapFloat[(r18+2)];
        f10 = heapFloat[(r2+2)];
        f8 = f8-f10;
        f4 = f4 < f7 ? f7 : f4;
        f7 = f8*f2;
        if(f4 >f7) //_LBB155_64
{
__label__ = 55;
}
else{
        r18 = r10 << 4;
        r17 = (r17 + r18)|0;
        r17 = r17 >> 2;
        f8 = heapFloat[(r17+2)];
        f8 = f8-f10;
        f5 = f6 < f5 ? f6 : f5;
        f6 = f8*f2;
        if(f6 >f5) //_LBB155_64
{
__label__ = 55;
}
else{
        f4 = f4 < f6 ? f6 : f4;
        f5 = f7 < f5 ? f7 : f5;
        r17 = f4 >= f3;
        r18 = f5 <= f9;
        r17 = r17 | r18;
        r18 = r17 & 1;
        r17 = r18 ^ 1;
        if(r16 != 0) //_LBB155_71
{
__label__ = 65;
}
else{
        if(r18 ==0) //_LBB155_72
{
        r18 = heap32[(r15+3)];
        if(r18 >-1) //_LBB155_74
{
        r19 = r1 >> 2;
        r19 = heap32[(r19)];
        r19 = r19 >> 2;
        r19 = heap32[(r19+2)];
        r20 = r18 >> 21;
        r18 = r18 & 2097151;
        heap32[(g0)] = r1;
        heap32[(g0+1)] = r20;
        heap32[(g0+2)] = r18;
        __FUNCTION_TABLE__[(r19)>>2](i7);
__label__ = 65;
}
else{
__label__ = 63;
break _66;
}
}
else{
__label__ = 65;
}
}
}
}
}
}
}
else{
__label__ = 55;
}
}
}
}
}
}
if (__label__ == 55){
        r17 = 0;
}
if(!(r17 !=0)) //_LBB155_77
{
        if(r16 != 0) //_LBB155_78
{
        r15 = heap32[(r15+3)];
        if(r15 <0) //_LBB155_80
{
        r16 = r15 << 4;
        r8 = (r8 - r16)|0;
        r13 = (r13 - r15)|0;
continue _66;
}
else{
__label__ = 69;
break _66;
}
}
}
        r8 = (r8 + 16)|0;
        r13 = (r13 + 1)|0;
}
else{
__label__ = 48;
break _66;
}
}
else{
__label__ = 72;
break _66;
}
}
switch(__label__ ){//multiple entries
case 72:
        r0 = maxIterations;
        r0 = r0 >> 2;
        r1 = heap32[(r0)];
if(!(r1 >=r14)) //_LBB155_84
{
        heap32[(r0)] = r14;
}
        return;
break;
case 69:
        r0 = _2E_str941;
        r1 = _2E_str313;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 75;
        _assert(i7);
break;
case 63:
        r0 = _2E_str739;
        r1 = _2E_str313;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 80;
        _assert(i7);
break;
case 48:
        r0 = _2E_str1143;
        r1 = _2E_str537;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 606;
        _assert(i7);
break;
}
}
}
        r0 = _2E_str212;
        r1 = _2E_str313;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 420;
        _assert(i7);
}
else{
        r0 = _2E_str212;
        r1 = _2E_str537;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 543;
        _assert(i7);
}
}

function _ZNK14btQuantizedBvh27walkStacklessTreeAgainstRayEP21btNodeOverlapCallbackRK9btVector3S4_S4_S4_ii(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var r12;
        var r13;
        var r14;
        var r15;
        var r16;
        var r17;
        var r18;
        var r19;
        var f0;
        var f1;
        var f2;
        var f3;
        var f4;
        var f5;
        var f6;
        var f7;
        var f8;
        var f9;
        var f10;
        var f11;
        var f12;
        var f13;
        var f14;
        var f15;
        var f16;
        var f17;
        var f18;
        var f19;
        var f20;
        var f21;
var __label__ = 0;
        i7 = sp + -48;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = heapU8[r0+60];
        if(r1 ==0) //_LBB156_2
{
        r1 = heap32[(fp+1)];
        r2 = heap32[(fp+2)];
        r3 = heap32[(fp+3)];
        r4 = heap32[(fp+4)];
        r5 = heap32[(fp+5)];
        r3 = r3 >> 2;
        r2 = r2 >> 2;
        f0 = heapFloat[(r2)];
        f1 = heapFloat[(r3)];
        f2 = heapFloat[(r3+1)];
        f3 = heapFloat[(r2+1)];
        r4 = r4 >> 2;
        r5 = r5 >> 2;
        f4 = f1-f0;
        r0 = r0 >> 2;
        f5 = f2-f3;
        f6 = heapFloat[(r3+2)];
        f7 = heapFloat[(r2+2)];
        r6 = heap32[(r0+24)];
        f8 = f6-f7;
        f9 = f4*f4;
        f10 = f5*f5;
        f11 = heapFloat[(r4)];
        f12 = heapFloat[(r4+1)];
        f13 = heapFloat[(r4+2)];
        f14 = heapFloat[(r5)];
        f15 = heapFloat[(r5+1)];
        f16 = heapFloat[(r5+2)];
        f9 = f9+f10;
        f10 = f8*f8;
        f9 = f9+f10;
        heapFloat[(g0)] = f9;
        sqrtf(i7);
        f10 =                         1;
        f9 = f10/f_g0;
        f17 = heapFloat[(r3)];
        f18 = heapFloat[(r2)];
        f19 = heapFloat[(r3+1)];
        f20 = heapFloat[(r2+1)];
        f4 = f4*f9;
        f17 = f17-f18;
        f5 = f5*f9;
        f18 = f19-f20;
        f19 = heapFloat[(r3+2)];
        f20 = heapFloat[(r2+2)];
        f8 = f8*f9;
        f9 = f19-f20;
        f17 = f4*f17;
        f18 = f5*f18;
        f19 = f1 < f0 ? f1 : f0;
        f20 = f2 < f3 ? f2 : f3;
        f21 = f6 < f7 ? f6 : f7;
        f0 = f0 < f1 ? f1 : f0;
        f1 = f3 < f2 ? f2 : f3;
        f2 = f7 < f6 ? f6 : f7;
        f3 = f17+f18;
        f6 = f8*f9;
        f7 = f19+f11;
        f9 = f20+f12;
        f11 = f21+f13;
        f0 = f0+f14;
        f1 = f1+f15;
        f2 = f2+f16;
        f3 = f3+f6;
        f6 =                         0;
        if(f4 !=f6) //_LBB156_4
{
        f4 = f10/f4;
}
else{
        f4 =        999999984306749440;
}
        if(f5 !=f6) //_LBB156_7
{
        f5 = f10/f5;
}
else{
        f5 =        999999984306749440;
}
        if(f8 !=f6) //_LBB156_10
{
        f8 = f10/f8;
}
else{
        f8 =        999999984306749440;
}
        r3 = f4 < f6;
        r7 = f5 < f6;
        r8 = f8 < f6;
        r3 = r3 & 1;
        r7 = r7 & 1;
        r8 = r8 & 1;
        r9 = r3 ^ 1;
        r10 = r7 ^ 1;
        r11 = r8 ^ 1;
        r12 = 0;
        r13 = r12;
_15: while(true){
        r14 = heap32[(r0+14)];
        if(r14 >r12) //_LBB156_12
{
        if(r14 >r13) //_LBB156_14
{
        r14 = r6 >> 2;
        f10 = heapFloat[(r14)];
        r15 = sp + -32;
        heapFloat[(fp+-8)] = f10;
        r16 = r15 >> 2;
        f12 = heapFloat[(r14+1)];
        heapFloat[(r16+1)] = f12;
        f13 = heapFloat[(r14+2)];
        heapFloat[(r16+2)] = f13;
        heap32[(r16+3)] = heap32[(r14+3)];
        f14 = heapFloat[(r14+4)];
        heapFloat[(r16+4)] = f14;
        f15 = heapFloat[(r14+5)];
        heapFloat[(r16+5)] = f15;
        f16 = heapFloat[(r14+6)];
        heapFloat[(r16+6)] = f16;
        heap32[(r16+7)] = heap32[(r14+7)];
        f17 = heapFloat[(r5)];
        f10 = f10-f17;
        heapFloat[(fp+-8)] = f10;
        f10 = heapFloat[(r5+1)];
        f10 = f12-f10;
        heapFloat[(r16+1)] = f10;
        f10 = heapFloat[(r5+2)];
        f10 = f13-f10;
        heapFloat[(r16+2)] = f10;
        f10 = heapFloat[(r4)];
        f10 = f14-f10;
        heapFloat[(r16+4)] = f10;
        f10 = heapFloat[(r4+1)];
        f10 = f15-f10;
        heapFloat[(r16+5)] = f10;
        f10 = heapFloat[(r4+2)];
        r13 = (r13 + 1)|0;
        f10 = f16-f10;
        heapFloat[(r16+6)] = f10;
        f10 = heapFloat[(r14+4)];
        if(f7 >f10) //_LBB156_17
{
__label__ = 17;
}
else{
        f10 = heapFloat[(r14)];
        if(f0 <f10) //_LBB156_17
{
__label__ = 17;
}
else{
        r16 = 1;
__label__ = 18;
}
}
if (__label__ == 17){
        r16 = 0;
}
        f10 = heapFloat[(r14+6)];
        if(f11 >f10) //_LBB156_21
{
__label__ = 20;
}
else{
        f10 = heapFloat[(r14+2)];
        if(f2 <f10) //_LBB156_21
{
__label__ = 20;
}
else{
__label__ = 21;
}
}
if (__label__ == 20){
        r16 = 0;
}
        f10 = heapFloat[(r14+5)];
        if(f9 >f10) //_LBB156_32
{
__label__ = 30;
}
else{
        f10 = heapFloat[(r14+1)];
        if(f1 <f10) //_LBB156_32
{
__label__ = 30;
}
else{
        r16 = r16 & 255;
        if(r16 ==0) //_LBB156_32
{
__label__ = 30;
}
else{
        r16 = r3 << 4;
        r17 = r10 << 4;
        r16 = (r15 + r16)|0;
        r17 = (r15 + r17)|0;
        r16 = r16 >> 2;
        r17 = r17 >> 2;
        f10 = heapFloat[(r16)];
        f12 = heapFloat[(r2)];
        f13 = heapFloat[(r17+1)];
        f14 = heapFloat[(r2+1)];
        f10 = f10-f12;
        f13 = f13-f14;
        f10 = f10*f4;
        f13 = f13*f5;
        if(f10 >f13) //_LBB156_32
{
__label__ = 30;
}
else{
        r16 = r9 << 4;
        r17 = r7 << 4;
        r16 = (r15 + r16)|0;
        r17 = (r15 + r17)|0;
        r16 = r16 >> 2;
        r17 = r17 >> 2;
        f15 = heapFloat[(r16)];
        f16 = heapFloat[(r17+1)];
        f12 = f15-f12;
        f14 = f16-f14;
        f12 = f12*f4;
        f14 = f14*f5;
        if(f14 >f12) //_LBB156_32
{
__label__ = 30;
}
else{
        r16 = r11 << 4;
        r16 = (r15 + r16)|0;
        r16 = r16 >> 2;
        f15 = heapFloat[(r16+2)];
        f16 = heapFloat[(r2+2)];
        f15 = f15-f16;
        f10 = f10 < f14 ? f14 : f10;
        f14 = f15*f8;
        if(f10 >f14) //_LBB156_32
{
__label__ = 30;
}
else{
        r16 = r8 << 4;
        r15 = (r15 + r16)|0;
        r15 = r15 >> 2;
        f15 = heapFloat[(r15+2)];
        f15 = f15-f16;
        f12 = f13 < f12 ? f13 : f12;
        f13 = f15*f8;
        if(f13 >f12) //_LBB156_32
{
__label__ = 30;
}
else{
        f10 = f10 < f13 ? f13 : f10;
        f12 = f14 < f12 ? f14 : f12;
        r15 = f10 >= f3;
        r16 = f12 <= f6;
        r15 = r15 | r16;
        r16 = r15 & 1;
        r17 = heap32[(r14+8)];
        r18 = -1;
        r16 = r16 ^ 1;
        r19 = r17 == r18;
        if(r15 != 0) //_LBB156_31
{
__label__ = 32;
}
else{
        r15 = r17 != r18;
        r15 = r15 & 1;
        if(r15 ==0) //_LBB156_33
{
        r15 = r1 >> 2;
        r15 = heap32[(r15)];
        r15 = r15 >> 2;
        r15 = heap32[(r15+2)];
        r17 = heap32[(r14+10)];
        r18 = heap32[(r14+9)];
        heap32[(g0)] = r1;
        heap32[(g0+1)] = r18;
        heap32[(g0+2)] = r17;
        __FUNCTION_TABLE__[(r15)>>2](i7);
__label__ = 32;
}
else{
__label__ = 32;
}
}
}
}
}
}
}
}
}
if (__label__ == 30){
        r16 = heap32[(r14+8)];
        r19 = -1;
        r19 = r16 == r19;
        r16 = 0;
}
if(!(r16 !=0)) //_LBB156_36
{
        r15 = r19 & 1;
        if(r15 ==0) //_LBB156_37
{
        r14 = heap32[(r14+8)];
        r15 = r14 << 6;
        r6 = (r6 + r15)|0;
        r12 = (r14 + r12)|0;
continue _15;
}
}
        r6 = (r6 + 64)|0;
        r12 = (r12 + 1)|0;
}
else{
__label__ = 13;
break _15;
}
}
else{
__label__ = 37;
break _15;
}
}
switch(__label__ ){//multiple entries
case 37:
        r0 = maxIterations;
        r0 = r0 >> 2;
        r1 = heap32[(r0)];
if(!(r1 >=r13)) //_LBB156_41
{
        heap32[(r0)] = r13;
}
        return;
break;
case 13:
        r0 = _2E_str1921;
        r1 = _2E_str537;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 490;
        _assert(i7);
break;
}
}
else{
        r0 = _2E_str1844;
        r1 = _2E_str537;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 451;
        _assert(i7);
}
}

function _ZN30btActivatingCollisionAlgorithmD0Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV30btActivatingCollisionAlgorithm;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        heap32[(g0)] = r0;
        _ZdlPv(i7);
        return;
}

function _ZN30btActivatingCollisionAlgorithmD1Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV30btActivatingCollisionAlgorithm;
        r0 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r0)] = r1;
        return;
}

function _ZN26btBoxBoxCollisionAlgorithm21calculateTimeOfImpactEP17btCollisionObjectS1_RK16btDispatcherInfoP16btManifoldResult(sp)
{
        var i7;
        var fp = sp>>2;
        var f0;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        f0 =                         1;
        f_g0 = f0;
        return;
}

function _ZN26btBoxBoxCollisionAlgorithm22getAllContactManifoldsER20btAlignedObjectArrayIP20btPersistentManifoldE(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = r0 >> 2;
        r2 = heap32[(r1+3)];
if(!(r2 ==0)) //_LBB160_23
{
        r0 = heapU8[r0+8];
if(!(r0 ==0)) //_LBB160_23
{
        r0 = heap32[(fp+1)];
        r3 = r0 >> 2;
        r4 = heap32[(r3+2)];
        r5 = heap32[(r3+1)];
        if(r4 ==r5) //_LBB160_4
{
        r6 = 1;
        r7 = r5 << 1;
        r7 = r5 == 0 ? r6 : r7;
if(!(r4 >=r7)) //_LBB160_3
{
        if(r7 !=0) //_LBB160_7
{
        r2 = gNumAlignedAllocs;
        r2 = r2 >> 2;
        r4 = heap32[(r2)];
        r8 = r7 << 2;
        r4 = (r4 + 1)|0;
        r8 = r8 | 3;
        heap32[(r2)] = r4;
        r2 = (r8 + 16)|0;
        heap32[(g0)] = r2;
        malloc(i7);
        r2 = r_g0;
        if(r2 !=0) //_LBB160_9
{
        r4 = 0;
        r8 = (r2 + 4)|0;
        r4 = (r4 - r8)|0;
        r4 = r4 & 15;
        r4 = (r2 + r4)|0;
        r8 = (r4 + 4)|0;
        r4 = r4 >> 2;
        heap32[(r4)] = r2;
        r2 = r8;
}
}
else{
        r2 = 0;
}
        r4 = (r0 + 12)|0;
        if(r5 <1) //_LBB160_12
{
        r8 = r4 >> 2;
        r9 = heap32[(r8)];
}
else{
        r8 = 0;
_15: while(true){
        r9 = r4 >> 2;
        r9 = heap32[(r9)];
        r10 = r8 << 2;
        r11 = (r9 + r10)|0;
        r11 = r11 >> 2;
        r10 = (r2 + r10)|0;
        r11 = heap32[(r11)];
        r8 = (r8 + 1)|0;
        r10 = r10 >> 2;
        heap32[(r10)] = r11;
if(!(r5 !=r8)) //_LBB160_13
{
break _15;
}
}
        r4 = (r0 + 12)|0;
}
        if(r9 !=0) //_LBB160_17
{
        r8 = heapU8[r0+16];
        if(r8 !=0) //_LBB160_19
{
        r5 = gNumAlignedFree;
        r5 = r5 >> 2;
        r8 = heap32[(r5)];
        r8 = (r8 + 1)|0;
        r9 = r9 >> 2;
        heap32[(r5)] = r8;
        r5 = heap32[(r9+-1)];
        heap32[(g0)] = r5;
        free(i7);
        r5 = heap32[(r3+1)];
}
        r8 = r4 >> 2;
        heap32[(r8)] = 0;
}
        r4 = r4 >> 2;
        heap8[r0+16] = r6;
        heap32[(r4)] = r2;
        heap32[(r3+2)] = r7;
        r2 = heap32[(r1+3)];
}
}
        r0 = r5 << 2;
        r1 = heap32[(r3+3)];
        r0 = (r1 + r0)|0;
        r0 = r0 >> 2;
        heap32[(r0)] = r2;
        r0 = heap32[(r3+1)];
        r0 = (r0 + 1)|0;
        heap32[(r3+1)] = r0;
}
}
        return;
}

function _ZN26btBoxBoxCollisionAlgorithm16processCollisionEP17btCollisionObjectS1_RK16btDispatcherInfoP16btManifoldResult(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
var __label__ = 0;
        i7 = sp + -176;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = r0 >> 2;
        r1 = heap32[(r1+3)];
if(!(r1 ==0)) //_LBB161_8
{
        r2 = heap32[(fp+1)];
        r3 = heap32[(fp+2)];
        r4 = heap32[(fp+3)];
        r5 = heap32[(fp+4)];
        r3 = r3 >> 2;
        r2 = r2 >> 2;
        r6 = heap32[(r3+48)];
        r7 = heap32[(r2+48)];
        r8 = sp + -136;
        r9 = r5 >> 2;
        r10 = r8 >> 2;
        heap32[(r9+1)] = r1;
        heap32[(r10+33)] = 0;
        heap32[(r10+32)] = 1566444395;
        heap32[(fp+-34)] = heap32[(r2+1)];
        heap32[(r10+1)] = heap32[(r2+2)];
        heap32[(r10+2)] = heap32[(r2+3)];
        heap32[(r10+3)] = heap32[(r2+4)];
        heap32[(r10+4)] = heap32[(r2+5)];
        heap32[(r10+5)] = heap32[(r2+6)];
        heap32[(r10+6)] = heap32[(r2+7)];
        heap32[(r10+7)] = heap32[(r2+8)];
        heap32[(r10+8)] = heap32[(r2+9)];
        heap32[(r10+9)] = heap32[(r2+10)];
        heap32[(r10+10)] = heap32[(r2+11)];
        heap32[(r10+11)] = heap32[(r2+12)];
        heap32[(r10+12)] = heap32[(r2+13)];
        heap32[(r10+13)] = heap32[(r2+14)];
        heap32[(r10+14)] = heap32[(r2+15)];
        heap32[(r10+15)] = heap32[(r2+16)];
        heap32[(r10+16)] = heap32[(r3+1)];
        heap32[(r10+17)] = heap32[(r3+2)];
        heap32[(r10+18)] = heap32[(r3+3)];
        heap32[(r10+19)] = heap32[(r3+4)];
        heap32[(r10+20)] = heap32[(r3+5)];
        heap32[(r10+21)] = heap32[(r3+6)];
        heap32[(r10+22)] = heap32[(r3+7)];
        heap32[(r10+23)] = heap32[(r3+8)];
        heap32[(r10+24)] = heap32[(r3+9)];
        heap32[(r10+25)] = heap32[(r3+10)];
        heap32[(r10+26)] = heap32[(r3+11)];
        heap32[(r10+27)] = heap32[(r3+12)];
        heap32[(r10+28)] = heap32[(r3+13)];
        heap32[(r10+29)] = heap32[(r3+14)];
        r1 = _ZTV16btBoxBoxDetector;
        heap32[(r10+30)] = heap32[(r3+15)];
        r2 = sp + -152;
        r1 = (r1 + 8)|0;
        heap32[(r10+31)] = heap32[(r3+16)];
        r3 = r2 >> 2;
        heap32[(fp+-38)] = r1;
        heap32[(r3+1)] = r7;
        r1 = r4 >> 2;
        heap32[(r3+2)] = r6;
        r1 = heap32[(r1+5)];
        heap32[(g0)] = r2;
        heap32[(g0+1)] = r8;
        heap32[(g0+2)] = r5;
        heap32[(g0+3)] = r1;
        heap32[(g0+4)] = 0;
        _ZN16btBoxBoxDetector16getClosestPointsERKN36btDiscreteCollisionDetectorInterface17ClosestPointInputERNS0_6ResultEP12btIDebugDrawb(i7);
        r0 = heapU8[r0+8];
if(!(r0 ==0)) //_LBB161_8
{
        r0 = heap32[(r9+1)];
        if(r0 !=0) //_LBB161_4
{
        r1 = r0 >> 2;
        r2 = heap32[(r1+279)];
if(!(r2 ==0)) //_LBB161_8
{
        r1 = heap32[(r1+277)];
        r2 = heap32[(r9+34)];
        if(r1 ==r2) //_LBB161_7
{
        r1 = (r5 + 8)|0;
        r2 = (r5 + 72)|0;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = r2;
        _ZN20btPersistentManifold20refreshContactPointsERK11btTransformS2_(i7);
}
else{
        r1 = (r5 + 72)|0;
        r5 = (r5 + 8)|0;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = r5;
        _ZN20btPersistentManifold20refreshContactPointsERK11btTransformS2_(i7);
        return;
}
}
}
else{
        r0 = _2E_str59;
        r5 = _2E_str160;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r5;
        heap32[(g0+2)] = 101;
        _assert(i7);
}
}
}
        return;
}

function _ZN26btBoxBoxCollisionAlgorithmD0Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV26btBoxBoxCollisionAlgorithm;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        r1 = heapU8[r0+8];
if(!(r1 ==0)) //_LBB162_3
{
        r1 = heap32[(r2+3)];
if(!(r1 ==0)) //_LBB162_3
{
        r3 = heap32[(r2+1)];
        r4 = r3 >> 2;
        r4 = heap32[(r4)];
        r4 = r4 >> 2;
        r4 = heap32[(r4+4)];
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r1;
        __FUNCTION_TABLE__[(r4)>>2](i7);
}
}
        r1 = _ZTV30btActivatingCollisionAlgorithm;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        heap32[(g0)] = r0;
        _ZdlPv(i7);
        return;
}

function _ZN26btBoxBoxCollisionAlgorithmD1Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV26btBoxBoxCollisionAlgorithm;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        r0 = heapU8[r0+8];
if(!(r0 ==0)) //_LBB163_3
{
        r0 = heap32[(r2+3)];
if(!(r0 ==0)) //_LBB163_3
{
        r1 = heap32[(r2+1)];
        r3 = r1 >> 2;
        r3 = heap32[(r3)];
        r3 = r3 >> 2;
        r3 = heap32[(r3+4)];
        heap32[(g0)] = r1;
        heap32[(g0+1)] = r0;
        __FUNCTION_TABLE__[(r3)>>2](i7);
}
}
        r0 = _ZTV30btActivatingCollisionAlgorithm;
        r0 = (r0 + 8)|0;
        heap32[(r2)] = r0;
        return;
}

function _ZN16btBoxBoxDetectorD1Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV16btBoxBoxDetector;
        r0 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r0)] = r1;
        return;
}

function _ZN16btBoxBoxDetectorD0Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV16btBoxBoxDetector;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        heap32[(g0)] = r0;
        _ZdlPv(i7);
        return;
}

function _ZN16btBoxBoxDetector16getClosestPointsERKN36btDiscreteCollisionDetectorInterface17ClosestPointInputERNS0_6ResultEP12btIDebugDrawb(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var r12;
        var r13;
        var r14;
        var r15;
        var r16;
        var r17;
        var r18;
        var r19;
        var r20;
        var r21;
        var r22;
        var r23;
        var r24;
        var r25;
        var f0;
        var f1;
        var f2;
        var f3;
        var f4;
        var f5;
        var f6;
        var f7;
        var f8;
        var f9;
        var f10;
        var f11;
        var f12;
        var f13;
        var f14;
        var f15;
        var f16;
        var f17;
        var f18;
        var f19;
        var f20;
        var f21;
        var f22;
        var f23;
        var f24;
        var f25;
        var f26;
        var f27;
        var f28;
        var f29;
        var f30;
var __label__ = 0;
        i7 = sp + -808;var g0 = i7>>2; // save stack
        r0 = heap32[(fp+1)];
        r1 = r0 >> 2;
        r2 = sp + -640;
        heap32[(fp+-160)] = heap32[(r1)];
        r3 = sp + -688;
        heap32[(fp+-172)] = heap32[(r1+16)];
        r4 = r2 >> 2;
        r5 = r3 >> 2;
        heap32[(r4+1)] = heap32[(r1+1)];
        heap32[(r5+1)] = heap32[(r1+17)];
        heap32[(r4+2)] = heap32[(r1+2)];
        heap32[(r5+2)] = heap32[(r1+18)];
        heap32[(r4+4)] = heap32[(r1+4)];
        heap32[(r5+4)] = heap32[(r1+20)];
        heap32[(r4+5)] = heap32[(r1+5)];
        heap32[(r5+5)] = heap32[(r1+21)];
        heap32[(r4+6)] = heap32[(r1+6)];
        heap32[(r5+6)] = heap32[(r1+22)];
        heap32[(r4+8)] = heap32[(r1+8)];
        heap32[(r5+8)] = heap32[(r1+24)];
        heap32[(r4+9)] = heap32[(r1+9)];
        heap32[(r5+9)] = heap32[(r1+25)];
        r6 = heap32[(fp)];
        heap32[(r4+10)] = heap32[(r1+10)];
        r6 = r6 >> 2;
        heap32[(r5+10)] = heap32[(r1+26)];
        r7 = heap32[(r6+2)];
        r8 = r7 >> 2;
        r9 = heap32[(r8)];
        r9 = r9 >> 2;
        r9 = heap32[(r9+11)];
        f0 = heapFloat[(r8+7)];
        f1 = heapFloat[(r8+8)];
        f2 = heapFloat[(r8+9)];
        heap32[(g0)] = r7;
        __FUNCTION_TABLE__[(r9)>>2](i7);
        f3 = f_g0;
        r9 = heap32[(r8)];
        r9 = r9 >> 2;
        r9 = heap32[(r9+11)];
        heap32[(g0)] = r7;
        __FUNCTION_TABLE__[(r9)>>2](i7);
        f4 = f_g0;
        r8 = heap32[(r8)];
        r8 = r8 >> 2;
        r8 = heap32[(r8+11)];
        heap32[(g0)] = r7;
        __FUNCTION_TABLE__[(r8)>>2](i7);
        f5 = f_g0;
        r6 = heap32[(r6+1)];
        r7 = r6 >> 2;
        r8 = heap32[(r7)];
        r8 = r8 >> 2;
        f6 = heapFloat[(r7+8)];
        r8 = heap32[(r8+11)];
        f7 = heapFloat[(r7+9)];
        f8 = heapFloat[(r7+7)];
        heap32[(g0)] = r6;
        __FUNCTION_TABLE__[(r8)>>2](i7);
        f9 = f_g0;
        r8 = heap32[(r7)];
        r8 = r8 >> 2;
        r8 = heap32[(r8+11)];
        heap32[(g0)] = r6;
        __FUNCTION_TABLE__[(r8)>>2](i7);
        f10 = f_g0;
        r7 = heap32[(r7)];
        r7 = r7 >> 2;
        r7 = heap32[(r7+11)];
        heap32[(g0)] = r6;
        __FUNCTION_TABLE__[(r7)>>2](i7);
        f8 = f8+f_g0;
        f6 = f6+f10;
        f8 = f8+f8;
        f10 =                       0.5;
        heapFloat[(fp+-187)] = f10;
        f11 = heapFloat[(r1+30)];
        f12 = heapFloat[(r1+14)];
        f13 = heapFloat[(r1+29)];
        f14 = heapFloat[(r1+13)];
        f15 = heapFloat[(r1+28)];
        f16 = heapFloat[(r1+12)];
        f17 = heapFloat[(fp+-160)];
        f18 = heapFloat[(r4+4)];
        f19 = heapFloat[(r4+8)];
        f20 = heapFloat[(r4+1)];
        f21 = heapFloat[(r4+5)];
        f22 = heapFloat[(r4+9)];
        f23 = heapFloat[(r4+2)];
        f24 = heapFloat[(r4+6)];
        f25 = heapFloat[(r4+10)];
        r4 = sp + -236;
        f7 = f7+f9;
        f6 = f6+f6;
        f8 = f8*f10;
        heapFloat[(fp+-179)] = f8;
        f0 = f0+f5;
        f5 = f7+f7;
        f6 = f6*f10;
        heapFloat[(fp+-178)] = f6;
        r6 = r4 >> 2;
        heapFloat[(fp+-59)] = f8;
        f1 = f1+f4;
        f0 = f0+f0;
        f4 = f5*f10;
        heapFloat[(fp+-177)] = f4;
        heapFloat[(r6+1)] = f6;
        r7 = sp + -248;
        heap32[(fp+-186)] = r7;
        f2 = f2+f3;
        f1 = f1+f1;
        f0 = f0*f10;
        heapFloat[(fp+-181)] = f0;
        heapFloat[(r6+2)] = f4;
        f2 = f2+f2;
        f1 = f1*f10;
        heapFloat[(fp+-182)] = f1;
        r6 = r7 >> 2;
        heapFloat[(fp+-62)] = f0;
        f0 = f2*f10;
        heapFloat[(fp+-180)] = f0;
        heapFloat[(r6+1)] = f1;
        heapFloat[(r6+2)] = f0;
        f0 = f15-f16;
        heapFloat[(fp+-183)] = f0;
        f1 = f13-f14;
        heapFloat[(fp+-184)] = f1;
        f2 = heapFloat[(fp+-172)];
        heapFloat[(fp+-194)] = f2;
        f3 = heapFloat[(r5+4)];
        heapFloat[(fp+-195)] = f3;
        f4 = heapFloat[(r5+1)];
        heapFloat[(fp+-191)] = f4;
        f5 = heapFloat[(r5+5)];
        heapFloat[(fp+-192)] = f5;
        f6 = heapFloat[(r5+2)];
        heapFloat[(fp+-188)] = f6;
        f7 = heapFloat[(r5+6)];
        heapFloat[(fp+-189)] = f7;
        f8 = f11-f12;
        heapFloat[(fp+-185)] = f8;
        f9 = heapFloat[(r5+8)];
        heapFloat[(fp+-196)] = f9;
        f10 = heapFloat[(r5+9)];
        heapFloat[(fp+-193)] = f10;
        f11 = heapFloat[(r5+10)];
        heapFloat[(fp+-190)] = f11;
        f12 = f17*f2;
        f13 = f18*f3;
        f14 = f17*f0;
        f15 = f18*f1;
        f16 = f20*f0;
        f26 = f21*f1;
        f0 = f23*f0;
        heapFloat[(fp+-198)] = f0;
        f1 = f24*f1;
        f27 = f17*f4;
        f28 = f18*f5;
        f17 = f17*f6;
        f18 = f18*f7;
        f29 = f20*f2;
        f30 = f21*f3;
        f0 = f20*f4;
        heapFloat[(fp+-197)] = f0;
        f0 = f21*f5;
        f20 = f20*f6;
        f21 = f21*f7;
        f2 = f23*f2;
        f3 = f24*f3;
        f4 = f23*f4;
        f5 = f24*f5;
        f6 = f23*f6;
        f7 = f24*f7;
        f12 = f12+f13;
        f13 = f19*f9;
        f14 = f14+f15;
        f15 = f19*f8;
        f16 = f16+f26;
        f23 = f22*f8;
        f24 = heapFloat[(fp+-198)];
        f1 = f24+f1;
        f8 = f25*f8;
        f24 = f27+f28;
        f26 = f19*f10;
        f17 = f17+f18;
        f18 = f19*f11;
        f19 = f29+f30;
        f27 = f22*f9;
        f28 = heapFloat[(fp+-197)];
        f0 = f28+f0;
        f28 = f22*f10;
        f20 = f20+f21;
        f21 = f22*f11;
        f2 = f2+f3;
        f3 = f25*f9;
        f4 = f4+f5;
        f5 = f25*f10;
        f6 = f6+f7;
        f7 = f25*f11;
        f9 = f12+f13;
        r5 = heap32[(fp+2)];
        r6 = (r2 + 4)|0;
        r7 = (r3 + 4)|0;
        r8 = (r2 + 8)|0;
        r9 = (r3 + 8)|0;
        r10 = (r0 + 112)|0;
        r0 = (r0 + 48)|0;
        f10 = f14+f15;
        f11 = f16+f23;
        f1 = f1+f8;
        f8 = f24+f26;
        f12 = f17+f18;
        f13 = f19+f27;
        f0 = f0+f28;
        f14 = f20+f21;
        f2 = f2+f3;
        f3 = f4+f5;
        f4 = f6+f7;
        f5 =                         0;
        if(f9 <f5) //_LBB166_2
{
        f6 = -f9;
}
else{
        f6 = f9;
}
        if(f8 <f5) //_LBB166_5
{
        f7 = -f8;
}
else{
        f7 = f8;
}
        if(f12 <f5) //_LBB166_8
{
        f15 = -f12;
}
else{
        f15 = f12;
}
        if(f13 <f5) //_LBB166_11
{
        f16 = -f13;
}
else{
        f16 = f13;
}
        if(f0 <f5) //_LBB166_14
{
        f17 = -f0;
}
else{
        f17 = f0;
}
        if(f14 <f5) //_LBB166_17
{
        f18 = -f14;
}
else{
        f18 = f14;
}
        if(f2 <f5) //_LBB166_20
{
        f19 = -f2;
}
else{
        f19 = f2;
}
        if(f3 <f5) //_LBB166_23
{
        f20 = -f3;
}
else{
        f20 = f3;
}
        if(f4 <f5) //_LBB166_26
{
        f21 = -f4;
}
else{
        f21 = f4;
}
        if(f10 <f5) //_LBB166_29
{
        f22 = -f10;
}
else{
        f22 = f10;
}
        f23 = heapFloat[(fp+-181)];
        f23 = f23*f6;
        f24 = heapFloat[(fp+-179)];
        f23 = f24+f23;
        f24 = heapFloat[(fp+-182)];
        f24 = f24*f7;
        f23 = f23+f24;
        f24 = heapFloat[(fp+-180)];
        f24 = f24*f15;
        f23 = f23+f24;
        f22 = f22-f23;
_41: do {
if(!(f22 >f5)) //_LBB166_268
{
        f23 =  -3.4028234663852886e+038;
        if(f22 >f23) //_LBB166_33
{
        f23 =                         0;
        r11 = f10 < f23;
        r13 = sp + -640;
        r11 = r11 & 1;
        r12 = 1;
        f23 = f22;
}
else{
        r11 = 0;
        r12 = r11;
        r13 = r11;
}
        if(f11 <f5) //_LBB166_36
{
        f22 = -f11;
}
else{
        f22 = f11;
}
        f24 = heapFloat[(fp+-181)];
        f24 = f24*f16;
        f25 = heapFloat[(fp+-178)];
        f24 = f25+f24;
        f25 = heapFloat[(fp+-182)];
        f25 = f25*f17;
        f24 = f24+f25;
        f25 = heapFloat[(fp+-180)];
        f25 = f25*f18;
        f24 = f24+f25;
        f22 = f22-f24;
if(!(f22 >f5)) //_LBB166_268
{
        if(f22 >f23) //_LBB166_40
{
        f23 =                         0;
        r11 = f11 < f23;
        r11 = r11 & 1;
        r12 = 2;
        f23 = f22;
        r13 = r6;
}
        if(f1 <f5) //_LBB166_43
{
        f22 = -f1;
}
else{
        f22 = f1;
}
        f24 = heapFloat[(fp+-181)];
        f24 = f24*f19;
        f25 = heapFloat[(fp+-177)];
        f24 = f25+f24;
        f25 = heapFloat[(fp+-182)];
        f25 = f25*f20;
        f24 = f24+f25;
        f25 = heapFloat[(fp+-180)];
        f25 = f25*f21;
        f24 = f24+f25;
        f22 = f22-f24;
if(!(f22 >f5)) //_LBB166_268
{
        if(f22 >f23) //_LBB166_47
{
        f23 =                         0;
        r11 = f1 < f23;
        r11 = r11 & 1;
        r12 = 3;
        f23 = f22;
        r13 = r8;
}
        f24 = heapFloat[(fp+-183)];
        f22 = heapFloat[(fp+-194)];
        f22 = f22*f24;
        f25 = heapFloat[(fp+-184)];
        f24 = heapFloat[(fp+-195)];
        f24 = f24*f25;
        f22 = f22+f24;
        f25 = heapFloat[(fp+-185)];
        f24 = heapFloat[(fp+-196)];
        f24 = f24*f25;
        f22 = f22+f24;
        if(f22 <f5) //_LBB166_50
{
        f24 = -f22;
}
else{
        f24 = f22;
}
        f25 = heapFloat[(fp+-179)];
        f25 = f25*f6;
        f26 = heapFloat[(fp+-178)];
        f26 = f26*f16;
        f25 = f25+f26;
        f26 = heapFloat[(fp+-177)];
        f26 = f26*f19;
        f25 = f25+f26;
        f26 = heapFloat[(fp+-181)];
        f25 = f25+f26;
        f24 = f24-f25;
if(!(f24 >f5)) //_LBB166_268
{
        if(f24 >f23) //_LBB166_54
{
        f23 =                         0;
        r11 = f22 < f23;
        r13 = sp + -688;
        r11 = r11 & 1;
        r12 = 4;
        f23 = f24;
}
        f24 = heapFloat[(fp+-183)];
        f22 = heapFloat[(fp+-191)];
        f22 = f22*f24;
        f25 = heapFloat[(fp+-184)];
        f24 = heapFloat[(fp+-192)];
        f24 = f24*f25;
        f22 = f22+f24;
        f25 = heapFloat[(fp+-185)];
        f24 = heapFloat[(fp+-193)];
        f24 = f24*f25;
        f22 = f22+f24;
        if(f22 <f5) //_LBB166_57
{
        f24 = -f22;
}
else{
        f24 = f22;
}
        f25 = heapFloat[(fp+-179)];
        f25 = f25*f7;
        f26 = heapFloat[(fp+-178)];
        f26 = f26*f17;
        f25 = f25+f26;
        f26 = heapFloat[(fp+-177)];
        f26 = f26*f20;
        f25 = f25+f26;
        f26 = heapFloat[(fp+-182)];
        f25 = f25+f26;
        f24 = f24-f25;
if(!(f24 >f5)) //_LBB166_268
{
        if(f24 >f23) //_LBB166_61
{
        f23 =                         0;
        r11 = f22 < f23;
        r11 = r11 & 1;
        r12 = 5;
        f23 = f24;
        r13 = r7;
}
        f24 = heapFloat[(fp+-183)];
        f22 = heapFloat[(fp+-188)];
        f22 = f22*f24;
        f25 = heapFloat[(fp+-184)];
        f24 = heapFloat[(fp+-189)];
        f24 = f24*f25;
        f22 = f22+f24;
        f25 = heapFloat[(fp+-185)];
        f24 = heapFloat[(fp+-190)];
        f24 = f24*f25;
        f22 = f22+f24;
        if(f22 <f5) //_LBB166_64
{
        f24 = -f22;
}
else{
        f24 = f22;
}
        f25 = heapFloat[(fp+-179)];
        f25 = f25*f15;
        f26 = heapFloat[(fp+-178)];
        f26 = f26*f18;
        f25 = f25+f26;
        f26 = heapFloat[(fp+-177)];
        f26 = f26*f21;
        f25 = f25+f26;
        f26 = heapFloat[(fp+-180)];
        f25 = f25+f26;
        f24 = f24-f25;
if(!(f24 >f5)) //_LBB166_268
{
        if(f24 >f23) //_LBB166_68
{
        f23 =                         0;
        r11 = f22 < f23;
        r11 = r11 & 1;
        r12 = 6;
        f23 = f24;
        r13 = r9;
}
        f22 = f1*f13;
        f24 = f11*f2;
        f25 =   9.9999997473787516e-006;
        f22 = f22-f24;
        f6 = f6+f25;
        f7 = f7+f25;
        f15 = f15+f25;
        heapFloat[(fp+-183)] = f15;
        f15 = f16+f25;
        f16 = f17+f25;
        f17 = f18+f25;
        f18 = f19+f25;
        heapFloat[(fp+-184)] = f18;
        f18 = f20+f25;
        heapFloat[(fp+-185)] = f18;
        f18 = f21+f25;
        if(f22 <f5) //_LBB166_71
{
        f19 = -f22;
}
else{
        f19 = f22;
}
        f21 = heapFloat[(fp+-184)];
        f20 = heapFloat[(fp+-178)];
        f20 = f20*f21;
        f21 = heapFloat[(fp+-177)];
        f21 = f21*f15;
        f20 = f20+f21;
        f24 = heapFloat[(fp+-183)];
        f21 = heapFloat[(fp+-182)];
        f21 = f21*f24;
        f20 = f20+f21;
        f21 = heapFloat[(fp+-180)];
        f21 = f21*f7;
        f20 = f20+f21;
        f19 = f19-f20;
        f20 =   1.1920928955078125e-007;
if(!(f19 >f20)) //_LBB166_268
{
        f21 = f2*f2;
        f21 = f21+f5;
        f24 = f13*f13;
        heapFloat[(fp+-188)] = f24;
        f24 = f21+f24;
        heapFloat[(g0)] = f24;
        sqrtf(i7);
        f24 = f_g0;
        if(f24 >f20) //_LBB166_75
{
        f25 = f19/f24;
        f19 =        1.0499999523162842;
        f19 = f25*f19;
        if(f19 <=f23) //_LBB166_74
{
__label__ = 69;
}
else{
        f19 =                         0;
        f23 = -f2;
        r11 = f22 < f19;
        f19 = f19/f24;
        f22 = f23/f24;
        f24 = f13/f24;
        r11 = r11 & 1;
        r12 = 7;
        r13 = 0;
        f23 = f25;
__label__ = 72;
}
}
else{
__label__ = 69;
}
if (__label__ == 69){
        f19 = f5;
        f22 = f5;
        f24 = f5;
}
        f25 = f1*f0;
        f26 = f11*f3;
        f25 = f25-f26;
        if(f25 <f5) //_LBB166_79
{
        f5 = -f25;
}
else{
        f5 = f25;
}
        f27 = heapFloat[(fp+-185)];
        f26 = heapFloat[(fp+-178)];
        f26 = f26*f27;
        f27 = heapFloat[(fp+-177)];
        f27 = f27*f16;
        f26 = f26+f27;
        f28 = heapFloat[(fp+-183)];
        f27 = heapFloat[(fp+-181)];
        f27 = f27*f28;
        f26 = f26+f27;
        f27 = heapFloat[(fp+-180)];
        f27 = f27*f6;
        f26 = f26+f27;
        f5 = f5-f26;
if(!(f5 >f20)) //_LBB166_268
{
        f26 = f3*f3;
        f27 =                         0;
        f26 = f26+f27;
        f28 = f0*f0;
        heapFloat[(fp+-189)] = f28;
        f28 = f26+f28;
        heapFloat[(g0)] = f28;
        sqrtf(i7);
        f28 = f_g0;
        if(f28 >f20) //_LBB166_83
{
        f5 = f5/f28;
        f29 =        1.0499999523162842;
        f29 = f5*f29;
if(!(f29 <=f23)) //_LBB166_82
{
        f19 =                         0;
        f22 = -f3;
        r11 = f25 < f19;
        f19 = f19/f28;
        f22 = f22/f28;
        f24 = f0/f28;
        r11 = r11 & 1;
        r12 = 8;
        r13 = 0;
        f23 = f5;
}
}
        f5 = f1*f14;
        f25 = f11*f4;
        f5 = f5-f25;
        if(f5 <f27) //_LBB166_87
{
        f25 = -f5;
}
else{
        f25 = f5;
}
        f28 = heapFloat[(fp+-178)];
        f28 = f28*f18;
        f29 = heapFloat[(fp+-177)];
        f29 = f29*f17;
        f28 = f28+f29;
        f29 = heapFloat[(fp+-181)];
        f29 = f29*f7;
        f28 = f28+f29;
        f29 = heapFloat[(fp+-182)];
        f29 = f29*f6;
        f28 = f28+f29;
        f25 = f25-f28;
if(!(f25 >f20)) //_LBB166_268
{
        f28 = f4*f4;
        f28 = f28+f27;
        f29 = f14*f14;
        heapFloat[(fp+-190)] = f29;
        f29 = f28+f29;
        heapFloat[(g0)] = f29;
        sqrtf(i7);
        f29 = f_g0;
        if(f29 >f20) //_LBB166_91
{
        f25 = f25/f29;
        f30 =        1.0499999523162842;
        f30 = f25*f30;
if(!(f30 <=f23)) //_LBB166_90
{
        f19 =                         0;
        f22 = -f4;
        r11 = f5 < f19;
        f19 = f19/f29;
        f22 = f22/f29;
        f24 = f14/f29;
        r11 = r11 & 1;
        r12 = 9;
        r13 = 0;
        f23 = f25;
}
}
        f5 = f10*f2;
        f25 = f1*f9;
        f5 = f5-f25;
        if(f5 <f27) //_LBB166_95
{
        f25 = -f5;
}
else{
        f25 = f5;
}
        f30 = heapFloat[(fp+-184)];
        f29 = heapFloat[(fp+-179)];
        f29 = f29*f30;
        f30 = heapFloat[(fp+-177)];
        f30 = f30*f6;
        f29 = f29+f30;
        f30 = heapFloat[(fp+-182)];
        f30 = f30*f17;
        f29 = f29+f30;
        f30 = heapFloat[(fp+-180)];
        f30 = f30*f16;
        f29 = f29+f30;
        f25 = f25-f29;
if(!(f25 >f20)) //_LBB166_268
{
        f29 = f9*f9;
        f21 = f21+f29;
        heapFloat[(g0)] = f21;
        sqrtf(i7);
        f21 = f_g0;
        if(f21 >f20) //_LBB166_99
{
        f25 = f25/f21;
        f30 =        1.0499999523162842;
        f30 = f25*f30;
if(!(f30 <=f23)) //_LBB166_98
{
        f22 =                         0;
        f23 = -f9;
        r11 = f5 < f22;
        f19 = f2/f21;
        f22 = f22/f21;
        f24 = f23/f21;
        r11 = r11 & 1;
        r12 = 10;
        r13 = 0;
        f23 = f25;
}
}
        f2 = f10*f3;
        f5 = f1*f8;
        f2 = f2-f5;
        if(f2 <f27) //_LBB166_103
{
        f5 = -f2;
}
else{
        f5 = f2;
}
        f25 = heapFloat[(fp+-185)];
        f21 = heapFloat[(fp+-179)];
        f21 = f21*f25;
        f25 = heapFloat[(fp+-177)];
        f25 = f25*f7;
        f21 = f21+f25;
        f25 = heapFloat[(fp+-181)];
        f25 = f25*f17;
        f21 = f21+f25;
        f25 = heapFloat[(fp+-180)];
        f25 = f25*f15;
        f21 = f21+f25;
        f5 = f5-f21;
if(!(f5 >f20)) //_LBB166_268
{
        f21 = f8*f8;
        f25 = f26+f21;
        heapFloat[(g0)] = f25;
        sqrtf(i7);
        f25 = f_g0;
        if(f25 >f20) //_LBB166_107
{
        f5 = f5/f25;
        f26 =        1.0499999523162842;
        f26 = f5*f26;
if(!(f26 <=f23)) //_LBB166_106
{
        f22 =                         0;
        f23 = -f8;
        r11 = f2 < f22;
        f19 = f3/f25;
        f22 = f22/f25;
        f24 = f23/f25;
        r11 = r11 & 1;
        r12 = 11;
        r13 = 0;
        f23 = f5;
}
}
        f2 = f10*f4;
        f1 = f1*f12;
        f1 = f2-f1;
        if(f1 <f27) //_LBB166_111
{
        f2 = -f1;
}
else{
        f2 = f1;
}
        f3 = heapFloat[(fp+-179)];
        f3 = f3*f18;
        f25 = heapFloat[(fp+-183)];
        f5 = heapFloat[(fp+-177)];
        f5 = f5*f25;
        f3 = f3+f5;
        f5 = heapFloat[(fp+-181)];
        f5 = f5*f16;
        f3 = f3+f5;
        f5 = heapFloat[(fp+-182)];
        f5 = f5*f15;
        f3 = f3+f5;
        f2 = f2-f3;
if(!(f2 >f20)) //_LBB166_268
{
        f3 = f12*f12;
        f5 = f28+f3;
        heapFloat[(g0)] = f5;
        sqrtf(i7);
        f5 = f_g0;
        if(f5 >f20) //_LBB166_115
{
        f2 = f2/f5;
        f25 =        1.0499999523162842;
        f25 = f2*f25;
if(!(f25 <=f23)) //_LBB166_114
{
        f22 =                         0;
        f23 = -f12;
        r11 = f1 < f22;
        f19 = f4/f5;
        f22 = f22/f5;
        f24 = f23/f5;
        r11 = r11 & 1;
        r12 = 12;
        r13 = 0;
        f23 = f2;
}
}
        f1 = f11*f9;
        f2 = f10*f13;
        f1 = f1-f2;
        if(f1 <f27) //_LBB166_119
{
        f2 = -f1;
}
else{
        f2 = f1;
}
        f4 = heapFloat[(fp+-179)];
        f4 = f4*f15;
        f5 = heapFloat[(fp+-178)];
        f5 = f5*f6;
        f4 = f4+f5;
        f5 = heapFloat[(fp+-182)];
        f5 = f5*f18;
        f4 = f4+f5;
        f6 = heapFloat[(fp+-185)];
        f5 = heapFloat[(fp+-180)];
        f5 = f5*f6;
        f4 = f4+f5;
        f2 = f2-f4;
if(!(f2 >f20)) //_LBB166_268
{
        f4 = heapFloat[(fp+-188)];
        f4 = f4+f29;
        f4 = f4+f27;
        heapFloat[(g0)] = f4;
        sqrtf(i7);
        f4 = f_g0;
        if(f4 >f20) //_LBB166_123
{
        f2 = f2/f4;
        f5 =        1.0499999523162842;
        f5 = f2*f5;
if(!(f5 <=f23)) //_LBB166_122
{
        f23 =                         0;
        f19 = -f13;
        r11 = f1 < f23;
        f19 = f19/f4;
        f22 = f9/f4;
        f24 = f23/f4;
        r11 = r11 & 1;
        r12 = 13;
        r13 = 0;
        f23 = f2;
}
}
        f1 = f11*f8;
        f2 = f10*f0;
        f1 = f1-f2;
        if(f1 <f27) //_LBB166_127
{
        f2 = -f1;
}
else{
        f2 = f1;
}
        f4 = heapFloat[(fp+-179)];
        f4 = f4*f16;
        f5 = heapFloat[(fp+-178)];
        f5 = f5*f7;
        f4 = f4+f5;
        f5 = heapFloat[(fp+-181)];
        f5 = f5*f18;
        f4 = f4+f5;
        f6 = heapFloat[(fp+-184)];
        f5 = heapFloat[(fp+-180)];
        f5 = f5*f6;
        f4 = f4+f5;
        f2 = f2-f4;
if(!(f2 >f20)) //_LBB166_268
{
        f4 = heapFloat[(fp+-189)];
        f4 = f4+f21;
        f4 = f4+f27;
        heapFloat[(g0)] = f4;
        sqrtf(i7);
        f4 = f_g0;
        if(f4 >f20) //_LBB166_131
{
        f2 = f2/f4;
        f5 =        1.0499999523162842;
        f5 = f2*f5;
if(!(f5 <=f23)) //_LBB166_130
{
        f23 =                         0;
        f19 = -f0;
        r11 = f1 < f23;
        f19 = f19/f4;
        f22 = f8/f4;
        f24 = f23/f4;
        r11 = r11 & 1;
        r12 = 14;
        r13 = 0;
        f23 = f2;
}
}
        f0 = f11*f12;
        f1 = f10*f14;
        f0 = f0-f1;
        if(f0 <f27) //_LBB166_135
{
        f1 = -f0;
}
else{
        f1 = f0;
}
        f2 = heapFloat[(fp+-179)];
        f2 = f2*f17;
        f5 = heapFloat[(fp+-183)];
        f4 = heapFloat[(fp+-178)];
        f4 = f4*f5;
        f2 = f2+f4;
        f5 = heapFloat[(fp+-185)];
        f4 = heapFloat[(fp+-181)];
        f4 = f4*f5;
        f2 = f2+f4;
        f5 = heapFloat[(fp+-184)];
        f4 = heapFloat[(fp+-182)];
        f4 = f4*f5;
        f2 = f2+f4;
        f1 = f1-f2;
if(!(f1 >f20)) //_LBB166_268
{
        f2 = heapFloat[(fp+-190)];
        f2 = f2+f3;
        f2 = f2+f27;
        heapFloat[(g0)] = f2;
        sqrtf(i7);
        f2 = f_g0;
        if(f2 <=f20) //_LBB166_140
{
__label__ = 128;
}
else{
        f1 = f1/f2;
        heapFloat[(fp+-183)] = f1;
        f3 =        1.0499999523162842;
        f1 = f1*f3;
        if(f1 <=f23) //_LBB166_140
{
__label__ = 128;
}
else{
        f23 =                         0;
        f19 = -f14;
        r11 = f0 < f23;
        f19 = f19/f2;
        f22 = f12/f2;
        f24 = f23/f2;
        r11 = r11 & 1;
        r13 = sp + -704;
        r12 = 15;
__label__ = 132;
}
}
if (__label__ == 128){
        if(r12 ==0) //_LBB166_268
{
break _41;
}
else{
        if(r13 !=0) //_LBB166_143
{
        r13 = r13 >> 2;
        f0 = heapFloat[(r13)];
        r6 = sp + -704;
        heapFloat[(fp+-176)] = f0;
        f1 = heapFloat[(r13+4)];
        r6 = r6 >> 2;
        heapFloat[(r6+1)] = f1;
        f19 = heapFloat[(r13+8)];
        heapFloat[(r6+2)] = f19;
        heapFloat[(fp+-183)] = f23;
__label__ = 133;
}
else{
        r13 = sp + -704;
        heapFloat[(fp+-183)] = f23;
__label__ = 132;
}
}
}
if (__label__ == 132){
        r6 = sp + -640;
        r6 = r6 >> 2;
        f0 = heapFloat[(fp+-160)];
        f1 = heapFloat[(r6+1)];
        f2 = heapFloat[(r6+2)];
        f0 = f0*f19;
        f1 = f1*f22;
        f0 = f0+f1;
        f1 = f2*f24;
        f0 = f0+f1;
        r7 = r13 >> 2;
        heapFloat[(r7)] = f0;
        f1 = heapFloat[(r6+4)];
        f2 = heapFloat[(r6+5)];
        f3 = heapFloat[(r6+6)];
        f1 = f1*f19;
        f2 = f2*f22;
        r7 = sp + -704;
        f1 = f1+f2;
        f2 = f3*f24;
        f1 = f1+f2;
        r7 = r7 >> 2;
        heapFloat[(r7+1)] = f1;
        f2 = heapFloat[(r6+8)];
        f3 = heapFloat[(r6+9)];
        f4 = heapFloat[(r6+10)];
        f19 = f2*f19;
        f2 = f3*f22;
        f19 = f19+f2;
        f2 = f4*f24;
        f19 = f19+f2;
        heapFloat[(r7+2)] = f19;
}
        if(r11 !=0) //_LBB166_147
{
        f0 = -f0;
        r6 = sp + -704;
        f1 = -f1;
        r6 = r6 >> 2;
        heapFloat[(fp+-176)] = f0;
        f19 = -f19;
        heapFloat[(r6+1)] = f1;
        heapFloat[(r6+2)] = f19;
}
        if(r12 <7) //_LBB166_153
{
        if(r12 >3) //_LBB166_155
{
        f0 = -f0;
        f1 = -f1;
        f19 = -f19;
        r1 = r0;
        r0 = r10;
}
else{
        r3 = sp + -640;
        r2 = sp + -688;
        r4 = sp + -236;
        heap32[(fp+-186)] = r4;
        r4 = sp + -248;
        r1 = r10;
}
        r6 = r2 >> 2;
        f2 = heapFloat[(r6)];
        f3 = heapFloat[(r6+4)];
        f4 = heapFloat[(r6+8)];
        f2 = f2*f0;
        f3 = f3*f1;
        f2 = f2+f3;
        f3 = f4*f19;
        f2 = f2+f3;
        heapFloat[(fp+-66)] = f2;
        f3 = heapFloat[(r6+1)];
        f4 = heapFloat[(r6+5)];
        f5 = heapFloat[(r6+9)];
        f3 = f3*f0;
        f4 = f4*f1;
        r7 = sp + -264;
        f3 = f3+f4;
        f4 = f5*f19;
        f3 = f3+f4;
        r8 = r7 >> 2;
        heapFloat[(r8+1)] = f3;
        f4 = heapFloat[(r6+2)];
        f5 = heapFloat[(r6+6)];
        f6 = heapFloat[(r6+10)];
        f4 = f4*f0;
        f5 = f5*f1;
        f4 = f4+f5;
        f5 = f6*f19;
        f4 = f4+f5;
        heapFloat[(r8+2)] = f4;
        if(f2 <f27) //_LBB166_158
{
        f2 = -f2;
}
        if(f3 <f27) //_LBB166_161
{
        f3 = -f3;
}
        if(f4 <f27) //_LBB166_164
{
        f4 = -f4;
}
        if(f3 <=f2) //_LBB166_169
{
        if(f2 <=f4) //_LBB166_171
{
        r6 = 2;
        r8 = 0;
        r9 = 1;
}
else{
        r6 = 0;
        r8 = 1;
        r9 = 2;
}
}
else{
        if(f3 <=f4) //_LBB166_168
{
        r6 = 2;
        r8 = 0;
        r9 = 1;
}
else{
        r6 = 1;
        r8 = 0;
        r9 = 2;
}
}
        r10 = r6 << 2;
        r11 = (r4 + r10)|0;
        r13 = (r2 + r10)|0;
        r1 = r1 >> 2;
        r0 = r0 >> 2;
        r11 = r11 >> 2;
        r13 = r13 >> 2;
        f2 = heapFloat[(r11)];
        f3 = heapFloat[(r13)];
        f4 = heapFloat[(r1)];
        f5 = heapFloat[(r0)];
        r7 = (r7 + r10)|0;
        f4 = f4-f5;
        f3 = f2*f3;
        r7 = r7 >> 2;
        f5 = heapFloat[(r7)];
        if(f5 <f27) //_LBB166_174
{
        r7 = r6 | 4;
        r6 = r6 | 8;
        r7 = r7 << 2;
        r6 = r6 << 2;
        r7 = (r2 + r7)|0;
        r6 = (r2 + r6)|0;
        r7 = r7 >> 2;
        r6 = r6 >> 2;
        f5 = heapFloat[(r1+1)];
        f6 = heapFloat[(r0+1)];
        f7 = heapFloat[(r7)];
        f8 = heapFloat[(r6)];
        f9 = heapFloat[(r1+2)];
        f10 = heapFloat[(r0+2)];
        f5 = f5-f6;
        f6 = f2*f7;
        f7 = f9-f10;
        f2 = f2*f8;
        f3 = f4+f3;
        f4 = f5+f6;
        f2 = f7+f2;
}
else{
        r7 = r6 | 4;
        r6 = r6 | 8;
        r7 = r7 << 2;
        r6 = r6 << 2;
        r7 = (r2 + r7)|0;
        r6 = (r2 + r6)|0;
        r7 = r7 >> 2;
        r6 = r6 >> 2;
        f5 = heapFloat[(r1+1)];
        f6 = heapFloat[(r0+1)];
        f7 = heapFloat[(r7)];
        f8 = heapFloat[(r6)];
        f9 = heapFloat[(r1+2)];
        f10 = heapFloat[(r0+2)];
        f5 = f5-f6;
        f6 = f2*f7;
        f7 = f9-f10;
        f2 = f2*f8;
        f3 = f4-f3;
        f4 = f5-f6;
        f2 = f7-f2;
}
        r1 = -1;
        r6 = -4;
        r6 = r12 < 4 ? r1 : r6;
        r6 = (r6 + r12)|0;
        if(r6 ==1) //_LBB166_178
{
        r7 = 0;
        r10 = 2;
}
else{
        if(r6 !=0) //_LBB166_179
{
        r7 = 0;
        r10 = 1;
}
else{
        r7 = 1;
        r10 = 2;
}
}
        r11 = r8 | 4;
        r13 = r7 | 4;
        r14 = r8 | 8;
        r15 = r9 | 4;
        r16 = r7 | 8;
        r17 = r10 | 4;
        r7 = r7 << 2;
        r13 = r13 << 2;
        r11 = r11 << 2;
        r8 = r8 << 2;
        r18 = r9 | 8;
        r19 = (r3 + r7)|0;
        r13 = (r3 + r13)|0;
        r16 = r16 << 2;
        r20 = r10 << 2;
        r17 = r17 << 2;
        r10 = r10 | 8;
        r21 = (r2 + r8)|0;
        r11 = (r2 + r11)|0;
        r14 = r14 << 2;
        r15 = r15 << 2;
        r9 = r9 << 2;
        r19 = r19 >> 2;
        r13 = r13 >> 2;
        r16 = (r3 + r16)|0;
        r22 = (r3 + r20)|0;
        r17 = (r3 + r17)|0;
        r10 = r10 << 2;
        r21 = r21 >> 2;
        heap32[(fp+-178)] = r21;
        r11 = r11 >> 2;
        heap32[(fp+-177)] = r11;
        r14 = (r2 + r14)|0;
        r23 = (r2 + r9)|0;
        r15 = (r2 + r15)|0;
        r18 = r18 << 2;
        r16 = r16 >> 2;
        r22 = r22 >> 2;
        r17 = r17 >> 2;
        r3 = (r3 + r10)|0;
        r10 = r14 >> 2;
        heap32[(fp+-179)] = r10;
        r14 = r23 >> 2;
        heap32[(fp+-180)] = r14;
        r15 = r15 >> 2;
        r2 = (r2 + r18)|0;
        f5 = heapFloat[(r19)];
        f6 = heapFloat[(r21)];
        f7 = heapFloat[(r13)];
        f8 = heapFloat[(r11)];
        f9 = heapFloat[(r22)];
        f10 = heapFloat[(r14)];
        f11 = heapFloat[(r17)];
        f12 = heapFloat[(r15)];
        r3 = r3 >> 2;
        r2 = r2 >> 2;
        r8 = (r4 + r8)|0;
        f13 = f5*f6;
        f14 = f7*f8;
        f15 = heapFloat[(r16)];
        f16 = heapFloat[(r10)];
        f17 = heapFloat[(r3)];
        f18 = heapFloat[(r2)];
        r3 = r8 >> 2;
        r4 = (r4 + r9)|0;
        f6 = f9*f6;
        f8 = f11*f8;
        f21 = f3*f5;
        f22 = f4*f7;
        f13 = f13+f14;
        f14 = f15*f16;
        f5 = f5*f10;
        f7 = f7*f12;
        f13 = f13+f14;
        f14 = heapFloat[(r3)];
        r3 = r4 >> 2;
        f23 = f3*f9;
        f24 = f4*f11;
        f6 = f6+f8;
        f8 = f17*f16;
        f9 = f9*f10;
        f10 = f11*f12;
        f11 = f21+f22;
        f12 = f2*f15;
        f5 = f5+f7;
        f7 = f15*f18;
        f6 = f6+f8;
        f8 = f11+f12;
        f11 = f14*f13;
        f5 = f5+f7;
        f7 = heapFloat[(r3)];
        f12 = f23+f24;
        f15 = f2*f17;
        f9 = f9+f10;
        f10 = f17*f18;
        f9 = f9+f10;
        f10 = f12+f15;
        f12 = f14*f6;
        f14 = f8-f11;
        f15 = f7*f5;
        r3 = sp + -296;
        f16 = f10-f12;
        f7 = f7*f9;
        f17 = f14-f15;
        r4 = r3 >> 2;
        f18 = f16-f7;
        heapFloat[(fp+-74)] = f17;
        f14 = f14+f15;
        heapFloat[(r4+1)] = f18;
        f11 = f8+f11;
        f16 = f16+f7;
        heapFloat[(r4+2)] = f14;
        f12 = f10+f12;
        f14 = f11+f15;
        heapFloat[(r4+3)] = f16;
        f16 = f12+f7;
        heapFloat[(r4+4)] = f14;
        f11 = f11-f15;
        heapFloat[(r4+5)] = f16;
        r8 = heap32[(fp+-186)];
        r7 = (r8 + r7)|0;
        f7 = f12-f7;
        heapFloat[(r4+6)] = f11;
        r9 = sp + -304;
        r8 = (r8 + r20)|0;
        r7 = r7 >> 2;
        heapFloat[(r4+7)] = f7;
        r4 = sp + -368;
        r10 = 0;
        r11 = 4;
        r13 = r9 >> 2;
        r8 = r8 >> 2;
        heap32[(fp+-76)] = heap32[(r7)];
        heap32[(r13+1)] = heap32[(r8)];
        r7 = r10;
_217: while(true){
        if(r7 <2) //_LBB166_199
{
        r18 = 0;
        r17 = (r18 - r7)|0;
        r14 = r1;
        r8 = r4;
_220: while(true){
        r4 = r8;
        if(r14 >1) //_LBB166_196
{
break _220;
}
else{
        f7 = r14; //fitos r14, f7
        r16 = (r3 + 8)|0;
        r8 = r18;
        r10 = r18;
        r13 = r4;
_223: while(true){
        if(r11 >0) //_LBB166_181
{
        r19 = (r3 + r8)|0;
        r20 = (r16 + r8)|0;
        r21 = r7 << 2;
        r22 = (r9 + r21)|0;
        r23 = (r19 + r21)|0;
        r22 = r22 >> 2;
        r23 = r23 >> 2;
        f11 = heapFloat[(r22)];
        f12 = heapFloat[(r23)];
        f14 = f7*f12;
        if(f14 <f11) //_LBB166_183
{
        r24 = r13 >> 2;
        r25 = r19 >> 2;
        r10 = (r10 + 1)|0;
        heap32[(r24)] = heap32[(r25)];
        heap32[(r24+1)] = heap32[(r25+1)];
        r24 = r10 & 8;
        if(r24 ==0) //_LBB166_185
{
        r13 = (r13 + 8)|0;
        f12 = heapFloat[(r23)];
        f11 = heapFloat[(r22)];
}
else{
__label__ = 179;
break _217;
}
}
        r20 = r11 > 1 ? r20 : r3;
        r23 = (r20 + r21)|0;
        r23 = r23 >> 2;
        f14 = heapFloat[(r23)];
        f15 = f7*f12;
        f16 = f7*f14;
        r23 = f15 < f11;
        r24 = f16 < f11;
        r23 = r23 ^ r24;
        if(r23 != 0) //_LBB166_188
{
        r23 = r17 << 2;
        r19 = (r19 + r23)|0;
        r20 = (r20 + r23)|0;
        r19 = r19 >> 2;
        r20 = r20 >> 2;
        f15 = heapFloat[(r19+1)];
        f16 = heapFloat[(r20+1)];
        f11 = f7*f11;
        f16 = f16-f15;
        f14 = f14-f12;
        f14 = f16/f14;
        f11 = f11-f12;
        r19 = (r13 + r23)|0;
        f11 = f14*f11;
        r19 = r19 >> 2;
        f11 = f15+f11;
        heapFloat[(r19+1)] = f11;
        r19 = (r13 + r21)|0;
        f11 = heapFloat[(r22)];
        r10 = (r10 + 1)|0;
        r19 = r19 >> 2;
        f11 = f7*f11;
        heapFloat[(r19)] = f11;
        r19 = r10 & 8;
        if(r19 ==0) //_LBB166_190
{
        r13 = (r13 + 8)|0;
}
else{
__label__ = 179;
break _217;
}
}
        r11 = (r11 + -1)|0;
        r8 = (r8 + 8)|0;
}
else{
break _223;
}
}
        r3 = sp + -368;
        r8 = sp + -128;
        r8 = r4 == r3 ? r8 : r3;
        r14 = (r14 + 2)|0;
        r3 = r4;
        r11 = r10;
}
}
        r7 = (r7 + 1)|0;
}
else{
__label__ = 177;
break _217;
}
}
if (__label__ == 177){
        r4 = r3;
}
        r1 = sp + -368;
if(!(r4 ==r1)) //_LBB166_202
{
        r3 = r10 << 3;
        heap32[(g0)] = r1;
        heap32[(g0+1)] = r4;
        heap32[(g0+2)] = r3;
        memcpy(i7);
}
        if(r10 <1) //_LBB166_268
{
break _41;
}
else{
        f7 = f13*f9;
        f11 = f5*f6;
        f12 =                         1;
        f7 = f7-f11;
        r3 = r6 << 2;
        f7 = f12/f7;
        r4 = heap32[(fp+-186)];
        r3 = (r4 + r3)|0;
        f6 = f6*f7;
        r3 = r3 >> 2;
        f11 = f13*f7;
        f5 = f5*f7;
        f7 = f9*f7;
        f6 = -f6;
        f9 = heapFloat[(r3)];
        r3 = heap32[(fp+-178)];
        f13 = heapFloat[(r3)];
        r3 = heap32[(fp+-180)];
        f14 = heapFloat[(r3)];
        r3 = heap32[(fp+-177)];
        f15 = heapFloat[(r3)];
        f16 = heapFloat[(r15)];
        r3 = heap32[(fp+-179)];
        f17 = heapFloat[(r3)];
        f18 = heapFloat[(r2)];
        r2 = (r1 + 4)|0;
        r3 = 0;
_243: while(true){
        r4 = r2 >> 2;
        f21 = heapFloat[(r4+-1)];
        f22 = heapFloat[(r4)];
        f23 = f21-f8;
        f24 = f22-f10;
        f25 = f23*f7;
        f26 = f24*f5;
        f25 = f25-f26;
        f23 = f23*f6;
        f24 = f24*f11;
        r4 = (r3 * 3)|0;
        f26 = f15*f25;
        f23 = f23+f24;
        f24 = f13*f25;
        r6 = sp + -464;
        r4 = r4 << 2;
        f25 = f17*f25;
        f26 = f4+f26;
        f28 = f16*f23;
        f24 = f3+f24;
        f29 = f14*f23;
        r4 = (r6 + r4)|0;
        f24 = f24+f29;
        f26 = f26+f28;
        f25 = f2+f25;
        f23 = f18*f23;
        f28 = f0*f24;
        f29 = f1*f26;
        f23 = f25+f23;
        r4 = r4 >> 2;
        heapFloat[(r4)] = f24;
        f24 = f28+f29;
        f25 = f19*f23;
        r7 = sp + -496;
        r8 = r3 << 2;
        r8 = (r7 + r8)|0;
        heapFloat[(r4+1)] = f26;
        f24 = f24+f25;
        f24 = f9-f24;
        r8 = r8 >> 2;
        heapFloat[(r4+2)] = f23;
        heapFloat[(r8)] = f24;
        if(f24 >=f27) //_LBB166_206
{
        r4 = r3 << 3;
        r8 = r4 | 4;
        r4 = (r1 + r4)|0;
        r8 = (r1 + r8)|0;
        r4 = r4 >> 2;
        r3 = (r3 + 1)|0;
        r8 = r8 >> 2;
        heapFloat[(r4)] = f21;
        heapFloat[(r8)] = f22;
}
        r10 = (r10 + -1)|0;
        r2 = (r2 + 8)|0;
if(!(r10 !=0)) //_LBB166_204
{
break _243;
}
}
        if(r3 <1) //_LBB166_268
{
break _41;
}
else{
        r2 = 4;
        r2 = r3 < 4 ? r3 : r2;
        r4 = 1;
        r8 = r2 < 1 ? r4 : r2;
        if(r3 >r8) //_LBB166_215
{
_252: do {
        if(r3 >1) //_LBB166_217
{
        f0 = heapFloat[(fp+-124)];
        r9 = 1;
        r7 = 0;
_254: while(true){
        r10 = sp + -496;
        r11 = r9 << 2;
        r10 = (r10 + r11)|0;
        r10 = r10 >> 2;
        f1 = heapFloat[(r10)];
        r10 = (r9 + 1)|0;
        r7 = f1 > f0 ? r9 : r7;
        f0 = f1 > f0 ? f1 : f0;
        r9 = r10;
if(!(r3 !=r10)) //_LBB166_218
{
break _252;
}
}
}
else{
        r7 = 0;
}
} while(0);
_258: do {
        if(r3 ==1) //_LBB166_224
{
        r9 = r1 >> 2;
        f0 = heapFloat[(fp+-92)];
        f1 = heapFloat[(r9+1)];
__label__ = 214;
break _258;
}
else{
        if(r3 ==2) //_LBB166_225
{
        r9 = r1 >> 2;
        f0 = heapFloat[(fp+-92)];
        f1 = heapFloat[(r9+2)];
        f2 = heapFloat[(r9+1)];
        f12 = heapFloat[(r9+3)];
        f0 = f0+f1;
        f1 = f2+f12;
        f2 = heapFloat[(fp+-187)];
        f0 = f0*f2;
        f1 = f1*f2;
__label__ = 214;
break _258;
}
else{
        r9 = (r3 + -1)|0;
_264: do {
        if(r9 >0) //_LBB166_223
{
        r10 = (r1 + 8)|0;
        f0 =                         0;
        f1 = f0;
        f2 = f0;
_266: while(true){
        r11 = r10 >> 2;
        f3 = heapFloat[(r11+-2)];
        f4 = heapFloat[(r11+1)];
        f5 = heapFloat[(r11)];
        f6 = heapFloat[(r11+-1)];
        f7 = f3*f4;
        f8 = f5*f6;
        f3 = f3+f5;
        f5 = f7-f8;
        f4 = f6+f4;
        f3 = f3*f5;
        f4 = f4*f5;
        r9 = (r9 + -1)|0;
        f2 = f2+f5;
        f1 = f3+f1;
        f0 = f4+f0;
        r10 = (r10 + 8)|0;
if(!(r9 !=0)) //_LBB166_226
{
break _264;
}
}
}
else{
        f0 = f27;
        f1 = f27;
        f2 = f27;
}
} while(0);
        r9 = r3 << 3;
        r9 = (r9 + r1)|0;
        r9 = r9 >> 2;
        r10 = r1 >> 2;
        f3 = heapFloat[(r9+-2)];
        f4 = heapFloat[(r10+1)];
        f5 = heapFloat[(fp+-92)];
        f6 = heapFloat[(r9+-1)];
        f7 = f3*f4;
        f8 = f5*f6;
        f7 = f7-f8;
        f2 = f2+f7;
        if(f2 <f27) //_LBB166_229
{
        f8 = -f2;
}
else{
        f8 = f2;
}
        if(f8 >f20) //_LBB166_232
{
        f8 =                         3;
        f2 = f2*f8;
        f2 = f12/f2;
}
else{
        f2 =        999999984306749440;
}
        if(r3 >0) //_LBB166_235
{
        f3 = f3+f5;
        f4 = f6+f4;
        f3 = f3*f7;
        f4 = f4*f7;
        f1 = f3+f1;
        f3 = f4+f0;
        f0 = f1*f2;
        f1 = f3*f2;
__label__ = 214;
}
else{
        r1 = 0;
__label__ = 219;
}
}
}
} while(0);
if (__label__ == 214){
        r1 = (r1 + 4)|0;
        r9 = sp + -32;
        r10 = r3;
_283: while(true){
        r11 = r1 >> 2;
        f2 = heapFloat[(r11+-1)];
        f3 = heapFloat[(r11)];
        f3 = f3-f1;
        f2 = f2-f0;
        heapFloat[(g0)] = f3;
        heapFloat[(g0+1)] = f2;
        r10 = (r10 + -1)|0;
        r1 = (r1 + 8)|0;
        r11 = (r9 + 4)|0;
        r9 = r9 >> 2;
        atan2f(i7);
        heapFloat[(r9)] = f_g0;
        r9 = r11;
if(!(r10 !=0)) //_LBB166_237
{
break _283;
}
}
        r1 = sp + -64;
        r9 = r3;
_286: while(true){
        r9 = (r9 + -1)|0;
        r10 = (r1 + 4)|0;
        r1 = r1 >> 2;
        heap32[(r1)] = 1;
        r1 = r10;
if(!(r9 !=0)) //_LBB166_239
{
break _286;
}
}
        r1 = 1;
}
        r9 = sp + -64;
        r10 = r7 << 2;
        r11 = (r9 + r10)|0;
        r11 = r11 >> 2;
        f0 =        6.2831854820251465;
        f1 = r8; //fitos r8, f1
        f1 = f0/f1;
        heap32[(r11)] = 0;
        heap32[(fp+-144)] = r7;
        f6 =                         0;
_290: while(true){
        if(r4 <r8) //_LBB166_242
{
        r11 = sp + -32;
        r13 = (r11 + r10)|0;
        r13 = r13 >> 2;
        f2 = r4; //fitos r4, f2
        f2 = f2*f1;
        f3 = heapFloat[(r13)];
        f2 = f2+f3;
        f3 =        3.1415927410125732;
        if(f2 >f3) //_LBB166_244
{
        f4 =       -6.2831854820251465;
        f2 = f2+f4;
}
        r13 = sp + -576;
        r14 = r4 << 2;
        r13 = (r13 + r14)|0;
        r13 = r13 >> 2;
        r14 = r1 & 1;
        heap32[(r13)] = r7;
        if(r14 ==0) //_LBB166_259
{
__label__ = 233;
break _290;
}
else{
        f4 =                1000000000;
        r14 = 0;
        r15 = r7;
_297: while(true){
        r16 = r14 << 2;
        r17 = (r9 + r16)|0;
        r17 = r17 >> 2;
        r17 = heap32[(r17)];
        if(r17 !=0) //_LBB166_249
{
        r16 = (r11 + r16)|0;
        r16 = r16 >> 2;
        f5 = heapFloat[(r16)];
        f5 = f5-f2;
        if(f5 <f6) //_LBB166_251
{
        f5 = -f5;
}
        if(f5 >f3) //_LBB166_254
{
        f5 = f0-f5;
}
if(!(f5 >=f4)) //_LBB166_248
{
        heap32[(r13)] = r14;
        r15 = r14;
        f4 = f5;
}
}
        r14 = (r14 + 1)|0;
if(!(r3 !=r14)) //_LBB166_247
{
break _297;
}
}
        if(r15 !=r7) //_LBB166_260
{
        r11 = r15 << 2;
        r11 = (r9 + r11)|0;
        r4 = (r4 + 1)|0;
        r11 = r11 >> 2;
        heap32[(r11)] = 0;
}
else{
__label__ = 233;
break _290;
}
}
}
else{
__label__ = 236;
break _290;
}
}
switch(__label__ ){//multiple entries
case 233:
        r0 = _2E_str65;
        r1 = _2E_str166;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 248;
        _assert(i7);
break;
case 236:
        if(r8 >0) //_LBB166_267
{
        r1 = 0;
_315: while(true){
        r3 = sp + -576;
        r4 = r1 << 2;
        r3 = (r3 + r4)|0;
        r3 = r3 >> 2;
        r3 = heap32[(r3)];
        r4 = (r3 * 3)|0;
        r4 = r4 << 2;
        r4 = (r6 + r4)|0;
        r4 = r4 >> 2;
        f0 = heapFloat[(r4)];
        f1 = heapFloat[(r0)];
        f0 = f0+f1;
        heapFloat[(fp+-148)] = f0;
        r7 = sp + -592;
        f1 = heapFloat[(r4+1)];
        f2 = heapFloat[(r0+1)];
        f1 = f1+f2;
        r8 = r7 >> 2;
        heapFloat[(r8+1)] = f1;
        f2 = heapFloat[(r4+2)];
        f3 = heapFloat[(r0+2)];
        f2 = f2+f3;
        r4 = sp + -496;
        r3 = r3 << 2;
        r3 = (r4 + r3)|0;
        r4 = r5 >> 2;
        heapFloat[(r8+2)] = f2;
        r4 = heap32[(r4)];
        r3 = r3 >> 2;
        r8 = sp + -704;
        f3 = heapFloat[(r3)];
        r3 = r4 >> 2;
        r4 = r8 >> 2;
        r3 = heap32[(r3+4)];
        f4 = -f3;
        f5 = heapFloat[(r4+2)];
        if(r12 >3) //_LBB166_265
{
        f6 = heapFloat[(fp+-176)];
        f7 = heapFloat[(r4+1)];
        f8 = f6*f3;
        r4 = sp + -160;
        f9 = f7*f3;
        f0 = f0-f8;
        r7 = r4 >> 2;
        f3 = f5*f3;
        f1 = f1-f9;
        heapFloat[(fp+-40)] = f0;
        f0 = f2-f3;
        heapFloat[(r7+1)] = f1;
        heapFloat[(r7+2)] = f0;
        r8 = sp + -144;
        f0 = -f6;
        heap32[(r7+3)] = 0;
        r7 = r8 >> 2;
        f1 = -f7;
        heapFloat[(fp+-36)] = f0;
        f0 = -f5;
        heapFloat[(r7+1)] = f1;
        heapFloat[(r7+2)] = f0;
        heap32[(r7+3)] = 0;
        heap32[(g0)] = r5;
        heap32[(g0+1)] = r8;
        heap32[(g0+2)] = r4;
        heapFloat[(g0+3)] = f4;
        __FUNCTION_TABLE__[(r3)>>2](i7);
}
else{
        f0 = heapFloat[(r4+1)];
        f1 = heapFloat[(fp+-176)];
        r4 = sp + -176;
        f1 = -f1;
        r8 = r4 >> 2;
        f0 = -f0;
        heapFloat[(fp+-44)] = f1;
        f1 = -f5;
        heapFloat[(r8+1)] = f0;
        heapFloat[(r8+2)] = f1;
        heap32[(r8+3)] = 0;
        heap32[(g0)] = r5;
        heap32[(g0+1)] = r4;
        heap32[(g0+2)] = r7;
        heapFloat[(g0+3)] = f4;
        __FUNCTION_TABLE__[(r3)>>2](i7);
}
        r1 = (r1 + 1)|0;
        if(r1 >=r2) //_LBB166_268
{
break _41;
}
else{
continue _315;
}
}
}
else{
break _41;
}
break;
}
}
else{
        if(r12 <4) //_LBB166_212
{
        r1 = (r6 + 8)|0;
        r2 = sp + -496;
_324: while(true){
        r4 = r1 >> 2;
        f12 = heapFloat[(r4+-2)];
        f20 = heapFloat[(r0)];
        f12 = f12+f20;
        heapFloat[(fp+-132)] = f12;
        r6 = sp + -528;
        f12 = heapFloat[(r4+-1)];
        f20 = heapFloat[(r0+1)];
        r7 = r6 >> 2;
        f12 = f12+f20;
        heapFloat[(r7+1)] = f12;
        f12 = heapFloat[(r4)];
        f20 = heapFloat[(r0+2)];
        f12 = f12+f20;
        heapFloat[(r7+2)] = f12;
        r4 = r5 >> 2;
        r4 = heap32[(r4)];
        r7 = sp + -704;
        r7 = r7 >> 2;
        r4 = r4 >> 2;
        r8 = r2 >> 2;
        f12 = heapFloat[(r8)];
        f20 = heapFloat[(r7+2)];
        f27 = heapFloat[(r7+1)];
        f0 = heapFloat[(fp+-176)];
        r4 = heap32[(r4+4)];
        r7 = sp + -208;
        f0 = -f0;
        r8 = r7 >> 2;
        f27 = -f27;
        heapFloat[(fp+-52)] = f0;
        f20 = -f20;
        heapFloat[(r8+1)] = f27;
        heapFloat[(r8+2)] = f20;
        heap32[(r8+3)] = 0;
        f12 = -f12;
        heap32[(g0)] = r5;
        heap32[(g0+1)] = r7;
        heap32[(g0+2)] = r6;
        heapFloat[(g0+3)] = f12;
        r3 = (r3 + -1)|0;
        r2 = (r2 + 4)|0;
        r1 = (r1 + 12)|0;
        __FUNCTION_TABLE__[(r4)>>2](i7);
        if(r3 ==0) //_LBB166_268
{
break _41;
}
else{
continue _324;
}
}
}
else{
        r1 = (r6 + 8)|0;
_327: while(true){
        r2 = r7 >> 2;
        r4 = r1 >> 2;
        f12 = heapFloat[(r2)];
        f20 = heapFloat[(fp+-176)];
        f27 = heapFloat[(r4+-2)];
        f0 = heapFloat[(r0)];
        f27 = f27+f0;
        f0 = f20*f12;
        r2 = sp + -704;
        f27 = f27-f0;
        heapFloat[(fp+-136)] = f27;
        r2 = r2 >> 2;
        f27 = heapFloat[(r2+1)];
        f0 = heapFloat[(r4+-1)];
        f1 = heapFloat[(r0+1)];
        r6 = sp + -544;
        f0 = f0+f1;
        f1 = f27*f12;
        r8 = r6 >> 2;
        f0 = f0-f1;
        heapFloat[(r8+1)] = f0;
        f0 = heapFloat[(r2+2)];
        f1 = heapFloat[(r4)];
        f2 = heapFloat[(r0+2)];
        f1 = f1+f2;
        f2 = f0*f12;
        f1 = f1-f2;
        r2 = r5 >> 2;
        heapFloat[(r8+2)] = f1;
        r2 = heap32[(r2)];
        r2 = r2 >> 2;
        r2 = heap32[(r2+4)];
        r4 = sp + -192;
        f20 = -f20;
        r8 = r4 >> 2;
        f27 = -f27;
        heapFloat[(fp+-48)] = f20;
        f20 = -f0;
        heapFloat[(r8+1)] = f27;
        heapFloat[(r8+2)] = f20;
        heap32[(r8+3)] = 0;
        f12 = -f12;
        heap32[(g0)] = r5;
        heap32[(g0+1)] = r4;
        heap32[(g0+2)] = r6;
        heapFloat[(g0+3)] = f12;
        r3 = (r3 + -1)|0;
        r7 = (r7 + 4)|0;
        r1 = (r1 + 12)|0;
        __FUNCTION_TABLE__[(r2)>>2](i7);
        if(r3 ==0) //_LBB166_268
{
break _41;
}
else{
continue _327;
}
}
}
}
}
}
}
else{
        r0 = sp + -688;
        r2 = r0 >> 2;
        f20 = heapFloat[(fp+-172)];
        f27 = heapFloat[(r2+4)];
        f2 = heapFloat[(r2+5)];
        f3 = heapFloat[(r2+1)];
        f4 = f0*f20;
        f5 = f1*f27;
        f6 = heapFloat[(r2+8)];
        f7 = heapFloat[(r2+9)];
        f8 = heapFloat[(r2+6)];
        f9 = heapFloat[(r2+2)];
        f4 = f4+f5;
        f5 = f19*f6;
        f10 = f0*f3;
        f11 = f1*f2;
        f12 = heapFloat[(r2+10)];
        f4 = f4+f5;
        f5 = f10+f11;
        f10 = f19*f7;
        f11 = f0*f9;
        f13 = f1*f8;
        f14 =                         0;
        f15 =                        -1;
        f16 =                         1;
        f5 = f5+f10;
        f4 = f4 > f14 ? f15 : f16;
        f10 = f11+f13;
        f11 = f19*f12;
        f13 = heapFloat[(fp+-181)];
        f4 = f13*f4;
        f10 = f10+f11;
        f5 = f5 > f14 ? f15 : f16;
        f11 = heapFloat[(fp+-182)];
        f5 = f11*f5;
        r2 = sp + -640;
        f20 = f4*f20;
        f11 = heapFloat[(r1+28)];
        f10 = f10 > f14 ? f15 : f16;
        r3 = r2 >> 2;
        f13 = heapFloat[(fp+-180)];
        f10 = f13*f10;
        f27 = f4*f27;
        f13 = heapFloat[(r1+29)];
        f20 = f11+f20;
        f3 = f5*f3;
        r4 = (r12 + -7)|0;
        f11 = heapFloat[(r3+4)];
        f17 = heapFloat[(fp+-160)];
        f18 = heapFloat[(r3+8)];
        f21 = heapFloat[(r3+5)];
        f22 = heapFloat[(r3+1)];
        f23 = heapFloat[(r3+9)];
        f24 = heapFloat[(r3+6)];
        f25 = heapFloat[(r3+2)];
        f26 = heapFloat[(r3+10)];
        f4 = f4*f6;
        f6 = heapFloat[(r1+30)];
        f27 = f13+f27;
        f2 = f5*f2;
        f20 = f20+f3;
        f3 = f10*f9;
        f9 = heapFloat[(r1+12)];
        heapFloat[(fp+-182)] = f9;
        f9 = heapFloat[(r1+13)];
        heapFloat[(fp+-181)] = f9;
        f9 = heapFloat[(r1+14)];
        heapFloat[(fp+-180)] = f9;
        r3 = (r4 / 3)|0;
        r4 = (r4 % 3)|0;
        r10 = sp + -512;
        f20 = f20+f3;
        f3 = f6+f4;
        f4 = f5*f7;
        f27 = f27+f2;
        f2 = f10*f8;
        r3 = r3 << 2;
        r4 = r4 << 2;
        f27 = f27+f2;
        r12 = r10 >> 2;
        heapFloat[(fp+-128)] = f20;
        f2 = f3+f4;
        f3 = f10*f12;
        r2 = (r2 + r3)|0;
        r0 = (r0 + r4)|0;
        f2 = f2+f3;
        heapFloat[(r12+1)] = f27;
        r2 = r2 >> 2;
        heapFloat[(r12+2)] = f2;
        r0 = r0 >> 2;
        f3 = heapFloat[(r2)];
        f4 = heapFloat[(r0)];
        f5 = heapFloat[(r2+4)];
        f6 = heapFloat[(r0+4)];
        f7 = heapFloat[(r2+8)];
        f8 = heapFloat[(r0+8)];
        f9 = f3*f4;
        f10 = f5*f6;
        f9 = f9+f10;
        f10 = f7*f8;
        f9 = f9+f10;
        f10 = f9*f9;
        f10 = f16-f10;
        f12 =   9.9999997473787516e-005;
        if(f10 >f12) //_LBB166_151
{
        f12 = f0*f17;
        f13 = f1*f11;
        f12 = f12+f13;
        f13 = f19*f18;
        f28 = f0*f22;
        f29 = f1*f21;
        f12 = f12+f13;
        f13 = f28+f29;
        f28 = f19*f23;
        f29 = f0*f25;
        f30 = f1*f24;
        f13 = f13+f28;
        f12 = f12 > f14 ? f16 : f15;
        f28 = f29+f30;
        f29 = f19*f26;
        f30 = heapFloat[(fp+-179)];
        f12 = f30*f12;
        f28 = f28+f29;
        f13 = f13 > f14 ? f16 : f15;
        f29 = heapFloat[(fp+-178)];
        f13 = f29*f13;
        f17 = f12*f17;
        f11 = f12*f11;
        f12 = f12*f18;
        f14 = f28 > f14 ? f16 : f15;
        f15 = heapFloat[(fp+-177)];
        f14 = f15*f14;
        f15 = heapFloat[(fp+-182)];
        f15 = f15+f17;
        f17 = f13*f22;
        f18 = heapFloat[(fp+-181)];
        f11 = f18+f11;
        f18 = f13*f21;
        f21 = heapFloat[(fp+-180)];
        f12 = f21+f12;
        f13 = f13*f23;
        f15 = f15+f17;
        f17 = f14*f25;
        f11 = f11+f18;
        f18 = f14*f24;
        f12 = f12+f13;
        f14 = f14*f26;
        f13 = f15+f17;
        f11 = f11+f18;
        f14 = f12+f14;
        f12 = f20-f13;
        f11 = f27-f11;
        f14 = f2-f14;
        f3 = f3*f12;
        f5 = f5*f11;
        f3 = f3+f5;
        f5 = f7*f14;
        f7 = f4*f12;
        f11 = f6*f11;
        f3 = f3+f5;
        f5 = f7+f11;
        f14 = f8*f14;
        f3 = f9*f3;
        f14 = f5+f14;
        f14 = f3-f14;
        f3 = f16/f10;
        f14 = f14*f3;
}
        f3 = f4*f14;
        f20 = f20+f3;
        f3 = f6*f14;
        f4 = f8*f14;
        f27 = f27+f3;
        heapFloat[(fp+-128)] = f20;
        f20 = f2+f4;
        heapFloat[(r12+1)] = f27;
        r0 = r5 >> 2;
        heapFloat[(r12+2)] = f20;
        r0 = heap32[(r0)];
        r0 = r0 >> 2;
        r0 = heap32[(r0+4)];
        r2 = sp + -224;
        f0 = -f0;
        r3 = r2 >> 2;
        f1 = -f1;
        heapFloat[(fp+-56)] = f0;
        f0 = -f19;
        heapFloat[(r3+1)] = f1;
        heapFloat[(r3+2)] = f0;
        heap32[(r3+3)] = 0;
        heap32[(g0)] = r5;
        heap32[(g0+1)] = r2;
        heap32[(g0+2)] = r10;
        f0 = heapFloat[(fp+-183)];
        heapFloat[(g0+3)] = f0;
        __FUNCTION_TABLE__[(r0)>>2](i7);
        return;
}
}
}
}
}
}
}
}
}
}
}
}
}
}
}
}
} while(0);
        return;
}

function _ZN21btCollisionDispatcher13findAlgorithmEP17btCollisionObjectS1_P20btPersistentManifold(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
var __label__ = 0;
        i7 = sp + -24;var g0 = i7>>2; // save stack
        r0 = sp + -8;
        r1 = heap32[(fp)];
        r2 = heap32[(fp+1)];
        r3 = r0 >> 2;
        r4 = heap32[(fp+3)];
        heap32[(fp+-2)] = r1;
        r5 = heap32[(fp+2)];
        r6 = r2 >> 2;
        heap32[(r3+1)] = r4;
        r3 = heap32[(r6+48)];
        r4 = r5 >> 2;
        r3 = r3 >> 2;
        r4 = heap32[(r4+48)];
        r3 = heap32[(r3+1)];
        r4 = r4 >> 2;
        r4 = heap32[(r4+1)];
        r3 = (r3 * 144)|0;
        r1 = (r1 + r3)|0;
        r3 = r4 << 2;
        r1 = (r1 + r3)|0;
        r1 = r1 >> 2;
        r1 = heap32[(r1+50)];
        r3 = r1 >> 2;
        r3 = heap32[(r3)];
        r3 = r3 >> 2;
        r3 = heap32[(r3+2)];
        heap32[(g0)] = r1;
        heap32[(g0+1)] = r0;
        heap32[(g0+2)] = r2;
        heap32[(g0+3)] = r5;
        __FUNCTION_TABLE__[(r3)>>2](i7);
        return;
}

function _ZN21btCollisionDispatcher13needsResponseEP17btCollisionObjectS1_(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp+1)];
        r0 = r0 >> 2;
        r0 = heap32[(r0+51)];
        r1 = r0 & 4;
if(!(r1 !=0)) //_LBB168_4
{
        r1 = heap32[(fp+2)];
        r1 = r1 >> 2;
        r1 = heap32[(r1+51)];
        r2 = r1 & 4;
if(!(r2 !=0)) //_LBB168_4
{
        r0 = r0 & 3;
        if(r0 ==0) //_LBB168_5
{
        r0 = 1;
        r_g0 = r0;
        return;
}
else{
        r0 = r1 & 3;
        r1 = 0;
        r0 = r0 == r1;
        r0 = r0 & 1;
        r_g0 = r0;
        return;
}
}
}
        r0 = 0;
        r_g0 = r0;
        return;
}

function _ZN23btCollisionPairCallbackD1Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV23btCollisionPairCallback;
        r0 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r0)] = r1;
        return;
}

function _ZN23btCollisionPairCallbackD0Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV23btCollisionPairCallback;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        heap32[(g0)] = r0;
        _ZdlPv(i7);
        return;
}

function _ZN23btCollisionPairCallback14processOverlapER16btBroadphasePair(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
var __label__ = 0;
        i7 = sp + -16;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = r0 >> 2;
        r1 = heap32[(r0+2)];
        r2 = r1 >> 2;
        r2 = heap32[(r2+47)];
        r0 = heap32[(r0+1)];
        r3 = heap32[(fp+1)];
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = r0;
        __FUNCTION_TABLE__[(r2)>>2](i7);
        r0 = 0;
        r_g0 = r0;
        return;
}

function _ZN21btCollisionDispatcher25dispatchAllCollisionPairsEP22btOverlappingPairCacheRK16btDispatcherInfoP12btDispatcher(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
var __label__ = 0;
        i7 = sp + -32;var g0 = i7>>2; // save stack
        r0 = _ZTV23btCollisionPairCallback;
        r1 = sp + -16;
        r0 = (r0 + 8)|0;
        r2 = r1 >> 2;
        r3 = heap32[(fp+2)];
        heap32[(fp+-4)] = r0;
        r0 = heap32[(fp+1)];
        r4 = heap32[(fp)];
        heap32[(r2+1)] = r3;
        r3 = r0 >> 2;
        heap32[(r2+2)] = r4;
        r2 = heap32[(r3)];
        r2 = r2 >> 2;
        r2 = heap32[(r2+12)];
        r3 = heap32[(fp+3)];
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = r3;
        __FUNCTION_TABLE__[(r2)>>2](i7);
        return;
}

function _ZNK21btCollisionDispatcher15getNumManifoldsEv(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = r0 >> 2;
        r0 = heap32[(r0+3)];
        r_g0 = r0;
        return;
}

function _ZN21btCollisionDispatcher26getInternalManifoldPointerEv(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = r0 >> 2;
        r0 = heap32[(r0+5)];
        r_g0 = r0;
        return;
}

function _ZN21btCollisionDispatcher26getManifoldByIndexInternalEi(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = r0 >> 2;
        r1 = heap32[(fp+1)];
        r0 = heap32[(r0+5)];
        r1 = r1 << 2;
        r0 = (r0 + r1)|0;
        r0 = r0 >> 2;
        r0 = heap32[(r0)];
        r_g0 = r0;
        return;
}

function _ZN21btCollisionDispatcher22freeCollisionAlgorithmEPv(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp+1)];
if(!(r0 ==0)) //_LBB176_5
{
        r1 = heap32[(fp)];
        r1 = r1 >> 2;
        r1 = heap32[(r1+48)];
        r1 = r1 >> 2;
        r2 = heap32[(r1+4)];
if(!(uint(r2) >uint(r0))) //_LBB176_4
{
        r3 = heap32[(r1)];
        r4 = heap32[(r1+1)];
        r3 = (r3 * r4)|0;
        r2 = (r2 + r3)|0;
if(!(uint(r2) <=uint(r0))) //_LBB176_4
{
        r2 = r0 >> 2;
        r3 = heap32[(r1+3)];
        heap32[(r2)] = r3;
        heap32[(r1+3)] = r0;
        r0 = heap32[(r1+2)];
        r0 = (r0 + 1)|0;
        heap32[(r1+2)] = r0;
        return;
}
}
        r1 = gNumAlignedFree;
        r1 = r1 >> 2;
        r2 = heap32[(r1)];
        r2 = (r2 + 1)|0;
        r0 = r0 >> 2;
        heap32[(r1)] = r2;
        r0 = heap32[(r0+-1)];
        heap32[(g0)] = r0;
        free(i7);
}
        return;
}

function _ZN21btCollisionDispatcher15releaseManifoldEP20btPersistentManifold(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
var __label__ = 0;
        i7 = sp + -16;var g0 = i7>>2; // save stack
        r0 = gNumManifold;
        r0 = r0 >> 2;
        r1 = heap32[(r0)];
        r2 = heap32[(fp)];
        r1 = (r1 + -1)|0;
        r3 = r2 >> 2;
        heap32[(r0)] = r1;
        r0 = heap32[(r3)];
        r0 = r0 >> 2;
        r0 = heap32[(r0+5)];
        r1 = heap32[(fp+1)];
        heap32[(g0)] = r2;
        heap32[(g0+1)] = r1;
        __FUNCTION_TABLE__[(r0)>>2](i7);
        r0 = r1 >> 2;
        r2 = heap32[(r3+3)];
        r4 = heap32[(r0+284)];
        if(r2 >r4) //_LBB177_2
{
        r2 = (r2 + -1)|0;
        r5 = r4 << 2;
        r6 = heap32[(r3+5)];
        r2 = r2 << 2;
        r7 = (r6 + r5)|0;
        r6 = (r6 + r2)|0;
        r7 = r7 >> 2;
        r6 = r6 >> 2;
        r8 = heap32[(r7)];
        r6 = heap32[(r6)];
        heap32[(r7)] = r6;
        r6 = heap32[(r3+5)];
        r2 = (r6 + r2)|0;
        r2 = r2 >> 2;
        heap32[(r2)] = r8;
        r2 = heap32[(r3+5)];
        r2 = (r2 + r5)|0;
        r2 = r2 >> 2;
        r2 = heap32[(r2)];
        r2 = r2 >> 2;
        heap32[(r2+284)] = r4;
        r2 = heap32[(r3+3)];
        r2 = (r2 + -1)|0;
        heap32[(r3+3)] = r2;
if(!(r1 ==0)) //_LBB177_7
{
        r2 = heap32[(r3+49)];
        r2 = r2 >> 2;
        r3 = heap32[(r2+4)];
if(!(uint(r3) >uint(r1))) //_LBB177_6
{
        r4 = heap32[(r2)];
        r5 = heap32[(r2+1)];
        r4 = (r4 * r5)|0;
        r3 = (r3 + r4)|0;
if(!(uint(r3) <=uint(r1))) //_LBB177_6
{
        r3 = heap32[(r2+3)];
        heap32[(r0)] = r3;
        heap32[(r2+3)] = r1;
        r0 = heap32[(r2+2)];
        r0 = (r0 + 1)|0;
        heap32[(r2+2)] = r0;
        return;
}
}
        r1 = gNumAlignedFree;
        r1 = r1 >> 2;
        r2 = heap32[(r1)];
        r2 = (r2 + 1)|0;
        heap32[(r1)] = r2;
        r0 = heap32[(r0+-1)];
        heap32[(g0)] = r0;
        free(i7);
}
        return;
}
else{
        r0 = _2E_str472;
        r1 = _2E_str573;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 120;
        _assert(i7);
}
}

function _ZN21btCollisionDispatcherD1Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV21btCollisionDispatcher;
        r2 = _ZTV16btManifoldResult;
        r3 = r0 >> 2;
        r1 = (r1 + 8)|0;
        r2 = (r2 + 8)|0;
        heap32[(r3)] = r1;
        heap32[(r3+7)] = r2;
        r1 = heap32[(r3+5)];
if(!(r1 ==0)) //_LBB178_4
{
        r2 = heapU8[r0+24];
if(!(r2 ==0)) //_LBB178_3
{
        r2 = gNumAlignedFree;
        r2 = r2 >> 2;
        r4 = heap32[(r2)];
        r4 = (r4 + 1)|0;
        r1 = r1 >> 2;
        heap32[(r2)] = r4;
        r1 = heap32[(r1+-1)];
        heap32[(g0)] = r1;
        free(i7);
}
        heap32[(r3+5)] = 0;
}
        r1 = 1;
        heap8[r0+24] = r1;
        heap32[(r3+5)] = 0;
        r0 = _ZTV12btDispatcher;
        heap32[(r3+3)] = 0;
        r0 = (r0 + 8)|0;
        heap32[(r3+4)] = 0;
        heap32[(r3)] = r0;
        return;
}

function _ZN21btCollisionDispatcherD0Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV21btCollisionDispatcher;
        r2 = _ZTV16btManifoldResult;
        r3 = r0 >> 2;
        r1 = (r1 + 8)|0;
        r2 = (r2 + 8)|0;
        heap32[(r3)] = r1;
        heap32[(r3+7)] = r2;
        r1 = heap32[(r3+5)];
if(!(r1 ==0)) //_LBB179_4
{
        r2 = heapU8[r0+24];
if(!(r2 ==0)) //_LBB179_3
{
        r2 = gNumAlignedFree;
        r2 = r2 >> 2;
        r4 = heap32[(r2)];
        r4 = (r4 + 1)|0;
        r1 = r1 >> 2;
        heap32[(r2)] = r4;
        r1 = heap32[(r1+-1)];
        heap32[(g0)] = r1;
        free(i7);
}
        heap32[(r3+5)] = 0;
}
        r1 = 1;
        heap8[r0+24] = r1;
        heap32[(r3+5)] = 0;
        r1 = _ZTV12btDispatcher;
        heap32[(r3+3)] = 0;
        r1 = (r1 + 8)|0;
        heap32[(r3+4)] = 0;
        heap32[(r3)] = r1;
        heap32[(g0)] = r0;
        _ZdlPv(i7);
        return;
}

function _ZN21btCollisionDispatcher14getNewManifoldEPvS0_(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var f0;
        var f1;
        var f2;
var __label__ = 0;
        i7 = sp + -24;var g0 = i7>>2; // save stack
        r0 = gNumManifold;
        r0 = r0 >> 2;
        r1 = heap32[(r0)];
        r1 = (r1 + 1)|0;
        r2 = heap32[(fp)];
        r3 = heap32[(fp+1)];
        r4 = heap32[(fp+2)];
        heap32[(r0)] = r1;
        r0 = heapU8[r2+4];
        r0 = r0 & 2;
        if(r0 !=0) //_LBB180_2
{
        r0 = r4 >> 2;
        r0 = heap32[(r0+48)];
        r1 = r0 >> 2;
        r1 = heap32[(r1)];
        r1 = r1 >> 2;
        r1 = heap32[(r1+5)];
        heap32[(g0)] = r0;
        heap32[(g0+1)] = 1017370378;
        __FUNCTION_TABLE__[(r1)>>2](i7);
        f0 = f_g0;
        r0 = r3 >> 2;
        heapFloat[(fp+-2)] = f0;
        r0 = heap32[(r0+48)];
        r1 = r0 >> 2;
        r1 = heap32[(r1)];
        r1 = r1 >> 2;
        r1 = heap32[(r1+5)];
        heap32[(g0)] = r0;
        heap32[(g0+1)] = 1017370378;
        __FUNCTION_TABLE__[(r1)>>2](i7);
        r0 = sp + -4;
        r1 = sp + -8;
        r0 = f_g0 < f0 ? r0 : r1;
        heapFloat[(fp+-1)] = f_g0;
}
else{
        r0 = gContactBreakingThreshold;
}
        r1 = r2 >> 2;
        r5 = heap32[(r1+49)];
        r6 = r3 >> 2;
        r7 = r4 >> 2;
        r5 = r5 >> 2;
        r0 = r0 >> 2;
        f0 = heapFloat[(r6+46)];
        f1 = heapFloat[(r7+46)];
        r6 = heap32[(r5+2)];
        f2 = heapFloat[(r0)];
        f0 = f0 < f1 ? f0 : f1;
_5: do {
        if(r6 ==0) //_LBB180_9
{
        r0 = gNumAlignedAllocs;
        r0 = r0 >> 2;
        r5 = heap32[(r0)];
        r5 = (r5 + 1)|0;
        heap32[(r0)] = r5;
        heap32[(g0)] = 1159;
        malloc(i7);
        r0 = r_g0;
        if(r0 !=0) //_LBB180_11
{
        r5 = 0;
        r6 = (r0 + 4)|0;
        r5 = (r5 - r6)|0;
        r5 = r5 & 15;
        r5 = (r0 + r5)|0;
        r6 = (r5 + 4)|0;
        r5 = r5 >> 2;
        heap32[(r5)] = r0;
        r0 = r6;
}
else{
break _5;
}
}
else{
        r0 = heap32[(r5)];
        if(r0 >1139) //_LBB180_6
{
        if(r6 >0) //_LBB180_8
{
        r0 = heap32[(r5+3)];
        r7 = r0 >> 2;
        r7 = heap32[(r7)];
        r6 = (r6 + -1)|0;
        heap32[(r5+3)] = r7;
        heap32[(r5+2)] = r6;
}
else{
        r1 = _2E_str371;
        r2 = _2E_str169;
        heap32[(g0)] = r1;
        heap32[(g0+1)] = r2;
        heap32[(g0+2)] = 70;
        _assert(i7);
}
}
else{
        r1 = _2E_str270;
        r2 = _2E_str169;
        heap32[(g0)] = r1;
        heap32[(g0+1)] = r2;
        heap32[(g0+2)] = 69;
        _assert(i7);
}
}
} while(0);
        r5 = r0 >> 2;
        heap32[(r5)] = 1;
        heap32[(r5+28)] = 0;
        r6 = 0;
        heap32[(r5+29)] = 0;
        heap8[r0+120] = r6;
        heap32[(r5+31)] = 0;
        heap32[(r5+32)] = 0;
        heap32[(r5+33)] = 0;
        heap32[(r5+34)] = 0;
        heap32[(r5+35)] = 0;
        heap32[(r5+36)] = 0;
        heap32[(r5+37)] = 0;
        heap32[(r5+97)] = 0;
        heap32[(r5+98)] = 0;
        heap8[r0+396] = r6;
        heap32[(r5+100)] = 0;
        heap32[(r5+101)] = 0;
        heap32[(r5+102)] = 0;
        heap32[(r5+103)] = 0;
        heap32[(r5+104)] = 0;
        heap32[(r5+105)] = 0;
        heap32[(r5+106)] = 0;
        heap32[(r5+166)] = 0;
        heap32[(r5+167)] = 0;
        heap8[r0+672] = r6;
        heap32[(r5+169)] = 0;
        heap32[(r5+170)] = 0;
        heap32[(r5+171)] = 0;
        heap32[(r5+172)] = 0;
        heap32[(r5+173)] = 0;
        heap32[(r5+174)] = 0;
        heap32[(r5+175)] = 0;
        heap32[(r5+235)] = 0;
        heap32[(r5+236)] = 0;
        heap8[r0+948] = r6;
        heap32[(r5+238)] = 0;
        heap32[(r5+239)] = 0;
        heap32[(r5+240)] = 0;
        heap32[(r5+241)] = 0;
        heap32[(r5+242)] = 0;
        heap32[(r5+243)] = 0;
        heap32[(r5+244)] = 0;
        heap32[(r5+277)] = r3;
        heap32[(r5+278)] = r4;
        heap32[(r5+279)] = 0;
        heapFloat[(r5+280)] = f2;
        heapFloat[(r5+281)] = f0;
        r3 = heap32[(r1+3)];
        heap32[(r5+284)] = r3;
        r3 = heap32[(r1+4)];
        r4 = heap32[(r1+3)];
        if(r3 ==r4) //_LBB180_14
{
        r5 = 1;
        r7 = r4 << 1;
        r7 = r4 == 0 ? r5 : r7;
if(!(r3 >=r7)) //_LBB180_13
{
        if(r7 !=0) //_LBB180_17
{
        r3 = gNumAlignedAllocs;
        r3 = r3 >> 2;
        r8 = heap32[(r3)];
        r9 = r7 << 2;
        r8 = (r8 + 1)|0;
        r9 = r9 | 3;
        heap32[(r3)] = r8;
        r3 = (r9 + 16)|0;
        heap32[(g0)] = r3;
        malloc(i7);
        r3 = r_g0;
        if(r3 !=0) //_LBB180_19
{
        r8 = (r3 + 4)|0;
        r8 = (r6 - r8)|0;
        r8 = r8 & 15;
        r8 = (r3 + r8)|0;
        r9 = (r8 + 4)|0;
        r8 = r8 >> 2;
        heap32[(r8)] = r3;
        r3 = r9;
}
}
else{
        r3 = 0;
}
        r8 = (r2 + 20)|0;
        if(r4 <1) //_LBB180_22
{
        r6 = r8 >> 2;
        r9 = heap32[(r6)];
}
else{
_26: while(true){
        r9 = r8 >> 2;
        r9 = heap32[(r9)];
        r10 = r6 << 2;
        r11 = (r9 + r10)|0;
        r11 = r11 >> 2;
        r10 = (r3 + r10)|0;
        r11 = heap32[(r11)];
        r6 = (r6 + 1)|0;
        r10 = r10 >> 2;
        heap32[(r10)] = r11;
if(!(r4 !=r6)) //_LBB180_23
{
break _26;
}
}
        r8 = (r2 + 20)|0;
}
        if(r9 !=0) //_LBB180_27
{
        r6 = heapU8[r2+24];
        if(r6 !=0) //_LBB180_29
{
        r4 = gNumAlignedFree;
        r4 = r4 >> 2;
        r6 = heap32[(r4)];
        r6 = (r6 + 1)|0;
        r9 = r9 >> 2;
        heap32[(r4)] = r6;
        r4 = heap32[(r9+-1)];
        heap32[(g0)] = r4;
        free(i7);
        r4 = heap32[(r1+3)];
}
        r6 = r8 >> 2;
        heap32[(r6)] = 0;
}
        r6 = r8 >> 2;
        heap8[r2+24] = r5;
        heap32[(r6)] = r3;
        heap32[(r1+4)] = r7;
}
}
        r2 = r4 << 2;
        r3 = heap32[(r1+5)];
        r2 = (r3 + r2)|0;
        r2 = r2 >> 2;
        heap32[(r2)] = r0;
        r2 = heap32[(r1+3)];
        r2 = (r2 + 1)|0;
        heap32[(r1+3)] = r2;
        r_g0 = r0;
        return;
}

function _ZN21btCollisionDispatcher13clearManifoldEP20btPersistentManifold(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp+1)];
        r0 = r0 >> 2;
        r1 = heap32[(r0+279)];
if(!(r1 <1)) //_LBB181_3
{
        r2 = 0;
_3: while(true){
        r2 = (r2 + 1)|0;
        if(r1 >r2) //_LBB181_2
{
continue _3;
}
else{
break _3;
}
}
}
        heap32[(r0+279)] = 0;
        return;
}

function _ZN21btCollisionDispatcher14needsCollisionEP17btCollisionObjectS1_(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
var __label__ = 0;
        i7 = sp + -16;var g0 = i7>>2; // save stack
        r0 = heap32[(fp+1)];
        if(r0 !=0) //_LBB182_2
{
        r1 = heap32[(fp+2)];
        if(r1 !=0) //_LBB182_4
{
        r2 = heap32[(fp)];
        r2 = r2 >> 2;
        r3 = heap32[(r2+1)];
        r4 = r3 & 1;
if(!(r4 != 0)) //_LBB182_8
{
        r4 = heapU8[r0+204];
        r4 = r4 & 3;
if(!(r4 ==0)) //_LBB182_8
{
        r4 = heapU8[r1+204];
        r4 = r4 & 3;
if(!(r4 ==0)) //_LBB182_8
{
        r3 = r3 | 1;
        heap32[(r2+1)] = r3;
        r2 = _2E_str977;
        heap32[(g0)] = r2;
        printf(i7);
}
}
}
        r2 = r0 >> 2;
        r3 = heap32[(r2+54)];
        if(r3 ==2) //_LBB182_10
{
__label__ = 10;
}
else{
        if(r3 !=5) //_LBB182_12
{
__label__ = 12;
}
else{
__label__ = 10;
}
}
_12: do {
if (__label__ == 10){
        r3 = r1 >> 2;
        r3 = heap32[(r3+54)];
if(!(r3 ==2)) //_LBB182_16
{
if(!(r3 ==5)) //_LBB182_16
{
break _12;
}
}
        r0 = 0;
        r_g0 = r0;
        return;
}
} while(0);
        r3 = heap32[(r2+63)];
        if(r3 !=0) //_LBB182_14
{
        r2 = heap32[(r2)];
        r2 = r2 >> 2;
        r2 = heap32[(r2)];
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        r0 = 0;
        __FUNCTION_TABLE__[(r2)>>2](i7);
        r0 = r_g0 == r0;
}
else{
        r0 = 0;
}
        r0 = r0 & 1;
        r0 = r0 ^ 1;
        r_g0 = r0;
        return;
}
else{
        r0 = _2E_str876;
        r1 = _2E_str573;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 167;
        _assert(i7);
}
}
else{
        r0 = _2E_str775;
        r1 = _2E_str573;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 166;
        _assert(i7);
}
}

function _ZN21btCollisionDispatcher26allocateCollisionAlgorithmEi(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
var __label__ = 0;
        i7 = sp + -16;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = r0 >> 2;
        r0 = heap32[(r0+48)];
        r0 = r0 >> 2;
        r1 = heap32[(r0+2)];
        r2 = heap32[(fp+1)];
        if(r1 ==0) //_LBB183_7
{
        r0 = gNumAlignedAllocs;
        r0 = r0 >> 2;
        r1 = heap32[(r0)];
        r1 = (r1 + 1)|0;
        heap32[(r0)] = r1;
        r0 = (r2 + 19)|0;
        heap32[(g0)] = r0;
        malloc(i7);
        r0 = r_g0;
if(!(r0 ==0)) //_LBB183_9
{
        r1 = 0;
        r2 = (r0 + 4)|0;
        r1 = (r1 - r2)|0;
        r1 = r1 & 15;
        r1 = (r0 + r1)|0;
        r2 = r1 >> 2;
        heap32[(r2)] = r0;
        r0 = (r1 + 4)|0;
}
        r_g0 = r0;
        return;
}
else{
if(!(r2 ==0)) //_LBB183_4
{
        r3 = heap32[(r0)];
if(!(r3 >=r2)) //_LBB183_4
{
        r0 = _2E_str270;
        r1 = _2E_str169;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 69;
        _assert(i7);
}
}
        if(r1 >0) //_LBB183_6
{
        r2 = heap32[(r0+3)];
        r3 = r2 >> 2;
        r3 = heap32[(r3)];
        r1 = (r1 + -1)|0;
        heap32[(r0+3)] = r3;
        heap32[(r0+2)] = r1;
        r_g0 = r2;
        return;
}
else{
        r0 = _2E_str371;
        r1 = _2E_str169;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 70;
        _assert(i7);
}
}
}

function _ZN21btCollisionDispatcher19defaultNearCallbackER16btBroadphasePairRS_RK16btDispatcherInfo(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var f0;
        var f1;
var __label__ = 0;
        i7 = sp + -184;var g0 = i7>>2; // save stack
        r0 = heap32[(fp+1)];
        r1 = heap32[(fp)];
        r1 = r1 >> 2;
        r2 = r0 >> 2;
        r3 = heap32[(r2)];
        r4 = heap32[(r1)];
        r5 = heap32[(r1+1)];
        r3 = r3 >> 2;
        r4 = r4 >> 2;
        r5 = r5 >> 2;
        r4 = heap32[(r4)];
        r5 = heap32[(r5)];
        r3 = heap32[(r3+6)];
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r4;
        heap32[(g0+2)] = r5;
        __FUNCTION_TABLE__[(r3)>>2](i7);
        r3 = r_g0;
_1: do {
if(!(r3 ==0)) //_LBB184_8
{
        r3 = heap32[(fp+2)];
        r6 = heap32[(r1+2)];
        if(r6 ==0) //_LBB184_3
{
        r6 = heap32[(r2)];
        r6 = r6 >> 2;
        r6 = heap32[(r6+2)];
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r4;
        heap32[(g0+2)] = r5;
        heap32[(g0+3)] = 0;
        __FUNCTION_TABLE__[(r6)>>2](i7);
        r6 = r_g0;
        heap32[(r1+2)] = r6;
        if(r6 ==0) //_LBB184_8
{
break _1;
}
}
        r0 = _ZTV16btManifoldResult;
        r1 = sp + -160;
        r0 = (r0 + 8)|0;
        r2 = r1 >> 2;
        heap32[(fp+-40)] = r0;
        heap32[(r2+1)] = 0;
        heap32[(r2+34)] = r4;
        r0 = r4 >> 2;
        heap32[(r2+35)] = r5;
        heap32[(r2+2)] = heap32[(r0+1)];
        heap32[(r2+3)] = heap32[(r0+2)];
        heap32[(r2+4)] = heap32[(r0+3)];
        heap32[(r2+5)] = heap32[(r0+4)];
        heap32[(r2+6)] = heap32[(r0+5)];
        heap32[(r2+7)] = heap32[(r0+6)];
        heap32[(r2+8)] = heap32[(r0+7)];
        heap32[(r2+9)] = heap32[(r0+8)];
        heap32[(r2+10)] = heap32[(r0+9)];
        heap32[(r2+11)] = heap32[(r0+10)];
        heap32[(r2+12)] = heap32[(r0+11)];
        heap32[(r2+13)] = heap32[(r0+12)];
        heap32[(r2+14)] = heap32[(r0+13)];
        heap32[(r2+15)] = heap32[(r0+14)];
        heap32[(r2+16)] = heap32[(r0+15)];
        r7 = r5 >> 2;
        heap32[(r2+17)] = heap32[(r0+16)];
        heap32[(r2+18)] = heap32[(r7+1)];
        heap32[(r2+19)] = heap32[(r7+2)];
        heap32[(r2+20)] = heap32[(r7+3)];
        heap32[(r2+21)] = heap32[(r7+4)];
        heap32[(r2+22)] = heap32[(r7+5)];
        heap32[(r2+23)] = heap32[(r7+6)];
        heap32[(r2+24)] = heap32[(r7+7)];
        heap32[(r2+25)] = heap32[(r7+8)];
        heap32[(r2+26)] = heap32[(r7+9)];
        heap32[(r2+27)] = heap32[(r7+10)];
        heap32[(r2+28)] = heap32[(r7+11)];
        heap32[(r2+29)] = heap32[(r7+12)];
        heap32[(r2+30)] = heap32[(r7+13)];
        heap32[(r2+31)] = heap32[(r7+14)];
        heap32[(r2+32)] = heap32[(r7+15)];
        r0 = r6 >> 2;
        heap32[(r2+33)] = heap32[(r7+16)];
        r0 = heap32[(r0)];
        r2 = r3 >> 2;
        r7 = heap32[(r2+2)];
        if(r7 !=1) //_LBB184_6
{
        r0 = r0 >> 2;
        r0 = heap32[(r0+3)];
        heap32[(g0)] = r6;
        heap32[(g0+1)] = r4;
        heap32[(g0+2)] = r5;
        heap32[(g0+3)] = r3;
        heap32[(g0+4)] = r1;
        __FUNCTION_TABLE__[(r0)>>2](i7);
        f0 = f_g0;
        f1 = heapFloat[(r2+3)];
if(!(f1 <=f0)) //_LBB184_8
{
        heapFloat[(r2+3)] = f0;
}
}
else{
        r0 = r0 >> 2;
        r0 = heap32[(r0+2)];
        heap32[(g0)] = r6;
        heap32[(g0+1)] = r4;
        heap32[(g0+2)] = r5;
        heap32[(g0+3)] = r3;
        heap32[(g0+4)] = r1;
        __FUNCTION_TABLE__[(r0)>>2](i7);
        return;
}
}
} while(0);
        return;
}

function _ZN17btCollisionObject24checkCollideWithOverrideEPS_(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = 1;
        r_g0 = r0;
        return;
}

function _ZN17btCollisionObject17setCollisionShapeEP16btCollisionShape(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = r0 >> 2;
        r1 = heap32[(fp+1)];
        heap32[(r0+48)] = r1;
        heap32[(r0+50)] = r1;
        return;
}

function _ZNK17btCollisionObject28calculateSerializeBufferSizeEv(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = 248;
        r_g0 = r0;
        return;
}

function _ZN17btCollisionObjectD1Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV17btCollisionObject;
        r0 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r0)] = r1;
        return;
}

function _ZN17btCollisionObjectD0Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV17btCollisionObject;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
if(!(r0 ==0)) //_LBB189_2
{
        r0 = gNumAlignedFree;
        r0 = r0 >> 2;
        r1 = heap32[(r0)];
        r1 = (r1 + 1)|0;
        heap32[(r0)] = r1;
        r0 = heap32[(r2+-1)];
        heap32[(g0)] = r0;
        free(i7);
}
        return;
}

function _ZNK17btCollisionObject9serializeEPvP12btSerializer(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = heap32[(fp+1)];
        r2 = r0 >> 2;
        r1 = r1 >> 2;
        heap32[(r1+4)] = heap32[(r2+1)];
        heap32[(r1+5)] = heap32[(r2+2)];
        heap32[(r1+6)] = heap32[(r2+3)];
        heap32[(r1+7)] = heap32[(r2+4)];
        heap32[(r1+8)] = heap32[(r2+5)];
        heap32[(r1+9)] = heap32[(r2+6)];
        heap32[(r1+10)] = heap32[(r2+7)];
        heap32[(r1+11)] = heap32[(r2+8)];
        heap32[(r1+12)] = heap32[(r2+9)];
        heap32[(r1+13)] = heap32[(r2+10)];
        heap32[(r1+14)] = heap32[(r2+11)];
        heap32[(r1+15)] = heap32[(r2+12)];
        heap32[(r1+16)] = heap32[(r2+13)];
        heap32[(r1+17)] = heap32[(r2+14)];
        heap32[(r1+18)] = heap32[(r2+15)];
        heap32[(r1+19)] = heap32[(r2+16)];
        heap32[(r1+20)] = heap32[(r2+17)];
        heap32[(r1+21)] = heap32[(r2+18)];
        heap32[(r1+22)] = heap32[(r2+19)];
        heap32[(r1+23)] = heap32[(r2+20)];
        heap32[(r1+24)] = heap32[(r2+21)];
        heap32[(r1+25)] = heap32[(r2+22)];
        heap32[(r1+26)] = heap32[(r2+23)];
        heap32[(r1+27)] = heap32[(r2+24)];
        heap32[(r1+28)] = heap32[(r2+25)];
        heap32[(r1+29)] = heap32[(r2+26)];
        heap32[(r1+30)] = heap32[(r2+27)];
        heap32[(r1+31)] = heap32[(r2+28)];
        heap32[(r1+32)] = heap32[(r2+29)];
        heap32[(r1+33)] = heap32[(r2+30)];
        heap32[(r1+34)] = heap32[(r2+31)];
        heap32[(r1+35)] = heap32[(r2+32)];
        heap32[(r1+36)] = heap32[(r2+33)];
        heap32[(r1+37)] = heap32[(r2+34)];
        heap32[(r1+38)] = heap32[(r2+35)];
        heap32[(r1+39)] = heap32[(r2+36)];
        heap32[(r1+40)] = heap32[(r2+37)];
        heap32[(r1+41)] = heap32[(r2+38)];
        heap32[(r1+42)] = heap32[(r2+39)];
        heap32[(r1+43)] = heap32[(r2+40)];
        heap32[(r1+44)] = heap32[(r2+41)];
        heap32[(r1+45)] = heap32[(r2+42)];
        heap32[(r1+46)] = heap32[(r2+43)];
        heap32[(r1+47)] = heap32[(r2+44)];
        r3 = heap32[(r2+45)];
        heap32[(r1+55)] = r3;
        r3 = heap32[(fp+2)];
        heap32[(r1+48)] = heap32[(r2+46)];
        heap32[(r1)] = 0;
        r4 = r3 >> 2;
        r5 = heap32[(r4)];
        r5 = r5 >> 2;
        r5 = heap32[(r5+7)];
        r6 = heap32[(r2+48)];
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r6;
        __FUNCTION_TABLE__[(r5)>>2](i7);
        heap32[(r1+1)] = r_g0;
        heap32[(r1+2)] = 0;
        r5 = heap32[(r2+51)];
        heap32[(r1+56)] = r5;
        r5 = heap32[(r2+52)];
        heap32[(r1+57)] = r5;
        r5 = heap32[(r2+53)];
        heap32[(r1+58)] = r5;
        r5 = heap32[(r2+54)];
        heap32[(r1+59)] = r5;
        r5 = heap32[(r2+54)];
        heap32[(r1+59)] = r5;
        heap32[(r1+49)] = heap32[(r2+55)];
        heap32[(r1+50)] = heap32[(r2+56)];
        heap32[(r1+51)] = heap32[(r2+57)];
        r5 = heap32[(r2+58)];
        heap32[(r1+60)] = r5;
        r5 = heap32[(r4)];
        r5 = r5 >> 2;
        r5 = heap32[(r5+10)];
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r0;
        __FUNCTION_TABLE__[(r5)>>2](i7);
        r0 = r_g0;
        r5 = heap32[(r4)];
        r5 = r5 >> 2;
        r5 = heap32[(r5+7)];
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r0;
        __FUNCTION_TABLE__[(r5)>>2](i7);
        r5 = r_g0;
        heap32[(r1+3)] = r5;
if(!(r5 ==0)) //_LBB190_2
{
        r4 = heap32[(r4)];
        r4 = r4 >> 2;
        r4 = heap32[(r4+12)];
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r0;
        __FUNCTION_TABLE__[(r4)>>2](i7);
}
        heap32[(r1+52)] = heap32[(r2+60)];
        heap32[(r1+53)] = heap32[(r2+61)];
        heap32[(r1+54)] = heap32[(r2+62)];
        heap32[(r1+54)] = heap32[(r2+62)];
        r0 = heap32[(r2+63)];
        heap32[(r1+61)] = r0;
        r0 = _2E_str78;
        r_g0 = r0;
        return;
}

function _ZNK17btCollisionObject21serializeSingleObjectEP12btSerializer(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
var __label__ = 0;
        i7 = sp + -24;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = r0 >> 2;
        r2 = heap32[(r1)];
        r2 = r2 >> 2;
        r2 = heap32[(r2+4)];
        heap32[(g0)] = r0;
        r3 = heap32[(fp+1)];
        __FUNCTION_TABLE__[(r2)>>2](i7);
        r4 = r3 >> 2;
        r5 = heap32[(r4)];
        r5 = r5 >> 2;
        r5 = heap32[(r5+4)];
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r_g0;
        heap32[(g0+2)] = 1;
        __FUNCTION_TABLE__[(r5)>>2](i7);
        r2 = r_g0;
        r1 = heap32[(r1)];
        r1 = r1 >> 2;
        r5 = r2 >> 2;
        r1 = heap32[(r1+5)];
        r5 = heap32[(r5+2)];
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r5;
        heap32[(g0+2)] = r3;
        __FUNCTION_TABLE__[(r1)>>2](i7);
        r4 = heap32[(r4)];
        r4 = r4 >> 2;
        r4 = heap32[(r4+5)];
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r2;
        heap32[(g0+2)] = r_g0;
        heap32[(g0+3)] = 1245859651;
        heap32[(g0+4)] = r0;
        __FUNCTION_TABLE__[(r4)>>2](i7);
        return;
}

function _ZN16btManifoldResult20setShapeIdentifiersAEii(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = r0 >> 2;
        r1 = heap32[(fp+1)];
        r2 = heap32[(fp+2)];
        heap32[(r0+36)] = r1;
        heap32[(r0+38)] = r2;
        return;
}

function _ZN16btManifoldResult20setShapeIdentifiersBEii(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = r0 >> 2;
        r1 = heap32[(fp+1)];
        r2 = heap32[(fp+2)];
        heap32[(r0+37)] = r1;
        heap32[(r0+39)] = r2;
        return;
}

function _ZN16btCollisionWorld14setDebugDrawerEP12btIDebugDraw(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = r0 >> 2;
        r1 = heap32[(fp+1)];
        heap32[(r0+21)] = r1;
        return;
}

function _ZN16btCollisionWorld14getDebugDrawerEv(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r0 = r0 >> 2;
        r0 = heap32[(r0+21)];
        r_g0 = r0;
        return;
}

function _ZNK16btCollisionWorld20ConvexResultCallback14needsCollisionEP17btBroadphaseProxy(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = heap32[(fp+1)];
        r2 = heapU16[(r0+10)>>1];
        r3 = heapU16[(r1+4)>>1];
        r2 = r2 & r3;
        r2 = r2 & 65535;
        if(r2 ==0) //_LBB196_2
{
        r0 = 0;
        r_g0 = r0;
        return;
}
else{
        r1 = heapU16[(r1+6)>>1];
        r0 = heapU16[(r0+8)>>1];
        r0 = r1 & r0;
        r0 = r0 & 65535;
        r1 = 0;
        r0 = r0 != r1;
        r0 = r0 & 1;
        r_g0 = r0;
        return;
}
}

function _ZN12btConvexCast10CastResult9DebugDrawEf(sp)
{
        var i7;
        var fp = sp>>2;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        return;
}

function _ZN12btConvexCast10CastResult15drawCoordSystemERK11btTransform(sp)
{
        var i7;
        var fp = sp>>2;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        return;
}

function _ZN12btConvexCast10CastResultD1Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTVN12btConvexCast10CastResultE;
        r0 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r0)] = r1;
        return;
}

function _ZN12btConvexCast10CastResultD0Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTVN12btConvexCast10CastResultE;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        heap32[(g0)] = r0;
        _ZdlPv(i7);
        return;
}

function _ZZN16btCollisionWorld13rayTestSingleERK11btTransformS2_P17btCollisionObjectPK16btCollisionShapeS2_RNS_17RayResultCallbackEEN29BridgeTriangleRaycastCallback9reportHitERK9btVector3fii(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var f0;
        var f1;
        var f2;
        var f3;
        var f4;
        var f5;
        var f6;
        var f7;
        var f8;
        var f9;
        var f10;
        var f11;
var __label__ = 0;
        i7 = sp + -56;var g0 = i7>>2; // save stack
        r0 = sp + -8;
        r1 = heap32[(fp+3)];
        r2 = heap32[(fp+1)];
        r3 = heap32[(fp)];
        r4 = r0 >> 2;
        r5 = heap32[(fp+4)];
        heap32[(fp+-2)] = r1;
        r1 = r2 >> 2;
        heap32[(r4+1)] = r5;
        r2 = r3 >> 2;
        r3 = heap32[(r2+12)];
        f0 = heapFloat[(r2+22)];
        f1 = heapFloat[(r1)];
        f2 = heapFloat[(r2+18)];
        f3 = heapFloat[(r2+14)];
        f4 = heapFloat[(r2+23)];
        f5 = heapFloat[(r1+1)];
        f6 = heapFloat[(r2+19)];
        f7 = heapFloat[(r2+15)];
        f8 = heapFloat[(r2+24)];
        f9 = heapFloat[(r1+2)];
        f10 = heapFloat[(r2+20)];
        f11 = heapFloat[(r2+16)];
        r1 = sp + -40;
        f3 = f3*f1;
        f7 = f7*f5;
        r4 = r1 >> 2;
        heap32[(fp+-10)] = r3;
        f2 = f2*f1;
        f6 = f6*f5;
        f3 = f3+f7;
        f7 = f11*f9;
        f0 = f0*f1;
        f1 = f4*f5;
        f2 = f2+f6;
        f4 = f10*f9;
        f3 = f3+f7;
        heap32[(r4+1)] = r0;
        f0 = f0+f1;
        f1 = f8*f9;
        f2 = f2+f4;
        heapFloat[(r4+2)] = f3;
        f0 = f0+f1;
        heapFloat[(r4+3)] = f2;
        heapFloat[(r4+4)] = f0;
        heap32[(r4+5)] = 0;
        heap32[(r4+6)] = heap32[(fp+2)];
        r0 = heap32[(r2+11)];
        r2 = r0 >> 2;
        r2 = heap32[(r2)];
        r2 = r2 >> 2;
        r2 = heap32[(r2+3)];
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 1;
        __FUNCTION_TABLE__[(r2)>>2](i7);
        return;
}

function _ZZN16btCollisionWorld13rayTestSingleERK11btTransformS2_P17btCollisionObjectPK16btCollisionShapeS2_RNS_17RayResultCallbackEEN29BridgeTriangleRaycastCallback9reportHitE_0RK9btVector3fii(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var f0;
        var f1;
        var f2;
        var f3;
        var f4;
        var f5;
        var f6;
        var f7;
        var f8;
        var f9;
        var f10;
        var f11;
var __label__ = 0;
        i7 = sp + -56;var g0 = i7>>2; // save stack
        r0 = sp + -8;
        r1 = heap32[(fp+3)];
        r2 = heap32[(fp+1)];
        r3 = heap32[(fp)];
        r4 = r0 >> 2;
        r5 = heap32[(fp+4)];
        heap32[(fp+-2)] = r1;
        r1 = r2 >> 2;
        heap32[(r4+1)] = r5;
        r2 = r3 >> 2;
        r3 = heap32[(r2+12)];
        f0 = heapFloat[(r2+22)];
        f1 = heapFloat[(r1)];
        f2 = heapFloat[(r2+18)];
        f3 = heapFloat[(r2+14)];
        f4 = heapFloat[(r2+23)];
        f5 = heapFloat[(r1+1)];
        f6 = heapFloat[(r2+19)];
        f7 = heapFloat[(r2+15)];
        f8 = heapFloat[(r2+24)];
        f9 = heapFloat[(r1+2)];
        f10 = heapFloat[(r2+20)];
        f11 = heapFloat[(r2+16)];
        r1 = sp + -40;
        f3 = f3*f1;
        f7 = f7*f5;
        r4 = r1 >> 2;
        heap32[(fp+-10)] = r3;
        f2 = f2*f1;
        f6 = f6*f5;
        f3 = f3+f7;
        f7 = f11*f9;
        f0 = f0*f1;
        f1 = f4*f5;
        f2 = f2+f6;
        f4 = f10*f9;
        f3 = f3+f7;
        heap32[(r4+1)] = r0;
        f0 = f0+f1;
        f1 = f8*f9;
        f2 = f2+f4;
        heapFloat[(r4+2)] = f3;
        f0 = f0+f1;
        heapFloat[(r4+3)] = f2;
        heapFloat[(r4+4)] = f0;
        heap32[(r4+5)] = 0;
        heap32[(r4+6)] = heap32[(fp+2)];
        r0 = heap32[(r2+11)];
        r2 = r0 >> 2;
        r2 = heap32[(r2)];
        r2 = r2 >> 2;
        r2 = heap32[(r2+3)];
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 1;
        __FUNCTION_TABLE__[(r2)>>2](i7);
        return;
}

function _ZZN16btCollisionWorld13rayTestSingleERK11btTransformS2_P17btCollisionObjectPK16btCollisionShapeS2_RNS_17RayResultCallbackEEN15LocalInfoAdder2D1Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTVZN16btCollisionWorld13rayTestSingleERK11btTransformS2_P17btCollisionObjectPK16btCollisionShapeS2_RNS_17RayResultCallbackEE15LocalInfoAdder2;
        r0 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r0)] = r1;
        return;
}

function _ZZN16btCollisionWorld13rayTestSingleERK11btTransformS2_P17btCollisionObjectPK16btCollisionShapeS2_RNS_17RayResultCallbackEEN15LocalInfoAdder2D0Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTVZN16btCollisionWorld13rayTestSingleERK11btTransformS2_P17btCollisionObjectPK16btCollisionShapeS2_RNS_17RayResultCallbackEE15LocalInfoAdder2;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        heap32[(g0)] = r0;
        _ZdlPv(i7);
        return;
}

function _ZZN16btCollisionWorld13rayTestSingleERK11btTransformS2_P17btCollisionObjectPK16btCollisionShapeS2_RNS_17RayResultCallbackEEN15LocalInfoAdder215addSingleResultERNS_14LocalRayResultEb(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
var __label__ = 0;
        i7 = sp + -24;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = sp + -8;
        r0 = r0 >> 2;
        heap32[(fp+-2)] = -1;
        r2 = heap32[(fp+1)];
        r3 = r1 >> 2;
        r4 = heap32[(r0+6)];
        r5 = r2 >> 2;
        heap32[(r3+1)] = r4;
        r3 = heap32[(r5+1)];
if(!(r3 !=0)) //_LBB205_2
{
        heap32[(r5+1)] = r1;
}
        r1 = heap32[(fp+2)];
        r3 = heap32[(r0+5)];
        r4 = r3 >> 2;
        r4 = heap32[(r4)];
        r4 = r4 >> 2;
        r4 = heap32[(r4+3)];
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r2;
        heap32[(g0+2)] = r1;
        __FUNCTION_TABLE__[(r4)>>2](i7);
        r1 = heap32[(r0+5)];
        r1 = r1 >> 2;
        heap32[(r0+1)] = heap32[(r1+1)];
        return;
}

function _ZZN16btCollisionWorld17objectQuerySingleEPK13btConvexShapeRK11btTransformS5_P17btCollisionObjectPK16btCollisionShapeS5_RNS_20ConvexResultCallbackEfEN32BridgeTriangleConvexcastCallback9reportHitERK9btVector3SG_fii(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var f0;
        var f1;
var __label__ = 0;
        i7 = sp + -72;var g0 = i7>>2; // save stack
        r0 = sp + -8;
        r1 = heap32[(fp+4)];
        r2 = heap32[(fp)];
        r3 = r0 >> 2;
        r4 = heap32[(fp+5)];
        heap32[(fp+-2)] = r1;
        r1 = r2 >> 2;
        heap32[(r3+1)] = r4;
        r2 = heap32[(r1+52)];
        f0 = heapFloat[(fp+3)];
        r3 = r2 >> 2;
        f1 = heapFloat[(r3+1)];
        if(f1 <f0) //_LBB206_2
{
        f_g0 = f0;
        return;
}
else{
        r4 = heap32[(fp+1)];
        r5 = heap32[(fp+2)];
        r1 = heap32[(r1+53)];
        r6 = sp + -56;
        r7 = r6 >> 2;
        heap32[(fp+-14)] = r1;
        r1 = r4 >> 2;
        heap32[(r7+1)] = r0;
        heap32[(r7+2)] = heap32[(r1)];
        heap32[(r7+3)] = heap32[(r1+1)];
        heap32[(r7+4)] = heap32[(r1+2)];
        r0 = r5 >> 2;
        heap32[(r7+5)] = heap32[(r1+3)];
        heap32[(r7+6)] = heap32[(r0)];
        heap32[(r7+7)] = heap32[(r0+1)];
        heap32[(r7+8)] = heap32[(r0+2)];
        heap32[(r7+9)] = heap32[(r0+3)];
        heapFloat[(r7+10)] = f0;
        r0 = heap32[(r3)];
        r0 = r0 >> 2;
        r0 = heap32[(r0+3)];
        heap32[(g0)] = r2;
        heap32[(g0+1)] = r6;
        heap32[(g0+2)] = 1;
        __FUNCTION_TABLE__[(r0)>>2](i7);
        return;
}
}

function _ZZN16btCollisionWorld17objectQuerySingleEPK13btConvexShapeRK11btTransformS5_P17btCollisionObjectPK16btCollisionShapeS5_RNS_20ConvexResultCallbackEfEN32BridgeTriangleConvexcastCallback9reportHitE_0RK9btVector3SG_fii(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var f0;
        var f1;
var __label__ = 0;
        i7 = sp + -72;var g0 = i7>>2; // save stack
        r0 = sp + -8;
        r1 = heap32[(fp+4)];
        r2 = heap32[(fp)];
        r3 = r0 >> 2;
        r4 = heap32[(fp+5)];
        heap32[(fp+-2)] = r1;
        r1 = r2 >> 2;
        heap32[(r3+1)] = r4;
        r2 = heap32[(r1+52)];
        f0 = heapFloat[(fp+3)];
        r3 = r2 >> 2;
        f1 = heapFloat[(r3+1)];
        if(f1 <f0) //_LBB207_2
{
        f_g0 = f0;
        return;
}
else{
        r4 = heap32[(fp+1)];
        r5 = heap32[(fp+2)];
        r1 = heap32[(r1+53)];
        r6 = sp + -56;
        r7 = r6 >> 2;
        heap32[(fp+-14)] = r1;
        r1 = r4 >> 2;
        heap32[(r7+1)] = r0;
        heap32[(r7+2)] = heap32[(r1)];
        heap32[(r7+3)] = heap32[(r1+1)];
        heap32[(r7+4)] = heap32[(r1+2)];
        r0 = r5 >> 2;
        heap32[(r7+5)] = heap32[(r1+3)];
        heap32[(r7+6)] = heap32[(r0)];
        heap32[(r7+7)] = heap32[(r0+1)];
        heap32[(r7+8)] = heap32[(r0+2)];
        heap32[(r7+9)] = heap32[(r0+3)];
        heapFloat[(r7+10)] = f0;
        r0 = heap32[(r3)];
        r0 = r0 >> 2;
        r0 = heap32[(r0+3)];
        heap32[(g0)] = r2;
        heap32[(g0+1)] = r6;
        heap32[(g0+2)] = 0;
        __FUNCTION_TABLE__[(r0)>>2](i7);
        return;
}
}

function _ZZN16btCollisionWorld17objectQuerySingleEPK13btConvexShapeRK11btTransformS5_P17btCollisionObjectPK16btCollisionShapeS5_RNS_20ConvexResultCallbackEfEN14LocalInfoAdderD1Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTVZN16btCollisionWorld17objectQuerySingleEPK13btConvexShapeRK11btTransformS5_P17btCollisionObjectPK16btCollisionShapeS5_RNS_20ConvexResultCallbackEfE14LocalInfoAdder;
        r0 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r0)] = r1;
        return;
}

function _ZZN16btCollisionWorld17objectQuerySingleEPK13btConvexShapeRK11btTransformS5_P17btCollisionObjectPK16btCollisionShapeS5_RNS_20ConvexResultCallbackEfEN14LocalInfoAdderD0Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTVZN16btCollisionWorld17objectQuerySingleEPK13btConvexShapeRK11btTransformS5_P17btCollisionObjectPK16btCollisionShapeS5_RNS_20ConvexResultCallbackEfE14LocalInfoAdder;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        heap32[(g0)] = r0;
        _ZdlPv(i7);
        return;
}

function _ZZN16btCollisionWorld17objectQuerySingleEPK13btConvexShapeRK11btTransformS5_P17btCollisionObjectPK16btCollisionShapeS5_RNS_20ConvexResultCallbackEfEN14LocalInfoAdder15addSingleResultERNS_17LocalConvexResultEb(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
var __label__ = 0;
        i7 = sp + -24;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = sp + -8;
        r0 = r0 >> 2;
        heap32[(fp+-2)] = -1;
        r2 = heap32[(fp+1)];
        r3 = r1 >> 2;
        r4 = heap32[(r0+4)];
        r5 = r2 >> 2;
        heap32[(r3+1)] = r4;
        r3 = heap32[(r5+1)];
if(!(r3 !=0)) //_LBB210_2
{
        heap32[(r5+1)] = r1;
}
        r1 = heap32[(fp+2)];
        r3 = heap32[(r0+3)];
        r4 = r3 >> 2;
        r4 = heap32[(r4)];
        r4 = r4 >> 2;
        r4 = heap32[(r4+3)];
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r2;
        heap32[(g0+2)] = r1;
        __FUNCTION_TABLE__[(r4)>>2](i7);
        r1 = heap32[(r0+3)];
        r1 = r1 >> 2;
        heap32[(r0+1)] = heap32[(r1+1)];
        return;
}

function _ZN17DebugDrawcallbackD1Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = _ZTV17DebugDrawcallback;
        r1 = heap32[(fp)];
        r1 = r1 >> 2;
        r2 = (r0 + 8)|0;
        r3 = _ZTV31btInternalTriangleIndexCallback;
        r0 = (r0 + 32)|0;
        heap32[(r1)] = r2;
        r2 = _ZTV18btTriangleCallback;
        r3 = (r3 + 8)|0;
        heap32[(r1+1)] = r0;
        r0 = (r2 + 8)|0;
        heap32[(r1+1)] = r3;
        heap32[(r1)] = r0;
        return;
}

function _ZN17DebugDrawcallbackD0Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = _ZTV17DebugDrawcallback;
        r1 = heap32[(fp)];
        r2 = r1 >> 2;
        r3 = (r0 + 8)|0;
        r4 = _ZTV31btInternalTriangleIndexCallback;
        r0 = (r0 + 32)|0;
        heap32[(r2)] = r3;
        r3 = _ZTV18btTriangleCallback;
        r4 = (r4 + 8)|0;
        heap32[(r2+1)] = r0;
        r0 = (r3 + 8)|0;
        heap32[(r2+1)] = r4;
        heap32[(r2)] = r0;
        heap32[(g0)] = r1;
        _ZdlPv(i7);
        return;
}

function _ZN17DebugDrawcallback15processTriangleEP9btVector3ii(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var f0;
        var f1;
        var f2;
        var f3;
        var f4;
        var f5;
        var f6;
        var f7;
        var f8;
        var f9;
        var f10;
        var f11;
        var f12;
        var f13;
        var f14;
        var f15;
        var f16;
        var f17;
        var f18;
        var f19;
        var f20;
        var f21;
var __label__ = 0;
        i7 = sp + -112;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = heap32[(fp+1)];
        r2 = r0 >> 2;
        r1 = r1 >> 2;
        f0 = heapFloat[(r1)];
        f1 = heapFloat[(r2+7)];
        f2 = heapFloat[(r1+1)];
        f3 = heapFloat[(r2+8)];
        f4 = heapFloat[(r2+11)];
        f5 = heapFloat[(r2+12)];
        f6 = f1*f0;
        f7 = f3*f2;
        f8 = heapFloat[(r1+2)];
        f9 = heapFloat[(r2+9)];
        f10 = heapFloat[(r2+15)];
        f11 = heapFloat[(r2+16)];
        f12 = heapFloat[(r2+13)];
        f13 = f4*f0;
        f14 = f5*f2;
        f6 = f6+f7;
        f7 = f9*f8;
        f15 = heapFloat[(r2+17)];
        f0 = f10*f0;
        f2 = f11*f2;
        f13 = f13+f14;
        f14 = f12*f8;
        f6 = f6+f7;
        f7 = heapFloat[(r2+19)];
        f16 = heapFloat[(r2+21)];
        f17 = heapFloat[(r2+20)];
        r3 = sp + -32;
        f13 = f13+f14;
        f0 = f0+f2;
        f2 = f15*f8;
        f6 = f6+f7;
        f0 = f0+f2;
        r4 = r3 >> 2;
        f2 = f13+f17;
        heapFloat[(fp+-8)] = f6;
        f0 = f0+f16;
        heapFloat[(r4+1)] = f2;
        heapFloat[(r4+2)] = f0;
        heap32[(r4+3)] = 0;
        f8 = heapFloat[(r1+4)];
        f13 = heapFloat[(r1+5)];
        f14 = heapFloat[(r1+6)];
        f18 = f1*f8;
        f19 = f3*f13;
        f20 = f4*f8;
        f21 = f5*f13;
        f18 = f18+f19;
        f19 = f9*f14;
        f18 = f18+f19;
        f8 = f10*f8;
        f13 = f11*f13;
        f19 = f20+f21;
        f20 = f12*f14;
        f18 = f18+f7;
        r4 = sp + -48;
        f19 = f19+f20;
        f8 = f8+f13;
        f13 = f15*f14;
        f14 = f19+f17;
        f8 = f8+f13;
        r5 = r4 >> 2;
        heapFloat[(fp+-12)] = f18;
        f8 = f8+f16;
        heapFloat[(r5+1)] = f14;
        heapFloat[(r5+2)] = f8;
        heap32[(r5+3)] = 0;
        f13 = heapFloat[(r1+8)];
        f19 = heapFloat[(r1+9)];
        f20 = heapFloat[(r1+10)];
        f1 = f1*f13;
        f3 = f3*f19;
        f4 = f4*f13;
        f5 = f5*f19;
        f1 = f1+f3;
        f3 = f9*f20;
        f1 = f1+f3;
        f3 = f10*f13;
        f9 = f11*f19;
        f4 = f4+f5;
        f5 = f12*f20;
        r1 = sp + -64;
        f4 = f4+f5;
        f1 = f1+f7;
        f3 = f3+f9;
        f5 = f15*f20;
        f3 = f3+f5;
        f4 = f4+f17;
        r5 = r1 >> 2;
        heapFloat[(fp+-16)] = f1;
        f5 = f6+f18;
        f3 = f3+f16;
        heapFloat[(r5+1)] = f4;
        f7 = f2+f14;
        heapFloat[(r5+2)] = f3;
        f5 = f5+f1;
        f9 =        0.3333333432674408;
        r6 = sp + -80;
        f10 = f0+f8;
        f7 = f7+f4;
        f8 = f8-f0;
        f4 = f4-f2;
        f11 = f18-f6;
        f0 = f3-f0;
        f2 = f14-f2;
        f1 = f1-f6;
        f5 = f5*f9;
        heap32[(r5+3)] = 0;
        f3 = f10+f3;
        f6 = f2*f0;
        f10 = f8*f4;
        f8 = f8*f1;
        f0 = f11*f0;
        r5 = r6 >> 2;
        f7 = f7*f9;
        heapFloat[(fp+-20)] = f5;
        f5 = f6-f10;
        f0 = f8-f0;
        f4 = f11*f4;
        f1 = f2*f1;
        f2 = f3*f9;
        heapFloat[(r5+1)] = f7;
        f1 = f4-f1;
        heapFloat[(r5+2)] = f2;
        f2 = f5*f5;
        f3 = f0*f0;
        heap32[(r5+3)] = 0;
        f2 = f2+f3;
        f3 = f1*f1;
        f2 = f2+f3;
        heapFloat[(g0)] = f2;
        r7 = sp + -96;
        sqrtf(i7);
        r8 = r7 >> 2;
        heap32[(fp+-24)] = 1065353216;
        heap32[(r8+1)] = 1065353216;
        heap32[(r8+2)] = 0;
        heap32[(r8+3)] = 0;
        r8 = heap32[(r2+2)];
        r9 = r8 >> 2;
        f3 =                         1;
        r9 = heap32[(r9)];
        f2 = f3/f_g0;
        r9 = r9 >> 2;
        r9 = heap32[(r9+2)];
        f3 = f5*f2;
        f4 = heapFloat[(fp+-20)];
        f5 = heapFloat[(r5+2)];
        f6 = heapFloat[(r5+1)];
        r5 = sp + -16;
        f0 = f0*f2;
        f3 = f4+f3;
        f1 = f1*f2;
        r10 = r5 >> 2;
        f0 = f6+f0;
        heapFloat[(fp+-4)] = f3;
        f1 = f5+f1;
        heapFloat[(r10+1)] = f0;
        heapFloat[(r10+2)] = f1;
        heap32[(r10+3)] = 0;
        heap32[(g0)] = r8;
        heap32[(g0+1)] = r6;
        heap32[(g0+2)] = r5;
        heap32[(g0+3)] = r7;
        __FUNCTION_TABLE__[(r9)>>2](i7);
        r5 = heap32[(r2+2)];
        r6 = r5 >> 2;
        r6 = heap32[(r6)];
        r6 = r6 >> 2;
        r6 = heap32[(r6+2)];
        r0 = (r0 + 12)|0;
        heap32[(g0)] = r5;
        heap32[(g0+1)] = r3;
        heap32[(g0+2)] = r4;
        heap32[(g0+3)] = r0;
        __FUNCTION_TABLE__[(r6)>>2](i7);
        r5 = heap32[(r2+2)];
        r6 = r5 >> 2;
        r6 = heap32[(r6)];
        r6 = r6 >> 2;
        r6 = heap32[(r6+2)];
        heap32[(g0)] = r5;
        heap32[(g0+1)] = r4;
        heap32[(g0+2)] = r1;
        heap32[(g0+3)] = r0;
        __FUNCTION_TABLE__[(r6)>>2](i7);
        r2 = heap32[(r2+2)];
        r4 = r2 >> 2;
        r4 = heap32[(r4)];
        r4 = r4 >> 2;
        r4 = heap32[(r4+2)];
        heap32[(g0)] = r2;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = r3;
        heap32[(g0+3)] = r0;
        __FUNCTION_TABLE__[(r4)>>2](i7);
        return;
}

function _ZN17DebugDrawcallback28internalProcessTriangleIndexEP9btVector3ii(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
var __label__ = 0;
        i7 = sp + -16;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = r0 >> 2;
        r1 = heap32[(r1)];
        r1 = r1 >> 2;
        r1 = heap32[(r1+2)];
        r2 = heap32[(fp+1)];
        r3 = heap32[(fp+2)];
        r4 = heap32[(fp+3)];
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r2;
        heap32[(g0+2)] = r3;
        heap32[(g0+3)] = r4;
        __FUNCTION_TABLE__[(r1)>>2](i7);
        return;
}

function _ZThn4_N17DebugDrawcallbackD1Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
var __label__ = 0;
        i7 = sp + 0;var g0 = i7>>2; // save stack
        r0 = _ZTV17DebugDrawcallback;
        r1 = heap32[(fp)];
        r1 = r1 >> 2;
        r2 = (r0 + 8)|0;
        r3 = _ZTV31btInternalTriangleIndexCallback;
        r0 = (r0 + 32)|0;
        heap32[(r1+-1)] = r2;
        r2 = _ZTV18btTriangleCallback;
        r3 = (r3 + 8)|0;
        heap32[(r1)] = r0;
        r0 = (r2 + 8)|0;
        heap32[(r1)] = r3;
        heap32[(r1+-1)] = r0;
        return;
}

function _ZThn4_N17DebugDrawcallbackD0Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV17DebugDrawcallback;
        r2 = r0 >> 2;
        r3 = (r1 + 8)|0;
        r4 = _ZTV31btInternalTriangleIndexCallback;
        r1 = (r1 + 32)|0;
        heap32[(r2+-1)] = r3;
        r3 = _ZTV18btTriangleCallback;
        r4 = (r4 + 8)|0;
        heap32[(r2)] = r1;
        r1 = (r3 + 8)|0;
        heap32[(r2)] = r4;
        heap32[(r2+-1)] = r1;
        r0 = (r0 + -4)|0;
        heap32[(g0)] = r0;
        _ZdlPv(i7);
        return;
}

function _ZThn4_N17DebugDrawcallback28internalProcessTriangleIndexEP9btVector3ii(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
var __label__ = 0;
        i7 = sp + -16;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = r0 >> 2;
        r1 = heap32[(r1+-1)];
        r1 = r1 >> 2;
        r1 = heap32[(r1+2)];
        r0 = (r0 + -4)|0;
        r2 = heap32[(fp+1)];
        r3 = heap32[(fp+2)];
        r4 = heap32[(fp+3)];
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r2;
        heap32[(g0+2)] = r3;
        heap32[(g0+3)] = r4;
        __FUNCTION_TABLE__[(r1)>>2](i7);
        return;
}

function _ZN16btCollisionWorldD1Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
var __label__ = 0;
        i7 = sp + -8;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        heap32[(g0)] = r0;
        _ZN16btCollisionWorldD2Ev(i7);
        return;
}

function _ZN16btCollisionWorldD0Ev(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
var __label__ = 0;
        i7 = sp + -16;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = _ZTV16btCollisionWorld;
        r2 = r0 >> 2;
        r1 = (r1 + 8)|0;
        heap32[(r2)] = r1;
        r1 = heap32[(r2+4)];
        r3 = heap32[(r2+2)];
        if(r3 >0) //_LBB219_2
{
        r3 = 0;
_3: while(true){
        r4 = r3 << 2;
        r1 = (r1 + r4)|0;
        r1 = r1 >> 2;
        r1 = heap32[(r1)];
        r1 = r1 >> 2;
        r4 = heap32[(r1+47)];
if(!(r4 ==0)) //_LBB219_5
{
        r5 = heap32[(r2+20)];
        r6 = r5 >> 2;
        r6 = heap32[(r6)];
        r6 = r6 >> 2;
        r6 = heap32[(r6+9)];
        heap32[(g0)] = r5;
        __FUNCTION_TABLE__[(r6)>>2](i7);
        r6 = r_g0 >> 2;
        r6 = heap32[(r6)];
        r6 = r6 >> 2;
        r6 = heap32[(r6+10)];
        r7 = heap32[(r2+6)];
        heap32[(g0)] = r_g0;
        heap32[(g0+1)] = r4;
        heap32[(g0+2)] = r7;
        __FUNCTION_TABLE__[(r6)>>2](i7);
        r5 = heap32[(r2+20)];
        r6 = r5 >> 2;
        r6 = heap32[(r6)];
        r6 = r6 >> 2;
        r6 = heap32[(r6+3)];
        r7 = heap32[(r2+6)];
        heap32[(g0)] = r5;
        heap32[(g0+1)] = r4;
        heap32[(g0+2)] = r7;
        __FUNCTION_TABLE__[(r6)>>2](i7);
        heap32[(r1+47)] = 0;
}
        r3 = (r3 + 1)|0;
        r1 = heap32[(r2+4)];
        r4 = heap32[(r2+2)];
        if(r4 >r3) //_LBB219_3
{
continue _3;
}
else{
break _3;
}
}
}
if(!(r1 ==0)) //_LBB219_10
{
        r3 = heapU8[r0+20];
if(!(r3 ==0)) //_LBB219_9
{
        r3 = gNumAlignedFree;
        r3 = r3 >> 2;
        r4 = heap32[(r3)];
        r4 = (r4 + 1)|0;
        r1 = r1 >> 2;
        heap32[(r3)] = r4;
        r1 = heap32[(r1+-1)];
        heap32[(g0)] = r1;
        free(i7);
}
        heap32[(r2+4)] = 0;
}
        r1 = 1;
        heap8[r0+20] = r1;
        heap32[(r2+4)] = 0;
        heap32[(r2+2)] = 0;
        heap32[(r2+3)] = 0;
        heap32[(g0)] = r0;
        _ZdlPv(i7);
        return;
}

function _ZN16btCollisionWorld11updateAabbsEv(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var f0;
        var f1;
        var f2;
        var f3;
        var f4;
        var f5;
        var f6;
var __label__ = 0;
        i7 = sp + -64;var g0 = i7>>2; // save stack
        r0 = _2E_str1998;
        r1 = heap32[(fp)];
        heap32[(g0)] = r0;
        r0 = r1 >> 2;
        _ZN15CProfileManager13Start_ProfileEPKc(i7);
        r2 = heap32[(r0+2)];
if(!(r2 <1)) //_LBB220_14
{
        r2 = 0;
_3: while(true){
        r3 = heap32[(r0+4)];
        r4 = r2 << 2;
        r3 = (r3 + r4)|0;
        r3 = r3 >> 2;
        r2 = (r2 + 1)|0;
        r3 = heap32[(r3)];
        r4 = heapU8[r1+88];
        if(r4 !=0) //_LBB220_5
{
__label__ = 5;
}
else{
        r4 = r3 >> 2;
        r4 = heap32[(r4+54)];
        if(r4 ==2) //_LBB220_13
{
__label__ = 13;
}
else{
        if(r4 ==5) //_LBB220_13
{
__label__ = 13;
}
else{
__label__ = 5;
}
}
}
_8: do {
if (__label__ == 5){
        r4 = r3 >> 2;
        r5 = heap32[(r4+48)];
        r6 = r5 >> 2;
        r6 = heap32[(r6)];
        r6 = r6 >> 2;
        r6 = heap32[(r6+2)];
        r7 = sp + -40;
        r8 = sp + -24;
        r9 = (r3 + 4)|0;
        heap32[(g0)] = r5;
        heap32[(g0+1)] = r9;
        heap32[(g0+2)] = r8;
        heap32[(g0+3)] = r7;
        __FUNCTION_TABLE__[(r6)>>2](i7);
        f0 =     -0.019999999552965164;
        f1 = heapFloat[(fp+-6)];
        f1 = f1+f0;
        r5 = r8 >> 2;
        heapFloat[(fp+-6)] = f1;
        f2 = heapFloat[(r5+1)];
        f2 = f2+f0;
        heapFloat[(r5+1)] = f2;
        f3 = heapFloat[(r5+2)];
        f0 = f3+f0;
        heapFloat[(r5+2)] = f0;
        f3 =      0.019999999552965164;
        f4 = heapFloat[(fp+-10)];
        f4 = f4+f3;
        r5 = r7 >> 2;
        heapFloat[(fp+-10)] = f4;
        f5 = heapFloat[(r5+1)];
        f5 = f5+f3;
        heapFloat[(r5+1)] = f5;
        f6 = heapFloat[(r5+2)];
        f3 = f6+f3;
        heapFloat[(r5+2)] = f3;
        r5 = heap32[(r0+20)];
        r3 = heapU8[r3+204];
        r3 = r3 & 1;
if(!(r3 != 0)) //_LBB220_10
{
        f1 = f4-f1;
        f2 = f5-f2;
        f0 = f3-f0;
        f1 = f1*f1;
        f2 = f2*f2;
        f1 = f1+f2;
        f0 = f0*f0;
        f0 = f1+f0;
        f1 =              999999995904;
if(!(f0 <f1)) //_LBB220_10
{
        r5 = heap32[(r4+54)];
        r5 = (r5 + -4)|0;
if(!(uint(r5) <uint(2))) //_LBB220_9
{
        heap32[(r4+54)] = 5;
}
        r4 = _ZZN16btCollisionWorld16updateSingleAabbEP17btCollisionObjectE8reportMe_2E_b;
        r5 = heapU8[r4];
        if(r5 != 0) //_LBB220_13
{
break _8;
}
else{
        r3 = heap32[(r0+21)];
        if(r3 ==0) //_LBB220_13
{
break _8;
}
else{
        r5 = 1;
        r6 = r3 >> 2;
        heap8[r4] = r5;
        r4 = heap32[(r6)];
        r4 = r4 >> 2;
        r4 = heap32[(r4+9)];
        r5 = _2E_str1594;
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r5;
        __FUNCTION_TABLE__[(r4)>>2](i7);
        r3 = heap32[(r0+21)];
        r4 = r3 >> 2;
        r4 = heap32[(r4)];
        r4 = r4 >> 2;
        r4 = heap32[(r4+9)];
        r5 = _2E_str1695;
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r5;
        __FUNCTION_TABLE__[(r4)>>2](i7);
        r3 = heap32[(r0+21)];
        r4 = r3 >> 2;
        r4 = heap32[(r4)];
        r4 = r4 >> 2;
        r4 = heap32[(r4+9)];
        r5 = _2E_str1796;
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r5;
        __FUNCTION_TABLE__[(r4)>>2](i7);
        r3 = heap32[(r0+21)];
        r4 = r3 >> 2;
        r4 = heap32[(r4)];
        r4 = r4 >> 2;
        r4 = heap32[(r4+9)];
        r5 = _2E_str1897;
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r5;
        __FUNCTION_TABLE__[(r4)>>2](i7);
break _8;
}
}
}
}
        r3 = r5 >> 2;
        r3 = heap32[(r3)];
        r3 = r3 >> 2;
        r3 = heap32[(r3+4)];
        r6 = heap32[(r0+6)];
        r4 = heap32[(r4+47)];
        heap32[(g0)] = r5;
        heap32[(g0+1)] = r4;
        heap32[(g0+2)] = r8;
        heap32[(g0+3)] = r7;
        heap32[(g0+4)] = r6;
        __FUNCTION_TABLE__[(r3)>>2](i7);
}
} while(0);
        r3 = heap32[(r0+2)];
        if(r3 >r2) //_LBB220_2
{
continue _3;
}
else{
break _3;
}
}
}
        r0 = _ZN15CProfileManager11CurrentNodeE;
        r0 = r0 >> 2;
        r1 = heap32[(r0)];
        r2 = r1 >> 2;
        r3 = heap32[(r2+4)];
        r3 = (r3 + -1)|0;
        heap32[(r2+4)] = r3;
_21: do {
if(!(r3 !=0)) //_LBB220_20
{
        r3 = heap32[(r2+1)];
        if(r3 !=0) //_LBB220_17
{
        r1 = sp + -8;
        heap32[(g0)] = r1;
        heap32[(g0+1)] = 0;
        r3 = _ZL13gProfileClock_2E_0;
        gettimeofday(i7);
        r3 = r3 >> 2;
        r3 = heap32[(r3)];
        r3 = r3 >> 2;
        r1 = r1 >> 2;
        r4 = heap32[(fp+-2)];
        r5 = heap32[(r3)];
        r4 = (r4 - r5)|0;
        r1 = heap32[(r1+1)];
        r3 = heap32[(r3+1)];
        r1 = (r1 - r3)|0;
        r3 = (r4 * 1000000)|0;
        r1 = (r1 + r3)|0;
        r3 = heap32[(r2+3)];
        r1 = (r1 - r3)|0;
        f0 = uint(r1); //fuitos r1, f0
        f1 =                      1000;
        f2 = heapFloat[(r2+2)];
        f0 = f0/f1;
        f0 = f2+f0;
        heapFloat[(r2+2)] = f0;
        r1 = heap32[(r2+4)];
        if(r1 !=0) //_LBB220_20
{
break _21;
}
else{
        r1 = heap32[(r0)];
}
}
        r1 = r1 >> 2;
        r1 = heap32[(r1+5)];
        heap32[(r0)] = r1;
}
} while(0);
        return;
}

function _ZN16btCollisionWorld14debugDrawWorldEv(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var r12;
        var r13;
        var r14;
        var r15;
        var f0;
var __label__ = 0;
        i7 = sp + -104;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = r0 >> 2;
        r2 = heap32[(r1)];
        r2 = r2 >> 2;
        r2 = heap32[(r2+4)];
        heap32[(g0)] = r0;
        __FUNCTION_TABLE__[(r2)>>2](i7);
        r2 = r_g0;
_1: do {
if(!(r2 ==0)) //_LBB221_10
{
        r2 = heap32[(r1)];
        r2 = r2 >> 2;
        r2 = heap32[(r2+4)];
        heap32[(g0)] = r0;
        __FUNCTION_TABLE__[(r2)>>2](i7);
        r3 = r_g0 >> 2;
        r3 = heap32[(r3)];
        r3 = r3 >> 2;
        r3 = heap32[(r3+12)];
        heap32[(g0)] = r_g0;
        __FUNCTION_TABLE__[(r3)>>2](i7);
        r2 = r_g0 & 8;
if(!(r2 ==0)) //_LBB221_10
{
        r2 = heap32[(r1+6)];
        r3 = r2 >> 2;
        r3 = heap32[(r3)];
        r3 = r3 >> 2;
        r3 = heap32[(r3+9)];
        heap32[(g0)] = r2;
        __FUNCTION_TABLE__[(r3)>>2](i7);
        r2 = r_g0;
        r3 = sp + -16;
        r4 = r3 >> 2;
        heap32[(fp+-4)] = 0;
        heap32[(r4+1)] = 0;
        r5 = 0;
        heap32[(r4+2)] = 0;
        heap32[(r4+3)] = 0;
_4: while(true){
        if(r5 <r2) //_LBB221_3
{
        r4 = heap32[(r1+6)];
        r6 = r4 >> 2;
        r6 = heap32[(r6)];
        r6 = r6 >> 2;
        r6 = heap32[(r6+10)];
        heap32[(g0)] = r4;
        heap32[(g0+1)] = r5;
        __FUNCTION_TABLE__[(r6)>>2](i7);
        r4 = r_g0;
        r6 = r4 >> 2;
        r6 = heap32[(r6+279)];
        r7 = 36;
        r8 = 68;
        r9 = 0;
_7: while(true){
        if(r9 <r6) //_LBB221_4
{
        r10 = r4 >> 2;
        r10 = heap32[(r10+279)];
        if(r10 >r9) //_LBB221_6
{
        r10 = (r9 * 69)|0;
        r11 = (r4 + r7)|0;
        r12 = (r4 + r8)|0;
        r13 = heap32[(r1)];
        r13 = r13 >> 2;
        r13 = heap32[(r13+4)];
        heap32[(g0)] = r0;
        __FUNCTION_TABLE__[(r13)>>2](i7);
        r14 = r_g0 >> 2;
        r15 = r10 << 2;
        r10 = r10 << 2;
        r14 = heap32[(r14)];
        r15 = (r4 + r15)|0;
        r10 = (r4 + r10)|0;
        r14 = r14 >> 2;
        r15 = r15 >> 2;
        r10 = r10 >> 2;
        r14 = heap32[(r14+8)];
        r15 = heap32[(r15+37)];
        f0 = heapFloat[(r10+21)];
        heap32[(g0)] = r_g0;
        heap32[(g0+1)] = r11;
        heap32[(g0+2)] = r12;
        heapFloat[(g0+3)] = f0;
        heap32[(g0+4)] = r15;
        heap32[(g0+5)] = r3;
        r9 = (r9 + 1)|0;
        r8 = (r8 + 276)|0;
        r7 = (r7 + 276)|0;
        __FUNCTION_TABLE__[(r14)>>2](i7);
}
else{
break _4;
}
}
else{
break _7;
}
}
        r5 = (r5 + 1)|0;
}
else{
break _1;
}
}
        r0 = _2E_str382;
        r1 = _2E_str483;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r1;
        heap32[(g0+2)] = 122;
        _assert(i7);
}
}
} while(0);
        r2 = heap32[(r1)];
        r2 = r2 >> 2;
        r2 = heap32[(r2+4)];
        heap32[(g0)] = r0;
        __FUNCTION_TABLE__[(r2)>>2](i7);
        r2 = r_g0;
_14: do {
if(!(r2 ==0)) //_LBB221_13
{
        r2 = heap32[(r1)];
        r2 = r2 >> 2;
        r2 = heap32[(r2+4)];
        heap32[(g0)] = r0;
        __FUNCTION_TABLE__[(r2)>>2](i7);
        r3 = r_g0 >> 2;
        r3 = heap32[(r3)];
        r3 = r3 >> 2;
        r3 = heap32[(r3+12)];
        heap32[(g0)] = r_g0;
        __FUNCTION_TABLE__[(r3)>>2](i7);
        r2 = r_g0 & 3;
if(!(r2 ==0)) //_LBB221_13
{
        r2 = heap32[(r1+2)];
        if(r2 >0) //_LBB221_14
{
        r2 = 0;
_18: while(true){
        r3 = heap32[(r1+4)];
        r4 = r2 << 2;
        r3 = (r3 + r4)|0;
        r3 = r3 >> 2;
        r3 = heap32[(r3)];
        r4 = heapU8[r3+204];
        r4 = r4 & 32;
if(!(r4 !=0)) //_LBB221_34
{
        r4 = heap32[(r1)];
        r4 = r4 >> 2;
        r4 = heap32[(r4+4)];
        heap32[(g0)] = r0;
        __FUNCTION_TABLE__[(r4)>>2](i7);
        r4 = r_g0;
if(!(r4 ==0)) //_LBB221_31
{
        r4 = heap32[(r1)];
        r4 = r4 >> 2;
        r4 = heap32[(r4+4)];
        heap32[(g0)] = r0;
        __FUNCTION_TABLE__[(r4)>>2](i7);
        r5 = r_g0 >> 2;
        r5 = heap32[(r5)];
        r5 = r5 >> 2;
        r5 = heap32[(r5+12)];
        heap32[(g0)] = r_g0;
        __FUNCTION_TABLE__[(r5)>>2](i7);
        r4 = r_g0 & 1;
if(!(r4 ==0)) //_LBB221_31
{
        r4 = sp + -32;
        r5 = r4 >> 2;
        heap32[(fp+-8)] = 1065353216;
        heap32[(r5+1)] = 1065353216;
        heap32[(r5+2)] = 1065353216;
        r6 = r3 >> 2;
        heap32[(r5+3)] = 0;
        r7 = heap32[(r6+54)];
_25: do {
        if(r7 >2) //_LBB221_21
{
        if(r7 ==3) //_LBB221_26
{
        heap32[(fp+-8)] = 0;
        heap32[(r5+1)] = 1065353216;
        heap32[(r5+2)] = 1065353216;
        heap32[(r5+3)] = 0;
__label__ = 30;
break _25;
}
else{
        if(r7 ==4) //_LBB221_27
{
        heap32[(fp+-8)] = 1065353216;
        heap32[(r5+1)] = 0;
        heap32[(r5+2)] = 0;
        heap32[(r5+3)] = 0;
__label__ = 30;
break _25;
}
else{
        if(r7 ==5) //_LBB221_28
{
        heap32[(fp+-8)] = 1065353216;
        heap32[(r5+1)] = 1065353216;
        heap32[(r5+2)] = 0;
        heap32[(r5+3)] = 0;
__label__ = 30;
break _25;
}
else{
__label__ = 29;
break _25;
}
}
}
}
else{
        if(r7 ==1) //_LBB221_24
{
        heap32[(fp+-8)] = 1065353216;
        heap32[(r5+1)] = 1065353216;
        heap32[(r5+2)] = 1065353216;
        heap32[(r5+3)] = 0;
__label__ = 30;
}
else{
        if(r7 ==2) //_LBB221_25
{
        heap32[(fp+-8)] = 0;
        heap32[(r5+1)] = 1065353216;
        heap32[(r5+2)] = 0;
        heap32[(r5+3)] = 0;
__label__ = 30;
}
else{
__label__ = 29;
}
}
}
} while(0);
if (__label__ == 29){
        heap32[(fp+-8)] = 1065353216;
        heap32[(r5+1)] = 0;
        heap32[(r5+2)] = 0;
        heap32[(r5+3)] = 0;
}
        r5 = heap32[(r1)];
        r5 = r5 >> 2;
        r5 = heap32[(r5+6)];
        r6 = heap32[(r6+48)];
        r7 = (r3 + 4)|0;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r7;
        heap32[(g0+2)] = r6;
        heap32[(g0+3)] = r4;
        __FUNCTION_TABLE__[(r5)>>2](i7);
}
}
        r4 = heap32[(r1+21)];
if(!(r4 ==0)) //_LBB221_34
{
        r5 = r4 >> 2;
        r5 = heap32[(r5)];
        r5 = r5 >> 2;
        r5 = heap32[(r5+12)];
        heap32[(g0)] = r4;
        __FUNCTION_TABLE__[(r5)>>2](i7);
        r4 = r_g0 & 2;
if(!(r4 ==0)) //_LBB221_34
{
        r4 = sp + -80;
        r5 = r4 >> 2;
        heap32[(fp+-20)] = 1065353216;
        heap32[(r5+1)] = 0;
        heap32[(r5+2)] = 0;
        r6 = r3 >> 2;
        heap32[(r5+3)] = 0;
        r5 = heap32[(r6+48)];
        r6 = r5 >> 2;
        r6 = heap32[(r6)];
        r6 = r6 >> 2;
        r6 = heap32[(r6+2)];
        r3 = (r3 + 4)|0;
        r7 = sp + -48;
        r8 = sp + -64;
        heap32[(g0)] = r5;
        heap32[(g0+1)] = r3;
        heap32[(g0+2)] = r7;
        heap32[(g0+3)] = r8;
        __FUNCTION_TABLE__[(r6)>>2](i7);
        r3 = heap32[(r1+21)];
        r5 = r3 >> 2;
        r5 = heap32[(r5)];
        r5 = r5 >> 2;
        r5 = heap32[(r5+13)];
        heap32[(g0)] = r3;
        heap32[(g0+1)] = r7;
        heap32[(g0+2)] = r8;
        heap32[(g0+3)] = r4;
        __FUNCTION_TABLE__[(r5)>>2](i7);
}
}
}
        r2 = (r2 + 1)|0;
        r3 = heap32[(r1+2)];
        if(r3 >r2) //_LBB221_15
{
continue _18;
}
else{
break _14;
}
}
}
}
}
} while(0);
        return;
}

function _ZN16btCollisionWorld15debugDrawObjectERK11btTransformPK16btCollisionShapeRK9btVector3(sp)
{
        var i7;
        var fp = sp>>2;
        var r0;
        var r1;
        var r2;
        var r3;
        var r4;
        var r5;
        var r6;
        var r7;
        var r8;
        var r9;
        var r10;
        var r11;
        var r12;
        var r13;
        var r14;
        var r15;
        var f0;
        var f1;
        var f2;
        var f3;
        var f4;
        var f5;
        var f6;
        var f7;
        var f8;
        var f9;
        var f10;
        var f11;
        var f12;
        var f13;
        var f14;
        var f15;
        var f16;
        var f17;
        var f18;
        var f19;
        var f20;
        var f21;
        var f22;
        var f23;
        var f24;
        var f25;
        var f26;
        var f27;
        var f28;
        var f29;
var __label__ = 0;
        i7 = sp + -1376;var g0 = i7>>2; // save stack
        r0 = heap32[(fp)];
        r1 = r0 >> 2;
        r2 = heap32[(r1)];
        r2 = r2 >> 2;
        r2 = heap32[(r2+4)];
        heap32[(g0)] = r0;
        __FUNCTION_TABLE__[(r2)>>2](i7);
        r3 = r_g0 >> 2;
        r3 = heap32[(r3)];
        r3 = r3 >> 2;
        r3 = heap32[(r3+14)];
        r4 = heap32[(fp+1)];
        heap32[(g0)] = r_g0;
        heap32[(g0+1)] = r4;
        heap32[(g0+2)] = 1065353216;
        r2 = heap32[(fp+2)];
        r5 = r2 >> 2;
        __FUNCTION_TABLE__[(r3)>>2](i7);
        r3 = heap32[(r5+1)];
        r6 = heap32[(fp+3)];
_1: do {
        if(r3 >10) //_LBB222_6
{
        if(r3 >27) //_LBB222_9
{
        if(r3 ==28) //_LBB222_23
{
        f0 = heapFloat[(r5+14)];
        f1 = heapFloat[(r5+13)];
        f2 = heapFloat[(r5+16)];
        f3 = heapFloat[(r5+12)];
        f4 = f0*f2;
        f5 = f1*f2;
        f2 = f3*f2;
        f6 =                         0;
        if(f0 <f6) //_LBB222_25
{
        f7 = -f0;
}
else{
        f7 = f0;
}
        f8 =       0.70710676908493042;
        if(f7 <=f8) //_LBB222_28
{
        f0 = f3*f3;
        f1 = f1*f1;
        f0 = f0+f1;
        heapFloat[(g0)] = f0;
        sqrtf(i7);
        f3 =                         1;
        f7 = heapFloat[(r5+13)];
        f9 = heapFloat[(r5+14)];
        f10 = f3/f_g0;
        f1 = heapFloat[(r5+12)];
        f3 = -f7;
        f8 = f1*f10;
        f1 = -f9;
        f11 = f10*f3;
        f12 =                       100;
        f1 = f8*f1;
        f3 = f9*f11;
        f7 = f0*f10;
        f0 = f11*f12;
}
else{
        f1 = f1*f1;
        f3 = f0*f0;
        f1 = f1+f3;
        heapFloat[(g0)] = f1;
        sqrtf(i7);
        f6 = heapFloat[(r5+14)];
        f0 =                         1;
        f7 = heapFloat[(r5+12)];
        f3 = f0/f_g0;
        f6 = -f6;
        f0 = heapFloat[(r5+13)];
        f8 = f3*f6;
        f6 = -f7;
        f0 = f0*f3;
        f9 =                       100;
        f1 = f1*f3;
        f3 = f0*f6;
        f7 = f7*f8;
        f6 = f0*f9;
        f0 =                         0;
}
        r2 = heap32[(r1)];
        r2 = r2 >> 2;
        r2 = heap32[(r2+4)];
        heap32[(g0)] = r0;
        f9 =                       100;
        __FUNCTION_TABLE__[(r2)>>2](i7);
        r3 = r4 >> 2;
        f8 = f8*f9;
        r4 = r_g0 >> 2;
        f10 = f2-f0;
        f11 = heapFloat[(r3)];
        f12 = f5-f8;
        f13 = heapFloat[(r3+1)];
        r4 = heap32[(r4)];
        f14 = heapFloat[(r3+4)];
        f15 = heapFloat[(r3+5)];
        f16 = f11*f10;
        f17 = f13*f12;
        f18 = f4-f6;
        f19 = heapFloat[(r3+2)];
        r4 = r4 >> 2;
        f20 = heapFloat[(r3+8)];
        f21 = heapFloat[(r3+9)];
        f22 = heapFloat[(r3+6)];
        f23 = f14*f10;
        f24 = f15*f12;
        f16 = f16+f17;
        f17 = f19*f18;
        f25 = heapFloat[(r3+10)];
        r4 = heap32[(r4+2)];
        f10 = f20*f10;
        f12 = f21*f12;
        f23 = f23+f24;
        f24 = f22*f18;
        f16 = f16+f17;
        f17 = heapFloat[(r3+12)];
        f26 = heapFloat[(r3+14)];
        f27 = heapFloat[(r3+13)];
        f0 = f2+f0;
        f8 = f5+f8;
        r5 = sp + -64;
        f23 = f23+f24;
        f10 = f10+f12;
        f12 = f25*f18;
        f16 = f16+f17;
        f6 = f4+f6;
        f10 = f10+f12;
        f11 = f11*f0;
        f12 = f13*f8;
        r7 = r5 >> 2;
        f13 = f23+f27;
        heapFloat[(fp+-16)] = f16;
        f14 = f14*f0;
        f15 = f15*f8;
        f11 = f11+f12;
        f12 = f19*f6;
        f10 = f10+f26;
        heapFloat[(r7+1)] = f13;
        f11 = f11+f12;
        heapFloat[(r7+2)] = f10;
        f0 = f20*f0;
        f8 = f21*f8;
        f10 = f14+f15;
        f12 = f22*f6;
        r8 = sp + -48;
        f10 = f10+f12;
        f0 = f0+f8;
        f6 = f25*f6;
        f8 = f11+f17;
        heap32[(r7+3)] = 0;
        f0 = f0+f6;
        r7 = r8 >> 2;
        f6 = f10+f27;
        heapFloat[(fp+-12)] = f8;
        f0 = f0+f26;
        heapFloat[(r7+1)] = f6;
        heapFloat[(r7+2)] = f0;
        heap32[(r7+3)] = 0;
        heap32[(g0)] = r_g0;
        heap32[(g0+1)] = r8;
        heap32[(g0+2)] = r5;
        heap32[(g0+3)] = r6;
        __FUNCTION_TABLE__[(r4)>>2](i7);
        r1 = heap32[(r1)];
        r1 = r1 >> 2;
        r1 = heap32[(r1+4)];
        heap32[(g0)] = r0;
        __FUNCTION_TABLE__[(r1)>>2](i7);
        f0 = f1*f9;
        f1 = f3*f9;
        f3 = f7*f9;
        r1 = r_g0 >> 2;
        f6 = f2-f0;
        f7 = heapFloat[(r3)];
        f8 = f5-f1;
        f9 = heapFloat[(r3+1)];
        r1 = heap32[(r1)];
        f10 = heapFloat[(r3+4)];
        f11 = heapFloat[(r3+5)];
        f12 = f7*f6;
        f13 = f9*f8;
        f14 = f4-f3;
        f15 = heapFloat[(r3+2)];
        r1 = r1 >> 2;
        f16 = heapFloat[(r3+8)];
        f17 = heapFloat[(r3+9)];
        f18 = heapFloat[(r3+6)];
        f19 = f10*f6;
        f20 = f11*f8;
        f12 = f12+f13;
        f13 = f15*f14;
        f21 = heapFloat[(r3+10)];
        r1 = heap32[(r1+2)];
        f6 = f16*f6;
        f8 = f17*f8;
        f19 = f19+f20;
        f20 = f18*f14;
        f12 = f12+f13;
        f13 = heapFloat[(r3+12)];
        f22 = heapFloat[(r3+14)];
        f23 = heapFloat[(r3+13)];
        f0 = f2+f0;
        f1 = f5+f1;
        r2 = sp + -32;
        f2 = f19+f20;
        f5 = f6+f8;
        f6 = f21*f14;
        f8 = f12+f13;
        f3 = f4+f3;
        f4 = f5+f6;
        f5 = f7*f0;
        f6 = f9*f1;
        r3 = r2 >> 2;
        f2 = f2+f23;
        heapFloat[(fp+-8)] = f8;
        f7 = f10*f0;
        f8 = f11*f1;
        f5 = f5+f6;
        f6 = f15*f3;
        f4 = f4+f22;
        heapFloat[(r3+1)] = f2;
        f2 = f5+f6;
        heapFloat[(r3+2)] = f4;
        f0 = f16*f0;
        f1 = f17*f1;
        f4 = f7+f8;
        f5 = f18*f3;
        r4 = sp + -16;
        f4 = f4+f5;
        f0 = f0+f1;
        f1 = f21*f3;
        f2 = f2+f13;
        heap32[(r3+3)] = 0;
        f0 = f0+f1;
        r3 = r4 >> 2;
        f1 = f4+f23;
        heapFloat[(fp+-4)] = f2;
        f0 = f0+f22;
        heapFloat[(r3+1)] = f1;
        heapFloat[(r3+2)] = f0;
        heap32[(r3+3)] = 0;
        heap32[(g0)] = r_g0;
        heap32[(g0+1)] = r4;
__label__ = 30;
break _1;
}
else{
        if(r3 !=31) //_LBB222_32
{
__label__ = 32;
break _1;
}
else{
        r2 = heap32[(r5+4)];
        r3 = (r2 + -1)|0;
        if(r3 <0) //_LBB222_42
{
__label__ = 40;
break _1;
}
else{
        r3 = 1;
        r2 = (r3 - r2)|0;
_18: while(true){
        r3 = r2;
        r2 = (r3 * -20)|0;
        r7 = heap32[(r5+6)];
        r2 = r2 << 2;
        r2 = (r7 + r2)|0;
        r2 = r2 >> 2;
        r7 = r4 >> 2;
        r8 = heap32[(r1)];
        f0 = heapFloat[(r2)];
        f1 = heapFloat[(r7)];
        f2 = heapFloat[(r2+4)];
        f3 = heapFloat[(r7+1)];
        r8 = r8 >> 2;
        f4 = heapFloat[(r2+1)];
        f5 = heapFloat[(r2+5)];
        f6 = f0*f1;
        f7 = f2*f3;
        f8 = heapFloat[(r2+8)];
        f9 = heapFloat[(r7+2)];
        f10 = heapFloat[(r7+8)];
        f11 = heapFloat[(r2+2)];
        f12 = heapFloat[(r7+4)];
        f13 = heapFloat[(r2+12)];
        f14 = heapFloat[(r7+9)];
        f15 = heapFloat[(r2+6)];
        f16 = heapFloat[(r7+5)];
        f17 = heapFloat[(r2+13)];
        f18 = heapFloat[(r2+9)];
        f19 = heapFloat[(r7+10)];
        f20 = heapFloat[(r2+10)];
        f21 = heapFloat[(r7+6)];
        f22 = heapFloat[(r2+14)];
        r8 = heap32[(r8+6)];
        f23 = f4*f1;
        f24 = f5*f3;
        f6 = f6+f7;
        f7 = f8*f9;
        r2 = heap32[(r2+16)];
        f25 = heapFloat[(r7+14)];
        f26 = heapFloat[(r7+13)];
        f27 = heapFloat[(r7+12)];
        r7 = sp + -688;
        f28 = f11*f1;
        f29 = f15*f3;
        f23 = f23+f24;
        f24 = f18*f9;
        f6 = f6+f7;
        r9 = r7 >> 2;
        f7 = f28+f29;
        f28 = f20*f9;
        f23 = f23+f24;
        heapFloat[(fp+-172)] = f6;
        f6 = f0*f12;
        f24 = f2*f16;
        f7 = f7+f28;
        heapFloat[(r9+1)] = f23;
        heapFloat[(r9+2)] = f7;
        f7 = f4*f12;
        f23 = f5*f16;
        f6 = f6+f24;
        f24 = f8*f21;
        f28 = f11*f12;
        f29 = f15*f16;
        f7 = f7+f23;
        f23 = f18*f21;
        f6 = f6+f24;
        heap32[(r9+3)] = 0;
        f24 = f28+f29;
        f28 = f20*f21;
        f7 = f7+f23;
        heapFloat[(r9+4)] = f6;
        f0 = f0*f10;
        f2 = f2*f14;
        f6 = f24+f28;
        heapFloat[(r9+5)] = f7;
        heapFloat[(r9+6)] = f6;
        f4 = f4*f10;
        f5 = f5*f14;
        f0 = f0+f2;
        f2 = f8*f19;
        f6 = f11*f10;
        f7 = f15*f14;
        f4 = f4+f5;
        f5 = f18*f19;
        f0 = f0+f2;
        heap32[(r9+7)] = 0;
        f1 = f1*f13;
        f2 = f3*f17;
        f3 = f6+f7;
        f6 = f20*f19;
        f4 = f4+f5;
        heapFloat[(r9+8)] = f0;
        f0 = f12*f13;
        f5 = f16*f17;
        f1 = f1+f2;
        f2 = f9*f22;
        f3 = f3+f6;
        heapFloat[(r9+9)] = f4;
        f1 = f1+f2;
        heapFloat[(r9+10)] = f3;
        f2 = f10*f13;
        f3 = f14*f17;
        f0 = f0+f5;
        f4 = f21*f22;
        f0 = f0+f4;
        f2 = f2+f3;
        f3 = f19*f22;
        f1 = f1+f27;
        heap32[(r9+11)] = 0;
        f2 = f2+f3;
        f0 = f0+f26;
        heapFloat[(r9+12)] = f1;
        f1 = f2+f25;
        heapFloat[(r9+13)] = f0;
        heapFloat[(r9+14)] = f1;
        heap32[(r9+15)] = 0;
        heap32[(g0)] = r0;
        heap32[(g0+1)] = r7;
        heap32[(g0+2)] = r2;
        heap32[(g0+3)] = r6;
        r2 = (r3 + 1)|0;
        __FUNCTION_TABLE__[(r8)>>2](i7);
        if(r3 ==0) //_LBB222_42
{
__label__ = 40;
break _1;
}
else{
continue _18;
}
}
}
}
}
}
else{
        if(r3 ==11) //_LBB222_20
{
        r2 = r4 >> 2;
        r3 = heap32[(r5+17)];
        f0 = heapFloat[(r5+14)];
        f1 = heapFloat[(r5+15)];
        f2 = heapFloat[(r2+12)];
        f3 = heapFloat[(r2+13)];
        f4 = heapFloat[(r2+14)];
        r4 = sp + -880;
        r5 = r4 >> 2;
        heap32[(fp+-220)] = 0;
        heap32[(r5+1)] = 0;
        r7 = r3 << 2;
        r4 = (r4 + r7)|0;
        f5 =                       0.5;
        heap32[(r5+2)] = 0;
        r4 = r4 >> 2;
        f1 = f1*f5;
        heap32[(r5+3)] = 0;
        r8 = sp + -896;
        heapFloat[(r4)] = f1;
        r4 = (r3 + 1)|0;
        r4 = (r4 % 3)|0;
        r9 = r8 >> 2;
        heap32[(fp+-224)] = 0;
        heap32[(r9+1)] = 0;
        r4 = r4 << 2;
        r8 = (r8 + r4)|0;
        heap32[(r9+2)] = 0;
        r8 = r8 >> 2;
        heap32[(r9+3)] = 0;
        r3 = (r3 + 2)|0;
        r10 = sp + -912;
        heapFloat[(r8)] = f0;
        r3 = (r3 % 3)|0;
        r8 = r10 >> 2;
        heap32[(fp+-228)] = 0;
        r3 = r3 << 2;
        heap32[(r8+1)] = 0;
        r3 = (r10 + r3)|0;
        heap32[(r8+2)] = 0;
        r3 = r3 >> 2;
        heap32[(r8+3)] = 0;
        heapFloat[(r3)] = f0;
        r3 = heap32[(r1)];
        r3 = r3 >> 2;
        r3 = heap32[(r3+4)];
        heap32[(g0)] = r0;
        __FUNCTION_TABLE__[(r3)>>2](i7);
        f1 = heapFloat[(r5+1)];
        f5 = heapFloat[(fp+-220)];
        f6 = heapFloat[(r9+1)];
        f7 = heapFloat[(fp+-224)];
        r10 = r_g0 >> 2;
        f8 = heapFloat[(r5+2)];
        f9 = heapFloat[(r9+2)];
        f10 = heapFloat[(r2)];
        f11 = f7-f5;
        f12 = heapFloat[(r2+1)];
        f13 = f6-f1;
        f14 = heapFloat[(r2+4)];
        f15 = heapFloat[(r2+5)];
        r5 = heap32[(r10)];
        f16 = heapFloat[(r2+2)];
        f17 = f9-f8;
        f18 = f10*f11;
        f19 = f12*f13;
        f20 = heapFloat[(r2+8)];
        f21 = heapFloat[(r2+9)];
        f22 = heapFloat[(r2+6)];
        r5 = r5 >> 2;
        f23 = f14*f11;
        f24 = f15*f13;
        f18 = f18+f19;
        f19 = f16*f17;
        f25 = heapFloat[(r2+10)];
        r5 = heap32[(r5+2)];
        f18 = f18+f19;
        f11 = f20*f11;
        f13 = f21*f13;
        f19 = f23+f24;
        f23 = f22*f17;
        r9 = sp + -416;
        f19 = f19+f23;
        f11 = f11+f13;
        f13 = f25*f17;
        f17 = f2+f18;
        f11 = f11+f13;
        f10 = f10*f5;
        f12 = f12*f1;
        r10 = r9 >> 2;
        f13 = f3+f19;
        heapFloat[(fp+-104)] = f17;
        f14 = f14*f5;
        f15 = f15*f1;
        f10 = f10+f12;
        f12 = f16*f8;
        f11 = f4+f11;
        heapFloat[(r10+1)] = f13;
        f10 = f10+f12;
        heapFloat[(r10+2)] = f11;
        f11 = f20*f5;
        f12 = f21*f1;
        f13 = f14+f15;
        f14 = f22*f8;
        r11 = sp + -400;
        f13 = f13+f14;
        f11 = f11+f12;
        f12 = f25*f8;
        f10 = f2+f10;
        heap32[(r10+3)] = 0;
        f11 = f11+f12;
        r10 = r11 >> 2;
        f12 = f3+f13;
        heapFloat[(fp+-100)] = f10;
        f10 = f4+f11;
        heapFloat[(r10+1)] = f12;
        heapFloat[(r10+2)] = f10;
        heap32[(r10+3)] = 0;
        heap32[(g0)] = r_g0;
        heap32[(g0+1)] = r11;
        heap32[(g0+2)] = r9;
        heap32[(g0+3)] = r6;
        __FUNCTION_TABLE__[(r5)>>2](i7);
        r3 = heap32[(r1)];
        r3 = r3 >> 2;
        r3 = heap32[(r3+4)];
        heap32[(g0)] = r0;
        __FUNCTION_TABLE__[(r3)>>2](i7);
        f10 = -f5;
        f11 = -f1;
        f12 = -f8;
        r5 = r_g0 >> 2;
        f13 = heapFloat[(r2)];
        f7 = f10-f7;
        f14 = heapFloat[(r2+1)];
        f6 = f11-f6;
        f15 = heapFloat[(r2+4)];
        f16 = heapFloat[(r2+5)];
        r5 = heap32[(r5)];
        f17 = heapFloat[(r2+2)];
        f9 = f12-f9;
        f18 = f13*f7;
        f19 = f14*f6;
        f20 = heapFloat[(r2+8)];
        f21 = heapFloat[(r2+9)];
        f22 = heapFloat[(r2+6)];
        r5 = r5 >> 2;
        f23 = f15*f7;
        f24 = f16*f6;
        f18 = f18+f19;
        f19 = f17*f9;
        f25 = heapFloat[(r2+10)];
        r5 = heap32[(r5+2)];
        f18 = f18+f19;
        f7 = f20*f7;
        f6 = f21*f6;
        f19 = f23+f24;
        f23 = f22*f9;
        r9 = sp + -384;
        f19 = f19+f23;
        f6 = f7+f6;
        f7 = f25*f9;
        f9 = f2+f18;
        f6 = f6+f7;
        f7 = f13*f5;
        f13 = f14*f1;
        r10 = r9 >> 2;
        f14 = f3+f19;
        heapFloat[(fp+-96)] = f9;
        f9 = f15*f5;
        f15 = f16*f1;
        f7 = f7+f13;
        f13 = f17*f8;
        f6 = f4+f6;
        heapFloat[(r10+1)] = f14;
        f7 = f7+f13;
        heapFloat[(r10+2)] = f6;
        f6 = f20*f5;
        f13 = f21*f1;
        f9 = f9+f15;
        f14 = f22*f8;
        r11 = sp + -368;
        f9 = f9+f14;
        f6 = f6+f13;
        f13 = f25*f8;
        f7 = f2+f7;
        heap32[(r10+3)] = 0;
        f6 = f6+f13;
        r10 = r11 >> 2;
        f9 = f3+f9;
        heapFloat[(fp+-92)] = f7;
        f6 = f4+f6;
        heapFloat[(r10+1)] = f9;
        heapFloat[(r10+2)] = f6;
        heap32[(r10+3)] = 0;
        heap32[(g0)] = r_g0;
        heap32[(g0+1)] = r11;
        heap32[(g0+2)] = r9;
        heap32[(g0+3)] = r6;
        __FUNCTION_TABLE__[(r5)>>2](i7);
        r3 = heap32[(r1)];
        r3 = r3 >> 2;
        r3 = heap32[(r3+4)];
        heap32[(g0)] = r0;
        __FUNCTION_TABLE__[(r3)>>2](i7);
        f6 = heapFloat[(r8+1)];
        f7 = heapFloat[(fp+-228)];
        f9 = heapFloat[(r8+2)];
        r5 = r_g0 >> 2;
        f13 = heapFloat[(r2)];
        f14 = f7-f5;
        f15 = heapFloat[(r2+1)];
        f16 = f6-f1;
        f17 = heapFloat[(r2+4)];
        f18 = heapFloat[(r2+5)];
        r5 = heap32[(r5)];
        f19 = heapFloat[(r2+2)];
        f20 = f9-f8;
        f21 = f13*f14;
        f22 = f15*f16;
        f23 = heapFloat[(r2+8)];
        f24 = heapFloat[(r2+9)];
        f25 = heapFloat[(r2+6)];
        r5 = r5 >> 2;
        f26 = f17*f14;
        f27 = f18*f16;
        f21 = f21+f22;
        f22 = f19*f20;
        f28 = heapFloat[(r2+10)];
        r5 = heap32[(r5+2)];
        f21 = f21+f22;
        f14 = f23*f14;
        f16 = f24*f16;
        f22 = f26+f27;
        f26 = f25*f20;
        r8 = sp + -352;
        f22 = f22+f26;
        f14 = f14+f16;
        f16 = f28*f20;
        f20 = f2+f21;
        f14 = f14+f16;
        f13 = f13*f5;
        f15 = f15*f1;
        r9 = r8 >> 2;
        f16 = f3+f22;
        heapFloat[(fp+-88)] = f20;
        f17 = f17*f5;
        f18 = f18*f1;
        f13 = f13+f15;
        f15 = f19*f8;
        f14 = f4+f14;
        heapFloat[(r9+1)] = f16;
        f13 = f13+f15;
        heapFloat[(r9+2)] = f14;
        f14 = f23*f5;
        f15 = f24*f1;
        f16 = f17+f18;
        f17 = f25*f8;
        r10 = sp + -336;
        f16 = f16+f17;
        f14 = f14+f15;
        f15 = f28*f8;
        f13 = f2+f13;
        heap32[(r9+3)] = 0;
        f14 = f14+f15;
        r9 = r10 >> 2;
        f15 = f3+f16;
        heapFloat[(fp+-84)] = f13;
        f13 = f4+f14;
        heapFloat[(r9+1)] = f15;
        heapFloat[(r9+2)] = f13;
        heap32[(r9+3)] = 0;
        heap32[(g0)] = r_g0;
        heap32[(g0+1)] = r10;
        heap32[(g0+2)] = r8;
        heap32[(g0+3)] = r6;
        __FUNCTION_TABLE__[(r5)>>2](i7);
        r3 = heap32[(r1)];
        r3 = r3 >> 2;
        r3 = heap32[(r3+4)];
        heap32[(g0)] = r0;
        __FUNCTION_TABLE__[(r3)>>2](i7);
        r5 = r_g0 >> 2;
        f13 = heapFloat[(r2)];
        f7 = f10-f7;
        f10 = heapFloat[(r2+1)];
        f6 = f11-f6;
        f11 = heapFloat[(r2+4)];
        f14 = heapFloat[(r2+5)];
        r5 = heap32[(r5)];
        f15 = heapFloat[(r2+2)];
        f9 = f12-f9;
        f12 = f13*f7;
        f16 = f10*f6;
        f17 = heapFloat[(r2+8)];
        f18 = heapFloat[(r2+9)];
        f19 = heapFloat[(r2+6)];
        r5 = r5 >> 2;
        f20 = f11*f7;
        f21 = f14*f6;
        f12 = f12+f16;
        f16 = f15*f9;
        f22 = heapFloat[(r2+10)];
        r5 = heap32[(r5+2)];
        f12 = f12+f16;
        f7 = f17*f7;
        f6 = f18*f6;
        f16 = f20+f21;
        f20 = f19*f9;
        r8 = sp + -320;
        f16 = f16+f20;
        f6 = f7+f6;
        f7 = f22*f9;
        f9 = f2+f12;
        f6 = f6+f7;
        f7 = f13*f5;
        f10 = f10*f1;
        r9 = r8 >> 2;
        f12 = f3+f16;
        heapFloat[(fp+-80)] = f9;
        f9 = f11*f5;
        f11 = f14*f1;
        f7 = f7+f10;
        f10 = f15*f8;
        f6 = f4+f6;
        heapFloat[(r9+1)] = f12;
        f7 = f7+f10;
        heapFloat[(r9+2)] = f6;
        f6 = f17*f5;
        f10 = f18*f1;
        f9 = f9+f11;
        f11 = f19*f8;
        r10 = sp + -304;
        f9 = f9+f11;
        f6 = f6+f10;
        f10 = f22*f8;
        f7 = f2+f7;
        heap32[(r9+3)] = 0;
        f6 = f6+f10;
        r9 = r10 >> 2;
        f9 = f3+f9;
        heapFloat[(fp+-76)] = f7;
        f6 = f4+f6;
        heapFloat[(r9+1)] = f9;
        heapFloat[(r9+2)] = f6;
        heap32[(r9+3)] = 0;
        heap32[(g0)] = r_g0;
        heap32[(g0+1)] = r10;
        heap32[(g0+2)] = r8;
        heap32[(g0+3)] = r6;
        r3 = sp + -928;
        __FUNCTION_TABLE__[(r5)>>2](i7);
        r5 = r3 >> 2;
        heap32[(fp+-232)] = 0;
        heap32[(r5+1)] = 0;
        r3 = (r3 + r7)|0;
        heap32[(r5+2)] = 0;
        r3 = r3 >> 2;
        heap32[(r5+3)] = 0;
        r7 = sp + -944;
        heap32[(r3)] = 1065353216;
        r3 = r7 >> 2;
        heap32[(fp+-236)] = 0;
        heap32[(r3+1)] = 0;
        r4 = (r7 + r4)|0;
        heap32[(r3+2)] = 0;
        r4 = r4 >> 2;
        heap32[(r3+3)] = 0;
        heap32[(r4)] = 1065353216;
        r1 = heap32[(r1)];
        r1 = r1 >> 2;
        r1 = heap32[(r1+4)];
        heap32[(g0)] = r0;
        __FUNCTION_TABLE__[(r1)>>2](i7);
        r1 = r_g0 >> 2;
        r1 = heap32[(r1)];
        f6 = heapFloat[(r2)];
        f7 = heapFloat[(fp+-236)];
        f9 = heapFloat[(r2+1)];
        f10 = heapFloat[(r3+1)];
        f11 = heapFloat[(r2+4)];
        f12 = heapFloat[(r2+5)];
        r1 = r1 >> 2;
        f13 = heapFloat[(r2+2)];
        f14 = heapFloat[(r3+2)];
        f15 = f6*f7;
        f16 = f9*f10;
        f17 = heapFloat[(r2+8)];
        f18 = heapFloat[(r2+9)];
        f19 = heapFloat[(r2+10)];
        f20 = heapFloat[(r2+6)];
        r1 = heap32[(r1+15)];
        f21 = f11*f7;
        f22 = f12*f10;
        f15 = f15+f16;
        f16 = f13*f14;
        r2 = sp + -288;
        f7 = f17*f7;
        f10 = f18*f10;
        f21 = f21+f22;
        f22 = f20*f14;
        f15 = f15+f16;
        r3 = r2 >> 2;
        f7 = f7+f10;
        f10 = f19*f14;
        f14 = f21+f22;
        heapFloat[(fp+-72)] = f15;
        f7 = f7+f10;
        heapFloat[(r3+1)] = f14;
        heapFloat[(r3+2)] = f7;
        heap32[(r3+3)] = 0;
        f7 = heapFloat[(fp+-232)];
        f10 = heapFloat[(r5+1)];
        f14 = heapFloat[(r5+2)];
        f15 = f6*f7;
        f16 = f9*f10;
        f21 = f11*f7;
        f22 = f12*f10;
        f15 = f15+f16;
        f16 = f13*f14;
        r3 = sp + -272;
        f7 = f17*f7;
        f10 = f18*f10;
        f21 = f21+f22;
        f22 = f20*f14;
        f15 = f15+f16;
        f6 = f6*f5;
        f9 = f9*f1;
        r4 = r3 >> 2;
        f7 = f7+f10;
        f10 = f19*f14;
        f14 = f21+f22;
        heapFloat[(fp+-68)] = f15;
        f11 = f11*f5;
        f12 = f12*f1;
        f6 = f6+f9;
        f9 = f13*f8;
        f7 = f7+f10;
        heapFloat[(r4+1)] = f14;
        f6 = f6+f9;
        heapFloat[(r4+2)] = f7;
        f5 = f17*f5;
        f1 = f18*f1;
        f7 = f11+f12;
        f9 = f20*f8;
        r5 = sp + -256;
        f7 = f7+f9;
        f1 = f5+f1;
        f5 = f19*f8;
        f2 = f2-f6;
        heap32[(r4+3)] = 0;
        f1 = f1+f5;
        r4 = r5 >> 2;
        f3 = f3-f7;
        heapFloat[(fp+-64)] = f2;
        f1 = f4-f1;
        heapFloat[(r4+1)] = f3;
        heapFloat[(r4+2)] = f1;
        heap32[(r4+3)] = 0;
        heap32[(g0)] = r_g0;
        heap32[(g0+1)] = r5;
        heap32[(g0+2)] = r3;
}
else{
        if(r3 ==13) //_LBB222_21
{
        r3 = heap32[(r5)];
        r3 = r3 >> 2;
        r7 = heap32[(r5+13)];
        r3 = heap32[(r3+21)];
        heap32[(g0)] = r2;
        __FUNCTION_TABLE__[(r3)>>2](i7);
        f0 = f_g0;
        r3 = sp + -240;
        f1 = heapFloat[(r5+8)];
        f2 = heapFloat[(r5+9)];
        f3 = heapFloat[(r5+7)];
        r8 = r3 >> 2;
        heap32[(r8+3)] = heap32[(r5+10)];
        r9 = heap32[(r5)];
        r9 = r9 >> 2;
        r9 = heap32[(r9+11)];
        heap32[(g0)] = r2;
        __FUNCTION_TABLE__[(r9)>>2](i7);
        f4 = f_g0;
        r9 = heap32[(r5)];
        r9 = r9 >> 2;
        r9 = heap32[(r9+11)];
        heap32[(g0)] = r2;
        __FUNCTION_TABLE__[(r9)>>2](i7);
        f5 = f_g0;
        r5 = heap32[(r5)];
        r5 = r5 >> 2;
        r5 = heap32[(r5+11)];
        heap32[(g0)] = r2;
        __FUNCTION_TABLE__[(r5)>>2](i7);
        f3 = f3+f_g0;
        r2 = r7 << 2;
        f1 = f1+f5;
        heapFloat[(fp+-60)] = f3;
        r3 = (r3 + r2)|0;
        f2 = f2+f4;
        heapFloat[(r8+1)] = f1;
        r3 = r3 >> 2;
        heapFloat[(r8+2)] = f2;
        r4 = r4 >> 2;
        f1 = heapFloat[(r3)];
        f2 = heapFloat[(r4+12)];
        f3 = heapFloat[(r4+13)];
        f4 = heapFloat[(r4+14)];
        r3 = sp + -960;
        r5 = r3 >> 2;
        heap32[(fp+-240)] = 0;
        heap32[(r5+1)] = 0;
        r3 = (r3 + r2)|0;
        heap32[(r5+2)] = 0;
        r3 = r3 >> 2;
        heap32[(r5+3)] = 0;
        r8 = sp + -976;
        heapFloat[(r3)] = f1;
        r3 = (r7 + 1)|0;
        r3 = (r3 % 3)|0;
        r7 = r8 >> 2;
        heap32[(fp+-244)] = 0;
        heap32[(r7+1)] = 0;
        r3 = r3 << 2;
        r8 = (r8 + r3)|0;
        heap32[(r7+2)] = 0;
        r8 = r8 >> 2;
        heap32[(r7+3)] = 0;
        heapFloat[(r8)] = f0;
        r8 = heap32[(r1)];
        r8 = r8 >> 2;
        r8 = heap32[(r8+4)];
        heap32[(g0)] = r0;
        __FUNCTION_TABLE__[(r8)>>2](i7);
        f1 = heapFloat[(r5+1)];
        f5 = heapFloat[(r7+1)];
        f6 = heapFloat[(fp+-240)];
        f7 = heapFloat[(fp+-244)];
        r9 = r_g0 >> 2;
        f8 = heapFloat[(r5+2)];
        f9 = heapFloat[(r7+2)];
        f10 = f7-f6;
        f11 = heapFloat[(r4)];
        f12 = f5-f1;
        f13 = heapFloat[(r4+1)];
        r5 = heap32[(r9)];
        f14 = heapFloat[(r4+4)];
        f15 = heapFloat[(r4+5)];
        f16 = f11*f10;
        f17 = f13*f12;
        f18 = f9-f8;
        f19 = heapFloat[(r4+2)];
        r5 = r5 >> 2;
        f20 = heapFloat[(r4+8)];
        f21 = heapFloat[(r4+9)];
        f22 = heapFloat[(r4+6)];
        f23 = f14*f10;
        f24 = f15*f12;
        f16 = f16+f17;
        f17 = f19*f18;
        f16 = f16+f17;
        f17 = heapFloat[(r4+10)];
        r5 = heap32[(r5+2)];
        f10 = f20*f10;
        f12 = f21*f12;
        f23 = f23+f24;
        f24 = f22*f18;
        f25 = f6+f7;
        f26 = f1+f5;
        r7 = sp + -224;
        f23 = f23+f24;
        f10 = f10+f12;
        f12 = f17*f18;
        f16 = f2+f16;
        f18 = f8+f9;
        f10 = f10+f12;
        f11 = f11*f25;
        f12 = f13*f26;
        r9 = r7 >> 2;
        f13 = f3+f23;
        heapFloat[(fp+-56)] = f16;
        f14 = f14*f25;
        f15 = f15*f26;
        f11 = f11+f12;
        f12 = f19*f18;
        f10 = f4+f10;
        heapFloat[(r9+1)] = f13;
        f11 = f11+f12;
        heapFloat[(r9+2)] = f10;
        f10 = f20*f25;
        f12 = f21*f26;
        f13 = f14+f15;
        f14 = f22*f18;
        r10 = sp + -208;
        f13 = f13+f14;
        f10 = f10+f12;
        f12 = f17*f18;
        f11 = f2+f11;
        heap32[(r9+3)] = 0;
        f10 = f10+f12;
        r9 = r10 >> 2;
        f12 = f3+f13;
        heapFloat[(fp+-52)] = f11;
        f10 = f4+f10;
        heapFloat[(r9+1)] = f12;
        heapFloat[(r9+2)] = f10;
        heap32[(r9+3)] = 0;
        heap32[(g0)] = r_g0;
        heap32[(g0+1)] = r10;
        heap32[(g0+2)] = r7;
        heap32[(g0+3)] = r6;
        __FUNCTION_TABLE__[(r5)>>2](i7);
        r5 = heap32[(r1)];
        r5 = r5 >> 2;
        r5 = heap32[(r5+4)];
        heap32[(g0)] = r0;
        __FUNCTION_TABLE__[(r5)>>2](i7);
        f10 = -f6;
        f11 = -f1;
        f12 = -f8;
        r7 = r_g0 >> 2;
        f10 = f10-f7;
        f13 = heapFloat[(r4)];
        f11 = f11-f5;
        f14 = heapFloat[(r4+1)];
        r7 = heap32[(r7)];
        f15 = heapFloat[(r4+4)];
        f16 = heapFloat[(r4+5)];
        f17 = f13*f10;
        f18 = f14*f11;
        f12 = f12-f9;
        f19 = heapFloat[(r4+2)];
        r7 = r7 >> 2;
        f20 = heapFloat[(r4+8)];
        f21 = heapFloat[(r4+9)];
        f22 = heapFloat[(r4+6)];
        f23 = f15*f10;
        f24 = f16*f11;
        f17 = f17+f18;
        f18 = f19*f12;
        f17 = f17+f18;
        f18 = heapFloat[(r4+10)];
        r7 = heap32[(r7+2)];
        f10 = f20*f10;
        f11 = f21*f11;
        f23 = f23+f24;
        f24 = f22*f12;
        f7 = f6-f7;
        f5 = f1-f5;
        r8 = sp + -192;
        f23 = f23+f24;
        f10 = f10+f11;
        f11 = f18*f12;
        f12 = f2+f17;
        f9 = f8-f9;
        f10 = f10+f11;
        f11 = f13*f7;
        f13 = f14*f5;
        r9 = r8 >> 2;
        f14 = f3+f23;
        heapFloat[(fp+-48)] = f12;
        f12 = f15*f7;
        f15 = f16*f5;
        f11 = f11+f13;
        f13 = f19*f9;
        f10 = f4+f10;
        heapFloat[(r9+1)] = f14;
        f11 = f11+f13;
        heapFloat[(r9+2)] = f10;
        f7 = f20*f7;
        f5 = f21*f5;
        f10 = f12+f15;
        f12 = f22*f9;
        r10 = sp + -176;
        f10 = f10+f12;
        f5 = f7+f5;
        f7 = f18*f9;
        f9 = f2+f11;
        heap32[(r9+3)] = 0;
        f5 = f5+f7;
        r9 = r10 >> 2;
        f7 = f3+f10;
        heapFloat[(fp+-44)] = f9;
        f5 = f4+f5;
        heapFloat[(r9+1)] = f7;
        heapFloat[(r9+2)] = f5;
        heap32[(r9+3)] = 0;
        heap32[(g0)] = r_g0;
        heap32[(g0+1)] = r10;
        heap32[(g0+2)] = r8;
        heap32[(g0+3)] = r6;
        r5 = sp + -992;
        __FUNCTION_TABLE__[(r7)>>2](i7);
        r7 = r5 >> 2;
        heap32[(fp+-248)] = 0;
        heap32[(r7+1)] = 0;
        r2 = (r5 + r2)|0;
        heap32[(r7+2)] = 0;
        r2 = r2 >> 2;
        heap32[(r7+3)] = 0;
        r5 = sp + -1008;
        heap32[(r2)] = 1065353216;
        r2 = r5 >> 2;
        heap32[(fp+-252)] = 0;
        heap32[(r2+1)] = 0;
        r3 = (r5 + r3)|0;
        heap32[(r2+2)] = 0;
        r3 = r3 >> 2;
        heap32[(r2+3)] = 0;
        heap32[(r3)] = 1065353216;
        r3 = heap32[(r1)];
        r3 = r3 >> 2;
        r3 = heap32[(r3+4)];
        heap32[(g0)] = r0;
        __FUNCTION_TABLE__[(r3)>>2](i7);
        r5 = r_g0 >> 2;
        r5 = heap32[(r5)];
        f5 = heapFloat[(r4)];
        f7 = heapFloat[(fp+-252)];
        f9 = heapFloat[(r4+1)];
        f10 = heapFloat[(r2+1)];
        f11 = heapFloat[(r4+4)];
        f12 = heapFloat[(r4+5)];
        r5 = r5 >> 2;
        f13 = heapFloat[(r4+2)];
        f14 = heapFloat[(r2+2)];
        f15 = f5*f7;
        f16 = f9*f10;
        f17 = heapFloat[(r4+8)];
        f18 = heapFloat[(r4+9)];
        f19 = heapFloat[(r4+10)];
        f20 = heapFloat[(r4+6)];
        r2 = heap32[(r5+15)];
        f21 = f11*f7;
        f22 = f12*f10;
        f15 = f15+f16;
        f16 = f13*f14;
        r5 = sp + -160;
        f23 = f17*f7;
        f24 = f18*f10;
        f21 = f21+f22;
        f22 = f20*f14;
        f15 = f15+f16;
        r8 = r5 >> 2;
        f16 = f23+f24;
        f23 = f19*f14;
        f21 = f21+f22;
        heapFloat[(fp+-40)] = f15;
        f15 = f16+f23;
        heapFloat[(r8+1)] = f21;
        heapFloat[(r8+2)] = f15;
        heap32[(r8+3)] = 0;
        f15 = heapFloat[(fp+-248)];
        f16 = heapFloat[(r7+1)];
        f21 = heapFloat[(r7+2)];
        f22 = f5*f15;
        f23 = f9*f16;
        f24 = f11*f15;
        f25 = f12*f16;
        f22 = f22+f23;
        f23 = f13*f21;
        r7 = sp + -144;
        f26 = f17*f15;
        f27 = f18*f16;
        f24 = f24+f25;
        f25 = f20*f21;
        f22 = f22+f23;
        f5 = f5*f6;
        f9 = f9*f1;
        r8 = r7 >> 2;
        f23 = f26+f27;
        f26 = f19*f21;
        f24 = f24+f25;
        heapFloat[(fp+-36)] = f22;
        f11 = f11*f6;
        f12 = f12*f1;
        f5 = f5+f9;
        f9 = f13*f8;
        f13 = f23+f26;
        heapFloat[(r8+1)] = f24;
        f5 = f5+f9;
        heapFloat[(r8+2)] = f13;
        f9 = f17*f6;
        f13 = f18*f1;
        f11 = f11+f12;
        f12 = f20*f8;
        r9 = sp + -128;
        f11 = f11+f12;
        f9 = f9+f13;
        f12 = f19*f8;
        f5 = f2-f5;
        heap32[(r8+3)] = 0;
        f9 = f9+f12;
        r8 = r9 >> 2;
        f11 = f3-f11;
        heapFloat[(fp+-32)] = f5;
        f5 = f4-f9;
        heapFloat[(r8+1)] = f11;
        heapFloat[(r8+2)] = f5;
        heap32[(r8+3)] = 0;
        heap32[(g0)] = r_g0;
        heap32[(g0+1)] = r9;
        heap32[(g0+2)] = r7;
        heap32[(g0+3)] = r5;
        heapFloat[(g0+4)] = f0;
        heapFloat[(g0+5)] = f0;
        heap32[(g0+6)] = 0;
        heap32[(g0+7)] = 1086918619;
        heap32[(g0+8)] = r6;
        heap32[(g0+9)] = 0;
        heap32[(g0+10)] = 1092616192;
        __FUNCTION_TABLE__[(r2)>>2](i7);
        r1 = heap32[(r1)];
        r1 = r1 >> 2;
        r1 = heap32[(r1+4)];
        heap32[(g0)] = r0;
        __FUNCTION_TABLE__[(r1)>>2](i7);
        r1 = r_g0 >> 2;
        f5 = heapFloat[(r4)];
        f9 = heapFloat[(r4+1)];
        r1 = heap32[(r1)];
        f11 = heapFloat[(r4+4)];
        f12 = heapFloat[(r4+5)];
        f13 = heapFloat[(r4+2)];
        r1 = r1 >> 2;
        f17 = f5*f7;
        f18 = f9*f10;
        f19 = heapFloat[(r4+8)];
        f20 = heapFloat[(r4+9)];
        f22 = heapFloat[(r4+10)];
        f23 = heapFloat[(r4+6)];
        r1 = heap32[(r1+15)];
        f24 = f11*f7;
        f25 = f12*f10;
        f17 = f17+f18;
        f18 = f13*f14;
        r2 = sp + -112;
        f7 = f19*f7;
        f10 = f20*f10;
        f24 = f24+f25;
        f25 = f23*f14;
        f17 = f17+f18;
        r3 = r2 >> 2;
        f7 = f7+f10;
        f10 = f22*f14;
        f14 = f24+f25;
        heapFloat[(fp+-28)] = f17;
        f17 = f5*f15;
        f18 = f9*f16;
        f7 = f7+f10;
        heapFloat[(r3+1)] = f14;
        heapFloat[(r3+2)] = f7;
        f7 = f11*f15;
        f10 = f12*f16;
        f14 = f17+f18;
        f17 = f13*f21;
        r4 = sp + -96;
        f15 = f19*f15;
        f16 = f20*f16;
        f7 = f7+f10;
        f10 = f23*f21;
        f14 = f14+f17;
        heap32[(r3+3)] = 0;
        f5 = f5*f6;
        f9 = f9*f1;
        r3 = r4 >> 2;
        f15 = f15+f16;
        f16 = f22*f21;
        f7 = f7+f10;
        heapFloat[(fp+-24)] = f14;
        f10 = f11*f6;
        f11 = f12*f1;
        f5 = f5+f9;
        f9 = f13*f8;
        f12 = f15+f16;
        heapFloat[(r3+1)] = f7;
        f5 = f5+f9;
        heapFloat[(r3+2)] = f12;
        f6 = f19*f6;
        f1 = f20*f1;
        f7 = f10+f11;
        f9 = f23*f8;
        r5 = sp + -80;
        f7 = f7+f9;
        f1 = f6+f1;
        f6 = f22*f8;
        f2 = f2+f5;
        heap32[(r3+3)] = 0;
        f1 = f1+f6;
        r3 = r5 >> 2;
        f3 = f3+f7;
        heapFloat[(fp+-20)] = f2;
        f1 = f4+f1;
        heapFloat[(r3+1)] = f3;
        heapFloat[(r3+2)] = f1;
        heap32[(r3+3)] = 0;
        heap32[(g0)] = r_g0;
        heap32[(g0+1)] = r5;
        heap32[(g0+2)] = r4;
}
else{
__label__ = 32;
break _1;
}
}
        heap32[(g0+3)] = r2;
        heapFloat[(g0+4)] = f0;
        heapFloat[(g0+5)] = f0;
        heap32[(g0+6)] = 0;
        heap32[(g0+7)] = 1086918619;
        heap32[(g0+8)] = r6;
        heap32[(g0+9)] = 0;
        heap32[(g0+10)] = 1092616192;
        __FUNCTION_TABLE__[(r1)>>2](i7);
        return;
}
}
else{
        if(r3 >8) //_LBB222_4
{
        if(r3 ==9) //_LBB222_16
{
        r2 = heap32[(r5+23)];
        r3 = (r2 + -1)|0;
        if(r3 <0) //_LBB222_42
{
__label__ = 40;
break _1;
}
else{
        r3 = 1;
        r2 = (r3 - r2)|0;
_32: while(true){
        r3 = r2;
        r2 = heap32[(r5+25)];
        r7 = r3 << 4;
        r8 = heap32[(r1)];
        r2 = (r2 - r7)|0;
        r2 = r2 >> 2;
        r7 = r8 >> 2;
        r7 = heap32[(r7+4)];
        f0 = heapFloat[(r2)];
        f1 = heapFloat[(r2+1)];
        f2 = heapFloat[(r2+2)];
        heap32[(g0)] = r0;
        __FUNCTION_TABLE__[(r7)>>2](i7);
        r7 = r4 >> 2;
        r8 = r_g0 >> 2;
        r8 = heap32[(r8)];
        f3 = heapFloat[(r7+1)];
        f4 =                         0;
        f5 = heapFloat[(r7)];
        f6 = heapFloat[(r7+2)];
        r8 = r8 >> 2;
        f7 = f3*f4;
        f8 = f5*f4;
        f9 = heapFloat[(r7+8)];
        f10 = heapFloat[(r7+9)];
        f11 = heapFloat[(r7+10)];
        f12 = heapFloat[(r7+4)];
        f13 = heapFloat[(r7+5)];
        f14 = heapFloat[(r7+6)];
        r8 = heap32[(r8+4)];
        f15 = f5+f7;
        f16 = f6*f4;
        f17 = heapFloat[(r7+14)];
        f18 = heapFloat[(r7+13)];
        f19 = heapFloat[(r7+12)];
        r7 = sp + -608;
        f20 = f8+f3;
        f15 = f15+f16;
        f7 = f8+f7;
        r9 = r7 >> 2;
        f8 = f20+f16;
        heapFloat[(fp+-152)] = f15;
        f15 = f13*f4;
        f7 = f7+f6;
        heapFloat[(r9+1)] = f8;
        f8 = f12*f4;
        heapFloat[(r9+2)] = f7;
        f7 = f12+f15;
        f16 = f14*f4;
        f20 = f8+f13;
        f7 = f7+f16;
        heap32[(r9+3)] = 0;
        f8 = f8+f15;
        f15 = f20+f16;
        heapFloat[(r9+4)] = f7;
        f7 = f10*f4;
        f8 = f8+f14;
        heapFloat[(r9+5)] = f15;
        f15 = f9*f4;
        heapFloat[(r9+6)] = f8;
        f8 = f9+f7;
        f4 = f11*f4;
        f16 = f15+f10;
        f8 = f8+f4;
        heap32[(r9+7)] = 0;
        f7 = f15+f7;
        f5 = f5*f0;
        f3 = f3*f1;
        f4 = f16+f4;
        heapFloat[(r9+8)] = f8;
        f8 = f12*f0;
        f12 = f13*f1;
        f3 = f5+f3;
        f5 = f6*f2;
        f6 = f7+f11;
        heapFloat[(r9+9)] = f4;
        f3 = f3+f5;
        heapFloat[(r9+10)] = f6;
        f0 = f9*f0;
        f1 = f10*f1;
        f4 = f8+f12;
        f5 = f14*f2;
        f4 = f4+f5;
        f0 = f0+f1;
        f1 = f11*f2;
        f2 = f3+f19;
        heap32[(r9+11)] = 0;
        f0 = f0+f1;
        f1 = f4+f18;
        heapFloat[(r9+12)] = f2;
        f0 = f0+f17;
        heapFloat[(r9+13)] = f1;
        heapFloat[(r9+14)] = f0;
        heap32[(r9+15)] = 0;
        r9 = heap32[(r5+30)];
        r10 = r3 << 2;
        r9 = (r9 - r10)|0;
        r9 = r9 >> 2;
        f0 = heapFloat[(r9)];
        heap32[(g0)] = r_g0;
        heapFloat[(g0+1)] = f0;
        heap32[(g0+2)] = r7;
        heap32[(g0+3)] = r6;
        r2 = (r3 + 1)|0;
        __FUNCTION_TABLE__[(r8)>>2](i7);
        if(r3 ==0) //_LBB222_42
{
__label__ = 40;
break _1;
}
else{
continue _32;
}
}
}
}
else{
        if(r3 ==10) //_LBB222_19
{
        r3 = heap32[(r5+13)];
        r5 = (r3 + 2)|0;
        r5 = (r5 % 3)|0;
        r7 = r3 << 2;
        r2 = (r2 + 28)|0;
        r5 = r5 << 2;
        r8 = (r2 + r7)|0;
        r2 = (r2 + r5)|0;
        r8 = r8 >> 2;
        r2 = r2 >> 2;
        f0 = heapFloat[(r8)];
        f1 = heapFloat[(r2)];
        r2 = sp + -720;
        r8 = r2 >> 2;
        heap32[(fp+-180)] = 0;
        heap32[(r8+1)] = 0;
        r9 = (r2 + r7)|0;
        heap32[(r8+2)] = 0;
        r9 = r9 >> 2;
        f2 = -f0;
        heap32[(r8+3)] = 0;
        r10 = sp + -736;
        heapFloat[(r9)] = f2;
        heap32[(fp+-184)] = 0;
        r9 = r10 >> 2;
        heap32[(r9+1)] = 0;
        r7 = (r10 + r7)|0;
        heap32[(r9+2)] = 0;
        r7 = r7 >> 2;
        heap32[(r9+3)] = 0;
        heapFloat[(r7)] = f0;
        r4 = r4 >> 2;
        f0 = heapFloat[(r4)];
        r7 = sp + -800;
        heapFloat[(fp+-200)] = f0;
        r11 = r7 >> 2;
        f2 = heapFloat[(r4+1)];
        heapFloat[(r11+1)] = f2;
        f3 = heapFloat[(r4+2)];
        heapFloat[(r11+2)] = f3;
        heap32[(r11+3)] = heap32[(r4+3)];
        f4 = heapFloat[(r4+4)];
        heapFloat[(r11+4)] = f4;
        f5 = heapFloat[(r4+5)];
        heapFloat[(r11+5)] = f5;
        f6 = heapFloat[(r4+6)];
        heapFloat[(r11+6)] = f6;
        heap32[(r11+7)] = heap32[(r4+7)];
        f7 = heapFloat[(r4+8)];
        heapFloat[(r11+8)] = f7;
        f8 = heapFloat[(r4+9)];
        heapFloat[(r11+9)] = f8;
        f9 = heapFloat[(r4+10)];
        heapFloat[(r11+10)] = f9;
        heap32[(r11+11)] = heap32[(r4+11)];
        f10 = heapFloat[(fp+-180)];
        f11 = heapFloat[(r8+1)];
        f12 = heapFloat[(r8+2)];
        f0 = f0*f10;
        f2 = f2*f11;
        f4 = f4*f10;
        f5 = f5*f11;
        f0 = f0+f2;
        f2 = f3*f12;
        f3 = f7*f10;
        f7 = f8*f11;
        f4 = f4+f5;
        f5 = f6*f12;
        f0 = f0+f2;
        f2 = heapFloat[(r4+12)];
        f6 = heapFloat[(r4+13)];
        f8 = heapFloat[(r4+14)];
        f4 = f4+f5;
        f3 = f3+f7;
        f5 = f9*f12;
        f0 = f0+f2;
        f2 = f3+f5;
        f3 = f4+f6;
        heapFloat[(r11+12)] = f0;
        f0 = f2+f8;
        heapFloat[(r11+13)] = f3;
        heapFloat[(r11+14)] = f0;
        heap32[(r11+15)] = 0;
        r11 = heap32[(r1)];
        r11 = r11 >> 2;
        r11 = heap32[(r11+4)];
        heap32[(g0)] = r0;
        __FUNCTION_TABLE__[(r11)>>2](i7);
        r12 = r_g0 >> 2;
        r12 = heap32[(r12)];
        r12 = r12 >> 2;
        r12 = heap32[(r12+4)];
        heap32[(g0)] = r_g0;
        heapFloat[(g0+1)] = f1;
        heap32[(g0+2)] = r7;
        heap32[(g0+3)] = r6;
        __FUNCTION_TABLE__[(r12)>>2](i7);
        f0 = heapFloat[(r4)];
        r7 = sp + -864;
        heapFloat[(fp+-216)] = f0;
        r11 = r7 >> 2;
        f2 = heapFloat[(r4+1)];
        heapFloat[(r11+1)] = f2;
        f3 = heapFloat[(r4+2)];
        heapFloat[(r11+2)] = f3;
        heap32[(r11+3)] = heap32[(r4+3)];
        f4 = heapFloat[(r4+4)];
        heapFloat[(r11+4)] = f4;
        f5 = heapFloat[(r4+5)];
        heapFloat[(r11+5)] = f5;
        f6 = heapFloat[(r4+6)];
        heapFloat[(r11+6)] = f6;
        heap32[(r11+7)] = heap32[(r4+7)];
        f7 = heapFloat[(r4+8)];
        heapFloat[(r11+8)] = f7;
        f8 = heapFloat[(r4+9)];
        heapFloat[(r11+9)] = f8;
        f9 = heapFloat[(r4+10)];
        heapFloat[(r11+10)] = f9;
        heap32[(r11+11)] = heap32[(r4+11)];
        f10 = heapFloat[(fp+-184)];
        f11 = heapFloat[(r9+1)];
        f12 = heapFloat[(r9+2)];
        f0 = f0*f10;
        f2 = f2*f11;
        f4 = f4*f10;
        f5 = f5*f11;
        f0 = f0+f2;
        f2 = f3*f12;
        f3 = f7*f10;
        f7 = f8*f11;
        f4 = f4+f5;
        f5 = f6*f12;
        f0 = f0+f2;
        f2 = heapFloat[(r4+12)];
        f6 = heapFloat[(r4+13)];
        f8 = heapFloat[(r4+14)];
        f4 = f4+f5;
        f3 = f3+f7;
        f5 = f9*f12;
        f0 = f0+f2;
        f2 = f3+f5;
        f3 = f4+f6;
        heapFloat[(r11+12)] = f0;
        f0 = f2+f8;
        heapFloat[(r11+13)] = f3;
        heapFloat[(r11+14)] = f0;
        heap32[(r11+15)] = 0;
        r11 = heap32[(r1)];
        r11 = r11 >> 2;
        r11 = heap32[(r11+4)];
        heap32[(g0)] = r0;
        __FUNCTION_TABLE__[(r11)>>2](i7);
        r12 = r_g0 >> 2;
        r12 = heap32[(r12)];
        r12 = r12 >> 2;
        r12 = heap32[(r12+4)];
        r3 = (r3 + 1)|0;
        heap32[(g0)] = r_g0;
        heapFloat[(g0+1)] = f1;
        heap32[(g0+2)] = r7;
        heap32[(g0+3)] = r6;
        r3 = (r3 % 3)|0;
        __FUNCTION_TABLE__[(r12)>>2](i7);
        r3 = r3 << 2;
        r7 = (r2 + r3)|0;
        f0 = heapFloat[(r4+12)];
        f2 = heapFloat[(r4+13)];
        f3 = heapFloat[(r4+14)];
        r3 = (r10 + r3)|0;
        r7 = r7 >> 2;
        heapFloat[(r7)] = f1;
        r3 = r3 >> 2;
        heapFloat[(r3)] = f1;
        r11 = heap32[(r1)];
        r11 = r11 >> 2;
        r11 = heap32[(r11+4)];
        heap32[(g0)] = r0;
        __FUNCTION_TABLE__[(r11)>>2](i7);
        r12 = r_g0 >> 2;
        f4 = heapFloat[(fp+-184)];
        f5 = heapFloat[(r4)];
        f6 = heapFloat[(r9+1)];
        f7 = heapFloat[(r4+1)];
        r12 = heap32[(r12)];
        f8 = heapFloat[(r4+4)];
        f9 = heapFloat[(r4+5)];
        f10 = f5*f4;
        f11 = f7*f6;
        f12 = heapFloat[(r9+2)];
        f13 = heapFloat[(r4+2)];
        r12 = r12 >> 2;
        f14 = heapFloat[(r4+8)];
        f15 = heapFloat[(r4+9)];
        f16 = heapFloat[(r4+6)];
        f17 = f8*f4;
        f18 = f9*f6;
        f10 = f10+f11;
        f11 = f13*f12;
        f10 = f10+f11;
        f11 = heapFloat[(r4+10)];
        r12 = heap32[(r12+2)];
        f4 = f14*f4;
        f6 = f15*f6;
        f17 = f17+f18;
        f18 = f16*f12;
        r13 = sp + -544;
        f17 = f17+f18;
        f4 = f4+f6;
        f6 = f11*f12;
        f10 = f0+f10;
        f4 = f4+f6;
        r14 = r13 >> 2;
        f6 = f2+f17;
        heapFloat[(fp+-136)] = f10;
        f4 = f3+f4;
        heapFloat[(r14+1)] = f6;
        heapFloat[(r14+2)] = f4;
        heap32[(r14+3)] = 0;
        f4 = heapFloat[(fp+-180)];
        f6 = heapFloat[(r8+1)];
        f10 = heapFloat[(r8+2)];
        f5 = f5*f4;
        f7 = f7*f6;
        f8 = f8*f4;
        f9 = f9*f6;
        f5 = f5+f7;
        f7 = f13*f10;
        f5 = f5+f7;
        f4 = f14*f4;
        f6 = f15*f6;
        f7 = f8+f9;
        f8 = f16*f10;
        r14 = sp + -528;
        f7 = f7+f8;
        f4 = f4+f6;
        f6 = f11*f10;
        f5 = f0+f5;
        f4 = f4+f6;
        r15 = r14 >> 2;
        f6 = f2+f7;
        heapFloat[(fp+-132)] = f5;
        f4 = f3+f4;
        heapFloat[(r15+1)] = f6;
        heapFloat[(r15+2)] = f4;
        heap32[(r15+3)] = 0;
        heap32[(g0)] = r_g0;
        heap32[(g0+1)] = r14;
        heap32[(g0+2)] = r13;
        heap32[(g0+3)] = r6;
        __FUNCTION_TABLE__[(r12)>>2](i7);
        f4 = -f1;
        heapFloat[(r7)] = f4;
        heapFloat[(r3)] = f4;
        r11 = heap32[(r1)];
        r11 = r11 >> 2;
        r11 = heap32[(r11+4)];
        heap32[(g0)] = r0;
        __FUNCTION_TABLE__[(r11)>>2](i7);
        r12 = r_g0 >> 2;
        f5 = heapFloat[(fp+-184)];
        f6 = heapFloat[(r4)];
        f7 = heapFloat[(r9+1)];
        f8 = heapFloat[(r4+1)];
        r12 = heap32[(r12)];
        f9 = heapFloat[(r4+4)];
        f10 = heapFloat[(r4+5)];
        f11 = f6*f5;
        f12 = f8*f7;
        f13 = heapFloat[(r9+2)];
        f14 = heapFloat[(r4+2)];
        r12 = r12 >> 2;
        f15 = heapFloat[(r4+8)];
        f16 = heapFloat[(r4+9)];
        f17 = heapFloat[(r4+6)];
        f18 = f9*f5;
        f19 = f10*f7;
        f11 = f11+f12;
        f12 = f14*f13;
        f11 = f11+f12;
        f12 = heapFloat[(r4+10)];
        r12 = heap32[(r12+2)];
        f5 = f15*f5;
        f7 = f16*f7;
        f18 = f18+f19;
        f19 = f17*f13;
        r13 = sp + -512;
        f18 = f18+f19;
        f5 = f5+f7;
        f7 = f12*f13;
        f11 = f0+f11;
        f5 = f5+f7;
        r14 = r13 >> 2;
        f7 = f2+f18;
        heapFloat[(fp+-128)] = f11;
        f5 = f3+f5;
        heapFloat[(r14+1)] = f7;
        heapFloat[(r14+2)] = f5;
        heap32[(r14+3)] = 0;
        f5 = heapFloat[(fp+-180)];
        f7 = heapFloat[(r8+1)];
        f11 = heapFloat[(r8+2)];
        f6 = f6*f5;
        f8 = f8*f7;
        f9 = f9*f5;
        f10 = f10*f7;
        f6 = f6+f8;
        f8 = f14*f11;
        f6 = f6+f8;
        f5 = f15*f5;
        f7 = f16*f7;
        f8 = f9+f10;
        f9 = f17*f11;
        r14 = sp + -496;
        f8 = f8+f9;
        f5 = f5+f7;
        f7 = f12*f11;
        f6 = f0+f6;
        f5 = f5+f7;
        r15 = r14 >> 2;
        f7 = f2+f8;
        heapFloat[(fp+-124)] = f6;
        f5 = f3+f5;
        heapFloat[(r15+1)] = f7;
        heapFloat[(r15+2)] = f5;
        heap32[(r15+3)] = 0;
        heap32[(g0)] = r_g0;
        heap32[(g0+1)] = r14;
        heap32[(g0+2)] = r13;
        heap32[(g0+3)] = r6;
        __FUNCTION_TABLE__[(r12)>>2](i7);
        heap32[(r7)] = 0;
        r2 = (r2 + r5)|0;
        r5 = (r10 + r5)|0;
        heap32[(r3)] = 0;
        r2 = r2 >> 2;
        heapFloat[(r2)] = f1;
        r3 = r5 >> 2;
        heapFloat[(r3)] = f1;
        r5 = heap32[(r1)];
        r5 = r5 >> 2;
        r5 = heap32[(r5+4)];
        heap32[(g0)] = r0;
        __FUNCTION_TABLE__[(r5)>>2](i7);
        r7 = r_g0 >> 2;
        f1 = heapFloat[(fp+-184)];
        f5 = heapFloat[(r4)];
        f6 = heapFloat[(r9+1)];
        f7 = heapFloat[(r4+1)];
        r7 = heap32[(r7)];
        f8 = heapFloat[(r4+4)];
        f9 = heapFloat[(r4+5)];
        f10 = f5*f1;
        f11 = f7*f6;
        f12 = heapFloat[(r9+2)];
        f13 = heapFloat[(r4+2)];
        r7 = r7 >> 2;
        f14 = heapFloat[(r4+8)];
        f15 = heapFloat[(r4+9)];
        f16 = heapFloat[(r4+6)];
        f17 = f8*f1;
        f18 = f9*f6;
        f10 = f10+f11;
        f11 = f13*f12;
        f10 = f10+f11;
        f11 = heapFloat[(r4+10)];
        r7 = heap32[(r7+2)];
        f1 = f14*f1;
        f6 = f15*f6;
        f17 = f17+f18;
        f18 = f16*f12;
        r10 = sp + -480;
        f17 = f17+f18;
        f1 = f1+f6;
        f6 = f11*f12;
        f10 = f0+f10;
        f1 = f1+f6;
        r11 = r10 >> 2;
        f6 = f2+f17;
        heapFloat[(fp+-120)] = f10;
        f1 = f3+f1;
        heapFloat[(r11+1)] = f6;
        heapFloat[(r11+2)] = f1;
        heap32[(r11+3)] = 0;
        f1 = heapFloat[(fp+-180)];
        f6 = heapFloat[(r8+1)];
        f10 = heapFloat[(r8+2)];
        f5 = f5*f1;
        f7 = f7*f6;
        f8 = f8*f1;
        f9 = f9*f6;
        f5 = f5+f7;
        f7 = f13*f10;
        f5 = f5+f7;
        f1 = f14*f1;
        f6 = f15*f6;
        f7 = f8+f9;
        f8 = f16*f10;
        r11 = sp + -464;
        f7 = f7+f8;
        f1 = f1+f6;
        f6 = f11*f10;
        f5 = f0+f5;
        f1 = f1+f6;
        r12 = r11 >> 2;
        f6 = f2+f7;
        heapFloat[(fp+-116)] = f5;
        f1 = f3+f1;
        heapFloat[(r12+1)] = f6;
        heapFloat[(r12+2)] = f1;
        heap32[(r12+3)] = 0;
        heap32[(g0)] = r_g0;
        heap32[(g0+1)] = r11;
        heap32[(g0+2)] = r10;
        heap32[(g0+3)] = r6;
        __FUNCTION_TABLE__[(r7)>>2](i7);
        heapFloat[(r2)] = f4;
        heapFloat[(r3)] = f4;
        r1 = heap32[(r1)];
        r1 = r1 >> 2;
        r1 = heap32[(r1+4)];
        heap32[(g0)] = r0;
        __FUNCTION_TABLE__[(r1)>>2](i7);
        r1 = r_g0 >> 2;
        f1 = heapFloat[(fp+-184)];
        f4 = heapFloat[(r4)];
        f5 = heapFloat[(r9+1)];
        f6 = heapFloat[(r4+1)];
        r1 = heap32[(r1)];
        f7 = heapFloat[(r4+4)];
        f8 = heapFloat[(r4+5)];
        f9 = f4*f1;
        f10 = f6*f5;
        f11 = heapFloat[(r9+2)];
        f12 = heapFloat[(r4+2)];
        r1 = r1 >> 2;
        f13 = heapFloat[(r4+8)];
        f14 = heapFloat[(r4+9)];
        f15 = heapFloat[(r4+6)];
        f16 = f7*f1;
        f17 = f8*f5;
        f9 = f9+f10;
        f10 = f12*f11;
        f9 = f9+f10;
        f10 = heapFloat[(r4+10)];
        r1 = heap32[(r1+2)];
        f1 = f13*f1;
        f5 = f14*f5;
        f16 = f16+f17;
        f17 = f15*f11;
        r2 = sp + -448;
        f16 = f16+f17;
        f1 = f1+f5;
        f5 = f10*f11;
        f9 = f0+f9;
        f1 = f1+f5;
        r3 = r2 >> 2;
        f5 = f2+f16;
        heapFloat[(fp+-112)] = f9;
        f1 = f3+f1;
        heapFloat[(r3+1)] = f5;
        heapFloat[(r3+2)] = f1;
        heap32[(r3+3)] = 0;
        f1 = heapFloat[(fp+-180)];
        f5 = heapFloat[(r8+1)];
        f9 = heapFloat[(r8+2)];
        f4 = f4*f1;
        f6 = f6*f5;
        f7 = f7*f1;
        f8 = f8*f5;
        f4 = f4+f6;
        f6 = f12*f9;
        f4 = f4+f6;
        f1 = f13*f1;
        f5 = f14*f5;
        f6 = f7+f8;
        f7 = f15*f9;
        r3 = sp + -432;
        f6 = f6+f7;
        f1 = f1+f5;
        f5 = f10*f9;
        f0 = f0+f4;
        f1 = f1+f5;
        r4 = r3 >> 2;
        f2 = f2+f6;
        heapFloat[(fp+-108)] = f0;
        f0 = f3+f1;
        heapFloat[(r4+1)] = f2;
        heapFloat[(r4+2)] = f0;
        heap32[(r4+3)] = 0;
        heap3