/***************************************************************************
 *     (c)2007-2014 Broadcom Corporation
 *
 *  This program is the proprietary software of Broadcom Corporation and/or its licensors,
 *  and may only be used, duplicated, modified or distributed pursuant to the terms and
 *  conditions of a separate, written license agreement executed between you and Broadcom
 *  (an "Authorized License").  Except as set forth in an Authorized License, Broadcom grants
 *  no license (express or implied), right to use, or waiver of any kind with respect to the
 *  Software, and Broadcom expressly reserves all rights in and to the Software and all
 *  intellectual property rights therein.  IF YOU HAVE NO AUTHORIZED LICENSE, THEN YOU
 *  HAVE NO RIGHT TO USE THIS SOFTWARE IN ANY WAY, AND SHOULD IMMEDIATELY
 *  NOTIFY BROADCOM AND DISCONTINUE ALL USE OF THE SOFTWARE.
 *
 *  Except as expressly set forth in the Authorized License,
 *
 *  1.     This program, including its structure, sequence and organization, constitutes the valuable trade
 *  secrets of Broadcom, and you shall use all reasonable efforts to protect the confidentiality thereof,
 *  and to use this information only in connection with your use of Broadcom integrated circuit products.
 *
 *  2.     TO THE MAXIMUM EXTENT PERMITTED BY LAW, THE SOFTWARE IS PROVIDED "AS IS"
 *  AND WITH ALL FAULTS AND BROADCOM MAKES NO PROMISES, REPRESENTATIONS OR
 *  WARRANTIES, EITHER EXPRESS, IMPLIED, STATUTORY, OR OTHERWISE, WITH RESPECT TO
 *  THE SOFTWARE.  BROADCOM SPECIFICALLY DISCLAIMS ANY AND ALL IMPLIED WARRANTIES
 *  OF TITLE, MERCHANTABILITY, NONINFRINGEMENT, FITNESS FOR A PARTICULAR PURPOSE,
 *  LACK OF VIRUSES, ACCURACY OR COMPLETENESS, QUIET ENJOYMENT, QUIET POSSESSION
 *  OR CORRESPONDENCE TO DESCRIPTION. YOU ASSUME THE ENTIRE RISK ARISING OUT OF
 *  USE OR PERFORMANCE OF THE SOFTWARE.
 *
 *  3.     TO THE MAXIMUM EXTENT PERMITTED BY LAW, IN NO EVENT SHALL BROADCOM OR ITS
 *  LICENSORS BE LIABLE FOR (i) CONSEQUENTIAL, INCIDENTAL, SPECIAL, INDIRECT, OR
 *  EXEMPLARY DAMAGES WHATSOEVER ARISING OUT OF OR IN ANY WAY RELATING TO YOUR
 *  USE OF OR INABILITY TO USE THE SOFTWARE EVEN IF BROADCOM HAS BEEN ADVISED OF
 *  THE POSSIBILITY OF SUCH DAMAGES; OR (ii) ANY AMOUNT IN EXCESS OF THE AMOUNT
 *  ACTUALLY PAID FOR THE SOFTWARE ITSELF OR U.S. $1, WHICHEVER IS GREATER. THESE
 *  LIMITATIONS SHALL APPLY NOTWITHSTANDING ANY FAILURE OF ESSENTIAL PURPOSE OF
 *  ANY LIMITED REMEDY.
 *
 * ***************************************************************************/

#include <unistd.h>
#include <string.h>
#include <stdlib.h>

#include "bcmnexus.h"
#include "bcmnexus_screen.h"
#include "bcmnexus_layer.h"

#include "directfb.h"
#include "directfb_strings.h"

#include "core/screens_internal.h"
#include "core/system.h"

#include "fusion/shmalloc.h"
#include "misc/conf.h"


D_DEBUG_DOMAIN( bcmnexusScreen, "bcmNexus/Screen",   "Broadcom NEXUS Screen driver" );
D_DEBUG_DOMAIN( bcmnexusScreenX, "bcmNexus/ScreenX", "Broadcom NEXUS Screen Xtra debug" );

/* Strings for Screen are defined here */
const DirectFBScreenOutputConnectorsNames  ( dfb_screen_output_connectors_names    )
const DirectFBScreenOutputResolutionNames  ( dfb_screen_output_resolution_names    )
const DirectFBScreenEncoderTVStandardsNames( dfb_screen_encoder_tv_standards_names )
const DirectFBScreenEncoderScanModeNames   ( dfb_screen_encoder_scan_mode_names    )
const DirectFBScreenEncoderFrequencyNames  ( dfb_screen_encoder_frequency_names    )

static int
bcmScreenDataSize( void )
{
    return sizeof(bcmScreenData);
}

#if !BCMNEXUS_NSC_SUPPORT
static void
bcmVsyncCallback( void *context,
                  int   param )
{
    DFBBCMNEXUS *dfb_bcmnexus = (DFBBCMNEXUS*)context;

    pthread_mutex_t *pLock = &GET_DISPLAY_VSYNC_MUTEX(dfb_bcmnexus, param);
    pthread_cond_t  *pSem =  &GET_DISPLAY_VSYNC_SEM(dfb_bcmnexus, param);

    D_DEBUG_AT( bcmnexusScreenX, "  -> VSync[%d] callback handled\n", param );

    PTHREAD_ROBUST_MUTEX_LOCK( pLock );
    pthread_cond_signal( pSem );
    pthread_mutex_unlock( pLock );

}

static void
bcmInstallVSyncCallback( DFBBCMNEXUS                    *dfb_bcmnexus,
                         DFBScreenID                     screen,
                         DFB_Platform_P_DisplaySettings *pDisplayCfg )
{
    D_DEBUG_AT( bcmnexusScreen, "%s() screen %d\n", __FUNCTION__, screen );

    if (GET_DISPLAY_VSYNC_CB(dfb_bcmnexus, screen) == NULL)
    {
        SET_DISPLAY_VSYNC_CB(dfb_bcmnexus, bcmVsyncCallback, screen);
        pDisplayCfg->vsyncCallback.callback = GET_DISPLAY_VSYNC_CB(dfb_bcmnexus, screen);
        pDisplayCfg->vsyncCallback.context  = dfb_bcmnexus;
        pDisplayCfg->vsyncCallback.param    = screen;
    }
}
#endif

static DFBResult
bcmSetDisplayOutput( UNUSED_ DFBBCMNEXUS                  *dfb_bcmnexus,
                     UNUSED_ DFBScreenID                   screen,
                     UNUSED_ DFB_PlatformDisplayOutputType outputType,
                     UNUSED_ int                           outputIndex,
                     UNUSED_ bool                          enable )
{
    DFBResult               res = DFB_OK;

#if !BCMNEXUS_NSC_SUPPORT
    DFB_PlatformResult      err;
    DFB_PlatformNexusHandle display;

    D_DEBUG_AT( bcmnexusScreen, "%s() screen %d, output type=%d, output index=%d, enable=%d\n",
                __FUNCTION__, screen, outputType, outputIndex, enable );

    display = GET_DISPLAY_HND(dfb_bcmnexus, screen);

    if (enable)
        err = DFB_Platform_AddDisplayOutput(display, outputType, outputIndex);
    else
        err = DFB_Platform_RemoveDisplayOutput(display, outputType, outputIndex);

    res = (err == DFB_PLATFORM_OK) ? DFB_OK : DFB_FAILURE;

    D_DEBUG_AT( bcmnexusScreen, "%s() screen %d, output type=%d, output index=%d, enable=%d Exit[%d]\n",
                __FUNCTION__, screen, outputType, outputIndex, enable, res );

#endif
    return res;
}

static DFBResult
setVideoOutput( UNUSED_ DFBBCMNEXUS                   *dfb_bcmnexus,
                UNUSED_ bcmScreenData                 *sd,
                UNUSED_ int                            output,
                UNUSED_ DFB_PlatformVideoFormat        format,
                UNUSED_ DFBScreenEncoderPictureFraming framing,
                UNUSED_ DFBScreenEncoderConfigFlags    aspect_ratio,
                        bool                           setSettings )
{
    DFBResult                      res    = DFB_OK;
    DFBScreenID                    screen = sd->id;
    DFB_PlatformNexusHandle        display;
    DFB_Platform_P_DisplaySettings displaySettings;

    D_DEBUG_AT( bcmnexusScreen, "%s() screen %d, output=%d, format=%d\n", __FUNCTION__, screen, output, format );

    display = GET_DISPLAY_HND(dfb_bcmnexus, screen);

    DFB_Platform_P_GetDisplaySettings(display, &displaySettings);

#if NEXUS_NUM_HDMI_OUTPUTS && NEXUS_NUM_COMPONENT_OUTPUTS
    if (screen == PRIMARY_DISPLAY_ID)
    {
        DFB_PlatformNexusHandle        componentHandle;

        if (DFB_Platform_GetDisplayOutput(display, DFB_PlatformDisplayOutputType_eComponent, output, &componentHandle) == DFB_PLATFORM_OK)
        {
            /* Disconnect the component output is the video format cannot be handled by the component output... */
            if (componentHandle)
            {
                if (format >= DFB_PlatformVideoFormat_eVesa640x480p60hz)
            {
                res = bcmSetDisplayOutput(dfb_bcmnexus, screen, DFB_PlatformDisplayOutputType_eComponent, output, false);
                    sd->componentDisconnected = true;
                }
            }
            /* Connect the component output is the video format can be handled by the component output and the component should
               be re-connected... */
            else
            {
                if (format < DFB_PlatformVideoFormat_eVesa640x480p60hz && sd->componentDisconnected)
                {
                    res = bcmSetDisplayOutput(dfb_bcmnexus, screen, DFB_PlatformDisplayOutputType_eComponent, output, true);
                    sd->componentDisconnected = false;
                }
            }
        }
    }
#endif

    if (res == DFB_OK)
    {
#if !BCMNEXUS_NSC_SUPPORT
        bcmInstallVSyncCallback(dfb_bcmnexus, screen, &displaySettings);
#endif
        /* Update the video format here.
           TODO: Add support for setting the aspect ratio too */
        displaySettings.format = format;
        displaySettings.aspectRatio= aspect_ratio;

        switch (framing)
        {
            case DSEPF_STEREO_SIDE_BY_SIDE_HALF:
                displaySettings.stereoMode = DFB_PlatformStereoscopicMode_eSideBySideHalf;
                break;
            case DSEPF_STEREO_TOP_AND_BOTTOM:
                displaySettings.stereoMode = DFB_PlatformStereoscopicMode_eTopBottom;
                break;
            case DSEPF_STEREO_FRAME_PACKING:
                displaySettings.stereoMode = DFB_PlatformStereoscopicMode_eFramePacking;
                break;
            case DSEPF_STEREO_SIDE_BY_SIDE_FULL:
                displaySettings.stereoMode = DFB_PlatformStereoscopicMode_eSideBySideFull;
                break;
            case DSEPF_MONO:
            default:
                displaySettings.stereoMode = DFB_PlatformStereoscopicMode_e2d;
                break;
        }

        res = (DFB_Platform_P_SetDisplaySettings( display, &displaySettings, setSettings) == DFB_PLATFORM_OK) ? DFB_OK : DFB_FAILURE;

    }
    D_DEBUG_AT( bcmnexusScreen, "%s() screen %d, output=%d, format=%d Exit[%d]\n", __FUNCTION__, screen, output, format, res );

    return res;
}

static void
getDisplayOutputCfg( DFB_PlatformNexusHandle  displayHandle,
                     DFBScreenEncoderConfig  *config )
{
    DFB_PlatformNexusHandle outputHandle;

#if !BCMNEXUS_NSC_SUPPORT
    if (!displayHandle)
#endif
        return;

    config->out_connectors  = DSOC_UNKNOWN;

    if ((DFB_Platform_GetDisplayOutput(displayHandle, DFB_PlatformDisplayOutputType_eComposite, 0, &outputHandle) == DFB_PLATFORM_OK) &&
        (outputHandle != NULL))
    {
        config->out_connectors  |= DSOC_CVBS;
        config->flags           |= DSECONF_CONNECTORS;
    }
    if ((DFB_Platform_GetDisplayOutput(displayHandle, DFB_PlatformDisplayOutputType_eSvideo, 0, &outputHandle) == DFB_PLATFORM_OK) &&
        (outputHandle != NULL))
    {
        config->out_connectors  |= DSOC_YC;
        config->flags           |= DSECONF_CONNECTORS;
    }
    if ((DFB_Platform_GetDisplayOutput(displayHandle, DFB_PlatformDisplayOutputType_eScart, 0, &outputHandle) == DFB_PLATFORM_OK) &&
        (outputHandle != NULL))
    {
        config->out_connectors  |= DSOC_SCART;
        config->flags           |= DSECONF_CONNECTORS;
    }
    if ((DFB_Platform_GetDisplayOutput(displayHandle, DFB_PlatformDisplayOutputType_eScart, 1, &outputHandle) == DFB_PLATFORM_OK) &&
        (outputHandle != NULL))
    {
        config->out_connectors  |= DSOC_SCART2;
        config->flags           |= DSECONF_CONNECTORS;
    }
    if ((DFB_Platform_GetDisplayOutput(displayHandle, DFB_PlatformDisplayOutputType_eComponent, 0, &outputHandle) == DFB_PLATFORM_OK) &&
        (outputHandle != NULL))
    {
        config->out_connectors  |= DSOC_COMPONENT;
        config->flags           |= DSECONF_CONNECTORS;
    }
    if ((DFB_Platform_GetDisplayOutput(displayHandle, DFB_PlatformDisplayOutputType_eHdmi, 0, &outputHandle) == DFB_PLATFORM_OK) &&
        (outputHandle != NULL))
    {
        config->out_connectors  |= DSOC_HDMI;
        config->flags           |= DSECONF_CONNECTORS;
    }
}


static void
getDefaultEncoderCfg( unsigned int             displayId,
                      DFBScreenEncoderConfig  *config )
{
    D_ASSERT(config != NULL);

    DFBBCMNEXUS                 *pBrcmDfb = (DFBBCMNEXUS *) dfb_system_data();
    BCMNEXUS_Options            *brcm_config = &pBrcmDfb->options;

    DFB_PlatformNexusHandle      displayHandle = GET_DISPLAY_HND(pBrcmDfb,displayId);

    memset(config, 0, sizeof(DFBScreenEncoderConfig));
    config->flags = (DFBScreenEncoderConfigFlags)(DSECONF_TV_STANDARD | DSECONF_SCANMODE |
                                                  DSECONF_FREQUENCY | DSECONF_CONNECTORS |
                                                  DSECONF_RESOLUTION | DSECONF_FRAMING | DSECONF_ASPECT_RATIO);
    config->tv_standard     = DSETV_DIGITAL;
    config->framing         = DSEPF_MONO;
    config->aspect_ratio    = DFB_PlatformDisplayAspectRatio_eAuto;

    if (strcmp(brcm_config->res,"480i") == 0) {
        config->tv_standard     = DSETV_NTSC;
        config->resolution      = DSOR_720_480;
        config->frequency       = DSEF_29_97HZ;
        config->scanmode        = DSESM_INTERLACED;
    }
    else if (strcmp(brcm_config->res,"480p") == 0) {
        config->resolution      = DSOR_720_480;
        config->frequency       = DSEF_59_94HZ;
        config->scanmode        = DSESM_PROGRESSIVE;
    }
    else if (strcmp(brcm_config->res,"576i") == 0) {
        config->tv_standard     = DSETV_PAL;
        config->resolution      = DSOR_720_576;
        config->frequency       = DSEF_25HZ;
        config->scanmode        = DSESM_INTERLACED;
    }
    else if (strcmp(brcm_config->res,"576p") == 0) {
        config->resolution      = DSOR_720_576;
        config->frequency       = DSEF_50HZ;
        config->scanmode        = DSESM_PROGRESSIVE;
    }
    else if (strcmp(brcm_config->res,"720p") == 0 || strcmp(brcm_config->res,"720p60") == 0) {
        config->resolution      = DSOR_1280_720;
        config->frequency       = DSEF_60HZ;
        config->scanmode        = DSESM_PROGRESSIVE;
    }
    else if (strcmp(brcm_config->res,"720ps") == 0 || strcmp(brcm_config->res,"720p60s") == 0) {
        config->resolution      = DSOR_1280_720;
        config->frequency       = DSEF_60HZ;
        config->scanmode        = DSESM_PROGRESSIVE;
        config->framing         = DSEPF_STEREO_FRAME_PACKING;
    }
    else if (strcmp(brcm_config->res,"720p50") == 0) {
        config->resolution      = DSOR_1280_720;
        config->frequency       = DSEF_50HZ;
        config->scanmode        = DSESM_PROGRESSIVE;
    }
    else if (strcmp(brcm_config->res,"720p50s") == 0) {
        config->resolution      = DSOR_1280_720;
        config->frequency       = DSEF_50HZ;
        config->scanmode        = DSESM_PROGRESSIVE;
        config->framing         = DSEPF_STEREO_FRAME_PACKING;
    }
    else if (strcmp(brcm_config->res,"1080i") == 0 || strcmp(brcm_config->res,"1080i30") == 0) {
        config->resolution      = DSOR_1920_1080;
        config->frequency       = DSEF_30HZ;
        config->scanmode        = DSESM_INTERLACED;
    }
    else if(strcmp(brcm_config->res,"1080i25") == 0) {
        config->resolution      = DSOR_1920_1080;
        config->frequency       = DSEF_25HZ;
        config->scanmode        = DSESM_INTERLACED;
    }
    else if (strcmp(brcm_config->res,"1080p") == 0 || strcmp(brcm_config->res,"1080p60") == 0) {
        config->resolution      = DSOR_1920_1080;
        config->frequency       = DSEF_60HZ;
        config->scanmode        = DSESM_PROGRESSIVE;
    }
    else if (strcmp(brcm_config->res,"1080p50") == 0) {
        config->resolution      = DSOR_1920_1080;
        config->frequency       = DSEF_50HZ;
        config->scanmode        = DSESM_PROGRESSIVE;
    }
    else if (strcmp(brcm_config->res,"1080p24") == 0) {
        config->resolution      = DSOR_1920_1080;
        config->frequency       = DSEF_24HZ;
        config->scanmode        = DSESM_PROGRESSIVE;
    }
    else if (strcmp(brcm_config->res,"1080p24s") == 0) {
        config->resolution      = DSOR_1920_1080;
        config->frequency       = DSEF_24HZ;
        config->scanmode        = DSESM_PROGRESSIVE;
        config->framing         = DSEPF_STEREO_FRAME_PACKING;
    }
    else if (strcmp(brcm_config->res,"1080p25") == 0) {
        config->resolution      = DSOR_1920_1080;
        config->frequency       = DSEF_25HZ;
        config->scanmode        = DSESM_PROGRESSIVE;
    }
    else if (strcmp(brcm_config->res,"1080p30") == 0) {
        config->resolution      = DSOR_1920_1080;
        config->frequency       = DSEF_30HZ;
        config->scanmode        = DSESM_PROGRESSIVE;
    }
#if BCMNEXUS_HAS_4K_DISPLAY_SUPPORT
    else if (strcmp(brcm_config->res,"2160p30") == 0) {
        config->resolution      = DSOR_3840_2160;
        config->frequency       = DSEF_30HZ;
        config->scanmode        = DSESM_PROGRESSIVE;
    }
    else if (strcmp(brcm_config->res,"2160p25") == 0) {
        config->resolution      = DSOR_3840_2160;
        config->frequency       = DSEF_25HZ;
        config->scanmode        = DSESM_PROGRESSIVE;
    }
    else if (strcmp(brcm_config->res,"2160p24") == 0) {
        config->resolution      = DSOR_3840_2160;
        config->frequency       = DSEF_24HZ;
        config->scanmode        = DSESM_PROGRESSIVE;
    }
#endif
    else if (strcmp(brcm_config->res,"VGA") == 0) {
        config->resolution      = DSOR_640_480;
        config->frequency       = DSEF_60HZ;
        config->scanmode        = DSESM_PROGRESSIVE;
    }
    else if (strcmp(brcm_config->res,"SVGA") == 0) {
        config->resolution      = DSOR_800_600;
        config->frequency       = DSEF_60HZ;
        config->scanmode        = DSESM_PROGRESSIVE;
    }
    else if (strcmp(brcm_config->res,"XGA") == 0) {
        config->resolution      = DSOR_1024_768;
        config->frequency       = DSEF_60HZ;
        config->scanmode        = DSESM_PROGRESSIVE;
    }
    else {
        /* Default is 720p/60Hz */
        if (displayId == 0)
        {
            config->resolution      = DSOR_1280_720;
            config->frequency       = DSEF_60HZ;
            config->scanmode        = DSESM_PROGRESSIVE;
        }
        else
        {
            config->tv_standard     = DSETV_NTSC;
            config->resolution      = DSOR_720_480;
            config->frequency       = DSEF_29_97HZ;
            config->scanmode        = DSESM_INTERLACED;
        }
    }

    getDisplayOutputCfg(displayHandle, config);
}

static void
getEncoderCfgFromDFBDisplayCfg( DFB_PlatformNexusHandle             displayHandle,
                                const DFB_PlatformDisplaySettings  *displayCfg,
                                DFBScreenEncoderConfig             *config )
{
    D_ASSERT(displayCfg != NULL);
    D_ASSERT(config != NULL);

    memset(config, 0, sizeof(DFBScreenEncoderConfig));
    config->flags = (DFBScreenEncoderConfigFlags)(DSECONF_TV_STANDARD | DSECONF_SCANMODE |
                                                  DSECONF_FREQUENCY | DSECONF_RESOLUTION |
                                                  DSECONF_FRAMING);
    config->framing = DSEPF_MONO;

    switch (displayCfg->format)
    {
    case DFB_PlatformVideoFormat_eNtsc:
        config->tv_standard = DSETV_NTSC;
        config->resolution  = DSOR_720_480;
        config->frequency   = DSEF_29_97HZ;
        config->scanmode    = DSESM_INTERLACED;
        break;

    case DFB_PlatformVideoFormat_eNtscJapan:
        config->tv_standard = DSETV_NTSC_M_JPN;
        config->resolution  = DSOR_720_480;
        config->frequency   = DSEF_29_97HZ;
        config->scanmode    = DSESM_INTERLACED;
        break;

    case DFB_PlatformVideoFormat_ePalG:
    case DFB_PlatformVideoFormat_ePalH:
    case DFB_PlatformVideoFormat_ePalI:
    case DFB_PlatformVideoFormat_ePalK:
        config->tv_standard = DSETV_PAL;
        config->resolution  = DSOR_720_576;
        config->frequency   = DSEF_25HZ;
        config->scanmode    = DSESM_INTERLACED;
        break;

    case DFB_PlatformVideoFormat_ePalM:
        config->tv_standard = DSETV_PAL_M;
        config->resolution  = DSOR_720_480;
        config->frequency   = DSEF_30HZ;
        break;

    case DFB_PlatformVideoFormat_ePalN:
        config->tv_standard = DSETV_PAL_N;
        config->resolution  = DSOR_720_576;
        config->frequency   = DSEF_25HZ;
        break;

    case DFB_PlatformVideoFormat_ePalNc:
        config->tv_standard = DSETV_PAL_NC;
        config->resolution  = DSOR_720_576;
        config->frequency   = DSEF_25HZ;
        break;

    case DFB_PlatformVideoFormat_eSecamL:
        config->tv_standard = DSETV_SECAM;
        config->resolution  = DSOR_720_576;
        config->frequency   = DSEF_25HZ;
        break;

    case DFB_PlatformVideoFormat_e480p:
        config->tv_standard = DSETV_DIGITAL;
        config->resolution  = DSOR_720_480;
        config->frequency   = DSEF_60HZ;
        config->scanmode    = DSESM_PROGRESSIVE;
        break;

    case DFB_PlatformVideoFormat_e576p:
        config->tv_standard = DSETV_DIGITAL;
        config->resolution  = DSOR_720_576;
        config->frequency   = DSEF_50HZ;
        config->scanmode    = DSESM_PROGRESSIVE;
        break;

    case DFB_PlatformVideoFormat_e720p24hz:
        config->tv_standard = DSETV_DIGITAL;
        config->resolution  = DSOR_1280_720;
        config->frequency   = DSEF_24HZ;
        config->scanmode    = DSESM_PROGRESSIVE;
        break;

    case DFB_PlatformVideoFormat_e720p25hz:
        config->tv_standard = DSETV_DIGITAL;
        config->resolution  = DSOR_1280_720;
        config->frequency   = DSEF_25HZ;
        config->scanmode    = DSESM_PROGRESSIVE;
        break;

    case DFB_PlatformVideoFormat_e720p30hz:
        config->tv_standard = DSETV_DIGITAL;
        config->resolution  = DSOR_1280_720;
        config->frequency   = DSEF_30HZ;
        config->scanmode    = DSESM_PROGRESSIVE;
        break;

    case DFB_PlatformVideoFormat_e720p50hz:
        config->tv_standard = DSETV_DIGITAL;
        config->resolution  = DSOR_1280_720;
        config->frequency   = DSEF_50HZ;
        config->scanmode    = DSESM_PROGRESSIVE;
        break;

    case DFB_PlatformVideoFormat_e720p50hz_3DOU_AS:
        config->tv_standard = DSETV_DIGITAL;
        config->resolution  = DSOR_1280_720;
        config->frequency   = DSEF_50HZ;
        config->scanmode    = DSESM_PROGRESSIVE;
        config->framing     = DSEPF_STEREO_FRAME_PACKING;
        break;

    case DFB_PlatformVideoFormat_e720p:
        config->tv_standard = DSETV_DIGITAL;
        config->resolution  = DSOR_1280_720;
        config->frequency   = DSEF_60HZ;
        config->scanmode    = DSESM_PROGRESSIVE;
        break;

    case DFB_PlatformVideoFormat_e720p_3DOU_AS:
        config->tv_standard = DSETV_DIGITAL;
        config->resolution  = DSOR_1280_720;
        config->frequency   = DSEF_60HZ;
        config->scanmode    = DSESM_PROGRESSIVE;
        config->framing     = DSEPF_STEREO_FRAME_PACKING;
        break;

    case DFB_PlatformVideoFormat_e1080i50hz:
        config->tv_standard = DSETV_DIGITAL;
        config->resolution  = DSOR_1920_1080;
        config->frequency   = DSEF_50HZ;
        config->scanmode    = DSESM_INTERLACED;
        break;

    case DFB_PlatformVideoFormat_e1080i:
        config->tv_standard = DSETV_DIGITAL;
        config->resolution  = DSOR_1920_1080;
        config->frequency   = DSEF_60HZ;
        config->scanmode    = DSESM_INTERLACED;
        break;

    case DFB_PlatformVideoFormat_e1080p24hz:
        config->tv_standard = DSETV_DIGITAL;
        config->resolution  = DSOR_1920_1080;
        config->frequency   = DSEF_24HZ;
        config->scanmode    = DSESM_PROGRESSIVE;
        break;

    case DFB_PlatformVideoFormat_e1080p24hz_3DOU_AS:
        config->tv_standard = DSETV_DIGITAL;
        config->resolution  = DSOR_1920_1080;
        config->frequency   = DSEF_24HZ;
        config->scanmode    = DSESM_PROGRESSIVE;
        config->framing     = DSEPF_STEREO_FRAME_PACKING;
        break;
/* removed ?? CJT
    case DFB_PlatformVideoFormat_1080p25hz:
        config->tv_standard = DSETV_DIGITAL;
        config->resolution  = DSOR_1920_1080;
        config->frequency   = DSEF_25HZ;
        config->scanmode    = DSESM_PROGRESSIVE;
        break;
*/
    case DFB_PlatformVideoFormat_e1080p30hz:
        config->tv_standard = DSETV_DIGITAL;
        config->resolution  = DSOR_1920_1080;
        config->frequency   = DSEF_30HZ;
        config->scanmode    = DSESM_PROGRESSIVE;
        break;

    case DFB_PlatformVideoFormat_e1080p50hz:
        config->tv_standard = DSETV_DIGITAL;
        config->resolution  = DSOR_1920_1080;
        config->frequency   = DSEF_50HZ;
        config->scanmode    = DSESM_PROGRESSIVE;
        break;

    case DFB_PlatformVideoFormat_e1080p60hz:
        config->tv_standard = DSETV_DIGITAL;
        config->resolution  = DSOR_1920_1080;
        config->frequency   = DSEF_60HZ;
        config->scanmode    = DSESM_PROGRESSIVE;
        break;

    case DFB_PlatformVideoFormat_eVesa640x480p60hz:
        config->tv_standard = DSETV_DIGITAL;
        config->resolution  = DSOR_640_480;
        config->frequency   = DSEF_60HZ;
        config->scanmode    = DSESM_PROGRESSIVE;
        break;

    case DFB_PlatformVideoFormat_eVesa800x600p60hz:
        config->tv_standard = DSETV_DIGITAL;
        config->resolution  = DSOR_800_600;
        config->frequency   = DSEF_60HZ;
        config->scanmode    = DSESM_PROGRESSIVE;
        break;

    case DFB_PlatformVideoFormat_eVesa1024x768p60hz:
        config->tv_standard = DSETV_DIGITAL;
        config->resolution  = DSOR_1024_768;
        config->frequency   = DSEF_60HZ;
        config->scanmode    = DSESM_PROGRESSIVE;
        break;
#if BCMNEXUS_HAS_4K_DISPLAY_SUPPORT
    case DFB_PlatformVideoFormat_e3840x2160p24hz:
        config->tv_standard = DSETV_DIGITAL;
        config->resolution  = DSOR_3840_2160;
        config->frequency   = DSEF_24HZ;
        config->scanmode    = DSESM_PROGRESSIVE;
        break;

    case DFB_PlatformVideoFormat_e3840x2160p25hz:
        config->tv_standard = DSETV_DIGITAL;
        config->resolution  = DSOR_3840_2160;
        config->frequency   = DSEF_25HZ;
        config->scanmode    = DSESM_PROGRESSIVE;
        break;

    case DFB_PlatformVideoFormat_e3840x2160p30hz:
        config->tv_standard = DSETV_DIGITAL;
        config->resolution  = DSOR_3840_2160;
        config->frequency   = DSEF_30HZ;
        config->scanmode    = DSESM_PROGRESSIVE;
        break;

    case DFB_PlatformVideoFormat_e3840x2160p50hz:
        config->tv_standard = DSETV_DIGITAL;
        config->resolution  = DSOR_3840_2160;
        config->frequency   = DSEF_50HZ;
        config->scanmode    = DSESM_PROGRESSIVE;
        break;

    case DFB_PlatformVideoFormat_e3840x2160p60hz:
        config->tv_standard = DSETV_DIGITAL;
        config->resolution  = DSOR_3840_2160;
        config->frequency   = DSEF_60HZ;
        config->scanmode    = DSESM_PROGRESSIVE;
        break;

    case DFB_PlatformVideoFormat_e4096x2160p24hz:
        config->tv_standard = DSETV_DIGITAL;
        config->resolution  = DSOR_4096_2160;
        config->frequency   = DSEF_24HZ;
        config->scanmode    = DSESM_PROGRESSIVE;
        break;

    case DFB_PlatformVideoFormat_e4096x2160p25hz:
        config->tv_standard = DSETV_DIGITAL;
        config->resolution  = DSOR_4096_2160;
        config->frequency   = DSEF_25HZ;
        config->scanmode    = DSESM_PROGRESSIVE;
        break;

    case DFB_PlatformVideoFormat_e4096x2160p30hz:
        config->tv_standard = DSETV_DIGITAL;
        config->resolution  = DSOR_4096_2160;
        config->frequency   = DSEF_30HZ;
        config->scanmode    = DSESM_PROGRESSIVE;
        break;

    case DFB_PlatformVideoFormat_e4096x2160p50hz:
        config->tv_standard = DSETV_DIGITAL;
        config->resolution  = DSOR_4096_2160;
        config->frequency   = DSEF_50HZ;
        config->scanmode    = DSESM_PROGRESSIVE;
        break;

    case DFB_PlatformVideoFormat_e4096x2160p60hz:
        config->tv_standard = DSETV_DIGITAL;
        config->resolution  = DSOR_4096_2160;
        config->frequency   = DSEF_60HZ;
        config->scanmode    = DSESM_PROGRESSIVE;
        break;

#endif
    default:
        D_ERROR("%s() Unknown video_format enumeration, %d.\n", __FUNCTION__, displayCfg->format);
        config->tv_standard     = DSETV_UNKNOWN;
        config->resolution      = DSOR_UNKNOWN;
        config->frequency       = DSEF_UNKNOWN;
        config->scanmode        = DSESM_UNKNOWN;
        config->flags           = DSECONF_NONE;
        config->framing         = DSEPF_UNKNOWN;
        break;
    }

    getDisplayOutputCfg(displayHandle, config);
}

static DFBResult
getDFBDisplayCfgFromEncoderCfg( const DFBScreenEncoderConfig   *config,
                                DFB_PlatformDisplaySettings    *displayCfg,
                                DFBScreenEncoderConfigFlags    *failFlags )
{
    D_ASSERT(displayCfg != NULL);
    D_ASSERT(config != NULL);

    memset(displayCfg, 0, sizeof(*displayCfg));

    if (config->flags & DSECONF_TV_STANDARD)
    {
        switch (config->tv_standard)
        {
            case DSETV_NTSC:
                displayCfg->format = DFB_PlatformVideoFormat_eNtsc;
                displayCfg->aspectRatio = DFB_PlatformDisplayAspectRatio_e4x3;
                break;

            case DSETV_NTSC_M_JPN:
                displayCfg->format = DFB_PlatformVideoFormat_eNtscJapan;
                displayCfg->aspectRatio = DFB_PlatformDisplayAspectRatio_e4x3;
                break;

            case DSETV_PAL:
                displayCfg->format = DFB_PlatformVideoFormat_ePalG;
                displayCfg->aspectRatio = DFB_PlatformDisplayAspectRatio_e4x3;
                break;

            case DSETV_PAL_M:
                displayCfg->format = DFB_PlatformVideoFormat_ePalM;
                displayCfg->aspectRatio = DFB_PlatformDisplayAspectRatio_e4x3;
                break;

            case DSETV_PAL_N:
                displayCfg->format = DFB_PlatformVideoFormat_ePalN;
                displayCfg->aspectRatio = DFB_PlatformDisplayAspectRatio_e4x3;
                break;

            case DSETV_PAL_NC:
                displayCfg->format = DFB_PlatformVideoFormat_ePalNc;
                displayCfg->aspectRatio = DFB_PlatformDisplayAspectRatio_e4x3;
                break;

            case DSETV_SECAM:
                displayCfg->format = DFB_PlatformVideoFormat_eSecamL;
                displayCfg->aspectRatio = DFB_PlatformDisplayAspectRatio_e4x3;
                break;

            case DSETV_DIGITAL:
                if (!(config->flags & DSECONF_RESOLUTION) ||
                    !(config->flags & DSECONF_FREQUENCY) ||
                    !(config->flags & DSECONF_SCANMODE))
                {
                    D_ERROR("%s() Must specify resolution, frequency, and scanmode for DSETV_DIGITAL "
                            "TV standard\n", __FUNCTION__);
                    *failFlags = (DFBScreenEncoderConfigFlags)(DSECONF_RESOLUTION | DSECONF_FREQUENCY |
                                                               DSECONF_SCANMODE | DSECONF_TV_STANDARD);
                    return DFB_INVARG;
                }

                displayCfg->format = -1;
                displayCfg->aspectRatio = DFB_PlatformDisplayAspectRatio_e16x9;

                switch (config->frequency)
                {
                    case DSEF_23_976HZ:
                    case DSEF_24HZ:
                        if (config->scanmode == DSESM_PROGRESSIVE)
                        {
                            if (config->resolution == DSOR_1280_720)
                                displayCfg->format = DFB_PlatformVideoFormat_e720p24hz;
                            else if (config->resolution == DSOR_1920_1080)
                            {
                                if (config->framing == DSEPF_STEREO_FRAME_PACKING)
                                    displayCfg->format = DFB_PlatformVideoFormat_e1080p24hz_3DOU_AS;
                                else
                                    displayCfg->format = DFB_PlatformVideoFormat_e1080p24hz;
                            }
#if BCMNEXUS_HAS_4K_DISPLAY_SUPPORT
                            else if (config->resolution == DSOR_3840_2160)
                                displayCfg->format = DFB_PlatformVideoFormat_e3840x2160p24hz;
                            else if (config->resolution == DSOR_4096_2160)
                                displayCfg->format = DFB_PlatformVideoFormat_e4096x2160p24hz;
#endif
                        }
                        break;

                    case DSEF_25HZ:
                        if (config->scanmode == DSESM_PROGRESSIVE)
                        {
                            if (config->resolution == DSOR_1280_720)
                                displayCfg->format = DFB_PlatformVideoFormat_e720p25hz;
                            else if (config->resolution == DSOR_1920_1080)
                                displayCfg->format = DFB_PlatformVideoFormat_e1080p25hz;
#if BCMNEXUS_HAS_4K_DISPLAY_SUPPORT
                            else if (config->resolution == DSOR_3840_2160)
                                displayCfg->format = DFB_PlatformVideoFormat_e3840x2160p25hz;
                            else if(config->resolution == DSOR_4096_2160)
                                displayCfg->format = DFB_PlatformVideoFormat_e4096x2160p25hz;
#endif
                        }
                        else
                        {
                            if (config->resolution == DSOR_720_576)
                                displayCfg->format = DFB_PlatformVideoFormat_ePalG;
                            else if (config->resolution == DSOR_1920_1080)
                                displayCfg->format = DFB_PlatformVideoFormat_e1080i50hz;
                        }
                        break;

                    case DSEF_29_97HZ:
                        if (config->scanmode == DSESM_PROGRESSIVE)
                        {
                            if (config->resolution == DSOR_1280_720)
                                displayCfg->format = DFB_PlatformVideoFormat_e720p30hz;
                            else if (config->resolution == DSOR_1920_1080)
                                displayCfg->format = DFB_PlatformVideoFormat_e1080p30hz;
                        }
                        else
                        {
                            if (config->resolution == DSOR_720_480)
                                displayCfg->format = DFB_PlatformVideoFormat_eNtsc;
                            else if (config->resolution == DSOR_1920_1080)
                                displayCfg->format = DFB_PlatformVideoFormat_e1080i;
                        }
                        break;

                    case DSEF_30HZ:
                        if (config->scanmode == DSESM_PROGRESSIVE)
                        {
                            if (config->resolution == DSOR_1280_720)
                                displayCfg->format = DFB_PlatformVideoFormat_e720p30hz;
                            else if (config->resolution == DSOR_1920_1080)
                                displayCfg->format = DFB_PlatformVideoFormat_e1080p30hz;
#if BCMNEXUS_HAS_4K_DISPLAY_SUPPORT
                            else if (config->resolution == DSOR_3840_2160)
                                displayCfg->format = DFB_PlatformVideoFormat_e3840x2160p30hz;
                            else if(config->resolution == DSOR_4096_2160)
                                displayCfg->format = DFB_PlatformVideoFormat_e4096x2160p30hz;
#endif
                        }
                        else
                        {
                            if (config->resolution == DSOR_1920_1080)
                                displayCfg->format = DFB_PlatformVideoFormat_e1080i;
                        }
                        break;

                    case DSEF_50HZ:
                        if (config->scanmode == DSESM_PROGRESSIVE)
                        {
                            if (config->resolution == DSOR_720_576)
                                displayCfg->format = DFB_PlatformVideoFormat_e576p;
                            else if (config->resolution == DSOR_1280_720)
                                if (config->framing == DSEPF_STEREO_FRAME_PACKING)
                                    displayCfg->format = DFB_PlatformVideoFormat_e720p50hz_3DOU_AS;
                                else
                                    displayCfg->format = DFB_PlatformVideoFormat_e720p50hz;
                            else if (config->resolution == DSOR_1920_1080)
                                displayCfg->format = DFB_PlatformVideoFormat_e1080p50hz;
#if BCMNEXUS_HAS_4K_DISPLAY_SUPPORT
                            else if (config->resolution == DSOR_3840_2160)
                                displayCfg->format = DFB_PlatformVideoFormat_e3840x2160p50hz;
                            else if(config->resolution == DSOR_4096_2160)
                                displayCfg->format = DFB_PlatformVideoFormat_e4096x2160p50hz;
#endif
                        }
                        else
                        {
                            if (config->resolution == DSOR_1920_1080)
                                displayCfg->format = DFB_PlatformVideoFormat_e1080i50hz;
                            else if (config->resolution == DSOR_720_576)
                                displayCfg->format = DFB_PlatformVideoFormat_ePalI;
                        }
                        break;

                    case DSEF_59_94HZ:
                        if (config->scanmode == DSESM_PROGRESSIVE)
                        {
                            if (config->resolution == DSOR_720_480)
                                displayCfg->format = DFB_PlatformVideoFormat_e480p;
                            else if (config->resolution == DSOR_1280_720)
                                if (config->framing == DSEPF_STEREO_FRAME_PACKING)
                                    displayCfg->format = DFB_PlatformVideoFormat_e720p_3DOU_AS;
                                else
                                    displayCfg->format = DFB_PlatformVideoFormat_e720p;
                            else if (config->resolution == DSOR_1920_1080)
                                displayCfg->format = DFB_PlatformVideoFormat_e1080p60hz;
                        }
                        break;

                    case DSEF_60HZ:
                        if (config->scanmode == DSESM_PROGRESSIVE)
                        {
                            if (config->resolution == DSOR_640_480)
                                displayCfg->format = DFB_PlatformVideoFormat_eVesa640x480p60hz;
                            else if (config->resolution == DSOR_720_480)
                                displayCfg->format = DFB_PlatformVideoFormat_e480p;
                            else if (config->resolution == DSOR_800_600)
                                displayCfg->format = DFB_PlatformVideoFormat_eVesa800x600p60hz;
                            else if (config->resolution == DSOR_1280_720)
                            {
                                if (config->framing == DSEPF_STEREO_FRAME_PACKING)
                                    displayCfg->format = DFB_PlatformVideoFormat_e720p_3DOU_AS;
                                else
                                    displayCfg->format = DFB_PlatformVideoFormat_e720p;
                            }
                            else if (config->resolution == DSOR_1024_768)
                                displayCfg->format = DFB_PlatformVideoFormat_eVesa1024x768p60hz;
                            else if (config->resolution == DSOR_1920_1080)
                                displayCfg->format = DFB_PlatformVideoFormat_e1080p60hz;
#if BCMNEXUS_HAS_4K_DISPLAY_SUPPORT
                            else if (config->resolution == DSOR_3840_2160)
                                displayCfg->format = DFB_PlatformVideoFormat_e3840x2160p60hz;
                            else if(config->resolution == DSOR_4096_2160)
                                displayCfg->format = DFB_PlatformVideoFormat_e4096x2160p60hz;
#endif
                        }
                        else
                        {
                            if (config->resolution == DSOR_1920_1080)
                                displayCfg->format = DFB_PlatformVideoFormat_e1080i;
                            else if (config->resolution == DSOR_720_480)
                                displayCfg->format = DFB_PlatformVideoFormat_eNtsc;
                        }
                        break;

                    default:
                        displayCfg->format = DFB_PlatformVideoFormat_eUnknown;
                        break;
                }

                if (displayCfg->format == DFB_PlatformVideoFormat_eUnknown)
                {
                    D_ERROR("%s() Unsupported resolution/frequency/scanmode specified %d/%d/%d.\n",
                            __FUNCTION__, config->resolution, config->frequency, config->scanmode);
                    *failFlags = (DFBScreenEncoderConfigFlags)(DSECONF_RESOLUTION | DSECONF_FREQUENCY |
                                                               DSECONF_SCANMODE | DSECONF_TV_STANDARD);
                    return DFB_UNSUPPORTED;
                }
                break;

            default:
                D_ERROR("%s() Unsupported TV standard specified %d.\n", __FUNCTION__, config->tv_standard);
                *failFlags = DSECONF_TV_STANDARD;
                return DFB_UNSUPPORTED;
        }
    }
    else
    {
        if (config->flags & DSECONF_ASPECT_RATIO)
        {
            displayCfg->aspectRatio = (DFB_PlatformDisplayAspectRatio)(config->aspect_ratio);
        }
    }

    if (config->flags & DSECONF_CONNECTORS)
    {
        if (config->out_connectors & ~(DSOC_YC | DSOC_CVBS | DSOC_COMPONENT | DSOC_HDMI | DSOC_SCART | DSOC_SCART2))
        {
            D_ERROR("%s() Unsupported output connector.\n", __FUNCTION__);
            *failFlags = DSECONF_CONNECTORS;
            return DFB_FAILURE;
        }

        if ((config->out_connectors & DSOC_CVBS) && (config->resolution > DSOR_800_600))
        {
            D_ERROR("%s() Can't connect such a high resolution to CVBS, res %x\n", __FUNCTION__, config->resolution);
            *failFlags = DSECONF_CONNECTORS;
            return DFB_UNSUPPORTED;
        }
    }

    return DFB_OK;
}

static DFBResult
primaryInitScreen( UNUSED_ CoreScreen         *screen,
                   UNUSED_ CoreGraphicsDevice *device,
                   void                       *driver_data,
                   void                       *screen_data,
                   DFBScreenDescription       *description )
{
    DFBBCMNEXUS                *dfb_bcmnexus = driver_data;
    bcmScreenData              *sd           = screen_data;
    DFB_PlatformDisplaySettings displaySettings;
    DFBScreenEncoderConfig      encoderCfg;
    DFBScreenEncoderConfigFlags failFlags;

    D_DEBUG_AT( bcmnexusScreen, "%s()\n", __FUNCTION__ );

    D_ASSERT( dfb_bcmnexus != NULL );

    /* Screen ID = 0 => Primary output ON */
    sd->id = PRIMARY_DISPLAY_ID;
    sd->powerMode = DSPM_ON;

    /* Set the screen capabilities. */
    description->caps       = DSCCAPS_ENCODERS | DSCCAPS_VSYNC | DSCCAPS_MIXERS;
    description->encoders   = 1;
    description->mixers  = 1;

    /* Set the screen name. */
    snprintf( description->name, DFB_SCREEN_DESC_NAME_LENGTH, "BCMNEXUS Primary Screen" );

    D_DEBUG_AT( bcmnexusScreen, "%s(): display0 handle=%p\n", __FUNCTION__, (void *)GET_DISPLAY_HND(dfb_bcmnexus, PRIMARY_DISPLAY_ID));

    getDefaultEncoderCfg(sd->id, &encoderCfg);
    getDFBDisplayCfgFromEncoderCfg(&encoderCfg, &displaySettings, &failFlags);
#if BCMNEXUS_NSC_SUPPORT
    sd->fixedResolution = true;
#else
    sd->fixedResolution = false;
#endif

    D_DEBUG_AT( bcmnexusScreen, "%s() screen %d, format=%d Exit\n", __FUNCTION__, sd->id, displaySettings.format );

    return DFB_OK;
}

static DFBResult
bcmInitEncoder ( UNUSED_ CoreScreen                   *screen,
                         void                         *driver_data,
                         void                         *screen_data,
                         int                           encoder,
                         DFBScreenEncoderDescription  *description,
                         DFBScreenEncoderConfig       *config )
{
    DFBResult                       res = DFB_OK;
    DFBBCMNEXUS                    *dfb_bcmnexus = driver_data;
    bcmScreenData                  *sd  = screen_data;
    DFB_PlatformNexusHandle         display;
    DFB_PlatformDisplaySettings     displayCfg;
    DFBBCMNEXUS                 *pBrcmDfb = (DFBBCMNEXUS *) dfb_system_data();
    BCMNEXUS_Options            *brcm_config = &pBrcmDfb->options;

    D_DEBUG_AT( bcmnexusScreen, "%s() screen %d, encoder=%d, flags=0x%08x\n", __FUNCTION__, sd->id, encoder, config->flags );

    if (encoder > 0)
    {
        D_ERROR("%s() Unsupported encoder %d.\n", __FUNCTION__,encoder);
        return DFB_UNSUPPORTED;
    }

    /* Set the encoder capabilities. */
    description->caps           = (DFBScreenEncoderCapabilities)(DSECAPS_TV_STANDARDS | DSECAPS_SCANMODE   |
                                                                 DSECAPS_FREQUENCY    | DSECAPS_CONNECTORS |
                                                                 DSECAPS_RESOLUTION   | DSECAPS_FRAMING | DSECAPS_ASPECT_RATIO);
    description->type           = (DFBScreenEncoderType)(DSET_TV | DSET_DIGITAL);
    description->tv_standards   = (DFBScreenEncoderTVStandards)(DSETV_NTSC  | DSETV_NTSC_M_JPN | DSETV_PAL    |
                                                                DSETV_PAL_M | DSETV_PAL_N      | DSETV_PAL_NC |
                                                                DSETV_SECAM | DSETV_DIGITAL);

    description->all_connectors = (DFBScreenOutputConnectors)(
#if NEXUS_NUM_COMPOSITE_OUTPUTS
                                   DSOC_CVBS      |
#endif
#if NEXUS_NUM_COMPONENT_OUTPUTS
                                   DSOC_COMPONENT |
#endif
#if NEXUS_NUM_HDMI_OUTPUTS
                                   DSOC_HDMI      |
#endif
#if NEXUS_NUM_SCART_INPUTS
                                   DSOC_SCART     |
#endif
#if NEXUS_NUM_SCART_INPUTS > 1
                                   DSOC_SCART2    |
#endif
#if NEXUS_NUM_SVIDEO_OUTPUTS
                                   DSOC_YC        |
#endif
                                   DSOC_UNKNOWN);

    description->all_resolutions= (DFBScreenOutputResolution)(DSOR_640_480  | DSOR_720_480  | DSOR_720_576 | DSOR_800_600 |
                                                              DSOR_1024_768 | DSOR_1280_720 | DSOR_1920_1080);
#if BCMNEXUS_HAS_4K_DISPLAY_SUPPORT
    description->all_resolutions |= (DFBScreenOutputResolution)(DSOR_3840_2160 | DSOR_4096_2160);
#endif

    description->all_framing    = DSEPF_ALL;

    if (sd->id > 0)
    {
        /* Make the secondary display SD only, not allow HDMI to be connected and no stereoscopic support*/
        description->all_resolutions = (DFBScreenOutputResolution)(DSOR_720_480  | DSOR_720_576 );
        description->all_connectors &= ~DSOC_HDMI;
        description->all_framing     = DSEPF_MONO;
    }

    description->all_aspect_ratio= DFB_ASPECT_RATIO_eAuto | DFB_ASPECT_RATIO_e4x3 | DFB_ASPECT_RATIO_e16x9;
    snprintf( description->name, DFB_SCREEN_ENCODER_DESC_NAME_LENGTH, "BCMNEXUS encoder %d",sd->id);

    D_DEBUG_AT( bcmnexusScreen, "Encoder description - caps 0x%x - type 0x%x - standards 0x%x - res 0x%x - connectors 0x%x - framing 0x%x - aspect rations 0x%x\n",
               description->caps,
               description->type,
               description->tv_standards,
               description->all_resolutions,
               description->all_connectors,
               description->all_framing,
               description->all_aspect_ratio);

    /* Set configuration. */
    display = GET_DISPLAY_HND(dfb_bcmnexus, sd->id);
#if BCMNEXUS_NSC_SUPPORT
    if (0)
#else
    if (!display)
#endif
    {
        DFB_PlatformDisplayInitSettings displayInitSettings;
        int clientId = -1; /*Dummy NSC clientId in XS mode used only in Trellis Slave mode.*/
        D_DEBUG_AT( bcmnexusScreen, "%s() Using default encoder configuration\n", __FUNCTION__ );

        /* Display hasn't been configured yet, so specify default configuration. */
        DFB_Platform_P_Display_GetDefaultInitSettings(sd->id, &displayInitSettings);

        res = (DFB_Platform_P_Display_Init( sd->id, &displayInitSettings, &display, DFB_Platform_SlaveType_None, clientId) == DFB_PLATFORM_OK) ? DFB_OK : DFB_FAILURE;
        if (res != DFB_OK)
        {
            D_ERROR("%s{} Error opening primary display%d!\n", __FUNCTION__, sd->id);
            return DFB_FAILURE;
        }
        SET_DISPLAY_HND(dfb_bcmnexus, display, sd->id);

        /* Get the default format (or overide if specified). */
        getDefaultEncoderCfg(sd->id, config);
    }
    else if (strcmp(brcm_config->res, " "))
    {
        /* Get the default format (or overide if specified). */
        if (sd->id == 0)
        {
            D_DEBUG_AT( bcmnexusScreen, "%s() Using encoder configuration set externally by DirectFB \"res=%s\" option.\n", __FUNCTION__, brcm_config->res );
            getDefaultEncoderCfg(sd->id, config);
        }
        else
        {
            /*dfb_config->res doesn't reflect how the secondary display was setup.*/
            DFB_Platform_GetDisplaySettings(display, &displayCfg);

            D_DEBUG_AT( bcmnexusScreen, "%s() Using default configuration for screen %d (format=%d).\n", __FUNCTION__, sd->id, displayCfg.format );

            getEncoderCfgFromDFBDisplayCfg(display, &displayCfg, config);
        }
    }
    else
    {
        DFB_Platform_GetDisplaySettings(display, &displayCfg);

        D_DEBUG_AT( bcmnexusScreen, "%s() Using encoder configuration set externally by application (format=%d).\n", __FUNCTION__, displayCfg.format );

        getEncoderCfgFromDFBDisplayCfg(display, &displayCfg, config);
    }

    D_DEBUG_AT( bcmnexusScreen, "%s() screen %d defaults: outputs=0x%08x, std=%s, res=%s, freq=%s, scan=%s\n", __FUNCTION__, sd->id,
                config->out_connectors,
                dfb_screen_encoder_tv_standards_names[D_BITn32(config->tv_standard)].name,
                dfb_screen_output_resolution_names[D_BITn32(config->resolution)].name,
                dfb_screen_encoder_frequency_names[D_BITn32(config->frequency)].name,
                dfb_screen_encoder_scan_mode_names[D_BITn32(config->scanmode)].name
              );

    return res;
}

static DFBResult
bcmTestEncoderConfig( UNUSED_ CoreScreen                   *screen,
                      UNUSED_ void                         *driver_data,
                      UNUSED_ void                         *screen_data,
                              int                           encoder,
                              const DFBScreenEncoderConfig *config,
                              DFBScreenEncoderConfigFlags  *failed )
{
    DFBResult                   rc  = DFB_OK;
    DFB_PlatformDisplaySettings displayCfg;

    D_DEBUG_AT( bcmnexusScreen, "%s()\n", __FUNCTION__ );

    D_ASSERT((DFBBCMNEXUS*)driver_data != NULL);
    D_ASSERT((bcmScreenData*)screen_data != NULL);
    D_ASSERT(encoder == 0);
    D_ASSERT(config != NULL);
    D_ASSERT(failed != NULL);

    if (encoder != 0)
    {
        D_ERROR("%s() Invalid encoder ID %d\n", __FUNCTION__, encoder);
        return DFB_UNSUPPORTED;
    }

    *failed = DSECONF_NONE;
    rc = getDFBDisplayCfgFromEncoderCfg(config, &displayCfg, failed);
    if (rc != DFB_OK)
    {
        D_ERROR("%s() Couldn't determine DFB display config from desired encoder config\n", __FUNCTION__);
    }

    return rc;
}

static DFBResult
bcmSetEncoderConfig ( UNUSED_ CoreScreen                   *screen,
                              void                         *driver_data,
                              void                         *screen_data,
                              int                           encoder,
                              const DFBScreenEncoderConfig *config )
{
    DFBResult                     rc = DFB_OK;
    DFBBCMNEXUS                  *dfb_bcmnexus = driver_data;
    bcmScreenData                *sd  = screen_data;
    bcmLayerData                 *layerData = NULL;
    DFB_PlatformNexusHandle       display;
    DFB_PlatformDisplaySettings   displayCfg;
    DFB_PlatformNexusHandle       outputHandle;
    DFBScreenEncoderConfigFlags   failFlags;
    DFB_PlatformDimension         displaySize;
    bool                          primaryDisabled;

    D_DEBUG_AT( bcmnexusScreen, "%s()\n", __FUNCTION__ );

    D_ASSERT(dfb_bcmnexus != NULL);
    D_ASSERT(sd != NULL);
    D_ASSERT(encoder == 0);
    D_ASSERT(config != NULL);

    if (encoder != 0)
    {
        D_ERROR("%s() Invalid encoder ID %d\n", __FUNCTION__, encoder);
        return DFB_UNSUPPORTED;
    }

    if (!(config->flags & bcmnexus_sys_SUPPORTED_SCREEN_ENCODER_FLAGS))
    {
        D_ERROR("%s() Unsupported Option!\n", __FUNCTION__);
        return DFB_UNSUPPORTED;
    }

    if (!memcmp(config, &sd->encoderCfg, sizeof(DFBScreenEncoderConfig)))
    {
        return DFB_OK;
    }

    display = GET_DISPLAY_HND(dfb_bcmnexus, sd->id);

    rc = getDFBDisplayCfgFromEncoderCfg(config, &displayCfg, &failFlags);
    if (rc != DFB_OK)
    {
        D_ERROR("%s() Couldn't determine DFB display config from desired encoder config\n", __FUNCTION__);
        return rc;
    }

#if BCMNEXUS_NSC_SUPPORT
    if (0)
#else
    if (config->flags & DSECONF_CONNECTORS)
#endif
    {
        if (config->out_connectors & ~(DSOC_YC | DSOC_CVBS | DSOC_COMPONENT | DSOC_HDMI | DSOC_SCART | DSOC_SCART2))
        {
            D_ERROR("%s() Cannot support setting output connector(s) 0x%08x for display%d!\n", __FUNCTION__, config->out_connectors, sd->id);
            return DFB_UNSUPPORTED;
        }

        /* Composite output */
        if (config->out_connectors & DSOC_CVBS)
        {
            if ((DFB_Platform_GetDisplayOutput(display, DFB_PlatformDisplayOutputType_eComposite, 0, &outputHandle) == DFB_PLATFORM_OK) &&
                (outputHandle == NULL))
            {
                rc = bcmSetDisplayOutput( dfb_bcmnexus, sd->id, DFB_PlatformDisplayOutputType_eComposite, 0, true);
                if (rc != DFB_OK)
                {
                    D_ERROR("%s() Couldn't add composite%d output to display%d!\n", __FUNCTION__, 0, sd->id);
                    return rc;
                }
            }
        }
        else
        {
            if ((DFB_Platform_GetDisplayOutput(display, DFB_PlatformDisplayOutputType_eComposite, 0, &outputHandle) == DFB_PLATFORM_OK) &&
                (outputHandle != NULL))
            {
                rc = bcmSetDisplayOutput( dfb_bcmnexus, sd->id, DFB_PlatformDisplayOutputType_eComposite, 0, false);
                if (rc != DFB_OK)
                {
                    D_ERROR("%s() Couldn't remove composite%d output to display%d!\n", __FUNCTION__, 0, sd->id);
                    return rc;
                }
            }
        }
        /* Scart output */
        if (config->out_connectors & DSOC_SCART)
        {
            if ((DFB_Platform_GetDisplayOutput(display, DFB_PlatformDisplayOutputType_eScart, 0, &outputHandle) == DFB_PLATFORM_OK) &&
                (outputHandle == NULL))
            {
                rc = bcmSetDisplayOutput( dfb_bcmnexus, sd->id, DFB_PlatformDisplayOutputType_eScart, 0, true);
                if (rc != DFB_OK)
                {
                    D_ERROR("%s() Couldn't add scart%d output to display%d!\n", __FUNCTION__, 0, sd->id);
                    return rc;
                }
            }
        }
        else
        {
            if ((DFB_Platform_GetDisplayOutput(display, DFB_PlatformDisplayOutputType_eScart, 0, &outputHandle) == DFB_PLATFORM_OK) &&
                (outputHandle != NULL))
            {
                rc = bcmSetDisplayOutput( dfb_bcmnexus, sd->id, DFB_PlatformDisplayOutputType_eScart, 0, false);
                if (rc != DFB_OK)
                {
                    D_ERROR("%s() Couldn't remove scart%d output to display%d!\n", __FUNCTION__, 0, sd->id);
                    return rc;
                }
            }
        }

        /* Second Scart output */
        if (config->out_connectors & DSOC_SCART2)
        {
            if ((DFB_Platform_GetDisplayOutput(display, DFB_PlatformDisplayOutputType_eScart, 1, &outputHandle) == DFB_PLATFORM_OK) &&
                (outputHandle == NULL))
            {
                rc = bcmSetDisplayOutput( dfb_bcmnexus, sd->id, DFB_PlatformDisplayOutputType_eScart, 1, true);
                if (rc != DFB_OK)
                {
                    D_ERROR("%s() Couldn't add scart%d output to display%d!\n", __FUNCTION__, 1, sd->id);
                    return rc;
                }
            }
        }
        else
        {
            if ((DFB_Platform_GetDisplayOutput(display, DFB_PlatformDisplayOutputType_eScart, 1, &outputHandle) == DFB_PLATFORM_OK) &&
                (outputHandle != NULL))
            {
                rc = bcmSetDisplayOutput( dfb_bcmnexus, sd->id, DFB_PlatformDisplayOutputType_eScart, 1, false);
                if (rc != DFB_OK)
                {
                    D_ERROR("%s() Couldn't remove scart%d output to display%d!\n", __FUNCTION__, 1, sd->id);
                    return rc;
                }
            }
        }

        /* Svideo output */
        if (config->out_connectors & DSOC_YC)
        {
            if ((DFB_Platform_GetDisplayOutput(display, DFB_PlatformDisplayOutputType_eSvideo, 0, &outputHandle) == DFB_PLATFORM_OK) &&
                (outputHandle == NULL))
            {
                rc = bcmSetDisplayOutput( dfb_bcmnexus, sd->id, DFB_PlatformDisplayOutputType_eSvideo, 0, true);
                if (rc != DFB_OK)
                {
                    D_ERROR("%s() Couldn't add svideo%d output to display%d!\n", __FUNCTION__, 0, sd->id);
                    return rc;
                }
            }
        }
        else
        {
            if ((DFB_Platform_GetDisplayOutput(display, DFB_PlatformDisplayOutputType_eSvideo, 0, &outputHandle) == DFB_PLATFORM_OK) &&
                (outputHandle != NULL))
            {
                rc = bcmSetDisplayOutput( dfb_bcmnexus, sd->id, DFB_PlatformDisplayOutputType_eSvideo, 0, false);
                if (rc != DFB_OK)
                {
                    D_ERROR("%s() Couldn't remove svideo%d output to display%d!\n", __FUNCTION__, 0, sd->id);
                    return rc;
                }
            }
        }

        /* Component output */
        if (config->out_connectors & DSOC_COMPONENT)
        {
            if ((DFB_Platform_GetDisplayOutput(display, DFB_PlatformDisplayOutputType_eComponent, 0, &outputHandle) == DFB_PLATFORM_OK) &&
                (outputHandle == NULL))
            {
                rc = bcmSetDisplayOutput( dfb_bcmnexus, sd->id, DFB_PlatformDisplayOutputType_eComponent, 0, true);
                if (rc != DFB_OK)
                {
                    D_ERROR("%s() Couldn't add component%d output to display%d!\n", __FUNCTION__, 0, sd->id);
                    return rc;
                }
            }
        }
        else
        {
            if ((DFB_Platform_GetDisplayOutput(display, DFB_PlatformDisplayOutputType_eComponent, 0, &outputHandle) == DFB_PLATFORM_OK) &&
                (outputHandle != NULL))
            {
                rc = bcmSetDisplayOutput( dfb_bcmnexus, sd->id, DFB_PlatformDisplayOutputType_eComponent, 0, false);
                if (rc != DFB_OK)
                {
                    D_ERROR("%s() Couldn't remove component%d output to display%d!\n", __FUNCTION__, 0, sd->id);
                    return rc;
                }
            }
        }

        /* HDMI output */
        if (config->out_connectors & DSOC_HDMI)
        {
            if ((DFB_Platform_GetDisplayOutput(display, DFB_PlatformDisplayOutputType_eHdmi, 0, &outputHandle) == DFB_PLATFORM_OK) &&
                (outputHandle == NULL))
            {
                rc = bcmSetDisplayOutput( dfb_bcmnexus, sd->id, DFB_PlatformDisplayOutputType_eHdmi, 0, true);
                if (rc != DFB_OK)
                {
                    D_ERROR("%s() Couldn't add hdmi%d output to display%d!\n", __FUNCTION__, 0, sd->id);
                    return rc;
                }
            }
        }
        else
        {
            if ((DFB_Platform_GetDisplayOutput(display, DFB_PlatformDisplayOutputType_eHdmi, 0, &outputHandle) == DFB_PLATFORM_OK) &&
                (outputHandle != NULL))
            {
                rc = bcmSetDisplayOutput( dfb_bcmnexus, sd->id, DFB_PlatformDisplayOutputType_eHdmi, 0, false);
                if (rc != DFB_OK)
                {
                    D_ERROR("%s() Couldn't remove hdmi%d output to display%d!\n", __FUNCTION__, 0, sd->id);
                    return rc;
                }
            }
        }
    }

    if (rc != DFB_OK)
    {
        D_ERROR("%s() Couldn't determine DFB display config from desired encoder config\n", __FUNCTION__);
        return rc;
    }

    /*
     * We need to have a nasty little test here ...
     * If the region hasn't been realised yet there won't be any region data available inside the layer data
     * which will cause the layer calls below to barf.
     */

    layerData = bcmGetLayerData(GET_DISPLAY_GFX_LAYER_ID(dfb_bcmnexus, PRIMARY_GFX_LAYER_INDEX, sd->id));

    if ( layerData == NULL || layerData->regionData == NULL)
        primaryDisabled = true;
    else
        primaryDisabled = false;

    /* Disable graphics first to help prevent unwanted artefacts */
    if (!primaryDisabled)
        rc = bcmSetLayerState(dfb_bcmnexus, GET_DISPLAY_GFX_LAYER_ID(dfb_bcmnexus, PRIMARY_GFX_LAYER_INDEX, sd->id), false);

    if (rc == DFB_OK)
    {
        usleep(20*1000);    /* Wait 20ms */
        /*When using NxClient API, do not set display settings during initialization,
          to avoid display resolution changes already set by server or other
          DirectFB application.*/
#if BCMNEXUS_NXCLIENT_SUPPORT
        rc = setVideoOutput(dfb_bcmnexus, sd, 0, displayCfg.format, config->framing, config->aspect_ratio, !primaryDisabled);
#else
        rc = setVideoOutput(dfb_bcmnexus, sd, 0, displayCfg.format, config->framing, config->aspect_ratio, true);
#endif
        if (rc == DFB_OK)
        {
            /* Save new settings */
            sd->encoderCfg = *config;
        }

        if (!primaryDisabled)
        {
            DFBRectangle       destRectangle;

            DFB_Platform_P_GetDisplaySize(sd->id, &displaySize);

            destRectangle.x = 0;
            destRectangle.y = 0;
            destRectangle.w = displaySize.w;
            destRectangle.h = displaySize.h;

            D_DEBUG_AT( bcmnexusScreen, "%s Setting display rect (%d,%d) %d x %d\n", __FUNCTION__,
                        destRectangle.x,destRectangle.y,destRectangle.w,destRectangle.h );

            /* Now set the new destination rectangle and framebuffer */
            rc = bcmSetLayerDestRectangle(dfb_bcmnexus, GET_DISPLAY_GFX_LAYER_ID(dfb_bcmnexus, PRIMARY_GFX_LAYER_INDEX, sd->id), &destRectangle);
        }
    }

    usleep(20*1000);    /* Wait 20ms */

    /* Now re-enable graphics */
    if (!primaryDisabled)
        bcmSetLayerState(dfb_bcmnexus, GET_DISPLAY_GFX_LAYER_ID(dfb_bcmnexus, PRIMARY_GFX_LAYER_INDEX, sd->id), true);

    D_DEBUG_AT( bcmnexusScreen, "%s() screen %d, format=%d aspect_ratio=%d Exit [%d]\n", __FUNCTION__, sd->id, displayCfg.format, displayCfg.aspectRatio, rc );

    return rc;
}

DFBResult bcmInitMixer( UNUSED_ CoreScreen                *screen,
                        UNUSED_ void                      *driver_data,
                                void                      *screen_data,
                                int                        mixer,
                                DFBScreenMixerDescription *description,
                                DFBScreenMixerConfig      *config )
{
    DFBResult                result       = DFB_OK;

    D_DEBUG_AT( bcmnexusScreen, "%s() screen %d, mixer=%d, flags=0x%08x\n", __FUNCTION__, ((bcmScreenData*)screen_data)->id, mixer, config->flags );

    if (mixer > 0)
    {
        D_ERROR("%s() Unsupported mixer.\n", __FUNCTION__);
        return DFB_UNSUPPORTED;
    }

    /* Set the mixer capabilities. */
    description->caps = (DFBScreenMixerCapabilities)(DSMCAPS_BACKGROUND);

    snprintf( description->name, DFB_SCREEN_MIXER_DESC_NAME_LENGTH, "BCMNEXUS Pri. Mixer" );

    config->flags = DSMCONF_BACKGROUND;
    config->background.a = 0;
    config->background.r = 0;
    config->background.g = 0;
    config->background.b = 0;

    return result;
}

DFBResult bcmTestMixerConfig( UNUSED_ CoreScreen                 *screen,
                              UNUSED_ void                       *driver_data,
                              UNUSED_ void                       *screen_data,
                                      int                         mixer,
                                      const DFBScreenMixerConfig *config,
                                      DFBScreenMixerConfigFlags  *failed )
{
    D_DEBUG_AT( bcmnexusScreen, "%s()\n", __FUNCTION__ );

    *failed = DSECONF_NONE;

    if (mixer > 0)
    {
        D_ERROR("%s() Unsupported mixer.\n", __FUNCTION__);
        return DFB_UNSUPPORTED;
    }

    if (config->flags | (DSMCONF_TREE & DSMCONF_LEVEL & DSMCONF_LAYERS) )
    {
        *failed = ( config->flags | (DSMCONF_TREE & DSMCONF_LEVEL & DSMCONF_LAYERS) );
        return DFB_UNSUPPORTED;
    }

    if (config->flags & DSMCONF_BACKGROUND)
    {
        /* check the background colour is valid ? */
    }


    return DFB_OK;
}

/*
 * Set new configuration.
 */
DFBResult bcmSetMixerConfig( UNUSED_ CoreScreen                 *screen,
                             UNUSED_ void                       *driver_data,
                             UNUSED_ void                       *screen_data,
                                     int                         mixer,
                             UNUSED_ const DFBScreenMixerConfig *config )
{
    DFBResult               result = DFB_UNSUPPORTED;
#if !BCMNEXUS_NSC_SUPPORT
    DFB_PlatformNexusHandle        display;
    DFB_Platform_P_DisplaySettings displaySettings;
    DFBBCMNEXUS         *dfb_bcmnexus = driver_data;
    bcmScreenData       *sd           = screen_data;
#endif
    D_DEBUG_AT( bcmnexusScreen, "%s()\n", __FUNCTION__ );

    if (mixer > 0)
    {
        D_ERROR("%s() Unsupported mixer.\n", __FUNCTION__);
        return DFB_UNSUPPORTED;
    }

#if !BCMNEXUS_NSC_SUPPORT
    if (config->flags & DSMCONF_BACKGROUND)
    {
        display = GET_DISPLAY_HND(dfb_bcmnexus, sd->id);

        DFB_Platform_P_GetDisplaySettings(display, &displaySettings);

        displaySettings.background.a = config->background.a;
        displaySettings.background.r = config->background.r;
        displaySettings.background.g = config->background.g;
        displaySettings.background.b = config->background.b;

        result = (DFB_Platform_P_SetDisplaySettings( display, &displaySettings, true) == DFB_PLATFORM_OK) ? DFB_OK : DFB_FAILURE;
    }
#endif

    return result;
}

DFBResult bcmGetMixerState( UNUSED_ CoreScreen     *screen,
                            UNUSED_ void           *driver_data,
                            UNUSED_ void           *screen_data,
                            UNUSED_ int             mixer,
                            UNUSED_ CoreMixerState *ret_state )
{
    DFBResult                result = DFB_UNSUPPORTED;

    D_DEBUG_AT( bcmnexusScreen, "%s()\n", __FUNCTION__ );

    return result;
}


#if (NUM_DISPLAYS > 1)

static DFBResult
secondaryInitScreen( UNUSED_ CoreScreen           *screen,
                     UNUSED_ CoreGraphicsDevice   *device,
                             void                 *driver_data,
                             void                 *screen_data,
                             DFBScreenDescription *description )
{
    DFBResult                    result;
    DFBBCMNEXUS                 *dfb_bcmnexus = driver_data;
    bcmScreenData               *sd           = screen_data;
    DFB_PlatformVideoFormat      secondaryFormat;
    DFB_PlatformDisplaySettings  displaySettings;

    D_DEBUG_AT( bcmnexusScreen, "%s()\n", __FUNCTION__ );

    D_ASSERT( dfb_bcmnexus != NULL );

    /* Screen ID = 1 => Secondary output ON */
    sd->id = SECONDARY_DISPLAY_ID;
    sd->powerMode = DSPM_ON;

    /* Set the screen capabilities. */
    description->caps = DSCCAPS_VSYNC | DSCCAPS_ENCODERS;
    /* description->caps = DSCCAPS_OUTPUTS | DSCCAPS_VSYNC | DSCCAPS_ENCODERS; */
    /* description->outputs = 1; old system pre encoders*/
    description->encoders = 1;

    /* Set the screen name. */
    snprintf( description->name, DFB_SCREEN_DESC_NAME_LENGTH, "BCMNEXUS Secondary Screen" );

#if BCMNEXUS_NSC_SUPPORT
    secondaryFormat = DFB_PlatformVideoFormat_eNtsc;
#else
    /* Read back the video format of the primary display and slave the SD output to it
       in terms of display resolution (and frequency) */
    DFB_Platform_GetDisplaySettings( GET_DISPLAY_HND(dfb_bcmnexus, PRIMARY_DISPLAY_ID), &displaySettings);

    switch (displaySettings.format)
    {
    case DFB_PlatformVideoFormat_e1080p50hz:
    case DFB_PlatformVideoFormat_e1080i50hz:
    case DFB_PlatformVideoFormat_e1250i50hz:
    case DFB_PlatformVideoFormat_e720p50hz:
    case DFB_PlatformVideoFormat_e720p50hz_3DOU_AS:
    case DFB_PlatformVideoFormat_eCustom_3D_720p_50hz:
    case DFB_PlatformVideoFormat_e576p:
    case DFB_PlatformVideoFormat_e1440x576p50hz:
    case DFB_PlatformVideoFormat_eCustomer1366x768p50hz:
    case DFB_PlatformVideoFormat_ePalG:
    case DFB_PlatformVideoFormat_e288p50hz:
        secondaryFormat = DFB_PlatformVideoFormat_ePalG;
        break;
    default:
        secondaryFormat = DFB_PlatformVideoFormat_eNtsc;
    }
#endif

    result = setVideoOutput(dfb_bcmnexus, sd, 0, secondaryFormat, DSEPF_MONO, displaySettings.aspectRatio, true);
    D_DEBUG_AT( bcmnexusScreen, "%s() screen %d format=%d Exit [%d].\n", __FUNCTION__, sd->id, secondaryFormat, result );
    return result;
}

#endif

static DFBResult
bcmGetScreenSize( UNUSED_ CoreScreen *screen,
                          void       *driver_data,
                          void       *screen_data,
                          int        *ret_width,
                          int        *ret_height )
{
    bcmScreenData         *sd = screen_data;
    DFB_PlatformDimension  displaySize;

    D_ASSERT( (DFBBCMNEXUS*)driver_data != NULL );

    DFB_Platform_P_GetDisplaySize(sd->id, &displaySize);

    *ret_width  = displaySize.w;
    *ret_height = displaySize.h;

    D_DEBUG_AT( bcmnexusScreen, "%s() screen %d: %dx%d\n", __FUNCTION__, sd->id, *ret_width, *ret_height );

    return DFB_OK;
}

#if !BCMNEXUS_NSC_SUPPORT
static DFBResult
bcmWaitForVSync( UNUSED_ CoreScreen *screen,
                         void       *driver_data,
                         void       *screen_data )
{
    DFBBCMNEXUS    *dfb_bcmnexus = driver_data;
    bcmScreenData  *sd           = screen_data;
    pthread_mutex_t *mutex;
    pthread_cond_t  *sem;

    D_ASSERT( dfb_bcmnexus != NULL );

    D_DEBUG_AT( bcmnexusScreenX, "Wait for VSync[%d]\n", sd->id );

    mutex = &GET_DISPLAY_VSYNC_MUTEX(dfb_bcmnexus, sd->id);
    sem = &GET_DISPLAY_VSYNC_SEM(dfb_bcmnexus, sd->id);

    PTHREAD_ROBUST_MUTEX_LOCK(mutex);
    pthread_cond_wait(sem, mutex);
    pthread_mutex_unlock(mutex);

    D_DEBUG_AT( bcmnexusScreenX, "Received VSync[%d]\n", sd->id );

    return DFB_OK;
}
#endif

static DFBResult
bcmSetPowerMode( UNUSED_ CoreScreen         *screen,
                         void               *driver_data,
                         void               *screen_data,
                         DFBScreenPowerMode  mode )
{
    DFBResult      res = DFB_OK;
    DFBBCMNEXUS   *dfb_bcmnexus = driver_data;
    bcmScreenData *sd           = screen_data;

    D_ASSERT( dfb_bcmnexus != NULL );

    D_DEBUG_AT( bcmnexusScreen, "%s: screen %d power mode=%d\n", __FUNCTION__, sd->id, mode );

    if (mode != sd->powerMode) {
        sd->powerMode = mode;
        res = bcmSetLayerState(dfb_bcmnexus, GET_DISPLAY_GFX_LAYER_ID(dfb_bcmnexus, PRIMARY_GFX_LAYER_INDEX, sd->id), mode==DSPM_ON);
    }
    D_DEBUG_AT( bcmnexusScreen, "%s() screen %d Exit [%d].\n", __FUNCTION__, sd->id, res );
    return res;
}

DFBScreenPowerMode
bcmGetPowerMode( CoreScreen *screen )
{
    bcmScreenData *sd = screen->screen_data;
    return sd->powerMode;
}

ScreenFuncs bcmnexusPrimaryScreenFuncs = {
    .ScreenDataSize =    bcmScreenDataSize,
    .InitScreen =        primaryInitScreen,
    .InitEncoder =       bcmInitEncoder,
    .TestEncoderConfig = bcmTestEncoderConfig,
    .SetEncoderConfig =  bcmSetEncoderConfig,
    .GetScreenSize =     bcmGetScreenSize,
#if !BCMNEXUS_NSC_SUPPORT
    .WaitVSync =         bcmWaitForVSync,
#endif
    .SetPowerMode =      bcmSetPowerMode,
    .InitMixer =         bcmInitMixer,
    .TestMixerConfig =   bcmTestMixerConfig,
    .SetMixerConfig =    bcmSetMixerConfig,
    .GetMixerState =     bcmGetMixerState};


#if (NUM_DISPLAYS > 1)
ScreenFuncs bcmnexusSecondaryScreenFuncs = {
    .ScreenDataSize =    bcmScreenDataSize,
    .InitScreen =        secondaryInitScreen,
    .InitEncoder =       bcmInitEncoder,
    .TestEncoderConfig = bcmTestEncoderConfig,
    .SetEncoderConfig =  bcmSetEncoderConfig,
    .GetScreenSize =     bcmGetScreenSize,
#if !BCMNEXUS_NSC_SUPPORT
    .WaitVSync =         bcmWaitForVSync,
#endif
    .SetPowerMode =      bcmSetPowerMode
};
#endif
