/*
* ============================================================================
* RDK MANAGEMENT, LLC CONFIDENTIAL AND PROPRIETARY
* ============================================================================
* This file (and its contents) are the intellectual property of RDK Management, LLC.
* It may not be used, copied, distributed or otherwise  disclosed in whole or in
* part without the express written permission of RDK Management, LLC.
* ============================================================================
* Copyright (C) 2017 Broadcom. The term "Broadcom" refers to Broadcom Limited and/or its subsidiaries.
* Copyright (c) 2017 RDK Management, LLC. All rights reserved.
* ============================================================================
*/

///////////////////////////////////////////////////////////
//  rmf_SectionFilter_INB.cpp
//  Implementation of the Class rmf_SectionFilter_INB
//  Created on:      20-Jul-2012 12:35:12 PM
//  Original author: Sunil S
///////////////////////////////////////////////////////////

#include "rmf_sectionfilter_inb.h"
#include "rmf_osal_mem.h"
#include "rmf_osal_thread.h"
#include "rmf_osal_util.h"
#include "rdk_debug.h"
#include "rmf_osal_memmap.h"

#include <assert.h>
#include <string.h>

#undef DEBUG_ERROR
#define DEBUG_ERROR
//#define CAPMT_ENABLED 1
#define CRC_INIT    0xFFFFFFFF
#define POLYNOMIAL        0x04c11db7        // for Crc32

#ifdef RMF_SF_SINGLE_THREAD_ENABLE
#define RMF_SF_DEFAULT_TIMEOUT       100000 //Default timeout for filter sections (in microseconds)
#endif

#if (NEXUS_PLATFORM == 97425) 
/* MOT7425-7248: 6 Tuner support. Add software rate-smoothing. Loop back Tuner 0 through Recpump & Playpump */
/* NOTE: these values *MUST* remain synchronized with the values in gstqamtunersrc_priv.h and pod_api.h */
#define SW_RS_LOOPBACK_PLAYPUMP_INDEX 15
#define SW_RS_LOOPBACK_PLAYPUMP_LTSID (SW_RS_LOOPBACK_PLAYPUMP_INDEX + 0x40) /* fixed mapping on 74xx */
#define SW_RS_LOOPBACK_PARSER_BAND 0
static int bInitRSTest = false;
static int bSWRateSmooth = false;
#include <vlEnv.h>

#endif


typedef struct rmf_thread_args_s
{
	void *pSectionFilter;
	void *pData;
}rmf_thread_args_t;

static unsigned    guiCrcTable[CRC_TABLE_SIZE];

rmf_SectionFilter_INB::rmf_SectionFilter_INB(void* pFilterSource):rmf_SectionFilter(pFilterSource) 
{
	rmf_FilterSource_INB *pInbFilterSource = (rmf_FilterSource_INB*)pFilterSource;

	m_SourceID 			= pInbFilterSource->tunerId;

	//Not required as demux handle is passed ar part of pInbFilterSource argument to the constructor
#if 0
	vlMedia_getDemuxHandle(	MEDIA_DECODE_SESSION_BROADCAST, 
			m_SourceID,
			&(((Inband_SectionFilter_Info_t*)(m_pSectionFilterInfo))->dmxHdl) );
#endif
	parserHandle = pInbFilterSource->dmxHdl;
	tunerId = pInbFilterSource->tunerId;
	freq = pInbFilterSource->freq;
	tsid = pInbFilterSource->tsId;
	//ltsid = pInbFilterSource->ltsId;
#ifdef CAPMT_ENABLED
	m_PmtCaptureTunerId = 0;
	m_PmtCapturePidId = 0;
#endif
	m_DVR_SMD_BUF_SIZE = 0;
	g_stThreadId = 100;
	FilterSessionListInit();
}

rmf_SectionFilter_INB::~rmf_SectionFilter_INB()
{
	FilterSessionListClose();
}

void rmf_SectionFilter_INB::GenerateCrcTable()
{
    register unsigned    crcAccum = 0;
    register int    i, j;

    for (i = 0; i < CRC_TABLE_SIZE; i++)
    {
        crcAccum = (unsigned)i << 24;
        for (j = 0; j < 8; j++)
        {
            if (crcAccum & 0x80000000)
            {
                crcAccum <<= 1;
                crcAccum  = (crcAccum ) ^ POLYNOMIAL;
            }
            else
               // crcAccum  = (crcAccum <<= 1);
             crcAccum <<= 1;//Murali: Coverity fix
        }
        guiCrcTable[i] = crcAccum;
    }
}
/* ----------------------------------------------------------------------------
 *    Calculate the CRC32 on the data block one byte at a time.
 *    Accumulation is useful when computing over all ATM cells in an AAL5 packet.
 *    NOTE: The initial value of crcAccum must be 0xffffffff
 * ----------------------------------------------------------------------------
 */
unsigned long rmf_SectionFilter_INB::Crc32(unsigned crcAccum, unsigned char *pData, int len)
{
   static unsigned    long tableDone = 0;

    register int i, j;

    if ( !tableDone)
    {
        GenerateCrcTable();
        tableDone = 1;
    }

    for (j = 0; j < len; j++ )
    {
        i = ((int)(crcAccum >> 24) ^ *pData++) & 0xff;
        crcAccum = (crcAccum << 8) ^ guiCrcTable[i];
    }
    return ~crcAccum;
}

void rmf_SectionFilter_INB::SectionFilterThreadFn(void *arg)
{
        RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER", "<%s: %d> - Enter\n",
             __FUNCTION__, __LINE__);
        rmf_thread_args_t *pArgs = (rmf_thread_args_t*)arg;
        rmf_SectionFilter_INB *pInbSectionFilter = (rmf_SectionFilter_INB*)pArgs->pSectionFilter;
        pInbSectionFilter->SectionMonitor(pArgs->pData);
        delete pArgs;
}

void rmf_SectionFilter_INB::SectionMonitor(void *pData)
{
#ifdef RMF_SF_SINGLE_THREAD_ENABLE
    rmf_FilterSession_t *pFilterSession = (rmf_FilterSession_t *)pData;
    rmf_osal_event_handle_t eventHandle;
    rmf_osal_event_params_t eventParams;
    uint32_t eventType;
    int32_t timeout = RMF_SF_DEFAULT_TIMEOUT;
    bool bShutDown = FALSE;
    int ret = 0;
    rmf_Error err = RMF_SUCCESS;

    RDK_LOG(RDK_LOG_DEBUG, "LOG.RDK.FILTER", "<%s: %d> - Enter\n",
             __FUNCTION__, __LINE__);

   while(!bShutDown)
    {
        RDK_LOG(RDK_LOG_DEBUG, "LOG.RDK.FILTER", "<%s: %d> - Waiting for new event...\n",
                 __FUNCTION__, __LINE__);

        eventParams.data = NULL; //Coverity 
        err = rmf_osal_eventqueue_get_next_event_timed (pFilterSession->sectionMonitorQueue, &eventHandle,
                                             NULL, &eventType, &eventParams, timeout);

        if(RMF_SUCCESS != err )
        {
            if(err == RMF_OSAL_ENODATA)
            {
                RDK_LOG(RDK_LOG_DEBUG, "LOG.RDK.FILTER", "<%s: %d> - Invalid event \n",
                              __FUNCTION__, __LINE__, eventType);
                continue;
            }
            else
            {
                RDK_LOG(RDK_LOG_ERROR,
                    "LOG.RDK.FILTER",
                    "<%s: %d> - unable to get event,... terminating thread. err = 0x%x\n",
                     __FUNCTION__, __LINE__, err);

                pFilterSession->threadStatus = SF_THREAD_QUIT;
                break;
            }
        }
	else
	{
		RDK_LOG(RDK_LOG_DEBUG, "LOG.RDK.FILTER", "<%s: %d> - New Event Received - %d\n",
				__FUNCTION__, __LINE__, eventType);

		switch (eventType)
		{
			case RMF_INB_SF_EVENT_NEWSECTION:
				{
					RDK_LOG(RDK_LOG_DEBUG, "LOG.RDK.FILTER","<%s: %d> - New Section\n",
							__FUNCTION__, __LINE__);

					available_data_t* pData = (available_data_t*)eventParams.data;
                    if ( NULL != pData ) { //Coverity
    					section_avail_cb(pData->filterID, pData->pSectionData);
    					rmf_osal_memFreeP(RMF_OSAL_MEM_FILTER, pData);
                    }
				}
				break;

			case RMF_INB_SF_EVENT_SHUTDOWN:
				{
					RDK_LOG(RDK_LOG_DEBUG, "LOG.RDK.FILTER", "<%s: %d> - Shutting down the section monitor thread...\n", __FUNCTION__, __LINE__);
					bShutDown = TRUE;
				}
				break;

			default:
				break; 
		}

		if ( eventHandle) {
		    rmf_osal_event_delete(eventHandle);
		}	  
	}
    }

    RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER",
        "<%s:%d> - Posting pMonitorThreadStopSem.\n", __FUNCTION__, __LINE__);
 
    ret = sem_post( pFilterSession->pMonitorThreadStopSem);
    if (0 != ret )
    {
        RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER",
        "<%s:%d> - sem_post failed.\n", __FUNCTION__, __LINE__);
    }

    RDK_LOG( RDK_LOG_TRACE1, "LOG.RDK.FILTER", "<%s:%d> - Exit\n", __FUNCTION__, __LINE__);

#else
    available_data_t* castdata = (available_data_t*)pData;
    if(m_filterState == SECTFLT_STARTED)
    {
        RDK_LOG(RDK_LOG_DEBUG, "LOG.RDK.FILTER", "%s: calling section_avail_cb()...\n", __FUNCTION__);
        section_avail_cb(castdata->filterID, castdata->pSectionData);
    }
    else
    {
            if (NULL != castdata->pSectionData)
            {
                if (NULL != castdata->pSectionData->sectionData)
                {
                    rmf_osal_memFreeP(RMF_OSAL_MEM_FILTER, castdata->pSectionData->sectionData);
                }

                rmf_osal_memFreeP(RMF_OSAL_MEM_FILTER, castdata->pSectionData);
            }
    }

    rmf_osal_memFreeP(RMF_OSAL_MEM_FILTER, pData);
#endif
}

void rmf_SectionFilter_INB::NexusMessageCallback( void *context, int param)
{
    NEXUS_MessageHandle sectionFilterHandle;
    unsigned char* sectionDataBuffer;
    size_t bufSize = 0;
    uint32_t filterID = 0;
    unsigned long parserHandle;
    NEXUS_PidChannelHandle pidFilterHandle;
    NEXUS_Error nexusApiResult;
    context_struct_t * retContext = (context_struct_t*)context;
    rmf_Error retOsal = RMF_SUCCESS;
    rmf_Section_Data *pSectionData = NULL;
    rmf_SectionFilter_INB *pInbFilter = (rmf_SectionFilter_INB*)retContext->pFilter;
    available_data_t *pData = NULL;

    RDK_LOG(RDK_LOG_DEBUG, "LOG.RDK.FILTER", "%s: entering...\n", __FUNCTION__);

    filterID = retContext->filterID;
    parserHandle = retContext->parserHandle;
    pidFilterHandle = retContext->pidFilterHandle;
    sectionFilterHandle = retContext->sectionFilterHandle;

    RDK_LOG(RDK_LOG_DEBUG, "LOG.RDK.FILTER", "%s: filterID is %d\n", __FUNCTION__, filterID);

    if (0 == sectionFilterHandle)
    {
        RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER", "%s: Nexus message handle is NULL\n", __FUNCTION__);
        return;
    }

    nexusApiResult = NEXUS_Message_GetBuffer(sectionFilterHandle, (const void **)&sectionDataBuffer, &bufSize);
    if(NEXUS_SUCCESS != nexusApiResult)
    {
	RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER", "%s: NEXUS_Message_GetBuffer returned error : %d\n", __FUNCTION__, nexusApiResult);
	return;
    }

    if(bufSize == 0)
    {
      RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER", "%s: Section Data size 0 returned from NEXUS_Message_Buffer\n", __FUNCTION__);
      return;
    }

    {
        //Check if current parserband is disabled or not, especially for MTSIF interface
        NEXUS_ParserBandSettings parserBandSettings;
        NEXUS_ParserBand_GetSettings((NEXUS_ParserBand)parserHandle, &parserBandSettings);

        if ( parserBandSettings.sourceType == NEXUS_ParserBandSourceType_eMtsif 
           && parserBandSettings.sourceTypeSettings.mtsif == NULL ) {
            //RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER", "%s:ignore data\n", __FUNCTION__);
            NEXUS_Message_ReadComplete(sectionFilterHandle, bufSize);
            return;
        }
    }

    RDK_LOG(RDK_LOG_DEBUG, "LOG.RDK.FILTER", "%s: NEXUS_Message_GetBuffer successful\n", __FUNCTION__);

    // Load section data to temporary buffer so we can release the Nexus Message buffer.
    retOsal = rmf_osal_memAllocP(RMF_OSAL_MEM_FILTER, sizeof(rmf_Section_Data), (void**)&pSectionData);
    if((RMF_SUCCESS != retOsal) || (NULL == pSectionData))
    {
      RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER", "%s: OUT OF MEMORY - while allocating %d bytes of memory for vl_Section_Data\n", __FUNCTION__, sizeof(rmf_Section_Data));
      NEXUS_Message_ReadComplete(sectionFilterHandle, bufSize);
      return;
    }

    // If buffer is bigger than the max message size, truncate it
    if (bufSize > SECTFLT_MAX_MESSAGE_SIZE)
    {
        RDK_LOG(RDK_LOG_WARN, "LOG.RDK.FILTER", "%s: Truncating buffer size\n", __FUNCTION__);
        bufSize = SECTFLT_MAX_MESSAGE_SIZE;
    }

    pSectionData->sectionID = pInbFilter->getNextSectionID();

    pSectionData->sectionLength = bufSize;

    retOsal = rmf_osal_memAllocP(RMF_OSAL_MEM_FILTER, bufSize, (void**)&(pSectionData->sectionData));
    if((RMF_SUCCESS != retOsal) || (NULL == pSectionData->sectionData))
    {
        RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER", "%s: OUT OF MEMORY - while allocating %d of memory for storing section filter data\n", __FUNCTION__, bufSize);
        NEXUS_Message_ReadComplete(sectionFilterHandle, bufSize);
        rmf_osal_memFreeP(RMF_OSAL_MEM_FILTER, pSectionData);
        return;
    }

    memcpy(pSectionData->sectionData, sectionDataBuffer, bufSize);

    RDK_LOG(RDK_LOG_DEBUG, "LOG.RDK.FILTER", "%s: stored filter data to section data structure. tableid: 0x%x\n", __FUNCTION__, sectionDataBuffer[0]);
    // Feb-12-2013: BPV-3387: Added section CRC-check error logging.
    {
        int nSectionSize = (((sectionDataBuffer[1]<<8)|(sectionDataBuffer[2]<<0))&0xFFF)+3;
        int table_id = sectionDataBuffer[0];
        RDK_LOG(RDK_LOG_DEBUG, "LOG.RDK.FILTER", "%s: section size = %d, buf size = %d, table_id: %d\n", __FUNCTION__, nSectionSize, bufSize, table_id);
        if(table_id == 2)
        {
	    int prog_no = (((sectionDataBuffer[3]<<8)|(sectionDataBuffer[4]<<0))&0xFFF);
            RDK_LOG(RDK_LOG_DEBUG, "LOG.RDK.FILTER", "%s: prog_no: %d\n", __FUNCTION__, prog_no);
        }
        unsigned long nCalculatedCrc32 = pInbFilter->Crc32(CRC_INIT, sectionDataBuffer, nSectionSize);
        int nCrcOffset = (nSectionSize-4);
        unsigned long nSectionCrc32 = (sectionDataBuffer[nCrcOffset+0]<<24)|(sectionDataBuffer[nCrcOffset+1]<<16)|(sectionDataBuffer[nCrcOffset+2]<<8)|(sectionDataBuffer[nCrcOffset+3]<<0);
        if(CRC_INIT != nCalculatedCrc32)
        {
            RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER", "%s: CRC CHECK FAILED for section of size = %d, padded size = %d, CRC_INIT = 0x%08X, section crc = 0x%08X, calculated crc = 0x%08X\n", __FUNCTION__, nSectionSize, bufSize, CRC_INIT, nSectionCrc32,  nCalculatedCrc32);
            //vlMpeosDumpBuffer(RDK_LOG_ERROR, "LOG.RDK.FILTER", sectionDataBuffer, bufSize);
        }

	if ((table_id == 0) || (table_id == 2))
	{

		RDK_LOG(RDK_LOG_DEBUG, "LOG.RDK.FILTER",
				"%s:: Adjusting section length, tableid=%u, len=%d --> %d\n",
				__FUNCTION__, table_id, pSectionData->sectionLength, nSectionSize);

		pSectionData->sectionLength = nSectionSize;
	}
    }

    nexusApiResult = NEXUS_Message_ReadComplete(sectionFilterHandle, bufSize);

    RDK_LOG(RDK_LOG_DEBUG, "LOG.RDK.FILTER", "%s: NEXUS_Message_ReadComplete issued. nexusApiResult = %d\n", __FUNCTION__, nexusApiResult);

#if 0
    // Enable to print out message buffer
    int i = 0;          
    for(i=0; i<bufSize; i++)
    {
        printf("%02X ", pSectionData->sectionData[i]);
    }
    printf("\n\n");
#endif

#ifdef CAPMT_ENABLED
    VL_SECTFLT_RESULT result;
    rmf_FilterSession_t *pFilterSessionInfo = NULL;
    result = vl_get_Filter_SessionObj_WithHandle(parserHandle, pidFilterHandle, &pFilterSessionInfo, filterID);
    if (NULL != pFilterSessionInfo)
    {
        if (pFilterSessionInfo->tableId == 0x02) //PMT
        {
            uint32_t TunerId;
            RDK_LOG(RDK_LOG_TRACE7, "LOG.RDK.FILTER","SectionFilterThreadFn:>>> Pmt Capture>>> \n");

            if(0 == vlGetTunerIdForFilterId( filterID, &TunerId ) )
            {
                vlhal_sectionfilter_CollectPmts(pSectionData,TunerId);
            }
        }
        else
        {
            RDK_LOG(RDK_LOG_TRACE7, "LOG.RDK.FILTER", "%s: tableId is not 0x02, it is 0x%x\n",
                      __FUNCTION__, pFilterSessionInfo->tableId);
        }
    }
    else
    {
        RDK_LOG(RDK_LOG_WARN, "LOG.RDK.FILTER", "Inband_SECTFLT_MessageCallback: pFilterSessionInfo is 0\n");
    }
#endif

    if(0 != filterID)
    {
        retOsal = rmf_osal_memAllocP(RMF_OSAL_MEM_FILTER, sizeof(available_data_t), (void**)&pData);
        if (NULL != pData)
        {
            pData->filterID = filterID;
            pData->pSectionData = pSectionData;

#ifdef RMF_SF_SINGLE_THREAD_ENABLE
            rmf_osal_event_params_t event_params = {0};
            rmf_osal_event_handle_t event_handle;
	    rmf_FilterSession_t     *pFilterSession = NULL;
            rmf_Error ret = RMF_SUCCESS;

            rmf_osal_mutexAcquire(pInbFilter->m_stFilterSessionMutex);

            pFilterSession = (rmf_FilterSession_t*)pInbFilter->get_section_filter_info(filterID);

            if((pFilterSession != NULL) && (pFilterSession->threadStatus == SF_THREAD_ACTIVE) && (pFilterSession->sectionMonitorQueue != 0))
            {
                event_params.priority = 0xFF;
                event_params.data = (void*)pData;

		RDK_LOG(RDK_LOG_DEBUG, "LOG.RDK.FILTER","%s::  Creating NEWSECTION event \n", __FUNCTION__);
		ret = rmf_osal_event_create(RMF_OSAL_EVENT_CATEGORY_MPEG, RMF_INB_SF_EVENT_NEWSECTION, &event_params, &event_handle);
		if(ret != RMF_SUCCESS)
		{
                    RDK_LOG(
                             RDK_LOG_ERROR,
			     "LOG.RDK.FILTER",
			     "<%s: %d> Could not create event handle: %x\n",
			      __FUNCTION__, __LINE__, ret);

		    if(NULL != pData)
		    {
			    rmf_osal_memFreeP(RMF_OSAL_MEM_FILTER, pData);
		    }

		    if (NULL != pSectionData)
		    {
                        if (NULL != pSectionData->sectionData)
			{
				rmf_osal_memFreeP(RMF_OSAL_MEM_FILTER, pSectionData->sectionData);
			}

			rmf_osal_memFreeP(RMF_OSAL_MEM_FILTER, pSectionData);
		    }
			//return ret;
		}
		else
		{
		    RDK_LOG(RDK_LOG_DEBUG, "LOG.RDK.FILTER","%s::  Sending NEWSECTION event to SectionMonitor thread\n", __FUNCTION__);
		    rmf_osal_eventqueue_add(pFilterSession->sectionMonitorQueue, event_handle);
		}
            }
            else
            {
		RDK_LOG(RDK_LOG_DEBUG, "LOG.RDK.FILTER","%s::  Not a valid session\n", __FUNCTION__);
                if(NULL != pData)
		{   
		    rmf_osal_memFreeP(RMF_OSAL_MEM_FILTER, pData);
		}

		if (NULL != pSectionData)
		{
		    if (NULL != pSectionData->sectionData)
		    {
			    rmf_osal_memFreeP(RMF_OSAL_MEM_FILTER, pSectionData->sectionData);
		    }

		    rmf_osal_memFreeP(RMF_OSAL_MEM_FILTER, pSectionData);
		}
	    }
            
            rmf_osal_mutexRelease(pInbFilter->m_stFilterSessionMutex);
#else
            rmf_osal_ThreadId available_data_thread_id;
            /* Feb-08-2013: BPV-3194: Use cThreadCreateSimple() instead of mpeos_threadCreate() to reduce chance of heap corruption.
            mpeos_threadCreate((void*)available_data_thread, (void*)(data),
                               MPE_THREAD_PRIOR_MAX, MPE_THREAD_STACK_SIZE,
                               &available_data_thread_id, "available_data_thread");*/
            //cThreadCreateSimple("available_data_thread", (THREADFP)available_data_thread, (unsigned long)data);
	    rmf_thread_args_t* threadargs = new rmf_thread_args_t;
            if (NULL != threadargs)
            {
                threadargs->pSectionFilter = (void*)pInbFilter;
                threadargs->pData = (void*)pData;

                rmf_osal_threadCreate( SectionFilterThreadFn, (void*)threadargs, RMF_OSAL_THREAD_PRIOR_DFLT, RMF_OSAL_THREAD_STACK_SIZE, &available_data_thread_id, "Section_Filter_Thread" );
                RDK_LOG(RDK_LOG_DEBUG, "LOG.RDK.FILTER", "%s: Started available_data thread for filterID %u\n",
                      __FUNCTION__, filterID);
            }
            else
            {
                RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER","%s:: threadargs couldnot be created.\n", __FUNCTION__);
            }
#endif
        }
        else
        {
            if (NULL != pSectionData)
            {
                if (NULL != pSectionData->sectionData)
                {
                    rmf_osal_memFreeP(RMF_OSAL_MEM_FILTER, pSectionData->sectionData);
                }

                rmf_osal_memFreeP(RMF_OSAL_MEM_FILTER, pSectionData);
            }

            RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER", "%s: Failed to allocate available data structure\n", __FUNCTION__);
        }
    }
    else
    {
        if (NULL != pSectionData)
        {
            if (NULL != pSectionData->sectionData)
            {
                rmf_osal_memFreeP(RMF_OSAL_MEM_FILTER, pSectionData->sectionData);
            }

            rmf_osal_memFreeP(RMF_OSAL_MEM_FILTER, pSectionData);
        }

        RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER", "%s: Not a Valid Filter id\n", __FUNCTION__);
    }

    RDK_LOG(RDK_LOG_DEBUG, "LOG.RDK.FILTER", "%s: Exiting...\n", __FUNCTION__);
}

void rmf_SectionFilter_INB::InitializeNexusMessage(TRNSPT_CALLBACK_t SECTFLT_MessageCallback, int index)
{
    rmf_Error retOsal = RMF_SUCCESS;
    NEXUS_MessageSettings   msgSettings;
    context_struct_t *context = NULL;

    // Free when closing Nexus message
    retOsal = rmf_osal_memAllocP(RMF_OSAL_MEM_FILTER, sizeof(context_struct_t), (void**)&context);
    if((RMF_SUCCESS != retOsal) || (NULL == context))
    {
        RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER","%s: context FAILED\n", __FUNCTION__);
    }

    // Get default Nexus message settings
    NEXUS_Message_GetDefaultSettings(&msgSettings);

    // Initialize the filterID to 0, this will indicate an unused message handle
    context->filterID = 0;
    context->pFilter = (void*)this;

    msgSettings.dataReady.callback          = SECTFLT_MessageCallback;
    msgSettings.dataReady.context           = (void*)context;
    msgSettings.bufferSize                  = 0;  //we allocated the memory using NEXUS_Memory_Allocate earlier when pidchannel got opened
    msgSettings.maxContiguousMessageSize    = 4096;

    // Open nexus message filter and assign to a filter session structure
    m_stFilterSessionList[index].sectionFilterHandle = NEXUS_Message_Open(&msgSettings);

    if(NULL == m_stFilterSessionList[index].sectionFilterHandle)
    {
        RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER","%s: NEXUS_Message_Open FAILED\n", __FUNCTION__);
        rmf_osal_memFreeP(RMF_OSAL_MEM_FILTER, context);
    }
    else
    {
        RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER", "%s: %p\n",
                  __FUNCTION__, m_stFilterSessionList[index].sectionFilterHandle);
    }
}

void rmf_SectionFilter_INB::CloseNexusMessage(int index)
{
    NEXUS_MessageSettings msgSettings;

    // Get the message settings so we can free the message context memory
    NEXUS_Message_GetSettings(m_stFilterSessionList[index].sectionFilterHandle, &msgSettings);
    context_struct_t* context = (context_struct_t*)msgSettings.dataReady.context;
    // Now close the message handle
    NEXUS_Message_Close(m_stFilterSessionList[index].sectionFilterHandle);
    RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER", "%s: %p\n",
              __FUNCTION__, m_stFilterSessionList[index].sectionFilterHandle);

    rmf_osal_memFreeP(RMF_OSAL_MEM_FILTER, context);
}

unsigned long rmf_SectionFilter_INB::AllocateFilterMemory()
{
    void* pMsgBufMem = NULL;
    unsigned long msgBufSize = 0;
    NEXUS_Error nexusApiResult;
    NEXUS_MemoryAllocationSettings memSettings;

#ifdef RDK_USE_NXCLIENT
    NEXUS_ClientConfiguration platformConfig;
    NEXUS_Platform_GetClientConfiguration(&platformConfig);

    NEXUS_Memory_GetDefaultAllocationSettings(&memSettings);
    memSettings.heap = platformConfig.heap[1]; /* protected and un-prot */
#else    
    memSettings.heap = NULL;
#endif
    
    memSettings.alignment = 1024;
    msgBufSize = 32*1024;

    if ((nexusApiResult = NEXUS_Memory_Allocate(msgBufSize, &memSettings, &pMsgBufMem)) != NEXUS_SUCCESS)
    {
        RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER", "AllocateFilterMemory Failed. Error = %d \n", nexusApiResult);
    }
    return (unsigned long)pMsgBufMem;
}

rmf_FilterSession_t* rmf_SectionFilter_INB::GetAvailableNexusMessage()
{
    int filterIndex = 0;
    rmf_FilterSession_t *pFilterSession = NULL;

    while ((filterIndex < MAX_FILTER_COUNT) && (pFilterSession == NULL))
    {
        if (INVALID_FILTER == m_stFilterSessionList[filterIndex].filterState)
        {
            pFilterSession = &m_stFilterSessionList[filterIndex];
            RDK_LOG(RDK_LOG_TRACE1, "LOG.RDK.FILTER", "%s: returning filter index %d\n", __FUNCTION__, filterIndex);
        }
        else
        {
            filterIndex++;
        }
    }

    return pFilterSession;
}

void rmf_SectionFilter_INB::FilterSessionListInit()
{
        int i =0;
        m_stFilterSessionCnt = 0;
        rmf_osal_mutexNew(&m_stFilterSessionMutex);
        memset(m_stFilterSessionList, 0, sizeof(rmf_FilterSession_t)*MAX_FILTER_COUNT);

        for (i = 0; i < MAX_FILTER_COUNT; i++)
        {
                m_stFilterSessionList[i].filterState = INVALID_FILTER;
                m_stFilterSessionList[i].threadStatus = SF_THREAD_INACTIVE;
		// allocate the NEXUS memory for this filter now
		m_stFilterSessionList[i].ulMsgBufAddr = AllocateFilterMemory();

		InitializeNexusMessage(NexusMessageCallback, i);

		RDK_LOG(RDK_LOG_TRACE8, "LOG.RDK.FILTER", "%s: filter session location %d\n", __FUNCTION__, i);
		RDK_LOG(RDK_LOG_TRACE8, "LOG.RDK.FILTER", "%s: filter session message address 0x%lu\n",
		          __FUNCTION__, m_stFilterSessionList[i].ulMsgBufAddr);
		RDK_LOG(RDK_LOG_TRACE8, "LOG.RDK.FILTER", "%s: filter session message handle %p\n",
		          __FUNCTION__, m_stFilterSessionList[i].sectionFilterHandle);

        }

        //m_sectionHandleMap = new rmf_SymbolMapTable();
        //rmf_osal_mutexNew(&m_sectionHandleMapMutex);

    rmf_osal_mutexNew(&m_stPmtDataMutex);
}

void rmf_SectionFilter_INB::FilterSessionListClose()
{
        int i = 0;

        RDK_LOG(RDK_LOG_TRACE8, "LOG.RDK.FILTER","%s: begin \n", __FUNCTION__);

        //Stop all pending Callbacks before acquire mutex to avoid deadlock
        for (i = 0; i < MAX_FILTER_COUNT; i++)
        {
            if ( NULL != m_stFilterSessionList[i].sectionFilterHandle )
            { 
                NEXUS_StopCallbacks(m_stFilterSessionList[i].sectionFilterHandle);
            }		
	} 

        rmf_osal_mutexAcquire(m_stFilterSessionMutex);

        for (i = 0; i < MAX_FILTER_COUNT; i++)
        {
            m_stFilterSessionList[i].filterState = INVALID_FILTER;

            CloseNexusMessage(i);

            // Free the filter's NEXUS memory now
            NEXUS_Memory_Free((void*)m_stFilterSessionList[i].ulMsgBufAddr);
  	    m_stFilterSessionList[i].ulMsgBufAddr = NULL;
        }

        memset(m_stFilterSessionList, 0, sizeof(rmf_FilterSession_t)*MAX_FILTER_COUNT);

	rmf_osal_mutexAcquire(m_stFilterSessionMutex);
        rmf_osal_mutexDelete(m_stFilterSessionMutex);
        
	//rmf_osal_mutexAcquire(m_sectionHandleMapMutex);
	//delete m_sectionHandleMap;
        //rmf_osal_mutexRelease(m_sectionHandleMapMutex);

        //rmf_osal_mutexDelete(m_sectionHandleMapMutex);

    	rmf_osal_mutexDelete(m_stPmtDataMutex);

        m_stFilterSessionCnt = 0;
}

RMF_SECTFLT_RESULT rmf_SectionFilter_INB::get_Filter_SessionObj_WithPid(unsigned long parserHandle,
                                                 uint32_t pid,
                                                 PDT_PidChannelType ePidChanType,
                                                 rmf_FilterSession_t **ppFilterSession)
{
        rmf_FilterSession_t *pFilter = NULL;
        int i = 0;

        if (NULL == ppFilterSession)
                return RMF_SECTFLT_Error;

//      rmf_osal_mutexAcquire(m_stFilterSessionMutex);

        for (i=0; i < MAX_FILTER_COUNT; i++)
        {
                pFilter = m_stFilterSessionList+i;
                if (INVALID_FILTER == pFilter->filterState)
                        continue;

                //if (SF_THREAD_QUIT == pFilter->threadStatus)
                  //      continue;

                if ( (pFilter->parserHandle == parserHandle)  && (pFilter->pid == pid) &&
                         (pFilter->pidType == ePidChanType) )
                {
                        if(pFilter->RefCnt <= 0)        //Already marked for release
                                continue;

                        *ppFilterSession = pFilter; // Content of this may be modified outside 
//                      rmf_osal_mutexRelease(m_stFilterSessionMutex);
                        return RMF_SECTFLT_Success;
                }
        }

//      rmf_osal_mutexRelease(m_stFilterSessionMutex);
        return RMF_SECTFLT_Error;
}

RMF_SECTFLT_RESULT rmf_SectionFilter_INB::TSMGR_FilterCreate(uint16_t pid, PDT_PidChannelType pidType, unsigned long parserHandle, uint32_t filterID, rmf_FilterSession_t** pFilterSession)
{
    NEXUS_PidChannelSettings pidChanSettings;
    NEXUS_PidChannelHandle pidFilterHandle = NULL;
    unsigned long msgBufSize = 0;
    RMF_SECTFLT_RESULT returnCode = RMF_SECTFLT_Success;
    rmf_FilterSession_t *availableNexusMessage;

    RDK_LOG(RDK_LOG_TRACE1, "LOG.RDK.FILTER","%s entering...\n", __FUNCTION__);
    rmf_osal_mutexAcquire(m_stFilterSessionMutex);

    if (MAX_FILTER_COUNT > m_stFilterSessionCnt)
    {
        NEXUS_PidChannel_GetDefaultSettings(&pidChanSettings);

        if (pidType == PDT_PidChannelType_AudioVideo)
        {
            RDK_LOG(RDK_LOG_TRACE1, "LOG.RDK.FILTER","%s::pidType is PDT_PidChannelType_AudioVideo\n", __FUNCTION__);
            /* requireMessageBuffer is deprecated*/
            //pidChanSettings.requireMessageBuffer = false;
            /*
            *As per nexus_pid_channel.h :
            *
            *NEXUS_PID_CHANNEL_OPEN_NOT_MESSAGE_CAPABLE will select a hardware PID channel index which is not capable of message filtering.
            *This allows an application to reserve message-capable pid channels for message filtering.
            *Otherwise, Nexus will use the pidChannelIndex if there is not a pid conflict on the same HW PID channel.
            */
            pidChanSettings.pidChannelIndex = NEXUS_PID_CHANNEL_OPEN_NOT_MESSAGE_CAPABLE;
        }
        else
        {
            msgBufSize = NEXUS_PSI_PKT_LENGTH;
            /* requireMessageBuffer is deprecated. use NEXUS_PID_CHANNEL_OPEN_MESSAGE_CAPABLE instead. */
            //pidChanSettings.requireMessageBuffer = true;

            /*\
             *As per nexus_pid_channel.h :
             *
             *NEXUS_PID_CHANNEL_OPEN_MESSAGE_CAPABLE will select a hardware PID channel index which is capable of message filtering. 
             *Only the lower 128 pid channels are capable of message filtering. 
             */
            pidChanSettings.pidChannelIndex = NEXUS_PID_CHANNEL_OPEN_MESSAGE_CAPABLE;
        }

        availableNexusMessage = GetAvailableNexusMessage();
        if (NULL != availableNexusMessage)
        {
            // Send the filter session pointer back to the caller
            *pFilterSession = availableNexusMessage;

#if (NEXUS_PLATFORM == 97425)
/* MOT7425-7248: 6 Tuner support. Add software rate-smoothing. Loop back Tuner 0 through Recpump & Playpump */
            if((bSWRateSmooth) && (SW_RS_LOOPBACK_PARSER_BAND == (int)parserHandle))
            {
                NEXUS_PlaypumpOpenPidChannelSettings playpumpPidChanSettings;

                NEXUS_Playpump_GetDefaultOpenPidChannelSettings(&playpumpPidChanSettings);		
            	playpumpPidChanSettings.pidSettings.pidChannelIndex= pidChanSettings.pidChannelIndex;
                pidFilterHandle = NEXUS_Playpump_OpenPidChannel(NEXUS_Playpump_RetrieveHandle(SW_RS_LOOPBACK_PLAYPUMP_INDEX), pid, &playpumpPidChanSettings);
            }
            else
            {
#endif    

            pidFilterHandle = NEXUS_PidChannel_Open((NEXUS_ParserBand)parserHandle, pid, &pidChanSettings);
            RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER","%s: NEXUS_PidChannel_Open 0x%p %d(0x%04x)\n",
                      __FUNCTION__, pidFilterHandle, (int)parserHandle, pid);
#if (NEXUS_PLATFORM == 97425)
            }
#endif

            if ((NULL != pidFilterHandle) )
            {
                // Load the filter session
                availableNexusMessage->RefCnt++;
                availableNexusMessage->parserHandle = parserHandle;
                availableNexusMessage->pidFilterHandle = pidFilterHandle;
                availableNexusMessage->pid = pid;
                availableNexusMessage->pidType = pidType;
                availableNexusMessage->filterID = filterID;
                availableNexusMessage->ulMsgBufSize = msgBufSize;

                m_stFilterSessionCnt++;
                availableNexusMessage->filterState = VALID_FILTER;
            }
            else
            {
                RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER", "%s: NEXUS_PidChannel_Open ret NULL \n", __FUNCTION__);
                returnCode = RMF_SECTFLT_NoFilterRes;
            }
        }
        else
        {
            RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER", "%s: No Available Filters\n", __FUNCTION__);
            returnCode = RMF_SECTFLT_NoFilterRes;
        }
    }
    else
    {
        RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER","%s::No more filter resource available.....\n", __FUNCTION__);
        returnCode = RMF_SECTFLT_NoFilterRes;
    }

    rmf_osal_mutexRelease(m_stFilterSessionMutex);

    RDK_LOG(RDK_LOG_TRACE1, "LOG.RDK.FILTER","%s end \n",__FUNCTION__);

    return returnCode;
}


/********************************************************************************************/
// Caller is already acquired the filter resource and got the filter handle (hPsiFilter)
// Now requesting a section filter resource on top of that.....
// No need of book keeping the filter handles.
/********************************************************************************************/
RMF_SECTFLT_RESULT rmf_SectionFilter_INB::TSMGR_SectionFilterCreate(rmf_FilterSession_t* nexusMessageInfo)
{
    NEXUS_MessageSettings msgSettings;
    RMF_SECTFLT_RESULT returnCode = RMF_SECTFLT_Success;

    RDK_LOG(RDK_LOG_TRACE1, "LOG.RDK.FILTER","%s begin \n",__FUNCTION__);

    if(nexusMessageInfo != NULL)
    {
        NEXUS_Message_GetSettings(nexusMessageInfo->sectionFilterHandle, &msgSettings);

        context_struct_t* context = (context_struct_t*)msgSettings.dataReady.context;

        // Add section filter handle to data callback context so that callback does not have to
        // lock mutex to look up the handle  (forbidden in Nexus documentation)
        context->filterID = nexusMessageInfo->filterID;
        context->parserHandle = nexusMessageInfo->parserHandle;
        context->pidFilterHandle = nexusMessageInfo->pidFilterHandle;
        context->sectionFilterHandle = nexusMessageInfo->sectionFilterHandle;
        NEXUS_Message_SetSettings(nexusMessageInfo->sectionFilterHandle, &msgSettings);
    }
    else
    {
        RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER","%s: NEXUS_Message_Open FAILED\n", __FUNCTION__);
        returnCode = RMF_SECTFLT_Error;
    }

    RDK_LOG(RDK_LOG_TRACE1, "LOG.RDK.FILTER","%s end \n",__FUNCTION__);
    return returnCode;

}

RMF_SECTFLT_RESULT rmf_SectionFilter_INB::TSMGR_SectionFilterStart(rmf_FilterSession_t *pFilterSession, const rmf_FilterSpec* filterSpec)
{
    NEXUS_Error nexusApiResult;
    NEXUS_MessageStartSettings startSettings;
    RMF_SECTFLT_RESULT returnCode = RMF_SECTFLT_Success;
    int i;

    RDK_LOG(RDK_LOG_TRACE2, "LOG.RDK.FILTER", "%s begin \n", __FUNCTION__);

    NEXUS_Message_GetDefaultStartSettings(pFilterSession->sectionFilterHandle, &startSettings);
    
    startSettings.pidChannel    = pFilterSession->pidFilterHandle;
    startSettings.buffer        = (void *)pFilterSession->ulMsgBufAddr;
    startSettings.bufferSize    = pFilterSession->ulMsgBufSize;

    if (filterSpec->pos.length > 0)
    {
        pFilterSession->tableId = filterSpec->pos.vals[0];

        RDK_LOG(RDK_LOG_TRACE1, "LOG.RDK.FILTER", "%s tableId:0x%02x, len=%d \n", __FUNCTION__,
                filterSpec->pos.vals[0], filterSpec->pos.length
                );
    }
    else
    {
        pFilterSession->tableId = 0; //a value other than 0x02
    }

    for (i = 0; i < filterSpec->pos.length; i++)
    {
        if(i < 2) // Nexus skips byte 2
        {
            startSettings.filter.coefficient[i] = (filterSpec->pos.vals[i] & filterSpec->pos.mask[i]);
            startSettings.filter.mask[i] = ~(filterSpec->pos.mask[i]);
        }
        else if(i == 2)
        {
            continue;
        }
        else
        {
            startSettings.filter.coefficient[i-1] = (filterSpec->pos.vals[i] & filterSpec->pos.mask[i]);
            startSettings.filter.mask[i-1] = ~(filterSpec->pos.mask[i]);
        }
    }

    for (i = 0; i < filterSpec->neg.length; i++)
    {
        if(i < 2) // Nexus skips byte 2
        {
            startSettings.filter.coefficient[i] |= (filterSpec->neg.vals[i] & filterSpec->neg.mask[i]);
            startSettings.filter.exclusion[i] = ~(filterSpec->neg.mask[i]);
        }
        else if(i == 2)
        {
            continue;
        }
        else
        {
            startSettings.filter.coefficient[i-1] |= (filterSpec->neg.vals[i] & filterSpec->neg.mask[i]);
            startSettings.filter.exclusion[i-1] = ~(filterSpec->neg.mask[i]);
        }
    }
    nexusApiResult = NEXUS_Message_Start(pFilterSession->sectionFilterHandle, &startSettings);

    if(nexusApiResult != NEXUS_SUCCESS)
    {
        RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER", "%s: NEXUS_Message_Start FAILED. nexusApiResult = %d\n",
                __FUNCTION__, nexusApiResult);
        returnCode = RMF_SECTFLT_Error;
    }
    else
    {
        RDK_LOG(RDK_LOG_TRACE2, "LOG.RDK.FILTER", "NEXUS_Message_Start SUCCEEDED\n");
        //NEXUS_StartCallbacks(pFilterSession->sectionFilterHandle);
    }

    RDK_LOG(RDK_LOG_TRACE2, "LOG.RDK.FILTER", "%s end \n", __FUNCTION__);

    return returnCode;

}

RMF_SECTFLT_RESULT rmf_SectionFilter_INB::TSMGR_SectionFilterStop(unsigned long parserHandle, NEXUS_MessageHandle sectionFilterHandle)
{
    RDK_LOG(RDK_LOG_INFO, "LOG.RDK.FILTER","%s: begin \n", __FUNCTION__);

    if (sectionFilterHandle != NULL)
    {
        NEXUS_Message_Stop(sectionFilterHandle);
        //NEXUS_StopCallbacks(sectionFilterHandle);
    } else {
        return RMF_SECTFLT_InvalidParams;
    }

    RDK_LOG(RDK_LOG_TRACE3, "LOG.RDK.FILTER","%s end \n", __FUNCTION__);

    return RMF_SECTFLT_Success;

}

RMF_SECTFLT_RESULT rmf_SectionFilter_INB::TSMGR_SectionFilterRelease(NEXUS_MessageHandle sectionFilterHandle)
{
    RDK_LOG(RDK_LOG_TRACE3, "LOG.RDK.FILTER","%s: begin \n", __FUNCTION__);

    NEXUS_MessageSettings msgSettings;

    NEXUS_Message_GetSettings(sectionFilterHandle, &msgSettings);
    context_struct_t* context = (context_struct_t*)msgSettings.dataReady.context;
    context->filterID = 0;
    msgSettings.dataReady.context = (void*)context;
    NEXUS_Message_SetSettings(sectionFilterHandle, &msgSettings);

    RDK_LOG(RDK_LOG_TRACE3, "LOG.RDK.FILTER", "%s: sectionFilterHandle=%p\n", __FUNCTION__, sectionFilterHandle);

    RDK_LOG(RDK_LOG_TRACE3, "LOG.RDK.FILTER","%s: end \n",__FUNCTION__);

    return RMF_SECTFLT_Success;
}

RMF_SECTFLT_RESULT rmf_SectionFilter_INB::TSMGR_FilterRelease(rmf_FilterSession_t* pInbandFilterInfo, int *pRefCnt)
{
    RMF_SECTFLT_RESULT result = RMF_SECTFLT_Success;
#ifdef RMF_SF_SINGLE_THREAD_ENABLE
    rmf_Error ret = RMF_SUCCESS;
#endif

    RDK_LOG(RDK_LOG_TRACE3, "LOG.RDK.FILTER","%s: begin \n", __FUNCTION__);

    rmf_osal_mutexAcquire(m_stFilterSessionMutex);

    if (NULL != pInbandFilterInfo)
    {
#ifdef RMF_SF_SINGLE_THREAD_ENABLE
        if ( NULL != pInbandFilterInfo->pMonitorThreadStopSem ) 
        {
            if(pInbandFilterInfo->threadStatus == SF_THREAD_ACTIVE)
            {
                rmf_osal_event_params_t event_params = {0};
                rmf_osal_event_handle_t event_handle;

    	    event_params.priority = 0xFF;

    	    RDK_LOG(RDK_LOG_DEBUG, "LOG.RDK.FILTER","<%s: %d> - Sending SHUTDOWN event to SectionMonitor thread\n", __FUNCTION__, __LINE__);
    	    ret = rmf_osal_event_create(RMF_OSAL_EVENT_CATEGORY_MPEG, RMF_INB_SF_EVENT_SHUTDOWN, &event_params, &event_handle);
    	    if(ret != RMF_SUCCESS)
    	    {
    		    RDK_LOG(
    				    RDK_LOG_ERROR,
    				    "LOG.RDK.FILTER",
    				    "<%s: %d> Could not create event handle: %x\n",
    				     __FUNCTION__, __LINE__, ret);
    		    //return ret;
    	    }
    	    else
    	    {
    		    rmf_osal_eventqueue_add(pInbandFilterInfo->sectionMonitorQueue, event_handle);
    		    RDK_LOG(RDK_LOG_DEBUG, "LOG.RDK.FILTER","<%s: %d> Waiting for SectionMonitor thread to exit\n", __FUNCTION__, __LINE__);
    		    ret = sem_wait( pInbandFilterInfo->pMonitorThreadStopSem);
    		    if (0 != ret )
    		    {
    		        RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER",
    					    "<%s: %d> - sem_wait failed.\n", __FUNCTION__, __LINE__);
    		    }
    		    pInbandFilterInfo->threadStatus = SF_THREAD_QUIT;
 
    	    }

           }
            RDK_LOG(RDK_LOG_DEBUG, "LOG.RDK.FILTER","<%s: %d> - Deleting pMonitorThreadStopSem\n", __FUNCTION__, __LINE__);
            ret = sem_destroy( pInbandFilterInfo->pMonitorThreadStopSem);
            if (0 != ret )
            {
                RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER",
                "<%s: %d> - sem_destroy failed.\n", __FUNCTION__, __LINE__);
            }

            delete pInbandFilterInfo->pMonitorThreadStopSem;
            pInbandFilterInfo->pMonitorThreadStopSem = NULL;
        

            RDK_LOG(RDK_LOG_DEBUG, "LOG.RDK.FILTER","<%s: %d> - Deleting pInbandFilterInfo->sectionMonitorQueue\n", __FUNCTION__, __LINE__);
            if(RMF_SUCCESS != rmf_osal_eventqueue_delete (pInbandFilterInfo->sectionMonitorQueue))
            {
                RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER", "<%s: %d> - unable to delete event queue.\n",
                         __FUNCTION__, __LINE__);
            }

            pInbandFilterInfo->sectionMonitorQueue = 0;
        }
#endif

        if (pInbandFilterInfo->RefCnt != 0)
        {
            pInbandFilterInfo->RefCnt--;
        }

        RDK_LOG(RDK_LOG_TRACE3, "LOG.RDK.FILTER", "%s: filter id is %d, refcnt is %d\n",
                  __FUNCTION__, pInbandFilterInfo->filterID, pInbandFilterInfo->RefCnt);

        if (NULL != pRefCnt) // NULL is a valid argument only. So do not return error if it is NULL. 
        {
            *pRefCnt = pInbandFilterInfo->RefCnt;
        }

        RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER","%s: NEXUS_PidChannel_Close 0x%p refcnt=%d\n",
                  __FUNCTION__, pInbandFilterInfo->pidFilterHandle, pInbandFilterInfo->RefCnt);
                  
        if (0 == pInbandFilterInfo->RefCnt)
        {
            if (NULL != pInbandFilterInfo->pidFilterHandle)
            {

#if (NEXUS_PLATFORM == 97425)
/* MOT7425-7248: 6 Tuner support. Add software rate-smoothing. Loop back Tuner 0 through Recpump & Playpump */
                if((bSWRateSmooth) && (SW_RS_LOOPBACK_PARSER_BAND == (int)parserHandle))
                {
            		NEXUS_Playpump_ClosePidChannel(NEXUS_Playpump_RetrieveHandle(SW_RS_LOOPBACK_PLAYPUMP_INDEX), pInbandFilterInfo->pidFilterHandle);
                }
                else
                {
#endif

                NEXUS_PidChannel_Close(pInbandFilterInfo->pidFilterHandle);
#if (NEXUS_PLATFORM == 97425)
                }
#endif
                          
                // Mark Nexus message filter available
                pInbandFilterInfo->filterState = INVALID_FILTER;
                m_stFilterSessionCnt--;
            }
        }
    }
    else
    {
        result = RMF_SECTFLT_Error;
    }

    rmf_osal_mutexRelease(m_stFilterSessionMutex);

    RDK_LOG(RDK_LOG_TRACE3, "LOG.RDK.FILTER","%s: end \n",__FUNCTION__);

    return result;

}


RMF_SECTFLT_RESULT rmf_SectionFilter_INB::Create(uint16_t pid, uint8_t* pos_mask, uint8_t* pos_value, uint16_t pos_length, uint8_t* neg_mask, uint8_t* neg_value, uint16_t neg_length, void **pSectionFilterInfo, uint32_t* pFilterID)
{
	// Check the availability of filter -> This will be done in TSMGR_FilterCreate function
	// Check whether Section filter resource is available or not
	// Looks like there is no limit in section filter. If we need to put a limit, we can
	// add restrictions in mpeos layer itself.
	RMF_SECTFLT_RESULT returnCode	=  RMF_SECTFLT_Error;

	//Inband_SectionFilter_Info_t 	*pInbandFilterInfo 	= 	NULL;
        rmf_FilterSession_t                     *pFilterSession = NULL;

	rmf_osal_mutexAcquire(m_sfMutex);
	RDK_LOG(RDK_LOG_INFO, "LOG.RDK.FILTER","%s::begin\n", __FUNCTION__);
	// 	printSessionList();
#if (NEXUS_PLATFORM == 97425)
    if(!bInitRSTest)
    {
		//bSWRateSmooth = vl_env_get_bool(false, "FEATURE.SW_RATE_SMOOTH");
		const char * value = rmf_osal_envGet("FEATURE.SW_RATE_SMOOTH");
		  if (NULL != value)
               {
                    bSWRateSmooth = (strcasecmp(value, "TRUE") == 0);
               }
        bInitRSTest = true;
    }
#endif

	// check whether a filter is already created with the same pid and pid channel type.
        rmf_osal_mutexAcquire(m_stFilterSessionMutex);

        get_Filter_SessionObj_WithPid(parserHandle,
                                pid,
				PDT_PidChannelType_ePsi,
                                &pFilterSession);
                
	rmf_osal_mutexRelease(m_stFilterSessionMutex);

        if(pFilterSession != NULL)
	{
		// Filter is not initialized...Return
		rmf_osal_mutexRelease(m_sfMutex);
		RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER","%s:: filter already available. Cannot create another one\n", __FUNCTION__);
		return RMF_SECTFLT_Error;
	}

	*pFilterID = getNextSectionID();  
	returnCode = TSMGR_FilterCreate(pid, PDT_PidChannelType_ePsi, parserHandle, *pFilterID, &pFilterSession);
	if (RMF_SECTFLT_Success != returnCode)
	{
		rmf_osal_mutexRelease(m_sfMutex);
		RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER","%s:: (RMF_SECTFLT_Success != TSMGR_FilterCreate) released mutex, pInbandFilterInfo->dmxHdl : 0x%x, pid : 0x%x\n", __FUNCTION__, parserHandle, pid);
		return RMF_SECTFLT_Error;
	}

	returnCode = TSMGR_SectionFilterCreate(pFilterSession);
	if (RMF_SECTFLT_Success != returnCode)
	{
		TSMGR_FilterRelease(pFilterSession, NULL);

		rmf_osal_mutexRelease(m_sfMutex);
		RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER","%s:: (RMF_SECTFLT_Success != TSMGR_SectionFilterCreate) released mutex\n", __FUNCTION__);
		return RMF_SECTFLT_Error;
	}


    // Note: Here we only copy pointer of mask/vals, instead of malloc/memcpy
    // It should be ok even mask/vals is in rmf_InbSiMgr::set_filter()'s stack, since Start(),
    // which uses this configuration to set Nexus filter, will be called under same caller-tree,
    // so those pointer/data is still valid.
    pFilterSession->filterSpec.pos.length = pos_length;
    pFilterSession->filterSpec.pos.mask = pos_mask;
    pFilterSession->filterSpec.pos.vals = pos_value;
    pFilterSession->filterSpec.neg.length = neg_length;
    pFilterSession->filterSpec.neg.mask = neg_mask;
    pFilterSession->filterSpec.neg.vals = neg_value;

#ifdef RMF_SF_SINGLE_THREAD_ENABLE
        int err = 0;
	pFilterSession->pMonitorThreadStopSem = new sem_t;

	if(NULL == pFilterSession->pMonitorThreadStopSem)
	{
		TSMGR_SectionFilterRelease(pFilterSession->sectionFilterHandle);
		TSMGR_FilterRelease(pFilterSession, NULL);
		rmf_osal_mutexRelease(m_sfMutex);
		return RMF_SECTFLT_Error;
	}

	err  = sem_init( pFilterSession->pMonitorThreadStopSem, 0 , 0);
	if (0 != err )
	{
		RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER",
				"<%s: %d> - sem_init failed.\n", __FUNCTION__, __LINE__);
		
              TSMGR_SectionFilterRelease(pFilterSession->sectionFilterHandle);
		TSMGR_FilterRelease(pFilterSession, NULL);
		if(NULL != pFilterSession->pMonitorThreadStopSem)
		{
              	delete pFilterSession->pMonitorThreadStopSem;
              	pFilterSession->pMonitorThreadStopSem = NULL;
		}
                
		rmf_osal_mutexRelease(m_sfMutex);
		return RMF_SECTFLT_Error;

	}

	if(RMF_SUCCESS != rmf_osal_eventqueue_create ((const uint8_t* )"SectionMonitor",
				&pFilterSession->sectionMonitorQueue))
	{
		RDK_LOG(RDK_LOG_ERROR,
				"LOG.RDK.FILTER",
				"<%s: %d> - unable to create event queue,... terminating thread.\n",
				 __FUNCTION__, __LINE__);

                sem_destroy( pFilterSession->pMonitorThreadStopSem);
                TSMGR_SectionFilterRelease(pFilterSession->sectionFilterHandle);
                TSMGR_FilterRelease(pFilterSession, NULL);
                if(NULL != pFilterSession->pMonitorThreadStopSem)
                {
                    delete pFilterSession->pMonitorThreadStopSem;
                    pFilterSession->pMonitorThreadStopSem = NULL;
                }
                
                rmf_osal_mutexRelease(m_sfMutex);
		return RMF_SECTFLT_Error;
	}


	/* threadargs is deleted in SectionFilterThreadFn*/
        rmf_osal_ThreadId sfThread;
	rmf_thread_args_t* threadargs = new rmf_thread_args_t;
    rmf_Error ret;
	if (NULL != threadargs)
	{
		threadargs->pSectionFilter = (void*)this;
		threadargs->pData = (void*)pFilterSession;

 	        RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER",
				"<%s: %d> - Creating thread.\n", __FUNCTION__, __LINE__);

		ret = rmf_osal_threadCreate( SectionFilterThreadFn, (void*)threadargs, RMF_OSAL_THREAD_PRIOR_DFLT, RMF_OSAL_THREAD_STACK_SIZE, &sfThread, "Section_Filter_Thread" );

        if ( RMF_SUCCESS != ret ) {
            RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER","<%s: %d> - SectionFilterThreadFn can't be created. FATAL ERROR!!!.\n", __FUNCTION__, __LINE__);
        } else {
            pFilterSession->threadStatus = SF_THREAD_ACTIVE;
        }
	}
	if ( NULL == threadargs || RMF_SUCCESS != ret ) {
                RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER","<%s: %d> - threadargs couldnot be created.\n", __FUNCTION__, __LINE__);

                rmf_osal_eventqueue_delete (pFilterSession->sectionMonitorQueue);
                pFilterSession->sectionMonitorQueue = 0;
                sem_destroy( pFilterSession->pMonitorThreadStopSem);
                TSMGR_SectionFilterRelease(pFilterSession->sectionFilterHandle);
                TSMGR_FilterRelease(pFilterSession, NULL);
                if(NULL != pFilterSession->pMonitorThreadStopSem)
                {
                    delete pFilterSession->pMonitorThreadStopSem;
                    pFilterSession->pMonitorThreadStopSem = NULL;
                }
                
		rmf_osal_mutexRelease(m_sfMutex);
		return RMF_SECTFLT_Error;
	}
#endif
	RDK_LOG(RDK_LOG_INFO, "LOG.RDK.FILTER","%s:: *pFilterID = %d, handle = %x\n", __FUNCTION__, *pFilterID, (uint32_t)pFilterSession->sectionFilterHandle);

	//rmf_osal_mutexAcquire(m_sectionHandleMapMutex);
	//m_sectionHandleMap->Insert((rmf_SymbolMapKey)(pInbandFilterInfo->hPsiSectionFilter), (void*)*pFilterID);
	//rmf_osal_mutexRelease(m_sectionHandleMapMutex);
	m_filterState = SECTFLT_CREATED;

	rmf_osal_mutexRelease(m_sfMutex);
	RDK_LOG(RDK_LOG_INFO, "LOG.RDK.FILTER","%s:: returned RMF_SECTFLT_Success\n", __FUNCTION__);

	// 	printSessionList();
	RDK_LOG(RDK_LOG_TRACE1, "LOG.RDK.FILTER","\n>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>\n");
	RDK_LOG(RDK_LOG_TRACE1, "LOG.RDK.FILTER","%s:: setting section filter Tuner id = %d dmxHdl = 0x%x\n", __FUNCTION__,m_SourceID, parserHandle);
	RDK_LOG(RDK_LOG_TRACE1, "LOG.RDK.FILTER",">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>\n");

#ifdef CAPMT_ENABLED	
	if( (pos_length > 0) && (pos_value != NULL) && (pos_value[0] == 0x02) )
	{
		RDK_LOG(RDK_LOG_TRACE1, "LOG.RDK.FILTER",">>>>>>>>>>>>> %s:: Pmt table Is found \n", __FUNCTION__);
		m_PmtCaptureTunerId = m_SourceID;
		// 		vlg_PmtCaptuteFilterId = *pFilterID;
		m_PmtCapturePidId = pid;
	}
#endif

	*pSectionFilterInfo = (void*)pFilterSession;
	RDK_LOG(RDK_LOG_INFO, "LOG.RDK.FILTER","%s::end\n", __FUNCTION__);

	return RMF_SECTFLT_Success;	
}


RMF_SECTFLT_RESULT rmf_SectionFilter_INB::Start(uint32_t filterID)
{
	RMF_SECTFLT_RESULT 	returnCode 	= 	RMF_SECTFLT_Success;

	rmf_osal_mutexAcquire(m_sfMutex);

	if (m_filterState != SECTFLT_CREATED)
	{
		rmf_osal_mutexRelease(m_sfMutex);
		RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER","%s:: filter not SECTFLT_CREATED\n", __FUNCTION__);
		return RMF_SECTFLT_NotReady;
	}


	rmf_FilterSession_t *pInbandFilterInfo = (rmf_FilterSession_t*)get_section_filter_info(filterID);
	if (NULL == pInbandFilterInfo)
	{
		rmf_osal_mutexRelease(m_sfMutex);
		RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER","%s:: pInbandFilterInfo is NULL\n", __FUNCTION__);
		return RMF_SECTFLT_NotReady;
	}

    if ( pInbandFilterInfo->filterState != FILTER_STARTED ) {
    	RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER","%s:: filter %d, filterState=%d \n", __FUNCTION__, filterID, pInbandFilterInfo->filterState);

    	returnCode = TSMGR_SectionFilterStart(pInbandFilterInfo, &(pInbandFilterInfo->filterSpec));
    	if (RMF_SECTFLT_Success != returnCode)
    	{
    		rmf_osal_mutexRelease(m_sfMutex);
    		TSMGR_FilterRelease(pInbandFilterInfo, NULL);
    		RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER","%s:: pInbandFilterInfo is NULL\n", __FUNCTION__);
    		return RMF_SECTFLT_NotReady;
    	}

        pInbandFilterInfo->filterState = FILTER_STARTED;
    }
    //m_filterState = SECTFLT_STARTED;

	rmf_osal_mutexRelease(m_sfMutex);
	// 	RDK_LOG(RDK_LOG_INFO, "LOG.RDK.FILTER","Inband_SECTFLT_Start::  end\n");

	return returnCode;
}


RMF_SECTFLT_RESULT rmf_SectionFilter_INB::Stop(uint32_t filterID)
{
	// ASSERT:  m_sfMutex is held by the caller

	RMF_SECTFLT_RESULT 	returnCode 	= 	RMF_SECTFLT_Success;
	if (m_filterState != SECTFLT_CREATED /*SECTFLT_STARTED*/)
	{
		return RMF_SECTFLT_NotReady;
	}

	//Inband_SectionFilter_Info_t *pInbandFilterInfo = (Inband_SectionFilter_Info_t*)m_pSectionFilterInfo;
	rmf_FilterSession_t *pInbandFilterInfo = (rmf_FilterSession_t*)get_section_filter_info(filterID);

	if (NULL == pInbandFilterInfo)
	{
		return RMF_SECTFLT_NotReady;
	}

    if ( pInbandFilterInfo->filterState == FILTER_STARTED ) {
    	RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER","%s:: filter %d, filterState=%d \n", __FUNCTION__, filterID, pInbandFilterInfo->filterState);
    	returnCode = TSMGR_SectionFilterStop(pInbandFilterInfo->parserHandle, pInbandFilterInfo->sectionFilterHandle);
        pInbandFilterInfo->filterState = VALID_FILTER;
    } else {
		returnCode = RMF_SECTFLT_NotStarted;
	}

	return returnCode;
}


RMF_SECTFLT_RESULT rmf_SectionFilter_INB::Release(uint32_t filterID){
	RMF_SECTFLT_RESULT 	returnCode 	= 	RMF_SECTFLT_Success;
	//int					nRefCnt		= 0;
        rmf_FilterSession_t                     *pFilterSession = NULL;

	rmf_osal_mutexAcquire(m_sfMutex);
	RDK_LOG(RDK_LOG_INFO, "LOG.RDK.FILTER","%s::begin\n", __FUNCTION__);
	// 	printSessionList();
	
	pFilterSession = (rmf_FilterSession_t*)get_section_filter_info(filterID);

	if( (pFilterSession != NULL) &&  (pFilterSession->filterState != INVALID_FILTER ))
	{
		//rmf_osal_mutexAcquire(m_sectionHandleMapMutex);
		//m_sectionHandleMap->Remove((rmf_SymbolMapKey)pFilterSession->sectionFilterHandle);
		//rmf_osal_mutexRelease(m_sectionHandleMapMutex);

		Stop(filterID);
		RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.FILTER","%s:: filter %d\n", __FUNCTION__, filterID);
		TSMGR_SectionFilterRelease(pFilterSession->sectionFilterHandle);
		TSMGR_FilterRelease(pFilterSession, NULL);
		returnCode = RMF_SECTFLT_Success;
	}
	RDK_LOG(RDK_LOG_INFO, "LOG.RDK.FILTER","%s::here\n", __FUNCTION__);
	//m_filterState = SECTFLT_RELEASED;

	RDK_LOG(RDK_LOG_INFO, "LOG.RDK.FILTER","%s:: m_filterState: 0x%x\n", __FUNCTION__, m_filterState);

	rmf_osal_mutexRelease(m_sfMutex);
	RDK_LOG(RDK_LOG_INFO, "LOG.RDK.FILTER","%s::end\n", __FUNCTION__);
	// 	printSessionList();
	return returnCode;
}


RMF_SECTFLT_RESULT rmf_SectionFilter_INB::Read(uint32_t filterID, rmf_Section_Data** ppSectionData)
{
	RMF_SECTFLT_RESULT 	returnCode 	= 	RMF_SECTFLT_Success;
	rmf_sf_SectionRequest_t*         pFilter_Request         = NULL;
        pFilter_Request = (rmf_sf_SectionRequest_t*)m_sectionRequests->Lookup((rmf_SymbolMapKey)filterID);

        // If we can not find the request, or if it has already been
        // canceled, let the caller know
        if (pFilter_Request == NULL || pFilter_Request->state == RMF_SF_REQSTATE_CANCELLED)
        {
                return RMF_SECTFLT_InvalidParams;
        }

#if 0
    *ppSectionData = (rmf_Section_Data*)queue_pop_head(pFilter_Request->sectionsDataQueue);
#else
    *ppSectionData = (rmf_Section_Data*)rmf_queue_pop_head(pFilter_Request->m_SectionsDataQueue);
#endif
	return returnCode;
}
