/* Copyright (c) 2018, Google Inc.
 *
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
 * OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE. */

#include <gtest/gtest.h>
#include <stdint.h>

#include "sike.h"
#include "fpx.h"
#include "../../crypto/test/abi_test.h"

TEST(SIKE, RoundTrip) {
  uint8_t sk[SIKE_PRV_BYTESZ] = {0};
  uint8_t pk[SIKE_PUB_BYTESZ] = {0};
  uint8_t ct[SIKE_CT_BYTESZ] = {0};
  uint8_t ss_enc[SIKE_SS_BYTESZ] = {0};
  uint8_t ss_dec[SIKE_SS_BYTESZ] = {0};

  for (size_t i = 0; i < 30; i++) {
    EXPECT_EQ(SIKE_keypair(sk, pk), 1);
    SIKE_encaps(ss_enc, ct, pk);
    SIKE_decaps(ss_dec, ct, pk, sk);

    EXPECT_EQ(memcmp(ss_enc, ss_dec, SIKE_SS_BYTESZ), 0);
  }
}

TEST(SIKE, Decapsulation) {
  const uint8_t sk[SIKE_PRV_BYTESZ] = {
      0xB1, 0xFD, 0x34, 0x42, 0xDB, 0x02, 0xBC, 0x9D, 0x4C, 0xD0,
      0x72, 0x34, 0x4D, 0xBD, 0x06, 0xDF, 0x1C, 0x7D, 0x0A, 0x88,
      0xB2, 0x50, 0xC4, 0xF6, 0xAE, 0xE8, 0x25, 0x01};

  const uint8_t pk[SIKE_PUB_BYTESZ] = {
      0x6D, 0x8D, 0xF5, 0x7B, 0xCD, 0x47, 0xCA, 0xCB, 0x7A, 0x38, 0xB7, 0xA6,
      0x90, 0xB7, 0x37, 0x03, 0xD4, 0x6F, 0x27, 0x73, 0x74, 0x17, 0x5A, 0xA4,
      0x0D, 0xC6, 0x81, 0xAD, 0xDB, 0xF7, 0x18, 0xB2, 0x3C, 0x30, 0xCF, 0xAA,
      0x08, 0x11, 0x91, 0xCC, 0x27, 0x4E, 0xF1, 0xA6, 0xB7, 0xDA, 0xD2, 0xCF,
      0x99, 0x7F, 0xF7, 0xE1, 0xD0, 0xCE, 0x00, 0xD2, 0x4B, 0xA4, 0x33, 0xB4,
      0x87, 0x01, 0x3F, 0x02, 0xF7, 0xF9, 0xDE, 0xC3, 0x60, 0x62, 0xDA, 0x3F,
      0x74, 0xA9, 0x44, 0xBE, 0x19, 0xD5, 0x03, 0x2A, 0x79, 0x8C, 0xA7, 0xFF,
      0xEA, 0xB3, 0xBB, 0xB5, 0xD4, 0x1D, 0x8F, 0x92, 0xCE, 0x62, 0x6E, 0x99,
      0x24, 0xD7, 0x57, 0xFA, 0xCD, 0xB6, 0xE2, 0x8E, 0xFD, 0x22, 0x0E, 0x31,
      0x21, 0x01, 0x8D, 0x79, 0xF8, 0x3E, 0x27, 0xEC, 0x43, 0x40, 0xDB, 0x82,
      0xE5, 0xEB, 0x6C, 0x97, 0x66, 0x29, 0x15, 0x68, 0xB7, 0x4D, 0x84, 0xD1,
      0x8A, 0x0B, 0x12, 0x36, 0x2C, 0x0C, 0x0A, 0x6E, 0x4E, 0xDE, 0xA5, 0x8A,
      0xDE, 0x77, 0xDD, 0x70, 0x49, 0x73, 0xAC, 0x27, 0x6D, 0x8D, 0x25, 0x9A,
      0xE4, 0x25, 0xE8, 0x95, 0x8F, 0xFE, 0x90, 0x3B, 0x00, 0x69, 0x20, 0xE8,
      0x7C, 0xA5, 0xF5, 0x79, 0xC0, 0x61, 0x51, 0x91, 0x35, 0x25, 0x3F, 0x17,
      0x2F, 0x70, 0x73, 0xF0, 0x89, 0xB5, 0xC8, 0x25, 0xB8, 0xE5, 0x7E, 0x34,
      0xDD, 0x11, 0xE5, 0xD6, 0xC3, 0xD5, 0x29, 0x89, 0xC6, 0x2C, 0x99, 0x53,
      0x1D, 0x2C, 0x77, 0xB0, 0xB6, 0xA1, 0xBD, 0x79, 0xFB, 0x4A, 0xC2, 0x48,
      0x4C, 0x62, 0x51, 0x00, 0xE3, 0x91, 0x2A, 0xCB, 0x84, 0x03, 0x5D, 0x2D,
      0xC8, 0x33, 0xE9, 0x14, 0xBF, 0x74, 0x21, 0xBC, 0xF4, 0x76, 0xE5, 0x42,
      0xB8, 0xBD, 0xE2, 0xE7, 0x20, 0x95, 0x54, 0xF2, 0xED, 0xC0, 0x79, 0x38,
      0x1E, 0xD2, 0xEA, 0x1A, 0x63, 0x85, 0xE7, 0x3A, 0xDA, 0xAD, 0xAB, 0x1B,
      0x1E, 0x19, 0x9E, 0x73, 0xD0, 0x10, 0x2E, 0x38, 0xAC, 0x8B, 0x00, 0x6A,
      0x30, 0x2C, 0x3D, 0x70, 0x8E, 0x39, 0x6D, 0xC0, 0x12, 0x61, 0x7D, 0x2A,
      0x0A, 0x04, 0x95, 0x8E, 0x09, 0x3C, 0x7B, 0xEC, 0x2E, 0xBC, 0xE8, 0xE8,
      0xE8, 0x37, 0x29, 0xC4, 0x7E, 0x76, 0x48, 0xB9, 0x3B, 0x72, 0xE5, 0x99,
      0x9B, 0xF9, 0xE3, 0x99, 0x72, 0x3F, 0x35, 0x29, 0x85, 0xE0, 0xC8, 0xBF,
      0xB1, 0x6B, 0xB1, 0x6E, 0x72, 0x00};

  const uint8_t ct[SIKE_CT_BYTESZ] = {
      0xFF, 0xEB, 0xEF, 0x4A, 0xC0, 0x57, 0x0F, 0x26, 0xAC, 0x76, 0xA8, 0xB0,
      0xA3, 0x5D, 0x9C, 0xD9, 0x25, 0xD1, 0x7F, 0x92, 0x5D, 0xF4, 0x23, 0x34,
      0xC3, 0x03, 0x10, 0xE1, 0xB0, 0x24, 0x9B, 0x44, 0x58, 0x26, 0x13, 0x56,
      0x83, 0x43, 0x72, 0x69, 0x28, 0x0D, 0x55, 0x07, 0x1F, 0xDB, 0xC0, 0x23,
      0x34, 0x83, 0x1A, 0x09, 0x9B, 0x80, 0x00, 0x64, 0x56, 0xDC, 0x79, 0x7A,
      0xD2, 0xCE, 0x23, 0xC9, 0x72, 0x27, 0xFC, 0x8D, 0xAB, 0xBF, 0xD3, 0x17,
      0xF6, 0x91, 0x7B, 0x15, 0x93, 0x83, 0x8A, 0x4F, 0x6C, 0xCA, 0x4A, 0x94,
      0xDA, 0xC7, 0x9D, 0xB6, 0xD6, 0xBA, 0xBD, 0x81, 0x9A, 0x78, 0xE5, 0xE5,
      0xBE, 0x17, 0xBC, 0xCB, 0xC8, 0x23, 0x80, 0x5F, 0x75, 0xF8, 0xDB, 0x51,
      0x55, 0x00, 0x25, 0x33, 0x52, 0x64, 0xB2, 0xD6, 0xD8, 0x9A, 0x2A, 0x9E,
      0x29, 0x99, 0x13, 0x33, 0xE2, 0xA7, 0x98, 0xAC, 0xD7, 0x79, 0x5C, 0x2F,
      0xBA, 0x07, 0xC3, 0x03, 0x37, 0xD6, 0xE6, 0xB5, 0xA1, 0xF5, 0x29, 0xB6,
      0xF6, 0xC0, 0x5C, 0x44, 0x68, 0x2B, 0x0B, 0xF5, 0x00, 0x01, 0x44, 0xD5,
      0xCC, 0x23, 0xB5, 0x27, 0x4F, 0xCA, 0xB4, 0x05, 0x01, 0xF9, 0xD4, 0x41,
      0xE0, 0xE1, 0x1E, 0xCF, 0xA9, 0xBC, 0x79, 0xD7, 0xD5, 0xF5, 0x3C, 0xE6,
      0x93, 0xF4, 0x6C, 0x84, 0x5A, 0x2C, 0x4B, 0xE4, 0x91, 0xB2, 0xB2, 0xB8,
      0xAD, 0x74, 0x9A, 0x69, 0x79, 0x4C, 0x84, 0xB7, 0xBF, 0xF1, 0x68, 0x4B,
      0xAE, 0x0F, 0x7F, 0x45, 0x3B, 0x18, 0x3F, 0xFA, 0x00, 0x48, 0xE0, 0x3A,
      0xE2, 0xC0, 0xAE, 0x00, 0xCE, 0x90, 0x28, 0xA4, 0x1B, 0xBE, 0xCA, 0x0C,
      0x21, 0x29, 0x64, 0x30, 0x5E, 0x35, 0xAD, 0xFD, 0x83, 0x47, 0x40, 0x6D,
      0x15, 0x56, 0xFC, 0xF8, 0x5F, 0xAB, 0x81, 0xFE, 0x6B, 0xE9, 0x6B, 0xED,
      0x27, 0x35, 0x7C, 0xD8, 0x2C, 0xD4, 0xF2, 0x11, 0xE6, 0xAF, 0xDF, 0xB8,
      0x91, 0x96, 0xEB, 0xF7, 0x4C, 0x8D, 0x70, 0x77, 0x90, 0x81, 0x00, 0x09,
      0x19, 0x27, 0x8A, 0x9E, 0xB6, 0x1A, 0xE9, 0xAC, 0x6C, 0xC9, 0xF8, 0xEA,
      0xA2, 0x34, 0xB8, 0xAC, 0xB3, 0xB3, 0x68, 0xA1, 0xB7, 0x29, 0x55, 0xCA,
      0x40, 0x23, 0x92, 0x5C, 0x0C, 0x79, 0x6B, 0xD6, 0x9F, 0x5B, 0xD2, 0xE6,
      0xAE, 0x04, 0xCB, 0xEC, 0xC7, 0x88, 0x18, 0xDB, 0x7A, 0xE6, 0xD6, 0xC9,
      0x39, 0xFD, 0x93, 0x9B, 0xC8, 0x01, 0x6F, 0x3E, 0x6C, 0x90, 0x3E, 0x73,
      0x76, 0x99, 0x7C, 0x48, 0xDA, 0x68, 0x48, 0x80, 0x2B, 0x63};

  const uint8_t ss_exp[SIKE_SS_BYTESZ] = {0xA1, 0xF9, 0x5A, 0x67, 0xB9, 0x3D,
                                          0x1E, 0x72, 0xE8, 0xC5, 0x71, 0xF1,
                                          0x4C, 0xB2, 0xAA, 0x6D};

  uint8_t ss_dec[SIKE_SS_BYTESZ] = {0};
  SIKE_decaps(ss_dec, ct, pk, sk);
  EXPECT_EQ(memcmp(ss_dec, ss_exp, sizeof(ss_exp)), 0);
}

// SIKE_encaps and SIKE_keypair doesn't return zeros.
TEST(SIKE, NonZero) {
  uint8_t sk[SIKE_PRV_BYTESZ] = {0};
  uint8_t pk[SIKE_PUB_BYTESZ] = {0};
  uint8_t ct[SIKE_CT_BYTESZ] = {0};
  uint8_t ss[SIKE_SS_BYTESZ] = {0};

  // Check secret and public key returned by SIKE_keypair
  EXPECT_EQ(SIKE_keypair(sk, pk), 1);
  uint8_t tmp = 0;
  for (size_t i = 0; i < sizeof(sk); i++) {
    tmp |= sk[i];
  }
  EXPECT_NE(tmp, 0);

  tmp = 0;
  for (size_t i = 0; i < sizeof(pk); i++) {
    tmp |= pk[i];
  }
  EXPECT_NE(tmp, 0);

  // Check shared secret and ciphertext returned by SIKE_encaps
  SIKE_encaps(ss, ct, pk);
  tmp = 0;
  for (size_t i = 0; i < sizeof(ct); i++) {
    tmp |= ct[i];
  }
  EXPECT_NE(tmp, 0);

  tmp = 0;
  for (size_t i = 0; i < sizeof(ss); i++) {
    tmp |= ss[i];
  }
  EXPECT_NE(tmp, 0);
}

TEST(SIKE, Negative) {
  uint8_t sk[SIKE_PRV_BYTESZ] = {0};
  uint8_t pk[SIKE_PUB_BYTESZ] = {0};
  uint8_t ct[SIKE_CT_BYTESZ] = {0};
  uint8_t ss_enc[SIKE_SS_BYTESZ] = {0};
  uint8_t ss_dec[SIKE_SS_BYTESZ] = {0};

  EXPECT_EQ(SIKE_keypair(sk, pk), 1);
  SIKE_encaps(ss_enc, ct, pk);

  // Change cipertext
  uint8_t ct_tmp[SIKE_CT_BYTESZ] = {0};
  memcpy(ct_tmp, ct, sizeof(ct));
  ct_tmp[0] = ~ct_tmp[0];
  SIKE_decaps(ss_dec, ct_tmp, pk, sk);
  EXPECT_NE(memcmp(ss_enc, ss_dec, SIKE_SS_BYTESZ), 0);

  // Change secret key
  uint8_t sk_tmp[SIKE_PRV_BYTESZ] = {0};
  memcpy(sk_tmp, sk, sizeof(sk));
  sk_tmp[0] = ~sk_tmp[0];
  SIKE_decaps(ss_dec, ct, pk, sk_tmp);
  EXPECT_NE(memcmp(ss_enc, ss_dec, SIKE_SS_BYTESZ), 0);

  // Change public key
  uint8_t pk_tmp[SIKE_PUB_BYTESZ] = {0};
  memcpy(pk_tmp, pk, sizeof(pk));
  pk_tmp[0] = ~pk_tmp[0];
  SIKE_decaps(ss_dec, ct, pk_tmp, sk);
  EXPECT_NE(memcmp(ss_enc, ss_dec, SIKE_SS_BYTESZ), 0);
}

TEST(SIKE, Unaligned) {
  alignas(4) uint8_t priv[SIKE_PRV_BYTESZ + 1];
  alignas(4) uint8_t pub[SIKE_PUB_BYTESZ + 1];
  alignas(4) uint8_t shared_key1[SIKE_SS_BYTESZ + 1];
  alignas(4) uint8_t ciphertext[SIKE_CT_BYTESZ + 1];
  alignas(4) uint8_t shared_key2[SIKE_SS_BYTESZ + 1];

  ASSERT_TRUE(SIKE_keypair(priv + 1, pub + 1));
  SIKE_encaps(shared_key1 + 1, ciphertext + 1, pub + 1);
  SIKE_decaps(shared_key2 + 1, ciphertext + 1, pub + 1, priv + 1);

  EXPECT_EQ(memcmp(shared_key1 + 1, shared_key2 + 1, SIKE_SS_BYTESZ), 0);
}

#if defined(SUPPORTS_ABI_TEST) && \
    (defined(OPENSSL_X86_64) || defined(OPENSSL_AARCH64))
TEST(SIKE, ABI) {
  felm_t a, b, c;
  dfelm_t d, e, f;
  CHECK_ABI(sike_fpadd, a, b, c);
  CHECK_ABI(sike_fpsub, a, b, c);
  CHECK_ABI(sike_mpmul, a, b, d);
  CHECK_ABI(sike_fprdc, d, a);
  CHECK_ABI(sike_mpadd_asm, a, b, c);
  CHECK_ABI(sike_mpsubx2_asm, d, e, f);
  CHECK_ABI(sike_mpdblsubx2_asm, d, e, f);
}

// Additional tests for checking if assembly implementation
// of MUL and REDC handles carry chains correctly.
TEST(SIKE, CarryChains) {
  // Expected results
  const dfelm_t exp_mul = {
    0x0000000000000001, 0x0000000000000000, 0x0000000000000000,
    0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
    0x0000000000000000, 0xFFFFFFFFFFFFFFFE, 0xFFFFFFFFFFFFFFFF,
    0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF,
    0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF,
  };

  const felm_t exp_redc = {
    0x93AA0C8C2D3235BE, 0xA8CD35DDDE399B46, 0xB9BBA5469509CA65,
    0x6B2FB3A5A2FB86E4, 0x585591BA6DBE862C, 0xD92D3FF5FE0938F2,
    0x0001E1F0EE75A1E1
  };

  // Input
  dfelm_t in14 = {
    0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF,
    0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF,
    0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF,
    0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF,
    0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF
  };

  felm_t in7 = {
    0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF,
    0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF,
    0xFFFFFFFFFFFFFFFF
  };

  dfelm_t res;
  sike_mpmul(in7, in7, res);
  EXPECT_EQ(memcmp(exp_mul, res, sizeof(exp_mul)), 0);

  // modifies in14 and in7
  sike_fprdc(in14, in7);
  EXPECT_EQ(memcmp(exp_redc, in7, sizeof(exp_redc)), 0);
}
#endif  // SUPPORTS_ABI_TEST && (X86_64 || AARCH64)
