/*
* ============================================================================
* RDK MANAGEMENT, LLC CONFIDENTIAL AND PROPRIETARY
* ============================================================================
* This file (and its contents) are the intellectual property of RDK Management, LLC.
* It may not be used, copied, distributed or otherwise  disclosed in whole or in
* part without the express written permission of RDK Management, LLC.
* ============================================================================
* Copyright (C) 2017 Broadcom. The term "Broadcom" refers to Broadcom Limited and/or its subsidiaries.
* Copyright (c) 2017 RDK Management, LLC. All rights reserved.
* ============================================================================
*/

#include "mediaplayersink.h"
#include "rmfprivate.h"

#include <cassert>
#include <cstdio>
#include <gst/gst.h>

#if USE_SYSRES_MLT
#include "rpl_new.h"
#endif

#define VIDEO_PLANE 6 // B
#define DEBUG_TRICKPLAY 0

//-- Callback -----------------------------------------------------------------

template <typename CB> class Callback
{
public:
    Callback() : m_cb(NULL), m_data(NULL)
    {
    }

    void set(CB cb, void* data)
    {
        m_cb = cb;
        m_data = data;
    }

    void operator()() // only works if the callback has no arguments besides 'data'
    {
        if (m_cb)
            m_cb(m_data);
    }

    operator bool()
    {
        return m_cb != NULL;
    }

    CB cb() const { return m_cb; }
    void* data() const { return m_data; }

private:
    CB m_cb;
    void* m_data;
};


//-- MediaPlayerSinkPrivate ---------------------------------------------------

class MediaPlayerSinkPrivate: public RMFMediaSinkPrivate
{
public:
    MediaPlayerSinkPrivate(MediaPlayerSink* parent);
    ~MediaPlayerSinkPrivate();
    void onSpeedChange(float new_speed);
    void* createElement();
    RMFResult setVideoRectangle(unsigned x, unsigned y, unsigned w, unsigned h, bool apply_now);
    void setMuted(bool muted);
    bool getMuted();
    void setVolume(float volume);
    float getVolume();
    RMFResult getMediaTime(double& t);
    GstElement* getVideoRenderer() const { return m_vsink; }
    long getVideoDecoderHandle() const;
    void setAudioLanguage (const char* pAudioLang);

    // from RMFMediaSinkBase
    /*virtual*/ RMFResult setSource(IRMFMediaSource* source);

    void setHaveVideoCallback(MediaPlayerSink::callback_t cb, void* data);
    void setHaveAudioCallback(MediaPlayerSink::callback_t cb, void* data);

private:
    class Rect
    {
    public:
        Rect() : m_x(0), m_y(0), m_w(0), m_h(0) {}
        bool isSet() const { return m_w && m_h; }
        void set(unsigned x, unsigned y, unsigned w, unsigned h)
        {
            m_x = x;
            m_y = y;
            m_w = w;
            m_h = h;
        }
        unsigned m_x, m_y, m_w, m_h; // video rect
    };

    void setMutedPriv(bool muted);
    bool wantAudio() const;
    void enableAudio(bool enable);
    void destroyAudio();
    void readMetadata();
    void configureTrickplay();
    void linkDemuxPad(GstPad *demux_src_pad);
    void applyVideoRectangle();

    static void onDemuxSrcPadAdded(GstElement*, GstPad *demux_src_pad, gpointer data);

    typedef Callback<MediaPlayerSink::callback_t> GenericCallback;

    GenericCallback m_haveVideoCallback;
    GenericCallback m_haveAudioCallback;

    Rect m_videoRect;
    bool m_haveAudio;
    bool m_haveVideo;
    bool m_contentBlocked;
    GstElement* m_demux;
    GstElement* m_vdec;
    GstElement* m_vsink;
    GstElement* m_asink;
#ifdef NEXUS_PLATFORM
    GstElement* m_adec;
#endif

    // stream props
    unsigned long m_currentPTS;
    unsigned int m_gopSize;
};

MediaPlayerSinkPrivate::MediaPlayerSinkPrivate(MediaPlayerSink* parent)
:   RMFMediaSinkPrivate(parent)
,   m_haveAudio(false)
,   m_haveVideo(false)
,   m_contentBlocked (false)
,   m_demux(NULL)
,   m_vdec(NULL)
,   m_vsink(NULL)
,   m_asink(NULL)
,   m_currentPTS(0)
,   m_gopSize(0)
{
}

MediaPlayerSinkPrivate::~MediaPlayerSinkPrivate()
{
    destroyAudio();
}

void MediaPlayerSinkPrivate::onSpeedChange(float new_speed)
{
    setMutedPriv(true);

    //Flush player bin
    GstPad* vdec_sink_pad = gst_element_get_static_pad(m_vdec, "sink");
    gst_pad_send_event(vdec_sink_pad, gst_event_new_flush_start());
    gst_pad_send_event(vdec_sink_pad, gst_event_new_flush_stop());
    gst_object_unref (vdec_sink_pad);

#ifdef NEXUS_PLATFORM
    GstPad* adec_sink_pad = gst_element_get_static_pad(m_adec, "sink");
    gst_pad_send_event(adec_sink_pad, gst_event_new_flush_start());
    gst_pad_send_event(adec_sink_pad, gst_event_new_flush_stop());
    gst_object_unref (adec_sink_pad);
#endif

    GstPad* demux_sink_pad = gst_element_get_static_pad(m_demux, "sink");
    gst_pad_send_event(demux_sink_pad, gst_event_new_flush_start());
    gst_pad_send_event(demux_sink_pad, gst_event_new_flush_stop());
    gst_object_unref (demux_sink_pad);
    
    // TODO: notify new_speed for fine-tune (or any special trick mode)
    if ((new_speed == 1) && (!m_contentBlocked))
    {
        setMutedPriv(false);
    }
}

void* MediaPlayerSinkPrivate::createElement()
{
    // init stream props
    m_currentPTS = 0;
    m_gopSize = 0;

#ifdef NEXUS_PLATFORM
#ifdef RDK_USE_NXCLIENT
    const char DEMUX[] = "brcmtsdemux";
#else
    const char DEMUX[] = "brcmdemux";
#endif    
    const char ASINK[] = "brcmaudiosink";
    const char VDEC[] = "brcmvideodecoder";
    const char VSINK[] = "brcmvideosink";
    const char ADEC[] = "brcmaudiodecoder";
#else	
#ifdef USE_HW_DEMUX
    const char DEMUX[] = "ismd_demux";
#else
    const char DEMUX[] = "flutsdemux";
#endif
    const char ASINK[] = "ismd_audio_sink";
    const char VDEC[] = "ismd_mpeg2_viddec";
    const char VSINK[] = "ismd_vidsink";
#endif
    
    // Create a bin to contain the sink elements.
    GstElement* bin = gst_bin_new("player_bin");

    // Add a demuxer to the bin.
    // NOTE: Use mpegtsdemux for seek to work.
    m_demux = gst_element_factory_make(DEMUX, "player_demux");
    if (!m_demux)
    {
        g_print("Failed to instantiate demuxer (%s)", DEMUX);
        RMF_ASSERT(m_demux);
        return NULL;
    }
#if defined(NEXUS_PLATFORM) && !defined(RDK_USE_NXCLIENT)
    g_object_set (G_OBJECT (m_demux), "rmf_support", 1, NULL);
#endif    
    g_signal_connect(m_demux, "pad-added", G_CALLBACK (onDemuxSrcPadAdded), this);

    gst_bin_add(GST_BIN(bin), m_demux);

    // Add a ghostpad to the bin so it can proxy to the demuxer.
    GstPad* pad = gst_element_get_static_pad(m_demux, "sink");
    gst_element_add_pad(bin, gst_ghost_pad_new("sink", pad));
    gst_object_unref(GST_OBJECT(pad));

    // Create video pipeline.
    m_vdec = gst_element_factory_make(VDEC, "player_vdec");
    m_vsink = gst_element_factory_make(VSINK, "player_vsink");
    if (!m_vdec || !m_vsink)
    {
        if (!m_vdec) g_print("Failed to instantiate video decoder (%s)\n", VDEC);
        if (!m_vsink) g_print("Failed to instantiate video sink (%s)\n", VSINK);
        RMF_ASSERT(m_vdec && m_vsink);
        return NULL;
    }

#if defined(NEXUS_PLATFORM) && !defined(RDK_USE_NXCLIENT)
    g_object_set (G_OBJECT (m_vdec), "rmf_support", 1, NULL);
#endif

    gst_bin_add_many(GST_BIN(bin), m_vdec, m_vsink, NULL);
    if (gst_element_link_many(m_vdec, m_vsink, NULL) != true)
    {
        g_print("Failed to link media sink elements\n");
        RMF_ASSERT(0);
        return NULL;
    }

    /*TODO, add gdl-plane support */
//    g_object_set(m_vsink, "gdl-plane", VIDEO_PLANE, NULL);

    // Create audio pipeline.
    // TODO: may need to reset m_asink and m_vsink on term()
    m_asink = gst_element_factory_make(ASINK, "player_asink");
#ifdef NEXUS_PLATFORM
    m_adec = gst_element_factory_make(ADEC, "player_adec");
    if (!m_asink || !m_adec)
    {
        g_print("Failed to instantiate audio sink (%s) or decoder (%s)\n", ASINK, ADEC);
        RMF_ASSERT(m_asink && m_adec);
        return NULL;
    }
#if !defined(RDK_USE_NXCLIENT)
    g_object_set (G_OBJECT (m_adec), "rmf_support", 1, NULL);
#endif    
    gst_bin_add_many(GST_BIN(bin), m_asink, m_adec, NULL);

    if (gst_element_link_many(m_adec, m_asink, NULL) != true)
    {
        g_print("Failed to link media sink elements\n");
        RMF_ASSERT(0);
        return NULL;
    }
#else
    if (!m_asink)
    {
        g_print("Failed to instantiate audio sink (%s)\n", ASINK);
        RMF_ASSERT(m_asink);
        return NULL;
    }
    gst_bin_add(GST_BIN(bin), m_asink);
#endif

    return bin;
}

RMFResult MediaPlayerSinkPrivate::setVideoRectangle(unsigned x, unsigned y, unsigned w, unsigned h, bool apply_now)
{
    if (!w || !h)
        return RMF_RESULT_INVALID_ARGUMENT;

    m_videoRect.set(x, y, w, h);
    if (apply_now)
        applyVideoRectangle();
    return RMF_RESULT_SUCCESS;
}

void MediaPlayerSinkPrivate::setMuted(bool muted)
{
    m_contentBlocked = muted;
    setMutedPriv (muted);
}

void MediaPlayerSinkPrivate::setMutedPriv(bool muted)
{
    if (!m_asink) return;
    g_object_set(m_asink, "mute", muted, NULL);
}

bool MediaPlayerSinkPrivate::getMuted()
{
    if (!m_asink) return false;
    bool muted = false;
    g_object_get(m_asink, "mute", &muted, NULL);
    return muted;
}

void MediaPlayerSinkPrivate::setVolume(float volume)
{
    if (!m_asink) return;
    g_object_set(m_asink, "volume", volume, NULL);
}

float MediaPlayerSinkPrivate::getVolume()
{
    if (!m_asink) return 1.0f;
    double volume = 1.0f;
    g_object_get(m_asink, "volume", &volume, NULL);
    return (float) volume;
}

RMFResult MediaPlayerSinkPrivate::getMediaTime(double& t)
{
    RMFResult result= RMF_RESULT_FAILURE;
    gboolean rc;
    gint64 cur_pos= 0.0;
    GstFormat cur_pos_fmt= GST_FORMAT_TIME;
    if ( m_haveVideo )
    {
        rc= gst_element_query_position(m_vsink, &cur_pos_fmt, &cur_pos );
    }
    else
    {
        rc= gst_element_query_position(getSink(), &cur_pos_fmt, &cur_pos );
    }

    if ( rc )
    {
        t = GST_TIME_AS_SECONDS((float)cur_pos);
        t = std::max(0., t); // prevent negative offsets
    }
    
    return result;
}

void MediaPlayerSinkPrivate::setAudioLanguage (const char* pAudioLang)
{
    if (pAudioLang)
    {
        std::string name = pAudioLang;
        g_print ("Language Selected = %s\n", name.c_str());
#ifdef USE_GENERIC_PLAYERSINKBIN
        if (m_playersinkbin)
            g_object_set(m_playersinkbin, "preferred-language", name.c_str(), NULL);
#endif /* USE_GENERIC_PLAYERSINKBIN */
    }
    else
        g_print ("The audio language selected is empty. Ignore it\n");
    return;
}

#define G_VALUE_INIT {0,{{0}}}
long MediaPlayerSinkPrivate::getVideoDecoderHandle() const
{
    //Closed Caption
    GValue gVideoDecoder = G_VALUE_INIT;
    g_value_init(&gVideoDecoder, G_TYPE_POINTER);
    g_object_get_property(G_OBJECT(m_vdec), "videodecoder", &gVideoDecoder);
    
    gpointer hVideoDecoderHwdl = g_value_get_pointer(&gVideoDecoder);
    g_print("The Video Decoder Handle:%p\n", hVideoDecoderHwdl);
    return (long) hVideoDecoderHwdl;
}

// virtual
RMFResult MediaPlayerSinkPrivate::setSource(IRMFMediaSource* source)
{
    if (source)
        applyVideoRectangle();


    if ((m_demux) && (source))
    {
        if (source->getPrivateSourceImpl()->isLiveSource())
            g_object_set (G_OBJECT (m_demux), "live_streaming_mode", true, NULL);
        else
            g_object_set (G_OBJECT (m_demux), "live_streaming_mode", false, NULL);
    }
    return RMFMediaSinkPrivate::setSource(source);
}

void MediaPlayerSinkPrivate::setHaveVideoCallback(MediaPlayerSink::callback_t cb, void* data)
{
    m_haveVideoCallback.set(cb, data);
}

void MediaPlayerSinkPrivate::setHaveAudioCallback(MediaPlayerSink::callback_t cb, void* data)
{
    m_haveAudioCallback.set(cb, data);
}

bool MediaPlayerSinkPrivate::wantAudio() const
{
    float play_speed = 0;
    getSource()->getSpeed(play_speed);
    return play_speed == 1.f; // disable audio during trickplay
}

// Add/remove audio elements from/to the pipeline.
// Useful for trickplay as pipeline simply does not work when playback speed != 1,
// i.e. we're fast forwarding or rewinding.
void MediaPlayerSinkPrivate::enableAudio(bool enable)
{
    if ((GST_ELEMENT_PARENT(m_asink) == getSink()) == enable)
    {
        // nothing to do
        return;
    }

    if (enable)
    {
#if DEBUG_TRICKPLAY
        g_print("Enabling audio\n");
#endif
        gst_bin_add_many(GST_BIN(getSink()), m_asink, NULL);
        assert(gst_element_sync_state_with_parent(m_asink) == true);
        gst_object_unref(m_asink);
    }
    else
    {
#if DEBUG_TRICKPLAY
        g_print("Disabling audio\n");
#endif
        gst_element_set_state(m_asink, GST_STATE_NULL);
        gst_object_ref(m_asink);
        gst_bin_remove(GST_BIN(getSink()), m_asink);
    }
}

void MediaPlayerSinkPrivate::destroyAudio()
{
    // If audio elements don't belong to the pipeline
    // (which happens if were doing trickplay before pipeline destruction)
    // destroy them manually. See enableAudio().
    if (GST_ELEMENT_PARENT(m_asink) != getSink())
    {
#if DEBUG_TRICKPLAY
        g_print("Destroing audio elements manually\n");
#endif
        assert(GST_OBJECT_REFCOUNT_VALUE(m_asink) == 1);
        gst_object_unref(m_asink);
    }
    else
    {
#if DEBUG_TRICKPLAY
        g_print("Audio elements will be destroyed by the pipeline\n");
#endif
    }
}

void MediaPlayerSinkPrivate::readMetadata()
{
    if (!m_vdec) // pipeline has not been fully constructed yet
        return;

    g_object_get(G_OBJECT(m_vsink),  "currentPTS", &m_currentPTS,     NULL);

    // FIXME: is this needed?
    // g_object_get(G_OBJECT(source), "GopSize",    &m_gopSize,          NULL);
    // if (sGopSize > 18)
    // {
    //     g_object_set(G_OBJECT (vdec), "gopsize", sGopSize, NULL);
    // }

    //g_print("metadata currentPTS=%u GopSize=%u\n", (unsigned) m_currentPTS, m_gopSize);
}

void MediaPlayerSinkPrivate::configureTrickplay()
{
    //readMetadata();

    //new segment event for rewind
    float play_speed = 0;
    getSource()->getSpeed(play_speed);
    if (play_speed < 0)
    {
#if DEBUG_TRICKPLAY
        g_print("Rewind: sending new segment event\n");
#endif

        GstElement* pipeline = getSource()->getPrivateSourceImpl()->getPipeline();
        GstPad* demux_sink_pad = gst_element_get_static_pad(m_demux, "sink");

        gst_element_set_state(pipeline, GST_STATE_PAUSED);

        //in vl-gst-app the last parameter is curr position in seconds instead of currentPTS (I didn't see any difference in FF > RW playback)
        gst_pad_send_event (demux_sink_pad, gst_event_new_new_segment(FALSE, play_speed, GST_FORMAT_BYTES, 0, m_currentPTS, m_currentPTS));

        gst_element_set_state (pipeline, GST_STATE_PLAYING);

        gst_object_unref (demux_sink_pad);
    }

//#if DEBUG_TRICKPLAY
//    g_print("Setting trick-rate to %f\n", play_speed);
//#endif
//    g_object_set(G_OBJECT (m_vdec), "trick-rate", play_speed, NULL);
}

void MediaPlayerSinkPrivate::linkDemuxPad(GstPad *demux_src_pad)
{
    std::string name;
    {
        GstCaps* caps = gst_pad_get_caps(demux_src_pad);
        name = gst_structure_get_name(gst_caps_get_structure (caps, 0));
        gst_caps_unref(caps);
    }

    g_print("Got %s\n", name.c_str());
    if (name.find("video/") == 0)
    {
        //configureTrickplay();

        GstPad* sink_pad = gst_element_get_static_pad(m_vdec, "sink");
        RMF_ASSERT(sink_pad);
        if (!GST_PAD_LINK_SUCCESSFUL(gst_pad_link(demux_src_pad, sink_pad)))
        {
            g_print("Failed to connect video pad\n");
            RMF_ASSERT(0);
        }
        else
        {
            m_haveVideo = true;
            m_haveVideoCallback();
        }
        gst_object_unref(sink_pad);
        applyVideoRectangle();
    }
    else if (name.find("audio/") == 0) // we only handle one audio stream
    {
        if (!wantAudio())
        {
#if DEBUG_TRICKPLAY
            g_print("Ignoring audio\n");
#endif
            return;
        }

        // The following breaks playback after the pipeliene goes to NULL state.
        // FIXME: Come up with a better solution like a callback using which
        // the source can notify its sinks whenever it transitions to NULL.
        //
        // Disabling this block causes runtime warnings
        // ("Failed to connect audio pad") when the stream has multiple audio tracks.
#if 0
        if (m_haveAudio)
        {
            g_print("Already configured audio, skipping audio stream\n");
            return;
        }
#endif
#ifdef NEXUS_PLATFORM
        GstPad* sink_pad = gst_element_get_static_pad(m_adec, "sink");
#else
        GstPad* sink_pad = gst_element_get_static_pad(m_asink, "sink");
#endif
        RMF_ASSERT(sink_pad);
        if (!GST_PAD_LINK_SUCCESSFUL(gst_pad_link(demux_src_pad, sink_pad)))
        {
            g_print("Failed to connect audio pad\n");
        }
        else
        {
            m_haveAudio = true;
            m_haveAudioCallback();
        }
        gst_object_unref(sink_pad);
    }
}

void MediaPlayerSinkPrivate::applyVideoRectangle()
{
    if (m_videoRect.isSet())
    {
        Rect& rect = m_videoRect;
        //g_print("Playing in rectangle: %dx%d at %dx%d\n", rect.m_w, rect.m_h, rect.m_x, rect.m_y);
        char rect_str[64];
        sprintf(rect_str, "%d,%d,%d,%d", rect.m_x, rect.m_y, rect.m_w, rect.m_h);
        g_object_set (GST_OBJECT(m_vsink), "window-set", rect_str, NULL);
    }
    else
        g_print("Playing full screen\n");

}

// connect dynamically created demuxer's video source pad to the video decoder
// static
void MediaPlayerSinkPrivate::onDemuxSrcPadAdded(GstElement*, GstPad *demux_src_pad, gpointer data)
{
    MediaPlayerSinkPrivate* self = (MediaPlayerSinkPrivate*) data;
    self->linkDemuxPad(demux_src_pad);
}

//-- MediaPlayerSink ----------------------------------------------------------

#define IMPL static_cast<MediaPlayerSinkPrivate*>(getPrivateSinkImpl())

RMFResult MediaPlayerSink::setVideoRectangle(unsigned x, unsigned y, unsigned w, unsigned h, bool apply_now)
{
    return IMPL->setVideoRectangle(x, y, w, h, apply_now);
}

// virtual
void MediaPlayerSink::onSpeedChange(float new_speed)
{
    IMPL->onSpeedChange(new_speed);
}

// virtual
void* MediaPlayerSink::createElement()
{
    return IMPL->createElement();
}

void MediaPlayerSink::setMuted(bool muted)
{
    return IMPL->setMuted(muted);
}

bool MediaPlayerSink::getMuted()
{
    return IMPL->getMuted();
}

void MediaPlayerSink::setVolume(float volume)
{
    return IMPL->setVolume(volume);
}

float MediaPlayerSink::getVolume()
{
    return IMPL->getVolume();
}

RMFResult MediaPlayerSink::getMediaTime(double& t)
{
   return IMPL->getMediaTime(t);
}

void MediaPlayerSink::setHaveVideoCallback(callback_t cb, void* data)
{
    return IMPL->setHaveVideoCallback(cb, data);
}

void MediaPlayerSink::setHaveAudioCallback(callback_t cb, void* data)
{
    return IMPL->setHaveAudioCallback(cb, data);
}

void MediaPlayerSink::setVolumeChangedCallback(callback_t cb, void* data)
{
    // NOP. TODO: Implement this?
    (void) cb;
    (void) data;
}

void MediaPlayerSink::setMuteChangedCallback(callback_t cb, void* data)
{
    // NOP. TODO: Implement this?
    (void) cb;
    (void) data;
}

void* MediaPlayerSink::getVideoRenderer() const
{
    return IMPL->getVideoRenderer();
}

long MediaPlayerSink::getVideoDecoderHandle() const
{
    return IMPL->getVideoDecoderHandle();
}

void MediaPlayerSink::setAudioLanguage (const char* pAudioLang)
{
    IMPL->setAudioLanguage (pAudioLang);
}

// virtual
RMFMediaSinkPrivate* MediaPlayerSink::createPrivateSinkImpl()
{
    return new MediaPlayerSinkPrivate(this);
}
