/***************************************************************************
 *     (c)2007-2013 Broadcom Corporation
 *
 *  This program is the proprietary software of Broadcom Corporation and/or its licensors,
 *  and may only be used, duplicated, modified or distributed pursuant to the terms and
 *  conditions of a separate, written license agreement executed between you and Broadcom
 *  (an "Authorized License").  Except as set forth in an Authorized License, Broadcom grants
 *  no license (express or implied), right to use, or waiver of any kind with respect to the
 *  Software, and Broadcom expressly reserves all rights in and to the Software and all
 *  intellectual property rights therein.  IF YOU HAVE NO AUTHORIZED LICENSE, THEN YOU
 *  HAVE NO RIGHT TO USE THIS SOFTWARE IN ANY WAY, AND SHOULD IMMEDIATELY
 *  NOTIFY BROADCOM AND DISCONTINUE ALL USE OF THE SOFTWARE.
 *
 *  Except as expressly set forth in the Authorized License,
 *
 *  1.     This program, including its structure, sequence and organization, constitutes the valuable trade
 *  secrets of Broadcom, and you shall use all reasonable efforts to protect the confidentiality thereof,
 *  and to use this information only in connection with your use of Broadcom integrated circuit products.
 *
 *  2.     TO THE MAXIMUM EXTENT PERMITTED BY LAW, THE SOFTWARE IS PROVIDED "AS IS"
 *  AND WITH ALL FAULTS AND BROADCOM MAKES NO PROMISES, REPRESENTATIONS OR
 *  WARRANTIES, EITHER EXPRESS, IMPLIED, STATUTORY, OR OTHERWISE, WITH RESPECT TO
 *  THE SOFTWARE.  BROADCOM SPECIFICALLY DISCLAIMS ANY AND ALL IMPLIED WARRANTIES
 *  OF TITLE, MERCHANTABILITY, NONINFRINGEMENT, FITNESS FOR A PARTICULAR PURPOSE,
 *  LACK OF VIRUSES, ACCURACY OR COMPLETENESS, QUIET ENJOYMENT, QUIET POSSESSION
 *  OR CORRESPONDENCE TO DESCRIPTION. YOU ASSUME THE ENTIRE RISK ARISING OUT OF
 *  USE OR PERFORMANCE OF THE SOFTWARE.
 *
 *  3.     TO THE MAXIMUM EXTENT PERMITTED BY LAW, IN NO EVENT SHALL BROADCOM OR ITS
 *  LICENSORS BE LIABLE FOR (i) CONSEQUENTIAL, INCIDENTAL, SPECIAL, INDIRECT, OR
 *  EXEMPLARY DAMAGES WHATSOEVER ARISING OUT OF OR IN ANY WAY RELATING TO YOUR
 *  USE OF OR INABILITY TO USE THE SOFTWARE EVEN IF BROADCOM HAS BEEN ADVISED OF
 *  THE POSSIBILITY OF SUCH DAMAGES; OR (ii) ANY AMOUNT IN EXCESS OF THE AMOUNT
 *  ACTUALLY PAID FOR THE SOFTWARE ITSELF OR U.S. $1, WHICHEVER IS GREATER. THESE
 *  LIMITATIONS SHALL APPLY NOTWITHSTANDING ANY FAILURE OF ESSENTIAL PURPOSE OF
 *  ANY LIMITED REMEDY.
 *
 ***************************************************************************/

#include "bcmnexus.h"
#include "bcmnexus_screen.h"
#include "bcmnexus_layer.h"
#include "bcmnexus_utils.h"
#include "bcmnexus_pool.h"
#include "nexus_core_utils.h"

#include "core/system.h"
#include "core/layers_internal.h"
#include "core/screens_internal.h"
#include "core/screens.h"
#include "misc/conf.h"
#include "gfx/util.h"

#if BCM_DFB_USE_TASK_MANAGER
#include <core/Task.h>
#endif

D_DEBUG_DOMAIN( bcmnexusLayer,  "bcmNexus/Layer",  "Broadcom NEXUS Layer driver" );
D_DEBUG_DOMAIN( bcmnexusLayerX, "bcmNexus/LayerX", "Broadcom NEXUS Layer driver (eXtra debug)" );


/**********************************************************************************************************************/

bcmLayerData *
bcmGetLayerData(DFBDisplayLayerID layerId)
{
    CoreLayer *layer;

    layer = dfb_layer_at(layerId);

    if (layer)
        return layer->layer_data;
    else
        return NULL;
}

#if BCM_DFB_USE_NOTIFY_DISPLAY_THREAD
static void *
notifyDisplayThread( UNUSED_ DirectThread *thread, void *arg )
{
    bcmLayerData      *layerData        = arg;
    CoreSurface       *frameBuffer      = NULL;
    CoreSurfaceBuffer *flipPendingLeft  = NULL;
    CoreSurfaceBuffer *flipPendingRight = NULL;

#if (((DIRECTFB_MAJOR_VERSION * 1000000 +       \
       DIRECTFB_MINOR_VERSION * 1000 +          \
       DIRECTFB_MICRO_VERSION) <= 1006000) &&   \
    FUSION_BUILD_MULTI )
    CoreTLS *tls = Core_GetTLS();
    if (!tls)
    {
        D_WARN( "TLS error" );
    }
    else
        tls->no_wait_on_shutdown = 1;
#endif

    /* Signal to thread which spawned us that we're started */
    PTHREAD_ROBUST_MUTEX_LOCK( &layerData->notifyDisplayThreadLock );
    pthread_cond_broadcast( &layerData->notifyDisplayThreadSem );

    D_DEBUG_AT( bcmnexusLayer, "%s: Starting... (layerData = %p)\n", __FUNCTION__,(void *) layerData );

    while (!layerData->notifyDisplayThreadQuit && !core_dfb->shutdown_tid)
    {
        while (!layerData->notifyDisplayThreadQuit && !layerData->notifyDisplayFrameBuffer && !layerData->notifyDisplayFlipPendingLeft && !layerData->notifyDisplayFlipPendingRight)
        {
            pthread_cond_wait( &layerData->notifyDisplayThreadSem, &layerData->notifyDisplayThreadLock );
        }

        if (layerData->notifyDisplayFrameBuffer)
        {
            if (!layerData->notifyDisplayThreadQuit && !core_dfb->shutdown_tid)
            {
                if (layerData->notifyDisplayFlipPendingLeft)
                {
                    flipPendingLeft = layerData->notifyDisplayFlipPendingLeft;
                    layerData->notifyDisplayFlipPendingLeft = NULL;
                }

                if (layerData->notifyDisplayFlipPendingRight)
                {
                    flipPendingRight = layerData->notifyDisplayFlipPendingRight;
                    layerData->notifyDisplayFlipPendingRight = NULL;
                }
            }

            frameBuffer = layerData->notifyDisplayFrameBuffer;
            layerData->notifyDisplayFrameBuffer = NULL;
            pthread_mutex_unlock( &layerData->notifyDisplayThreadLock );

            if (flipPendingLeft)
            {
                dfb_surface_notify_display( frameBuffer, flipPendingLeft );
                flipPendingLeft = NULL;
            }

            if (flipPendingRight)
            {
                dfb_surface_notify_display( frameBuffer, flipPendingRight );
                flipPendingRight = NULL;
            }

            PTHREAD_ROBUST_MUTEX_LOCK( &layerData->notifyDisplayThreadLock );
            dfb_surface_unref( frameBuffer );
        }
    }

    pthread_cond_broadcast( &layerData->notifyDisplayThreadSem );
    pthread_mutex_unlock( &layerData->notifyDisplayThreadLock );

    D_DEBUG_AT( bcmnexusLayer, "%s: Exiting... (layerData = %p)\n", __FUNCTION__, (void*)layerData );

    return NULL;
}
#endif

static void
frameBufferCallback( void *context, int param )
{
    bcmLayerData      *layerData = context;
    CoreSurface       *frameBuffer;
    CoreSurfaceBuffer *flipPendingLeft;
    CoreSurfaceBuffer *flipPendingRight;
#if BCM_DFB_USE_TASK_MANAGER
    DFB_DisplayTask   *taskLeft;
    DFB_DisplayTask   *taskRight;
#endif
    int ret = 0;

    D_DEBUG_AT( bcmnexusLayerX, "  -> screen %d gfx layer %d framebuffer callback handled\n", layerData->displayId, param);

    PTHREAD_ROBUST_MUTEX_LOCK( &layerData->frameBufferSemLock );
    frameBuffer      = layerData->frameBuffer;
    flipPendingLeft  = layerData->flipPendingLeft;
    flipPendingRight = layerData->flipPendingRight;

#if BCM_DFB_USE_TASK_MANAGER
    taskLeft         = layerData->taskLeft;
    taskRight        = layerData->taskRight;
    layerData->taskLeft = NULL;
    layerData->taskRight = NULL;
#endif

    layerData->flipPendingLeft  = NULL;
    layerData->flipPendingRight = NULL;

    ret = pthread_cond_broadcast( &layerData->frameBufferSem );
    pthread_mutex_unlock( &layerData->frameBufferSemLock );

#if BCM_DFB_USE_NOTIFY_DISPLAY_THREAD
    /* Inform the notifyDisplayThread that the buffer has been displayed */
    PTHREAD_ROBUST_MUTEX_LOCK( &layerData->notifyDisplayThreadLock );
    layerData->notifyDisplayFlipPendingLeft = flipPendingLeft;
    layerData->notifyDisplayFlipPendingRight = flipPendingRight;
    dfb_surface_ref( frameBuffer );
    layerData->notifyDisplayFrameBuffer = frameBuffer;
    pthread_cond_broadcast( &layerData->notifyDisplayThreadSem );
    pthread_mutex_unlock( &layerData->notifyDisplayThreadLock );
#else

#if BCM_DFB_USE_TASK_MANAGER
    D_DEBUG_AT( bcmnexusLayerX, "  -> taskLeft %p, oldTaskLeft %p\n", taskLeft, layerData->oldTaskLeft);

    if (layerData->oldTaskLeft)
        Task_Done( layerData->oldTaskLeft );

    layerData->oldTaskLeft = taskLeft;

    if (layerData->oldTaskRight)
        Task_Done( layerData->oldTaskRight );

    layerData->oldTaskRight = taskRight;
#endif
#endif
}

static void
bcmDropFrame( bcmLayerData *layerData )
{
#if BCM_DFB_USE_TASK_MANAGER
    DFB_DisplayTask   *taskLeft         = layerData->taskLeft;
    DFB_DisplayTask   *taskRight        = layerData->taskRight;
#else
    CoreSurfaceBuffer *flipPendingLeft  = layerData->flipPendingLeft;
    CoreSurfaceBuffer *flipPendingRight = layerData->flipPendingRight;

    D_DEBUG_AT( bcmnexusLayer, "%s: screen %d gfx layer %d\n", __FUNCTION__, layerData->displayId, layerData->layerId);
#endif

    layerData->flipPendingLeft  = NULL;
    layerData->flipPendingRight = NULL;
#if BCM_DFB_USE_TASK_MANAGER
    layerData->taskLeft = NULL;
    layerData->taskRight = NULL;

    D_DEBUG_AT( bcmnexusLayer, "%s: screen %d gfx layer %d: taskLeft %p\n", __FUNCTION__, layerData->displayId, layerData->layerId, layerData->taskLeft );

#endif

    /* Inform DirectFB that the buffer has been displayed */
#if BCM_DFB_USE_TASK_MANAGER
    if (taskLeft)
        Task_Done( taskLeft );
#else
    if (flipPendingLeft)
        dfb_surface_notify_display( layerData->frameBuffer, flipPendingLeft );
#endif

#if BCM_DFB_USE_TASK_MANAGER
    if (taskRight)
        Task_Done( taskRight );
#else
    if (flipPendingRight)
        dfb_surface_notify_display( layerData->frameBuffer, flipPendingRight );
#endif

}

static void
bcmGetClipRectangle(bcmLayerData *layerData, int dispWidth, int dispHeight, int defaultWidth, int defaultHeight, int* clipWidth, int* clipHeight)
{
    DFBBCMNEXUS                 *pBrcmDfb = (DFBBCMNEXUS *) dfb_system_data();
    BCMNEXUS_Options            *brcm_config = &pBrcmDfb->options;

    if(layerData->ibuffer)
    {
        long int ibufArea = brcm_config->ibuffer.width * brcm_config->ibuffer.height;
        long int dstArea = dispWidth * dispHeight;

        if ((layerData->srcRectangle.w > dispWidth ) || (layerData->srcRectangle.h > dispHeight) )
        {
            *clipWidth        = MIN(brcm_config->ibuffer.width, dispWidth);
            *clipHeight       = MIN(brcm_config->ibuffer.height, dispHeight);
        }
        else if((layerData->srcRectangle.w <= dispWidth) || (layerData->srcRectangle.h <= dispHeight) )
        {
            if(dstArea > ibufArea)
            {
                *clipWidth        = MIN(brcm_config->ibuffer.width, dispWidth);
                *clipHeight       = MIN(brcm_config->ibuffer.height, dispHeight);

            }
            else
            {
                *clipWidth        = defaultWidth;
                *clipHeight       = defaultHeight;
            }
        }
        else
        {
            *clipWidth  = defaultWidth;
            *clipHeight = defaultHeight;
        }
    }
    else
    {
        *clipWidth  = defaultWidth;
        *clipHeight = defaultHeight;
    }
}

static DFBResult
bcmGetLayerGraphicsSettings(DFBBCMNEXUS *dfb_bcmnexus, DFBDisplayLayerID layerId, DFB_Platform_P_GraphicsSettings *pGraphicsSettings)
{
    int                    source_height, source_width;
    int                    display_width, display_height;
    int                    canvas_width,  canvas_height;
    bcmLayerData          *layerData;
    CoreLayerRegionConfig *regionConfig;
    DFBScreenID            displayId;
    DFB_PlatformDimension  displaySize;
    int                    width;
    int                    height;

    layerData      = bcmGetLayerData(layerId);
    if (layerData == NULL)
        return DFB_FAILURE;

    regionConfig   = (CoreLayerRegionConfig *)&(layerData->regionData->config);
    displayId      = layerData->displayId;

    /* Get Canvas size */
    DFB_Platform_P_GetDisplaySize(displayId, &displaySize);

    canvas_width   = displaySize.w;
    canvas_height  = displaySize.h;

    /* Get Display Rectangle size */
    display_width  = MIN(layerData->destRectangle.w, canvas_width);
    display_height = MIN(layerData->destRectangle.h, canvas_height);

    /* Get Source Rectangle size */
    source_width   = layerData->srcRectangle.w;
    if (layerData->hasVscaler)
        source_height = layerData->srcRectangle.h;
    else
        source_height = display_height;

    DFB_Platform_P_GetGraphicsSettings(GET_DISPLAY_HND(dfb_bcmnexus, displayId), pGraphicsSettings);

    pGraphicsSettings->enabled           = layerData->layerEnabled;
    pGraphicsSettings->zorder            = layerData->level + BCMNEXUS_Z_ORDER_MAX/2;
    pGraphicsSettings->position.width    = display_width;
    pGraphicsSettings->position.height   = display_height;
    pGraphicsSettings->position.x        = (layerData->destRectangle.x + display_width  > canvas_width)  ? 0 : layerData->destRectangle.x;
    pGraphicsSettings->position.y        = (layerData->destRectangle.y + display_height > canvas_height) ? 0 : layerData->destRectangle.y;
    pGraphicsSettings->clip.x            = layerData->srcRectangle.x;
    pGraphicsSettings->clip.y            = layerData->srcRectangle.y;

    bcmGetClipRectangle(layerData, display_width, display_height, source_width, source_height, &width, &height);

    pGraphicsSettings->clip.width  = (unsigned short) width;
    pGraphicsSettings->clip.height = (unsigned short) height;


    pGraphicsSettings->lowerChromakey.a  = 0x00;
    pGraphicsSettings->lowerChromakey.r  = regionConfig->src_key.r;
    pGraphicsSettings->lowerChromakey.g  = regionConfig->src_key.g;
    pGraphicsSettings->lowerChromakey.b  = regionConfig->src_key.b;

    pGraphicsSettings->upperChromakey.a  = 0xff;
    pGraphicsSettings->upperChromakey.r  = regionConfig->src_key.r;
    pGraphicsSettings->upperChromakey.g  = regionConfig->src_key.g;
    pGraphicsSettings->upperChromakey.b  = regionConfig->src_key.b;

    pGraphicsSettings->chromakeyEnabled  = layerData->srcColorKeyOn;
    pGraphicsSettings->sourceBlendFactor = layerData->srcBlendFunc;
    pGraphicsSettings->destBlendFactor   = layerData->destBlendFunc;
    pGraphicsSettings->alpha             = layerData->alpha;
    pGraphicsSettings->constantAlpha     = layerData->constantAlpha;

    pGraphicsSettings->horizontalFilter  = DFB_PlatformFilterCoeffs_eBilinear;

    if (DFB_Platform_P_CheckGfxVScl(0) != false)
        pGraphicsSettings->verticalFilter = DFB_PlatformFilterCoeffs_eBilinear;

    if (layerData->layerEnabled) {
        pGraphicsSettings->frameBufferCallback.callback = layerData->layerCallback;
        pGraphicsSettings->frameBufferCallback.context  = layerData;
        pGraphicsSettings->frameBufferCallback.param    = layerId;
    } else {
        pGraphicsSettings->frameBufferCallback.callback = NULL;
    }

    CoreScreen *screen = dfb_screens_at_translated(layerData->displayId);
    bcmScreenData* screenData = (bcmScreenData*)(screen->screen_data);

    /* Modify display sizes for half res 3d modes */
    if (screenData->encoderCfg.framing == DSEPF_STEREO_SIDE_BY_SIDE_HALF)
        pGraphicsSettings->position.width    = display_width/2;

    if (screenData->encoderCfg.framing == DSEPF_STEREO_TOP_AND_BOTTOM)
        pGraphicsSettings->position.height    = display_height/2;

    /* Set Z offset if required */
    if ( regionConfig->options & (DLOP_LR_MONO | DLOP_STEREO) )
    {
        int scaledZ;

        if (screenData->encoderCfg.framing != DSEPF_MONO)
        {

            if (screenData->encoderCfg.framing == DSEPF_STEREO_SIDE_BY_SIDE_HALF)
                scaledZ = layerData->z * display_width / (int)(display_width/2);
            else
                scaledZ = layerData->z;

            pGraphicsSettings->stereoOffset = scaledZ;
            pGraphicsSettings->clip.width -= abs(scaledZ);
            pGraphicsSettings->position.width -= abs(scaledZ);
        }
        else
            pGraphicsSettings->stereoOffset = 0;
    }
    else
            pGraphicsSettings->stereoOffset = 0;

    return DFB_OK;
}

static DFBResult
bcmSetLayerFrameBuffer(UNUSED_ DFBBCMNEXUS *dfb_bcmnexus, DFBDisplayLayerID layerId, bool lock)
{
    DFBResult            result = DFB_OK;
    bcmLayerData        *layerData;
    bool                 single_buffered = false;

    D_DEBUG_AT( bcmnexusLayer, "%s() gfx layer %d (master=%c)\n",
                __FUNCTION__, layerId, dfb_core_is_master( dfb_bcmnexus_core ) ? 'y' : 'n');

    layerData = bcmGetLayerData(layerId);
    if (layerData == NULL) {
        result = DFB_FAILURE;
    } else {
        D_DEBUG_AT( bcmnexusLayer, "%s() screen %d gfx layer %d (enabled=%d, master=%c)\n",
                    __FUNCTION__, layerData->displayId, layerId, layerData->layerEnabled, dfb_core_is_master( dfb_bcmnexus_core ) ? 'y' : 'n');

        if (layerData->regionData->config.buffermode == DLBM_FRONTONLY)
        {
            D_DEBUG_AT( bcmnexusLayer, "%s() Single buffered layer : 0x%08x\n", __FUNCTION__, layerData->regionData->config.buffermode );
            single_buffered = true;
        }

        if (layerData->layerEnabled)
        {
            if (lock)
            {
                PTHREAD_ROBUST_MUTEX_LOCK( &layerData->frameBufferSemLock );
            }

            DFB_Platform_P_DisplayGraphicsFramebuffer3D frameBuffer3D;
            NEXUS_DisplayHandle hDisplay;
            NEXUS_SurfaceHandle  hLeftSurface, hRightSurface;
            DFB_PlatformRect left_rect, right_rect;

            CoreScreen *screen = dfb_screens_at_translated(layerData->displayId);
            bcmScreenData* screenData = (bcmScreenData*)(screen->screen_data);

            hLeftSurface = layerData->regionData->leftBufferLock.handle;
            hRightSurface = layerData->regionData->rightBufferLock.handle;

            hDisplay = GET_DISPLAY_HND(dfb_bcmnexus, layerData->displayId);
            DFB_Platform_P_GetDefaultDisplayGraphicsFramebuffer3D(&frameBuffer3D);

            frameBuffer3D.main = hLeftSurface;

            if (layerData->regionData->config.options & DLOP_STEREO)
            {

                frameBuffer3D.right = hRightSurface;

                switch (screenData->encoderCfg.framing)
                {

                    case DSEPF_STEREO_SIDE_BY_SIDE_HALF:
                        frameBuffer3D.orientation = DFB_Platform_P_VideoOrientation_e3D_LeftRight;
                        D_DEBUG_AT(bcmnexusLayer,"Using Left/Right packed stereoscopic graphics buffers\n");
                        break;
                    case DSEPF_STEREO_SIDE_BY_SIDE_FULL:
                    case DSEPF_STEREO_TOP_AND_BOTTOM:
                    case DSEPF_STEREO_FRAME_PACKING:
                        frameBuffer3D.orientation = DFB_Platform_P_VideoOrientation_e3D_OverUnder;
                        D_DEBUG_AT(bcmnexusLayer,"Using Over/Under packed stereoscopic graphics buffers\n");
                        break;
                    case DSEPF_MONO:
                    default:
                        frameBuffer3D.orientation = DFB_Platform_P_VideoOrientation_e2D;
                        D_DEBUG_AT(bcmnexusLayer,"Using single graphics buffer\n");
                        break;
                }

            }
            else
            {
                frameBuffer3D.orientation = DFB_Platform_P_VideoOrientation_e2D;
                frameBuffer3D.right = NULL;
            }

            bcmDropFrame( layerData );

            layerData->flipPendingLeft  = layerData->regionData->leftBufferLock.buffer;
            layerData->flipPendingRight = layerData->regionData->rightBufferLock.buffer;
#if BCM_DFB_USE_TASK_MANAGER
            layerData->taskLeft         = layerData->regionData->leftBufferLock.task;
            layerData->taskRight        = layerData->regionData->rightBufferLock.task;
#endif
            right_rect.height = layerData->regionData->right_update.h;
            right_rect.width  = layerData->regionData->right_update.w;
            right_rect.x      = layerData->regionData->right_update.x;
            right_rect.y      = layerData->regionData->right_update.y;

            left_rect.height = layerData->regionData->left_update.h;
            left_rect.width  = layerData->regionData->left_update.w;
            left_rect.x      = layerData->regionData->left_update.x;
            left_rect.y      = layerData->regionData->left_update.y;

#if BCM_DFB_USE_TASK_MANAGER
            if (layerData->regionData->leftBufferLock.task)
                Task_Log( layerData->regionData->leftBufferLock.task, "SetDisplay()" );
#endif

            result = (DFB_Platform_P_SetDisplayGraphicsFramebuffer3D(hDisplay, &frameBuffer3D,
                                                                     (left_rect.height  > 0 ? &left_rect  : NULL),
                                                                     (right_rect.height > 0 ? &right_rect : NULL),
                                                                     single_buffered) == DFB_PLATFORM_OK) ? DFB_OK : DFB_FAILURE;
            if (lock)
            {
                pthread_mutex_unlock( &layerData->frameBufferSemLock );
            }
        }
    }

    return result;
}

static DFBResult
bcmSetLayerGraphicsSettings(DFBBCMNEXUS *dfb_bcmnexus, DFBDisplayLayerID layerId)
{
    DFB_Platform_P_GraphicsSettings graphicsSettings;
    bcmLayerData          *layerData;
    NEXUS_DisplayHandle    hDisplay;
    DFBResult              result = DFB_OK;

    layerData = bcmGetLayerData(layerId);
    if (layerData == NULL) {
        result = DFB_FAILURE;
    } else {
        D_DEBUG_AT( bcmnexusLayer, "%s() screen %d gfx layer %d (master=%c)\n",
                __FUNCTION__, layerData->displayId, layerId, dfb_core_is_master( dfb_bcmnexus_core ) ? 'y' : 'n');

        bcmGetLayerGraphicsSettings(dfb_bcmnexus, layerId, &graphicsSettings);

        D_DEBUG_AT( bcmnexusLayer, "-> position (%d, %d, %d x %d), clip (%d, %d, %d x %d), opacity %d\n",
                    graphicsSettings.position.x, graphicsSettings.position.y,
                    graphicsSettings.position.width, graphicsSettings.position.height,
                    graphicsSettings.clip.x, graphicsSettings.clip.y,
                    graphicsSettings.clip.width, graphicsSettings.clip.height,
                    graphicsSettings.alpha );

        hDisplay = GET_DISPLAY_HND(dfb_bcmnexus, layerData->displayId);
        result = (DFB_Platform_P_SetGraphicsSettings(hDisplay, &graphicsSettings) == NEXUS_SUCCESS) ? DFB_OK : DFB_FAILURE;

    }
    return result;
}

DFBResult
bcmSetLayerState(DFBBCMNEXUS *dfb_bcmnexus, DFBDisplayLayerID layerId, bool enable)
{
    DFBResult               result;
    bcmLayerData           *layerData;
    DFBScreenID             displayId = 0xff;

    layerData = bcmGetLayerData(layerId);

    D_DEBUG_AT( bcmnexusLayer, "%s() screen %d gfx layer %d: enable=%d\n", __FUNCTION__, displayId, layerId, enable);

    if (layerData == NULL) {
        result = DFB_FAILURE;
    } else {
        /* Lock access to the layerData */
        PTHREAD_ROBUST_MUTEX_LOCK( &layerData->lock );

        displayId = layerData->displayId;

        /* Get whether the screen/display is on */
        layerData->layerEnabled = (bcmGetPowerMode(dfb_screens_at(displayId)) == DSPM_ON) && enable;

        result = bcmSetLayerGraphicsSettings(dfb_bcmnexus, layerId);

#if BCM_DFB_USE_TASK_MANAGER
        layerData->regionData->leftBufferLock.task = NULL;
        layerData->regionData->rightBufferLock.task = NULL;
#endif

        bcmSetLayerFrameBuffer(dfb_bcmnexus, layerId,false);

        /* Unlock access to the layerData */
        pthread_mutex_unlock( &layerData->lock );
    }
    D_DEBUG_AT( bcmnexusLayer, "%s() screen %d gfx layer %d: Exit [%d].\n", __FUNCTION__, displayId, layerId, result );
    return result;
}

DFBResult
bcmSetLayerDestRectangle(DFBBCMNEXUS *dfb_bcmnexus, DFBDisplayLayerID layerId, DFBRectangle *destRectangle)
{
    DFBResult               result;
    DFBScreenID             displayId = 0xff;
    bcmLayerData           *layerData;
    bcmRegionData          *regionData;
    DFB_Platform_P_GraphicsSettings  graphicsSettings;

    D_DEBUG_AT( bcmnexusLayer, "%s() screen %d gfx layer %d: rect=%dx%d@%d,%d\n", __FUNCTION__,
                    displayId, layerId, destRectangle->w, destRectangle->h, destRectangle->x, destRectangle->y);

    layerData = bcmGetLayerData(layerId);

    if (layerData == NULL) {
        result = DFB_FAILURE;
    } else {
        /* Lock access to the layerData */
        PTHREAD_ROBUST_MUTEX_LOCK( &layerData->lock );

        displayId = layerData->layerId;

        /* Save new destination rectangle settings */
        layerData->destRectangle = *destRectangle;

        bcmGetLayerGraphicsSettings(dfb_bcmnexus, layerId, &graphicsSettings);

        regionData = layerData->regionData;
        regionData->vscale = false;

        if (graphicsSettings.clip.height != layerData->srcRectangle.h)
        {
            regionData->vscale = true;
        }

        result = bcmSetLayerGraphicsSettings(dfb_bcmnexus, layerId);

#if BCM_DFB_USE_TASK_MANAGER
        layerData->regionData->leftBufferLock.task = NULL;
        layerData->regionData->rightBufferLock.task = NULL;
#endif
        bcmSetLayerFrameBuffer(dfb_bcmnexus, layerId,false);

        /* Unlock access to the layerData */
        pthread_mutex_unlock( &layerData->lock );
    }
    D_DEBUG_AT( bcmnexusLayer, "%s() screen %d gfx layer %d: Exit [%d].\n", __FUNCTION__, displayId, layerId, result );
    return result;
}

static void
bcmGetLayerDefaultSize(DFBDisplayLayerID layerId, int *pWidth, int *pHeight)
{
    int                   width, height;
    DFBScreenID           displayId;
    DFB_PlatformDimension displaySize;

    displayId = dfb_screen_id_translated(dfb_layer_screen(dfb_layer_at_translated(layerId)));

    /*Secondary layer frame buffer size need to be equal to maximum secondary disp format size. Secondary GFD does not have V-scaler
       IMS ID 528599 */
    if(displayId == SECONDARY_DISPLAY_ID)
    {
        DFB_PlatformDimension dispFormatSize;

        DFB_Platform_P_DFBFormatToDisplaySize(DFB_PlatformVideoFormat_ePalG, &dispFormatSize);
        width  = dispFormatSize.w;
        height = dispFormatSize.h;
    }
    else
    {
        DFB_Platform_P_GetDisplaySize(displayId, &displaySize);
        width  = displaySize.w;
        height = displaySize.h;
    }

    if (dfb_config->layers[layerId].config.flags & DLCONF_WIDTH)
        width = dfb_config->layers[layerId].config.width;

    if (dfb_config->layers[layerId].config.flags & DLCONF_HEIGHT)
        height = dfb_config->layers[layerId].config.height;

    *pWidth = width;
    *pHeight = height;

}

static void
bcmGetLayerDefaultFormat(DFBDisplayLayerID layerId, DFBSurfacePixelFormat *pFormat)
{
    DFBSurfacePixelFormat pixelFormat = DSPF_ARGB;

    if (dfb_config->layers[layerId].config.flags & DLCONF_PIXELFORMAT)
        pixelFormat = dfb_config->layers[layerId].config.pixelformat;

    *pFormat = pixelFormat;
}

static void
bcmGetLayerDefaultBufferMode(DFBDisplayLayerID layerId, DFBDisplayLayerBufferMode *pMode)
{
    DFBDisplayLayerBufferMode bufferMode = DLBM_FRONTONLY;

    if (dfb_config->layers[layerId].config.flags & DLCONF_BUFFERMODE)
        bufferMode = dfb_config->layers[layerId].config.buffermode;

    *pMode = bufferMode;
}

static void
bcmGetLayerDefaultOptions(DFBDisplayLayerID layerId, DFBDisplayLayerOptions *pOptions)
{
    DFBDisplayLayerOptions options = DLOP_OPACITY | DLOP_ALPHACHANNEL;

    if (dfb_config->layers[layerId].config.flags & DLCONF_OPTIONS)
        options = dfb_config->layers[layerId].config.options;

    *pOptions = options;
}

static void
bcmGetLayerDefaultSurfaceCaps(UNUSED_ DFBDisplayLayerID layerId, DFBSurfaceCapabilities *pCaps)
{
    DFBSurfaceCapabilities caps = DSCAPS_PRIMARY | DSCAPS_VIDEOONLY | DSCAPS_PREMULTIPLIED;

    *pCaps = caps;
}

static void
bcmGetLayerDefaultLevel(UNUSED_ DFBDisplayLayerID layerId, int *pLevel)
{
    *pLevel = 0;
}

static int
bcmGetLayerDefaultUpdateSkipCnt(DFBDisplayLayerID layerId)
{
#if ((DIRECTFB_MAJOR_VERSION * 1000000 + \
      DIRECTFB_MINOR_VERSION * 1000 + \
      DIRECTFB_MICRO_VERSION) >= 1006000)
    (void)layerId;
    return direct_config_get_int_value( "layer-update-skip" );

#else
    return dfb_config->layers[layerId].update_skip_cnt;
#endif
}

static int bcmLayerDataSize( void )
{
    return sizeof(bcmLayerData);
}

static int bcmRegionDataSize( void )
{
    return sizeof(bcmRegionData);
}

/********************************************
 * Common functions to setup layers/regions *
 ********************************************/

/* Common function to help initialize the layer data. */
static DFBResult
bcmInitLayer( DFBBCMNEXUS                *dfb_bcmnexus,
                          bcmLayerData               *layerData,
                          DFBDisplayLayerDescription *description,
                          DFBDisplayLayerConfig      *config)
{
    DFBScreenID         displayId;
    DFBDisplayLayerID   layerId;
    pthread_mutexattr_t mutexAttr;
    pthread_condattr_t  condAttr;

#if BCM_DFB_USE_NOTIFY_DISPLAY_THREAD
    char                thread_name[24];
#endif

    layerId   = layerData->layerId;
    displayId = layerData->displayId;

    D_DEBUG_AT( bcmnexusLayer, "%s() %s display graphics layer %d\n", __FUNCTION__,
                (displayId == PRIMARY_DISPLAY_ID) ? "primary" : "secondary", layerId) ;

    /* Save the graphics layer ID for the display */
    SET_DISPLAY_GFX_LAYER_ID(dfb_bcmnexus, layerId, GET_DISPLAY_NUM_GFX_LAYERS(dfb_bcmnexus, displayId)++, displayId);

    /* semaphore init */
    pthread_mutexattr_init(&mutexAttr);
    pthread_mutexattr_setpshared(&mutexAttr, PTHREAD_PROCESS_SHARED);
    PTHREAD_MUTEXATTR_SETROBUST(&mutexAttr, PTHREAD_MUTEX_ROBUST);

    pthread_condattr_init(&condAttr);
    pthread_condattr_setpshared(&condAttr, PTHREAD_PROCESS_SHARED);

    pthread_mutex_init(&layerData->lock, &mutexAttr);
    pthread_mutex_init(&layerData->frameBufferSemLock, &mutexAttr);
    pthread_cond_init(&layerData->frameBufferSem, &condAttr);

#if BCM_DFB_USE_NOTIFY_DISPLAY_THREAD
    pthread_mutex_init( &layerData->notifyDisplayThreadLock, &mutexAttr );
    pthread_cond_init( &layerData->notifyDisplayThreadSem, &condAttr );

    PTHREAD_ROBUST_MUTEX_LOCK( &layerData->notifyDisplayThreadLock );
    layerData->notifyDisplayThreadQuit = false;
    layerData->notifyDisplayFlipPendingLeft = NULL;
    layerData->notifyDisplayFlipPendingRight = NULL;

    snprintf( thread_name, D_ARRAY_SIZE(thread_name), "BRCM NotifyDisplay" );
    layerData->notifyDisplayThread = direct_thread_create( DTT_OUTPUT, notifyDisplayThread, (void *)layerData, thread_name );

    /* Wait for thread to start and to get into 'ready and waiting state' */
    pthread_cond_wait( &layerData->notifyDisplayThreadSem, &layerData->notifyDisplayThreadLock );
    pthread_mutex_unlock( &layerData->notifyDisplayThreadLock );
#endif

    pthread_mutexattr_destroy(&mutexAttr);
    pthread_condattr_destroy(&condAttr);

    /* Set type and capabilities */
    description->type = DLTF_GRAPHICS;
    description->caps = DLCAPS_SURFACE         | DLCAPS_OPACITY      | DLCAPS_ALPHACHANNEL |
                        DLCAPS_PREMULTIPLIED   | DLCAPS_SRC_COLORKEY | DLCAPS_LEVELS |
                        DLCAPS_SCREEN_POSITION | DLCAPS_SCREEN_SIZE  | DLCAPS_SOURCES;

    description->caps |= DLCAPS_LR_MONO | DLCAPS_STEREO;

    /* Set the layer name. */
    snprintf( description->name, DFB_DISPLAY_LAYER_DESC_NAME_LENGTH,
              "BRCM %s disp gfx layer %d", (displayId == PRIMARY_DISPLAY_ID) ? "primary" : "secondary", layerId );

    /* fill out the default configuration */
    config->flags = DLCONF_WIDTH | DLCONF_HEIGHT | DLCONF_PIXELFORMAT | DLCONF_BUFFERMODE | DLCONF_OPTIONS |
        DLCONF_SURFACE_CAPS | DLCONF_SOURCE | DLCONF_COLORSPACE;

    /* Set default level, number of regions, sources and clipping regions */
    description->level        = 0;  /* Default level is 0. Negative means below and positive means above */
    description->regions      = 1;  /* We only support a single region per layer context */
    description->sources      = 2;  /* HD and SD graphics sources */
    description->clip_regions = 0;

    /* Set the default level for this layer */
    bcmGetLayerDefaultLevel(layerId, &layerData->level);

    /* Set the default size of this layer */
    bcmGetLayerDefaultSize(layerId, &config->width, &config->height);

    /* Set the default pixel format and colorspace for this layer */
    bcmGetLayerDefaultFormat(layerId, &config->pixelformat);
    config->colorspace = DFB_COLORSPACE_DEFAULT(config->pixelformat);

    /* Set the default buffer mode for this layer */
    bcmGetLayerDefaultBufferMode(layerId, &config->buffermode);

    /* Set the default options for this layer */
    bcmGetLayerDefaultOptions(layerId, &config->options);

    /* Set the default surface capabilities for this layer */
    bcmGetLayerDefaultSurfaceCaps(layerId, &config->surface_caps);

    /* Set the default source for this layer */
    config->source = layerData->sourceId;

    /* Set layer callback... */
    layerData->layerCallback = frameBufferCallback;

    D_DEBUG_AT( bcmnexusLayer, "%s() screen %d gfx layer %d: flags=0x%08x, %dx%d, format=%s, bm=%d, opt=0x%08x, surf.caps=0x%08x\n", __FUNCTION__,
            displayId, layerId, config->flags,
            config->width, config->height, dfb_pixelformat_name(config->pixelformat),
            config->buffermode, config->options, config->surface_caps );

    return DFB_OK;
}

static DFBResult
bcmShutdownLayer( UNUSED_ CoreLayer *layer,
                  UNUSED_ void      *driver_data,
                          void      *layer_data )
{
    DFBBCMNEXUS                     *dfb_bcmnexus;
    DFB_PlatformNexusHandle          hDisplay;
    DFB_Platform_P_GraphicsSettings  graphicsSettings;
    bcmLayerData                    *layerData = layer_data;

    D_DEBUG_AT( bcmnexusLayer, "%s() screen %d gfx layer %d\n", __FUNCTION__, layerData->displayId, layerData->layerId );

    dfb_bcmnexus = dfb_system_data();
    hDisplay = GET_DISPLAY_HND(dfb_bcmnexus, layerData->displayId);
    DFB_Platform_P_GetGraphicsSettings(hDisplay, &graphicsSettings);
    graphicsSettings.frameBufferCallback.callback = NULL;
    graphicsSettings.frameBufferCallback.context= NULL;
    DFB_Platform_P_SetGraphicsSettings(hDisplay, &graphicsSettings);

#if BCM_DFB_USE_NOTIFY_DISPLAY_THREAD
    D_DEBUG_AT( bcmnexusLayer, "%s() screen %d gfx layer %d: waiting for notifyDisplayThreadLock thread to end...\n", __FUNCTION__,
            layerData->displayId, layerData->layerId );

    PTHREAD_ROBUST_MUTEX_LOCK( &layerData->notifyDisplayThreadLock );
    layerData->notifyDisplayThreadQuit = true;
    pthread_cond_broadcast( &layerData->notifyDisplayThreadSem );
    pthread_mutex_unlock( &layerData->notifyDisplayThreadLock );
    direct_thread_join( layerData->notifyDisplayThread );
    direct_thread_destroy( layerData->notifyDisplayThread );
    pthread_mutex_destroy( &layerData->notifyDisplayThreadLock );
    pthread_cond_destroy( &layerData->notifyDisplayThreadSem );

#endif

    pthread_mutex_destroy( &layerData->lock );
    pthread_mutex_destroy( &layerData->frameBufferSemLock );
    pthread_cond_destroy( &layerData->frameBufferSem );

    D_DEBUG_AT( bcmnexusLayer, "%s() screen %d gfx layer %d: Exit.\n", __FUNCTION__, layerData->displayId, layerData->layerId );
    return DFB_OK;
}

/* Common function to test whether the region can be modified to these settings. */
static DFBResult
bcmTestRegion( UNUSED_ CoreLayer                  *layer,
                       void                       *driver_data,
                       void                       *layer_data,
                       CoreLayerRegionConfig      *config,
                       CoreLayerRegionConfigFlags *failed )
{
    bcmLayerData               *layerData    = layer_data;
    CoreLayerRegionConfigFlags  fail         = CLRCF_NONE;
    int                         source_height, source_width;
    int                         display_width, display_height;
    DFBScreenID                 displayId;
    DFBDisplayLayerID           layerId;
    DFB_PlatformDimension       displaySize;

    D_ASSERT( (DFBBCMNEXUS*)driver_data != NULL );

    displayId  = layerData->displayId;
    layerId    = layerData->layerId;

    D_DEBUG_AT( bcmnexusLayer, "%s() screen %d gfx layer %d (%dx%d) format=%s, caps=0x%08x, bm=%d,\n"
                               "options=0x%08x, source_id=%d, opacity=%d [%dx%d@%d,%d]=>[%dx%d@%d,%d]\n",
                               __FUNCTION__, displayId, layerId, config->width, config->height, dfb_pixelformat_name(config->format),
                               config->surface_caps, config->buffermode, config->options, config->source_id, config->opacity,
                               config->source.w, config->source.h, config->source.x, config->source.y,
                               config->dest.w, config->dest.h, config->dest.x, config->dest.y);

    /* Get Display Rectangle size */
    DFB_Platform_P_GetDisplaySize(displayId, &displaySize);

    display_width  = MIN(config->dest.w, displaySize.w);
    display_height = MIN(config->dest.h, displaySize.h);

    /* Get Source Rectangle size */
    source_width = config->source.w;

    /* Lock access to the layerData */
    PTHREAD_ROBUST_MUTEX_LOCK( &layerData->lock );

    if (layerData->hasVscaler)
        source_height = config->source.h;
    else
        source_height = display_height;

    /* Unlock access to the layerData */
    pthread_mutex_unlock( &layerData->lock );

    /* Check format */
    switch (config->format) {
          case DSPF_A1:
          case DSPF_A4:
          case DSPF_A8:
          case DSPF_LUT2:
          case DSPF_LUT4:
          case DSPF_LUT8:
          case DSPF_ALUT8:
          case DSPF_RGB444:
          case DSPF_RGB555:
          case DSPF_BGR555:
          case DSPF_RGBA4444:
          case DSPF_ARGB4444:
          case DSPF_ARGB1555:
          case DSPF_RGB16:
          case DSPF_RGB24:
          case DSPF_RGB32:
          case DSPF_ARGB:
          case DSPF_ABGR:
          case DSPF_YUY2:
          case DSPF_UYVY:
          case DSPF_AYUV:
               break;

          default:
               fail |= CLRCF_FORMAT;
    }

    /* Check options */
    if (!D_FLAGS_ARE_IN( config->options, DLOP_OPACITY | DLOP_ALPHACHANNEL | DLOP_SRC_COLORKEY |
                                          DLOP_LR_MONO | DLOP_STEREO))
         fail |= CLRCF_OPTIONS;

    if (D_FLAGS_ARE_SET( config->options, DLOP_LR_MONO | DLOP_STEREO ))
         fail |= CLRCF_OPTIONS;

    if (failed)
         *failed = fail;

    if (fail)
    {
         D_ERROR("bcmNexus/Layer: %s() screen %d gfx layer %d: Failed (flags = 0x%08x)!\n", __FUNCTION__, displayId, layerId, fail);
         return DFB_UNSUPPORTED;
    }

    D_DEBUG_AT( bcmnexusLayer, "%s() screen %d gfx layer %d: Exit.\n", __FUNCTION__, displayId, layerId );
    return DFB_OK;
}

/* Common function to add a region to a layer */
static DFBResult
bcmAddRegion( CoreLayer             *layer,
              bcmLayerData          *layerData,
              bcmRegionData         *regionData,
              CoreLayerRegionConfig *config,
              int                    width,
              int                    height)
{
    DFBResult         res = DFB_OK;
    CoreSurface      *tmp_surface;
    DFBScreenID       displayId;
    DFBDisplayLayerID layerId;

    DFBBCMNEXUS                 *pBrcmDfb = (DFBBCMNEXUS *) dfb_system_data();
    BCMNEXUS_Options            *brcm_config = &pBrcmDfb->options;

    displayId = layerData->displayId;
    layerId   = layerData->layerId;

    D_DEBUG_AT( bcmnexusLayer, "%s() screen %d gfx layer %d: %dx%d, format=%s, region=%dx%d, caps=0x%08x,\n"
                               "bm=%d, options=0x%08x, source_id=%d, opacity=%d [%dx%d@%d,%d] -> [%dx%d@%d,%d]\n",
                               __FUNCTION__, displayId, layerId, width, height, dfb_pixelformat_name(config->format),
                               config->width, config->height, config->surface_caps, config->buffermode,
                               config->options, config->source_id, config->opacity,
                               config->source.w, config->source.h, config->source.x, config->source.y,
                               config->dest.w, config->dest.h, config->dest.x, config->dest.y);

    /* Lock access to the layerData */
    PTHREAD_ROBUST_MUTEX_LOCK( &layerData->lock );

    /* Save current region pointer in layer data */
    layerData->regionData = regionData;

    /* Ensure the update skip count is 0 initially */
    layerData->updateSkipCnt = 0;

    /* Store new region configuration */
    regionData->config = *config;

    /* Save Layer's source screen/display rectangle if not already set */
    if (layerData->srcRectangle.w == 0)
        layerData->srcRectangle = config->source;

    /* Save Layer's destination screen/display rectangle if not already set */
    if (layerData->destRectangle.w == 0)
        layerData->destRectangle = config->dest;

    if (brcm_config->ibuffer.width && brcm_config->ibuffer.height)
        layerData->ibuffer = true;

    /* If graphics feeder doesn't have a vertical scaler, then we need to create
       a temporary framebuffer to allow vertical scaling to occur */
    if (!layerData->hasVscaler || layerData->ibuffer)
    {
#if BCM_DFB_SURFACE_CREATE_WITH_HINT
        res = dfb_surface_create_simple( layer->core, width, height, config->format, config->colorspace,
                DSCAPS_VIDEOONLY | DSCAPS_STEREO, CSTF_SHARED, layerId, DSHF_LAYER, NULL, &tmp_surface );
#else
        res = dfb_surface_create_simple( layer->core, width, height, config->format, config->colorspace,
                DSCAPS_VIDEOONLY | DSCAPS_STEREO, CSTF_SHARED, layerId, NULL, &tmp_surface );
#endif

        if (res == DFB_OK)
        {
            D_DEBUG_AT( bcmnexusLayer, "%s() screen %d gfx layer %d: Created framebuffer %p: %dx%d (format=%s).\n", __FUNCTION__,
                        displayId, layerId, (void *)tmp_surface, width, height, dfb_pixelformat_name(config->format) );

            layerData->frameBuffer = tmp_surface;

            /* Make the surface global for all processes to access the reference count */
            res = dfb_surface_globalize( layerData->frameBuffer );
        }
    }
    /* Unlock access to the layerData */
    pthread_mutex_unlock( &layerData->lock );

    D_ASSERT( res == DFB_OK );

    D_DEBUG_AT( bcmnexusLayer, "%s() screen %d gfx layer %d: Exit [%d].\n", __FUNCTION__, displayId, layerId, res );
    return res;
}

/* Common function to set the primary region to the new settings. */
static DFBResult
bcmSetRegion( UNUSED_ CoreLayer                  *layer,
                      void                       *driver_data,
                      void                       *layer_data,
                      void                       *region_data,
                      CoreLayerRegionConfig      *config,
                      CoreLayerRegionConfigFlags  updated,
                      CoreSurface                *surface,
                      CorePalette                *palette,
                      CoreSurfaceBufferLock      *left_lock,
                      CoreSurfaceBufferLock      *right_lock )
{
    DFBResult              res;
    DFBBCMNEXUS           *dfb_bcmnexus = driver_data;
    bcmLayerData          *layerData    = layer_data;
    bcmRegionData         *regionData   = region_data;
    CoreSurfaceBufferLock  leftFrameBufferLock, rightFrameBufferLock;
    CoreLayerRegionConfig *regionConfig;
    NEXUS_SurfaceHandle    hLeftFrameBuffer, hRightFrameBuffer;
    DFB_Platform_P_GraphicsSettings graphicsSettings;
    DFBScreenID            displayId;
    DFBDisplayLayerID      layerId;
    DFB_PlatformDimension  displaySize;
    int                    source_width, source_height;
    bool                   reenable_gfx;
    bool                   displayOn;
#if BCM_DFB_USE_SURFACE_SET_STEREO_EYE
    DFBSurfaceStereoEye    surfaceEye = DSSE_LEFT, frameBufferEye = DSSE_LEFT;
#endif
    CoreSurfaceBuffer     *buffer;

    D_ASSERT( dfb_bcmnexus != NULL );

    /* Lock access to the layerData */
    PTHREAD_ROBUST_MUTEX_LOCK( &layerData->lock );

    displayId    = layerData->displayId;
    layerId      = layerData->layerId;
    regionConfig = &regionData->config;

    D_DEBUG_AT( bcmnexusLayer, "%s() screen %d gfx layer %d: updated flags=0x%08x, surface=%p,\n"
                               "handle=%p, format=%s, region=%dx%d, caps=0x%08x, bm=%d,\n"
                               "options=0x%08x, source_id=%d, opacity=%d [%dx%d@%d,%d] -> [%dx%d@%d,%d]\n",
                               __FUNCTION__, displayId, layerId, updated, (void *)surface, left_lock->handle, dfb_pixelformat_name(config->format),
                               config->width, config->height, config->surface_caps, config->buffermode,
                               config->options, config->source_id, config->opacity,
                               config->source.w, config->source.h, config->source.x, config->source.y,
                               config->dest.w, config->dest.h, config->dest.x, config->dest.y);

    if (regionData->leftBufferLock.allocation)
        dfb_surface_allocation_unref( regionData->leftBufferLock.allocation );

    regionData->leftBufferLock = *left_lock;

    dfb_surface_allocation_ref( regionData->leftBufferLock.allocation );

    if (config->options & DLOP_STEREO) {
        if (regionData->rightBufferLock.allocation)
            dfb_surface_allocation_unref( regionData->rightBufferLock.allocation );

        regionData->rightBufferLock = *right_lock;

        dfb_surface_allocation_ref( regionData->rightBufferLock.allocation );
    }

    hLeftFrameBuffer = left_lock->handle;
    hRightFrameBuffer = (config->options & DLOP_STEREO) ? right_lock->handle : NULL;

    if (D_FLAGS_IS_SET(updated, CLRCF_PALETTE) && palette) {
        /* iterate through all connected buffers */
        int i;

        /* Lock the surface prior to modifying the palette */
        if (dfb_surface_lock( surface )) {
            /* Unlock access to the layerData */
            pthread_mutex_unlock( &layerData->lock );
            return DFB_FUSION;
        }

        for( i=0; i<surface->num_buffers; i++ ) {
            CoreSurfaceBufferLock  lock;

            buffer = surface->left_buffers[i];
            res = dfb_surface_buffer_lock( buffer, CSAID_CPU, CSAF_WRITE | CSAF_READ, &lock );
            if ( res == DFB_OK ) {
                copy_palette( surface, lock.handle );
                dfb_surface_buffer_unlock( &lock );
            }
            else
                fprintf(stderr,"FIX THIS!!!\n");

            if (surface->config.caps & DSCAPS_STEREO)
            {
                buffer = surface->right_buffers[i];
                res = dfb_surface_buffer_lock( buffer, CSAID_CPU, CSAF_WRITE | CSAF_READ, &lock );
                if ( res == DFB_OK ) {
                    copy_palette( surface, lock.handle );
                    dfb_surface_buffer_unlock( &lock );
                }
                else
                    fprintf(stderr,"FIX THIS!!!\n");
            }
        }
        dfb_surface_unlock( surface );
    }

    if (!layerData->layerCallback) {
        layerData->layerCallback = frameBufferCallback;
    }

    if (!layerData->frameBuffer) {
        layerData->frameBuffer = surface;
    }

    /* Save Region Configuration */
    if (D_FLAGS_IS_SET(updated, CLRCF_WIDTH))
        regionConfig->width = config->width;

    if (D_FLAGS_IS_SET(updated, CLRCF_HEIGHT))
            regionConfig->height = config->height;

    if (D_FLAGS_IS_SET(updated, CLRCF_FORMAT))
        regionConfig->format = config->format;

    if (D_FLAGS_IS_SET(updated, CLRCF_SURFACE_CAPS))
        regionConfig->surface_caps = config->surface_caps;

    if (D_FLAGS_IS_SET(updated, CLRCF_BUFFERMODE))
        regionConfig->buffermode = config->buffermode;

    if (D_FLAGS_IS_SET(updated, CLRCF_OPTIONS))
        regionConfig->options = config->options;

    if (D_FLAGS_IS_SET(updated, CLRCF_SOURCE_ID))
        regionConfig->source_id = config->source_id;

    if (D_FLAGS_IS_SET(updated, CLRCF_SOURCE)) {
        regionConfig->source = config->source;
        layerData->srcRectangle = config->source;
    }

    if (D_FLAGS_IS_SET(updated, CLRCF_DEST)) {
        regionConfig->dest = config->dest;
        layerData->destRectangle = config->dest;
    }

    if (D_FLAGS_IS_SET(updated, CLRCF_CLIPS)) {
        regionConfig->clips     = config->clips;
        regionConfig->num_clips = config->num_clips;
        regionConfig->positive  = config->positive;
    }

    if (D_FLAGS_IS_SET(updated, CLRCF_OPACITY))
        regionConfig->opacity = config->opacity;

    if (D_FLAGS_IS_SET(updated, CLRCF_ALPHA_RAMP)) {
        regionConfig->alpha_ramp[0] = config->alpha_ramp[0];
        regionConfig->alpha_ramp[1] = config->alpha_ramp[1];
        regionConfig->alpha_ramp[2] = config->alpha_ramp[2];
        regionConfig->alpha_ramp[3] = config->alpha_ramp[3];
    }

    if (D_FLAGS_IS_SET(updated, CLRCF_SRCKEY))
        regionConfig->src_key = config->src_key;

    if (D_FLAGS_IS_SET(updated, CLRCF_DSTKEY))
        regionConfig->dst_key = config->dst_key;

    if (D_FLAGS_IS_SET(updated, CLRCF_PARITY))
        regionConfig->parity = config->parity;

    /* Setup Source Configuration */
    layerData->sourceId = config->source_id;

    /* Get whether the screen/display is on */
    displayOn = (bcmGetPowerMode(dfb_screens_at(displayId)) == DSPM_ON);

    /* Check if we need to re-enable the graphics layer */
    reenable_gfx = ((layerData->layerEnabled == false) && displayOn);

    layerData->layerEnabled  = displayOn;
    layerData->srcColorKeyOn = D_FLAGS_IS_SET(regionConfig->options, DLOP_SRC_COLORKEY);

    /* Case 1: Per-pixel alpha blending enabled on a non-premultiplied source surface.
               In this case, we need to use the following blending equation:
               Oc = (Sc * Sa) + (1 - Sa) * Dc

       Case 2: Per-pixel alpha blending enabled on a pre-multiplied source surface.
               In this case, we don't need to multiply the Sc by the Sa as this has
               already been completed.  The Sa needs to be set to 1 by using the constant alpha
               value of 0xff:
               Oc = (Sc * Ca) + (1 - Sa) * Dc
               Ca = 0xff
       Case 3: Global alpha blending enabled on a source surface.
               In this case, we need to use the constant alpha as the "global" alpha value:
               Oc = (Sc * Ca) + (1 - Ca) * Dc
       Case 4: Both per-pixel alpha blending and global alpha value are set on a non-premultiplied source surface.
               In this case, we can use the standard blending equation as well as the independent alpha value:
               Oc = (Sc * Sa) + (1 - Sa) * Dc
       Case 5: Both per-pixel alpha blending and global alpha value are set on a premultiplied source surface.
               In this case, we can use the independent alpha value along with constant source alpha of 0xff:
               Oc = (Sc * Ca) + (1 - Sa) * Dc
               Ca = 0xff
    */

    layerData->srcBlendFunc             = DFB_PlatformBlendFactor_eOne;
    layerData->destBlendFunc            = DFB_PlatformBlendFactor_eZero;
    layerData->alpha                    = 0xFF;
    layerData->constantAlpha            = 0xFF;

    if (regionConfig->options & DLOP_ALPHACHANNEL) {
        layerData->srcBlendFunc         = DFB_PlatformBlendFactor_eSourceAlpha;
        layerData->destBlendFunc        = DFB_PlatformBlendFactor_eInverseSourceAlpha;

        if (surface->config.caps & DSCAPS_PREMULTIPLIED)
        {
            layerData->srcBlendFunc     = DFB_PlatformBlendFactor_eConstantAlpha;
        }

        if (!DFB_PIXELFORMAT_HAS_ALPHA(surface->config.format))
        {
            layerData->srcBlendFunc     = DFB_PlatformBlendFactor_eConstantAlpha;

        }
        layerData->alpha = (regionConfig->options & DLOP_OPACITY) ? regionConfig->opacity : 0xff;
    }
    else if (regionConfig->options & DLOP_OPACITY) {
        layerData->constantAlpha        = regionConfig->opacity;
        layerData->srcBlendFunc         = DFB_PlatformBlendFactor_eConstantAlpha;
        /* There is a bug in the VDC on some chips where setting the dest blend func to (1-Ca) causes
           a green tint.  The temporary solution is to set this to (1-Sa) if the constant alpha
           value is 0xFF. */
        /*constantAlpha needs to be set to 0xFF to avod vdc invalid blend errors*/
        layerData->constantAlpha = 0xFF;
        layerData->alpha = regionConfig->opacity;
        layerData->destBlendFunc    = DFB_PlatformBlendFactor_eInverseSourceAlpha;
    }

    /* Color keying doesn't work if the blending equations use constants
       e.g. Sa=Ca, Da=(1-Ca)  or Sa=1, Da=0 */
    if (layerData->srcColorKeyOn) {
        if(layerData->destBlendFunc != DFB_PlatformBlendFactor_eInverseSourceAlpha)
        {
        layerData->srcBlendFunc         = DFB_PlatformBlendFactor_eSourceAlpha;
        layerData->destBlendFunc        = DFB_PlatformBlendFactor_eInverseSourceAlpha;

        /* Ensure the opacity can still be changed, as this uses the alpha value */
        if (layerData->constantAlpha != 0xFF)
            layerData->alpha = layerData->constantAlpha;
    }

    }

    /* Now get the updated graphics settings */
    bcmGetLayerGraphicsSettings(dfb_bcmnexus, layerId, &graphicsSettings);

    source_width  = graphicsSettings.clip.width;
    source_height = graphicsSettings.clip.height;

    DFB_Platform_P_GetDisplaySize(displayId, &displaySize);

    D_DEBUG_AT( bcmnexusLayer, "%s() screen %d gfx layer %d: Requested: %dx%d: src:[%dx%d@%d,%d] => dst:[%dx%d@%d,%d], zorder=%d\n", __FUNCTION__,
                displayId, layerId,
                regionConfig->width, regionConfig->height,
                regionConfig->source.w, regionConfig->source.h, regionConfig->source.x, regionConfig->source.y,
                regionConfig->dest.w, regionConfig->dest.h, regionConfig->dest.x, regionConfig->dest.y,
                graphicsSettings.zorder );

    D_DEBUG_AT( bcmnexusLayer, "%s() screen %d gfx layer %d: Actual:    %dx%d: src:[%dx%d@%d,%d] => dst:[%dx%d@%d,%d], zorder=%d\n", __FUNCTION__,
                displayId, layerId,
                displaySize.w, displaySize.h,
                source_width, source_height, graphicsSettings.clip.x, graphicsSettings.clip.y,
                graphicsSettings.position.width, graphicsSettings.position.height, graphicsSettings.position.x, graphicsSettings.position.y,
                graphicsSettings.zorder );

    if (layerData->srcColorKeyOn)
        D_DEBUG_AT( bcmnexusLayer, "%s() screen %d gfx layer %d: ColorKey[RGB]=0x%02x,0x%02x,0x%02x\n", __FUNCTION__,
                    displayId, layerId, regionConfig->src_key.r, regionConfig->src_key.g, regionConfig->src_key.b);

    D_DEBUG_AT( bcmnexusLayer, "%s() screen %d gfx layer %d: sourceBlendFactor=%d,destBlendFactor=%d,alpha=%d,constantAlpha=%d\n", __FUNCTION__,
                displayId, layerId, graphicsSettings.sourceBlendFactor, graphicsSettings.destBlendFactor,
                graphicsSettings.alpha, graphicsSettings.constantAlpha);

    regionData->vscale = false;
    if ((graphicsSettings.clip.height != layerData->srcRectangle.h) || layerData->ibuffer)
    {
        DFBRectangle rect;

        D_DEBUG_AT( bcmnexusLayer, "%s() screen %d gfx layer %d: using blitter to vscale layer\n", __FUNCTION__, displayId, layerId);

        if (graphicsSettings.clip.height != layerData->srcRectangle.h)
        regionData->vscale = true;

        /* Copy the source surface to the layer's framebuffer */
        D_DEBUG_AT( bcmnexusLayer, "%s() screen %d gfx layer %d: Going to flush %p...\n", __FUNCTION__, displayId, layerId, (void *)hLeftFrameBuffer );
        NEXUS_Surface_Flush( hLeftFrameBuffer );
        if ( hRightFrameBuffer )
            NEXUS_Surface_Flush( hRightFrameBuffer );

        if (dfb_surface_lock( layerData->frameBuffer )) {
            /* Unlock access to the layerData */
            pthread_mutex_unlock( &layerData->lock );
            return DFB_FUSION;
        }

        /* We use the M2MC to scale only the vertical and the GFD for the horizontal */
        rect.x = 0;
        rect.y = 0;

        bcmGetClipRectangle(layerData, displaySize.w, displaySize.h, source_width, source_height, &rect.w, &rect.h);

#if BCM_DFB_USE_SURFACE_SET_STEREO_EYE
        surfaceEye = dfb_surface_get_stereo_eye( surface );
        frameBufferEye = dfb_surface_get_stereo_eye( layerData->frameBuffer );
#endif

        bcm_gfx_stretch_stereo( surface, DSSE_LEFT, layerData->frameBuffer, DSSE_LEFT, &layerData->srcRectangle, &rect, false);

        if ( surface->config.caps & DSCAPS_STEREO )
            bcm_gfx_stretch_stereo( surface, DSSE_RIGHT, layerData->frameBuffer, DSSE_RIGHT, &layerData->srcRectangle, &rect, false);

#if BCM_DFB_USE_SURFACE_SET_STEREO_EYE
        dfb_surface_set_stereo_eye( surface, surfaceEye );
#endif

        dfb_gfxcard_sync();

        /* Need to instantiate a real surface buffer for the framebuffer */
        buffer = bcm_surface_get_buffer2( layerData->frameBuffer, CSBR_FRONT, DSSE_LEFT );
        D_MAGIC_ASSERT( buffer, CoreSurfaceBuffer );
        res = dfb_surface_buffer_lock( buffer, CSAID_GPU, CSAF_READ, &leftFrameBufferLock );
        if (res)
        {
            D_ERROR( "%s() screen %d gfx layer %d: Could NOT lock framebuffer surface! (res=%d)\n", __FUNCTION__, displayId, layerId, res );
#if BCM_DFB_USE_SURFACE_SET_STEREO_EYE
            dfb_surface_set_stereo_eye( layerData->frameBuffer, frameBufferEye );
#endif
            dfb_surface_unlock( layerData->frameBuffer );
            /* Unlock access to the layerData */
            pthread_mutex_unlock( &layerData->lock );
            return res;
        }

        if (regionData->leftBufferLock.allocation)
            dfb_surface_allocation_unref( regionData->leftBufferLock.allocation );

        regionData->leftBufferLock = leftFrameBufferLock;

        dfb_surface_allocation_ref( regionData->leftBufferLock.allocation );

        hLeftFrameBuffer = leftFrameBufferLock.handle;

        if ( surface->config.caps & DSCAPS_STEREO )
        {
            buffer = bcm_surface_get_buffer2( layerData->frameBuffer, CSBR_FRONT, DSSE_RIGHT );
            D_MAGIC_ASSERT( buffer, CoreSurfaceBuffer );
            res = dfb_surface_buffer_lock( buffer, CSAID_GPU, CSAF_READ, &rightFrameBufferLock );
            if ( res )
            {
                D_ERROR( "%s() screen %d gfx layer %d: Could NOT lock right framebuffer surface! (res=%d)\n", __FUNCTION__, displayId, layerId, res );
                dfb_surface_buffer_unlock( &leftFrameBufferLock );
#if BCM_DFB_USE_SURFACE_SET_STEREO_EYE
                dfb_surface_set_stereo_eye( layerData->frameBuffer, frameBufferEye );
#endif
                dfb_surface_unlock( layerData->frameBuffer );
                /* Unlock access to the layerData */
                pthread_mutex_unlock( &layerData->lock );
                return res;
            }
        }
#if BCM_DFB_USE_SURFACE_SET_STEREO_EYE
        dfb_surface_set_stereo_eye( layerData->frameBuffer, frameBufferEye );
#endif
    }

    /* Ensure the surface is flushed prior to the graphics feeder reading it. */
    D_DEBUG_AT( bcmnexusLayer, "%s() screen %d gfx layer %d: Going to flush %p...\n", __FUNCTION__, displayId, layerId, (void *)hLeftFrameBuffer );
    NEXUS_Surface_Flush( hLeftFrameBuffer );
    if ( hRightFrameBuffer )
        NEXUS_Surface_Flush( hRightFrameBuffer );

    /* If the graphics was disabled, then we need to call NEXUS_Display_SetGraphicsSettings first before calling
       NEXUS_Display_SetGraphicsFramebuffer.  If not, then we need to make sure we do this in the opposite order
       to prevent a stale surface buffer being used in the call to NEXUS_Display_SetGraphicsSettings! */

#if BCM_DFB_USE_TASK_MANAGER
    layerData->regionData->leftBufferLock.task = NULL;
    layerData->regionData->rightBufferLock.task = NULL;
#endif
    if (reenable_gfx == true)
    {
        bcmSetLayerGraphicsSettings(dfb_bcmnexus, layerId);
        bcmSetLayerFrameBuffer(dfb_bcmnexus, layerId,false);
    }
    else
    {
        bcmSetLayerFrameBuffer(dfb_bcmnexus, layerId,false);
        bcmSetLayerGraphicsSettings(dfb_bcmnexus, layerId);
    }

    regionData->updated = false;


    if (regionData->vscale == true)
    {
        dfb_surface_buffer_unlock(&leftFrameBufferLock);
        if (regionConfig->surface_caps & DSCAPS_STEREO)
        {
            dfb_surface_buffer_unlock(&rightFrameBufferLock);
        }

        if (dfb_surface_unlock( layerData->frameBuffer )) {
            /* Unlock access to the layerData */
            pthread_mutex_unlock( &layerData->lock );
            return DFB_FUSION;
        }
    }

    /* Unlock access to the layerData */
    pthread_mutex_unlock( &layerData->lock );

    D_DEBUG_AT( bcmnexusLayer, "%s() screen %d gfx layer %d: Exit.\n", __FUNCTION__, displayId, layerId );
    return DFB_OK;
}

/* Common function to remove a region from a layer */
static DFBResult
bcmRemoveRegion( UNUSED_ CoreLayer *layer,
                         void      *driver_data,
                         void      *layer_data,
                 UNUSED_ void      *region_data )
{
    DFBBCMNEXUS            *dfb_bcmnexus = driver_data;
    bcmLayerData           *layerData    = layer_data;
    DFBDisplayLayerID       layerId;
    DFBScreenID             displayId;

    D_ASSERT( dfb_bcmnexus != NULL );

    layerId      = layerData->layerId;
    displayId    = layerData->displayId;

    PTHREAD_ROBUST_MUTEX_LOCK( &layerData->lock );

    D_DEBUG_AT( bcmnexusLayer, "%s() screen %d gfx layer %d\n", __FUNCTION__, displayId, layerId );

    layerData->layerEnabled = false;
    bcmSetLayerGraphicsSettings(dfb_bcmnexus, layerId);

    if (!layerData->hasVscaler)
    {
        D_DEBUG_AT( bcmnexusLayer, "%s() screen %d gfx layer %d: Removing framebuffer %p.\n", __FUNCTION__,
                    displayId, layerId, (void *)layerData->frameBuffer);

        /* Decrement the internal reference count and if 0, then really destroy the surface */
        dfb_surface_unlink( &layerData->frameBuffer );
    }
    layerData->frameBuffer = NULL;
    layerData->regionData  = NULL;


#if BCM_DFB_USE_TASK_MANAGER
    DFB_DisplayTask *oldTaskLeft  = D_SYNC_FETCH_AND_CLEAR( &layerData->oldTaskLeft );
    DFB_DisplayTask *oldTaskRight = D_SYNC_FETCH_AND_CLEAR( &layerData->oldTaskRight );

    if (oldTaskLeft)
        Task_Done( oldTaskLeft );

    if (oldTaskRight)
        Task_Done( oldTaskRight );
#endif
    pthread_mutex_unlock( &layerData->lock );

    D_DEBUG_AT( bcmnexusLayer, "%s() screen %d gfx layer %d: Exit\n", __FUNCTION__, displayId, layerId );

    return DFB_OK;
}

/* Common function to invalidate a region */
static DFBResult
bcmInvalidateRegion( DFBBCMNEXUS           *dfb_bcmnexus,
                     bcmLayerData          *layerData,
                     bcmRegionData         *regionData,
                     CoreSurface           *surface,
                     DFBSurfaceFlipFlags    flags,
                     CoreSurfaceBufferLock *left_lock,
                     CoreSurfaceBufferLock *right_lock,
                     bool                   flipping)
{
    DFBResult              res = DFB_OK;
    DFBDisplayLayerID      layerId;
    DFBScreenID            displayId;
    DFB_PlatformDimension  displaySize;
    CoreSurfaceBufferLock  leftFrameBufferLock, rightFrameBufferLock;
    CoreSurfaceBufferLock *pLeftFrameBufferLock, *pRightFrameBufferLock;
    DFBRectangle           rect;
    bool                   vscale;
#if BCM_DFB_USE_SURFACE_SET_STEREO_EYE
    DFBSurfaceStereoEye    surfaceEye, frameBufferEye;
#endif
    CoreSurfaceBuffer     *buffer;
#if BCMNEXUS_NSC_SUPPORT == BCMNEXUS_NSC_MULTIBUFFER_MODE

    struct timespec   ts;
    struct timeval    tp;
    int               rc;
#endif

    D_ASSERT( dfb_bcmnexus != NULL );

    layerId   = layerData->layerId;
    displayId = layerData->displayId;

    pLeftFrameBufferLock = left_lock;
    pRightFrameBufferLock = right_lock;

    vscale = regionData->vscale;

    D_DEBUG_AT( bcmnexusLayerX, "%s() screen %d gfx layer %d\n", __FUNCTION__, displayId, layerId);

    if (((vscale == true) || layerData->ibuffer ) && layerData->frameBuffer)
    {
        D_DEBUG_AT( bcmnexusLayerX, "%s() screen %d gfx layer %d: using blitter to vscale layer\n", __FUNCTION__, displayId, layerId);
        if (dfb_surface_lock( layerData->frameBuffer )) {
            return DFB_FUSION;
        }

        DFB_Platform_P_GetDisplaySize(displayId, &displaySize);

        /* We use the M2MC to scale only the vertical and the GFD for the horizontal */
        rect.x = 0;
        rect.y = 0;

        bcmGetClipRectangle(layerData, displaySize.w, displaySize.h, layerData->srcRectangle.w, MIN(layerData->destRectangle.h, displaySize.h), &rect.w, &rect.h);

        if (((rect.w == layerData->srcRectangle.w)) && (rect.h == MIN(layerData->destRectangle.h, displaySize.h)))
        {
            if (vscale != true)
            {
                /*No scaling required*/
                goto no_scale;
            }
        }

#if BCM_DFB_USE_SURFACE_SET_STEREO_EYE
        surfaceEye = dfb_surface_get_stereo_eye( surface );
        frameBufferEye = dfb_surface_get_stereo_eye( layerData->frameBuffer );
#endif

        bcm_gfx_stretch_stereo( surface, DSSE_LEFT, layerData->frameBuffer, DSSE_LEFT, &layerData->srcRectangle, &rect, flipping );

        if (surface->config.caps & DSCAPS_STEREO)
            bcm_gfx_stretch_stereo( surface, DSSE_RIGHT, layerData->frameBuffer, DSSE_RIGHT, &layerData->srcRectangle, &rect, flipping );

#if BCM_DFB_USE_SURFACE_SET_STEREO_EYE
        dfb_surface_set_stereo_eye( surface, surfaceEye );
#endif
        dfb_gfxcard_sync();

        pLeftFrameBufferLock = &leftFrameBufferLock;
        buffer = bcm_surface_get_buffer2( layerData->frameBuffer, CSBR_FRONT, DSSE_LEFT );

        D_MAGIC_ASSERT( buffer, CoreSurfaceBuffer );
        res = dfb_surface_buffer_lock( buffer, CSAID_GPU, CSAF_READ, pLeftFrameBufferLock );
        if (res)
        {
            D_ERROR( "%s() screen %d gfx layer %d: Could NOT lock framebuffer! (res=%d)\n", __FUNCTION__, displayId, layerId, res );
#if BCM_DFB_USE_SURFACE_SET_STEREO_EYE
            dfb_surface_set_stereo_eye( layerData->frameBuffer, frameBufferEye );
#endif
            dfb_surface_unlock( layerData->frameBuffer );
            return res;
        }

        if (surface->config.caps & DSCAPS_STEREO)
        {
#if BCM_DFB_USE_TASK_MANAGER
            rightFrameBufferLock.task = right_lock->task;
#endif
            pRightFrameBufferLock = &rightFrameBufferLock;
            buffer = bcm_surface_get_buffer2( layerData->frameBuffer, CSBR_FRONT, DSSE_RIGHT );

            D_MAGIC_ASSERT( buffer, CoreSurfaceBuffer );
            res = dfb_surface_buffer_lock( buffer, CSAID_GPU, CSAF_READ, pRightFrameBufferLock );
            if (res)
            {
                D_ERROR( "%s() screen %d gfx layer %d: Could NOT lock right framebuffer! (res=%d)\n", __FUNCTION__, displayId, layerId, res );
                dfb_surface_buffer_unlock( pLeftFrameBufferLock );
#if BCM_DFB_USE_SURFACE_SET_STEREO_EYE
                dfb_surface_set_stereo_eye( layerData->frameBuffer, frameBufferEye );
#endif
                dfb_surface_unlock( layerData->frameBuffer );
                return res;
            }
        }
#if BCM_DFB_USE_SURFACE_SET_STEREO_EYE
        dfb_surface_set_stereo_eye( layerData->frameBuffer, frameBufferEye );
#endif
    }
no_scale:

    D_DEBUG_AT( bcmnexusLayerX, "  -> enabled %d\n", layerData->layerEnabled );
    D_DEBUG_AT( bcmnexusLayerX, "  -> pLeftFrameBufferLock->handle %p\n", pLeftFrameBufferLock->handle );
    D_DEBUG_AT( bcmnexusLayerX, "  -> regionData->leftBufferLock.handle %p\n", regionData->leftBufferLock.handle );

    /* Only Set the Graphics Framebuffer if the layer is enabled and the framebuffer has changed. */
    if (layerData->layerEnabled &&
        (!regionData->updated ||
         ((pLeftFrameBufferLock->handle != regionData->leftBufferLock.handle) ||
          (pRightFrameBufferLock && (pRightFrameBufferLock->handle != regionData->rightBufferLock.handle)))))
    {
        regionData->updated = true;

        if (regionData->leftBufferLock.allocation)
            dfb_surface_allocation_unref( regionData->leftBufferLock.allocation );

        regionData->leftBufferLock = *pLeftFrameBufferLock;

        dfb_surface_allocation_ref( regionData->leftBufferLock.allocation );

        if (pRightFrameBufferLock) {
            if (regionData->rightBufferLock.allocation)
                dfb_surface_allocation_unref( regionData->rightBufferLock.allocation );

            regionData->rightBufferLock = *pRightFrameBufferLock;

            dfb_surface_allocation_ref( regionData->rightBufferLock.allocation );
        }
        else {
            if (regionData->rightBufferLock.allocation)
                dfb_surface_allocation_unref( regionData->rightBufferLock.allocation );

            memset(&regionData->rightBufferLock, 0, sizeof(regionData->rightBufferLock));
        }


        /* Inform DirectFB that the buffers have changed function */
#if BCM_DFB_USE_TASK_MANAGER
        if (flipping && !dfb_config->task_manager)
#else
        if (flipping)
#endif
        {
            dfb_surface_flip( surface, false );
        }

#if BCM_DFB_USE_TASK_MANAGER
        if (left_lock->task)
            Task_Log( left_lock->task, "bcmInvalidateRegion()" );
#endif

        res = bcmSetLayerFrameBuffer(dfb_bcmnexus, layerId, false);
        if (res == DFB_OK && flags & DSFLIP_WAIT) {
            D_DEBUG_AT( bcmnexusLayerX, "%s() screen %d gfx layer %d res %d: waiting for framebuffer %p callback...\n", __FUNCTION__,
                    displayId, layerId, res, (void *)layerData->frameBuffer);
        }

        if ((res == DFB_OK) && (flags & DSFLIP_WAIT)) {
            PTHREAD_ROBUST_MUTEX_LOCK( &layerData->frameBufferSemLock );
            while (layerData->flipPendingLeft || layerData->flipPendingRight)
            {
#if BCMNEXUS_NSC_SUPPORT == BCMNEXUS_NSC_MULTIBUFFER_MODE
                rc =  gettimeofday(&tp, NULL);
                /* Convert from timeval to timespec */
                ts.tv_sec  = tp.tv_sec;
                ts.tv_nsec = tp.tv_usec * 1000;
                ts.tv_sec += 1;
                rc = pthread_cond_timedwait(&layerData->frameBufferSem, &layerData->frameBufferSemLock, &ts);
                if (rc == ETIMEDOUT)
                {
                    break;
                }
#else
                pthread_cond_wait( &layerData->frameBufferSem, &layerData->frameBufferSemLock );

#endif
            }

            pthread_mutex_unlock( &layerData->frameBufferSemLock );

            D_DEBUG_AT( bcmnexusLayerX, "%s() screen %d gfx layer %d: received callback for framebuffer %p.\n", __FUNCTION__,
                    displayId, layerId, (void *)layerData->frameBuffer);
        }

    }
    else {
        /* Inform DirectFB that the buffers have changed function */
#if BCM_DFB_USE_TASK_MANAGER
                if (flipping && !dfb_config->task_manager)
#else
        if (flipping)
#endif
                {
            dfb_surface_flip( surface, false );
                }

#if BCM_DFB_USE_TASK_MANAGER
        if (left_lock->task)
            Task_Done( left_lock->task );

        if (pRightFrameBufferLock && right_lock->task)
            Task_Done( right_lock->task );
#endif
    }

    if ((vscale == true) && layerData->frameBuffer)
    {
        D_DEBUG_AT( bcmnexusLayerX, "%s() screen %d gfx layer %d: unlocking framebuffer %p\n", __FUNCTION__,
                    displayId, layerId, (void *)layerData->frameBuffer);

        dfb_surface_buffer_unlock( pLeftFrameBufferLock );
        if (surface->config.caps & DSCAPS_STEREO)
        {
            dfb_surface_buffer_unlock( pRightFrameBufferLock );
        }
        if (dfb_surface_unlock( layerData->frameBuffer )) {
            return DFB_FUSION;
        }
    }
    return res;
}

/* Common function to flip a region */
static DFBResult
bcmFlipRegion( DFBBCMNEXUS           *dfb_bcmnexus,
               bcmLayerData          *layerData,
               bcmRegionData         *regionData,
               CoreSurface           *surface,
               DFBSurfaceFlipFlags    flags,
               CoreSurfaceBufferLock *left_lock,
               CoreSurfaceBufferLock *right_lock)
{
    DFBResult res;

    res = bcmInvalidateRegion(dfb_bcmnexus, layerData, regionData, surface, flags, left_lock, right_lock, true);

    return res;
}

static DFBResult
bcmGetLevel( UNUSED_ CoreLayer *layer,
             UNUSED_ void      *driver_data,
                     void      *layer_data,
                     int       *level )
{
    bcmLayerData *layerData = layer_data;

    *level = layerData->level;

    D_DEBUG_AT(bcmnexusLayer, "%s() screen %d gfx layer %d: level=%d\n", __FUNCTION__, layerData->displayId, layerData->layerId, *level);

    return DFB_OK;
}

/* Set the z-order of this layer.
   0 is the default layer
   +ve are higher than this layer
   -ve are lower than this layer

   Nexus levels go from 0 (lowest) to BCMNEXUS_Z_ORDER_MAX (highest)
   Default level is 2 for graphics (which is 0 level for DFB)
*/
static DFBResult
bcmSetLevel( UNUSED_ CoreLayer *layer,
                     void      *driver_data,
                     void      *layer_data,
                     int        level )
{
    DFBResult              res          = DFB_INVARG;
    DFBBCMNEXUS           *dfb_bcmnexus = driver_data;
    bcmLayerData          *layerData    = layer_data;
    DFBDisplayLayerID      layerId;
    DFBScreenID            displayId;

    layerId   = layerData->layerId;
    displayId = layerData->displayId;

    D_DEBUG_AT(bcmnexusLayer, "%s() screen %d gfx layer %d: level=%d\n", __FUNCTION__, displayId, layerId, level);

    if ((level <= BCMNEXUS_Z_ORDER_MAX/2) && (level >= -BCMNEXUS_Z_ORDER_MAX/2))
    {
        /* Lock access to the layerData */
        PTHREAD_ROBUST_MUTEX_LOCK( &layerData->lock );

        layerData->level = level;
        res = bcmSetLayerGraphicsSettings(dfb_bcmnexus, layerId);
        if (res != DFB_OK)
            D_ERROR("%s() screen %d gfx layer %d: Cannot set layer level to %d!\n", __FUNCTION__,
                    displayId, layerId, level);

        /* Unlock access to the layerData */
        pthread_mutex_unlock( &layerData->lock );
    } else
        D_ERROR("%s() screen %d gfx layer %d: Invalid level %d! It needs to be between %d and -%d.\n", __FUNCTION__,
                displayId, layerId, level, BCMNEXUS_Z_ORDER_MAX/2, BCMNEXUS_Z_ORDER_MAX/2);

    D_DEBUG_AT( bcmnexusLayer, "%s() screen %d gfx layer %d: Exit [%d].\n", __FUNCTION__, displayId, layerId, res );
    return res;
}

/*
 * Set the stereo depth of this layer.
 *
 * The left buffer will be shifted +z and the right buffer
 * will be shifted -z.
 */
static DFBResult
bcmSetStereoDepth( UNUSED_ CoreLayer  *layer,
                    void               *driver_data,
                    void               *layer_data,
                    bool                follow_video,
                    int                 z )
{
    DFBResult              res = DFB_OK;
    DFBBCMNEXUS           *dfb_bcmnexus = driver_data;
    bcmLayerData          *layerData = layer_data;
    DFBScreenID            displayId;
    DFBDisplayLayerID      layerId;

    layerId   = layerData->layerId;
    displayId = layerData->displayId;

    D_DEBUG_AT(bcmnexusLayerX, "%s() screen %d gfx layer %d: %s z (%d)\n", __FUNCTION__, displayId, layerId,
        follow_video ? "metadata" : "fixed", z);

    if (follow_video) {
        D_ERROR("%s() follow_video only supported on BluRay platforms! %d %d\n", __FUNCTION__,
            displayId, layerId);
        return DFB_UNSUPPORTED;
    }

    /* Lock access to the layerData */
    PTHREAD_ROBUST_MUTEX_LOCK( &layerData->lock );

    layerData->follow_video = follow_video;
    layerData->z = z;
    if (layerData->layerEnabled) {
        res = bcmSetLayerGraphicsSettings(dfb_bcmnexus, layerId);
        if (res != DFB_OK)
        {
                D_ERROR("%s() screen %d gfx layer %d: Cannot set layer stereo depth (%d)!\n", __FUNCTION__,
                displayId, layerId, z);
        }
    }
    /* Unlock access to the layerData */
    pthread_mutex_unlock( &layerData->lock );

    return res;
}

/**************************************************
 * Setup the Primary Display Primary Layer/Region *
 **************************************************/

/* initialize the primary graphics layer data
 * this function is called for the master fusionee only
 */
static DFBResult
bcmInitPrimaryDisplayPrimaryLayer( CoreLayer                  *layer,
                                   void                       *driver_data,
                                   void                       *layer_data,
                                   DFBDisplayLayerDescription *description,
                                   DFBDisplayLayerConfig      *config,
                           UNUSED_ DFBColorAdjustment         *adjustment )
{
    DFBBCMNEXUS  *dfb_bcmnexus = driver_data;
    bcmLayerData *layerData    = layer_data;

    D_ASSERT( dfb_bcmnexus != NULL );

    layerData->layerId   = DLID_PRIMARY;
    layerData->displayId = dfb_screen_id_translated(dfb_layer_screen(layer));
    layerData->sourceId  = DLSID_SURFACE;
    layerData->hasVscaler = DFB_Platform_P_CheckGfxVScl(0);

    return bcmInitLayer(dfb_bcmnexus, layerData, description, config);
}

/* Add a primary region to the primary display's primary layer */
static DFBResult
bcmAddPrimaryDisplayPrimaryRegion(         CoreLayer             *layer,
                                   UNUSED_ void                  *driver_data,
                                           void                  *layer_data,
                                           void                  *region_data,
                                           CoreLayerRegionConfig *config )
{
    return bcmAddRegion(layer, layer_data, region_data, config, BCMNEXUS_MAX_HD_DISPLAY_WIDTH, BCMNEXUS_MAX_HD_DISPLAY_HEIGHT);
}

/* Flip the primary region */
static DFBResult
bcmFlipPrimaryDisplayPrimaryRegion( UNUSED_ CoreLayer             *layer,
                                            void                  *driver_data,
                                            void                  *layer_data,
                                            void                  *region_data,
                                            CoreSurface           *surface,
                                            DFBSurfaceFlipFlags    flags,
                                            const DFBRegion       *left_update,
                                            CoreSurfaceBufferLock *left_lock,
                                            const DFBRegion       *right_update,
                                            CoreSurfaceBufferLock *right_lock )
{
    DFBResult             res;
    DFBBCMNEXUS          *dfb_bcmnexus = driver_data;
    bcmLayerData         *layerData    = layer_data;
    DFBDisplayLayerID     layerId;
    DFBScreenID           displayId;
#if (NUM_DISPLAYS > 1)
    bcmLayerData         *layerDataSec;
    bcmRegionData        *regionData;
    DFBRectangle          rect;
    CoreSurfaceBufferLock leftFrameBufferLock, rightFrameBufferLock;
#if BCM_DFB_USE_SURFACE_SET_STEREO_EYE
    DFBSurfaceStereoEye   surfaceEye, frameBufferEye;
#endif
    CoreSurfaceBuffer    *buffer;
    DFB_PlatformDimension displaySize;
#endif
    DFBRectangle left_update_rect,right_update_rect;

    layerId   = layerData->layerId;
    displayId = layerData->displayId;

    if (left_update)
    {
        dfb_rectangle_from_region(&left_update_rect, left_update);
    }
    else
    {
        left_update_rect.x = layerData->srcRectangle.x;
        left_update_rect.y = layerData->srcRectangle.y;
        left_update_rect.h = layerData->srcRectangle.h;
        left_update_rect.w = layerData->srcRectangle.w;
    }

    if (right_update)
    {
        dfb_rectangle_from_region(&right_update_rect, right_update);
    }
    else
    {
        right_update_rect.x = layerData->srcRectangle.x;
        right_update_rect.y = layerData->srcRectangle.y;
        right_update_rect.h = layerData->srcRectangle.h;
        right_update_rect.w = layerData->srcRectangle.w;
    }

    D_DEBUG_AT( bcmnexusLayerX, "%s() screen %d gfx layer %d: surface=%p, flags=0x%08x, "
        "left_handle=%p right_handle=%p\n", __FUNCTION__, layerData->displayId, layerData->layerId,
        (void *)surface, flags, (void *)left_lock->handle, right_lock ? (void *)right_lock->handle : NULL );

    res = bcmFlipRegion(dfb_bcmnexus, layer_data, region_data, surface, flags, left_lock, right_lock);
    if (res != DFB_OK)
    {
        /*In some cases DirectFB-1.7 sets framebuffer surface without receiving recycle callback.*/
#if !defined(BCMNEXUS_NSC_SUPPORT)
        D_ERROR("%s() screen %d gfx layer %d: Could NOT flip primary region (res=%d)!\n", __FUNCTION__, displayId, layerId, res);
#endif
        return res;
    }

#if (NUM_DISPLAYS > 1)
    /* Check to see whether we need to use the M2MC to blit from the primary graphics source path
       to the secondary path when we are in mirroring mode. */
    layerDataSec = bcmGetLayerData(GET_DISPLAY_GFX_LAYER_ID(dfb_bcmnexus, PRIMARY_GFX_LAYER_INDEX, SECONDARY_DISPLAY_ID));
    if (layerDataSec != NULL)
    {

        PTHREAD_ROBUST_MUTEX_LOCK( &layerData->lock );

        regionData = layerDataSec->regionData;

        if (regionData && (layerDataSec->sourceId == DLSID_SURFACE) && layerDataSec->frameBuffer)
        {
            if (dfb_surface_lock( layerDataSec->frameBuffer )) {
                /* Unlock access to the layerData */
                pthread_mutex_unlock( &layerData->lock );
                return DFB_FUSION;
            }

            float mod_h = 0, mod_w = 0;

            /* Use the M2MC to scale the primary gfx layer H size to the H size of the secondary gfx layer.
               Also use the M2MC to scale the primary gfx layer V size to the V size of the secondary gfx layer
               if we have a vertical scaler in the graphics feeder (GFD), otherwise we need to use the M2MC to
               scale it vertically to the display height (which isn't necessarily the canvas height) */
            rect.x = 0;
            rect.y = 0;
            rect.w = regionData->config.width;
            if (layerDataSec->hasVscaler)
                rect.h = regionData->config.height;
            else
            {
                DFB_Platform_P_GetDisplaySize(SECONDARY_DISPLAY_ID, &displaySize);
                rect.h = MIN(layerDataSec->destRectangle.h, displaySize.h);
            }

            D_DEBUG_AT( bcmnexusLayerX, "%s() screen %d gfx layer %d: About to stretch blit surface %p to framebuffer %p [%dx%d@%d,%d]\n", __FUNCTION__,
                    displayId, layerId, (void *)surface, (void *)layerDataSec->frameBuffer, rect.w, rect.h, rect.x, rect.y);

            if (left_update)
            {
                /* layerData->SrcRectangle is the primary layer, regionData->config is the secondary */
                /* Calculate the difference to reduce output by*/
                mod_h = ((float)rect.h / (float) layerData->srcRectangle.h);
                mod_w = ((float)regionData->config.width / (float) layerData->srcRectangle.w);

                /* left_update_rect is the partial source retangle */
                rect.x = (left_update_rect.x * mod_w) + 0.5;
                rect.y = (left_update_rect.y * mod_h) + 0.5;;
                rect.h = (left_update_rect.h * mod_h) ;
                rect.w = (left_update_rect.w * mod_w) ;
            }

#if BCM_DFB_USE_SURFACE_SET_STEREO_EYE
            surfaceEye = dfb_surface_get_stereo_eye( surface );
            frameBufferEye = dfb_surface_get_stereo_eye( layerData->frameBuffer );
#endif

            bcm_gfx_stretch_stereo( surface, DSSE_LEFT, layerDataSec->frameBuffer, DSSE_LEFT, &left_update_rect, &rect, false );

            if (surface->config.caps & DSCAPS_STEREO)
                bcm_gfx_stretch_stereo( surface, DSSE_RIGHT, layerDataSec->frameBuffer, DSSE_RIGHT, &right_update_rect, &rect, false );

#if BCM_DFB_USE_SURFACE_SET_STEREO_EYE
            dfb_surface_set_stereo_eye( surface, surfaceEye );
#endif
            dfb_gfxcard_sync();

            buffer = bcm_surface_get_buffer2( layerDataSec->frameBuffer, CSBR_FRONT, DSSE_LEFT );

            D_MAGIC_ASSERT( buffer, CoreSurfaceBuffer );
            res = dfb_surface_buffer_lock( buffer, CSAID_GPU, CSAF_WRITE, &leftFrameBufferLock );
            if (res)
            {
                D_ERROR( "%s() screen %d gfx layer %d: Could NOT lock framebuffer! (res=%d)\n", __FUNCTION__, displayId, layerId, res );
#if BCM_DFB_USE_SURFACE_SET_STEREO_EYE
                dfb_surface_set_stereo_eye(layerDataSec->frameBuffer, frameBufferEye);
#endif
                dfb_surface_unlock(layerDataSec->frameBuffer);
                /* Unlock access to the layerData */
                pthread_mutex_unlock( &layerData->lock );
                return res;
            }

            memset( &rightFrameBufferLock, 0, sizeof(rightFrameBufferLock) );
            if (surface->config.caps & DSCAPS_STEREO)
            {
                buffer = bcm_surface_get_buffer2( layerDataSec->frameBuffer, CSBR_FRONT, DSSE_RIGHT );

                D_MAGIC_ASSERT( buffer, CoreSurfaceBuffer);
                res = dfb_surface_buffer_lock( buffer, CSAID_GPU, CSAF_WRITE, &rightFrameBufferLock );
                if (res)
                {
                    D_ERROR( "%s() screen %d gfx layer %d: Could NOT lock right framebuffer! (res=%d)\n", __FUNCTION__, displayId, layerId, res );
                    dfb_surface_buffer_unlock( &leftFrameBufferLock );
#if BCM_DFB_USE_SURFACE_SET_STEREO_EYE
                    dfb_surface_set_stereo_eye( layerDataSec->frameBuffer, frameBufferEye );
#endif
                    dfb_surface_unlock( layerDataSec->frameBuffer );
                    /* Unlock access to the layerData */
                    pthread_mutex_unlock( &layerData->lock );
                    return res;
                }
            }
#if BCM_DFB_USE_SURFACE_SET_STEREO_EYE
            dfb_surface_set_stereo_eye( layerDataSec->frameBuffer, frameBufferEye );
#endif

            if (layerDataSec->layerEnabled &&
                ((leftFrameBufferLock.handle != regionData->leftBufferLock.handle) ||
                 (rightFrameBufferLock.handle != regionData->rightBufferLock.handle)))
            {
                if (regionData->leftBufferLock.allocation)
                    dfb_surface_allocation_unref( regionData->leftBufferLock.allocation );

                regionData->leftBufferLock = leftFrameBufferLock;

                dfb_surface_allocation_ref( regionData->leftBufferLock.allocation );

                if (regionData->rightBufferLock.allocation)
                    dfb_surface_allocation_unref( regionData->rightBufferLock.allocation );

                regionData->rightBufferLock = rightFrameBufferLock;

                dfb_surface_allocation_ref( regionData->rightBufferLock.allocation );

                bcmSetLayerFrameBuffer( dfb_bcmnexus, SECONDARY_GFX_LAYER_INDEX, true );
            }

            dfb_surface_buffer_unlock( &leftFrameBufferLock );
            if (surface->config.caps & DSCAPS_STEREO)
            {
                dfb_surface_buffer_unlock( &rightFrameBufferLock );
            }

            if (dfb_surface_unlock( layerDataSec->frameBuffer )) {
                /* Unlock access to the layerData */
                pthread_mutex_unlock( &layerData->lock );
                return DFB_FUSION;
            }
        }
    }

    pthread_mutex_unlock( &layerData->lock );

#endif


    D_DEBUG_AT( bcmnexusLayerX, "%s() screen %d gfx layer %d: Exit.\n", __FUNCTION__, displayId, layerId );
    return DFB_OK;
}

/* Update the primary region */
static DFBResult
bcmUpdatePrimaryDisplayPrimaryRegion( UNUSED_ CoreLayer             *layer,
                                              void                  *driver_data,
                                              void                  *layer_data,
                                              void                  *region_data,
                                              CoreSurface           *surface,
                                              const DFBRegion       *left_update,
                                              CoreSurfaceBufferLock *left_lock,
                                              const DFBRegion       *right_update,
                                              CoreSurfaceBufferLock *right_lock )
{
    DFBResult             res;
    DFBBCMNEXUS          *dfb_bcmnexus = driver_data;
    bcmLayerData         *layerData    = layer_data;
    DFBDisplayLayerID     layerId;
    DFBScreenID           displayId;
    bool                  primaryUpdated;
    bcmRegionData        *regionData = region_data;
#if (NUM_DISPLAYS > 1)
    bcmLayerData         *layerDataSec;
    DFBRectangle          rect;
    CoreSurfaceBufferLock leftFrameBufferLock, rightFrameBufferLock;
#if BCM_DFB_USE_SURFACE_SET_STEREO_EYE
    DFBSurfaceStereoEye   surfaceEye, frameBufferEye;
#endif
    CoreSurfaceBuffer    *buffer;
    DFB_PlatformDimension displaySize;
#endif

    layerId   = layerData->layerId;
    displayId = layerData->displayId;

    D_DEBUG_AT( bcmnexusLayerX, "%s() screen %d gfx layer %d: surf=%p, left_handle=%p [%d,%d->%d,%d], right_handle=%p [%d,%d->%d,%d]\n",
        __FUNCTION__, displayId, layerId, (void *)surface,
        (void *)left_lock->handle, left_update->x1, left_update->y1, left_update->x2, left_update->y2,
        right_lock ? (void *)right_lock->handle : NULL, right_update ? right_update->x1 : 0, right_update ? right_update->y1 : 0,
        right_update ? right_update->x2 : 0, right_update ? right_update->y2 : 0);

    /* Lock access to the layerData */
    PTHREAD_ROBUST_MUTEX_LOCK( &layerData->lock );

    if(left_update)
    {
        dfb_rectangle_from_region(&regionData->left_update, left_update);
    }

    if(right_update)
    {
        dfb_rectangle_from_region(&regionData->right_update, right_update);
    }

    /* If update skip count is set to a value greater than 0, it means we won't
       update this layer region as often.  This is useful for reducing memory
       bandwidth and improving graphics performance.
    */
    if (layerData->updateSkipCnt) {
        layerData->updateSkipCnt--;
        primaryUpdated = false;
    } else {
#if BCMNEXUS_NSC_SUPPORT
        res = bcmFlipRegion(dfb_bcmnexus, layer_data, region_data, surface, DSFLIP_NONE, left_lock, right_lock);
#else
        res = bcmInvalidateRegion(dfb_bcmnexus, layer_data, region_data, surface, DSFLIP_NONE, left_lock, right_lock, false);
#endif

        if (res != DFB_OK)
        {
            /*In some cases DirectFB-1.7 sets framebuffer surface without receiving recycle callback.*/
#if !defined(BCMNEXUS_NSC_SUPPORT)
            D_ERROR("%s() screen %d gfx layer %d: Could NOT update primary region (res=%d)!\n", __FUNCTION__, displayId, layerId, res);
#endif
            /* Unlock access to the layerData */
            pthread_mutex_unlock( &layerData->lock );
            return res;
        }
        layerData->updateSkipCnt = bcmGetLayerDefaultUpdateSkipCnt(layerData->layerId);
        primaryUpdated = true;
    }

#if (NUM_DISPLAYS > 1)
    /* Check to see whether we need to use the M2MC to blit from the primary graphics source path
       to the secondary path when we are in mirroring mode. */
    layerDataSec = bcmGetLayerData(GET_DISPLAY_GFX_LAYER_ID(dfb_bcmnexus, PRIMARY_GFX_LAYER_INDEX, SECONDARY_DISPLAY_ID));
    if (layerDataSec && primaryUpdated)
    {
        regionData = layerDataSec->regionData;

        if (regionData && (layerDataSec->sourceId == DLSID_SURFACE) && layerDataSec->frameBuffer)
        {
            /* If update skip count is set to a value greater than 0, it means we won't
               update the secondary display with the contents of the primary as often.
               This is useful for reducing memory bandwidth and improving graphics
               performance.
            */
            if (layerDataSec->updateSkipCnt) {
                layerDataSec->updateSkipCnt--;
            } else {
                if (dfb_surface_lock( layerDataSec->frameBuffer )) {
                    /* Unlock access to the layerData */
                    pthread_mutex_unlock( &layerData->lock );
                    return DFB_FUSION;
                }

                /* Use the M2MC to scale the primary gfx layer H size to the H size of the secondary gfx layer.
                   Also use the M2MC to scale the primary gfx layer V size to the V size of the secondary gfx layer
                   if we have a vertical scaler in the graphics feeder (GFD), otherwise we need to use the M2MC to
                   scale it vertically to the display height (which isn't necessarily the canvas height) */
                rect.x = 0;
                rect.y = 0;
                rect.w = regionData->config.width;
                if (layerDataSec->hasVscaler)
                    rect.h = regionData->config.height;
                else
                {
                    DFB_Platform_P_GetDisplaySize(SECONDARY_DISPLAY_ID, &displaySize);
                    rect.h = MIN(layerDataSec->destRectangle.h, displaySize.h);
                }

                D_DEBUG_AT( bcmnexusLayerX, "%s() screen %d gfx layer %d: About to stretch blit surface %p to framebuffer %p [%dx%d@%d,%d]\n",
                            __FUNCTION__, displayId, layerId, (void *)surface, (void *)layerDataSec->frameBuffer,
                            rect.w, rect.h, rect.x, rect.y);


#if BCM_DFB_USE_SURFACE_SET_STEREO_EYE
                surfaceEye = dfb_surface_get_stereo_eye( surface );
                frameBufferEye = dfb_surface_get_stereo_eye( layerData->frameBuffer );
#endif

                bcm_gfx_stretch_stereo( surface, DSSE_LEFT, layerDataSec->frameBuffer, DSSE_LEFT, &layerData->srcRectangle, &rect, false );

                if (surface->config.caps & DSCAPS_STEREO)
                    bcm_gfx_stretch_stereo( surface, DSSE_RIGHT, layerDataSec->frameBuffer, DSSE_RIGHT, &layerData->srcRectangle, &rect, false );

#if BCM_DFB_USE_SURFACE_SET_STEREO_EYE
                dfb_surface_set_stereo_eye( surface, surfaceEye );
#endif
                dfb_gfxcard_sync();

                buffer = bcm_surface_get_buffer2( layerDataSec->frameBuffer, CSBR_FRONT, DSSE_LEFT );

                D_MAGIC_ASSERT( buffer, CoreSurfaceBuffer );
                res = dfb_surface_buffer_lock( buffer, CSAID_GPU, CSAF_WRITE, &leftFrameBufferLock );
                if (res)
                {
                    D_ERROR( "%s() screen %d gfx layer %d: Could NOT lock framebuffer! (res=%d)\n", __FUNCTION__, displayId, layerId, res );
#if BCM_DFB_USE_SURFACE_SET_STEREO_EYE
                    dfb_surface_set_stereo_eye(layerDataSec->frameBuffer, frameBufferEye);
#endif
                    dfb_surface_unlock(layerDataSec->frameBuffer);
                    /* Unlock access to the layerData */
                    pthread_mutex_unlock( &layerData->lock );
                    return res;
                }

                memset( &rightFrameBufferLock, 0, sizeof(rightFrameBufferLock) );
                if (surface->config.caps & DSCAPS_STEREO)
                {
                    buffer = bcm_surface_get_buffer2( layerDataSec->frameBuffer, CSBR_FRONT, DSSE_RIGHT );

                    D_MAGIC_ASSERT( buffer, CoreSurfaceBuffer);
                    res = dfb_surface_buffer_lock( buffer, CSAID_GPU, CSAF_WRITE, &rightFrameBufferLock );
                    if (res)
                    {
                        D_ERROR( "%s() screen %d gfx layer %d: Could NOT lock right framebuffer! (res=%d)\n", __FUNCTION__, displayId, layerId, res );
                        dfb_surface_buffer_unlock( &leftFrameBufferLock );
#if BCM_DFB_USE_SURFACE_SET_STEREO_EYE
                        dfb_surface_set_stereo_eye( layerDataSec->frameBuffer, frameBufferEye );
#endif
                        dfb_surface_unlock( layerDataSec->frameBuffer );
                        /* Unlock access to the layerData */
                        pthread_mutex_unlock( &layerData->lock );
                        return res;
                    }
                }
#if BCM_DFB_USE_SURFACE_SET_STEREO_EYE
                dfb_surface_set_stereo_eye( layerDataSec->frameBuffer, frameBufferEye );
#endif

                if (layerDataSec->layerEnabled &&
                    ((leftFrameBufferLock.handle != regionData->leftBufferLock.handle) ||
                     (rightFrameBufferLock.handle != regionData->rightBufferLock.handle)))
                {
                    if (regionData->leftBufferLock.allocation)
                        dfb_surface_allocation_unref( regionData->leftBufferLock.allocation );

                    regionData->leftBufferLock = leftFrameBufferLock;

                    dfb_surface_allocation_ref( regionData->leftBufferLock.allocation );

                    if (regionData->rightBufferLock.allocation)
                        dfb_surface_allocation_unref( regionData->rightBufferLock.allocation );

                    regionData->rightBufferLock = rightFrameBufferLock;

                    dfb_surface_allocation_ref( regionData->rightBufferLock.allocation );

                    bcmSetLayerFrameBuffer( dfb_bcmnexus, SECONDARY_GFX_LAYER_INDEX, true );
                }

                dfb_surface_buffer_unlock( &leftFrameBufferLock );
                if (surface->config.caps & DSCAPS_STEREO)
                {
                    dfb_surface_buffer_unlock( &rightFrameBufferLock );
                }

                if (dfb_surface_unlock( layerDataSec->frameBuffer )) {
                    /* Unlock access to the layerData */
                    pthread_mutex_unlock( &layerData->lock );
                    return DFB_FUSION;
                }
                layerDataSec->updateSkipCnt = bcmGetLayerDefaultUpdateSkipCnt(layerDataSec->layerId);
            }
        }
    }
#endif
    /* Unlock access to the layerData */
    pthread_mutex_unlock( &layerData->lock );

    D_DEBUG_AT( bcmnexusLayerX, "%s() screen %d gfx layer %d: Exit.\n", __FUNCTION__, displayId, layerId );
    return DFB_OK;
}

static DFBResult
bcmInitPrimaryMainLayerSource( UNUSED_ CoreLayer                         *layer,
                               UNUSED_ void                              *driver_data,
                                       void                              *layer_data,
                                       int                                source,
                                       DFBDisplayLayerSourceDescription  *description )
{
    bcmLayerData *layerData = layer_data;

    /* We know that this is called for the Primary Main layer */
    description->source_id = (source == 0) ? layerData->sourceId : (layerData->sourceId + 1);
    snprintf(description->name, DFB_DISPLAY_LAYER_SOURCE_DESC_NAME_LENGTH, "BCMNEXUS Gfx Source %d", description->source_id);
    description->caps = DDLSCAPS_SURFACE;

    return DFB_OK;
}


DisplayLayerFuncs bcmnexusPrimaryGfxLayerFuncs = {
    .LayerDataSize         = bcmLayerDataSize,
    .RegionDataSize        = bcmRegionDataSize,
    .InitLayer             = bcmInitPrimaryDisplayPrimaryLayer,
    .ShutdownLayer         = bcmShutdownLayer,
    .TestRegion            = bcmTestRegion,
    .AddRegion             = bcmAddPrimaryDisplayPrimaryRegion,
    .SetRegion             = bcmSetRegion,
    .RemoveRegion          = bcmRemoveRegion,
    .FlipRegion            = bcmFlipPrimaryDisplayPrimaryRegion,
    .UpdateRegion          = bcmUpdatePrimaryDisplayPrimaryRegion,
    .InitSource            = bcmInitPrimaryMainLayerSource,
    .SetLevel              = bcmSetLevel,
    .GetLevel              = bcmGetLevel,
    .SetStereoDepth        = bcmSetStereoDepth
};


/******************************************************
 * Setup the Secondary Display's Primary Layer/Region *
 ******************************************************/

#if (NUM_DISPLAYS > 1)
/* initialize the secondary graphics layer data
 * this function is called for the master fusionee only
 */
static DFBResult
bcmInitSecondaryDisplayPrimaryLayer(         CoreLayer                  *layer,
                                             void                       *driver_data,
                                             void                       *layer_data,
                                             DFBDisplayLayerDescription *description,
                                             DFBDisplayLayerConfig      *config,
                                     UNUSED_ DFBColorAdjustment         *adjustment )
{
    DFBBCMNEXUS  *dfb_bcmnexus = driver_data;
    bcmLayerData *layerData    = layer_data;

    D_ASSERT( dfb_bcmnexus != NULL );

    layerData->layerId   = DLID_PRIMARY+1;
    layerData->displayId = dfb_screen_id_translated(dfb_layer_screen(layer));
    layerData->sourceId  = DLSID_SURFACE+1;

    layerData->hasVscaler = DFB_Platform_P_CheckGfxVScl(SECONDARY_DISPLAY_ID);

    return bcmInitLayer(dfb_bcmnexus, layerData, description, config);
}

/* Add a region to the secondary display's primary layer */
static DFBResult
bcmAddSecondaryDisplayPrimaryRegion(         CoreLayer             *layer,
                                     UNUSED_ void                  *driver_data,
                                             void                  *layer_data,
                                             void                  *region_data,
                                             CoreLayerRegionConfig *config )
{
    return bcmAddRegion(layer, layer_data, region_data, config, BCMNEXUS_MAX_SD_DISPLAY_WIDTH, BCMNEXUS_MAX_SD_DISPLAY_HEIGHT);
}

/* Flip the secondary region */
static DFBResult
bcmFlipSecondaryDisplayPrimaryRegion( UNUSED_ CoreLayer             *layer,
                                              void                  *driver_data,
                                              void                  *layer_data,
                                              void                  *region_data,
                                              CoreSurface           *surface,
                                              DFBSurfaceFlipFlags    flags,
                                      UNUSED_ const DFBRegion       *left_update,
                                              CoreSurfaceBufferLock *left_lock,
                                      UNUSED_ const DFBRegion       *right_update,
                                              CoreSurfaceBufferLock *right_lock )
{
    DFBResult          res;
    bcmLayerData      *layerData = layer_data;
    DFBDisplayLayerID  layerId;
    DFBScreenID        displayId;

    layerId   = layerData->layerId;
    displayId = layerData->displayId;

    D_DEBUG_AT( bcmnexusLayerX, "%s() screen %d gfx layer %d: surface=%p, flags=0x%08x, left_handle=%p, right_handle=%p\n", __FUNCTION__,
                displayId, layerId, (void *)surface, flags, (void *)left_lock->handle, right_lock ? (void *)right_lock->handle : NULL );

    /* Lock access to the layerData */
    PTHREAD_ROBUST_MUTEX_LOCK( &layerData->lock );

    res = bcmFlipRegion(driver_data, layer_data, region_data, surface, flags, left_lock, right_lock);

    /* Unlock access to the layerData */
    pthread_mutex_unlock( &layerData->lock );

    if (res != DFB_OK) {
        D_ERROR("%s() screen %d gfx layer %d: Could NOT flip secondary region (res=%d)!\n", __FUNCTION__,  displayId, layerId, res);
        return res;
    }

    D_DEBUG_AT( bcmnexusLayerX, "%s() screen %d gfx layer %d: Exit.\n", __FUNCTION__, displayId, layerId );
    return DFB_OK;
}

/* Update the secondary region */
static DFBResult
bcmUpdateSecondaryDisplayPrimaryRegion( UNUSED_ CoreLayer             *layer,
                                                void                  *driver_data,
                                                void                  *layer_data,
                                                void                  *region_data,
                                                CoreSurface           *surface,
                                                const DFBRegion       *left_update,
                                                CoreSurfaceBufferLock *left_lock,
                                                const DFBRegion       *right_update,
                                                CoreSurfaceBufferLock *right_lock )
{
    DFBResult          res       = DFB_OK;
    bcmLayerData      *layerData = layer_data;
    DFBDisplayLayerID  layerId;
    DFBScreenID        displayId;

    layerId   = layerData->layerId;
    displayId = layerData->displayId;

    D_DEBUG_AT( bcmnexusLayerX, "%s() screen %d gfx layer %d: surf=%p, left_handle=%p [%d,%d->%d,%d], right_handle=%p [%d,%d->%d,%d]\n",
        __FUNCTION__, displayId, layerId, (void *)surface,
        (void *)left_lock->handle, left_update->x1, left_update->y1, left_update->x2, left_update->y2,
        right_lock ? (void *)right_lock->handle : NULL, right_update ? right_update->x1 : 0, right_update ? right_update->y1 : 0,
        right_update ? right_update->x2 : 0, right_update ? right_update->y2 : 0);

    /* Lock access to the layerData */
    PTHREAD_ROBUST_MUTEX_LOCK( &layerData->lock );

    /* If update skip count is set to a value greater than 0, it means we won't
       update this layer region as often.  This is useful for reducing memory
       bandwidth and improving graphics performance.
    */
    if (layerData->updateSkipCnt) {
        layerData->updateSkipCnt--;
    } else {

        res = bcmInvalidateRegion(driver_data, layer_data, region_data, surface, DSFLIP_NONE, left_lock, right_lock, false);

        if (res != DFB_OK)
            D_ERROR("%s() screen %d gfx layer %d: Could NOT update secondary region (res=%d)!\n", __FUNCTION__, displayId, layerId, res);

        layerData->updateSkipCnt = bcmGetLayerDefaultUpdateSkipCnt(layerData->layerId);
    }
    /* Unlock access to the layerData */
    pthread_mutex_unlock( &layerData->lock );

    D_DEBUG_AT( bcmnexusLayerX, "%s() screen %d gfx layer %d: Exit [%d].\n", __FUNCTION__, displayId, layerId, res );
    return res;
}

static DFBResult
bcmInitSecondaryMainLayerSource( UNUSED_ CoreLayer                         *layer,
                                 UNUSED_ void                              *driver_data,
                                         void                              *layer_data,
                                         int                                source,
                                         DFBDisplayLayerSourceDescription  *description )
{
    bcmLayerData *layerData = layer_data;

    /* We know that this is called for the Secondary Main layer */
    description->source_id = (source == 0) ? layerData->sourceId : (layerData->sourceId - 1);
    snprintf(description->name, DFB_DISPLAY_LAYER_DESC_NAME_LENGTH, "BCMNEXUS Gfx Source %d", description->source_id);
    description->caps = DDLSCAPS_SURFACE;

    return DFB_OK;
}

DisplayLayerFuncs bcmnexusSecondaryGfxLayerFuncs = {
    .LayerDataSize         = bcmLayerDataSize,
    .RegionDataSize        = bcmRegionDataSize,
    .InitLayer             = bcmInitSecondaryDisplayPrimaryLayer,
    .ShutdownLayer         = bcmShutdownLayer,
    .TestRegion            = bcmTestRegion,
    .AddRegion             = bcmAddSecondaryDisplayPrimaryRegion,
    .SetRegion             = bcmSetRegion,
    .RemoveRegion          = bcmRemoveRegion,
    .FlipRegion            = bcmFlipSecondaryDisplayPrimaryRegion,
    .UpdateRegion          = bcmUpdateSecondaryDisplayPrimaryRegion,
    .InitSource            = bcmInitSecondaryMainLayerSource,
    .SetLevel              = bcmSetLevel,
    .GetLevel              = bcmGetLevel,
};
#endif

/* Helper function to convert between old and new stereosopic APIs*/
void bcm_gfx_stretch_stereo( CoreSurface         *source,
                             DFBSurfaceStereoEye  source_eye,
                             CoreSurface         *destination,
                             DFBSurfaceStereoEye  destination_eye,
                             const DFBRectangle  *srect,
                             const DFBRectangle  *drect,
                             bool                 flipping )
{
#if BCM_DFB_USE_SURFACE_SET_STEREO_EYE
    dfb_surface_set_stereo_eye( source, source_eye );
    dfb_surface_set_stereo_eye( destination, destination_eye );
    dfb_gfx_stretch_to( source, destination, srect, drect, flipping );
#else
    dfb_gfx_stretch_stereo( source, source_eye, destination, destination_eye, srect, drect, flipping );
#endif
}

CoreSurfaceBuffer *
bcm_surface_get_buffer2( CoreSurface           *surface,
                         CoreSurfaceBufferRole  role,
                         DFBSurfaceStereoEye    eye )
{
#if BCM_DFB_USE_SURFACE_SET_STEREO_EYE
                    dfb_surface_set_stereo_eye( surface, eye );
                    return dfb_surface_get_buffer( surface, role );
#else
                    return dfb_surface_get_buffer2( surface, role, eye );
#endif
}
