/******************************************************************************
 *    (c)2011-2014 Broadcom Corporation
 *
 * This program is the proprietary software of Broadcom Corporation and/or its licensors,
 * and may only be used, duplicated, modified or distributed pursuant to the terms and
 * conditions of a separate, written license agreement executed between you and Broadcom
 * (an "Authorized License").  Except as set forth in an Authorized License, Broadcom grants
 * no license (express or implied), right to use, or waiver of any kind with respect to the
 * Software, and Broadcom expressly reserves all rights in and to the Software and all
 * intellectual property rights therein.  IF YOU HAVE NO AUTHORIZED LICENSE, THEN YOU
 * HAVE NO RIGHT TO USE THIS SOFTWARE IN ANY WAY, AND SHOULD IMMEDIATELY
 * NOTIFY BROADCOM AND DISCONTINUE ALL USE OF THE SOFTWARE.
 *
 * Except as expressly set forth in the Authorized License,
 *
 * 1.     This program, including its structure, sequence and organization, constitutes the valuable trade
 * secrets of Broadcom, and you shall use all reasonable efforts to protect the confidentiality thereof,
 * and to use this information only in connection with your use of Broadcom integrated circuit products.
 *
 * 2.     TO THE MAXIMUM EXTENT PERMITTED BY LAW, THE SOFTWARE IS PROVIDED "AS IS"
 * AND WITH ALL FAULTS AND BROADCOM MAKES NO PROMISES, REPRESENTATIONS OR
 * WARRANTIES, EITHER EXPRESS, IMPLIED, STATUTORY, OR OTHERWISE, WITH RESPECT TO
 * THE SOFTWARE.  BROADCOM SPECIFICALLY DISCLAIMS ANY AND ALL IMPLIED WARRANTIES
 * OF TITLE, MERCHANTABILITY, NONINFRINGEMENT, FITNESS FOR A PARTICULAR PURPOSE,
 * LACK OF VIRUSES, ACCURACY OR COMPLETENESS, QUIET ENJOYMENT, QUIET POSSESSION
 * OR CORRESPONDENCE TO DESCRIPTION. YOU ASSUME THE ENTIRE RISK ARISING OUT OF
 * USE OR PERFORMANCE OF THE SOFTWARE.
 *
 * 3.     TO THE MAXIMUM EXTENT PERMITTED BY LAW, IN NO EVENT SHALL BROADCOM OR ITS
 * LICENSORS BE LIABLE FOR (i) CONSEQUENTIAL, INCIDENTAL, SPECIAL, INDIRECT, OR
 * EXEMPLARY DAMAGES WHATSOEVER ARISING OUT OF OR IN ANY WAY RELATING TO YOUR
 * USE OF OR INABILITY TO USE THE SOFTWARE EVEN IF BROADCOM HAS BEEN ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGES; OR (ii) ANY AMOUNT IN EXCESS OF THE AMOUNT
 * ACTUALLY PAID FOR THE SOFTWARE ITSELF OR U.S. $1, WHICHEVER IS GREATER. THESE
 * LIMITATIONS SHALL APPLY NOTWITHSTANDING ANY FAILURE OF ESSENTIAL PURPOSE OF
 * ANY LIMITED REMEDY.
 *
 *****************************************************************************/

#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <dlfcn.h>

#include "platform_nexus_init.h"
#include "platform_display.h"
#include "platform_nexus_surface_compositor_display.h"

#if NEXUS_HAS_SURFACE_COMPOSITOR
#include "nexus_surface_compositor.h"
#endif

DFB_P_DisplaySettings *dfb_p_displaysettings;

// Copy rectangle contents from src to dest
#define SET_RECT(dest,src) { (dest).x = (src).x;          \
                             (dest).y = (src).y;          \
                             (dest).width = (src).width;  \
                             (dest).height = (src).height; }

BDBG_MODULE(surface_compositor_init);

static void recycle_all(NEXUS_SurfaceClientHandle blit_client);

static BKNI_EventHandle recycledEvent = NULL;
#if BCMNEXUS_NSC_SUPPORT != BCMNEXUS_NSC_TUNNELLED_MODE
static bool pushRecycleMode = true;
static void *lastFramebufferSet = NULL;
#endif
static DFB_PlatformCallbackDesc recycledCallbackContext = { NULL, NULL, 0 };
static NxClient_AllocResults nxClientAllocResults;
static unsigned nxClientId = UINT_MAX;

static void DFB_Platform_P_RecycledCallbackFunction(void *data, int blitClient)
{
    DFB_PlatformCallbackDesc *pCtx = (DFB_PlatformCallbackDesc *)data;

    BDBG_MSG(("------>%s: callback=%p >>>", __FUNCTION__, pCtx->callback));

    recycle_all((NEXUS_SurfaceClientHandle)blitClient);

    BKNI_SetEvent(recycledEvent);

    if (pCtx->callback)
    {
        (*pCtx->callback)(pCtx->context, pCtx->param);
    }
}

#if BCMNEXUS_NSC_SUPPORT == BCMNEXUS_NSC_MULTIBUFFER_MODE
static void recycle_all(NEXUS_SurfaceClientHandle blit_client)
{
    unsigned num;
    do
    {
        NEXUS_SurfaceHandle surface[MAX_NSC_RECYCLE];
        NEXUS_Error rc;
        unsigned i;

        rc = DFB_Platform_SurfaceClient_RecycleSurface(blit_client, surface, MAX_NSC_RECYCLE, &num);
        BDBG_ASSERT(!rc);
        for (i=0;i<num;i++)
        {
            unsigned j;
            /* if submitted infront, they may return out of order */
            for (j=0;j<NUM_NSC_SURFACES;j++)
            {
                if (surfaceQueue.surface[j].handle == surface[i])
                {
                    BDBG_MSG(("%s: surface=%p\n", __FUNCTION__, surfaceQueue.surface[j].handle));
                    BDBG_ASSERT(surfaceQueue.surface[j].submitted);
                    surfaceQueue.surface[j].submitted = false;
                    surfaceQueue.surface[j].handle = NULL;
                    surfaceQueue.depth--;
                }
            }
        }
    } while (num >= MAX_NSC_RECYCLE);
}
#endif

void
DFB_Platform_P_HDMI_Hotplug_Callback( void *pParam,
                                      int   iParam )
{
    BSTD_UNUSED(pParam);
    BSTD_UNUSED(iParam);
}

DFB_PlatformResult
DFB_Platform_P_HdmiHotplugHandler( DFB_PlatformNexusHandle displayHandle,
                                   DFB_PlatformNexusHandle hdmiHandle )
{
    BSTD_UNUSED(displayHandle);
    BSTD_UNUSED(hdmiHandle);
    return DFB_PLATFORM_OK;
}

DFB_PlatformResult
DFB_Platform_P_AddDisplayOutput_Hdmi( DFB_PlatformNexusHandle displayHandle,
                                      int                     outputIndex )
{
    BSTD_UNUSED(displayHandle);
    BSTD_UNUSED(outputIndex);
    return DFB_PLATFORM_UNSUPPORTED;
}

DFB_PlatformResult
DFB_Platform_P_RemoveDisplayOutput_Hdmi( DFB_PlatformNexusHandle displayHandle,
                                         int                     outputIndex )
{
    BSTD_UNUSED(displayHandle);
    BSTD_UNUSED(outputIndex);
    return DFB_PLATFORM_UNSUPPORTED;
}

DFB_PlatformResult
DFB_Platform_P_AddDisplayOutput_Component( DFB_PlatformNexusHandle displayHandle,
                                           int                     outputIndex )
{
    BSTD_UNUSED(displayHandle);
    BSTD_UNUSED(outputIndex);
    return DFB_PLATFORM_UNSUPPORTED;
}

DFB_PlatformResult
DFB_Platform_P_RemoveDisplayOutput_Component( DFB_PlatformNexusHandle displayHandle,
                                              int                     outputIndex )
{
    BSTD_UNUSED(displayHandle);
    BSTD_UNUSED(outputIndex);
    return DFB_PLATFORM_UNSUPPORTED;
}

DFB_PlatformResult
DFB_Platform_P_AddDisplayOutput_Composite( DFB_PlatformNexusHandle displayHandle,
                                           int                     outputIndex )
{
    BSTD_UNUSED(displayHandle);
    BSTD_UNUSED(outputIndex);
    return DFB_PLATFORM_UNSUPPORTED;
}

DFB_PlatformResult
DFB_Platform_P_RemoveDisplayOutput_Composite( DFB_PlatformNexusHandle displayHandle,
                                              int                     outputIndex )
{
    BSTD_UNUSED(displayHandle);
    BSTD_UNUSED(outputIndex);
    return DFB_PLATFORM_UNSUPPORTED;
}

DFB_PlatformResult
DFB_Platform_P_AddDisplayOutput_Svideo( DFB_PlatformNexusHandle displayHandle,
                                        int                     outputIndex )
{
    BSTD_UNUSED(displayHandle);
    BSTD_UNUSED(outputIndex);
    return DFB_PLATFORM_UNSUPPORTED;
}

DFB_PlatformResult
DFB_Platform_P_RemoveDisplayOutput_Svideo( DFB_PlatformNexusHandle displayHandle,
                                           int                     outputIndex )
{
    BSTD_UNUSED(displayHandle);
    BSTD_UNUSED(outputIndex);
    return DFB_PLATFORM_UNSUPPORTED;
}

DFB_PlatformResult
DFB_Platform_P_AddDisplayOutput_Scart( DFB_PlatformNexusHandle displayHandle,
                                       int                     outputIndex )
{
    BSTD_UNUSED(displayHandle);
    BSTD_UNUSED(outputIndex);
    return DFB_PLATFORM_UNSUPPORTED;
}

DFB_PlatformResult
DFB_Platform_P_RemoveDisplayOutput_Scart( DFB_PlatformNexusHandle displayHandle,
                                          int                     outputIndex )
{
    BSTD_UNUSED(displayHandle);
    BSTD_UNUSED(outputIndex);
    return DFB_PLATFORM_UNSUPPORTED;
}

DFB_PlatformResult
DFB_Platform_P_AddDisplayOutput_Rfm( DFB_PlatformNexusHandle displayHandle,
                                     int                     outputIndex )
{
    BSTD_UNUSED(displayHandle);
    BSTD_UNUSED(outputIndex);
    return DFB_PLATFORM_UNSUPPORTED;
}

DFB_PlatformResult
DFB_Platform_P_RemoveDisplayOutput_Rfm( DFB_PlatformNexusHandle displayHandle,
                                        int                     outputIndex )
{
    BSTD_UNUSED(displayHandle);
    BSTD_UNUSED(outputIndex);
    return DFB_PLATFORM_UNSUPPORTED;
}

void
DFB_Platform_P_DFBFormatToDisplaySize( DFB_PlatformVideoFormat format,
                                       DFB_PlatformDimension  *pSize )
{
    NEXUS_VideoFormatInfo info;

    if (format < DFB_PlatformVideoFormat_eMax)
    {
        NEXUS_VideoFormat_GetInfo(DFB_Platform_P_DFBToNexusVideoFormat(format), &info);
        pSize->w = info.width;
        pSize->h = info.height;
    }
}

DFB_PlatformResult
DFB_Platform_P_SetDisplaySettings( DFB_PlatformNexusHandle         displayHandle,
                                   DFB_Platform_P_DisplaySettings *pSettings,
                                   bool                            setSettings)
{
    DFB_PlatformResult       res = DFB_PLATFORM_OK;
    int                      displayIndex = -1;
#if BCMNEXUS_NXCLIENT_SUPPORT
    int                      rc;
    DFB_PlatformSettings     dfbPlatformSettings;
    NxClient_DisplaySettings displaySettings;
#endif

    displayIndex = DFB_Platform_P_GetDisplayIndex(displayHandle);

    if (displayIndex < 0)
    {
        return DFB_PLATFORM_INVARG;
    }
    else
    {
#if BCMNEXUS_NXCLIENT_SUPPORT
         res = DFB_Platform_GetSettings(&dfbPlatformSettings);
        /*NxClient server supports diplay settings changes by client apps.*/
        if(res != DFB_PLATFORM_OK)
        {
            BDBG_ERR(("%s() Failed to get DFB platform settings",__FUNCTION__));
            return DFB_PLATFORM_FAILURE;
        }

        if (dfbPlatformSettings.slave == DFB_Platform_SlaveType_NxClient)
        {
            char *pDisplayControl = NULL;
            unsigned long displayControl = 0;

            /*When using NxClient API, display setting changes made during initialization of DirectFB app
              can interere with display setting that are currently set by other app(s) in the system.
              The display setting changes by a DirectFB app are enabled if the environment variable
              dfb_display_control=1 is set.*/
            if ((pDisplayControl= getenv("dfb_display_control")) != NULL)
            {
                displayControl = strtoul(pDisplayControl, NULL, 10);
            }

            if(displayControl && setSettings)
            {
                NxClient_GetDisplaySettings(&displaySettings);
                displaySettings.format      = DFB_Platform_P_DFBToNexusVideoFormat(pSettings->format);
                displaySettings.aspectRatio = DFB_Platform_P_DFBToNexusDisplayAspectRatio(pSettings->aspectRatio);
#if BCMNEXUS_NSC_DISPLAY_BGCOLOR_SUPPORT
                displaySettings.backgroundColor  = DFB_Platform_P_DFBToNexusColor(pSettings->background);
#endif
                switch (pSettings->stereoMode)
                {
                    case DFB_PlatformStereoscopicMode_e2d:
                    {
                        BDBG_MSG_TRACE(("Trying to set non stereoscopic mode - 2D"));
#if BCMNEXUS_NSC_DISPLAY_3D_SUPPORT
                        displaySettings.display3DSettings.orientation = NEXUS_VideoOrientation_e2D;
#endif
                        break;
                    }

                    case DFB_PlatformStereoscopicMode_eSideBySideHalf:
                    {
                        BDBG_MSG_TRACE(("Using stereoscopic mode - Side by Side - Half"));
#if BCMNEXUS_NSC_DISPLAY_3D_SUPPORT
                        displaySettings.display3DSettings.orientation = NEXUS_VideoOrientation_e3D_LeftRight;
#endif
                        break;
                    }

                    case DFB_PlatformStereoscopicMode_eTopBottom:
                    {
                        BDBG_MSG_TRACE(("Using stereoscopic mode - Top Bottom - Half"));
#if BCMNEXUS_NSC_DISPLAY_3D_SUPPORT
                        displaySettings.display3DSettings.orientation = NEXUS_VideoOrientation_e3D_OverUnder;
#endif
                        break;
                    }

                    case DFB_PlatformStereoscopicMode_eFramePacking:
                    {
                        /* If frame packed we need to make sure we are in the correct 3D mode not 2D version. */
                        /* This relies on the ordering of the Nexus video formats (3D formats are at the end of the enum). */
                        if (pSettings->format < DFB_PlatformVideoFormat_e720p_3DOU_AS)
                        {
                            BDBG_MSG_TRACE(("HDMI 3D Mode and Nexus Display mode not compatible"));
                            return DFB_PLATFORM_INVARG;
                        }
                        BDBG_MSG_TRACE(("Using stereoscopic mode - Frame packed (Updating the display timing generator settings)"));
                        break;
                    }

                    case DFB_PlatformStereoscopicMode_eSideBySideFull:
                    {
                        break;
                    }

                    default:
                    {
                        BDBG_ERR(("%s() Unknown stereoscopic framing mode %d\n", __FUNCTION__, pSettings->stereoMode));
                        return DFB_PLATFORM_INVARG;
                    }
                }

                rc = NxClient_SetDisplaySettings(&displaySettings);
                if (!rc)
                {
                    memcpy(&dfb_p_displaysettings->display[displayIndex].displaySettings, pSettings, sizeof(*pSettings));
                    DFB_Platform_P_DFBFormatToDisplaySize(pSettings->format, &dfb_p_displaysettings->display[displayIndex].displaySize);
                }
                else
                {
                    BDBG_ERR(("%s() Failed to set Nexus display settings!\n",__FUNCTION__));
                    res = DFB_PLATFORM_FAILURE;
                }
            } /*displayControl*/
            else
            {
                /*Display settings changes are enabled only if the envar dfb_display_control=1 is set.*/
                DFB_Platform_P_DFBFormatToDisplaySize(pSettings->format, &dfb_p_displaysettings->display[displayIndex].displaySize);
            }
        }/*clientType*/
        else
#endif /*BCMNEXUS_NXCLIENT_SUPPORT*/
        {
            /*Display settings changes not supported when using non NxClient NSC APIs*/
            DFB_Platform_P_DFBFormatToDisplaySize(pSettings->format, &dfb_p_displaysettings->display[displayIndex].displaySize);
        }
    }

    return res;
}

/**
Summary:
Get current graphics compositing settings.
**/
DFB_PlatformResult DFB_Platform_P_GetGraphicsSettings(
    DFB_PlatformNexusHandle display,
    DFB_Platform_P_GraphicsSettings *pSettings /* [out] */
    )
{
    NEXUS_SurfaceClientSettings nexusSettings;

    DFB_Platform_SurfaceClient_GetSettings(display, &nexusSettings);

    BDBG_MSG_TRACE((">>%s", __FUNCTION__));

    pSettings->enabled = nexusSettings.composition.visible;
    pSettings->alpha   = 0xFF;
    pSettings->zorder  = nexusSettings.composition.zorder;

    pSettings->chromakeyEnabled = false;

    SET_RECT(pSettings->position, nexusSettings.composition.position);
    SET_RECT(pSettings->clip, nexusSettings.composition.position);

    pSettings->frameBufferCallback.callback = recycledCallbackContext.callback;
    pSettings->frameBufferCallback.context  = recycledCallbackContext.context;
    pSettings->frameBufferCallback.param    = recycledCallbackContext.param;

    pSettings->sourceBlendFactor = DFB_PlatformBlendFactor_eSourceAlpha;
    pSettings->destBlendFactor = DFB_PlatformBlendFactor_eInverseSourceAlpha;

    pSettings->constantAlpha = nexusSettings.composition.constantColor;

    pSettings->stereoOffset = nexusSettings.composition.rightViewOffset;

    BDBG_MSG_TRACE(("<<%s", __FUNCTION__));

    return DFB_PLATFORM_OK;
}

/**
Summary:
Set graphics compositing settings.
**/
DFB_PlatformResult DFB_Platform_P_SetGraphicsSettings(
    DFB_PlatformNexusHandle display,
    const DFB_Platform_P_GraphicsSettings *pSettings
    )
{
    DFB_PlatformResult rc;
    NEXUS_SurfaceClientSettings nexusSettings;

    BDBG_MSG_TRACE((">>%s", __FUNCTION__));

    if (recycledEvent == NULL)
    {
        BKNI_CreateEvent(&recycledEvent);
    }

    DFB_Platform_SurfaceClient_GetSettings(display, &nexusSettings);

    recycledCallbackContext.callback = pSettings->frameBufferCallback.callback;
    recycledCallbackContext.context  = pSettings->frameBufferCallback.context;
    recycledCallbackContext.param    = pSettings->frameBufferCallback.param;

#if BCMNEXUS_NSC_SUPPORT == BCMNEXUS_NSC_MULTIBUFFER_MODE
    /*Need the recycled callback in all cases to signal to DFB that somthing's been flipped.*/
    if (pushRecycleMode)
    {
        nexusSettings.recycled.callback = DFB_Platform_P_RecycledCallbackFunction;
        nexusSettings.recycled.context  = &recycledCallbackContext;
        nexusSettings.recycled.param    = (int) display;
    }
    else
    {
        nexusSettings.displayed.callback = DFB_Platform_P_RecycledCallbackFunction;
        nexusSettings.displayed.context  = &recycledCallbackContext;
    }

#else
    /* Tunnelled. */
    nexusSettings.displayed.callback = DFB_Platform_P_RecycledCallbackFunction;
    nexusSettings.displayed.context  = &recycledCallbackContext;
#endif

    rc = DFB_Platform_SurfaceClient_SetSettings(display, &nexusSettings);

    BDBG_MSG_TRACE(("<<%s", __FUNCTION__));

    return rc;
}

void
DFB_Platform_P_GetDefaultDisplayGraphicsFramebuffer3D( DFB_Platform_P_DisplayGraphicsFramebuffer3D *pFrameBuffer3D )
{
    BSTD_UNUSED(pFrameBuffer3D);
}

DFB_PlatformResult
DFB_Platform_P_SetDisplayGraphicsFramebuffer3D(       DFB_PlatformNexusHandle                      display,
                                                const DFB_Platform_P_DisplayGraphicsFramebuffer3D *pFrameBuffer3D,
                                                const DFB_PlatformRect *pLeftUpdateRect,
                                                const DFB_PlatformRect *pRightUpdateRect,
                                                bool  singleBuffered)
{
    DFB_PlatformResult err = DFB_PLATFORM_OK;
    NEXUS_Error rc;
    int i;
    /*Until we do 3D NSC*/
    BSTD_UNUSED(*pRightUpdateRect);

#if BCMNEXUS_NSC_SUPPORT == BCMNEXUS_NSC_TUNNELLED_MODE
    BSTD_UNUSED(singleBuffered);
    BDBG_MSG(("%s: NEXUS_SurfaceClient_PushTunnelledSurface %p ...", __FUNCTION__, (void *)pFrameBuffer3D->main));
    err = DFB_Platform_SurfaceClient_PushTunneledSurface(display ,pFrameBuffer3D->main, pLeftUpdateRect);
#else

    BDBG_MSG(("%s: NEXUS_SurfaceClient_Push/Set/UpdateSurface %p (submitPtr=%d depth=%d)", __FUNCTION__, (void *)pFrameBuffer3D->main, surfaceQueue.submitPtr, surfaceQueue.depth));

    if (pLeftUpdateRect)
    {
        BDBG_MSG(("%s: Updating Left rectangle %d x %d @ %d x %d",__FUNCTION__,
                  pLeftUpdateRect->width,pLeftUpdateRect->height,pLeftUpdateRect->x,pLeftUpdateRect->y));
    }
    else
    {
        BDBG_MSG(("%s: No Left update rectangle",__FUNCTION__));
    }

    if (pRightUpdateRect)
    {
        BDBG_MSG(("%s: Updating Right rectangle %d x %d @ %d x %d",__FUNCTION__,
                  pRightUpdateRect->width,pRightUpdateRect->height,pRightUpdateRect->x,pRightUpdateRect->y));
    }
    else
    {
        BDBG_MSG(("%s: No Right update rectangle",__FUNCTION__));
    }

    if (pushRecycleMode)
    {
        if ((surfaceQueue.surface[surfaceQueue.submitPtr].submitted) || (surfaceQueue.depth >= MAX_NSC_DEPTH))
        {
            recycle_all((NEXUS_SurfaceClientHandle) display);
            if (surfaceQueue.depth >= MAX_NSC_DEPTH)
            {
                rc = BKNI_WaitForEvent(recycledEvent, BKNI_INFINITE);
                BDBG_ASSERT(!rc);
            }
        }

        for(i = 0;i<NUM_NSC_SURFACES;++i)
        {
            if(surfaceQueue.surface[i].handle == pFrameBuffer3D->main)
            {
                if(surfaceQueue.surface[i].submitted)
                {
                    /*surface should be recycled before submitting again.*/
                    recycle_all(display);
                    BDBG_MSG(("%s: Duplicate surface ", __FUNCTION__));
                    return DFB_PLATFORM_UNSUPPORTED;
                }
                else
                {
                    BDBG_MSG(("%s: Invalid condition", __FUNCTION__));
                    return DFB_PLATFORM_FAILURE;
                }
                break;
            }
        }

        if(i == NUM_NSC_SURFACES)
        {
            surfaceQueue.surface[surfaceQueue.submitPtr].handle = pFrameBuffer3D->main;
            surfaceQueue.surface[surfaceQueue.submitPtr].submitted = true;

            /*partial region updates not yet supported in refSW using push/recycle mode*/
            /* err = DFB_Platform_SurfaceClient_PushSurface(display, pFrameBuffer3D->main, pLeftUpdateRect); */
            err = DFB_Platform_SurfaceClient_PushSurface(display, surfaceQueue.surface[surfaceQueue.submitPtr].handle, NULL);
            BDBG_ASSERT(!err);

            if (++surfaceQueue.submitPtr == NUM_NSC_SURFACES)
            {
                surfaceQueue.submitPtr = 0;
            }
            surfaceQueue.depth++;

              /*recycle surface from recycle callback.See nexus/docs/Nexus_SurfaceCompositor.pdf*/
//            recycle_all(display);

            if (singleBuffered)
            {
                /* Applications that draw into the framebuffer
                   without flipping can't be seen without this */
                err = DFB_Platform_SurfaceClient_PublishSurface(display);
            }
        }
    }
    else {

        BDBG_MSG(("%s: NewSurface 0x%08x OldSurface 0x%08x", __FUNCTION__, pFrameBuffer3D->main, lastFramebufferSet));

        /*Must wait for this because it means that the old buffer has been copied out*/
        if (!surfaceQueue.depth)
        {
            err = BKNI_WaitForEvent(recycledEvent, BKNI_INFINITE);
            BDBG_ASSERT(!err);
            surfaceQueue.depth--;
        }

        if (surfaceQueue.surface[surfaceQueue.submitPtr].handle != (NEXUS_SurfaceHandle)&pFrameBuffer3D->main) {
            err = DFB_Platform_SurfaceClient_SetSurface(display, pFrameBuffer3D->main);
            surfaceQueue.surface[surfaceQueue.submitPtr].handle = (void *) pFrameBuffer3D->main;
            surfaceQueue.submitPtr++;
            surfaceQueue.depth++;
            if (++surfaceQueue.submitPtr == NUM_NSC_SURFACES)
            {
                surfaceQueue.submitPtr = 0;
            }
            surfaceQueue.depth++;
            if (++surfaceQueue.depth >= MAX_NSC_DEPTH)
            {
                surfaceQueue.depth = 0;
            }
        }
        else
        {
            err = DFB_Platform_SurfaceClient_UpdateSurface(display, pLeftUpdateRect);
        }
    }

#endif

    return err;
}

DFB_PlatformResult
DFB_Platform_P_ClearDisplayGraphicsFramebuffer3D(DFB_PlatformNexusHandle                      display)
{

    BDBG_MSG(("%s: Clearing.", __FUNCTION__));

    DFB_Platform_SurfaceClient_Clear(display);
    memset(&surfaceQueue, 0, sizeof(surfaceQueue));
    if (recycledEvent)
    {
        /* Release threads waiting
           for the recycled event. */
        BKNI_SetEvent(recycledEvent);
    }

    return DFB_PLATFORM_OK;
}

bool
DFB_Platform_P_DisplayWindow_GetDefaultInitSettings( unsigned                               windowSlot,
                                                     DFB_PlatformDisplayWindowInitSettings *pSettings )
{
    BSTD_UNUSED(windowSlot);
    BSTD_UNUSED(pSettings);
    return false;
}

DFB_PlatformResult
DFB_Platform_P_DisplayWindow_Init( DFB_PlatformDisplayWindowInitSettings *pSettings,
                                   DFB_PlatformNexusHandle               *pHandle )
{
    BSTD_UNUSED(pSettings);
    BSTD_UNUSED(pHandle);
    return DFB_PLATFORM_UNSUPPORTED;
}

void
DFB_Platform_P_DisplayWindow_Uninit( DFB_PlatformNexusHandle handle)
{
    BSTD_UNUSED(handle);
}

DFB_PlatformResult
DFB_Platform_P_DisplayOutput_Init(DFB_PlatformDisplayOutputInitSettings *pSettings,
                                  DFB_PlatformNexusHandle               *pHandle )
{
    BSTD_UNUSED(pSettings);
    BSTD_UNUSED(pHandle);
    return DFB_PLATFORM_UNSUPPORTED;
}

/*Dummy function to check GFD vscale support*/
bool
DFB_Platform_P_CheckGfxVScl( int displayIndex )
{
    BSTD_UNUSED(displayIndex);
    return true;
}

DFB_PlatformResult
DFB_Platform_P_Display_Init( unsigned                         displayIndex,
                             DFB_PlatformDisplayInitSettings *pSettings,
                             DFB_PlatformNexusHandle         *pHandle,
                             DFB_Platform_SlaveType           slaveType,
                             int                              clientId)
{
    DFB_PlatformResult             res;
//    int   clientId;
    char *pClientId;

    BDBG_MSG_TRACE((">>%s", __FUNCTION__));

    BDBG_ASSERT(pSettings);
    BDBG_ASSERT(pHandle);

    if (displayIndex != 0)
        return DFB_PLATFORM_OK;

    /* For connections to other servers than Trellis if we haven't been given a
       client ID we can choose something sensible for ourselves. */
    if ( (slaveType != DFB_Platform_SlaveType_Trellis) && (clientId == INT_MAX))
    {
        if ((pClientId = getenv("dfb_clientid")) != NULL)
        {
            clientId = strtoul(pClientId, NULL, 10);
        }
        else
            clientId = displayIndex;
    }

    BDBG_MSG(("%s: Acquiring surface client id=%d slaveType=%d ", __FUNCTION__, clientId, slaveType));

    switch (slaveType)
    {
        case DFB_Platform_SlaveType_Trellis:
        {
            /* Use the ID passed in on initialisation */
            if (clientId != INT_MAX)
            {
                *pHandle = DFB_Platform_SurfaceClient_Acquire(clientId,slaveType);
            }
            else
            {
                BDBG_ERR(("%s: Invalid clientId - You must specifiy a client ID on initialisation.", __FUNCTION__));
                res = DFB_PLATFORM_FAILURE;
            }
            break;
        }
#if BCMNEXUS_NXCLIENT_SUPPORT
        case DFB_Platform_SlaveType_NxClient:
        {
            if(clientId == 0)
            {
                NxClient_AllocSettings allocSettings;
                NEXUS_Error rc = NEXUS_SUCCESS;

                NxClient_GetDefaultAllocSettings(&allocSettings);

                allocSettings.surfaceClient = 1;

                rc = NxClient_Alloc(&allocSettings, &nxClientAllocResults);

                if (rc)
                {
                    BDBG_ERR(("%s: Failed during NxClient_Alloc() for display %d rc(%d)", __FUNCTION__, displayIndex, rc));
                    break;
                }
                clientId = nxClientAllocResults.surfaceClient[0].id;
                *pHandle = DFB_Platform_SurfaceClient_Acquire(nxClientAllocResults.surfaceClient[0].id, slaveType);
            }
            else /* Use the ID passed in on initialisation */
            {
                *pHandle = DFB_Platform_SurfaceClient_Acquire(clientId,slaveType);
            }
            /*Store for use in DFB_Platform_P_Get/SetClientComposition API*/
            nxClientId = clientId;
            break;
        }
#endif
        default:
        {
            /* Use Nexus Surface Compositor API directly */
            *pHandle = DFB_Platform_SurfaceClient_Acquire(clientId,slaveType);
        }
    }

    if (*pHandle == NULL)
    {
        BDBG_ERR(("%s: Cannot open surface compositor for display %d!", __FUNCTION__, displayIndex));
        res = DFB_PLATFORM_FAILURE;
    }
    else
    {
        BDBG_MSG(("%s: Display %d Handle=%p", __FUNCTION__, displayIndex, *pHandle));
        dfb_p_displaysettings->display[displayIndex].displayHandle               = *pHandle;
        res = DFB_PLATFORM_OK;
    }

#if BCMNEXUS_NSC_SUPPORT == BCMNEXUS_NSC_TUNNELLED_MODE
    BDBG_MSG(("%s: Display %d: Running NSC in Tunnelled mode", __FUNCTION__, displayIndex));
#else
    if (getenv("dfb_nexus_compositor_set"))
    {
        BDBG_MSG(("%s: Display %d: Running NSC in Set/Update mode", __FUNCTION__, displayIndex));
        pushRecycleMode = false;
    }
    else
    {
        BDBG_MSG(("%s: Display %d: Running NSC in Push/Recycle mode", __FUNCTION__, displayIndex));
        pushRecycleMode = true;
    }
#endif

    BDBG_MSG_TRACE(("<<%s", __FUNCTION__));
    return res;
}

void
DFB_Platform_P_Display_Uninit( DFB_PlatformNexusHandle handle, DFB_Platform_SlaveType slaveType)
{
    int displayIndex = 0;

    BDBG_MSG_TRACE((">>%s", __FUNCTION__));

    BDBG_ASSERT(handle);

    DFB_Platform_SurfaceClient_Release(handle);

    if(DFB_Platform_SlaveType_NxClient == slaveType)
    {
        NxClient_Free(&nxClientAllocResults);
        nxClientId = UINT_MAX;
    }

    if (recycledEvent)
    {
        BKNI_DestroyEvent(recycledEvent);
        recycledEvent = NULL;
    }

    dfb_p_displaysettings->display[displayIndex].displayHandle = NULL;

    BDBG_MSG_TRACE(("<<%s", __FUNCTION__));
}

bool
DFB_Platform_P_DisplayOutput_GetDefaultInitSettings( unsigned                               outputSlot,
                                                     DFB_PlatformDisplayOutputInitSettings *pSettings )
{
    bool init = false;

    BDBG_MSG_TRACE((">>%s", __FUNCTION__));

    BSTD_UNUSED(outputSlot);
    BSTD_UNUSED(pSettings);

    BDBG_MSG_TRACE(("<<%s", __FUNCTION__));
    return init;
}

DFB_PlatformResult DFB_Platform_P_GetClientComposition(DFB_PlatformSurfaceComposition *pClientComposition)
{
    DFB_PlatformResult       res;
    DFB_PlatformSettings     dfbPlatformSettings;
    NEXUS_SurfaceComposition comp;

    if(!pClientComposition)
    {
        BDBG_ERR(("%s() Invalid argument",__FUNCTION__));
        return DFB_PLATFORM_INVARG;
    }

    res = DFB_Platform_GetSettings(&dfbPlatformSettings);
    /*NxClient server supports diplay settings changes by client apps.*/
    if(res != DFB_PLATFORM_OK)
    {
        BDBG_ERR(("%s() Failed to get DFB platform settings",__FUNCTION__));
        return DFB_PLATFORM_FAILURE;
    }

    /*Verify client is initialized*/
    if(nxClientId == UINT_MAX)
    {
        BDBG_ERR(("%s() Invalid client ID %u",__FUNCTION__, nxClientId));
        return DFB_PLATFORM_FAILURE;
    }

    /*Client composition setting is supported using NxClient API*/
    if (dfbPlatformSettings.slave == DFB_Platform_SlaveType_NxClient)
    {
        NxClient_GetSurfaceClientComposition(nxClientId, &comp);

        /*Copy NxClient composition into DFB platform composition structure.*/
        pClientComposition->virtualDisplay.width           = comp.virtualDisplay.width;
        pClientComposition->virtualDisplay.height          = comp.virtualDisplay.height;

        pClientComposition->position.x                     = comp.position.x;
        pClientComposition->position.y                     = comp.position.y;
        pClientComposition->position.width                 = comp.position.width;
        pClientComposition->position.height                = comp.position.height;

        pClientComposition->clipRect.x                     = comp.clipRect.x;
        pClientComposition->clipRect.y                     = comp.clipRect.y;
        pClientComposition->clipRect.width                 = comp.clipRect.width;
        pClientComposition->clipRect.height                = comp.clipRect.height;

        pClientComposition->zorder                         = comp.zorder;
        pClientComposition->visible                        = comp.visible;
        pClientComposition->displayCache                   = comp.displayCache;

        pClientComposition->colorBlend.a                   = comp.colorBlend.a;
        pClientComposition->colorBlend.b                   = comp.colorBlend.b;
        pClientComposition->colorBlend.subtract_cd         = comp.colorBlend.subtract_cd;
        pClientComposition->colorBlend.c                   = comp.colorBlend.c;
        pClientComposition->colorBlend.d                   = comp.colorBlend.d;
        pClientComposition->colorBlend.subtract_e          = comp.colorBlend.subtract_e;
        pClientComposition->colorBlend.e                   = comp.colorBlend.e;

        pClientComposition->alphaBlend.a                   = comp.alphaBlend.a;
        pClientComposition->alphaBlend.b                   = comp.alphaBlend.b;
        pClientComposition->alphaBlend.subtract_cd         = comp.alphaBlend.subtract_cd;
        pClientComposition->alphaBlend.c                   = comp.alphaBlend.c;
        pClientComposition->alphaBlend.d                   = comp.alphaBlend.d;
        pClientComposition->alphaBlend.subtract_e          = comp.alphaBlend.subtract_e;
        pClientComposition->alphaBlend.e                   = comp.alphaBlend.e;
#if BCMNEXUS_NSC_COLORKEY_SUPPORT
        pClientComposition->colorKey.source.enabled        = comp.colorKey.source.enabled;
        pClientComposition->colorKey.source.lower          = comp.colorKey.source.lower;
        pClientComposition->colorKey.source.upper          = comp.colorKey.source.upper;
        pClientComposition->colorKey.source.mask           = comp.colorKey.source.mask;
        pClientComposition->colorKey.source.replace        = comp.colorKey.source.replace;
        pClientComposition->colorKey.source.replaceMask    = comp.colorKey.source.replaceMask;

        pClientComposition->colorKey.dest.enabled          = comp.colorKey.dest.enabled;
        pClientComposition->colorKey.dest.lower            = comp.colorKey.dest.lower;
        pClientComposition->colorKey.dest.upper            = comp.colorKey.dest.upper;
        pClientComposition->colorKey.dest.mask             = comp.colorKey.dest.mask;
        pClientComposition->colorKey.dest.replace          = comp.colorKey.dest.replace;
        pClientComposition->colorKey.dest.replaceMask      = comp.colorKey.dest.replaceMask;

        pClientComposition->horizontalFilter               = comp.horizontalFilter;

        pClientComposition->verticalFilter                 = comp.verticalFilter;

        pClientComposition->alphaPremultiplySourceEnabled  = comp.alphaPremultiplySourceEnabled;
#endif

        pClientComposition->constantColor                  = comp.constantColor;
        pClientComposition->rightViewOffset                = comp.rightViewOffset;
        pClientComposition->colorMatrixEnabled             = comp.colorMatrixEnabled;

        pClientComposition->colorMatrix.shift              = comp.colorMatrix.shift;
        if((sizeof(pClientComposition->colorMatrix.coeffMatrix)) == (sizeof(comp.colorMatrix.coeffMatrix)))
        {
            memcpy(comp.colorMatrix.coeffMatrix, pClientComposition->colorMatrix.coeffMatrix, sizeof(pClientComposition->colorMatrix.coeffMatrix));
        }
        else
        {
            BDBG_ERR(("%s() coeffMatrix size mismatch",__FUNCTION__));
            return DFB_PLATFORM_FAILURE;
        }

        pClientComposition->contentMode                    = comp.contentMode;

        res = DFB_PLATFORM_OK;
    }
    else
    {
        BDBG_ERR(("%s() Not supported for the non-NxCleint slave types",__FUNCTION__));
        res = DFB_PLATFORM_UNSUPPORTED;
    }

    return res;
}

DFB_PlatformResult DFB_Platform_P_SetClientComposition(DFB_PlatformSurfaceComposition *pClientComposition)
{
    DFB_PlatformResult       res;
    DFB_PlatformSettings     dfbPlatformSettings;
    NEXUS_SurfaceComposition comp;
    NEXUS_Error              rc;

    if(!pClientComposition)
    {
        BDBG_ERR(("%s() Invalid argument",__FUNCTION__));
        return DFB_PLATFORM_INVARG;
    }

    /*Verify client is initialized*/
    if(nxClientId == UINT_MAX)
    {
        BDBG_ERR(("%s() Invalid client ID %u",__FUNCTION__, nxClientId));
        return DFB_PLATFORM_FAILURE;
    }

    res = DFB_Platform_GetSettings(&dfbPlatformSettings);
    /*NxClient server supports diplay settings changes by client apps.*/
    if(res != DFB_PLATFORM_OK)
    {
        BDBG_ERR(("%s() Failed to get DFB platform settings",__FUNCTION__));
        return DFB_PLATFORM_FAILURE;
    }

    /*Client composition setting is supported using NxClient API*/
    if (dfbPlatformSettings.slave == DFB_Platform_SlaveType_NxClient)
    {
        /*Copy NxClient composition into DFB platform composition structure.*/
        comp.virtualDisplay.width          = pClientComposition->virtualDisplay.width;
        comp.virtualDisplay.height         = pClientComposition->virtualDisplay.height;

        comp.position.x                    = pClientComposition->position.x;
        comp.position.y                    = pClientComposition->position.y;
        comp.position.width                = pClientComposition->position.width;
        comp.position.height               = pClientComposition->position.height;

        comp.clipRect.x                    = pClientComposition->clipRect.x;
        comp.clipRect.y                    = pClientComposition->clipRect.y;
        comp.clipRect.width                = pClientComposition->clipRect.width;
        comp.clipRect.height               = pClientComposition->clipRect.height;

        comp.zorder                        = pClientComposition->zorder;
        comp.visible                       = pClientComposition->visible;
        comp.displayCache                  = pClientComposition->displayCache;

        comp.colorBlend.a                  = pClientComposition->colorBlend.a;
        comp.colorBlend.b                  = pClientComposition->colorBlend.b;
        comp.colorBlend.subtract_cd        = pClientComposition->colorBlend.subtract_cd;
        comp.colorBlend.c                  = pClientComposition->colorBlend.c;
        comp.colorBlend.d                  = pClientComposition->colorBlend.d;
        comp.colorBlend.subtract_e         = pClientComposition->colorBlend.subtract_e;
        comp.colorBlend.e                  = pClientComposition->colorBlend.e;

        comp.alphaBlend.a                  = pClientComposition->alphaBlend.a;
        comp.alphaBlend.b                  = pClientComposition->alphaBlend.b;
        comp.alphaBlend.subtract_cd        = pClientComposition->alphaBlend.subtract_cd;
        comp.alphaBlend.c                  = pClientComposition->alphaBlend.c;
        comp.alphaBlend.d                  = pClientComposition->alphaBlend.d;
        comp.alphaBlend.subtract_e         = pClientComposition->alphaBlend.subtract_e;
        comp.alphaBlend.e                  = pClientComposition->alphaBlend.e;
#if BCMNEXUS_NSC_COLORKEY_SUPPORT
        comp.colorKey.source.enabled       = pClientComposition->colorKey.source.enabled;
        comp.colorKey.source.lower         = pClientComposition->colorKey.source.lower;
        comp.colorKey.source.upper         = pClientComposition->colorKey.source.upper;
        comp.colorKey.source.mask          = pClientComposition->colorKey.source.mask;
        comp.colorKey.source.replace       = pClientComposition->colorKey.source.replace;
        comp.colorKey.source.replaceMask   = pClientComposition->colorKey.source.replaceMask;

        comp.colorKey.dest.enabled         = pClientComposition->colorKey.dest.enabled;
        comp.colorKey.dest.lower           = pClientComposition->colorKey.dest.lower;
        comp.colorKey.dest.upper           = pClientComposition->colorKey.dest.upper;
        comp.colorKey.dest.mask            = pClientComposition->colorKey.dest.mask;
        comp.colorKey.dest.replace         = pClientComposition->colorKey.dest.replace;
        comp.colorKey.dest.replaceMask     = pClientComposition->colorKey.dest.replaceMask;

        comp.horizontalFilter              = pClientComposition->horizontalFilter;

        comp.verticalFilter                = pClientComposition->verticalFilter;

        comp.alphaPremultiplySourceEnabled = pClientComposition->alphaPremultiplySourceEnabled;
#endif
        comp.constantColor                 = pClientComposition->constantColor;
        comp.rightViewOffset               = pClientComposition->rightViewOffset;
        comp.colorMatrixEnabled            = pClientComposition->colorMatrixEnabled;

        comp.colorMatrix.shift             = pClientComposition->colorMatrix.shift;
        if((sizeof(pClientComposition->colorMatrix.coeffMatrix)) == (sizeof(comp.colorMatrix.coeffMatrix)))
        {
            memcpy(comp.colorMatrix.coeffMatrix, pClientComposition->colorMatrix.coeffMatrix, sizeof(pClientComposition->colorMatrix.coeffMatrix));
        }
        else
        {
            BDBG_ERR(("%s() coeffMatrix size mismatch",__FUNCTION__));
            return DFB_PLATFORM_FAILURE;
        }

        comp.contentMode                   = pClientComposition->contentMode;
        rc = NxClient_SetSurfaceClientComposition(nxClientId, &comp);

        if(rc == NEXUS_SUCCESS)
        {
            res = DFB_PLATFORM_OK;
        }
        else
        {
            BDBG_ERR(("%s() Failed to set NxClient composition",__FUNCTION__));
            res = DFB_PLATFORM_FAILURE;
        }
    }
    else
    {
        BDBG_ERR(("%s() Not supported for the non-NxCleint slave types",__FUNCTION__));
        res = DFB_PLATFORM_UNSUPPORTED;
    }

    return res;
}
