/******************************************************************************
 *    (c)2010-2013 Broadcom Corporation
 *
 * This program is the proprietary software of Broadcom Corporation and/or its licensors,
 * and may only be used, duplicated, modified or distributed pursuant to the terms and
 * conditions of a separate, written license agreement executed between you and Broadcom
 * (an "Authorized License").  Except as set forth in an Authorized License, Broadcom grants
 * no license (express or implied), right to use, or waiver of any kind with respect to the
 * Software, and Broadcom expressly reserves all rights in and to the Software and all
 * intellectual property rights therein.  IF YOU HAVE NO AUTHORIZED LICENSE, THEN YOU
 * HAVE NO RIGHT TO USE THIS SOFTWARE IN ANY WAY, AND SHOULD IMMEDIATELY
 * NOTIFY BROADCOM AND DISCONTINUE ALL USE OF THE SOFTWARE.
 *
 * Except as expressly set forth in the Authorized License,
 *
 * 1.     This program, including its structure, sequence and organization, constitutes the valuable trade
 * secrets of Broadcom, and you shall use all reasonable efforts to protect the confidentiality thereof,
 * and to use this information only in connection with your use of Broadcom integrated circuit products.
 *
 * 2.     TO THE MAXIMUM EXTENT PERMITTED BY LAW, THE SOFTWARE IS PROVIDED "AS IS"
 * AND WITH ALL FAULTS AND BROADCOM MAKES NO PROMISES, REPRESENTATIONS OR
 * WARRANTIES, EITHER EXPRESS, IMPLIED, STATUTORY, OR OTHERWISE, WITH RESPECT TO
 * THE SOFTWARE.  BROADCOM SPECIFICALLY DISCLAIMS ANY AND ALL IMPLIED WARRANTIES
 * OF TITLE, MERCHANTABILITY, NONINFRINGEMENT, FITNESS FOR A PARTICULAR PURPOSE,
 * LACK OF VIRUSES, ACCURACY OR COMPLETENESS, QUIET ENJOYMENT, QUIET POSSESSION
 * OR CORRESPONDENCE TO DESCRIPTION. YOU ASSUME THE ENTIRE RISK ARISING OUT OF
 * USE OR PERFORMANCE OF THE SOFTWARE.
 *
 * 3.     TO THE MAXIMUM EXTENT PERMITTED BY LAW, IN NO EVENT SHALL BROADCOM OR ITS
 * LICENSORS BE LIABLE FOR (i) CONSEQUENTIAL, INCIDENTAL, SPECIAL, INDIRECT, OR
 * EXEMPLARY DAMAGES WHATSOEVER ARISING OUT OF OR IN ANY WAY RELATING TO YOUR
 * USE OF OR INABILITY TO USE THE SOFTWARE EVEN IF BROADCOM HAS BEEN ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGES; OR (ii) ANY AMOUNT IN EXCESS OF THE AMOUNT
 * ACTUALLY PAID FOR THE SOFTWARE ITSELF OR U.S. $1, WHICHEVER IS GREATER. THESE
 * LIMITATIONS SHALL APPLY NOTWITHSTANDING ANY FAILURE OF ESSENTIAL PURPOSE OF
 * ANY LIMITED REMEDY.
 *
 *****************************************************************************/

#ifndef __BCMNEXUS_COLORSPACE__
#define __BCMNEXUS_COLORSPACE__

/*
 * Color space conversion matricies.
 */
#define BCMNEXUS_COLOR_MATRIX_SHIFT     10

/* RGB -> BT.601 */
/* Y  = R *  0.257 + G *  0.504 + B *  0.098 + 16  */
/* Cb = R * -0.148 + G * -0.291 + B *  0.439 + 128 */
/* Cr = R *  0.439 + G * -0.368 + B * -0.071 + 128 */
static NEXUS_Graphics2DColorMatrix RGB_to_BT601 =
    MAKE_CSC_MATRIX(BCMNEXUS_COLOR_MATRIX_SHIFT,
         0.257,  0.504,  0.098,  0.000,    16.000,
        -0.148, -0.291,  0.439,  0.000,   128.000,
         0.439, -0.368, -0.071,  0.000,   128.000,
         0.000,  0.000,  0.000,  1.000,     0.000 );

/* RGB -> BT.709 */
/* Y  = R *  0.184 + G *  0.614 + B *  0.061 + 16  */
/* Cb = R * -0.101 + G * -0.339 + B *  0.440 + 128 */
/* Cr = R *  0.440 + G * -0.400 + B * -0.040 + 128 */
static NEXUS_Graphics2DColorMatrix RGB_to_BT709 =
    MAKE_CSC_MATRIX(BCMNEXUS_COLOR_MATRIX_SHIFT,
         0.184,  0.614,  0.061,  0.000,    16.000,
        -0.101, -0.339,  0.440,  0.000,   128.000,
         0.440, -0.400, -0.040,  0.000,   128.000,
         0.000,  0.000,  0.000,  1.000,     0.000 );

/* BT.601 -> RGB */
/* R = Y * 1.164 + Cb * 0.000 + Cr * 1.596 - 223 */
/* G = Y * 1.164 - Cb * 0.391 - Cr * 0.813 + 135 */
/* B = Y * 1.164 + Cb * 2.018 + Cr * 0.000 - 277 */
static NEXUS_Graphics2DColorMatrix BT601_to_RGB =
    MAKE_CSC_MATRIX(BCMNEXUS_COLOR_MATRIX_SHIFT,
         1.164,  0.000,  1.596,  0.000,  -223.000,
         1.164, -0.391, -0.813,  0.000,   135.000,
         1.164,  2.018,  0.000,  0.000,  -277.000,
         0.000,  0.000,  0.000,  1.000,     0.000 );

/* BT.709 -> RGB */
/* R = Y * 1.164 + Cb * 0.000 + Cr * 1.787 - 247 */
/* G = Y * 1.164 - Cb * 0.531 - Cr * 0.222 + 76 */
/* B = Y * 1.164 + Cb * 2.112 + Cr * 0.000 - 289 */
static NEXUS_Graphics2DColorMatrix BT709_to_RGB =
    MAKE_CSC_MATRIX(BCMNEXUS_COLOR_MATRIX_SHIFT,
         1.164,  0.000,  1.787,  0.000,  -247.000,
         1.164, -0.531, -0.222,  0.000,    76.000,
         1.164,  2.112,  0.000,  0.000,  -289.000,
         0.000,  0.000,  0.000,  1.000,     0.000 );

/* BT.601 -> BT.709 */
static NEXUS_Graphics2DColorMatrix BT601_to_BT709 =
    MAKE_CSC_MATRIX(BCMNEXUS_COLOR_MATRIX_SHIFT,
         1.000, -0.117, -0.206,  0.000,    40.961,
         0.000,  1.020,  0.114,  0.000,   -17.122,
         0.000,  0.076,  1.027,  0.000,   -13.040,
         0.000,  0.000,  0.000,  1.000,     0.000 );

/* BT.709 -> BT.601 */
static NEXUS_Graphics2DColorMatrix BT709_to_BT601 =
    MAKE_CSC_MATRIX(BCMNEXUS_COLOR_MATRIX_SHIFT,
         1.000, -0.061,  0.347,  0.000,   -37.497,
         0.000,  1.082, -0.200,  0.000,    15.569,
         0.000,  0.045,  0.866,  0.000,    12.118,
         0.000,  0.000,  0.000,  1.000,     0.000 );

/* BT.709 -> to Full Range BT.601 */
static NEXUS_Graphics2DColorMatrix BT709_to_BT601FR =
    MAKE_CSC_MATRIX(BCMNEXUS_COLOR_MATRIX_SHIFT,
         1.164, -0.071,  0.404,  0.000,   -62.187,
         0.000,  1.232, -0.229,  0.000,     0.087,
         0.000,  0.051,  0.987,  0.000,    -3.935,
         0.000,  0.000,  0.000,  1.000,     0.000 );

/* BT.601 -> Full Range BT.601 */
static NEXUS_Graphics2DColorMatrix BT601_to_BT601FR =
    MAKE_CSC_MATRIX(BCMNEXUS_COLOR_MATRIX_SHIFT,
         1.164,  0.000,  0.000,  0.000,   -19.010,
         0.000,  1.138,  0.000,  0.000,   -17.498,
         0.000,  0.000,  1.139,  0.000,   -17.628,
         0.000,  0.000,  0.000,  1.000,     0.000 );

/* Full Range BT.601 -> to BT.709 */
static NEXUS_Graphics2DColorMatrix BT601FR_to_BT709 =
    MAKE_CSC_MATRIX(BCMNEXUS_COLOR_MATRIX_SHIFT,
         0.859, -0.103, -0.180,  0.000,    52.317,
         0.000,  0.896,  0.100,  0.000,     0.405,
         0.000,  0.067,  0.902,  0.000,     3.928,
         0.000,  0.000,  0.000,  1.000,     0.000 );

/* Full Range BT.601 -> BT.601 */
static NEXUS_Graphics2DColorMatrix BT601FR_to_BT601 =
    MAKE_CSC_MATRIX(BCMNEXUS_COLOR_MATRIX_SHIFT,
         0.859,  0.000,  0.000,  0.000,    16.000,
         0.000,  0.878,  0.000,  0.000,    16.000,
         0.000,  0.000,  0.878,  0.000,    16.000,
         0.000,  0.000,  0.000,  1.000,     0.000 );

/* Full Range BT.601 -> RGB */
static NEXUS_Graphics2DColorMatrix BT601FR_to_RGB =
    MAKE_CSC_MATRIX(BCMNEXUS_COLOR_MATRIX_SHIFT,
         1.000,  0.000,  1.402,  0.000,  -179.456,
         1.000, -0.344, -0.714,  0.000,   135.460,
         1.000,  1.772,  0.000,  0.000,  -226.816,
         0.000,  0.000,  0.000,  1.000,     0.000 );

/* RGB -> Full Range BT.601 */
static NEXUS_Graphics2DColorMatrix RGB_to_BT601FR =
    MAKE_CSC_MATRIX(BCMNEXUS_COLOR_MATRIX_SHIFT,
         0.299,  0.587,  0.114,  0.000,     0.000,
        -0.169, -0.331,  0.500,  0.000,   128.000,
         0.500, -0.419, -0.081,  0.000,   128.000,
         0.000,  0.000,  0.000,  1.000,     0.000 );


/***************************************************************************
Description:
    This matrix is used to select which color conversion matrix should be
    used to convert from the input surface color space to the output surface
    color space.
****************************************************************************/
typedef enum {
    CS_UNKNOWN          = 0,    /* Indicies to matrix below */
    CS_BT601            = 1,
    CS_BT709            = 2,
    CS_BT601_FULLRANGE  = 3,
    CS_RGB              = 4,
    CS_LAST             = CS_RGB
}bcmnexus_ColorSpace;

#define NUM_COLORSPACES (CS_LAST - CS_UNKNOWN + 1)

static NEXUS_Graphics2DColorMatrix *bcmnexus_CSCMatrix[NUM_COLORSPACES][NUM_COLORSPACES] =
{
    {NULL,  NULL,                   NULL,                   NULL,                   NULL                },
    {NULL,  NULL,                   &BT601_to_BT709,        &BT601_to_BT601FR,      &BT601_to_RGB       },
    {NULL,  &BT709_to_BT601,        NULL,                   &BT709_to_BT601FR,      &BT709_to_RGB       },
    {NULL,  &BT601FR_to_BT601,      &BT601FR_to_BT709,      NULL,                   &BT601FR_to_RGB     },
    {NULL,  &RGB_to_BT601,          &RGB_to_BT709,          &RGB_to_BT601FR,        NULL                }
};

static __inline__ bcmnexus_ColorSpace
cs_to_index(DFBSurfaceColorSpace ColorSpace)
{
    switch (ColorSpace)
    {
        case DSCS_UNKNOWN:
            return CS_UNKNOWN;
        case DSCS_BT601:
            return CS_BT601;
        case DSCS_BT709:
            return CS_BT709;
        case DSCS_BT601_FULLRANGE:
            return CS_BT601_FULLRANGE;
        case DSCS_RGB:
            return CS_RGB;
        default:
            return CS_UNKNOWN;
    }
}

static __inline__ void
dfb_convert_colorspace(
    DFBSurfaceColorSpace fromCS, u16 C2, u16 C1, u16 C0,
    DFBSurfaceColorSpace toCS, u16 *pC2, u16 *pC1, u16 *pC0)
{
    NEXUS_Graphics2DColorMatrix *M;
    bcmnexus_ColorSpace from, to;
    u32 c2, c1, c0;

    from = cs_to_index(fromCS);
    to = cs_to_index(toCS);

    M = bcmnexus_CSCMatrix[from][to];
    if (M == NULL)
    {
        *pC2 = C2;
        *pC1 = C1;
        *pC0 = C0;
        return;
    }

    c2 = (M->coeffMatrix[0]*C2  + M->coeffMatrix[1]*C1  + M->coeffMatrix[2]*C0  + M->coeffMatrix[4])  >> M->shift;
    c1 = (M->coeffMatrix[5]*C2  + M->coeffMatrix[6]*C1  + M->coeffMatrix[7]*C0  + M->coeffMatrix[9])  >> M->shift;
    c0 = (M->coeffMatrix[10]*C2 + M->coeffMatrix[11]*C1 + M->coeffMatrix[12]*C0 + M->coeffMatrix[14]) >> M->shift;

    *pC2 = c2 > 0x100 ? 0x100 : (u16)c2;
    *pC1 = c1 > 0x100 ? 0x100 : (u16)c1;
    *pC0 = c0 > 0x100 ? 0x100 : (u16)c0;
}


#endif /* __BCMNEXUS_COLORSPACE__ */
