/***************************************************************************
 *     (c)2007-2013 Broadcom Corporation
 *
 *  This program is the proprietary software of Broadcom Corporation and/or its licensors,
 *  and may only be used, duplicated, modified or distributed pursuant to the terms and
 *  conditions of a separate, written license agreement executed between you and Broadcom
 *  (an "Authorized License").  Except as set forth in an Authorized License, Broadcom grants
 *  no license (express or implied), right to use, or waiver of any kind with respect to the
 *  Software, and Broadcom expressly reserves all rights in and to the Software and all
 *  intellectual property rights therein.  IF YOU HAVE NO AUTHORIZED LICENSE, THEN YOU
 *  HAVE NO RIGHT TO USE THIS SOFTWARE IN ANY WAY, AND SHOULD IMMEDIATELY
 *  NOTIFY BROADCOM AND DISCONTINUE ALL USE OF THE SOFTWARE.
 *
 *  Except as expressly set forth in the Authorized License,
 *
 *  1.     This program, including its structure, sequence and organization, constitutes the valuable trade
 *  secrets of Broadcom, and you shall use all reasonable efforts to protect the confidentiality thereof,
 *  and to use this information only in connection with your use of Broadcom integrated circuit products.
 *
 *  2.     TO THE MAXIMUM EXTENT PERMITTED BY LAW, THE SOFTWARE IS PROVIDED "AS IS"
 *  AND WITH ALL FAULTS AND BROADCOM MAKES NO PROMISES, REPRESENTATIONS OR
 *  WARRANTIES, EITHER EXPRESS, IMPLIED, STATUTORY, OR OTHERWISE, WITH RESPECT TO
 *  THE SOFTWARE.  BROADCOM SPECIFICALLY DISCLAIMS ANY AND ALL IMPLIED WARRANTIES
 *  OF TITLE, MERCHANTABILITY, NONINFRINGEMENT, FITNESS FOR A PARTICULAR PURPOSE,
 *  LACK OF VIRUSES, ACCURACY OR COMPLETENESS, QUIET ENJOYMENT, QUIET POSSESSION
 *  OR CORRESPONDENCE TO DESCRIPTION. YOU ASSUME THE ENTIRE RISK ARISING OUT OF
 *  USE OR PERFORMANCE OF THE SOFTWARE.
 *
 *  3.     TO THE MAXIMUM EXTENT PERMITTED BY LAW, IN NO EVENT SHALL BROADCOM OR ITS
 *  LICENSORS BE LIABLE FOR (i) CONSEQUENTIAL, INCIDENTAL, SPECIAL, INDIRECT, OR
 *  EXEMPLARY DAMAGES WHATSOEVER ARISING OUT OF OR IN ANY WAY RELATING TO YOUR
 *  USE OF OR INABILITY TO USE THE SOFTWARE EVEN IF BROADCOM HAS BEEN ADVISED OF
 *  THE POSSIBILITY OF SUCH DAMAGES; OR (ii) ANY AMOUNT IN EXCESS OF THE AMOUNT
 *  ACTUALLY PAID FOR THE SOFTWARE ITSELF OR U.S. $1, WHICHEVER IS GREATER. THESE
 *  LIMITATIONS SHALL APPLY NOTWITHSTANDING ANY FAILURE OF ESSENTIAL PURPOSE OF
 *  ANY LIMITED REMEDY.
 *
 ***************************************************************************/

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <limits.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>

#include "directfb.h"
#include "directfb_keynames.h"

#include "core/input.h"
#include "core/core.h"
#include "core/input_driver.h"
#include "core/system.h"

#include "misc/conf.h"

/*
** Broadcom specific includes.
*/
#include "platform_init.h"

#include "bcmnexus_ir.h"

#include "bcmnexus.h"

D_DEBUG_DOMAIN( bcmnexusIrInput, "bcmNexus/IrInput", "Broadcom NEXUS IR Input Event driver" );

DEFINE_MODULE_DIRECTORY( dfb_bcmnexus_ir_keycodes, "inputdrivers/bcmnexus", DFB_BCMNEXUS_IR_KEYCODES_ABI_VERSION );

DFB_INPUT_DRIVER( bcmNexusIrInputDevice )

/*
** type definitions.
*/

/*
** A container for the IR receiver variables.
*/
typedef struct {

    DFB_PlatformNexusHandle     irInput;
    struct timeval              timePrevEvent;  /* For hardware that doesn't implement "repeat" detection */
    CoreInputDevice            *pDfbDevice;     /* For access to the DFB event queue */
    DirectThread               *thread;         /* Input IR event handling thread */
    FusionSkirmish              lock;           /* Lock */
    bool                        cancel_thread;  /* Set to true to cancel thread */
    unsigned int                deviceId;       /* Device ID for Multi- IR support*/
    DFB_PlatformClientResources clientID;      /* Client ID for use with virtualised input devices */
} BDFB_ir_info;

/*
** Constant definitions.
*/

/*
** If hw events are arriving faster than every "DEFAULT_REPEAT_MILLI_SECS",
** then we'll make the gross assumption that the key is being held down.
** This is used in the event handlers to debounce the keys.
*/
#define DEFAULT_REPEAT_MILLI_SECS   150

/*
** Define maximum number of events that can be consumed in a single callback.
*/
#define NUM_EVENTS 10

/*
** Define the number of initial repeats to skip/discard before issuing
** an IR key event.
*/
#define DEFAULT_IR_SKIP_REPEAT_COUNT 2

/*Multi-IR support tested with RemoteA and cirnec codes. To support more remote inputs, increase
  the MAX_IR_SUPPORTED define.
  To test multiple Remotes following changes are required
  1.In /usr/local/etc/directfbrc file, add required remote formats as comma separated list
    bcmnexus-ir-protocol=CirNec,RemoteA
    bcmnexus-ir-keycodes=cirnec,RemoteA
  2.Change Nexus pin mux to allow reception of multiple IR inputs (The change is required in Nexus refSW)
    For example, on BCM97425 platform using Broad Band Studio:
    SYS_CTRL.SYS_AON.AON_UPGCORE.UPG_AON_27.PM_AON irr1_in: setting to IR_IN0. from it's default value IR_IN1.
    IMS ID #607356
*/
#define MAX_IR_SUPPORTED 2
/*
** Global data.
*/

static int g_iIrNumBasicCodes[MAX_IR_SUPPORTED];          /* these variables store the size of the IR keycode arrays */
static int g_iIrNumAdvanCodes[MAX_IR_SUPPORTED];

bcmnexus_IR_keycodes irKeycodes[MAX_IR_SUPPORTED];
KeyMapElement *g_pIrBasicCodeMap[MAX_IR_SUPPORTED];
KeyMapElement *g_pIrAdvanCodeMap[MAX_IR_SUPPORTED];
static char *irProtocolList[MAX_IR_SUPPORTED];
static char *irKeycodeList[MAX_IR_SUPPORTED];

/*
** Broadcom specific routines.
*/

DFB_PlatformIrInputMode BDFB_lookup_ir_protocol(
    const char *name )
{
    DFB_PlatformIrInputMode inputMode;

    D_DEBUG_AT(bcmnexusIrInput, "%s: looking for IR protocol \"%s\"\n", __FUNCTION__, name);

    if(!name)
    {
        D_ERROR("bcmNexus/IrInput: BDFB_lookup_ir_protocol(): Invalid parameter \n");
        return DFB_PlatformIrInputMode_eMax;
    }

    inputMode = DFB_PlatformLookupIrInputmode(name);

    if (inputMode >= DFB_PlatformIrInputMode_eMax )
    {
        /* should stop recursive lookup if things are bad */
        if ( strcasecmp(name, DIRECTFB_IR_PROTOCOL ) )
        {
            D_WARN("bcmNexus/IrInput: Cannot find IR protocol \"%s\" - defaulting to \"%s\"!\n", name, DIRECTFB_IR_PROTOCOL);
            inputMode = BDFB_lookup_ir_protocol(DIRECTFB_IR_PROTOCOL);
        }
        else
            D_ERROR("bcmNexus/IrInput: Cannot find IR protocol\n");
    }
    else
        D_DEBUG_AT(bcmnexusIrInput, "%s: found IR protocol \"%s\" [%d]\n", __FUNCTION__, name, inputMode);

    return inputMode;
}

static bool BDFB_load_ir_keycodes (
    const char *name,
    bcmnexus_IR_keycodes *pIRKeys)
{
    DirectLink *link;

    D_DEBUG_AT(bcmnexusIrInput, "%s: Looking for IR keycodes module: '%s'\n", __FUNCTION__,name);

    if((!name) || (!pIRKeys)){
        D_ERROR("bcmNexus/IrInput: %s(): Invalid parameter \n", __FUNCTION__);
        return false;
    }

    direct_modules_explore_directory( &dfb_bcmnexus_ir_keycodes );

    for ( link = (DirectLink*)dfb_bcmnexus_ir_keycodes.entries;
           link != NULL ;
          link = link->next )
    {
        DirectModuleEntry *module = (DirectModuleEntry*)link;
        IrKeycodesMapFuncs *funcs = NULL;

        funcs = (void*)direct_module_ref( module );
        if (!funcs)
            continue;

        if (!name || !strcasecmp( name, module->name ))
        {
            if (pIRKeys->module)
                direct_module_unref( pIRKeys->module );

            pIRKeys->module = module;
            pIRKeys->funcs  = funcs;
        }
        else
            direct_module_unref( module );
    }

    if (pIRKeys->module && pIRKeys->funcs)
    {
        D_DEBUG_AT(bcmnexusIrInput, "%s: Loaded IR keycodes module: '%s'\n", __FUNCTION__,name);
        return true;
    }

    /* Try default value - if we're not already trying the default. */
    if (strcasecmp(DIRECTFB_IR_KEYCODES,name))
        return BDFB_load_ir_keycodes(DIRECTFB_IR_KEYCODES,pIRKeys);

    D_ERROR( "bcmNexus/IrInput: IR keycodes module '%s' not found!\n", name ? name : "" );

    return false;
}


/*
** Routine to map the IR hw codes to DirectFB keycodes.
*/
static void BDFB_lookup_ir_keycodes(
    uint32_t                uiEventCode,
    DFBInputEvent *     pDfbEvent,
    unsigned int        udeviceId
    )
{
    int i;

    if(udeviceId >= MAX_IR_SUPPORTED)
    {
        D_ERROR("%s: Invalid Device ID\n", __FUNCTION__);
        return;
    }

    if(!pDfbEvent) {
        D_ERROR("%s: Invalid Input Event\n", __FUNCTION__);
        return;
    }

    /*
    **  Hardware keycode, i.e. no mapping applied.
    */
    pDfbEvent->key_code = uiEventCode;

    /*
    ** Basic mapping, modifier independent.
    ** If we don't get a "hit" on the hardware key code, we'll return "DIKI_UNKNOWN"
    */

    pDfbEvent->key_id = DIKI_UNKNOWN;

    for( i=0; i < g_iIrNumBasicCodes[udeviceId]; i++ )
    {
        if ( uiEventCode == g_pIrBasicCodeMap[udeviceId][ i ].hwEventCode )
        {
            pDfbEvent->key_id = g_pIrBasicCodeMap[udeviceId][ i ].dfbKeySymbol;
            break;
        }
    }

    /*
    ** Advanced mapping, unicode compatible, modified dependent.
    ** If we don't get a "hit" on the hardware key code, we'll return "DIKS_NULL".
    ** Is this the right solution for the real code?
    */

    pDfbEvent->key_symbol = DIKS_NULL;


    for( i=0; i < g_iIrNumAdvanCodes[udeviceId]; i++ )
    {
        if ( uiEventCode == g_pIrAdvanCodeMap[udeviceId][ i ].hwEventCode )
        {
            pDfbEvent->key_symbol = g_pIrAdvanCodeMap[udeviceId][ i ].dfbKeySymbol;
            break;
        }
    }

    return;

}       /* end of BDFB_lookup_ir_keycodes */


/*
**  Data ready callback for the IR receiver.
*/
static void BDFB_input_dataReady(void *context, int param)
{
    FusionSkirmish *lock = (FusionSkirmish *)context;

    BSTD_UNUSED(param);

    fusion_skirmish_prevail( lock );
    fusion_skirmish_notify(  lock );
    fusion_skirmish_dismiss( lock );
}       /* end of BDFB_input_dataReady() */

static void *BDFB_IrfraThread(DirectThread *thread, void *arg)
{
    DirectResult                res;
    DFB_PlatformResult          rc;
    DFB_PlatformInputDeviceEvent irEvent[NUM_EVENTS];
    size_t                      numEventsRead;
    bool                        overflow;
    uint32_t                    eventCode, lastEventCode = 0;
    int                         skipRepeatCnt = 0;
    bool                        bKeyRepeated = false;
    DFBInputEvent               dfbEvent;
    CoreInputDevice *           pDfbDevice;
    size_t                      i;
    DFBBCMNEXUS                 *pBrcmDfb = (DFBBCMNEXUS *) dfb_system_data();
    BCMNEXUS_Options            *brcm_config = &pBrcmDfb->options;
    unsigned int                timeout = brcm_config->bcmnexus_ir_timeout;

    BSTD_UNUSED(thread);

    BDFB_ir_info  *pInputInfo = (BDFB_ir_info *)arg;

    memset( &dfbEvent, 0, sizeof( dfbEvent ) );

    while ( pInputInfo && pInputInfo->cancel_thread == false)
    {
        /* Wait for Nexus IR Callback notification */
        fusion_skirmish_prevail( &pInputInfo->lock );
        res = fusion_skirmish_wait( &pInputInfo->lock, timeout );
        fusion_skirmish_dismiss( &pInputInfo->lock );

        if ( pInputInfo->cancel_thread == false )
        {
            /*
            ** If there is queue available to receive the data, send it along.
            ** This should always be true for this driver, but will check just to
            ** be certain.
            */
            pDfbDevice = (CoreInputDevice *)( pInputInfo->pDfbDevice );

            if ( pDfbDevice )
            {
                /* If no keycode event received for a given timeout, then generate a KEYRELEASE event */
                if (res == DR_TIMEOUT)
                {
                    if (dfbEvent.type & DIET_KEYPRESS)
                    {
                        dfbEvent.type = DIET_KEYRELEASE;
                        dfb_input_dispatch( pDfbDevice, &dfbEvent );
                    }
                }
                else
                {
                    rc = DFB_PlatformInputDevice_GetEvents(DFB_PlatformInputDevice_eInfraRed, pInputInfo->irInput, irEvent, NUM_EVENTS, &numEventsRead, &overflow);
                    if ( rc != NEXUS_SUCCESS)
                    {
                        D_ERROR("%s: NEXUS_IrInput_GetEvents failed %d\n", __FUNCTION__, rc);
                        continue;
                    }
                    if ( numEventsRead<1)
                    {
                        continue;
                    }
                    else if (overflow || numEventsRead > NUM_EVENTS)
                    {
                        D_ERROR("%s: NEXUS_IrInput_GetEvents under/overflowed events read=%d\n",__FUNCTION__, numEventsRead);
                    }

                    for (i=0; i<numEventsRead; i++)
                    {
                        D_DEBUG_AT(bcmnexusIrInput, "%s: IR event %d (%d,%d)\n", __FUNCTION__, i, irEvent[i].code, irEvent[i].repeat);

                        eventCode = irEvent[i].code;

                        /* If we've not received a KEYRELEASE timeout and we are pressing a new key
                           then we need to generate a KEYRELEASE event first. */
                        if (timeout && (eventCode != lastEventCode) && (dfbEvent.type == DIET_KEYPRESS))
                        {
                            dfbEvent.type = DIET_KEYRELEASE;
                            dfb_input_dispatch( pDfbDevice, &dfbEvent );
                        }

                        memset( &dfbEvent, 0, sizeof( dfbEvent ) );

                        lastEventCode = eventCode;

                        /* Setup the timestamp */
                        gettimeofday(&dfbEvent.timestamp, NULL);

                        /* Initialise Skip Repeat Count if first repeat noticed */
                        if ((bKeyRepeated == false) && (irEvent[i].repeat == true))
                        {
                            skipRepeatCnt = brcm_config->bcmnexus_ir_repeat_skip ? brcm_config->bcmnexus_ir_repeat_skip : DEFAULT_IR_SKIP_REPEAT_COUNT;
                        }

                        bKeyRepeated = irEvent[i].repeat;

                        dfbEvent.flags = ( DIEF_KEYID | DIEF_KEYSYMBOL | DIEF_KEYCODE | DIEF_TIMESTAMP );
                        dfbEvent.type  = DIET_KEYPRESS;

                        if (i < (numEventsRead-1))
                            dfbEvent.flags |= DIEF_FOLLOW;

                        if ( bKeyRepeated )
                            dfbEvent.flags |= DIEF_REPEAT;

                        BDFB_lookup_ir_keycodes( eventCode, &dfbEvent, pInputInfo->deviceId );

                        /*
                        ** DirectFB wants to know about both the key press and the release.
                        ** We'll stretch the truth here and generate both events even
                        ** though it might only be a key press.
                        **
                        ** There appears to be some support in the hardware to detect the
                        ** key release, i.e. the "lflag" in the "KBD1_STATUS" register.
                        ** This logic is NOT currently wired up in the software, we'll stick with
                        ** our approximation of always sending a key release event.
                        **
                        ** We will also skip a number of initial repeats to help prevent applications
                        ** having to have the same logic of discarding the first N initial key repeats.
                        */
                        if (!bKeyRepeated || (bKeyRepeated && (skipRepeatCnt == 0)))
                        {
                            dfb_input_dispatch( pDfbDevice, &dfbEvent );

                            if (!timeout)
                            {
                                dfbEvent.type = DIET_KEYRELEASE;
                                dfb_input_dispatch( pDfbDevice, &dfbEvent );
                            }
                        }

                        if (skipRepeatCnt)
                            skipRepeatCnt--;
                    }
                }
            }
        }
    }
    return NULL;
}       /* end of BDFB_IrfraThread() */


/*
** DirectFB "standard' input driver interface routines.
*/

static int driver_get_available( void )
{
    DFBBCMNEXUS                 *pBrcmDfb = (DFBBCMNEXUS *) dfb_system_data();
    BCMNEXUS_Options            *options = &pBrcmDfb->options;
    char                        *irProtocol = NULL;
    char                        *irKeycode = NULL;
    unsigned int                 numProtocols = 0, numKeycodes = 0, index = 0;
    bool                         ret = true;
    DFB_PlatformSettings         dfbPlatformSettings;

    DFB_Platform_GetSettings(&dfbPlatformSettings);

    if(dfbPlatformSettings.slave == DFB_Platform_SlaveType_Trellis)
    {
        D_DEBUG_AT(bcmnexusIrInput, "%s: Disabled IR input for Trellis \n", __FUNCTION__);
        return 0;
    }

    /*Initialize IR protocol and Keycode*/
    memset(irProtocolList, 0, MAX_IR_SUPPORTED*sizeof(char*));
    memset(irKeycodeList, 0, MAX_IR_SUPPORTED*sizeof(char*));

    /*Parse IR Protocol list*/
    irProtocol = strtok(options->bcmnexus_ir_protocol,",");
    while(1) {
        if(irProtocol)
        {
            irProtocolList[numProtocols] = D_STRDUP(irProtocol);
            irProtocol= strtok (NULL, " , -");
            ++numProtocols;
            if(numProtocols >= MAX_IR_SUPPORTED)
            {
                D_ERROR("bcmNexus/IrInput %s Max number of IR protocols support limit exceeded \n", __FUNCTION__);
                ret = false;
                break;
            }
        }
        else { /*Done with protocol parsing*/
            break;
        }
    }

    /*Parse IR Keycode list*/
    irKeycode = strtok(options->bcmnexus_ir_keycodes,",");
    while (1)
    {
        if(irKeycode)
        {
            irKeycodeList[numKeycodes] =  D_STRDUP(irKeycode);
            irKeycode= strtok (NULL, " ,.-");
            ++numKeycodes;
            if(numKeycodes >= MAX_IR_SUPPORTED)
            {
                D_ERROR("bcmNexus/IrInput %s Max number of IR keycodes support limit exceeded \n", __FUNCTION__);
                ret = false;
                break;
            }
        }
        else {/*Done with keycode parsing*/
            break;
        }
    }

    /*IR Number of Protocols and Keycodes must match*/
    if(numProtocols != numKeycodes)
    {
        D_ERROR("bcmNexus/IrInput %s IR Protocols and Keycodes count must match\n", __FUNCTION__);
        ret = false;
    }

    D_DEBUG_AT(bcmnexusIrInput, "%s: Num IR Protocols supported: %d \n", __FUNCTION__, numProtocols);

    if(!ret)
    {
        /*Free list*/
        for (index = 0; index < numProtocols; ++index)
        {
            if(irProtocolList[index])
            {
                D_FREE(irProtocolList[index]);
                irProtocolList[index] = NULL;
            }
        }

        for (index = 0; index < numKeycodes; ++index)
        {
            if(irKeycodeList[index])
            {
                D_FREE(irKeycodeList[index]);
                irKeycodeList[index] = NULL;
            }
        }

        return 0;
    }

     return numProtocols;
}

static void driver_get_info( InputDriverInfo *pDeviceInfo )
{
    /*
    ** These values are simply a guess.
    */
    if(!pDeviceInfo){
        D_ERROR("bcmNexus/IrInput %s Invalid param pDeviceInfo\n", __FUNCTION__);
        return;
    }
    snprintf( pDeviceInfo->name,
               DFB_INPUT_DRIVER_INFO_NAME_LENGTH,
               "BRCM IR driver" );

    snprintf( pDeviceInfo->vendor,
               DFB_INPUT_DRIVER_INFO_VENDOR_LENGTH,
               "Broadcom" );

    pDeviceInfo->version.major = 3;
    pDeviceInfo->version.minor = 1;
}       /* end of driver_get_info() */

static DFBResult driver_open_device(
    CoreInputDevice *   pDfbDevice,
    unsigned int        uiDeviceId,
    InputDeviceInfo *   pDfbDeviceInfo,
    void **             pDriverData
    )
{
    DFB_PlatformInputDeviceSettings  irInputSettings;
    BDFB_ir_info                    *pInputInfra;

    DFBBCMNEXUS                 *pBrcmDfb = (DFBBCMNEXUS *) dfb_system_data();
    BCMNEXUS_Options            *brcm_config = &pBrcmDfb->options;


    if((!pDfbDevice) || (!pDfbDeviceInfo) || (!pDriverData))
    {
        D_ERROR("bcmNexus/IrInput %s Invalid param \n", __FUNCTION__);
        return DR_INVARG;
    }

    if(uiDeviceId >= MAX_IR_SUPPORTED)
    {
        D_ERROR("bcmNexus/IrInput %s Invalid Device ID \n", __FUNCTION__);
        return DR_INVARG;
    }
    /*
    ** Allocate and initialize the device and "global" data.
    */

    if (!BDFB_load_ir_keycodes(irKeycodeList[uiDeviceId], &irKeycodes[uiDeviceId]))
    {
        D_ERROR("bcmNexus/IrInput %s failed to load IR keycodes module: %s\n", __FUNCTION__, irKeycodeList[uiDeviceId]);
        return DFB_INIT;
    }

    g_iIrNumBasicCodes[uiDeviceId]  = irKeycodes[uiDeviceId].funcs->GetBasicKeycodesSize();
    g_iIrNumAdvanCodes[uiDeviceId]  = irKeycodes[uiDeviceId].funcs->GetAdvanKeycodesSize();

    g_pIrBasicCodeMap[uiDeviceId] = D_CALLOC(g_iIrNumBasicCodes[uiDeviceId], sizeof(KeyMapElement));

    if (g_pIrBasicCodeMap[uiDeviceId] == NULL)
    {
        D_ERROR("bcmNexus/IrInput: %s failed to allocate basic code map memory\n", __FUNCTION__);
        return DFB_INIT;
    }

    g_pIrAdvanCodeMap[uiDeviceId] = D_CALLOC(g_iIrNumAdvanCodes[uiDeviceId], sizeof(KeyMapElement));

    if (g_pIrAdvanCodeMap[uiDeviceId] == NULL)
    {
        D_ERROR("bcmNexus/IrInput: %s failed to allocate advanced code map memory\n", __FUNCTION__);
        D_FREE(g_pIrBasicCodeMap[uiDeviceId]);

        D_FREE(irProtocolList[uiDeviceId]);
        irProtocolList[uiDeviceId] = NULL;

        D_FREE(irKeycodeList[uiDeviceId]);
        irKeycodeList[uiDeviceId] = NULL;

        return DFB_INIT;
    }

    irKeycodes[uiDeviceId].funcs->GetBasicKeycodes(g_pIrBasicCodeMap[uiDeviceId]);
    irKeycodes[uiDeviceId].funcs->GetAdvanKeycodes(g_pIrAdvanCodeMap[uiDeviceId]);

    pInputInfra = D_CALLOC( 1, sizeof(BDFB_ir_info) );

    if (pInputInfra == NULL)
    {
        D_ERROR("bcmNexus/IrInput: %s failed to allocate internal structure pInputInfra\n", __FUNCTION__);
        D_FREE(g_pIrBasicCodeMap[uiDeviceId]);
        D_FREE(g_pIrAdvanCodeMap[uiDeviceId]);

        D_FREE(irProtocolList[uiDeviceId]);
        irProtocolList[uiDeviceId] = NULL;

        D_FREE(irKeycodeList[uiDeviceId]);
        irKeycodeList[uiDeviceId] = NULL;

        return DFB_INIT;
    }

    pInputInfra->cancel_thread = false;

    /*
    ** We'll use " pDfbDevice " to queue the events within callbacks
    */
    pInputInfra->pDfbDevice  = pDfbDevice;

    DFB_PlatformInputDevice_GetDefaultSettings(DFB_PlatformInputDevice_eInfraRed,&irInputSettings);
    irInputSettings.dataReady.callback = BDFB_input_dataReady;
    irInputSettings.dataReady.context = &pInputInfra->lock;
    irInputSettings.repeatFilterTime = brcm_config->bcmnexus_ir_repeat_time ? brcm_config->bcmnexus_ir_repeat_time : DEFAULT_REPEAT_MILLI_SECS;
    irInputSettings.eventQueueSize = (irInputSettings.eventQueueSize < NUM_EVENTS) ? NUM_EVENTS : irInputSettings.eventQueueSize;
    irInputSettings.device.ir.mode = BDFB_lookup_ir_protocol(irProtocolList[uiDeviceId]);
    irInputSettings.device.ir.channel_number = uiDeviceId;
    pInputInfra->irInput = DFB_PlatformInputDevice_Open(DFB_PlatformInputDevice_eInfraRed,uiDeviceId, &irInputSettings, &pInputInfra->clientID);
    pInputInfra->deviceId = uiDeviceId;

    if (pInputInfra->irInput == NULL)
    {
        D_ERROR("bcmNexus/IrInput: %s failed to initialize hardware driver\n", __FUNCTION__);

        D_FREE(pInputInfra);
        D_FREE(g_pIrBasicCodeMap[uiDeviceId]);
        D_FREE(g_pIrAdvanCodeMap[uiDeviceId]);

        D_FREE(irProtocolList[uiDeviceId]);
        irProtocolList[uiDeviceId] = NULL;

        D_FREE(irKeycodeList[uiDeviceId]);
        irKeycodeList[uiDeviceId] = NULL;

        return DFB_INIT;
    }

    /* Create the locks/semaphores */
    fusion_skirmish_init( &pInputInfra->lock,  "Broadcom BDFB_Infra Semaphore",  dfb_core_world(NULL) );

    /*
    ** Fill in the driver info with the supported features.
    */
    pDfbDeviceInfo->prefered_id = DIDID_REMOTE;
    pDfbDeviceInfo->desc.type   = DIDTF_REMOTE;
    pDfbDeviceInfo->desc.caps   = DICAPS_KEYS;

    *pDriverData = pInputInfra;

    /*
    ** Create threads to handle the input IR and key events.
    */
    pInputInfra->thread = direct_thread_create(DTT_DEFAULT, BDFB_IrfraThread,  pInputInfra, "BDFB_Irfra");

    return DFB_OK;
}       /* end of driver_open_device() */

static DFBResult driver_get_keymap_entry(
    CoreInputDevice *           pDfbDevice,
    void *                      pDriverData,
    DFBInputDeviceKeymapEntry * pKeymapEntry
    )
{
    BSTD_UNUSED(pDfbDevice);
    BSTD_UNUSED(pDriverData);
    BSTD_UNUSED(pKeymapEntry);
    return DFB_UNSUPPORTED;
}       /* end of driver_get_keymap_entry() */

static void driver_close_device( void *pDriverData )
{
    BDFB_ir_info *pInputInfra = (BDFB_ir_info *) pDriverData;

    if(pInputInfra)
    {
        /* Cancel IR Input thread... */
        pInputInfra->cancel_thread = true;
        fusion_skirmish_prevail( &pInputInfra->lock );
        fusion_skirmish_notify( &pInputInfra->lock );
        fusion_skirmish_dismiss( &pInputInfra->lock );
        direct_thread_join(pInputInfra->thread);
        direct_thread_destroy(pInputInfra->thread);

        /* Close the ir receiver interface */
        if ( pInputInfra->irInput ) DFB_PlatformInputDevice_Close(DFB_PlatformInputDevice_eInfraRed, pInputInfra->irInput, &pInputInfra->clientID);

        /* Destroy the locks */
        fusion_skirmish_destroy(&pInputInfra->lock);

        if (g_pIrBasicCodeMap[pInputInfra->deviceId])
            D_FREE(g_pIrBasicCodeMap[pInputInfra->deviceId]);

        if (g_pIrAdvanCodeMap[pInputInfra->deviceId])
            D_FREE(g_pIrAdvanCodeMap[pInputInfra->deviceId]);

        if (irProtocolList[pInputInfra->deviceId])
            D_FREE(irProtocolList[pInputInfra->deviceId]);

        if (irKeycodeList[pInputInfra->deviceId])
            D_FREE(irKeycodeList[pInputInfra->deviceId]);

        /* Free data structure */
        D_FREE( pInputInfra );
    }

    return;
}       /* end of driver_close_device() */

