/******************************************************************************
 *    (c)2008-2014 Broadcom Corporation
 *
 * This program is the proprietary software of Broadcom Corporation and/or its licensors,
 * and may only be used, duplicated, modified or distributed pursuant to the terms and
 * conditions of a separate, written license agreement executed between you and Broadcom
 * (an "Authorized License").  Except as set forth in an Authorized License, Broadcom grants
 * no license (express or implied), right to use, or waiver of any kind with respect to the
 * Software, and Broadcom expressly reserves all rights in and to the Software and all
 * intellectual property rights therein.  IF YOU HAVE NO AUTHORIZED LICENSE, THEN YOU
 * HAVE NO RIGHT TO USE THIS SOFTWARE IN ANY WAY, AND SHOULD IMMEDIATELY
 * NOTIFY BROADCOM AND DISCONTINUE ALL USE OF THE SOFTWARE.
 *
 * Except as expressly set forth in the Authorized License,
 *
 * 1.     This program, including its structure, sequence and organization, constitutes the valuable trade
 * secrets of Broadcom, and you shall use all reasonable efforts to protect the confidentiality thereof,
 * and to use this information only in connection with your use of Broadcom integrated circuit products.
 *
 * 2.     TO THE MAXIMUM EXTENT PERMITTED BY LAW, THE SOFTWARE IS PROVIDED "AS IS"
 * AND WITH ALL FAULTS AND BROADCOM MAKES NO PROMISES, REPRESENTATIONS OR
 * WARRANTIES, EITHER EXPRESS, IMPLIED, STATUTORY, OR OTHERWISE, WITH RESPECT TO
 * THE SOFTWARE.  BROADCOM SPECIFICALLY DISCLAIMS ANY AND ALL IMPLIED WARRANTIES
 * OF TITLE, MERCHANTABILITY, NONINFRINGEMENT, FITNESS FOR A PARTICULAR PURPOSE,
 * LACK OF VIRUSES, ACCURACY OR COMPLETENESS, QUIET ENJOYMENT, QUIET POSSESSION
 * OR CORRESPONDENCE TO DESCRIPTION. YOU ASSUME THE ENTIRE RISK ARISING OUT OF
 * USE OR PERFORMANCE OF THE SOFTWARE.
 *
 * 3.     TO THE MAXIMUM EXTENT PERMITTED BY LAW, IN NO EVENT SHALL BROADCOM OR ITS
 * LICENSORS BE LIABLE FOR (i) CONSEQUENTIAL, INCIDENTAL, SPECIAL, INDIRECT, OR
 * EXEMPLARY DAMAGES WHATSOEVER ARISING OUT OF OR IN ANY WAY RELATING TO YOUR
 * USE OF OR INABILITY TO USE THE SOFTWARE EVEN IF BROADCOM HAS BEEN ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGES; OR (ii) ANY AMOUNT IN EXCESS OF THE AMOUNT
 * ACTUALLY PAID FOR THE SOFTWARE ITSELF OR U.S. $1, WHICHEVER IS GREATER. THESE
 * LIMITATIONS SHALL APPLY NOTWITHSTANDING ANY FAILURE OF ESSENTIAL PURPOSE OF
 * ANY LIMITED REMEDY.
 *
 *****************************************************************************/

#include "bcmnexus.h"
#include "bstd.h"
#include "platform_init.h"
#include "nexus_platform.h"
#include "fusion/build.h"

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <pthread.h>

#include "bkni.h"
#include "bkni_multi.h"
#if FUSION_BUILD_MULTI
#include "platform_ipc_session_simple_client.h"
#endif

BDBG_MODULE(platform_init);

/* Used to indicate whether DFB_Platform_Init() was called successfully or not.
   Each process should call DFB_Platform_Init() to ensure coherency. */
static bool dfb_platform_initialised = false;

/* Indicates whether this is the first ever application/process to
   initialise the dfb platform code. */
static bool dfb_is_master = false;

/* Used to access the shared public handles (this gets set per process) */
static DFB_PlatformSettings  *dfb_platform_shared = NULL;

/* Used to access the shared platform status (this gets set per process) */
static DFB_PlatformStatus    *dfb_platform_status_shared = NULL;

/* Used to access the shared private handles (this gets set per process) */
static DFB_P_DisplaySettings *dfb_p_display_shared = NULL;

/* Used to access the shared private handles (this gets set per process) */
static DFB_P_PlatformSharedData *dfb_p_platform_shared_data = NULL;

/* Not shared across processes, but indicates whether a device was initialised
   in the current process or not. */
static DFB_P_DeviceStatus     dfb_device_status;


#ifdef BCMNEXUS_NSC_SUPPORT
static const bool isXS = true;
#else
static const bool isXS = false;
#endif

#ifdef BCMNEXUS_NIR_SUPPORT
static const bool isNIR = true;
#else
static const bool isNIR = false;
#endif

/********* Private Functions *********/

/* returns false if process is in slave mode
 *         true if process is in master mode */
static bool
DFB_Platform_P_InitSharedMemory( size_t shMemSz )
{
#if FUSION_BUILD_MULTI
   char * client_name = "dfb";

    BSTD_UNUSED(shMemSz);
    int rc = simple_client_init(client_name);
    if(rc < 0) {
        BDBG_MSG_TRACE(("Could not connect to IPC server. DFB master can ignore this message."));
        return true;
    }
    else {
        return false;
    }
#else
    (void)shMemSz;
#endif
    return true;
}

/*
* This function will create the shared memory and give you the
* void pointer to that memory
*/
static void *
DFB_Platform_P_OpenSharedMemory( size_t shMemSz,
                                 bool bCreate )
{
    char    *shm;
#if FUSION_BUILD_MULTI
    shm = calloc(1, shMemSz);
    if (bCreate) {
        DFB_PlatformResult rc = DFB_PLATFORM_OK ;
        memset(shm, 0, shMemSz);
        rc = DFB_Platform_P_StartIpcServer();

        if(rc) {
            BDBG_ERR(("Failed to start platform ipc server"));
            return NULL;
        }
    }
    else {
        DFB_PlatformResult res;
        DFB_PlatformSettings *pSettings = (DFB_PlatformSettings  *)   ((uint8_t *)shm );
        /*Check received platform settings are valid*/
        while(pSettings->platformSettings.platforSettingsValid != (unsigned int)PLATFORM_SETTINGS_VALID) {
            res = brc_client_get_settings_client(simple_client_state.rca_client,(DFB_PlatformShared *) shm);
            if(res) {
                BDBG_ERR(("Failed to get shared settings from ipc server"));
                free (shm);
                return NULL;
            }
            usleep(100000);
        }
    }

#else
    (void)bCreate;
    shm = (char*)calloc(1,shMemSz);
#endif /* FUSION_BUILD_MULTI */

    BDBG_MSG(("Returning Shared Memory Address:%p\n",shm));

    return shm;
}

static void
DFB_Platform_P_CloseSharedMemory( void * shMem )
{

#if FUSION_BUILD_MULTI
    if(dfb_is_master) {
        DFB_Platform_P_StopIpcServer();
    }
#endif /* FUSION_BUILD_MULTI */
    free(shMem);

}

static DFB_PlatformResult
DFB_Platform_P_SetupSharedMemory( void )
{
    int   shareMemSize = (sizeof(DFB_PlatformSettings)      ) +
                         (sizeof(DFB_PlatformStatus)        ) +
                         (DFB_Platform_P_GetSharedMemSize() ) +
                         (sizeof(DFB_P_PlatformSharedData)  );

    void *pShareMem    = NULL;

    if (dfb_platform_shared == NULL)
    {
        /* If the shared memory has already been initialised, then it means
        we are a slave application. */
        dfb_is_master = DFB_Platform_P_InitSharedMemory(shareMemSize);
        pShareMem = DFB_Platform_P_OpenSharedMemory(shareMemSize, dfb_is_master);

        if (!pShareMem)
        {
            BDBG_ERR(("%s: Setting Up Shared Memory Failed\n", __FUNCTION__));
            return DFB_PLATFORM_NOSHAREDMEMORY;
        }

        dfb_platform_shared         = (DFB_PlatformSettings  *)   ((uint8_t *)pShareMem );
        dfb_platform_status_shared  = (DFB_PlatformStatus    *)   ((uint8_t *)dfb_platform_shared        + (sizeof(DFB_PlatformSettings )));
        dfb_p_display_shared        = (DFB_P_DisplaySettings *)   ((uint8_t *)dfb_platform_status_shared + (sizeof(DFB_PlatformStatus   )));
        dfb_p_platform_shared_data  = (DFB_P_PlatformSharedData *)((uint8_t *)dfb_p_display_shared       + (sizeof(DFB_P_DisplaySettings)));
    }
    return DFB_PLATFORM_OK;
}

DFB_PlatformResult
DFB_Platform_P_GetShared(DFB_PlatformShared *shm)
{
    DFB_PlatformResult res = DFB_PLATFORM_OK;

    int   shareMemSize = (sizeof(DFB_PlatformSettings)      ) +
                         (sizeof(DFB_PlatformStatus)        ) +
                         (DFB_Platform_P_GetSharedMemSize() ) +
                         (sizeof(DFB_P_PlatformSharedData)  );

    if(!shm)
    {
        BDBG_ERR(("%s() Invalid parameter!", __FUNCTION__));
        res = DFB_PLATFORM_INIT;
    }
    else
    {
        memcpy(shm, dfb_platform_shared, shareMemSize );
    }
    return res;
}

DFB_PlatformResult
DFB_Platform_P_GetStatus( DFB_PlatformStatus *pStatus )
{
    DFB_PlatformResult res;

    res = DFB_Platform_P_SetupSharedMemory();

    if (res != DFB_PLATFORM_OK) {
        BDBG_ERR(("%s() failed to setup shared memory!", __FUNCTION__));
    }
    else {
        dfb_platform_status_shared->isXS     = isXS;
        dfb_platform_status_shared->isNIR    = isNIR;
        dfb_platform_status_shared->isMaster = dfb_is_master;
        memcpy(pStatus, dfb_platform_status_shared, sizeof(*pStatus));
    }

    return res;
}

DFB_PlatformResult
DFB_Platform_P_GetSettings( DFB_PlatformSettings *pSettings )
{
    memcpy(pSettings, dfb_platform_shared, sizeof(DFB_PlatformSettings));
    return DFB_PLATFORM_OK;
}

unsigned long
DFB_Platform_P_GetGraphicsHeapSize( void )
{
    unsigned long heapSize = DEFAULT_GFX_HEAP_SIZE;
    char *pValue;
    char invalid = '\0';
    char *pInvalid = &invalid;

    pValue = getenv("gfx_heap_size");
    if (pValue != NULL)
    {
        heapSize = strtoul(pValue, &pInvalid, 0);
        if ((heapSize == 0) || (*pInvalid != '\0'))
        {
            BDBG_WRN(("Invalid \"gfx_heap_size\" value specified - defaulting to %ld bytes.\n", heapSize));
            heapSize = DEFAULT_GFX_HEAP_SIZE;
        }
    }

    if (heapSize % 4096)
    {
        heapSize -= (heapSize % 4096);
        BDBG_WRN(("Invalid \"gfx_heap_size\" value specified - must be 4k aligned rounding down to %ld bytes.\n", heapSize));
    }

    return heapSize;
}

DFB_PlatformNexusHandle
DFB_Platform_P_GetGraphicsHeap( void *pMemory )
{
    bool isMaster = dfb_platform_status_shared->isMaster || dfb_platform_shared->trustUnauthenticatedClients;

    return DFB_Platform_P_GetHeap(pMemory, isMaster);
}

/** Returns -1 if no display found with given handle */
int
DFB_Platform_P_GetDisplayIndex( DFB_PlatformNexusHandle displayHandle )
{
    int displayIndex;

    for (displayIndex = 0; displayIndex < NUM_DISPLAYS; displayIndex ++)
    {
        if (dfb_p_display_shared->display[displayIndex].displayHandle == displayHandle)
            break;
    }

    if (displayIndex >= NUM_DISPLAYS)
        displayIndex = -1;

    return displayIndex;
}

DFB_PlatformNexusHandle
DFB_Platform_P_GetDisplayHandle( int displayIndex)
{
    DFB_PlatformNexusHandle handle = NULL;

    if (displayIndex < NUM_DISPLAYS)
        handle = dfb_p_display_shared->display[displayIndex].displayHandle;

    return handle;
}

DFB_PlatformResult
DFB_Platform_P_GetDisplayOutputStatus( DFB_PlatformDisplayOutputType outputType,
                                       int                           outputIndex)
{
    int                        outputSlot;
    DFB_P_DisplayOutputStatus *pOutputStatus;

    pOutputStatus = &dfb_p_display_shared->output[0];

    for (outputSlot = 0; outputSlot < DFB_PLATFORM_NUM_DISPLAY_OUTPUTS_MAX; outputSlot++)
    {
        if (pOutputStatus->outputType   == outputType &&
            pOutputStatus->outputIndex  == outputIndex)
        {
            return DFB_PLATFORM_BUSY;
        }
        pOutputStatus++;
    }

    return DFB_PLATFORM_OK;
}

DFB_PlatformNexusHandle
DFB_Platform_P_GetDisplayOutputHandle( DFB_PlatformNexusHandle       displayHandle,
                                       DFB_PlatformDisplayOutputType outputType,
                                       int                           outputIndex )
{
    int                        outputSlot;
    int                        displayIndex;
    DFB_PlatformNexusHandle    outputHandle = NULL;
    DFB_P_DisplayOutputStatus *pOutputStatus;

    displayIndex = DFB_Platform_P_GetDisplayIndex(displayHandle);

    if (displayIndex >= 0 && displayIndex < NUM_DISPLAYS)
    {
        pOutputStatus = &dfb_p_display_shared->output[0];

        for (outputSlot = 0; outputSlot < DFB_PLATFORM_NUM_DISPLAY_OUTPUTS_MAX; outputSlot++)
        {
            if (pOutputStatus->displayIndex == displayIndex &&
                pOutputStatus->outputType   == outputType   &&
                pOutputStatus->outputIndex  == outputIndex)
            {
                outputHandle = pOutputStatus->outputHandle;
                break;
            }
            pOutputStatus++;
        }
    }
    return outputHandle;
}

DFB_PlatformResult
DFB_Platform_P_SetDisplayOutputHandle( DFB_PlatformNexusHandle       displayHandle,
                                       DFB_PlatformDisplayOutputType outputType,
                                       int                           outputIndex,
                                       DFB_PlatformNexusHandle       outputHandle )
{
    DFB_PlatformResult         res = DFB_PLATFORM_FAILURE;
    int                        outputSlot;
    int                        displayIndex;
    DFB_P_DisplayOutputStatus *pOutputStatus;

    displayIndex = DFB_Platform_P_GetDisplayIndex(displayHandle);

    if (displayIndex >= 0 && displayIndex < NUM_DISPLAYS)
    {
        pOutputStatus = &dfb_p_display_shared->output[0];

        for (outputSlot = 0; outputSlot < DFB_PLATFORM_NUM_DISPLAY_OUTPUTS_MAX; outputSlot++)
        {
            /* If the output Handle is NULL, then we need to find an already registered slot
               and then remove it. */
            if (outputHandle == NULL)
            {
                if (pOutputStatus->outputType   == outputType &&
                    pOutputStatus->outputIndex  == outputIndex &&
                    pOutputStatus->displayIndex == displayIndex)
                {
                    memset(pOutputStatus, 0, sizeof(*pOutputStatus));
                    res = DFB_PLATFORM_OK;
                    break;
                }
            }
            /* If we are adding a handle, then we need to find a spare slot... */
            else
            {
                if (pOutputStatus->outputHandle == NULL)
                {
                    pOutputStatus->outputHandle = outputHandle;
                    pOutputStatus->outputType   = outputType;
                    pOutputStatus->outputIndex  = outputIndex;
                    pOutputStatus->displayIndex = displayIndex;
                    res = DFB_PLATFORM_OK;
                    break;
                }
            }
            pOutputStatus++;
        }
    }
    return res;
}

/** Returns -1 if no display window found with given handle */
int
DFB_Platform_P_GetDisplayWindowSlot( DFB_PlatformNexusHandle windowHandle )
{
    int windowSlot;

    for (windowSlot = 0; windowSlot < DFB_PLATFORM_NUM_DISPLAY_WINDOWS_MAX; windowSlot++)
    {
        if (dfb_p_display_shared->window[windowSlot].windowHandle == windowHandle)
            break;
    }

    if (windowSlot >= DFB_PLATFORM_NUM_DISPLAY_WINDOWS_MAX)
        windowSlot = -1;

    return windowSlot;
}

DFB_PlatformNexusHandle
DFB_Platform_P_GetDisplayWindowHandle( DFB_PlatformNexusHandle displayHandle,
                                       int                     windowIndex )
{
    int                        displayIndex;
    int                        windowSlot;
    DFB_PlatformNexusHandle    windowHandle = NULL;
    DFB_P_DisplayWindowStatus *pWindowStatus;

    displayIndex = DFB_Platform_P_GetDisplayIndex(displayHandle);

    if (displayIndex >= 0 && displayIndex < NUM_DISPLAYS)
    {
        pWindowStatus = &dfb_p_display_shared->window[0];

        for (windowSlot = 0; windowSlot < DFB_PLATFORM_NUM_DISPLAY_WINDOWS_MAX; windowSlot++)
        {
            if (pWindowStatus->displayIndex == displayIndex && pWindowStatus->windowIndex  == windowIndex)
            {
                windowHandle = pWindowStatus->windowHandle;
                break;
            }
            pWindowStatus++;
        }
    }
    return windowHandle;
}

DFB_PlatformColor
DFB_Platform_P_NexusToDFB_PlatformColor( uint32_t color )
{
    DFB_PlatformColor retColor = {color>>24, (color&0xff0000>>16), (color&0xff00)>>8, color&0xff};
    return retColor;
}

uint32_t
DFB_Platform_P_DFBToNexusColor( DFB_PlatformColor color )
{
    uint32_t retColor = (color.a << 24) | (color.r << 16) | (color.g <<8) |color.b;
    return retColor;
}

NEXUS_VideoFormat
DFB_Platform_P_DFBToNexusVideoFormat( DFB_PlatformVideoFormat format )
{
    return DFB_Platform_P_DFBToNexusVideoFormatTable[format];
}

DFB_PlatformVideoFormat
DFB_Platform_P_NexusToDFBVideoFormat( NEXUS_VideoFormat format )
{
    return DFB_Platform_P_NexusToDFBVideoFormatTable[format];
}

#if NEXUS_HAS_DISPLAY
NEXUS_DisplayType
DFB_Platform_P_DFBToNexusDisplayType( DFB_PlatformDisplayType type )
{
    return DFB_Platform_P_DFBToNexusDisplayTypeTable[type];
}

DFB_PlatformDisplayType
DFB_Platform_P_NexusToDFBDisplayType( NEXUS_DisplayType type )
{
    return DFB_Platform_P_NexusToDFBDisplayTypeTable[type];
}

NEXUS_Timebase
DFB_Platform_P_DFBToNexusTimebase( DFB_PlatformTimebase timebase)
{
    return DFB_Platform_P_DFBToNexusTimebaseTable[timebase];
}

#if BCMNEXUS_SET_HDMI_COLORSPACE
DFB_PlatformColorSpace
DFB_Platform_P_NexusToDFBColorSpace( NEXUS_ColorSpace colorspace)
{
    return DFB_Platform_P_NexusToDFBColorSpaceTable[colorspace];
}

NEXUS_ColorSpace
DFB_Platform_P_DFBToNexusColorSpace( DFB_PlatformColorSpace colorspace)
{
    return DFB_Platform_P_DFBToNexusColorSpaceTable[colorspace];
}

DFB_PlatformHdmiColorDepth
DFB_Platform_P_NexusToDFBHdmiColorDepth( NEXUS_HdmiColorDepth colordepth)
{
    return DFB_Platform_P_NexusToDFBHdmiColorDepthTable[colordepth];
}

NEXUS_HdmiColorDepth
DFB_Platform_P_DFBToNexusHdmiColorDepth( DFB_PlatformHdmiColorDepth colorepth)
{
    return DFB_Platform_P_DFBToNexusHdmiColorDepthTable[colorepth];
}
#endif /*BCMNEXUS_SET_HDMI_COLORSPACE*/
#endif

NEXUS_AspectRatio
DFB_Platform_P_DFBToNexusDisplayAspectRatio( DFB_PlatformDisplayAspectRatio ratio )
{
    return DFB_Platform_P_DFBToNexusDisplayAspectRatioTable[ratio];
}

DFB_PlatformDisplayAspectRatio
DFB_Platform_P_NexusToDFBDisplayAspectRatio( NEXUS_AspectRatio ratio )
{
    return DFB_Platform_P_NexusToDFBDisplayAspectRatioTable[ratio];
}

/********* Public Functions *********/

/* Get the default DFB platform settings */
void
DFB_Platform_GetDefaultSettings_tagged( DFB_PlatformClientType clientType,
                                        DFB_PlatformSettings  *pSettings,
                                        bool                   initOrJoinNexus,
                                        unsigned               versionCheck,
                                        unsigned               structSizeCheck )
{
    DFB_PlatformResult res;
    unsigned  i;

    BDBG_MSG_TRACE((">>%s", __FUNCTION__));

    /* Check for binary compatibility */
    if (versionCheck != DFB_PLATFORM_GET_VERSION()) {
        BDBG_ERR(("DFB_Platform_GetDefaultSettings() failed with version mismatch (dfb=%d.%d, caller=%d.%d). Please recompile application and/or DirectFB.",
            (DFB_PLATFORM_GET_VERSION() & DFB_PLATFORM_P_VERSION_MAJ_MASK) >> DFB_PLATFORM_P_VERSION_MAJ_SHIFT,
            (DFB_PLATFORM_GET_VERSION() & DFB_PLATFORM_P_VERSION_MIN_MASK) >> DFB_PLATFORM_P_VERSION_MIN_SHIFT,
            (versionCheck                 & DFB_PLATFORM_P_VERSION_MAJ_MASK) >> DFB_PLATFORM_P_VERSION_MAJ_SHIFT,
            (versionCheck                 & DFB_PLATFORM_P_VERSION_MIN_MASK) >> DFB_PLATFORM_P_VERSION_MIN_SHIFT));
        BDBG_ASSERT(false);
    }

    if (structSizeCheck != DFB_PLATFORM_P_GET_STRUCT_SIZE()) {
        BDBG_ERR(("DFB_Platform_GetDefaultSettings() failed with struct size mismatch (dfb=%d, caller=%d). Please recompile application and/or DirectFB.",
            DFB_PLATFORM_P_GET_STRUCT_SIZE(), structSizeCheck));
        BDBG_ASSERT(false);
    }

    memset(pSettings, 0, sizeof(*pSettings));

    res = DFB_Platform_P_SetupSharedMemory();

    if (res != DFB_PLATFORM_OK) {
        BDBG_ERR(("DFB_Platform_GetDefaultSettings() failed to setup shared memory!"));
        BDBG_ASSERT(false);
    }

    pSettings->initOrJoinNexus = initOrJoinNexus;

    if (clientType >= DFB_PlatformClientType_eMax)
    {
        BDBG_ERR(("DFB_Platform_GetDefaultSettings() Invalid client type %d!\n", clientType));
        BDBG_ASSERT(false);
    }
    else if (clientType == DFB_PlatformClientType_eNxClient)
    {
        pSettings->slave = DFB_Platform_SlaveType_NxClient;
    }

    /* If we have already initialised the shared memory, then it means we are definitely a slave.
      or if the dfb_slave env var is set to y. */
    if ( (dfb_is_master == false) || (getenv("dfb_slave") && !strcmp(getenv("dfb_slave"), "y")) )
    {
        if (clientType != DFB_PlatformClientType_eNxClient)
            pSettings->slave = DFB_Platform_SlaveType_Nexus;
    }

    pSettings->allowUnauthenticatedClients = true;
    pSettings->trustUnauthenticatedClients = false;
    pSettings->clientId = INT_MAX;

    if (dfb_is_master)
    {
        /* If we are a DFB master application, then get the default settings */
        DFB_Platform_P_GetDefaultSettings(pSettings);
    }
    else
    {
        /* If we are not a DFB master application, then copy the platform settings from the shared memory. */
        memcpy(&pSettings->platformSettings, &dfb_platform_shared->platformSettings, sizeof(dfb_platform_shared->platformSettings));

        /* Also copy over the server-side settings from shared memory */
        pSettings->allowUnauthenticatedClients = dfb_platform_shared->allowUnauthenticatedClients;
        pSettings->trustUnauthenticatedClients = dfb_platform_shared->trustUnauthenticatedClients;

        /* and copy over the heap handles... */
        pSettings->displayHeapHandle          = dfb_platform_shared->displayHeapHandle;
        pSettings->secondaryDisplayHeapHandle = dfb_platform_shared->secondaryDisplayHeapHandle;
        pSettings->offscreenHeapHandle        = dfb_platform_shared->offscreenHeapHandle;
        pSettings->paletteHeapHandle          = dfb_platform_shared->paletteHeapHandle;
        pSettings->videoHeapHandle            = dfb_platform_shared->videoHeapHandle;
        pSettings->pictureDecoderHeapHandle   = dfb_platform_shared->pictureDecoderHeapHandle;
    }

#if BCMNEXUS_NSC_SUPPORT
    /*When DirectFB is built in XS mode but Nexus nxClient API is not used*/
    if (clientType != DFB_PlatformClientType_eNxClient)
    {
            pSettings->slave = DFB_Platform_SlaveType_Nexus;
    }
#endif

    if (dfb_platform_shared->graphics2d[0].handle == NULL)
        pSettings->graphics2d[0].init = DFB_Platform_P_Graphics2D_GetDefaultInitSettings(&pSettings->graphics2d[0].initSettings);

    if (dfb_platform_shared->graphics2d[1].handle == NULL)
        pSettings->graphics2d[1].init = DFB_Platform_P_Graphics2D_GetDefaultInitSettings(&pSettings->graphics2d[1].initSettings);

    if (dfb_platform_shared->graphics3d.handle == NULL)
        pSettings->graphics3d.init = DFB_Platform_P_Graphics3D_GetDefaultInitSettings(&pSettings->graphics3d.initSettings);

    if (dfb_platform_shared->pictureDecoder.handle == NULL)
        pSettings->pictureDecoder.init = DFB_Platform_P_PictureDecoder_GetDefaultInitSettings(&pSettings->pictureDecoder.initSettings);

    /* Initialise the displays if we are a master */
    if (dfb_is_master)
    {
        for (i = 0; i < DFB_PLATFORM_NUM_DISPLAYS_MAX; i++)
        {
            if (dfb_platform_shared->display[i].handle == NULL)
                pSettings->display[i].init = DFB_Platform_P_Display_GetDefaultInitSettings( i, &pSettings->display[i].initSettings );
        }

        /* Initialise the display windows */
        for (i = 0; i < DFB_PLATFORM_NUM_DISPLAY_WINDOWS_MAX; i++)
        {
            if (dfb_platform_shared->displayWindow[i].handle == NULL)
                pSettings->displayWindow[i].init = DFB_Platform_P_DisplayWindow_GetDefaultInitSettings( i, &pSettings->displayWindow[i].initSettings );
        }

        /* Initialise the display outputs */
        for (i = 0; i < DFB_PLATFORM_NUM_DISPLAY_OUTPUTS_MAX; i++)
        {
            if (dfb_platform_shared->displayOutput[i].handle == NULL)
                pSettings->displayOutput[i].init = DFB_Platform_P_DisplayOutput_GetDefaultInitSettings( i, &pSettings->displayOutput[i].initSettings );
        }
    }

    BDBG_MSG_TRACE(("<<%s", __FUNCTION__));
}

DFB_PlatformResult
DFB_Platform_GetStatus( DFB_PlatformStatus *pStatus )
{
    DFB_PlatformResult res;

    BDBG_MSG_TRACE((">>%s", __FUNCTION__));

    BDBG_ASSERT(pStatus);

    res = DFB_Platform_P_GetStatus(pStatus);

    BDBG_MSG_TRACE(("<<%s", __FUNCTION__));
    return res;
}

DFB_PlatformResult
DFB_Platform_GetSettings( DFB_PlatformSettings *pSettings )
{
    DFB_PlatformResult res;
    DFB_PlatformStatus platformStatus;

    BDBG_MSG_TRACE((">>%s", __FUNCTION__));

    BDBG_ASSERT(pSettings);

    res = DFB_Platform_P_GetStatus(&platformStatus);

    if (res == DFB_PLATFORM_OK)
    {
        if (!platformStatus.initOrJoinNexus)
        {
            BDBG_ERR(("%s: DFB Platform not initialised. Please ensure DFB_Platform_Init() is called first.\n", __FUNCTION__));
            res = DFB_PLATFORM_INIT;
        }
        else
        {
            res = DFB_Platform_P_GetSettings(pSettings);
        }
    }

    BDBG_MSG_TRACE(("<<%s", __FUNCTION__));
    return res;
}

void
DFB_Platform_GetVersion( unsigned *pMajor,
                         unsigned *pMinor )
{

    BDBG_MSG_TRACE((">>%s", __FUNCTION__));

    BDBG_ASSERT(pMajor);
    BDBG_ASSERT(pMinor);

    *pMajor = (DFB_PLATFORM_GET_VERSION() & DFB_PLATFORM_P_VERSION_MAJ_MASK) >> DFB_PLATFORM_P_VERSION_MAJ_SHIFT;
    *pMinor = (DFB_PLATFORM_GET_VERSION() & DFB_PLATFORM_P_VERSION_MIN_MASK) >> DFB_PLATFORM_P_VERSION_MIN_SHIFT;

    BDBG_MSG_TRACE(("<<%s", __FUNCTION__));
}

DFB_PlatformResult
DFB_Platform_GetDisplaySettings( DFB_PlatformNexusHandle displayHandle, DFB_PlatformDisplaySettings *pSettings)
{
    DFB_PlatformResult             res;
    DFB_Platform_P_DisplaySettings displaySettings;

    BDBG_MSG_TRACE((">>%s", __FUNCTION__));

    BDBG_ASSERT(pSettings);

    if (!dfb_platform_initialised)
    {
        BDBG_ERR(("%s: DFB Platform not initialised. Please ensure DFB_Platform_Init() is called first.\n", __FUNCTION__));
        res = DFB_PLATFORM_INIT;
    }
    else
    {
         if (DFB_Platform_P_GetDisplayIndex(displayHandle) == -1)
        {
            BDBG_ERR(("%s: Invalid display handle %p!\n", __FUNCTION__, (void *)displayHandle));
            res = DFB_PLATFORM_INVARG;
        }
        else
        {
            res = DFB_Platform_P_GetDisplaySettings(displayHandle, &displaySettings);
            if (res != DFB_PLATFORM_OK)
                BDBG_ERR(("%s: Cannot get private display settings (res=%d)!\n", __FUNCTION__, res));
            else
                memcpy(pSettings, &displaySettings, sizeof(*pSettings));
        }
     }

    BDBG_MSG_TRACE(("<<%s", __FUNCTION__));
    return res;
}

DFB_PlatformResult
DFB_Platform_SetDisplaySettings( DFB_PlatformNexusHandle displayHandle, DFB_PlatformDisplaySettings *pSettings)
{
    DFB_PlatformResult             res;
    DFB_Platform_P_DisplaySettings displaySettings;

    BDBG_MSG_TRACE((">>%s", __FUNCTION__));

    BDBG_ASSERT(pSettings);

    if (!dfb_platform_initialised)
    {
        BDBG_ERR(("%s: DFB Platform not initialised. Please ensure DFB_Platform_Init() is called first.\n", __FUNCTION__));
        res = DFB_PLATFORM_INIT;
    }
    else
    {
         if (DFB_Platform_P_GetDisplayIndex(displayHandle) == -1)
        {
            BDBG_ERR(("%s: Invalid display handle %p!\n", __FUNCTION__, (void *)displayHandle));
            res = DFB_PLATFORM_INVARG;
        }
        else
        {
            memset(&displaySettings, 0, sizeof(displaySettings));
            memcpy(&displaySettings, pSettings, sizeof(*pSettings));
            displaySettings.vsyncCallback.callback = (DFB_PlatformCallback)-1; /* Signal don't set */

            res = DFB_Platform_P_SetDisplaySettings( displayHandle, &displaySettings, true );
            if (res != DFB_PLATFORM_OK)
                BDBG_ERR(("%s: Cannot set private display settings (res=%d)!\n", __FUNCTION__, res));
        }
    }
    BDBG_MSG_TRACE(("<<%s", __FUNCTION__));
    return res;
}

DFB_PlatformResult
DFB_Platform_GetDisplayWindow( DFB_PlatformNexusHandle       displayHandle,
                               int                           windowIndex,
                               DFB_PlatformNexusHandle      *pHandle )
{
    DFB_PlatformResult res = DFB_PLATFORM_OK;
    int       displayIndex;

    BDBG_MSG_TRACE((">>%s", __FUNCTION__));

    BDBG_ASSERT(displayHandle);
    BDBG_ASSERT(pHandle);

    if (!dfb_platform_initialised)
    {
        BDBG_ERR(("%s: DFB Platform not initialised. Please ensure DFB_Platform_Init() is called first.\n", __FUNCTION__));
        res = DFB_PLATFORM_INIT;
    }
    else
    {
        displayIndex = DFB_Platform_P_GetDisplayIndex(displayHandle);

        if (displayIndex == -1)
        {
            BDBG_ERR(("%s: Invalid display handle %p!\n", __FUNCTION__, (void *)displayHandle));
            res = DFB_PLATFORM_INVARG;
        }
        else
        {
            *pHandle = DFB_Platform_P_GetDisplayWindowHandle(displayHandle, windowIndex);

            if (*pHandle == NULL)
            {
                BDBG_ERR(("%s: Cannot get window handle for display %d and window index %d (res=%d)!\n", __FUNCTION__, displayIndex, windowIndex));
                res = DFB_PLATFORM_FAILURE;
            }
        }
    }

    BDBG_MSG_TRACE(("<<%s", __FUNCTION__));
    return res;

}

DFB_PlatformResult
DFB_Platform_AddDisplayOutput( DFB_PlatformNexusHandle       displayHandle,
                               DFB_PlatformDisplayOutputType outputType,
                               int                           outputIndex )
{
    DFB_PlatformResult      res;

    BDBG_MSG_TRACE((">>%s", __FUNCTION__));

    BDBG_ASSERT(displayHandle);

    if (!dfb_platform_initialised)
    {
        BDBG_ERR(("%s: DFB Platform not initialised. Please ensure DFB_Platform_Init() is called first.\n", __FUNCTION__));
        res = DFB_PLATFORM_INIT;
    }
    else
    {
        if (DFB_Platform_P_GetDisplayIndex(displayHandle) == -1)
        {
            BDBG_ERR(("%s: Invalid display handle %p!\n", __FUNCTION__, (void *)displayHandle));
            res = DFB_PLATFORM_INVARG;
        }
        else
        {
            if (outputType >= DFB_PlatformDisplayOutputType_eMax)
            {
                BDBG_ERR(("%s: Invalid output connector type %d!\n", __FUNCTION__, outputType));
                res = DFB_PLATFORM_INVARG;
            }
            else
            {
                res = DFB_Platform_P_GetDisplayOutputStatus(outputType, outputIndex);

                if (res == DFB_PLATFORM_OK)
                {
                    res = DFB_Platform_P_AddDisplayOutput(displayHandle, outputType, outputIndex);
                    if (res != DFB_PLATFORM_OK)
                        BDBG_ERR(("%s: Cannot add output of type %d and index %d to display (res=%d)!\n", __FUNCTION__,
                                   outputType, outputIndex, res));
                }
                else
                    BDBG_ERR(("%s: Output handle for output of type %d and index %d busy! (res=%d)!\n", __FUNCTION__,
                               outputType, outputIndex, res));
            }
        }
    }

    BDBG_MSG_TRACE(("<<%s", __FUNCTION__));
    return res;
}

DFB_PlatformResult
DFB_Platform_GetDisplayOutput( DFB_PlatformNexusHandle       displayHandle,
                               DFB_PlatformDisplayOutputType outputType,
                               int                           outputIndex,
                               DFB_PlatformNexusHandle      *pHandle )
{
    DFB_PlatformResult res;

    BDBG_MSG_TRACE((">>%s", __FUNCTION__));

    BDBG_ASSERT(displayHandle);
    BDBG_ASSERT(pHandle);

    if (!dfb_platform_initialised)
    {
        BDBG_ERR(("%s: DFB Platform not initialised. Please ensure DFB_Platform_Init() is called first.\n", __FUNCTION__));
        res = DFB_PLATFORM_INIT;
    }
    else
    {
         if (DFB_Platform_P_GetDisplayIndex(displayHandle) == -1)
        {
            BDBG_ERR(("%s: Invalid display handle %p!\n", __FUNCTION__, (void *)displayHandle));
            res = DFB_PLATFORM_INVARG;
        }
        else
        {
            if (outputType >= DFB_PlatformDisplayOutputType_eMax)
            {
                BDBG_ERR(("%s: Invalid output connector type %d!\n", __FUNCTION__, outputType));
                res = DFB_PLATFORM_INVARG;
            }
            else
            {
                res = DFB_Platform_P_GetDisplayOutput(displayHandle, outputType, outputIndex, pHandle);
                if (res != DFB_PLATFORM_OK)
                    BDBG_ERR(("%s: Cannot get output handle for output of type %d and index %d (res=%d)!\n", __FUNCTION__,
                               outputType, outputIndex, res));
            }
        }
     }

    BDBG_MSG_TRACE(("<<%s", __FUNCTION__));
    return res;

}

DFB_PlatformResult
DFB_Platform_SetDisplayOutput( DFB_PlatformNexusHandle       displayHandle,
                               DFB_PlatformDisplayOutputType outputType,
                               int                           outputIndex,
                               DFB_PlatformNexusHandle       handle )
{
    DFB_PlatformResult res;

    BDBG_MSG_TRACE((">>%s", __FUNCTION__));

    BDBG_ASSERT(displayHandle);

    if (!dfb_platform_initialised)
    {
        BDBG_ERR(("%s: DFB Platform not initialised. Please ensure DFB_Platform_Init() is called first.\n", __FUNCTION__));
        res = DFB_PLATFORM_INIT;
    }
    else
    {
        if (DFB_Platform_P_GetDisplayIndex(displayHandle) == -1)
        {
            BDBG_ERR(("%s: Invalid display handle %p!\n", __FUNCTION__, (void *)displayHandle));
            res = DFB_PLATFORM_INVARG;
        }
        else
        {
            if (outputType >= DFB_PlatformDisplayOutputType_eMax)
            {
                BDBG_ERR(("%s: Invalid output connector type %d!\n", __FUNCTION__, outputType));
                res = DFB_PLATFORM_INVARG;
            }
            else
            {
                res = DFB_Platform_P_SetDisplayOutput(displayHandle, outputType, outputIndex, handle);
                if (res != DFB_PLATFORM_OK)
                    BDBG_ERR(("%s: Cannot set output handle %p for output of type %d and index %d (res=%d)!\n", __FUNCTION__,
                               (void *)handle, outputType, outputIndex, res));
            }
        }
    }

    BDBG_MSG_TRACE(("<<%s", __FUNCTION__));
    return res;
}

DFB_PlatformResult
DFB_Platform_RemoveDisplayOutput( DFB_PlatformNexusHandle       displayHandle,
                                  DFB_PlatformDisplayOutputType outputType,
                                  int                           outputIndex )
{
    DFB_PlatformResult      res;

    BDBG_MSG_TRACE((">>%s", __FUNCTION__));

    BDBG_ASSERT(displayHandle);

    if (!dfb_platform_initialised)
    {
        BDBG_ERR(("%s: DFB Platform not initialised. Please ensure DFB_Platform_Init() is called first.\n", __FUNCTION__));
        res = DFB_PLATFORM_INIT;
    }
    else
    {
        if (DFB_Platform_P_GetDisplayIndex(displayHandle) == -1)
        {
            BDBG_ERR(("%s: Invalid display handle %p!\n", __FUNCTION__, (void *)displayHandle));
            res = DFB_PLATFORM_INVARG;
        }
        else
        {
            if (outputType >= DFB_PlatformDisplayOutputType_eMax)
            {
                BDBG_ERR(("%s: Invalid output connector type %d!\n", __FUNCTION__, outputType));
                res = DFB_PLATFORM_INVARG;
            }
            else
            {
                res =  DFB_Platform_P_GetDisplayOutputStatus(outputType, outputIndex);
                if (res != DFB_PLATFORM_BUSY)
                {
                    BDBG_ERR(("%s: Output of type %d and index %d has already been removed!\n", __FUNCTION__, outputType, outputIndex));
                    res = DFB_PLATFORM_FAILURE;
                }
                else
                {
                    res = DFB_Platform_P_RemoveDisplayOutput(displayHandle, outputType, outputIndex);
                    if (res != DFB_PLATFORM_OK)
                        BDBG_ERR(("%s: Cannot remove output of type %d and index %d from display (res=%d)!\n", __FUNCTION__,
                                    outputType, outputIndex, res));
                }
            }
        }
    }

    BDBG_MSG_TRACE(("<<%s", __FUNCTION__));
    return res;
}

DFB_PlatformResult
DFB_Platform_RemoveDisplayOutputs( DFB_PlatformNexusHandle displayHandle )
{
    DFB_PlatformResult            res = DFB_PLATFORM_OK;
    int                           outputSlot;
    int                           outputIndex;
    int                           displayIndex;
    DFB_PlatformDisplayOutputType outputType;
    DFB_P_DisplayOutputStatus    *pOutputStatus;


    BDBG_MSG_TRACE((">>%s", __FUNCTION__));

    BDBG_ASSERT(displayHandle);

    if (!dfb_platform_initialised)
    {
        BDBG_ERR(("%s: DFB Platform not initialised. Please ensure DFB_Platform_Init() is called first.\n", __FUNCTION__));
        res = DFB_PLATFORM_INIT;
    }
    else
    {
        displayIndex = DFB_Platform_P_GetDisplayIndex(displayHandle);

        if (displayIndex == -1)
        {
            BDBG_ERR(("%s: Invalid display handle %p!\n", __FUNCTION__, (void *)displayHandle));
            res = DFB_PLATFORM_INVARG;
        }
        else
        {
            pOutputStatus = &dfb_p_display_shared->output[0];

            for (outputSlot = 0; outputSlot < DFB_PLATFORM_NUM_DISPLAY_OUTPUTS_MAX; outputSlot++)
            {
                if (pOutputStatus->outputHandle && (displayIndex == pOutputStatus->displayIndex))
                {
                    outputType  = pOutputStatus->outputType;
                    outputIndex = pOutputStatus->outputIndex;

                    res = DFB_Platform_P_RemoveDisplayOutput(displayHandle, outputType, outputIndex);
                    if (res != DFB_PLATFORM_OK)
                    {
                        BDBG_ERR(("%s: Cannot remove output of type %d and index %d from display%d (res=%d)!\n", __FUNCTION__,
                                    outputType, outputIndex, displayIndex, res));
                        break;
                    }
                }
                pOutputStatus++;
            }
        }
    }

    BDBG_MSG_TRACE(("<<%s", __FUNCTION__));
    return res;
}

DFB_PlatformResult
DFB_Platform_HdmiHotplugHandler( DFB_PlatformNexusHandle displayHandle,
                                 DFB_PlatformNexusHandle hdmiHandle )
{
    DFB_PlatformResult res;

    BDBG_MSG_TRACE((">>%s", __FUNCTION__));

    BDBG_ASSERT(displayHandle);
    BDBG_ASSERT(hdmiHandle);

    if (!dfb_platform_initialised)
    {
        BDBG_ERR(("%s: DFB Platform not initialised. Please ensure DFB_Platform_Init() is called first.\n", __FUNCTION__));
        res = DFB_PLATFORM_INIT;
    }
    else
    {
        if (DFB_Platform_P_GetDisplayIndex(displayHandle) == -1)
        {
            BDBG_ERR(("%s: Invalid display handle %p!\n", __FUNCTION__, (void *)displayHandle));
            res = DFB_PLATFORM_INVARG;
        }
        else
        {
            res = DFB_Platform_P_HdmiHotplugHandler(displayHandle, hdmiHandle);
            if (res != DFB_PLATFORM_OK)
                BDBG_ERR(("%s: Problem encountered setting HDMI hotplug handler (res=%d)!\n", __FUNCTION__, res));
        }
    }

    BDBG_MSG_TRACE(("<<%s", __FUNCTION__));
    return res;
}

/* An external application can call this function (via DFB_Platform_Init()) to pass information about handles, heaps, etc. to DFB */
DFB_PlatformResult
DFB_Platform_Init_tagged( DFB_PlatformSettings *pSettings,
                          unsigned              versionCheck,
                          unsigned              structSizeCheck )
{
    DFB_PlatformResult   err = DFB_PLATFORM_OK;
    int                  displayIndex;
    int                  windowSlot;
    int                  outputSlot;
    int                  i;

    BDBG_MSG_TRACE((">>%s", __FUNCTION__));

    BDBG_Init();

    /* Check for binary compatibility */
    if (versionCheck != DFB_PLATFORM_GET_VERSION()) {
        BDBG_ERR(("DFB_Platform_Init() failed with version mismatch (dfb=%d.%d, caller=%d.%d). Please recompile application and/or DirectFB.",
            (DFB_PLATFORM_GET_VERSION() & DFB_PLATFORM_P_VERSION_MAJ_MASK) >> DFB_PLATFORM_P_VERSION_MAJ_SHIFT,
            (DFB_PLATFORM_GET_VERSION() & DFB_PLATFORM_P_VERSION_MIN_MASK) >> DFB_PLATFORM_P_VERSION_MIN_SHIFT,
            (versionCheck                 & DFB_PLATFORM_P_VERSION_MAJ_MASK) >> DFB_PLATFORM_P_VERSION_MAJ_SHIFT,
            (versionCheck                 & DFB_PLATFORM_P_VERSION_MIN_MASK) >> DFB_PLATFORM_P_VERSION_MIN_SHIFT));
        return DFB_PLATFORM_VERSIONMISMATCH;
    }

    if (structSizeCheck != DFB_PLATFORM_P_GET_STRUCT_SIZE()) {
        BDBG_ERR(("DFB_Platform_Init() failed with struct size mismatch (dfb=%d, caller=%d). Please recompile application and/or DirectFB.",
            DFB_PLATFORM_P_GET_STRUCT_SIZE(), structSizeCheck));
        return DFB_PLATFORM_INVARG;
    }

    /* If the client hasn't specified any settings, then this is also an error! */
    if (pSettings == NULL)
    {
        BDBG_ERR(("DFB_Platform_Init() failed because no settings were passed!"));
        return DFB_PLATFORM_INVARG;
    }

    /* If we have already initialised the platform code in the same process, then no need to do it again! */
    if (dfb_platform_initialised == true)
    {
        BDBG_WRN(("DFB_Platform_Init(): DFB Platform already initialised!"));
        return DFB_PLATFORM_OK;
    }

    BDBG_WRN(("Broadcom Corporation DirectFB%s Platform Layer Version %d.%d",
              (isXS ? "-XS" : ""), DFB_PLATFORM_VERSION_MAJOR, DFB_PLATFORM_VERSION_MINOR));

    err = DFB_Platform_P_SetupSharedMemory();

    if (err != DFB_PLATFORM_OK) {
        BDBG_ERR(("DFB_Platform_Init() failed to setup shared memory!"));
        return err;
    }

    /* Ensure per process device status is cleared */
    memset(&dfb_device_status, 0, sizeof(dfb_device_status));

     /* Perform initialisation */
    err = DFB_Platform_P_Init(pSettings, dfb_p_display_shared);

    if (err != DFB_PLATFORM_OK) {
        BDBG_ERR(("DFB_Platform_Init() failed to initialise private settings!"));

        return err;
    }

    /* Always ensure that the heap handles are initialised */
    DFB_Platform_P_SetHeapHandles(pSettings);

    /* Display the heaps */
    DFB_Platform_P_DisplayHeaps(pSettings);

    /*Register the user Hdmi hotplug callback here in case display init is not done here.*/
    memcpy(&dfb_p_display_shared->display[displayIndex].displaySettings.hdmiHotplugDescriptor,
           &pSettings->platformSettings.hdmiHotplugDescriptor,
           sizeof(DFB_PlatformCallbackDesc));

    /* Setup the displays */
    for (displayIndex = 0; displayIndex < NUM_DISPLAYS; displayIndex++)
    {
        /* Should we initialise the display? */
        if (pSettings->display[displayIndex].init)
        {
#if !BCMNEXUS_NSC_SUPPORT
            /* Check to make sure that another process hasn't already opened the display */
            if (dfb_p_display_shared->display[displayIndex].displayHandle != NULL)
            {
                BDBG_WRN(("DFB_Platform_Init() Display module already opened!\n"));
                continue;
            }
#endif

            /* Do we need to actually open the display? */
            if (pSettings->display[displayIndex].handle == NULL)
            {
                BDBG_MSG(("DFB_Platform_Init() Opening display%d...\n", displayIndex));
                err = DFB_Platform_P_Display_Init(displayIndex, &pSettings->display[displayIndex].initSettings,
                                                  &pSettings->display[displayIndex].handle, pSettings->slave, pSettings->clientId);

                /* Indicate that the display was opened by this process... */
                dfb_device_status.IsDisplayOpened[displayIndex] = (err == DFB_PLATFORM_OK);
            }

            if (err == DFB_PLATFORM_OK)
            {
                if (pSettings->display[displayIndex].handle == NULL)
                {
                    BDBG_ERR(("DFB_Platform_Init() Cannot open display %d!\n", displayIndex));
                    err = DFB_PLATFORM_FAILURE;
                    goto err_init;
                }
                else
                {
                    DFB_PlatformVideoFormat format;

                    /* Save the display state if it is really open */
                    memcpy(&dfb_platform_shared->display[displayIndex], &pSettings->display[displayIndex], sizeof(pSettings->display[0]));

                    format = pSettings->display[displayIndex].initSettings.format;

                    /* Also ensure that the display shared state is consistent */
                    dfb_p_display_shared->display[displayIndex].displayHandle               = pSettings->display[displayIndex].handle;
                    dfb_p_display_shared->display[displayIndex].displaySettings.format      = format;
                    dfb_p_display_shared->display[displayIndex].displaySettings.aspectRatio = pSettings->display[displayIndex].initSettings.aspectRatio;
                    memcpy(&dfb_p_display_shared->display[displayIndex].displaySettings.background,
                           &pSettings->display[displayIndex].initSettings.background,
                           sizeof(dfb_p_display_shared->display[displayIndex].displaySettings.background));

                    DFB_Platform_P_DFBFormatToDisplaySize(format, &dfb_p_display_shared->display[displayIndex].displaySize);
                }
            }
        }
    }

    /* Setup the display windows */
    for (windowSlot = 0; windowSlot < DFB_PLATFORM_NUM_DISPLAY_WINDOWS_MAX; windowSlot++)
    {
        /* Should we initialise the output? */
        if (pSettings->displayWindow[windowSlot].init)
        {
            /* Check to make sure that another process hasn't already init the display window */
            if (dfb_platform_shared->displayWindow[windowSlot].handle != NULL)
            {
                BDBG_WRN(("DFB_Platform_Init() Display Window slot %d already opened!\n", windowSlot));
                continue;
            }

            /* Do we need to actually init the display window? */
            if (pSettings->displayWindow[windowSlot].handle == NULL)
            {
                BDBG_MSG(("DFB_Platform_Init() Opening display window slot%d...\n", windowSlot));
                err = DFB_Platform_P_DisplayWindow_Init(&pSettings->displayWindow[windowSlot].initSettings, &pSettings->displayWindow[windowSlot].handle);

                /* Indicate that the window was opened by this process... */
                dfb_device_status.IsWindowOpened[windowSlot] = (err == DFB_PLATFORM_OK);
            }

            if (err == DFB_PLATFORM_OK)
            {
                if (pSettings->displayWindow[windowSlot].handle == NULL)
                {
                    BDBG_ERR(("DFB_Platform_Init() Cannot open display window slot %d!\n", windowSlot));
                    err = DFB_PLATFORM_FAILURE;
                    goto err_init;
                }
                else
                {
                    /* Save the display window state if it is really open */
                    memcpy(&dfb_platform_shared->displayWindow[windowSlot], &pSettings->displayWindow[windowSlot], sizeof(pSettings->displayWindow[0]));

                    /* Also ensure that the display window shared state is consistent */
                    dfb_p_display_shared->window[windowSlot].windowHandle = pSettings->displayWindow[windowSlot].handle;
                    dfb_p_display_shared->window[windowSlot].windowIndex  = pSettings->displayWindow[windowSlot].initSettings.windowIndex;
                    dfb_p_display_shared->window[windowSlot].displayIndex = pSettings->displayWindow[windowSlot].initSettings.displayIndex;
                }
            }
        }
    }

    /* Setup the display outputs */
    for (outputSlot = 0; outputSlot < DFB_PLATFORM_NUM_DISPLAY_OUTPUTS_MAX; outputSlot++)
    {
        /* Should we initialise the output? */
        if (pSettings->displayOutput[outputSlot].init)
        {
            /* Check to make sure that another process hasn't already init the output */
            if (dfb_p_display_shared->output[outputSlot].outputHandle != NULL)
            {
                BDBG_WRN(("DFB_Platform_Init() Output slot %d already opened!\n", outputSlot));
                continue;
            }

            /* Do we need to actually init the output? */
            if (pSettings->displayOutput[outputSlot].handle == NULL)
            {
                BDBG_MSG(("DFB_Platform_Init() Opening output slot%d...\n", outputSlot));
                err = DFB_Platform_P_DisplayOutput_Init(&pSettings->displayOutput[outputSlot].initSettings, &pSettings->displayOutput[outputSlot].handle);
            }

            if (err == DFB_PLATFORM_OK)
            {
                if (pSettings->displayOutput[outputSlot].handle == NULL)
                {
                    BDBG_ERR(("DFB_Platform_Init() Cannot open output slot %d!\n", outputSlot));
                    err = DFB_PLATFORM_FAILURE;
                    goto err_init;
                }
                else
                {
                    /* Save the output state if it is really open */
                    memcpy(&dfb_platform_shared->displayOutput[outputSlot], &pSettings->displayOutput[outputSlot], sizeof(pSettings->displayOutput[0]));

                    /* Also ensure that the output shared state is consistent */
                    dfb_p_display_shared->output[outputSlot].outputHandle = pSettings->displayOutput[outputSlot].handle;
                    dfb_p_display_shared->output[outputSlot].outputType   = pSettings->displayOutput[outputSlot].initSettings.outputType;
                    dfb_p_display_shared->output[outputSlot].outputIndex  = pSettings->displayOutput[outputSlot].initSettings.outputIndex;
                    dfb_p_display_shared->output[outputSlot].displayIndex = pSettings->displayOutput[outputSlot].initSettings.displayIndex;
                }
            }
        }
    }

    /* Should we initialise the graphics 2D module? */
    for (i=0; i < NEXUS_NUM_2D_ENGINES; ++i)
    {
        if (pSettings->graphics2d[i].init == true)
        {
#if !BCMNEXUS_NSC_SUPPORT
            /* Check to make sure that another process hasn't already opened the graphics2d module */
            if (dfb_platform_shared->graphics2d[i].handle != NULL)
            {
                BDBG_WRN(("DFB_Platform_Init() Graphics2d [%d] module already opened!\n", i));
            }
            else
#endif
            {
                /* Do we need to actually open the graphics2d module? */
                if (pSettings->graphics2d[i].handle == NULL)
                {
                    BDBG_MSG(("DFB_Platform_Init() Opening graphics2d [%d] module...\n", i));
                    err = DFB_Platform_P_Graphics2D_Init(&pSettings->graphics2d[i].initSettings, &pSettings->graphics2d[i].handle, i);

                    /* Indicate successfull open of the graphics2d device in the current process... */
                    dfb_device_status.IsGraphics2DOpened = (err == DFB_PLATFORM_OK);
                }

                if (err == DFB_PLATFORM_OK)
                {
                    if (pSettings->graphics2d[i].handle == NULL)
                    {
                        BDBG_ERR(("DFB_Platform_Init() Cannot open graphics2d [%d]!\n", i));
                        err = DFB_PLATFORM_FAILURE;
                        goto err_init;
                    }
                    else
                    {
                        /* Save the graphics2d module state if it is really open */
                        memcpy(&dfb_platform_shared->graphics2d, &pSettings->graphics2d, sizeof(pSettings->graphics2d));
                    }
                }
            }
        }
    }

    /* Should we initialise the graphics 3D module? */
    if (pSettings->graphics3d.init == true)
    {
        /* Check to make sure that another process hasn't already opened the graphics3d module */
        if (dfb_platform_shared->graphics3d.handle != NULL)
        {
            BDBG_WRN(("DFB_Platform_Init() Graphics3d module already opened!\n"));
        }
        else
        {
            /* Do we need to actually open the graphics3d module? */
            if (pSettings->graphics3d.handle == NULL)
            {
                BDBG_MSG(("DFB_Platform_Init() Opening graphics3d module...\n"));
                err = DFB_Platform_P_Graphics3D_Init(&pSettings->graphics3d.initSettings, &pSettings->graphics3d.handle);

                /* Indicate successfull open of the graphics3d device in the current process... */
                dfb_device_status.IsGraphics3DOpened = (err == DFB_PLATFORM_OK);
            }

            if (err == DFB_PLATFORM_OK)
            {
                if (pSettings->graphics3d.handle == NULL)
                {
                    BDBG_ERR(("DFB_Platform_Init() Cannot open graphics3d!\n"));
                    err = DFB_PLATFORM_FAILURE;
                    goto err_init;
                }
                else
                {
                    /* Save the graphics3d module state if it is really open */
                    memcpy(&dfb_platform_shared->graphics3d, &pSettings->graphics3d, sizeof(pSettings->graphics3d));
                }
            }
        }
    }

    /* Should we initialise the picture decoder module? */
    if (pSettings->pictureDecoder.init == true && !getenv("sw_picture_decode"))
    {
        /* Check to make sure that another process hasn't already opened the picture decoder module */
        if (dfb_platform_shared->pictureDecoder.handle != NULL)
        {
            BDBG_WRN(("DFB_Platform_Init() Picture Decoder module already opened!\n"));
        }
        else
        {
            /* Do we need to actually open the picture decoder module? */
            if (pSettings->pictureDecoder.handle == NULL)
            {
                BDBG_MSG(("DFB_Platform_Init() Opening picture decoder module...\n"));
                err = DFB_Platform_P_PictureDecoder_Init(&pSettings->pictureDecoder.initSettings, &pSettings->pictureDecoder.handle);

                /* Indicate successfull open of the picture decoder device in the current process... */
                dfb_device_status.IsPictureDecoderOpened = (err == DFB_PLATFORM_OK);
            }

            if (err == DFB_PLATFORM_OK)
            {
                if (pSettings->pictureDecoder.handle == NULL)
                {
                    BDBG_ERR(("DFB_Platform_Init() Cannot open picture decoder!\n"));
                    err = DFB_PLATFORM_FAILURE;
                    goto err_init;
                }
                else
                {
                    /* Save the picture decoder module state if it is really open */
                    memcpy(&dfb_platform_shared->pictureDecoder, &pSettings->pictureDecoder, sizeof(pSettings->pictureDecoder));
                }
            }
        }
    }


    /* Only copy all settings over to shared memory if we are the master process */
    if (dfb_is_master)
    {
        BDBG_MSG(("DFB_Platform_Init() Copying platform settings to shared memory...\n"));
        pSettings->platformSettings.platforSettingsValid =(unsigned int) PLATFORM_SETTINGS_VALID;
        *dfb_platform_shared = *pSettings;
        dfb_platform_status_shared->isMaster = true;
    }
    else
    {
        dfb_platform_status_shared->isMaster = false;
    }

    /* Indicate that we have at this point successfully initialised the DFB Platform code */
    dfb_platform_initialised = true;                        /* Local process state */
    dfb_platform_status_shared->initOrJoinNexus = true;     /* Global process state */


    goto err_ok;

err_init:

    DFB_Platform_P_CloseSharedMemory(dfb_platform_shared);
err_ok:
    BDBG_MSG_TRACE(("<<%s", __FUNCTION__));
    return err;
}

DFB_PlatformResult
DFB_Platform_Uninit( void )
{
    DFB_PlatformResult res;
    int       displayIndex;
    int       windowSlot;

    BDBG_MSG_TRACE((">>%s", __FUNCTION__));

    if (!dfb_platform_initialised)
    {
        BDBG_WRN(("%s(): DFB Platform already uninitialised.\n", __FUNCTION__));
        res = DFB_PLATFORM_OK;
    }
    else
    {
        if (dfb_device_status.IsGraphics3DOpened)
        {
            DFB_Platform_P_Graphics3D_Uninit(dfb_platform_shared->graphics3d.handle);
            dfb_platform_shared->graphics3d.handle = NULL;
            dfb_device_status.IsGraphics3DOpened  = false;
        }

        if (dfb_device_status.IsGraphics2DOpened)
        {

                        if (dfb_platform_shared->graphics2d[1].handle != NULL)
                        {
                DFB_Platform_P_Graphics2D_Uninit(dfb_platform_shared->graphics2d[1].handle);
                dfb_platform_shared->graphics2d[1].handle = NULL;
                        }
            DFB_Platform_P_Graphics2D_Uninit(dfb_platform_shared->graphics2d[0].handle);
            dfb_platform_shared->graphics2d[0].handle = NULL;
            dfb_device_status.IsGraphics2DOpened  = false;
        }

        if (dfb_device_status.IsPictureDecoderOpened)
        {
            DFB_Platform_P_PictureDecoder_Uninit(dfb_platform_shared->pictureDecoder.handle);
            dfb_platform_shared->pictureDecoder.handle = NULL;
            dfb_device_status.IsPictureDecoderOpened  = false;
        }

        for (windowSlot = 0; windowSlot < DFB_PLATFORM_NUM_DISPLAY_WINDOWS_MAX; windowSlot++)
        {
            if (dfb_device_status.IsWindowOpened[windowSlot])
            {
                DFB_Platform_P_DisplayWindow_Uninit(dfb_platform_shared->displayWindow[windowSlot].handle);
                dfb_platform_shared->displayWindow[windowSlot].handle = NULL;
                dfb_device_status.IsWindowOpened[windowSlot] = false;
            }
        }

        for (displayIndex = 0; displayIndex < NUM_DISPLAYS; displayIndex++)
        {
            if (dfb_device_status.IsDisplayOpened[displayIndex])
            {
                DFB_Platform_RemoveDisplayOutputs(dfb_platform_shared->display[displayIndex].handle);

                DFB_Platform_P_Display_Uninit(dfb_platform_shared->display[displayIndex].handle, dfb_platform_shared->slave);
                dfb_platform_shared->display[displayIndex].handle = NULL;
                dfb_device_status.IsDisplayOpened[displayIndex]  = false;
            }
        }

        res = DFB_Platform_P_Uninit(dfb_platform_shared->slave);

        if (res != DFB_PLATFORM_OK)
        {
            BDBG_ERR(("%s: Cannot uninitialise platform (res=%d)!\n", __FUNCTION__, res));
        }
        else
        {
            if (dfb_is_master)
            {
                dfb_platform_status_shared->initOrJoinNexus = false;
            }
#if FUSION_BUILD_MULTI
            else
            {
                simple_client_shutdown();
            }
#endif
            BDBG_MSG(("%s: Destroying The Shared Memory\n", __FUNCTION__));
            DFB_Platform_P_CloseSharedMemory(dfb_platform_shared);
            dfb_platform_shared = NULL;
            dfb_platform_initialised = false;
        }
    }

    BDBG_MSG_TRACE(("<<%s", __FUNCTION__));

    BDBG_Uninit();
    return res;
}

DFB_PlatformResult DFB_Platform_P_GetSharedSettings(DFB_PlatformSharedSetting setting, void* pValue)
{

    DFB_PlatformResult res = DFB_PLATFORM_UNSUPPORTED;

    if (!dfb_platform_initialised)
    {
        BDBG_ERR(("%s: DFB Platform not initialised. Please ensure DFB_Platform_Init() is called first.\n", __FUNCTION__));
        res = DFB_PLATFORM_INIT;
    }
    else if (setting < DFB_PlatformSharedSetting_eMax )
    {
        switch (setting)
        {
            case DFB_PlatformSharedSetting_eSurfacePool:
                *(unsigned int*)pValue = dfb_p_platform_shared_data->surfacePool;
                BDBG_MSG_TRACE(("%s(%d) Returning value 0x%x\n",__FUNCTION__,__LINE__,pValue));
                res = DFB_PLATFORM_OK;
                break;
            case DFB_PlatformSharedSetting_e3DGraphicsPool:
                *(unsigned int*)pValue = dfb_p_platform_shared_data->graphics3DPool;
                BDBG_MSG_TRACE(("%s(%d) Returning value 0x%x\n",__FUNCTION__,__LINE__,pValue));
                res = DFB_PLATFORM_OK;
                break;
            default:
                break;
        }
    }

    return res;
}

DFB_PlatformResult DFB_Platform_P_SetSharedSettings(DFB_PlatformSharedSetting setting, void* pValue)
{

    DFB_PlatformResult res = DFB_PLATFORM_UNSUPPORTED;

    /* Have to be master application to set data */
    if (!dfb_is_master)
        return DFB_PLATFORM_ACCESSDENIED;

    if (!dfb_platform_initialised)
    {
        BDBG_ERR(("%s: DFB Platform not initialised. Please ensure DFB_Platform_Init() is called first.\n", __FUNCTION__));
        res = DFB_PLATFORM_INIT;
    }
    else if (setting < DFB_PlatformSharedSetting_eMax )
    {
        switch (setting)
        {
            case DFB_PlatformSharedSetting_eSurfacePool:
                BDBG_MSG_TRACE(("%s(%d) Saving value 0x%x\n",__FUNCTION__,__LINE__,*(unsigned int*)pValue));
                dfb_p_platform_shared_data->surfacePool = *(unsigned int*)pValue;
                res = DFB_PLATFORM_OK;
                break;
            case DFB_PlatformSharedSetting_e3DGraphicsPool:
                BDBG_MSG_TRACE(("%s(%d) Saving value 0x%x\n",__FUNCTION__,__LINE__,*(unsigned int*)pValue));
                dfb_p_platform_shared_data->graphics3DPool = *(unsigned int*)pValue;
                res = DFB_PLATFORM_OK;
                break;
            default:
                break;
        }
    }

    return res;
}

DFB_PlatformResult DFB_Platform_SetClientComposition(DFB_PlatformSurfaceComposition *pClientComposition)
{
    return DFB_Platform_P_SetClientComposition(pClientComposition);
}

DFB_PlatformResult DFB_Platform_GetClientComposition(DFB_PlatformSurfaceComposition *pClientComposition)
{
    return DFB_Platform_P_GetClientComposition(pClientComposition);
}
