/******************************************************************************
 *    (c)2010-2013 Broadcom Corporation
 *
 * This program is the proprietary software of Broadcom Corporation and/or its licensors,
 * and may only be used, duplicated, modified or distributed pursuant to the terms and
 * conditions of a separate, written license agreement executed between you and Broadcom
 * (an "Authorized License").  Except as set forth in an Authorized License, Broadcom grants
 * no license (express or implied), right to use, or waiver of any kind with respect to the
 * Software, and Broadcom expressly reserves all rights in and to the Software and all
 * intellectual property rights therein.  IF YOU HAVE NO AUTHORIZED LICENSE, THEN YOU
 * HAVE NO RIGHT TO USE THIS SOFTWARE IN ANY WAY, AND SHOULD IMMEDIATELY
 * NOTIFY BROADCOM AND DISCONTINUE ALL USE OF THE SOFTWARE.
 *
 * Except as expressly set forth in the Authorized License,
 *
 * 1.     This program, including its structure, sequence and organization, constitutes the valuable trade
 * secrets of Broadcom, and you shall use all reasonable efforts to protect the confidentiality thereof,
 * and to use this information only in connection with your use of Broadcom integrated circuit products.
 *
 * 2.     TO THE MAXIMUM EXTENT PERMITTED BY LAW, THE SOFTWARE IS PROVIDED "AS IS"
 * AND WITH ALL FAULTS AND BROADCOM MAKES NO PROMISES, REPRESENTATIONS OR
 * WARRANTIES, EITHER EXPRESS, IMPLIED, STATUTORY, OR OTHERWISE, WITH RESPECT TO
 * THE SOFTWARE.  BROADCOM SPECIFICALLY DISCLAIMS ANY AND ALL IMPLIED WARRANTIES
 * OF TITLE, MERCHANTABILITY, NONINFRINGEMENT, FITNESS FOR A PARTICULAR PURPOSE,
 * LACK OF VIRUSES, ACCURACY OR COMPLETENESS, QUIET ENJOYMENT, QUIET POSSESSION
 * OR CORRESPONDENCE TO DESCRIPTION. YOU ASSUME THE ENTIRE RISK ARISING OUT OF
 * USE OR PERFORMANCE OF THE SOFTWARE.
 *
 * 3.     TO THE MAXIMUM EXTENT PERMITTED BY LAW, IN NO EVENT SHALL BROADCOM OR ITS
 * LICENSORS BE LIABLE FOR (i) CONSEQUENTIAL, INCIDENTAL, SPECIAL, INDIRECT, OR
 * EXEMPLARY DAMAGES WHATSOEVER ARISING OUT OF OR IN ANY WAY RELATING TO YOUR
 * USE OF OR INABILITY TO USE THE SOFTWARE EVEN IF BROADCOM HAS BEEN ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGES; OR (ii) ANY AMOUNT IN EXCESS OF THE AMOUNT
 * ACTUALLY PAID FOR THE SOFTWARE ITSELF OR U.S. $1, WHICHEVER IS GREATER. THESE
 * LIMITATIONS SHALL APPLY NOTWITHSTANDING ANY FAILURE OF ESSENTIAL PURPOSE OF
 * ANY LIMITED REMEDY.
 *
 *****************************************************************************/

//#define DIRECT_ENABLE_DEBUG


#include "bcm_config.h"

#include <direct/LockWQ.h>

#include <core/Renderer.h>
#include <core/Task.h>
#include <core/Util.h>

extern "C" {
#include <directfb.h>

#include <direct/debug.h>
#include <direct/messages.h>

#include <core/surface_allocation.h>
#include <core/surface_pool.h>
#include <core/system.h>

#include <gfx/clip.h>
#include <gfx/convert.h>

#include "bcmnexus_gfx.h"
#include "bcmnexus_pool.h"
}


#if BCM_DFB_USE_TASK_MANAGER

#include "M2MC_Core.h"
#include "M2MC_Engine.h"
#include "M2MC_Task.h"


#define GFX_FULL_PACKET_SIZE (2 * \
    sizeof (BM2MC_PACKET_PacketSourceFeeder) + \
    sizeof (BM2MC_PACKET_PacketDestinationFeeder) + \
    sizeof (BM2MC_PACKET_PacketOutputFeeder) + \
    sizeof (BM2MC_PACKET_PacketOutputControl) + \
    sizeof (BM2MC_PACKET_PacketBlend) + \
    sizeof (BM2MC_PACKET_PacketRop) + \
    sizeof (BM2MC_PACKET_PacketSourceColorkey) * 2 + \
    sizeof (BM2MC_PACKET_PacketSourceColorkeyEnable) * 2 + \
    sizeof (BM2MC_PACKET_PacketFilter) + \
    sizeof (BM2MC_PACKET_PacketFilterEnable) + \
    sizeof (BM2MC_PACKET_PacketSourceColorMatrix) + \
    sizeof (BM2MC_PACKET_PacketSourceColorMatrixEnable) + \
    sizeof (BM2MC_PACKET_PacketSourcePalette) + \
    sizeof (BM2MC_PACKET_PacketAlphaPremultiply) + \
    sizeof (BM2MC_PACKET_PacketMirror) + \
    sizeof (BM2MC_PACKET_PacketScaleBlendBlit))

#define GFX_FILL_PACKET_SIZE   sizeof(BM2MC_PACKET_PacketFillBlit)

#define GFX_DRAW_PACKET_SIZE 4*sizeof(BM2MC_PACKET_PacketFillBlit)


#define GFX_BLIT_PACKET_SIZE   sizeof(BM2MC_PACKET_PacketScaleBlendBlit) + \
                               sizeof(BM2MC_PACKET_PacketFilter) + \
                               sizeof(BM2MC_PACKET_PacketFilterEnable)

#define GFX_TWO_PASSES_BLIT_PACKET_SIZE ( \
    sizeof(BM2MC_PACKET_PacketSourceFeeder) * 2 + \
    sizeof(BM2MC_PACKET_PacketOutputFeeder) * 2 + \
    sizeof(BM2MC_PACKET_PacketRop) * 2 + \
    sizeof(BM2MC_PACKET_PacketSourceColorkeyEnable) * 2 + \
    sizeof(BM2MC_PACKET_PacketDestinationColorkeyEnable) * 2 + \
    sizeof(BM2MC_PACKET_PacketSourceColorMatrixEnable) * 2 + \
    sizeof(BM2MC_PACKET_PacketAlphaPremultiply) * 2 + \
    sizeof(BM2MC_PACKET_PacketMirror) * 2 + \
    sizeof(BM2MC_PACKET_PacketFilter) + \
    sizeof(BM2MC_PACKET_PacketFilterEnable) * 2 + \
    sizeof(BM2MC_PACKET_PacketScaleBlendBlit) * 2)



D_DEBUG_DOMAIN( M2MC_Engine,       "M2MC/Engine",       "M2MC Engine" );
D_DEBUG_DOMAIN( M2MC_Task,         "M2MC/Task",         "M2MC Task" );
D_DEBUG_DOMAIN( M2MC_Core,         "M2MC/Core",         "M2MC Core" );

/*********************************************************************************************************************/

namespace Broadcom {


M2MCEngine::M2MCEngine( unsigned int num_cores )
     :
     threads( "M2MC", num_cores ) // Have a thread per hardware core
{
     NEXUS_MemoryAllocationSettings surfaceMemorySettings;

     D_DEBUG_AT( M2MC_Engine, "M2MCEngine::%s() - cores %d\n", __FUNCTION__ , num_cores);

     D_ASSERT( num_cores > 0 );

     caps.cores            = num_cores;
     caps.clipping         = (DFBAccelerationMask)(DFXL_ALL /*& ~DFXL_FILLRECTANGLE*/);
     caps.render_options   = (DFBSurfaceRenderOptions)(DSRO_SMOOTH_DOWNSCALE | DSRO_SMOOTH_UPSCALE);
     caps.max_scale_down_x = BCMNEXUS_GFX_SCALE_DOWN_MAX_X;
     caps.max_scale_down_y = BCMNEXUS_GFX_SCALE_DOWN_MAX_Y;
     caps.max_operations   = 7000;

     pBrcmDfb = (DFBBCMNEXUS *) dfb_system_data();

     for (unsigned int i=0; i<num_cores; i++) {
          M2MCCore *t = new M2MCCore( i );
          cores.push_back( t );
     }

     /* Preallocate memory for intermediate surface for two pass down scaling */
     intermediateSurfaceSize = ((4096 + BCMNEXUS_GFX_SCALE_DOWN_MAX_X - 1) / BCMNEXUS_GFX_SCALE_DOWN_MAX_X) *
                               ((2048 + BCMNEXUS_GFX_SCALE_DOWN_MAX_Y - 1) / BCMNEXUS_GFX_SCALE_DOWN_MAX_Y) * 4;

     NEXUS_Memory_GetDefaultAllocationSettings(&surfaceMemorySettings);
     surfaceMemorySettings.alignment = 4096;  /* Surface needs to be aligned on a 4K boundary */
     surfaceMemorySettings.heap = GET_OFFSCREEN_HEAP_HND(pBrcmDfb);

     if (NEXUS_Memory_Allocate((size_t)intermediateSurfaceSize, &surfaceMemorySettings, &intermediateSurfaceAddr) != NEXUS_SUCCESS)
     {
         D_ERROR( "bcmNexus/Pool: Insufficient memory to preallocate intermediate surface memory (%d bytes)\n", intermediateSurfaceSize);
         // FIXME: return DFB_NOVIDEOMEMORY;
     }
}

DFBResult
M2MCEngine::bind( DirectFB::Renderer::Setup *setup )
{
     D_DEBUG_AT( M2MC_Engine, "M2MCEngine::%s()\n", __FUNCTION__ );

     if (direct_config_get_int_value( "m2mc-stats" )) {
          static long long last;
          long long        now = direct_clock_get_time( DIRECT_CLOCK_MONOTONIC );

          if (last && now - last >= 300000) {
               Direct::String str;
               long long      total_busy = 0;
               size_t         num = cores.size();

               for (unsigned int i=0; i<num; i++) {
                    long long time_total = cores[i]->GetTotal();
                    long long core_busy  = 1000 - cores[i]->GetIdle( true );

                    total_busy += core_busy;

                    str.PrintF( "Core%u %3lld.%lld%% busy (%lldms) %u checkpoints   ", i,
                                core_busy / 10, core_busy % 10, time_total / 1000, cores[i]->checkpoint2d_count );
               }

               str.PrintF( "(total %3lld.%lld%%)", total_busy / 10, total_busy % 10 );

               printf( "%s\n", str.buffer() );

               last = now;
          }
          else if (!last)
               last = now;
     }

     for (unsigned int i=0; i<setup->tiles; i++) {
          setup->tasks[i] = new M2MCTask( this, i );
     }

     return DFB_OK;
}

DFBResult
M2MCEngine::check( DirectFB::Renderer::Setup *setup )
{
     D_DEBUG_AT( M2MC_Engine, "M2MCEngine::%s()\n", __FUNCTION__ );

     for (unsigned int i=0; i<setup->tiles; i++) {
          M2MCTask *mytask = (M2MCTask *) setup->tasks[i];

          if (mytask->buffer.GetLength() >= 0x17800)
               return DFB_LIMITEXCEEDED;
     }

     return DFB_OK;
}

DFBResult
M2MCEngine::CheckState( CardState              *state,
                        DFBAccelerationMask     accel )
{
     D_DEBUG_AT( M2MC_Engine, "M2MCEngine::%s()\n", __FUNCTION__ );

     if (state->destination->config.caps & DSCAPS_SYSTEMONLY)
          return DFB_UNSUPPORTED;

     if (DFB_BLITTING_FUNCTION( accel )) {
          if (state->source->config.caps & DSCAPS_SYSTEMONLY)
               return DFB_UNSUPPORTED;

          if ((state->blittingflags & (DSBLIT_SRC_MASK_ALPHA | DSBLIT_SRC_MASK_COLOR)) && state->source_mask->config.caps & DSCAPS_SYSTEMONLY)
               return DFB_UNSUPPORTED;

          if (accel == DFXL_BLIT2 &&
              (state->source2->config.caps & DSCAPS_SYSTEMONLY))
               return DFB_UNSUPPORTED;
     }

     bcmnexus_GraphicsFuncs->CheckState( NULL, NULL, state, accel );

     if (!(state->accel & accel))
          return DFB_UNSUPPORTED;

     return DFB_OK;
}

DFBResult
M2MCEngine::SetState( DirectFB::SurfaceTask  *task,
                      CardState              *state,
                      StateModificationFlags  modified,
                      DFBAccelerationMask     accel )
{
     M2MCTask *mytask = (M2MCTask *)task;

     (void)modified;

     D_DEBUG_AT( M2MC_Engine, "M2MCEngine::%s()\n", __FUNCTION__ );

     mytask->Bdriver.packetSettings.packetBufNextPtr = mytask->buffer.GetBuffer( GFX_FULL_PACKET_SIZE * 2 ); // FIXME: enough?

     bcmnexus_GraphicsFuncs->SetState( &mytask->Bdriver, &mytask->Bdevice, NULL, state, accel );

     mytask->clip = state->clip;

     mytask->buffer.PutBuffer( mytask->Bdriver.packetSettings.packetBufNextPtr );

     return DFB_OK;
}


DFBResult
M2MCEngine::DrawRectangles( DirectFB::SurfaceTask  *task,
                            const DFBRectangle     *rects,
                            unsigned int           &num_rects )
{
     M2MCTask *mytask = (M2MCTask *)task;

     D_DEBUG_AT( M2MC_Engine, "M2MCEngine::%s( %d )\n", __FUNCTION__, num_rects );

     mytask->Bdriver.packetSettings.packetBufNextPtr = mytask->buffer.GetBuffer( GFX_FILL_PACKET_SIZE * num_rects ); // FIXME: enough?

     for (unsigned int i=0; i<num_rects; i++) {
          mytask->Bdevice.count = 0;
          bcmnexus_GraphicsFuncs->DrawRectangle( &mytask->Bdriver, &mytask->Bdevice, (DFBRectangle*) &rects[i] );
     }

     mytask->buffer.PutBuffer( mytask->Bdriver.packetSettings.packetBufNextPtr );

     return DFB_OK;
}

DFBResult
M2MCEngine::DrawLines( DirectFB::SurfaceTask  *task,
                       const DFBRegion        *lines,
                       unsigned int           &num_lines )
{
     M2MCTask *mytask = (M2MCTask *)task;

     D_DEBUG_AT( M2MC_Engine, "M2MCEngine::%s( %d )\n", __FUNCTION__, num_lines );

     mytask->Bdriver.packetSettings.packetBufNextPtr = mytask->buffer.GetBuffer( GFX_FILL_PACKET_SIZE * num_lines ); // FIXME: enough?

     for (unsigned int i=0; i<num_lines; i++) {
          mytask->Bdevice.count = 0;
          bcmnexus_GraphicsFuncs->DrawLine( &mytask->Bdriver, &mytask->Bdevice, (DFBRegion*) &lines[i] );
     }

     mytask->buffer.PutBuffer( mytask->Bdriver.packetSettings.packetBufNextPtr );

     return DFB_OK;
}

DFBResult
M2MCEngine::FillRectangles( DirectFB::SurfaceTask  *task,
                            const DFBRectangle     *rects,
                            unsigned int           &num_rects )
{
     M2MCTask *mytask = (M2MCTask *)task;

     D_DEBUG_AT( M2MC_Engine, "M2MCEngine::%s( %d )\n", __FUNCTION__, num_rects );

     mytask->Bdriver.packetSettings.packetBufNextPtr = mytask->buffer.GetBuffer( GFX_FILL_PACKET_SIZE * num_rects ); // FIXME: enough?

     for (unsigned int i=0; i<num_rects; i++) {
          mytask->Bdevice.count = 0;
          bcmnexus_GraphicsFuncs->FillRectangle( &mytask->Bdriver, &mytask->Bdevice, (DFBRectangle*) &rects[i] );
     }

     mytask->buffer.PutBuffer( mytask->Bdriver.packetSettings.packetBufNextPtr );

     return DFB_OK;
}

DFBResult
M2MCEngine::FillTriangles( DirectFB::SurfaceTask  *task,
                           const DFBTriangle      *tris,
                           unsigned int           &num_tris )
{
     M2MCTask *mytask = (M2MCTask *)task;

     D_DEBUG_AT( M2MC_Engine, "M2MCEngine::%s( %d )\n", __FUNCTION__, num_tris );

     mytask->Bdriver.packetSettings.packetBufNextPtr = mytask->buffer.GetBuffer( GFX_FILL_PACKET_SIZE * num_tris ); // FIXME: enough?

     for (unsigned int i=0; i<num_tris; i++) {
          mytask->Bdevice.count = 0;
          bcmnexus_GraphicsFuncs->FillTriangle( &mytask->Bdriver, &mytask->Bdevice, (DFBTriangle*) &tris[i] );
     }

     mytask->buffer.PutBuffer( mytask->Bdriver.packetSettings.packetBufNextPtr );

     return DFB_OK;
}

DFBResult
M2MCEngine::FillTrapezoids( DirectFB::SurfaceTask  *task,
                            const DFBTrapezoid     *traps,
                            unsigned int           &num_traps )
{
     M2MCTask *mytask = (M2MCTask *)task;

     D_DEBUG_AT( M2MC_Engine, "M2MCEngine::%s( %d )\n", __FUNCTION__, num_traps );

     for (unsigned int i=0; i<num_traps; i++) {
          mytask->Bdriver.packetSettings.packetBufNextPtr = mytask->buffer.GetBuffer( GFX_FILL_PACKET_SIZE * (traps[i].y2 - traps[i].y1 + 1) ); // FIXME: enough?

          mytask->Bdevice.count = 0;
          bcmnexus_GraphicsFuncs->FillTrapezoid( &mytask->Bdriver, &mytask->Bdevice, (DFBTrapezoid*) &traps[i] );

          mytask->buffer.PutBuffer( mytask->Bdriver.packetSettings.packetBufNextPtr );
     }

     return DFB_OK;
}

DFBResult
M2MCEngine::FillSpans( DirectFB::SurfaceTask  *task,
                       int                     y,
                       const DFBSpan          *spans,
                       unsigned int           &num_spans )
{
     M2MCTask *mytask = (M2MCTask *)task;

     D_DEBUG_AT( M2MC_Engine, "M2MCEngine::%s( %d )\n", __FUNCTION__, num_spans );

     mytask->Bdriver.packetSettings.packetBufNextPtr = mytask->buffer.GetBuffer( GFX_FILL_PACKET_SIZE * num_spans ); // FIXME: enough?

     bcmnexus_GraphicsFuncs->FillSpans( &mytask->Bdriver, &mytask->Bdevice, y, spans, num_spans );

     mytask->buffer.PutBuffer( mytask->Bdriver.packetSettings.packetBufNextPtr );

     return DFB_OK;
}


DFBResult
M2MCEngine::Blit( DirectFB::SurfaceTask  *task,
                  const DFBRectangle     *rects,
                  const DFBPoint         *points,
                  u32                    &num )
{
     M2MCTask *mytask = (M2MCTask *)task;

     D_DEBUG_AT( M2MC_Engine, "M2MCEngine::%s( %d )\n", __FUNCTION__, num );

     mytask->Bdriver.packetSettings.packetBufNextPtr = mytask->buffer.GetBuffer( (GFX_TWO_PASSES_BLIT_PACKET_SIZE) * num ); // FIXME: enough?

     for (unsigned int i=0; i<num; i++) {
          mytask->Bdevice.count = 0;

          if (dfb_clip_blit_precheck( &mytask->clip, rects[i].w, rects[i].h, points[i].x, points[i].y )) {
               bcmnexus_GraphicsFuncs->Blit( &mytask->Bdriver, &mytask->Bdevice, (DFBRectangle*) &rects[i], points[i].x, points[i].y );
          }
     }

     mytask->buffer.PutBuffer( mytask->Bdriver.packetSettings.packetBufNextPtr );

     return DFB_OK;
}

DFBResult
M2MCEngine::Blit2( DirectFB::SurfaceTask  *task,
                   const DFBRectangle     *rects,
                   const DFBPoint         *points1,
                   const DFBPoint         *points2,
                   u32                    &num )
{
     M2MCTask *mytask = (M2MCTask *)task;

     D_DEBUG_AT( M2MC_Engine, "M2MCEngine::%s( %d )\n", __FUNCTION__, num );

     mytask->Bdriver.packetSettings.packetBufNextPtr = mytask->buffer.GetBuffer( (GFX_TWO_PASSES_BLIT_PACKET_SIZE) * num ); // FIXME: enough?

     for (unsigned int i=0; i<num; i++) {
          mytask->Bdevice.count = 0;
          bcmnexus_GraphicsFuncs->Blit2( &mytask->Bdriver, &mytask->Bdevice,
                                         (DFBRectangle*) &rects[i], points1[i].x, points1[i].y, points2[i].x, points2[i].y );
     }

     mytask->buffer.PutBuffer( mytask->Bdriver.packetSettings.packetBufNextPtr );

     return DFB_OK;
}

DFBResult
M2MCEngine::StretchBlit( DirectFB::SurfaceTask  *task,
                         const DFBRectangle     *srects,
                         const DFBRectangle     *drects,
                         u32                    &num )
{
     DFBResult  ret                = DFB_OK;
     M2MCTask  *mytask             = (M2MCTask *)task;
     bool       twoPassesDownscale = false;

     D_DEBUG_AT( M2MC_Engine, "M2MCEngine::%s( %d )\n", __FUNCTION__, num );

     mytask->Bdriver.packetSettings.packetBufNextPtr = mytask->buffer.GetBuffer( (GFX_TWO_PASSES_BLIT_PACKET_SIZE) * num ); // FIXME: enough?

     for (unsigned int i=0; i<num; i++) {
          mytask->Bdevice.count = 0;

          if (drects[i].w && drects[i].h) {
               /* Check width/height scale limit */
               if ((srects[i].w > drects[i].w * BCMNEXUS_GFX_SCALE_DOWN_MAX_X) || (srects[i].h > drects[i].h * BCMNEXUS_GFX_SCALE_DOWN_MAX_Y))
               {
                   /* Ensure that with 2 passes we don't exceed the maximum scale down.  If we do, then we have to fall
                      back to using the generic software implementation instead... */
                   if ((srects[i].w <= drects[i].w * BCMNEXUS_GFX_SCALE_DOWN_MAX_X * BCMNEXUS_GFX_SCALE_DOWN_MAX_X) &&
                       (srects[i].h <= drects[i].h * BCMNEXUS_GFX_SCALE_DOWN_MAX_Y * BCMNEXUS_GFX_SCALE_DOWN_MAX_Y))
                   {
                       /* If we exceed maximum vertical or horizontal downscale, need to perform two passes using
                          an intermediate buffer */
                       twoPassesDownscale = true;
                   }
                   else
                       continue;   /* FIXME: let s/w do the blit */
               }

               if (twoPassesDownscale)
                    mytask->Bdevice.mode = BCMNEXUS_GFX_IDLE;

               bcmnexus_GraphicsFuncs->StretchBlit( &mytask->Bdriver, &mytask->Bdevice, (DFBRectangle*) &srects[i], (DFBRectangle*) &drects[i] );

               if (twoPassesDownscale) {
                    num = i+1;
                    ret = DFB_LIMITEXCEEDED;

                    break;
               }
          }
     }

     mytask->buffer.PutBuffer( mytask->Bdriver.packetSettings.packetBufNextPtr );

     return DFB_OK;
}

DFBResult
M2MCEngine::TileBlit( DirectFB::SurfaceTask  *task,
                      const DFBRectangle     *rects,
                      const DFBPoint         *points1,
                      const DFBPoint         *points2,
                      u32                    &num )
{
     M2MCTask *mytask = (M2MCTask *)task;

     D_DEBUG_AT( M2MC_Engine, "M2MCEngine::%s( %d )\n", __FUNCTION__, num );

     (void)rects;
     (void)points1;
     (void)points2;

     mytask->Bdriver.packetSettings.packetBufNextPtr = mytask->buffer.GetBuffer( (GFX_TWO_PASSES_BLIT_PACKET_SIZE) * num ); // FIXME: enough?

     D_UNIMPLEMENTED();

     for (unsigned int i=0; i<num; i++) {
          mytask->Bdevice.count = 0;
     }

     mytask->buffer.PutBuffer( mytask->Bdriver.packetSettings.packetBufNextPtr );

     return DFB_OK;
}

DFBResult
M2MCEngine::TextureTriangles( DirectFB::SurfaceTask  *task,
                              const DFBVertex        *vertices,
                              int                    &num,
                              DFBTriangleFormation    formation )
{
     M2MCTask *mytask = (M2MCTask *)task;

     D_DEBUG_AT( M2MC_Engine, "M2MCEngine::%s( %d )\n", __FUNCTION__, num );

     mytask->Bdriver.packetSettings.packetBufNextPtr = mytask->buffer.GetBuffer( (GFX_TWO_PASSES_BLIT_PACKET_SIZE) * num ); // FIXME: enough?

     bcmnexus_GraphicsFuncs->TextureTriangles( &mytask->Bdriver, &mytask->Bdevice, (DFBVertex*) vertices, num, formation );

     mytask->buffer.PutBuffer( mytask->Bdriver.packetSettings.packetBufNextPtr );

     return DFB_OK;
}



extern "C" {
     void
     register_m2mc()
     {
          int m2mc_cores = direct_config_get_int_value_with_default( "m2mc-cores", 1 );

          D_DEBUG_AT(M2MC_Engine,"%s: Using %d M2MC cores\n",__FUNCTION__,m2mc_cores);

          if (m2mc_cores && dfb_config->task_manager)
               DirectFB::Renderer::RegisterEngine( new M2MCEngine( m2mc_cores ) );
     }
}

}

#endif /* BCM_DFB_USE_TASK_MANAGER */
