#ifdef QT_WEBKITWIDGETS_LIB
#include "browser.h"
#endif

#ifdef QT_WEBENGINEWIDGETS_LIB
#include "browser2.h"
#endif

#include <QApplication>
#include <QNetworkProxy>
#include <QtGui>
#include <QObject>

#include <QtWidgets>

#ifdef QT_OPENGL_LIB
#include <QtOpenGL>
#endif

#ifdef CROSS_COMPILED_FOR_DEVICE
# include <signal.h>
# include <sys/types.h>
# include <unistd.h>
# define STACK_SIZE 192
#else
# define STACK_SIZE 1024
#endif

#if defined(ENABLE_GOOGLE_BREAKPAD)
#include "client/linux/handler/exception_handler.h"
using namespace google_breakpad;
bool breakpadCallback(const MinidumpDescriptor& descriptor,void* context, bool succeeded)
{
    Q_UNUSED(descriptor);
    Q_UNUSED(context);
    fprintf(stderr, "Caught exception, see minidump at '%s'\n", descriptor.path());
    return succeeded;
}
void setupBreakpad()
{
    new ExceptionHandler(MinidumpDescriptor("/opt/minidumps"), NULL, breakpadCallback, NULL, true, -1);
}
#endif

#ifdef ENABLE_CPU_PROFILER
#include <gperftools/profiler.h>
#endif

#ifdef ENABLE_HEAP_PROFILER
#include <gperftools/heap-profiler.h>
#endif

#include <gst/gst.h>
#include <cstdio>
#include <sys/time.h>     // getrlimit(2)
#include <sys/resource.h> // getrlimit(2)
#include "main.h"

unsigned sDisplayWidth  = 1280;
unsigned sDisplayHeight = 720;

#ifdef CROSS_COMPILED_FOR_DEVICE
static void sighandler(int signo)
{
    QCoreApplication::quit();

    // work around a crash in QWSTtyKeyboardHandler
    signal(signo, SIG_DFL);
    // kill(getpid(), signo);
}

#if defined (QT_WEBKITWIDGETS_LIB)
static void usr1handler(int)
{
    Browser::emitClearWebCoreCaches();
    Browser::emitDumpWebCoreStatistics();
}

static void usr2handler(int)
{
    TestApp::getInstance()->emitQuitApp();
}
#endif

#endif

static void setStackSizeInKb(long sizeInKb)
{
    printf("Setting stack size to %ld\n", sizeInKb);
    const rlim_t kStackSize = sizeInKb * 1024L;
    struct rlimit rl;
    int result;

    result = getrlimit(RLIMIT_STACK, &rl);
    if (result == 0)
    {
        rl.rlim_cur = kStackSize;
        result = setrlimit(RLIMIT_STACK, &rl);
        if (result != 0)
        {
            printf("setrlimit returned result = %d\n", result);
        }
    }
}

static void startProfiler() {
#ifdef ENABLE_HEAP_PROFILER
    static bool heap_profile_tried = false;
    static const char* heap_profile_prefix = getenv("HEAP_PROFILE_PREFIX");
    if (!heap_profile_tried && heap_profile_prefix && !IsHeapProfilerRunning()) {
        HeapProfilerStart(heap_profile_prefix);
    }
    heap_profile_tried = true;
#endif

#ifdef ENABLE_CPU_PROFILER
    static bool cpu_profile_tried = false;
    static const char* cpu_profile_prefix = getenv("CPU_PROFILE_PREFIX");
    if (!cpu_profile_tried && cpu_profile_prefix) {
        ProfilerStart(cpu_profile_prefix);
    }
    cpu_profile_tried = true;
#endif
}

static void dumpProfile(const char* reason) {
#ifdef ENABLE_HEAP_PROFILER
    if (IsHeapProfilerRunning()) {
        HeapProfilerDump(reason);
    }
#endif
}

#if defined (QT_WEBKITWIDGETS_LIB)
static TestApp* gInstance = NULL;

TestApp* TestApp::getInstance() {
    return gInstance;
}

TestApp::TestApp (
    QGraphicsScene* scene,
    const QList<QString>& urls,
    bool is_video,
    bool enable_gui,
    bool enable_tiled_backing_store,
    const QString& user_agent,
    bool ignore_ssl_errors)
    : m_scene(scene)
    , m_urls(urls)
    , m_isVideo(is_video)
    , m_enableGUI(enable_gui)
    , m_enableTBS(enable_tiled_backing_store)
    , m_userAgent(user_agent)
    , m_ignoreSSLErrors(ignore_ssl_errors) {
    gInstance = this;
    createNewBrowser();
}

TestApp::~TestApp() {
    gInstance = NULL;
}

void TestApp::emitQuitApp() {
    QMetaObject::invokeMethod(TestApp::getInstance(), "quitApp", Qt::QueuedConnection);
}

void TestApp::quitApp() {
    if (qGuiApp->thread() != this->thread()) {
        printf("***Error: Cannot handle 'quitApp' on non main thread\n");
        return;
    }
    m_browser.reset();
    QWebSettings::clearMemoryCaches();
    QWebSettings::dumpWebCoreStatistics();
    fprintf(stderr, "Exit in 10 seconds\n");
    QTimer::singleShot(10 * 1000, qGuiApp, SLOT(quit()));
}

void TestApp::createNewBrowser() {
    static int loadNum = 0;
    if (loadNum >= 3) { // m_urls.size()
        loadNum = 0;
        QWebSettings::clearMemoryCaches();
        QWebSettings::dumpWebCoreStatistics();
        dumpProfile(__FUNCTION__);
    } else {
        ++loadNum;
    }

    if (loadNum >= 3) { // m_urls.size()
        startProfiler();
    }

    if (m_urls.empty()) {
        emitQuitApp();
        return;
    }

    QString url = m_urls.takeFirst();
    m_browser.reset(new Browser(*m_scene, url, m_isVideo, m_enableGUI,
                                m_enableTBS, m_userAgent, m_ignoreSSLErrors));
    m_browser->setScreenshotPath(m_screenshotPath);
    QObject::connect(m_browser.data(),SIGNAL(documentLoaded()),SLOT(loadNextUrl()));
    m_urls.append(url);
}

void TestApp::loadNextUrl() {
    if (m_urls.size() > 1) {
        m_browser.reset();
        QTimer::singleShot(1000, this, SLOT(createNewBrowser()));
    }
}

void TestApp::setScreenshotPath(QString path) {
    m_screenshotPath = path;
}
#endif

int main(int argc, char **argv)
{
    setStackSizeInKb(STACK_SIZE); // workaround for XONE-10286
#ifdef CROSS_COMPILED_FOR_DEVICE
    signal(SIGINT, sighandler);
#if defined (QT_WEBKITWIDGETS_LIB)
    signal(SIGUSR1, usr1handler);
    signal(SIGUSR2, usr2handler);
#endif
#endif
    gst_init(0, 0);

    qRegisterMetaType<QVariantList>("QVariantList");

    bool enable_gui = true;
    QString scrPath;

    QApplication app(argc, argv);
    QRectF sceneRect(0,0,sDisplayWidth, sDisplayHeight);
    QGraphicsScene scene;

    scene.setSceneRect(sceneRect);
    QGraphicsView view(&scene);
    view.setBackgroundBrush(QBrush(Qt::white));

#ifdef CROSS_COMPILED_FOR_DEVICE
    enable_gui = false; // because of a keyboard focus issue

    //intelce requires fullscreen opengl exclusively
    //note that when we don't disable both depth and stencil and we run with ocap video, the screen gets garbled
    //TODO - determine what the issue is here
#ifdef QT_OPENGL_LIB
    view.setViewport(new QGLWidget(QGLFormat(QGL::NoDepthBuffer | //dont need depth buffer - scene renders back to front
        QGL::NoStencilBuffer | //not needed either - could artifacts result?
        QGL::NoSampleBuffers)));//antialiasing really slow down the scene
    view.setViewportUpdateMode(QGraphicsView::FullViewportUpdate);//opengl only support full screen updates
#endif
    //hide scrollbars and window frame and run exclusively in fullscreen
    //because out custom gfxdriver cannot draw anything except the QGLWidget viewport
    view.setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    view.setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    view.setFrameStyle(0);
    view.resize(sDisplayWidth, sDisplayHeight);
    if (qGuiApp->platformName() == QStringLiteral("eglfs") ||
        qGuiApp->platformName() == QStringLiteral("directfb")) {
        view.showFullScreen();
    } else {
        view.show();
    }
#else
    view.setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    view.setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    view.setWindowTitle(QString::fromUtf8("Qt Browser"));
    view.setRenderHint(QPainter::Antialiasing);
    view.setGeometry(0, 0, sDisplayWidth, sDisplayHeight);
    view.setFrameStyle(0);
    view.show();
#endif

    QUrl proxy_url;
    
    // Handle environment variables.
    {
        const char* proxy_env = getenv("http_proxy");
        if (proxy_env && *proxy_env)
            proxy_url = proxy_env;
    }

    // Parse command line.
    QList<QString> urlList;
    QString url("http://google.com"), user_agent;

    bool is_video = false;

    //FIXME tiled backing store (original) causes some weirdness when openening another page, 
    //the previous is still shown below (transparency)
    bool enable_tiled_backing_store = false;

    bool ignore_ssl_errors = false;
    bool repaint_test = false;

    for (int i = 0; i < argc; ++i)
    {
        QString cur_arg(argv[i]);

        if (cur_arg == "-path" && i < argc-1) // -path <url>
        {
            url = argv[++i];
            urlList.append(url);
        }
        else if (cur_arg == "-videopath" && i < argc-1) // -videopath <url>
        {
            url = argv[++i];
            is_video = true;
        }
        else if (cur_arg == "-tbs" && i < argc-1) // -tbs <0|1>
        {
            const char* arg = argv[++i];
            enable_tiled_backing_store = !(!strcmp(arg, "false") || !strcmp(arg, "0"));
        }
        else if (cur_arg == "-gui" && i < argc-1) // -gui <0|1>
        {
            const char* arg = argv[++i];
            enable_gui = !(!strcmp(arg, "false") || !strcmp(arg, "0"));
            printf("GUI %s\n", enable_gui ? "enabled" : "disabled");
        }
        else if (cur_arg == "-ua" && i < argc-1) // -ua <user_agent_string>
        {
            user_agent = argv[++i];
            printf("User-Agent set to '%s'\n", user_agent.toUtf8().constData());
        }
        else if (cur_arg == "-proxy" && i < argc-1) // -proxy <proxy_url>
        {
            proxy_url = argv[++i];
        }
        else if (cur_arg == "-k") {
            ignore_ssl_errors = true;
        }
        else if (cur_arg == "-scr_path"){
            scrPath = argv[++i];
            printf("Screenshot path is %s\n", scrPath.toLatin1().constData());
        }
        else if (cur_arg == "-repaint_test") {
            repaint_test = true;
            printf("Force repaint 30 times per second\n");
        }
    }

    if (proxy_url.port() > 0)
    {
        struct NetworkProxyFactory : public  QNetworkProxyFactory {
            QNetworkProxy m_proxy;
            NetworkProxyFactory(QNetworkProxy proxy) : m_proxy(proxy) {
            }
            virtual QList<QNetworkProxy>  queryProxy(const QNetworkProxyQuery & query) {
                if (query.url().toString().isEmpty() ||
                    query.url().toString().contains("192.168.201") ||
                    query.url().toString().contains("d2lkq7nlcrdi7q.cloudfront.net") ||
                    query.url().toString().contains("s3.lvlt.dash.us.aiv-cdn.net")) {
                    // fprintf(stderr, "Return No Proxy for url: %s\n", qPrintable(query.url().toString()));
                    return QList<QNetworkProxy>() << QNetworkProxy(QNetworkProxy::NoProxy);
                }
                // fprintf(stderr, "Return proxy for url: %s\n", qPrintable(query.url().toString()));
                return QList<QNetworkProxy>() << m_proxy;
            }
        };
        QNetworkProxy proxy;
        proxy.setType(QNetworkProxy::HttpProxy);
        proxy.setHostName(proxy_url.host());
        proxy.setPort(proxy_url.port());
        QNetworkProxyFactory* appNetworkProxyFactory = new NetworkProxyFactory(proxy);
        QNetworkProxyFactory::setApplicationProxyFactory(appNetworkProxyFactory);
        printf("Proxy set to %s:%d\n", proxy_url.host().toUtf8().constData(), proxy_url.port());
    }

#if defined (QT_WEBENGINEWIDGETS_LIB)
    Browser2 browser(scene, url, is_video, enable_gui, enable_tiled_backing_store, user_agent, ignore_ssl_errors);
#elif defined (QT_WEBKITWIDGETS_LIB)
    // same capacities as in receiver
    unsigned int cacheTotalCapacity = 64 * 1024 * 1024;
    unsigned int cacheMinDeadCapacity = cacheTotalCapacity / 4;
    unsigned int cacheMaxDeadCapacity = cacheTotalCapacity / 2;
    QWebSettings::setObjectCacheCapacities(cacheMinDeadCapacity, cacheMaxDeadCapacity, cacheTotalCapacity);
    QScopedPointer<TestApp> testapp;
    testapp.reset(new TestApp(&scene, urlList, is_video,
                              enable_gui, enable_tiled_backing_store,
                              user_agent, ignore_ssl_errors));
    testapp->setScreenshotPath(scrPath);
#else
#error "No WebKit nor WebEngine backend is available"
#endif

#if defined(ENABLE_GOOGLE_BREAKPAD)
    // QWebEngine overrides breakpads signal handlers so setup it here
    setupBreakpad();
#endif

    if (repaint_test) {
        QTimer* repaint_timer = new QTimer(&app);
        QObject::connect(repaint_timer, SIGNAL(timeout()), &scene, SLOT(update()));
        repaint_timer->start(1000/30);
    }

    int rv = app.exec();

#ifdef ENABLE_CPU_PROFILER
    ProfilerStop();
#endif

#if defined (QT_WEBKITWIDGETS_LIB)
    testapp.reset();
    QWebSettings::clearMemoryCaches();
    QWebSettings::dumpWebCoreStatistics();
#endif

    gst_deinit();
    return rv;
}
