/******************************************************************************
**
** Copyright (C) 2015 The Qt Company Ltd.
** Contact: http://www.qt.io/licensing/
**
** This file is part of the Qt Virtual Keyboard module.
**
** $QT_BEGIN_LICENSE:COMM$
**
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and The Qt Company. For licensing terms
** and conditions see http://www.qt.io/terms-conditions. For further
** information use the contact form at http://www.qt.io/contact-us.
**
** $QT_END_LICENSE$
**
******************************************************************************/

import QtQuick.Layouts 1.0

import QtQuick 2.0
import QtQuick.VirtualKeyboard 2.1
import QtQuick.VirtualKeyboard.Styles 2.1

// Definitions
// ===========
//
// highlight - the glass box thing that follows the cursor
// sketchout - the black bit which darkens the background under it

KeyboardStyle {
    id: currentStyle

    readonly property string fontFamily: "Sky Text"
    readonly property real keyBackgroundMargin: 2
    readonly property real keyContentMargin: 0
    readonly property real keyIconScale: scaleHint * 0.3
    readonly property string resourcePrefix: ""
    readonly property int fontPixelSize: Math.round(34 * scaleHint)
    readonly property color fontColor: "#ebebeb"
    readonly property color keybackColor: "#d9111111"
    readonly property color specialKeybackColor: "#d9191919"
    readonly property color sketchoutColor: "#66000000"
    readonly property real hoverScaleFactorImage: 1.1

    readonly property int keyHeight: 72 // Warning - you will have to change these in the layouts too.
    readonly property int dblKeyWidth: 148
    readonly property int dblhKeyWidth: 186
    readonly property int interKeyGap: 4

    // Co-ordinates for the rectangles which make up the background
    readonly property int firstRowY: 278
    readonly property int secondRowY: 354
    readonly property int thirdRowY: 430
    readonly property int fourthRowY: 518

    // Main panel (i.e. the left hand side)
    readonly property int mainPanelLeft: 388
    readonly property int mainPanelWidth: 751
    readonly property int mainPanelWidthSearch: 452

    // Number panel (i.e. the right hand side)
    readonly property int numPanelLeft: 1156
    readonly property int numPanelWidth: 375
    readonly property int numPanelWidthSearch: 375

    // Values in symbol mode
    readonly property int symPanelWidth: 678
    readonly property int symNumPanelWidth: 350

    // Key x positions for bottom row.
    readonly property int shiftKeyLeft: mainPanelLeft
    readonly property int symKeyLeft: shiftKeyLeft+dblKeyWidth+interKeyGap
    readonly property int spaceKeyLeft: symKeyLeft+dblKeyWidth+interKeyGap
    readonly property int delKeyLeft: numPanelLeft
    readonly property int subKeyLeft: numPanelLeft+dblhKeyWidth+interKeyGap

    // The spacebar is in the remainder of the width
    readonly property int spaceKeyWidth: (mainPanelLeft+mainPanelWidth)-spaceKeyLeft

    keyboardDesignWidth: 1920
    keyboardDesignHeight: 643
    keyboardRelativeLeftMargin: 389 / keyboardDesignWidth
    keyboardRelativeRightMargin: 389 / keyboardDesignWidth
    keyboardRelativeTopMargin: (276) / keyboardDesignHeight
    keyboardRelativeBottomMargin: 51 / keyboardDesignHeight

    keyboardBackground: Item {
        anchors.fill: parent
        // This is the background image.
        Item {
            anchors.fill: parent
            // This is the top band, the shiny "rim"
            Image {
                x:0
                y:643-450
                cache: false
                source: "backga1.png"
            }
            // This is the alpha only mask which sits below it.
            // We downscaled it cause it was a waste of gpu ram.
            Image {
                x: 0
                y: (643-450)+67
                width: 1920
                height: 450-67
                cache: false
                source: "backga2.png"
            }
            // This is the original, a combination of the two images above.
            //Image {
            //    source: "backga.png"
            //    x:0
            //    y:643-450
            //    cache:false
            //}

            // This is a wierd one pixel wide, full height band which is swept across the screen.
            Image {
                anchors.fill: parent
                fillMode: Image.Stretch
                cache: false
                source: "backgb.png"
            }
            z:-1
        }
        // We have to draw the key backs manually, unfortunately, to get them in the correct z order for the design.
        // Main panel.  The three rows to the left of the screen.  The last row varies in width depending on layout and symbol mode.
        Rectangle {
            color: keybackColor
            x: mainPanelLeft
            y: firstRowY
            width: mainPanelWidth
            height: keyHeight
        }
        Rectangle {
            color: keybackColor
            x: mainPanelLeft
            y: secondRowY
            width: mainPanelWidth
            height: keyHeight
        }
        Rectangle {
            color: keybackColor
            x: mainPanelLeft
            y: thirdRowY
            width: symbolMode ? symPanelWidth : (((InputContext.inputMethodHints & Qt.ImhPreferUppercase) && ( InputContext.inputMethodHints & Qt.ImhNoAutoUppercase )) ? mainPanelWidthSearch:mainPanelWidth);
            height: keyHeight
        }
        // Number panel.  The three rows to the right hand side.
        Rectangle {
            color: keybackColor
            x: numPanelLeft
            y: firstRowY
            width: numPanelWidth
            height: keyHeight
        }
        Rectangle {
            color: keybackColor
            x: numPanelLeft
            y: secondRowY
            width: numPanelWidth
            height: keyHeight
        }
        Rectangle {
            color: keybackColor
            x: numPanelLeft
            y: thirdRowY
            width: symbolMode ? symNumPanelWidth : (((InputContext.inputMethodHints & Qt.ImhPreferUppercase) && ( InputContext.inputMethodHints & Qt.ImhNoAutoUppercase )) ? symNumPanelWidth:numPanelWidthSearch);
            height: keyHeight
        }
        // Control key row.  The row at the bottom of the screen.
        // shift key back.
        Rectangle {
            color: keybackColor
            x: shiftKeyLeft
            y: fourthRowY
            width: dblKeyWidth
            height: keyHeight
        }
        // symbol key back.
        Rectangle {
            color: keybackColor
            x: symKeyLeft
            y: fourthRowY
            width: dblKeyWidth
            height: keyHeight
        }
        // space key back.
        Rectangle {
            color: keybackColor
            x: spaceKeyLeft
            y: fourthRowY
            width: spaceKeyWidth
            height: keyHeight
        }
        // del key back.
        Rectangle {
            color: keybackColor
            x: delKeyLeft
            y: fourthRowY
            width: dblhKeyWidth
            height: keyHeight
        }
        // submit key back.
        Rectangle {
            color: specialKeybackColor
            x: subKeyLeft
            y: fourthRowY
            width: dblhKeyWidth-1
            height: keyHeight
        }
    }

    keyPanel: KeyPanel {
        Text {
            id: keyText
            text: control.displayText
            color: fontColor
            horizontalAlignment: Text.AlignHCenter
            verticalAlignment: Text.AlignVCenter
            anchors.fill: parent
            anchors.leftMargin: keyContentMargin
            anchors.topMargin:  keyContentMargin
            anchors.rightMargin: keyContentMargin
            anchors.bottomMargin: keyContentMargin
            font {
                family: fontFamily
                styleName: "Regular"
                pixelSize: fontPixelSize
                capitalization: (control.uppercased && (control.displayText.length == 1)) ? Font.AllUppercase : Font.MixedCase
            }
            states: [
                State {
                    name: "zoom"
                    when: control == keyboardInputArea.initialKey
                    PropertyChanges {
                        target: keyText
                        font.styleName: "Medium"
                    }
                }
            ]
        }
    }

    backspaceKeyPanel: KeyPanel {
        Text {
            id: backspaceKeyText
            visible: text.length !== 0
            anchors.fill: parent
            anchors.topMargin: keyBackgroundMargin
            anchors.bottomMargin: keyBackgroundMargin
            anchors.leftMargin: 0
            anchors.rightMargin: 2
            text: control.displayText
            clip: true
            fontSizeMode: Text.HorizontalFit
            horizontalAlignment: Text.AlignHCenter
            verticalAlignment: Text.AlignVCenter
            color: fontColor
            font {
                family: fontFamily
                styleName: "Regular"
                pixelSize: fontPixelSize
            }
            states: [
                State {
                    name: "zoom"
                    when: control == keyboardInputArea.initialKey
                    PropertyChanges {
                        target: backspaceKeyText
                        font.styleName: "Medium"
                    }
                }
            ]
        }
    }

    enterKeyPanel: KeyPanel {
        Text {
            anchors.fill: parent
            anchors.topMargin: keyBackgroundMargin
            anchors.bottomMargin: keyBackgroundMargin
            anchors.leftMargin: 2
            anchors.rightMargin: 0
            id: enterKeyText
            visible: text.length !== 0
            text: control.displayText
            fontSizeMode: Text.HorizontalFit
            horizontalAlignment: Text.AlignHCenter
            verticalAlignment: Text.AlignVCenter
            color: fontColor
            font {
                family: fontFamily
                styleName: "Regular"
                pixelSize: fontPixelSize
            }
            states: [
                State {
                    name: "zoom"
                    when: control == keyboardInputArea.initialKey
                    PropertyChanges {
                        target: enterKeyText
                        font.styleName: "Medium"
                    }
                }
            ]
        }
    }

    shiftKeyPanel: KeyPanel {
            Image {
                id: shiftKeyIcon
                anchors.centerIn: parent
                anchors.topMargin: keyBackgroundMargin
                anchors.bottomMargin: keyBackgroundMargin
                anchors.leftMargin: 0
                anchors.rightMargin: 2
                source: resourcePrefix + "capsu.png"
            }
            states: [
                // fill in the arrow when shift is selected.
                State {
                    name: "shift"
                    when: InputContext.shift
                    PropertyChanges {
                        target: shiftKeyIcon
                        source: resourcePrefix + "capsl.png"
                    }
                },
                State {
                    // Make it a little bigger when hovered.
                    name: "zoom"
                    when: control == keyboardInputArea.initialKey
                    PropertyChanges {
                        target: shiftKeyIcon
                        width: sourceSize.width * hoverScaleFactorImage
                        height: sourceSize.height * hoverScaleFactorImage
                    }
                }
            ]
    }

    spaceKeyPanel: KeyPanel {
        Text {
            id: enterKeyText
            anchors.fill: parent
            anchors.topMargin: keyBackgroundMargin
            anchors.bottomMargin: keyBackgroundMargin
            anchors.leftMargin: 2
            anchors.rightMargin: 0
            visible: text.length !== 0
            text: control.displayText
            clip: true
            horizontalAlignment: Text.AlignHCenter
            verticalAlignment: Text.AlignVCenter
            color: fontColor
            font {
                family: fontFamily
                styleName: "Regular"
                pixelSize: fontPixelSize
            }
            states: [
                State {
                    name: "zoom"
                    when: control == keyboardInputArea.initialKey
                    PropertyChanges {
                        target: enterKeyText
                        font.styleName: "Medium"
                    }
                }
            ]
        }
    }

    symbolKeyPanel: KeyPanel {
        Text {
            id: symbolKeyText
            anchors.fill: parent
            anchors.topMargin: keyBackgroundMargin
            anchors.bottomMargin: keyBackgroundMargin
            anchors.leftMargin: 2
            anchors.rightMargin: 2
            text: control.displayText
            color: "white"
            horizontalAlignment: Text.AlignHCenter
            verticalAlignment: Text.AlignVCenter
            font {
                family: fontFamily
                styleName: "Regular"
                pixelSize: fontPixelSize
            }
            states: [
                State {
                    name: "zoom"
                    when: control == keyboardInputArea.initialKey
                    PropertyChanges {
                        target: symbolKeyText
                        font.styleName: "Medium"
                    }
                }
            ]
        }
    }
    // Navigation highlight is the glass rectangle with follows the cursor.
    // Navigation background is the dark sketchout behind the key text but in front of the
    // key background.  Note there is no way of accomplishing this with Qt out of the box -
    // I (jm) added this functionality.
    navigationBackground: Item {
        anchors.fill: parent

        Rectangle {
            anchors.topMargin: 2
            anchors.bottomMargin: 2
            anchors.fill: parent
            color: sketchoutColor
        }
    }

    navigationHighlight: Item {
        anchors.fill: parent
        anchors.leftMargin: -10
        anchors.rightMargin: -13
        anchors.topMargin: -30
        anchors.bottomMargin: -30

        // Glass square top to highlight is a 9 point image
        BorderImage {
            cache: false
            anchors.fill: parent
            anchors.topMargin: 19
            anchors.bottomMargin:15
            border.left: 30
            border.right: 30
            border.top: 20
            border.bottom: 20
            source: "glass.png"
        }

        // Rectangle containing glints. (The "lighty" bits at the top and bottom of the glass).
        Rectangle {
            anchors.fill: parent
            anchors.rightMargin: 4
            clip: true
            color: "transparent"
            Image {
                cache: false
                anchors.top: parent.top
                anchors.horizontalCenter: parent.horizontalCenter
                fillMode: Image.Pad
                clip: true
                source: "glint.png"
            }

            Image {
                cache: false
                anchors.bottom: parent.bottom
                anchors.horizontalCenter: parent.horizontalCenter
                fillMode: Image.Pad
                clip: true
                source: "glint.png"
            }
        }
    }
}
