/*
 * Copyright (C) 2017 Broadcom
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 */

#include <linux/device.h>
#include <linux/errno.h>
#include <linux/extcon.h>
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/miscdevice.h>
#include <linux/mm.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/slab.h>
#include <linux/uaccess.h>
#include <linux/brcmstb/hdmi_hpd_switch.h>

static DEFINE_MUTEX(hpd_switch_lock);

static const unsigned int brcmstb_extcon_hdmi_cable[] = {
	EXTCON_DISP_HDMI,
	EXTCON_NONE,
};
static struct extcon_dev *hdmi_hpd_switch;

static long brcmstb_extcon_hdmi_hpd_ioctl(struct file *file,
	unsigned int cmd, unsigned long arg)
{
	enum hdmi_state switch_state;

	switch (cmd) {
	case HDMI_HPD_IOCTL_SET_SWITCH:
		if (copy_from_user(&switch_state, (void __user *)arg,
			sizeof(switch_state)))
			return -EFAULT;
		if (switch_state != HDMI_UNPLUGGED &&
		    switch_state != HDMI_CONNECTED) {
			pr_err("%s: invalid switch state (%d)\n", __func__,
				switch_state);
			return -EINVAL;
		}
		mutex_lock(&hpd_switch_lock);
		extcon_set_state(hdmi_hpd_switch, EXTCON_DISP_HDMI,
			(switch_state == HDMI_CONNECTED) ? true : false);
		mutex_unlock(&hpd_switch_lock);
		break;
	default:
		pr_err("%s: invalid command\n", __func__);
		return -ENOIOCTLCMD;
	};

	return 0;
}

#ifdef CONFIG_COMPAT
static long brcmstb_extcon_hdmi_hpd_compat_ioctl(struct file *file,
		unsigned int cmd, unsigned long arg)
{
	return brcmstb_extcon_hdmi_hpd_ioctl(file, cmd,
				(unsigned long)compat_ptr(arg));
}
#endif

static const struct file_operations brcmstb_extcon_hdmi_hpd_fops = {
	.owner		= THIS_MODULE,
	.unlocked_ioctl = brcmstb_extcon_hdmi_hpd_ioctl,
#ifdef CONFIG_COMPAT
	.compat_ioctl	= brcmstb_extcon_hdmi_hpd_compat_ioctl,
#endif
};

static struct miscdevice brcmstb_extcon_hdmi_hpd_miscdev = {
	.minor	= MISC_DYNAMIC_MINOR,
	.name	= "hdmi_hpd",
	.fops	= &brcmstb_extcon_hdmi_hpd_fops,
};

static int brcmstb_extcon_hdmi_hpd_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	int ret;

	ret = misc_register(&brcmstb_extcon_hdmi_hpd_miscdev);
	if (ret < 0) {
		dev_err(dev, "failed to register misc device\n");
		return ret;
	}

	hdmi_hpd_switch = devm_extcon_dev_allocate(dev,
		brcmstb_extcon_hdmi_cable);
	if (IS_ERR(hdmi_hpd_switch)) {
		dev_err(dev, "failed to allocate extcon device\n");
		return -ENOMEM;
	}

	ret = devm_extcon_dev_register(dev, hdmi_hpd_switch);
	if (ret < 0) {
		dev_err(dev, "failed to register extcon device\n");
		return ret;
	}

	return 0;
}

static int brcmstb_extcon_hdmi_hpd_remove(struct platform_device *pdev)
{
	misc_deregister(&brcmstb_extcon_hdmi_hpd_miscdev);
	return 0;
}

static struct platform_driver brcmstb_extcon_hdmi_hpd_driver = {
	.probe		= brcmstb_extcon_hdmi_hpd_probe,
	.remove		= brcmstb_extcon_hdmi_hpd_remove,
	.driver		= {
		.name	= "extcon-brcmstb-hdmi-hpd",
	},
};

static struct platform_device brcmstb_extcon_hdmi_hpd_dev = {
	.name = "extcon-brcmstb-hdmi-hpd",
	.id = -1,
};

static int __init brcmstb_extcon_hdmi_hpd_init(void)
{
	int ret;

	ret = platform_driver_register(&brcmstb_extcon_hdmi_hpd_driver);
	if (ret) {
		pr_err("failed to register brcmstb_extcon_hdmi_hpd_driver\n");
		return ret;
	}

	ret = platform_device_register(&brcmstb_extcon_hdmi_hpd_dev);
	if (ret) {
		pr_err("failed to register brcmstb_extcon_hdmi_hpd_dev\n");
		platform_driver_unregister(&brcmstb_extcon_hdmi_hpd_driver);
		return ret;
	}

	pr_info("successfully registered brcmstb_extcon_hdmi_hpd_driver\n");
	return 0;
}

static void __exit brcmstb_extcon_hdmi_hpd_exit(void)
{
	platform_device_unregister(&brcmstb_extcon_hdmi_hpd_dev);
	platform_driver_unregister(&brcmstb_extcon_hdmi_hpd_driver);
}

module_init(brcmstb_extcon_hdmi_hpd_init);
module_exit(brcmstb_extcon_hdmi_hpd_exit);

MODULE_AUTHOR("Broadcom LTD");
MODULE_LICENSE("GPL v2");
