#!/bin/bash

#######################################
#
# Build Framework standard script for
#
# QT5 component

# use -e to fail on any shell issue
# -e is the requirement from Build Framework
set -e


# default PATHs - use `man readlink` for more info
# the path to combined build
export RDK_PROJECT_ROOT_PATH=${RDK_PROJECT_ROOT_PATH-`readlink -m ../../..`}
export COMBINED_ROOT=$RDK_PROJECT_ROOT_PATH

# path to build script (this script)
export RDK_SCRIPTS_PATH=${RDK_SCRIPTS_PATH-`readlink -m $0 | xargs dirname`}

# path to components sources and target
export RDK_SOURCE_PATH=${RDK_SOURCE_PATH-`readlink -m .`}
export RDK_TARGET_PATH=${RDK_TARGET_PATH-$RDK_SOURCE_PATH}

# fsroot and toolchain (valid for all devices)
export RDK_FSROOT_PATH=${RDK_FSROOT_PATH-`readlink -m $RDK_PROJECT_ROOT_PATH/sdk/fsroot/ramdisk`}
export RDK_TOOLCHAIN_PATH=${RDK_TOOLCHAIN_PATH-`readlink -m $RDK_PROJECT_ROOT_PATH/sdk/toolchain/staging_dir`}


# default component name
export RDK_COMPONENT_NAME=${RDK_COMPONENT_NAME-`basename $RDK_SOURCE_PATH`}

export RDK_SUDO_SAFE=${RDK_SUDO_SAFE-${RDK_PROJECT_ROOT_PATH}/build_framework/sudo_safe.sh}

set -o pipefail
DLNA=0
DLNACLIENTTSB=0
HYBRID=0
LOG_TO_STDOUT=1
REBUILD=0
OPTIMIZE=0
NO_GUI=0
VERBOSE=0
NO_CONFIG_TEST=0
DISABLE_STRIP=0



# parse arguments
INITIAL_ARGS=$@

function usage()
{
    set +x
    echo "Usage: `basename $0` [-e feature] [-suxogvf] [-h|--help] [-v|--verbose] [action]"
    echo "    -h    --help                  : this help"
    echo "    -v    --verbose               : verbose output"
    echo "    -e                            : enable named feature"
    echo "    -s                            : log to standard output"
    echo "    -u                            : upload debug symbols"
    echo "    -x                            : build hybrid version"
    echo "    -o                            : build optimized Qt"
    echo "    -g                            : don't build the QtGui library and dependencies"
    echo "    -v                            : verbose build"
    echo "    -f                            : skip config change test"
    echo "    -t                            : debug tracing"
    echo
    echo "Supported features:"
    echo "      dlna: build DLNA video player for XG1-Delia"
    echo "      dlnaclienttsb: build DLNA video player for XG1-MC, XG1-HLMC"
    echo
    echo "Supported actions:"
    echo "      configure, clean, build (DEFAULT), rebuild, install"
}

function log()
{
    if [[ "$LOG_TO_STDOUT" = "1" ]]; then
        cat
    else
        cat >> $buildReport
    fi
}

function trap_exit() {
    exit_code=$?
    if [ $exit_code -ne 0 ]; then
        echo "BUILD QT FAILED" 2>&1 | log
    fi
    trap - INT TERM EXIT
    exit $exit_code
}
trap 'trap_exit' INT TERM EXIT


# options may be followed by one colon to indicate they have a required argument
if ! GETOPT=$(getopt -n "build.sh" -o hvxcuset:ogvf -l help,verbose -- "$@")
then
    usage
    exit 1
fi

eval set -- "$GETOPT"

while true; do
  case "$1" in
    -h | --help ) usage; exit 0 ;;
    -v | --verbose ) set -x ;;
         x)
             HYBRID=1
             ;;
         c)
             COMBINED=1
             ;;
         u)
             UPLOAD=1
             ;;
         s)
             LOG_TO_STDOUT=1
             ;;
         e)
             if [ "$OPTARG" = "dlna" ]; then
                 DLNA=1
             elif [ "$OPTARG" = "dlnaclienttsb" ]; then
                 DLNACLIENTTSB=1
             else
                 usage
             fi
             ;;
         o)
             OPTIMIZE=1
             ;;
         g)
             NO_GUI=1
             ;;
         v)
             VERBOSE=1
             ;;
         f)
             NO_CONFIG_TEST=1
             ;;
         t)
             DISABLE_STRIP=1
             ;;
    -- ) shift; break;;
    * ) break;;
  esac
  shift
done

ARGS=$@


# component-specific vars
SCRIPTS_DIR=$RDK_SCRIPTS_PATH
BUILDS_DIR=$RDK_PROJECT_ROOT_PATH


# functional modules

function configure()
{
    true #use this function to perform any pre-build configuration
}

function clean()
{
    true #use this function to provide instructions to clean workspace
}

function build()
{
    . $RDK_SUDO_SAFE

    qtPath=$BUILDS_DIR/opensource/src/qt
    sdkTool=$BUILDS_DIR/sdk/toolchain
    sdkFsroot=$BUILDS_DIR/sdk/fsroot/ramdisk
    xrePath=$BUILDS_DIR/xre/src
    export DUMP_SYMS=$SCRIPTS_DIR/tools/linux/dump_syms

    canmoreSdk=$sdkTool/i686-linux-elf

    cd $qtPath

    qtSrc=$qtPath/source

    sudo_safe rm -rf $sdkFsroot/usr/local/Qt
    sudo_safe mkdir $sdkFsroot/usr/local/Qt
    sudo_safe chmod -R 777 $sdkFsroot/usr/local/Qt
    sudo_safe rm -rf $sdkFsroot/usr/local/share/fonts
    sudo_safe mkdir $sdkFsroot/usr/local/share/fonts
    sudo_safe chmod -R 777 $sdkFsroot/usr/local/share/fonts
    #Make sure hybrid is set.

      if [[ "$HYBRID" -eq "1" ]];then
      export HYBRID_FLAG=1
      else
      export HYBRID_FLAG=0
      fi
      export QT_SRC_ROOT=$qtSrc
      export TOOLCHAIN_DIR=$canmoreSdk
      export FSROOT=$sdkFsroot

      if [[ "$DLNA" -eq "1" ]]; then
          export CUSTOM_QT_CONFIG_OPTS="-DENABLE_DLNA_VIDEO_PLAYER $CUSTOM_QT_CONFIG_OPTS"
      fi

      if [[ "$DLNACLIENTTSB" -eq "1" ]]; then
          export CUSTOM_QT_CONFIG_OPTS="-DENABLE_DLNA_VIDEO_PLAYER -DCLIENT_TSB $CUSTOM_QT_CONFIG_OPTS"
      fi

      export WEBKITSRC=$QT_SRC_ROOT/src/3rdparty/webkit-2.3
      if [ -d $WEBKITSRC/Tools/qmake ]; then
          rm -f "$qtPath/build/mkspecs/modules/qt_webkit_version.pri"
          export CUSTOM_QT_CONFIG_OPTS="-no-webkit -no-declarative $CUSTOM_QT_CONFIG_OPTS"
      fi

      if [[ "$OPTIMIZE" -eq "1" ]]; then
          export CUSTOM_QT_CONFIG_OPTS="-optimize $CUSTOM_QT_CONFIG_OPTS"
      fi

      if [[ "$NO_GUI" -eq "1" ]]; then
          export CUSTOM_QT_CONFIG_OPTS="-no-gui $CUSTOM_QT_CONFIG_OPTS"
      fi

      if [[ "$VERBOSE" -eq "1" ]]; then
          export QT_BUILD_CONFIG_VERBOSE=true
      fi

      if [[ "$NO_CONFIG_TEST" -eq "1" ]]; then
          export QT_NO_CONFIG_CHANGE_TEST=true
      fi

      if [[ "$DISABLE_STRIP" -eq "1" ]]; then
          export STRIP="echo skip stripping"
          export CUSTOM_QT_CONFIG_OPTS="-force-debug-info -no-strip $CUSTOM_QT_CONFIG_OPTS"
      fi

      source ./setenv.sh

      if [[ "$UPLOAD" -eq "1" ]] ; then
          export UPLOAD_FLAG=1
          export QT_BREAKPAD_ROOT_PATH=$SCRIPTS_DIR
      fi

      if [[ "$LOG_TO_STDOUT" = "0" ]]; then
          rm -f $buildReport
          echo " Please see the satus of Qt build at $buildReport"
      fi

      if [ $REBUILD -eq 1 ]; then
          echo "Cleaning up..."
          make clean
      fi

      export QT_BUILD_NUM_CPUS=$((`grep -c ^processor /proc/cpuinfo` + 1))
      make << START_HERE 2>&1 | log

START_HERE
      . $RDK_SUDO_SAFE
      if [ $? -ne 0 ]; then
      echo "BUILD QT FAILED" 2>&1 | log
      exit -1
      else
      echo "BUILD QT SUCCESS" 2>&1 | log

      if [ -d $WEBKITSRC/Tools/qmake ]; then
          cd ${RDK_PROJECT_ROOT_PATH}/build_scripts
          export DLNA REBUILD UPLOAD LOG_TO_STDOUT DLNACLIENTTSB
          sh buildWebKit.sh
      fi

      if [[ -z $STRIP ]]; then
              export STRIP=$canmoreSdk/bin/i686-cm-linux-strip
          fi

      cd ${RDK_PROJECT_ROOT_PATH}/build_scripts
      stageDir=stage/usr/local/Qt/
      echo "extracting symbols and stripping the binaries"

      for BINARY in `find $qtPath/$stageDir -type f -perm +111 -print` ; do
          file=${BINARY##*/}
          if [[ `file $BINARY | grep "ELF"` ]] ; then
          if [[ `echo libQt{,5}{Core,Network,WebKit,OpenGL,Script,Concurrent} | grep ${file%%.*}` ]] ; then
              opt="-es"
          else
              opt="-s"
          fi
          sh extractSymbols.sh $opt $BINARY 2>&1 | tee -a $buildReport
          else
          echo "$file is not ELF, skipping"
          fi
      done
      cd -

      fi
}

function rebuild()
{
    REBUILD=1
    build
}

function install()
{
    . $RDK_SUDO_SAFE
    QT_PATH=$RDK_PROJECT_ROOT_PATH/opensource/src/qt
    SDK_PATH=${RDK_PROJECT_ROOT_PATH}/sdk
    FSROOT_TOP_PATH=${SDK_PATH}/fsroot
    if [ ! -d "${RDK_FSROOT_PATH}/usr/local/Qt" ]; then
      sudo mkdir ${RDK_FSROOT_PATH}/usr/local/Qt
    fi
    sudo chmod -R 777 ${RDK_FSROOT_PATH}/usr/local/Qt

    cd ${QT_PATH}/stage/usr/local/Qt/
    CURRENT_ROOT=`grep "RDK_PLATFORM_PWD" mkspecs/qdevice.pri | cut -d "=" -f2 | cut -d "." -f1`
    OLD_ROOT=${CURRENT_ROOT/ /}
    NEW_BUILD_PATH=`echo $RDK_PROJECT_ROOT_PATH | cut -d "." -f1`
    find lib/ -name "*.prl" -type f -exec sed -i -e "s|$OLD_ROOT|$NEW_BUILD_PATH|g" "{}" \; -print
    find lib/ -name "*.la" -type f -exec sed -i -e "s|$OLD_ROOT|$NEW_BUILD_PATH|g" "{}" \; -print
    find lib/pkgconfig/ -name "*.pc" -type f -exec sed -i -e "s|$OLD_ROOT|$NEW_BUILD_PATH|g" "{}" \; -print
    find mkspecs/ -name "*.pri" -type f -exec sed -i -e "s|$OLD_ROOT|$NEW_BUILD_PATH|g" "{}" \; -print
    find mkspecs/modules/ -name "*.pri" -type f -exec sed -i -e "s|$OLD_ROOT|$NEW_BUILD_PATH|g" "{}" \; -print

    cd $QT_PATH
    sudo cp -Rp stage/usr/local/Qt/* ${FSROOT_TOP_PATH}/ramdisk/usr/local/Qt

    fontSrcDir=$QT_PATH/fonts
    fontDstDir=${FSROOT_TOP_PATH}/ramdisk/usr/local/Qt/lib/fonts
    fontCacheDir=${FSROOT_TOP_PATH}/ramdisk/fontconfig/var/cache/fontconfig

    if [ -d "$fontCacheDir" ] ; then
        rm -rf $fontCacheDir
    fi

    mkdir -p $fontCacheDir
    cd $fontCacheDir
    tar zxvf $fontSrcDir/fontconfig.tgz

    usrShareDir=${FSROOT_TOP_PATH}/ramdisk/usr/local/share/fonts
    if [ -d "$usrShareDir" ] ; then
        rm -rf $usrShareDir
    fi

    mkdir -p $usrShareDir
    cd $usrShareDir

    tar zxvf $fontSrcDir/usr_local_share_fonts_patch.tar.gz
    unzip -o $fontSrcDir/XFINITYSansTT.zip
    unzip -o $fontSrcDir/DejaVuFonts.zip

    if [ -d "$fontDstDir" ] ; then
        rm -rf $fontDstDir
    fi

    mkdir -p $fontDstDir
    cd $fontDstDir

    rm -f *
    ln -s /usr/local/share/fonts/* .


    platformDir=${FSROOT_TOP_PATH}/ramdisk/mnt/nfs/env

    if [ ! -d "$platformDir" ] ; then
        mkdir -p $platformDir
    fi

    cp $fontSrcDir/platform_fonts.conf $platformDir/platform_fonts.conf

    cp $QT_PATH/components/fsroot/usr/local/Qt/fontconfig/etc/fonts/local.conf $RDK_FSROOT_PATH/usr/local/Qt/fontconfig/etc/fonts/local.conf
}


# run the logic

#these args are what left untouched after parse_args
HIT=false

for i in "$ARGS"; do
    case $i in
        configure)  HIT=true; configure ;;
        clean)      HIT=true; clean ;;
        build)      HIT=true; build ;;
        rebuild)    HIT=true; rebuild ;;
        install)    HIT=true; install ;;
        *)
            #skip unknown
        ;;
    esac
done

# if not HIT do build by default
if ! $HIT; then
  build
fi
