// COMCAST MODIFICATION BEGIN [GSTREAMER_HARDWARE_VIDEO_DECODING]
#ifdef GSTREAMER_HARDWARE_VIDEO_DECODING

// Copyright 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#define USE_GSTREAMER_APP_SRC 1
//#define WRITE_VIDEO_DEBUG_OUTPUT 1
//#define WRITE_AUDIO_DEBUG_OUTPUT 1

#include "content/renderer/media/webmediaplayer_gstreamer.h"

#include <assert.h>

#include <algorithm>
#include <limits>
#include <string>
#include <vector>

#include "base/bind.h"
#include "base/callback.h"
#include "base/command_line.h"
#include "base/debug/crash_logging.h"
#include "base/file_util.h"
#include "base/json/json_reader.h"
#include "base/message_loop/message_loop_proxy.h"
#include "base/metrics/histogram.h"
#include "base/strings/stringprintf.h"
#include "base/strings/string_number_conversions.h"
#include "base/synchronization/waitable_event.h"
#include "base/values.h"
#include "cc/layers/layer.h"
#include "cc/layers/video_layer.h"
#include "content/public/common/content_switches.h"
#include "content/renderer/media/buffered_data_source.h"
#include "content/renderer/media/crypto/key_systems.h"
#include "content/renderer/media/webmediaplayer_delegate.h"
#include "content/renderer/media/webmediaplayer_params.h"
#include "content/renderer/media/webmediaplayer_util.h"
#include "content/renderer/media/webmediasource_impl.h"
#include "content/renderer/pepper/pepper_webplugin_impl.h"
#include "gpu/GLES2/gl2extchromium.h"
#include "media/base/bind_to_loop.h"
#include "media/base/data_source.h"
#include "media/base/filter_collection.h"
#include "media/base/limits.h"
#include "media/base/media_log.h"
#include "media/base/media_switches.h"
#include "media/base/pipeline.h"
#include "media/base/video_frame.h"
#include "media/base/video_decoder_config.h"
#include "media/filters/chunk_demuxer.h"
#include "media/filters/video_renderer_impl.h"
#include "media/filters/vpx_video_decoder.h"
#include "third_party/WebKit/public/platform/WebMediaSource.h"
#include "third_party/WebKit/public/platform/WebString.h"
#include "third_party/WebKit/public/platform/WebURL.h"
#include "third_party/WebKit/public/web/WebDocument.h"
#include "third_party/WebKit/public/web/WebRuntimeFeatures.h"
#include "third_party/WebKit/public/web/WebSecurityOrigin.h"
#include "third_party/WebKit/public/web/WebView.h"
#include "v8/include/v8.h"
#include "webkit/renderer/compositor_bindings/web_layer_impl.h"

using base::StringPrintf;
using base::TimeDelta;
using base::ReadFileToString;
using base::JSONReader;
using base::FilePath;
using blink::WebCanvas;
using blink::WebMediaPlayer;
using blink::WebRect;
using blink::WebSecurityOrigin;
using blink::WebSize;
using blink::WebString;
using media::DecoderBuffer;
using media::DemuxerStream;
using media::PipelineStatus;
using media::VideoDecoderConfig;
using media::VideoFrame;

namespace {

// Amount of extra memory used by each player instance reported to V8.
// It is not exact number -- first, it differs on different platforms,
// and second, it is very hard to calculate. Instead, use some arbitrary
// value that will cause garbage collection from time to time. We don't want
// it to happen on every allocation, but don't want 5k players to sit in memory
// either. Looks that chosen constant achieves both goals, at least for audio
// objects. (Do not worry about video objects yet, JS programs do not create
// thousands of them...)
const int kPlayerExtraMemory = 1024 * 1024;

// Limits the range of playback rate.
//
// TODO(kylep): Revisit these.
//
// Vista has substantially lower performance than XP or Windows7.  If you speed
// up a video too much, it can't keep up, and rendering stops updating except on
// the time bar. For really high speeds, audio becomes a bottleneck and we just
// use up the data we have, which may not achieve the speed requested, but will
// not crash the tab.
//
// A very slow speed, ie 0.00000001x, causes the machine to lock up. (It seems
// like a busy loop). It gets unresponsive, although its not completely dead.
//
// Also our timers are not very accurate (especially for ogg), which becomes
// evident at low speeds and on Vista. Since other speeds are risky and outside
// the norms, we think 1/16x to 16x is a safe and useful range for now.
const double kMinRate = 0.0625;
const double kMaxRate = 16.0;

// Prefix for histograms related to Encrypted Media Extensions.
const char* kMediaEme = "Media.EME.";

// GstPlayFlags flags from playbin2. It is the policy of GStreamer to
// not publicly expose element-specific enums. That's why this
// GstPlayFlags enum has been copied here.
typedef enum {
  GST_PLAY_FLAG_VIDEO = 0x00000001,
  GST_PLAY_FLAG_AUDIO = 0x00000002,
  GST_PLAY_FLAG_TEXT = 0x00000004,
  GST_PLAY_FLAG_VIS = 0x00000008,
  GST_PLAY_FLAG_SOFT_VOLUME = 0x00000010,
  GST_PLAY_FLAG_NATIVE_AUDIO = 0x00000020,
  GST_PLAY_FLAG_NATIVE_VIDEO = 0x00000040,
  GST_PLAY_FLAG_DOWNLOAD = 0x00000080,
  GST_PLAY_FLAG_BUFFERING = 0x000000100
} GstPlayFlags;

#ifdef GST_API_VERSION_1
static const char* gPlaybinName = "playbin";
#else
static const char* gPlaybinName = "playbin2";
#endif

const int kVideo = 0;
const int kAudio = 1;

const int kCallPrintDebugLevel = 20;

}  // namespace

namespace media {

GStreamerDemuxerHost::GStreamerDemuxerHost() : total_bytes_(0), duration_() {}

void GStreamerDemuxerHost::SetTotalBytes(int64 total_bytes) {
  VLOG(kCallPrintDebugLevel)
      << "GStreamerDataHost::SetTotalBytes(): " << total_bytes;

  total_bytes_ = total_bytes;
}

void GStreamerDemuxerHost::AddBufferedByteRange(int64 start, int64 end) {
  VLOG(kCallPrintDebugLevel) << "GStreamerDataHost::AddBufferedByteRange(): ("
                             << start << "," << end << ")";

  buffered_byte_ranges_.Add(start, end);
}

void GStreamerDemuxerHost::AddBufferedTimeRange(base::TimeDelta start,
                                                base::TimeDelta end) {
  VLOG(kCallPrintDebugLevel) << "GStreamerDataHost::AddBufferedTimeRange(): ("
                             << start.InSecondsF() << "," << end.InSecondsF()
                             << ")";

  buffered_time_ranges_.Add(start, end);
}

void GStreamerDemuxerHost::SetDuration(base::TimeDelta duration) {
  VLOG(kCallPrintDebugLevel)
      << "GStreamerDemuxerHost::SetDuration(): " << duration.InSecondsF();

  duration_ = duration;
}

void GStreamerDemuxerHost::OnDemuxerError(PipelineStatus error) {
  // stop playback here, treat like a read error in the data source
}

void GStreamerDemuxerHost::AddTextStream(DemuxerStream* text_stream,
                                         const TextTrackConfig& config) {
  // TODO(mzuber): what do I do about text tracks?
}

void GStreamerDemuxerHost::RemoveTextStream(DemuxerStream* text_stream) {
  // TODO(mzuber): what do I do about text tracks?
}

const media::Ranges<int64>& GStreamerDemuxerHost::GetBufferedByteRanges() {
  return buffered_byte_ranges_;
}

const media::Ranges<base::TimeDelta>&
GStreamerDemuxerHost::GetBufferedTimeRanges() {
  return buffered_time_ranges_;
}

int64 GStreamerDemuxerHost::GetTotalBytes() { return total_bytes_; }

base::TimeDelta GStreamerDemuxerHost::GetDuration() { return duration_; }

}  // namespace media

namespace content {

#define COMPILE_ASSERT_MATCHING_ENUM(name)                              \
  COMPILE_ASSERT(static_cast<int>(WebMediaPlayer::CORSMode##name) ==    \
                     static_cast<int>(BufferedResourceLoader::k##name), \
                 mismatching_enums)
COMPILE_ASSERT_MATCHING_ENUM(Unspecified);
COMPILE_ASSERT_MATCHING_ENUM(Anonymous);
COMPILE_ASSERT_MATCHING_ENUM(UseCredentials);
#undef COMPILE_ASSERT_MATCHING_ENUM

#define BIND_TO_RENDER_LOOP(function) \
  media::BindToLoop(main_loop_, base::Bind(function, AsWeakPtr()))

#define BIND_TO_RENDER_LOOP_1(function, arg1) \
  media::BindToLoop(main_loop_, base::Bind(function, AsWeakPtr(), arg1))

#define BIND_TO_RENDER_LOOP_2(function, arg1, arg2) \
  media::BindToLoop(main_loop_, base::Bind(function, AsWeakPtr(), arg1, arg2))

static void LogMediaSourceError(const scoped_refptr<media::MediaLog>& media_log,
                                const std::string& error) {
  media_log->AddEvent(media_log->CreateMediaSourceErrorEvent(error));
}

static gboolean mediaPlayerPrivateMessageCallback(
    GstBus*,
    GstMessage* message,
    WebMediaPlayerGStreamer* player) {
  VLOG(kCallPrintDebugLevel) << "mediaPlayerPrivateMessageCallback\n";
  fflush(stderr);

  const CommandLine* command_line = CommandLine::ForCurrentProcess();
  bool is_single_process = command_line->HasSwitch(switches::kSingleProcess);

  // when not in single-process mode we create our own gmainloop in another
  // thread
  // so we must pass the messages to the main thread for handling
  if (!is_single_process)
    return player->handleMessageMultiThreadedThread(message);
  else
    return player->handleMessage(message);
}

static void mediaPlayerPrivateOnAutoElementAdded(
    GstBin* decodebin2,
    GstElement* element,
    WebMediaPlayerGStreamer* player) {
  player->onAutoElementAdded(element);
}

static void mediaPlayerPrivateOnAutoPadAdded(GstElement* decodebin2,
                                             GstPad* pad,
                                             gboolean last,
                                             WebMediaPlayerGStreamer* player) {
  player->onAutoPadAdded(pad);
}

bool initializeGStreamerAndRegisterWebKitElements() { return true; }

// appsrc buffered data source additions
static void mediaPlayerPrivateStartFeed(GstAppSrc* appsrc,
                                        guint size,
                                        WebMediaPlayerGStreamer* player) {
  player->StartFeedingAppSource(appsrc);
}

static void mediaPlayerPrivateStopFeed(GstAppSrc* appsrc,
                                       WebMediaPlayerGStreamer* player) {
  player->StopFeedingAppSource(appsrc);
}

static gboolean mediaPlayerPrivateSeekData(GstAppSrc* appsrc,
                                           guint64 position,
                                           WebMediaPlayerGStreamer* player) {
  player->SetNewAppSourceReadPosition(appsrc, position);
  return TRUE;
}

static void decryptingDemuxerStopped() { }

// single static glib main loop running in a static thread for all media player
// instances
// once started the thread and main loop will exist for the life of the render
// process
GMainLoop* WebMediaPlayerGStreamer::gMainLoop = NULL;
base::Thread* WebMediaPlayerGStreamer::gMainLoopThread =
    new base::Thread("Glib Main Loop");

const std::string WebMediaPlayerGStreamer::kStringsFilePath =
    "/etc/receiver_chromium_media_player_settings.json";

WebMediaPlayerGStreamer::WebMediaPlayerGStreamer(
    content::RenderView* render_view,
    blink::WebFrame* frame,
    blink::WebMediaPlayerClient* client,
    base::WeakPtr<WebMediaPlayerDelegate> delegate,
    const WebMediaPlayerParams& params)
    : content::RenderViewObserver(render_view),
      frame_(frame),
      main_loop_(base::MessageLoopProxy::current()),
      media_loop_(params.message_loop_proxy()),
      client_(client),
      delegate_(delegate),
      defer_load_cb_(params.defer_load_cb()),
      media_log_(params.media_log()),
      incremented_externally_allocated_memory_(false),
      chunk_demuxer_(NULL),
      video_frame_provider_client_(NULL),

      // gstreamer additions
      play_bin_(0),
      webkit_video_sink_(0),
      fps_sink_(0),
      pipeline_(0),
      source_(0),
      decbin_(0),
      audio_decoder_(0),
      video_decoder_(0),
      audio_sink_(0),
      video_sink_(0),
      seek_time_(0.0f),
      changing_rate_(false),
      end_time_(std::numeric_limits<float>::infinity()),
      is_end_reached_(false),
      network_state_(WebMediaPlayer::NetworkStateEmpty),
      ready_state_(WebMediaPlayer::ReadyStateHaveNothing),
      // is_streaming_(false),
      size_(WebSize()),
      buffer_(0),
      media_locations_(0),
      media_location_current_index_(0),
      reset_pipeline_(false),
      paused_(true),
      seeking_(false),
      buffering_(false),
      playback_rate_(1),
      error_occured_(false),
      media_duration_(0),
      started_buffering_(false),
      have_ever_started_play_(false)
      // , fill_timer_(this, &MediaPlayerPrivateGStreamer::fillTimerFired)
      ,
      max_time_loaded_(0),
      buffering_percentage_(0),
      preload_(WebMediaPlayer::PreloadAuto),
      delaying_load_(false),
      media_duration_known_(true),
      max_time_loaded_at_last_did_loading_progress_(0),
      volume_timer_handler_(0),
      mute_timer_handler_(0),
      has_video_(false),
      has_audio_(false),
      audio_timer_handler_(0),
      video_timer_handler_(0),
      video_seek_timer_handler_(0),
      webkit_audio_sink_(0),
      original_preload_was_auto_and_was_overridden_(false),
      message_map_locked_(false),
      gstreamer_strings_(NULL),
      appsrc_source_(0),
      aborted_(true, false),  // We never want to reset |aborted_|.
      last_read_bytes_(0),
      read_position_(0),
      supports_save_(true),
      is_local_source_(false),
      decrypted_video_stream_initialized_(false),
      decrypted_audio_stream_initialized_(false) {
  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer::WebMediaPlayerGStreamer\n";

  LoadGstreamerStrings();

  if (blink::WebRuntimeFeatures::isPrefixedEncryptedMediaEnabled()) {
    decryptor_.reset(new ProxyDecryptor(
#if defined(ENABLE_PEPPER_CDMS)
        client,
        frame,
#endif
        BIND_TO_RENDER_LOOP(&WebMediaPlayerGStreamer::OnKeyAdded),
        BIND_TO_RENDER_LOOP(&WebMediaPlayerGStreamer::OnKeyError),
        BIND_TO_RENDER_LOOP(&WebMediaPlayerGStreamer::OnKeyMessage)));

    media::SetDecryptorReadyCB set_decryptor_ready_cb;
    set_decryptor_ready_cb = base::Bind(&ProxyDecryptor::SetDecryptorReadyCB,
                                        base::Unretained(decryptor_.get()));

    decrypted_video_stream_.reset(new media::DecryptingDemuxerStream(
        media_loop_, set_decryptor_ready_cb));
    decrypted_audio_stream_.reset(new media::DecryptingDemuxerStream(
        media_loop_, set_decryptor_ready_cb));
  }

  initializeGStreamerAndRegisterWebKitElements();

  memset(&read_in_progress_, 0, sizeof(read_in_progress_));
  memset(&should_be_reading_, 0, sizeof(should_be_reading_));
}

WebMediaPlayerGStreamer::~WebMediaPlayerGStreamer() {
  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer::~WebMediaPlayerGStreamer\n";

  DCHECK(main_loop_->BelongsToCurrentThread());

  Destroy();

  // base::MessageLoop::current()->RunUntilIdle();

  SetVideoFrameProviderClient(NULL);
  GetClient()->setWebLayer(NULL);

  if (delegate_.get())
    delegate_->PlayerGone(this);

  VLOG(20) << "webmedia player destroyed!\n";
  fflush(stderr);
}

void WebMediaPlayerGStreamer::Destroy() {
  DCHECK(main_loop_->BelongsToCurrentThread());
  // setting the window size to nothing on destruction seems to get rid
  // of issues where a phantom window appears when we start another
  // video
  std::string window_set = GetGStreamerString("video-sink-window-set");
  if (video_sink_) {
    char buf[100];
    snprintf(buf, sizeof(buf), "%d,%d,%d,%d", 0, 0, 0, 0);
    g_object_set(GST_OBJECT(video_sink_), window_set.c_str(), buf, NULL);
  }

  if (decrypted_video_stream_initialized_) {
    decrypted_video_stream_->Stop(base::Bind(&decryptingDemuxerStopped));
  }
  if (decrypted_audio_stream_initialized_) {
    decrypted_audio_stream_->Stop(base::Bind(&decryptingDemuxerStopped));
  }

  if (data_source_)
    data_source_->Abort();
  if (chunk_demuxer_) {
    chunk_demuxer_->Shutdown();
    base::WaitableEvent waiter(false, false);
    chunk_demuxer_->Stop(
        base::Bind(&base::WaitableEvent::Signal, base::Unretained(&waiter)));
    waiter.Wait();
    chunk_demuxer_ = NULL;
  }

  if (buffer_)
    gst_buffer_unref(buffer_);
  buffer_ = 0;

  if (media_locations_) {
    gst_structure_free(media_locations_);
    media_locations_ = 0;
  }
  // m_player = 0;

  if (load_type_ != LoadTypeMediaSource)
    DestroyNormalPipeline();
  else
    DestroyMediaSourcePipeline();

  if (mute_timer_handler_)
    g_source_remove(mute_timer_handler_);

  if (volume_timer_handler_)
    g_source_remove(volume_timer_handler_);

  if (video_timer_handler_)
    g_source_remove(video_timer_handler_);

  if (audio_timer_handler_)
    g_source_remove(audio_timer_handler_);

  if (video_seek_timer_handler_)
    g_source_remove(video_seek_timer_handler_);

  // lock the message map so the gmainloop thread can't add
  // any more messages
  lockMessageMap(true);

  // if there are any gstreamer messages that haven't been handled
  // by the main loop, destroy them here to avoid a memory leak
  clearMessageMap();

  data_source_.reset();
  demuxer_.reset();
  demuxer_host_.reset();
}

void WebMediaPlayerGStreamer::LoadGstreamerStrings() {
  DCHECK(main_loop_->BelongsToCurrentThread());

  std::string strings_file;
  if (!ReadFileToString(FilePath(kStringsFilePath), &strings_file)) {
    VLOG(1) << "Error reading gstreamer strings file!";
    return;
  }

  JSONReader reader;
  gstreamer_json_data_.reset(reader.ReadToValue(strings_file));
  if (!gstreamer_json_data_.get() ||
      !gstreamer_json_data_->GetAsDictionary(&gstreamer_strings_)) {
    VLOG(1) << "Error parsing gstreamer strings file!";
    return;
  }
}

std::string WebMediaPlayerGStreamer::GetGStreamerString(std::string key) {
  // DCHECK(main_loop_->BelongsToCurrentThread());

  std::string val;

  if (gstreamer_strings_) {
    gstreamer_strings_->GetString(key, &val);
  }

  return val;
}

void WebMediaPlayerGStreamer::load(LoadType load_type,
                                   const blink::WebURL& url,
                                   CORSMode cors_mode) {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::load()\n";

  load_type_ = load_type;

  // Media source pipelines can start immediately.
  if (load_type == LoadTypeMediaSource) {
    supports_save_ = false;
    VLOG(kCallPrintDebugLevel) << "Load type is media source\n";
    // DoLoad(url);

    chunk_demuxer_ = new media::ChunkDemuxer(
        BIND_TO_RENDER_LOOP(&WebMediaPlayerGStreamer::OnDemuxerOpened),
        BIND_TO_RENDER_LOOP(&WebMediaPlayerGStreamer::OnNeedKey),
        base::Bind(&LogMediaSourceError, media_log_));
    demuxer_.reset(chunk_demuxer_);
    demuxer_host_.reset(new media::GStreamerDemuxerHost);
    bool text_renderer =
        false;  // TODO(mzuber): What do I do about text rendering

    media::PipelineStatusCB demuxer_init = base::Bind(
        &WebMediaPlayerGStreamer::OnDemuxerInit, base::Unretained(this));
    demuxer_->Initialize(demuxer_host_.get(), demuxer_init, text_renderer);
  } else {
    VLOG(kCallPrintDebugLevel) << "Load type is NOT media source\n";

    GURL gurl(url);
    data_source_.reset(new BufferedDataSource(
        main_loop_,
        frame_,
        media_log_.get(),
        base::Bind(&WebMediaPlayerGStreamer::NotifyDownloading, AsWeakPtr())));
    data_source_->Initialize(
        url,
        static_cast<BufferedResourceLoader::CORSMode>(cors_mode),
        base::Bind(&WebMediaPlayerGStreamer::DataSourceInitialized,
                   AsWeakPtr(),
                   gurl));
    demuxer_host_.reset(new media::GStreamerDemuxerHost);
    data_source_->set_host(demuxer_host_.get());
    is_local_source_ = !gurl.SchemeIsHTTPOrHTTPS();
  }
}

void WebMediaPlayerGStreamer::DoLoad(const blink::WebURL& url) {
  DCHECK(main_loop_->BelongsToCurrentThread());
  // make sure the message map isn't locked so we receive gstreamer
  // messages on the main thread
  lockMessageMap(false);

  scr_rect_ = WebRect(0, 0, 0, 0);

  // start single static glib main loop if it isn't running
  // and we aren't in single-process mode where a gmainloop will be inherited
  const CommandLine* command_line = CommandLine::ForCurrentProcess();
  bool is_single_process = command_line->HasSwitch(switches::kSingleProcess);
  if (!gMainLoopThread->IsRunning() && !is_single_process) {
    gMainLoopThread->Start();
    gMainLoopThread->message_loop()->PostTask(
        FROM_HERE, base::Bind(&WebMediaPlayerGStreamer::GlibMainLoopFunc));
  }

  //----------------------------------------------------------------------
  // Reconstruct
  //----------------------------------------------------------------------
  if (load_type_ != LoadTypeMediaSource)
    ConstructNormalPipeline(url);
  else
    ConstructMediaSourcePipeline();

  if (preload_ == WebMediaPlayer::PreloadNone) {
    DVLOG(1) << ("Delaying load.");
    delaying_load_ = true;
  }

  // Reset network and ready states. Those will be set properly once
  // the pipeline pre-rolled.
  network_state_ = WebMediaPlayer::NetworkStateLoading;
  client_->networkStateChanged();
  ready_state_ = WebMediaPlayer::ReadyStateHaveNothing;
  client_->readyStateChanged();

  // GStreamer needs to have the pipeline set to a paused state to
  // start providing anything useful.
  gst_element_set_state(pipeline_, GST_STATE_PAUSED);

#define WAIT_FOR_INITIAL_CONSRUCTION (0)
/* Below error handling must be disabled for ::loadNextLocation() to work
 * properly */
#if WAIT_FOR_INITIAL_CONSRUCTION
  GstState currentState = GST_STATE_READY;
  int cnt = 50; /* wait 5 seconds */
  while ((currentState != GST_STATE_PAUSED) && cnt--) {
    GstStateChangeReturn ret = gst_element_get_state(
        GST_ELEMENT(pipeline_), &currentState, NULL, 100 * GST_MSECOND);
    if (cnt < 10) {
      DBG_MSG("gst_element_get_state ret=%s currentState=%s cnt=%d\n",
              gst_element_state_change_return_get_name(ret),
              gst_element_state_get_name(currentState),
              cnt);
    }
  }

  if (cnt < 1) {
    DBG_MSG("GST_STATE_CHANGE_FAILURE, loadingFailed!!!!!!!!!!!!!!!!\n");
    /* use decode error for now, because network error may not get reported ?*/
    loadingFailed(MediaPlayer::DecodeError);

    /*chz revisit when XRE error handling is improved */
    gst_element_set_state(pipeline_, GST_STATE_NULL);
    gst_object_unref(GST_OBJECT(pipeline_));

    pipeline_ = NULL;
    source_ = NULL;
    decbin_ = NULL;
    video_decoder_ = NULL;
    audio_decoder_ = NULL;
    video_sink_ = NULL;
    audio_sink_ = NULL;
    return;
  }
#endif

  if (!delaying_load_)
    commitLoad();
}

void WebMediaPlayerGStreamer::SetNetworkState(WebMediaPlayer::NetworkState state) {
  DCHECK(main_loop_->BelongsToCurrentThread());
  DVLOG(1) << "SetNetworkState: " << state;
  network_state_ = state;
  // Always notify to ensure client has the latest value.
  client_->networkStateChanged();
}

void WebMediaPlayerGStreamer::SetReadyState(WebMediaPlayer::ReadyState state) {
  DCHECK(main_loop_->BelongsToCurrentThread());
  DVLOG(1) << "SetReadyState: " << state;

  if (state == WebMediaPlayer::ReadyStateHaveEnoughData &&
      is_local_source_ &&
      network_state_ == WebMediaPlayer::NetworkStateLoading)
    SetNetworkState(WebMediaPlayer::NetworkStateLoaded);

  ready_state_ = state;
  // Always notify to ensure client has the latest value.
  client_->readyStateChanged();
}

void WebMediaPlayerGStreamer::ConstructNormalPipeline(
    const blink::WebURL& url) {
  //----------------------------------------------------------------------
  // Destroy
  //----------------------------------------------------------------------
  DestroyNormalPipeline();

  /* Create the empty pipeline */
  pipeline_ = gst_pipeline_new("pipeline");

  std::string video_sink = GetGStreamerString("video-sink");
  std::string audio_sink = GetGStreamerString("audio-sink");

#ifndef USE_GSTREAMER_APP_SRC
  source_ = gst_element_factory_make("souphttpsrc", "http-source");
#else
  appsrc_source_ = GST_APP_SRC(gst_element_factory_make("appsrc", "app-src"));
#endif
  decbin_ = gst_element_factory_make("decodebin2", "decode-bin");
  video_sink_ = gst_element_factory_make(video_sink.c_str(), "video-output");
  audio_sink_ = gst_element_factory_make(audio_sink.c_str(), "audio-output");

  if (!pipeline_ || !(source_ || appsrc_source_) || !decbin_ || !video_sink_ ||
      !audio_sink_) {
    g_printerr(
        "Not all elements could be created. pipeline_=%p source_=%p "
        "decbin_=%p video_sink_=%p audio_sink_=%p\n",
        pipeline_,
        source_,
        decbin_,
        video_sink_,
        audio_sink_);
    return;
  }

  // Must reference and _sink the pipeline as per gplayback.c and
  // http://gstreamer.freedesktop.org/data/doc/gstreamer/head/gstreamer/html/GstObject.html#gst-object-sink
  gst_object_ref(GST_OBJECT(pipeline_));
  gst_object_sink(GST_OBJECT(pipeline_));

  GstBus* bus = gst_pipeline_get_bus(GST_PIPELINE(pipeline_));
  gst_bus_add_signal_watch(bus);
  g_signal_connect(
      bus, "message", G_CALLBACK(mediaPlayerPrivateMessageCallback), this);
  gst_object_unref(bus);

  g_signal_connect(decbin_,
                   "element-added",
                   G_CALLBACK(mediaPlayerPrivateOnAutoElementAdded),
                   this);
  g_signal_connect(decbin_,
                   "new-decoded-pad",
                   G_CALLBACK(mediaPlayerPrivateOnAutoPadAdded),
                   this);

  /* Modify the source's properties */
  VLOG(kCallPrintDebugLevel) << StringPrintf("WebMediaPlayerGStreamer URL:%s\n",
                                             url.string().utf8().data());
  if (source_) {
    g_object_set(
        G_OBJECT(source_), "location", url.string().utf8().data(), NULL);
  }

  /*Modifiy decode bin properties */
  /* tell decodebin2 to stop searching once it finds platform video/audio caps
   * output */
  std::string decode_video_caps = GetGStreamerString("decode-video-caps");
  std::string decode_audio_caps = GetGStreamerString("decode-audio-caps");
  std::string caps_string = decode_video_caps + "; " + decode_audio_caps;
  g_object_set(G_OBJECT(decbin_),
               "caps",
               gst_caps_from_string(caps_string.c_str()),
               NULL);

  /* Build the pipeline */
  if (appsrc_source_) {
    g_signal_connect(appsrc_source_,
                     "need-data",
                     G_CALLBACK(mediaPlayerPrivateStartFeed),
                     this);
    g_signal_connect(appsrc_source_,
                     "enough-data",
                     G_CALLBACK(mediaPlayerPrivateStopFeed),
                     this);

    // set whether we can seek the stream or not
    if (isLiveStream()) {  // no seeking on live streams
      gst_app_src_set_stream_type(appsrc_source_, GST_APP_STREAM_TYPE_STREAM);
    } else {  // seekable stream
      g_object_set(G_OBJECT(appsrc_source_),
                   "stream-type",
                   GST_APP_STREAM_TYPE_SEEKABLE,
                   "format",
                   GST_FORMAT_BYTES,
                   NULL);
      g_signal_connect(appsrc_source_,
                       "seek-data",
                       G_CALLBACK(mediaPlayerPrivateSeekData),
                       this);
      if (totalBytes() > 0)
        gst_app_src_set_size(appsrc_source_, totalBytes());
    }

    gst_bin_add_many(
        GST_BIN(pipeline_), GST_ELEMENT(appsrc_source_), decbin_, NULL);
    if (gst_element_link(GST_ELEMENT(appsrc_source_), decbin_) == FALSE) {
      g_printerr("could not link source_, decbin_.\n");
      return;
    }
  } else {
    gst_bin_add_many(GST_BIN(pipeline_), source_, decbin_, NULL);
    if (gst_element_link(source_, decbin_) == FALSE) {
      g_printerr("could not link source_, decbin_.\n");
      return;
    }
  }
}

void WebMediaPlayerGStreamer::ConstructMediaSourcePipeline() {
  //----------------------------------------------------------------------
  // Destroy
  //----------------------------------------------------------------------
  DestroyMediaSourcePipeline();

  /* Create the empty pipeline */
  pipeline_ = gst_pipeline_new("pipeline");

  std::string queue_val         = GetGStreamerString("queue");
  std::string video_filter_val  = GetGStreamerString("video-filter");
  std::string video_decoder_val = GetGStreamerString("video-decoder");
  std::string audio_filter_val  = GetGStreamerString("audio-filter");
  std::string audio_decoder_val = GetGStreamerString("audio-decoder");
  std::string video_sink_val    = GetGStreamerString("video-sink");
  std::string audio_sink_val    = GetGStreamerString("audio-sink");

  appsrc_source_ = GST_APP_SRC(gst_element_factory_make("appsrc", "app-src"));
  appsrc_source_audio_ = GST_APP_SRC(gst_element_factory_make("appsrc", "app-src-audio"));

  GstElement* video_queue = gst_element_factory_make(queue_val.c_str(), "video-queue");
  GstElement* audio_queue = gst_element_factory_make(queue_val.c_str(), "audio-queue");

  GstElement* video_filter = gst_element_factory_make(video_filter_val.c_str(), "video-filter");
  GstElement* audio_filter = gst_element_factory_make(audio_filter_val.c_str(), "audio-filter");

  video_decoder_ = gst_element_factory_make(video_decoder_val.c_str(), "video-decoder");
  audio_decoder_ = gst_element_factory_make(audio_decoder_val.c_str(), "audio-decoder");

  video_sink_ = gst_element_factory_make(video_sink_val.c_str(), "video-output");
  audio_sink_ = gst_element_factory_make(audio_sink_val.c_str(), "audio-output");

  if (!pipeline_ || !appsrc_source_ || !appsrc_source_audio_ || !video_queue || !audio_queue || !video_filter ||
      !audio_filter || !video_decoder_ || !audio_decoder_ || !video_sink_ || !audio_sink_) {
    g_printerr(
        "Not all elements could be created. pipeline_=%p source_=%p audio_source =%p"
        "video_queue =%p audio_queue =%p video_filter=%p audio_filter=%p video_decoder_=%p"
        "audio_decoder=%p video_sink_=%p audio_sink=%p\n",
        pipeline_,
        appsrc_source_,
        appsrc_source_audio_,
        video_queue,
        audio_queue,
        video_filter,
        audio_filter,
        video_decoder_,
        audio_decoder_,
        video_sink_,
        audio_sink_);
    return;
  }

  // Must reference and _sink the pipeline as per gplayback.c and
  // http://gstreamer.freedesktop.org/data/doc/gstreamer/head/gstreamer/html/GstObject.html#gst-object-sink
  gst_object_ref(GST_OBJECT(pipeline_));
  gst_object_sink(GST_OBJECT(pipeline_));

  GstBus* bus = gst_pipeline_get_bus(GST_PIPELINE(pipeline_));
  gst_bus_add_signal_watch(bus);
  g_signal_connect(
      bus, "message", G_CALLBACK(mediaPlayerPrivateMessageCallback), this);
  gst_object_unref(bus);

  /* Build the pipeline */
  g_signal_connect(appsrc_source_,
                   "need-data",
                   G_CALLBACK(mediaPlayerPrivateStartFeed),
                   this);
  g_signal_connect(appsrc_source_,
                   "enough-data",
                   G_CALLBACK(mediaPlayerPrivateStopFeed),
                   this);

  g_signal_connect(appsrc_source_audio_,
                   "need-data",
                   G_CALLBACK(mediaPlayerPrivateStartFeed),
                   this);
  g_signal_connect(appsrc_source_audio_,
                   "enough-data",
                   G_CALLBACK(mediaPlayerPrivateStopFeed),
                   this);

  // set whether we can seek the stream or not
  if (isLiveStream()) {  // no seeking on live streams
    gst_app_src_set_stream_type(appsrc_source_, GST_APP_STREAM_TYPE_STREAM);
    gst_app_src_set_stream_type(appsrc_source_audio_, GST_APP_STREAM_TYPE_STREAM);
  } else {  // seekable stream
    g_object_set(G_OBJECT(appsrc_source_),
                 "stream-type",
                 GST_APP_STREAM_TYPE_SEEKABLE,
                 "format",
                 GST_FORMAT_TIME,
                 NULL);
    g_object_set(G_OBJECT(appsrc_source_audio_),
                 "stream-type",
                 GST_APP_STREAM_TYPE_SEEKABLE,
                 "format",
                 GST_FORMAT_TIME,
                 NULL);
    g_signal_connect(appsrc_source_,
                     "seek-data",
                     G_CALLBACK(mediaPlayerPrivateSeekData),
                     this);
    g_signal_connect(appsrc_source_audio_,
                      "seek-data",
                      G_CALLBACK(mediaPlayerPrivateSeekData),
                      this);
    if (totalBytes() > 0) {
      gst_app_src_set_size(appsrc_source_, totalBytes());
      gst_app_src_set_size(appsrc_source_audio_, totalBytes());
    }
  }

  gst_bin_add_many(GST_BIN(pipeline_),
                   GST_ELEMENT(appsrc_source_),
                   GST_ELEMENT(appsrc_source_audio_),
                   video_queue,
                   audio_queue,
                   video_filter,
                   audio_filter,
                   video_decoder_,
                   audio_decoder_,
                   video_sink_,
                   audio_sink_,
                   NULL);

  if (gst_element_link_many(GST_ELEMENT(appsrc_source_),
                            video_queue,
                            video_filter,
                            video_decoder_,
                            video_sink_,
                            NULL) == FALSE)
    g_printerr("could not link media source video gstreamer pipeline!\n");

  if (gst_element_link_many(GST_ELEMENT(appsrc_source_audio_),
                            audio_queue,
                            audio_filter,
                            audio_decoder_,
                            audio_sink_,
                            NULL) == FALSE)
    g_printerr("could not link media source audio gstreamer pipeline!\n");
}

void WebMediaPlayerGStreamer::DestroyNormalPipeline() {
  if (pipeline_) {
    // DBG_MSG("destroying pipeline in load");

    gst_element_set_state(pipeline_, GST_STATE_NULL);
    gst_object_unref(GST_OBJECT(pipeline_));

    pipeline_ = NULL;
    source_ = NULL;
    appsrc_source_ = NULL;
    decbin_ = NULL;
    video_decoder_ = NULL;
    audio_decoder_ = NULL;
    video_sink_ = NULL;
    audio_sink_ = NULL;
  }
}

void WebMediaPlayerGStreamer::DestroyMediaSourcePipeline() {
  if (pipeline_) {
    // DBG_MSG("destroying pipeline in load");

    gst_element_set_state(pipeline_, GST_STATE_NULL);
    gst_object_unref(GST_OBJECT(pipeline_));

    pipeline_ = NULL;
    source_ = NULL;
    appsrc_source_ = NULL;
    decbin_ = NULL;
    video_decoder_ = NULL;
    audio_decoder_ = NULL;
    video_sink_ = NULL;
    audio_sink_ = NULL;
  }
}

// Playback controls.
void WebMediaPlayerGStreamer::play() {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::play()\n";

  if (data_source_)
    data_source_->MediaIsPlaying();

  if (changePipelineState(GST_STATE_PLAYING)) {
    is_end_reached_ = false;
    DVLOG(1) << ("Play");
  }
  have_ever_started_play_ = true;

  // at this point we know we can hole punch
  createWebLayerAndHolePunchFrame();
}

void WebMediaPlayerGStreamer::createWebLayerAndHolePunchFrame() {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer::createWebLayerAndHolePunchFrame()\n";

  // the size of the hole frame doesn't seem to matter as far as
  // i can tell, the hole punching just uses the size of the
  // video layer, which gets determined either by the web code
  // or if its just a link to a video file it is the reported
  // natural size, follow android code by making it naturalSize()
  current_frame_ = VideoFrame::CreateHoleFrame(naturalSize());
  client_->setOpaque(true);

  // TODO(mzuber): Broadcom broke hasVideo, put this back when its fixed
  if (/*hasVideo() &&*/ !video_weblayer_ && client_->needsWebLayerForVideo()) {
    video_weblayer_.reset(
        new webkit::WebLayerImpl(cc::VideoLayer::Create(this)));
    client_->setWebLayer(video_weblayer_.get());
  }

  // force the grabbing of the hole punch frame
  client_->repaint();
}

void WebMediaPlayerGStreamer::pause() {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::pause()\n";

  if (data_source_)
    data_source_->MediaIsPaused();

  if (is_end_reached_)
    return;

  if (changePipelineState(GST_STATE_PAUSED))
    DVLOG(1) << ("Pause");
}

bool WebMediaPlayerGStreamer::supportsFullscreen() const {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer::supportsFullscreen()\n";

  return true;
}

bool WebMediaPlayerGStreamer::supportsSave() const {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::supportsSave()\n";

  return supports_save_;
}

void WebMediaPlayerGStreamer::seek(double seconds) {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::seek()\n";

  if (ready_state_ > WebMediaPlayer::ReadyStateHaveMetadata)
    SetReadyState(WebMediaPlayer::ReadyStateHaveMetadata);

  changePipelineState(GST_STATE_PAUSED);

  SetShouldBeReading(false, kVideo);
  SetShouldBeReading(false, kAudio);

  base::TimeDelta seek_time = ConvertSecondsToTimestamp(seconds);
  if (seeking_) {
    if (chunk_demuxer_)
      chunk_demuxer_->CancelPendingSeek(seek_time);
  }

  media_log_->AddEvent(media_log_->CreateSeekEvent(seconds));

  // cast to float to match webkit gstreamer code
  float time = static_cast<float>(seconds);

  if (!pipeline_)
    return;

  if (error_occured_)
    return;

  VLOG(kCallPrintDebugLevel) << StringPrintf("Seek attempt to %f secs\n", time);

  // Avoid useless seeking.
  if (time == currentTimeF())
    return;

  seeking_ = true;
  seek_time_ = time;

  if (chunk_demuxer_)
    chunk_demuxer_->StartWaitingForSeek(seek_time);

  // Extract the integer part of the time (seconds) and the
  // fractional part (microseconds). Attempt to round the
  // microseconds so no floating point precision is lost and we can
  // perform an accurate seek.
  float seconds_part;
  float micro_seconds = modff(time, &seconds_part) * 1000000;
  GTimeVal time_value;
  time_value.tv_sec = static_cast<glong>(seconds_part);
  time_value.tv_usec =
      static_cast<glong>(roundf(micro_seconds / 10000) * 10000);

  GstClockTime clock_time = GST_TIMEVAL_TO_TIME(time_value);
  VLOG(kCallPrintDebugLevel)
      << StringPrintf("Seek: %" GST_TIME_FORMAT, GST_TIME_ARGS(clock_time));

  // TODO(mzuber): chromium client_ doesn't have rate function
  // just using stored playback_rate_
  if (!gst_element_seek(
           pipeline_,
           playback_rate_,
           GST_FORMAT_TIME,
           (GstSeekFlags)(GST_SEEK_FLAG_FLUSH | GST_SEEK_FLAG_ACCURATE),
           GST_SEEK_TYPE_SET,
           clock_time,
           GST_SEEK_TYPE_NONE,
           GST_CLOCK_TIME_NONE)) {
  } else {
    SetNetworkState(WebMediaPlayer::NetworkStateLoading);
    main_loop_->PostTask(
        FROM_HERE,
        base::Bind(&WebMediaPlayerGStreamer::notifyPlayerSeekFinished,
                   AsWeakPtr()));
  }
}

void WebMediaPlayerGStreamer::setRate(double rate) {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::setRate()\n";

  /*
      // Avoid useless playback rate update.
      if (playback_rate_ == rate)
          return;

      GstState state;
      GstState pending;

      gst_element_get_state(play_bin_, &state, &pending, 0);
      if ((state != GST_STATE_PLAYING && state != GST_STATE_PAUSED)
          || (pending == GST_STATE_PAUSED))
          return;

      if (isLiveStream())
          return;

      playback_rate_ = rate;
      changing_rate_ = true;

      if (!rate) {
          gst_element_set_state(play_bin_, GST_STATE_PAUSED);
          return;
      }

      float currentPosition = static_cast<float>(playbackPosition() *
     GST_SECOND);
      GstSeekFlags flags = (GstSeekFlags)(GST_SEEK_FLAG_FLUSH);
      gint64 start, end;
      bool mute = false;

      LOG_MEDIA_MESSAGE("Set Rate to %f", rate);
      if (rate > 0) {
          // Mute the sound if the playback rate is too extreme.
          // TODO: in other cases we should perform pitch adjustments.
          mute = (bool) (rate < 0.8 || rate > 2);
          start = currentPosition;
          end = GST_CLOCK_TIME_NONE;
      } else {
          start = 0;
          mute = true;

          // If we are at beginning of media, start from the end to
          // avoid immediate EOS.
          if (currentPosition <= 0)
              end = static_cast<gint64>(duration() * GST_SECOND);
          else
              end = currentPosition;
      }

      LOG_MEDIA_MESSAGE("Need to mute audio: %d", (int) mute);

      if (!gst_element_seek(play_bin_, rate, GST_FORMAT_TIME, flags,
                            GST_SEEK_TYPE_SET, start,
                            GST_SEEK_TYPE_SET, end))
          LOG_MEDIA_MESSAGE("Set rate to %f failed", rate);
      else
          g_object_set(play_bin_, "mute", mute, NULL);
  */

  if (data_source_)
    data_source_->MediaPlaybackRateChanged(rate);
}

void WebMediaPlayerGStreamer::setVolume(double volume) {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::setVolume()\n";

  std::string volume_string = GetGStreamerString("audio-sink-volume");

  if (!audio_sink_)
    return;
  g_object_set(audio_sink_, volume_string.c_str(), volume, NULL);
}

void WebMediaPlayerGStreamer::setPreload(
    blink::WebMediaPlayer::Preload preload) {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::setPreload()\n";
  /*
      original_preload_was_auto_and_was_overridden_ = preload_ != preload_ &&
     preload_ == MediaPlayer::Auto;

      preload_ = preload_;

      ASSERT(play_bin_);

      GstPlayFlags flags;
      g_object_get(play_bin_, "flags", &flags, NULL);
      if (preload_ == MediaPlayer::Auto) {
          LOG_MEDIA_MESSAGE("Enabling on-disk buffering");
          g_object_set(play_bin_, "flags", flags | GST_PLAY_FLAG_DOWNLOAD,
     NULL);
      } else {
          LOG_MEDIA_MESSAGE("Disabling on-disk buffering");
          g_object_set(play_bin_, "flags", flags & ~GST_PLAY_FLAG_DOWNLOAD,
     NULL);
      }

      if (delaying_load_ && preload_ != MediaPlayer::None) {
          delaying_load_ = false;
          commitLoad();
      }
  */

  if (data_source_)
    data_source_->SetPreload(static_cast<content::Preload>(preload));
}

TimeDelta WebMediaPlayerGStreamer::TimeForByteOffset(int64 byte_offset) const {
  DCHECK(main_loop_->BelongsToCurrentThread());
  // Use floating point to avoid potential overflow when using 64 bit integers.
  TimeDelta duration_val =
      TimeDelta::FromMilliseconds(static_cast<int64>(duration() * 1000.0));
  int64 total_bytes = demuxer_host_->GetTotalBytes();

  double time_offset_in_ms = duration_val.InMilliseconds() *
                             (static_cast<double>(byte_offset) / total_bytes);
  TimeDelta time_offset(
      TimeDelta::FromMilliseconds(static_cast<int64>(time_offset_in_ms)));
  // Since the byte->time calculation is approximate, fudge the beginning &
  // ending areas to look better.
  TimeDelta epsilon = duration_val / 100;
  if (time_offset < epsilon)
    return TimeDelta();
  if (time_offset + epsilon > duration_val)
    return duration_val;
  return time_offset;
}

media::Ranges<base::TimeDelta>
WebMediaPlayerGStreamer::GetBufferedTimeRanges() {
  DCHECK(main_loop_->BelongsToCurrentThread());
  media::Ranges<TimeDelta> time_ranges;

  if (demuxer_host_) {
    int64 total_bytes = totalBytes();
    const media::Ranges<int64>& buffered_byte_ranges =
        demuxer_host_->GetBufferedByteRanges();
    const media::Ranges<base::TimeDelta>& buffered_time_ranges =
        demuxer_host_->GetBufferedTimeRanges();

    for (size_t i = 0; i < buffered_time_ranges.size(); ++i) {
      time_ranges.Add(buffered_time_ranges.start(i),
                      buffered_time_ranges.end(i));
    }
    if (duration() == TimeDelta().InSecondsF() || total_bytes == 0)
      return time_ranges;
    for (size_t i = 0; i < buffered_byte_ranges.size(); ++i) {
      TimeDelta start = TimeForByteOffset(buffered_byte_ranges.start(i));
      TimeDelta end = TimeForByteOffset(buffered_byte_ranges.end(i));
      // Cap approximated buffered time at the length of the video.
      end = std::min(
          end,
          TimeDelta::FromMilliseconds(static_cast<int64>(duration() * 1000.0)));
      time_ranges.Add(start, end);
    }
  }

  return time_ranges;
}

const blink::WebTimeRanges& WebMediaPlayerGStreamer::buffered() {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::buffered()\n";

  blink::WebTimeRanges web_ranges(
      ConvertToWebTimeRanges(GetBufferedTimeRanges()));
  buffered_.swap(web_ranges);
  return buffered_;
}

float WebMediaPlayerGStreamer::maxTimeSeekableF() const {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::maxTimeSeekable()\n";

  if (error_occured_)
    return 0.0f;

  DVLOG(1) << ("maxTimeSeekable");
  // infinite duration means live stream
  if (std::isinf(durationF()))
    return 0.0f;

  // If we haven't even gotten to ReadyStateHaveMetadata yet then just
  // return 0 so that the seekable range is empty.
  if (ready_state_ < WebMediaPlayer::ReadyStateHaveMetadata)
    return 0.0;

  // We don't support seeking in streaming media.
  if (data_source_ && data_source_->IsStreaming())
    return 0.0;

  return durationF();
}

double WebMediaPlayerGStreamer::maxTimeSeekable() const {
  DCHECK(main_loop_->BelongsToCurrentThread());
  return maxTimeSeekableF();
}

// Methods for painting.
void WebMediaPlayerGStreamer::paint(blink::WebCanvas* canvas,
                                    const blink::WebRect& rect,
                                    unsigned char alpha) {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel)
      << StringPrintf("WebMediaPlayerGStreamer::paint(), (%d,%d)\n",
                      rect.width,
                      rect.height);

  // paint isn't used when using hardware accelerated rendering
}

// True if the loaded media has a playable video/audio track.
bool WebMediaPlayerGStreamer::hasVideo() const {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::hasVideo()\n";
  return has_video_;
}
bool WebMediaPlayerGStreamer::hasAudio() const {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::hasAudio()\n";
  return has_audio_;
}

// Dimensions of the video.
blink::WebSize WebMediaPlayerGStreamer::naturalSize() const {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::naturalSize()\n";

#if 0 /*!BCM_QTWEBKIT*/
  if (!hasVideo())
    return IntSize();

  if (!video_size_.isEmpty())
    return video_size_;

  GRefPtr<GstCaps> caps = webkitGstGetPadCaps(m_videoSinkPad.get());
  if (!caps)
    return IntSize();

  // TODO(webkit_dev): handle possible clean aperture data. See
  // https://bugzilla.gnome.org/show_bug.cgi?id=596571
  // TODO(webkit_dev): handle possible transformation matrix. See
  // https://bugzilla.gnome.org/show_bug.cgi?id=596326

  // Get the video PAR and original size, if this fails the
  // video-sink has likely not yet negotiated its caps.
  int pixelAspectRatioNumerator, pixelAspectRatioDenominator, stride;
  IntSize originalSize;
  GstVideoFormat format;
  if (!getVideoSizeAndFormatFromCaps(caps.get(),
                                     originalSize,
                                     format,
                                     pixelAspectRatioNumerator,
                                     pixelAspectRatioDenominator,
                                     stride))
    return IntSize();

  LOG_MEDIA_MESSAGE("Original video size: %dx%d",
                    originalSize.width(),
                    originalSize.height());
  LOG_MEDIA_MESSAGE("Pixel aspect ratio: %d/%d",
                    pixelAspectRatioNumerator,
                    pixelAspectRatioDenominator);

  // Calculate DAR based on PAR and video size.
  int displayWidth = originalSize.width() * pixelAspectRatioNumerator;
  int displayHeight = originalSize.height() * pixelAspectRatioDenominator;

  // Divide display width and height by their GCD to avoid possible overflows.
  int displayAspectRatioGCD =
      greatestCommonDivisor(displayWidth, displayHeight);
  displayWidth /= displayAspectRatioGCD;
  displayHeight /= displayAspectRatioGCD;

  // Apply DAR to original video size. This is the same behavior as in
  // xvimagesink's setcaps function.
  guint64 width = 0, height = 0;
  if (!(originalSize.height() % displayHeight)) {
    LOG_MEDIA_MESSAGE("Keeping video original height");
    width = gst_util_uint64_scale_int(
        originalSize.height(), displayWidth, displayHeight);
    height = static_cast<guint64>(originalSize.height());
  } else if (!(originalSize.width() % displayWidth)) {
    LOG_MEDIA_MESSAGE("Keeping video original width");
    height = gst_util_uint64_scale_int(
        originalSize.width(), displayHeight, displayWidth);
    width = static_cast<guint64>(originalSize.width());
  } else {
    LOG_MEDIA_MESSAGE("Approximating while keeping original video height");
    width = gst_util_uint64_scale_int(
        originalSize.height(), displayWidth, displayHeight);
    height = static_cast<guint64>(originalSize.height());
  }

  LOG_MEDIA_MESSAGE(
      "Natural size: %" G_GUINT64_FORMAT "x%" G_GUINT64_FORMAT, width, height);
  video_size_ = IntSize(static_cast<int>(width), static_cast<int>(height));
  return video_size_;
#else
  // int width = 320;
  // int height = 176;
  int width = 640;
  int height = 360;
  return blink::WebSize(width, height);
#endif
}

// Getters of playback state.
bool WebMediaPlayerGStreamer::paused() const {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::paused()\n";

  if (is_end_reached_) {
    DVLOG(1) << ("Ignoring pause at EOS");
    return true;
  }

  if (pipeline_ == NULL) {
    DVLOG(1) << ("pipeline_==NULL");
    return true;
  }

  GstState state;

  gst_element_get_state(pipeline_, &state, 0, 0);
  return state == GST_STATE_PAUSED;
}

bool WebMediaPlayerGStreamer::seeking() const {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::seeking()\n";

  return seeking_;
}

float WebMediaPlayerGStreamer::durationF() const {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::duration()\n";

  if (!pipeline_)
    return 0.0f;

  if (error_occured_)
    return 0.0f;

  // Media duration query failed already, don't attempt new useless queries.
  if (!media_duration_known_)
    return std::numeric_limits<float>::infinity();

  if (media_duration_)
    return media_duration_;

  if (load_type_ == LoadTypeMediaSource)
    return demuxer_host_->GetDuration().InSecondsF();

  GstFormat time_format = GST_FORMAT_TIME;
  gint64 time_length = 0;

#ifdef GST_API_VERSION_1
  bool failure =
      !gst_element_query_duration(pipeline_, timeFormat, &timeLength) ||
      static_cast<guint64>(timeLength) == GST_CLOCK_TIME_NONE;
#else
  bool failure =
      !gst_element_query_duration(pipeline_, &time_format, &time_length) ||
      time_format != GST_FORMAT_TIME ||
      static_cast<guint64>(time_length) == GST_CLOCK_TIME_NONE;
#endif
  if (failure) {
    DVLOG(1) << StringPrintf("Time duration query failed for %s",
                             url_.string().utf8().data());
    return std::numeric_limits<float>::infinity();
  }

  VLOG(kCallPrintDebugLevel) << StringPrintf("Duration: %" GST_TIME_FORMAT,
                                             GST_TIME_ARGS(time_length));

  return static_cast<double>(time_length) / GST_SECOND;
  // FIXME: handle 3.14.9.5 properly
}

double WebMediaPlayerGStreamer::duration() const { return durationF(); }

float WebMediaPlayerGStreamer::currentTimeF() const {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel + 5) << "WebMediaPlayerGStreamer::currentTime()\n";

  if (!pipeline_)
    return 0.0f;

  if (error_occured_)
    return 0.0f;

  if (seeking_)
    return seek_time_;

  // Workaround for
  // https://bugzilla.gnome.org/show_bug.cgi?id=639941 In GStreamer
  // 0.10.35 basesink reports wrong duration in case of EOS and
  // negative playback rate. There's no upstream accepted patch for
  // this bug yet, hence this temporary workaround.
  if (is_end_reached_ && playback_rate_ < 0)
    return 0.0f;

  float position = playbackPosition();

  return position;
}

double WebMediaPlayerGStreamer::currentTime() const {
  DCHECK(main_loop_->BelongsToCurrentThread());
  return currentTimeF();
}

// Internal states of loading and network.
blink::WebMediaPlayer::NetworkState WebMediaPlayerGStreamer::networkState()
    const {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel)
      << StringPrintf("WebMediaPlayerGStreamer::networkState() :%d\n",
                      network_state_);
  return network_state_;
}
blink::WebMediaPlayer::ReadyState WebMediaPlayerGStreamer::readyState() const {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel)
      << StringPrintf("WebMediaPlayerGStreamer::readyState() :%d\n",
                      ready_state_);
  return ready_state_;
}

bool WebMediaPlayerGStreamer::didLoadingProgress() const {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer::didLoadingProgress()\n";
  VLOG(kCallPrintDebugLevel) << StringPrintf(
                                    "progress vals: pipeline:%p, "
                                    "media_duration_:%f, totalBytes:%lld\n",
                                    pipeline_,
                                    media_duration_,
                                    totalBytes());

  bool data_loaded = false;
  if (load_type_ == LoadTypeMediaSource)
    data_loaded = demuxer_host_->GetDuration().InSecondsF() > 0;
  else
    data_loaded = totalBytes() > 0;

  if (!pipeline_ || !media_duration_ || !data_loaded)
    return false;
  float current_max_time_loaded = maxTimeLoaded();
  bool did_loading_progress =
      current_max_time_loaded != max_time_loaded_at_last_did_loading_progress_;
  max_time_loaded_at_last_did_loading_progress_ = current_max_time_loaded;
  DVLOG(1) << StringPrintf("didLoadingProgress: %d", did_loading_progress);
  return did_loading_progress;
}

bool WebMediaPlayerGStreamer::hasSingleSecurityOrigin() const {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer::hasSingleSecurityOrigin()\n";

  if (data_source_)
    return data_source_->HasSingleOrigin();
  return true;
}

bool WebMediaPlayerGStreamer::didPassCORSAccessCheck() const {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer::didPassCORSAccessCheck()\n";

  if (data_source_)
    return data_source_->DidPassCORSAccessCheck();
  return false;
}

double WebMediaPlayerGStreamer::mediaTimeForTimeValue(double time_value) const {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer::mediaTimeForTimeValue()\n";
  return ConvertSecondsToTimestamp(time_value).InSecondsF();
}

unsigned WebMediaPlayerGStreamer::decodedFrameCount() const {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer::decodedFrameCount()\n";

  guint64 decoded_frames = 0;
  if (fps_sink_)
    g_object_get(fps_sink_, "frames-rendered", &decoded_frames, NULL);
  return static_cast<unsigned>(decoded_frames);
}
unsigned WebMediaPlayerGStreamer::droppedFrameCount() const {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer::droppedFrameCount()\n";

  guint64 frames_dropped = 0;
  if (fps_sink_)
    g_object_get(fps_sink_, "frames-dropped", &frames_dropped, NULL);
  return static_cast<unsigned>(frames_dropped);
}
unsigned WebMediaPlayerGStreamer::audioDecodedByteCount() const {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer::audioDecodedByteCount()\n";

  GstQuery* query = gst_query_new_position(GST_FORMAT_BYTES);
  gint64 position = 0;

  if (webkit_audio_sink_ && gst_element_query(webkit_audio_sink_, query))
    gst_query_parse_position(query, 0, &position);

  gst_query_unref(query);
  return static_cast<unsigned>(position);
}
unsigned WebMediaPlayerGStreamer::videoDecodedByteCount() const {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer::videoDecodedByteCount()\n";

  GstQuery* query = gst_query_new_position(GST_FORMAT_BYTES);
  gint64 position = 0;

  if (gst_element_query(webkit_video_sink_, query))
    gst_query_parse_position(query, 0, &position);

  gst_query_unref(query);
  return static_cast<unsigned>(position);
}

void WebMediaPlayerGStreamer::SetVideoFrameProviderClient(
    cc::VideoFrameProvider::Client* client) {
  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer::SetVideoFrameProviderClient()\n";

  if (video_frame_provider_client_)
    video_frame_provider_client_->StopUsingProvider();
  video_frame_provider_client_ = client;
}

scoped_refptr<media::VideoFrame> WebMediaPlayerGStreamer::GetCurrentFrame() {
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::GetCurrentFrame()\n";

  // this is called before a frame is drawn when compositing is on
  // this is essentially the hook that replaces paint() when
  // compositing is on, once a video layer is set, paint() will
  // not be called, for us the frame is a simple hole punching
  // frame

  VLOG(kCallPrintDebugLevel)
      << StringPrintf(
             "video layer position and size,(%s,%s)\n",
             gfx::PointF(video_weblayer_->position()).ToString().c_str(),
             gfx::Size(video_weblayer_->bounds()).ToString().c_str());

  // get the video layer's screen coordinate transformation matrix
  const gfx::Transform& transform =
      video_weblayer_->layer()->screen_space_transform();

  gfx::RectF bounds_rect(video_weblayer_->layer()->bounds());
  // gfx::PointF position(video_weblayer_->layer()->position());
  // gfx::Point posi(position.x(),position.y());

  // transform the video layer position and bounds
  // to screen coordinates
  transform.TransformRect(&bounds_rect);
  // transform.TransformPoint(&posi);

  VLOG(kCallPrintDebugLevel)
      << StringPrintf("screen rect:%s\n", bounds_rect.ToString().c_str());

  // tell gstreamer where to place the video and what size
  // WebKit::WebRect
  // screenRect(posi.x(),posi.y(),bounds_rect.width(),bounds_rect.height());
  blink::WebRect screen_rect(bounds_rect.x(),
                             bounds_rect.y(),
                             bounds_rect.width(),
                             bounds_rect.height());
  setScreenRect(screen_rect);

  return current_frame_;
}

void WebMediaPlayerGStreamer::PutCurrentFrame(
    const scoped_refptr<media::VideoFrame>& frame) {
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::PutCurrentFrame()\n";

  // client_->repaint();
}

// As we are closing the tab or even the browser, |main_loop_| is destroyed
// even before this object gets destructed, so we need to know when
// |main_loop_| is being destroyed and we can stop posting repaint task
// to it.
void WebMediaPlayerGStreamer::OnDestruct() {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer:WillDestoryCurrentMessageLoop()\n";

  Destroy();
}

blink::WebMediaPlayerClient* WebMediaPlayerGStreamer::GetClient() {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::GetClient()\n";
  DCHECK(main_loop_->BelongsToCurrentThread());
  DCHECK(client_);
  return client_;
}


gboolean WebMediaPlayerGStreamer::handleMessageMultiThreadedThread(
    GstMessage* message) {

  // this is called on the gmainloop thread

  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer::handleMessageMultiThreadedThread\n";

  // ref the message so it isn't deleted before we post to the render main loop
  addToMessageMap(message);

  // do the actual handling on the main thread
  // according to the message loop documentation in the chromium code
  // messages are processed FIFO inter-mixed with normal IO and UI updates
  main_loop_->PostTask(
      FROM_HERE,
      base::Bind(&WebMediaPlayerGStreamer::handleMessageMultiThreadedMain,
                 AsWeakPtr(),
                 message));

  return TRUE;
}

void WebMediaPlayerGStreamer::handleMessageMultiThreadedMain(
    GstMessage* message) {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer::handleMessageMultiThreadedMain";

  // we are now in the main thread, so handle the gstreamer message like normal
  handleMessage(message);

  // unref the extra ref we added to keep the message
  removeFromMessageMap(message);
}

gboolean WebMediaPlayerGStreamer::handleMessage(GstMessage* message) {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::handleMessage()\n";

  GError* err = NULL;
  gchar* debug = NULL;
  WebMediaPlayer::NetworkState error;
  bool issue_error = true;
  bool attempt_next_location = false;
  const GstStructure* structure = gst_message_get_structure(message);

  if (structure) {
    //        const gchar* messageTypeName = gst_structure_get_name(structure);

    /* BCM: To avoid gst_structure_get_name that cause crash in Multi-Process
     * environment */
    // const gchar* messageTypeName =
    // gst_structure_get_name(message->structure);
    const gchar* message_type_name = "";

    // Redirect messages are sent from elements, like qtdemux, to
    // notify of the new location(s) of the media.
    if (!g_strcmp0(message_type_name, "redirect")) {
      mediaLocationChanged(message);
      return TRUE;
    }
  }

  DVLOG(1) << "Message received from element " << GST_MESSAGE_SRC_NAME(message);
  switch (GST_MESSAGE_TYPE(message)) {
    case GST_MESSAGE_ERROR:
      DVLOG(1) << "GST_MESSAGE_ERROR\n";
      if (reset_pipeline_)
        break;
      gst_message_parse_error(message, &err, &debug);
      // LOG_MEDIA_MESSAGE("Error %d: %s (url=%s)", err->code, err->message,
      // url_.string().utf8().data());
      DVLOG(1) << StringPrintf("Error %d: %s (url=%s)",
                               err->code,
                               err->message,
                               url_.string().utf8().data());

      GST_DEBUG_BIN_TO_DOT_FILE_WITH_TS(
          GST_BIN(pipeline_), GST_DEBUG_GRAPH_SHOW_ALL, "webkit-video.error");

      error = WebMediaPlayer::NetworkStateEmpty;
      if (err->code == GST_STREAM_ERROR_CODEC_NOT_FOUND ||
          err->code == GST_STREAM_ERROR_WRONG_TYPE ||
          err->code == GST_STREAM_ERROR_FAILED ||
          err->code == GST_CORE_ERROR_MISSING_PLUGIN ||
          err->code == GST_RESOURCE_ERROR_NOT_FOUND)
        error = WebMediaPlayer::NetworkStateFormatError;
      else if (err->domain == GST_STREAM_ERROR) {
        // Let the mediaPlayerClient handle the stream error, in
        // this case the HTMLMediaElement will emit a stalled
        // event.
        if (err->code == GST_STREAM_ERROR_TYPE_NOT_FOUND) {
          DVLOG(1)
              << "Decode error, let the Media element emit a stalled event.";
          break;
        }
        error = WebMediaPlayer::NetworkStateDecodeError;
        attempt_next_location = true;
      } else if (err->domain == GST_RESOURCE_ERROR) {
        error = WebMediaPlayer::NetworkStateNetworkError;
      }

      if (attempt_next_location)
        issue_error = !loadNextLocation();
      if (issue_error)
        loadingFailed(error);
      break;
    case GST_MESSAGE_EOS:
      DVLOG(1) << "GST_MESSAGE_EOS\n";
      DVLOG(1) << "End of Stream";
      didEnd();
      break;
    case GST_MESSAGE_STATE_CHANGED:
      // Ignore state changes if load is delayed (preload_=none). The
      // player state will be updated once commitLoad() is called.
      if (delaying_load_) {
        DVLOG(1)
            << "Media load has been delayed. Ignoring state changes for now";
        break;
      }

      // Ignore state changes from internal elements. They are
      // forwarded to playbin2 anyway.
      if (GST_MESSAGE_SRC(message) == reinterpret_cast<GstObject*>(pipeline_)) {
        DVLOG(1) << "GST_MESSAGE_STATE_CHANGED for pipeline_\n";
        updateStates();

        // Construct a filename for the graphviz dot file output.
        GstState old_state, new_state;
        gst_message_parse_state_changed(message, &old_state, &new_state, 0);

        std::string dot_file_name =
            StringPrintf("webkit-video.%s_%s",
                         gst_element_state_get_name(old_state),
                         gst_element_state_get_name(new_state));

        GST_DEBUG_BIN_TO_DOT_FILE_WITH_TS(
            GST_BIN(pipeline_), GST_DEBUG_GRAPH_SHOW_ALL, dot_file_name.data());
      }
      break;
    case GST_MESSAGE_BUFFERING:
      DVLOG(1) << "GST_MESSAGE_BUFFERING\n";
      processBufferingStats(message);
      break;
#ifdef GST_API_VERSION_1
    case GST_MESSAGE_DURATION_CHANGED:
#else
    case GST_MESSAGE_DURATION:
#endif
      DVLOG(1) << "GST_MESSAGE_DURATION\n";
      DVLOG(1) << "Duration changed\n";
      durationChanged();
      break;
    default:
      DVLOG(1) << StringPrintf("GST_MESSAGE - Unknown '%s'\n",
                               GST_MESSAGE_TYPE_NAME(message));
      DVLOG(1) << StringPrintf("Unhandled GStreamer message type: %s",
                               GST_MESSAGE_TYPE_NAME(message));
      break;
  }

  if (err)
    g_error_free(err);
  if (debug)
    g_free(debug);

  return TRUE;
}

void WebMediaPlayerGStreamer::mediaLocationChanged(GstMessage* message) {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer::mediaLocationChanged()\n";
  if (media_locations_)
    gst_structure_free(media_locations_);

  const GstStructure* structure = gst_message_get_structure(message);
  if (structure) {
    // This structure can contain:
    // - both a new-location string and embedded locations structure
    // - or only a new-location string.
    media_locations_ = gst_structure_copy(structure);
    const GValue* locations =
        gst_structure_get_value(media_locations_, "locations");

    if (locations)
      media_location_current_index_ =
          static_cast<int>(gst_value_list_get_size(locations)) - 1;

    loadNextLocation();
  }
}

bool WebMediaPlayerGStreamer::loadNextLocation() {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::loadNextLocation()\n";
  if (!media_locations_)
    return false;

  const GValue* locations =
      gst_structure_get_value(media_locations_, "locations");
  const gchar* new_location = 0;

  if (!locations) {
    // Fallback on new-location string.
    new_location = gst_structure_get_string(media_locations_, "new-location");
    if (!new_location)
      return false;
  }

  if (!new_location) {
    if (media_location_current_index_ < 0) {
      media_locations_ = 0;
      return false;
    }

    const GValue* location =
        gst_value_list_get_value(locations, media_location_current_index_);
    const GstStructure* structure = gst_value_get_structure(location);

    if (!structure) {
      media_location_current_index_--;
      return false;
    }

    new_location = gst_structure_get_string(structure, "new-location");
  }

  if (new_location) {
    // Found a candidate. new-location is not always an absolute url
    // though. We need to take the base of the current url and
    // append the value of new-location to it.

    gchar* current_location = 0;
    g_object_get(pipeline_, "uri", &current_location, NULL);

    blink::WebURL currentUrl(GURL(current_location));
    g_free(current_location);

    blink::WebURL new_url;

    if (gst_uri_is_valid(new_location)) {
      new_url = blink::WebURL(GURL(new_location));
    } else {
      VLOG(kCallPrintDebugLevel) << "TODO implement this change location code "
                                    "using chromium classes!\n";
      // TODO(mzuber): Implement this location change code using chromium
      // classes
      // baseAsString function doesn't exist in chromium url classes
      // newUrl = WebKit::WebURL(currentUrl.baseAsString() + newLocation);
    }

    // RefPtr<SecurityOrigin> securityOrigin =
    // SecurityOrigin::create(currentUrl);
    WebSecurityOrigin security_origin = WebSecurityOrigin::create(new_url);
    if (security_origin.canRequest(new_url)) {
      DVLOG(1) << StringPrintf("New media url: %s",
                               new_url.string().utf8().data());

      // Reset player states.
      network_state_ = WebMediaPlayer::NetworkStateLoading;
      client_->networkStateChanged();
      ready_state_ = WebMediaPlayer::ReadyStateHaveNothing;
      client_->readyStateChanged();

      // Reset pipeline state.
      reset_pipeline_ = true;
      gst_element_set_state(pipeline_, GST_STATE_READY);

      GstState state;
      gst_element_get_state(pipeline_, &state, 0, 0);
      if (state <= GST_STATE_READY) {
        // Set the new uri and start playing.
        g_object_set(G_OBJECT(source_),
                     "location",
                     new_url.string().utf8().data(),
                     NULL);
        gst_element_set_state(pipeline_, GST_STATE_PLAYING);
        return true;
      }
    }
  }
  media_location_current_index_--;
  return false;
}

void WebMediaPlayerGStreamer::loadingFailed(
    WebMediaPlayer::NetworkState error) {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::loadingFailed()\n";
  error_occured_ = true;
  if (network_state_ != error) {
    network_state_ = error;
    client_->networkStateChanged();
  }
  if (ready_state_ != WebMediaPlayer::ReadyStateHaveNothing) {
    ready_state_ = WebMediaPlayer::ReadyStateHaveNothing;
    client_->readyStateChanged();
  }
}

void WebMediaPlayerGStreamer::didEnd() {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::didEnd()\n";

  // Synchronize position and duration values to not confuse the
  // HTMLMediaElement. In some cases like reverse playback the
  // position is not always reported as 0 for instance.
  float now = std::min(currentTimeF(), durationF());
  if (now > 0 && now <= durationF() && media_duration_ != now) {
    media_duration_known_ = true;
    media_duration_ = now;
    DVLOG(1) << StringPrintf(
                    "chz media_duration_=%f  calling durationChanged\n",
                    media_duration_);
    client_->durationChanged();
  }

  is_end_reached_ = true;
  timeChanged();

  // TODO(mzuber): How do I tell if video is looping in chromium?
  // if (!m_player->mediaPlayerClient()->mediaPlayerIsLooping()) {
  // paused_ = true;
  // gst_element_set_state(pipeline_, GST_STATE_NULL);
  // }

  seek_time_ = 0.0f;
}

void WebMediaPlayerGStreamer::updateStates() {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::updateStates()\n";

  if (!pipeline_)
    return;

  if (error_occured_)
    return;

  WebMediaPlayer::NetworkState oldNetworkState = network_state_;
  WebMediaPlayer::ReadyState oldReadyState = ready_state_;
  GstState state;
  GstState pending;

  GstStateChangeReturn ret =
      gst_element_get_state(pipeline_, &state, &pending, 250 * GST_NSECOND);

  bool should_update_after_seek = false;
  switch (ret) {
    case GST_STATE_CHANGE_SUCCESS:
      DVLOG(1) << StringPrintf("State: %s, pending: %s",
                               gst_element_state_get_name(state),
                               gst_element_state_get_name(pending));

      reset_pipeline_ = state <= GST_STATE_READY;

      // Try to figure out ready and network states.
      if (state == GST_STATE_READY) {
        DVLOG(1) << ("===== GST_STATE_CHANGE_SUCCESS: GST_STATE_READY\n");
        ready_state_ = WebMediaPlayer::ReadyStateHaveMetadata;
        network_state_ = WebMediaPlayer::NetworkStateEmpty;
        // Cache the duration without emiting the durationchange
        // event because it's taken care of by the media element
        // in this precise case.
        if (!is_end_reached_)
          cacheDuration();
      } else if ((state == GST_STATE_NULL) ||
                 (maxTimeLoaded() == durationF())) {
        network_state_ = WebMediaPlayer::NetworkStateLoaded;
        ready_state_ = WebMediaPlayer::ReadyStateHaveEnoughData;
      } else {
        ready_state_ = currentTimeF() < maxTimeLoaded()
                           ? WebMediaPlayer::ReadyStateHaveFutureData
                           : WebMediaPlayer::ReadyStateHaveCurrentData;
        network_state_ = WebMediaPlayer::NetworkStateLoading;
      }

      if (buffering_ && state != GST_STATE_READY) {
        ready_state_ = WebMediaPlayer::ReadyStateHaveCurrentData;
        network_state_ = WebMediaPlayer::NetworkStateLoading;
      }

      // Now let's try to get the states in more detail using
      // information from GStreamer, while we sync states where
      // needed.
      if (state == GST_STATE_PAUSED) {
        DVLOG(1) << ("===== GST_STATE_CHANGE_SUCCESS: GST_STATE_PAUSED\n");
        if (!webkit_audio_sink_)
          updateAudioSink();
        if (buffering_ && buffering_percentage_ == 100) {
          buffering_ = false;
          buffering_percentage_ = 0;
          ready_state_ = WebMediaPlayer::ReadyStateHaveEnoughData;

          DVLOG(1) << ("[Buffering] Complete.");

          if (!paused_) {
            DVLOG(1) << ("[Buffering] Restarting playback.");
            gst_element_set_state(pipeline_, GST_STATE_PLAYING);
          }
        } else if (!buffering_ && (currentTimeF() < durationF())) {
          paused_ = true;
        }
      } else if (state == GST_STATE_PLAYING) {
        DVLOG(1) << ("===== GST_STATE_CHANGE_SUCCESS: GST_STATE_PLAYING\n");
        ready_state_ = WebMediaPlayer::ReadyStateHaveEnoughData;
        paused_ = false;

        if (buffering_ && !isLiveStream()) {
          ready_state_ = WebMediaPlayer::ReadyStateHaveCurrentData;
          network_state_ = WebMediaPlayer::NetworkStateLoading;

          DVLOG(1) << ("[Buffering] Pausing stream for buffering.");

          gst_element_set_state(pipeline_, GST_STATE_PAUSED);
        }
      } else {
        DVLOG(1) << ("===== GST_STATE_CHANGE_SUCCESS: Unknown\n");
        paused_ = true;
      }

      // Is on-disk buffering in progress?
      if (fill_timer_.IsRunning())
        network_state_ = WebMediaPlayer::NetworkStateLoading;

      if (changing_rate_) {
        // TODO(mzuber): chromium media client doesn't have rateChange, is this
        // a problem?
        // client_->rateChanged();
        changing_rate_ = false;
      }

      //if (seeking_) {
      //  should_update_after_seek = true;
      //  seeking_ = false;
      //}

      break;
    case GST_STATE_CHANGE_ASYNC:
      DVLOG(1) << ("===== GST_STATE_CHANGE_ASYNC\n");
      DVLOG(1) << StringPrintf("Async: State: %s, pending: %s",
                               gst_element_state_get_name(state),
                               gst_element_state_get_name(pending));
      // Change in progress

      // On-disk buffering was attempted but the media is live. This
      // can't work so disable on-disk buffering and reset the
      // pipeline.
      if (state == GST_STATE_READY && isLiveStream() &&
          preload_ == WebMediaPlayer::PreloadAuto) {
        setPreload(WebMediaPlayer::PreloadNone);
        gst_element_set_state(pipeline_, GST_STATE_NULL);
        gst_element_set_state(pipeline_, GST_STATE_PAUSED);
      }

      // A live stream was paused, reset the pipeline.
      if (state == GST_STATE_PAUSED && pending == GST_STATE_PLAYING &&
          isLiveStream()) {
        gst_element_set_state(pipeline_, GST_STATE_NULL);
        gst_element_set_state(pipeline_, GST_STATE_PLAYING);
      }

      if (!isLiveStream() && !buffering_)
        return;

      //if (seeking_) {
      //  should_update_after_seek = true;
      //  seeking_ = false;
      //}
      break;
    case GST_STATE_CHANGE_FAILURE:
      DVLOG(1) << ("===== GST_STATE_CHANGE_FAILURE\n");
      DVLOG(1) << StringPrintf("Failure: State: %s, pending: %s",
                               gst_element_state_get_name(state),
                               gst_element_state_get_name(pending));
      // Change failed
      return;
    case GST_STATE_CHANGE_NO_PREROLL:
      DVLOG(1) << ("===== GST_STATE_CHANGE_NO_PREROLL\n");
      DVLOG(1) << StringPrintf("No preroll: State: %s, pending: %s",
                               gst_element_state_get_name(state),
                               gst_element_state_get_name(pending));

      if (state == GST_STATE_READY)
        ready_state_ = WebMediaPlayer::ReadyStateHaveNothing;
      else if (state == GST_STATE_PAUSED) {
        ready_state_ = WebMediaPlayer::ReadyStateHaveEnoughData;
        paused_ = true;
        // Live pipelines go in PAUSED without prerolling.
        // is_streaming_ = true;
      } else if (state == GST_STATE_PLAYING) {
        paused_ = false;
      }

      //if (seeking_) {
      //  should_update_after_seek = true;
      //  seeking_ = false;
      //  if (!paused_) {
      //    gst_element_set_state(pipeline_, GST_STATE_PLAYING);
      //  }
      //} else
      if (!paused_) {
        gst_element_set_state(pipeline_, GST_STATE_PLAYING);
      }

      network_state_ = WebMediaPlayer::NetworkStateLoading;
      break;
    default:
      DVLOG(1) << ("===== Unknown\n");
      DVLOG(1) << StringPrintf("Else : %d", ret);
      break;
  }

  //if (seeking())
  //  ready_state_ = WebMediaPlayer::ReadyStateHaveNothing;

  if (should_update_after_seek)
    timeChanged();

  if (network_state_ != oldNetworkState) {
    DVLOG(1) << StringPrintf("Network State Changed from %u to %u",
                             oldNetworkState,
                             network_state_);
    client_->networkStateChanged();
  }
  if (ready_state_ != oldReadyState) {
    DVLOG(1) << StringPrintf("Ready State Changed from %u to %u",
                             oldReadyState,
                             ready_state_);
    client_->readyStateChanged();
  }
}

void WebMediaPlayerGStreamer::processBufferingStats(GstMessage* message) {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer::processBufferingStats()\n";

  // This is the immediate buffering that needs to happen so we have
  // enough to play right now.
  buffering_ = true;
  const GstStructure* structure = gst_message_get_structure(message);
  gst_structure_get_int(structure, "buffer-percent", &buffering_percentage_);

  DVLOG(1) << StringPrintf("[Buffering] Buffering: %d%%.",
                           buffering_percentage_);

  GstBufferingMode mode;
  gst_message_parse_buffering_stats(message, &mode, 0, 0, 0);
  if (mode != GST_BUFFERING_DOWNLOAD) {
    updateStates();
    return;
  }

  // This is on-disk buffering, that allows us to download much more
  // than needed for right now.
  if (!started_buffering_) {
    DVLOG(1) << ("[Buffering] Starting on-disk buffering.");

    started_buffering_ = true;

    if (fill_timer_.IsRunning())
      fill_timer_.Stop();

    fill_timer_.Start(FROM_HERE,
                      TimeDelta::FromMilliseconds(200),
                      this,
                      &WebMediaPlayerGStreamer::fillTimerFired);
  }
}

void WebMediaPlayerGStreamer::durationChanged() {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::durationChanged()\n";

  float previousDuration = media_duration_;

  cacheDuration();
  // Avoid emiting durationchanged in the case where the previous
  // duration was 0 because that case is already handled by the
  // HTMLMediaElement.
  if (previousDuration && media_duration_ != previousDuration)
    client_->durationChanged();

  if (preload_ == WebMediaPlayer::PreloadNone &&
      original_preload_was_auto_and_was_overridden_) {
    // total_bytes_ = -1;
    if (totalBytes() && !isLiveStream()) {
      setPreload(WebMediaPlayer::PreloadAuto);
      gst_element_set_state(pipeline_, GST_STATE_NULL);
      gst_element_set_state(pipeline_, GST_STATE_PAUSED);
    }
  }
}

void WebMediaPlayerGStreamer::timeChanged() {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::timeChanged()\n";

  updateStates();
  client_->timeChanged();
}

void WebMediaPlayerGStreamer::cacheDuration() {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::cacheDuration()\n";
  /*
      // Reset cached media duration
      media_duration_ = 0;

      // And re-cache it if possible.
      GstState state;
      gst_element_get_state(play_bin_, &state, 0, 0);
      float newDuration = duration();

      if (state <= GST_STATE_READY) {
          // Don't set media_duration_known_ yet if the pipeline is not
          // paused. This allows duration() query to fail at least once
          // before playback starts and duration becomes known.
          if (!isinf(newDuration))
              media_duration_ = newDuration;
      } else {
          media_duration_known_ = !isinf(newDuration);
          if (media_duration_known_)
              media_duration_ = newDuration;
      }

      if (!isinf(newDuration))
          media_duration_ = newDuration;
  */
}

float WebMediaPlayerGStreamer::maxTimeLoaded() const {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::maxTimeLoaded()\n";

  if (error_occured_)
    return 0.0f;

  float loaded = max_time_loaded_;
  if (!loaded && !fill_timer_.IsRunning())
    loaded = durationF();
  DVLOG(1) << StringPrintf("maxTimeLoaded: %f", loaded);
  return loaded;
}

void WebMediaPlayerGStreamer::updateAudioSink() {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::updateAudioSink()\n";
  /*
      if (!play_bin_)
          return;

      GstElement* sinkPtr = 0;

      g_object_get(play_bin_, "audio-sink", &sinkPtr, NULL);
      webkit_audio_sink_ = adoptGRef(sinkPtr);
  */
}

void WebMediaPlayerGStreamer::fillTimerFired() {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::fillTimerFired()\n";

  GstQuery* query = gst_query_new_buffering(GST_FORMAT_PERCENT);
  /*
      if (!gst_element_query(play_bin_, query)) {
          gst_query_unref(query);
          return;
      }
  */
  gint64 start, stop;
  gdouble fill_status = 100.0;

  gst_query_parse_buffering_range(query, 0, &start, &stop, 0);
  gst_query_unref(query);

  if (stop != -1)
    fill_status = 100.0 * stop / GST_FORMAT_PERCENT_MAX;

  DVLOG(1) << StringPrintf("[Buffering] Download buffer filled up to %f%%",
                           fill_status);

  if (!media_duration_)
    durationChanged();

  // Update maxTimeLoaded only if the media duration is
  // available. Otherwise we can't compute it.
  if (media_duration_) {
    if (fill_status == 100.0)
      max_time_loaded_ = media_duration_;
    else
      max_time_loaded_ =
          static_cast<float>((fill_status * media_duration_) / 100.0);
    DVLOG(1) << StringPrintf("[Buffering] Updated maxTimeLoaded: %f",
                             max_time_loaded_);
  }

  if (fill_status != 100.0) {
    updateStates();
    return;
  }

  // Media is now fully loaded. It will play even if network
  // connection is cut. Buffering is done, remove the fill source
  // from the main loop.
  fill_timer_.Stop();
  started_buffering_ = false;
  updateStates();
}

int64 WebMediaPlayerGStreamer::totalBytes() const {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::totalBytes()\n";

  if (demuxer_host_)
    return demuxer_host_->GetTotalBytes();
  else
    return 0;
}

void WebMediaPlayerGStreamer::onAutoElementAdded(GstElement* element) {
  // this is called on the gmainloop thread

  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer::onAutoElementAdded()\n";

  std::string video_decoder = GetGStreamerString("video-decoder");
  std::string audio_decoder = GetGStreamerString("audio-decoder");

  if (g_strrstr(GST_ELEMENT_NAME(element), "multiqueue")) {
    DVLOG(1) << ("multiqueue Auto Element Added...g_object_set \n");
    g_object_set(G_OBJECT(element),
                 "max-size-bytes",
                 1024,
                 "max-size-time",
                 5 * GST_SECOND,
                 "max-size-buffers",
                 5,
                 "use-buffering",
                 FALSE,
                 NULL);
  } else if (g_strrstr(GST_ELEMENT_NAME(element), video_decoder.c_str())) {
    DVLOG(1) << ("platform video decoder added\n");
    video_decoder_ = element;
  } else if (g_strrstr(GST_ELEMENT_NAME(element), audio_decoder.c_str())) {
    DVLOG(1) << ("platform audio decoder added\n");
    audio_decoder_ = element;
  } else {
    VLOG(1) << StringPrintf("element added %s \n", GST_ELEMENT_NAME(element));
  }
}

void WebMediaPlayerGStreamer::onAutoPadAdded(GstPad* pad) {
  // this is called on the gmainloop thread

  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::onAutoPadAdded()\n";

  GstCaps* caps;
  GstStructure* structure;
  const gchar* name;

  caps = gst_pad_get_caps(pad);
  structure = gst_caps_get_structure(caps, 0);
  name = gst_structure_get_name(structure);

  VLOG(kCallPrintDebugLevel) << "auto pad caps: " << gst_caps_to_string(caps);

  std::string decode_video_caps = GetGStreamerString("decode-video-caps");
  std::string decode_audio_caps = GetGStreamerString("decode-audio-caps");

  if (g_strrstr(name, decode_video_caps.c_str())) {
    if (gst_bin_add(GST_BIN(pipeline_), video_sink_) == FALSE) {
      g_print("Could not add video sink to bin\n");
      return;
    }
    if (gst_element_link(decbin_, video_sink_) == FALSE) {
      g_print("Could not make link video sink to bin\n");
      return;
    }
  }

  if (g_strrstr(name, decode_audio_caps.c_str())) {
    if (gst_bin_add(GST_BIN(pipeline_), audio_sink_) == FALSE) {
      g_print("Could not add audio sink to bin\n");
      return;
    }
    if (gst_element_link(decbin_, audio_sink_) == FALSE) {
      g_print("Could not make link video sink to bin\n");
      return;
    }
  }

  gst_caps_unref(caps);
}

void WebMediaPlayerGStreamer::commitLoad() {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::commitLoad()\n";

  assert(!delaying_load_);
  DVLOG(1) << ("Committing load.");
  updateStates();
}

float WebMediaPlayerGStreamer::playbackPosition() const {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel + 5) << "WebMediaPlayerGStreamer::playbackPosition()\n";

  if (is_end_reached_) {
    // Position queries on a null pipeline return 0. If we're at
    // the end of the stream the pipeline is null but we want to
    // report either the seek time or the duration because this is
    // what the Media element spec expects us to do.
    if (seeking_)
      return seek_time_;
    if (media_duration_) {
      DVLOG(1) << StringPrintf("media_duration_ %f\n", media_duration_);
      return media_duration_;
    }
  }

  float ret = 0.0f;

  GstFormat fmt = GST_FORMAT_TIME;
  gint64 position = -1;

  if (!gst_element_query_position(
           static_cast<GstElement*>(pipeline_), &fmt, &position)) {
    //g_printerr("Could not query current position.\n");
  }

  // Position is available only if the pipeline is not in GST_STATE_NULL or
  // GST_STATE_READY state.
  if (position != static_cast<gint64>(GST_CLOCK_TIME_NONE))
    ret = /*seek_time_ +*/(static_cast<double>(position) / GST_SECOND);
  // TODO(mzuber): The above added seek time was a fix in a recent broadcom
  // webkit mediaPlayerPrivateGstreamer fix
  // Though it seems to cause issues here, because once seeking is done
  // seek_time_ doesn't get cleared.
  // Really this shouldn't need to be set at all, the seeking in gstreamer
  // (broadcom only?) takes a really long
  // time before gst_element_query_position, starts returning the correct time
  // again.

  DVLOG(1) << StringPrintf("Position %" GST_TIME_FORMAT,
                           GST_TIME_ARGS(position));

  return ret;
}

bool WebMediaPlayerGStreamer::changePipelineState(GstState new_state) {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer::changePipelineState()\n";

  assert(new_state == GST_STATE_PLAYING || new_state == GST_STATE_PAUSED);

  GstState current_state;
  GstState pending;

  if (pipeline_ == NULL) {
    DVLOG(1) << ("pipeline_ == NULL\n");
    return false;
  }

  gst_element_get_state(pipeline_, &current_state, &pending, 0);
  DVLOG(1) << StringPrintf("Current state: %s, pending: %s",
                           gst_element_state_get_name(current_state),
                           gst_element_state_get_name(pending));
  if (current_state == new_state || pending == new_state)
    return true;

  printf(
      "*** *%s %d: changing to newState = %s\n",
      __FUNCTION__,
      __LINE__,
      (new_state == GST_STATE_PLAYING)
          ? "GST_STATE_PLAYING"
          : ((new_state == GST_STATE_PAUSED) ? "GST_STATE_PAUSED" : "Unknown"));
  GstStateChangeReturn set_state_result =
      gst_element_set_state(pipeline_, new_state);
  GstState paused_or_playing =
      new_state == GST_STATE_PLAYING ? GST_STATE_PAUSED : GST_STATE_PLAYING;
  if (current_state != paused_or_playing &&
      set_state_result == GST_STATE_CHANGE_FAILURE) {
    DVLOG(1) << ("GST_STATE_CHANGE_FILURE, loadingFailed!!!!!!!!!!!!!!!!\n");
    loadingFailed(WebMediaPlayer::NetworkStateEmpty);
    return false;
  }
  return true;
}

void WebMediaPlayerGStreamer::notifyPlayerSeekFinished() {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer::notifyPlayerSeekFinished()\n";

  video_seek_timer_handler_ = 0;

  if (chunk_demuxer_) {
    media::PipelineStatusCB seek_status =
        base::Bind(&WebMediaPlayerGStreamer::PipelineStatusHandler,
                   base::Unretained(this));
    base::TimeDelta seek_time = ConvertSecondsToTimestamp(seek_time_);
    VLOG(kCallPrintDebugLevel) << "Seeking chunk_demuxer to " << seek_time_;
    chunk_demuxer_->Seek(seek_time, seek_status);
  }

  changePipelineState(GST_STATE_PLAYING);

  GetClient()->timeChanged();
  updateStates();
}

void WebMediaPlayerGStreamer::lockMessageMap(bool lock) {
  base::AutoLock auto_lock(lock_);
  message_map_locked_ = lock;
}

bool WebMediaPlayerGStreamer::isMessageMapLocked() {
  base::AutoLock auto_lock(lock_);
  return message_map_locked_;
}

void WebMediaPlayerGStreamer::addToMessageMap(GstMessage* message) {
  base::AutoLock auto_lock(lock_);
  if (!message_map_locked_) {
    gst_message_ref(message);
    message_map_[message] = message;
  }
}

void WebMediaPlayerGStreamer::removeFromMessageMap(GstMessage* message) {
  base::AutoLock auto_lock(lock_);
  message_map_.erase(message);
  gst_message_unref(message);
}

void WebMediaPlayerGStreamer::clearMessageMap() {
  base::AutoLock auto_lock(lock_);

  VLOG(kCallPrintDebugLevel)
      << StringPrintf("left over message count:%d\n", message_map_.size());
  for (std::map<GstMessage*, GstMessage*>::iterator it = message_map_.begin();
       it != message_map_.end();
       ++it) {
    gst_message_unref(it->first);
  }

  message_map_.clear();
}

// #if BCM_QTWEBKIT
void WebMediaPlayerGStreamer::setScreenRect(const blink::WebRect& r) {
  VLOG(kCallPrintDebugLevel) << "setScreenRect()\n";

  if (!pipeline_)
    return;

  // return if it has not been started yet
  GstState current_state;
  GstState pending;
  gst_element_get_state(pipeline_, &current_state, &pending, 0);
  if (!(current_state == GST_STATE_PLAYING ||
        current_state == GST_STATE_PAUSED))
    return;

  std::string window_set = GetGStreamerString("video-sink-window-set");

  // update only when it is new video position
  if (scr_rect_ != r) {
    if (video_sink_ && r.width > 0 && r.height > 0) {
      char buf[100];
      DVLOG(1) << StringPrintf("x = %d, y = %d, width = %d, height = %d\n",
                               r.x,
                               r.y,
                               r.width,
                               r.height);

      snprintf(buf, sizeof(buf), "%d,%d,%d,%d", r.x, r.y, r.width, r.height);
      g_object_set(GST_OBJECT(video_sink_), window_set.c_str(), buf, NULL);
    }
    scr_rect_ = r;
  }
}
// #endif

void WebMediaPlayerGStreamer::GlibMainLoopFunc() {
  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::GlibMainLoopFunc()";

  // start the gmainloop, it will run for the entirety
  // of the render process
  GMainLoop* loop = g_main_loop_new(NULL, FALSE);
  g_main_loop_run(loop);
}

// TODO(mzuber): This hook isn't called in chromium
// like it was in webkit, when should we hook
// into calling it ourselves?  Seems like it is
// used in conjunction with delaying a load
// and that functionality won't function
// properly unless this is called at the right
// time
void WebMediaPlayerGStreamer::prepareToPlay() {
  is_end_reached_ = false;
  seeking_ = false;

  if (delaying_load_) {
    delaying_load_ = false;
    commitLoad();
  }
}

// TODO(mzuber): This hook isn't called in chromium
// like it was in webkit, is there going to be a problem
// because of that?
void WebMediaPlayerGStreamer::cancelLoad() {
  if (network_state_ < WebMediaPlayer::NetworkStateLoading ||
      network_state_ == WebMediaPlayer::NetworkStateLoaded)
    return;

  if (pipeline_)
    gst_element_set_state(pipeline_, GST_STATE_NULL);
}

// TODO(mzuber): This hook isn't called in chromium
// like it was in webkit, is there going to be a problem
// because of that?
void WebMediaPlayerGStreamer::loadStateChanged() { updateStates(); }

// TODO(mzuber): There are no mute hooks in chromium as far as I
// can tell, is there going to be a problem because of that?
// bool MediaPlayerPrivateGStreamer::supportsMuting() const
// void MediaPlayerPrivateGStreamer::setMuted(bool muted)
// void MediaPlayerPrivateGStreamer::notifyPlayerOfMute()
// void MediaPlayerPrivateGStreamer::muteChanged()

// TODO(mzuber): No setSize hook in chromium,
// void MediaPlayerPrivateGStreamer::setSize(const IntSize& size)

// TODO(mzuber): The following 3 functions are replaced in chromium by
// code in: chromium/net/base/mime_util.cc
// both broadcom and android use #defines in there to declare what types of
// media they support

// MediaPlayer::SupportsType MediaPlayerPrivateGStreamer::supportsType(const
// String& type, const String& codecs, const KURL&)
// PlatformMedia MediaPlayerPrivateGStreamer::platformMedia() const
// MediaPlayer::MovieLoadType MediaPlayerPrivateGStreamer::movieLoadType() const

// buffered data source additions

void WebMediaPlayerGStreamer::NotifyDownloading(bool is_downloading) {
  DCHECK(main_loop_->BelongsToCurrentThread());

  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::NotifyDownloading()";

  if (!is_downloading &&
      network_state_ == WebMediaPlayer::NetworkStateLoading) {
    network_state_ = WebMediaPlayer::NetworkStateIdle;
    client_->networkStateChanged();
  } else if (is_downloading &&
             network_state_ == WebMediaPlayer::NetworkStateIdle) {
    network_state_ = WebMediaPlayer::NetworkStateLoading;
    client_->networkStateChanged();
  }

  /*
  media_log_->AddEvent(
      media_log_->CreateBooleanEvent(
          media::MediaLogEvent::NETWORK_ACTIVITY_SET,
          "is_downloading_data", is_downloading));
          */
}

void WebMediaPlayerGStreamer::DataSourceInitialized(const blink::WebURL& url,
                                                    bool success) {
  DCHECK(main_loop_->BelongsToCurrentThread());

  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer::DataSourceInitialized() :" << success;

  if (!success) {
    network_state_ = WebMediaPlayer::NetworkStateFormatError;
    client_->networkStateChanged();
    // Repaint();
    return;
  }

  DoLoad(url);
}

// clean up, turn back on vp8 and vp9

void WebMediaPlayerGStreamer::PushDataToAppSource() {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer::PushDataToAppSource()";
  // I used blocking_url_protocol.cc as a reference to see how to read data from
  // the
  // data_source

  if (IsReadInProgress())
    return;

  if (aborted_.IsSignaled()) {
    GstFlowReturn ret;
    ret = gst_app_src_end_of_stream(appsrc_source_);
    g_debug("eos returned %d at %d\n", ret, __LINE__);
    return;
  }

  int64 file_size;
  if (data_source_ && data_source_->GetSize(&file_size) &&
      read_position_ >= file_size) {
    GstFlowReturn ret;
    ret = gst_app_src_end_of_stream(appsrc_source_);
    g_debug("eos returned %d at %d\n", ret, __LINE__);
    return;
  }

  SetReadInProgress(true);
  appsrc_data_ = static_cast<guint8*>(g_malloc(kReadSizeBytes));

  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer::before read position: (): "
      << read_position_;

  if (data_source_) {
    // read will happen inside the data source on another thread
    // once the read completes SignalReadCompleted will be called on the main
    // thread
    // from there we will push the data to the gstreamer appsrc
    data_source_->Read(read_position_,
                       kReadSizeBytes,
                       appsrc_data_,
                       base::Bind(&WebMediaPlayerGStreamer::SignalReadCompleted,
                                  base::Unretained(this)));
  }
}

void WebMediaPlayerGStreamer::PushDataToVideoAppSourceChunkDemuxer() {
  DCHECK(main_loop_->BelongsToCurrentThread());

  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer::PushDataToVideoAppSourceChunkDemuxer()";

  if (IsReadInProgress(kVideo))
      return;

  if (aborted_.IsSignaled()) {
      GstFlowReturn ret;
    ret = gst_app_src_end_of_stream(appsrc_source_);
    g_debug("eos returned %d at %d\n", ret, __LINE__);
    return;
  }

  DemuxerStream* video_stream = chunk_demuxer_->GetStream(DemuxerStream::VIDEO);

  // TODO(mzuber): h264 under broadcom doesn't seem to need caps
  // but the intel version probably does, we will have to get the video
  // decoder config video_stream->video_decoder_config(), and then convert
  // it to appropriate caps (this is what the ffmpeg version does, except
  // it converts to ffmpeg types)

  /*
  static bool test_bool = false;
  if (!test_bool) {
    test_bool = true;
    GstCaps* caps;

    caps = gst_caps_new_simple("video/x-h264",
                               "format",
                               GST_TYPE_FOURCC,
                               GST_MAKE_FOURCC('Y', 'V', '1', '2'),
                               "framerate",
                               GST_TYPE_FRACTION,
                               25,
                               1,
                               "pixel-aspect-ratio",
                               GST_TYPE_FRACTION,
                               1,
                               1,
                               "width",
                               G_TYPE_INT,
                               1280,
                               "height",
                               G_TYPE_INT,
                               720,
                               NULL);


     VLOG(kCallPrintDebugLevel) << "Setting appsrc caps:";
     gst_app_src_set_caps(appsrc_source_,caps);
  }
  */

  if (video_stream) {
    bool is_encrypted = video_stream->video_decoder_config().is_encrypted();

    if (is_encrypted && !decrypted_video_stream_initialized_) {
      decrypted_video_stream_->Initialize(
          video_stream,
          base::Bind(&WebMediaPlayerGStreamer::DecryptedVideoStreamInitialized,
                     AsWeakPtr()));
    } else {
      SetReadInProgress(true, kVideo);
      VLOG(kCallPrintDebugLevel)
          << "Chunk Demuxer video decoder info:"
          << video_stream->video_decoder_config().AsHumanReadableString();

      if (!is_encrypted) {
        video_stream->Read(base::Bind(
            &WebMediaPlayerGStreamer::OnDemuxerVideoBufferReady, AsWeakPtr()));
      } else {
        decrypted_video_stream_->Read(base::Bind(
            &WebMediaPlayerGStreamer::OnDemuxerVideoBufferReady, AsWeakPtr()));
      }
    }
  }
}

void WebMediaPlayerGStreamer::PushDataToAudioAppSourceChunkDemuxer() {
  DCHECK(main_loop_->BelongsToCurrentThread());

  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer::PushDataToAudioAppSourceChunkDemuxer()";

  if (IsReadInProgress(kAudio))
    return;

  if (aborted_.IsSignaled()) {
    GstFlowReturn ret;
    ret = gst_app_src_end_of_stream(appsrc_source_audio_);
    g_debug("eos returned %d at %d\n", ret, __LINE__);
    return;
  }

  DemuxerStream* audio_stream = chunk_demuxer_->GetStream(DemuxerStream::AUDIO);

  if (audio_stream) {
    bool is_encrypted = audio_stream->audio_decoder_config().is_encrypted();

    if (is_encrypted && !decrypted_audio_stream_initialized_) {
      decrypted_audio_stream_->Initialize(
          audio_stream,
          base::Bind(&WebMediaPlayerGStreamer::DecryptedAudioStreamInitialized,
                     AsWeakPtr()));
    } else {
      SetReadInProgress(true, kAudio);
      VLOG(kCallPrintDebugLevel) << "Chunk Demuxer audio decoder info:"
                                 << GetAudioConfigAsHumanReadableString(
                                      audio_stream->audio_decoder_config());
      if (!is_encrypted) {
        audio_stream->Read(base::Bind(
          &WebMediaPlayerGStreamer::OnDemuxerAudioBufferReady, AsWeakPtr()));
      } else {
        decrypted_audio_stream_->Read(base::Bind(
          &WebMediaPlayerGStreamer::OnDemuxerAudioBufferReady, AsWeakPtr()));
      }
    }
  }
}


std::string WebMediaPlayerGStreamer::GetAudioConfigAsHumanReadableString(
    const media::AudioDecoderConfig& audio_config) {
  std::ostringstream s;
  s << "codec: " << audio_config.codec()
    << " bytes ber channel: " << audio_config.bytes_per_channel()
    << " channel layout: " << audio_config.channel_layout()
    << " samples per second: " << audio_config.samples_per_second()
    << " sample format: " << audio_config.sample_format()
    << " bytes per frame: " << audio_config.bytes_per_frame()
    << " has extra data? " << (audio_config.extra_data() ? "true" : "false")
    << " encrypted? " << (audio_config.is_encrypted() ? "true" : "false");
  return s.str();
}

void WebMediaPlayerGStreamer::WriteRawVideoFramesToFile(
    float time_till_stop_seconds,
    const scoped_refptr<DecoderBuffer>& buffer) {
  static bool do_write_file = true;
  static FILE* test_file = NULL;
  if (do_write_file) {
    do_write_file = false;
    test_file = fopen("/raw_video_frames.bin", "wb");
  }

  if (buffer->timestamp().InSeconds() < time_till_stop_seconds && test_file) {
    VLOG(kCallPrintDebugLevel) << "test file: " << test_file;
    guint8* data_to_write = static_cast<guint8*>(g_malloc(buffer->data_size()));
    memcpy(data_to_write, buffer->data(), buffer->data_size());
    fwrite(data_to_write, 1, buffer->data_size(), test_file);
    g_free(data_to_write);
  } else {
    if (test_file) {
      fclose(test_file);
      test_file = NULL;
    }
  }
}

void WebMediaPlayerGStreamer::WriteRawAudioFramesToFile(
    float time_till_stop_seconds,
    const scoped_refptr<media::DecoderBuffer>& buffer) {
  static bool do_write_file = true;
  static FILE* test_file = NULL;
  if (do_write_file) {
    do_write_file = false;
    test_file = fopen("/raw_audio_frames.bin", "wb");
  }

  if (buffer->timestamp().InSeconds() < time_till_stop_seconds && test_file) {
    VLOG(kCallPrintDebugLevel) << "test file: " << test_file;
    guint8* data_to_write = static_cast<guint8*>(g_malloc(buffer->data_size()));
    memcpy(data_to_write, buffer->data(), buffer->data_size());
    fwrite(data_to_write, 1, buffer->data_size(), test_file);
    g_free(data_to_write);
  } else {
    if (test_file) {
      fclose(test_file);
      test_file = NULL;
    }
  }
}

void WebMediaPlayerGStreamer::OnDemuxerAudioBufferReady(
    media::DemuxerStream::Status status,
    const scoped_refptr<media::DecoderBuffer>& buffer) {

    if (status == DemuxerStream::kConfigChanged && ShouldBeReading(kAudio)) {
      // Config has changed, so we need to re-ask for the video decoder stream
      // for its new video config before it will gives us buffers again
      main_loop_->PostTask(
          FROM_HERE,
          base::Bind(
              &WebMediaPlayerGStreamer::PushDataToAudioAppSourceChunkDemuxer,
              AsWeakPtr()));
      SetReadInProgress(false, kAudio);
      return;
    }

    if (status == DemuxerStream::kOk)
      VLOG(kCallPrintDebugLevel) << buffer->AsHumanReadableString();

    GstBuffer* gst_buffer;
    GstFlowReturn ret;

    if (status == DemuxerStream::kAborted) {
      // aborted_.Signal();
      // ret = gst_app_src_end_of_stream(appsrc_source_);
      // g_debug("eos returned %d at %d\n", ret, __LINE__);
      main_loop_->PostTask(
          FROM_HERE,
          base::Bind(
              &WebMediaPlayerGStreamer::PushDataToAudioAppSourceChunkDemuxer,
              AsWeakPtr()));
      SetReadInProgress(false, kAudio);
      return;
    }

    if (status == DemuxerStream::kOk && buffer->end_of_stream()) {
      GstFlowReturn ret;
      ret = gst_app_src_end_of_stream(appsrc_source_audio_);
      g_debug("eos returned %d at %d\n", ret, __LINE__);
      return;
    }

  #ifdef WRITE_AUDIO_DEBUG_OUTPUT
    float seconds_wanted = 29.0;
    WriteRawAudioFramesToFile(seconds_wanted, buffer);
  #endif

    appsrc_data_ = static_cast<guint8*>(g_malloc(buffer->data_size()));
    memcpy(appsrc_data_, buffer->data(), buffer->data_size());
    gst_buffer = gst_buffer_new();
    GST_BUFFER_MALLOCDATA(gst_buffer) = appsrc_data_;
    GST_BUFFER_SIZE(gst_buffer) = buffer->data_size();
    GST_BUFFER_DATA(gst_buffer) = GST_BUFFER_MALLOCDATA(gst_buffer);
    GST_BUFFER_TIMESTAMP(gst_buffer) =
        buffer->timestamp().InMicroseconds() * 1000;

    VLOG(kCallPrintDebugLevel) << "Push audio buffer " << buffer->data_size() << " bytes";
    ret = gst_app_src_push_buffer(appsrc_source_audio_, gst_buffer);

    if (ret != GST_FLOW_OK) {
      g_debug("push buffer returned %d for %d bytes \n", ret, last_read_bytes_);
      SetReadInProgress(false, kAudio);
      return;
    }

    SetReadInProgress(false, kAudio);

    if (status == DemuxerStream::kOk && ShouldBeReading(kAudio)) {
      main_loop_->PostTask(
          FROM_HERE,
          base::Bind(
              &WebMediaPlayerGStreamer::PushDataToAudioAppSourceChunkDemuxer,
              AsWeakPtr()));
    }
}

void WebMediaPlayerGStreamer::OnDemuxerVideoBufferReady(
    DemuxerStream::Status status,
    const scoped_refptr<DecoderBuffer>& buffer) {
  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer::OnDemuxerVideoBufferReady(), status:" << status;

  if (status == DemuxerStream::kConfigChanged && ShouldBeReading(kVideo)) {
    // Config has changed, so we need to re-ask for the video decoder stream
    // for its new video config before it will gives us buffers again
    main_loop_->PostTask(
        FROM_HERE,
        base::Bind(
            &WebMediaPlayerGStreamer::PushDataToVideoAppSourceChunkDemuxer,
            AsWeakPtr()));
    SetReadInProgress(false, kVideo);
    return;
  }

  if (status == DemuxerStream::kOk)
    VLOG(kCallPrintDebugLevel) << buffer->AsHumanReadableString();

  GstBuffer* gst_buffer;
  GstFlowReturn ret;

  if (status == DemuxerStream::kAborted) {
    // aborted_.Signal();
    // ret = gst_app_src_end_of_stream(appsrc_source_);
    // g_debug("eos returned %d at %d\n", ret, __LINE__);
    main_loop_->PostTask(
        FROM_HERE,
        base::Bind(
            &WebMediaPlayerGStreamer::PushDataToVideoAppSourceChunkDemuxer,
            AsWeakPtr()));
    SetReadInProgress(false, kVideo);
    return;
  }

  if (status == DemuxerStream::kOk && buffer->end_of_stream()) {
    GstFlowReturn ret;
    ret = gst_app_src_end_of_stream(appsrc_source_);
    g_debug("eos returned %d at %d\n", ret, __LINE__);
    SetReadInProgress(false);
    return;
  }

#ifdef WRITE_VIDEO_DEBUG_OUTPUT
  float seconds_wanted = 29.0;
  WriteRawVideoFramesToFile(seconds_wanted, buffer);
#endif

  appsrc_data_ = static_cast<guint8*>(g_malloc(buffer->data_size()));
  memcpy(appsrc_data_, buffer->data(), buffer->data_size());
  gst_buffer = gst_buffer_new();
  GST_BUFFER_MALLOCDATA(gst_buffer) = appsrc_data_;
  GST_BUFFER_SIZE(gst_buffer) = buffer->data_size();
  GST_BUFFER_DATA(gst_buffer) = GST_BUFFER_MALLOCDATA(gst_buffer);
  GST_BUFFER_TIMESTAMP(gst_buffer) =
      buffer->timestamp().InMicroseconds() * 1000;

  ret = gst_app_src_push_buffer(appsrc_source_, gst_buffer);

  if (ret != GST_FLOW_OK) {
    g_debug("push buffer returned %d for %d bytes \n", ret, last_read_bytes_);
    SetReadInProgress(false, kVideo);
    return;
  }

  // read_position_ += last_read_bytes_;
  SetReadInProgress(false, kVideo);

  if (status == DemuxerStream::kOk && ShouldBeReading(kVideo)) {
    main_loop_->PostTask(
        FROM_HERE,
        base::Bind(
            &WebMediaPlayerGStreamer::PushDataToVideoAppSourceChunkDemuxer,
            AsWeakPtr()));
  }
}

bool WebMediaPlayerGStreamer::IsReadInProgress(int av) {
  base::AutoLock auto_lock(read_lock_);
  return read_in_progress_[av];
}

void WebMediaPlayerGStreamer::SetReadInProgress(bool in_progress, int av) {
  base::AutoLock auto_lock(read_lock_);
  read_in_progress_[av] = in_progress;
}

void WebMediaPlayerGStreamer::StartFeedingAppSource(GstAppSrc* pSrc) {
  // this is called on the gmainloop thread

  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer::StartFeedingAppSource(" << (pSrc==appsrc_source_?"video":"audio") << ")";

  if (!ShouldBeReading(pSrc == appsrc_source_ ? kVideo : kAudio)) {
    // start the reading on the main thread
    SetShouldBeReading(true, pSrc == appsrc_source_ ? kVideo : kAudio);

    if (load_type_ != LoadTypeMediaSource) {
      main_loop_->PostTask(
          FROM_HERE,
          base::Bind(&WebMediaPlayerGStreamer::PushDataToAppSource,
                     AsWeakPtr()));
    } else {
      main_loop_->PostTask(
          FROM_HERE,
          base::Bind(
              pSrc == appsrc_source_ ?
                      &WebMediaPlayerGStreamer::PushDataToVideoAppSourceChunkDemuxer :
                      &WebMediaPlayerGStreamer::PushDataToAudioAppSourceChunkDemuxer,
              AsWeakPtr()));
    }
  }
}

void WebMediaPlayerGStreamer::StopFeedingAppSource(GstAppSrc* pSrc) {
  // this is called on the gmainloop thread

  if (pSrc==appsrc_source_)
  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer::StopFeedingAppSource()";

  // gstreamer signaled it doesn't want more data, so stop reading
  SetShouldBeReading(false, pSrc == appsrc_source_ ? kVideo : kAudio);
}

void WebMediaPlayerGStreamer::SetNewAppSourceReadPosition(GstAppSrc* pSrc, guint64 position) {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer::SetNewAppSourceReadPosition()";

  VLOG(kCallPrintDebugLevel) << "Changing read position from:" << read_position_
                             << " to:" << position;

  SetShouldBeReading(false, pSrc == appsrc_source_ ? kVideo : kAudio);  // this lets the PushDataToAppSrc function know
  // it should start the read ping/pong again
  // at the new read position
  read_position_ = position;
}

void WebMediaPlayerGStreamer::SignalReadCompleted(int size) {
  DCHECK(main_loop_->BelongsToCurrentThread());
  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer::SignalReadCompleted()";
  last_read_bytes_ = size;

  GstBuffer* buffer;
  GstFlowReturn ret;

  if (last_read_bytes_ == media::DataSource::kReadError) {
    g_free(appsrc_data_);
    appsrc_data_ = NULL;
    aborted_.Signal();
    ret = gst_app_src_end_of_stream(appsrc_source_);
    g_debug("eos returned %d at %d\n", ret, __LINE__);
    SetReadInProgress(false);
    return;
  }

  buffer = gst_buffer_new();
  GST_BUFFER_MALLOCDATA(buffer) = appsrc_data_;
  GST_BUFFER_SIZE(buffer) = last_read_bytes_;
  GST_BUFFER_DATA(buffer) = GST_BUFFER_MALLOCDATA(buffer);

  ret = gst_app_src_push_buffer(appsrc_source_, buffer);
  appsrc_data_ = NULL;

  if (ret != GST_FLOW_OK) {
    g_debug("push buffer returned %d for %d bytes \n", ret, last_read_bytes_);
    SetReadInProgress(false);
    return;
  }

  read_position_ += last_read_bytes_;
  SetReadInProgress(false);

  if (ShouldBeReading()) {
    // continue to ping/pong reads until we are told to stop
    main_loop_->PostTask(
        FROM_HERE,
        base::Bind(&WebMediaPlayerGStreamer::PushDataToAppSource, AsWeakPtr()));
  }
}

bool WebMediaPlayerGStreamer::isLiveStream() const {
  DCHECK(main_loop_->BelongsToCurrentThread());
  if (data_source_)
    return data_source_->IsStreaming();
  else
    return false;
}

void WebMediaPlayerGStreamer::SetShouldBeReading(bool is_reading, int av) {
  base::AutoLock auto_lock(read_lock_);
  should_be_reading_[av] = is_reading;
}

bool WebMediaPlayerGStreamer::ShouldBeReading(int av) {
  base::AutoLock auto_lock(read_lock_);
  return should_be_reading_[av];
}

// additions because of chunk demuxer
void WebMediaPlayerGStreamer::OnDemuxerOpened() {
  DCHECK(main_loop_->BelongsToCurrentThread());
  GetClient()->mediaSourceOpened(new WebMediaSourceImpl(
      chunk_demuxer_, base::Bind(&LogMediaSourceError, media_log_)));
}

// encryption additions

// Helper functions to report media EME related stats to UMA. They follow the
// convention of more commonly used macros UMA_HISTOGRAM_ENUMERATION and
// UMA_HISTOGRAM_COUNTS. The reason that we cannot use those macros directly is
// that UMA_* macros require the names to be constant throughout the process'
// lifetime.
static void EmeUMAHistogramEnumeration(const blink::WebString& key_system,
                                       const std::string& method,
                                       int sample,
                                       int boundary_value) {
  base::LinearHistogram::FactoryGet(
      kMediaEme + KeySystemNameForUMA(key_system) + "." + method,
      1,
      boundary_value,
      boundary_value + 1,
      base::Histogram::kUmaTargetedHistogramFlag)->Add(sample);
}

static void EmeUMAHistogramCounts(const blink::WebString& key_system,
                                  const std::string& method,
                                  int sample) {
  // Use the same parameters as UMA_HISTOGRAM_COUNTS.
  base::Histogram::FactoryGet(
      kMediaEme + KeySystemNameForUMA(key_system) + "." + method,
      1,
      1000000,
      50,
      base::Histogram::kUmaTargetedHistogramFlag)->Add(sample);
}

// Helper enum for reporting generateKeyRequest/addKey histograms.
enum MediaKeyException {
  kUnknownResultId,
  kSuccess,
  kKeySystemNotSupported,
  kInvalidPlayerState,
  kMaxMediaKeyException
};

static MediaKeyException MediaKeyExceptionForUMA(
    WebMediaPlayer::MediaKeyException e) {
  switch (e) {
    case WebMediaPlayer::MediaKeyExceptionKeySystemNotSupported:
      return kKeySystemNotSupported;
    case WebMediaPlayer::MediaKeyExceptionInvalidPlayerState:
      return kInvalidPlayerState;
    case WebMediaPlayer::MediaKeyExceptionNoError:
      return kSuccess;
    default:
      return kUnknownResultId;
  }
}

// Helper for converting |key_system| name and exception |e| to a pair of enum
// values from above, for reporting to UMA.
static void ReportMediaKeyExceptionToUMA(const std::string& method,
                                         const WebString& key_system,
                                         WebMediaPlayer::MediaKeyException e) {
  MediaKeyException result_id = MediaKeyExceptionForUMA(e);
  DCHECK_NE(result_id, kUnknownResultId) << e;
  EmeUMAHistogramEnumeration(
      key_system, method, result_id, kMaxMediaKeyException);
}

WebMediaPlayer::MediaKeyException WebMediaPlayerGStreamer::generateKeyRequest(
    const WebString& key_system,
    const unsigned char* init_data,
    unsigned init_data_length) {
  WebMediaPlayer::MediaKeyException e =
      GenerateKeyRequestInternal(key_system, init_data, init_data_length);
  ReportMediaKeyExceptionToUMA("generateKeyRequest", key_system, e);
  return e;
}

WebMediaPlayer::MediaKeyException
WebMediaPlayerGStreamer::GenerateKeyRequestInternal(
    const WebString& key_system,
    const unsigned char* init_data,
    unsigned init_data_length) {
  VLOG(1) << "generateKeyRequest: " << key_system.utf8().data() << ": "
          << std::string(reinterpret_cast<const char*>(init_data),
                         static_cast<size_t>(init_data_length));

  if (!IsConcreteSupportedKeySystem(key_system))
    return WebMediaPlayer::MediaKeyExceptionKeySystemNotSupported;

  // We do not support run-time switching between key systems for now.
  if (current_key_system_.isEmpty()) {
    if (!decryptor_->InitializeCDM(key_system.utf8(), frame_->document().url()))
      return WebMediaPlayer::MediaKeyExceptionKeySystemNotSupported;
    current_key_system_ = key_system;
  } else if (key_system != current_key_system_) {
    return WebMediaPlayer::MediaKeyExceptionInvalidPlayerState;
  }

  // TODO(xhwang): We assume all streams are from the same container (thus have
  // the same "type") for now. In the future, the "type" should be passed down
  // from the application.
  if (!decryptor_->GenerateKeyRequest(
           init_data_type_, init_data, init_data_length)) {
    current_key_system_.reset();
    return WebMediaPlayer::MediaKeyExceptionKeySystemNotSupported;
  }

  return WebMediaPlayer::MediaKeyExceptionNoError;
}

WebMediaPlayer::MediaKeyException WebMediaPlayerGStreamer::addKey(
    const WebString& key_system,
    const unsigned char* key,
    unsigned key_length,
    const unsigned char* init_data,
    unsigned init_data_length,
    const WebString& session_id) {
  WebMediaPlayer::MediaKeyException e = AddKeyInternal(
      key_system, key, key_length, init_data, init_data_length, session_id);
  ReportMediaKeyExceptionToUMA("addKey", key_system, e);
  return e;
}

WebMediaPlayer::MediaKeyException WebMediaPlayerGStreamer::AddKeyInternal(
    const WebString& key_system,
    const unsigned char* key,
    unsigned key_length,
    const unsigned char* init_data,
    unsigned init_data_length,
    const WebString& session_id) {
  DCHECK(key);
  DCHECK_GT(key_length, 0u);
  DVLOG(1) << "addKey: " << key_system.utf8().data() << ": "
           << std::string(reinterpret_cast<const char*>(key),
                          static_cast<size_t>(key_length)) << ", "
           << std::string(reinterpret_cast<const char*>(init_data),
                          static_cast<size_t>(init_data_length)) << " ["
           << session_id.utf8().data() << "]";

  if (!IsConcreteSupportedKeySystem(key_system))
    return WebMediaPlayer::MediaKeyExceptionKeySystemNotSupported;

  if (current_key_system_.isEmpty() || key_system != current_key_system_)
    return WebMediaPlayer::MediaKeyExceptionInvalidPlayerState;

  decryptor_->AddKey(
      key, key_length, init_data, init_data_length, session_id.utf8());
  return WebMediaPlayer::MediaKeyExceptionNoError;
}

WebMediaPlayer::MediaKeyException WebMediaPlayerGStreamer::cancelKeyRequest(
    const WebString& key_system,
    const WebString& session_id) {
  WebMediaPlayer::MediaKeyException e =
      CancelKeyRequestInternal(key_system, session_id);
  ReportMediaKeyExceptionToUMA("cancelKeyRequest", key_system, e);
  return e;
}

WebMediaPlayer::MediaKeyException
WebMediaPlayerGStreamer::CancelKeyRequestInternal(const WebString& key_system,
                                                  const WebString& session_id) {
  if (!IsConcreteSupportedKeySystem(key_system))
    return WebMediaPlayer::MediaKeyExceptionKeySystemNotSupported;

  if (current_key_system_.isEmpty() || key_system != current_key_system_)
    return WebMediaPlayer::MediaKeyExceptionInvalidPlayerState;

  decryptor_->CancelKeyRequest(session_id.utf8());
  return WebMediaPlayer::MediaKeyExceptionNoError;
}

void WebMediaPlayerGStreamer::OnKeyAdded(const std::string& session_id) {
  DCHECK(main_loop_->BelongsToCurrentThread());

  VLOG(kCallPrintDebugLevel) << "WebMediaPlayerGStreamer::OnKeyAdded()";

  EmeUMAHistogramCounts(current_key_system_, "KeyAdded", 1);
  GetClient()->keyAdded(current_key_system_, WebString::fromUTF8(session_id));
}

void WebMediaPlayerGStreamer::OnNeedKey(const std::string& type,
                                        const std::vector<uint8>& init_data) {
  DCHECK(main_loop_->BelongsToCurrentThread());

  VLOG(kCallPrintDebugLevel)
      << "WebMediaPlayerGStreamer::OnNeedKey() of type: " << type;

  // Do not fire NeedKey event if encrypted media is not enabled.
  if (!decryptor_)
    return;

  UMA_HISTOGRAM_COUNTS(kMediaEme + std::string("NeedKey"), 1);

  DCHECK(init_data_type_.empty() || type.empty() || type == init_data_type_);
  if (init_data_type_.empty())
    init_data_type_ = type;

  const uint8* init_data_ptr = init_data.empty() ? NULL : &init_data[0];
  GetClient()->keyNeeded(
      WebString(), WebString(), init_data_ptr, init_data.size());
}

void WebMediaPlayerGStreamer::OnKeyError(const std::string& session_id,
                                         media::MediaKeys::KeyError error_code,
                                         int system_code) {
  DCHECK(main_loop_->BelongsToCurrentThread());

  EmeUMAHistogramEnumeration(current_key_system_,
                             "KeyError",
                             error_code,
                             media::MediaKeys::kMaxKeyError);

  GetClient()->keyError(
      current_key_system_,
      WebString::fromUTF8(session_id),
      static_cast<blink::WebMediaPlayerClient::MediaKeyErrorCode>(error_code),
      system_code);
}

void WebMediaPlayerGStreamer::OnKeyMessage(const std::string& session_id,
                                           const std::vector<uint8>& message,
                                           const std::string& default_url) {
  DCHECK(main_loop_->BelongsToCurrentThread());

  const GURL default_url_gurl(default_url);
  DLOG_IF(WARNING, !default_url.empty() && !default_url_gurl.is_valid())
      << "Invalid URL in default_url: " << default_url;

  GetClient()->keyMessage(current_key_system_,
                          WebString::fromUTF8(session_id),
                          message.empty() ? NULL : &message[0],
                          message.size(),
                          default_url_gurl);
}

void WebMediaPlayerGStreamer::OnDemuxerInit(media::PipelineStatus status) {
  /*
  if (!success) {
    network_state_ = WebMediaPlayer::NetworkStateFormatError;
    client_->networkStateChanged();
    // Repaint();
    return;
  }
  */

  // maybe check for a status error here...
  VLOG(kCallPrintDebugLevel) << "Demuxer Init Status: " << status;

  DoLoad(GURL(""));
}

void WebMediaPlayerGStreamer::PipelineStatusHandler(media::PipelineStatus status) {
    VLOG(kCallPrintDebugLevel) << "PipelineStatusHandler " << status;
    seeking_ = false;
}

void WebMediaPlayerGStreamer::DecryptedVideoStreamInitialized(
    media::PipelineStatus status) {
  // maybe check for a status error here...
  VLOG(kCallPrintDebugLevel)
      << "Decrypted Video Stream Init Status: " << status;

  decrypted_video_stream_initialized_ = true;

  main_loop_->PostTask(
      FROM_HERE,
      base::Bind(&WebMediaPlayerGStreamer::PushDataToVideoAppSourceChunkDemuxer,
                 AsWeakPtr()));
}

void WebMediaPlayerGStreamer::DecryptedAudioStreamInitialized(
    media::PipelineStatus status) {
  // maybe check for a status error here...
  VLOG(kCallPrintDebugLevel)
      << "Decrypted Audio Stream Init Status: " << status;

  decrypted_audio_stream_initialized_ = true;

  main_loop_->PostTask(
      FROM_HERE,
      base::Bind(&WebMediaPlayerGStreamer::PushDataToAudioAppSourceChunkDemuxer,
                 AsWeakPtr()));
}

}  // namespace content

#endif  // #ifdef GSTREAMER_HARDWARE_VIDEO_DECODING
// COMCAST MODIFICATION END [GSTREAMER_HARDWARE_VIDEO_DECODING]
