// COMCAST MODIFICATION BEGIN [GSTREAMER_HARDWARE_VIDEO_DECODING]
#ifdef GSTREAMER_HARDWARE_VIDEO_DECODING

// Copyright 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

// Delegate calls from WebCore::MediaPlayerPrivate to Chrome's video player.
// It contains Pipeline which is the actual media player pipeline, it glues
// the media player pipeline, data source, audio renderer and renderer.
// Pipeline would creates multiple threads and access some public methods
// of this class, so we need to be extra careful about concurrent access of
// methods and members.
//
// Other issues:
// During tear down of the whole browser or a tab, the DOM tree may not be
// destructed nicely, and there will be some dangling media threads trying to
// the main thread, so we need this class to listen to destruction event of the
// main thread and cleanup the media threads when the even is received. Also
// at destruction of this class we will need to unhook it from destruction event
// list of the main thread.

#ifndef CONTENT_RENDERER_MEDIA_WEBMEDIAPLAYER_GSTREAMER_H_
#define CONTENT_RENDERER_MEDIA_WEBMEDIAPLAYER_GSTREAMER_H_

#include <gst/app/gstappsrc.h>
#include <gst/gst.h>

#include <map>
#include <string>
#include <vector>

#include "base/basictypes.h"
#include "base/memory/ref_counted.h"
#include "base/memory/scoped_ptr.h"
#include "base/memory/weak_ptr.h"
#include "base/synchronization/waitable_event.h"
#include "base/threading/thread.h"
#include "base/timer/timer.h"
#include "cc/layers/video_frame_provider.h"
#include "content/public/renderer/render_view_observer.h"
#include "content/renderer/media/crypto/proxy_decryptor.h"
#include "media/base/audio_renderer_sink.h"
#include "media/base/decoder_buffer.h"
#include "media/base/decryptor.h"
#include "media/base/demuxer_stream.h"
#include "media/base/media_keys.h"
#include "media/base/pipeline.h"
#include "media/base/pipeline_status.h"
#include "media/base/ranges.h"
#include "media/base/text_track.h"
#include "media/filters/skcanvas_video_renderer.h"
#include "media/filters/decrypting_demuxer_stream.h"
#include "skia/ext/platform_canvas.h"
#include "third_party/WebKit/public/platform/WebGraphicsContext3D.h"
#include "third_party/WebKit/public/platform/WebAudioSourceProvider.h"
#include "third_party/WebKit/public/platform/WebMediaPlayer.h"
#include "third_party/WebKit/public/platform/WebMediaPlayerClient.h"
#include "third_party/WebKit/public/platform/WebRect.h"
#include "third_party/WebKit/public/platform/WebSize.h"
#include "url/gurl.h"

class RenderAudioSourceProvider;

namespace blink {
class WebFrame;
}

namespace base {
class MessageLoopProxy;
class Value;
class DictionaryValue;
}

namespace media {
class ChunkDemuxer;
class GpuVideoAcceleratorFactories;
class MediaLog;
// class DecryptingDemuxerStream;

class MEDIA_EXPORT GStreamerDemuxerHost : public DemuxerHost {
 public:
  GStreamerDemuxerHost();
  // Set the total size of the media file.
  virtual void SetTotalBytes(int64 total_bytes);

  // Notify the host that byte range [start,end] has been buffered.
  // TODO(fischman): remove this method when demuxing is push-based instead of
  // pull-based.  http://crbug.com/131444
  virtual void AddBufferedByteRange(int64 start, int64 end);

  // Notify the host that time range [start,end] has been buffered.
  virtual void AddBufferedTimeRange(base::TimeDelta start, base::TimeDelta end);

  const media::Ranges<int64>& GetBufferedByteRanges();
  const media::Ranges<base::TimeDelta>& GetBufferedTimeRanges();
  int64 GetTotalBytes();
  base::TimeDelta GetDuration();

  // Sets the duration of the media in microseconds.
  // Duration may be kInfiniteDuration() if the duration is not known.
  virtual void SetDuration(base::TimeDelta duration);

  // Stops execution of the pipeline due to a fatal error.  Do not call this
  // method with PIPELINE_OK.
  virtual void OnDemuxerError(PipelineStatus error);

  // Add |text_stream| to the collection managed by the text renderer.
  virtual void AddTextStream(DemuxerStream* text_stream,
                             const TextTrackConfig& config);

  // Remove |text_stream| from the presentation.
  virtual void RemoveTextStream(DemuxerStream* text_stream);

  // protected:
  //  virtual ~GStreamerDataHost() {};
 private:
  // variables for DataSourceHost callbacks
  // Amount of available buffered data.  Set by filters.
  media::Ranges<int64> buffered_byte_ranges_;
  media::Ranges<base::TimeDelta> buffered_time_ranges_;

  // Total size of the media.  Set by filters.
  int64 total_bytes_;

  // variables for DemuxerHost callbacks
  base::TimeDelta duration_;
};

}  // namespace media

namespace webkit {
class WebLayerImpl;
}

namespace content {
class BufferedDataSource;
class WebMediaPlayerDelegate;
class WebMediaPlayerParams;

class WebMediaPlayerGStreamer
    : public blink::WebMediaPlayer,
      public cc::VideoFrameProvider,
      public content::RenderViewObserver,
      public base::SupportsWeakPtr<WebMediaPlayerGStreamer> {
 public:
  // Constructs a WebMediaPlayer implementation using gstreamer as a backend.
  //
  // |delegate| may be null.
  WebMediaPlayerGStreamer(content::RenderView* render_view,
                          blink::WebFrame* frame,
                          blink::WebMediaPlayerClient* client,
                          base::WeakPtr<WebMediaPlayerDelegate> delegate,
                          const WebMediaPlayerParams& params);
  virtual ~WebMediaPlayerGStreamer();

  virtual void load(LoadType load_type,
                    const blink::WebURL& url,
                    CORSMode cors_mode) OVERRIDE;
  // Playback controls.
  virtual void play();
  virtual void pause();
  virtual bool supportsFullscreen() const;
  virtual bool supportsSave() const;
  virtual void seek(double seconds);
  virtual void setRate(double rate);
  virtual void setVolume(double volume);
  virtual void setPreload(blink::WebMediaPlayer::Preload preload);
  virtual const blink::WebTimeRanges& buffered();
  virtual double maxTimeSeekable() const;

  // Methods for painting.
  virtual void paint(blink::WebCanvas* canvas,
                     const blink::WebRect& rect,
                     unsigned char alpha);

  // True if the loaded media has a playable video/audio track.
  virtual bool hasVideo() const;
  virtual bool hasAudio() const;

  // Dimensions of the video.
  virtual blink::WebSize naturalSize() const;

  // Getters of playback state.
  virtual bool paused() const;
  virtual bool seeking() const;
  virtual double duration() const;
  virtual double currentTime() const;

  // Internal states of loading and network.
  virtual blink::WebMediaPlayer::NetworkState networkState() const;
  virtual blink::WebMediaPlayer::ReadyState readyState() const;

  virtual bool didLoadingProgress() const;

  virtual bool hasSingleSecurityOrigin() const;
  virtual bool didPassCORSAccessCheck() const;

  virtual double mediaTimeForTimeValue(double time_value) const;

  virtual unsigned decodedFrameCount() const;
  virtual unsigned droppedFrameCount() const;
  virtual unsigned audioDecodedByteCount() const;
  virtual unsigned videoDecodedByteCount() const;

  // cc::VideoFrameProvider implementation.
  virtual void SetVideoFrameProviderClient(
      cc::VideoFrameProvider::Client* client) OVERRIDE;
  virtual scoped_refptr<media::VideoFrame> GetCurrentFrame() OVERRIDE;
  virtual void PutCurrentFrame(const scoped_refptr<media::VideoFrame>& frame)
      OVERRIDE;

  // content::RenderViewObserver implementation.
  virtual void OnDestruct() OVERRIDE;

  // additions for decryption
  virtual MediaKeyException generateKeyRequest(
      const blink::WebString& key_system,
      const unsigned char* init_data,
      unsigned init_data_length);

  virtual MediaKeyException addKey(const blink::WebString& key_system,
                                   const unsigned char* key,
                                   unsigned key_length,
                                   const unsigned char* init_data,
                                   unsigned init_data_length,
                                   const blink::WebString& session_id);

  virtual MediaKeyException cancelKeyRequest(
      const blink::WebString& key_system,
      const blink::WebString& session_id);

  void OnKeyAdded(const std::string& session_id);
  void OnKeyError(const std::string& session_id,
                  media::MediaKeys::KeyError error_code,
                  int system_code);
  void OnKeyMessage(const std::string& session_id,
                    const std::vector<uint8>& message,
                    const std::string& default_url);
  void OnNeedKey(const std::string& type, const std::vector<uint8>& init_data);

  // gstreamer additions:
  gboolean handleMessage(GstMessage* message);
  gboolean handleMessageMultiThreadedThread(GstMessage* message);
  void onAutoElementAdded(GstElement* element);
  void onAutoPadAdded(GstPad* pad);
  void notifyPlayerSeekFinished();

  // buffered data source / appsrc additions
  void PushDataToAppSource();
  void StartFeedingAppSource(GstAppSrc*);
  void StopFeedingAppSource(GstAppSrc*);
  void SetNewAppSourceReadPosition(GstAppSrc*, guint64 position);

  // chunk demuxer additions
  void PushDataToVideoAppSourceChunkDemuxer();
  void PushDataToAudioAppSourceChunkDemuxer();

 private:
  // Getter method to |client_|.
  blink::WebMediaPlayerClient* GetClient();
  void Destroy();

  // Helpers that set the network/ready state and notifies the client if
  // they've changed.
  void SetNetworkState(blink::WebMediaPlayer::NetworkState state);
  void SetReadyState(blink::WebMediaPlayer::ReadyState state);

  // added gstreamer specific functions
  bool loadNextLocation();
  void mediaLocationChanged(GstMessage* message);
  void loadingFailed(blink::WebMediaPlayer::NetworkState error);
  void didEnd();
  void updateStates();
  void processBufferingStats(GstMessage* message);
  void durationChanged();
  void timeChanged();
  void cacheDuration();
  float maxTimeLoaded() const;
  void updateAudioSink();
  bool isLiveStream() const;
  void fillTimerFired();
  int64 totalBytes() const;
  void commitLoad();
  float playbackPosition() const;
  bool changePipelineState(GstState new_state);
  void prepareToPlay();
  void cancelLoad();
  void loadStateChanged();
  void setScreenRect(const blink::WebRect& r);
  void createWebLayerAndHolePunchFrame();
  static void GlibMainLoopFunc();
  void handleMessageMultiThreadedMain(GstMessage* message);

  // methods to store gstreamer messages arriving on another thread
  // all these methods use a lock and are thread safe
  void lockMessageMap(bool lock);
  bool isMessageMapLocked();
  void addToMessageMap(GstMessage* message);
  void removeFromMessageMap(GstMessage* message);
  void clearMessageMap();

  // methods to load platform specific gstreamer strings
  void LoadGstreamerStrings();
  std::string GetGStreamerString(std::string key);

  // additional float versions since gstreamer plugin code
  // expects floats otherwise there are rounding issues
  // make sure that all code internal to plugin calls the
  // float versions
  float maxTimeSeekableF() const;
  float durationF() const;
  float currentTimeF() const;

  // additional functions because of buffered data source and appsrc

  // Called when the data source is downloading or paused.
  void NotifyDownloading(bool is_downloading);

  // Called after asynchronous initialization of a data source completed.
  void DataSourceInitialized(const blink::WebURL& url, bool success);
  void DoLoad(const blink::WebURL& url);
  void SignalReadCompleted(int size);
  bool IsReadInProgress(int av = 0);
  void SetReadInProgress(bool in_progress, int av = 0);
  void SetShouldBeReading(bool is_reading, int av = 0);
  bool ShouldBeReading(int av = 0);
  media::Ranges<base::TimeDelta> GetBufferedTimeRanges();
  base::TimeDelta TimeForByteOffset(int64 byte_offset) const;

  // additional functions because of chunk demuxer
  void OnDemuxerOpened();
  void OnDemuxerInit(media::PipelineStatus status);
  void PipelineStatusHandler(media::PipelineStatus status);
  void OnDemuxerVideoBufferReady(
      media::DemuxerStream::Status status,
      const scoped_refptr<media::DecoderBuffer>& buffer);
  void OnDemuxerAudioBufferReady(
      media::DemuxerStream::Status status,
      const scoped_refptr<media::DecoderBuffer>& buffer);
  void WriteRawVideoFramesToFile(
      float time_till_stop_seconds,
      const scoped_refptr<media::DecoderBuffer>& buffer);
  void WriteRawAudioFramesToFile(
      float time_till_stop_seconds,
      const scoped_refptr<media::DecoderBuffer>& buffer);
  std::string GetAudioConfigAsHumanReadableString(
      const media::AudioDecoderConfig& audio_config);
  void ConstructNormalPipeline(const blink::WebURL& url);
  void ConstructMediaSourcePipeline();
  void DestroyNormalPipeline();
  void DestroyMediaSourcePipeline();

  // additions becuase of decryption

  // Actually do the work for generateKeyRequest/addKey so they can easily
  // report results to UMA.
  MediaKeyException GenerateKeyRequestInternal(
      const blink::WebString& key_system,
      const unsigned char* init_data,
      unsigned init_data_length);
  MediaKeyException AddKeyInternal(const blink::WebString& key_system,
                                   const unsigned char* key,
                                   unsigned key_length,
                                   const unsigned char* init_data,
                                   unsigned init_data_length,
                                   const blink::WebString& session_id);
  MediaKeyException CancelKeyRequestInternal(
      const blink::WebString& key_system,
      const blink::WebString& session_id);

  void DecryptedVideoStreamInitialized(media::PipelineStatus status);
  void DecryptedAudioStreamInitialized(media::PipelineStatus status);

  blink::WebFrame* frame_;

  // Keep a list of buffered time ranges.
  blink::WebTimeRanges buffered_;

  // Message loops for posting tasks on Chrome's main thread. Also used
  // for DCHECKs so methods calls won't execute in the wrong thread.
  const scoped_refptr<base::MessageLoopProxy> main_loop_;

  // scoped_ptr<media::Pipeline> pipeline_;
  scoped_refptr<base::MessageLoopProxy> media_loop_;

  // The currently selected key system. Empty string means that no key system
  // has been selected.
  blink::WebString current_key_system_;

  // The LoadType passed in the |load_type| parameter of the load() call.
  LoadType load_type_;

  blink::WebMediaPlayerClient* client_;

  base::WeakPtr<WebMediaPlayerDelegate> delegate_;

  base::Callback<void(const base::Closure&)> defer_load_cb_;

  scoped_refptr<media::MediaLog> media_log_;

  bool incremented_externally_allocated_memory_;

  // The decryptor that manages decryption keys and decrypts encrypted frames.
  scoped_ptr<ProxyDecryptor> decryptor_;

  // These two are mutually exclusive:
  //   |data_source_| is used for regular resource loads.
  //   |chunk_demuxer_| is used for Media Source resource loads.
  //
  // |demuxer_| will contain the appropriate demuxer based on which resource
  // load strategy we're using.
  scoped_ptr<BufferedDataSource> data_source_;
  scoped_ptr<media::Demuxer> demuxer_;
  media::ChunkDemuxer* chunk_demuxer_;

  // Video frame rendering members.
  scoped_refptr<media::VideoFrame> current_frame_;

  // The compositor layer for displaying the video content when using composited
  // playback.
  scoped_ptr<webkit::WebLayerImpl> video_weblayer_;

  // A pointer back to the compositor to inform it about state changes. This is
  // not NULL while the compositor is actively using this webmediaplayer.
  cc::VideoFrameProvider::Client* video_frame_provider_client_;

  // webkit gstreamer additions
  GstElement* play_bin_;
  GstElement* webkit_video_sink_;
  GstElement* video_sink_bin_;
  GstElement* fps_sink_;

  // GRefPtr<GstElement> source_;
  GstElement* pipeline_;
  GstElement* source_;
  GstElement* decbin_;
  GstElement* audio_decoder_;
  GstElement* video_decoder_;
  GstElement* audio_sink_;
  GstElement* video_sink_;

  blink::WebRect scr_rect_;

  float seek_time_;
  bool changing_rate_;
  float end_time_;
  bool is_end_reached_;
  blink::WebMediaPlayer::NetworkState network_state_;
  blink::WebMediaPlayer::ReadyState ready_state_;
  blink::WebSize size_;
  GstBuffer* buffer_;
  GstStructure* media_locations_;
  int media_location_current_index_;
  bool reset_pipeline_;
  bool paused_;
  bool seeking_;
  bool buffering_;
  float playback_rate_;
  bool error_occured_;
  gfloat media_duration_;
  bool started_buffering_;
  bool have_ever_started_play_;
  base::RepeatingTimer<WebMediaPlayerGStreamer> fill_timer_;
  float max_time_loaded_;
  int buffering_percentage_;
  WebMediaPlayer::Preload preload_;
  bool delaying_load_;
  bool media_duration_known_;
  mutable float max_time_loaded_at_last_did_loading_progress_;
#ifndef GST_API_VERSION_1
// RefPtr<GStreamerGWorld> m_gstGWorld;
#endif
  guint volume_timer_handler_;
  guint mute_timer_handler_;
  bool has_video_;
  bool has_audio_;
  guint audio_timer_handler_;
  guint video_timer_handler_;
  guint video_seek_timer_handler_;
  GstElement* webkit_audio_sink_;
  // GRefPtr<GstPad> m_videoSinkPad;
  mutable blink::WebSize video_size_;
  blink::WebURL url_;
  bool original_preload_was_auto_and_was_overridden_;

  static GMainLoop* gMainLoop;
  static base::Thread* gMainLoopThread;

  base::Lock lock_;
  std::map<GstMessage*, GstMessage*> message_map_;
  bool message_map_locked_;

  static const std::string kStringsFilePath;
  scoped_ptr<base::Value> gstreamer_json_data_;
  base::DictionaryValue* gstreamer_strings_;

  // additions because of buffered data source and app_src
  static const int32_t kReadSizeBytes = 32768;  // 1024;
  scoped_ptr<media::GStreamerDemuxerHost> demuxer_host_;
  GstAppSrc* appsrc_source_;
  GstAppSrc* appsrc_source_audio_;
  base::WaitableEvent aborted_;
  // Cached number of bytes last read from the data source.
  int32_t last_read_bytes_;
  // Cached position within the data source.
  int64 read_position_;
  base::Lock read_lock_;
  guint8* appsrc_data_;
  bool read_in_progress_[2];
  bool should_be_reading_[2];

  // additions because of chunk demuxer

  bool supports_save_;
  bool is_local_source_;

  // Temporary for EME v0.1. In the future the init data type should be passed
  // through GenerateKeyRequest() directly from WebKit.
  std::string init_data_type_;

  scoped_ptr<media::DecryptingDemuxerStream> decrypted_video_stream_;
  scoped_ptr<media::DecryptingDemuxerStream> decrypted_audio_stream_;
  bool decrypted_video_stream_initialized_;
  bool decrypted_audio_stream_initialized_;

  DISALLOW_COPY_AND_ASSIGN(WebMediaPlayerGStreamer);
};

}  // namespace content

#endif  // CONTENT_RENDERER_MEDIA_WEBMEDIAPLAYER_GSTREAMER_H_

#endif  // GSTREAMER_HARDWARE_VIDEO_DECODING
// COMCAST MODIFICATION END [GSTREAMER_HARDWARE_VIDEO_DECODING]
