/*
 * Copyright (C) 2009 Google Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above
 * copyright notice, this list of conditions and the following disclaimer
 * in the documentation and/or other materials provided with the
 * distribution.
 *     * Neither the name of Google Inc. nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef WebDocument_h
#define WebDocument_h

#include "../platform/WebReferrerPolicy.h"
#include "../platform/WebVector.h"
#include "WebDraggableRegion.h"
#include "WebExceptionCode.h"
#include "WebNode.h"
#include "WebSecurityOrigin.h"

#if BLINK_IMPLEMENTATION
namespace WebCore {
class Document;
class DocumentType;
}
namespace WTF { template <typename T> class PassRefPtr; }
#endif

namespace v8 {
class Value;
template <class T> class Handle;
}

namespace blink {
class WebAXObject;
class WebDocumentType;
class WebElement;
class WebFormElement;
class WebFrame;
class WebNodeCollection;
class WebNodeList;
class WebString;
class WebURL;

// Provides readonly access to some properties of a DOM document.
class WebDocument : public WebNode {
public:
    // FIXME: Stop using this from Chromium code and get rid of this enum.
    enum UserStyleLevel {
        UserStyleAuthorLevel
    };

    WebDocument() { }
    WebDocument(const WebDocument& e) : WebNode(e) { }

    WebDocument& operator=(const WebDocument& e)
    {
        WebNode::assign(e);
        return *this;
    }
    void assign(const WebDocument& e) { WebNode::assign(e); }

    BLINK_EXPORT WebURL url() const;
    // Note: Security checks should use the securityOrigin(), not url().
    BLINK_EXPORT WebSecurityOrigin securityOrigin() const;

    BLINK_EXPORT WebString encoding() const;
    BLINK_EXPORT WebString contentLanguage() const;
    BLINK_EXPORT WebString referrer() const;

    // The url of the OpenSearch Desription Document (if any).
    BLINK_EXPORT WebURL openSearchDescriptionURL() const;

    // Returns the frame the document belongs to or 0 if the document is frameless.
    BLINK_EXPORT WebFrame* frame() const;
    BLINK_EXPORT bool isHTMLDocument() const;
    BLINK_EXPORT bool isXHTMLDocument() const;
    BLINK_EXPORT bool isPluginDocument() const;
    BLINK_EXPORT WebURL baseURL() const;

    // The firstPartyForCookies is used to compute whether this document
    // appears in a "third-party" context for the purpose of third-party
    // cookie blocking.
    BLINK_EXPORT WebURL firstPartyForCookies() const;

    BLINK_EXPORT WebElement documentElement() const;
    BLINK_EXPORT WebElement body() const;
    BLINK_EXPORT WebElement head();
    BLINK_EXPORT WebString title() const;
    BLINK_EXPORT WebNodeCollection all();
    BLINK_EXPORT void forms(WebVector<WebFormElement>&) const;
    BLINK_EXPORT void images(WebVector<WebElement>&);
    BLINK_EXPORT WebURL completeURL(const WebString&) const;
    BLINK_EXPORT WebElement getElementById(const WebString&) const;
    BLINK_EXPORT WebNode focusedNode() const;
    BLINK_EXPORT WebDocumentType doctype() const;
    BLINK_EXPORT void cancelFullScreen();
    BLINK_EXPORT WebElement fullScreenElement() const;
    BLINK_EXPORT WebDOMEvent createEvent(const WebString& eventType);
    BLINK_EXPORT WebReferrerPolicy referrerPolicy() const;
    BLINK_EXPORT WebElement createElement(const WebString& tagName);

    // Accessibility support. These methods should only be called on the
    // top-level document, because one accessibility cache spans all of
    // the documents on the page.

    // Gets the accessibility object for this document.
    BLINK_EXPORT WebAXObject accessibilityObject() const;

    // Gets the accessibility object for an object on this page by ID.
    BLINK_EXPORT WebAXObject accessibilityObjectFromID(int axID) const;
    // Inserts the given CSS source code as a stylesheet in the document.
    // FIXME: Delete insertUserStyleSheet once Chromium code stops calling it.
    BLINK_EXPORT void insertUserStyleSheet(const WebString& sourceCode, UserStyleLevel);
    BLINK_EXPORT void insertStyleSheet(const WebString& sourceCode);

    // Arranges to call WebFrameClient::didMatchCSS(frame(), ...) when one of
    // the selectors matches or stops matching an element in this document.
    // Each call to this method overrides any previous calls.
    BLINK_EXPORT void watchCSSSelectors(const WebVector<WebString>& selectors);

    BLINK_EXPORT WebVector<WebDraggableRegion> draggableRegions() const;

    BLINK_EXPORT v8::Handle<v8::Value> registerEmbedderCustomElement(const WebString& name, v8::Handle<v8::Value> options, WebExceptionCode&);

#if BLINK_IMPLEMENTATION
    WebDocument(const WTF::PassRefPtr<WebCore::Document>&);
    WebDocument& operator=(const WTF::PassRefPtr<WebCore::Document>&);
    operator WTF::PassRefPtr<WebCore::Document>() const;
#endif
};

} // namespace blink

#endif
