/*
 * Copyright (C) 2013 TopCoder Inc., All Rights Reserved.
 */

#include "usermediapermissionwindow.h"
#include <QtWebRTC/qinclude.h>
#include <QtWebKitWidgets/qwebpage.h>

#ifdef WIN32
static const int kDefaultAudioDeviceId = -1;
static const int kDefaultSoundclipDeviceId = -2;
#else
static const int kDefaultAudioDeviceId = 0;
#endif

using namespace WebCore;

QUserMediaPermissionWindow::QUserMediaPermissionWindow(QWebPage* webPage, QWebFrame* webFrame, QWidget* parent)
    : QDialog(parent), m_webPage(webPage), m_webFrame(webFrame)
{
}

bool QUserMediaPermissionWindow::init()
{
    QRTCPeerConnectionHandlerWrapper::getVideoCaptureDevices(m_videoDevices);
    QRTCPeerConnectionHandlerWrapper::getAudioInputDevices(m_audioDevices);

    if (m_videoDevices.count() == 0) {
        denyRequest();
        return false;
    }

    if (m_audioDevices.count() == 0) {
        // add the default device
        m_audioDevices.append(QSharedPointer<QDevice>(new QDevice(QString::fromUtf8(""), QString::fromUtf8("0"))));
    }

    m_yesButton = createButton(tr("&Yes"), SLOT(acceptRequest()));
    m_noButton = createButton(tr("&No"), SLOT(denyRequest()));

    m_audioDevice = m_audioDevices.at(0);
    m_videoDevice = m_videoDevices.at(0);

    m_videoComboBox = createComboBox(SLOT(videoDeviceChanged(int)));
    m_audioComboBox = createComboBox(SLOT(audioDeviceChanged(int)));

    for (int i = 0; i < m_videoDevices.count(); i++) {
        m_videoComboBox->addItem(m_videoDevices.at(i).data()->name());
    }

    for (int i = 0; i < m_audioDevices.count(); i++) {
        m_audioComboBox->addItem(m_audioDevices.at(i).data()->id().toInt() != kDefaultAudioDeviceId &&
            !m_audioDevices.at(i).data()->name().isEmpty() ?
            m_audioDevices.at(i).data()->name() : QString::fromUtf8("Default device"));
    }

    m_videoComboBox->setCurrentIndex(0);
    m_audioComboBox->setCurrentIndex(0);

    m_firstLabel = new QLabel(tr("Do you want to allow access to camera "));
    m_secondLabel = new QLabel(tr(" and microphone "));
    QGridLayout *mainLayout = new QGridLayout;

    mainLayout->addWidget(m_firstLabel, 0, 0);
    mainLayout->addWidget(m_videoComboBox, 0, 1);
    mainLayout->addWidget(m_secondLabel, 1, 0);
    mainLayout->addWidget(m_audioComboBox, 1, 1);

    mainLayout->addWidget(m_yesButton, 2, 0);
    mainLayout->addWidget(m_noButton, 2, 1);

    setLayout(mainLayout);

    setWindowTitle(tr("Page is requesting permission to access your hardware"));

    resize(700, 100);

    return true;
}

void QUserMediaPermissionWindow::audioDeviceChanged(int index)
{
    m_audioDevice = m_audioDevices.at(index);
}

void QUserMediaPermissionWindow::videoDeviceChanged(int index)
{
    m_videoDevice = m_videoDevices.at(index);
}

void QUserMediaPermissionWindow::acceptRequest()
{
    if (!m_webPage) {
        return;
    }

    // TODO: implement setPreferredAVDevices api
    // m_webPage->setPreferredAVDevices(m_webFrame, m_audioDevice, m_videoDevice);
    m_webPage->setFeaturePermission(m_webFrame, QWebPage::WebRTC, QWebPage::PermissionGrantedByUser);
    m_webPage = NULL;
    close();
}

void QUserMediaPermissionWindow::denyRequest()
{
    if (!m_webPage) {
        return;
    }    

    m_webPage->setFeaturePermission(m_webFrame, QWebPage::WebRTC, QWebPage::PermissionDeniedByUser);
    m_webPage = NULL;
    close();
}

void QUserMediaPermissionWindow::closeEvent(QCloseEvent* )
{
    if (!m_webPage) {
        return;
    }

    m_webPage->setFeaturePermission(m_webFrame, QWebPage::WebRTC, QWebPage::PermissionDeniedByUser);
    m_webPage = NULL;
}

QPushButton *QUserMediaPermissionWindow::createButton(const QString &text, const char *member)
{
    QPushButton *button = new QPushButton(text);
    connect(button, SIGNAL(clicked()), this, member);
    return button;
}

QComboBox *QUserMediaPermissionWindow::createComboBox(const char *member)
{
    QComboBox *comboBox = new QComboBox;
    comboBox->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Preferred);
    connect(comboBox, SIGNAL(currentIndexChanged(int)), this, member);
    return comboBox;
}
