#include <iostream>
#include <sstream>

#include "ppapi/c/pp_errors.h"
#include "ppapi/c/ppb_console.h"
#include "ppapi/cpp/core.h"
#include "ppapi/cpp/graphics_3d.h"
#include "ppapi/cpp/graphics_3d_client.h"
#include "ppapi/cpp/instance.h"
#include "ppapi/cpp/module.h"
#include "ppapi/cpp/rect.h"
#include "ppapi/lib/gl/gles2/gl2ext_ppapi.h"
#include "ppapi/lib/gl/include/GLES2/gl2.h"
#include "ppapi/utility/completion_callback_factory.h"

namespace {

class HolePunchInstance : public pp::Instance, public pp::Graphics3DClient {
 public:
  explicit HolePunchInstance(PP_Instance instance);
  virtual ~HolePunchInstance();

  // pp::Instance implementation (see PPP_Instance).
  virtual bool Init(uint32_t argc, const char* argn[], const char* argv[]);
  virtual void DidChangeView(const pp::Rect& position,
                             const pp::Rect& clip);

  // pp::Graphics3DClient implementation.
  virtual void Graphics3DContextLost();

 private:
  class LogError {
   public:
    LogError(HolePunchInstance* instance) : instance_(instance) {}
    ~LogError() {
      const std::string& msg = stream_.str();
      instance_->console_if_->Log(
          instance_->pp_instance(), PP_LOGLEVEL_ERROR, pp::Var(msg).pp_var());
      std::cerr << msg << std::endl;
    }
    std::ostringstream& s() { return stream_; }
   private:
    HolePunchInstance* instance_;
    std::ostringstream stream_;
  };

  // GL-related functions.
  void InitGL(int32_t result);
  void Paint(int32_t result);
  void PaintFinished(int32_t result);

  bool is_painting_;
  bool needs_paint_;
  pp::Size plugin_size_;
  pp::CompletionCallbackFactory<HolePunchInstance> callback_factory_;

  // Unowned pointers.
  const PPB_Console* console_if_;

  // Owned data.
  pp::Graphics3D* context_;
};

// This object is the global object representing this plugin library as long
// as it is loaded.
class HolePunchModule : public pp::Module {
 public:
  HolePunchModule() : pp::Module() {}
  virtual ~HolePunchModule() {}

  // Override CreateInstance to create your customized Instance object.
  virtual pp::Instance* CreateInstance(PP_Instance instance) {
    return new HolePunchInstance(instance);
  }
};

HolePunchInstance::HolePunchInstance(PP_Instance instance)
    : pp::Instance(instance),
      pp::Graphics3DClient(this),
      is_painting_(false),
      needs_paint_(false),
      callback_factory_(this),
      context_(NULL) {
  console_if_ = static_cast<const PPB_Console*>(
      pp::Module::Get()->GetBrowserInterface(PPB_CONSOLE_INTERFACE));
}

HolePunchInstance::~HolePunchInstance() {
  glTerminatePPAPI();
  delete context_;
}

bool HolePunchInstance::Init(uint32_t /*argc*/,
                             const char* /*argn*/[],
                             const char* /*argv*/[]) {
  return !!glInitializePPAPI(pp::Module::Get()->get_browser_interface());
}

void HolePunchInstance::DidChangeView(
    const pp::Rect& position, const pp::Rect& /*clip*/) {
  if (position.width() == 0 || position.height() == 0)
    return;
  plugin_size_ = position.size();

  // Initialize graphics.
  InitGL(0);
}

void HolePunchInstance::Graphics3DContextLost() {
  delete context_;
  context_ = NULL;
  pp::CompletionCallback cb = callback_factory_.NewCallback(
      &HolePunchInstance::InitGL);
  pp::Module::Get()->core()->CallOnMainThread(0, cb, 0);
}

void HolePunchInstance::InitGL(int32_t /*result*/) {
  PP_DCHECK(plugin_size_.width() && plugin_size_.height());

  if (context_) {
    context_->ResizeBuffers(plugin_size_.width(), plugin_size_.height());
    Paint(PP_OK);
    return;
  }
  int32_t context_attributes[] = {
    PP_GRAPHICS3DATTRIB_ALPHA_SIZE, 8,
    PP_GRAPHICS3DATTRIB_BLUE_SIZE, 8,
    PP_GRAPHICS3DATTRIB_GREEN_SIZE, 8,
    PP_GRAPHICS3DATTRIB_RED_SIZE, 8,
    PP_GRAPHICS3DATTRIB_DEPTH_SIZE, 0,
    PP_GRAPHICS3DATTRIB_STENCIL_SIZE, 0,
    PP_GRAPHICS3DATTRIB_SAMPLES, 0,
    PP_GRAPHICS3DATTRIB_SAMPLE_BUFFERS, 0,
    PP_GRAPHICS3DATTRIB_WIDTH, plugin_size_.width(),
    PP_GRAPHICS3DATTRIB_HEIGHT, plugin_size_.height(),
    PP_GRAPHICS3DATTRIB_HOLE_PUNCH, 1,
    PP_GRAPHICS3DATTRIB_NONE,
  };
  context_ = new pp::Graphics3D(this, context_attributes);
  PP_DCHECK(!context_->is_null());
  BindGraphics(*context_);

  glSetCurrentContextPPAPI(context_->pp_resource());
  Paint(PP_OK);
}

void HolePunchInstance::Paint(int32_t result) {
  if (result != PP_OK || !context_)
    return;

  if (is_painting_) {
      needs_paint_ = true;
      return;
  }
  is_painting_ = true;
  needs_paint_ = false;

  // Punch a hole
  glViewport(0, 0, plugin_size_.width(), plugin_size_.height());
  glEnable(GL_SCISSOR_TEST);
  glScissor(0, 0, plugin_size_.width(), plugin_size_.height());
  glClearColor(0, 0, 0, 0);
  glClear(GL_COLOR_BUFFER_BIT);
  glDisable(GL_SCISSOR_TEST);

  GLenum err;
  while ((err = glGetError()) != GL_NO_ERROR) {
      LogError(this).s() << "XXX glError:" << err;
  }

  pp::CompletionCallback cb = callback_factory_.NewCallback(
      &HolePunchInstance::PaintFinished);
  context_->SwapBuffers(cb);
}

void HolePunchInstance::PaintFinished(int32_t result) {
  is_painting_ = false;
  if (needs_paint_)
    Paint(result);
}

}  // anonymous namespace

namespace pp {

// Factory function for your specialization of the Module object.
Module* CreateModule() {
  return new HolePunchModule();
}

}  // namespace pp
