/*
 * If not stated otherwise in this file or this component's Licenses.txt file the
 * following copyright and licenses apply:
 *
 * Copyright 2017 RDK Management
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "essos.h"

#include <stdlib.h>
#include <stdio.h>
#include <stdint.h>
#include <string.h>
#include <memory.h>
#include <assert.h>
#include <errno.h>
#include <unistd.h>
#include <linux/input.h>
#include <linux/joystick.h>
#include <dlfcn.h>
#include <poll.h>
#include <pthread.h>
#include <sys/time.h>

#include <vector>
#include <map>
#include <string>

#ifdef HAVE_WAYLAND
#include <xkbcommon/xkbcommon.h>
#include <sys/mman.h>
#include "wayland-client.h"
#include "wayland-egl.h"
#endif

#ifdef HAVE_WESTEROS
#include "westeros-gl.h"
#include <dirent.h>
#include <fcntl.h>
#include <sys/inotify.h>
#include <sys/stat.h>
#include <sys/types.h>
#include "simpleshell-client-protocol.h"
#endif

#include "westeros-version.h"

#define ESS_UNUSED(x) ((void)x)
#define ESS_MAX_ERROR_DETAIL (512)
#define DEFAULT_PLANE_WIDTH (1280)
#define DEFAULT_PLANE_HEIGHT (720)
#define DEFAULT_PLANE_SAFE_BORDER_PERCENT (5)
#define DEFAULT_KEY_REPEAT_DELAY (500)
#define DEFAULT_KEY_REPEAT_PERIOD (100)

#define INT_FATAL(FORMAT, ...)      essLog(0, "Essos Fatal: " FORMAT "\n", ##__VA_ARGS__)
#define INT_ERROR(FORMAT, ...)      essLog(0, "Essos Error: " FORMAT "\n", ##__VA_ARGS__)
#define INT_WARNING(FORMAT, ...)    essLog(1, "Essos Warning: " FORMAT "\n",  ##__VA_ARGS__)
#define INT_INFO(FORMAT, ...)       essLog(2, "Essos Info: " FORMAT "\n",  ##__VA_ARGS__)
#define INT_DEBUG(FORMAT, ...)      essLog(3, "Essos Debug: " FORMAT "\n",  ##__VA_ARGS__)
#define INT_TRACE(FORMAT, ...)      essLog(4, "Essos Trace: " FORMAT "\n",  ##__VA_ARGS__)

#define FATAL(FORMAT, ...)          INT_FATAL(FORMAT, ##__VA_ARGS__)
#define ERROR(FORMAT, ...)          INT_ERROR(FORMAT, ##__VA_ARGS__)
#define WARNING(FORMAT, ...)        INT_WARNING(FORMAT, ##__VA_ARGS__)
#define INFO(FORMAT, ...)           INT_INFO(FORMAT, ##__VA_ARGS__)
#define DEBUG(FORMAT, ...)          INT_DEBUG(FORMAT, ##__VA_ARGS__)
#define TRACE(FORMAT, ...)          INT_TRACE(FORMAT, ##__VA_ARGS__)

#define ESS_INPUT_POLL_LIMIT (10)
#define ESS_MAX_TOUCH (10)

typedef struct _EssTouchInfo
{
   int id;
   int x;
   int y;
   bool valid;
   bool starting;
   bool stopping;
   bool moved;
} EssTouchInfo;

typedef struct _EssInputDeviceScanCode
{
   uint8_t filterByte;
   uint8_t prevFilterByte;
   uint8_t shortCustomerCode;
} EssInputDeviceScanCode;

typedef struct _EssInputDevice
{
   EssCtx *ctx;
   const char *devicePath;
   int fd;
} EssInputDevice;

typedef struct _EssGamepad
{
   EssCtx *ctx;
   const char *devicePath;
   const char *name;
   int fd;
   int buttonCount;
   uint16_t buttonMap[KEY_MAX - BTN_MISC + 1];
   int buttonState[KEY_MAX - BTN_MISC + 1];
   int axisCount;
   uint8_t axisMap[ABS_CNT];
   int axisState[ABS_CNT];
   void *eventListenerUserData;
   EssGamepadEventListener *eventListener;
} EssGamepad;

typedef struct _EssCtx
{
   pthread_mutex_t mutex;
   bool autoMode;
   bool isWayland;
   bool isInitialized;
   bool isRunning;
   bool isExternalEGL;
   char lastErrorDetail[ESS_MAX_ERROR_DETAIL];

   bool haveMode;
   int planeWidth;
   int planeHeight;
   int planeSafeX;
   int planeSafeY;
   int planeSafeW;
   int planeSafeH;
   int windowX;
   int windowY;
   int windowWidth;
   int windowHeight;
   bool pendingGeometryChange;
   bool fullScreen;
   const char *appName;
   uint32_t appSurfaceId;
   int waylandFd;
   pollfd wlPollFd;
   int notifyFd;
   int watchFd;
   std::vector<pollfd> inputDeviceFds;
   std::map<int, EssInputDeviceMetadata*> inputDeviceMetadata;
   std::map<int, EssInputDeviceScanCode> inputDeviceScanCode;
   std::vector<EssInputDevice*> inputDevices;
   std::vector<EssGamepad*> gamepads;
   int eventLoopPeriodMS;
   long long eventLoopLastTimeStamp;

   int pointerX;
   int pointerY;

   long long lastKeyTime;
   int lastKeyCode;
   bool keyPressed;
   bool keyRepeating;
   int keyRepeatInitialDelay;
   int keyRepeatPeriod;

   EssTouchInfo touch[ESS_MAX_TOUCH];

   void *keyListenerUserData;
   EssKeyListener *keyListener;
   void *keyAndMetadataListenerUserData;
   EssInputDeviceMetadata *keyAndMetadataListenerMetadata;
   EssKeyAndMetadataListener *keyAndMetadataListener;
   void *pointerListenerUserData;
   EssPointerListener *pointerListener;
   void *touchListenerUserData;
   EssTouchListener *touchListener;
   void *gamepadConnectionListenerUserData;
   EssGamepadConnectionListener *gamepadConnectionListener;
   void *settingsListenerUserData;
   EssSettingsListener *settingsListener;
   void *terminateListenerUserData;
   EssTerminateListener *terminateListener;

   NativeDisplayType displayType;
   NativeWindowType nativeWindow;
   EGLDisplay eglDisplay;
   EGLint eglVersionMajor;
   EGLint eglVersionMinor;
   EGLint *eglCfgAttr;
   EGLint eglCfgAttrSize;
   EGLConfig eglConfig;
   EGLint *eglCtxAttr;
   EGLint eglCtxAttrSize;
   EGLint *eglSurfAttr;
   EGLint eglSurfAttrSize;
   EGLContext eglContext;
   EGLSurface eglSurfaceWindow;
   EGLint eglSwapInterval;

   bool resizePending;
   int resizeWidth;
   int resizeHeight;

   #ifdef HAVE_WAYLAND
   struct wl_display *wldisplay;
   struct wl_registry *wlregistry;
   struct wl_compositor *wlcompositor;
   struct wl_seat *wlseat;
   struct wl_output *wloutput;
   struct wl_keyboard *wlkeyboard;
   struct wl_pointer *wlpointer;
   struct wl_touch *wltouch;
   struct wl_surface *wlsurface;
   struct wl_simple_shell *shell;
   struct wl_egl_window *wleglwindow;
   struct wl_shell_surface *shellSurface;
   struct wl_shell *wlshell;

   struct xkb_context *xkbCtx;
   struct xkb_keymap *xkbKeymap;
   struct xkb_state *xkbState;
   xkb_mod_index_t modAlt;
   xkb_mod_index_t modCtrl;
   xkb_mod_index_t modShift;
   xkb_mod_index_t modCaps;
   unsigned int modMask;
   #endif
   #ifdef HAVE_WESTEROS
   WstGLCtx *glCtx;
   #endif
} EssCtx;

static void essLog( int level, const char *fmt, ... );
static long long essGetCurrentTimeMillis(void);
static bool essPlatformInit( EssCtx *ctx );
static void essPlatformTerm( EssCtx *ctx );
static bool essEGLInit( EssCtx *ctx );
static void essEGLTerm( EssCtx *ctx );
static void essInitInput( EssCtx *ctx );
static void essSetDisplaySize( EssCtx *ctx, int width, int height, bool customSafe, int safeX, int safeY, int safeW, int safeH );
static bool essCreateNativeWindow( EssCtx *ctx, int width, int height );
static bool essDestroyNativeWindow( EssCtx *ctx, NativeWindowType nw );
static bool essResize( EssCtx *ctx, int width, int height );
static void essRunEventLoopOnce( EssCtx *ctx );
static void essProcessKeyPressed( EssCtx *ctx, int linuxKeyCode );
static void essProcessKeyReleased( EssCtx *ctx, int linuxKeyCode );
static void essProcessKeyRepeat( EssCtx *ctx, int linuxKeyCode );
static void essProcessPointerMotion( EssCtx *ctx, int x, int y );
static void essProcessPointerButtonPressed( EssCtx *ctx, int button );
static void essProcessPointerButtonReleased( EssCtx *ctx, int button );
static void essProcessTouchDown( EssCtx *ctx, int id, int x, int y );
static void essProcessTouchUp( EssCtx *ctx, int id );
static void essProcessTouchMotion( EssCtx *ctx, int id, int x, int y );
static void essProcessTouchFrame( EssCtx *ctx );
static void essProcessGamepadButtonPressed( EssGamepad *gp, int button );
static void essProcessGamepadButtonReleased( EssGamepad *gp, int button );
static void essProcessGamepadAxisChanged( EssGamepad *gp, int axis, int value );
#ifdef HAVE_WAYLAND
static bool essPlatformInitWayland( EssCtx *ctx );
static void essPlatformTermWayland( EssCtx *ctx );
static void essProcessRunWaylandEventLoopOnce( EssCtx *ctx );
#endif
#ifdef HAVE_WESTEROS
static bool essPlatformInitDirect( EssCtx *ctx );
static void essPlatformTermDirect( EssCtx *ctx );
static bool essPlatformSetDisplayModeDirect( EssCtx *ctx, const char *mode );
static int essOpenInputDevice( EssCtx *ctx, const char *devPathName );
static char *essGetInputDevice( EssCtx *ctx, const char *path, char *devName );
static void essGetInputDevices( EssCtx *ctx );
static void essMonitorInputDevicesLifecycleBegin( EssCtx *ctx );
static void essMonitorInputDevicesLifecycleEnd( EssCtx *ctx );
static void essReleaseInputDevices( EssCtx *ctx );
static void essProcessInputDevices( EssCtx *ctx );
static void essProcessGamepad( EssCtx *ctx, EssGamepad *gp );
static void essFreeInputDevice( EssInputDevice *idev );
static void essFreeInputDevices( EssCtx *ctx );
static void essValidateInputDevices( EssCtx *ctx );
static EssInputDevice *essGetInputDeviceFromPath( EssCtx *ctx, const char *path );
static void essFreeGamepad( EssCtx *ctx, EssGamepad *gp );
static void essFreeGamepads( EssCtx *ctx );
static void essValidateGamepads( EssCtx *ctx );
static EssGamepad *essGetGamepadFromPath( EssCtx *ctx, const char *path );
static EssGamepad *essGetGamepadFromFd( EssCtx *ctx, int fd );
static void essGamepadNotifyConnected( EssCtx *ctx, EssGamepad *gp );
static void essGamepadNotifyDisconnected( EssCtx *ctx, EssGamepad *gp );
#endif

static EGLint gDefaultEGLCfgAttrSize= 17;
static EGLint gDefaultEGLCfgAttr[]=
{
  EGL_RED_SIZE, 8,
  EGL_GREEN_SIZE, 8,
  EGL_BLUE_SIZE, 8,
  EGL_ALPHA_SIZE, 8,
  EGL_DEPTH_SIZE, 0,
  EGL_STENCIL_SIZE, 0,
  EGL_SURFACE_TYPE, EGL_WINDOW_BIT,
  EGL_RENDERABLE_TYPE, EGL_OPENGL_ES2_BIT,
  EGL_NONE
};

static EGLint gDefaultEGLSurfAttrSize= 1;
static EGLint gDefaultEGLSurfAttr[]=
{
  EGL_NONE
};

static EGLint gDefaultEGLCtxAttrSize= 3;
static EGLint gDefaultEGLCtxAttr[]=
{
  EGL_CONTEXT_CLIENT_VERSION, 2,
  EGL_NONE
};

static int g_activeLevel= 2;

static void essLog( int level, const char *fmt, ... )
{
   if ( level <= g_activeLevel )
   {
      va_list argptr;
      va_start( argptr, fmt );
      vfprintf( stderr, fmt, argptr );
      va_end( argptr );
   }
}

EssCtx* EssContextCreate()
{
   EssCtx *ctx= 0;

   INFO("westeros (essos) version " WESTEROS_VERSION_FMT, WESTEROS_VERSION );

   const char *env= getenv( "ESSOS_DEBUG" );
   if ( env )
   {
      int level= atoi( env );
      g_activeLevel= level;
   }

   ctx= (EssCtx*)calloc( 1, sizeof(EssCtx) );
   if ( ctx )
   {
      pthread_mutex_init( &ctx->mutex, 0 );

      essSetDisplaySize( ctx, DEFAULT_PLANE_WIDTH, DEFAULT_PLANE_HEIGHT, false, 0, 0, 0, 0);
      ctx->windowWidth= DEFAULT_PLANE_WIDTH;
      ctx->windowHeight= DEFAULT_PLANE_HEIGHT;
      ctx->autoMode= true;
      ctx->notifyFd= -1;
      ctx->watchFd= -1;
      ctx->waylandFd= -1;
      ctx->eventLoopPeriodMS= 16;
      if ( getenv("ESSOS_NO_EVENT_LOOP_THROTTLE") )
      {
         ctx->eventLoopPeriodMS= 0;
      }

      ctx->keyRepeatInitialDelay= DEFAULT_KEY_REPEAT_DELAY;
      ctx->keyRepeatPeriod= DEFAULT_KEY_REPEAT_PERIOD;

      ctx->displayType= EGL_DEFAULT_DISPLAY;
      ctx->eglDisplay= EGL_NO_DISPLAY;
      ctx->eglCfgAttr= gDefaultEGLCfgAttr;
      ctx->eglCfgAttrSize= gDefaultEGLCfgAttrSize;
      ctx->eglCtxAttr= gDefaultEGLCtxAttr;
      ctx->eglCtxAttrSize= gDefaultEGLCtxAttrSize;
      ctx->eglContext= EGL_NO_CONTEXT;
      ctx->eglSurfaceWindow= EGL_NO_SURFACE;
      ctx->eglSwapInterval= 1;

      ctx->inputDeviceFds= std::vector<pollfd>();
      ctx->inputDevices= std::vector<EssInputDevice*>();
      ctx->gamepads= std::vector<EssGamepad*>();
      ctx->inputDeviceMetadata = std::map<int, EssInputDeviceMetadata*>();
      ctx->inputDeviceScanCode = std::map<int, EssInputDeviceScanCode>();

      INFO("westeros (essos) config supports: direct %d wayland %d", EssContextSupportDirect(ctx), EssContextSupportWayland(ctx));
   }

   return ctx;
}

void EssContextDestroy( EssCtx *ctx )
{
   if ( ctx )
   {
      if ( ctx->isRunning )
      {
         EssContextStop( ctx );
      }

      if ( ctx->isInitialized )
      {
         essPlatformTerm( ctx );
      }

      if ( ctx->appName )
      {
         free( (char*)ctx->appName );
      }

      pthread_mutex_destroy( &ctx->mutex );
      
      free( ctx );
   }
}

const char *EssContextGetLastErrorDetail( EssCtx *ctx )
{
   const char *detail= 0;
   
   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );
      
      detail= ctx->lastErrorDetail;
      
      pthread_mutex_unlock( &ctx->mutex );
   }
   
   return detail;
}

bool EssContextSupportWayland( EssCtx *ctx )
{
   #ifdef HAVE_WAYLAND
   return true;
   #else
   return false;
   #endif
}

bool EssContextSupportDirect( EssCtx *ctx )
{
   #ifdef HAVE_WESTEROS
   // We use westeros-gl to hide SOC specifics of EGL
   return true;
   #else
   return false;
   #endif
}

bool EssContextSetUseWayland( EssCtx *ctx, bool useWayland )
{
   bool result= false;
                  
   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );

      if ( ctx->isInitialized )
      {
         sprintf( ctx->lastErrorDetail,
                  "Bad state.  Can't change application type when already initialized" );
         pthread_mutex_unlock( &ctx->mutex );
         goto exit;
      }

      #ifndef HAVE_WAYLAND
      if ( useWayland )
      {
         sprintf( ctx->lastErrorDetail,
                  "Error.  Wayland mode is not available" );
         pthread_mutex_unlock( &ctx->mutex );
         goto exit;
      }
      #endif

      #ifndef HAVE_WESTEROS
      if ( !useWayland )
      {
         sprintf( ctx->lastErrorDetail,
                  "Error.  Direct mode is not available" );
         pthread_mutex_unlock( &ctx->mutex );
         goto exit;
      }
      #endif

      ctx->autoMode= false;
      ctx->isWayland= useWayland;

      pthread_mutex_unlock( &ctx->mutex );

      result= true;
   }

exit:   

   return result;
}

bool EssContextGetUseWayland( EssCtx *ctx )
{
   bool result= false;
                  
   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );

      result= ctx->isWayland;

      pthread_mutex_unlock( &ctx->mutex );
   }

exit:   

   return result;      
}

bool EssContextSetUseDirect( EssCtx *ctx, bool useDirect )
{
   return EssContextSetUseWayland( ctx, !useDirect );
}

bool EssContextGetUseDirect( EssCtx *ctx )
{
   return !EssContextGetUseWayland( ctx );
}

bool EssContextSetName( EssCtx *ctx, const char *name )
{
   bool result= false;

   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );

      if ( ctx->isInitialized )
      {
         sprintf( ctx->lastErrorDetail,
                  "Bad state.  Can't change application name when already initialized" );
         pthread_mutex_unlock( &ctx->mutex );
         goto exit;
      }

      if ( !name )
      {
         sprintf( ctx->lastErrorDetail,
                  "Invalid parameter: name is null" );
         pthread_mutex_unlock( &ctx->mutex );
         goto exit;
      }

      ctx->appName= strdup(name);;

      pthread_mutex_unlock( &ctx->mutex );

      result= true;
   }

exit:

   return result;
}

bool EssContextSetEGLConfigAttributes( EssCtx *ctx, EGLint *attrs, EGLint size )
{
   bool result= false;

   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );

      ctx->eglCfgAttr= (attrs ? attrs : gDefaultEGLCfgAttr);
      ctx->eglCfgAttrSize= (attrs ? size : gDefaultEGLCfgAttrSize);

      pthread_mutex_unlock( &ctx->mutex );

      result= true;
   }

   return result;
}

bool EssContextGetEGLConfigAttributes( EssCtx *ctx, EGLint **attrs, EGLint *size )
{
   bool result= false;

   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );

      if ( attrs && size )
      {
         *attrs= ctx->eglCfgAttr;
         *size= ctx->eglCfgAttrSize;
         result= true;
      }

      pthread_mutex_unlock( &ctx->mutex );
   }

   return result;
}

bool EssContextSetEGLSurfaceAttributes( EssCtx *ctx, EGLint *attrs, EGLint size )
{
   bool result= false;

   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );

      ctx->eglSurfAttr= (attrs ? attrs : gDefaultEGLSurfAttr);
      ctx->eglSurfAttrSize= (attrs ? size : gDefaultEGLSurfAttrSize);

      pthread_mutex_unlock( &ctx->mutex );

      result= true;
   }

   return result;
}

bool EssContextGetEGLSurfaceAttributes( EssCtx *ctx, EGLint **attrs, EGLint *size )
{
   bool result= false;

   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );

      if ( attrs && size )
      {
         *attrs= ctx->eglSurfAttr;
         *size= ctx->eglSurfAttrSize;
         result= true;
      }

      pthread_mutex_unlock( &ctx->mutex );
   }

   return result;
}

bool EssContextSetEGLContextAttributes( EssCtx *ctx, EGLint *attrs, EGLint size )
{
   bool result= false;

   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );

      ctx->eglCtxAttr= (attrs ? attrs : gDefaultEGLCtxAttr);
      ctx->eglCtxAttrSize= (attrs ? size : gDefaultEGLCtxAttrSize);

      pthread_mutex_unlock( &ctx->mutex );

      result= true;
   }

   return result;
}

bool EssContextGetEGLContextAttributes( EssCtx *ctx, EGLint **attrs, EGLint *size )
{
   bool result= false;

   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );

      if ( attrs && size )
      {
         *attrs= ctx->eglCtxAttr;
         *size= ctx->eglCtxAttrSize;
         result= true;
      }

      pthread_mutex_unlock( &ctx->mutex );
   }

   return result;
}

bool EssContextSetDisplayMode( EssCtx *ctx, const char *mode )
{
   bool result= false;

   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );

      if ( !mode )
      {
         sprintf( ctx->lastErrorDetail,
                  "Invalid parameter: mode is null" );
         pthread_mutex_unlock( &ctx->mutex );
         goto exit;
      }

      if ( ctx->isWayland )
      {
         WARNING("EssContextSetDisplayMode ignored for Wayland");
         result= true;
      }
      #ifdef HAVE_WESTEROS
      else
      {
         result= essPlatformSetDisplayModeDirect( ctx, mode );
      }
      #endif

      pthread_mutex_unlock( &ctx->mutex );
   }

exit:

   return result;
}

bool EssContextSetInitialWindowSize( EssCtx *ctx, int width, int height )
{
   bool result= false;
                  
   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );
      
      if ( ctx->isRunning )
      {
         pthread_mutex_unlock( &ctx->mutex );
         WARNING("EssContextSetInitialWindowSize: already running: calling resize (%d,%d)", width, height);
         result= EssContextResizeWindow( ctx, width, height );
         goto exit;
      }

      ctx->haveMode= true;
      ctx->windowWidth= width;
      ctx->windowHeight= height;

      pthread_mutex_unlock( &ctx->mutex );

      result= true;
   }

exit:   

   return result;
}

bool EssContextSetSwapInterval( EssCtx *ctx, EGLint swapInterval )
{
   bool result= false;
                  
   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );
      
      if ( ctx->isRunning )
      {
         sprintf( ctx->lastErrorDetail,
                  "Bad state.  Context is already running" );
         pthread_mutex_unlock( &ctx->mutex );
         goto exit;
      }

      ctx->eglSwapInterval= swapInterval;

      pthread_mutex_unlock( &ctx->mutex );

      result= true;
   }

exit:   

   return result;
}

bool EssContextInit( EssCtx *ctx )
{
   bool result= false;

   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );
      
      if ( ctx->isRunning )
      {
         sprintf( ctx->lastErrorDetail,
                  "Bad state.  Context is already running" );
         pthread_mutex_unlock( &ctx->mutex );
         goto exit;
      }

      if ( ctx->autoMode )
      {
         ctx->isWayland= ((getenv("WAYLAND_DISPLAY") != 0) ? true : false );
         INFO("auto set mode: isWayland %d", ctx->isWayland);
      }

      pthread_mutex_unlock( &ctx->mutex );

      if ( !EssContextSetUseWayland( ctx, ctx->isWayland ) )
      {
         goto exit;
      }

      result= essPlatformInit(ctx);
      if ( result )
      {
         ctx->isInitialized= true;
      }
   }

exit:   

   return result;
}

bool EssContextGetEGLDisplayType( EssCtx *ctx, NativeDisplayType *displayType )
{
   bool result= false;

   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );

      if ( !ctx->isInitialized )
      {
         sprintf( ctx->lastErrorDetail,
                  "Bad state.  Must initialize before getting display type" );
         pthread_mutex_unlock( &ctx->mutex );
         goto exit;
      }

      *displayType= ctx->displayType;

      pthread_mutex_unlock( &ctx->mutex );

      result= true;
   }

exit:
   return result;
}

EssAppPlatformDisplayType EssContextGetAppPlatformDisplayType( EssCtx *ctx )
{
   EssAppPlatformDisplayType displayType= EssAppPlatformDisplayType_direct;

   bool useWayland= EssContextGetUseWayland(ctx);

   if (useWayland)
   {
      displayType= EssAppPlatformDisplayType_wayland;
      #ifdef EGL_PLATFORM_WAYLAND_EXT
       #ifdef USE_ESS_BRCM_UPSTREAM_WAYLAND
         displayType= EssAppPlatformDisplayType_waylandExtension;
         INFO("prefering platform extensions");
       #endif //USE_ESS_BRCM_UPSTREAM_WAYLAND
      #endif //EGL_PLATFORM_WAYLAND_EXT
   }
   return displayType;
}

bool EssContextCreateNativeWindow( EssCtx *ctx, int width, int height, NativeWindowType *nativeWindow )
{
   bool result= false;

   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );

      if ( !ctx->isInitialized )
      {
         sprintf( ctx->lastErrorDetail,
                  "Bad state.  Must initialize before creating native window" );
         pthread_mutex_unlock( &ctx->mutex );
         goto exit;
      }

      result= essCreateNativeWindow( ctx, width, height );
      if ( result )
      {
         *nativeWindow= ctx->nativeWindow;

         /*
          * App is creating its EGL environment outside of Essos
          */
         if ( !ctx->isExternalEGL )
         {
            INFO("essos: app using external EGL");
            ctx->isExternalEGL= true;
         }
      }

      pthread_mutex_unlock( &ctx->mutex );
   }

exit:
   return result;
}

bool EssContextDestroyNativeWindow( EssCtx *ctx, NativeWindowType nativeWindow )
{
   bool result= false;

   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );

      if ( !ctx->isInitialized )
      {
         sprintf( ctx->lastErrorDetail,
                  "Bad state.  Must initialize context before calling" );
         pthread_mutex_unlock( &ctx->mutex );
         goto exit;
      }

      essDestroyNativeWindow( ctx, nativeWindow );

      result= true;

      pthread_mutex_unlock( &ctx->mutex );
   }

exit:
   return result;
}

void* EssContextGetWaylandDisplay( EssCtx *ctx )
{
   void *wldisplay= 0;

   #ifdef HAVE_WAYLAND
   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );

      wldisplay= (void*)ctx->wldisplay;   

      pthread_mutex_unlock( &ctx->mutex );
   }
   #endif

   return wldisplay;
}

bool EssContextSetKeyListener( EssCtx *ctx, void *userData, EssKeyListener *listener )
{
   bool result= false;
                  
   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );

      ctx->keyListenerUserData= userData;
      ctx->keyListener= listener;

      result= true;

      pthread_mutex_unlock( &ctx->mutex );
   }

   return result;
}

bool EssContextSetKeyAndMetadataListener( EssCtx *ctx, void *userData, EssKeyAndMetadataListener *listener, EssInputDeviceMetadata *metadata )
{
   bool result= false;

   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );

      ctx->keyAndMetadataListenerUserData= userData;
      ctx->keyAndMetadataListenerMetadata= metadata;
      ctx->keyAndMetadataListener= listener;

      result= true;

      pthread_mutex_unlock( &ctx->mutex );
   }

   return result;
}

bool EssContextSetPointerListener( EssCtx *ctx, void *userData, EssPointerListener *listener )
{
   bool result= false;
                  
   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );

      ctx->pointerListenerUserData= userData;
      ctx->pointerListener= listener;

      result= true;

      pthread_mutex_unlock( &ctx->mutex );
   }

   return result;
}

bool EssContextSetTouchListener( EssCtx *ctx, void *userData, EssTouchListener *listener )
{
   bool result= false;

   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );

      ctx->touchListenerUserData= userData;
      ctx->touchListener= listener;

      result= true;

      pthread_mutex_unlock( &ctx->mutex );
   }

   return result;
}

bool EssContextSetSettingsListener( EssCtx *ctx, void *userData, EssSettingsListener *listener )
{
   bool result= false;

   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );

      ctx->settingsListenerUserData= userData;
      ctx->settingsListener= listener;

      result= true;

      pthread_mutex_unlock( &ctx->mutex );
   }

   return result;
}

bool EssContextSetTerminateListener( EssCtx *ctx, void *userData, EssTerminateListener *listener )
{
   bool result= false;
                  
   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );

      ctx->terminateListenerUserData= userData;
      ctx->terminateListener= listener;

      result= true;

      pthread_mutex_unlock( &ctx->mutex );
   }

   return result;
}

bool EssContextSetGamepadConnectionListener( EssCtx *ctx, void *userData, EssGamepadConnectionListener *listener )
{
   bool result= false;

   if ( ctx )
   {
      std::vector<EssGamepad*> pads= std::vector<EssGamepad*>();

      pthread_mutex_lock( &ctx->mutex );

      ctx->gamepadConnectionListenerUserData= userData;
      ctx->gamepadConnectionListener= listener;

      if ( listener && listener->connected )
      {
         // Create a snapshot of any existing gamepads
         for( int i= 0; i < ctx->gamepads.size(); ++i )
         {
            EssGamepad *gp= ctx->gamepads[i];
            pads.push_back( gp );
         }
      }

      result= true;

      pthread_mutex_unlock( &ctx->mutex );

      #ifdef HAVE_WESTEROS
      // Notify listener of any existing gamepads
      for( int i= 0; i < pads.size(); ++i )
      {
         EssGamepad *gp= pads[i];
         essGamepadNotifyConnected( ctx, gp );
      }
      #endif
   }

   return result;
}

bool EssGamepadSetEventListener( EssGamepad *gp, void *userData, EssGamepadEventListener *listener )
{
   bool result= false;

   #ifdef HAVE_WESTEROS
   if ( gp )
   {
      EssCtx *ctx= gp->ctx;

      pthread_mutex_lock( &ctx->mutex );

      gp->eventListenerUserData= userData;
      gp->eventListener= listener;

      result= true;

      pthread_mutex_unlock( &ctx->mutex );
   }
   #endif

   return result;
}

const char *EssGamepadGetDeviceName( EssGamepad *gp )
{
   const char *name= 0;

   #ifdef HAVE_WESTEROS
   if ( gp )
   {
      EssCtx *ctx= gp->ctx;
      int rc, len;
      char work[256];

      if ( ctx )
      {
         pthread_mutex_lock( &ctx->mutex );

         if ( !gp->name )
         {
            work[0]= '\0';
            rc= ioctl( gp->fd, EVIOCGNAME(sizeof(work)), work);
            if ( rc > 0 )
            {
               len= strlen(work);
               if ( (len > 0) && (len < sizeof(work)) )
               {
                  gp->name= strdup(work);
               }
               else
               {
                  sprintf( ctx->lastErrorDetail,
                           "Error.  Bad gamepad name length: %d", len );
               }
            }
            else
            {
               sprintf( ctx->lastErrorDetail,
                        "Error.  Failed to get gamepad name: rc %d", rc );
            }
         }

         name= gp->name;
      }

      pthread_mutex_unlock( &ctx->mutex );
   }
   #endif

   return name;
}

unsigned int EssGamepadGetDriverVersion( EssGamepad *gp )
{
   unsigned int version= 0;

   #ifdef HAVE_WESTEROS
   if ( gp )
   {
      EssCtx *ctx= gp->ctx;
      int rc;

      if ( ctx )
      {
         pthread_mutex_lock( &ctx->mutex );

         rc= ioctl( gp->fd, EVIOCGVERSION, &version );
         if ( rc != 0 )
         {
            sprintf( ctx->lastErrorDetail,
                     "Error.  Failed to get gamepad driver version: rc %d", rc );
         }

         pthread_mutex_unlock( &ctx->mutex );
      }
   }
   #endif

   return version;
}

bool EssGamepadGetButtonMap( EssGamepad *gp, int *count, int *map )
{
   bool result= false;

   #ifdef HAVE_WESTEROS
   if ( gp )
   {
      EssCtx *ctx= gp->ctx;
      int buttonCount= 0;

      if ( ctx )
      {
         pthread_mutex_lock( &ctx->mutex );

         if ( count )
         {
            *count= gp->buttonCount;
         }

         if ( map )
         {
            for( int i= 0; i < gp->buttonCount; ++i )
            {
               int id;
               id= gp->buttonMap[i];
               map[i]= id;
            }
         }

         result= true;

         pthread_mutex_unlock( &ctx->mutex );
      }
   }
   #endif

   return result;
}

bool EssGamepadGetAxisMap( EssGamepad *gp, int *count, int *map )
{
   bool result= false;

   #ifdef HAVE_WESTEROS
   if ( gp )
   {
      EssCtx *ctx= gp->ctx;
      int axisCount= 0;

      if ( ctx )
      {
         pthread_mutex_lock( &ctx->mutex );

         if ( count )
         {
            *count= gp->axisCount;
         }

         if ( map )
         {
            for( int i= 0; i < gp->axisCount; ++i )
            {
               int id;
               id= gp->axisMap[i];
               map[i]= id;
            }
         }

         result= true;

         pthread_mutex_unlock( &ctx->mutex );
      }
   }
   #endif

   return result;
}

bool EssGamepadGetState( EssGamepad *gp, int *buttonState, int *axisState )
{
   bool result= false;

   #ifdef HAVE_WESTEROS
   if ( gp )
   {
      EssCtx *ctx= gp->ctx;

      if ( ctx )
      {
         pthread_mutex_lock( &ctx->mutex );

         if ( buttonState )
         {
            memcpy( buttonState, gp->buttonState, gp->buttonCount*sizeof(int) );
         }

         if ( axisState )
         {
            memcpy( axisState, gp->axisState, gp->axisCount*sizeof(int) );
         }

         result= true;

         pthread_mutex_unlock( &ctx->mutex );
      }
   }
   #endif

   return result;
}

bool EssContextSetKeyRepeatInitialDelay( EssCtx *ctx, int delay )
{
   bool result= false;
                  
   if ( ctx )
   {
      int rc;

      pthread_mutex_lock( &ctx->mutex );

      ctx->keyRepeatInitialDelay= delay;

      result= true;

      pthread_mutex_unlock( &ctx->mutex );
   }

   return result;
}

bool EssContextSetKeyRepeatPeriod( EssCtx *ctx, int period )
{
   bool result= false;

   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );

      ctx->keyRepeatPeriod= period;

      result= true;

      pthread_mutex_unlock( &ctx->mutex );
   }

   return result;
}

bool EssContextStart( EssCtx *ctx )
{
   bool result= false;

   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );
      
      if ( ctx->isRunning )
      {
         sprintf( ctx->lastErrorDetail,
                  "Bad state.  Context is already running" );
         pthread_mutex_unlock( &ctx->mutex );
         goto exit;
      }

      if ( !ctx->isInitialized )
      {
         pthread_mutex_unlock( &ctx->mutex );
         result= EssContextInit( ctx );
         if ( !result ) goto exit;
         pthread_mutex_lock( &ctx->mutex );
      }

      if ( !ctx->isExternalEGL )
      {
         result= essEGLInit( ctx );
         if ( !result )
         {
            pthread_mutex_unlock( &ctx->mutex );
            goto exit;
         }
      }

      essInitInput( ctx );

      ctx->isRunning= true;

      pthread_mutex_unlock( &ctx->mutex );

      essRunEventLoopOnce( ctx );

      result= true;
   }

exit:   

   return result;
}

void EssContextStop( EssCtx *ctx )
{
   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );

      if ( ctx->isRunning )
      {
         #ifdef HAVE_WESTEROS
         if  (!ctx->isWayland )
         {
            essMonitorInputDevicesLifecycleEnd( ctx );
            essReleaseInputDevices( ctx );
            essFreeInputDevices( ctx );
            essFreeGamepads( ctx );
         }
         #endif

         essEGLTerm( ctx );

         ctx->isRunning= false;
      }

      pthread_mutex_unlock( &ctx->mutex );
   }
}

bool EssContextSetDisplaySize( EssCtx *ctx, int width, int height )
{
   bool result= false;

   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );

      if ( !ctx->isInitialized )
      {
         sprintf( ctx->lastErrorDetail,
                  "Bad state.  Must initialize before setting display size" );
         pthread_mutex_unlock( &ctx->mutex );
         goto exit;
      }

      if ( ctx->isWayland )
      {
         // ignore
      }
      else
      {
         ctx->haveMode= true;
         essSetDisplaySize( ctx, width, height, false, 0, 0, 0, 0);
      }

      result= true;

      pthread_mutex_unlock( &ctx->mutex );
   }

exit:

   return result;
}

bool EssContextGetDisplaySize( EssCtx *ctx, int *width, int *height )
{
   bool result= false;

   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );

      if ( !ctx->isInitialized )
      {
         sprintf( ctx->lastErrorDetail,
                  "Bad state.  Must initialize before querying display size" );
         pthread_mutex_unlock( &ctx->mutex );
         goto exit;
      }

      if ( width )
      {
         *width= ctx->planeWidth;
      }
      if ( height )
      {
         *height= ctx->planeHeight;
      }

      result= true;

      pthread_mutex_unlock( &ctx->mutex );
   }

exit:

   return result;
}

bool EssContextGetDisplaySafeArea( EssCtx *ctx, int *x, int *y, int *width, int *height )
{
   bool result= false;

   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );

      if ( !ctx->isInitialized )
      {
         sprintf( ctx->lastErrorDetail,
                  "Bad state.  Must initialize before querying display safe area" );
         pthread_mutex_unlock( &ctx->mutex );
         goto exit;
      }

      if ( x )
      {
         *x= ctx->planeSafeX;
      }
      if ( y )
      {
         *y= ctx->planeSafeY;
      }
      if ( width )
      {
         *width= ctx->planeSafeW;
      }
      if ( height )
      {
         *height= ctx->planeSafeH;
      }

      result= true;

      pthread_mutex_unlock( &ctx->mutex );
   }

exit:

   return result;
}

bool EssContextSetWindowPosition( EssCtx *ctx, int x, int y )
{
   bool result= false;

   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );

      ctx->windowX= x;
      ctx->windowY= y;

      if ( ctx->isWayland )
      {
         #ifdef HAVE_WAYLAND
         if ( ctx->shell && ctx->appSurfaceId )
         {
            if ( !ctx->fullScreen )
            {
               #ifdef HAVE_WESTEROS
               wl_simple_shell_set_geometry( ctx->shell, ctx->appSurfaceId,
                                             ctx->windowX, ctx->windowY,
                                             ctx->windowWidth, ctx->windowHeight );
               #endif
            }
         }
         else
         {
            ctx->pendingGeometryChange= true;
         }
         #endif
      }
      else
      {
         // ignore
      }

      result= true;

      pthread_mutex_unlock( &ctx->mutex );
   }

exit:

   return result;
}

bool EssContextResizeWindow( EssCtx *ctx, int width, int height )
{
   bool result= false;

   if ( ctx )
   {
      pthread_mutex_lock( &ctx->mutex );

      result= essResize( ctx, width, height );

      pthread_mutex_unlock( &ctx->mutex );
   }

exit:

   return result;
}

void EssContextRunEventLoopOnce( EssCtx *ctx )
{
   if ( ctx )
   {
      long long start, end, diff, delay;
      if ( ctx->eventLoopPeriodMS )
      {
         start= essGetCurrentTimeMillis();
      }

      essRunEventLoopOnce( ctx );

      if ( ctx->eventLoopPeriodMS )
      {
         if ( ctx->eventLoopLastTimeStamp )
         {
            diff= start-ctx->eventLoopLastTimeStamp;
            delay= ((long long)ctx->eventLoopPeriodMS - diff);
            if ( (delay > 0) && (delay <= ctx->eventLoopPeriodMS) )
            {
               usleep( delay*1000 );
            }
         }
         ctx->eventLoopLastTimeStamp= start;
      }
   }
}

void EssContextUpdateDisplay( EssCtx *ctx )
{
   if ( ctx )
   {
      if ( ctx->haveMode &&
           (ctx->eglDisplay != EGL_NO_DISPLAY) &&
           (ctx->eglSurfaceWindow != EGL_NO_SURFACE) )
      {
         eglSwapBuffers( ctx->eglDisplay, ctx->eglSurfaceWindow );
      }
   }
}

static long long essGetCurrentTimeMillis(void)
{
   struct timeval tv;
   long long utcCurrentTimeMillis;

   gettimeofday(&tv,0);
   utcCurrentTimeMillis= tv.tv_sec*1000LL+(tv.tv_usec/1000LL);

   return utcCurrentTimeMillis;
}

static bool essPlatformInit( EssCtx *ctx )
{
   bool result= false;

   if ( ctx->isWayland )
   {
      #ifdef HAVE_WAYLAND
      result= essPlatformInitWayland( ctx );
      #endif
   }
   else
   {
      #ifdef HAVE_WESTEROS
      ctx->haveMode= true;
      result= essPlatformInitDirect( ctx );
      #endif
   }

   return result;   
}

static void essPlatformTerm( EssCtx *ctx )
{
   if ( ctx->isWayland )
   {
      #ifdef HAVE_WAYLAND
      essPlatformTermWayland( ctx );
      #endif
   }
   else
   {
      #ifdef HAVE_WESTEROS
      essPlatformTermDirect( ctx );
      #endif
   }
}

static bool essEGLInit( EssCtx *ctx )
{
   bool result= false;
   EGLBoolean b;
   EGLConfig *eglConfigs= 0;
   EGLint configCount= 0;
   EGLint redSizeNeed, greenSizeNeed, blueSizeNeed, alphaSizeNeed, depthSizeNeed;
   EGLint redSize, greenSize, blueSize, alphaSize, depthSize;
   int i;
   
   DEBUG("essEGLInit: displayType %p", ctx->displayType);
   #ifdef EGL_PLATFORM_WAYLAND_EXT
   PFNEGLGETPLATFORMDISPLAYEXTPROC eglGetPlatformDisplayEXT= (PFNEGLGETPLATFORMDISPLAYEXTPROC)eglGetProcAddress("eglGetPlatformDisplayEXT");
   printf("eglGetPlatformDisplayEXT %p isWayland %d\n", eglGetPlatformDisplayEXT, ctx->isWayland);
   if (ctx->isWayland && eglGetPlatformDisplayEXT)
   {
      ctx->eglDisplay = eglGetPlatformDisplayEXT(EGL_PLATFORM_WAYLAND_EXT, (NativeDisplayType)ctx->displayType, NULL);
      if ( ctx->eglDisplay == EGL_NO_DISPLAY )
      {
         ctx->eglDisplay = eglGetDisplay((NativeDisplayType)ctx->displayType);
      }
   }
   else
   #endif
   ctx->eglDisplay= eglGetDisplay( ctx->displayType );
   if ( ctx->eglDisplay == EGL_NO_DISPLAY )
   {
      sprintf( ctx->lastErrorDetail,
               "Error. Unable to get EGL display: eglError %X", eglGetError() );
      goto exit;
   }

   b= eglInitialize( ctx->eglDisplay, &ctx->eglVersionMajor, &ctx->eglVersionMinor );
   if ( !b )
   {
      sprintf( ctx->lastErrorDetail,
               "Error: Unable to initialize EGL display: eglError %X", eglGetError() );
      goto exit;
   }

   b= eglChooseConfig( ctx->eglDisplay, ctx->eglCfgAttr, 0, 0, &configCount );
   if ( !b || (configCount == 0) )
   {
      sprintf( ctx->lastErrorDetail,
               "Error: eglChooseConfig failed to return number of configurations: count: %d eglError %X\n", configCount, eglGetError() );
      goto exit;
   }

   eglConfigs= (EGLConfig*)malloc( configCount*sizeof(EGLConfig) );
   if ( !eglConfigs )
   {
      sprintf( ctx->lastErrorDetail,
               "Error: Unable to allocate memory for %d EGL configurations", configCount );
      goto exit;
   }

   b= eglChooseConfig( ctx->eglDisplay, ctx->eglCfgAttr, eglConfigs, configCount, &configCount );
   if ( !b || (configCount == 0) )
   {
      sprintf( ctx->lastErrorDetail,
               "Error: eglChooseConfig failed to return list of configurations: count: %d eglError %X\n", configCount, eglGetError() );
      goto exit;
   }

   redSizeNeed= greenSizeNeed= blueSizeNeed= alphaSizeNeed= depthSizeNeed= 0;
   for( i= 0; i < ctx->eglCfgAttrSize; i += 2 )
   {
      EGLint type= ctx->eglCfgAttr[i];
      if ( (type != EGL_NONE) && (i+1 < ctx->eglCfgAttrSize) )
      {
         EGLint value= ctx->eglCfgAttr[i+1];
         switch( ctx->eglCfgAttr[i] )
         {
            case EGL_RED_SIZE:
               redSizeNeed= value;
               break;
            case EGL_GREEN_SIZE:
               greenSizeNeed= value;
               break;
            case EGL_BLUE_SIZE:
               blueSizeNeed= value;
               break;
            case EGL_ALPHA_SIZE:
               alphaSizeNeed= value;
               break;
            case EGL_DEPTH_SIZE:
               depthSizeNeed= value;
               break;
            default:
               break;
         }
      }
   }

   for( i= 0; i < configCount; ++i )
   {
      eglGetConfigAttrib( ctx->eglDisplay, eglConfigs[i], EGL_RED_SIZE, &redSize );
      eglGetConfigAttrib( ctx->eglDisplay, eglConfigs[i], EGL_GREEN_SIZE, &greenSize );
      eglGetConfigAttrib( ctx->eglDisplay, eglConfigs[i], EGL_BLUE_SIZE, &blueSize );
      eglGetConfigAttrib( ctx->eglDisplay, eglConfigs[i], EGL_ALPHA_SIZE, &alphaSize );
      eglGetConfigAttrib( ctx->eglDisplay, eglConfigs[i], EGL_DEPTH_SIZE, &depthSize );

      DEBUG("essEGLInit: config %d: red: %d green: %d blue: %d alpha: %d depth: %d",
              i, redSize, greenSize, blueSize, alphaSize, depthSize );
      if ( (redSize == redSizeNeed) &&
           (greenSize == greenSizeNeed) &&
           (blueSize == blueSizeNeed) &&
           (alphaSize == alphaSizeNeed) &&
           (depthSize >= depthSizeNeed) )
      {
         DEBUG( "essEGLInit: choosing config %d", i);
         break;
      }
   }

   if ( i == configCount )
   {
      sprintf( ctx->lastErrorDetail,
               "Error: no suitable configuration available\n");
      goto exit;
   }

   ctx->eglConfig= eglConfigs[i];

   ctx->eglContext= eglCreateContext( ctx->eglDisplay, ctx->eglConfig, EGL_NO_CONTEXT, ctx->eglCtxAttr );
   if ( ctx->eglContext == EGL_NO_CONTEXT )
   {
      sprintf( ctx->lastErrorDetail,
               "Error: eglCreateContext failed: eglError %X\n", eglGetError() );
      goto exit;
   }
   DEBUG("essEGLInit: eglContext %p", ctx->eglContext );

   if ( !essCreateNativeWindow( ctx, ctx->windowWidth, ctx->windowHeight ) )
   {
      goto exit;
   }
   DEBUG("essEGLInit: nativeWindow %p", ctx->nativeWindow );

   ctx->eglSurfaceWindow= eglCreateWindowSurface( ctx->eglDisplay,
                                                  ctx->eglConfig,
                                                  ctx->nativeWindow,
                                                  ctx->eglSurfAttr );
   if ( ctx->eglSurfaceWindow == EGL_NO_SURFACE )
   {
      sprintf( ctx->lastErrorDetail,
               "Error: eglCreateWindowSurface failed: eglError %X\n", eglGetError() );
      goto exit;
   }
   DEBUG("essEGLInit: eglSurfaceWindow %p", ctx->eglSurfaceWindow );

   b= eglMakeCurrent( ctx->eglDisplay, ctx->eglSurfaceWindow, ctx->eglSurfaceWindow, ctx->eglContext );
   if ( !b )
   {
      sprintf( ctx->lastErrorDetail,
               "Error: eglMakeCurrent failed: eglError %X\n", eglGetError() );
      goto exit;
   }
    
   eglSwapInterval( ctx->eglDisplay, ctx->eglSwapInterval );

   result= true;

exit:
   if ( !result )
   {
      essEGLTerm(ctx);
   }

   if ( eglConfigs )
   {
      free( eglConfigs );
   }

   return result;
}

static void essEGLTerm( EssCtx *ctx )
{
   if ( ctx )
   {
      if ( ctx->eglDisplay != EGL_NO_DISPLAY )
      {
         eglMakeCurrent( ctx->eglDisplay, EGL_NO_SURFACE, EGL_NO_SURFACE, EGL_NO_CONTEXT );

         if ( ctx->eglSurfaceWindow != EGL_NO_SURFACE )
         {
            eglDestroySurface( ctx->eglDisplay, ctx->eglSurfaceWindow );
            ctx->eglSurfaceWindow= EGL_NO_SURFACE;
         }

         if ( ctx->eglContext != EGL_NO_CONTEXT )
         {
            eglDestroyContext( ctx->eglDisplay, ctx->eglContext );
            ctx->eglContext= EGL_NO_CONTEXT;
         }
         
         eglTerminate( ctx->eglDisplay );
         ctx->eglDisplay= EGL_NO_DISPLAY;

         eglReleaseThread();
      }
   }
}

static void essInitInput( EssCtx *ctx )
{
   if ( ctx )
   {
      if ( ctx->isWayland )
      {
         // Setup during wayland registry processing
      }
      #ifdef HAVE_WESTEROS
      else
      {
         essGetInputDevices( ctx );
         essMonitorInputDevicesLifecycleBegin( ctx );
      }
      #endif
   }
}

static void essSetDisplaySize( EssCtx *ctx, int width, int height, bool customSafe, int safeX, int safeY, int safeW, int safeH )
{
   ctx->planeWidth= width;
   ctx->planeHeight= height;
   if ( customSafe )
   {
      ctx->planeSafeX= safeX;
      ctx->planeSafeY= safeY;
      ctx->planeSafeW= safeW;
      ctx->planeSafeH= safeH;
   }
   else
   {
      ctx->planeSafeX= width*DEFAULT_PLANE_SAFE_BORDER_PERCENT/100;
      ctx->planeSafeY= height*DEFAULT_PLANE_SAFE_BORDER_PERCENT/100;
      ctx->planeSafeW= width-2*ctx->planeSafeX;
      ctx->planeSafeH= height-2*ctx->planeSafeY;
   }
}

static void essShellSurfacePing(void *data, struct wl_shell_surface *shell_surface, uint32_t serial)
{
   ESS_UNUSED(data);
   wl_shell_surface_pong(shell_surface, serial);
}

static void essShellSurfaceConfigure(void *data, struct wl_shell_surface *shell_surface, uint32_t edges, int32_t width, int32_t height)
{
   ESS_UNUSED(shell_surface);
   ESS_UNUSED(edges);
   EssCtx *ctx= (EssCtx*)data;
   wl_egl_window_resize(ctx->wleglwindow, width, height, 0, 0);
}

static void essShellSurfacePopupDone(void *data, struct wl_shell_surface *shell_surface)
{
   ESS_UNUSED(data);
   ESS_UNUSED(shell_surface);
}

static const struct wl_shell_surface_listener essShellSurfaceListener= {
   essShellSurfacePing,
   essShellSurfaceConfigure,
   essShellSurfacePopupDone
};

static bool essCreateNativeWindow( EssCtx *ctx, int width, int height )
{
   bool result= false;

   if ( ctx )
   {
      if ( ctx->isWayland )
      {
         #ifdef HAVE_WAYLAND
         ctx->wlsurface= wl_compositor_create_surface(ctx->wlcompositor);
         if ( !ctx->wlsurface )
         {
            sprintf( ctx->lastErrorDetail,
                     "Error.  Unable to create wayland surface" );
            goto exit;
         }

         ctx->shellSurface= wl_shell_get_shell_surface(ctx->wlshell, ctx->wlsurface);
         if ( !ctx->shellSurface )
         {
            sprintf( ctx->lastErrorDetail,
                     "Error.  Unable to get shell surface" );
            goto exit;
         }

         wl_shell_surface_add_listener(ctx->shellSurface, &essShellSurfaceListener, ctx);
         wl_shell_surface_set_toplevel(ctx->shellSurface);

         ctx->wleglwindow= wl_egl_window_create(ctx->wlsurface, width, height);
         if ( !ctx->wleglwindow )
         {
            sprintf( ctx->lastErrorDetail,
                     "Error.  Unable to create wayland egl window" );
            goto exit;
         }

         ctx->nativeWindow= (NativeWindowType)ctx->wleglwindow;         

         result= true;
         #endif
      }
      else
      {
         #ifdef HAVE_WESTEROS
         ctx->nativeWindow= (NativeWindowType)WstGLCreateNativeWindow( ctx->glCtx, 
                                                                       0,
                                                                       0,
                                                                       width,
                                                                       height );
#ifndef WESTEROS_PLATFORM_QEMUX86
         if ( !ctx->nativeWindow )
         {
            sprintf( ctx->lastErrorDetail,
                     "Error.  Unable to create native egl window" );
            goto exit;
         }
#endif
         result= true;
         #endif
      }
   }

exit:
   return result;
}

static bool essDestroyNativeWindow( EssCtx *ctx, NativeWindowType nw )
{
   bool result= false;

   if ( ctx )
   {
      if ( nw == ctx->nativeWindow )
      {
         if ( ctx->isWayland )
         {
            #ifdef HAVE_WAYLAND
            if ( ctx->wleglwindow )
            {
               wl_egl_window_destroy( ctx->wleglwindow );
               ctx->wleglwindow= 0;
            }

            if ( ctx->wlsurface )
            {
               wl_surface_destroy( ctx->wlsurface );
               ctx->wlsurface= 0;
            }
            #endif
         }
         else
         {
            #ifdef HAVE_WESTEROS
            if ( ctx->nativeWindow )
            {
               WstGLDestroyNativeWindow( ctx->glCtx, (void*)ctx->nativeWindow );
               ctx->nativeWindow= 0;
            }
            #endif
         }

         ctx->nativeWindow= 0;
         result= true;
      }
   }

exit:
   return result;
}

static bool essResize( EssCtx *ctx, int width, int height )
{
   bool result= false;

   INFO("essResize %dx%d", width, height);
   if ( ctx->isWayland )
   {
      #ifdef HAVE_WAYLAND
      if ( !ctx->fullScreen && ctx->wleglwindow )
      {
         wl_egl_window_resize( ctx->wleglwindow, width, height, 0, 0 );

         result= true;
      }
      #endif
   }
   #ifdef HAVE_WESTEROS
   else
   {
      if ( ctx->eglDisplay != EGL_NO_DISPLAY )
      {
         eglMakeCurrent( ctx->eglDisplay, EGL_NO_SURFACE, EGL_NO_SURFACE, EGL_NO_CONTEXT );

         if ( ctx->eglSurfaceWindow != EGL_NO_SURFACE )
         {
            eglDestroySurface( ctx->eglDisplay, ctx->eglSurfaceWindow );
            ctx->eglSurfaceWindow= EGL_NO_SURFACE;
         }

         if ( ctx->nativeWindow )
         {
            WstGLDestroyNativeWindow( ctx->glCtx, (void*)ctx->nativeWindow );
            ctx->nativeWindow= 0;
         }

         if ( essCreateNativeWindow( ctx, width, height ) )
         {
            ctx->eglSurfaceWindow= eglCreateWindowSurface( ctx->eglDisplay,
                                                           ctx->eglConfig,
                                                           ctx->nativeWindow,
                                                           NULL );
            if ( ctx->eglSurfaceWindow != EGL_NO_SURFACE )
            {
               DEBUG("essResize: eglSurfaceWindow %p", ctx->eglSurfaceWindow );

               if ( eglMakeCurrent( ctx->eglDisplay, ctx->eglSurfaceWindow, ctx->eglSurfaceWindow, ctx->eglContext ) )
               {
                  result= true;
               }
               else
               {
                  ERROR("Error: eglResize: eglMakeCurrent failed: eglError %X", eglGetError());
               }
            }
            else
            {
               ERROR("Error: eglResize: eglCreateWindowSurface failed: eglError %X", eglGetError());
            }
         }
         else
         {
            ERROR("Error: eglResize: essCreateNativeWindow failed");
         }
      }
   }
   #endif

   if ( result )
   {
      ctx->windowWidth= width;
      ctx->windowHeight= height;
   }

   return result;
}

static void essRunEventLoopOnce( EssCtx *ctx )
{
   if ( ctx )
   {
      if ( ctx->isWayland )
      {
         #ifdef HAVE_WAYLAND
         essProcessRunWaylandEventLoopOnce( ctx );
         #endif
      }
      else
      {
         #ifdef HAVE_WESTEROS
         essProcessInputDevices( ctx );
         #endif
      }

      if ( ctx->keyPressed )
      {
         long long now= essGetCurrentTimeMillis();
         long long diff= now-ctx->lastKeyTime;
         if (
              (ctx->keyRepeating && (diff >= ctx->keyRepeatPeriod)) ||
              (!ctx->keyRepeating && (diff >= ctx->keyRepeatInitialDelay))
            )
         {
            ctx->lastKeyTime= now;
            ctx->keyRepeating= true;
            essProcessKeyRepeat( ctx, ctx->lastKeyCode );
         }
      }

      if ( ctx->resizePending )
      {
         ctx->resizePending= false;
         if ( ctx->settingsListener )
         {
            if ( ctx->settingsListener->displaySize )
            {
               ctx->settingsListener->displaySize( ctx->settingsListenerUserData, ctx->resizeWidth, ctx->resizeHeight );
            }
            if ( ctx->settingsListener->displaySafeArea )
            {
               ctx->settingsListener->displaySafeArea( ctx->settingsListenerUserData,
                                                       ctx->planeSafeX, ctx->planeSafeY,
                                                       ctx->planeSafeW, ctx->planeSafeH );
            }
         }
      }
   }
}

static void essProcessKeyPressed( EssCtx *ctx, int linuxKeyCode )
{
   if ( ctx )
   {
      DEBUG("essProcessKeyPressed: key %d", linuxKeyCode);
      if ( ctx->keyListener && ctx->keyListener->keyPressed )
      {
         ctx->keyListener->keyPressed( ctx->keyListenerUserData, linuxKeyCode );
      }
      if ( ctx->keyAndMetadataListener && ctx->keyAndMetadataListener->keyPressed )
      {
         ctx->keyAndMetadataListener->keyPressed( ctx->keyAndMetadataListenerUserData, linuxKeyCode,
                                                  ctx->keyAndMetadataListenerMetadata );
      }
   }
}

static void essProcessKeyReleased( EssCtx *ctx, int linuxKeyCode )
{
   if ( ctx )
   {
      DEBUG("essProcessKeyReleased: key %d", linuxKeyCode);
      if ( ctx->keyListener && ctx->keyListener->keyReleased )
      {
         ctx->keyListener->keyReleased( ctx->keyListenerUserData, linuxKeyCode );
      }
      if ( ctx->keyAndMetadataListener && ctx->keyAndMetadataListener->keyReleased )
      {
         ctx->keyAndMetadataListener->keyReleased( ctx->keyAndMetadataListenerUserData, linuxKeyCode,
                                                   ctx->keyAndMetadataListenerMetadata );
      }
   }
}

static void essProcessKeyRepeat( EssCtx *ctx, int linuxKeyCode )
{
   if ( ctx )
   {
      DEBUG("essProcessKeyRepeat: key %d", linuxKeyCode);
      if ( ctx->keyListener )
      {
         if ( ctx->keyListener->keyRepeat )
         {
            ctx->keyListener->keyRepeat( ctx->keyListenerUserData, linuxKeyCode );
         }
         else if ( ctx->keyListener->keyPressed )
         {
            ctx->keyListener->keyPressed( ctx->keyListenerUserData, linuxKeyCode );
         }
      }
      if ( ctx->keyAndMetadataListener )
      {
         if ( ctx->keyAndMetadataListener->keyRepeat )
         {
            ctx->keyAndMetadataListener->keyRepeat( ctx->keyAndMetadataListenerUserData, linuxKeyCode,
                                                    ctx->keyAndMetadataListenerMetadata );
         }
         else if ( ctx->keyAndMetadataListener->keyPressed )
         {
            ctx->keyAndMetadataListener->keyPressed( ctx->keyAndMetadataListenerUserData, linuxKeyCode,
                                                     ctx->keyAndMetadataListenerMetadata );
         }
      }
   }
}

static void essProcessPointerMotion( EssCtx *ctx, int x, int y )
{
   if ( ctx )
   {
      TRACE("essProcessKeyPointerMotion (%d, %d)", x, y );
      ctx->pointerX= x;
      ctx->pointerY= y;
      if ( ctx->pointerListener && ctx->pointerListener->pointerMotion )
      {
         ctx->pointerListener->pointerMotion( ctx->pointerListenerUserData, x,  y );
      }
   }
}

static void essProcessPointerButtonPressed( EssCtx *ctx, int button )
{
   if ( ctx )
   {
      DEBUG("essProcessKeyPointerPressed %d", button );
      if ( ctx->pointerListener && ctx->pointerListener->pointerButtonPressed )
      {
         ctx->pointerListener->pointerButtonPressed( ctx->pointerListenerUserData, button, ctx->pointerX, ctx->pointerY );
      }
   }
}

static void essProcessPointerButtonReleased( EssCtx *ctx, int button )
{
   if ( ctx )
   {
      DEBUG("essos: essProcessKeyPointerReleased %d", button );
      if ( ctx->pointerListener && ctx->pointerListener->pointerButtonReleased )
      {
         ctx->pointerListener->pointerButtonReleased( ctx->pointerListenerUserData, button, ctx->pointerX, ctx->pointerY );
      }
   }
}

static void essProcessTouchDown( EssCtx *ctx, int id, int x, int y )
{
   if ( ctx )
   {
      DEBUG("essos: essProcessTouchDown id %d (%d,%d)", id, x, y );
      if ( ctx->touchListener && ctx->touchListener->touchDown )
      {
         ctx->touchListener->touchDown( ctx->touchListenerUserData, id, x, y );
      }
   }
}

static void essProcessTouchUp( EssCtx *ctx, int id )
{
   if ( ctx )
   {
      DEBUG("essos: essProcessTouchUp id %d", id );
      if ( ctx->touchListener && ctx->touchListener->touchUp )
      {
         ctx->touchListener->touchUp( ctx->touchListenerUserData, id );
      }
   }
}

static void essProcessTouchMotion( EssCtx *ctx, int id, int x, int y )
{
   if ( ctx )
   {
      DEBUG("essos: essProcessTouchMotion id %d (%d,%d)", id, x, y );
      if ( ctx->touchListener && ctx->touchListener->touchMotion )
      {
         ctx->touchListener->touchMotion( ctx->touchListenerUserData, id, x, y );
      }
   }
}

static void essProcessTouchFrame( EssCtx *ctx )
{
   if ( ctx )
   {
      DEBUG("essos: essProcessTouchFrame" );
      if ( ctx->touchListener && ctx->touchListener->touchFrame )
      {
         ctx->touchListener->touchFrame( ctx->touchListenerUserData );
      }
   }
}

static void essProcessGamepadButtonPressed( EssGamepad *gp, int buttonId )
{
   if ( gp )
   {
      DEBUG("essos: essProcessGamepadButtonPressed: buttonId %d", buttonId);
      if ( gp->eventListener && gp->eventListener->buttonPressed )
      {
         gp->eventListener->buttonPressed( gp->eventListenerUserData, buttonId );
      }
   }
}

static void essProcessGamepadButtonReleased( EssGamepad *gp, int buttonId )
{
   if ( gp )
   {
      DEBUG("essos: essProcessGamepadButtonReleased: buttonId %d", buttonId);
      if ( gp->eventListener && gp->eventListener->buttonReleased )
      {
         gp->eventListener->buttonReleased( gp->eventListenerUserData, buttonId );
      }
   }
}

static void essProcessGamepadAxisChanged( EssGamepad *gp, int axisId, int value )
{
   if ( gp )
   {
      DEBUG("essos: essProcessGamepadAxisChanged: axisId %d value %d", axisId, value);
      if ( gp->eventListener && gp->eventListener->axisChanged )
      {
         gp->eventListener->axisChanged( gp->eventListenerUserData, axisId, value );
      }
   }
}

#ifdef HAVE_WAYLAND
static void essKeyboardKeymap( void *data, struct wl_keyboard *keyboard, uint32_t format, int32_t fd, uint32_t size )
{
   EssCtx *ctx= (EssCtx*)data;

   if ( format == WL_KEYBOARD_KEYMAP_FORMAT_XKB_V1 )
   {
      void *map= mmap( 0, size, PROT_READ, MAP_SHARED, fd, 0 );
      if ( map != MAP_FAILED )
      {
         if ( !ctx->xkbCtx )
         {
            ctx->xkbCtx= xkb_context_new( XKB_CONTEXT_NO_FLAGS );
         }
         else
         {
            ERROR("essKeyboardKeymap: xkb_context_new failed");
         }
         if ( ctx->xkbCtx )
         {
            if ( ctx->xkbKeymap )
            {
               xkb_keymap_unref( ctx->xkbKeymap );
               ctx->xkbKeymap= 0;
            }
            ctx->xkbKeymap= xkb_keymap_new_from_string( ctx->xkbCtx, (char*)map, XKB_KEYMAP_FORMAT_TEXT_V1, XKB_KEYMAP_COMPILE_NO_FLAGS);
            if ( !ctx->xkbKeymap )
            {
               ERROR("essKeyboardKeymap: xkb_keymap_new_from_string failed");
            }
            if ( ctx->xkbState )
            {
               xkb_state_unref( ctx->xkbState );
               ctx->xkbState= 0;
            }
            ctx->xkbState= xkb_state_new( ctx->xkbKeymap );
            if ( !ctx->xkbState )
            {
               ERROR("essKeyboardKeymap: xkb_state_new failed");
            }
            if ( ctx->xkbKeymap )
            {
               ctx->modAlt= xkb_keymap_mod_get_index( ctx->xkbKeymap, XKB_MOD_NAME_ALT );
               ctx->modCtrl= xkb_keymap_mod_get_index( ctx->xkbKeymap, XKB_MOD_NAME_CTRL );
               ctx->modShift= xkb_keymap_mod_get_index( ctx->xkbKeymap, XKB_MOD_NAME_SHIFT );
               ctx->modCaps= xkb_keymap_mod_get_index( ctx->xkbKeymap, XKB_MOD_NAME_CAPS );
            }
         }
         munmap( map, size );
      }
   }

   close( fd );
}

static void essKeyboardEnter( void *data, struct wl_keyboard *keyboard, uint32_t serial,
                              struct wl_surface *surface, struct wl_array *keys )
{
   ESS_UNUSED(data);
   ESS_UNUSED(keyboard);
   ESS_UNUSED(serial);
   ESS_UNUSED(keys);

   DEBUG("essKeyboardEnter: keyboard enter surface %p", surface );
}

static void essKeyboardLeave( void *data, struct wl_keyboard *keyboard, uint32_t serial, struct wl_surface *surface )
{
   ESS_UNUSED(data);
   ESS_UNUSED(keyboard);
   ESS_UNUSED(serial);

   DEBUG("esKeyboardLeave: keyboard leave surface %p", surface );
}

static void essKeyboardKey( void *data, struct wl_keyboard *keyboard, uint32_t serial,
                            uint32_t time, uint32_t key, uint32_t state )
{
   EssCtx *ctx= (EssCtx*)data;
   ESS_UNUSED(keyboard);
   ESS_UNUSED(serial);
   ESS_UNUSED(time);

   switch( key )
   {
      case KEY_CAPSLOCK:
      case KEY_LEFTSHIFT:
      case KEY_RIGHTSHIFT:
      case KEY_LEFTCTRL:
      case KEY_RIGHTCTRL:
      case KEY_LEFTALT:
      case KEY_RIGHTALT:
         break;
      default:
         if ( state == WL_KEYBOARD_KEY_STATE_PRESSED )
         {
            if ( !(ctx->keyPressed && (ctx->lastKeyCode == key)) )
            {
               ctx->lastKeyCode= key;
               ctx->keyPressed= true;
               ctx->keyRepeating= false;
               ctx->lastKeyTime= essGetCurrentTimeMillis();
               essProcessKeyPressed( ctx, key );
            }
         }
         else if ( state == WL_KEYBOARD_KEY_STATE_RELEASED )
         {
            ctx->keyPressed= false;
            essProcessKeyReleased( ctx, key );
         }
         break;
   }
}

static void essUpdateModifierKey( EssCtx *ctx, bool active, int key )
{
   if ( active )
   {
      essProcessKeyPressed( ctx, key );
   }
   else
   {
      essProcessKeyReleased( ctx, key );
   }
}

static void essKeyboardModifiers( void *data, struct wl_keyboard *keyboard, uint32_t serial,
                                  uint32_t mods_depressed, uint32_t mods_latched,
                                  uint32_t mods_locked, uint32_t group )
{
   EssCtx *ctx= (EssCtx*)data;
   if ( ctx->xkbState )
   {
      int wasActive, nowActive, key;

      xkb_state_update_mask( ctx->xkbState, mods_depressed, mods_latched, mods_locked, 0, 0, group );
      DEBUG("essKeyboardModifiers: mods_depressed %X mods locked %X", mods_depressed, mods_locked);

      wasActive= (ctx->modMask & (1<<ctx->modCaps));
      nowActive= (mods_locked & (1<<ctx->modCaps));
      key= KEY_CAPSLOCK;
      if ( nowActive != wasActive )
      {
         ctx->modMask ^= (1<<ctx->modCaps);
         essProcessKeyPressed( ctx, key );
         essProcessKeyReleased( ctx, key );
      }

      wasActive= (ctx->modMask & (1<<ctx->modShift));
      nowActive= (mods_depressed & (1<<ctx->modShift));
      key= KEY_RIGHTSHIFT;
      if ( nowActive != wasActive )
      {
         ctx->modMask ^= (1<<ctx->modShift);
         essUpdateModifierKey( ctx, nowActive, key );
      }

      wasActive= (ctx->modMask & (1<<ctx->modCtrl));
      nowActive= (mods_depressed & (1<<ctx->modCtrl));
      key= KEY_RIGHTCTRL;
      if ( nowActive != wasActive )
      {
         ctx->modMask ^= (1<<ctx->modCtrl);
         essUpdateModifierKey( ctx, nowActive, key );
      }

      wasActive= (ctx->modMask & (1<<ctx->modAlt));
      nowActive= (mods_depressed & (1<<ctx->modAlt));
      key= KEY_RIGHTALT;
      if ( nowActive != wasActive )
      {
         ctx->modMask ^= (1<<ctx->modAlt);
         essUpdateModifierKey( ctx, nowActive, key );
      }
   }
}

static void essKeyboardRepeatInfo( void *data, struct wl_keyboard *keyboard, int32_t rate, int32_t delay )
{
   ESS_UNUSED(data);
   ESS_UNUSED(keyboard);
   ESS_UNUSED(rate);
   ESS_UNUSED(delay);
}

static const struct wl_keyboard_listener essKeyboardListener= {
   essKeyboardKeymap,
   essKeyboardEnter,
   essKeyboardLeave,
   essKeyboardKey,
   essKeyboardModifiers,
   essKeyboardRepeatInfo
};

static void essPointerEnter( void* data, struct wl_pointer *pointer, uint32_t serial,
                             struct wl_surface *surface, wl_fixed_t sx, wl_fixed_t sy )
{
   ESS_UNUSED(pointer);
   ESS_UNUSED(serial);
   EssCtx *ctx= (EssCtx*)data;
   int x, y;

   x= wl_fixed_to_int( sx );
   y= wl_fixed_to_int( sy );

   DEBUG("essPointerEnter: pointer enter surface %p (%d,%d)", surface, x, y );
}

static void essPointerLeave( void* data, struct wl_pointer *pointer, uint32_t serial, struct wl_surface *surface )
{
   ESS_UNUSED(data);
   ESS_UNUSED(pointer);
   ESS_UNUSED(serial);

   DEBUG("essPointerLeave: pointer leave surface %p", surface );
}

static void essPointerMotion( void *data, struct wl_pointer *pointer, uint32_t time, wl_fixed_t sx, wl_fixed_t sy )
{
   ESS_UNUSED(pointer);
   EssCtx *ctx= (EssCtx*)data;
   int x, y;

   x= wl_fixed_to_int( sx );
   y= wl_fixed_to_int( sy );

   essProcessPointerMotion( ctx, x, y );
}

static void essPointerButton( void *data, struct wl_pointer *pointer, uint32_t serial,
                              uint32_t time, uint32_t button, uint32_t state )
{
   ESS_UNUSED(pointer);
   ESS_UNUSED(serial);
   EssCtx *ctx= (EssCtx*)data;

   if ( state == WL_POINTER_BUTTON_STATE_PRESSED )
   {
      essProcessPointerButtonPressed( ctx, button );
   }
   else
   {
      essProcessPointerButtonReleased( ctx, button );
   }
}

static void essPointerAxis( void *data, struct wl_pointer *pointer, uint32_t time,
                            uint32_t axis, wl_fixed_t value )
{
   ESS_UNUSED(data);
   ESS_UNUSED(pointer);
   ESS_UNUSED(time);
   ESS_UNUSED(value);
}

static const struct wl_pointer_listener essPointerListener = {
   essPointerEnter,
   essPointerLeave,
   essPointerMotion,
   essPointerButton,
   essPointerAxis
};

static void essTouchDown( void *data, struct wl_touch *touch,
                          uint32_t serial, uint32_t time, struct wl_surface *surface,
                          int32_t id, wl_fixed_t sx, wl_fixed_t sy )
{
   ESS_UNUSED(touch);
   ESS_UNUSED(serial);
   ESS_UNUSED(time);
   ESS_UNUSED(surface);
   EssCtx *ctx= (EssCtx*)data;

   int x, y;

   x= wl_fixed_to_int( sx );
   y= wl_fixed_to_int( sy );

   essProcessTouchDown( ctx, id, x, y );
}

static void essTouchUp( void *data, struct wl_touch *touch,
                        uint32_t serial, uint32_t time, int32_t id )
{
   ESS_UNUSED(touch);
   ESS_UNUSED(serial);
   ESS_UNUSED(time);
   EssCtx *ctx= (EssCtx*)data;

   essProcessTouchUp( ctx, id );
}

static void essTouchMotion( void *data, struct wl_touch *touch,
                            uint32_t time, int32_t id, wl_fixed_t sx, wl_fixed_t sy )
{
   ESS_UNUSED(touch);
   ESS_UNUSED(time);
   EssCtx *ctx= (EssCtx*)data;

   int x, y;

   x= wl_fixed_to_int( sx );
   y= wl_fixed_to_int( sy );

   essProcessTouchMotion( ctx, id, x, y );
}

static void essTouchFrame( void *data, struct wl_touch *touch )
{
   ESS_UNUSED(touch);
   EssCtx *ctx= (EssCtx*)data;

   essProcessTouchFrame( ctx );
}

static const struct wl_touch_listener essTouchListener= {
   essTouchDown,
   essTouchUp,
   essTouchMotion,
   essTouchFrame
};

static void essSeatCapabilities( void *data, struct wl_seat *seat, uint32_t capabilities )
{
   EssCtx *ctx= (EssCtx*)data;

   DEBUG("essSeatCapabilities: seat %p caps: %X", seat, capabilities );
   
   if ( capabilities & WL_SEAT_CAPABILITY_KEYBOARD )
   {
      DEBUG("essSeatCapabilities:  seat has keyboard");
      ctx->wlkeyboard= wl_seat_get_keyboard( ctx->wlseat );
      DEBUG("essSeatCapabilities:  keyboard %p", ctx->wlkeyboard );
      wl_keyboard_add_listener( ctx->wlkeyboard, &essKeyboardListener, ctx );
   }
   if ( capabilities & WL_SEAT_CAPABILITY_POINTER )
   {
      DEBUG("essSeatCapabilities:  seat has pointer");
      ctx->wlpointer= wl_seat_get_pointer( ctx->wlseat );
      DEBUG("essSeatCapabilities:  pointer %p", ctx->wlpointer );
      wl_pointer_add_listener( ctx->wlpointer, &essPointerListener, ctx );
   }
   if ( capabilities & WL_SEAT_CAPABILITY_TOUCH )
   {
      DEBUG("essSeatCapabilities:  seat has touch");
      ctx->wltouch= wl_seat_get_touch( ctx->wlseat );
      DEBUG("essSeatCapabilities:  touch %p", ctx->wltouch );
      wl_touch_add_listener( ctx->wltouch, &essTouchListener, ctx );
   }   
}

static void essSeatName( void *data, struct wl_seat *seat, const char *name )
{
   ESS_UNUSED(data);
   ESS_UNUSED(seat);
   ESS_UNUSED(name);
}

static const struct wl_seat_listener essSeatListener = {
   essSeatCapabilities,
   essSeatName 
};

static void essOutputGeometry( void *data, struct wl_output *output, int32_t x, int32_t y,
                               int32_t physical_width, int32_t physical_height, int32_t subpixel,
                               const char *make, const char *model, int32_t transform )
{
   ESS_UNUSED(output);
   ESS_UNUSED(x);
   ESS_UNUSED(y);
   ESS_UNUSED(physical_width);
   ESS_UNUSED(physical_height);
   ESS_UNUSED(subpixel);
   ESS_UNUSED(transform);

   EssCtx *ctx= (EssCtx*)data;
   if ( data && make && model )
   {
      int lenMake= strlen(make);
      int lenModel= strlen(model);
      if (
           ((lenMake == 8) && !strncmp( make, "Westeros", lenMake) ) &&
           ((lenModel == 17) && !strncmp( model, "Westeros-embedded", lenModel) )
         )
      {
         ctx->fullScreen= true;
      }
   }
}

static void essOutputMode( void *data, struct wl_output *output, uint32_t flags,
                        int32_t width, int32_t height, int32_t refresh )
{
   EssCtx *ctx= (EssCtx*)data;

   INFO("essOutputMode: outputMode: mode %d(%d) (%dx%d)", flags, WL_OUTPUT_MODE_CURRENT, width, height);
   if ( flags & WL_OUTPUT_MODE_CURRENT )
   {
      ctx->haveMode= true;
      if ( (width != ctx->planeWidth) || (height != ctx->planeHeight) )
      {
         essSetDisplaySize( ctx, width, height, false, 0, 0, 0, 0);
         ctx->windowWidth= width;
         ctx->windowHeight= height;

         if ( ctx->fullScreen && ctx->wleglwindow )
         {
            wl_egl_window_resize( ctx->wleglwindow, width, height, 0, 0 );
         }

         if ( ctx->settingsListener )
         {
            if ( ctx->settingsListener->displaySize )
            {
               ctx->settingsListener->displaySize( ctx->settingsListenerUserData, width, height );
            }
            if ( ctx->settingsListener->displaySafeArea )
            {
               ctx->settingsListener->displaySafeArea( ctx->settingsListenerUserData,
                                                       ctx->planeSafeX, ctx->planeSafeY,
                                                       ctx->planeSafeW, ctx->planeSafeH );
            }
         }
      }
   }
}

static void essOutputDone( void *data, struct wl_output *output )
{
   ESS_UNUSED(data);
   ESS_UNUSED(output);
}

static void essOutputScale( void *data, struct wl_output *output, int32_t factor )
{
   ESS_UNUSED(data);
   ESS_UNUSED(output);
   ESS_UNUSED(factor);
}

static const struct wl_output_listener essOutputListener = {
   essOutputGeometry,
   essOutputMode,
   essOutputDone,
   essOutputScale
};

#ifdef HAVE_WESTEROS
static void essShellSurfaceId(void *data,
                           struct wl_simple_shell *wl_simple_shell,
                           struct wl_surface *surface,
                           uint32_t surfaceId)
{
   EssCtx *ctx= (EssCtx*)data;
   char name[32];

   DEBUG("shell: surface created: %p id %x", surface, surfaceId);
   ctx->appSurfaceId= surfaceId;
   if ( ctx->appName )
   {
      wl_simple_shell_set_name( ctx->shell, surfaceId, ctx->appName );
   }
   else
   {
      sprintf( name, "essos-app-%x", surfaceId );
      wl_simple_shell_set_name( ctx->shell, surfaceId, name );
   }
   if ( ctx->pendingGeometryChange )
   {
      ctx->pendingGeometryChange= false;
      if ( !ctx->fullScreen )
      {
         wl_simple_shell_set_geometry( ctx->shell, ctx->appSurfaceId,
                                       ctx->windowX, ctx->windowY,
                                       ctx->windowWidth, ctx->windowHeight );
      }
   }
}

static void essShellSurfaceCreated(void *data,
                                struct wl_simple_shell *wl_simple_shell,
                                uint32_t surfaceId,
                                const char *name)
{
   ESS_UNUSED(data);
   ESS_UNUSED(wl_simple_shell);
   ESS_UNUSED(name);
}

static void essShellSurfaceDestroyed(void *data,
                                  struct wl_simple_shell *wl_simple_shell,
                                  uint32_t surfaceId,
                                  const char *name)
{
   ESS_UNUSED(data);
   ESS_UNUSED(wl_simple_shell);
   ESS_UNUSED(surfaceId);
   ESS_UNUSED(name);
}

static void essShellSurfaceStatus(void *data,
                               struct wl_simple_shell *wl_simple_shell,
                               uint32_t surfaceId,
                               const char *name,
                               uint32_t visible,
                               int32_t x,
                               int32_t y,
                               int32_t width,
                               int32_t height,
                               wl_fixed_t opacity,
                               wl_fixed_t zorder)
{
   ESS_UNUSED(data);
   ESS_UNUSED(wl_simple_shell);
   ESS_UNUSED(surfaceId);
   ESS_UNUSED(name);
   ESS_UNUSED(visible);
   ESS_UNUSED(x);
   ESS_UNUSED(y);
   ESS_UNUSED(width);
   ESS_UNUSED(height);
   ESS_UNUSED(opacity);
   ESS_UNUSED(zorder);
}

static void essShellGetSurfacesDone(void *data,
                                 struct wl_simple_shell *wl_simple_shell)
{
   ESS_UNUSED(data);
   ESS_UNUSED(wl_simple_shell);
}

static const struct wl_simple_shell_listener shellListener =
{
   essShellSurfaceId,
   essShellSurfaceCreated,
   essShellSurfaceDestroyed,
   essShellSurfaceStatus,
   essShellGetSurfacesDone
};
#endif

static void essRegistryHandleGlobal(void *data, 
                                    struct wl_registry *registry, uint32_t id,
                                    const char *interface, uint32_t version)
{
   EssCtx *ctx= (EssCtx*)data;
   int len;

   DEBUG("essRegistryHandleGlobal: id %d interface (%s) version %d", id, interface, version );

   len= strlen(interface);
   if ( (len==13) && !strncmp(interface, "wl_compositor", len) ) {
      ctx->wlcompositor= (struct wl_compositor*)wl_registry_bind(registry, id, &wl_compositor_interface, 1);
      DEBUG("essRegistryHandleGlobal: wlcompositor %p", ctx->wlcompositor);
   } 
   else if ( (len==7) && !strncmp(interface, "wl_seat", len) ) {
      ctx->wlseat= (struct wl_seat*)wl_registry_bind(registry, id, &wl_seat_interface, 4);
      DEBUG("essRegistryHandleGlobal: wlseat %p", ctx->wlseat);
      wl_seat_add_listener(ctx->wlseat, &essSeatListener, ctx);
   }
   else if ( (len==8) && !strncmp(interface, "wl_shell", len) ) {
      ctx->wlshell= (struct wl_shell*)wl_registry_bind(registry, id, &wl_shell_interface, 1);
      DEBUG("essRegistryHandleGlobal: wlshell %p", ctx->wlshell);
   }
   else if ( (len==9) && !strncmp(interface, "wl_output", len) ) {
      ctx->wloutput= (struct wl_output*)wl_registry_bind(registry, id, &wl_output_interface, 2);
      DEBUG("essRegistryHandleGlobal: wloutput %p", ctx->wloutput);
      wl_output_add_listener(ctx->wloutput, &essOutputListener, ctx);
      wl_display_roundtrip(ctx->wldisplay);
   }
   #ifdef HAVE_WESTEROS
   else if ( (len==15) && !strncmp(interface, "wl_simple_shell", len) ) {
      ctx->shell= (struct wl_simple_shell*)wl_registry_bind(registry, id, &wl_simple_shell_interface, 1);
      DEBUG("shell %p", ctx->shell );
      wl_simple_shell_add_listener(ctx->shell, &shellListener, ctx);
   }
   #endif
}

static void essRegistryHandleGlobalRemove(void *data, 
                                          struct wl_registry *registry,
                                          uint32_t name)
{
   ESS_UNUSED(data);
   ESS_UNUSED(registry);
   ESS_UNUSED(name);
}

static const struct wl_registry_listener essRegistryListener = 
{
   essRegistryHandleGlobal,
   essRegistryHandleGlobalRemove
};

static bool essPlatformInitWayland( EssCtx *ctx )
{
   bool result= false;

   if ( ctx )
   {
      ctx->wldisplay= wl_display_connect( NULL );
      if ( !ctx->wldisplay )
      {
         sprintf( ctx->lastErrorDetail,
                  "Error.  Failed to connect to wayland display" );
         goto exit;
      }

      DEBUG("essPlatformInitWayland: wldisplay %p", ctx->wldisplay);

      ctx->wlregistry= wl_display_get_registry( ctx->wldisplay );
      if ( !ctx->wlregistry )
      {
         sprintf( ctx->lastErrorDetail,
                  "Error.  Failed to get wayland display registry" );
         goto exit;
      }

      wl_registry_add_listener( ctx->wlregistry, &essRegistryListener, ctx );
      wl_display_roundtrip( ctx->wldisplay );

      ctx->waylandFd= wl_display_get_fd( ctx->wldisplay );
      if ( ctx->waylandFd < 0 )
      {
         sprintf( ctx->lastErrorDetail,
                  "Error.  Failed to get wayland display fd" );
         goto exit;
      }
      ctx->wlPollFd.fd= ctx->waylandFd;
      ctx->wlPollFd.events= POLLIN | POLLERR | POLLHUP;
      ctx->wlPollFd.revents= 0;

      ctx->displayType= (NativeDisplayType)ctx->wldisplay;
      #ifndef HAVE_WESTEROS
      if( ctx->eglDisplay == EGL_NO_DISPLAY )
      {
         ctx->eglDisplay= eglGetDisplay( ctx->displayType );
         if( !ctx->eglDisplay )
         {
            sprintf( ctx->lastErrorDetail,
                     "Error. Unable to get EGL display: eglError %X", eglGetError() );
            goto exit;
         }
      }
      #endif

      result= true;
   }

exit:
   return result;
}

static void essPlatformTermWayland( EssCtx *ctx )
{
   if ( ctx )
   {
      if ( ctx->wldisplay )
      {
         if ( ctx->wleglwindow )
         {
            wl_egl_window_destroy( ctx->wleglwindow );
            ctx->wleglwindow= 0;
         }

         if ( ctx->shellSurface )
         {
            wl_shell_surface_destroy( ctx->shellSurface );
            ctx->shellSurface= 0;
         }

         if ( ctx->wlsurface )
         {
            wl_surface_destroy( ctx->wlsurface );
            ctx->wlsurface= 0;
         }

         if ( ctx->wlcompositor )
         {
            wl_compositor_destroy( ctx->wlcompositor );
            ctx->wlcompositor= 0;
         }

         if ( ctx->wlshell )
         {
            wl_shell_destroy(ctx->wlshell);
            ctx->wlshell= 0;
         }

         if ( ctx->wlseat )
         {
            wl_seat_destroy(ctx->wlseat);
            ctx->wlseat= 0;
         }

         if ( ctx->wloutput )
         {
            wl_output_destroy(ctx->wloutput);
            ctx->wloutput= 0;
         }

         if ( ctx->wlregistry )
         {
            wl_registry_destroy( ctx->wlregistry );
            ctx->wlregistry= 0;
         }

         wl_display_disconnect( ctx->wldisplay );
         ctx->wldisplay= 0;
      }
   }
}

static void essInitInputWayland( EssCtx *ctx )
{
   if ( ctx )
   {
      #ifdef HAVE_WAYLAND
      if ( ctx->wlseat )
      {
         wl_seat_add_listener(ctx->wlseat, &essSeatListener, ctx);
         wl_display_roundtrip(ctx->wldisplay);
      }
      #endif
   }
}

static int essReadAndDispatchWaylandDisplayQueue( EssCtx *ctx )
{
   if ( wl_display_prepare_read( ctx->wldisplay ) == 0 )
   {
      if ( wl_display_read_events( ctx->wldisplay ) == -1 )
      {
         return -1;
      }
   }
   return wl_display_dispatch_pending( ctx->wldisplay );
}

static void essProcessRunWaylandEventLoopOnce( EssCtx *ctx )
{
   int n;
   bool error= false;

   wl_display_flush( ctx->wldisplay );
   wl_display_dispatch_pending( ctx->wldisplay );
   ctx->wlPollFd.revents= 0;
   n= poll(&ctx->wlPollFd, 1, 0);
   if ( n > 0 )
   {
      if ( ctx->wlPollFd.revents & POLLIN )
      {
         if ( essReadAndDispatchWaylandDisplayQueue( ctx ) == -1 )
         {
            error= true;
         }
      }
      if ( ctx->wlPollFd.revents & (POLLERR|POLLHUP) )
      {
         error= true;
      }
      if ( error )
      {
         if ( ctx->terminateListener && ctx->terminateListener->terminated )
         {
            ctx->terminateListener->terminated( ctx->terminateListenerUserData );
         }
      }
   }
}
#endif

#ifdef HAVE_WESTEROS
extern "C"
{
   typedef void (*DisplaySizeCallback)( void *userData, int width, int height );
   typedef bool (*AddDisplaySizeListener)( WstGLCtx *ctx, void *userData, DisplaySizeCallback listener );
   typedef bool (*GetDisplaySafeArea)( WstGLCtx *ctx, int *x, int *y, int *w, int *h );
   typedef bool (*GetDisplayCaps)( WstGLCtx *ctx, unsigned int *caps );
   typedef bool (*SetDisplayMode)( WstGLCtx *ctx, const char *mode );

}

static GetDisplaySafeArea gGetDisplaySafeArea= 0;
static GetDisplayCaps gGetDisplayCaps= 0;
static SetDisplayMode gSetDisplayMode= 0;

void displaySizeCallback( void *userData, int width, int height )
{
   EssCtx *ctx= (EssCtx*)userData;
   int safex= 0, safey= 0, safew= 0, safeh= 0;
   bool customSafe= false;
   INFO("displaySizeCallback: display size %dx%d", width, height );
   if ( gGetDisplaySafeArea )
   {
      if ( gGetDisplaySafeArea( ctx->glCtx, &safex, &safey, &safew, &safeh ) )
      {
         INFO("displaySizeCallback: display safe (%d,%d,%d,%d)", safex, safey, safew, safeh );
         customSafe= true;
      }
      else
      {
         ERROR("failure to get display safe area");
      }
   }
   essSetDisplaySize( ctx, width, height, customSafe, safex, safey, safew, safeh);
   if ( !ctx->haveMode )
   {
      ctx->windowWidth= width;
      ctx->windowHeight= height;
   }
   ctx->resizePending= true;
   ctx->resizeWidth= width;
   ctx->resizeHeight= height;
}

static bool essPlatformInitDirect( EssCtx *ctx )
{
   bool result= false;

   if ( ctx )
   {
      ctx->glCtx= WstGLInit();
      if ( !ctx->glCtx )
      {
         sprintf( ctx->lastErrorDetail,
                  "Error.  Failed to create a platform context" );
         goto exit;
      }
      DEBUG("essPlatformInitDirect: glCtx %p", ctx->glCtx);

      if ( getenv("ESSOS_DIRECT_NO_EVENT_LOOP_THROTTLE") )
      {
         INFO("essos no event loop throttle for direct");
         ctx->eventLoopPeriodMS= 0;
      }

      {
         void *module= dlopen( "libwesteros_gl.so.0.0.0", RTLD_NOW );
         if ( module )
         {
            AddDisplaySizeListener addDisplaySizeListener= 0;
            addDisplaySizeListener= (AddDisplaySizeListener)dlsym( module, "_WstGLAddDisplaySizeListener" );
            gGetDisplaySafeArea= (GetDisplaySafeArea)dlsym( module, "_WstGLGetDisplaySafeArea" );
            gGetDisplayCaps= (GetDisplayCaps)dlsym( module, "_WstGLGetDisplayCaps" );
            gSetDisplayMode= (SetDisplayMode)dlsym( module, "_WstGLSetDisplayMode" );
            if ( addDisplaySizeListener )
            {
               addDisplaySizeListener( ctx->glCtx, ctx, displaySizeCallback );
            }
            dlclose( module );
         }
      }

      ctx->displayType= EGL_DEFAULT_DISPLAY;

      result= true;
   }

exit:
   return result;
}

static void essPlatformTermDirect( EssCtx *ctx )
{
   if ( ctx )
   {
      if ( ctx->nativeWindow )
      {
         WstGLDestroyNativeWindow( ctx->glCtx, (void*)ctx->nativeWindow );
         ctx->nativeWindow= 0;
      }
      if ( ctx->glCtx )
      {
         WstGLTerm( ctx->glCtx );
         ctx->glCtx= 0;
      }
   }
}

static bool essPlatformSetDisplayModeDirect( EssCtx *ctx, const char *mode )
{
   bool result= false;
   bool canSetMode= false;

   #ifdef WESTEROS_GL_DISPLAY_CAPS
   if ( gGetDisplayCaps && gSetDisplayMode )
   {
      unsigned int displayCaps= 0;
      if ( gGetDisplayCaps( ctx->glCtx, &displayCaps ) )
      {
         if ( displayCaps & WstGLDisplayCap_modeset )
         {
            canSetMode= true;
         }
      }
   }
   #endif

   if ( !canSetMode )
   {
      sprintf( ctx->lastErrorDetail,
               "Error.  Mode setting not supported" );
      goto exit;
   }

   result= gSetDisplayMode( ctx->glCtx, mode );
   if ( !result )
   {
      sprintf( ctx->lastErrorDetail,
               "Error.  Mode setting for (%s) failed", mode );
      goto exit;
   }

exit:

   return result;
}

static const char *inputPath= "/dev/input/";

static bool essCheckBit( unsigned char *bits, int bit )
{
   int i= (bit/8);
   int m= (bit%8);
   return (bits[i] & (1<<m));
}

static void essGamepadBuildButtonMap( EssCtx *ctx, EssGamepad *gp, unsigned char *bits, int numBits )
{
   int buttonCount= 0;
   int i;

   for( i= BTN_MISC; i < numBits; ++i )
   {
      if ( essCheckBit( bits, i ) )
      {
         gp->buttonMap[buttonCount++]= i;
      }
   }
   gp->buttonCount= buttonCount;
}

static void essGamepadBuildAxisMap( EssCtx *ctx, EssGamepad *gp, unsigned char *bits, int numBits )
{
   int axisCount= 0;
   int i;

   for( i= 0; i < numBits; ++i )
   {
      if ( essCheckBit( bits, i ) )
      {
         gp->axisMap[axisCount++]= i;
      }
   }
   gp->axisCount= axisCount;
}

static EssGamepad* essCheckForGamepad( EssCtx *ctx, int fd, const char *devPathName )
{
   EssGamepad *gpReturn= 0;
   int rc;
   unsigned int evbits= 0;
   int buttonMapSize= KEY_CNT;
   int axisMapSize= ABS_CNT;
   unsigned char *bits= 0;
   int bitsSize;

   rc= ioctl( fd, EVIOCGBIT(0,sizeof(evbits)), &evbits);
   if ( rc >= 0 )
   {
      bool hasKey= essCheckBit((unsigned char*)&evbits, EV_KEY);
      bool hasAbs= essCheckBit((unsigned char*)&evbits, EV_ABS);
      if ( hasKey && hasAbs )
      {
         bitsSize= ((buttonMapSize+7)/8);
         bits= (unsigned char*)calloc( 1, bitsSize );
         if ( !bits )
         {
            ERROR("No memory for EV key bits");
            goto exit;
         }

         rc= ioctl( fd, EVIOCGBIT(EV_KEY, bitsSize), bits );
         if ( rc >= 0 )
         {
            if ( essCheckBit( bits, BTN_SOUTH ) )
            {
               EssGamepad *gp;

               gp= (EssGamepad*)calloc( 1, sizeof(EssGamepad));
               if ( gp )
               {
                  gp->ctx= ctx;
                  gp->devicePath= strdup(devPathName);
                  gp->fd= fd;
                  ctx->gamepads.push_back( gp );

                  essGamepadBuildButtonMap( ctx, gp, bits, buttonMapSize );

                  if ( axisMapSize <= buttonMapSize )
                  {
                     memset( bits, 0, bitsSize );
                  }
                  else
                  {
                     free( bits );
                     bitsSize= ((axisMapSize+7)/8);
                     bits= (unsigned char*)calloc( 1, bitsSize );
                     if ( !bits )
                     {
                        ERROR("No memory for EV axis bits");
                        goto exit;
                     }
                  }

                  rc= ioctl( fd, EVIOCGBIT(EV_ABS, bitsSize), bits );
                  if ( rc >= 0 )
                  {
                     essGamepadBuildAxisMap( ctx, gp, bits, axisMapSize );
                  }
                  DEBUG("essCheckForGamepad: gp %p has %d buttons %d axes", gp, gp->buttonCount, gp->axisCount);
                  INFO("essCheckForGamepad: gp %p has %d buttons %d axes", gp, gp->buttonCount, gp->axisCount);

                  pthread_mutex_unlock( &ctx->mutex );
                  essGamepadNotifyConnected( ctx, gp );
                  pthread_mutex_lock( &ctx->mutex );

                  gpReturn= gp;
               }
               else
               {
                  ERROR("Error: unable to allocate gamepad");
               }
            }
         }
         else
         {
            ERROR("EVIOGCBIT for keymap failed");
         }
      }
   }
   else
   {
      ERROR("EVIOGCBIT for caps failed");
   }

exit:
   if ( bits )
   {
      free( bits );
   }
   return gpReturn;
}

static void essReadInputDeviceMetaData(EssCtx *ctx, int fd, const char * devicePathName)
{
   DEBUG("essReadInputDeviceMetaData: Read metadata for %s", devicePathName);

   EssInputDeviceMetadata * meta = (EssInputDeviceMetadata*)calloc( 1, sizeof(EssInputDeviceMetadata));

   struct stat buf = {};
   if (fstat(fd, &buf) != 0 || !S_ISCHR(buf.st_mode))
   {
      ERROR("essReadInputDeviceMetaData: failed to get input device number of '%s'", devicePathName);
   }

   meta->deviceNumber = buf.st_rdev;

   // try and get the device id, we use this to determine if any special
   // handling needs to be done for the input events
   struct input_id id = {};
   if (ioctl(fd, EVIOCGID, &id) != 0)
   {
      ERROR("essReadInputDeviceMetaData: failed to get input device id");
   }

   meta->id = id;

   // try and get the device 'physical address' for bluetooth this is the
   // BD_ADDR in string form
   std::string physAddress(256, '\0');
   int ret = ioctl(fd, EVIOCGPHYS(physAddress.size()), physAddress.data());
   if (ret < 0)
   {
      DEBUG("essReadInputDeviceMetaData: failed to get physical address");

      // if the kernel hasn't given us a physical address then try using the
      // unique id
      ret = ioctl(fd, EVIOCGUNIQ(physAddress.size()), physAddress.data());
      if (ret < 0)
      {
         DEBUG("essReadInputDeviceMetaData: failed to get unique identifier");

         // getting the unique id also failed then use the device path
         physAddress = devicePathName;
      }
   }

   if (ret >= 0)
   {
      physAddress.resize(ret);
   }

   // trim any trailing null terminating chars
   while (!physAddress.empty() && (physAddress.at(physAddress.size()-1) == '\0'))
   {
      physAddress.erase(physAddress.size()-1,1);
   }

   meta->devicePhysicalAddress = strdup(physAddress.c_str());

   ctx->inputDeviceMetadata[fd] = meta;

   DEBUG("essReadInputDeviceMetaData: metadata for input device type %d, details 0x%04x:0x%04x:0x%04x, physical '%s'",
           meta->id.bustype, meta->id.vendor, meta->id.product, meta->id.version, meta->devicePhysicalAddress);
}

static void essReleaseInputDeviceMetaData(EssCtx *ctx, int fd)
{
   std::map<int, EssInputDeviceMetadata*>::iterator metaIter = ctx->inputDeviceMetadata.find(fd);

   if (metaIter != ctx->inputDeviceMetadata.end())
   {
      DEBUG("essReleaseInputDeviceMetaData: Release metadata for '%s'", metaIter->second->devicePhysicalAddress );

      if (metaIter->second->devicePhysicalAddress != 0)
      {
         free((char *)metaIter->second->devicePhysicalAddress);
         metaIter->second->devicePhysicalAddress = 0;
      }
      ctx->inputDeviceMetadata.erase(fd);
   }
   else
   {
      WARNING("essReleaseInputDeviceMetaData: failed to find metadata for fd %d", fd);
   }
}

static int essOpenInputDevice( EssCtx *ctx, const char *devPathName )
{
   int fd= -1;   
   struct stat buf;
   
   if ( stat( devPathName, &buf ) == 0 )
   {
      if ( S_ISCHR(buf.st_mode) )
      {
         EssGamepad *gp= essGetGamepadFromPath( ctx, devPathName );
         EssInputDevice *idev= essGetInputDeviceFromPath( ctx, devPathName );
         if ( gp )
         {
            fd= gp->fd;
         }
         else if ( idev )
         {
            fd= idev->fd;
         }
         else
         {
            fd= open( devPathName, O_RDONLY | O_CLOEXEC );
            DEBUG( "essOpenInputDevice: opened device %s : fd %d", devPathName, fd );
            if ( fd >= 0 )
            {
               essReadInputDeviceMetaData(ctx, fd, devPathName);
               ctx->inputDeviceScanCode[fd] = {};
            }
         }

         if ( fd < 0 )
         {
            snprintf( ctx->lastErrorDetail, ESS_MAX_ERROR_DETAIL,
            "Error: error opening device: %s\n", devPathName );
         }
         else
         {
            pollfd pfd;
            DEBUG( "essOpenInputDevice: have device %s : fd %d", devPathName, fd );
            pfd.fd= fd;
            ctx->inputDeviceFds.push_back( pfd );

            if ( !gp )
            {
               gp= essCheckForGamepad( ctx, fd, devPathName );
            }
            if ( !gp && !idev )
            {
               idev= (EssInputDevice*)calloc( 1, sizeof(EssInputDevice));
               if ( idev )
               {
                  idev->ctx= ctx;
                  idev->devicePath= strdup(devPathName);
                  idev->fd= fd;
                  ctx->inputDevices.push_back( idev );
               }
            }
         }
      }
      else
      {
         DEBUG("essOpenInputDevice: ignoring non character device %s", devPathName );
      }
   }
   else
   {
      DEBUG( "essOpenInputDevice: error performing stat on device: %s", devPathName );
   }
   
   return fd;
}

static char *essGetInputDevice( EssCtx *ctx, const char *path, char *devName )
{
   int len;
   char *devicePathName= 0;
   struct stat buffer;
   
   if ( !devName )
      return devicePathName; 
      
   len= strlen( devName );
   
   devicePathName= (char *)malloc( strlen(path)+len+1);
   if ( devicePathName )
   {
      strcpy( devicePathName, path );
      strcat( devicePathName, devName );     

      if ( !stat(devicePathName, &buffer) )
      {
         DEBUG( "essGetInputDevice: found %s", devicePathName );
      }
      else
      {
         free( devicePathName );
         devicePathName= 0;
      }
   }

   return devicePathName;
}

static void essGetInputDevices( EssCtx *ctx )
{
   DIR *dir;
   struct dirent *result;
   char *devPathName;
   if ( NULL != (dir = opendir( inputPath )) )
   {
      while( NULL != (result = readdir( dir )) )
      {
         if ( result->d_type != DT_DIR )
         {
            devPathName= essGetInputDevice( ctx, inputPath, result->d_name );
            if ( devPathName )
            {
               if ( !strncmp(result->d_name, "event", 5) )
               {
                  if (essOpenInputDevice( ctx, devPathName ) < 0 )
                  {
                     ERROR("essos: could not open device %s", devPathName);
                  }
               }
               free( devPathName );
            }
         }
      }

      closedir( dir );
   }
}

static void essMonitorInputDevicesLifecycleBegin( EssCtx *ctx )
{
   pollfd pfd;

   ctx->notifyFd= inotify_init();
   if ( ctx->notifyFd >= 0 )
   {
      pfd.fd= ctx->notifyFd;
      ctx->watchFd= inotify_add_watch( ctx->notifyFd, inputPath, IN_CREATE | IN_DELETE );
      ctx->inputDeviceFds.push_back( pfd );
   }
}

static void essMonitorInputDevicesLifecycleEnd( EssCtx *ctx )
{
   if ( ctx->notifyFd >= 0 )
   {
      if ( ctx->watchFd >= 0 )
      {
         inotify_rm_watch( ctx->notifyFd, ctx->watchFd );
         ctx->watchFd= -1;
      }
      ctx->inputDeviceFds.pop_back();
      close( ctx->notifyFd );
      ctx->notifyFd= -1;
   }
}

static void essReleaseInputDevices( EssCtx *ctx )
{
   while( ctx->inputDeviceFds.size() > 0 )
   {
      pollfd pfd= ctx->inputDeviceFds[0];
      DEBUG( "essos: closing device fd: %d", pfd.fd );
      essReleaseInputDeviceMetaData(ctx, pfd.fd);
      ctx->inputDeviceScanCode.erase(pfd.fd);
      close( pfd.fd );
      ctx->inputDeviceFds.erase( ctx->inputDeviceFds.begin() );
   }
}

static void essFillKeyAndMetadataListenerMetadata( EssCtx *ctx, int fd )
{
   if (ctx->keyAndMetadataListener != 0 && ctx->keyAndMetadataListenerMetadata != 0)
   {
      std::map<int, EssInputDeviceMetadata*>::iterator storedMetadataIter
            = ctx->inputDeviceMetadata.find(fd);

      if (storedMetadataIter != ctx->inputDeviceMetadata.end())
      {
         EssInputDeviceMetadata* storedMetadata = storedMetadataIter->second;

         if (ctx->keyAndMetadataListenerMetadata->devicePhysicalAddress != 0)
         {
            free((char*) ctx->keyAndMetadataListenerMetadata->devicePhysicalAddress);
            ctx->keyAndMetadataListenerMetadata->devicePhysicalAddress = 0;
         }

         ctx->keyAndMetadataListenerMetadata->devicePhysicalAddress = strdup(storedMetadata->devicePhysicalAddress);
         ctx->keyAndMetadataListenerMetadata->id = storedMetadata->id;
         ctx->keyAndMetadataListenerMetadata->deviceNumber = storedMetadata->deviceNumber;
         ctx->keyAndMetadataListenerMetadata->filterCode = storedMetadata->filterCode;
      }
      else
      {
         WARNING("essProcessInputDevices metadata not found for fd: %d", fd);
      }
   }
}

static const uint8_t SCC_PAIRING_MODE = 5;

void essUpdateMetadataFilterCode(EssCtx *ctx,  int fd)
{
   if (ctx->inputDeviceScanCode[fd].shortCustomerCode != SCC_PAIRING_MODE)
   {
      if (ctx->inputDeviceScanCode[fd].filterByte != ctx->inputDeviceScanCode[fd].prevFilterByte)
      {
         ctx->inputDeviceMetadata[fd]->filterCode = ctx->inputDeviceScanCode[fd].filterByte;
         ctx->inputDeviceScanCode[fd].prevFilterByte = ctx->inputDeviceScanCode[fd].filterByte;
         DEBUG("essUpdateMetadataFilterCode: update fd: %d, filterCode: 0x%02x", fd, ctx->inputDeviceMetadata[fd]->filterCode);
      }
   }
   ctx->inputDeviceScanCode[fd].filterByte = 0x0;
   ctx->inputDeviceScanCode[fd].shortCustomerCode = 0x0;
}

void essClearInputDeviceScanCode(EssCtx *ctx,  int fd)
{
   ctx->inputDeviceScanCode[fd].filterByte = 0x0;
   ctx->inputDeviceScanCode[fd].shortCustomerCode = 0x0;
}

void essReadInputDeviceScanCode(EssCtx *ctx,  int fd, int32_t inputEventValue)
{
   ctx->inputDeviceScanCode[fd].filterByte = ((inputEventValue >> 8) & 0xff);
   ctx->inputDeviceScanCode[fd].shortCustomerCode = ((inputEventValue >> 20) & 0x0f);
}

static void essProcessInputDevices( EssCtx *ctx )
{
   int deviceCount;
   int i, n;
   int pollCnt= 0;
   input_event e;
   char intfyEvent[512];
   static bool mouseMoved= false;
   static int mouseAccel= 1;
   static int mouseX= 0;
   static int mouseY= 0;
   static int currTouchSlot= 0;
   static bool touchChanges= false;
   static bool touchClean= false;

   deviceCount= ctx->inputDeviceFds.size();

   for( i= 0; i < deviceCount; ++i )
   {
      ctx->inputDeviceFds[i].events= POLLIN | POLLERR;
      ctx->inputDeviceFds[i].revents= 0;
   }

   n= poll(&ctx->inputDeviceFds[0], deviceCount, 0);
   while ( n >= 0 )
   {
      for( i= 0; i < deviceCount; ++i )
      {
         if ( ctx->inputDeviceFds[i].revents & POLLIN )
         {
            if ( ctx->inputDeviceFds[i].fd == ctx->notifyFd )
            {
               // A hotplug event has occurred
               n= read( ctx->notifyFd, &intfyEvent, sizeof(intfyEvent) );
               if ( n >= sizeof(struct inotify_event) )
               {
                  struct inotify_event *iev= (struct inotify_event*)intfyEvent;
                  {
                     essValidateGamepads( ctx );
                     essValidateInputDevices( ctx );

                     // Re-discover devices
                     DEBUG("essProcessInputDevices: inotify: mask %x (%s) wd %d (%d)", iev->mask, iev->name, iev->wd, ctx->watchFd );
                     pthread_mutex_lock( &ctx->mutex );
                     pollfd pfd= ctx->inputDeviceFds.back();
                     ctx->inputDeviceFds.pop_back();
                     essReleaseInputDevices( ctx );
                     usleep( 100000 );
                     essGetInputDevices( ctx );
                     ctx->inputDeviceFds.push_back( pfd );
                     deviceCount= ctx->inputDeviceFds.size();
                     pthread_mutex_unlock( &ctx->mutex );
                     break;
                  }
               }
            }
            else
            {
               EssGamepad *gp= essGetGamepadFromFd( ctx, ctx->inputDeviceFds[i].fd );
               if ( gp )
               {
                  essProcessGamepad( ctx, gp );
                  break;
               }

               n= read( ctx->inputDeviceFds[i].fd, &e, sizeof(input_event) );
               if ( n > 0 )
               {
                  switch( e.type )
                  {
                     case EV_KEY:

                        essUpdateMetadataFilterCode(ctx, ctx->inputDeviceFds[i].fd);

                        switch( e.code )
                        {
                           case BTN_LEFT:
                           case BTN_RIGHT:
                           case BTN_MIDDLE:
                           case BTN_SIDE:
                           case BTN_EXTRA:
                              {
                                 unsigned int keyCode= e.code;
                                 
                                 switch ( e.value )
                                 {
                                    case 0:
                                       essProcessPointerButtonReleased( ctx, keyCode );
                                       break;
                                    case 1:
                                       essProcessPointerButtonPressed( ctx, keyCode );
                                       break;
                                    default:
                                       break;
                                 }
                              }
                              break;
                           case BTN_TOUCH:
                              // Ignore
                              break;
                           default:
                              {
                                 int keyCode= e.code;
                                 long long timeMillis= e.time.tv_sec*1000LL+e.time.tv_usec/1000LL;

                                 switch ( e.value )
                                 {
                                    case 0:
                                       essFillKeyAndMetadataListenerMetadata(ctx, ctx->inputDeviceFds[i].fd);
                                       ctx->keyPressed= false;
                                       essProcessKeyReleased( ctx, keyCode );
                                       break;
                                    case 1:
                                       if ( !(ctx->keyPressed && (ctx->lastKeyCode == keyCode)) )
                                       {
                                          ctx->lastKeyCode= keyCode;
                                          ctx->keyPressed= true;
                                          ctx->keyRepeating= false;
                                          ctx->lastKeyTime= timeMillis;
                                          essFillKeyAndMetadataListenerMetadata(ctx, ctx->inputDeviceFds[i].fd);
                                          essProcessKeyPressed( ctx, keyCode );
                                       }
                                       break;
                                    default:
                                       break;
                                 }
                              }
                              break;
                        }
                        break;
                     case EV_REL:
                        switch( e.code )
                        {
                           case REL_X:
                              mouseX= mouseX + e.value * mouseAccel;
                              if ( mouseX < 0 ) mouseX= 0;
                              if ( mouseX > ctx->planeWidth ) mouseX= ctx->planeWidth;
                              mouseMoved= true;
                              break;
                           case REL_Y:
                              mouseY= mouseY + e.value * mouseAccel;
                              if ( mouseY < 0 ) mouseY= 0;
                              if ( mouseY > ctx->planeHeight ) mouseY= ctx->planeHeight;
                              mouseMoved= true;
                              break;
                           default:
                              break;
                        }
                        break;
                     case EV_SYN:
                        {
                           if ( mouseMoved )
                           {
                              essProcessPointerMotion( ctx, mouseX, mouseY );
                              
                              mouseMoved= false;
                           }
                           if ( touchChanges )
                           {
                              bool touchEvents= false;
                              for( int i= 0; i < ESS_MAX_TOUCH; ++i )
                              {
                                 if ( ctx->touch[i].valid )
                                 {
                                    if ( ctx->touch[i].starting )
                                    {
                                       essProcessTouchDown( ctx, ctx->touch[i].id, ctx->touch[i].x, ctx->touch[i].y );
                                       touchEvents= true;
                                    }
                                    else if ( ctx->touch[i].stopping )
                                    {
                                       essProcessTouchUp( ctx, ctx->touch[i].id );
                                       touchEvents= true;
                                    }
                                    else if ( ctx->touch[i].moved )
                                    {
                                       essProcessTouchMotion( ctx, ctx->touch[i].id, ctx->touch[i].x, ctx->touch[i].y );
                                       touchEvents= true;
                                    }
                                 }
                              }

                              if ( touchEvents )
                              {
                                 essProcessTouchFrame( ctx );
                              }

                              if ( touchClean )
                              {
                                 touchClean= false;
                                 for( int i= 0; i < ESS_MAX_TOUCH; ++i )
                                 {
                                    ctx->touch[i].starting= false;
                                    if ( ctx->touch[i].stopping )
                                    {
                                       ctx->touch[i].valid= false;
                                       ctx->touch[i].stopping= false;
                                       ctx->touch[i].id= -1;
                                    }
                                 }
                              }
                              touchChanges= false;
                           }

                           essClearInputDeviceScanCode(ctx, ctx->inputDeviceFds[i].fd);
                        }
                        break;
                     case EV_ABS:
                        switch( e.code )
                        {
                           case ABS_MT_SLOT:
                              currTouchSlot= e.value;
                              break;
                           case ABS_MT_POSITION_X:
                              if ( (currTouchSlot >= 0) && (currTouchSlot < ESS_MAX_TOUCH) )
                              {
                                 ctx->touch[currTouchSlot].x= e.value;
                                 ctx->touch[currTouchSlot].moved= true;
                                 touchChanges= true;
                              }
                              break;
                           case ABS_MT_POSITION_Y:
                              if ( (currTouchSlot >= 0) && (currTouchSlot < ESS_MAX_TOUCH) )
                              {
                                 ctx->touch[currTouchSlot].y= e.value;
                                 ctx->touch[currTouchSlot].moved= true;
                                 touchChanges= true;
                              }
                              break;
                           case ABS_MT_TRACKING_ID:
                              if ( (currTouchSlot >= 0) && (currTouchSlot < ESS_MAX_TOUCH) )
                              {
                                 ctx->touch[currTouchSlot].valid= true;
                                 if ( e.value >= 0 )
                                 {
                                    ctx->touch[currTouchSlot].id= e.value;
                                    ctx->touch[currTouchSlot].starting= true;
                                 }
                                 else
                                 {
                                    ctx->touch[currTouchSlot].stopping= true;
                                 }
                                 touchClean= true;
                                 touchChanges= true;
                              }
                              break;
                           default:
                              break;
                        }
                        break;

                     case EV_MSC:
                        if (e.code == MSC_SCAN)
                        {
                           essReadInputDeviceScanCode(ctx, ctx->inputDeviceFds[i].fd, e.value);
                        }
                        break;
                     default:
                        break;
                  }
               }
            }
         }
      }

      if ( ++pollCnt >= ESS_INPUT_POLL_LIMIT )
      {
         break;
      }
      n= poll(&ctx->inputDeviceFds[0], deviceCount, 0);
   }
}

static void essProcessGamepad( EssCtx *ctx, EssGamepad *gp )
{
   int rc;
   struct input_event ev;

   rc= read( gp->fd, &ev, sizeof(struct input_event) );
   if ( rc == sizeof(struct input_event) )
   {
      int i;
      switch( ev.type )
      {
         case EV_KEY:
            for( i= 0; i < gp->buttonCount; i++ )
            {
               if ( gp->buttonMap[i] == ev.code )
               {
                  gp->buttonState[i]= ev.value;
                  if ( ev.value )
                  {
                     essProcessGamepadButtonPressed( gp, ev.code );
                  }
                  else
                  {
                     essProcessGamepadButtonReleased( gp, ev.code );
                  }
                  break;
               }
            }
            break;
         case EV_ABS:
            for( i= 0; i < gp->axisCount; i++ )
            {
               if ( gp->axisMap[i] == ev.code )
               {
                  struct input_absinfo info;
                  rc= ioctl( gp->fd, EVIOCGABS(ev.code), &info );
                  if ( rc == 0 )
                  {
                     int value;
                     if ( ev.value == info.minimum )
                     {
                        value= -32768;
                     }
                     else if ( ev.value == info.maximum )
                     {
                        value= 32767;
                     }
                     else if ( ev.value == 0 )
                     {
                        value= 0;
                     }
                     else
                     {
                        value= ((ev.value-info.minimum)*65536)/(info.maximum-info.minimum+1)-32768;
                     }
                     gp->axisState[i]= value;
                     essProcessGamepadAxisChanged( gp, ev.code, value );
                  }
                  break;
               }
            }
            break;
      }
   }
}

static void essFreeInputDevice( EssInputDevice *idev )
{
   if ( idev->devicePath )
   {
      free( (char*)idev->devicePath );
   }
   if ( idev->fd >= 0 )
   {
      close( idev->fd );
   }
   free( idev );
}

static void essFreeInputDevices( EssCtx *ctx )
{
    for (EssInputDevice *idev : ctx->inputDevices)
    {
        if (idev)
        {
            essFreeInputDevice(idev);
        }
    }
    ctx->inputDevices.clear();
}

static void essValidateInputDevices( EssCtx *ctx )
{
   if ( ctx )
   {
      std::vector<EssInputDevice*> toRelease= std::vector<EssInputDevice*>();

      pthread_mutex_lock( &ctx->mutex );

      // Remove all input device fd's from inputDeviceFds
      for( std::vector<EssInputDevice*>::iterator it= ctx->inputDevices.begin();
           it != ctx->inputDevices.end();
           ++it )
      {
         EssInputDevice *idev= (*it);
         if ( idev )
         {
            for( std::vector<pollfd>::iterator it= ctx->inputDeviceFds.begin();
                 it != ctx->inputDeviceFds.end();
                 ++it )
            {
               if ( (*it).fd == idev->fd )
               {
                  ctx->inputDeviceFds.erase( it );
                  break;
               }
            }
         }
      }

      // Check each input device to see if it is still valid
      std::vector<EssInputDevice*>::iterator it= ctx->inputDevices.begin();
      while ( it != ctx->inputDevices.end() )
      {
         EssInputDevice *idev= (*it);
         if ( idev )
         {
            bool keep= false;
            struct stat buf;

            if ( stat( idev->devicePath, &buf ) == 0 )
            {
               if ( S_ISCHR(buf.st_mode) )
               {
                  keep= true;
               }
            }

            if ( !keep )
            {
               it= ctx->inputDevices.erase( it );
               toRelease.push_back( idev );
               continue;
            }
         }
         ++it;
      }

      pthread_mutex_unlock( &ctx->mutex );

      // Discard all gamepads that are no longer valid
      if ( toRelease.size() )
      {
         for( std::vector<EssInputDevice*>::iterator it= toRelease.begin();
              it != toRelease.end();
              ++it )
         {
            EssInputDevice *idev= (*it);
            if ( idev )
            {
               if ( idev->devicePath )
               {
                  free( (char*)idev->devicePath );
               }
               if ( idev->fd >= 0 )
               {
                  close( idev->fd );
               }
               free( idev );
            }
         }
      }
   }
}

// Must be called holding context mutex
static EssInputDevice *essGetInputDeviceFromPath( EssCtx *ctx, const char *path )
{
   EssInputDevice *idev= 0;
   if ( ctx && path )
   {
      int i, imax;

      imax= ctx->inputDevices.size();
      for( i= 0; i < imax; ++i )
      {
         if ( strcmp( path, ctx->inputDevices[i]->devicePath ) == 0 )
         {
            idev= ctx->inputDevices[i];
            break;
         }
      }
   }
   return idev;
}

static void essFreeGamepad( EssGamepad *gp )
{
   if ( gp->devicePath )
   {
      free( (char*)gp->devicePath );
   }
   if ( gp->name )
   {
      free( (char*)gp->name );
   }
   if ( gp->fd >= 0 )
   {
      close( gp->fd );
   }
   free( gp );
}

static void essFreeGamepads( EssCtx *ctx )
{
   std::vector<EssGamepad*>::iterator it= ctx->gamepads.begin();
   while ( it != ctx->gamepads.end() )
   {
      EssGamepad *gp= (*it);
      if ( gp )
      {
         it= ctx->gamepads.erase( it );
         essFreeGamepad( gp );
         continue;
      }
      ++it;
   }
}

static void essValidateGamepads( EssCtx *ctx )
{
   if ( ctx )
   {
      std::vector<EssGamepad*> toRelease= std::vector<EssGamepad*>();

      pthread_mutex_lock( &ctx->mutex );

      // Remove all gamepad fd's from inputDeviceFds
      for( std::vector<EssGamepad*>::iterator it= ctx->gamepads.begin();
           it != ctx->gamepads.end();
           ++it )
      {
         EssGamepad *gp= (*it);
         if ( gp )
         {
            for( std::vector<pollfd>::iterator it= ctx->inputDeviceFds.begin();
                 it != ctx->inputDeviceFds.end();
                 ++it )
            {
               if ( (*it).fd == gp->fd )
               {
                  ctx->inputDeviceFds.erase( it );
                  break;
               }
            }
         }
      }

      // Check each gamepad to see if it is still valid
      std::vector<EssGamepad*>::iterator it= ctx->gamepads.begin();
      while ( it != ctx->gamepads.end() )
      {
         EssGamepad *gp= (*it);
         if ( gp )
         {
            bool keep= false;
            struct stat buf;

            if ( stat( gp->devicePath, &buf ) == 0 )
            {
               if ( S_ISCHR(buf.st_mode) )
               {
                  keep= true;
               }
            }

            if ( !keep )
            {
               it= ctx->gamepads.erase( it );
               toRelease.push_back( gp );
               continue;
            }
         }
         ++it;
      }

      pthread_mutex_unlock( &ctx->mutex );

      // Discard all gamepads that are no longer valid
      if ( toRelease.size() )
      {
         for( std::vector<EssGamepad*>::iterator it= toRelease.begin();
              it != toRelease.end();
              ++it )
         {
            EssGamepad *gp= (*it);
            if ( gp )
            {
               essGamepadNotifyDisconnected( ctx, gp );

               essFreeGamepad( gp );
            }
         }
      }
   }
}

// Must be called holding context mutex
static EssGamepad *essGetGamepadFromPath( EssCtx *ctx, const char *path )
{
   EssGamepad *gp= 0;
   if ( ctx && path )
   {
      int i, imax;

      imax= ctx->gamepads.size();
      for( i= 0; i < imax; ++i )
      {
         if ( strcmp( path, ctx->gamepads[i]->devicePath ) == 0 )
         {
            gp= ctx->gamepads[i];
            break;
         }
      }
   }
   return gp;
}

static EssGamepad *essGetGamepadFromFd( EssCtx *ctx, int fd )
{
   EssGamepad *gp= 0;
   if ( ctx )
   {
      int i, imax;

      pthread_mutex_lock( &ctx->mutex );
      imax= ctx->gamepads.size();
      for( i= 0; i < imax; ++i )
      {
         if ( ctx->gamepads[i]->fd == fd )
         {
            gp= ctx->gamepads[i];
            break;
         }
      }
      pthread_mutex_unlock( &ctx->mutex );
   }
   return gp;
}

static void essGamepadNotifyConnected( EssCtx *ctx, EssGamepad *gp )
{
   if ( ctx )
   {
      DEBUG("essos: essGamepadNotifyConnected gp %p", gp );
      if ( ctx->gamepadConnectionListener && ctx->gamepadConnectionListener->connected )
      {
         ctx->gamepadConnectionListener->connected( ctx->gamepadConnectionListenerUserData, gp );
      }
   }
}

static void essGamepadNotifyDisconnected( EssCtx *ctx, EssGamepad *gp )
{
   if ( ctx )
   {
      DEBUG("essos: essGamepadNotifyDisconnected gp %p", gp );
      if ( ctx->gamepadConnectionListener && ctx->gamepadConnectionListener->disconnected )
      {
         ctx->gamepadConnectionListener->disconnected( ctx->gamepadConnectionListenerUserData, gp );
      }
   }
}
#endif

