/*
 * Broadcom oops capture and display (boops)
 *
 * Copyright (C) 2017 Broadcom
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */


#include <linux/proc_fs.h>
#include <linux/kmsg_dump.h>
#include <linux/platform_device.h>
#include <linux/vmalloc.h>
#include <linux/uaccess.h>
#include <linux/kthread.h>
#include <linux/spinlock.h>
#include <linux/bug.h>
#include <linux/delay.h>
#include <linux/seq_file.h>
#include <linux/version.h>

#include "boops_k.h"
#include "boops_k_proc.h"
#include "boops_k_common.h"

struct boops_proc_t {
    struct boops_t *boops;
    struct proc_dir_entry *boops_proc_dir;      /* Proc directroy containing other proc entries */
    struct proc_dir_entry *uuid_proc;           /* Proc entry to fetch the current boot UUID in userspace */
    struct proc_dir_entry *force_lock_proc;     /* Proc entry to trigger a soft/hard lockup */
    struct proc_dir_entry *capture_test_proc;   /* Proc entry to trigger a soft/hard lockup */
    struct proc_dir_entry *oops_log_line_proc;    /* Proc entry for passing previous oops prints back to kernel space */

    uint8_t* test_buff;
    uint8_t* test_buff_pos;
    uint32_t test_buff_size;
    uint32_t test_buff_remaining;
    uint8_t  log_line[BOOPS_PROC_MAX_LOG_LINE_SIZE+1];
};

static struct boops_proc_t *g_boops_proc = NULL;

#include <linux/module.h>    /* Specifically, a module */


///////////////////////////////////////////////////////////////////////////////////
// Capture test
//////////////////////
static ssize_t boops_proc_oops_log_line(struct file *filp,const char *buf,size_t size,loff_t *offp) {
    uint32_t data_size;
    int ret = 0;

    data_size = size < BOOPS_PROC_MAX_LOG_LINE_SIZE ? size : BOOPS_PROC_MAX_LOG_LINE_SIZE;
    ret=copy_from_user(g_boops_proc->log_line, buf, data_size);
    if (ret)
        return -EFAULT;
    g_boops_proc->log_line[data_size]='\0';

    boops_oops("%s", g_boops_proc->log_line);
    return data_size;
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(5,15,0)
static struct proc_ops boops_proc_oops_log_line_fops = {
    .proc_write   = boops_proc_oops_log_line,
};
#else
static struct file_operations boops_proc_oops_log_line_fops = {
    .owner   = THIS_MODULE,
    .write   = boops_proc_oops_log_line,
};
#endif

///////////////////////////////////////////////////////////////////////////////////
// Capture test
//////////////////////
static ssize_t boops_proc_capture_test_load_data(struct file *filp,const char *buf,size_t size,loff_t *offp) {
    uint32_t data_size;
    int ret = 0;

    if (!g_boops_proc->test_buff || g_boops_proc->test_buff_remaining == 0) {
        return 0;
    }

    data_size = size < g_boops_proc->test_buff_remaining ? size : g_boops_proc->test_buff_remaining;
    ret=copy_from_user(g_boops_proc->test_buff_pos, buf, data_size);
    if (ret)
        return -EFAULT;
    g_boops_proc->test_buff_pos+=data_size;
    g_boops_proc->test_buff_remaining-=data_size;

    return data_size;
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(5,15,0)
static bool boops_proc_capture_test_read_data(struct kmsg_dump_iter *iter, bool syslog, char * buf, size_t size, size_t * len) {
#else
static bool boops_proc_capture_test_read_data(struct kmsg_dumper * dumper, bool syslog, char * buf, size_t size, size_t * len) {
#endif
    uint32_t data_size;
    if (!g_boops_proc->test_buff || g_boops_proc->test_buff_remaining == 0) {
        return 0;
    }

    data_size = size < g_boops_proc->test_buff_remaining ? size : g_boops_proc->test_buff_remaining;
    g_boops_proc->test_buff_remaining-=data_size;
    memcpy(buf, &g_boops_proc->test_buff[g_boops_proc->test_buff_remaining], data_size);
    *len = data_size;
    boops_info("Reading %u bytes\n", data_size);
    return (g_boops_proc->test_buff_remaining>0);
}

static int boops_proc_capture_test_execute(void) {
    /* We've finished writng to the buffer, convert values for reading */
    g_boops_proc->test_buff_size = g_boops_proc->test_buff_size - g_boops_proc->test_buff_remaining;
    g_boops_proc->test_buff_remaining = g_boops_proc->test_buff_size;
    g_boops_proc->boops->simulation_mode = true;

    boops_info("Starting oops dump test with 'dmesg' buffer size %u\n", g_boops_proc->test_buff_size);
    boops_capture_oops(g_boops_proc->boops, boops_proc_capture_test_read_data, g_boops_proc, KMSG_DUMP_OOPS);
    return 0;
}

static void boops_proc_capture_test_destroy(void) {
    if (g_boops_proc->test_buff) {
        vfree(g_boops_proc->test_buff);
        g_boops_proc->test_buff_size = 0;
        g_boops_proc->test_buff_remaining = 0;
        g_boops_proc->test_buff_pos = NULL;
        g_boops_proc->test_buff = NULL;
    }
}

static int boops_proc_capture_test_init(void) {
    boops_proc_capture_test_destroy();

    g_boops_proc->test_buff_size = g_boops_proc->boops->block_buffer_size;
    g_boops_proc->test_buff = vmalloc(g_boops_proc->test_buff_size);
    if (!g_boops_proc->test_buff) {
        boops_err("Failed to allocate proc storage\n");
        g_boops_proc->test_buff_size = 0;
        return 0;
    }
    g_boops_proc->test_buff_remaining = g_boops_proc->test_buff_size;
    g_boops_proc->test_buff_pos = g_boops_proc->test_buff;
    return 0;
}

static int boops_proc_capture_test_open(struct inode *inode, struct file *file) {
    try_module_get(THIS_MODULE);
    boops_proc_capture_test_init();
    return 0;
}

static int boops_proc_capture_test_close(struct inode *inode, struct file *file) {
    boops_proc_capture_test_execute();
    boops_proc_capture_test_destroy();
    module_put(THIS_MODULE);
    return 0;
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(5,15,0)
static struct proc_ops boops_proc_capture_test_fops = {
    .proc_write   = boops_proc_capture_test_load_data,
    .proc_open    = boops_proc_capture_test_open,
    .proc_release = boops_proc_capture_test_close
};
#else
static struct file_operations boops_proc_capture_test_fops = {
    .owner   = THIS_MODULE,
    .write   = boops_proc_capture_test_load_data,
    .open    = boops_proc_capture_test_open,
    .release = boops_proc_capture_test_close
};
#endif

///////////////////////////////////////////////////////////////////////////////////
// Hard/soft lock
//////////////////////
#define BOOPS_HARD_LOCK "HARD"
#define BOOPS_SOFT_LOCK "SOFT"
#define BOOPS_SEGMENTATION_FAULT "SEG"
#define BOOPS_LOCK_HELP_STRING "Echo one of:\n"\
                                "    '" BOOPS_HARD_LOCK "'\n"\
                                "    '" BOOPS_SOFT_LOCK "'\n"\
                                "    '" BOOPS_SEGMENTATION_FAULT "\n"\
                                "into this proc to execute the oops test\n"

static DEFINE_SPINLOCK(g_testlock);
static struct task_struct *lockup_thread[8];

static int hard_lockup_thread(void *data) {
    bool hard_lock=data;
    msleep (1*1000);
    if (hard_lock) {
        printk("HARD LOCK: Trying spin_lock_irq ... on CPU [%d]\n", smp_processor_id());
        spin_lock_irq(&g_testlock);
    }
    else {
        printk("SOFT LOCK: Trying spin_lock ... on CPU [%d]\n", smp_processor_id());
        spin_lock(&g_testlock);
    }

    if (num_online_cpus() == 1) {
        printk("Only one CPU. Forcing a busy loop...\n");
        while(true) {}
    }

    /* We should not reach here */
    BUG();

    return 0;
}

static void boops_proc_do_lock_test(bool hard_lock, bool seg_fault) {
    int i;
    char thread_name[8];

    g_boops_proc->boops->simulation_mode = true;
    boops_err("\n\n\n\n\n******* boops forcing a %s lock on current boot [%pUB] as requested by the user *******\n\n\n\n\n", hard_lock ? "HARD" : "SOFT", g_boops_current_boot_uuid);

    /* Aquire spin_lock here, so that the kthreads should busy-wait */
    spin_lock(&g_testlock);

    /* Create kthreads per CPU */
    for (i=0; i<num_online_cpus(); i++) {
        snprintf(thread_name, 8, "slt%d", i);
        //lockup_thread[i] = kthread_create_on_cpu(hard_lockup_thread, NULL, i, thread_name);
        lockup_thread[i] = kthread_create(hard_lockup_thread, (void*)hard_lock, thread_name);
        if (lockup_thread[i]) {
            kthread_bind(lockup_thread[i], i);
        }
    }

    /* Start kthreads */
    for (i=0; i<num_online_cpus(); i++) {
        if (lockup_thread[i]) {
            wake_up_process(lockup_thread[i]);
        }
    }

    if (seg_fault) {
        char*seg_fault_ptr=NULL;
        boops_err("\n\n\n\n\n******* Forcing SEG FAULT *******\n\n\n\n\n");
        *seg_fault_ptr=1;
    }
}

ssize_t boops_proc_force_lock_write(struct file *filp,const char *buf,size_t count,loff_t *offp) {
    char msg[8];
    unsigned long actual_len = count < (sizeof(msg)) ? count : (sizeof(msg));
    if (copy_from_user(msg, buf, actual_len))
        return -EFAULT;

    if (strncmp(msg, BOOPS_HARD_LOCK, strlen(BOOPS_HARD_LOCK)) == 0) {
        boops_proc_do_lock_test(true, false);
    }
    else if (strncmp(msg, BOOPS_SOFT_LOCK, strlen(BOOPS_SOFT_LOCK)) == 0) {
        boops_proc_do_lock_test(false, false);
    }
    else if (strncmp(msg, BOOPS_SEGMENTATION_FAULT, strlen(BOOPS_SEGMENTATION_FAULT)) == 0) {
        boops_proc_do_lock_test(true, true);
    }
    else {
        boops_info(BOOPS_LOCK_HELP_STRING);
    }

    return count;
}

static ssize_t boops_proc_force_lock_read(struct file *file, char __user *buffer, size_t count, loff_t *offset) {
    boops_info(BOOPS_LOCK_HELP_STRING);
    return 0;
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(5,15,0)
static struct proc_ops boops_proc_force_lock_fops = {
    .proc_read = boops_proc_force_lock_read,
    .proc_write = boops_proc_force_lock_write,
};
#else
static struct file_operations boops_proc_force_lock_fops = {
    .owner = THIS_MODULE,
    .read = boops_proc_force_lock_read,
    .write = boops_proc_force_lock_write,
};
#endif
///////////////////////////////////////////////////////////////////////////////////
// UUID
//////////////////////
static int boops_proc_uuid_show(struct seq_file *m, void *v) {
    seq_printf(m, "%pUB\n", g_boops_current_boot_uuid);
    return 0;
}

static int boops_proc_uuid_open(struct inode *inode, struct file *file) {
    return single_open(file, boops_proc_uuid_show, NULL);
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(5,15,0)
static const struct proc_ops boops_proc_uuid_fops = {
    .proc_open       = boops_proc_uuid_open,
    .proc_read       = seq_read,
    .proc_lseek      = seq_lseek,
    .proc_release    = single_release,
};
#else
static const struct file_operations boops_proc_uuid_fops = {
    .owner = THIS_MODULE,
    .open       = boops_proc_uuid_open,
    .read       = seq_read,
    .llseek     = seq_lseek,
    .release    = single_release,
};
#endif

///////////////////////////////////////////////////////////////////////////////////
// Init
//////////////////////
struct boops_proc_t* boops_proc_init(struct boops_t *boops) {
    struct boops_proc_t* boops_proc= NULL;

    boops_proc = vmalloc(sizeof(*boops_proc));
    if (!boops_proc) {
        boops_err("Failed to allocate proc storage\n");
        return NULL;
    }
    memset(boops_proc, 0, sizeof(*boops_proc));
    boops_proc->boops = boops;

    boops_proc->boops_proc_dir = proc_mkdir(BOOPS_PROC_DIR, NULL);
    if (!boops_proc->boops_proc_dir)
        goto err_out;

    boops_proc->uuid_proc = proc_create(BOOPS_PROC_UUID,0, boops_proc->boops_proc_dir, &boops_proc_uuid_fops);
    if (!boops_proc->uuid_proc)
        goto err_out;

    boops_proc->force_lock_proc = proc_create(BOOPS_PROC_FORCE,0, boops_proc->boops_proc_dir, &boops_proc_force_lock_fops);
    if (!boops_proc->force_lock_proc)
        goto err_out;

    boops_proc->capture_test_proc = proc_create(BOOPS_PROC_TEST,0, boops_proc->boops_proc_dir, &boops_proc_capture_test_fops);
    if (!boops_proc->capture_test_proc)
        goto err_out;

    boops_proc->oops_log_line_proc = proc_create(BOOPS_PROC_SEND_LOG,0, boops_proc->boops_proc_dir, &boops_proc_oops_log_line_fops);
    if (!boops_proc->oops_log_line_proc)
        goto err_out;

    boops_info("Initialized boops proc interface\n");
    g_boops_proc=boops_proc;
    return g_boops_proc;

err_out:
    boops_err("Failed creating the boops procs. This feature will not be available\n");
    boops_proc_remove(boops_proc);
    return NULL;
}

void boops_proc_remove(struct boops_proc_t* boops_proc) {
    if (boops_proc) {
        boops_info("Destroying boops proc interface\n");
        if (boops_proc->oops_log_line_proc)
            proc_remove(boops_proc->oops_log_line_proc);

        if (boops_proc->capture_test_proc)
            proc_remove(boops_proc->capture_test_proc);

        if (boops_proc->force_lock_proc)
            proc_remove(boops_proc->force_lock_proc);

        if (boops_proc->uuid_proc)
            proc_remove(boops_proc->uuid_proc);

        if (boops_proc->boops_proc_dir)
            proc_remove(boops_proc->boops_proc_dir);

        g_boops_proc = NULL;
        vfree(boops_proc);
    }
}


