/*
 * MCDescriptorData.cpp
 *
 *  Created on: Jul 1, 2014
 *      Author: ktaubing
 */

#include "MCDescriptorData.h"

#include <arpa/inet.h>
#include <sstream>

/*
 * Mini Carousel Syntax
 *
 * Syntax                     No. of Bits          Value
 * mini_carousel_section ()
 * {
 *    table_id                      8              0xC9 (201)
 *    section_syntax_indicator      1              '0'
 *    private_indicator             1              '0'
 *    reserved                      2              '00'
 *    section_length                12             uimsbf
 *    transport_stream_id[2]        16             for MC 2.1 = Service Group ID; for MC 3.1.1 = 0
 *    reserved                      2              '11'
 *    version_number                5              uimsbf
 *    current_next_indicator        1              '1'
 *    section_number                8              uimsbf
 *    last_section_number           8              uimsbf
 *    protocol_version              8              0x00
 *    num_channels_in_section       8              uimsbf
 *    for(i=0; i<num_channels_in_section;i++)
 *    {
 *      short_name                  7*16           uimsbf
 *      reserved                    4              '1111'
 *      major_channel_number        10             '1111111111'
 *      minor_channel_number        10             '1111111111'
 *      modulation_mode             8              uimsbf
 *      carrier_frequency           32             uimsbf
 *      channel_TSID                16             uimsbf
 *      program_number              16             uimsbf
 *      ETM_location                2              '00'
 *      access_controlled           1              '0'
 *      hidden                      1              '0'
 *      path_select                 1              '0'
 *      out_of_band                 1              '0'
 *      hide_guide                  1              '0'
 *      reserved                    3              '111'
 *      service_type                6              uimsbf
 *      source_id                   16             uimsbf
 *      reserved                    6              '111111'
 *      ch_descriptors_length       10             '0000000000'
 *      for(i=0;i<N;i++)
 *      {
 *         ch_descriptor()
 *         {
 *            ch_descriptor_tag     8
 *            ch_descriptor_length  8
 *            ch_descriptor_data    *
 *         }
 *      }
 *    }
 *    reserved                      6              '111111'
 *    mc_descriptors_length         10             uimsbf
 *    for(j=0; j<N;j++)
 *    {
 *       mc _descriptor()
 *       {
 *          mc _descriptor_tag      8              usmsbf
 *          mc _descriptor_length   8              usmsbf
 *          mc _descriptor_data     *              usmsbf
 *       }
 *    }
 *    crc_32                        32             uimsbf
 * }
 */


using namespace sdv;


MCDescriptorData::MCDescriptorData(uint8_t* sectionData) {
	initIpData();
	parseDescriptors(sectionData);
}

MCDescriptorData::MCDescriptorData() {
	initIpData();
}

void MCDescriptorData::initIpData() {
	// Zero all IP related data since we don't know what descriptors will be available
	for (int n = 0; n < IPV4_BINARY_LENGTH; ++n) {
		ip4BinaryAddr[n] = 0;
	}
	for (int n = 0; n < IPV6_BINARY_LENGTH; ++n) {
		ip6BinaryAddr[n] = 0;
	}
	ip6Port = 0;
	ip4Port = 0;
}

void MCDescriptorData::parseDescriptors(uint8_t* sectionData) {
	uint8_t* fieldPtr;

	 /* Prior to MC3.1.1, transport ID field contained the service group ID. For backwards compatibility,
	    we will temporary save the service group ID from the transport stream ID field and only use it if
	    we don't find a service group ID descriptor later on in the parsing */
	fieldPtr = sectionData + TRANSPORT_STREAM_ID_OFFSET;
	serviceGroupId = ntohs( *((uint16_t*)fieldPtr) );

	 /* Get the number of channels that we will need to skip in our data block
	    since they were parsed out before we got the data. Then skip over the number of channels */
	fieldPtr = sectionData + NUM_CHANNELS_IN_SECTION_OFFSET;
	uint8_t numChannels = *fieldPtr;
	fieldPtr += sizeof(numChannels);

	 /* Increment field pointer past the channel block */
	fieldPtr += numChannels * CHANNEL_BLOCK_SIZE;

	 /* Get total length of MC descriptors. Upper bits are reserved and therefore need to be masked */
	uint16_t descriptorsLength = (ntohs( *((uint16_t*)fieldPtr) )) & MC_DESCRIPTORS_LENGTH_BIT_MASK;
	fieldPtr += sizeof(descriptorsLength);

	/* While descriptor data is available, get a tag and parse it */
	while (descriptorsLength > 0) {
		uint8_t descriptorTag = *fieldPtr;

		switch (descriptorTag) {
			case IP_RESOURCE_DESCRIPTOR_TAG:
				parseIp4Descriptor((MCMIS_IP4_RESOURCE_DESCRIPTOR *)fieldPtr);
				fieldPtr += sizeof(struct mcmis_ip4_resource_descriptor);
				descriptorsLength -= sizeof(struct mcmis_ip4_resource_descriptor);
				break;

			case CONFIG_PARAMS_DESCRIPTOR_TAG:
				parseConfigParamDescriptor((MCMIS_CONFIG_PARAMS_DESCRIPTOR *)fieldPtr);
				fieldPtr += sizeof(struct mcmis_config_params_descriptor);
				descriptorsLength -= sizeof(struct mcmis_config_params_descriptor);
				break;

			case SERVICE_GROUP_ID_DESCRIPTOR_TAG:
				parseServiceGroupIdDescriptor((MCMIS_SERVICE_GROUP_ID_DESCRIPTOR *)fieldPtr);
				fieldPtr += sizeof(struct mcmis_service_group_id_descriptor);
				descriptorsLength -= sizeof(struct mcmis_service_group_id_descriptor);
				break;

			case IPV6_RESOURCE_DESCRIPTOR_TAG:
				parseIp6Descriptor((MCMIS_IP6_RESOURCE_DESCRIPTOR *)fieldPtr);
				fieldPtr += sizeof(struct mcmis_ip6_resource_descriptor);
				descriptorsLength -= sizeof(struct mcmis_ip6_resource_descriptor);
				break;

			default:
				/* In case we get a future unsupported MCMIS descriptor, skip over it */
				fieldPtr += 1; 									// skip tag field
				uint8_t descLength = *fieldPtr;
				fieldPtr += sizeof(descLength) + descLength;	// skip length field and descriptor
				break;
		}
	}
}

bool MCDescriptorData::isEqual(MCDescriptorData* newData) {

	if (isServerChanged(newData)) {
		return false;
	}
	else if (isServiceGroupChanged(newData)) {
		return false;
	}
	else if (newData->lastUserActivityReportInterval != lastUserActivityReportInterval) {
		return false;
	}
	else if (newData->miniCarouselReadInterval != miniCarouselReadInterval) {
		return false;
	}
	else if (newData->messageResponseTimeout != messageResponseTimeout) {
		return false;
	}
	else if (newData->messageRequestMaxRetries != messageRequestMaxRetries) {
		return false;
	}
	else if (newData->messageRequestRetryInterval != messageRequestRetryInterval) {
		return false;
	}
	else if (newData->bandwidthReclaimUITimeout != bandwidthReclaimUITimeout) {
		return false;
	}
	else {
		return true;
	}
}

bool MCDescriptorData::isServerChanged(MCDescriptorData* newData) {

	for (int n = 0; n < IPV4_BINARY_LENGTH; ++n) {
		if (newData->ip4BinaryAddr[n] != ip4BinaryAddr[n]) {
			return true;
		}
	}
	for (int n = 0; n < IPV6_BINARY_LENGTH; ++n) {
		if (newData->ip6BinaryAddr[n] != ip6BinaryAddr[n]) {
			return true;
		}
	}
	if (newData->ip4Port != ip4Port) {
		return true;
	}
	else if (newData->ip6Port != ip6Port) {
		return true;
	}
	else {
		return false;
	}
}

bool MCDescriptorData::isServiceGroupChanged(MCDescriptorData* newData) {
	if (newData->serviceGroupId != serviceGroupId) {
		return true;
	}
	else {
		return false;
	}
}

bool MCDescriptorData::isValidIp6Descriptor() {
	for (int n = 0; n < IPV6_BINARY_LENGTH; n++) {
		if (ip6BinaryAddr[n] != 0) {
			return true;
		}
	}
	return false;
}

void MCDescriptorData::parseIp4Descriptor(MCMIS_IP4_RESOURCE_DESCRIPTOR* descriptor) {
	ip4Port = ntohs(descriptor->ipPort);
	memcpy(ip4BinaryAddr, descriptor->ipAddress, IPV4_BINARY_LENGTH);
	return;
}

void MCDescriptorData::parseConfigParamDescriptor(MCMIS_CONFIG_PARAMS_DESCRIPTOR* descriptor) {
	lastUserActivityReportInterval = ntohl(descriptor->luaReportingInterval);
	miniCarouselReadInterval = ntohl(descriptor->miniCarouselReadInterval);
	messageResponseTimeout = ntohl(descriptor->messageResponseTimeout);
	messageRequestMaxRetries = ntohl(descriptor->messageRequestMaxRetriesCount);
	messageRequestRetryInterval = ntohl(descriptor->messageRequestRetryInterval);
	bandwidthReclaimUITimeout = ntohl(descriptor->bandwidthReclaimUITimeout);
	return;
}

void MCDescriptorData::parseServiceGroupIdDescriptor(MCMIS_SERVICE_GROUP_ID_DESCRIPTOR* descriptor) {
	serviceGroupId = ntohl(descriptor->serviceGroupId);
}

void MCDescriptorData::parseIp6Descriptor(MCMIS_IP6_RESOURCE_DESCRIPTOR* descriptor) {
	ip6Port = ntohs(descriptor->ipPort);
	memcpy(ip6BinaryAddr, descriptor->ipAddress, IPV6_BINARY_LENGTH);
	return;
}

std::string MCDescriptorData::toDebugString() {
	std::stringstream stm;

	stm << "MC Descriptor Data:\n";

	char ipString[64];
	struct in_addr addr;
	memcpy(&addr, ip4BinaryAddr, sizeof(struct in_addr));
	inet_ntop(AF_INET, &addr, ipString, INET_ADDRSTRLEN);
	stm << "     server IP4= " << ipString << "  port= " << ip4Port << '\n';

	struct in6_addr addr6;
	memcpy(&addr6, ip6BinaryAddr, sizeof(struct in6_addr));
	inet_ntop(AF_INET6, &addr6, ipString, INET6_ADDRSTRLEN);
	stm << "     server IP6= " << ipString << "  port= " << ip6Port << '\n';

	stm << "     serviceGroupId= " << serviceGroupId << '\n';
	stm << "     lastUserActivityReportInterval= " << lastUserActivityReportInterval << '\n';
	stm << "     miniCarouselReadInterval= " << miniCarouselReadInterval << '\n';
	stm << "     messageResponseTimeout= " << messageResponseTimeout << '\n';
	stm << "     messageRequestMaxRetries= " << messageRequestMaxRetries <<'\n';
	stm << "     messageRequestRetryInterval= " << messageRequestRetryInterval << '\n';
	stm << "     bandwidthReclaimUITimeout= " << bandwidthReclaimUITimeout << '\n';

	return stm.str();
}


