/*
* ============================================================================
* RDK MANAGEMENT, LLC CONFIDENTIAL AND PROPRIETARY
* ============================================================================
* This file (and its contents) are the intellectual property of RDK Management, LLC.
* It may not be used, copied, distributed or otherwise  disclosed in whole or in
* part without the express written permission of RDK Management, LLC.
* ============================================================================
* Copyright (c) 2014 RDK Management, LLC. All rights reserved.
* ============================================================================
*/
#include "MessageService.h"
#include "InitConfirm.h"
#include "ProgramSelectConfirm.h"
#include "QueryRequest.h"
#include "EventIndication.h"
#include "ProgramSelectIndication.h"
#include "rdk_debug.h"

#include <netinet/in.h>


using namespace sdv;

#define LOG(...) RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", __VA_ARGS__)
#define LOG_ERR(...) RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.SDVAGENT", __VA_ARGS__);


MessageService::MessageService(ConfigurationParser * _configParser, SDVEventQueue * queue){
    this->_configParser = _configParser;
    this->_queue = queue;
}

SDVService::SERVICE_RESULT MessageService::start(){
    LOG("[MessageService] start()\n");

    this->_udpService = createUdpService(_configParser);
    return SDVService::SUCCESS;
}

SDVService::SERVICE_RESULT MessageService::stop(){
    LOG("[MessageService] stop()\n");
    delete _udpService;
    return SDVService::SUCCESS;
}

SDVService::SERVICE_RESULT MessageService::restart(){
    LOG("[MessageService] restart()\n");
    _queue->push(this, handleRestartEvent, NULL, 0);
    return SDVService::SUCCESS;
}

void MessageService::handleRestartEvent(SDVEventQueue::TASK_ID_t, void* ptrInstance, void* data){
    LOG("[MessageService] handleRestartEvent()\n");
    MessageService* obj = (MessageService *)ptrInstance;

	// Create a new UDP Service with current configuration parameters
	delete obj->_udpService;
	obj->_udpService = obj->createUdpService(obj->_configParser);

    LOG("[MessageService] handleRestartEvent() completed\n");
}

void MessageService::sendMessage(SDVMessage * msg, SDV_MESSAGE_CALLBACK_t callback, void* ptrInst, uint32_t delay){
    LOG("[MessageService] sendMessage() messageId=0x%x\n", msg->getMessageId());

    CALLBACK_DATA_t * handler = new CALLBACK_DATA_t();
    handler->ptrInst= ptrInst;
    handler->callback = callback;
    handler->outboundMsg = msg;

    if(callback != NULL){
    	TIMEOUT_DATA_t * timeoutCheck = new TIMEOUT_DATA_t();
        timeoutCheck->remaining_retries = _configParser->getMessageRequestMaxRetries();
        timeoutCheck->transactionId = msg->getTransactionId();
        handler->timeoutCheck = timeoutCheck;	// indicate response message expected within timeout
    }
    else {
        handler->timeoutCheck = NULL;			// indicate response message not expected
    }
    _queue->push(this, handleSendMessageEvent, handler, delay);
}

void MessageService::setMessageHandler(SDVMessage::MessageId messageId, SDV_MESSAGE_CALLBACK_t callback, void* ptrInst) {
	UNSOLICTED_MESSAGE_HANDLER* handler;
    MESSAGE_HANDLERS_t::iterator iter = _messageHandlers.find(messageId);

    // If handler already exists, either delete or replace it
    if (iter != _messageHandlers.end()) {
    	handler = iter->second;
    	if (callback == NULL) {
            LOG("[MessageService] setMessageHandler() delete handler for messageId 0x%x\n", messageId);
    		_messageHandlers.erase(messageId);
    		delete handler;
    	}
    	else {
            LOG("[MessageService] setMessageHandler() replace handler for messageId 0x%x\n", messageId);
        	handler->ptrInst = ptrInst;
            handler->callback = callback;
    	}
    }
    // Else create new handler
    else {
        LOG("[MessageService] setMessageHandler() create new handler for messageId 0x%x\n", messageId);
        handler = new UNSOLICTED_MESSAGE_HANDLER();
    	handler->ptrInst = ptrInst;
        handler->callback = callback;
        _messageHandlers[messageId] = handler;
    }
}

void MessageService::packetRecieved(UdpService::PACKET_BUFF * buff, void* callbackInstance){
	if (rdk_dbg_enabled("LOG.RDK.SDVAGENT", RDK_LOG_DEBUG)) {
	    LOG("[MessageService] packetRecieved! \n");
		hexDump(buff->data, buff->data_length);
	}

    SDVMessage * msg =  ((MessageService*)callbackInstance)->parseMessage(buff->data);
    ((MessageService*)callbackInstance)->_udpService->freePacket(buff);
    if(msg != NULL){
        ((MessageService*)callbackInstance)->_queue->push(callbackInstance, handleIncomingMessageEvent, msg, 0);
    }
    fflush(stdout);
}

void MessageService::handleSendMessageEvent(SDVEventQueue::TASK_ID_t, void* ptrInstance, void * data) {
    CALLBACK_DATA_t * callbackData = (CALLBACK_DATA_t *) data;
    SDVMessage * outboundMessage = callbackData->outboundMsg;

    UdpService::PACKET_BUFF* buffer = ((MessageService*) ptrInstance)->_udpService->getEmptyPacket();
    buffer->data_length = outboundMessage->serialize(buffer->data, buffer->max_length);

    LOG("[MessageService] handleSendMessageEvent() messageId=0x%x transactionId=%d\n",
            outboundMessage->getMessageId(), outboundMessage->getTransactionId());
	if (rdk_dbg_enabled("LOG.RDK.SDVAGENT", RDK_LOG_DEBUG)) {
		hexDump(buffer->data, buffer->data_length);
	}
    ((MessageService*) ptrInstance)->_udpService->sendPacket(buffer);

    //Response message is required
    if(callbackData->timeoutCheck != NULL){
        LOG("[MessageService] handleSendMessageEvent() tracking transaction %d for response \n", callbackData->outboundMsg->getTransactionId());
        ((MessageService*) ptrInstance)->_responseMessageHandlers[callbackData->outboundMsg->getTransactionId()] = callbackData;
        ((MessageService*) ptrInstance)->_queue->push(ptrInstance,
                handleResponseTimeoutCheck, callbackData->timeoutCheck,
                ((MessageService*) ptrInstance)->_configParser->getMessageResponseTimeout());
    }else{
        delete callbackData->outboundMsg;
        delete callbackData;
    }
}

void MessageService::handleIncomingMessageEvent(SDVEventQueue::TASK_ID_t, void* ptrInstance, void * data){
    SDVMessage * incomingMsg = (SDVMessage*)data;
    SDVMessage::TRANACTION_ID_t transactionId = incomingMsg->getTransactionId();
    RESPONSE_HANDLERS_t::iterator rhIter = ((MessageService*)ptrInstance)->_responseMessageHandlers.find(transactionId);

    // If message has no response handler, check if it has an unsolicited message handler
    if (rhIter == ((MessageService*)ptrInstance)->_responseMessageHandlers.end()) {
        SDVMessage::MessageId messageId = incomingMsg->getMessageId();
        MESSAGE_HANDLERS_t::iterator mhIter = ((MessageService*)ptrInstance)->_messageHandlers.find(messageId);

        if (mhIter != ((MessageService*)ptrInstance)->_messageHandlers.end()) {
            LOG("[MessageService] handleIncomingMessageEvent() invoke unsolicited message handler; messageId=0x%x transactionId=%d\n", messageId, transactionId);
            UNSOLICTED_MESSAGE_HANDLER* handler = mhIter->second;
            handler->callback(handler->ptrInst, SUCCESS, incomingMsg);
        }
        else {
            LOG_ERR("[MessageService] handleIncomingMessageEvent() handler not found for unsolicited message; messageId=0x%x transactionId=%d\n", messageId, transactionId);
        }
    }
    // Else invoke response handler
    else {
        LOG("[MessageService] handleIncomingMessageEvent() invoke response handler; transactionId=%d\n", transactionId);
        CALLBACK_DATA_t * handler = rhIter->second;
        handler->callback(handler->ptrInst, SUCCESS, incomingMsg);

        // Response handler only exists during the life of the message transaction, therefore delete it
        ((MessageService*)ptrInstance)->_responseMessageHandlers.erase(transactionId);
        delete handler->outboundMsg;
        delete handler;
    }
    delete incomingMsg;
}

void MessageService::handleResponseTimeoutCheck(SDVEventQueue::TASK_ID_t id, void* ptrInstance, void * data){
    TIMEOUT_DATA_t * timeout = (TIMEOUT_DATA_t *) data;
    LOG("[MessageService] handleResponseTimeoutCheck() transactionId=%d\n", timeout->transactionId);
    RESPONSE_HANDLERS_t::iterator iter = ((MessageService*)ptrInstance)->_responseMessageHandlers.find(timeout->transactionId);
    if(iter == ((MessageService*)ptrInstance)->_responseMessageHandlers.end()){
        LOG("[MessageService] handleResponseTimeoutCheck() Response received. Canceling timeout check\n");
        delete timeout;
    }else{
        CALLBACK_DATA_t * callback = iter->second;
        if(timeout->remaining_retries--){
            LOG_ERR("[MessageService] handleResponseTimeoutCheck() resend message; remaining retries=%d\n", timeout->remaining_retries);
            handleSendMessageEvent(id, ptrInstance, (void *)callback);
        }else{
            LOG_ERR("[MessageService] handleResponseTimeoutCheck() response failed, invoke callback\n");
            callback->callback(callback->ptrInst, TIMEOUT, NULL);
            ((MessageService*)ptrInstance)->_responseMessageHandlers.erase(timeout->transactionId);
            delete timeout;
            delete callback->outboundMsg;
            delete callback;
        }
    }
}

sdv::UdpService * MessageService::createUdpService(ConfigurationParser* config){
	return new UdpService(&packetRecieved, this, config->getServerIpString().c_str(), config->getServerIpPort());
}

sdv::SDVMessage * MessageService::parseMessage(uint8_t * data){
    SDVMessage::MessageId id = SDVMessage::getMessageType(data);
    SDVMessage * msg = NULL;

    switch(id){
    case SDVMessage::InitConfirm:
        LOG("[MessageService] parseMessage() Message is InitConfirm\n");
        msg = new InitConfirm(data);
        break;
    case SDVMessage::ProgramSelectConfirm:
        LOG("[MessageService] parseMessage() Message is ProgramSelectConfirm\n");
        msg = new ProgramSelectConfirm(data);
        break;
    case SDVMessage::QueryRequest:
        LOG("[MessageService] parseMessage() Message is QueryRequest\n");
        msg = new QueryRequest(data);
        break;
    case SDVMessage::EventIndication:
        LOG("[MessageService] parseMessage() Message is EventIndication (display barker)\n");
        msg = new EventIndication(data);
        break;
    case SDVMessage::ProgramSelectIndication:
        LOG("[MessageService] parseMessage() Message is ProgramSelectIndication\n");
        msg = new ProgramSelectIndication(data);
        break;
    default:
        LOG_ERR("[MessageService] parseMessage() Unknown message - 0x%x\n", id);
        break;
    };
    return msg;
}

void MessageService::hexDump(unsigned char * data, size_t len){
    int i;
    unsigned char buff[17];
    unsigned char *pc = (unsigned char*)data;

    printf ("%s:\n", " Packet:");
    for (i = 0; i < len; i++) {
        if ((i % 16) == 0) {
            if (i != 0){
                printf ("  %s\n", buff);
            }
            printf ("  %04x ", i);
        }
        printf (" %02x", pc[i]);
        if ((pc[i] < 0x20) || (pc[i] > 0x7e))
            buff[i % 16] = '.';
        else
            buff[i % 16] = pc[i];
        buff[(i % 16) + 1] = '\0';
    }
    printf ("\n");
}
