/*
* ============================================================================
* RDK MANAGEMENT, LLC CONFIDENTIAL AND PROPRIETARY
* ============================================================================
* This file (and its contents) are the intellectual property of RDK Management, LLC.
* It may not be used, copied, distributed or otherwise  disclosed in whole or in
* part without the express written permission of RDK Management, LLC.
* ============================================================================
* Copyright (c) 2014 RDK Management, LLC. All rights reserved.
* ============================================================================
*/

#ifndef TRMMONITOR_H_
#define TRMMONITOR_H_

#include <pthread.h>

#include "SDVService.h"
#include "SDVEventQueue.h"
#include "trm/Messages.h"




namespace sdv {

class TRMMonitorService : public SDVService {


public:

	/**
	 * @enum TUNER_USE
	 * @brief Bit flags to indicate usage of a tuner
	 * Most of the values correspond to definitions in CCMIS specification.
	 */
	typedef enum tuner_use {
		BACKGROUND	=	0x00,
		RECORDING 	= 	0x01,
		PPV			=	0x04,
		MAIN_TV		= 	0x08,
		FREE		=	BACKGROUND	// CCMIS does not support "FREE" usage, therefore behave as BACKGROUND
	} TUNER_USE;
    
    typedef struct trm_detailed_tuner_state{
        std::string locator;
        std::string state;
    } TRM_DETAILED_TUNER_STATE;

    typedef std::vector<TRM_DETAILED_TUNER_STATE> TRM_TUNER_STATES;
	/**
	 * @typedef TUNER_USAGE_CHANGE_CALLBACK
	 * @brief Callback to invoke when a tuner usage has changed.
	 * @details Callback function will be invoked on eventQueue thread
	 *
	 * @param [in] callbackInstance - pointer to instance of callback
	 * @param [in] sourceId - sourceId corresponding to the tuner usage that has changed
	 * @param [in] newUsage - indicates the new tuner usage
	 */
	typedef void (TUNER_USAGE_CHANGE_CALLBACK)(void* callbackInstance, uint32_t sourceId, uint8_t newUsage);

	TRMMonitorService(SDVEventQueue* queue);
	~TRMMonitorService();

	SDVService::SERVICE_RESULT start();
	SDVService::SERVICE_RESULT stop();

	/**
	 * @fn setTunerUsageChangeCallback
	 * @brief Set a callback to notify when a tuner usage has changed
	 *
	 * @param callbackInstance
	 * @param callbackFunction
	 */
	void setTunerUsageChangeCallback(void* callbackInstance, TUNER_USAGE_CHANGE_CALLBACK* callbackFunction);

	/**
	 * @fn getTunerUsage
	 * @brief Get the tuner usage for the specified source ID
	 *
	 * @param sourceId
	 * @return TUNER_USE value.  FREE indicates source ID is not being tuned
	 */
	uint8_t getTunerUsage(uint32_t sourceId);

    static void notifyTunerStatesUpdate(SDVEventQueue::TASK_ID_t id, void* ptrObj, void* data);

    /**
     * @fn updateTunerUseage
     * @brief Callback function for receiving NotifyTunerStatesUpdate messages from the TRM JSON decoder.
     *
     * @param tunerStateMap A map if thedetailed tuner state information
     */
    void updateTunerUsage(TRM_TUNER_STATES *tunerStateMap);

    private:

	#define CLASSNAME 	"TRMMonitorService"

	#define TRM_IP_ADDR "127.0.0.1"
	#define TRM_PORT	9987
	#define TRM_SDVAGENT_CONNECTION_ID 0xFFFFFF03	//TODO: how to determine ID?
    #define OCAP_LOCATOR_PREFIX "ocap://0x"
    #define PPV_LOCATOR_PREFIX "ppv://0x"

     /**
     * @typedef TUNER_STATE
     * @brief Struct containing the state information of a tuner
     */
    typedef struct tuner_state {
        uint16_t sourceId;
        uint8_t state;

        bool operator== (const tuner_state &ts1) {
            return (sourceId == ts1.sourceId && state == ts1.state);
        }
    } TUNER_STATE;

    typedef std::vector<TUNER_STATE> TUNER_USAGE_MAP;

    SDVEventQueue* eventQueue;

	bool isRunning;			//!> indicates this service has started and is running
	bool isConnected;		//!> indicates this service is connected to TRM server
	int socketFd;			//!> socket file descriptor
	pthread_t thread_id;	//!> read socket thread identifier
	pthread_attr_t thread_attr;
	TUNER_USAGE_CHANGE_CALLBACK* usageChangeCallback;	//!< callback function to notify tuner usage change
	void* usageChangeCallbackInstance;             		//!< pointer to instance of callback class
    static TUNER_USAGE_MAP tunerUseMap;
    static pthread_mutex_t mapLock;

	static void* readServerThread(void* arg);
	void connectToServer();
    void getTRMVersion();
    void processBuffer(const char* buf, int len);
	bool postToServer(const char *payload, int payloadLength);
    static uint8_t getTunerUseFromTRMTunerState(const std::string &trmTunerState);
};

} /* namespace sdv */

#endif /* TRMMONITOR_H_ */
