/*
* ============================================================================
* RDK MANAGEMENT, LLC CONFIDENTIAL AND PROPRIETARY
* ============================================================================
* This file (and its contents) are the intellectual property of RDK Management, LLC.
* It may not be used, copied, distributed or otherwise  disclosed in whole or in
* part without the express written permission of RDK Management, LLC.
* ============================================================================
* Copyright (c) 2014 RDK Management, LLC. All rights reserved.
* ============================================================================
*/

#include "EventIndication.h"
#include "UserActivityService.h"
#include "QueryConfirm.h"
#include "EventResponse.h"
#include "ProgramSelectResponse.h"
#include "UserActivityReport.h"
#include "rdk_debug.h"
#include "IARMProxyService.h"
#include <sys/time.h>



#define LOG(...) RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", __VA_ARGS__)
#define LOG_ERR(...) RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.SDVAGENT", __VA_ARGS__);

#define MILLIS_PER_SEC 1000

using namespace sdv;


UserActivityService::UserActivityService(IARMProxyService* iarm_proxy, MessageService* msg_service, SDVEventQueue* event_queue,
		ConfigurationParser* configuration, SDVSessionHandler* session_handler, TRMMonitorService* trm_service) {
	LOG("[UserActivityService] create\n");

	iarmProxy = iarm_proxy;
	msgService = msg_service;
	eventQueue = event_queue;
	config = configuration;
	sessionHandler = session_handler;
	trmService = trm_service;
}


SDVService::SERVICE_RESULT UserActivityService::start(){
	LOG("[UserActivityService] start()\n");
	isRunning = true;
    msgService->setMessageHandler(SDVMessage::QueryRequest, queryRequest, this);
    msgService->setMessageHandler(SDVMessage::EventIndication, eventIndication, this);
    msgService->setMessageHandler(SDVMessage::ProgramSelectIndication, programSelectIndication, this);

    trmService->setTunerUsageChangeCallback(this, notifyTunerUsageChange);

    // Start delay for reporting user activity
	eventQueue->push(this, sendReport, NULL, config->getLastUserActivityReportInterval() * MILLIS_PER_SEC);

	return sdv::SDVService::SUCCESS;
}

SDVService::SERVICE_RESULT UserActivityService::stop(){
	LOG("[UserActivityService] stop()\n");
    msgService->setMessageHandler(SDVMessage::QueryRequest, NULL, NULL);
    msgService->setMessageHandler(SDVMessage::EventIndication, NULL, NULL);
    msgService->setMessageHandler(SDVMessage::ProgramSelectIndication, NULL, NULL);
    isRunning = false;
	return sdv::SDVService::SUCCESS;
}

UserActivityService::~UserActivityService(){
	LOG("[UserActivityService] destroy\n");
}

void UserActivityService::sendReport(SDVEventQueue::TASK_ID_t qTaskId, void* ptrInstance, void* data) {
	UserActivityService* instance = (UserActivityService*)ptrInstance;

	if (!instance->isRunning) {
		LOG_ERR("[UserActivityService] sendReport() service is not running\n");
		return;
	}

	// If any session still active, create a fake user activity report using the current time
	SDVSessionHandler::SDV_SESSION session;
	if (instance->sessionHandler->getAnyOpenSession(&session) == 0) {
		uint32_t currentTime = instance->getCurrentTimeInSecs();
		UserActivityReport* report = new UserActivityReport(
				instance->config->getStbMac(),
				(uint8_t)session.tunerId,
				session.sourceId,
				currentTime);

		LOG("[UserActivityService] sendReport() send UserActivityReport; tunerId=%d sourceId=%d lastActivity=%d\n", session.tunerId, session.sourceId, currentTime);
		instance->msgService->sendMessage(report, NULL, NULL, 0);
	}
	else {
		LOG("[UserActivityService] sendReport() no open session found and therefore no activity to report\n");
	}

    // Start delay for reporting next user activity
	instance->eventQueue->push(instance, instance->sendReport, NULL, instance->config->getLastUserActivityReportInterval() * MILLIS_PER_SEC);
}

void UserActivityService::queryRequest(void* obj, MessageService::REPONSE_STATUS_t status, SDVMessage* receivedMsg) {
	UserActivityService* instance = (UserActivityService*)obj;
	uint8_t tunerUse;
	uint32_t sourceId;
	SDVSessionHandler::SDV_SESSION session;

	// If session still open for tuner, get actual usage and source ID
	if (instance->sessionHandler->getOpenSession(receivedMsg->getTunerIndex(), &session) == 0) {
		sourceId = session.sourceId;
		tunerUse = instance->trmService->getTunerUsage(sourceId);
	}
	else {
		LOG_ERR("[UserActivityService] queryRequest() session not open for tunerId %d\n", receivedMsg->getTunerIndex());
		tunerUse = TRMMonitorService::BACKGROUND;
		sourceId = 0;
	}
	QueryConfirm* confirm = new QueryConfirm(
			instance->config->getStbMac(),
			receivedMsg->getTunerIndex(),
			sourceId,
			tunerUse,
			instance->config->getServiceGroupId(),
			instance->getCurrentTimeInSecs(),
			receivedMsg->getTransactionId());

	LOG("[UserActivityService] queryRequest() send QueryConfirm; tunerId=%d sourceId=%d tunerUse=0x%x\n", receivedMsg->getTunerIndex(), sourceId, tunerUse);
	instance->msgService->sendMessage(confirm, NULL, NULL, 0);
}

void UserActivityService::eventIndication(void* obj, MessageService::REPONSE_STATUS_t status, SDVMessage* receivedMsg) {
	UserActivityService* instance = (UserActivityService*)obj;

	// Create and send response for display barker event indication message
	EventResponse* response = new EventResponse(
			instance->config->getStbMac(),
			receivedMsg->getTunerIndex(),
			receivedMsg->getTransactionId());

	LOG("[UserActivityService] eventIndication() send EventResponse; tunerId=%d\n", receivedMsg->getTunerIndex());
	instance->msgService->sendMessage(response, NULL, NULL, 0);

	// If session still open for the tuner, assume activity and therefore immediately send UserActivityReport per CCMIS
	SDVSessionHandler::SDV_SESSION session;
	if (instance->sessionHandler->getOpenSession(receivedMsg->getTunerIndex(), &session) == 0) {
		uint32_t currentTime = instance->getCurrentTimeInSecs();
		UserActivityReport* report = new UserActivityReport(
				instance->config->getStbMac(),
				(uint8_t)session.tunerId,
				session.sourceId,
				currentTime);

		LOG("[UserActivityService] eventIndication() send UserActivityReport; tunerId=%d sourceId=%d lastActivity=%d\n", session.tunerId, session.sourceId, currentTime);
		instance->msgService->sendMessage(report, NULL, NULL, 0);
	}
}

void UserActivityService::programSelectIndication(void* obj, MessageService::REPONSE_STATUS_t status, SDVMessage* receivedMsg) {
	UserActivityService* instance = (UserActivityService*)obj;
	ProgramSelectIndication* psi = (ProgramSelectIndication*)receivedMsg;
	SDVSessionHandler::SDV_SESSION session;

	LOG("[UserActivityService] programSelectIndication() tunerId=%d currentSourceId=%d reason=%s\n", psi->getTunerIndex(), psi->getCurrentSourceId(),
			((psi->getReason() == REASON_rspForceTune) ? "ForceTune" : "ProgNotAvail"));

	// If session not open (i.e. tuner inactive or on broadcast service), send no session response message
	if (instance->sessionHandler->getOpenSession(psi->getTunerIndex(), &session) != 0) {
		LOG_ERR("[UserActivityService] programSelectIndication() send response to server for session not open\n");
		ProgramSelectResponse* responseMsg = new ProgramSelectResponse(
				instance->config->getStbMac(),
				psi->getTunerIndex(),
				RESPONSE_rspNoSession,
				psi->getTransactionId());
		instance->msgService->sendMessage(responseMsg, NULL, NULL, 0);
	}
	// Else if session now on a different SDV service, no response message will be sent
	else if (session.sourceId != psi->getCurrentSourceId()) {
		LOG_ERR("[UserActivityService] programSelectIndication() no response; session now on sourceId=%d\n", session.sourceId);
	}
	// Else send response and generate force tune event
	else {
		LOG("[UserActivityService] handleProgramNotAvailable() send response to server and invoke forceTune\n");
		ProgramSelectResponse* responseMsg = new ProgramSelectResponse(
				instance->config->getStbMac(),
				psi->getTunerIndex(),
				RESPONSE_rspOk,
				psi->getTransactionId());
		instance->msgService->sendMessage(responseMsg, NULL, NULL, 0);

		instance->sessionHandler->forceTune(&session);
	}
}

uint32_t UserActivityService::getCurrentTimeInSecs() {
    struct timeval tv;
    gettimeofday(&tv, NULL);
    return tv.tv_sec;
}

void UserActivityService::notifyTunerUsageChange(void* obj, uint32_t sourceId, uint8_t tunerUse) {
	UserActivityService* instance = (UserActivityService*)obj;
	int tunerId = instance->sessionHandler->getOpenSessionTunerId(sourceId);

	// If sdv session still open for this source ID, create and send EventIndication to sdv server
	if (tunerId >= 0) {

		EventIndication* indication = new EventIndication(
				instance->config->getStbMac(),
				tunerId,
				sourceId,
				tunerUse);

        RDK_LOG(RDK_LOG_INFO,"LOG.RDK.SDVAGENT","[UserActivityService] notifyTunerUsageChange: send EventIndication; tunerId=%d sourceId=%d tunerUse=0x%x\n", tunerId, sourceId, tunerUse);
		instance->msgService->sendMessage(indication, NULL, NULL, 0);
	}
	else {
        RDK_LOG(RDK_LOG_WARN,"LOG.RDK.SDVAGENT","[UserActivityService] notifyTunerUsageChange: session not found for sourceId %d\n", sourceId);
	}
}


