/**
* @file sdv_iarm.h
*
* @brief SDVAgent IARM API definitions and declarations.
*
* Components such as RMF use the APIs declared in this file to access the SDVAgent via the IARM BUS.
* The API are used for notifying the Agent when an SDV tuning session starts and ends as well as providing
* tuning information originating from the SDV Server.
*
* @par Document
* Document reference.
*
* @par Open Issues (in no particular order)
* -# None
*
* @par Assumptions
* -# None
*
* @par Implementation Notes
* -# None
*
*/
#ifndef SDV_IARM_H_
#define SDV_IARM_H_

#include <stdint.h>
#include <string>

/**
 * @def IARM_BUS_SDVAGENT_NAME
 * @brief IARM Bus name used to identify the SDVAgent
 */
#define IARM_BUS_SDVAGENT_NAME "SDVAgent"

/**
 * @def IARM_BUS_SDVAGENT_API_OPEN_SESSION
 * @brief IARM RPC method name for opening an SDV session and acquiring tuning details.
 * @details Must invoke for both SDV and Linear tunes.  For linear tunes, the response info in the payload is ignored.
 *
 * @see IARM_Bus_Call for usage
 * @see IARM_SDVAGENT_OPEN_SESSION_PAYLOAD for parameter details.
 */
#define IARM_BUS_SDVAGENT_API_OPEN_SESSION 	"sdvAgentOpenSession"

/**
 * @def IARM_BUS_SDVAGENT_API_CLOSE_SESSION
 * @brief IARM RPC method name for indicating an SDV tuning has closed
 * @details Must invoke only for SDV tunes, not Linear tunes.
 *
 * @see IARM_Bus_Call for usage
 * @see IARM_SDVAGENT_CLOSE_SESSION_PAYLOAD for parameter details
 */
#define IARM_BUS_SDVAGENT_API_CLOSE_SESSION	"sdvAgentCloseSession"

/**
 * @def IARM_BUS_SDVAGENT_API_STREAM_START
 * @brief IARM RPC method name for indicating an SDV session has started streaming for a
 * specific client device.
 *
 * @see IARM_Bus_Call for usage
 * @see IARM_SDVAGENT_STREAM_START_PAYLOAD for parameter details
 */
#define IARM_BUS_SDVAGENT_API_STREAM_START "sdvAgentStreamStart"

/**
 * @def IARM_BUS_SDVAGENT_API_STREAM_STOP
 * @brief IARM RPC method name for indicating an SDV session has stopped streaming for a
 * specific client device.
 *
 * @see IARM_Bus_Call for usage
 * @see IARM_SDVAGENT_STREAM_STOP_PAYLOAD for parameter details
 */
#define IARM_BUS_SDVAGENT_API_STREAM_STOP "sdvAgentStreamStop"

/**
 * @def IARM_BUS_SDVAGENT_API_NEW_MC_DATA
 * @brief IARM RPC method name for pushing new mini-carousel data into the SDVAgent
 *
 * @see IARM_Bus_Call for usage
 * @see IARM_SDVAGENT_STREAM_STOP_PAYLOAD for parameter details
 */
#define IARM_BUS_SDVAGENT_API_NEW_MC_DATA "sdvAgentNewMcData"

/**
 * @enum IARM_BUS_SDVAGENT_EVENT
 * @brief enumerations defining SDVAgent events broadcasted over IARM bus.
 */
typedef enum iarm_bus_sdvagent_event {
    SDVAGENT_REQUEST_FRESH_MC_DATA = 0,	//!> request autodiscovery to push fresh MC data from carousel captured after this event
    SDVAGENT_REQUEST_LATEST_MC_DATA,	//!> request autodiscovery to push latest MC data captured from carousel (may be cached or fresh)
    SDVAGENT_REQUEST_FORCE_TUNE,		//!> indicates server is requesting us to tune away from the SDV service
    SDVAGENT_EVENT_MAX       			//!> Max event id from this component
} IARM_BUS_SDVAGENT_EVENT;


/**
 * @enum SDVAGENT_PROGRAM_SELECT_STATUS
 * Indicates the status of SDVAgents attempt to acquire tuning info from the SDV Server.
 * Additional details on these errors can be found in the TWC-SDV_CCMIS specification.
 */
typedef enum sdvagent_select_status {
	PROGRAM_SELECT_CONFIRM_rspOK = 0x0000,                  //!< server responded okay
	PROGRAM_SELECT_CONFIRM_rspFormatError = 0x0001,         //!< server responded with format error
	PROGRAM_SELECT_CONFIRM_rspProgramOutOfService = 0x0006,	//!< server responded with program out of service
	PROGRAM_SELECT_CONFIRM_rspRedirect = 0x0007,            //!< server responded with program redirect
	PROGRAM_SELECT_CONFIRM_rspInvalidSG = 0x8001,           //!< server responded with invalid service group
	PROGRAM_SELECT_CONFIRM_rspUnknownClient = 0x8002,       //!< server responded with unknown client
	PROGRAM_SELECT_CONFIRM_rspSeNoResource = 0x8003,        //!< server responded with session resource unavailable
	PROGRAM_SELECT_CONFIRM_rspNetBwNotAvail = 0x8004,       //!< server responded with bandwidth not available
	PROGRAM_SELECT_CONFIRM_rspSrvCapacityExceeded = 0x8005, //!< server responded with server capacity exceeded
	PROGRAM_SELECT_CONFIRM_rspUnknownError = 0x80ff,        //!< server responded unknown error
    PROGRAM_SELECT_ERROR_serviceNotSDV = 0xd000,            //!< Source locator is not a SDV service
	MESSAGE_SENT_rspTimedOut = 0xe000,                 		//!< SDVAgent timed out waiting for server response
	AGENT_NOT_RESPONDING = 0xdead							//!< indicates sdvAgent did not respond to IARM call
} SDVAGENT_PROGRAM_SELECT_STATUS;

/**
 * @struct SDVAGENT_SESSION_INFO
 * Information sent to the SDVAgent for opening a switched or linear tuning session
 */
typedef struct sdvagent_session_info {
	uint32_t sourceId;			//!< service source ID originating from app
	uint32_t tunerId;			//!< tuner ID originating from gstreamer element
	uint8_t isSdvService;		//!< 1 indicates session for SDV channel; 0 indicates linear channel
} SDVAGENT_SESSION_INFO;

/**
 * @struct SDVAGENT_TUNING_INFO
 * Information needed by RMF QAM Source to tune.
 * This tuning information is only valid if its status is PROGRAM_SELECT_CONFIRM_rspOK.
 */
typedef struct sdvagent_tuning_info {
	SDVAGENT_PROGRAM_SELECT_STATUS status;	//!< status of SDVAgent acquiring this tuning info
	uint32_t carrierFreq;				//!< QAM frequency in herz
	uint8_t modulationFormat;			//!< QAM modulation format as defined in TWC-SDV_CCMIS spec
	uint32_t programNum;				//!< MPEG program number defined in the transport stream
} SDVAGENT_TUNING_INFO;

/**
 * @struct IARM_SDVAGENT_OPEN_SESSION_PAYLOAD
 * Input and output data structures used in the @link IARM_BUS_SDVAGENT_API_OPEN_SESSION payload
 */
typedef struct iarm_sdvagent_open_session_payload {
	SDVAGENT_SESSION_INFO requestInfo;		//!< request data to SDVAgent
	SDVAGENT_TUNING_INFO responseInfo;		//!< response data from SDVAGent
} IARM_SDVAGENT_OPEN_SESSION_PAYLOAD;

/**
 * @struct IARM_SDVAGENT_CLOSE_SESSION_PAYLOAD
 * Input data structure of an @link IARM_BUS_SDVAGENT_API_CLOSE_SESSION payload
 */
typedef struct iarm_sdvagent_close_session_payload {
	uint32_t sourceId;			//!< service source ID originating from app
	uint32_t tunerId;			//!< tuner ID originating from gstreamer element
} IARM_SDVAGENT_CLOSE_SESSION_PAYLOAD;

/**
 * @struct IARM_SDVAGENT_STREAM_START_PAYLOAD
 * Input data structure of an @link IARM_BUS_SDVAGENT_API_STREAM_START payload
 */
typedef struct iarm_sdvagent_stream_start_payload {
	uint32_t sourceId;			//!< service source ID originating from app
    std::string clientIpAddr;	    //!< IP address of the client device that started streaming
} IARM_SDVAGENT_STREAM_START_PAYLOAD;

/**
 * @struct IARM_SDVAGENT_STREAM_STOP_PAYLOAD
 * Input data structure of an @link IARM_BUS_SDVAGENT_API_STREAM_STOP payload
 */
typedef struct iarm_sdvagent_stream_stop_payload {
	uint32_t sourceId;			//!< service source ID originating from app
    std::string clientIpAddr;    	//!< IP address of the client device that stopped streaming
} IARM_SDVAGENT_STREAM_STOP_PAYLOAD;


/**
 * @struct SDVAGENT_IPV6_MAP_ELEMENT
 *
 * @brief A mapping between an PV4 address and its IPV6 equivalent
 * @details The IPV4 to IPV6 mapping intented to provided an eqivellent IPV6
 * address where only an IPV4 address is known.
 */
typedef struct sdvagent_ipv6_map_element {
	uint8_t ipv4address[4];			//!< 4 byte IPV4 address
	uint8_t ipv6address[16];		//!< 16 byte IPV4 address
} SDVAGENT_IPV6_MAP_ELEMENT;

/**
 * @typedef IARM_SDVAGENT_NEW_MC_DATA_PAYLOAD
 *
 * @brief Input data type of an @link IARM_BUS_SDVAGENT_API_NEW_MC_DATA payload
 *
 * @details.  The ipv6MapElements elements are used when operating in an IPV6 environment where only MCMIS 2.1 is
 * used and IPV6 address information is not available in the mcpData. This list will only
 * be considered if greater then 0.
 *
 * The structure of the payload is as follows
 *
 *  The MCP data from the inband SDV carousel.
 *  bytes [0 -> mcpDataSize]
 *
 *  The IP Address map
 *  bytes [mcpDataSize -> (ipv6MapElementCount * sizeof(SDVAGENT_IPV6_M'AP_ELEMENT)) ]
 */
typedef struct  sdvagent_mcp_data_payload {
	uint16_t mcpDataSize;								//!< The size of the MCP carousel data
	uint16_t ipv6MapElementCount;						//!< The number of ipv6MapElements.
	uint8_t payload[];                                   //!< Payload containing both MCP data and SDVAGENT_IPV6_MAP_ELEMENTs
} IARM_SDVAGENT_NEW_MC_DATA_PAYLOAD;

/**
 * @typedef IARM_SDVAGENT_FORCE_TUNE_PAYLOAD
 * @brief Payload data structure for a @link SDVAGENT_REQUEST_FORCE_TUNE IARM_BUS_SDVAGENT_EVENT
 */
typedef struct iarm_sdvagent_force_tune_payload {
	uint32_t tunerId;			//!< ID of the tuner
	uint32_t currentSourceId;	//!< source ID the tuner is currently on
} IARM_SDVAGENT_FORCE_TUNE_PAYLOAD;



/**
 * @def IARM_BUS_RMFSDV_ATTEMPT_ENABLE
 * @brief IARM RPC method name used by sdvAgent to attempt to enable SDV in rmfStreamer.
 * @details RMF SDV can only successfully enable if it detects configuration URLs during this bus call
 *
 * @see IARM_Bus_Call for usage
 * @see IARM_BUS_RMFSDV_ENABLE_PAYLOAD for parameter details.
 */
#define IARM_BUS_RMFSDV_ATTEMPT_ENABLE "rmfSdvAttemptEnable"

/**
 * @typedef IARM_RMFSDV_ENABLE_PAYLOAD
 * @brief Payload data structure for @link IARM_BUS_RMFSDV_ATTEMPT_ENABLE
 */
typedef struct iarm_rmfsdv_enable_payload {
    enum {
        NO_RESPONSE =    0xdead,     //!< indicates RMF SDV component failed to respond
        ENABLE_SUCCESS = 0x1111,     //!< indicates RMF SDV component successfully enabled
        ENABLE_REJECT =  0x2222      //!< indicates RMF SDV component cannot be enabled
    } result;                        //!< [out] result of enable attempt
} IARM_RMFSDV_ENABLE_PAYLOAD;

#endif /* SDV_IARM_H_ */
