/*
 * reporter.cpp
 *
 *  Created on: Jun 2, 2014
 *      Author: jasbedar
 */

#include "UserActivityService.h"
#include "SDVSessionHandler.h"
#include "mocks/IARMProxyService_mock.h"
#include "mocks/TestMessageHelper.h"
#include "mocks/MessageService_mock.h"
#include "mocks/ConfigurationParser_mock.h"
#include "mocks/SDVChannelMap_mock.h"
#include "mocks/SDVEventQueue_mock.h"
#include "mocks/SDVSessionHandler_mock.h"
#include <gtest/gtest.h>
#include <gmock/gmock.h>

#include <stdio.h>

using ::testing::Test;
using ::testing::Invoke;
using ::testing::Return;
using ::testing::_;

namespace sdv {

class UserActivityServiceTest : public Test {
public:
    void SetUp(){
        iarmProxy = new MockIARMProxyService();
        sdv::MessageService::SDV_MESSAGE_CALLBACK_t reqCb = NULL;
        config = new MockConfigurationParser(iarmProxy, eventQueue, NULL, NULL);
        msgService = new MockMessageService(config, NULL);
        channelMap = new MockSDVChannelMap();
        eventQueue = new MockSDVEventQueue();
        sessionHandler = new MockSDVSessionHandler();

        SERVICE_GROUP_ID = 34;
    }

    void TearDown(){
        delete iarmProxy;
        delete config;
        delete msgService;
        delete eventQueue;
        delete channelMap;
        delete sessionHandler;
        delete testService;
    }

    static uint8_t MAC_ADDRESS[4];
    uint32_t SERVICE_GROUP_ID;

    MockIARMProxyService *iarmProxy;
    MockMessageService *msgService;
    MockSDVEventQueue * eventQueue;
    MockConfigurationParser * config;
    MockSDVChannelMap * channelMap;
    MockSDVSessionHandler * sessionHandler;
    SDVChannelMap::SDV_CHANNEL_INFO channelInfo;
    UserActivityService * testService;

private:

};

uint8_t MAC[] = {0x11,0x22,0x33,0x44, 0x55, 0x66};
#define TUNER_ID 3
#define TUNER_USE 8
#define SOURCE_ID 9876
#define LAST_USER_ACTIVITY 12345678



TEST_F(UserActivityServiceTest, ConstructorTest) {

    testService = new UserActivityService(iarmProxy, msgService,eventQueue, config, sessionHandler);

    ASSERT_TRUE(testService != NULL);
}

TEST_F(UserActivityServiceTest, StartAndstopTest) {
    testService = new UserActivityService(iarmProxy, msgService, eventQueue, config, sessionHandler);
    uint32_t activityReportIntervalSecs = 3333;
    uint32_t activityReportIntervalMillis = activityReportIntervalSecs * 1000;

    EXPECT_CALL(*config, getLastUserActivityReportInterval())
    .WillRepeatedly(Return(activityReportIntervalSecs));

    EXPECT_CALL(*eventQueue, push(testService,_,_,activityReportIntervalMillis))
    .WillRepeatedly(Invoke(eventQueue, &MockSDVEventQueue::pushAction));

    EXPECT_CALL(*sessionHandler, getAnyOpenSession(_))
    .WillRepeatedly(Return(NULL));

    // Start the service
    SDVService::SERVICE_RESULT result = testService->start();
    ASSERT_TRUE(SDVService::SUCCESS == result);

    // Stop the service
    result = testService->stop();
    ASSERT_TRUE(SDVService::SUCCESS == result);
}

TEST_F(UserActivityServiceTest, SendReportTest) {
    testService = new UserActivityService(iarmProxy, msgService, eventQueue, config, sessionHandler);
    uint32_t activityReportIntervalSecs = 1;
    uint32_t activityReportIntervalMillis = activityReportIntervalSecs * 1000;

    // Expectations for ConfigurationParser
    EXPECT_CALL(*config, getLastUserActivityReportInterval())
    .WillRepeatedly(Return(activityReportIntervalSecs));
    EXPECT_CALL(*config, getStbMac())
    .WillRepeatedly(Return(MAC));

    // Expectations for EventQueue
    EXPECT_CALL(*eventQueue, push(testService,_,_,activityReportIntervalMillis))
    .WillRepeatedly(Invoke(eventQueue, &MockSDVEventQueue::pushAction));

    // Setup SessionHandler to give us a session
    SDVSessionHandler::SDV_SESSION* session = new SDVSessionHandler::SDV_SESSION(SOURCE_ID, TUNER_ID, TUNER_USE);
    session->state = SDVSessionHandler::SDV_SESSION::ACTIVE;
    sessionHandler->presetSession(session);
    EXPECT_CALL(*sessionHandler, getAnyOpenSession(_))
    .WillRepeatedly(Invoke(sessionHandler, &MockSDVSessionHandler::getAnyOpenSessionAction));

    EXPECT_CALL(*msgService, sendMessage(_,_,_,_))
    .WillRepeatedly(Invoke(msgService, &MockMessageService::sendMessageAction));

    // Start the service
    SDVService::SERVICE_RESULT result = testService->start();
    ASSERT_TRUE(SDVService::SUCCESS == result);

    usleep((activityReportIntervalMillis * 1000) + 1000);  // allow event queue timeout to fire once

    // Stop the service
    result = testService->stop();
    ASSERT_TRUE(SDVService::SUCCESS == result);

    // Verify sent message
    ASSERT_TRUE(msgService->sentMsgId == SDVMessage::UserActivityReport);
    ASSERT_TRUE(msgService->sentMsgTunerIndex == TUNER_ID);

    delete session;
}

/*
TEST_F(UserActivityServiceTest, QueryRequestIfSessionOpenend) {
	//TODO
}

TEST_F(UserActivityServiceTest, QueryRequestIfSessionClosed) {
	//TODO
}

TEST_F(UserActivityServiceTest, EventIndicationIfSessionOpenend) {
	//TODO
}

TEST_F(UserActivityServiceTest, EventIndicationIfSessionClosed) {
	//TODO
}

TEST_F(UserActivityServiceTest, programSelectIndicationIfSessionClosed) {
	//TODO
}

TEST_F(UserActivityServiceTest, programSelectIndicationIfSessionOnDifferentService) {
	//TODO
}

TEST_F(UserActivityServiceTest, programSelectIndicationIfSessionOpened) {
	//TODO
}
*/

} //namespace sdv
