/******************************************************************************
 *    (c)2010-2013 Broadcom Corporation
 *
 * This program is the proprietary software of Broadcom Corporation and/or its licensors,
 * and may only be used, duplicated, modified or distributed pursuant to the terms and
 * conditions of a separate, written license agreement executed between you and Broadcom
 * (an "Authorized License").  Except as set forth in an Authorized License, Broadcom grants
 * no license (express or implied), right to use, or waiver of any kind with respect to the
 * Software, and Broadcom expressly reserves all rights in and to the Software and all
 * intellectual property rights therein.  IF YOU HAVE NO AUTHORIZED LICENSE, THEN YOU
 * HAVE NO RIGHT TO USE THIS SOFTWARE IN ANY WAY, AND SHOULD IMMEDIATELY
 * NOTIFY BROADCOM AND DISCONTINUE ALL USE OF THE SOFTWARE.
 *
 * Except as expressly set forth in the Authorized License,
 *
 * 1.     This program, including its structure, sequence and organization, constitutes the valuable trade
 * secrets of Broadcom, and you shall use all reasonable efforts to protect the confidentiality thereof,
 * and to use this information only in connection with your use of Broadcom integrated circuit products.
 *
 * 2.     TO THE MAXIMUM EXTENT PERMITTED BY LAW, THE SOFTWARE IS PROVIDED "AS IS"
 * AND WITH ALL FAULTS AND BROADCOM MAKES NO PROMISES, REPRESENTATIONS OR
 * WARRANTIES, EITHER EXPRESS, IMPLIED, STATUTORY, OR OTHERWISE, WITH RESPECT TO
 * THE SOFTWARE.  BROADCOM SPECIFICALLY DISCLAIMS ANY AND ALL IMPLIED WARRANTIES
 * OF TITLE, MERCHANTABILITY, NONINFRINGEMENT, FITNESS FOR A PARTICULAR PURPOSE,
 * LACK OF VIRUSES, ACCURACY OR COMPLETENESS, QUIET ENJOYMENT, QUIET POSSESSION
 * OR CORRESPONDENCE TO DESCRIPTION. YOU ASSUME THE ENTIRE RISK ARISING OUT OF
 * USE OR PERFORMANCE OF THE SOFTWARE.
 *
 * 3.     TO THE MAXIMUM EXTENT PERMITTED BY LAW, IN NO EVENT SHALL BROADCOM OR ITS
 * LICENSORS BE LIABLE FOR (i) CONSEQUENTIAL, INCIDENTAL, SPECIAL, INDIRECT, OR
 * EXEMPLARY DAMAGES WHATSOEVER ARISING OUT OF OR IN ANY WAY RELATING TO YOUR
 * USE OF OR INABILITY TO USE THE SOFTWARE EVEN IF BROADCOM HAS BEEN ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGES; OR (ii) ANY AMOUNT IN EXCESS OF THE AMOUNT
 * ACTUALLY PAID FOR THE SOFTWARE ITSELF OR U.S. $1, WHICHEVER IS GREATER. THESE
 * LIMITATIONS SHALL APPLY NOTWITHSTANDING ANY FAILURE OF ESSENTIAL PURPOSE OF
 * ANY LIMITED REMEDY.
 *
 *****************************************************************************/
#include "bcmnexus_sys_defines.h"
#include "idirectfb_media_player_priv.h"
#include "nexus_simple_video_decoder.h"
#include "nexus_simple_audio_decoder.h"
#include "nexus_simple_stc_channel.h"
#include "nexus_surface_client.h"
#include "nexus_platform_client.h"
#include "nexus_playback.h"
#include "nexus_file.h"
#include "nexus_core_utils.h"
#include "../../utils/namevalue.h"
#include "b_playback_ip_lib.h"
#include "blst_queue.h"

/* media probe */
#include "bmedia_probe.h"
#include "bmpeg2ts_probe.h"
#include "bfile_stdio.h"
#if B_HAS_ASF
#include "basf_probe.h"
#endif
#if B_HAS_AVI
#include "bavi_probe.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#include "bstd.h"
#include "bkni.h"

BDBG_MODULE(media_player);

#include "../../utils/namevalue.c"

#define IP_NETWORK_MAX_JITTER 300 /* in msec */

BDBG_OBJECT_ID(media_player);
struct media_player
{
    BDBG_OBJECT(media_player)
    media_player_create_settings create_settings;
    media_player_start_settings start_settings;
    bool setup_done;
    bool started;
    
    NEXUS_SimpleVideoDecoderHandle videoDecoder;
    NEXUS_SimpleAudioDecoderHandle audioDecoder;
    NEXUS_FilePlayHandle file;
    NEXUS_PlaypumpHandle playpump;
    NEXUS_PlaybackHandle playback;
    NEXUS_PidChannelHandle pcrPidChannel;
    NEXUS_SimpleStcChannelHandle stcChannel;
    NEXUS_TransportTimestampType timestampType;
    NEXUS_TransportType transportType;
    bmedia_probe_t probe;
    const bmedia_probe_stream *stream;
    unsigned connectId;
    NEXUS_SimpleVideoDecoderStartSettings videoProgram;
    NEXUS_SimpleAudioDecoderStartSettings audioProgram;
    B_PlaybackIpHandle playbackIp;
    bool playbackIpActive;
    bool playbackIpLiveMode;
    B_PlaybackIpPsiInfo playbackIpPsi;
};

void media_player_get_default_create_settings_priv( media_player_create_settings *psettings )
{
    memset(psettings, 0, sizeof(*psettings));
    psettings->decodeVideo = true;
    psettings->decodeAudio = true;
}

void media_player_get_default_start_settings_priv( media_player_start_settings *psettings )
{
    memset(psettings, 0, sizeof(*psettings));
}

static void endOfStreamCallback(void *context, int param)
{
    media_player_t player = context;
    BSTD_UNUSED(param);
    if (player->start_settings.eof) {
        (player->start_settings.eof)(player->start_settings.context);
    }
}

static media_player_t media_player_p_create(const media_player_create_settings *psettings)
{
    NEXUS_PlaypumpOpenSettings playpumpOpenSettings;
    NEXUS_PlaybackSettings playbackSettings;
    NEXUS_ClientConfiguration clientConfig;
    media_player_t player;
    int rc;
   
    player = malloc(sizeof(*player));
    if (!player) {
        return NULL;
    }
    memset(player, 0, sizeof(*player));
    BDBG_OBJECT_SET(player, media_player);
    player->create_settings = *psettings;

    NEXUS_Platform_GetClientConfiguration(&clientConfig);

    NEXUS_Playpump_GetDefaultOpenSettings(&playpumpOpenSettings);
    playpumpOpenSettings.heap = clientConfig.heap[1];
    playpumpOpenSettings.boundsHeap = clientConfig.heap[1];
    player->playpump = NEXUS_Playpump_Open(NEXUS_ANY_ID, &playpumpOpenSettings);
    BDBG_ASSERT(player->playpump);
    player->playback = NEXUS_Playback_Create();
    BDBG_ASSERT(player->playback);

    player->stcChannel = NEXUS_SimpleStcChannel_Create(NULL);
    if (!player->stcChannel) {
        BDBG_WRN(("stc channel not available"));
    }

    NEXUS_Playback_GetSettings(player->playback, &playbackSettings);
    playbackSettings.playpump = player->playpump;
    playbackSettings.simpleStcChannel = player->stcChannel;
    playbackSettings.stcTrick = player->stcChannel != NULL;
    playbackSettings.endOfStreamCallback.callback = endOfStreamCallback;
    playbackSettings.endOfStreamCallback.context = player;
    rc = NEXUS_Playback_SetSettings(player->playback, &playbackSettings);
    BDBG_ASSERT(!rc);

    if (psettings->decodeVideo) {
        player->videoDecoder = NEXUS_SimpleVideoDecoder_Acquire(psettings->videoId);
        if (!player->videoDecoder) {
            BDBG_WRN(("video decoder not available"));
        }
    }

    if (psettings->decodeAudio) {
        player->audioDecoder = NEXUS_SimpleAudioDecoder_Acquire(psettings->audioId);
        if (!player->audioDecoder) {
            BDBG_WRN(("audio decoder not available"));
        }
    }

    player->playbackIp = B_PlaybackIp_Open(NULL);
    BDBG_ASSERT(player->playbackIp);

    return player;
}

media_player_t media_player_create_priv( const media_player_create_settings *psettings )
{
    media_player_t player;
    media_player_create_settings default_settings;

    if (!psettings) {
        media_player_get_default_create_settings_priv(&default_settings);
        psettings = &default_settings;
    }

    player = media_player_p_create(psettings);
    if (!player) goto error_create_player;
    
    return player;
    
error_create_player:
    return NULL;
}

static void b_print_media_string(const bmedia_probe_stream *stream)
{
    char stream_info[512];
    bmedia_stream_to_string(stream, stream_info, sizeof(stream_info));
    BDBG_WRN(( "Media Probe: %s", stream_info));
}

/* 
syntax: "scheme://domain:port/path?query_string#fragment_id"
each char array is null-terminated
*/
struct url {
    char scheme[32];
    char domain[128];
    unsigned port;
    char path[256]; /* contains "/path?query_string#fragment_id" */
};

#undef min
#define min(A,B) ((A)<(B)?(A):(B))

/* parse_url()

example: http://player.vimeo.com:80/play_redirect?quality=hd&codecs=h264&clip_id=638324 

    scheme=http
    domain=player.vimeo.com
    port=80
    path=/play_redirect?quality=hd&codecs=h264&clip_id=638324
    
example: file://videos/cnnticker.mpg or videos/cnnticker.mpg

    scheme=file
    domain=
    port=0
    path=videos/cnnticker.mpg

example: udp://192.168.1.10:1234 || udp://224.1.1.10:1234 || udp://@:1234

    scheme=udp
    domain=live channel is streamed to this STB's IP address 192.168.1.10 || server is streaming to a multicast address 224.1.1.10 || this STB but user doesn't need to explicitly specify its IP address
    port=1234
    path=N/A as server is just streaming out a live channel, client doesn't get to pick a file
    
*/
static void parse_url(const char *s, struct url *url)
{
    const char *server, *file;
    
    memset(url, 0, sizeof(*url));
    
    server = strstr(s, "://");
    if (!server) {
        strcpy(url->scheme, "file");
        server = s;
    }
    else {
        strncpy(url->scheme, s, server-s);
        server += strlen("://"); /* points to the start of server name */
    }
        
    if (!strcmp(url->scheme, "file")) {
        strncpy(url->path, server, sizeof(url->path)-1);
    }
    else {
        char *port;
        file = strstr(server, "/"); /* should point to start of file name */
        if (file) {
            strncpy(url->domain, server, min(sizeof(url->domain)-1, (unsigned)(file-server)));
            strncpy(url->path, file, sizeof(url->path)-1);
        }
        else {
            strncpy(url->domain, server, sizeof(url->domain)-1);
        }
        
        /* now server string is null terminated, look for explicit port number in the format of server:port */
        port = strstr(url->domain, ":");
        if (port) {
            *port = 0;
            url->port = atoi(port+1);
        }
        else {
            url->port = 80; /* default port */
        }
    }
}

int media_player_start_setup_priv( media_player_t player, const media_player_start_settings *psettings )
{
    int rc;

    BDBG_OBJECT_ASSERT(player, media_player);
    if (player->setup_done || player->started) {
        return BERR_TRACE(NEXUS_NOT_AVAILABLE);
    }
    if (!psettings) {
        return BERR_TRACE(NEXUS_INVALID_PARAMETER);
    }
    if (!psettings->stream_url) {
        return BERR_TRACE(NEXUS_INVALID_PARAMETER);
    }

    player->setup_done = true;
    player->start_settings = *psettings;
    player->timestampType = NEXUS_TransportTimestampType_eNone;
    player->transportType = NEXUS_TransportType_eTs;

    /* open pid channels and configure start settings based on probe */
    NEXUS_SimpleVideoDecoder_GetDefaultStartSettings(&player->videoProgram);
    NEXUS_SimpleAudioDecoder_GetDefaultStartSettings(&player->audioProgram);

    /* probe the stream */
    {
        bfile_io_read_t fd = NULL;
        FILE *fin;
        bmedia_probe_config probe_config;
        struct url url;
    
        parse_url(psettings->stream_url, &url);
        
        if (!strcasecmp(url.scheme, "http") || !strcasecmp(url.scheme, "https")) {
            /* URL contains http src info, setup IP playback */
            B_PlaybackIpSessionOpenSettings ipSessionOpenSettings;
            B_PlaybackIpSessionOpenStatus ipSessionOpenStatus;
            B_PlaybackIpSessionSetupSettings ipSessionSetupSettings;
            B_PlaybackIpSessionSetupStatus ipSessionSetupStatus;
            NEXUS_Error rc;
    
            /* Setup socket setting structure used in the IP Session Open */
            B_PlaybackIp_GetDefaultSessionOpenSettings(&ipSessionOpenSettings);
            ipSessionOpenSettings.socketOpenSettings.protocol = B_PlaybackIpProtocol_eHttp;
            strncpy(ipSessionOpenSettings.socketOpenSettings.ipAddr, url.domain, sizeof(ipSessionOpenSettings.socketOpenSettings.ipAddr)-1);
            ipSessionOpenSettings.socketOpenSettings.port = url.port;
            ipSessionOpenSettings.socketOpenSettings.url = url.path;
            BDBG_WRN(("parsed url is http://%s:%d%s", ipSessionOpenSettings.socketOpenSettings.ipAddr, ipSessionOpenSettings.socketOpenSettings.port, ipSessionOpenSettings.socketOpenSettings.url));
            ipSessionOpenSettings.ipMode = B_PlaybackIpClockRecoveryMode_ePull;
            
            rc = B_PlaybackIp_SessionOpen(player->playbackIp, &ipSessionOpenSettings, &ipSessionOpenStatus);
            if (rc) { rc = BERR_TRACE(rc); goto error; }
            BDBG_MSG (("Session Open call succeeded, HTTP status code %d", ipSessionOpenStatus.u.http.statusCode));
    
            /* now do session setup */
            B_PlaybackIp_GetDefaultSessionSetupSettings(&ipSessionSetupSettings);
            /* if app needs to play multiple formats (such as a DLNA DMP/DMR) (e.g. TS, VOB/PES, MP4, ASF, etc.), then set this option to do deep payload inspection */
            ipSessionSetupSettings.u.http.enablePayloadScanning = true;
            /* set a limit on how long the psi parsing should continue before returning */
            ipSessionSetupSettings.u.http.psiParsingTimeLimit = 30000; /* 30sec */
            rc = B_PlaybackIp_SessionSetup(player->playbackIp, &ipSessionSetupSettings, &ipSessionSetupStatus);
            if (rc) { rc = BERR_TRACE(rc); goto error; }
            BDBG_MSG (("Session Setup call succeeded, file handle %p", ipSessionSetupStatus.u.http.file));
            player->stream = (bmedia_probe_stream *)(ipSessionSetupStatus.u.http.stream);
            player->file = ipSessionSetupStatus.u.http.file;
            player->playbackIpActive = true;
            if (ipSessionSetupStatus.u.udp.psi.liveChannel)
                player->playbackIpLiveMode = true;
            else
                player->playbackIpLiveMode = false;
            player->playbackIpPsi = ipSessionSetupStatus.u.udp.psi;
            player->transportType = player->playbackIpPsi.mpegType;
        }
        else if (!strcasecmp(url.scheme, "udp") || !strcasecmp(url.scheme, "rtp")) {
            /* URL contains a live IP channel info, setup IP playback */
            B_PlaybackIpSessionOpenSettings ipSessionOpenSettings;
            B_PlaybackIpSessionOpenStatus ipSessionOpenStatus;
            B_PlaybackIpSessionSetupSettings ipSessionSetupSettings;
            B_PlaybackIpSessionSetupStatus ipSessionSetupStatus;
            NEXUS_Error rc;
    
            /* Setup socket setting structure used in the IP Session Open */
            B_PlaybackIp_GetDefaultSessionOpenSettings(&ipSessionOpenSettings);
            ipSessionOpenSettings.maxNetworkJitter = IP_NETWORK_MAX_JITTER;
            ipSessionOpenSettings.networkTimeout = 1;  /* timeout in 1 sec during network outage events */
            if (!strcasecmp(url.scheme, "rtp")) {
                ipSessionOpenSettings.socketOpenSettings.protocol = B_PlaybackIpProtocol_eRtp;
            }
            else {
                ipSessionOpenSettings.socketOpenSettings.protocol = B_PlaybackIpProtocol_eUdp;
            }
            strncpy(ipSessionOpenSettings.socketOpenSettings.ipAddr, url.domain, sizeof(ipSessionOpenSettings.socketOpenSettings.ipAddr)-1);
            ipSessionOpenSettings.socketOpenSettings.port = url.port;
#if 0
            /* needed for RTSP */
            ipSessionOpenSettings.socketOpenSettings.url = url.path;
#endif
            BDBG_MSG(("parsed url is udp://%s:%d%s", ipSessionOpenSettings.socketOpenSettings.ipAddr, ipSessionOpenSettings.socketOpenSettings.port, ipSessionOpenSettings.socketOpenSettings.url));
            ipSessionOpenSettings.ipMode = B_PlaybackIpClockRecoveryMode_ePushWithPcrSyncSlip;
            
            rc = B_PlaybackIp_SessionOpen(player->playbackIp, &ipSessionOpenSettings, &ipSessionOpenStatus);
            if (rc) { rc = BERR_TRACE(rc); goto error; }
            BDBG_MSG(("Session Open call succeeded"));
    
            /* now do session setup */
            B_PlaybackIp_GetDefaultSessionSetupSettings(&ipSessionSetupSettings);
            /* set a limit on how long the psi parsing should continue before returning */
            ipSessionSetupSettings.u.udp.psiParsingTimeLimit = 3000; /* 3sec */
            rc = B_PlaybackIp_SessionSetup(player->playbackIp, &ipSessionSetupSettings, &ipSessionSetupStatus);
            if (rc) { rc = BERR_TRACE(rc); goto error; }
            BDBG_MSG(("Session Setup call succeeded"));
            player->stream = (bmedia_probe_stream *)(ipSessionSetupStatus.u.udp.stream);
            player->file = NULL;
            player->playbackIpActive = true;
            player->playbackIpLiveMode = true;
            player->playbackIpPsi = ipSessionSetupStatus.u.udp.psi;
            player->transportType = player->playbackIpPsi.mpegType;
            BDBG_MSG(("Video Pid %d, Video Codec %d, Audio Pid %d, Audio Codec %d, Pcr Pid %d, Transport Type %d", 
                player->playbackIpPsi.videoPid, player->playbackIpPsi.videoCodec, player->playbackIpPsi.audioPid, player->playbackIpPsi.audioCodec, player->playbackIpPsi.pcrPid, player->playbackIpPsi.mpegType));
        }
        else
        {
            const char *index_url = NULL;
            
            player->probe = bmedia_probe_create();
    
            fin = fopen64(url.path, "rb");
            if (!fin) {
                BDBG_ERR(("can't open media file '%s' for probing", url.path));
                return -1;
            }
    
            fd = bfile_stdio_read_attach(fin);
    
            bmedia_probe_default_cfg(&probe_config);
            probe_config.file_name = url.path;
            probe_config.type = bstream_mpeg_type_unknown;
            player->stream = bmedia_probe_parse(player->probe, fd, &probe_config);
    
            /* now stream is either NULL, or stream descriptor with linked list of audio/video tracks */
            bfile_stdio_read_detach(fd);
    
            fclose(fin);
            if(!player->stream) {
                BDBG_ERR(("media probe can't parse stream '%s'", url.path));
                bmedia_probe_destroy(player->probe);
                player->probe = NULL;
                return -1;
            }
    
            index_url = psettings->index_url;
            if (!index_url) {
                if (player->stream->index == bmedia_probe_index_available || player->stream->index == bmedia_probe_index_required) {
                    /* if user didn't specify an index, use the file as index if probe indicates */
                    index_url = url.path;
                }
            }
            player->file = NEXUS_FilePlay_OpenPosix(url.path, index_url);
            if (!player->file) {
                BDBG_ERR(("can't open files: %s %s", url.path, index_url));
                rc = -1;
                goto error;
            }
        }
    
        if (player->stream) {
            b_print_media_string(player->stream);
            player->transportType = b_mpegtype2nexus(player->stream->type);
            if (player->stream->type == bstream_mpeg_type_ts) {
                if ((((bmpeg2ts_probe_stream*)player->stream)->pkt_len) == 192) {
                    player->timestampType = NEXUS_TransportTimestampType_eMod300;
                }
            }
        }
    }

    return 0;

error:
    if (player->playbackIpActive) {
        B_PlaybackIp_SessionClose(player->playbackIp);
        player->playbackIpActive = false;
    }
    media_player_stop_priv(player);
    return -1;
}


NEXUS_Error media_player_get_streaminfo_priv(media_player_t player, NEXUS_VideoDecoderStreamInformation *stream_info)
{
    NEXUS_Error rc = NEXUS_SUCCESS;

    if(!player || !stream_info) {
        return NEXUS_INVALID_PARAMETER;
    }

    if(player->videoDecoder) {
        NEXUS_SimpleVideoDecoder_GetStreamInformation( player->videoDecoder, stream_info);
        return rc;
    }

    return rc;
}

int media_player_start_priv( media_player_t player )
{
    NEXUS_PlaybackPidChannelSettings playbackPidSettings;
    NEXUS_PlaybackSettings playbackSettings;
    int rc;

    BDBG_OBJECT_ASSERT(player, media_player);
    if (!player->setup_done || player->started) {
        return BERR_TRACE(NEXUS_NOT_AVAILABLE);
    }

    /* if we've created the probe, we must be stopped */
    player->started = true;

    if (player->playbackIpActive && player->playbackIpLiveMode) {
        NEXUS_PlaypumpSettings playpumpSettings;
        NEXUS_SimpleStcChannelSettings stcSettings;
        NEXUS_PidChannelHandle pcrPidChannel = NULL;
        NEXUS_PlaypumpOpenPidChannelSettings pidChannelSettings;
        /* playback IP channel is in the live mode, it differs from regular playback in following ways: */
        /* -use nexus playback to feed IP playloads */
        /* -configure simpleStc in the live mode to use the PCRs */
        /* for streams w/ *no* transport timestamps, following additional changes are done */
        /* -program highJitter value in the simpleStc. This tells simpleStc to increase the PCR error thresholds */
        /* -delay decode by ip jitter value to account to absorb high network jitter */
        /* -increase CDB size to make sure that additional space for dejitter buffer is available */
        /* -TODO: check if default CDB sizes are big enough for additional buffering of IP_NETWORK_MAX_JITTER msec */

        /* configure nexus playpump */
        NEXUS_Playpump_GetSettings(player->playpump, &playpumpSettings);
        playpumpSettings.transportType = player->playbackIpPsi.mpegType;
#if 0
        /* TODO: need this logic for TTS streams */
        if (player->playbackIpPsi.transportTimeStampEnabled) {
            playpumpSettings.timestamp.type = NEXUS_TransportTimestampType_e32_Binary;
            /* TODO: how do we handle the need of explicit timebase here */
            //playpumpSettings.timestamp.timebase = NEXUS_Timebase_e1;
            //playpumpSettings.timestamp.pacingMaxError = playbackIpSettings.ttsParams.pacingMaxError;
            playpumpSettings.timestamp.pacing = true;
            playpumpSettings.timestamp.pacingOffsetAdjustDisable = true;
            playpumpSettings.timestamp.parityCheckDisable = true;
            playpumpSettings.timestamp.resetPacing = true;
        }
#endif
        rc = NEXUS_Playpump_SetSettings(player->playpump, &playpumpSettings);
        if (rc) { rc = BERR_TRACE(rc); goto error; }

        /* Open the pid channels */
        if (player->playbackIpPsi.videoCodec != NEXUS_VideoCodec_eNone && player->playbackIpPsi.videoPid) {
            NEXUS_VideoDecoderSettings settings;

            NEXUS_Playpump_GetDefaultOpenPidChannelSettings(&pidChannelSettings);
            pidChannelSettings.pidType = NEXUS_PidType_eVideo;
            player->videoProgram.settings.pidChannel = NEXUS_Playpump_OpenPidChannel(player->playpump, player->playbackIpPsi.videoPid, &pidChannelSettings);
            if (!player->videoProgram.settings.pidChannel) { rc = BERR_TRACE(rc); goto error; }
            player->videoProgram.settings.codec = player->playbackIpPsi.videoCodec;
            if (!player->playbackIpPsi.transportTimeStampEnabled) { /* if timestamps are present, playpump buffer is used as de-jitter buffer */
                /* increase the ptsOffset so that CDB can be used at the de-jitter buffer */
                NEXUS_SimpleVideoDecoder_GetSettings(player->videoDecoder, &settings);
                settings.ptsOffset = IP_NETWORK_MAX_JITTER * 45;    /* In 45Khz clock */
                rc = NEXUS_SimpleVideoDecoder_SetSettings(player->videoDecoder, &settings);
                if (rc) { rc = BERR_TRACE(rc); goto error; }
            }
            BDBG_MSG(("%s: video pid %d, pid channel created", __FUNCTION__, player->playbackIpPsi.videoPid));
        }

        /* Open the extra video pid channel if present in the stream */
        if (player->playbackIpPsi.extraVideoCodec != NEXUS_VideoCodec_eNone && player->playbackIpPsi.extraVideoPid != 0) {
            NEXUS_Playpump_GetDefaultOpenPidChannelSettings(&pidChannelSettings);
            pidChannelSettings.pidType = NEXUS_PidType_eVideo;
            player->videoProgram.settings.enhancementPidChannel = NEXUS_Playpump_OpenPidChannel(player->playpump, player->playbackIpPsi.extraVideoPid, &pidChannelSettings);
            if (!player->videoProgram.settings.enhancementPidChannel) { rc = BERR_TRACE(rc); goto error; }
            BDBG_MSG(("%s: extra video pid %d, codec %d is present, pid channel created", __FUNCTION__, player->playbackIpPsi.extraVideoPid, player->playbackIpPsi.extraVideoCodec));
        }

        if (player->playbackIpPsi.audioCodec != NEXUS_AudioCodec_eUnknown && player->playbackIpPsi.audioPid) {
            NEXUS_SimpleAudioDecoderSettings settings;
            /* Open the audio pid channel */
            NEXUS_Playpump_GetDefaultOpenPidChannelSettings(&pidChannelSettings);
            pidChannelSettings.pidType = NEXUS_PidType_eAudio;
            pidChannelSettings.pidTypeSettings.audio.codec = player->playbackIpPsi.audioCodec;
            player->audioProgram.primary.pidChannel = NEXUS_Playpump_OpenPidChannel(player->playpump, player->playbackIpPsi.audioPid, &pidChannelSettings);
            if (!player->audioProgram.primary.pidChannel) { rc = BERR_TRACE(rc); goto error; }
            player->audioProgram.primary.codec = player->playbackIpPsi.audioCodec;
            if (!player->playbackIpPsi.transportTimeStampEnabled) { /* if timestamps are present, playpump buffer is used as de-jitter buffer */
                /* increase the ptsOffset so that CDB can be used at the de-jitter buffer */
                NEXUS_SimpleAudioDecoder_GetSettings(player->audioDecoder, &settings);
                settings.primary.ptsOffset = IP_NETWORK_MAX_JITTER * 45;    /* In 45Khz clock */
                rc = NEXUS_SimpleAudioDecoder_SetSettings(player->audioDecoder, &settings);
                if (rc) { rc = BERR_TRACE(rc); goto error; }
            }
        }

        if (player->playbackIpPsi.pcrPid && player->playbackIpPsi.pcrPid != player->playbackIpPsi.audioPid && player->playbackIpPsi.pcrPid != player->playbackIpPsi.videoPid) {
            /* Open the pcr pid channel */
            NEXUS_Playpump_GetDefaultOpenPidChannelSettings(&pidChannelSettings);
            pidChannelSettings.pidType = NEXUS_PidType_eUnknown;
            pcrPidChannel = NEXUS_Playpump_OpenPidChannel(player->playpump, player->playbackIpPsi.pcrPid, &pidChannelSettings);
            if (!pcrPidChannel) { rc = BERR_TRACE(rc); goto error; }
            BDBG_MSG(("%s: Opened pcr pid channel %u for pcr pid %u ", __FUNCTION__, pcrPidChannel, player->playbackIpPsi.pcrPid));
        }
        else 
        {
            if (player->playbackIpPsi.pcrPid == player->playbackIpPsi.audioPid)
                pcrPidChannel = player->audioProgram.primary.pidChannel;
            else
                pcrPidChannel = player->videoProgram.settings.pidChannel;
        }

        /* now configure the simple stc channel */
        if (!player->stcChannel) { rc = BERR_TRACE(rc); goto error; }
        NEXUS_SimpleStcChannel_GetSettings(player->stcChannel, &stcSettings);
        stcSettings.mode = NEXUS_StcChannelMode_ePcr;
        stcSettings.modeSettings.pcr.pidChannel = pcrPidChannel;
        if (player->playbackIpPsi.transportTimeStampEnabled) { 
            /* when timestamps are present, dejittering is done before feeding to xpt and */ 
            /* thus is treated as a live playback */
            stcSettings.modeSettings.highJitter.mode = NEXUS_SimpleStcChannelHighJitterMode_eTtsPacing;
        }
        else {
            /* when timestamps are not present, we directly feed the jittered packets to the transport */
            /* and set the max network jitter in highJitterThreshold. */
            /* This enables the SimpleStc to internally program the various stc & timebase related thresholds to account for network jitter */
            stcSettings.modeSettings.highJitter.threshold = IP_NETWORK_MAX_JITTER;
            stcSettings.modeSettings.highJitter.mode = NEXUS_SimpleStcChannelHighJitterMode_eDirect;
        }
        rc = NEXUS_SimpleStcChannel_SetSettings(player->stcChannel, &stcSettings);
        if (rc) { rc = BERR_TRACE(rc); goto error; }
        BDBG_MSG(("%s: live IP mode configuration complete...", __FUNCTION__));
    }
    else 
    {
        const bmedia_probe_track *track;
        int program_offset = -1;
        
        /* playback mode setup for either file or HTTP playback */
        if (player->stcChannel) {
            NEXUS_SimpleStcChannelSettings stcSettings;
            NEXUS_SimpleStcChannel_GetSettings(player->stcChannel, &stcSettings);
            if (stcSettings.modeSettings.Auto.transportType != player->transportType) {
                stcSettings.modeSettings.Auto.transportType = player->transportType;
                rc = NEXUS_SimpleStcChannel_SetSettings(player->stcChannel, &stcSettings);
                if (rc) { rc = BERR_TRACE(rc); goto error; }
            }
        }

        NEXUS_Playback_GetSettings(player->playback, &playbackSettings);
        playbackSettings.playpumpSettings.transportType = player->transportType;
        playbackSettings.playpumpSettings.timestamp.type = player->timestampType;
        playbackSettings.endOfStreamAction = player->start_settings.loop?NEXUS_PlaybackLoopMode_eLoop:NEXUS_PlaybackLoopMode_ePause;
        rc = NEXUS_Playback_SetSettings(player->playback, &playbackSettings);
        if (rc) { rc = BERR_TRACE(rc); goto error; }

        for(track=BLST_SQ_FIRST(&player->stream->tracks);track;track=BLST_SQ_NEXT(track, link)) {
            if (program_offset == -1) {
                program_offset = track->program;
            }
            if ((track->program - program_offset) != player->start_settings.program) continue;

            switch(track->type) {
                case bmedia_track_type_audio:
                    if(track->info.audio.codec != baudio_format_unknown && player->audioDecoder) {
                        if (!player->audioProgram.primary.pidChannel) {
                            NEXUS_Playback_GetDefaultPidChannelSettings(&playbackPidSettings);
                            playbackPidSettings.pidSettings.pidType = NEXUS_PidType_eAudio;
                            playbackPidSettings.pidTypeSettings.audio.simpleDecoder = player->audioDecoder;
                            player->audioProgram.primary.pidChannel = NEXUS_Playback_OpenPidChannel(player->playback, track->number, &playbackPidSettings);
                            player->audioProgram.primary.codec = b_audiocodec2nexus(track->info.audio.codec);
                        }
                    }
                    break;
                case bmedia_track_type_video:
                    if (player->videoDecoder) {
                        if (track->info.video.codec == bvideo_codec_h264_svc || track->info.video.codec == bvideo_codec_h264_mvc) {
                            if (player->videoProgram.settings.pidChannel && !player->videoProgram.settings.enhancementPidChannel) {
                                NEXUS_Playback_GetDefaultPidChannelSettings(&playbackPidSettings);
                                playbackPidSettings.pidSettings.pidType = NEXUS_PidType_eVideo;
                                playbackPidSettings.pidTypeSettings.video.codec = b_videocodec2nexus(track->info.video.codec);
                                playbackPidSettings.pidTypeSettings.video.index = true;
                                playbackPidSettings.pidTypeSettings.video.simpleDecoder = player->videoDecoder;
                                player->videoProgram.settings.enhancementPidChannel = NEXUS_Playback_OpenPidChannel(player->playback, track->number, &playbackPidSettings);
                                player->videoProgram.settings.codec = b_videocodec2nexus(track->info.video.codec); /* overwrite */
                            }
                            break;
                        }
                        else if (track->info.video.codec != bvideo_codec_unknown) {
                            if (!player->videoProgram.settings.pidChannel) {
                                NEXUS_Playback_GetDefaultPidChannelSettings(&playbackPidSettings);
                                playbackPidSettings.pidSettings.pidType = NEXUS_PidType_eVideo;
                                playbackPidSettings.pidTypeSettings.video.codec = b_videocodec2nexus(track->info.video.codec);
                                playbackPidSettings.pidTypeSettings.video.index = true;
                                playbackPidSettings.pidTypeSettings.video.simpleDecoder = player->videoDecoder;
                                player->videoProgram.settings.pidChannel = NEXUS_Playback_OpenPidChannel(player->playback, track->number, &playbackPidSettings);
                                player->videoProgram.settings.codec = b_videocodec2nexus(track->info.video.codec);
                                player->videoProgram.settings.timestampMode = track->info.video.timestamp_order;
                                if (player->create_settings.maxWidth && player->create_settings.maxHeight) {
                                    player->videoProgram.maxWidth = player->create_settings.maxWidth;
                                    player->videoProgram.maxHeight = player->create_settings.maxHeight;
                                }
                            }
                        }
                    }
                    break;
#if 0
                    /* TODO: need playback to handle duplicate w/ different settings in the case of eOther */
                case bmedia_track_type_pcr:
                    NEXUS_Playback_GetDefaultPidChannelSettings(&playbackPidSettings);
                    playbackPidSettings.pidSettings.pidType = NEXUS_PidType_eOther;
                    player->pcrPidChannel = NEXUS_Playback_OpenPidChannel(player->playback, track->number, &playbackPidSettings);
                    break;
#endif
                default:
                    break;
            }
        }

#if B_HAS_AVI
        if (player->stream->type == bstream_mpeg_type_avi && ((bavi_probe_stream *)player->stream)->video_framerate) {
            NEXUS_LookupFrameRate(((bavi_probe_stream *)player->stream)->video_framerate, &player->videoProgram.settings.frameRate);
        }
#endif
    }
    
    if (!player->videoProgram.settings.pidChannel && !player->audioProgram.primary.pidChannel) {
        BDBG_WRN(("no content found"));
        goto error;
    }
    
    /* set StcChannel to all decoders before starting any */
    if (player->videoProgram.settings.pidChannel) {
        rc = NEXUS_SimpleVideoDecoder_SetStcChannel(player->videoDecoder, player->stcChannel);
        if (rc) { rc = BERR_TRACE(rc); goto error; }
    }
    if (player->audioProgram.primary.pidChannel) {
        rc = NEXUS_SimpleAudioDecoder_SetStcChannel(player->audioDecoder, player->stcChannel);
        if (rc) { rc = BERR_TRACE(rc); goto error; }
    }

    if (player->videoProgram.settings.pidChannel) {
        rc = NEXUS_SimpleVideoDecoder_Start(player->videoDecoder, &player->videoProgram);
        if (rc) { rc = BERR_TRACE(rc); goto error; }
    }
    if (player->audioProgram.primary.pidChannel) {
        rc = NEXUS_SimpleAudioDecoder_Start(player->audioDecoder, &player->audioProgram);
        if (rc) { rc = BERR_TRACE(rc); goto error; }
        /* decode may fail if audio codec not supported */
    }

    if (player->playbackIpActive && player->playbackIpLiveMode) {
        BDBG_MSG(("starting nexus playpump"));
        rc = NEXUS_Playpump_Start(player->playpump);
    }
    else 
    {
        BDBG_MSG(("starting nexus playback"));
        rc = NEXUS_Playback_Start(player->playback, player->file, NULL);
    }
    if (rc) { rc = BERR_TRACE(rc); goto error; }

    if (player->playbackIpActive) {
        B_PlaybackIpSessionStartSettings ipSessionStartSettings;
        B_PlaybackIpSessionStartStatus ipSessionStartStatus;
        /* B_PlaybackIp_GetDefaultSessionStartSettings(&ipSessionStartSettings);*/
        memset(&ipSessionStartSettings, 0, sizeof(ipSessionStartSettings));
        /* set Nexus handles */
        ipSessionStartSettings.nexusHandles.playpump = player->playpump;
        ipSessionStartSettings.nexusHandles.playback = player->playback;
        ipSessionStartSettings.nexusHandles.simpleVideoDecoder = player->videoDecoder;
        ipSessionStartSettings.nexusHandles.simpleStcChannel = player->stcChannel;
        ipSessionStartSettings.nexusHandles.simpleAudioDecoder = player->audioDecoder;
        ipSessionStartSettings.nexusHandles.videoPidChannel = player->videoProgram.settings.pidChannel;
        ipSessionStartSettings.nexusHandles.extraVideoPidChannel = player->videoProgram.settings.enhancementPidChannel;
        ipSessionStartSettings.nexusHandles.audioPidChannel = player->audioProgram.primary.pidChannel;
        ipSessionStartSettings.nexusHandles.pcrPidChannel = player->pcrPidChannel;
        ipSessionStartSettings.nexusHandlesValid = true;
        ipSessionStartSettings.mpegType = player->transportType;
        rc = B_PlaybackIp_SessionStart(player->playbackIp, &ipSessionStartSettings, &ipSessionStartStatus);
        if (rc) { rc = BERR_TRACE(rc); goto error; }
        BDBG_MSG(("playback playback in %s mode is started", player->playbackIpLiveMode?"Live":"Playback"));
    }
    return 0;

error:
    if (player->playbackIpActive) {
        B_PlaybackIp_SessionStop(player->playbackIp);
        B_PlaybackIp_SessionClose(player->playbackIp);
        player->playbackIpActive = false;
    }
    media_player_stop_priv(player);
    return -1;
}

void media_player_stop_priv(media_player_t player)
{
    BDBG_OBJECT_ASSERT(player, media_player);

    if (!player->started) return;

    if (player->videoDecoder) {
        NEXUS_SimpleVideoDecoder_Stop(player->videoDecoder);
        NEXUS_SimpleVideoDecoder_SetStcChannel(player->videoDecoder, NULL);
    }
    if (player->audioDecoder) {
        NEXUS_SimpleAudioDecoder_Stop(player->audioDecoder);
        NEXUS_SimpleAudioDecoder_SetStcChannel(player->audioDecoder, NULL);
    }
    if (player->playbackIpActive) {
        if (player->playbackIpLiveMode) {
            if (player->playpump) {
                NEXUS_Playpump_Stop(player->playpump);
                NEXUS_Playpump_CloseAllPidChannels(player->playpump);
                player->playbackIpLiveMode = false;
            }
        }
        else {
            /* playback mode */
            if (player->playback) {
                NEXUS_Playback_Stop(player->playback);
                NEXUS_Playback_CloseAllPidChannels(player->playback);
            }
        }
        player->videoProgram.settings.pidChannel = NULL;
        player->audioProgram.primary.pidChannel = NULL;
        player->pcrPidChannel = NULL;
        B_PlaybackIp_SessionStop(player->playbackIp);
        B_PlaybackIp_SessionClose(player->playbackIp);
        player->playbackIpActive = false;
    }
    else
    {
        /* regular file playback case */
        if (player->playback) {
            NEXUS_Playback_Stop(player->playback);
            if (player->videoProgram.settings.pidChannel) {
                NEXUS_Playback_ClosePidChannel(player->playback, player->videoProgram.settings.pidChannel);
                player->videoProgram.settings.pidChannel = NULL;
            }
            if (player->audioProgram.primary.pidChannel) {
                NEXUS_Playback_ClosePidChannel(player->playback, player->audioProgram.primary.pidChannel);
                player->audioProgram.primary.pidChannel = NULL;
            }
            if (player->pcrPidChannel) {
                NEXUS_Playback_ClosePidChannel(player->playback, player->pcrPidChannel);
                player->pcrPidChannel = NULL;
            }
        }
        if (player->file) {
            NEXUS_FilePlay_Close(player->file);
        }
        if (player->probe) {
            if (player->stream) {
                bmedia_probe_stream_free(player->probe, player->stream);
            }
            bmedia_probe_destroy(player->probe);
        }
    }
    player->started = false;
    player->setup_done = false;
}

void media_player_destroy_priv(media_player_t player)
{
    BDBG_OBJECT_ASSERT(player, media_player);
    if (player->started) {
        media_player_stop_priv(player);
    }
    if (player->playback) {
        NEXUS_Playback_Destroy(player->playback);
    }
    if (player->playpump) {
        NEXUS_Playpump_Close(player->playpump);
    }
    if (player->videoDecoder) {
        NEXUS_SimpleVideoDecoder_Release(player->videoDecoder);
    }
    if (player->audioDecoder) {
        NEXUS_SimpleAudioDecoder_Release(player->audioDecoder);
    }
    if (player->stcChannel) {
        NEXUS_SimpleStcChannel_Destroy(player->stcChannel);
    }
    if (player->playbackIp) {
        B_PlaybackIp_Close(player->playbackIp);
    }
    BDBG_OBJECT_DESTROY(player, media_player);
    free(player);
}

int media_player_trick_priv( media_player_t player, int rate)
{
    BDBG_OBJECT_ASSERT(player, media_player);
    if (!player->started) {
        /* Call start and then trick */
        return BERR_TRACE(NEXUS_NOT_AVAILABLE);
    }
    if ( player->playbackIpActive && player->playbackIpLiveMode ) {
        /* In live mode, can't do trick modes */
        return BERR_TRACE(NEXUS_NOT_AVAILABLE);
    }
    if (rate == NEXUS_NORMAL_DECODE_RATE) {
        return NEXUS_Playback_Play(player->playback);
    }
    else if (rate == 0) {
        return NEXUS_Playback_Pause(player->playback);
    }
    else {
        NEXUS_PlaybackTrickModeSettings settings;
        NEXUS_Playback_GetDefaultTrickModeSettings(&settings);
        settings.rate = rate;
        return NEXUS_Playback_TrickMode(player->playback, &settings);
    }
}

int media_player_seek_priv( media_player_t player, unsigned int milliseconds)
{
    int rc;

    BDBG_OBJECT_ASSERT(player, media_player);

    if (!player->started) {
        /* Set start_position in start parameters */
        return BERR_TRACE(NEXUS_NOT_AVAILABLE);
    }

    if ( player->playbackIpActive ) {
        B_PlaybackIpTrickModesSettings ipTrickModeSettings;

        if ( player->playbackIpLiveMode ) {
            /* In live mode, can't do trick modes */
            return BERR_TRACE(NEXUS_NOT_AVAILABLE);
        }

        if ((rc = B_PlaybackIp_GetTrickModeSettings(player->playbackIp, &ipTrickModeSettings)) != B_ERROR_SUCCESS) {
            return BERR_TRACE(NEXUS_UNKNOWN);
        }

        ipTrickModeSettings.seekPositionIsRelative = false;
        ipTrickModeSettings.seekBackward = false;
        ipTrickModeSettings.seekPosition = milliseconds;
        ipTrickModeSettings.pauseMethod = B_PlaybackIpPauseMethod_UseConnectionStalling;

        if ((rc = B_PlaybackIp_Seek(player->playbackIp, &ipTrickModeSettings)) != B_ERROR_SUCCESS) {
            return BERR_TRACE(NEXUS_UNKNOWN);
        }
    }
    else
    {
        rc = NEXUS_Playback_Seek(player->playback, milliseconds);
    }
    return rc;
}

int media_player_get_playback_status_priv( media_player_t player, NEXUS_PlaybackStatus *pstatus )
{
    return NEXUS_Playback_GetStatus(player->playback, pstatus);
}

int media_player_get_status_priv( media_player_t player, media_player_status *pstatus )
{
    if (!player || !pstatus) {
        return BERR_TRACE(NEXUS_INVALID_PARAMETER);
    }

    memset(pstatus, 0, sizeof(*pstatus));

    if (!player->setup_done) {
        /* Can't give valid data */
        return NEXUS_NOT_INITIALIZED;
    }

    if (player->playbackIpActive) {
        pstatus->has_video = ((player->playbackIpPsi.videoCodec != NEXUS_VideoCodec_eNone) && player->playbackIpPsi.videoPid);
        if (pstatus->has_video) {
            pstatus->width = player->playbackIpPsi.videoWidth;
            pstatus->height = player->playbackIpPsi.videoHeight;
        }
        pstatus->has_audio = ((player->playbackIpPsi.audioCodec != NEXUS_AudioCodec_eUnknown) && player->playbackIpPsi.audioPid);
        pstatus->valid = true;
    }
    else
    {
        const bmedia_probe_track *track;
        int program_offset = -1;

        if (!player->stream) {
            return BERR_TRACE(NEXUS_UNKNOWN);
        }

        for(track=BLST_SQ_FIRST(&player->stream->tracks);track;track=BLST_SQ_NEXT(track, link)) {
            if (program_offset == -1) {
                program_offset = track->program;
            }
            if ((track->program - program_offset) != player->start_settings.program) continue;

            switch(track->type) {
                case bmedia_track_type_audio:
                    if (track->info.audio.codec != baudio_format_unknown) {
                        pstatus->has_audio = true;
                    }
                    break;
                case bmedia_track_type_video:
                    if (track->info.video.codec != bvideo_codec_unknown) {
                        pstatus->has_video = true;
                        pstatus->width = track->info.video.width;
                        pstatus->height = track->info.video.height;
                    }
                    break;
                default:
                    break;
            }
        }
        pstatus->valid = true;
    }

    if (pstatus->valid) {
        BDBG_WRN(("%s: video:%s %d,%d, audio:%s", __FUNCTION__, pstatus->has_video ? "yes" : "no", pstatus->width, pstatus->height, pstatus->has_audio ? "yes" : "no"));
    }
    else {
        BDBG_WRN(("%s: status invalid", __FUNCTION__));
    }

    return NEXUS_SUCCESS;
}

NEXUS_SimpleVideoDecoderHandle media_player_get_video_decoder_priv(media_player_t player)
{
    return player->videoDecoder;
}

int probe_media_priv(const char *streamname, const char *indexname, struct probe_results *results)
{
    bmedia_probe_t probe;
    bfile_io_read_t fd = NULL;
    FILE *fin;
    bmedia_probe_config probe_config;
    const bmedia_probe_track *track;
    const bmedia_probe_stream *stream;
    NEXUS_FilePlayHandle file;
    int rc = 0;
    
    memset(results, 0, sizeof(*results));
    
    probe = bmedia_probe_create();
    if (!probe) {
        return BERR_TRACE(-1);
    }

    fin = fopen64(streamname, "rb");
    if (!fin) {
        BDBG_ERR(("can't open media file '%s' for probing", streamname));
        return -1;
    }

    fd = bfile_stdio_read_attach(fin);

    bmedia_probe_default_cfg(&probe_config);
    probe_config.file_name = streamname;
    probe_config.type = bstream_mpeg_type_unknown;
    stream = bmedia_probe_parse(probe, fd, &probe_config);

    /* now stream is either NULL, or stream descriptor with linked list of audio/video tracks */
    bfile_stdio_read_detach(fd);

    fclose(fin);
    if(!stream) {
        BDBG_ERR(("media probe can't parse stream '%s'", streamname));
        bmedia_probe_destroy(probe);
        probe = NULL;
        return -1;
    }

    if (!indexname) {
        if (stream->index == bmedia_probe_index_available || stream->index == bmedia_probe_index_required) {
            /* if user didn't specify an index, use the file as index if probe indicates */
            indexname = streamname;
        }
    }
    file = NEXUS_FilePlay_OpenPosix(streamname, indexname);
    if (!file) {
        BDBG_ERR(("can't open files: %s %s", streamname, indexname));
        rc = -1;
        goto error;
    }

    results->transportType = b_mpegtype2nexus(stream->type);
    if (stream->type == bstream_mpeg_type_ts) {
        if ((((bmpeg2ts_probe_stream*)stream)->pkt_len) == 192) {
            results->timestampType = NEXUS_TransportTimestampType_eMod300;
        }
    }
    
    for(track=BLST_SQ_FIRST(&stream->tracks);track;track=BLST_SQ_NEXT(track, link)) {
        switch(track->type) {
            case bmedia_track_type_audio:
                if(track->info.audio.codec != baudio_format_unknown && results->num_audio < MAX_PIDS) {
                    results->audio[results->num_audio].pid = track->number;
                    results->audio[results->num_audio].codec = b_audiocodec2nexus(track->info.audio.codec);
                    results->num_audio++;
                }
                break;
            case bmedia_track_type_video:
                if (track->info.video.codec == bvideo_codec_h264_svc || track->info.video.codec == bvideo_codec_h264_mvc) {
                    /* TODO */
                }
                else if (track->info.video.codec != bvideo_codec_unknown && results->num_video < MAX_PIDS) {
                    results->video[results->num_video].pid = track->number;
                    results->video[results->num_video].codec = b_videocodec2nexus(track->info.video.codec);
                    results->num_video++;
                }
                break;
            default:
                break;
        }
    }
    
error:
    if (file) {
        NEXUS_FilePlay_Close(file);
    }
    if (probe) {
        if (stream) {
            bmedia_probe_stream_free(probe, stream);
        }
        bmedia_probe_destroy(probe);
    }

    return rc;
}
