/******************************************************************************
 *    (c)2009-2013 Broadcom Corporation
 *
 * This program is the proprietary software of Broadcom Corporation and/or its licensors,
 * and may only be used, duplicated, modified or distributed pursuant to the terms and
 * conditions of a separate, written license agreement executed between you and Broadcom
 * (an "Authorized License").  Except as set forth in an Authorized License, Broadcom grants
 * no license (express or implied), right to use, or waiver of any kind with respect to the
 * Software, and Broadcom expressly reserves all rights in and to the Software and all
 * intellectual property rights therein.  IF YOU HAVE NO AUTHORIZED LICENSE, THEN YOU
 * HAVE NO RIGHT TO USE THIS SOFTWARE IN ANY WAY, AND SHOULD IMMEDIATELY
 * NOTIFY BROADCOM AND DISCONTINUE ALL USE OF THE SOFTWARE.
 *
 * Except as expressly set forth in the Authorized License,
 *
 * 1.     This program, including its structure, sequence and organization, constitutes the valuable trade
 * secrets of Broadcom, and you shall use all reasonable efforts to protect the confidentiality thereof,
 * and to use this information only in connection with your use of Broadcom integrated circuit products.
 *
 * 2.     TO THE MAXIMUM EXTENT PERMITTED BY LAW, THE SOFTWARE IS PROVIDED "AS IS"
 * AND WITH ALL FAULTS AND BROADCOM MAKES NO PROMISES, REPRESENTATIONS OR
 * WARRANTIES, EITHER EXPRESS, IMPLIED, STATUTORY, OR OTHERWISE, WITH RESPECT TO
 * THE SOFTWARE.  BROADCOM SPECIFICALLY DISCLAIMS ANY AND ALL IMPLIED WARRANTIES
 * OF TITLE, MERCHANTABILITY, NONINFRINGEMENT, FITNESS FOR A PARTICULAR PURPOSE,
 * LACK OF VIRUSES, ACCURACY OR COMPLETENESS, QUIET ENJOYMENT, QUIET POSSESSION
 * OR CORRESPONDENCE TO DESCRIPTION. YOU ASSUME THE ENTIRE RISK ARISING OUT OF
 * USE OR PERFORMANCE OF THE SOFTWARE.
 *
 * 3.     TO THE MAXIMUM EXTENT PERMITTED BY LAW, IN NO EVENT SHALL BROADCOM OR ITS
 * LICENSORS BE LIABLE FOR (i) CONSEQUENTIAL, INCIDENTAL, SPECIAL, INDIRECT, OR
 * EXEMPLARY DAMAGES WHATSOEVER ARISING OUT OF OR IN ANY WAY RELATING TO YOUR
 * USE OF OR INABILITY TO USE THE SOFTWARE EVEN IF BROADCOM HAS BEEN ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGES; OR (ii) ANY AMOUNT IN EXCESS OF THE AMOUNT
 * ACTUALLY PAID FOR THE SOFTWARE ITSELF OR U.S. $1, WHICHEVER IS GREATER. THESE
 * LIMITATIONS SHALL APPLY NOTWITHSTANDING ANY FAILURE OF ESSENTIAL PURPOSE OF
 * ANY LIMITED REMEDY.
 *
 *****************************************************************************/

#include "bcmnexus.h"
#include "bcmnexus_pool.h"
#include "bcmnexus_utils.h"

#include "nexus_base_mmap.h"

#include "directfb_util.h"

#include "fusion/conf.h"

#include "core/core.h"
#include "core/system.h"
#include "core/surface_pool.h"
#include "core/palette.h"
#include "core/gfxcard.h"

/* Uncomment out the line below to use Nexus memory that has been pre-allocated
   for creating surfaces... */
/*#define PREALLOCATE_MEMORY */

#define CYAN_COLOUR "\033[36m"
#define BLUE_COLOUR "\033[34m"
#define NO_COLOUR "\033[0m"

D_DEBUG_DOMAIN( bcmnexusPool,        "bcmNexus/Pool",        "Broadcom NEXUS Surface Pool" );
D_DEBUG_DOMAIN( bcmnexusPoolX,       "bcmNexus/PoolX",       "Broadcom NEXUS Surface Pool eXtended Debug" );
D_DEBUG_DOMAIN( bcmnexusPoolPalette, "bcmNexus/PoolPalette", "Broadcom NEXUS Surface Pool Palette Debug" );

struct bcmPoolData {
     FusionSHMPoolShared        *pool;              /* this is where the memory is */
     DFBBCMNEXUS                *bcmnexus;          /* shared struct to NEXUS handles e.a. */
     int                         refs;              /* Reference count */
};

struct bcmPoolLocal
{
     CoreDFB                    *core;
     struct bcmPoolData         *pd;
     int                         refs;              /* Reference count */
     DFB_Platform_P_PoolHandle   gfx3DPoolHandle;   /* Handle for the 3D graphics pool */
};


/**********************************************************************************************************************/
static DFBResult
createNexusSurface( CoreDFB                  *core,
                    CoreSurfaceAllocation    *allocation,
                    struct bcmAllocationData *ad,
                    struct bcmPoolLocal      *pl )
{
     struct bcmPoolData            *pd = pl->pd;
     CoreSurfaceBuffer             *buffer;
     CoreSurface                   *surface;
     CorePalette                   *palette;
     NEXUS_SurfaceCreateSettings    createSettings;
     NEXUS_MemoryAllocationSettings paletteMemorySettings;
     NEXUS_PixelFormat              eFormat;
     NEXUS_SurfaceMemory            memory;
     NEXUS_MemoryStatus             memoryStatus;
     CoreGraphicsDevice            *device;
     CardLimitations                limits;

     buffer  = allocation->buffer;

     device = dfb_core_get_part( core, DFCP_GRAPHICS );
     D_ASSERT(device != NULL);
     dfb_gfxcard_get_limits(device, &limits);
     if (!limits.surface_byteoffset_alignment)
         limits.surface_byteoffset_alignment = 32;

     surface = buffer->surface;
     palette = surface->palette;

     eFormat = bcmnexus_getNexusPixelFormat(buffer->format);

     if (eFormat == NEXUS_PixelFormat_eUnknown)
     {
        D_ERROR( "%s: unsupported format %s\n",__FUNCTION__, dfb_pixelformat_name(surface->config.format) );
        return DFB_UNSUPPORTED;
     }

     ad->gfx3DPoolHandle = pl->gfx3DPoolHandle;

     /* IMPORTANT: The GFD doesn't support 15 or 24-bit formats, so we need to make sure
        that the pixelformat for the primary layer isn't one of these formats. */
     if (surface->type & CSTF_LAYER)
     {
         if (eFormat == NEXUS_PixelFormat_eR8_G8_B8)
         {
            D_WARN("bcmNexus/Pool: GFD cannot support DSPF_RGB24, defaulting to DSPF_RGB32 instead.\n");
            eFormat = bcmnexus_getNexusPixelFormat(DSPF_RGB32);
         }

#if BCMNEXUS_NSC_SUPPORT == BCMNEXUS_NSC_TUNNELLED_MODE
         if (surface->resource_id == DLID_PRIMARY)
         {
             BM2MC_PACKET_Plane framebufferPlane;
             BERR_Code rc;
             rc = NEXUS_SurfaceClient_AcquireTunneledSurface(GET_DISPLAY_HND(pd->bcmnexus,0), &ad->bcmSurfaceHandle);
             D_DEBUG_AT( bcmnexusPool,"acquired tunnel surface %#lx\n", (unsigned long)ad->bcmSurfaceHandle);
             if ( rc != NEXUS_SUCCESS || ad->bcmSurfaceHandle==NULL )
             {
                 D_ERROR( "bcmNexus/Pool: NEXUS_SurfaceClient_GetTunneledSurface failed!\n" );
                 return DFB_LIMITEXCEEDED;
             }
             NEXUS_Surface_GetCreateSettings(ad->bcmSurfaceHandle, &createSettings);
             NEXUS_Surface_GetMemory(ad->bcmSurfaceHandle, &memory);
             if( eFormat != createSettings.pixelFormat, surface->config.size.w != createSettings.width  || surface->config.size.h != createSettings.height) {
                 D_ERROR( "bcmNexus/Pool: invalid data pixel:%u,%u width:%u,%u height:%u,%u pitch:%u,%u !",
                          eFormat, createSettings.pixelFormat, surface->config.size.w, createSettings.width,
                          surface->config.size.h, createSettings.height, ad->pitch, memory.pitch );
                 D_ERROR( "bcmNexus/Pool: invalid data!\n");
                 return DFB_LIMITEXCEEDED;
             }

             /* Stop trying to be clever and let NEXUS figure these out.*/
             NEXUS_Surface_InitPlane(ad->bcmSurfaceHandle, &framebufferPlane);
             /* store buffer info */
             ad->pitch        = framebufferPlane.pitch;
             ad->allocation   = allocation;
             ad->address      = memory.buffer;
             ad->pixelformat  = framebufferPlane.format;
             ad->offsetInHeap = 0;
             ad->hwaddress    = framebufferPlane.address;
             ad->size = createSettings.height * memory.pitch;

             return DFB_OK;
         }
#endif

         if (DFB_Platform_P_CheckGfxVScl(0) == false)
         {
             /* Check to see if we are creating the actual framebuffer.  If so, then this needs to
                have a maximum height of all possible TV resolutions as we don't have a vertical
                scaler in the graphics feeder path.
             */
             if (surface->resource_id == DLID_PRIMARY)
             {
                 surface->config.min_size.h = MAX(surface->config.min_size.h, BCMNEXUS_MAX_HD_DISPLAY_HEIGHT);
             }
         }

         if (DFB_Platform_P_CheckGfxVScl(1) == false)
         {
             /* Check to see if we are creating the actual framebuffer.  If so, then this needs to
                have a maximum height of all possible TV resolutions as we don't have a vertical
                scaler in the graphics feeder path.
             */
             if (surface->resource_id == (DLID_PRIMARY+1))
             {
                 surface->config.min_size.h = MAX(surface->config.min_size.h, BCMNEXUS_MAX_SD_DISPLAY_HEIGHT);
             }
         }
     }

     NEXUS_Surface_GetDefaultCreateSettings(&createSettings);


     dfb_gfxcard_calc_buffer_size( dfb_gfxcard_get_primary(), buffer, &ad->pitch, &ad->size );

     /* Always ensure that the palette is created in heap 0 for off-screen surfaces, as otherwise the surface won't be
        blitted correctly on platforms such as the 7420 where RTS dictates that the palette and the M2MC
        packet-buffer must be on heap 0. If the surface is the actual framebuffer (GFD), then the palette must be on
        heap 1 for the 7420, because there is no bandwidth allocated on memc0 (client 29 on mem0 is disabled).
     */
     if (DFB_PIXELFORMAT_IS_INDEXED(surface->config.format)) {
         NEXUS_Memory_GetDefaultAllocationSettings(&paletteMemorySettings);
         paletteMemorySettings.alignment = 32;  /* Palette needs to be aligned on a 32-byte boundary */

         if (surface->type & CSTF_LAYER)
             if (surface->resource_id == DLID_PRIMARY)
                 paletteMemorySettings.heap = GET_DISPLAY_HEAP_HND(pd->bcmnexus);
             else
                 paletteMemorySettings.heap = GET_2ND_DISPLAY_HEAP_HND(pd->bcmnexus);
         else
             paletteMemorySettings.heap = GET_PALETTE_HEAP_HND(pd->bcmnexus);

         /* From bsur.c:
            M2MC HW accesses palette as 1024 bytes, regardless of palette format.
            so we must alloc a constant 1024 to avoid ARC violations. */
         if (NEXUS_Memory_Allocate(1024, &paletteMemorySettings, &createSettings.pPaletteMemory) != NEXUS_SUCCESS)
         {
             D_ERROR("%s: Error creating Memory of %d bytes for palette!\n", __FUNCTION__, palette->num_entries*sizeof(NEXUS_Pixel));
             return DFB_NOSHAREDMEMORY;
         }

         /* Save address of palette for later */
         ad->palette = createSettings.pPaletteMemory;

         D_DEBUG_AT( bcmnexusPool, CYAN_COLOUR"%s() Palette memory=%p heap=%p [%d entries]\n"NO_COLOUR,
                     __FUNCTION__, (void *)createSettings.pPaletteMemory, (void *)paletteMemorySettings.heap, palette->num_entries);
     }

     /* If we have created a pre-allocated surface in BMEM, then we don't have to allocate any surface data */
     if ((surface->type & CSTF_PREALLOCATED) && (buffer->policy == CSP_VIDEOONLY))
     {
         DFB_PlatformNexusHandle heap;
         NEXUS_MemoryStatus      memoryStatus;
         int                     index;

#if DFB_BCM_TASK_MANAGER
         index = allocation->index;
#else
         index = dfb_surface_buffer_index( buffer );
#endif

         D_ASSERT( surface->config.preallocated[index].handle != NULL );
         D_ASSERT( surface->config.preallocated[index].pitch != 0 );

         heap = surface->config.preallocated[index].handle;

         if (NEXUS_Heap_GetStatus( heap, &memoryStatus ) != NEXUS_SUCCESS)
             return DFB_FAILURE;

         /* Update the pitch to the actual provided one... */
         ad->pitch                = surface->config.preallocated[index].pitch;

         createSettings.height    = surface->config.size.h;
         createSettings.pMemory   = (u8*) memoryStatus.addr + surface->config.preallocated[index].offset;
         createSettings.heap      = heap;
     }
     else {
         createSettings.height    = MAX(surface->config.size.h, surface->config.min_size.h);

         /* This surface is allocated on video layer? */
         if ((surface->type & CSTF_LAYER) && (surface->resource_id == (DLID_PRIMARY+2))) {
             createSettings.heap = GET_VIDEO_HEAP_HND(pd->bcmnexus);
         }
     }
     createSettings.pixelFormat    = eFormat;
     createSettings.width          = surface->config.size.w;
     createSettings.pitch          = ad->pitch;
     createSettings.alignment      = 12; /* Ensure surface is aligned on 4096Kbyte boundary to help prevent "cache crosstalk" */

    /*4-byte alignment for odd width image decode with SID.*/
    createSettings.pitch = (createSettings.pitch + 0x3) & ~(0x3);

     if (createSettings.heap == NULL)
     {
          if(buffer->surface->config.caps & DSCAPS_GL)
          {
              createSettings.heap = GET_OFFSCREEN_HEAP_HND(pd->bcmnexus);
          }
          else if (surface->type & CSTF_LAYER)
          {
            /* Layer allocations must be from the display heap */
              createSettings.heap = (surface->resource_id == DLID_PRIMARY) ? GET_DISPLAY_HEAP_HND(pd->bcmnexus) : GET_2ND_DISPLAY_HEAP_HND(pd->bcmnexus);
          }
          else
              createSettings.heap = GET_OFFSCREEN_HEAP_HND(pd->bcmnexus);

          if (createSettings.heap == NULL)
          {
              D_ERROR( "bcmNexus/Pool: Invalid heap for surface allocation\n");
              return DFB_NOVIDEOMEMORY;
          }

          /* Check that we can actually allocate this amount of memory from our desired pool.
           * If not fall back to the secondry pool if possible */
          if (NEXUS_Heap_GetStatus(createSettings.heap, &memoryStatus) != NEXUS_SUCCESS)
               return DFB_FAILURE;

              if (memoryStatus.largestFreeBlock < (unsigned int)ad->size)
              {
                  D_ERROR( "bcmNexus/Pool: Insufficient memory to create surface (%d bytes short) - %s heap\n",
                           ad->size - memoryStatus.largestFreeBlock,
                       (createSettings.heap == GET_OFFSCREEN_HEAP_HND(pd->bcmnexus)) ? "offscreen" : "primary");
              return DFB_NOVIDEOMEMORY;
          }
     }
#ifdef PREALLOCATE_MEMORY
     if (createSettings.pMemory == NULL)
     {
          NEXUS_MemoryAllocationSettings surfaceMemorySettings;
          NEXUS_Memory_GetDefaultAllocationSettings(&surfaceMemorySettings);
          surfaceMemorySettings.alignment = 4096;  /* Surface needs to be aligned on a 4K boundary */
          surfaceMemorySettings.heap = createSettings.heap;

          if (NEXUS_Memory_Allocate((size_t)ad->size, &surfaceMemorySettings, &createSettings.pMemory) != NEXUS_SUCCESS)
          {
              D_ERROR( "bcmNexus/Pool: Insufficient preallocated memory to create surface (%d bytes)\n", ad->size);
              return DFB_NOVIDEOMEMORY;
          }
     }
#endif

     ad->bcmSurfaceHandle = NEXUS_Surface_Create(&createSettings);
     if (ad->bcmSurfaceHandle==NULL) {
         D_ERROR( "bcmNexus/Pool: NEXUS_Surface_Create failed!\n" );
         return DFB_NOVIDEOMEMORY;
     }

     NEXUS_Surface_GetMemory(ad->bcmSurfaceHandle, &memory);

     if (memory.buffer == NULL)
     {
         D_ERROR("Failed to get buffer information for surface %p\n", (void *)ad->bcmSurfaceHandle);
         return DFB_NOVIDEOMEMORY;
     }

     /* store buffer info */
     ad->allocation      = allocation;
     ad->address         = memory.buffer;
     ad->pixelformat     = createSettings.pixelFormat;
     ad->pitch = createSettings.pitch;

     D_ASSERT(createSettings.heap);

     if (NEXUS_Heap_GetStatus(createSettings.heap, &memoryStatus) != NEXUS_SUCCESS)
         return DFB_FAILURE;

     ad->heap         = createSettings.heap;
     ad->offsetInHeap = (uint8_t*)memory.buffer - (uint8_t*)memoryStatus.addr;
     ad->hwaddress    = memoryStatus.offset + ad->offsetInHeap;
#ifdef BCMNEXUS_GFX_PACKET_BUFFER
     NEXUS_Surface_InitPlane(ad->bcmSurfaceHandle, &ad->surfacePlane);
#endif
     D_DEBUG_AT( bcmnexusPool, CYAN_COLOUR"%s: Nexus_Surface_Created() %s: sur handle %p vaddr=%p hwaddr=%p [heap: handle %p, vaddr=%p hwaddr=%p]"
                               ", WxH=%dx%d, pitch=%d bytes, size=%d bytes, dfb_pixelformat=%s\n"NO_COLOUR, __FUNCTION__,
                               (surface->type & CSTF_LAYER) ? "framebuffer" : "offscreen",
                               (void*)(ad->bcmSurfaceHandle), (void *)ad->address, (void *)ad->hwaddress,
                               (void *)ad->heap, (void *)memoryStatus.addr, (void *)memoryStatus.offset,
                               createSettings.width, createSettings.height, ad->pitch, ad->size, dfb_pixelformat_name(buffer->format) );

     /* Load palette if it has one */
     if (DFB_PIXELFORMAT_IS_INDEXED(surface->config.format))
     {
          unsigned int i;

          D_ASSERT(paletteMemorySettings.heap);

          if (NEXUS_Heap_GetStatus(paletteMemorySettings.heap, &memoryStatus) != NEXUS_SUCCESS)
              return DFB_FAILURE;

          ad->hwpalette = memoryStatus.offset + (uint8_t*)memory.palette - (uint8_t*)memoryStatus.addr;

          for (i=0; i<palette->num_entries; i++)
          {
               memory.palette[i] = PIXEL_ARGB(
                    palette->entries[i].a,
                    palette->entries[i].r,
                    palette->entries[i].g,
                    palette->entries[i].b);
               D_DEBUG_AT( bcmnexusPoolPalette, CYAN_COLOUR"Palette entry %d: 0x%08x\n"NO_COLOUR, i, memory.palette[i] );
          }
          memory.numPaletteEntries = palette->num_entries;
          D_DEBUG_AT( bcmnexusPool, CYAN_COLOUR"Palette copied to %p [num entries=%d]\n"NO_COLOUR, (void *)memory.palette, memory.numPaletteEntries );
     }

     return DFB_OK;
}

/**********************************************************************************************************************/

static int bcmPoolDataSize( void )
{
     return sizeof(struct bcmPoolData);
}

static int bcmPoolLocalDataSize( void )
{
    return sizeof(struct bcmPoolLocal);
}

static int bcmAllocationDataSize( void )
{
     return sizeof(struct bcmAllocationData);
}

static DFBResult
bcmInitPoolCommon( CoreDFB             *core,
                   struct bcmPoolData  *pd,
                   struct bcmPoolLocal *pl,
                   DFBBCMNEXUS         *bcmnexus )
{
     DFBResult  result = DFB_OK;

     pl->core     = core;
     pd->pool     = dfb_core_shmpool( core );
     pd->bcmnexus = bcmnexus;
     pl->pd       = pd;

     return result;
}

static DFBResult
bcmInitPool( CoreDFB                    *core,
             CoreSurfacePool            *pool,
             void                       *pool_data,
             void                       *pool_local,
             void                       *system_data,
             CoreSurfacePoolDescription *ret_desc )
{
     DFBResult            result;
     struct bcmPoolData  *pd = pool_data;
     struct bcmPoolLocal *pl = pool_local;

     D_DEBUG_AT( bcmnexusPool, CYAN_COLOUR"%s()\n"NO_COLOUR, __FUNCTION__ );

     D_MAGIC_ASSERT( pool, CoreSurfacePool );
     D_ASSERT( ret_desc != NULL );

     snprintf( ret_desc->name, DFB_SURFACE_POOL_DESC_NAME_LENGTH, "BRCM Surface Pool" );

     result = bcmInitPoolCommon(core, pd, pl, system_data);

     if (result == DFB_OK)
     {
         ret_desc->caps = CSPCAPS_VIRTUAL | CSPCAPS_PHYSICAL;

         ret_desc->access[CSAID_CPU]    = CSAF_READ | CSAF_WRITE | CSAF_SHARED;
         ret_desc->access[CSAID_GPU]    = CSAF_READ | CSAF_WRITE | CSAF_SHARED;
         ret_desc->access[CSAID_LAYER0] = CSAF_READ;
         ret_desc->access[CSAID_LAYER1] = CSAF_READ;
         ret_desc->access[CSAID_LAYER2] = CSAF_READ;
         ret_desc->access[CSAID_LAYER3] = CSAF_READ;
         ret_desc->access[CSAID_LAYER4] = CSAF_READ;
         ret_desc->access[CSAID_LAYER5] = CSAF_READ;
         ret_desc->access[CSAID_LAYER6] = CSAF_READ;
         ret_desc->access[CSAID_LAYER7] = CSAF_READ;

         ret_desc->types    = CSTF_LAYER | CSTF_WINDOW | CSTF_CURSOR | CSTF_FONT | CSTF_SHARED | CSTF_EXTERNAL | CSTF_PREALLOCATED;
         ret_desc->priority = CSPP_PREFERED;
     }

     /* TODO: Add size field */

     return result;
}

static DFBResult
bcmInitGraphics3DPool( CoreDFB                    *core,
                       CoreSurfacePool            *pool,
                       void                       *pool_data,
                       void                       *pool_local,
                       void                       *system_data,
                       CoreSurfacePoolDescription *ret_desc )
{
     DFBResult                                 result;
     struct bcmPoolData                       *pd = pool_data;
     struct bcmPoolLocal                      *pl = pool_local;

     D_DEBUG_AT( bcmnexusPool, CYAN_COLOUR"%s()\n"NO_COLOUR, __FUNCTION__ );

     D_MAGIC_ASSERT( pool, CoreSurfacePool );
     D_ASSERT( ret_desc != NULL );

     snprintf( ret_desc->name, DFB_SURFACE_POOL_DESC_NAME_LENGTH, "BRCM Graphics3D Surface Pool" );

     result = bcmInitPoolCommon(core, pd, pl, system_data);

     if (result == DFB_OK)
     {
         /* We are intentionally not checking the return code of this function so that the register pool
            in bcmnexus.c always succeeds. We check in the test if the pool handle is valid instead */
         DFB_Platform_P_Graphics3D_PoolInit( &(pl->gfx3DPoolHandle));

         ret_desc->caps = CSPCAPS_VIRTUAL | CSPCAPS_PHYSICAL;

         ret_desc->access[CSAID_CPU]    = CSAF_READ | CSAF_WRITE | CSAF_SHARED;
         ret_desc->access[CSAID_GPU]    = CSAF_READ | CSAF_WRITE | CSAF_SHARED;
         ret_desc->access[CSAID_LAYER0] = CSAF_READ;
         ret_desc->access[CSAID_LAYER1] = CSAF_READ;
         ret_desc->access[CSAID_LAYER2] = CSAF_READ;
         ret_desc->access[CSAID_LAYER3] = CSAF_READ;
         ret_desc->access[CSAID_LAYER4] = CSAF_READ;
         ret_desc->access[CSAID_LAYER5] = CSAF_READ;
         ret_desc->access[CSAID_LAYER6] = CSAF_READ;
         ret_desc->access[CSAID_LAYER7] = CSAF_READ;

         ret_desc->types    = CSTF_LAYER | CSTF_WINDOW | CSTF_SHARED | CSTF_EXTERNAL;
         ret_desc->priority = CSPP_PREFERED;
     }

     /* TODO: Add size field */

     return result;
}


static DFBResult
bcmJoinPool( UNUSED_ CoreDFB         *core,
                     CoreSurfacePool *pool,
             UNUSED_ void            *pool_data,
             UNUSED_ void            *pool_local,
             UNUSED_ void            *system_data )
{
     D_DEBUG_AT( bcmnexusPool, CYAN_COLOUR"%s()\n"NO_COLOUR, __FUNCTION__ );

     D_MAGIC_ASSERT( pool, CoreSurfacePool );

     return DFB_OK;
}

static DFBResult
bcmJoinGraphics3DPool( UNUSED_ CoreDFB         *core,
                               CoreSurfacePool *pool,
                       UNUSED_ void            *pool_data,
                               void            *pool_local,
                       UNUSED_ void            *system_data )
{
     struct bcmPoolLocal                      *pl = pool_local;

     pl->gfx3DPoolHandle = NULL;

     D_DEBUG_AT( bcmnexusPool, CYAN_COLOUR"%s()\n"NO_COLOUR, __FUNCTION__ );

     D_MAGIC_ASSERT( pool, CoreSurfacePool );

     /* We are intentionally not checking the return code of this function so that the join pool
        in bcmnexus.c always succeeds. We check in the test if the pool handle is valid instead */
     DFB_Platform_P_Graphics3D_PoolInit( &(pl->gfx3DPoolHandle));

     return DFB_OK;

}

static DFBResult
bcmDestroyPool( CoreSurfacePool *pool,
       UNUSED_  void            *pool_data,
       UNUSED_  void            *pool_local )
{
     D_DEBUG_AT( bcmnexusPool, CYAN_COLOUR"%s()\n"NO_COLOUR, __FUNCTION__ );

     D_MAGIC_ASSERT( pool, CoreSurfacePool );

     return DFB_OK;
}

static DFBResult
bcmDestroyGraphics3DPool( CoreSurfacePool *pool,
                 UNUSED_  void            *pool_data,
                          void            *pool_local )
{
     struct bcmPoolLocal *pl = pool_local;

     D_DEBUG_AT( bcmnexusPool, CYAN_COLOUR"%s()\n"NO_COLOUR, __FUNCTION__ );

     D_MAGIC_ASSERT( pool, CoreSurfacePool );

     if (pl->gfx3DPoolHandle)
     {
        DFB_Platform_P_Graphics3D_PoolUninit(pl->gfx3DPoolHandle);
        pl->gfx3DPoolHandle = NULL;
     }

     return DFB_OK;
}

static DFBResult
bcmLeavePool( CoreSurfacePool *pool,
      UNUSED_ void            *pool_data,
      UNUSED_ void            *pool_local )
{
     D_DEBUG_AT( bcmnexusPool, CYAN_COLOUR"%s()\n"NO_COLOUR, __FUNCTION__ );

     D_MAGIC_ASSERT( pool, CoreSurfacePool );

     return DFB_OK;
}

static DFBResult
bcmLeaveGraphics3DPool( CoreSurfacePool *pool,
               UNUSED_  void            *pool_data,
                        void            *pool_local )
{
     struct bcmPoolLocal *pl = pool_local;

     D_DEBUG_AT( bcmnexusPool, CYAN_COLOUR"%s()\n"NO_COLOUR, __FUNCTION__ );

     D_MAGIC_ASSERT( pool, CoreSurfacePool );

     if (pl->gfx3DPoolHandle)
     {
        DFB_Platform_P_Graphics3D_PoolUninit(pl->gfx3DPoolHandle);
        pl->gfx3DPoolHandle = NULL;
     }

     return DFB_OK;
}

static DFBResult
bcmTestConfig( CoreSurfacePool         *pool,
     UNUSED_   void                    *pool_data,
     UNUSED_   void                    *pool_local,
               CoreSurfaceBuffer       *buffer,
               const CoreSurfaceConfig *config )
{
     NEXUS_PixelFormat        eFormat;
     DFBSurfaceCapabilities   caps;
     DFBResult                result    = DFB_OK;

     D_MAGIC_ASSERT( pool, CoreSurfacePool );
     D_MAGIC_ASSERT( buffer, CoreSurfaceBuffer );
     D_ASSERT( config != NULL );

     caps = buffer->surface->config.caps;

     D_DEBUG_AT( bcmnexusPool, CYAN_COLOUR"%s() format=%s, type=0x%08x, policy=0x%08x, caps=0x%08x\n"NO_COLOUR, __FUNCTION__,
                 dfb_pixelformat_name(config->format), buffer->type, buffer->policy, caps );

     eFormat = bcmnexus_getNexusPixelFormat(config->format);

     if (eFormat == NEXUS_PixelFormat_eUnknown)
     {
        D_DEBUG_AT( bcmnexusPool, CYAN_COLOUR"%s: unsupported format %s\n"NO_COLOUR, __FUNCTION__, dfb_pixelformat_name(config->format) );
        result = DFB_UNSUPPORTED;
     }

     /* We only support hardware acceleration of preallocated data from the external video memory */
     if (D_FLAGS_IS_SET(buffer->type, CSTF_PREALLOCATED) && !D_FLAGS_IS_SET(buffer->type, CSTF_EXTERNAL))
     {
          D_DEBUG_AT( bcmnexusPool, "%s: cannot support preallocated surface data from internal memory!\n", __FUNCTION__ );
          result = DFB_UNSUPPORTED;
     }

     if (caps & DSCAPS_GL)
     {
        D_DEBUG_AT( bcmnexusPool, CYAN_COLOUR"%s: unsupported surface caps 0x%08x\n"NO_COLOUR, __FUNCTION__, caps );
        result = DFB_UNSUPPORTED;
     }

     return result;
}

static DFBResult
bcmTestGraphics3DConfig( CoreSurfacePool         *pool,
                 UNUSED_ void                    *pool_data,
                 UNUSED_ void                    *pool_local,
                         CoreSurfaceBuffer       *buffer,
                         const CoreSurfaceConfig *config )
{
     NEXUS_PixelFormat        eFormat;
     DFBSurfaceCapabilities   caps;
     DFBResult                result    = DFB_OK;
     struct bcmPoolLocal     *pl        = pool_local;


     D_MAGIC_ASSERT( pool, CoreSurfacePool );
     D_MAGIC_ASSERT( buffer, CoreSurfaceBuffer );
     D_ASSERT( config != NULL );

     caps = buffer->surface->config.caps;

     D_DEBUG_AT( bcmnexusPool, CYAN_COLOUR"%s() format=%s, caps=0x%08x\n"NO_COLOUR, __FUNCTION__, dfb_pixelformat_name(config->format), caps );

     eFormat = bcmnexus_getNexusPixelFormat(config->format);

     if (eFormat == NEXUS_PixelFormat_eUnknown || DFB_PIXELFORMAT_IS_INDEXED(config->format))
     {
        D_DEBUG_AT( bcmnexusPool, CYAN_COLOUR"%s: unsupported format %s\n"NO_COLOUR, __FUNCTION__, dfb_pixelformat_name(config->format) );
        result = DFB_UNSUPPORTED;
     }

     if (!(caps & DSCAPS_GL))
     {
        D_DEBUG_AT( bcmnexusPool, CYAN_COLOUR"%s: unsupported surface caps 0x%08x\n"NO_COLOUR, __FUNCTION__, caps );
        result = DFB_UNSUPPORTED;
     }

     if (!(pl->gfx3DPoolHandle))
     {
        D_DEBUG_AT( bcmnexusPool, CYAN_COLOUR"%s: no valid platform pool!\n"NO_COLOUR, __FUNCTION__);
        result = DFB_UNSUPPORTED;
     }

     return result;
}

static DFBResult
bcmPreAlloc( CoreSurfacePool             *pool,
     UNUSED_ void                        *pool_data,
     UNUSED_ void                        *pool_local,
             const DFBSurfaceDescription *description,
             CoreSurfaceConfig           *config )
{
     unsigned int i, num = 1;

     D_MAGIC_ASSERT( pool, CoreSurfacePool );

     D_DEBUG_AT( bcmnexusPool,
                 CYAN_COLOUR"%s() description=%p, config=%p [master=%c]\n"NO_COLOUR,
                 __FUNCTION__, (void*) description, (void*) config, dfb_core_is_master( dfb_bcmnexus_core ) ? 'y' : 'n' );

     if (config->caps & DSCAPS_DOUBLE)
          num = 2;
     else if (config->caps & DSCAPS_TRIPLE)
          num = 3;

     if (!(config->caps & DSCAPS_VIDEOONLY))
          return DFB_UNSUPPORTED;

     for (i=0; i<num; i++) {
          DFB_PlatformNexusHandle heap;
          NEXUS_MemoryStatus      memoryStatus;

          heap = DFB_Platform_P_GetGraphicsHeap( description->preallocated[i].data );
          if (!heap) {
               D_ERROR( "bcmNexus/Pool: Failed to determine heap for surface pre-allocation (address %p)!\n",
                        description->preallocated[i].data );
               return DFB_UNSUPPORTED;
          }

          if (NEXUS_Heap_GetStatus( heap, &memoryStatus ) != NEXUS_SUCCESS)
              return DFB_FAILURE;

          config->preallocated[i].handle = heap;
          config->preallocated[i].offset = (long) description->preallocated[i].data - (long) memoryStatus.addr;
          config->preallocated[i].pitch  = description->preallocated[i].pitch;
     }

     return DFB_OK;
}

static DFBResult
bcmAllocateBuffer( CoreSurfacePool       *pool,
          UNUSED_  void                  *pool_data,
                   void                  *pool_local,
                   CoreSurfaceBuffer     *buffer,
                   CoreSurfaceAllocation *allocation,
                   void                  *alloc_data )
{
     DFBResult                     result;
     struct bcmAllocationData     *ad   = alloc_data;
     struct bcmPoolLocal          *pl   = pool_local;

     D_MAGIC_ASSERT( pool, CoreSurfacePool );
     D_MAGIC_ASSERT( buffer, CoreSurfaceBuffer );
     D_MAGIC_ASSERT( allocation, CoreSurfaceAllocation );

     D_DEBUG_AT( bcmnexusPool,
                 CYAN_COLOUR"%s() surface=%p, buffer=%p, allocation=%p, surface type=0x%08x, caps=0x%08x, size=%dx%d, format=%s, policy=0x%08x [master=%c]\n"
                 NO_COLOUR, __FUNCTION__, (void *)buffer->surface, (void *)buffer,  (void *)allocation, allocation->type, allocation->config.caps,
                 allocation->config.size.w, allocation->config.size.h, dfb_pixelformat_name(buffer->format), buffer->policy,
                 dfb_core_is_master( dfb_bcmnexus_core ) ? 'y' : 'n' );

     /* allocate the NEXUS surface */
     result = createNexusSurface( pl->core, allocation, ad, pl );
     if (result == DFB_OK) {
          allocation->size = ad->size;
          if (!((buffer->type & CSTF_LAYER) && (buffer->resource_id == (DLID_PRIMARY+2))))
          {
              D_DEBUG_AT( bcmnexusPool, CYAN_COLOUR"%s() allocation=%d, flush handle=%p\n"NO_COLOUR, __FUNCTION__,
                         allocation->size, (void *)ad->bcmSurfaceHandle);

              NEXUS_Surface_Flush(ad->bcmSurfaceHandle);
          }
     }
     return result;
}

static DFBResult
bcmDeallocateBuffer( CoreSurfacePool       *pool,
                     void                  *pool_data,
             UNUSED_ void                  *pool_local,
             UNUSED_ CoreSurfaceBuffer     *buffer,
                     CoreSurfaceAllocation *allocation,
                     void                  *alloc_data )
{
     struct bcmAllocationData     *ad   = alloc_data;
     struct bcmPoolData           *pd   = pool_data;

     D_MAGIC_ASSERT( pool, CoreSurfacePool );
     D_MAGIC_ASSERT( allocation, CoreSurfaceAllocation );

     D_DEBUG_AT( bcmnexusPool,
                 CYAN_COLOUR"%s() handle=%p [addr: %p], type=0x%08x, size=%dx%d, format=%s [master=%c]\n"NO_COLOUR,
                 __FUNCTION__, (void *)ad->bcmSurfaceHandle, ad->address, allocation->type,
                 allocation->config.size.w, allocation->config.size.h, dfb_pixelformat_name(allocation->config.format),
                 dfb_core_is_master( dfb_bcmnexus_core ) ? 'y' : 'n' );

     /* deallocate the NEXUS surface */
     if (ad->bcmSurfaceHandle) {

#if BCMNEXUS_NSC_SUPPORT == BCMNEXUS_NSC_MULTIBUFFER_MODE
         if (allocation &&
                allocation->type & CSTF_LAYER
                && allocation->resource_id == DLID_PRIMARY)
         {
             D_DEBUG_AT( bcmnexusPool,
                         CYAN_COLOUR"%s() Clearing NSC queue \n"NO_COLOUR,__FUNCTION__);
             DFB_Platform_P_ClearDisplayGraphicsFramebuffer3D(GET_DISPLAY_HND(pd->bcmnexus, PRIMARY_DISPLAY_ID));
         }
#endif

#if BCMNEXUS_NSC_SUPPORT == BCMNEXUS_NSC_TUNNELLED_MODE
         if (allocation &&
                allocation->type & CSTF_LAYER
                && allocation->resource_id == DLID_PRIMARY)
         {
             D_DEBUG_AT( bcmnexusPool, "releasing tunnel surface %#lx\n", (unsigned long)ad->bcmSurfaceHandle);
             NEXUS_SurfaceClient_ReleaseTunneledSurface(GET_DISPLAY_HND(pd->bcmnexus,0),ad->bcmSurfaceHandle);
             /* BDBG_ASSERT(!rc); */
             ad->bcmSurfaceHandle = NULL;
             goto done;
         }
#endif

#if BCM_DFB_USE_TASK_MANAGER
         if (!dfb_config->task_manager)
         {
          /* Ensure the graphics h/w has finished before removing memory! */
             if ((ad->syncIssued == false) || GET_GRAPHICS2Dn_SYNC(pd->bcmnexus, 0))
          {
              D_DEBUG_AT( bcmnexusPool, CYAN_COLOUR"%s() sync\n"NO_COLOUR, __FUNCTION__);
              dfb_gfxcard_sync();
          } else {
              ad->syncIssued = false;
          }

             if (!((allocation->type & CSTF_LAYER) && (allocation->resource_id == (DLID_PRIMARY+2))))
          {
              D_DEBUG_AT( bcmnexusPool, CYAN_COLOUR"%s() flush handle=%p\n"NO_COLOUR, __FUNCTION__, (void *)ad->bcmSurfaceHandle);
              NEXUS_Surface_Flush(ad->bcmSurfaceHandle);
          }
         }
#endif

          /* Free up the palette if it was created */
          if (ad->palette != NULL) {
              D_DEBUG_AT( bcmnexusPool, CYAN_COLOUR"%s() About to free palette memory %p\n"NO_COLOUR, __FUNCTION__, (void *)ad->palette );
              NEXUS_Memory_Free(ad->palette);
              ad->palette = NULL;
          }

          /* If not a pre-allocated surface, then we can free up the Nexus memory */
#ifdef PREALLOCATE_MEMORY
          if (!(allocation->type & CSTF_PREALLOCATED))
          {
              NEXUS_SurfaceMemory mem;

              D_DEBUG_AT( bcmnexusPool, CYAN_COLOUR"%s() About to get surface handle %p memory\n"NO_COLOUR, __FUNCTION__, (void *)ad->bcmSurfaceHandle );
              NEXUS_Surface_GetMemory(ad->bcmSurfaceHandle, &mem);
              D_DEBUG_AT( bcmnexusPool, CYAN_COLOUR"%s() About to free buffer memory %p\n"NO_COLOUR, __FUNCTION__, (void *)mem.buffer );
              NEXUS_Memory_Free(mem.buffer);
          }
#endif
          NEXUS_Surface_Destroy(ad->bcmSurfaceHandle );


          ad->bcmSurfaceHandle = NULL;
     }

#if BCMNEXUS_NSC_SUPPORT == BCMNEXUS_NSC_TUNNELLED_MODE
  done:
#endif

     D_DEBUG_AT( bcmnexusPool, CYAN_COLOUR"%s() exiting.\n"NO_COLOUR, __FUNCTION__ );
     return DFB_OK;
}

static DFBResult
bcmPreLock(         CoreSurfacePool        *pool,
                    void                   *pool_data,
            UNUSED_ void                   *pool_local,
                    CoreSurfaceAllocation  *allocation,
                    void                   *alloc_data,
                    CoreSurfaceAccessorID   accessor,
                    CoreSurfaceAccessFlags  access )
{
    struct bcmPoolData       *pd       = pool_data;
    struct bcmAllocationData *ad       = alloc_data;
    bool                      flush_cache = false;
    bool                      gfx_sync  = false;
    int                       locks;

    D_MAGIC_ASSERT( pool, CoreSurfacePool );
    D_MAGIC_ASSERT( allocation, CoreSurfaceAllocation );

    locks = dfb_surface_allocation_locks( allocation );

    D_DEBUG_AT( bcmnexusPoolX, BLUE_COLOUR"%s() surface=%p, allocation=%p, addr=%p, pitch=%d, handle=%p accessor=%d, access=0x%08x, CPU accessed=0x%08x, "
                "GPU accessed=0x%08x, type=0x%08x, flags=0x%08x, caps=0x%08x, locks=%d\n"NO_COLOUR, __FUNCTION__,
                (void *)allocation->surface, (void *)allocation, ad->address, ad->pitch, (void*)ad->bcmSurfaceHandle, accessor, access,
                allocation->accessed[CSAID_CPU], allocation->accessed[CSAID_GPU], allocation->type, allocation->config.flags,
                allocation->config.caps, locks );

    if  (accessor == CSAID_GPU)
    {
        // Previous SW write ?
        if (allocation->accessed[CSAID_CPU] & CSAF_WRITE)
        {
            flush_cache = true;

            if (!locks)
                allocation->accessed[CSAID_CPU] &= ~CSAF_WRITE;
        }
    }
    else if (accessor == CSAID_CPU)
    {
        // Check for previous HW access
        if (allocation->accessed[CSAID_GPU] & CSAF_WRITE)
        {

            gfx_sync = true; // Make sure HW op has finished
            flush_cache = true; // SW read after GPU write may need cache to be invalidated

            if (!locks)
                allocation->accessed[CSAID_GPU] &= ~( CSAF_WRITE | CSAF_READ); // clear read to save double sync
        }

        // SW write operation
        if (access & CSAF_WRITE)
        {
            // Has the GPU accessed ?
            if (allocation->accessed[CSAID_GPU] & CSAF_READ)
            {
                gfx_sync = true; // wait for op to finish

                if (!locks)
                    allocation->accessed[CSAID_GPU] &= ~CSAF_READ;
            }
        }
    }
    else if ( accessor == (CSAID_LAYER0 + allocation->resource_id)) // primary layer access
    {
        // Check if SW has written into buffer
        if (allocation->accessed[CSAID_CPU] & CSAF_WRITE)
        {
            flush_cache = true;

            if (!locks)
                allocation->accessed[CSAID_CPU] &= ~CSAF_WRITE;
        }
        // Next HW access
        if (allocation->accessed[CSAID_GPU] & CSAF_WRITE)
        {
            gfx_sync = flush_cache = true; // sync GFX and flush cache

            if (!locks)
                allocation->accessed[CSAID_GPU] &= ~CSAF_WRITE;
        }
    }

    // Do we need to perform a graphics checkpoint ?
     if (gfx_sync == true)
     {
         if ((ad->syncIssued == false) || GET_GRAPHICS2Dn_SYNC(pd->bcmnexus, 0))
         {
             D_DEBUG_AT( bcmnexusPoolX, BLUE_COLOUR"%s() sync\n"NO_COLOUR, __FUNCTION__);
             dfb_gfxcard_sync();
         }
         else
             ad->syncIssued = false;
     }

     // Do we need to flush the surface cache?
     if (flush_cache == true) {
          if (!((allocation->type & CSTF_LAYER) && (allocation->resource_id == (DLID_PRIMARY+2))))
          {
              D_DEBUG_AT( bcmnexusPoolX, BLUE_COLOUR"%s() flush handle=%p\n"NO_COLOUR, __FUNCTION__, (void *)ad->bcmSurfaceHandle);
              NEXUS_Surface_Flush(ad->bcmSurfaceHandle);
          }
     }

     D_DEBUG_AT( bcmnexusPoolX, BLUE_COLOUR"%s() Exit\n"NO_COLOUR, __FUNCTION__ );
     return DFB_OK;

}

#if BCM_DFB_USE_TASK_MANAGER
static DFBResult
bcmCacheOp( CoreSurfacePool        *pool,
            UNUSED_ void           *pool_data,
            UNUSED_ void           *pool_local,
            CoreSurfaceAllocation  *allocation,
            void                   *alloc_data,
            CoreSurfaceAccessorID   accessor,
            UNUSED_ bool            flush,
            UNUSED_ bool            invalidate )
{
     struct bcmAllocationData *ad = alloc_data;

     D_MAGIC_ASSERT( pool, CoreSurfacePool );
     D_MAGIC_ASSERT( allocation, CoreSurfaceAllocation );

     D_DEBUG_AT( bcmnexusPoolX, BLUE_COLOUR"%s() surface=%p, allocation=%p, addr=%p, pitch=%d, handle=%p accessor=%d, access=0x%08x, CPU accessed=0x%08x, "
                 "GPU accessed=0x%08x, type=0x%08x, flags=0x%08x, caps=0x%08x\n"NO_COLOUR, __FUNCTION__,
                 (void *)allocation->surface, (void *)allocation, ad->address, ad->pitch, (void*)ad->bcmSurfaceHandle, accessor, (unsigned int)access,
                 allocation->accessed[CSAID_CPU], allocation->accessed[CSAID_GPU], allocation->type, allocation->config.flags,
                 allocation->config.caps );

     if (accessor == CSAID_CPU)
     {
     if (!((allocation->type & CSTF_LAYER) && (allocation->resource_id == (DLID_PRIMARY+2))))
     {
         D_DEBUG_AT( bcmnexusPoolX, BLUE_COLOUR"%s() flush handle=%p\n"NO_COLOUR, __FUNCTION__, (void *)ad->bcmSurfaceHandle);
         NEXUS_Surface_Flush(ad->bcmSurfaceHandle);
          }
     }

     D_DEBUG_AT( bcmnexusPoolX, BLUE_COLOUR"%s() Exit\n"NO_COLOUR, __FUNCTION__ );
     return DFB_OK;
}
#endif

static DFBResult
bcmLock(         CoreSurfacePool       *pool,
         UNUSED_ void                  *pool_data,
         UNUSED_ void                  *pool_local,
                 CoreSurfaceAllocation *allocation,
                 void                  *alloc_data,
                 CoreSurfaceBufferLock *lock )
{
     struct bcmAllocationData *ad       = alloc_data;

     D_MAGIC_ASSERT( pool, CoreSurfacePool );
     D_MAGIC_ASSERT( allocation, CoreSurfaceAllocation );
     D_MAGIC_ASSERT( lock, CoreSurfaceBufferLock );

     D_DEBUG_AT( bcmnexusPoolX, BLUE_COLOUR"%s() surface=%p, allocation=%p, addr=%p, pitch=%d, handle=%p accessor=%d, access=0x%08x, CPU accessed=0x%08x, "
                                "GPU accessed=0x%08x, type=0x%08x, flags=0x%08x, caps=0x%08x\n"NO_COLOUR,
                 __FUNCTION__, (void *)allocation->surface, (void *)allocation, ad->address, ad->pitch, (void*)ad->bcmSurfaceHandle, lock->accessor,
                 lock->access, allocation->accessed[CSAID_CPU], allocation->accessed[CSAID_GPU], allocation->type, allocation->config.flags, allocation->config.caps );

     /* Setup handle to surface so that graphics driver can access it */
     lock->handle = ad->bcmSurfaceHandle;
     lock->addr   = ad->address;
     lock->phys   = (unsigned long)ad->hwaddress;
     lock->offset = 0; /* irrelevant */
     lock->pitch  = ad->pitch; /* ad->pitch */


     /* For secure fusion applications we need to do some address mapping as the virtual address space for each process is different */
     if (fusion_config->secure_fusion && !dfb_core_is_master(dfb_bcmnexus_core) )
     {
         lock->addr = NEXUS_OffsetToCachedAddr(ad->hwaddress);

         D_DEBUG_AT( bcmnexusPoolX, BLUE_COLOUR"%s() Secure fusion client fixing up virtual address from %p to %p \n"NO_COLOUR,
                     __FUNCTION__, ad->address, lock->addr);
     }

     D_DEBUG_AT( bcmnexusPoolX, BLUE_COLOUR"%s() Exit\n"NO_COLOUR, __FUNCTION__ );
     return DFB_OK;

}

static DFBResult
bcmUnlock(         CoreSurfacePool       *pool,
           UNUSED_ void                  *pool_data,
           UNUSED_ void                  *pool_local,
                   CoreSurfaceAllocation *allocation,
                   void                  *alloc_data,
                   CoreSurfaceBufferLock *lock )
{
     D_MAGIC_ASSERT( pool, CoreSurfacePool );
     D_MAGIC_ASSERT( allocation, CoreSurfaceAllocation );
     D_MAGIC_ASSERT( lock, CoreSurfaceBufferLock );

     D_DEBUG_AT( bcmnexusPoolX, BLUE_COLOUR"%s() surface=%p, allocation=%p, addr=%p, pitch=%d, handle=%p accessor=%d, access=0x%08x, CPU accessed=0x%08x, "
                 "GPU accessed=0x%08x, type=0x%08x, flags=0x%08x, caps=0x%08x\n"NO_COLOUR, __FUNCTION__, (void *)allocation->surface, (void *)allocation,
                 ((struct bcmAllocationData*)alloc_data)->address, ((struct bcmAllocationData*)alloc_data)->pitch,
                 (void*)((struct bcmAllocationData*)alloc_data)->bcmSurfaceHandle, lock->accessor, lock->access,
                 allocation->accessed[CSAID_CPU], allocation->accessed[CSAID_GPU], allocation->type, allocation->config.flags, allocation->config.caps );

     return DFB_OK;
}

const SurfacePoolFuncs bcmnexusPoolFunctions = {
     .PoolDataSize       = bcmPoolDataSize,
     .PoolLocalDataSize  = bcmPoolLocalDataSize,
     .AllocationDataSize = bcmAllocationDataSize,
     .InitPool           = bcmInitPool,
     .JoinPool           = bcmJoinPool,
     .DestroyPool        = bcmDestroyPool,
     .LeavePool          = bcmLeavePool,
     .TestConfig         = bcmTestConfig,
     .PreAlloc           = bcmPreAlloc,
     .AllocateBuffer     = bcmAllocateBuffer,
     .DeallocateBuffer   = bcmDeallocateBuffer,
#if BCM_DFB_USE_TASK_MANAGER
     .CacheOp            = bcmCacheOp,
#endif
     .PreLock            = bcmPreLock,
     .Lock               = bcmLock,
     .Unlock             = bcmUnlock

};

const SurfacePoolFuncs bcmnexusGraphics3DPoolFunctions = {
     .PoolDataSize       = bcmPoolDataSize,
     .PoolLocalDataSize  = bcmPoolLocalDataSize,
     .AllocationDataSize = bcmAllocationDataSize,
     .InitPool           = bcmInitGraphics3DPool,
     .JoinPool           = bcmJoinGraphics3DPool,
     .DestroyPool        = bcmDestroyGraphics3DPool,
     .LeavePool          = bcmLeaveGraphics3DPool,
     .TestConfig         = bcmTestGraphics3DConfig,
     .PreAlloc           = bcmPreAlloc,
     .AllocateBuffer     = bcmAllocateBuffer,
     .DeallocateBuffer   = bcmDeallocateBuffer,
#if BCM_DFB_USE_TASK_MANAGER
     .CacheOp            = bcmCacheOp,
#endif
     .PreLock            = bcmPreLock,
     .Lock               = bcmLock,
     .Unlock             = bcmUnlock
};
