//COMCAST MODIFICATION BEGIN [COMPOSITE TO MAILBOX]
#include "mailbox_renderer_qt.h"

// Chromium's
#include "base/message_loop/message_loop.h"
#include "base/bind.h"

#include "content/browser/renderer_host/dip_util.h"
#include "ui/compositor/compositor.h"

#include <QOpenGLContext>
#if !defined(QT_NO_EGL)
#include <EGL/egl.h>
#include <EGL/eglext.h>
#endif

static void waitAndDeleteChromiumSync(FenceSync *sync)
{
    // Chromium uses its own GL bindings and stores in in thread local storage.
    // For that reason, let chromium_gpu_helper.cpp contain the producing code that will run in the Chromium
    // GPU thread, and put the sync consuming code here that will run in the QtQuick SG or GUI thread.
    switch (sync->type) {
    case FenceSync::NoSync:
        break;
    case FenceSync::EglSync:
#ifdef EGL_KHR_fence_sync
    {
        static bool resolved = false;
        static PFNEGLCLIENTWAITSYNCKHRPROC eglClientWaitSyncKHR = 0;
        static PFNEGLDESTROYSYNCKHRPROC eglDestroySyncKHR = 0;

        if (!resolved) {
            QOpenGLContext *context = QOpenGLContext::currentContext();
            eglClientWaitSyncKHR = (PFNEGLCLIENTWAITSYNCKHRPROC)context->getProcAddress("eglClientWaitSyncKHR");
            eglDestroySyncKHR = (PFNEGLDESTROYSYNCKHRPROC)context->getProcAddress("eglDestroySyncKHR");
            resolved = true;
        }

        // FIXME: Use the less wasteful eglWaitSyncKHR once we have a device that supports EGL_KHR_wait_sync.
        EGLint ret = eglClientWaitSyncKHR(sync->egl.display, sync->egl.sync, EGL_SYNC_FLUSH_COMMANDS_BIT_KHR, EGL_FOREVER_KHR);
        if (ret != EGL_CONDITION_SATISFIED_KHR) {
            qWarning("eglClientWaitSyncKHR failed");
        }
        eglDestroySyncKHR(sync->egl.display, sync->egl.sync);
        sync->reset();
    }
#endif
        break;
    case FenceSync::ArbSync:
#ifdef GL_ARB_sync
        glWaitSync(sync->arb.sync, 0, GL_TIMEOUT_IGNORED);
        glDeleteSync(sync->arb.sync);
        sync->reset();
#endif
        break;
    default:
        sync->reset();
        break;
    }
    // If Chromium was able to create a sync, we should have been able to handle its type here too.
    Q_ASSERT(!*sync);
}

MailboxRendererQt::MailboxRendererQt()
    : m_program (NULL)
    , m_vertexCoordEntry(0)
    , m_textureCoordEntry(0)
    , m_textureId(0)
{
}

MailboxRendererQt::~MailboxRendererQt()
{
    delete m_program;
    m_program = NULL;
}

void MailboxRendererQt::setFrameData(cc::GLFrameData* gl_frame_data)
{
    if (!gl_frame_data)
        return;

    m_frontFrameData.reset(new cc::GLFrameData);
    m_frontFrameData->mailbox         = gl_frame_data->mailbox;
    m_frontFrameData->sync_point      = gl_frame_data->sync_point;
    m_frontFrameData->size            = gl_frame_data->size;
    m_frontFrameData->sub_buffer_rect = gl_frame_data->sub_buffer_rect;
}

cc::GLFrameData* MailboxRendererQt::takeFrameData()
{
    cc::GLFrameData* ret = m_backFrameData.release();
    m_backFrameData.swap(m_frontFrameData);
    return ret ? ret : new cc::GLFrameData;
}

void MailboxRendererQt::draw(const QRectF& viewport)
{
    // cleanup caller errors
    GLenum err;
    while ((err = glGetError()) != GL_NO_ERROR) {
        qWarning("XXX ERROR MailboxRendererQt::draw@%d, Qt.glError: 0x%x\n", __LINE__, err);
    }

    if (!prepareMailboxTexture()) {
        qWarning() << "XXX ERROR: MailboxRendererQt::draw failed to prepare mailbox texutre";
        return;
    }

    const GLfloat l = viewport.left();
    const GLfloat r = viewport.right();
    const GLfloat t = viewport.top();
    const GLfloat b = viewport.bottom();

    // render texture from flipped device
    // (0,1) 1 - 2 (1,1)
    //       | \ |
    // (0,0) 4 - 3 (1,0)
    const GLfloat textureCoordinates[] = {
        0, 1,
        1, 1,
        1, 0,
        0, 0
    };
    // (l,t) 1 - 2 (r,t)
    //       | \ |
    // (l,b) 4 - 3 (r,b)
    const GLfloat vertexCoordinates[] = {
        l, t,
        r, t,
        r, b,
        l, b
    };

    if (!m_program) {
        static const char *textureVertexProgram =
            "attribute highp vec2 vertexCoordEntry;             \n"
            "attribute highp vec2 textureCoordEntry;            \n"
            "varying highp vec2 textureCoord;                   \n"
            "void main() {                                      \n"
            "   textureCoord = textureCoordEntry;               \n"
            "   gl_Position = vec4(vertexCoordEntry, 0.0, 1.0); \n"
            "}                                                  \n";
        static const char *textureFragmentProgram =
            "uniform sampler2D texture;                          \n"
            "varying highp vec2 textureCoord;                    \n"
            "void main() {                                       \n"
            "   gl_FragColor = texture2D(texture, textureCoord); \n"
            "}                                                   \n";
        m_program = new QOpenGLShaderProgram;
        m_program->addShaderFromSourceCode(QOpenGLShader::Vertex, textureVertexProgram);
        m_program->addShaderFromSourceCode(QOpenGLShader::Fragment, textureFragmentProgram);
        if (!m_program->link()) {
            qWarning() << "XXX ERROR MailboxRendererQt::draw failed to link program";
            delete m_program;
            m_program = NULL;
            return;
        }
        m_vertexCoordEntry = m_program->attributeLocation("vertexCoordEntry");
        m_textureCoordEntry = m_program->attributeLocation("textureCoordEntry");
    }

    m_program->bind();
    glEnableVertexAttribArray(m_vertexCoordEntry);
    glEnableVertexAttribArray(m_textureCoordEntry);
    glVertexAttribPointer(m_vertexCoordEntry, 2, GL_FLOAT, GL_FALSE, 0, vertexCoordinates);
    glVertexAttribPointer(m_textureCoordEntry, 2, GL_FLOAT, GL_FALSE, 0, textureCoordinates);
    glActiveTexture(GL_TEXTURE0);
    glBindTexture(GL_TEXTURE_2D, m_textureId);
    glDrawArrays(GL_TRIANGLE_FAN, 0, 4);
    m_program->release();
    glBindTexture(GL_TEXTURE_2D, 0);
    glDisableVertexAttribArray(m_vertexCoordEntry);
    glDisableVertexAttribArray(m_textureCoordEntry);

    while ((err = glGetError()) != GL_NO_ERROR) {
        qWarning("XXX ERROR MailboxRendererQt::draw@%d, glError: 0x%x\n", __LINE__, err);
    }
}

void MailboxRendererQt::fetchTextureAndUnlockQt()
{
    gpu::gles2::MailboxManager *mailboxManager = mailbox_manager();
    gpu::gles2::Texture *tex = ConsumeTexture(
        mailboxManager,
        GL_TEXTURE_2D,
        *reinterpret_cast<const gpu::gles2::MailboxName*>(m_frontFrameData->mailbox.name));
    m_textureId = service_id(tex);
    FenceSync fence = createFence();
    QMutexLocker lock(&m_fetchMutex);
    Q_ASSERT(!m_GLFence);
    m_GLFence = fence;
    m_fetchWaitCond.wakeOne();
}

void MailboxRendererQt::syncPointRetired()
{
    QMutexLocker lock(&m_fetchMutex);
    base::MessageLoop::current()->PostTask(
        FROM_HERE, base::Bind(&MailboxRendererQt::fetchTextureAndUnlockQt, AsWeakPtr()));
}

bool MailboxRendererQt::prepareMailboxTexture()
{
    if (m_frontFrameData.get()
        && !m_frontFrameData->mailbox.IsZero()
        && (m_frontFrameData->sync_point != 0)) {
        QMutexLocker lock(&m_fetchMutex);
        base::MessageLoop *gpuMessageLoop = gpu_message_loop();
        content::SyncPointManager *syncPointManager = sync_point_manager();
        AddSyncPointCallbackOnGpuThread(
            gpuMessageLoop,
            syncPointManager,
            m_frontFrameData->sync_point,
            base::Bind(&MailboxRendererQt::syncPointRetired, AsWeakPtr()));
        m_fetchWaitCond.wait(&m_fetchMutex);
        m_frontFrameData->sync_point = 0;
        waitAndDeleteChromiumSync(&m_GLFence);
    }
    return m_textureId != 0;
}
//COMCAST MODIFICATION END [COMPOSITE TO MAILBOX]
