/******************************************************************************
 *  Copyright (c) 2017 RDK Management, LLC. All rights reserved.
 *  Copyright (C) 2017 Broadcom. The term "Broadcom" refers to Broadcom Limited and/or its subsidiaries.
 *
 *  This program is the proprietary software of Broadcom and/or its licensors,
 *  and may only be used, duplicated, modified or distributed pursuant to the terms and
 *  conditions of a separate, written license agreement executed between you and Broadcom
 *  (an "Authorized License").  Except as set forth in an Authorized License, Broadcom grants
 *  no license (express or implied), right to use, or waiver of any kind with respect to the
 *  Software, and Broadcom expressly reserves all rights in and to the Software and all
 *  intellectual property rights therein.  IF YOU HAVE NO AUTHORIZED LICENSE, THEN YOU
 *  HAVE NO RIGHT TO USE THIS SOFTWARE IN ANY WAY, AND SHOULD IMMEDIATELY
 *  NOTIFY BROADCOM AND DISCONTINUE ALL USE OF THE SOFTWARE.
 *
 *  Except as expressly set forth in the Authorized License,
 *
 *  1.     This program, including its structure, sequence and organization, constitutes the valuable trade
 *  secrets of Broadcom, and you shall use all reasonable efforts to protect the confidentiality thereof,
 *  and to use this information only in connection with your use of Broadcom integrated circuit products.
 *
 *  2.     TO THE MAXIMUM EXTENT PERMITTED BY LAW, THE SOFTWARE IS PROVIDED "AS IS"
 *  AND WITH ALL FAULTS AND BROADCOM MAKES NO PROMISES, REPRESENTATIONS OR
 *  WARRANTIES, EITHER EXPRESS, IMPLIED, STATUTORY, OR OTHERWISE, WITH RESPECT TO
 *  THE SOFTWARE.  BROADCOM SPECIFICALLY DISCLAIMS ANY AND ALL IMPLIED WARRANTIES
 *  OF TITLE, MERCHANTABILITY, NONINFRINGEMENT, FITNESS FOR A PARTICULAR PURPOSE,
 *  LACK OF VIRUSES, ACCURACY OR COMPLETENESS, QUIET ENJOYMENT, QUIET POSSESSION
 *  OR CORRESPONDENCE TO DESCRIPTION. YOU ASSUME THE ENTIRE RISK ARISING OUT OF
 *  USE OR PERFORMANCE OF THE SOFTWARE.
 *
 *  3.     TO THE MAXIMUM EXTENT PERMITTED BY LAW, IN NO EVENT SHALL BROADCOM OR ITS
 *  LICENSORS BE LIABLE FOR (i) CONSEQUENTIAL, INCIDENTAL, SPECIAL, INDIRECT, OR
 *  EXEMPLARY DAMAGES WHATSOEVER ARISING OUT OF OR IN ANY WAY RELATING TO YOUR
 *  USE OF OR INABILITY TO USE THE SOFTWARE EVEN IF BROADCOM HAS BEEN ADVISED OF
 *  THE POSSIBILITY OF SUCH DAMAGES; OR (ii) ANY AMOUNT IN EXCESS OF THE AMOUNT
 *  ACTUALLY PAID FOR THE SOFTWARE ITSELF OR U.S. $1, WHICHEVER IS GREATER. THESE
 *  LIMITATIONS SHALL APPLY NOTWITHSTANDING ANY FAILURE OF ESSENTIAL PURPOSE OF
 *  ANY LIMITED REMEDY.
 ******************************************************************************/

#include "dvrmanager-socif.h"
#include <memory.h>
#include "rdk_debug.h"

#if USE_SYSRES_MLT
#include "rpl_malloc.h"
#include "rpl_new.h"
#endif

#if (NEXUS_SECURITY_API_VERSION != 2)
#error "Only compatible to NEXUS_SECURITY_API_VERSION 2"
#endif

#if defined(NEXUS_HAS_SECURITY) && (NEXUS_PLATFORM!=97125)

#define USE_OTP_KEY 1
#include "nexus_dma.h"
#include "nexus_memory.h"
#include "nexus_security.h"
#include "nexus_keyladder.h"
#ifdef RDK_USE_NXCLIENT
#include "nexus_platform_client.h"
#include "nxclient.h"
#endif

#define INT_FATAL(FORMAT, ...)      RDK_LOG(RDK_LOG_FATAL, "LOG.RDK.DVR", FORMAT "\n", __VA_ARGS__)
#define INT_ERROR(FORMAT, ...)      RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.DVR", FORMAT "\n", __VA_ARGS__)
#define INT_WARNING(FORMAT, ...)    RDK_LOG(RDK_LOG_WARN, "LOG.RDK.DVR", FORMAT "\n",  __VA_ARGS__)
#define INT_INFO(FORMAT, ...)       RDK_LOG(RDK_LOG_INFO, "LOG.RDK.DVR", FORMAT "\n",  __VA_ARGS__)
#define INT_DEBUG(FORMAT, ...)      RDK_LOG(RDK_LOG_DEBUG, "LOG.RDK.DVR", FORMAT "\n", __VA_ARGS__)
#define INT_TRACE1(FORMAT, ...)     RDK_LOG(RDK_LOG_TRACE1, "LOG.RDK.DVR", FORMAT "\n",  __VA_ARGS__)
#define INT_TRACE2(FORMAT, ...)     RDK_LOG(RDK_LOG_TRACE2, "LOG.RDK.DVR", FORMAT "\n",  __VA_ARGS__)
#define INT_TRACE3(FORMAT, ...)     RDK_LOG(RDK_LOG_TRACE3, "LOG.RDK.DVR", FORMAT "\n",  __VA_ARGS__)
#define INT_TRACE4(FORMAT, ...)     RDK_LOG(RDK_LOG_TRACE4, "LOG.RDK.DVR", FORMAT "\n",  __VA_ARGS__)
#define INT_TRACE5(FORMAT, ...)     RDK_LOG(RDK_LOG_TRACE5, "LOG.RDK.DVR", FORMAT "\n",  __VA_ARGS__)

#define FATAL(...)                  INT_FATAL(__VA_ARGS__, "")
#define ERROR(...)                  INT_ERROR(__VA_ARGS__, "")
#define WARNING(...)                INT_WARNING(__VA_ARGS__, "")
#define INFO(...)                   INT_INFO(__VA_ARGS__, "")
#define DEBUG(...)                  INT_DEBUG(__VA_ARGS__, "")
#define TRACE1(...)                 INT_TRACE1(__VA_ARGS__, "")
#define TRACE2(...)                 INT_TRACE2(__VA_ARGS__, "")
#define TRACE3(...)                 INT_TRACE3(__VA_ARGS__, "")
#define TRACE4(...)                 INT_TRACE4(__VA_ARGS__, "")
#define TRACE5(...)                 INT_TRACE5(__VA_ARGS__, "")

static NEXUS_DmaHandle gDmaHandle = NULL;
static NEXUS_KeySlotHandle gEncKeySlotHandle = NULL;
static NEXUS_KeySlotHandle gDecKeySlotHandle = NULL;

// SOC specific crypto state data is stored in DVRCryptoState
class crypto_t{
public:
    crypto_t();
    ~crypto_t();

    pthread_mutex_t buffer_mutex;
    uint8_t *buffer;
    uint32_t buffer_size();  

    BKNI_EventHandle dma_event;
}; 

static crypto_t *gCryptoEncrypt = NULL;
static crypto_t *gCryptoDecrypt = NULL;

#ifdef USE_OTP_KEY
static const uint8_t ucProcInForKey3[16] = { 0x11, 0x11, 0x22, 0x22, 0x12, 0x34, 0x56, 0x78, 
                                             0x98, 0x76, 0x54, 0x32, 0x09, 0x23, 0x45, 0x56 };
static const uint8_t ucProcInForKey4[16] = { 0x2E, 0xF6, 0xB6, 0xCC, 0x5B, 0x6C, 0x86, 0xF7, 
                                             0x92, 0xA2, 0x48, 0x70, 0xAC, 0xD9, 0x46, 0x73 };
#else
static const uint8_t gClearKey[16] = { 0xBE, 0xF9, 0xB0, 0x67, 0x13, 0xB8, 0xBC, 0x87, 
                                       0xBC, 0xFB, 0xB2, 0x69, 0x13, 0xBA, 0xBE, 0x8b };
static const uint8_t gClearIv[16]  = { 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 
                                       0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff };
#endif

#define BUFFER_SIZE  (192*1024*2)

crypto_t::crypto_t()
{
    NEXUS_MemoryAllocationSettings memSettings;

#ifdef RDK_USE_NXCLIENT
    NEXUS_ClientConfiguration platformConfig;
    NEXUS_Platform_GetClientConfiguration(&platformConfig);

    NEXUS_Memory_GetDefaultAllocationSettings(&memSettings);
    memSettings.heap = platformConfig.heap[1]; /* protected and un-prot */
#else    
    memSettings.heap = NULL;
#endif
    NEXUS_Memory_Allocate(BUFFER_SIZE, &memSettings, (void **)&buffer);
    BDBG_ASSERT(buffer != NULL);

    pthread_mutex_init(&buffer_mutex, NULL); 

    BKNI_CreateEvent(&dma_event);
    BDBG_ASSERT(dma_event != NULL);
}

crypto_t::~crypto_t()
{
    BDBG_ASSERT(buffer != NULL);
    NEXUS_Memory_Free(buffer);
    pthread_mutex_destroy(&buffer_mutex);
    BDBG_ASSERT(dma_event != NULL);
    BKNI_DestroyEvent(dma_event);
}

uint32_t crypto_t::buffer_size(void)
{
    return BUFFER_SIZE;
}

static int configureKeySlot(NEXUS_KeySlotHandle *pKeySlotHandle, bool isEnc)
{
    NEXUS_KeySlotAllocateSettings keyslotAllocSettings;
    NEXUS_KeySlotSettings keyslotSettings;
    NEXUS_KeySlotEntrySettings keyslotEntrySettings;
    NEXUS_KeySlotHandle keyslotHandle = NULL;
    NEXUS_KeySlotBlockEntry entry;
    NEXUS_Error rc = NEXUS_UNKNOWN;

    /* allocate keyslot */
    NEXUS_KeySlot_GetDefaultAllocateSettings( &keyslotAllocSettings );
    keyslotAllocSettings.owner = NEXUS_SecurityCpuContext_eHost;
    keyslotAllocSettings.slotType = NEXUS_KeySlotType_eIvPerBlock;
    keyslotAllocSettings.useWithDma = true;
    keyslotHandle = NEXUS_KeySlot_Allocate( &keyslotAllocSettings );
    if( !keyslotHandle ) { return BERR_TRACE( NEXUS_NOT_AVAILABLE ); }

    /*configure keyslot parameters */
    NEXUS_KeySlot_GetSettings( keyslotHandle, &keyslotSettings );
    rc = NEXUS_KeySlot_SetSettings( keyslotHandle, &keyslotSettings );
    if( rc != NEXUS_SUCCESS ) { BERR_TRACE( rc ); goto exit; }

    /*configure keyslot entry parameters. setting for all ood, even, and clear */
    if(isEnc) { entry = NEXUS_KeySlotBlockEntry_eCpsOdd; }
    else      { entry = NEXUS_KeySlotBlockEntry_eCpdOdd; }
    for(int i = 0; i <= 2 ; i++) {
        NEXUS_KeySlot_GetEntrySettings( keyslotHandle, (NEXUS_KeySlotBlockEntry)(entry+i), 
                &keyslotEntrySettings );
        keyslotEntrySettings.algorithm = NEXUS_CryptographicAlgorithm_e3DesAba;
        keyslotEntrySettings.algorithmMode = NEXUS_CryptographicAlgorithmMode_eEcb;
        keyslotEntrySettings.terminationMode = NEXUS_KeySlotTerminationMode_eClear;
        keyslotEntrySettings.rPipeEnable = true;
        keyslotEntrySettings.gPipeEnable = true;
        rc = NEXUS_KeySlot_SetEntrySettings( keyslotHandle, (NEXUS_KeySlotBlockEntry)(entry+i), 
                &keyslotEntrySettings );
        if( rc != NEXUS_SUCCESS ) { BERR_TRACE( rc ); }
    }
exit:
    *pKeySlotHandle = keyslotHandle;
     return rc;
}

#ifdef USE_OTP_KEY
static int configureKeyladder(NEXUS_KeySlotHandle keySlotHandle, bool isEnc)
{
    NEXUS_KeyLadderAllocateSettings ladderAllocSettings;
    NEXUS_KeyLadderSettings ladderSettings;
    NEXUS_KeyLadderInfo keyLadderInfo;
    NEXUS_KeyLadderLevelKey levelKey;
    NEXUS_KeySlotBlockEntry entry;
    NEXUS_KeyLadderHandle handle = NULL;
    NEXUS_Error rc = NEXUS_UNKNOWN;

    NEXUS_KeyLadder_GetDefaultAllocateSettings( &ladderAllocSettings );
    ladderAllocSettings.owner = NEXUS_SecurityCpuContext_eHost;
    handle = NEXUS_KeyLadder_Allocate( NEXUS_ANY_ID, &ladderAllocSettings );
    if( !handle ) { return BERR_TRACE( NEXUS_NOT_AVAILABLE ); }

    rc = NEXUS_KeyLadder_GetInfo( handle, &keyLadderInfo );
    if( rc != NEXUS_SUCCESS ) { return BERR_TRACE( rc ); }

    INFO( "KEYLADDER: Allocate Index[%d].\n", keyLadderInfo.index );

    NEXUS_KeyLadder_GetSettings( handle, &ladderSettings );
    ladderSettings.algorithm = NEXUS_CryptographicAlgorithm_e3DesAba;
    ladderSettings.operation = NEXUS_CryptographicOperation_eDecrypt;
    ladderSettings.mode = NEXUS_KeyLadderMode_eCp_128_4;

    INFO( "KEYLADDER: Configure Index[%d].\n", keyLadderInfo.index );

    ladderSettings.root.type = NEXUS_KeyLadderRootType_eOtpDirect;
#if (BCHP_CHIP==74371)
    ladderSettings.root.otpKeyIndex = 1; /* otp key B*/
#else
    ladderSettings.root.otpKeyIndex = 0; /* otp key A*/
#endif

    rc = NEXUS_KeyLadder_SetSettings( handle, &ladderSettings );
    if( rc != NEXUS_SUCCESS ) { return BERR_TRACE( rc ); }

    INFO( "KEYLADDER: Set Key Level 3  Index[%d].\n", keyLadderInfo.index );

    NEXUS_KeyLadder_GetLevelKeyDefault(&levelKey);
    levelKey.level = 3;
    levelKey.ladderKeySize = 128; 
    BKNI_Memcpy(levelKey.ladderKey, ucProcInForKey3, sizeof(ucProcInForKey3));
    rc = NEXUS_KeyLadder_GenerateLevelKey(handle, &levelKey);
    if( rc != NEXUS_SUCCESS ) { return  BERR_TRACE( rc ); }

    NEXUS_KeyLadder_GetLevelKeyDefault(&levelKey);
    levelKey.level = 4;
    levelKey.ladderKeySize = 128; 
    levelKey.route.destination = NEXUS_KeyLadderDestination_eKeyslotKey;
    levelKey.route.keySlot.handle = keySlotHandle;
    BKNI_Memcpy(levelKey.ladderKey, ucProcInForKey4, sizeof(ucProcInForKey4));
    // Setting for all ood, even, and clear,
    if(isEnc) { entry = NEXUS_KeySlotBlockEntry_eCpsOdd; }
    else      { entry = NEXUS_KeySlotBlockEntry_eCpdOdd; }
    for(int i = 0; i <= 2; i++) {
        levelKey.route.keySlot.entry = (NEXUS_KeySlotBlockEntry)(entry+i);
        INFO( "KEYLADDER: Set Key Level 4  Index[%d]. destination[%d] entry[%d] \n", keyLadderInfo.index, 
                NEXUS_KeyLadderDestination_eKeyslotKey, levelKey.route.keySlot.entry);
        rc = NEXUS_KeyLadder_GenerateLevelKey(handle, &levelKey);
        if( rc != NEXUS_SUCCESS ) { return  BERR_TRACE( rc ); }
    }

    INFO( "KEYLADDER: Free Index[%d].\n", keyLadderInfo.index );
    NEXUS_KeyLadder_Free(handle);
    return 0;
exit:
    return 1;
}

#else /* #ifdef USE_OTP_KEY */

static int configureClearKey(NEXUS_KeySlotHandle keySlotHandle, bool isEnc)
{
    NEXUS_KeySlotKey slotKey;
    NEXUS_KeySlotBlockEntry entry;
    slotKey.size = sizeof(gClearKey);
    memcpy(slotKey.key, gClearKey, sizeof(gClearKey));
    if(isEnc) { entry = NEXUS_KeySlotBlockEntry_eCpsOdd; }
    else      { entry = NEXUS_KeySlotBlockEntry_eCpdOdd; }
    for(int i = 0; i <= 2; i++) {
        rc = NEXUS_KeySlot_SetEntryKey(keySlotHandle, (NEXUS_KeySlotBlockEntry)(entry+i), &slotKey);
        if( rc != NEXUS_SUCCESS ) { BERR_TRACE( rc ); goto exit; }
    }
    return 0;
exit:
    return 1;
}

#endif /* #ifdef USE_OTP_KEY */

static void completeCallback( void *pParam, int iParam )
{
    BSTD_UNUSED( iParam );
    BKNI_SetEvent( (BKNI_EventHandle)pParam );
    return;
}

bool dvrcryptoifc::initCrypto()
{
    NEXUS_Error                     rc = NEXUS_UNKNOWN;

#ifdef RDK_USE_NXCLIENT

    NxClient_JoinSettings joinSettings;

    NxClient_GetDefaultJoinSettings(&joinSettings);
    snprintf(joinSettings.name, NXCLIENT_MAX_NAME, "%s", "dvrmgr-hal-broadcom-initcrypto");
    rc = NxClient_Join(&joinSettings);
    if( rc != NEXUS_SUCCESS ) { 
        ERROR("NxClient_Join failed %d", rc);
        goto exit;
    }

#endif
    if ( gDmaHandle == NULL ) { 
#if (NEXUS_HAS_XPT_DMA == 1)
#if defined(NEXUS_DMA_ID_RESERVED_FOR_TSB)
        gDmaHandle = NEXUS_Dma_Open(NEXUS_DMA_ID_RESERVED_FOR_TSB, NULL);
#else
        gDmaHandle = NEXUS_Dma_Open(NEXUS_ANY_ID, NULL);
#endif
#else
        gDmaHandle = NEXUS_Dma_Open(0, NULL); 
#endif
        INFO("%s: gDmaHandle:%p\n", __FUNCTION__, gDmaHandle);
    }

    if ( gEncKeySlotHandle == NULL ) {
        rc = configureKeySlot(&gEncKeySlotHandle, true);
        if( rc != NEXUS_SUCCESS ) { BERR_TRACE( rc ); goto exit; }
        // Load recpump key
#ifndef USE_OTP_KEY
        rc = configureClearKey (gEncKeySlotHandle, true);
        if( rc != NEXUS_SUCCESS ) { BERR_TRACE( rc ); goto exit; }
#else
        rc = configureKeyladder(gEncKeySlotHandle, true);
        if( rc != NEXUS_SUCCESS ) { BERR_TRACE( rc ); goto exit; }
#endif
    }
    if ( gCryptoEncrypt == NULL ) { gCryptoEncrypt = new crypto_t; }

    if ( gDecKeySlotHandle == NULL ) {
        rc = configureKeySlot(&gDecKeySlotHandle, false);
        if( rc != NEXUS_SUCCESS ) { BERR_TRACE( rc ); goto exit; }

        // Load playpump key
#ifndef USE_OTP_KEY            
        rc = configureClearKey (gDecKeySlotHandle, false);
        if( rc != NEXUS_SUCCESS ) { BERR_TRACE( rc ); goto exit; }
#else
        rc = configureKeyladder(gDecKeySlotHandle, false);
        if( rc != NEXUS_SUCCESS ) { BERR_TRACE( rc ); goto exit; }
#endif
    }

    if ( gCryptoDecrypt == NULL ) { gCryptoDecrypt = new crypto_t; }

    return true;
exit:
    return false;
}

void dvrcryptoifc::termCrypto()
{
	if ( gCryptoEncrypt ) {
        delete (crypto_t *)gCryptoEncrypt;
		gCryptoEncrypt = NULL;
	}
  if ( gCryptoDecrypt ) {
        delete (crypto_t *)gCryptoDecrypt;
        gCryptoDecrypt = NULL;
	}
}

static void dma_crypto_engine(void *pSrc, int length, bool isEnc, bool isTTS, BKNI_EventHandle event)
{
    NEXUS_DmaJobSettings jobSettings;
    NEXUS_DmaJobHandle dmaJob;
    NEXUS_DmaJobBlockSettings blockSettings;
    NEXUS_DmaJobStatus jobStatus;
    NEXUS_Error rc = NEXUS_UNKNOWN;
    int i = 0;

    if((isTTS ? (length % 192) : (length % 188)) != 0) {
        ERROR("%s: pacakge [length = %d] is not multiple of  %d \n",__FUNCTION__, length, (isTTS ? 192 : 188));
        return;
    }

    BDBG_ASSERT(event != NULL);
    BKNI_ResetEvent(event);

    NEXUS_DmaJob_GetDefaultSettings(&jobSettings);
    jobSettings.numBlocks                   = 1;
    jobSettings.keySlot                     = isEnc ? gEncKeySlotHandle : gDecKeySlotHandle;
    if ( isTTS ) {
#if (NEXUS_HAS_XPT_DMA == 1)
        jobSettings.timestampType           = NEXUS_TransportTimestampType_eMod300;
#endif
    } else {
        jobSettings.timestampType           = NEXUS_TransportTimestampType_eNone;
    }
    jobSettings.dataFormat                  = NEXUS_DmaDataFormat_eMpeg;
    jobSettings.completionCallback.callback = completeCallback;
    jobSettings.completionCallback.context  = event;

    dmaJob = NEXUS_DmaJob_Create(gDmaHandle, &jobSettings);

    NEXUS_DmaJob_GetDefaultBlockSettings(&blockSettings);
    blockSettings.pSrcAddr                  = pSrc;
    blockSettings.pDestAddr                 = pSrc;
    blockSettings.blockSize                 = length;
    blockSettings.resetCrypto               = true;
    blockSettings.scatterGatherCryptoStart  = true;
    blockSettings.scatterGatherCryptoEnd    = true;
    blockSettings.cached                    = true;

    rc = NEXUS_DmaJob_ProcessBlocks(dmaJob, &blockSettings, 1);
    if( rc == NEXUS_DMA_QUEUED ) {
        /* BKNI_WaitForEvent( event, BKNI_INFINITE ); */
        BKNI_WaitForEvent( event, 1000);
        NEXUS_DmaJob_GetStatus( dmaJob, &jobStatus );
        BDBG_ASSERT( jobStatus.currentState == NEXUS_DmaJobState_eComplete );
        rc = NEXUS_SUCCESS;
    }

    if(dmaJob) NEXUS_DmaJob_Destroy(dmaJob);
    return;
}

void dvrcryptoifc::decryptPackets( unsigned char *packets, int packetSize, int headerSize, int length,unsigned char *physicalAddress)
{
    crypto_t *p = gCryptoDecrypt;
    uint8_t *buf;
    uint32_t n = 0;

    if(gCryptoDecrypt == NULL) { /* JUST for Testing */
        WARNING("InitCrypto should be called before decryptPackets\n");
        initCrypto();
        p = gCryptoDecrypt;
    }
    if(packets == NULL || packetSize == 0) return;

    BDBG_ASSERT(p != NULL); 
    pthread_mutex_lock( &p->buffer_mutex);
    buf = p->buffer;
    do {
        if( p->buffer_size() < length ) { n = p->buffer_size(); }
        else                            { n = length; }

        memcpy((void *)buf, packets, n);
        dma_crypto_engine((void *)buf, n, FALSE, (headerSize==16), p->dma_event);
        memcpy(packets, buf, n);

        packets += n;
        length  -= n;
    } while ( length != 0 );

    pthread_mutex_unlock( &p->buffer_mutex);

    return;
}

void dvrcryptoifc::encryptPackets( unsigned char *packets, int packetSize, int headerSize, int length,unsigned char *physicalAddress)
{
    crypto_t *p = gCryptoEncrypt;
    uint8_t *buf;
    uint32_t n = 0;

    if(gCryptoEncrypt == NULL) { /* JUST for Testing */
        WARNING("InitCrypto should be called before encryptPackets\n");
        initCrypto();
        p = gCryptoEncrypt;
    }
    if(packets == NULL || packetSize == 0) return;

    BDBG_ASSERT(p != NULL); 
    pthread_mutex_lock( &p->buffer_mutex);
    buf = p->buffer;
    do {
        if( p->buffer_size() < length ) { n = p->buffer_size(); }
        else                            { n = length; }

        memcpy((void *)buf, packets, n);
        dma_crypto_engine((void *)buf, n, TRUE, (headerSize==16), p->dma_event);
        memcpy(packets, buf, n);

        packets += n;
        length  -= n;
    } while ( length != 0 );

    pthread_mutex_unlock( &p->buffer_mutex);

    return;
}

#else  // #if defined(NEXUS_HAS_SECURITY) && (NEXUS_PLATFORM!=97125)
bool dvrcryptoifc::initCrypto(){return TRUE;}
void dvrcryptoifc::termCrypto(){}
void dvrcryptoifc::decryptPackets( unsigned char *packets, int packetSize, int headerSize, int length,unsigned char*physicalAddress){}
void dvrcryptoifc::encryptPackets( unsigned char *packets, int packetSize, int headerSize, int length,unsigned char* physicalAddress){}
#endif // #if defined(NEXUS_HAS_SECURITY) && (NEXUS_PLATFORM!=97125)

bool dvrcryptoifc::allocateDeviceBuffer( size_t size, void **virtualAddress, void **physicalAddress)
{
    bool result = false;
    void *buffer = 0;
    
    *physicalAddress = 0;
    buffer= malloc( size );
    if ( buffer ) {
        *virtualAddress = buffer;
        result = true;
    } else {
        *virtualAddress = 0;
        result = false;
    }
    return result;
}

bool dvrcryptoifc::allocateDeviceBufferAligned( size_t size, int alignment, void **virtualAddress, void **physicalAddress )
{
    bool result = false;
    void *buffer = 0;
    
    *physicalAddress = 0;
    if(0 == posix_memalign(virtualAddress, alignment, size))
        return true;
    else
        return false;
}
void dvrcryptoifc::freeDeviceBuffer( void *virtualAddress )
{
       free( virtualAddress );
}
