#!/bin/bash

#######################################
#
# Build Framework standard script for
#
# sdvagent component

# use -e to fail on any shell issue
# -e is the requirement from Build Framework
set -e

# default PATHs - use `man readlink` for more info
# the path to combined build
export RDK_PROJECT_ROOT_PATH=${RDK_PROJECT_ROOT_PATH-`readlink -m ..`}
export COMBINED_ROOT=$RDK_PROJECT_ROOT_PATH

# path to build script (this script)
export RDK_SCRIPTS_PATH=${RDK_SCRIPTS_PATH-`readlink -m $0 | xargs dirname`}

# path to components sources and target
export RDK_SOURCE_PATH=${RDK_SOURCE_PATH-`readlink -m .`}
export RDK_TARGET_PATH=${RDK_TARGET_PATH-$RDK_SOURCE_PATH}

# fsroot and toolchain (valid for all devices)
export RDK_FSROOT_PATH=${RDK_FSROOT_PATH-`readlink -m $RDK_PROJECT_ROOT_PATH/sdk/fsroot/ramdisk`}
export RDK_TOOLCHAIN_PATH=${RDK_TOOLCHAIN_PATH-`readlink -m $RDK_PROJECT_ROOT_PATH/sdk/toolchain/staging_dir`}


# default component name
export RDK_COMPONENT_NAME=${RDK_COMPONENT_NAME-`basename $RDK_SOURCE_PATH`}
export RDK_DIR=$RDK_PROJECT_ROOT_PATH

export BUILDS_DIR=$RDK_PROJECT_ROOT_PATH
export WORK_DIR=$BUILDS_DIR/work${RDK_PLATFORM_DEVICE^^}

export LDFLAGS="$LDFLAGS -Wl,-rpath-link=$RDK_FSROOT_PATH/usr/local/lib"
# parse arguments
INITIAL_ARGS=$@

function usage()
{
    set +x
    echo "Usage: `basename $0` [-h|--help] [-v|--verbose] [action]"
    echo "    -h    --help                  : this help"
    echo "    -v    --verbose               : verbose output"
    echo "    -p    --platform  =PLATFORM   : specify platform for sdvagent"
    echo "    -l    --local                 : local build, enables iarm stubs and code metrics"
    echo
    echo "Supported actions:"
    echo "      configure, clean, build (DEFAULT), rebuild, install"
}

# options may be followed by one colon to indicate they have a required argument
if ! GETOPT=$(getopt -n "build.sh" -o hvlp: -l help,verbose,local,platform: -- "$@")
then
    usage
    exit 1
fi

eval set -- "$GETOPT"

local_build=false

while true; do
  case "$1" in
    -h | --help ) usage; exit 0 ;;
    -v | --verbose ) set -x ;;
    -l | --local ) local_build=true;;
    -p | --platform ) CC_PLATFORM="$2" ; shift ;;
    -- ) shift; break;;
    * ) break;;
  esac
  shift
done

ARGS=$@
source ${RDK_PROJECT_ROOT_PATH}/build_scripts/setBCMenv.sh


# component-specific vars
if [ $local_build = "false" ]; then

    export CROSS_COMPILE=1
    export CXX=mipsel-linux-g++
    export AR=mipsel-linux-ar
    export LD=mipsel-linux-ld
    export NM=mipsel-linux-nm
    export RANLIB=mipsel-linux-ranlib
    export STRIP=mipsel-linux-strip

#export PATH=${PATH}:${RDK_PROJECT_ROOT_PATH}/workXG1/tools/stbgcc-4.5.4-2.8/bin
    export PATH=${PATH}:$BCM_TOOLCHAIN

fi



# functional modules

function configure()
{
    echo "======= Configure sdvAgent ======="

    cd ${RDK_SOURCE_PATH}

    if [ ! -d m4 ]; then
        mkdir m4
    fi

    autoreconf -ivf

    if [ "$local_build" = "true" ]; then
        echo "THIS IS A LOCAL BUILD"
        ./configure --prefix="/opt/test/" --enable-local="yes"
    else
        ./configure --prefix="${RDK_FSROOT_PATH}/usr"
    fi

}


function clean()
{
    echo "======= Clean sdvAgent ======="

    cd ${RDK_SOURCE_PATH}

    ./removeall.sh
}

function build()
{
    echo "======= Build sdvAgent ======="
    make -C ${RDK_SOURCE_PATH}
}


function rebuild()
{
    clean
    configure
    build
}

function install()
{
    echo "======= Install sdvAgent ======="

    cd ${RDK_SOURCE_PATH}
    make install

    # This was needed for dbus in the vss implementation it used to be in
    # /etc/dbus-1/system.d/ not sure where to put it now
    #if [ -f tools/sdvAgent.conf ]; then
#	mkdir -p ${RDK_PROJECT_ROOT_PATH}/sysint/devspec/etc/dbus-1/system.d/
#        cp tools/sdvAgent.conf ${RDK_PROJECT_ROOT_PATH}/sysint/devspec/etc/dbus-1/system.d/
#    fi

#    if [ -f tools/sdvAgent_init ]; then
#	mkdir -p ${RDK_FSROOT_PATH}/sysint/sdvAgent
#        cp tools/sdvAgent_init ${RDK_FSROOT_PATH}/sysint/sdvAgent
#    fi

}

function test()
{
    echo "======= Test sdvAgent ======="

    make check

    ./test/sdvagent_test --gtest_output=xml:build/gtestresults.xml

    ./tools/gcovr -r . --xml > build/coverage.xml
}


# run the logic

#these args are what left untouched after parse_args
HIT=false

for i in "$ARGS"; do
    case $i in
        configure)  HIT=true; configure ;;
        clean)      HIT=true; clean ;;
        build)      HIT=true; build ;;
        rebuild)    HIT=true; rebuild ;;
        install)    HIT=true; install ;;
        test)       HIT=true; test ;;
        *)
            #skip unknown
        ;;
    esac
done

# if not HIT do build by default
if ! $HIT; then
  build
fi
