/*
* ============================================================================
* RDK MANAGEMENT, LLC CONFIDENTIAL AND PROPRIETARY
* ============================================================================
* This file (and its contents) are the intellectual property of RDK Management, LLC.
* It may not be used, copied, distributed or otherwise  disclosed in whole or in
* part without the express written permission of RDK Management, LLC.
* ============================================================================
* Copyright (c) 2014 RDK Management, LLC. All rights reserved.
* ============================================================================
*/

#include "SDVAgent.h"

#include "IARMProxyService.h"
#include "TRMMonitorService.h"
#include "UserActivityService.h"

#include "rdk_debug.h"

using namespace sdv;

#define INIT_REQUEST_RETRY_BACKOFF_MS 60000   //One minute in ms

SdvAgent::SdvAgent(){
    RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "SdvAgent<constructor>\n");

    this->_eventQueue = new sdv::SDVEventQueue();
    this->iarmProxyService = new sdv::IARMProxyService();
    this->_configParser = new sdv::ConfigurationParser(iarmProxyService,_eventQueue, this, &SdvAgent::mcpUpdated);
    this->_messageService = new sdv::MessageService(_configParser, _eventQueue );
    this->trmMonitor = new TRMMonitorService(_eventQueue);

    this->_state = SDV_STATE_NOT_STARTED;
}

SdvAgent::~SdvAgent(){
    stop();

    delete this->userActivity;
    delete this->sessionHandler;
    delete this->_messageService;
    delete this->_configParser;
    delete this->trmMonitor;
    delete this->iarmProxyService;
    delete this->_eventQueue;
}

void SdvAgent::start(){
    RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "SdvAgent.start: begin\n");
    if(_eventQueue->start() && iarmProxyService->start()){
        _state = SDV_STATE_WAITING_FOR_MCP;

        trmMonitor->start();

        // Session handler cannot be created until after IARM Proxy Service has started
        sessionHandler = SDVSessionHandler::createInstance(iarmProxyService, _messageService, _eventQueue, _configParser, trmMonitor);

        // These services depend on SessionHandler and therefore must be created after handler created
        userActivity = new UserActivityService(iarmProxyService, _messageService, _eventQueue, _configParser, sessionHandler, trmMonitor);
    }
    else{
        iarmProxyService->stop();
        RDK_LOG(RDK_LOG_FATAL, "LOG.RDK.SDVAGENT", "SdvAgent.start: Failed to start core services service\n");
    }

    _configParser->start();

    uint32_t dummyPayload;
    IARM_Result_t result = IARM_Bus_BroadcastEvent(IARM_BUS_SDVAGENT_NAME, SDVAGENT_REQUEST_LATEST_MC_DATA, &dummyPayload, sizeof(dummyPayload));
    if (result != IARM_RESULT_SUCCESS) {
        RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.SDVAGENT", "SdvAgent.start: IARM_Bus_BroadcastEvent failed - %d\n", result);
    }
    RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "SdvAgent.start: completed\n");
}

void SdvAgent::stop(){
    RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "SdvAgent.stop\n");
    switch (this->_state){
    case SDV_STATE_READY:
        userActivity->stop();
        trmMonitor->stop();
        //no break
    case SDV_STATE_WAITING_FOR_INIT_CONFIRM:
        this->_messageService->stop();
        //no break
    case SDV_STATE_WAITING_FOR_MCP:
        //no break
    case SDV_STATE_NOT_STARTED:
        this->iarmProxyService->stop();
        this->_eventQueue->stop();
        break;
    }
}

void SdvAgent::mcpUpdated(void * ptrObject, bool isServiceGroupChange, bool isServerIpChange){
    sdv::ConfigurationParser * configParser = ((SdvAgent*)ptrObject)->_configParser;

    RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "SdvAgent.mcpUpdated: state=%d isServiceGroupChange=%d isServerIpChange=%d\n", ((SdvAgent*)ptrObject)->_state, isServiceGroupChange, isServerIpChange);

    switch (((SdvAgent*)ptrObject)->_state) {

    case SDV_STATE_READY:
        ((SdvAgent*)ptrObject)->_state = SDV_STATE_WAITING_FOR_INIT_CONFIRM;
        if (isServiceGroupChange) {
        	((SdvAgent*)ptrObject)->sessionHandler->forceTuneAllSessions();
        }
        if (isServerIpChange) {
            ((SdvAgent*)ptrObject)->_messageService->restart();
        }
        ((SdvAgent*)ptrObject)->sendInitRequest(0);
        break;

    case SDV_STATE_WAITING_FOR_INIT_CONFIRM:
        if (isServerIpChange) {
        	((SdvAgent*)ptrObject)->_messageService->restart();
        }
        // No sessions should be open during init confirm state, therefore no need to force tune
    	((SdvAgent*)ptrObject)->sendInitRequest(0);
        break;

    case SDV_STATE_WAITING_FOR_MCP:
        ((SdvAgent*)ptrObject)->_state = SDV_STATE_WAITING_FOR_INIT_CONFIRM;
        ((SdvAgent*)ptrObject)->_messageService->start();
        ((SdvAgent*)ptrObject)->sendInitRequest(0);

        /* Assumption made here that InitRequest will succeed and therefore we will start handling sessions now
         	 instead of waiting for InitConfirm */
        ((SdvAgent*)ptrObject)->sessionHandler->start();
        ((SdvAgent*)ptrObject)->userActivity->start();
        break;

    default:
        RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.SDVAGENT", "mcpUpdated Illegal state transition");
        break;
    }
}

void SdvAgent::initResponse(void * ptrObject, MessageService::REPONSE_STATUS_t status, SDVMessage * msg){
    if(	((SdvAgent*)ptrObject)->_state != SDV_STATE_WAITING_FOR_INIT_CONFIRM){
        RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.SDVAGENT", "SdvAgent.initResponse: Illegal State transition\n");
        return;
    }
    if(status == MessageService::TIMEOUT )	{	//Check for timeout
        RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.SDVAGENT", "SdvAgent.initResponse: timeout occurred, retry delay:%d\n", INIT_REQUEST_RETRY_BACKOFF_MS);
        ((SdvAgent*)ptrObject)->sendInitRequest(INIT_REQUEST_RETRY_BACKOFF_MS);
        return;
    }
    InitConfirm * confirm = (InitConfirm*)msg;
    RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "SdvAgent.initResponse: response code : %d\n", confirm->getResponseCode());

    switch(confirm->getResponseCode()){
    case sdv::InitConfirm::RspOK:{
        ((SdvAgent*)ptrObject)->_state = SDV_STATE_READY;
        break;
    }

    case sdv::InitConfirm::RspInvalidSG: {
        RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "SdvAgent.initResponse: invalid Service Group; request new MC data\n");
        uint32_t dummyPayload;
        IARM_Result_t result = IARM_Bus_BroadcastEvent(IARM_BUS_SDVAGENT_NAME, SDVAGENT_REQUEST_FRESH_MC_DATA, &dummyPayload, sizeof(dummyPayload));
        if (result != IARM_RESULT_SUCCESS) {
            RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.SDVAGENT", "SdvAgent.initResponse: IARM_Bus_BroadcastEvent failed - %d\n", result);
        }
        break;
    }

    case sdv::InitConfirm::RspSvrCapacityExceeded:
    case sdv::InitConfirm::RspVerNotSupported:
    case sdv::InitConfirm::RspUnkownError:
        ((SdvAgent*)ptrObject)->sendInitRequest(INIT_REQUEST_RETRY_BACKOFF_MS);
        break;
    }
}

void SdvAgent::sendInitRequest(int delay) {
    InitRequest * t = new InitRequest(_configParser->getStbMac(), _configParser->getServiceGroupId());
    _messageService->sendMessage(t, initResponse, this, delay);
}
