/*
 * @file SDVEventQueue.cpp
 * @Author Jason Bedard jasbedar@cisco.com
 * @Date Jun 20 2014
 *
 * @brief Implementation of the SDVEventQueue.
 */
#include "SDVEventQueue.h"
#include <ctime>
#include <sys/time.h>
#include <stdio.h>
#include "rdk_debug.h"

using namespace sdv;


SDVEventQueue::QueuedEvent::QueuedEvent(SDVEventQueue::TASK_ID_t id, SDVEventQueue::SDV_EVENT_CALLBACK_t callback, void* ptrInstance, void * data, long timeout_ms){
    _id = id;
    _callback = callback;
    _ptrInstance = ptrInstance;
    _data = data;
    _timeout = SDVEventQueue::getNowTimeInMillis() + timeout_ms;
}

bool SDVEventQueue::QueuedItemComparator::operator()(const SDVEventQueue::QueuedEvent * a, const SDVEventQueue::QueuedEvent * b) const{
    return (a->_timeout > b->_timeout);
}

void *SDVEventQueue::doLoop(void * eventQueueInstance ) {
    SDVEventQueue * instance = (SDVEventQueue*)eventQueueInstance;
    while(instance->_running){
        QueuedEvent* event = NULL;
        pthread_mutex_lock(instance->_events_mutex);
        RDK_LOG(RDK_LOG_DEBUG,"LOG.RDK.SDVAGENT","[SDVEventQueue] doLoop: events queued=%d\n", instance->_queuedEvents.size());

        if(!instance->_queuedEvents.empty()) {
            event = instance->_queuedEvents.top();
            uint64_t tnow =  getNowTimeInMillis();
            RDK_LOG(RDK_LOG_DEBUG,"LOG.RDK.SDVAGENT","[SDVEventQueue] doLoop: eventId=%d nowTime=%llu expireTime=%llu\n",  event->_id, tnow, event->_timeout);

            // If event timeout has not yet expired, block thread waiting for timeout or new push to queue
            if( tnow < event->_timeout) {
            	struct timespec waitTime;
            	waitTime.tv_sec = (event->_timeout / 1000);
            	waitTime.tv_nsec = (event->_timeout % 1000) * 1000 * 1000;

            	RDK_LOG(RDK_LOG_DEBUG,"LOG.RDK.SDVAGENT","[SDVEventQueue] doLoop: starting thread timed wait, %lu:%lu\n", waitTime.tv_sec, waitTime.tv_nsec);
                pthread_cond_timedwait(instance->_event_add_cv, instance->_events_mutex, &waitTime);
                event = NULL;
            }
            // Else remove the event from the queue
            else {
            	RDK_LOG(RDK_LOG_DEBUG,"LOG.RDK.SDVAGENT","[SDVEventQueue] doLoop: pop event from queue\n");
                instance->_queuedEvents.pop();
            }
        }
        else {
            pthread_cond_wait(instance->_event_add_cv, instance->_events_mutex);
        }
        pthread_mutex_unlock(instance->_events_mutex);

        //Fire the Event
        if(event != NULL){
            try{
                event->_callback(event->_id, event->_ptrInstance, event->_data);
            }catch(std::exception const & ex) {
            	RDK_LOG(RDK_LOG_ERROR,"LOG.RDK.SDVAGENT","[SDVEventQueue] doLoop: Exception from callback \n");
            }
            delete event;
        }
    }
    RDK_LOG(RDK_LOG_INFO,"LOG.RDK.SDVAGENT","[SDVEventQueue] doLoop: Exit\n");
    pthread_exit(NULL);
    return 0;
}

SDVService::SERVICE_RESULT SDVEventQueue::start(){
	RDK_LOG(RDK_LOG_INFO,"LOG.RDK.SDVAGENT","[SDVEventQueue] start\n");
    _running  = true;

    _runThread = (pthread_t *) malloc(sizeof(pthread_t));
    _attr = (pthread_attr_t *) malloc(sizeof(pthread_attr_t));
    _events_mutex = (pthread_mutex_t *) malloc(sizeof(pthread_mutex_t));
    _event_add_cv = (pthread_cond_t *) malloc(sizeof(pthread_cond_t));

    pthread_mutex_init(_events_mutex, NULL);
    pthread_cond_init(_event_add_cv, NULL);
    pthread_attr_init(_attr);

    pthread_attr_setdetachstate(_attr, PTHREAD_CREATE_JOINABLE);
    int rc = pthread_create(_runThread, _attr, doLoop, (void*)this);
    if(rc){
    	RDK_LOG(RDK_LOG_ERROR,"LOG.RDK.SDVAGENT","[SDVEventQueue] start: failed to create thread!\n");
        return FAILURE;
    }
    RDK_LOG(RDK_LOG_INFO,"LOG.RDK.SDVAGENT","[SDVEventQueue] start: Thread Created\n");
    return sdv::SDVService::SUCCESS;
}

SDVService::SERVICE_RESULT SDVEventQueue::stop(){
    if(!_running){
        return SUCCESS;
    }
    RDK_LOG(RDK_LOG_INFO,"LOG.RDK.SDVAGENT","[SDVEventQueue] stop: stopping thread\n");
    _running = false;
    pthread_mutex_lock(_events_mutex);
    pthread_cond_signal(_event_add_cv);
    pthread_mutex_unlock(_events_mutex);

    pthread_join(*_runThread, &_status);

    while(!_queuedEvents.empty()){
        QueuedEvent* event = _queuedEvents.top();
        _queuedEvents.pop();
        delete event;
    }

    pthread_mutex_destroy(_events_mutex);
    pthread_attr_destroy(_attr);
    pthread_cond_destroy(_event_add_cv);
    free(_runThread);
    free(_attr);
    free(_events_mutex);
    free(_event_add_cv);

    RDK_LOG(RDK_LOG_INFO,"LOG.RDK.SDVAGENT","[SDVEventQueue] stop: thread stopped\n");
    return SDVService::SUCCESS;
}

SDVEventQueue::TASK_ID_t SDVEventQueue::push(void* ptrInstance, SDVEventQueue::SDV_EVENT_CALLBACK_t callback,  void * data, uint32_t timeout_ms){
    pthread_mutex_lock(_events_mutex);
    TASK_ID_t id = _nextTaskId++;
    QueuedEvent * event = new QueuedEvent(id, callback, ptrInstance, data, timeout_ms );
    RDK_LOG(RDK_LOG_DEBUG,"LOG.RDK.SDVAGENT","[SDVEventQueue] push: id=%d timeout=%llu\n", id, event->_timeout);
    _queuedEvents.push(event);
    pthread_cond_signal(_event_add_cv);
    pthread_mutex_unlock(_events_mutex);
    return id;
}

uint64_t SDVEventQueue::getNowTimeInMillis() {
    struct timeval tv;
    gettimeofday(&tv, NULL);
    return ((uint64_t)tv.tv_sec * 1000) + ((uint64_t)tv.tv_usec / 1000);
}

