/*
* ============================================================================
* RDK MANAGEMENT, LLC CONFIDENTIAL AND PROPRIETARY
* ============================================================================
* This file (and its contents) are the intellectual property of RDK Management, LLC.
* It may not be used, copied, distributed or otherwise  disclosed in whole or in
* part without the express written permission of RDK Management, LLC.
* ============================================================================
* Copyright (c) 2014 RDK Management, LLC. All rights reserved.
* ============================================================================
*/

#include "SDVSessionHandler.h"
#include "ProgramSelectRequest.h"
#include "ProgramSelectConfirm.h"
#include "QueryRequest.h"
#include "QueryConfirm.h"
#include "InitRequest.h"
#include "InitConfirm.h"

#include <ctime>
#include <sys/time.h>
#include <assert.h>

#include "rdk_debug.h"


using namespace sdv;



static SDVSessionHandler* instance;

SDVSessionHandler *SDVSessionHandler::createInstance(IARMProxyService* iarmProxy, MessageService* messageService,
        SDVEventQueue* eventQueue, ConfigurationParser* config, TRMMonitorService* trmService){
    instance = new SDVSessionHandler(iarmProxy, messageService, eventQueue, config, trmService);
    return instance;
}

SDVSessionHandler::SDVSessionHandler(IARMProxyService* iarmProxy, MessageService* messageService, SDVEventQueue* queue, ConfigurationParser* config, TRMMonitorService* trmService) {
	isReady = false;
	_iarmProxy = iarmProxy;
    _messageService = messageService;
    _queue = queue;
    _config = config;
    _trmService = trmService;

    IARM_Result_t result = _iarmProxy->registerIARMRPC(IARM_BUS_SDVAGENT_API_OPEN_SESSION, (IARM_BusCall_t)openSession);
    assert(IARM_RESULT_SUCCESS == result);

    result = _iarmProxy->registerIARMRPC(IARM_BUS_SDVAGENT_API_CLOSE_SESSION, (IARM_BusCall_t)closeSession);
    assert(IARM_RESULT_SUCCESS == result);

    result = _iarmProxy->registerIARMRPC(IARM_BUS_SDVAGENT_API_STREAM_START, (IARM_BusCall_t)startStream);
    assert(IARM_RESULT_SUCCESS == result);

    result =_iarmProxy->registerIARMRPC(IARM_BUS_SDVAGENT_API_STREAM_STOP, (IARM_BusCall_t)stopStream);
    assert(IARM_RESULT_SUCCESS == result);

    pthread_mutexattr_t attr;
    assert(pthread_mutexattr_init(&attr) == 0);
    assert(pthread_mutexattr_settype(&attr, PTHREAD_MUTEX_ERRORCHECK) == 0);
    assert(pthread_mutex_init(&sessions_mutex, &attr) == 0);

    assert(pthread_cond_init(&sessions_cv, NULL) == 0);

    RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] created\n");
}

SDVSessionHandler::~SDVSessionHandler(){
    RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] destroyed\n");
}

SDVService::SERVICE_RESULT SDVSessionHandler::start(){
    pthread_mutex_lock(&instance->sessions_mutex);

    isReady = true;
    pthread_cond_signal(&instance->sessions_cv); // open thread may be waiting for handler to be ready during tune time autodiscovery

    pthread_mutex_unlock(&instance->sessions_mutex);

    RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] started\n");
    return SUCCESS;
}

SDVService::SERVICE_RESULT SDVSessionHandler::stop(){
    IARM_Result_t result;
    result = _iarmProxy->registerIARMRPC(IARM_BUS_SDVAGENT_API_OPEN_SESSION, (IARM_BusCall_t)openSession);

    pthread_mutex_destroy(&sessions_mutex);
    pthread_cond_destroy(&sessions_cv);

    RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] stopped\n");
    return SUCCESS;
}

SDVSessionHandler::SDV_SESSION* SDVSessionHandler::getSessionByTunerId(uint32_t tunerId) {
	SDV_SESSION_MAP::iterator iter = sessionList.find(tunerId);
	if(iter != sessionList.end()) {
		return iter->second;
	}
	else {
		return NULL;
	}
}

SDVSessionHandler::SDV_SESSION* SDVSessionHandler::getSessionBySourceId(uint32_t sourceId) {
	for(SDV_SESSION_MAP::iterator iter = sessionList.begin(); iter != sessionList.end(); iter++) {
	    if (iter->second->sourceId == sourceId) {
	    	return iter->second;
	    }
	}
	return NULL;
}

SDVSessionHandler::SDV_SESSION* SDVSessionHandler::createSession(uint32_t sourceId, uint32_t tunerId) {
	SDV_SESSION* session = new SDV_SESSION(sourceId, tunerId);
	sessionList[tunerId] = session;
    RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] createSession() session=%d sourceId=%d tunerId=%d\n", session, sourceId, tunerId);
    return session;
}

void SDVSessionHandler::deleteSession(uint32_t tunerId) {
	SDV_SESSION_MAP::iterator iter = sessionList.find(tunerId);
	if(iter != sessionList.end()) {
	    RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] deleteSession() session=%d sourceId=%d tunerId=%d\n", iter->second, iter->second->sourceId, iter->second->tunerId);
		delete iter->second;
		sessionList.erase(iter);
	}
}

void SDVSessionHandler::handleLinearOpen(uint32_t sourceId, uint32_t tunerId, uint8_t tunerUseFlags) {

	pthread_mutex_lock(&sessions_mutex);

	// If an SDV session exists for this tuner, delete it
	SDV_SESSION* session = getSessionByTunerId(tunerId);

	if (session != NULL) {
        RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] handleLinearOpen() delete existing sdv session for sourceId=%d tunerId=%d\n", session->sourceId, session->tunerId);
        deleteSession(tunerId);
    }

	// Send message to server without expectations of confirm message
    sendConfirmLessProgramSelectRequest(sourceId, tunerId, tunerUseFlags);

    pthread_mutex_unlock(&sessions_mutex);
}

IARM_Result_t SDVSessionHandler::openSession(IARM_SDVAGENT_OPEN_SESSION_PAYLOAD *sessionPayload){
    uint32_t sourceId = sessionPayload->requestInfo.sourceId;
    uint32_t tunerId = sessionPayload->requestInfo.tunerId;

    uint8_t tunerUse = instance->_trmService->getTunerUsage(sourceId);

    RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] openSession() sourceId=%u tunerId=%u usage=0x%x existing sessions=%d\n",
    		sourceId, tunerId, tunerUse, instance->sessionList.size());

    pthread_mutex_lock(&instance->sessions_mutex);

    // Return now if linear service and we've not received MC data
    if (!sessionPayload->requestInfo.isSdvService && !instance->isReady) {
        sessionPayload->responseInfo.status = PROGRAM_SELECT_ERROR_serviceNotSDV;
        pthread_mutex_unlock(&instance->sessions_mutex);
        return IARM_RESULT_SUCCESS;
    }

    if (!instance->isReady) {
    	// Set absolute time we will wait for handler to be readied by SdvAgent
    	struct timespec waitTime;
        struct timeval currentTime;
        gettimeofday(&currentTime, NULL);
        waitTime.tv_sec = currentTime.tv_sec + HANDLER_NOT_READY_DELAY_SECS;
        waitTime.tv_nsec = currentTime.tv_usec * 1000;

        RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] openSession() not ready, begin timed wait\n");
    	pthread_cond_timedwait(&instance->sessions_cv, &instance->sessions_mutex, &waitTime);

    	// If still not ready, return error
    	if (!instance->isReady) {
            RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] openSession() still not ready, return error\n");
            pthread_mutex_unlock(&instance->sessions_mutex);
            return IARM_RESULT_INVALID_STATE;
    	}
    }

    // If not an SDV channel, handle linear service and set error status in response
    if (!sessionPayload->requestInfo.isSdvService) {
        RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] openSession() service is not switched\n");
        instance->handleLinearOpen(sourceId, tunerId, tunerUse);
        sessionPayload->responseInfo.status = PROGRAM_SELECT_ERROR_serviceNotSDV;
        pthread_mutex_unlock(&instance->sessions_mutex);
        return IARM_RESULT_SUCCESS;
    }

    // If a session already exists for this tuner, update it now
    SDV_SESSION* session = instance->getSessionByTunerId(tunerId);
    if (session != NULL) {
        RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] openSession() update existing session having sourceId=%d state=%d\n",session->sourceId, session->state);
    		session->state = SDV_SESSION::OPENING;
    		session->sourceId = sourceId;
    }
    // Else create a new session
    else {
    	session = instance->createSession(sourceId, tunerId);
    }

    // Send request message to server and wait for response
    instance->sendProgramSelectRequest(session);
	RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] openSession() wait for ProgramSelectConfirm...\n");
	pthread_cond_wait(&instance->sessions_cv, &instance->sessions_mutex);
	RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] openSession() ProgramSelectConfirm received for sourceId %d\n", sourceId);

	// Copy tuning info into return payload and handle failure if necessary
	sessionPayload->responseInfo = session->tuningInfo;
	if (session->state == SDV_SESSION::FAILED) {
		instance->handleFailedOpen(session);
	}

    pthread_mutex_unlock(&instance->sessions_mutex);

    return IARM_RESULT_SUCCESS;
}

void SDVSessionHandler::handleFailedOpen(SDV_SESSION* session) {

	// If failure due to possible configuration data problem, request fresh MC data
	if (session->tuningInfo.status == PROGRAM_SELECT_CONFIRM_rspInvalidSG ||
			session->tuningInfo.status == PROGRAM_SELECT_CONFIRM_rspUnknownClient) {
        RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] handleFailedOpen() send IARM event to Autodiscover to request fresh MC data\n");
        uint32_t dummyPayload;
        IARM_Result_t result = IARM_Bus_BroadcastEvent(IARM_BUS_SDVAGENT_NAME, SDVAGENT_REQUEST_FRESH_MC_DATA, &dummyPayload, sizeof(dummyPayload));
        if (result != IARM_RESULT_SUCCESS) {
            RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] handleFailedOpen() IARM_Bus_BroadcastEvent failed - %d\n", result);
        }
	}
	deleteSession(session->tunerId);
}

IARM_Result_t SDVSessionHandler::closeSession(IARM_SDVAGENT_CLOSE_SESSION_PAYLOAD *sessionPayload){
	RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] closeSession() sourceId=%u tunerId=%u\n", sessionPayload->sourceId, sessionPayload->tunerId);

    pthread_mutex_lock(&instance->sessions_mutex);

    /* If session in list, queue delay for tune-away program select request.  The purpose of the delay is to prevent additional
     * outband server messages in case user is channel surfing on a single tuner */
	SDV_SESSION* session = instance->getSessionByTunerId(sessionPayload->tunerId);
	if (session != NULL) {
	    RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] closeSession() queue tune-away delay\n");
		session->state = SDV_SESSION::CLOSING;
		instance->_queue->push(instance, instance->sessionTuneAwayCallback, (void *)sessionPayload->tunerId, SESSION_CLOSE_TUNEAWAY_DELAY);
	}
	else {
		RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] closeSession() session not found\n");
	}

    pthread_mutex_unlock(&instance->sessions_mutex);

    return IARM_RESULT_SUCCESS;
}

void SDVSessionHandler::sessionTuneAwayCallback(SDVEventQueue::TASK_ID_t qTaskId, void* ptrInstance, void* data) {
	 // data pointer contains 32-bit tuner ID; union used to prevent compiler errors on 64-bit systems
	union {
		uint32_t tunerId;
		void* ptrData;
	} voidPtrConverter;	// converts void* to uint32 tunerId data independent of CPU word size
	voidPtrConverter.ptrData = data;

    pthread_mutex_lock(&instance->sessions_mutex);
    SDV_SESSION* session = instance->getSessionByTunerId(voidPtrConverter.tunerId);

    if (session != NULL) {
        // If session is still in the process of closing, notify server tuner no longer in use
    	if (session->state == SDV_SESSION::CLOSING) {
    		RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] sessionTuneAwayCallback() send tune-away and delete session for sourceid=%d tunerId=%d\n", session->sourceId, voidPtrConverter.tunerId);
    		instance->sendConfirmLessProgramSelectRequest(0x00, voidPtrConverter.tunerId, TRMMonitorService::BACKGROUND);
    		instance->deleteSession(session->tunerId);
    	}
    	// Else tuner must have been opened for another sdv session during tune-away delay
    	else {
    		RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] sessionTuneAwayCallback() session re-opened with sourceid=%d tunerId=%d\n", session->sourceId, voidPtrConverter.tunerId);
    	}
    }
    // Session would no longer exist if tuner was opened on linear channel during tune-away delay
    else {
		RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] sessionTuneAwayCallback() session not found for tunerId %d\n", voidPtrConverter.tunerId);
    }

    pthread_mutex_unlock(&instance->sessions_mutex);
}

void SDVSessionHandler::sendProgramSelectRequest(SDV_SESSION* session) {
    ProgramSelectRequest * request = new ProgramSelectRequest(
    		_config->getStbMac(),
            (uint8_t)session->tunerId,
            _trmService->getTunerUsage(session->sourceId),
            0xFF,   //TODO RETRY COUNT
            session->sourceId,
            _config->getServiceGroupId());

	_messageService->sendMessage(request, programSelectConfirm, session, 0);
}

void SDVSessionHandler::sendConfirmLessProgramSelectRequest(uint32_t sourceId, uint32_t tunerId, uint8_t tunerUseFlags){
    ProgramSelectRequest * request = new ProgramSelectRequest(
    		_config->getStbMac(),
            (uint8_t)tunerId,
            tunerUseFlags,
            0xFF,
            sourceId,
            _config->getServiceGroupId());

	_messageService->sendMessage(request, NULL, NULL, 0);
}

void SDVSessionHandler::programSelectConfirm(void* ptrObject,MessageService::REPONSE_STATUS_t status, sdv::SDVMessage* response) {
	SDV_SESSION* session = (SDV_SESSION *)ptrObject;

    pthread_mutex_lock(&instance->sessions_mutex);

	if (session != NULL) {
	    RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] programSelectConfirm() session=%d sourceId=%d tunerId=%d\n", session, session->sourceId, session->tunerId);

	    if (status == MessageService::TIMEOUT) {
			session->tuningInfo.status = MESSAGE_SENT_rspTimedOut;
			session->state = SDV_SESSION::FAILED;
		    RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] programSelectConfirm() failed due to timeOut\n");
		}

	    // If not timed out, we should have a ProgramSelectConfirm response message
	    else if (response != NULL) {
		    ProgramSelectConfirm* confirm = (ProgramSelectConfirm *)response;

		    // If SelectConfirm indicates server no longer recognizes us, initiate re-init and return keeping openSession thread waiting
            if (confirm->getResponseCode() == PROGRAM_SELECT_CONFIRM_rspUnknownClient) {
                RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] programSelectConfirm() failed due to rspUnknownClient, attempt re-init\n");
                instance->sendInitRequestForUnknownClient(session);
                pthread_mutex_unlock(&instance->sessions_mutex);
                return;
            }
            // Else if any other SelectConfirm failures
            else if (confirm->getResponseCode() != PROGRAM_SELECT_CONFIRM_rspOK) {
				session->tuningInfo.status = confirm->getResponseCode();
				session->state = SDV_SESSION::FAILED;
			    RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] programSelectConfirm() failed due to %d\n", session->tuningInfo.status);
			}
            // Else SelectConfirm was successful
			else {
				session->tuningInfo.status = PROGRAM_SELECT_CONFIRM_rspOK;
				session->tuningInfo.carrierFreq = confirm->getFrequency();
				session->tuningInfo.programNum = confirm->getMpegProgramNumber();
				session->tuningInfo.modulationFormat = confirm->getModulation();
				session->state = SDV_SESSION::ACTIVE;
			    RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] programSelectConfirm() freq=%d progNum=%d\n", session->tuningInfo.carrierFreq, session->tuningInfo.programNum);
			}
		}
		else {
			RDK_LOG(RDK_LOG_FATAL, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] programSelectConfirm() null SDVMessage object in callback!\n");
		}
	}
	else {
		RDK_LOG(RDK_LOG_FATAL, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] programSelectConfirm() null session object in callback!\n");
	}

	// Signal openSession thread of confirm
    pthread_cond_signal(&instance->sessions_cv);
    pthread_mutex_unlock(&instance->sessions_mutex);
}

IARM_Result_t SDVSessionHandler::startStream(iarm_sdvagent_stream_start_payload *sessionPayload){

	RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "Starting stream: srcId:%u client:%s\n", sessionPayload->sourceId, sessionPayload->clientIpAddr.c_str());

    return IARM_RESULT_SUCCESS;
}

IARM_Result_t SDVSessionHandler::stopStream(iarm_sdvagent_stream_stop_payload *sessionPayload){

	RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "Stopping stream: srcId:%u client:%s\n", sessionPayload->sourceId, sessionPayload->clientIpAddr.c_str());

    return IARM_RESULT_SUCCESS;
}

int SDVSessionHandler::getOpenSession(uint32_t tunerId, SDV_SESSION* session) {
	int rval = -1;
	pthread_mutex_lock(&sessions_mutex);

	// If opening or active session found, copy to callers storage and return 0
	SDV_SESSION* openSession = getSessionByTunerId(tunerId);
	if (openSession != NULL && (openSession->state == SDV_SESSION::OPENING || openSession->state == SDV_SESSION::ACTIVE)) {
		*session = *openSession;
		rval = 0;
	}
    pthread_mutex_unlock(&sessions_mutex);
    return rval;
}

int SDVSessionHandler::getAnyOpenSession(SDV_SESSION* session) {
	int rval = -1;
	pthread_mutex_lock(&sessions_mutex);

	// Find and copy the first open session in our map
	for(SDV_SESSION_MAP::iterator iter = sessionList.begin(); iter != sessionList.end(); iter++) {
		SDV_SESSION* openSession = iter->second;
		if ((openSession->state == SDV_SESSION::OPENING || openSession->state == SDV_SESSION::ACTIVE)) {
			*session = *openSession;
			rval = 0;
			break;
	    }
	}
    pthread_mutex_unlock(&sessions_mutex);
    return rval;
}

void SDVSessionHandler::forceTune(SDV_SESSION* session) {
    RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] forceTune() tunerId=%d  sourceId=%d\n", session->tunerId, session->sourceId);

	// Create event and send it to mediaframework
	IARM_SDVAGENT_FORCE_TUNE_PAYLOAD payload;
	payload.currentSourceId = session->sourceId;
	payload.tunerId = session->tunerId;
    IARM_Result_t result = IARM_Bus_BroadcastEvent(IARM_BUS_SDVAGENT_NAME, SDVAGENT_REQUEST_FORCE_TUNE, &payload, sizeof(IARM_SDVAGENT_FORCE_TUNE_PAYLOAD));
    if (result != IARM_RESULT_SUCCESS) {
        RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] forceTune() IARM_Bus_BroadcastEvent failed - %d\n", result);
    }
}

void SDVSessionHandler::forceTuneAllSessions() {
	pthread_mutex_lock(&sessions_mutex);

	for(SDV_SESSION_MAP::iterator iter = sessionList.begin(); iter != sessionList.end(); iter++) {
		SDV_SESSION* session = iter->second;
		if ((session->state == SDV_SESSION::OPENING || session->state == SDV_SESSION::ACTIVE)) {
			forceTune(session);
	    }
	}
    pthread_mutex_unlock(&sessions_mutex);
}

int SDVSessionHandler::getOpenSessionTunerId(uint32_t sourceId) {
	int rval;

	pthread_mutex_lock(&sessions_mutex);

	// If opening or active session found, return associated tuner ID
	SDV_SESSION* session = getSessionBySourceId(sourceId);
	if (session != NULL && (session->state == SDV_SESSION::OPENING || session->state == SDV_SESSION::ACTIVE)) {
		rval = session->tunerId;
	}
	else {
		rval = -1;
	}
    pthread_mutex_unlock(&sessions_mutex);

    return rval;
}

void SDVSessionHandler::sendInitRequestForUnknownClient(SDV_SESSION* session) {
    InitRequest * request = new InitRequest(_config->getStbMac(), _config->getServiceGroupId());
    _messageService->sendMessage(request, initConfirmForUnknownClient, session, 0);
}

void SDVSessionHandler::initConfirmForUnknownClient(void* ptrObject, MessageService::REPONSE_STATUS_t status, sdv::SDVMessage* msg) {
    SDV_SESSION* session = (SDV_SESSION *)ptrObject;
    InitConfirm* confirm = (InitConfirm *)msg;

    // If Init succeeded, resend ProgramSelectRequest
    if (confirm->getResponseCode() == InitConfirm::RspOK) {
        RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] %s() response ok; resend ProgramSelectRequest for sourceId %d\n", __FUNCTION__, session->sourceId);
        instance->sendProgramSelectRequest(session);
    }

    // Else Init failed, fail session open
    else {
        pthread_mutex_lock(&instance->sessions_mutex);
        session->tuningInfo.status = PROGRAM_SELECT_CONFIRM_rspUnknownClient;  // use original error code
        session->state = SDV_SESSION::FAILED;

        if (status == MessageService::TIMEOUT) {
            RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] %s() timeout failure for sourceId %d\n", __FUNCTION__, session->sourceId);
        }
        else {
            RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.SDVAGENT", "[SDVSessionHandler] %s() response failure %d for sourceId %d\n", __FUNCTION__, confirm->getResponseCode(), session->sourceId);
        }

        // Signal openSession thread of confirm
        pthread_cond_signal(&instance->sessions_cv);
        pthread_mutex_unlock(&instance->sessions_mutex);
    }
}



