/*
* @file UdpService.h
*
* @brief Provides UDP communications services for SDV
*
* @par Document
* Document reference.
*
* @par Open Issues (in no particular order)
* -# None
*
* @par Assumptions
* -# None
*
* @par Implementation Notes
* -# None
*/

#ifndef UDPSOCKETSERVICE_H_
#define UDPSOCKETSERVICE_H_

#include <stdint.h>
#include <pthread.h>
#include <netinet/in.h>


/**
 * @def UDP_PACKET_BUFF_LEN
 * @brief Length of packet buffer easily capable of holding any SDV message.
 */
#define UDP_PACKET_BUFF_LEN 512

/**
 * @def TOTAL_UDP_PACKET_BUFFS
 * @brief Total number of packet buffers available for UDP communications.
 */
#define TOTAL_UDP_PACKET_BUFFS 8

namespace sdv {

/**
 * @class UdpService
 * @brief Class provides UDP communication services and associated packet buffering.
 */
class UdpService {
public:

	/**
	 * @typedef PACKET_BUFF
	 * @brief UDP packet buffer control structure
	 */
	typedef struct packet_buff {
		uint8_t* data;				//!< packet data bytes
		uint32_t data_length;		//!< length in bytes of packet data
		uint32_t max_length;		//!< maximum length in bytes of packet buffer
		struct packet_buff* next;
	} PACKET_BUFF;

	/**
	 * @typedef DATA_RX_CALLBACK
	 * @brief Call back to invoke when UDP data packet is received from server.
	 *
	 * @details To minimize the risk of losing received packets, the callback function
	 * should quickly put the packet in some kind of queue and schedule a different
	 * thread to process it.
	 *
	 * @param [in] buff - pointer to PACKET_BUFF structure containing received UDP data
	 * @param [in] callbackInstance - pointer to instance of callback
	 */
	typedef void (DATA_RX_CALLBACK)(PACKET_BUFF* buff, void* callbackInstance);

	/**
	 * @constructor
	 * @brief Constructor will create and open both send and read sockets and start socket read thread.
	 * @details Assumes the send and receive ports are the same.
	 *
	 * @param [in] event_queue - callback to invoke when a packet is received
	 * @param [in] server_ip_addr - IP address string of remote host to send datagram packets.  String must be either
	 * 						   IPv4 format "xxx.xxx.xxx.xxx" or IPv6 format "xxxx:xxxx:xxxx:xxxx:xxxx:xxxx:xxxx:xxxx"
	 * @param [in] server_port - port to send and receive datagram packets
	 * @param [in] callbackInstance - instance of the callback
	 */
	UdpService(DATA_RX_CALLBACK* recv_callback, void* callbackInstance, const char* server_ip_addr, const uint16_t server_port);

	/**
	 * @Destrcutor
	 * @brief Destructor to kill thread and free all resources.
	 */
	~UdpService();

	/**
	 * @fn getEmptyPacket
	 * @brief Get an empty UDP packet buffer of size UDP_PACKET_BUFF_LEN.
	 *
	 * @return pointer to empty packet buffer control structure; NULL if out of buffers
	 */
	PACKET_BUFF* getEmptyPacket();

	/**
	 * @fn freePacket
	 * @brief Free the specified UDP packet buffer so it can be used for a future UDP transaction.
	 *
	 * @param [in] buff - pointer to PACKET_BUFF to free
	 */
	void freePacket(PACKET_BUFF* buff);

	/**
	 * @fn sendPacket
	 * @brief Send the specified UDP packet.  After sending, the passed buffer will be freed.
	 *
	 * @param [in] buff - pointer to PACKET_BUFF to be transmitted.
	 */
	void sendPacket(PACKET_BUFF* buff);


private:

	DATA_RX_CALLBACK* rx_callback;	        //!< callback function to notify buffer received
	void* rx_callback_instance;             //!< pointer to instance of callback class
	PACKET_BUFF *buff_link_head;	        //!< buffer packet linked list
	pthread_mutex_t lock;			        //!< mutex for linked list

	pthread_t thread_id;
	pthread_attr_t thread_attr;

    int send_sock;					//!< sending socket identifier
	int recv_sock;					//!< receiving socket identifier

    struct sockaddr* server_addr;	//!< socket address can use either IPv4 or IPv6
    socklen_t server_addr_len;		//!< length of the server_addr struct


    /**
     * @fn initBuffPool
     * @brief Allocate and initialize packet buffers and link them in a list.
     */
	void initBuffPool();

	/**
	 * @fn freeBuffPool
	 * @brief Deallocate the linked list of packet buffers.
	 */
	void freeBuffPool();

	/**
	 * @fn setSendSocket
	 * @brief Sets socket and address structure that will be used for sending packets to server.
	 *
	 * @param [in] server_port
	 * @param [in] server_ip_addr
	 */
	void setSendSocket(uint16_t server_port, const char* server_ip_addr);

	/**
	 * @fn createSocketReadThread
	 * @brief Create a pthread to handle the receiving of packets
	 */
	void createSocketReadThread();

	/**
	 * @fn createIp4SocketAddr
	 * @brief Create an IPv4 socket address for the specified port and IP string.
	 *
	 * @param [in] port - port number
	 * @param [in] ip4 - pointer to char array containing IP address; NULL if any address can be used
	 * @return pointer to sockaddr_in stuct allocated from heap
	 */
	struct sockaddr_in* createIp4SocketAddr(const uint16_t server_port, const char* ip4);

	/**
	 * @fn createIp6SocketAddr
	 * @brief Create an IPv6 socket address for the specified port and IP string.
	 *
	 * @param [in] port - port number
	 * @param [in] ip6 - pointer to char array containing IP address; NULL if any address can be used
	 * @return pointer to sockaddr_in6 stuct allocated from heap
	 */
	struct sockaddr_in6* createIp6SocketAddr(const uint16_t server_port, const char* ip6);

	/**
	 * @brief Socket reader thread.
	 *
	 * @details This function also creates the port and socket address.  Note the socket address is stored locally while
	 * socket ID is stored in the class so that another thread can close the socket while its blocked in a read.
	 *
	 * @param [in] arg - pointer to UdpService object using this thread
	 */
	static void* socketReader(void* arg);

	/**
	 * @brief For debug
	 * @param dual_addr
	 */
	void printSockAddrInfo(struct sockaddr* dual_addr);
};
};//namspace sdv
#endif /* UDPSOCKETSERVICE_H_ */
