/**
 * @file UserActivityService.h
 *
 */

#ifndef USERACTIVITYSERVICE_H_
#define USERACTIVITYSERVICE_H_

#include "SDVService.h"
#include "IARMProxyService.h"
#include "MessageService.h"
#include "SDVEventQueue.h"
#include "ConfigurationParser.h"
#include "SDVSessionHandler.h"
#include "ProgramSelectIndication.h"
#include "TRMMonitorService.h"

namespace sdv {

/**
 * @class UserActivityService
 * @brief Provides services for processing messages and operations relating to user activity and bandwidth reclamation.
 */
class UserActivityService : public SDVService {

public:

	UserActivityService(IARMProxyService* iarm_proxy, MessageService* msg_service, SDVEventQueue* event_queue,
			ConfigurationParser* configuration, SDVSessionHandler* session_handler, TRMMonitorService* trm_service);

	~UserActivityService();

	SDVService::SERVICE_RESULT start();

	SDVService::SERVICE_RESULT stop();


private:

	IARMProxyService* iarmProxy;
	MessageService* msgService;
	SDVEventQueue* eventQueue;
	ConfigurationParser* config;
	SDVSessionHandler* sessionHandler;
	TRMMonitorService* trmService;
	bool isRunning;

	/**
	 * @fn getCurrentTimeInSecs
	 * @brief Get the current EPOCH time in seconds
	 *
	 * @return current time in seconds
	 */
	uint32_t getCurrentTimeInSecs();

    /**
     * @fn queryRequest
     * @brief Callback from MessageService to handle query request messages from the server.
     *
     * @param [in] obj - pointer to UserActivityService object
     * @param [in] status - status of the received message
     * @param [in] receivedMsg - pointer to QueryRequest message received from server
     */
    static void queryRequest(void* obj, MessageService::REPONSE_STATUS_t status, SDVMessage* receivedMsg);

    /**
     * @fn eventIndication
     * @brief Callback from MessageService to handle display barker event indication messages from server.
     * @details User inactivity barker display is not supported in RDK.  Therefore we will only respond
     * to the server with EventResponse message upon receiving this message.
     *
     * @param [in] obj - pointer to UserActivityService object
     * @param [in] status - status of the received message
     * @param [in] receivedMsg - pointer to EventIndication message received from server
     */
    static void eventIndication(void* obj, MessageService::REPONSE_STATUS_t status, SDVMessage* receivedMsg);

    /**
     * @fn programSelectIndication
     * @brief Callback from MessageService to handle display program select indication messages from server.
     *
     * @param [in] obj - pointer to UserActivityService object
     * @param [in] status - status of the received message
     * @param [in] programIndication - pointer to ProgramSelectIndication message received from server
     */
    static void programSelectIndication(void* obj, MessageService::REPONSE_STATUS_t status, SDVMessage* receivedMsg);

    /**
     * @fn sendReport
     * @brief Callback from event queue to send a user activity report to the server.
     * @details RDK does not support reporting of user key presses to SDVAgent; therefore user activity reported to
     * the SDV server will be faked to keep the service in use until the QAMSrc is closed.
     *
     * @param qTaskId
     * @param ptrInstance
     * @param data
     */
    static void sendReport(SDVEventQueue::TASK_ID_t qTaskId, void* ptrInstance, void* data);

    /**
     * @fn notifyTunerUsageChange
     * @brief Callback function to notify this service that a tuner usage has changed
     *
     * @param obj - instance of UserActivityService object
     * @param sourceId - source ID corresponding to the tuner usage that has changed
     * @param tunerUse - TUNER_USE flags indicating the new usage of the tuner
     */
    static void notifyTunerUsageChange(void* obj, uint32_t sourceId, uint8_t tunerUse);

};

} /* namespace sdv */

#endif /* USERACTIVITYSERVICE_H_ */
