/**
 *  @file SDVSessionHandler_ut.cc
 *
 *  @date Jun 19, 2014
 *  @Author Spencer Gilroy (sgilroy@cisco.com)
 **/


#include "SDVSessionHandler.h"
#include "mocks/IARMProxyService_mock.h"
#include "mocks/TestMessageHelper.h"
#include "mocks/MessageService_mock.h"
#include "mocks/ConfigurationParser_mock.h"
#include "mocks/SDVChannelMap_mock.h"
#include "mocks/SDVEventQueue_mock.h"
#include <gtest/gtest.h>
#include <gmock/gmock.h>

#include <stdio.h>

using ::testing::Test;
using ::testing::Invoke;
using ::testing::Return;
using ::testing::_;

namespace sdv {

class SDVSessionHandlerTest : public Test{
public:
    void SetUp(){
        iarmProxy = new MockIARMProxyService();
        sdv::MessageService::SDV_MESSAGE_CALLBACK_t reqCb = NULL;
        config = new MockConfigurationParser(iarmProxy, eventQueue, NULL, NULL);
        msgService = new MockMessageService(config, NULL);
        channelMap = new MockSDVChannelMap();
        eventQueue = new MockSDVEventQueue();

        SERVICE_GROUP_ID = 34;
    }

    void TearDown(){
        delete iarmProxy;
        delete config;
        delete msgService;
        delete eventQueue;
        delete channelMap;
        delete testHandler;

    }

    static uint8_t MAC_ADDRESS[4];
    uint32_t SERVICE_GROUP_ID;

    MockIARMProxyService *iarmProxy;
    MockMessageService *msgService;
    MockSDVEventQueue * eventQueue;
    MockConfigurationParser * config;
    MockSDVChannelMap * channelMap;

    SDVSessionHandler * testHandler;
    SDVChannelMap::SDV_CHANNEL_INFO channelInfo;
};
uint8_t SDVSessionHandlerTest::MAC_ADDRESS[] = {1,2,3,4};

TEST_F(SDVSessionHandlerTest, ConstructorTest) {

    testHandler = SDVSessionHandler::createInstance(iarmProxy, msgService,eventQueue, config, channelMap);

    ASSERT_TRUE(testHandler != NULL);
}

TEST_F(SDVSessionHandlerTest, startTest){
    testHandler = SDVSessionHandler::createInstance(iarmProxy, msgService,eventQueue, config, channelMap);
    SDVService::SERVICE_RESULT result;

    //Test Successful returns
    EXPECT_CALL(*iarmProxy, registerIARMRPC(_, _))
    .WillRepeatedly(Invoke(iarmProxy, &MockIARMProxyService::registerIARMRPC_action ));

    result = testHandler->start();

    ASSERT_TRUE(SDVService::SUCCESS == result);
    ASSERT_TRUE(iarmProxy->getRegisteredHandler("sdvAgentOpenSession") != NULL);
    ASSERT_TRUE(iarmProxy->getRegisteredHandler("sdvAgentCloseSession") != NULL);
    ASSERT_TRUE(iarmProxy->getRegisteredHandler("sdvAgentStreamStart") != NULL);
    ASSERT_TRUE(iarmProxy->getRegisteredHandler("sdvAgentStreamStop") != NULL);
}


TEST_F(SDVSessionHandlerTest, OpenSessionTest) {
    testHandler =  SDVSessionHandler::createInstance(iarmProxy, msgService,eventQueue, config, channelMap);
    iarm_sdvagent_open_session_payload payload;

    payload.requestInfo.sourceId = 0x2aea;
    payload.requestInfo.tunerId = 1;

    EXPECT_CALL(*iarmProxy, registerIARMRPC(_, _))
    .WillRepeatedly(Invoke(iarmProxy, &MockIARMProxyService::registerIARMRPC_action ));

    testHandler->start();
    IARM_BusCall_t openSessionCallback = iarmProxy->getRegisteredHandler("sdvAgentOpenSession");
    ASSERT_TRUE(openSessionCallback != NULL);

    uint8_t * msgData = TestMessageHelper::getProgramSelectConfirm(PROGRAM_SELECT_CONFIRM_rspOK, 0x2aea, 16, 370000000,234,1 );

    ProgramSelectConfirm confirm = ProgramSelectConfirm(msgData);

    msgService->setResponse(MessageService::SUCCESS, &confirm);
    EXPECT_CALL(*msgService, sendMessage(_,_,_,_))
    .WillRepeatedly(Invoke(msgService, &MockMessageService::sendMessageAction ));

    EXPECT_CALL(*config, getStbMac())
    .WillOnce(Return(MAC_ADDRESS ));

    EXPECT_CALL(*config, getServiceGroupId())
    .WillOnce(Return(SERVICE_GROUP_ID ));

    EXPECT_CALL(*channelMap, getChannelStatus(10986))
    .WillRepeatedly(Return(&channelInfo ));

    openSessionCallback(&payload);

    delete msgData;

    ASSERT_EQ(payload.responseInfo.status, PROGRAM_SELECT_CONFIRM_rspOK);
    ASSERT_EQ(payload.responseInfo.carrierFreq, 370000000);
    ASSERT_EQ(payload.responseInfo.modulationFormat, 16);
    ASSERT_EQ(payload.responseInfo.programNum, 234);

    // Verify session is active in handler
    SDVSessionHandler::SDV_SESSION session;
    int rval = testHandler->getOpenSession(1, &session);
    ASSERT_TRUE(rval == 0);
    ASSERT_TRUE(session.state == SDVSessionHandler::SDV_SESSION::ACTIVE);
    ASSERT_TRUE(session.tunerId == 1);
    ASSERT_TRUE(session.sourceId == 0x2aea);
}

TEST_F(SDVSessionHandlerTest, OpenExistingSession) {
    testHandler = SDVSessionHandler::createInstance(iarmProxy, msgService,eventQueue, config, channelMap);

    EXPECT_CALL(*iarmProxy, registerIARMRPC(_, _))
    .WillRepeatedly(Invoke(iarmProxy, &MockIARMProxyService::registerIARMRPC_action ));

    // Set expectations for all Configuration get commands
    EXPECT_CALL(*config, getStbMac())
    .WillRepeatedly(Return(MAC_ADDRESS ));
    EXPECT_CALL(*config, getServiceGroupId())
    .WillRepeatedly(Return(SERVICE_GROUP_ID ));

    testHandler->start();
    IARM_BusCall_t openSessionCallback = iarmProxy->getRegisteredHandler("sdvAgentOpenSession");
    ASSERT_TRUE(openSessionCallback != NULL);

    // Create confirm message for 1st open
    uint8_t * msgData = TestMessageHelper::getProgramSelectConfirm(PROGRAM_SELECT_CONFIRM_rspOK, 10986, 16, 370000000,234,1 );
    ProgramSelectConfirm confirm = ProgramSelectConfirm(msgData);

    // Set expectations for all sendMessage commands
    msgService->setResponse(MessageService::SUCCESS, &confirm);
    EXPECT_CALL(*msgService, sendMessage(_,_,_,_))
    .WillRepeatedly(Invoke(msgService, &MockMessageService::sendMessageAction ));


    // Set expectations for 1st getChannelStatus
    EXPECT_CALL(*channelMap, getChannelStatus(10986))
    .WillOnce(Return(&channelInfo));

    // Create payload for 1st open
    IARM_SDVAGENT_OPEN_SESSION_PAYLOAD payload;
    payload.requestInfo.sourceId = 10986;
    payload.requestInfo.tunerId = 1;

    // Open 1st session
    openSessionCallback(&payload);

    // Verify 1st session response
    ASSERT_EQ(payload.responseInfo.status, PROGRAM_SELECT_CONFIRM_rspOK);
    ASSERT_EQ(payload.responseInfo.carrierFreq, 370000000);
    ASSERT_EQ(payload.responseInfo.modulationFormat, 16);
    ASSERT_EQ(payload.responseInfo.programNum, 234);
    delete msgData;


    // Set expectations for 2nd getChannelStatus
    EXPECT_CALL(*channelMap, getChannelStatus(2345))
    .WillOnce(Return(&channelInfo));

    // Create confirm message for 2nd open on same tuner
    msgData = TestMessageHelper::getProgramSelectConfirm(PROGRAM_SELECT_CONFIRM_rspOK, 2345, 18, 220000000, 222, 1);
    confirm = ProgramSelectConfirm(msgData);

    // Create payload for 2nd open
    IARM_SDVAGENT_OPEN_SESSION_PAYLOAD payload2;
    payload2.requestInfo.sourceId = 2345;
    payload2.requestInfo.tunerId = 1;

    // Open 2nd session
    openSessionCallback(&payload2);

    // Verify 2nd session response
    ASSERT_EQ(payload2.responseInfo.status, PROGRAM_SELECT_CONFIRM_rspOK);
    ASSERT_EQ(payload2.responseInfo.carrierFreq, 220000000);
    ASSERT_EQ(payload2.responseInfo.modulationFormat, 18);
    ASSERT_EQ(payload2.responseInfo.programNum, 222);

    // Verify session is open in handler
    SDVSessionHandler::SDV_SESSION session;
    int rval = testHandler->getOpenSession(1, &session);
    ASSERT_TRUE(rval == 0);
    ASSERT_TRUE(session.state == SDVSessionHandler::SDV_SESSION::ACTIVE);
    ASSERT_TRUE(session.tunerId == 1);
    ASSERT_TRUE(session.sourceId == 2345);


    delete msgData;
}

TEST_F(SDVSessionHandlerTest, CloseSessionTest) {
    testHandler = SDVSessionHandler::createInstance(iarmProxy, msgService, eventQueue, config, channelMap);

    EXPECT_CALL(*iarmProxy, registerIARMRPC(_, _))
    .WillRepeatedly(Invoke(iarmProxy, &MockIARMProxyService::registerIARMRPC_action ));

    testHandler->start();
    IARM_BusCall_t openSessionCallback = iarmProxy->getRegisteredHandler("sdvAgentOpenSession");
    ASSERT_TRUE(openSessionCallback != NULL);

    uint8_t * msgData = TestMessageHelper::getProgramSelectConfirm(PROGRAM_SELECT_CONFIRM_rspOK, 0x2aea, 16, 370000000,234,1 );
    ProgramSelectConfirm confirm = ProgramSelectConfirm(msgData);

    msgService->setResponse(MessageService::SUCCESS, &confirm);
    EXPECT_CALL(*msgService, sendMessage(_,_,_,_))
    .WillOnce(Invoke(msgService, &MockMessageService::sendMessageAction ));

    EXPECT_CALL(*config, getStbMac())
    .WillOnce(Return(MAC_ADDRESS ));

    EXPECT_CALL(*config, getServiceGroupId())
    .WillOnce(Return(SERVICE_GROUP_ID ));

    EXPECT_CALL(*channelMap, getChannelStatus(10986))
    .WillRepeatedly(Return(&channelInfo ));

    IARM_SDVAGENT_OPEN_SESSION_PAYLOAD payload;
    payload.requestInfo.sourceId = 0x2aea;
    payload.requestInfo.tunerId = 22;

    openSessionCallback(&payload);

    // Verify sent message
    ASSERT_EQ(msgService->sentMsgId, SDVMessage::ProgramSelectRequest);
    ASSERT_EQ(msgService->sentMsgTunerIndex, 22);

    delete msgData;

    IARM_BusCall_t closeSessionCallback = iarmProxy->getRegisteredHandler("sdvAgentCloseSession");
    ASSERT_TRUE(closeSessionCallback != NULL);

    EXPECT_CALL(*config, getStbMac())
    .WillOnce(Return(MAC_ADDRESS ));

    EXPECT_CALL(*config, getServiceGroupId())
    .WillOnce(Return(SERVICE_GROUP_ID ));

    EXPECT_CALL(*eventQueue, push(_,_,_,_))
    .WillOnce(Invoke(eventQueue, &MockSDVEventQueue::pushAction));

    EXPECT_CALL(*msgService, sendMessage(_,NULL,NULL,0))
    .WillOnce(Invoke(msgService, &MockMessageService::sendMessageAction ));

    IARM_SDVAGENT_CLOSE_SESSION_PAYLOAD closePayload;
    closePayload.sourceId = 0x2aea;
    closePayload.tunerId = 22;

    closeSessionCallback(&closePayload);

    // Verify session is not open
    SDVSessionHandler::SDV_SESSION session;
    int rval = testHandler->getOpenSession(22, &session);
    ASSERT_TRUE(rval == -1);

    sleep(11);  // allow event queue to fire

    // Verify sent tune-away message
    ASSERT_EQ(msgService->sentMsgId, SDVMessage::ProgramSelectRequest);
    ASSERT_EQ(msgService->sentMsgTunerIndex, 22);
}

TEST_F(SDVSessionHandlerTest, LinearServiceTest) {
	int tunerId = 33;
	int sourceId = 1234;

    testHandler = SDVSessionHandler::createInstance(iarmProxy, msgService, eventQueue, config, channelMap);

    EXPECT_CALL(*iarmProxy, registerIARMRPC(_, _))
    .WillRepeatedly(Invoke(iarmProxy, &MockIARMProxyService::registerIARMRPC_action ));

    testHandler->start();
    IARM_BusCall_t openSessionCallback = iarmProxy->getRegisteredHandler("sdvAgentOpenSession");
    ASSERT_TRUE(openSessionCallback != NULL);

    EXPECT_CALL(*msgService, sendMessage(_,_,_,_))
    .WillOnce(Invoke(msgService, &MockMessageService::sendMessageAction ));

    EXPECT_CALL(*config, getStbMac())
    .WillOnce(Return(MAC_ADDRESS ));

    EXPECT_CALL(*config, getServiceGroupId())
    .WillOnce(Return(SERVICE_GROUP_ID ));

    EXPECT_CALL(*channelMap, getChannelStatus(sourceId))
    .WillOnce(Return((SDVChannelMap::SDV_CHANNEL_INFO*)NULL));

    IARM_SDVAGENT_OPEN_SESSION_PAYLOAD payload;
    payload.requestInfo.sourceId = sourceId;
    payload.requestInfo.tunerId = tunerId;

    openSessionCallback(&payload);

    // Verify sent message
    ASSERT_EQ(msgService->sentMsgId, SDVMessage::ProgramSelectRequest);
    ASSERT_EQ(msgService->sentMsgTunerIndex, tunerId);

    // Verify response payload
    ASSERT_EQ(payload.responseInfo.status, PROGRAM_SELECT_ERROR_serviceNotSDV);

    // Verify session is not open
    SDVSessionHandler::SDV_SESSION session;
    int rval = testHandler->getOpenSession(tunerId, &session);
    ASSERT_TRUE(rval == -1);


}

TEST_F(SDVSessionHandlerTest, InvalidGroupIdTest) {
	int tunerId = 33;
	int sourceId = 1234;

    testHandler = SDVSessionHandler::createInstance(iarmProxy, msgService, eventQueue, config, channelMap);

    EXPECT_CALL(*iarmProxy, registerIARMRPC(_, _))
    .WillRepeatedly(Invoke(iarmProxy, &MockIARMProxyService::registerIARMRPC_action ));

    testHandler->start();
    IARM_BusCall_t openSessionCallback = iarmProxy->getRegisteredHandler("sdvAgentOpenSession");
    ASSERT_TRUE(openSessionCallback != NULL);

    uint8_t * msgData = TestMessageHelper::getProgramSelectConfirm(PROGRAM_SELECT_CONFIRM_rspInvalidSG, sourceId, 16, 370000000,234, tunerId);
    ProgramSelectConfirm confirm = ProgramSelectConfirm(msgData);

    msgService->setResponse(MessageService::SUCCESS, &confirm);
    EXPECT_CALL(*msgService, sendMessage(_,_,_,_))
    .WillOnce(Invoke(msgService, &MockMessageService::sendMessageAction));

    EXPECT_CALL(*config, getStbMac())
    .WillOnce(Return(MAC_ADDRESS ));

    EXPECT_CALL(*config, getServiceGroupId())
    .WillOnce(Return(SERVICE_GROUP_ID ));

    EXPECT_CALL(*channelMap, getChannelStatus(sourceId))
    .WillRepeatedly(Return(&channelInfo ));

    IARM_SDVAGENT_OPEN_SESSION_PAYLOAD payload;
    payload.requestInfo.sourceId = sourceId;
    payload.requestInfo.tunerId = tunerId;

    openSessionCallback(&payload);

    // Verify sent message
    ASSERT_EQ(msgService->sentMsgId, SDVMessage::ProgramSelectRequest);
    ASSERT_EQ(msgService->sentMsgTunerIndex, tunerId);

    // Verify response payload
    ASSERT_EQ(payload.responseInfo.status, PROGRAM_SELECT_CONFIRM_rspInvalidSG);

    // Verify IARM broadcast
    ASSERT_EQ(IARM_MOCK_broadcastedEvent.eventId, SDVAGENT_REQUEST_FRESH_MC_DATA);
    ASSERT_EQ(IARM_MOCK_broadcastedEvent.ownerName, IARM_BUS_SDVAGENT_NAME);

    // Verify session is not open
    SDVSessionHandler::SDV_SESSION session;
    int rval = testHandler->getOpenSession(tunerId, &session);
    ASSERT_TRUE(rval == -1);

    delete msgData;
}

TEST_F(SDVSessionHandlerTest, MessageTimeoutTest) {
	int tunerId = 33;
	int sourceId = 1234;

    testHandler = SDVSessionHandler::createInstance(iarmProxy, msgService, eventQueue, config, channelMap);

    EXPECT_CALL(*iarmProxy, registerIARMRPC(_, _))
    .WillRepeatedly(Invoke(iarmProxy, &MockIARMProxyService::registerIARMRPC_action ));

    testHandler->start();
    IARM_BusCall_t openSessionCallback = iarmProxy->getRegisteredHandler("sdvAgentOpenSession");
    ASSERT_TRUE(openSessionCallback != NULL);

    msgService->setResponse(MessageService::TIMEOUT, NULL);
    EXPECT_CALL(*msgService, sendMessage(_,_,_,_))
    .WillOnce(Invoke(msgService, &MockMessageService::sendMessageAction));

    EXPECT_CALL(*config, getStbMac())
    .WillOnce(Return(MAC_ADDRESS ));

    EXPECT_CALL(*config, getServiceGroupId())
    .WillOnce(Return(SERVICE_GROUP_ID ));

    EXPECT_CALL(*channelMap, getChannelStatus(sourceId))
    .WillRepeatedly(Return(&channelInfo ));

    IARM_SDVAGENT_OPEN_SESSION_PAYLOAD payload;
    payload.requestInfo.sourceId = sourceId;
    payload.requestInfo.tunerId = tunerId;

    openSessionCallback(&payload);

    // Verify sent message
    ASSERT_EQ(msgService->sentMsgId, SDVMessage::ProgramSelectRequest);
    ASSERT_EQ(msgService->sentMsgTunerIndex, tunerId);

    // Verify response payload
    ASSERT_EQ(payload.responseInfo.status, MESSAGE_SENT_rspTimedOut);

    // Verify session is not open
    SDVSessionHandler::SDV_SESSION session;
    int rval = testHandler->getOpenSession(tunerId, &session);
    ASSERT_TRUE(rval == -1);
}

} /* namespace sdv */

