/*
 * ConfigurationParser.cpp
 *
 *  Created on: Jun 3, 2014
 *      Author: jasbedar
 */

#include "ConfigurationParser.h"
#include "SDVEventQueue.h"
#include "sdv_iarm.h"
#include "stddef.h"
#include "rdk_debug.h"

#include <sys/socket.h>
#include <sys/ioctl.h>
#include <linux/if.h>
#include <netdb.h>
#include <string.h>
#include <netinet/in.h>
#include <assert.h>
#include <arpa/inet.h>
#include <ifaddrs.h>
#include <fstream>
#include <algorithm>


#define LOG(...) RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", __VA_ARGS__)
#define LOG_ERR(...) RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.SDVAGENT", __VA_ARGS__);


#define SDV_IPV6_CHECK_FILE

using namespace sdv;

const std::string ConfigurationParser::DEVICEPROPS = "/etc/device.properties";
const std::string ConfigurationParser::IPV6_CHECK_FILE = "/tmp/estb_ipv6";
const std::string ConfigurationParser::ESTBIFACE = "ESTB_INTERFACE";
std::string ConfigurationParser::rfIface = "eth0"; // We default to eth0 so as not to crash if its not defined


ConfigurationParser::ConfigurationParser(IARMProxyService* iarmProxy,SDVEventQueue* eventQueue, void* userData, MCP_UPDATE_NOTIFY userCallback){
	this->eventQueue = eventQueue;
	this->userData = userData;
	this->userCallback = userCallback;
	this->sectionFilterProxy = iarmProxy;
	this->mcData = new MCDescriptorData();	// initialize with empty data
	assert(pthread_mutex_init(&dataLock, NULL) == 0);
	setRfInterface();
	setStbMac();

	LOG("Configuration.<constructor>: created\n");
}

ConfigurationParser::~ConfigurationParser(){
	delete mcData;
	pthread_mutex_destroy(&dataLock);
	LOG("Configuration.<destructor>: deleted\n");
}

void ConfigurationParser::start() {
	LOG("Configuration.start: register IARM call\n");
	IARM_Result_t result = sectionFilterProxy->registerIARMRPC(IARM_BUS_SDVAGENT_API_NEW_MC_DATA, &newMcDataReceived, this);
	if (result != IARM_RESULT_SUCCESS) {
		LOG_ERR("Configuration.start: register IARM failed with %d\n", result);
	}
}

uint32_t ConfigurationParser::getMessageResponseTimeout() {
	uint32_t retVal;
	pthread_mutex_lock(&dataLock);
	retVal = mcData->messageResponseTimeout;
	pthread_mutex_unlock(&dataLock);
	return retVal;
}

uint32_t ConfigurationParser::getMessageRequestMaxRetries() {
	uint32_t retVal;
	pthread_mutex_lock(&dataLock);
	retVal = mcData->messageRequestMaxRetries;
	pthread_mutex_unlock(&dataLock);
	return retVal;
}

uint32_t ConfigurationParser::getMessageRequestRetryInterval() {
	uint32_t retVal;
	pthread_mutex_lock(&dataLock);
	retVal = mcData->messageRequestRetryInterval;
	pthread_mutex_unlock(&dataLock);
	return retVal;
}

uint32_t ConfigurationParser::getBandwidthReclaimUITimeout() {
	uint32_t retVal;
	pthread_mutex_lock(&dataLock);
	retVal = mcData->bandwidthReclaimUITimeout;
	pthread_mutex_unlock(&dataLock);
	return retVal;
}

uint32_t ConfigurationParser::getMiniCarouselReadInterval() {
	uint32_t retVal;
	pthread_mutex_lock(&dataLock);
	retVal = mcData->miniCarouselReadInterval;
	pthread_mutex_unlock(&dataLock);
	return retVal;
}

uint32_t ConfigurationParser::getLastUserActivityReportInterval() {
	uint32_t retVal;
	pthread_mutex_lock(&dataLock);
	retVal = mcData->lastUserActivityReportInterval;
	pthread_mutex_unlock(&dataLock);
	return retVal;
}

uint16_t ConfigurationParser::getServerIpPort() {
	uint16_t retVal;

	pthread_mutex_lock(&dataLock);
	if (isStbIp6AddressAvailable() && mcData->isValidIp6Descriptor() == true) {
		retVal = mcData->ip6Port;
	}
	else {
		retVal = mcData->ip4Port;
	}
	pthread_mutex_unlock(&dataLock);
	return retVal;
}

uint32_t ConfigurationParser::getServiceGroupId() {
	uint32_t retVal;
	pthread_mutex_lock(&dataLock);
	retVal = mcData->serviceGroupId;
	pthread_mutex_unlock(&dataLock);
	return retVal;
}

bool ConfigurationParser::isStbIp6AddressAvailable() {
    bool have_access = access( IPV6_CHECK_FILE.c_str(), F_OK ) != -1;

    LOG("Configuration.isStbIp6AddressAvailable() = %d\n", have_access);
	return have_access;
}

std::string ConfigurationParser::getServerIpString() {
	pthread_mutex_lock(&dataLock);

	if(isStbIp6AddressAvailable()){
	    if(mcData->isValidIp6Descriptor()){
	        // Convert ip6 binary address to string
	        char ipAddr[INET6_ADDRSTRLEN];
	        binaryIpToString(mcData->ip6BinaryAddr, ipAddr, INET6_ADDRSTRLEN);
	        RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.SDVAGENT", "Configuration.getServerIpString() Device is IPV6 and found ipv6 in MCP descriptor %s\n", ipAddr);

	        pthread_mutex_unlock(&dataLock);
	        return ipAddr;
	    }else{
	        RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.SDVAGENT", "Configuration.getServerIpString() Device is IPV6 but no descriptor in MCP,  ipv4->ipv6 map size():%d \n", ipAddressMap.size());
	        char ipv4Addr[INET_ADDRSTRLEN];
	        binaryIpToString(mcData->ip4BinaryAddr, ipv4Addr, INET_ADDRSTRLEN);

	        std::map<std::string, std::string>::iterator iterator = ipAddressMap.find(std::string(ipv4Addr));
	        if(iterator != ipAddressMap.end()) {
	            RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.SDVAGENT", "Configuration.getServerIpString() Found ipv6:%s mapped value for %s\n", iterator->second.c_str(),ipv4Addr);
	            pthread_mutex_unlock(&dataLock);
	            return iterator->second;
	        }
	        RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.SDVAGENT", "Configuration.getServerIpString() Could not find ipv6 map value for %s\n", ipv4Addr);
	    }
	}

    RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.SDVAGENT", "Configuration.getServerIpString() Device is IPV4 Or could not find IPV descriptor\n");

    //If we made it here we we must use IPV4 address
    char ipAddr[INET_ADDRSTRLEN];
    binaryIpToString(mcData->ip4BinaryAddr, ipAddr, INET_ADDRSTRLEN);

    RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.SDVAGENT", "Configuration.getServerIpString() Reutrn IPV4 %s\n", ipAddr);
    pthread_mutex_unlock(&dataLock);
    return ipAddr;
}

void ConfigurationParser::setRfInterface() {

    std::string line;
    std::string tmpIface;
    std::ifstream file(DEVICEPROPS.c_str());

    while(std::getline(file, line)) {
        if(line.find(ESTBIFACE) != std::string::npos){
            std::size_t pos = line.find("=");
            if(pos != std::string::npos) {
                if(line.size() > (pos +1)){
                    tmpIface = line.substr(pos + 1);
                    // Remove any spaces
                    tmpIface.erase(std::remove_if(tmpIface.begin(),tmpIface.end(),::isspace),tmpIface.end());
                }
            }
        }
    }
    if(tmpIface.empty()){
        RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.SDVAGENT", "Configuration.setRfInterface unable to find RF interface in %s defaulting to %s\n", DEVICEPROPS.c_str(),rfIface.c_str());
    }
    else{
        rfIface = tmpIface;
        RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "Configuration.setRfInterface RF interface set to %s\n", rfIface.c_str());
    }
}

std::string ConfigurationParser::getRfInterface() {

    return rfIface;
}


void ConfigurationParser::setStbMac() {
    int fd;
    struct ifreq ifr;
     
    fd = socket(AF_INET, SOCK_DGRAM, 0);
 
    ifr.ifr_addr.sa_family = AF_INET;
    strncpy(ifr.ifr_name, rfIface.c_str(), IFNAMSIZ-1);
    ioctl(fd, SIOCGIFHWADDR, &ifr);
    close(fd);

    stbMac[0] = ifr.ifr_hwaddr.sa_data[0];
    stbMac[1] = ifr.ifr_hwaddr.sa_data[1];
    stbMac[2] = ifr.ifr_hwaddr.sa_data[2];
    stbMac[3] = ifr.ifr_hwaddr.sa_data[3];
    stbMac[4] = ifr.ifr_hwaddr.sa_data[4];
    stbMac[5] = ifr.ifr_hwaddr.sa_data[5];

    RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "Configuration.setStbMac: %02x:%02x:%02x:%02x:%02x:%02x\n", stbMac[0],stbMac[1],stbMac[2],stbMac[3],stbMac[4],stbMac[5]);
}

uint8_t* ConfigurationParser::getStbMac() {
	return stbMac;
}

void ConfigurationParser::processNewMcData(SDVEventQueue::TASK_ID_t id, void* objInstance, void* data) {
	static const int TABLE_ID = 0xc9;			// MPEG2 CVCT Table ID
	static const int SECTION_NUMBER_OFFSET = 6;	// byte offset from very beginning of section transport stream

	LOG("Configuration.processNewMcData: event with new MC data at 0x%02x\n", data);

	ConfigurationParser* config = (ConfigurationParser*)objInstance;
	MCDescriptorData* newMcData = (MCDescriptorData*)data;

	// If any MC data has changed, replace our data object with the new one
	if(config->mcData->isEqual(newMcData) == false) {
		LOG("Configuration.processNewMcData: MC data changed\n");
		LOG(newMcData->toDebugString().c_str());

		bool isServerChanged = config->mcData->isServerChanged(newMcData);
		bool isServiceGroupChanged = config->mcData->isServiceGroupChanged(newMcData);

		// Swap to new data object before notifying user
		pthread_mutex_lock(&config->dataLock);
		delete config->mcData;
		config->mcData = newMcData;
		pthread_mutex_unlock(&config->dataLock);

		// If server or service group change, notify user using callback
		if(isServerChanged || isServiceGroupChanged) {
			LOG("Configuration.processNewMcData: server or service group changed, invoke callback\n");
			(*config->userCallback)(config->userData, isServiceGroupChanged, isServerChanged);
		}
	}
	else {
		delete newMcData;
	}
}

void ConfigurationParser::processNewIPV6Mapping(SDVEventQueue::TASK_ID_t id, void* objInstance, void* data) {

	LOG("Configuration.processNewIPV6Mapping: \n");

	ConfigurationParser* config = (ConfigurationParser*)objInstance;
	IARM_SDVAGENT_NEW_MC_DATA_PAYLOAD* newMcData = (IARM_SDVAGENT_NEW_MC_DATA_PAYLOAD*)data;

	//Get the array of map elements which will be after the MCP data in the payload.
	SDVAGENT_IPV6_MAP_ELEMENT * elements = (SDVAGENT_IPV6_MAP_ELEMENT * ) &newMcData->payload[newMcData->mcpDataSize];

	pthread_mutex_lock(&config->dataLock);

	config->ipAddressMap.clear();//clear all old data
	for(uint16_t i=0;i!=newMcData->ipv6MapElementCount;++i){
	    char ipv6Addr[INET6_ADDRSTRLEN];
	    config->binaryIpToString(elements[i].ipv6address, ipv6Addr, INET6_ADDRSTRLEN);

	    char ipAv4ddr[INET_ADDRSTRLEN];
	    config->binaryIpToString(elements[i].ipv4address, ipAv4ddr, INET_ADDRSTRLEN);

	    LOG("Configuration.processNewIPV6Mapping: Add IPV4->IPV6 map \t%s - > %s\n", ipAv4ddr, ipv6Addr );

	    config->ipAddressMap[ipAv4ddr] = ipv6Addr;
	}
	LOG("Configuration.processNewIPV6Mapping:IP Address Map size = %d \n", config->ipAddressMap.size() );

	pthread_mutex_unlock(&config->dataLock);

}

IARM_Result_t ConfigurationParser::newMcDataReceived(void *payload) {
	static const int TABLE_ID = 0xc9;			// MPEG2 CVCT Table ID
	static const int SECTION_NUMBER_OFFSET = 6;	// byte offset from very beginning of section transport stream

	IARM_SDVAGENT_NEW_MC_DATA_PAYLOAD * mcData = (IARM_SDVAGENT_NEW_MC_DATA_PAYLOAD*)payload;

	// If section 0, create new MC data
	if (mcData->payload[SECTION_NUMBER_OFFSET] == 0) {
		LOG("Configuration.newMcDataReceived: new MC descriptor data received\n");
		MCDescriptorData* newData = new MCDescriptorData(&mcData->payload[0]);

		// Get our instance of configuration and queue new mc data for processing
		ConfigurationParser* config = (ConfigurationParser*)IARMProxyService::getRpcObject(IARM_BUS_SDVAGENT_API_NEW_MC_DATA);

        //Process the IPV4 to IPV6 mapping used when running IPV6 with only MCMIS 2.1.
        config->eventQueue->push((void*)config, config->processNewIPV6Mapping, mcData, 0);

		LOG("Configuration.newMcDataReceived: queue event with new MC data at 0x%02x\n", newData);
		config->eventQueue->push((void*)config, config->processNewMcData, newData, 0);
	}
	// Else must not be section 0 data
	else {
		LOG("Configuration.newMcDataReceived: ignore data with sectionNumber=%d\n", mcData->payload[SECTION_NUMBER_OFFSET]);
	}
	return IARM_RESULT_SUCCESS;
}

void ConfigurationParser::binaryIpToString(uint8_t* ipBinary, char* ipString, uint32_t maxStringLength) {

	if (maxStringLength == INET_ADDRSTRLEN) {
		struct in_addr addr;
		memcpy(&addr, ipBinary, sizeof(struct in_addr));
		inet_ntop(AF_INET, &addr, ipString, INET_ADDRSTRLEN);
	}
	else if (maxStringLength == INET6_ADDRSTRLEN) {
		struct in6_addr addr;
		memcpy(&addr, ipBinary, sizeof(struct in6_addr));
		inet_ntop(AF_INET6, &addr, ipString, INET6_ADDRSTRLEN);
	}
	else {
		assert(false);
	}
}

