/*
 * Configuration.h
 *
 *  Created on: Jun 3, 2014
 *      Author: jasbedar
 */

#ifndef CONFIGURATIONPARSER_H_
#define CONFIGURATIONPARSER_H_

#include <stdint.h>
#include <pthread.h>
#include <string.h>
#include <map>

#include "IARMProxyService.h"
#include "SDVEventQueue.h"
#include "MCDescriptorData.h"

namespace sdv {


/**
 * @class ConfigurationParser
 * @brief Provides configuration data
 * @details  The MC data provided by this class can be updated at any time.  To maintain mutual exclusion
 */
class ConfigurationParser {

public:

	/**
	 * @enum PARSE_RESULT_t
	 * @brief Indicates result of parsing of MC data
	 * @TODO determine if this is really needed
	 */
	enum {CONFIG_PARSE_FAILURE,   		//!< CONIFG_PARSE_FAILURE
		  CONFIG_PARSE_SUCCESS,      	//!< CONFIG_PARSE_SUCCESS
		  CONFIG_PARSE_SERVER_UPDATED	//!< CONFIG_PARSE_SERVER_UPDATED
	} typedef PARSE_RESULT_t;

	/**
	 * @brief Callback to notify if MC Data has been updated.
	 * @details Callback will be invoked whenever server IP, server port, or service group has changed.
	 *
	 * @param userData - void pointer to data belonging to user of this class
	 * @param isServiceGroupChange - indicates if the service group has changed
	 * @param isServerIpChange - indicates if server IP or port has changed
	 */
	typedef void (*MCP_UPDATE_NOTIFY)(void* userData, bool isServiceGroupChange, bool isServerIpChange);

	/**
	 * @constructor
	 *
	 * @param iarmProxy
	 * @param eventQueue
	 * @param userData - void pointer for user data to be passed through the callback
	 * @param userCallback - callback to notify if SDV Server IP or port change
	 */
	ConfigurationParser(IARMProxyService* iarmProxy, SDVEventQueue* eventQueue, void* userData, MCP_UPDATE_NOTIFY userCallback);

	~ConfigurationParser();

	/**
	 * @fn start
	 * @brief Start Configuration.
	 * @details Allows Configuration to start receiving raw MC data from SDVAutodiscovery through IARM RPC
	 */
	virtual void start();

	/**
	 * @fn stop
	 * @brief API has no effect, currently Configuration cannot be stopped directly.
	 */
	virtual void stop(){};

	/**
	 * @fn getServerIpString
	 * @brief Get the SDV server IP address string.
	 * @details If dual IP is in use, IPv6 will be used if STB has an IPv6 address and MC data contains IPv6 descriptor.
	 * Otherwise we will use the servers IPv4 address.
	 *
	 * @return Server IP address string
	 */
	virtual std::string getServerIpString();

	/**
	 * @fn getServerIpPort
	 * @brief Get the SDV Server port number.
	 *
	 * @return port number
	 */
	virtual uint16_t getServerIpPort();

	/**
	 * @fn getServiceGroupId
	 * @brief See TWC-SDV-MCMIS 3.1.1 Specification for details
	 *
	 * @return service group ID
	 */
	virtual uint32_t getServiceGroupId();

	/**
	 * @fn getLastUserActivityReportInterval
	 * @brief See TWC-SDV-MCMIS 3.1.1 Specification for details
	 *
	 * @return last user activity report interval
	 */
	virtual uint32_t getLastUserActivityReportInterval();

	/**
	 * @fn getMiniCarouselReadInterval
	 * @brief See TWC-SDV-MCMIS 3.1.1 Specification for details
	 *
	 * @return mini-carousel read interval
	 */
	virtual uint32_t getMiniCarouselReadInterval();

	/**
	 * @fn getMessageResponseTimeout
	 * @brief See TWC-SDV-MCMIS 3.1.1 Specification for details
	 *
	 * @return message response timeout
	 */
	virtual uint32_t getMessageResponseTimeout();

	/**
	 * @fn getMessageRequestMaxRetries
	 * @brief See TWC-SDV-MCMIS 3.1.1 Specification for details
	 *
	 * @return message request maximum retries
	 */
	virtual uint32_t getMessageRequestMaxRetries();

	/**
	 * @fn getMessageRequestRetryInterval
	 * @brief See TWC-SDV-MCMIS 3.1.1 Specification for details
	 *
	 * @return message request retry interval
	 */
	virtual uint32_t getMessageRequestRetryInterval();

	/**
	 * @fn getBandwidthReclaimUITimeout
	 * @brief See TWC-SDV-MCMIS 3.1.1 Specification for details
	 *
	 * @return bandwidth reclaim UI timeout
	 */
	virtual uint32_t getBandwidthReclaimUITimeout();

        /**
         * @fn getRfInterface
         * @brief Helper function to get the RF/CPE eth interface name for this STB
         *
         * @return string of eth interface name
         */
        std::string getRfInterface();

	/**
	 * @fn getStbMac
	 * @brief Get this STBs 6-byte MAC address.  Also referred to as CPE, RF, and eSTB MAC.
	 *
	 * @return pointer to mac address
	 */
	virtual uint8_t* getStbMac();


private:
    static std::string rfIface;                     //!< interface to use for SDV Server communications
    static const std::string DEVICEPROPS;           //!< path and file name for the device properties file
    static const std::string IPV6_CHECK_FILE;       //!< path and file name for ipv6 file whose presence indicates IPV6 should be used
    static const std::string ESTBIFACE;             //!< propertie name for the estb (rf/cpe) eth interface
    MCDescriptorData* mcData;                       //!< current mc data

    std::map<std::string, std::string> ipAddressMap;      //!< optional map of ipv4 to ipv6 addresses if running IVP6 with MCMIS 2.1

    uint8_t stbMac[6];                      //!< CPE mac of this STB
    pthread_mutex_t dataLock;               //!< mutex for accessing MC data
    MCP_UPDATE_NOTIFY userCallback;         //!< user callback to notify with MC updates
    void *userData;                         //!< user provided data pointer to pass in callback
    IARMProxyService* sectionFilterProxy;
    SDVEventQueue* eventQueue;


	/**
	 * @fn processNewMcData
	 * @brief Handler invoked by the EventQueue for processing new MC data and notifying user
	 * of MC data update if necessary.
	 *
	 * @param objInstance - pointer to the instance of this ConfigurationParser object
	 * @param data - pointer to new MCDescriptorData object
	 */
	static void processNewMcData(SDVEventQueue::TASK_ID_t id, void* objInstance, void* data);

	/**
	 * @fn processNewMcData
	 * @brief Handler invoked by the EventQueue for processing new IPv4 to ipv6
	 * mappings
	 *
	 * @param objInstance - pointer to the instance of this ConfigurationParser object
	 * @param data - pointer to new iarm IARM_SDVAGENT_NEW_MC_DATA_PAYLOAD payload
	 */
	static void processNewIPV6Mapping(SDVEventQueue::TASK_ID_t id, void* objInstance, void* data);

	/**
	 * @fn newMcDataReceived
	 * @brief Remote Procedure callback from IARM to received new raw MC section data
	 *
	 * @param [i] payload - pointer to byte array containing raw MC section data
	 */
	static IARM_Result_t newMcDataReceived(void* payload);

        /**
         * @fn setRfInterface
         * @brief Helper function to set the RF/CPE eth interface for this STB
         */
        void setRfInterface();

	/**
	 * @fn setStbMac
	 * @brief Helper function to detect and set the CPE MAC for this STB
	 */
	void setStbMac();

	/**
	 * @fn binaryIpToString
	 * @brief Convert the specified binary IPv4 or IPv6 address array into a IP address string.
	 * @details Uses temporary socket address and inet library to do the conversion.
	 *
	 * @param [in] ipBinary - array containing IPv4 or IPv6 binary
	 * @param [out] ipString - memory location to store the output IP string
	 * @param [in] maxStringLength - maximum length of the output string (including terminator).
	 * Must equal INET_ADDRSTRLEN or INET6_ADDRSTRLEN.
	 */
	void binaryIpToString(uint8_t* ipBinary, char* ipString, uint32_t maxStringLength);

	/**
	 * @fn isStbIp6AddressAvailable
	 * @brief Determine if the STB has an IPv6 address for our rfInterface.
	 *
	 * @return true if an IPv6 address was detected; otherwise false
	 */
	bool isStbIp6AddressAvailable();
};

} /* namespace sdv */

#endif /* CONFIGURATIONPARSER_H_ */



