/*
 * MCDescriptorData.h
 *
 *  Created on: Jul 1, 2014
 *      Author: ktaubing
 */

#ifndef MCDESCRIPTORDATA_H_
#define MCDESCRIPTORDATA_H_

#include <stdint.h>
#include <string>
#include <string.h>


namespace sdv {

/**
 * @class MCDescriptorData
 * @brief Acquires and stores descriptor parameters parsed from MC raw MPEG section data.
 */
class MCDescriptorData {

public:

#define IPV6_BINARY_LENGTH 16 	//!> byte length of IPv6 binary address
#define IPV4_BINARY_LENGTH 4 	//!> byte length of IPv4 binary address


	/**
	 * MC Data described in TWC-SDV-MCMIS 3.1.1 Specification
	 */
	uint8_t ip4BinaryAddr[IPV4_BINARY_LENGTH];
	uint16_t ip4Port;
	uint8_t ip6BinaryAddr[IPV6_BINARY_LENGTH];
	uint16_t ip6Port;
	uint32_t serviceGroupId;
	uint32_t lastUserActivityReportInterval;
	uint32_t miniCarouselReadInterval;
	uint32_t messageResponseTimeout;
	uint32_t messageRequestMaxRetries;
	uint32_t messageRequestRetryInterval;
	uint32_t bandwidthReclaimUITimeout;

	/**
	 * @constructor
	 * @brief Create an object with descriptor data from the specified raw MPEG section data.
	 * @details Only section 0 contains the descriptor data.
	 *
	 * @param sectionData - pointer to raw section 0 data
	 */
	MCDescriptorData(uint8_t* sectionData);

	/**
	 * @constructor
	 * @brief Create an object with empty descriptor data
	 */
	MCDescriptorData();

	/**
	 * @fn isEqual
	 * @brief Determine if the specified MC descriptor data is equal to the data in this object.
	 *
	 * @param newData - pointer to MCDescriptorData to compare
	 * @return true if data equal; otherwise false
	 */
	bool isEqual(MCDescriptorData* newData);

	/**
	 * @fn isServerChanged
	 * @brief Indicates if the SDV server IP addresses or ports have changed in the specified MC descriptor data.
	 *
	 * @param newData - pointer to new descriptor data
	 * @return true if server has changed; otherwise false
	 */
	bool isServerChanged(MCDescriptorData* newData);

	/**
	 * @fn isServiceGroupChanged
	 * @brief Indicates if the service group has changed in the specified descriptor data.
	 *
	 * @param newData - pointer to new descriptor data
	 * @return true if service group ID has changed; otherwise false
	 */
	bool isServiceGroupChanged(MCDescriptorData* newData);

	/**
	 * @fn isValidIp6Descriptor
	 * @brief Indicates if this MC data contains a valid IPv6 descriptor
	 *
	 * @return true if valid IPv6 server address; otherwise false
	 */
	bool isValidIp6Descriptor();

	/**
	 * Generate debug formatted string for all descriptor data
	 */
	std::string toDebugString();

private:

	/**
	 * Byte offsets to data fields from very beginning of section transport stream
	 */
	static const int TRANSPORT_STREAM_ID_OFFSET = 3;
	static const int NUM_CHANNELS_IN_SECTION_OFFSET = 9;
	static const int SECTION_NUMBER_OFFSET = 6;

	/**
	 * Upper 6 bits of MC descriptors length are reserved and therefore need to be masked out
	 */
	static const uint16_t MC_DESCRIPTORS_LENGTH_BIT_MASK = 0x3ff;

	/**
	 * Bytes allocated per each channel in channel fields
	 */
	static const int CHANNEL_BLOCK_SIZE = 32;

	/**
	 * MC Descriptor tags.
	 */
	static const int IP_RESOURCE_DESCRIPTOR_TAG = 3;
	static const int CONFIG_PARAMS_DESCRIPTOR_TAG = 4;
	static const int SERVICE_GROUP_ID_DESCRIPTOR_TAG = 6;   // new in MCMIS 3.1.1
	static const int IPV6_RESOURCE_DESCRIPTOR_TAG = 7;      // new in MCMIS 3.1.1

	/**
	 * MCMIS Descriptor structures per TWC-SDV-MCMIS 3.1.1
	 * These structures are only used for parsing raw section data
	 */
#pragma pack(push,1)
	typedef struct mcmis_ip4_resource_descriptor {
		uint8_t tag;
		uint8_t length;
		uint16_t ipPort;
		uint8_t ipAddress[IPV4_BINARY_LENGTH];
		uint32_t reserved;
	} MCMIS_IP4_RESOURCE_DESCRIPTOR;

	typedef struct mcmis_config_params_descriptor {
		uint8_t tag;
		uint8_t length;
		uint16_t reserved;
		uint32_t luaReportingInterval;
		uint32_t miniCarouselReadInterval;
		uint32_t messageResponseTimeout;
		uint32_t messageRequestMaxRetriesCount;
		uint32_t messageRequestRetryInterval;
		uint32_t bandwidthReclaimUITimeout;
	} MCMIS_CONFIG_PARAMS_DESCRIPTOR;

	typedef struct mcmis_service_group_id_descriptor {
		uint8_t tag;
		uint8_t length;
		uint32_t serviceGroupId;
	} MCMIS_SERVICE_GROUP_ID_DESCRIPTOR;

	typedef struct mcmis_ip6_resource_descriptor {
		uint8_t tag;
		uint8_t length;
		uint16_t ipPort;
		uint8_t ipAddress[IPV6_BINARY_LENGTH];
	} MCMIS_IP6_RESOURCE_DESCRIPTOR;
#pragma pack(pop)


	/**
	 * @fn parseDescriptors
	 * @brief Parse descriptors in specified section data
	 *
	 * @param sectionData - pointer to MPEG section data containing Mini Carousel Descriptors.
	 */
	void parseDescriptors(uint8_t* sectionData);

	/**
	 * @fn parseIp4Descriptor
	 * @brief Extract the IPv4 Address and port from the specified MC descriptor
	 *
	 * @param descriptor = pointer to network byte order descriptor
	 */
	void parseIp4Descriptor(MCMIS_IP4_RESOURCE_DESCRIPTOR* descriptor);

	/**
	 * @fn parseConfigParamDescriptor
	 * @brief Extract the configuration parameters from the specified MC descriptor
	 *
	 * @param descriptor = pointer to network byte order descriptor
	 */
	void parseConfigParamDescriptor(MCMIS_CONFIG_PARAMS_DESCRIPTOR* descriptor);

	/**
	 * @fn parseServiceGroupIdDescriptor
	 * @param Extract the group ID from the specified MC descriptor
	 *
	 * @param descriptor = pointer to network byte order descriptor
	 */
	void parseServiceGroupIdDescriptor(MCMIS_SERVICE_GROUP_ID_DESCRIPTOR* descriptor);

	/**
	 * @fn parseIpv6Descriptor
	 * @brief Extract the IPv6 Address and port from the specified MC descriptor
	 *
	 * @param descriptor = pointer to network byte order descriptor
	 */
	void parseIp6Descriptor(MCMIS_IP6_RESOURCE_DESCRIPTOR* descriptor);

	/**
	 * @fn initIpData
	 * @brief Initialize all IP data to zero
	 */
	void initIpData();

};

} /* namespace sdv */

#endif /* MCDESCRIPTORDATA_H_ */
