/*
* ============================================================================
* RDK MANAGEMENT, LLC CONFIDENTIAL AND PROPRIETARY
* ============================================================================
* This file (and its contents) are the intellectual property of RDK Management, LLC.
* It may not be used, copied, distributed or otherwise  disclosed in whole or in
* part without the express written permission of RDK Management, LLC.
* ============================================================================
* Copyright (c) 2014 RDK Management, LLC. All rights reserved.
* ============================================================================
*/
#ifndef MESSAGESERVICE_H_
#define MESSAGESERVICE_H_

#include "SDVService.h"
#include "SDVMessage.h"
#include "ConfigurationParser.h"
#include "SDVEventQueue.h"
#include "UdpService.h"

#include <map>
#include <cstdlib>
#include "stddef.h"
#include <cstring>

namespace sdv {

/**
 * Service for sending and receiving CCP messages with the SDV Server.
 *
 * Message handlers can be set to receive all messages of a given type.
 * Only a single message handler can be set for a given message type.
 *
 * Additionally when sending a message an optional response message handler
 * can be provided.  This response handler will be notified a response
 * to the message being sent is received from the SDV Server.  If the
 * Server does not respond the handler will also be notified of the timeout.
 */
class MessageService : public SDVService{
public:

    /**@enum REPONSE_STATUS_t
     *
     * @brief Possible status codes resulting from transaction
     * with the SDV server that requires a response.
     */
    typedef enum{
        SUCCESS,
        TIMEOUT
    }REPONSE_STATUS_t;

    /** @brief MessageService callback Function template
     *
     * Template for callback function which gets notified when a message it has
     * been registered for is recieved.
     *
     * @parm [in] ptrInst - pointer to the instance of the class that owns the callback method
     * @parm response - pointer to message which has been recieved.  Message will go out out scope after
     *  callback returns.
     */
    typedef void (*SDV_MESSAGE_CALLBACK_t)(void* ptrInst, REPONSE_STATUS_t status, SDVMessage * message);

    /**@constrcutor
     *
     * @brief Constrcutor for MessageService
     *
     * @param [in] configParser - holds configutation needed for creating sockets
     *  and message retry informtation
     * @param [in] queue - sending of messages and handing of incoming packets will be managed though
     * the event queue
     */
    MessageService(ConfigurationParser * configParser, SDVEventQueue * queue);

    /**@fn sendMessage
     *
     * @brief Send a message asynchronoulsy to the SDV Server.
     *
     * @param [in] msg - pointer to the sdv message on the to send.
     *  It is expected the Message is allocated on the heap and will deleted after
     *  message is succesfully sent
     *
     * @param [in] callback - function to invoke when a response message is expected.
     *  If no response is expected this should be NULL.
     *
     * @param [in] ptrInst - pointer to the instance of the class that owns the callback function.
     *  If no response is expected this should be NULL.
     *
     * @param [in] delay - amount of time in milliseconds to wait before sending the message.
     *  If the message should be sent immediatly should be 0.
     */
    virtual void sendMessage(SDVMessage * msg, SDV_MESSAGE_CALLBACK_t callback, void* ptrInst, uint32_t delay);

    /**@fn setMessageHandler
     *
     * @brief Sets a message handler for all incoming messages of a given type.
     *
     * @details Only one message handler can be set for a given type, if another handler is
     * set, the original handler will replaced by the new one
     *
     * @param [in] messageID - The type of message the provided handler is interested in receiving.
     *
     * @param [in] callback - function to invoke when a response message is expected.
     *  To clear a previously set callback should be NULL
     *
     * @param [in] ptrInst - pointer to the instance of the class that owns the callback function.
     */
    void setMessageHandler(SDVMessage::MessageId messageID, SDV_MESSAGE_CALLBACK_t callback, void* ptrInst);

    /**@fn start
     *
     * @brief starts the Message service.
     *
     * Creates and binds datagrams socket for sending and recieving packets with
     * the SDVServer.
     *
     * Configutation information for the sockets will be re-read from ConfigurationParser each time
     * the service is started.
     *
     * @return SUCCESS if the service started correctly otherwise FAILURE
     */
    sdv::SDVService::SERVICE_RESULT start();

    /**@fn stop
     *
     * @brief stops the Message service.
     *
     * Destoryed sockets which where prevoulsy bound for communication with the sdv server.
     *
     * Registered handlers will not be removed and if service is started again
     * will continue to recived incoming messages events
     *
     * @return SUCCESS
     */
    sdv::SDVService::SERVICE_RESULT stop();

    /**
     * @fn restart
     * @brief Restart MessageService so in order for it to apply new server communication configurations
     *
     * @return SUCCESS
     */
    sdv::SDVService::SERVICE_RESULT restart();

private:

    /**@struct timeoutCheckData
     *
     * @brief Container for information about a timeout check task
     *
     * Timeout check tasks are posted to the event queue when messages are sent
     * that require a response.  The task keeps track of how many messages have been sent
     * and if any more retry attempts should be allowed.
     */
    typedef struct timeoutCheckData{
        SDVMessage::TRANACTION_ID_t transactionId; /**< tranaction id of expected response message */
        uint8_t remaining_retries;                 /**< The number of retry attempts reamining */
    }TIMEOUT_DATA_t;

    /**@struct callbackData
     *
     * @brief Container for information needed to track a send message operation and
     * invoke a callback after a response is received or timeout occurs.  Also used for
     * the callback on unsolicited inbound messages.
     *
     * Timeout check tasks are posted to the event queue when messages are sent
     * that require a response.  The task keeps track of how many messages have been sent
     * and if any more retry attempts should be allowed.
     */
    typedef struct callbackData{
        void* ptrInst;                      /**< Pointer to the instance of the callback function's class */
        SDV_MESSAGE_CALLBACK_t callback;    /**< callback function to invoke when message received or timeout */
        SDVMessage * outboundMsg;           /** <Pointer to the outbound message for this transaction */
        TIMEOUT_DATA_t * timeoutCheck;      /** <Pointer to timeout check data */
    }CALLBACK_DATA_t;

    /**
     * @typedef UNSOLICTED_MESSAGE_HANDLER
     * @brief Structure for storing details on the handler to be invoked on receiving of an unsolicited message
     */
    typedef struct unsolicted_message_handler {
        void* ptrInst;                      /**< Pointer to the instance of the handler function's class */
        SDV_MESSAGE_CALLBACK_t callback;    /**< Pointer to the the callback function to invoke when message  */
    } UNSOLICTED_MESSAGE_HANDLER;


private:
    /**
     * @fn packetRecieved
     *
     * @brief Callback from UDPService when a packet has been recived.
     *
     * Data from the incoming packet will be used to parse and SDVMessage.
     * the the Message type is known an event will be pushed to call handleIncomingPacketEvent
     * from the event queue in order to process the message.

     * @param [in] buff - buffer that holds incoming packet data and length
     * @param [in] callbackInstance - reference to this instance of this MessageService.
     */
    static void packetRecieved(UdpService::PACKET_BUFF * buff, void* callbackInstance);
private:
    /**
     * @fn handleIncomingPacketEvent
     *
     * @brief Event queue handler for incoming SDVMNessages
     *
     * Should only be run from within the Event queue thread.
     *
     * Will check if any solicited or un-solicted handlers are registered for the
     * incoming message.  If a handler is found they will be notified of the incoming message.
     * from within the event queue thread.
     *
     * The incomding message will be deleted after the callback returns.
     * If the incoming message is part of a tranasction the related outgoing message will
     * also be deleted after the callback has been invoked.
     *
     * @param [in] taskId - ID of the task which is being executed.
     * @param [in] ptrInstance - reference to this instance of this MessageService.
     * @param [in] data - Pointer to instance of SDVMessage on the heap.
     */
    static void handleIncomingMessageEvent(SDVEventQueue::TASK_ID_t taskId, void* ptrInstance, void * data);

    /**
     * @fn handleSendMessageEvent
     *
     * @brief Event queue handler for outgoing SDVMessages
     *
     * Should only be run from within the Event queue thread.
     *
     * Will serailize the provided SDVMessage and use the UDPService to send
     * a packet to the SDV Server.
     *
     * If a response is requried for the message an handleResponseTimeoutCheck
     * event will be scheduled to execute after the appropriate timeout.
     *
     * If a response is not required the outgoing message will be deleted after
     * after the packet is sent.  Otherwsie the message will be deleted when a
     * response is recieved or timeout occurs.
     *
     * @param [in] taskId - ID of the task which is being executed.
     * @param [in] ptrInstance - reference to this instance of this MessageService.
     * @param [in] data - Pointer to instance of CALLBACK_DATA_t on the heap.
     */
    static void handleSendMessageEvent(SDVEventQueue::TASK_ID_t taskId, void* ptrInstance, void * data);

    /**
     * @fn handleResponseTimeoutCheck
     *
     * @brief Event queue handler for checking if a response message
     * has been recieved.
     *
     * Should only be run from within the Event queue thread.
     *
     * Will check to see if a response is still pending for a transaction id,
     * if a response has not been recived a retry attempt will be mad x number of times.
     *
     * If all retry attempts fail the registered handler will be notified of the timeout
     * and the Message service will stop waiting for a response.
     *
     * If a response message has been recieved then the only action needed is to delete
     * the TIMEOUT_DATA_t from the heap.
     *
     * @param [in] taskId - ID of the task which is being executed.
     * @param [in] ptrInstance - reference to this instance of this MessageService.
     * @param [in] data - Pointer to instance of TIMEOUT_DATA_t on the heap.
     */
    static void handleResponseTimeoutCheck(SDVEventQueue::TASK_ID_t taskId, void* ptrInstance, void * data);

    /**
     * @fn handleRestartEvent
     * @brief Handle event to restart the MessageService to activate new server comm configuration.
     * @details The restart is executed through the event queue for thread safety usage of UdpService
     *
     * @param TASK_ID_t
     * @param ptrInstance
     * @param data
     */
    static void handleRestartEvent(SDVEventQueue::TASK_ID_t, void* ptrInstance, void * data);

private:
    ConfigurationParser * _configParser;    /**< holds configutation needed for creating sockets and message retry informtation  */
    SDVEventQueue * _queue;                 /**< sending/receiving messages will be managed in the event queue  */
    UdpService * _udpService;               /**< instance of UDPService for sending/recieving packets */

    /**
     * Type of map that holds messages handlers for un-solicited messages that
     * are not part of a transaction.
     */
    typedef std::map<sdv::SDVMessage::TRANACTION_ID_t, UNSOLICTED_MESSAGE_HANDLER*> MESSAGE_HANDLERS_t;
    MESSAGE_HANDLERS_t _messageHandlers;      /**< Map of un-solicited message handlers */

    /**
     * Type of map that holds messages handlers for un-solicited messages that
     * are not part of a transaction.
     */
    typedef std::map<sdv::SDVMessage::TRANACTION_ID_t, CALLBACK_DATA_t*> RESPONSE_HANDLERS_t;
    RESPONSE_HANDLERS_t _responseMessageHandlers ;  /**< Map of response message handlers */
protected:

    /**
     * @fn createUdpService
     *
     * @brief Creates a new instance of a UDP Service for sending/recieving of
     * packets with the SDV Server.
     *
     * @param [in] pointer to Configuration object
     *
     * @return the newly created UdpService.  It can be assumed the UdpService has started successfully
     * and is ready to be used.
     */
    virtual UdpService * createUdpService(ConfigurationParser* config);

    /**
     * @fn parseMessage
     *
     * @brief Parse an Incoming SDVMessage from raw bytes     *
     * @param [in] data - raw data to constrcut the SDV message from.
     *
     * @return parsed SDVMessage or NULL if message type could not be determined.
     */
    virtual SDVMessage * parseMessage(unsigned char * data);
private:
    static void hexDump(unsigned char * data, size_t len);
};

} /* namespace sdv */

#endif /* MESSAGESERVICE_H_ */
