/*
* @file UdpService.cpp
*
* @brief Provides UDP communications services for SDV.
*
* @par Open Issues (in no particular order)
* -# None
*
* @par Assumptions
* -# None
*
* @par Implementation Notes
* -# None
*/

#include <stdio.h>
#include <stdlib.h>
#include <string>
#include <string.h>
#include <assert.h>
#include <pthread.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <errno.h>

#include "UdpService.h"
#include "rdk_debug.h"


#define LOG(...) RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", __VA_ARGS__)
#define LOG_ERR(...) RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.SDVAGENT", __VA_ARGS__)


using namespace sdv;

UdpService::UdpService(DATA_RX_CALLBACK* recv_callback,void* callbackInstance, const char* server_ip_addr, const uint16_t server_port) : recv_sock(0) {
	LOG("UdpService.<constructor>: ip=%s port=%u\n", server_ip_addr, server_port);

	rx_callback = recv_callback;
	rx_callback_instance = callbackInstance;

	assert(pthread_mutex_init(&lock, NULL) == 0);
	initBuffPool();
	setSendSocket(server_port, server_ip_addr);
	createSocketReadThread();

	// To ensure socket receive is invoked before user initiates socket send, wait for thread to create socket
	int delayCnt = 10;
	while ((recv_sock == 0) && (delayCnt-- > 0)) {
		LOG("UdpService.<constructor>: waiting for socket read thread to start\n");
		sleep(1);
	}
	assert(delayCnt != 0);
}

UdpService::~UdpService() {
	LOG("UdpService.<destructor>: start shutdown\n");

	shutdown(send_sock, SHUT_RDWR);
	close(send_sock);

	shutdown(recv_sock, SHUT_RDWR);
	close(recv_sock);
	recv_sock = -1;  // indicates to socketReader we're shutting down

	pthread_join(thread_id, NULL);	// wait for socketReader thread to exit
	pthread_attr_destroy(&thread_attr);

	free(server_addr);

	pthread_mutex_destroy(&lock);

	freeBuffPool();
	LOG("UdpService.<destructor>: delete complete\n");
}

void UdpService::setSendSocket(uint16_t server_port, const char* server_ip_addr) {
	// If string contains IPv4 specific character, setup IPv4 server socket and address
	if (strchr(server_ip_addr, '.') != NULL) {
		server_addr = (struct sockaddr *) createIp4SocketAddr(server_port, server_ip_addr);
		server_addr_len = sizeof(struct sockaddr_in);
		send_sock = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
	}
	// Else setup for IPv6
	else {
		server_addr = (struct sockaddr *) createIp6SocketAddr(server_port, server_ip_addr);
		server_addr_len = sizeof(struct sockaddr_in6);
		send_sock = socket(AF_INET6, SOCK_DGRAM, IPPROTO_UDP);
	}
	assert(send_sock > 0);
}

void* UdpService::socketReader(void* arg) {
	struct sockaddr* recv_addr;
	socklen_t addr_len;
	PACKET_BUFF* pkt = NULL;
	UdpService* obj = (UdpService *)arg;

	LOG("UdpService.socketReader: starting...\n");

	// If IPv4 server address, create IPv4 socket address to receive on
	if (obj->server_addr->sa_family == AF_INET) {
		uint16_t net_port = ((struct sockaddr_in *) obj->server_addr)->sin_port;			// receive port same as send port
		recv_addr = (struct sockaddr *) obj->createIp4SocketAddr(ntohs(net_port), NULL);	// null indicates any address
		addr_len = sizeof(struct sockaddr_in);
		obj->recv_sock = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
	}
	// Else create IPv6 socket address to receive on
	else {
		uint16_t net_port = ((struct sockaddr_in6 *) obj->server_addr)->sin6_port;			// receive port same as send port
		recv_addr = (struct sockaddr *) obj->createIp6SocketAddr(ntohs(net_port), NULL);	// null indicates any address
		addr_len = sizeof(struct sockaddr_in6);
		obj->recv_sock = socket(AF_INET6, SOCK_DGRAM, IPPROTO_UDP);
	}
	assert(obj->recv_sock >= 0);

	if (bind(obj->recv_sock, recv_addr, addr_len) < 0) {
		LOG_ERR("UdpService.createIp4SocketAddr: bind failed\n");
		close(obj->recv_sock);
	}

	// Read from socket forever...
	while (true) {
		pkt = obj->getEmptyPacket();
		assert(pkt != NULL);

		LOG("UdpService.socketReader: wait for recvfrom\n");

		int recv_len = pkt->data_length = recvfrom(obj->recv_sock, pkt->data, pkt->max_length, 0, recv_addr, &addr_len);
		if (recv_len >= 1) {
			LOG("UdpService.socketReader: packet received; len=%u\n", recv_len);
			pkt->data_length = recv_len;
			(*(obj->rx_callback))(pkt, obj->rx_callback_instance);  // callback function now responsible for freeing packet
		}
		else {
			LOG_ERR("UdpService.socketReader: recvfrom failed with %u. errno: 0x%x, recv_len: %d\n", pkt->data_length, errno, recv_len);
			obj->freePacket(pkt);
			if (obj->recv_sock < 0) {
				LOG_ERR("UdpService.socketReader: socket was closed, exiting thread\n");
				break;
			}
			sleep(1);
		}
	}
	if (obj->recv_sock > 0) {
		close(obj->recv_sock);
	}
	free(recv_addr);
	pthread_exit(NULL);
	LOG("UdpService.socketReader: ending...\n");
}

struct sockaddr_in* UdpService::createIp4SocketAddr(const uint16_t port, const char* ip4) {

	struct sockaddr_in* new_sockaddr = (struct sockaddr_in *) malloc(sizeof(struct sockaddr_in));
	assert(new_sockaddr != NULL);

	memset(new_sockaddr, 0, sizeof(new_sockaddr));
	new_sockaddr->sin_family = AF_INET;
	new_sockaddr->sin_port = htons(port);

	// If we have an IPv4 address string, convert to network and set sin_addr
	if (ip4 != NULL) {
		inet_pton(AF_INET, ip4, &(new_sockaddr->sin_addr));
	}
	// Else use any address
	else {
		new_sockaddr->sin_addr.s_addr = htonl(INADDR_ANY);
	}

	LOG("UdpService.createIp4SocketAddr: socket address created for %s\n", ip4);
	return new_sockaddr;
}

struct sockaddr_in6* UdpService::createIp6SocketAddr(const uint16_t port, const char* ip6) {

	struct sockaddr_in6* new_sockaddr = (struct sockaddr_in6 *) malloc(sizeof(struct sockaddr_in6));
	assert(new_sockaddr != NULL);

	memset(new_sockaddr, 0, sizeof(new_sockaddr));
	new_sockaddr->sin6_family = AF_INET6;
	new_sockaddr->sin6_port = htons(port);

	// If we have an IPv6 address string, convert to network and set sin_addr
	if (ip6 != NULL) {
		inet_pton(AF_INET6, ip6, &(new_sockaddr->sin6_addr));
	}
	// Else use any address
	else {
		new_sockaddr->sin6_addr = in6addr_any;
	}

	LOG("UdpService.createIp6SocketAddr: socket address created for %s\n", ip6);
	return new_sockaddr;
}

void UdpService::createSocketReadThread() {
	// Create thread with default scheduling, priority, and stack.  May need to change???
	pthread_attr_init(&thread_attr);
	pthread_attr_setdetachstate(&thread_attr,  PTHREAD_CREATE_JOINABLE);

	pthread_create(&thread_id, &thread_attr, &UdpService::socketReader, this);

#if 0 // Check default stack size
	size_t stacksize;
	pthread_attr_init(&thread_attr);
	pthread_attr_getstacksize(&thread_attr, &stacksize);
	printf("UdpService::createSocketThread: default stack size = %lu\n", stacksize);
#endif
}

void UdpService::initBuffPool() {
	buff_link_head = NULL;

	for (int i = 0; i < TOTAL_UDP_PACKET_BUFFS; i++) {
		PACKET_BUFF* packet = (PACKET_BUFF*) malloc(sizeof(struct packet_buff));
		assert(packet != NULL);
		packet->data_length = 0;
		packet->max_length = UDP_PACKET_BUFF_LEN;
		packet->data = (uint8_t*) malloc(UDP_PACKET_BUFF_LEN);

		// Place new packet on head of linked list
		packet->next = buff_link_head;
		buff_link_head = packet;
	}
}

void UdpService::freeBuffPool() {
	uint32_t delete_count = 0;
	while (buff_link_head != NULL) {
		PACKET_BUFF * todelete = buff_link_head;
		buff_link_head = todelete->next;
		free(todelete->data);
		free(todelete);
		++delete_count;
	}
	LOG("UdpService.freeBuffPool: %u buffs deleted\n", delete_count);
	return;
}

UdpService::PACKET_BUFF* UdpService::getEmptyPacket() {
	PACKET_BUFF* buff;

	pthread_mutex_lock(&lock);

	if (buff_link_head != NULL) {
		buff = buff_link_head;
		buff_link_head = buff->next;
		buff->data_length = 0;
	} else {
		buff = NULL;
		LOG_ERR("UdpService.getEmptyPacket: out of packets!\n");
	}

	pthread_mutex_unlock(&lock);
	return buff;
}

void UdpService::freePacket(PACKET_BUFF* buff) {
	pthread_mutex_lock(&lock);

	// Add it to the list head
	buff->next = buff_link_head;
	buff_link_head = buff;

	pthread_mutex_unlock(&lock);
}

void UdpService::sendPacket(PACKET_BUFF* buff) {
	LOG("UdpService.sendPacket: len=%u\n", buff->data_length);

	if (sendto(send_sock, buff->data, buff->data_length, 0, server_addr, server_addr_len) < 0) {
		perror("UdpService.sendPacket: sendto failed");
	}

	freePacket(buff);
}

void UdpService::printSockAddrInfo(struct sockaddr* dual_addr) {
	char* addr_string;
	uint16_t port;
	if (dual_addr->sa_family ==  AF_INET) {
		struct sockaddr_in* addr = (struct sockaddr_in*)dual_addr;
		char str[INET_ADDRSTRLEN];
		addr_string = str;
		inet_ntop(AF_INET, &(addr->sin_addr), addr_string, INET_ADDRSTRLEN);
		port = ntohs(addr->sin_port);
	}
	else {
		struct sockaddr_in6* addr = (struct sockaddr_in6*)dual_addr;
		char str[INET6_ADDRSTRLEN];
		addr_string = str;
		inet_ntop(AF_INET, &(addr->sin6_addr), addr_string, INET6_ADDRSTRLEN);
		port = ntohs(addr->sin6_port);
	}
	LOG("UdpService.printSockAddrInfo: addr=%s  port=%u\n", addr_string, port);
}
