/**
* @file SDVMessage.cpp
* @Author Jason Bedard (jasbedar@cisco.com)
* @Date Jun 23 2014
*
* @brief Function implementations for SDVMessage class
*/
#include <string.h>
#include <netinet/in.h>
#include <assert.h>

#include "SDVMessage.h"

namespace sdv {


#define BYTE_MASK 0xFF
#define SDV_MESSAGE_ID_OFFSET 2                 /**< Offset into serialized SDVMessage where messageId can be found*/
#define PROTOCOL_DESCRIPTOR 0x11                /**< SDV protocol descriptor defined in CCMIS 3.14 */
#define PROTOCOL_DSCMCC_TYPE 0x04               /**< SDV DSMCC Type defined in CCMIS 3.14 */
#define SESSION_ID_SIZE 10	                /**< Size of the SessionID which is in all messages but not in DMSCC Header */

int SDVMessage::lastTransactionId = 0;

SDVMessage::SDVMessage(uint8_t* macAddress, MessageId messageId, uint8_t tunerIndex, uint16_t payloadLength) {
    memcpy(_macAddress, macAddress, sizeof(_macAddress));
    _messageId = messageId;
    _transactionId = lastTransactionId++ & 0x3fffffff;
    _tunerIndex = tunerIndex;
    _payloadLength = payloadLength + SESSION_ID_SIZE;
}

SDVMessage::SDVMessage(uint8_t* macAddress, MessageId messageId, uint8_t tunerIndex, uint16_t payloadLength, uint32_t transactionId) {
    memcpy(_macAddress, macAddress, sizeof(_macAddress));
    _messageId = messageId;
    _transactionId = transactionId;
    _tunerIndex = tunerIndex;
    _payloadLength = payloadLength + SESSION_ID_SIZE;
}

SDVMessage::SDVMessage(CCMIS_HEADER* header) {
	_messageId = (MessageId)ntohs(header->message_id);
    _transactionId = ntohl(header->transaction_id);
    memcpy(_macAddress, header->session_mac, sizeof(_macAddress));
    _tunerIndex = header->session_tuner_index;
    _payloadLength = header->message_len;
}

SDVMessage::MessageId SDVMessage::getMessageType(uint8_t* buffer){
    buffer+=SDV_MESSAGE_ID_OFFSET;
    uint16_t mId = ((*buffer++ & BYTE_MASK ) << 8);
    mId |= (*buffer++ & BYTE_MASK);
    return (MessageId)mId;
}

size_t SDVMessage::serialize(uint8_t* emptyBuff, size_t buffLength) {
	CCMIS_HEADER* header = (CCMIS_HEADER*)emptyBuff;
	assert(buffLength >= sizeof(struct ccmis_header));

	header->protocol = PROTOCOL_DESCRIPTOR;
	header->dsmcc_type = PROTOCOL_DSCMCC_TYPE;
	header->message_id = htons(_messageId);
	header->transaction_id = htonl(_transactionId);
	header->reserved1 = 0xff;
	header->adaptation_len = 0x00;
	header->message_len = htons(_payloadLength);
	memcpy(header->session_mac, _macAddress, sizeof(_macAddress));
	header->session_tuner_index = _tunerIndex;

	return sizeof(struct ccmis_header);
}

} // sdv namespace
