#include "libIBus.h"
#include <assert.h>
#include <map>
#include <iostream>


typedef std::map <const char*, void*> RPC_MAP;

RPC_MAP rpcMap;
pthread_mutex_t mapLock;				//!< mutex for accessing map data




struct libIBusMock_broadcastedEvent IARM_MOCK_broadcastedEvent;

/**
 * @brief Initialize the IARM-Bus library.
 *
 * This function initialize the IARM-Bus library and register the application to the bus.
 * The @p name is the <b>well-known</b> name of the application.  Once concatenated with the
 * IARM-Bus name, it becomes an universally unique name for the application.  This name can
 * be thought of as the namespace of the events or RPC methods exported by the application.
 *
 * After the library is initialized, the application is ready to access events and RPC methods
 * on the bus.
 *
 * @param name well-known application name..
 * @return IARM_Result_t Error Code.
 * @retval IARM_RESULT_SUCCESS on success
 */
IARM_Result_t IARM_Bus_Init(const char *name){
	IARM_MOCK_broadcastedEvent.eventId = 0xffffffff;
	IARM_MOCK_broadcastedEvent.ownerName = "";
    return IARM_RESULT_SUCCESS;
}
/**
 * @brief Terminate the IARM-Bus library.
 *
 * This function releases resources allocated by the IARM Bus Library. After it is called,
 * the library returns to the state prior to <tt>IARM_Bus_Init</tt> is called.
 *
 * @return IARM_Result_t Error Code.
 * @retval IARM_RESULT_SUCCESS on success
 */
IARM_Result_t IARM_Bus_Term(void){
    return IARM_RESULT_SUCCESS;
}
/**
 * @brief Connected to the IARM-Bus Daemon.
 *
 * The bus, when connected to the IARM-Bus, is not known to the <b>Bus-Daemon</b> until this API
 * is called.  The <tt>connect</tt> function register the application with the Bus daemon.
 * Bus Daemon only coordinate resource sharing among the applications connected to the bus.
 * Normally this API should be called right after <tt>IARM_Bus_Init()</tt>.
 *
 * @return IARM_Result_t Error Code.
 * @retval IARM_RESULT_SUCCESS on success
 */
IARM_Result_t IARM_Bus_Connect(void){
    return IARM_RESULT_SUCCESS;
}
/**
 * @brief Disconnect application from the IARM-Bus Daemon.
 *
 * @return IARM_Result_t Error Code.
 * @retval IARM_RESULT_SUCCESS on success
 */
IARM_Result_t IARM_Bus_Disconnect(void){
    return IARM_RESULT_SUCCESS;
}

/**
 * @brief Returns group context of the calling member
 *
 * @return IARM_Result_t Error Code.
 * @retval IARM_RESULT_SUCCESS on success
 */

IARM_Result_t IARM_Bus_GetContext(void **context){
    return IARM_RESULT_SUCCESS;
}
/**
 * @brief Send an event onto the bus.
 *
 * Event is always broadcast onto the bus. Only listening applications will be awakened.
 * The registered event handlers (via <tt>IARM_Bus_RegisterEventHandler</tt>) are invoked
 * asynchronously.
 *
 * @p ownerName is the well-known name of the application sending the event. It can be thought
 * of as the namespace of the event being sent. @p data holds the event data.  This data structure
 * is allocated internally by IARM and need not be freed by application.  The event data stored
 * at @p data is no longer available after the event handler is returned. <b>Application should make
 * a copy of the event data</b> if it wishes to access the data after event handler is returned.
 *
 * Normally the application use strcmp() to dispatch events based on ownerName and then a
 * switch block to dispatch events based on eventId.
 *
 * @param [in] ownerName is the source of the event.
 * @param [in] eventId is the enum value of the event.
 * @param [in] data is the data structure holding event data.
 * @param [in] len is the sizeof the event data. 
 *
 * @return IARM_Result_t Error Code.
 * @retval IARM_RESULT_SUCCESS on success
 */
IARM_Result_t IARM_Bus_BroadcastEvent(const char *ownerName, IARM_EventId_t eventId, void *data, size_t len){
	std::cout << "Mock IARM: BroadcastEvent for " << ownerName << '\n';

	IARM_MOCK_broadcastedEvent.ownerName = ownerName;
	IARM_MOCK_broadcastedEvent.eventId = eventId;
    return IARM_RESULT_SUCCESS;
}

/**
 * @brief Check if current process is registered with UI Manager.
 *
 * In the event of a crash at UI Manager, the manager may have lost memory of registered process.
 * A member process can call this function regularly to see if it is still registered.
 *
 * @param [in] memberName 
 * @param [out] isRegistered True if still registered.
 *
 * @return IARM_Result_t Error Code.
 * @retval IARM_RESULT_SUCCESS on success
 */
IARM_Result_t IARM_Bus_IsConnected(const char *memberName, int *isRegistered){
    return IARM_RESULT_SUCCESS;
}

/**
 * @brief Register Event Handler to handle events.
 *
 * Each event can only be associated with 1 event handler. If an event handler is already registered, the old one
 * will be replaced with the new one.
 *
 * @param [in] ownerName well-known name of the application sending the event.
 * @param [in] eventId is the enum value of the event.
 * @param [in] handler callback function to process the event.
 *
 * @return IARM_Result_t Error Code.
 * @retval IARM_RESULT_SUCCESS on success
 *
 * @see IARM_Bus_BroadcastEvent
 * @see IARM_EventHandler_t
 */
IARM_Result_t IARM_Bus_RegisterEventHandler(const char *ownerName, IARM_EventId_t eventId, IARM_EventHandler_t handler){
    return IARM_RESULT_SUCCESS;
}

/**
 * @brief Unregister to remove from listeners of the event.
 *
 * This API remove the process from the listeners of the event.
 
 * @param [in] ownerName well-known name of the application sending the event.
 * @param [in] eventId The event whose listener to remove.
 *
 * @return IARM_Result_t Error Code.
 * @retval IARM_RESULT_SUCCESS on success
 */
IARM_Result_t IARM_Bus_UnRegisterEventHandler(const char *ownerName, IARM_EventId_t eventId){
    return IARM_RESULT_SUCCESS;
}
/**
 * @brief Register an RPC Method.
 *
 * Register an RPC method that can be invoked by other applications.  The @p methodName is the string name
 * used to invoke the RPC method. @p handler is the implementation of the RPC method. When other application
 * invoke the method via its string name, the function pointed to by the handler is executed.
 *
 * @param [in] methodName well-known name of the RPC Method. This name lives within the name sapce of the application name.
 * @param [in] handler implementation of the RPC method..
 *
 * @return IARM_Result_t Error Code.
 * @retval IARM_RESULT_SUCCESS on success
 *
 * @see IARM_BusCall_t
 */
IARM_Result_t IARM_Bus_RegisterCall(const char *methodName, IARM_BusCall_t handler){
	rpcMap[methodName] = (void*)handler;
    return IARM_RESULT_SUCCESS;
}
/**
 * @brief Invoke an RPC Method.
 *
 * Invoke RPC method by its application name and method name.  The @p arg is data structure holding all input
 * and output parameters of the invocation. @p argLen is result of sizeof() applied on the @p arg data structure.
 *
 * @param [in] ownerName well-known name of the application sending the event.
 * @param [in] methodName well-known name of the RPC Method. 
 * @param [in] arg is data structure holding all input and output parameters of the invocation.
 * @param [in] argLen is the sizeof arg.
 *
 * @return IARM_Result_t Error Code.
 * @retval IARM_RESULT_SUCCESS on success
 *
 * @see IARM_BusCall_t
 */
IARM_Result_t IARM_Bus_Call(const char *ownerName,  const char *methodName, void *arg, size_t argLen){
	RPC_MAP::iterator iter = rpcMap.find(methodName);
	assert(iter != rpcMap.end());
	IARM_BusCall_t handler = (IARM_BusCall_t)iter->second;
	(*handler)(arg);
	return IARM_RESULT_SUCCESS;
}
/**
 * @brief Register Events.
 *
 * Register all the events that are published by the application.  An application can publish multipel events.
 * These events must each have an enumeration value starting 0. The @p maxEventId is the enum value of the last
 * event + 1.  This API register all events whose enum value is less than @p maxEventId at once.
 *
 * Once an event is registered, it can be broadcast on to the bus and received by listening applications.
 *
 * @param [in] maxEventId is the enum value of the last event + 1
 *
 * @return IARM_Result_t Error Code.
 * @retval IARM_RESULT_SUCCESS on success
 *
 * @see IARM_Bus_BroadcastEvent
 */
IARM_Result_t IARM_Bus_RegisterEvent(IARM_EventId_t maxEventId){
    return IARM_RESULT_SUCCESS;
}
