#include "SDVAgent.h"

#include "rdk_debug.h"
#include "rdk_error.h"
#include "sdv_iarm.h"
#include "libIBus.h"
#include <string>



#define DEFAULT_DEBUG_CONF_FILE "/mnt/nfs/env/debug.ini"

bool enableRmfSdv();

bool done = false;


//Callback for handling system signals
void signalHandler (int signum ){
    switch(signum){
        case SIGTERM:
        case SIGINT:
            done = true;
            break;
        default:
            break;
    }
}

/**
 * Main entry point for sdvAgent process.
 * Process will exit if it cannot enable RMF SDV.
 * @param argc
 * @param argv
 * @return
 */
int main(int argc, char ** argv) {
    const char* debugConfigFile = NULL;
    int itr=0;
    while (itr < argc){
        if(strcmp(argv[itr], "--debugconfig")==0){
            itr++;
            if(itr < argc) {
                debugConfigFile = argv[itr];
            }
            else {
                break;
            }
        }
        itr++;
    }

    if(rdk_logger_init(debugConfigFile) != RDK_SUCCESS) {
        rdk_logger_init(DEFAULT_DEBUG_CONF_FILE);
    }

    //Register callback for system signals
    signal(SIGTERM, &signalHandler);
    signal(SIGINT, &signalHandler);

    // Initialize and connect our IARM bus so we can use it to attempt to enable RMF SDV in rmfStreamer
    if (IARM_Bus_Init(IARM_BUS_SDVAGENT_NAME) != IARM_RESULT_SUCCESS) {
        RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.SDVAGENT", "[SDVAgent_main] %s: IARM_Bus_Init failed\n", __FUNCTION__);
        return 0;
    }
    if (IARM_Bus_Connect() != IARM_RESULT_SUCCESS) {
        RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.SDVAGENT", "[SDVAgent_main] %s: IARM_Bus_Connect failed\n", __FUNCTION__);
        return 0;
    }

    // If RMF SDV enable succeeds, start SDV Agent
    if (enableRmfSdv()) {
        RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "[SDVAgent_main] %s: RMF SDV enabled, continue starting agent\n", __FUNCTION__);
        SdvAgent myagent = SdvAgent();
        myagent.start();

        // Start SDV Agent crash monitor script if script file exists and is not already running
        if ((system("which /lib/rdk/monitorSDV.sh") == 0) && (system("pidof monitorSDV.sh") != 0)) {
            system("/lib/rdk/monitorSDV.sh &");
        } 

        // Main Loop, wait until a system signal is received then exit.
        while(!done){
            pause();
        }
    // Else assume we are on a non-sdv network and exit the agent
    } else {
        RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "[SDVAgent_main] %s: RMF SDV failed to enable, exiting process\n", __FUNCTION__);
    }

    IARM_Bus_Disconnect();
    IARM_Bus_Term();

    return 1;
}

/**
 * @fn enableRmfSdv
 *
 * @brief Attempt to enable mediaframework SDV subcomponent using IARM call.
 * @details RMF SDV will return an enable failure if we are on a non-SDV network.
 * To handle the case where the sdvAgent has started before RMF SDV has platform initialized,
 * this function continuously loops with a delay until the IARM_BUS call succeeds.
 *
 * @return true if RMF SDV successfully enabled; otherwise false
 */
bool enableRmfSdv() {
    bool isEnabled = FALSE;
    unsigned int retryCount = 2;

    // Initialize payload result in case IARM_Bus_Call fails without errors
    IARM_RMFSDV_ENABLE_PAYLOAD payload;
    payload.result = IARM_RMFSDV_ENABLE_PAYLOAD::NO_RESPONSE;

    // In case rmfStreamer has not yet loaded, loop until a result response is received
    while (payload.result == IARM_RMFSDV_ENABLE_PAYLOAD::NO_RESPONSE) {
        RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "[SDVAgent_main] %s: invoke IARM_BUS_RMFSDV_ENABLE\n", __FUNCTION__);
        IARM_Result_t iarmResult = IARM_Bus_Call("rmfStreamer", IARM_BUS_RMFSDV_ATTEMPT_ENABLE, &payload, sizeof(IARM_RMFSDV_ENABLE_PAYLOAD));

        if (IARM_RESULT_SUCCESS != iarmResult) {
            RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.SDVAGENT", "[SDVAgent_main] %s: IARM_Bus_Call failed with %d\n", __FUNCTION__, iarmResult);
            sleep(5);
        } else if (payload.result == IARM_RMFSDV_ENABLE_PAYLOAD::NO_RESPONSE) {
            if(0 != retryCount--) {
                RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.SDVAGENT", "[SDVAgent_main] %s: RMFSDV failed to get a response; retry after 1s.\n", __FUNCTION__);
                sleep(1);
            }
            else {
                RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.SDVAGENT", "[SDVAgent_main] %s: failed to get a response; retries exhausted. Return failure.\n", __FUNCTION__);
                return FALSE;
            }
        } else if (payload.result == IARM_RMFSDV_ENABLE_PAYLOAD::ENABLE_REJECT) {
            RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "[SDVAgent_main] %s: enable rejected\n", __FUNCTION__);
        } else if (payload.result == IARM_RMFSDV_ENABLE_PAYLOAD::ENABLE_SUCCESS) {
            isEnabled = TRUE;
            RDK_LOG(RDK_LOG_INFO, "LOG.RDK.SDVAGENT", "[SDVAgent_main] %s: enable succeeded\n", __FUNCTION__);
        } else {
            RDK_LOG(RDK_LOG_ERROR, "LOG.RDK.SDVAGENT", "[SDVAgent_main] %s: unknown payload result %d\n", __FUNCTION__, payload.result);
        }
    }
    return isEnabled;
}
