/**
 * @file SDVEventQueue.h
 *
 * @brief Event Queue for SDVAgent services to asynchronously execute functions.
 *
 * SDV Sub services need to delegate handling of external events to a processing
 * while not blocking the calling thread. An would be the delegation of handling
 * a UDP Socket read so as not to block the OS callback thread.
 *
 * Additionally some services will need to handle timeout used cases. An
 * example of this would be timeing out while waiting for a response message
 * from the SDVServer.
 *
 * All events which are posted to the SDVEventQueue are fired FIFO
 * from a single thread.
 */
#ifndef SDVEVENTQUEUE_H_
#define SDVEVENTQUEUE_H_

#define ONE_SECOND_IN_NANO 1000000000L
#define ONE_MILLI_IN_NANO 1000000

#include <queue>
#include <map>
#include <stddef.h>
#include <stdint.h>
#include <pthread.h>
#include <stdlib.h>
#include <unistd.h>

#include "SDVService.h"
#include "Log.h"

namespace sdv {

class SDVEventQueue : public sdv::SDVService {
public:
	/**
	 * @var TASK_ID_t
	 * Identifier of a task which has been posted to the SDVEventQueue
	 */
	typedef int TASK_ID_t;

	/** @brief SDVEventQueue callback Function template
	 *
	 * Template for callback function point which gets invoked when an event is fired.
	 */
	typedef void (*SDV_EVENT_CALLBACK_t)(TASK_ID_t, void* ptrInstance, void * data);

public:
	SDVEventQueue(): _nextTaskId(0), _running(false){}

	/**
	 * @fn start()
	 *
	 * @brief Starts the SDVEventQueue service and event thread.
	 * Once started events can be posted and will be fired.
	 *
	 * @return Status of SDVEventQueue service
	 */
	sdv::SDVService::SERVICE_RESULT start();

	/**
	 * @fn stop()
	 *
	 * @brief Stops the SDVEventQueue service and event thread.
	 * call will block until event queue thread has been killed.
	 *
	 * @return Status of SDVEventQueue service
	 */
	sdv::SDVService::SERVICE_RESULT stop();

	/**
	 * @fn push()
	 *
	 * @brief Push a new event to the queue to be executed asynchronously
	 *
	 * Event Will fire on a FIFO basis triggering the callback from within
	 * the SDVEventQueue thread.
	 *
	 * @param ptrInstance the instance of the object whom's callback will be invoked
	 * when the event is fired.
	 *
	 * @param callback function to be invoked when the event fires.
	 *
	 * @param data optional event specific structure of data. If no event data required can be NULL.
	 * Any memory used by data needs to be manager by the caller/reciver and will not be reclaimed
	 * by the SDVEventQueue.
	 *
	 * @param timeout_ms the amount of time in milliseconds to wait before event fires.
	 * A timeout of 0 indicates the event should be fired as soon as possible.
	 *
	 * @return the TASK_ID of the new event.
	 */
	virtual TASK_ID_t push(void* ptrInstance, SDV_EVENT_CALLBACK_t callback,  void * data, uint32_t timeout_ms);

private:

	/**
	 * @fn doLoop()
	 *
	 * @brief Main loop for execution of queued tasks.
	 *
	 * While running the event queue will be inspected for events.  If the queue
	 * has any events the first event will be inspected to determine if it is ready to fire.
	 *
	 * If event even is ready to fire the event's callback will be invoked from within the thread
	 * but from outside of the events_mutex.  after the event is fired the loop will continue.
	 *
	 * If the event is not ready to fire a wait will occur until either another event is posted to
	 * the queue or the event time is reached.
	 *
	 * If no events are in the queue a wait will occur until an even is posted to the queue.
	 *
	 * @param [in] eventQueueInstance pointer to the instance of the event queue.
	 */
	static void *doLoop(void * eventQueueInstance );
private:
	/**
	 * @struct QueuedEvent
	 * @brief Representation of an event which has been posted to the event queue
	 */
	struct QueuedEvent{
	    sdv::SDVEventQueue::TASK_ID_t _id;
	    sdv::SDVEventQueue::SDV_EVENT_CALLBACK_t _callback;
	    void* _ptrInstance;
	    void * _data;
	    uint64_t _timeout;	//!< absolute time event expires in milliseconds

	    /**
	     * @Constructor for QueuedEvents
	     *
	     * @param id the identifier assigned to the task
	     * @param callback function to be invoked when event is fired.
	     * @param ptrInstance pointer to the instance of a class if a member function was
	     *  used for the callback so state data is available
	     * @param data payload of the event to be forwarded to callback when fired
	     * @param the amount of time to wait for the event is fired in ms.
	     */
	    QueuedEvent(sdv::SDVEventQueue::TASK_ID_t id, sdv::SDVEventQueue::SDV_EVENT_CALLBACK_t callback, void* ptrInstance, void * data, long timeout_ms);
	};
	/**
	 * @struct QueuedItemComparator
	 * @brief  comparator for QueuedEvents on behalf of a priorty_queue
	 *
	 * Events will need to be sorted from low to high relative to how much
	 * time is left before the event should fire.
	 */
	struct QueuedItemComparator{
	    bool operator()(const sdv::SDVEventQueue::QueuedEvent * a, const sdv::SDVEventQueue::QueuedEvent * b) const;
	};

private:
    TASK_ID_t _nextTaskId;
    bool _running;                       /**< state of the Event Queue thread */
    pthread_mutex_t * _events_mutex;     /**< mutex used to control access to event queue */
    pthread_cond_t * _event_add_cv;      /**< pthread conditional to wait on/notify events */
    pthread_t * _runThread;              /**< reference to the event queue thread */
    pthread_attr_t * _attr;              /**< attributes for the event queue pthead */
    void * _status;                      /**< start of the pthread join operation */

	/**
	 * @var queuedEvents()
	 *
	 * @brief Queue that holds events which have been posted to the queue.
	 *
	 * Access to the event queue should be limited to within a critical section,
	 * Events will be sorted based on the time they should fire with shorted time
	 * at the top.
	 */
	std::priority_queue<sdv::SDVEventQueue::QueuedEvent*, std::vector<sdv::SDVEventQueue::QueuedEvent*>, QueuedItemComparator > _queuedEvents;

	static uint64_t getNowTimeInMillis();
};
}
#endif /* SDVEVENTQUEUE_H_ */
