/*
* ============================================================================
* RDK MANAGEMENT, LLC CONFIDENTIAL AND PROPRIETARY
* ============================================================================
* This file (and its contents) are the intellectual property of RDK Management, LLC.
* It may not be used, copied, distributed or otherwise  disclosed in whole or in
* part without the express written permission of RDK Management, LLC.
* ============================================================================
* Copyright (c) 2014 RDK Management, LLC. All rights reserved.
* ============================================================================
*/

#ifndef SDVSESSIONHANDLER_H_
#define SDVSESSIONHANDLER_H_

#include <stddef.h>
#include <pthread.h>
#include <map>
#include <cstdlib>
#include <cstring>
#include <assert.h>

#include "sdv_iarm.h"
#include "SDVService.h"
#include "IARMProxyService.h"
#include "ConfigurationParser.h"
#include "MessageService.h"
#include "SDVEventQueue.h"
#include "TRMMonitorService.h"


namespace sdv {

/**
 * @class SDVSessionHandler
 * @brief Handles and tracks the opening and closing of SDV tuning sessions.
 * @details Sessions are opened and closed via IARM remote procedure calls.  SDV tuning details are returned to the remote
 * caller in the open function.  Opening of linear channel tunes is also reported to fulfill CCMIS requirements.
 */
class SDVSessionHandler : SDVService {

public:

	/**
	 * @struct SDV_SESSION
	 * @brief Structure to maintain details on an SDV session
	 */
	typedef struct sdv_session {
		enum {
			OPENING,	//!> new session is waiting for ProgramSelectConfirm message from server
			ACTIVE,		//!> session is actively streaming
			CLOSING,	//!> session is waiting for timeout or re-use of its tuner ID
			FAILED		//!> session program select failed with server
		} state;		//!> state of this sdv session
		uint32_t sourceId;		//!> service ID
		uint32_t tunerId;		//!> tuner index
		SDVAGENT_TUNING_INFO tuningInfo;

		sdv_session() {};
		sdv_session(uint32_t source_id, uint32_t tuner_id) {
			sourceId = source_id;
			tunerId = tuner_id;
			state = OPENING;
		}
	} SDV_SESSION;

	/**
	 * @fn createInstance
	 * @brief Create a single instance of the SDVSessionHandler.
	 * @details Handler is not ready to open sessions until start has been invoked.
	 *
	 * @param iarmProxy
	 * @param messageService
	 * @param eventQueue
	 * @param config
	 * @param trmService
	 * @return SDVSessionHandler object
	 */
	static SDVSessionHandler* createInstance(IARMProxyService* iarmProxy, MessageService* messageService,
		   SDVEventQueue* eventQueue, ConfigurationParser* config, TRMMonitorService* trmService);

    /**
     * @fn start
     * @brief Readies session handler and allows it to open sessions.
     * @details Signals openSession thread in case its blocked waiting for ready
     *
     * @return SERVICE_RESULT  the result of trying to start the service  SUCCESS if started ok ortherwise FAILURE
     *
     * @retval SUCCESS           If registered correctly with IARM
     * @retval FAILURE           Registration with IARM Failed
     **/
   SDVService::SERVICE_RESULT start();

   SDVService::SERVICE_RESULT stop();

   /**
    * @fn getOpenSession
    * @brief Get the session data structure for the specified tuner
    *
    * @param [in] tunerId
    * @param [out] session - pointer to callers memory in which to store session (if found)
    * @return 0 if active or opening session found; -1 if session not found, not opening, or not active
    */
   virtual int getOpenSession(uint32_t tunerId, SDV_SESSION* session);

   /**
    * @fn getAnyOpenSession
    * @brief Get session data structure for any open session (for "fake" UserActivityReports)
    *
    * @param [out] session - pointer to callers memory in which to store session (if found)
    * @return 0 if active or opening session found; -1 if no opening or active sessions found
    */
   virtual int getAnyOpenSession(SDV_SESSION* session);

   /**
    * @fn forceTuneAllSessions
    * @brief Initiates all SDV sessions to be forced tune
    */
   virtual void forceTuneAllSessions();

   /**
    * @fn forceTune
    * @brief Initiates a force tune of the specified session
    * @details Broadcasts a force tune event to mediaframework.  Its expected that mediaframework will close the QAMSrc for the
    * tuner which shall result in the corresponding session being closed.
    *
    * @param session - pointer to SDV Session object
    */
   virtual void forceTune(SDV_SESSION* session);

   /**
    * @fn getOpenSessionTunerId
    * @brief Get the tuner index for the specified source ID of an open session
    *
    * @param sourceId
    * @return tuner index or -1 if source ID not found in open session list
    */
   virtual int getOpenSessionTunerId(uint32_t sourceId);

private:
    /**
     * @name openSession
     * @brief IARM RPC for opening a new SDV Session
     *
     * @param [in] *sessionPayload  pointer to session information needed to open new SDV session
     * @retval IARM_RESULT_SUCCESS     Successfully created new SDV Session
     * @retval Error Code              Failed to create new SDV Session
     *
     */
    static IARM_Result_t openSession(iarm_sdvagent_open_session_payload *sessionPayload);

    /**
     * @name closeSession
     * @brief IARM RPC for closing SDV Session
     *
     * @param [in] *sessionPayload  pointer to session information needed to close a SDV session
     * @retval IARM_SUCCESS     Successfully closed the SDV Session
     * @retval IARM_FAILURE     Failed to close SDV Session
     *
     */
    static IARM_Result_t closeSession(iarm_sdvagent_close_session_payload *sessionPayload);

    /**
     * @name startStream
     * @brief IARM RPC for starting a SDV Stream
     *
     * @param [in] *sessionPayload  pointer to session information needed to start a SDV Stream
     * @retval IARM_SUCCESS     Successfully started the SDV Stream
     * @retval IARM_FAILURE     Failed to start SDV Stream
     *
     */
    static IARM_Result_t startStream(iarm_sdvagent_stream_start_payload *sessionPayload);

    /**
     * @name stopStream
     * @brief IARM RPC for stopping a SDV Stream
     *
     * @param [in] *sessionPayload  pointer to session information needed to stop a SDV Stream
     * @retval IARM_SUCCESS     Successfully stopped the SDV Stream
     * @retval IARM_FAILURE     Failed to stop SDV Stream
     *
     */
    static IARM_Result_t stopStream(iarm_sdvagent_stream_stop_payload *sessionPayload);


protected:
    /**
     * @name SDVSessionHandler()
     *
     * @brief Constructor for the SDVSessionHandler
     */
    SDVSessionHandler(IARMProxyService* iarmProxy, MessageService* messageService, SDVEventQueue* queue, ConfigurationParser* config, TRMMonitorService* trmService);

public:
    ~SDVSessionHandler();


private:

    typedef struct callbackWrapper {
        IARMProxyService::IARMProxyEventHander_t _callback;
        void * _ptrObj;
        SDVEventQueue _queue;
        callbackWrapper(SDVEventQueue queue, IARMProxyService::IARMProxyEventHander_t callback, void * ptrObj)
            : _callback(callback), _ptrObj(ptrObj), _queue(queue){}
        void iarmInvoked(void * data){
            _callback(_ptrObj, data);
        }
    }CALLBACK_WRAPPER_TYPE;

	/**
	 * @def SESSION_CLOSE_TUNEAWAY_DELAY
	 * @brief Milliseconds to delay before sending tune away message to server for closing session
	 */
	#define SESSION_CLOSE_TUNEAWAY_DELAY 10000

    /**
     * @def HANDLER_NOT_READY_DELAY_SECS
     * @brief Delay to wait for session handler to be ready.
     * @details SdvAgent readies session handler when it receives InitConfirm
     */
	#define HANDLER_NOT_READY_DELAY_SECS 4  //IARM Timeout is 5 seconds

	/**
	 * @typedef SDV_SESSION_MAP
	 * @brief Map containing SDV sessions;
	 * @details first=tunerId, second=SDV_SESSION*
	 */
	typedef std::map<uint32_t, SDV_SESSION*> SDV_SESSION_MAP;
	SDV_SESSION_MAP sessionList;        	//!> list of SDV sessions

    IARMProxyService *_iarmProxy;       //!< reference to IARMProxyService
    MessageService *_messageService;    //!< reference to MessageService
    SDVEventQueue * _queue;             //!< reference to SDVEventQueue
    ConfigurationParser * _config;      //!< reference to SDVEventQueue
	TRMMonitorService* _trmService;
    pthread_mutex_t sessions_mutex;   	//!< mutex used to control access to session data
    pthread_cond_t sessions_cv;       	//!< pthread conditional to wait on/notify when session data is available
    uint32_t lastOpenedSessionTunerId;	//!< track the tuner ID of the last session opened
    bool isReady;						//!< indicates handler has started and is ready for open sessions


    /**
     * @fn sendProgramSelectRequest
     * @brief Send a program select request message to the sdv server with the specified parameters and expect
     * a callback for ProgramSelectConfirm.
     *
     * @param session
     */
    void sendProgramSelectRequest(SDV_SESSION* session);

    /**
    * @fn sendProgramSelectRequestWithCallback
    * @brief Send a program select request message to the sdv server with the specified parameters and not expect
    * a callback for ProgramSelectConfirm
    *
    * @param sourceId
    * @param tunerId
    * @param tunerUseFlags
    */
    void sendConfirmLessProgramSelectRequest(uint32_t sourceId, uint32_t tunerId, uint8_t tunerUseFlags);

    /**
     * @fn closingCallback
     * @brief Callback from Event Queue to indicate a session closing delay has completed.
     * @details Tune away ProgramSelectRequest will be sent to the server only if the tuner used
     * by the closing session is still not being used for sdv session
     *
     * @param qTaskId
     * @param ptrInstance
     * @param data
     */
    static void sessionTuneAwayCallback(SDVEventQueue::TASK_ID_t qTaskId, void* ptrInstance, void* data);

    /**
     * @fn getSessionBySourceId
     * @brief Get a reference to an existing session object for the specified source ID.
     *
     * @param sourceId
     * @return reference to SDV_SESSION; null if no sessions exist for the ID
     */
    SDV_SESSION* getSessionBySourceId(uint32_t sourceId);

    /**
     * @fn getSessionByTunerId
     * @brief Get a reference to an existing session object for the specified tuner ID.
     *
     * @param tunerId
     * @return reference to SDV_SESSION; null if no sessions exist for the ID
     */
    SDV_SESSION* getSessionByTunerId(uint32_t tunerId);

    /**
     * @fn createSession
     * @brief Create a new session for the specified source and tuner IDs.
     *
     * @param sourceId
     * @param tunerId
     * @return reference to new session object
     */
    SDV_SESSION* createSession(uint32_t sourceId, uint32_t tunerId);

    /**
     * @fn deleteSession
     * @brief Delete the session with the specified tuner ID
     *
     * @param tunerId
     */
    void deleteSession(uint32_t tunerId);

    /**
     * @fn handleLinearOpen
     * @brief Handle an open for a linear service
     *
     * @param sourceId
     * @param tunerId
     */
    void handleLinearOpen(uint32_t sourceId, uint32_t tunerId, uint8_t tunerUseFlags);

    /**
     * @fn handleFailedOpen
     * @brief Handle a session that failed to open
     * @details Session will fail to open if server returns a ProgramSelectConfirm response code error or if
     * confirm message never received after timeouts and retries.
     *
     * @param session - SDV_SESSION struct of the failed session
     */
    void handleFailedOpen(SDV_SESSION* session);

    /**
     * @fn programSelectConfirm
     * @brief Callback from MessageService to provide ProgramSelectConfirm response message.
     *
     * @param [in] ptrObject - pointer to our session handler object
     * @param [in] status - status of the message send
     * @param [in] response - ProgramSelectConfirm message; NULL if time-out and message never received
     */
    static void programSelectConfirm(void * ptrObject, MessageService::REPONSE_STATUS_t status, sdv::SDVMessage * response);

    /**
     * @fn sendInitRequestForUnknownClient
     * @brief Send an InitRequest message due to an unknown client error detected during a session open
     * @param session - SDV session that was being opened
     */
    void sendInitRequestForUnknownClient(SDV_SESSION* session);

    /**
     * @fn initConfirmForUnknownClient
     * @brief Callback to receive an InitConfirm message following an unknown client error detected during a session open
     * @details If the InitConfirm succeeded, resends the ProgramSelectRequest to open the session.  If the Init failed,
     * fails the session open.
     *
     * @param ptrObject - pointer to SDV_SESSION that was attempting to be opened
     * @param status - status of message send
     * @param msg - pointer to InitConfirm message received from SDV Server
     */
    static void initConfirmForUnknownClient(void * ptrObject, MessageService::REPONSE_STATUS_t status, sdv::SDVMessage * msg);
};
}

#endif /* SDVSESSIONHANDLER_H_ */
